/* packet-woww.c
 * Routines for World of Warcraft World dissection
 * Copyright 2021, Gtker <woww@gtker.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/*
 * The protocol is used for World of Warcraft World packets.
 * These are seen when a client is connected to a world server and plays the game.
 * The WOW protocol (no extra W) packets are Login packets, and they are handled in
 * the packet-wow.c file.
 *
 * The message definitions are automatically generated from the `wowm` files at
 * https://github.com/gtker/wow_messages.
 *
 * More info on world packets and login packets:
 * https://wowdev.wiki/World_Packet
 * https://wowdev.wiki/Login_Packet
 *
 * Currently this dissector is valid for 1.12.x, the most popular Vanilla version.
 *
 * All World packets contain a header with:
 * * A 16 bit big endian size field.
 * * A (32 or 16 bit) little endian opcode field.
 * Server to client opcodes are 16 bits while client to server opcodes are 32 bits.
 *
 * All world packets other than SMSG_AUTH_CHALLENGE and CMSG_AUTH_SESSION have
 * "encrypted" headers based on a 40 byte session key, however it is relatively
 * easily broken.
 *
 * SMSG packets are Server messages (from server) and CMSG packets are Client messages
 * (from client). MSG packets can be either.
 *
 * # SESSION KEY DEDUCTION:
 *
 * The header is encrypted through the formula `E = (x ^ S) + L` where:
 * * E is the encrypted value.
 * * x is the plain unencrypted value.
 * * S is a byte of the session key.
 * * L is the last encrypted value.
 *
 * The header is decrypted through the formula `x = (E - L) ^ S` with the same values.
 *
 * Notably, this allows us to deduce the session key value S if we know what the
 * unencrypted value x is. The L value is simply the last encrypted value sent.
 *
 * Fortunately, the client sends opcodes as 32bit little endian values, but there are no
 * opcodes that use the two most significant bytes meaning we can always count on them being 0.
 * This means we can now deduce the session key value S through `S = 0 ^ (E - L)` (where 0 is x).
 * Because of this we can deduce 2 bytes of the session key every client packet.
 */

#include <config.h>
#include <epan/packet.h>
#include <epan/conversation.h>
#include <epan/wmem_scopes.h>
#include <ptvcursor.h>

void proto_reg_handoff_woww(void);
void proto_register_woww(void);
static dissector_handle_t woww_handle;

static int proto_woww;

/* Fields that all packets have */
static int hf_woww_size;
static int hf_woww_opcode;

static int hf_woww_string_length;

static int hf_woww_aura_mask;
static int hf_woww_aura;
static int hf_woww_amount_of_masks;
static int hf_woww_update_mask;
static int hf_woww_update_mask_index;
static int hf_woww_update_mask_value;
static int hf_woww_amount_of_splines;
static int hf_woww_packed_spline;

/* AUTOGENERATED_START_HF */
static int hf_woww_absorb;
static int hf_woww_absorbed;
static int hf_woww_absorbed_damage;
static int hf_woww_account_data_type;
static int hf_woww_action;
static int hf_woww_action_bar;
static int hf_woww_action_bars;
static int hf_woww_action_type;
static int hf_woww_activate_taxi_reply;
static int hf_woww_actual_roll;
static int hf_woww_addon_crc;
static int hf_woww_addon_extra_crc;
static int hf_woww_addon_has_signature;
static int hf_woww_addon_name;
static int hf_woww_addon_type;
static int hf_woww_agility;
static int hf_woww_ai_reaction;
static int hf_woww_allow_movement;
static int hf_woww_allowed_class;
static int hf_woww_allowed_genders;
static int hf_woww_allowed_guild_id;
static int hf_woww_allowed_maximum_level;
static int hf_woww_allowed_minimum_level;
static int hf_woww_allowed_race;
static int hf_woww_ammo_display_id;
static int hf_woww_ammo_inventory_type;
static int hf_woww_ammo_type;
static int hf_woww_amount;
static int hf_woww_amount_bought;
static int hf_woww_amount_for_sale;
static int hf_woww_amount_of_accounts_in_guild;
static int hf_woww_amount_of_auras;
static int hf_woww_amount_of_carriers;
static int hf_woww_amount_of_characters;
static int hf_woww_amount_of_characters_in_guild;
static int hf_woww_amount_of_choice_item_rewards;
static int hf_woww_amount_of_cooldowns;
static int hf_woww_amount_of_damages;
static int hf_woww_amount_of_effects;
static int hf_woww_amount_of_emotes;
static int hf_woww_amount_of_entries;
static int hf_woww_amount_of_events;
static int hf_woww_amount_of_extra_fields;
static int hf_woww_amount_of_faction_standings;
static int hf_woww_amount_of_factions;
static int hf_woww_amount_of_friends;
static int hf_woww_amount_of_gossip_items;
static int hf_woww_amount_of_hits;
static int hf_woww_amount_of_ignored;
static int hf_woww_amount_of_item_rewards;
static int hf_woww_amount_of_items;
static int hf_woww_amount_of_logs;
static int hf_woww_amount_of_mails;
static int hf_woww_amount_of_members;
static int hf_woww_amount_of_misses;
static int hf_woww_amount_of_nodes;
static int hf_woww_amount_of_objects;
static int hf_woww_amount_of_outbid_items;
static int hf_woww_amount_of_petitions;
static int hf_woww_amount_of_pets;
static int hf_woww_amount_of_players;
static int hf_woww_amount_of_quests;
static int hf_woww_amount_of_raid_infos;
static int hf_woww_amount_of_reactions;
static int hf_woww_amount_of_records;
static int hf_woww_amount_of_required_items;
static int hf_woww_amount_of_rights;
static int hf_woww_amount_of_signatures;
static int hf_woww_amount_of_spells;
static int hf_woww_amount_of_states;
static int hf_woww_amount_of_strings;
static int hf_woww_amount_of_targets;
static int hf_woww_amount_of_teammates;
static int hf_woww_amount_of_zones;
static int hf_woww_angle;
static int hf_woww_animation_id;
static int hf_woww_answer;
static int hf_woww_answer_comment;
static int hf_woww_apply;
static int hf_woww_arcane_resistance;
static int hf_woww_area;
static int hf_woww_argument;
static int hf_woww_armor;
static int hf_woww_attacker;
static int hf_woww_auction_command_action;
static int hf_woww_auction_command_result;
static int hf_woww_auction_command_result_two;
static int hf_woww_auction_duration_in_minutes;
static int hf_woww_auction_house;
static int hf_woww_auction_id;
static int hf_woww_auction_main_category;
static int hf_woww_auction_out_bid;
static int hf_woww_auction_outbid;
static int hf_woww_auction_slot_id;
static int hf_woww_auction_sub_category;
static int hf_woww_auctioneer;
static int hf_woww_aura_duration;
static int hf_woww_aura_slot;
static int hf_woww_aura_type;
static int hf_woww_auto_decline_time;
static int hf_woww_auto_finish;
static int hf_woww_autocast_enabled;
static int hf_woww_average_wait_time_in_ms;
static int hf_woww_background_color;
static int hf_woww_backwards_running_speed;
static int hf_woww_backwards_swimming_speed;
static int hf_woww_bag;
static int hf_woww_bag_family;
static int hf_woww_bag_index;
static int hf_woww_bag_slot;
static int hf_woww_bag_type_subclass;
static int hf_woww_battlefield_port_action;
static int hf_woww_battleground_bracket;
static int hf_woww_battleground_end_status;
static int hf_woww_battleground_winner;
static int hf_woww_battlegrounds;
static int hf_woww_battlemaster;
static int hf_woww_bg_type_id;
static int hf_woww_bid;
static int hf_woww_bidder;
static int hf_woww_billing_flags;
static int hf_woww_billing_rested;
static int hf_woww_billing_time;
static int hf_woww_block;
static int hf_woww_blocked;
static int hf_woww_blocked_amount;
static int hf_woww_body;
static int hf_woww_body_text;
static int hf_woww_bonding;
static int hf_woww_bonus_honor;
static int hf_woww_border_color;
static int hf_woww_border_style;
static int hf_woww_bug_type;
static int hf_woww_build;
static int hf_woww_button;
static int hf_woww_buy_bank_slot_result;
static int hf_woww_buy_price;
static int hf_woww_buy_result;
static int hf_woww_buyback_slot;
static int hf_woww_buyout;
static int hf_woww_buyout_amount;
static int hf_woww_cash_on_delivery_amount;
static int hf_woww_cast_failure_reason;
static int hf_woww_cast_flags;
static int hf_woww_cast_item;
static int hf_woww_caster;
static int hf_woww_category_cooldown;
static int hf_woww_channel;
static int hf_woww_channel_flags;
static int hf_woww_channel_member_flags;
static int hf_woww_channel_name;
static int hf_woww_channel_password;
static int hf_woww_character_flags;
static int hf_woww_character_guid;
static int hf_woww_character_name;
static int hf_woww_charter_display_id;
static int hf_woww_charter_entry;
static int hf_woww_charter_owner;
static int hf_woww_chat_credit;
static int hf_woww_chat_data_line_count;
static int hf_woww_chat_notify;
static int hf_woww_chat_type;
static int hf_woww_checked_timestamp;
static int hf_woww_cinematic_sequence_id;
static int hf_woww_civilian;
static int hf_woww_class;
static int hf_woww_class_mask;
static int hf_woww_client_instance_id;
static int hf_woww_client_proof;
static int hf_woww_client_seed;
static int hf_woww_code;
static int hf_woww_coded;
static int hf_woww_comment;
static int hf_woww_compressed_chat_data;
static int hf_woww_compressed_data;
static int hf_woww_compressed_move_opcode;
static int hf_woww_container_slots;
static int hf_woww_content;
static int hf_woww_cooldown;
static int hf_woww_cooldown_count;
static int hf_woww_cooldown_time;
static int hf_woww_corpse;
static int hf_woww_corpse_query_result;
static int hf_woww_cos_angle;
static int hf_woww_cost_in_copper;
static int hf_woww_count;
static int hf_woww_countdown_time;
static int hf_woww_counter;
static int hf_woww_created_day;
static int hf_woww_created_month;
static int hf_woww_created_year;
static int hf_woww_creature;
static int hf_woww_creature_entry;
static int hf_woww_creature_family;
static int hf_woww_creature_id;
static int hf_woww_creature_rank;
static int hf_woww_creature_type;
static int hf_woww_critical;
static int hf_woww_current_health;
static int hf_woww_current_power;
static int hf_woww_damage;
static int hf_woww_damage_float;
static int hf_woww_damage_maximum;
static int hf_woww_damage_minimum;
static int hf_woww_damage_state;
static int hf_woww_damage_uint;
static int hf_woww_data;
static int hf_woww_data_type;
static int hf_woww_datetime;
static int hf_woww_days_since_last_updated;
static int hf_woww_days_since_oldest_ticket_creation;
static int hf_woww_days_since_ticket_creation;
static int hf_woww_deadline;
static int hf_woww_deaths;
static int hf_woww_debug_log_format;
static int hf_woww_decompressed_size;
static int hf_woww_delay;
static int hf_woww_delay_time;
static int hf_woww_description;
static int hf_woww_destination_bag;
static int hf_woww_destination_node;
static int hf_woww_destination_slot;
static int hf_woww_destionation_slot;
static int hf_woww_details;
static int hf_woww_dismount_result;
static int hf_woww_display_id;
static int hf_woww_duel_winner_reason;
static int hf_woww_durability;
static int hf_woww_duration;
static int hf_woww_eff;
static int hf_woww_emblem_color;
static int hf_woww_emblem_style;
static int hf_woww_emote_delay;
static int hf_woww_emote_enum;
static int hf_woww_emote_int;
static int hf_woww_enable_pvp;
static int hf_woww_enchantment;
static int hf_woww_encrypted_data;
static int hf_woww_end_text;
static int hf_woww_ended_without_interruption;
static int hf_woww_enemy;
static int hf_woww_energize_amount;
static int hf_woww_energize_power;
static int hf_woww_energy;
static int hf_woww_entry;
static int hf_woww_entry_id;
static int hf_woww_environmental_damage_type;
static int hf_woww_equip_error;
static int hf_woww_equipment_display_id;
static int hf_woww_equipped_item_class;
static int hf_woww_equipped_item_inventory_type_mask;
static int hf_woww_equipped_item_subclass_mask;
static int hf_woww_event_descriptions;
static int hf_woww_exp_group_bonus;
static int hf_woww_experience;
static int hf_woww_experience_award_type;
static int hf_woww_experience_reward;
static int hf_woww_experience_without_rested;
static int hf_woww_expiration_time;
static int hf_woww_extend_flag;
static int hf_woww_extra_attacks;
static int hf_woww_face;
static int hf_woww_facial_hair;
static int hf_woww_faction;
static int hf_woww_faction_flag;
static int hf_woww_fall_time;
static int hf_woww_far_sight_operation;
static int hf_woww_feed_pet_item;
static int hf_woww_fields;
static int hf_woww_fire_resistance;
static int hf_woww_first_bag_display_id;
static int hf_woww_first_bag_inventory_id;
static int hf_woww_first_login;
static int hf_woww_first_rank;
static int hf_woww_flags;
static int hf_woww_focus;
static int hf_woww_friend_result;
static int hf_woww_friend_status;
static int hf_woww_frost_resistance;
static int hf_woww_gain_multiplier;
static int hf_woww_gameobject;
static int hf_woww_gender;
static int hf_woww_gift_bag_index;
static int hf_woww_gift_slot;
static int hf_woww_gift_wrapper;
static int hf_woww_gm_ticket_escalation_status;
static int hf_woww_gm_ticket_queue_status;
static int hf_woww_gm_ticket_response;
static int hf_woww_gm_ticket_status;
static int hf_woww_gm_ticket_status_response;
static int hf_woww_gm_ticket_type;
static int hf_woww_gold;
static int hf_woww_gossip_list_id;
static int hf_woww_grade;
static int hf_woww_greeting;
static int hf_woww_group_loot_setting;
static int hf_woww_group_member_online_status;
static int hf_woww_group_number;
static int hf_woww_group_type;
static int hf_woww_group_update_flags;
static int hf_woww_guid;
static int hf_woww_guids;
static int hf_woww_guild;
static int hf_woww_guild_charter_cost;
static int hf_woww_guild_command;
static int hf_woww_guild_command_result;
static int hf_woww_guild_emblem_result;
static int hf_woww_guild_event;
static int hf_woww_guild_id;
static int hf_woww_guild_info;
static int hf_woww_guild_member_status;
static int hf_woww_guild_name;
static int hf_woww_hair_color;
static int hf_woww_hair_style;
static int hf_woww_happiness;
static int hf_woww_has_transport;
static int hf_woww_heal_amount;
static int hf_woww_heal_critical;
static int hf_woww_health;
static int hf_woww_higher_bidder;
static int hf_woww_highest_bid;
static int hf_woww_highest_bidder;
static int hf_woww_hit_info;
static int hf_woww_hits;
static int hf_woww_holy_resistance;
static int hf_woww_homebind_timer;
static int hf_woww_honor_points;
static int hf_woww_honorable_kills;
static int hf_woww_horizontal_speed;
static int hf_woww_icon;
static int hf_woww_id;
static int hf_woww_ignored;
static int hf_woww_inactive;
static int hf_woww_index;
static int hf_woww_info_block;
static int hf_woww_info_type;
static int hf_woww_initiator;
static int hf_woww_initiator_name;
static int hf_woww_instance_id;
static int hf_woww_instance_reset_failed_reason;
static int hf_woww_intellect;
static int hf_woww_interrupted_spell;
static int hf_woww_inventory_result;
static int hf_woww_inventory_type;
static int hf_woww_invited_player;
static int hf_woww_is_applied;
static int hf_woww_is_frozen;
static int hf_woww_is_online;
static int hf_woww_issue_date;
static int hf_woww_item;
static int hf_woww_item_bag_index;
static int hf_woww_item_charges;
static int hf_woww_item_class;
static int hf_woww_item_class_and_sub_class;
static int hf_woww_item_count;
static int hf_woww_item_creator;
static int hf_woww_item_display_id;
static int hf_woww_item_enchant_id;
static int hf_woww_item_enchantment;
static int hf_woww_item_flag;
static int hf_woww_item_icon;
static int hf_woww_item_id;
static int hf_woww_item_level;
static int hf_woww_item_limit_category_id;
static int hf_woww_item_name;
static int hf_woww_item_owner;
static int hf_woww_item_quality;
static int hf_woww_item_random_properties_id;
static int hf_woww_item_random_property_id;
static int hf_woww_item_random_suffix;
static int hf_woww_item_random_suffix_id;
static int hf_woww_item_set;
static int hf_woww_item_slot;
static int hf_woww_item_slot_int;
static int hf_woww_item_spell_charges;
static int hf_woww_item_stack_count;
static int hf_woww_item_stack_size;
static int hf_woww_item_stat_type;
static int hf_woww_item_sub_class_mask;
static int hf_woww_item_suffix_factor;
static int hf_woww_item_template;
static int hf_woww_item_text_id;
static int hf_woww_item_to_damage;
static int hf_woww_items_required;
static int hf_woww_join_as_group;
static int hf_woww_key_version;
static int hf_woww_kill_count;
static int hf_woww_killing_blows;
static int hf_woww_lag;
static int hf_woww_language;
static int hf_woww_last_week_honor;
static int hf_woww_last_week_honorable;
static int hf_woww_leader;
static int hf_woww_level;
static int hf_woww_level_played_time;
static int hf_woww_lifetime_dishonorable;
static int hf_woww_lifetime_honorable;
static int hf_woww_list_from;
static int hf_woww_list_start_item;
static int hf_woww_listed_players;
static int hf_woww_living_orientation;
static int hf_woww_location_name;
static int hf_woww_lock_id;
static int hf_woww_log_format;
static int hf_woww_logout_result;
static int hf_woww_logout_speed;
static int hf_woww_loot;
static int hf_woww_loot_master;
static int hf_woww_loot_method;
static int hf_woww_loot_method_error;
static int hf_woww_loot_slot;
static int hf_woww_loot_slot_type;
static int hf_woww_looted_target;
static int hf_woww_loyalty;
static int hf_woww_mail_action;
static int hf_woww_mail_id;
static int hf_woww_mail_result;
static int hf_woww_mail_result_two;
static int hf_woww_mail_template_id;
static int hf_woww_mail_type;
static int hf_woww_mailbox;
static int hf_woww_mailbox_id;
static int hf_woww_mana;
static int hf_woww_map;
static int hf_woww_master_loot;
static int hf_woww_material;
static int hf_woww_max_count;
static int hf_woww_max_durability;
static int hf_woww_max_health;
static int hf_woww_max_items;
static int hf_woww_max_level_money_reward;
static int hf_woww_max_power;
static int hf_woww_maximum;
static int hf_woww_maximum_level;
static int hf_woww_maximum_signatures;
static int hf_woww_meeting_stone_failure;
static int hf_woww_meeting_stone_status;
static int hf_woww_member;
static int hf_woww_message;
static int hf_woww_message_id;
static int hf_woww_message_of_the_day;
static int hf_woww_minimum;
static int hf_woww_minimum_bid;
static int hf_woww_minimum_level;
static int hf_woww_minimum_signatures;
static int hf_woww_misc;
static int hf_woww_misc_value;
static int hf_woww_money;
static int hf_woww_money_in_trade;
static int hf_woww_money_reward;
static int hf_woww_monster;
static int hf_woww_monster_move_type;
static int hf_woww_monster_name;
static int hf_woww_motd;
static int hf_woww_mount_result;
static int hf_woww_move_event;
static int hf_woww_movement_counter;
static int hf_woww_movement_flags;
static int hf_woww_multiplier;
static int hf_woww_name;
static int hf_woww_nature_resistance;
static int hf_woww_nearest_node;
static int hf_woww_new_bid;
static int hf_woww_new_guild_leader_name;
static int hf_woww_new_item_chat_alert;
static int hf_woww_new_item_creation_type;
static int hf_woww_new_item_source;
static int hf_woww_new_level;
static int hf_woww_new_name;
static int hf_woww_new_owner;
static int hf_woww_new_speed;
static int hf_woww_new_spell_id;
static int hf_woww_next_page_id;
static int hf_woww_next_quest_in_chain;
static int hf_woww_next_resurrect_time;
static int hf_woww_node_count;
static int hf_woww_nodes;
static int hf_woww_note;
static int hf_woww_notification;
static int hf_woww_npc;
static int hf_woww_number_of_battlegrounds;
static int hf_woww_number_of_choices;
static int hf_woww_object_type;
static int hf_woww_object_unk;
static int hf_woww_objective_text;
static int hf_woww_objective_texts;
static int hf_woww_objectives;
static int hf_woww_offer_reward_text;
static int hf_woww_officer_note;
static int hf_woww_old_mover;
static int hf_woww_old_spell_id;
static int hf_woww_online_players;
static int hf_woww_op;
static int hf_woww_opponent_name;
static int hf_woww_orientation;
static int hf_woww_out_bid;
static int hf_woww_outbid_item_ids;
static int hf_woww_outfit_id;
static int hf_woww_owner;
static int hf_woww_page_id;
static int hf_woww_page_text;
static int hf_woww_page_text_material;
static int hf_woww_party_operation;
static int hf_woww_party_result;
static int hf_woww_party_status;
static int hf_woww_periodic_log;
static int hf_woww_pet;
static int hf_woww_pet_command_state;
static int hf_woww_pet_current_health;
static int hf_woww_pet_current_power;
static int hf_woww_pet_display_id;
static int hf_woww_pet_enabled;
static int hf_woww_pet_feedback;
static int hf_woww_pet_level;
static int hf_woww_pet_max_health;
static int hf_woww_pet_max_power;
static int hf_woww_pet_name;
static int hf_woww_pet_name_timestamp;
static int hf_woww_pet_number;
static int hf_woww_pet_react_state;
static int hf_woww_pet_slot;
static int hf_woww_pet_talk_reason;
static int hf_woww_pet_tame_failure_reason;
static int hf_woww_petition;
static int hf_woww_petition_id;
static int hf_woww_petition_result;
static int hf_woww_pitch;
static int hf_woww_player;
static int hf_woww_player_chat_tag;
static int hf_woww_player_is_saved_to_a_raid;
static int hf_woww_player_name;
static int hf_woww_player_rank;
static int hf_woww_player_with_killing_blow;
static int hf_woww_point_map_id;
static int hf_woww_point_opt;
static int hf_woww_position;
static int hf_woww_position_x;
static int hf_woww_position_x_int;
static int hf_woww_position_y;
static int hf_woww_position_y_int;
static int hf_woww_power;
static int hf_woww_price;
static int hf_woww_probability;
static int hf_woww_public_key;
static int hf_woww_public_note;
static int hf_woww_pvp_corpse;
static int hf_woww_pvp_rank;
static int hf_woww_query;
static int hf_woww_quest_completable;
static int hf_woww_quest_failed_reason;
static int hf_woww_quest_flags;
static int hf_woww_quest_giver_status;
static int hf_woww_quest_icon;
static int hf_woww_quest_id;
static int hf_woww_quest_level;
static int hf_woww_quest_method;
static int hf_woww_quest_party_message;
static int hf_woww_quest_title;
static int hf_woww_quest_type;
static int hf_woww_question_id;
static int hf_woww_queue_position;
static int hf_woww_queue_slot;
static int hf_woww_race;
static int hf_woww_race_mask;
static int hf_woww_racial_leader;
static int hf_woww_rage;
static int hf_woww_raid_group_error;
static int hf_woww_raid_instance_message;
static int hf_woww_raid_target_index;
static int hf_woww_raid_target_update_type;
static int hf_woww_random_property;
static int hf_woww_random_property_id;
static int hf_woww_ranged_range_modification;
static int hf_woww_rank;
static int hf_woww_rank_id;
static int hf_woww_rank_name;
static int hf_woww_rank_names;
static int hf_woww_rank_progress_bar;
static int hf_woww_raw_data;
static int hf_woww_read_by_gm;
static int hf_woww_realm_name;
static int hf_woww_receiver;
static int hf_woww_records;
static int hf_woww_reputation_objective_value;
static int hf_woww_reputation_rank;
static int hf_woww_request_items_text;
static int hf_woww_requested_rank;
static int hf_woww_required_city_rank;
static int hf_woww_required_faction_rank;
static int hf_woww_required_honor_rank;
static int hf_woww_required_item_count;
static int hf_woww_required_item_id;
static int hf_woww_required_kill_count;
static int hf_woww_required_level;
static int hf_woww_required_money;
static int hf_woww_required_opposite_reputation_value;
static int hf_woww_required_skill_rank;
static int hf_woww_required_skill_value;
static int hf_woww_required_spell;
static int hf_woww_required_spell_focus;
static int hf_woww_required_spells;
static int hf_woww_reserved_for_future_use;
static int hf_woww_reset_time;
static int hf_woww_resist;
static int hf_woww_resisted;
static int hf_woww_reward;
static int hf_woww_reward_spell;
static int hf_woww_reward_spell_cast;
static int hf_woww_rights;
static int hf_woww_roll_number;
static int hf_woww_roll_vote;
static int hf_woww_round_time_in_ms;
static int hf_woww_running_speed;
static int hf_woww_scale;
static int hf_woww_search_strings;
static int hf_woww_searched_name;
static int hf_woww_self_player;
static int hf_woww_sell_item_result;
static int hf_woww_sell_price;
static int hf_woww_sender;
static int hf_woww_sender_id;
static int hf_woww_sender_name;
static int hf_woww_sequence_id;
static int hf_woww_server_id;
static int hf_woww_server_message_type;
static int hf_woww_server_seed;
static int hf_woww_set_assistant;
static int hf_woww_shadow_resistance;
static int hf_woww_sheath_state;
static int hf_woww_sheathe_type;
static int hf_woww_show_affiliation;
static int hf_woww_signer;
static int hf_woww_simple_spell_cast_result;
static int hf_woww_sin_angle;
static int hf_woww_size_struct;
static int hf_woww_skill;
static int hf_woww_skin;
static int hf_woww_skin_color;
static int hf_woww_slot;
static int hf_woww_slot_id;
static int hf_woww_slot_index;
static int hf_woww_sound_id;
static int hf_woww_source_bag;
static int hf_woww_source_item_id;
static int hf_woww_source_node;
static int hf_woww_source_slot;
static int hf_woww_speech_bubble_credit;
static int hf_woww_speed;
static int hf_woww_spell;
static int hf_woww_spell_art_kit;
static int hf_woww_spell_cast_result;
static int hf_woww_spell_cast_target_flags;
static int hf_woww_spell_category;
static int hf_woww_spell_category_cooldown;
static int hf_woww_spell_charges;
static int hf_woww_spell_cooldown;
static int hf_woww_spell_cost;
static int hf_woww_spell_count;
static int hf_woww_spell_data_id;
static int hf_woww_spell_effect;
static int hf_woww_spell_id;
static int hf_woww_spell_index;
static int hf_woww_spell_miss_info;
static int hf_woww_spell_on_lowest_slot;
static int hf_woww_spell_school;
static int hf_woww_spell_school_mask;
static int hf_woww_spell_trigger_type;
static int hf_woww_spell_visual_kit;
static int hf_woww_spells;
static int hf_woww_spirit;
static int hf_woww_spirit_released;
static int hf_woww_spline_elevation;
static int hf_woww_spline_flag;
static int hf_woww_spline_id;
static int hf_woww_stable_master;
static int hf_woww_stable_result;
static int hf_woww_stable_slots;
static int hf_woww_stack_count;
static int hf_woww_stackable;
static int hf_woww_stamina;
static int hf_woww_standing;
static int hf_woww_start_bid;
static int hf_woww_start_from_page;
static int hf_woww_start_quest;
static int hf_woww_starting_bid;
static int hf_woww_state;
static int hf_woww_stationery;
static int hf_woww_status;
static int hf_woww_status_id;
static int hf_woww_strength;
static int hf_woww_string;
static int hf_woww_sub_name;
static int hf_woww_subject;
static int hf_woww_suggestion;
static int hf_woww_summoner;
static int hf_woww_survey_id;
static int hf_woww_swap_with_name;
static int hf_woww_swimming_speed;
static int hf_woww_talent;
static int hf_woww_talent_point_cost;
static int hf_woww_talent_reset_cost;
static int hf_woww_target;
static int hf_woww_target_error;
static int hf_woww_target_player;
static int hf_woww_target_string;
static int hf_woww_targets;
static int hf_woww_taxi_mask_node_known;
static int hf_woww_text;
static int hf_woww_text_emote;
static int hf_woww_text_id;
static int hf_woww_texts;
static int hf_woww_this_week_honor;
static int hf_woww_this_week_honorable;
static int hf_woww_time;
static int hf_woww_time_in_queue_in_ms;
static int hf_woww_time_left;
static int hf_woww_time_offline;
static int hf_woww_time_passed;
static int hf_woww_time_remaining;
static int hf_woww_time_skipped;
static int hf_woww_time_to_bg_autoleave_in_ms;
static int hf_woww_time_to_bg_start_in_ms;
static int hf_woww_time_to_remove_in_queue_in_ms;
static int hf_woww_timer;
static int hf_woww_timer_type;
static int hf_woww_timescale;
static int hf_woww_timestamp;
static int hf_woww_title;
static int hf_woww_title_text_id;
static int hf_woww_today_honorable_and_dishonorable;
static int hf_woww_todo_amount_of_signers;
static int hf_woww_total_amount_of_auctions;
static int hf_woww_total_cost;
static int hf_woww_total_damage;
static int hf_woww_total_exp;
static int hf_woww_total_played_time;
static int hf_woww_trade_item;
static int hf_woww_trade_slot;
static int hf_woww_trade_slot_count;
static int hf_woww_trade_slot_number;
static int hf_woww_trade_status;
static int hf_woww_trainer_spell_state;
static int hf_woww_trainer_type;
static int hf_woww_training_failure_reason;
static int hf_woww_transfer_abort_reason;
static int hf_woww_transport;
static int hf_woww_transport_guid;
static int hf_woww_transport_orientation;
static int hf_woww_transport_progress_in_ms;
static int hf_woww_trigger_id;
static int hf_woww_turn_rate;
static int hf_woww_tutorial_data;
static int hf_woww_tutorial_flag;
static int hf_woww_type_flags;
static int hf_woww_unit_stand_state;
static int hf_woww_unit_target;
static int hf_woww_unknown_flags;
static int hf_woww_unknown_float;
static int hf_woww_unknown_guid;
static int hf_woww_unknown_int;
static int hf_woww_unread_mails;
static int hf_woww_unused;
static int hf_woww_update_available_flag;
static int hf_woww_update_flag;
static int hf_woww_update_type;
static int hf_woww_url;
static int hf_woww_url_info;
static int hf_woww_usable;
static int hf_woww_username;
static int hf_woww_v_cos;
static int hf_woww_v_sin;
static int hf_woww_value;
static int hf_woww_vendor;
static int hf_woww_vendor_slot;
static int hf_woww_vertical_speed;
static int hf_woww_victim;
static int hf_woww_walking_speed;
static int hf_woww_weather_change_type;
static int hf_woww_weather_type;
static int hf_woww_winning_player;
static int hf_woww_winning_roll;
static int hf_woww_wiping_npc;
static int hf_woww_won;
static int hf_woww_world_result;
static int hf_woww_wrapped;
static int hf_woww_x;
static int hf_woww_xy_speed;
static int hf_woww_y;
static int hf_woww_yesterday_honor;
static int hf_woww_yesterday_honorable;
static int hf_woww_z;
static int hf_woww_z_speed;
static int hf_woww_zone_or_sort;
static int hf_woww_zones;
/* AUTOGENERATED_END_HF */

#define WOWW_CLIENT_TO_SERVER pinfo->srcport > pinfo->destport
#define WOWW_SERVER_TO_CLIENT pinfo->srcport < pinfo->destport

// Allocate 8 because tree wants 32 bit aligned data
#define WOWW_HEADER_ARRAY_ALLOC_SIZE 8

#define WOWW_HEADER_SIZE_FIELD_WIDTH 2

// The session key is the result of two SHA-1 hashes appended, so it is
// _always_ 40 bytes.
#define WOWW_SESSION_KEY_LENGTH 40

static gint ett_woww;
static gint ett_message;
static gint ett_character;

// Packets that do not have at least a u16 size field and a u16 opcode field are not valid.
#define WOWW_MIN_LENGTH 4

// A participant can either be the server or a client.
typedef struct WowwParticipant {
    // The previous encrypted value sent. Persists through headers.
    guint8 last_encrypted_value;
    // Index into the session key. Must always be in [0; WOWW_SESSION_KEY_LENGTH - 1].
    // Named idx because there's a check for 'index'
    guint8 idx;
    // The first header is unencrypted. Tracks if that header has been encountered.
    gboolean unencrypted_packet_encountered;
    // If a server message is unable to be fully decrypted we stop decrypting any
    // any more, since it's impossible to know if the PDU contains multiple messages
    // and thus how many times the session key index should be incremented.
    guint64 stopped_at;
} WowwParticipant_t;

typedef struct WowwConversation {
    // Secret session key known to the client and host.
    guint8 session_key[WOWW_SESSION_KEY_LENGTH];
    // Which values of the session key have been deduced.
    bool known_indices[WOWW_SESSION_KEY_LENGTH];
    // Cache headers that have already been decrypted to save time
    // as well as reduce headaches from out of order packets.
    wmem_map_t* decrypted_headers;
    // Packets that are not fully decryptable when received will need
    // to be decrypted later.
    wmem_map_t* headers_need_decryption;
    // The client and server will have different indices/last values
    // because they send different amounts of packets and with different
    // header lengths.
    WowwParticipant_t client;
    WowwParticipant_t server;
} WowwConversation_t;

typedef struct {
    // Index into the session key, named idx because there's a check for 'index'
    guint8 idx;
    guint8 last_encrypted_value;
} WowwPreviousValues_t;

typedef struct {
    guint8 size[2];
    guint8 opcode[];
} WowwDecryptedHeader_t;

/* AUTOGENERATED_START_ENUM */
typedef enum {
    ADDON_TYPE_BANNED = 0x0,
    ADDON_TYPE_ENABLED = 0x1,
    ADDON_TYPE_BLIZZARD = 0x2,
} e_addon_type;
static const value_string e_addon_type_strings[] =  {
    { ADDON_TYPE_BANNED, "Banned" },
    { ADDON_TYPE_ENABLED, "Enabled" },
    { ADDON_TYPE_BLIZZARD, "Blizzard" },
    { 0, NULL }
};

typedef enum {
    INFO_BLOCK_UNAVAILABLE = 0x0,
    INFO_BLOCK_AVAILABLE = 0x1,
} e_info_block;
static const value_string e_info_block_strings[] =  {
    { INFO_BLOCK_UNAVAILABLE, "Unavailable" },
    { INFO_BLOCK_AVAILABLE, "Available" },
    { 0, NULL }
};

typedef enum {
    KEY_VERSION_ZERO = 0x0,
    KEY_VERSION_ONE = 0x1,
    KEY_VERSION_TWO = 0x2,
    KEY_VERSION_THREE = 0x3,
    KEY_VERSION_FOUR = 0x4,
    KEY_VERSION_FIVE = 0x5,
    KEY_VERSION_SIX = 0x6,
    KEY_VERSION_SEVEN = 0x7,
    KEY_VERSION_EIGHT = 0x8,
    KEY_VERSION_NINE = 0x9,
} e_key_version;
static const value_string e_key_version_strings[] =  {
    { KEY_VERSION_ZERO, "Zero" },
    { KEY_VERSION_ONE, "One" },
    { KEY_VERSION_TWO, "Two" },
    { KEY_VERSION_THREE, "Three" },
    { KEY_VERSION_FOUR, "Four" },
    { KEY_VERSION_FIVE, "Five" },
    { KEY_VERSION_SIX, "Six" },
    { KEY_VERSION_SEVEN, "Seven" },
    { KEY_VERSION_EIGHT, "Eight" },
    { KEY_VERSION_NINE, "Nine" },
    { 0, NULL }
};

typedef enum {
    URL_INFO_UNAVAILABLE = 0x0,
    URL_INFO_AVAILABLE = 0x1,
} e_url_info;
static const value_string e_url_info_strings[] =  {
    { URL_INFO_UNAVAILABLE, "Unavailable" },
    { URL_INFO_AVAILABLE, "Available" },
    { 0, NULL }
};

typedef enum {
    AURA_TYPE_NONE = 0x00,
    AURA_TYPE_BIND_SIGHT = 0x01,
    AURA_TYPE_MOD_POSSESS = 0x02,
    AURA_TYPE_PERIODIC_DAMAGE = 0x03,
    AURA_TYPE_DUMMY = 0x04,
    AURA_TYPE_MOD_CONFUSE = 0x05,
    AURA_TYPE_MOD_CHARM = 0x06,
    AURA_TYPE_MOD_FEAR = 0x07,
    AURA_TYPE_PERIODIC_HEAL = 0x08,
    AURA_TYPE_MOD_ATTACKSPEED = 0x09,
    AURA_TYPE_MOD_THREAT = 0x0A,
    AURA_TYPE_MOD_TAUNT = 0x0B,
    AURA_TYPE_MOD_STUN = 0x0C,
    AURA_TYPE_MOD_DAMAGE_DONE = 0x0D,
    AURA_TYPE_MOD_DAMAGE_TAKEN = 0x0E,
    AURA_TYPE_DAMAGE_SHIELD = 0x0F,
    AURA_TYPE_MOD_STEALTH = 0x10,
    AURA_TYPE_MOD_STEALTH_DETECT = 0x11,
    AURA_TYPE_MOD_INVISIBILITY = 0x12,
    AURA_TYPE_MOD_INVISIBILITY_DETECTION = 0x13,
    AURA_TYPE_OBS_MOD_HEALTH = 0x14,
    AURA_TYPE_OBS_MOD_MANA = 0x15,
    AURA_TYPE_MOD_RESISTANCE = 0x16,
    AURA_TYPE_PERIODIC_TRIGGER_SPELL = 0x17,
    AURA_TYPE_PERIODIC_ENERGIZE = 0x18,
    AURA_TYPE_MOD_PACIFY = 0x19,
    AURA_TYPE_MOD_ROOT = 0x1A,
    AURA_TYPE_MOD_SILENCE = 0x1B,
    AURA_TYPE_REFLECT_SPELLS = 0x1C,
    AURA_TYPE_MOD_STAT = 0x1D,
    AURA_TYPE_MOD_SKILL = 0x1E,
    AURA_TYPE_MOD_INCREASE_SPEED = 0x1F,
    AURA_TYPE_MOD_INCREASE_MOUNTED_SPEED = 0x20,
    AURA_TYPE_MOD_DECREASE_SPEED = 0x21,
    AURA_TYPE_MOD_INCREASE_HEALTH = 0x22,
    AURA_TYPE_MOD_INCREASE_ENERGY = 0x23,
    AURA_TYPE_MOD_SHAPESHIFT = 0x24,
    AURA_TYPE_EFFECT_IMMUNITY = 0x25,
    AURA_TYPE_STATE_IMMUNITY = 0x26,
    AURA_TYPE_SCHOOL_IMMUNITY = 0x27,
    AURA_TYPE_DAMAGE_IMMUNITY = 0x28,
    AURA_TYPE_DISPEL_IMMUNITY = 0x29,
    AURA_TYPE_PROC_TRIGGER_SPELL = 0x2A,
    AURA_TYPE_PROC_TRIGGER_DAMAGE = 0x2B,
    AURA_TYPE_TRACK_CREATURES = 0x2C,
    AURA_TYPE_TRACK_RESOURCES = 0x2D,
    AURA_TYPE_UNKNOWN46 = 0x2E,
    AURA_TYPE_MOD_PARRY_PERCENT = 0x2F,
    AURA_TYPE_UNKNOWN48 = 0x30,
    AURA_TYPE_MOD_DODGE_PERCENT = 0x31,
    AURA_TYPE_MOD_BLOCK_SKILL = 0x32,
    AURA_TYPE_MOD_BLOCK_PERCENT = 0x33,
    AURA_TYPE_MOD_CRIT_PERCENT = 0x34,
    AURA_TYPE_PERIODIC_LEECH = 0x35,
    AURA_TYPE_MOD_HIT_CHANCE = 0x36,
    AURA_TYPE_MOD_SPELL_HIT_CHANCE = 0x37,
    AURA_TYPE_TRANSFORM = 0x38,
    AURA_TYPE_MOD_SPELL_CRIT_CHANCE = 0x39,
    AURA_TYPE_MOD_INCREASE_SWIM_SPEED = 0x3A,
    AURA_TYPE_MOD_DAMAGE_DONE_CREATURE = 0x3B,
    AURA_TYPE_MOD_PACIFY_SILENCE = 0x3C,
    AURA_TYPE_MOD_SCALE = 0x3D,
    AURA_TYPE_PERIODIC_HEALTH_FUNNEL = 0x3E,
    AURA_TYPE_PERIODIC_MANA_FUNNEL = 0x3F,
    AURA_TYPE_PERIODIC_MANA_LEECH = 0x40,
    AURA_TYPE_MOD_CASTING_SPEED_NOT_STACK = 0x41,
    AURA_TYPE_FEIGN_DEATH = 0x42,
    AURA_TYPE_MOD_DISARM = 0x43,
    AURA_TYPE_MOD_STALKED = 0x44,
    AURA_TYPE_SCHOOL_ABSORB = 0x45,
    AURA_TYPE_EXTRA_ATTACKS = 0x46,
    AURA_TYPE_MOD_SPELL_CRIT_CHANCE_SCHOOL = 0x47,
    AURA_TYPE_MOD_POWER_COST_SCHOOL_PCT = 0x48,
    AURA_TYPE_MOD_POWER_COST_SCHOOL = 0x49,
    AURA_TYPE_REFLECT_SPELLS_SCHOOL = 0x4A,
    AURA_TYPE_MOD_LANGUAGE = 0x4B,
    AURA_TYPE_FAR_SIGHT = 0x4C,
    AURA_TYPE_MECHANIC_IMMUNITY = 0x4D,
    AURA_TYPE_MOUNTED = 0x4E,
    AURA_TYPE_MOD_DAMAGE_PERCENT_DONE = 0x4F,
    AURA_TYPE_MOD_PERCENT_STAT = 0x50,
    AURA_TYPE_SPLIT_DAMAGE_PCT = 0x51,
    AURA_TYPE_WATER_BREATHING = 0x52,
    AURA_TYPE_MOD_BASE_RESISTANCE = 0x53,
    AURA_TYPE_MOD_REGEN = 0x54,
    AURA_TYPE_MOD_POWER_REGEN = 0x55,
    AURA_TYPE_CHANNEL_DEATH_ITEM = 0x56,
    AURA_TYPE_MOD_DAMAGE_PERCENT_TAKEN = 0x57,
    AURA_TYPE_MOD_HEALTH_REGEN_PERCENT = 0x58,
    AURA_TYPE_PERIODIC_DAMAGE_PERCENT = 0x59,
    AURA_TYPE_MOD_RESIST_CHANCE = 0x5A,
    AURA_TYPE_MOD_DETECT_RANGE = 0x5B,
    AURA_TYPE_PREVENTS_FLEEING = 0x5C,
    AURA_TYPE_MOD_UNATTACKABLE = 0x5D,
    AURA_TYPE_INTERRUPT_REGEN = 0x5E,
    AURA_TYPE_GHOST = 0x5F,
    AURA_TYPE_SPELL_MAGNET = 0x60,
    AURA_TYPE_MANA_SHIELD = 0x61,
    AURA_TYPE_MOD_SKILL_TALENT = 0x62,
    AURA_TYPE_MOD_ATTACK_POWER = 0x63,
    AURA_TYPE_AURAS_VISIBLE = 0x64,
    AURA_TYPE_MOD_RESISTANCE_PCT = 0x65,
    AURA_TYPE_MOD_MELEE_ATTACK_POWER_VERSUS = 0x66,
    AURA_TYPE_MOD_TOTAL_THREAT = 0x67,
    AURA_TYPE_WATER_WALK = 0x68,
    AURA_TYPE_FEATHER_FALL = 0x69,
    AURA_TYPE_HOVER = 0x6A,
    AURA_TYPE_ADD_FLAT_MODIFIER = 0x6B,
    AURA_TYPE_ADD_PCT_MODIFIER = 0x6C,
    AURA_TYPE_ADD_TARGET_TRIGGER = 0x6D,
    AURA_TYPE_MOD_POWER_REGEN_PERCENT = 0x6E,
    AURA_TYPE_ADD_CASTER_HIT_TRIGGER = 0x6F,
    AURA_TYPE_OVERRIDE_CLASS_SCRIPTS = 0x70,
    AURA_TYPE_MOD_RANGED_DAMAGE_TAKEN = 0x71,
    AURA_TYPE_MOD_RANGED_DAMAGE_TAKEN_PCT = 0x72,
    AURA_TYPE_MOD_HEALING = 0x73,
    AURA_TYPE_MOD_REGEN_DURING_COMBAT = 0x74,
    AURA_TYPE_MOD_MECHANIC_RESISTANCE = 0x75,
    AURA_TYPE_MOD_HEALING_PCT = 0x76,
    AURA_TYPE_SHARE_PET_TRACKING = 0x77,
    AURA_TYPE_UNTRACKABLE = 0x78,
    AURA_TYPE_EMPATHY = 0x79,
    AURA_TYPE_MOD_OFFHAND_DAMAGE_PCT = 0x7A,
    AURA_TYPE_MOD_TARGET_RESISTANCE = 0x7B,
    AURA_TYPE_MOD_RANGED_ATTACK_POWER = 0x7C,
    AURA_TYPE_MOD_MELEE_DAMAGE_TAKEN = 0x7D,
    AURA_TYPE_MOD_MELEE_DAMAGE_TAKEN_PCT = 0x7E,
    AURA_TYPE_RANGED_ATTACK_POWER_ATTACKER_BONUS = 0x7F,
    AURA_TYPE_MOD_POSSESS_PET = 0x80,
    AURA_TYPE_MOD_SPEED_ALWAYS = 0x81,
    AURA_TYPE_MOD_MOUNTED_SPEED_ALWAYS = 0x82,
    AURA_TYPE_MOD_RANGED_ATTACK_POWER_VERSUS = 0x83,
    AURA_TYPE_MOD_INCREASE_ENERGY_PERCENT = 0x84,
    AURA_TYPE_MOD_INCREASE_HEALTH_PERCENT = 0x85,
    AURA_TYPE_MOD_MANA_REGEN_INTERRUPT = 0x86,
    AURA_TYPE_MOD_HEALING_DONE = 0x87,
    AURA_TYPE_MOD_HEALING_DONE_PERCENT = 0x88,
    AURA_TYPE_MOD_TOTAL_STAT_PERCENTAGE = 0x89,
    AURA_TYPE_MOD_MELEE_HASTE = 0x8A,
    AURA_TYPE_FORCE_REACTION = 0x8B,
    AURA_TYPE_MOD_RANGED_HASTE = 0x8C,
    AURA_TYPE_MOD_RANGED_AMMO_HASTE = 0x8D,
    AURA_TYPE_MOD_BASE_RESISTANCE_PCT = 0x8E,
    AURA_TYPE_MOD_RESISTANCE_EXCLUSIVE = 0x8F,
    AURA_TYPE_SAFE_FALL = 0x90,
    AURA_TYPE_CHARISMA = 0x91,
    AURA_TYPE_PERSUADED = 0x92,
    AURA_TYPE_MECHANIC_IMMUNITY_MASK = 0x93,
    AURA_TYPE_RETAIN_COMBO_POINTS = 0x94,
    AURA_TYPE_RESIST_PUSHBACK = 0x95,
    AURA_TYPE_MOD_SHIELD_BLOCKVALUE_PCT = 0x96,
    AURA_TYPE_TRACK_STEALTHED = 0x97,
    AURA_TYPE_MOD_DETECTED_RANGE = 0x98,
    AURA_TYPE_SPLIT_DAMAGE_FLAT = 0x99,
    AURA_TYPE_MOD_STEALTH_LEVEL = 0x9A,
    AURA_TYPE_MOD_WATER_BREATHING = 0x9B,
    AURA_TYPE_MOD_REPUTATION_GAIN = 0x9C,
    AURA_TYPE_PET_DAMAGE_MULTI = 0x9D,
    AURA_TYPE_MOD_SHIELD_BLOCKVALUE = 0x9E,
    AURA_TYPE_NO_PVP_CREDIT = 0x9F,
    AURA_TYPE_MOD_AOE_AVOIDANCE = 0xA0,
    AURA_TYPE_MOD_HEALTH_REGEN_IN_COMBAT = 0xA1,
    AURA_TYPE_POWER_BURN_MANA = 0xA2,
    AURA_TYPE_MOD_CRIT_DAMAGE_BONUS = 0xA3,
    AURA_TYPE_UNKNOWN164 = 0xA4,
    AURA_TYPE_MELEE_ATTACK_POWER_ATTACKER_BONUS = 0xA5,
    AURA_TYPE_MOD_ATTACK_POWER_PCT = 0xA6,
    AURA_TYPE_MOD_RANGED_ATTACK_POWER_PCT = 0xA7,
    AURA_TYPE_MOD_DAMAGE_DONE_VERSUS = 0xA8,
    AURA_TYPE_MOD_CRIT_PERCENT_VERSUS = 0xA9,
    AURA_TYPE_DETECT_AMORE = 0xAA,
    AURA_TYPE_MOD_SPEED_NOT_STACK = 0xAB,
    AURA_TYPE_MOD_MOUNTED_SPEED_NOT_STACK = 0xAC,
    AURA_TYPE_ALLOW_CHAMPION_SPELLS = 0xAD,
    AURA_TYPE_MOD_SPELL_DAMAGE_OF_STAT_PERCENT = 0xAE,
    AURA_TYPE_MOD_SPELL_HEALING_OF_STAT_PERCENT = 0xAF,
    AURA_TYPE_SPIRIT_OF_REDEMPTION = 0xB0,
    AURA_TYPE_AOE_CHARM = 0xB1,
    AURA_TYPE_MOD_DEBUFF_RESISTANCE = 0xB2,
    AURA_TYPE_MOD_ATTACKER_SPELL_CRIT_CHANCE = 0xB3,
    AURA_TYPE_MOD_FLAT_SPELL_DAMAGE_VERSUS = 0xB4,
    AURA_TYPE_MOD_FLAT_SPELL_CRIT_DAMAGE_VERSUS = 0xB5,
    AURA_TYPE_MOD_RESISTANCE_OF_STAT_PERCENT = 0xB6,
    AURA_TYPE_MOD_CRITICAL_THREAT = 0xB7,
    AURA_TYPE_MOD_ATTACKER_MELEE_HIT_CHANCE = 0xB8,
    AURA_TYPE_MOD_ATTACKER_RANGED_HIT_CHANCE = 0xB9,
    AURA_TYPE_MOD_ATTACKER_SPELL_HIT_CHANCE = 0xBA,
    AURA_TYPE_MOD_ATTACKER_MELEE_CRIT_CHANCE = 0xBB,
    AURA_TYPE_MOD_ATTACKER_RANGED_CRIT_CHANCE = 0xBC,
    AURA_TYPE_MOD_RATING = 0xBD,
    AURA_TYPE_MOD_FACTION_REPUTATION_GAIN = 0xBE,
    AURA_TYPE_USE_NORMAL_MOVEMENT_SPEED = 0xBF,
} e_aura_type;
static const value_string e_aura_type_strings[] =  {
    { AURA_TYPE_NONE, "None" },
    { AURA_TYPE_BIND_SIGHT, "Bind Sight" },
    { AURA_TYPE_MOD_POSSESS, "Mod Possess" },
    { AURA_TYPE_PERIODIC_DAMAGE, "Periodic Damage" },
    { AURA_TYPE_DUMMY, "Dummy" },
    { AURA_TYPE_MOD_CONFUSE, "Mod Confuse" },
    { AURA_TYPE_MOD_CHARM, "Mod Charm" },
    { AURA_TYPE_MOD_FEAR, "Mod Fear" },
    { AURA_TYPE_PERIODIC_HEAL, "Periodic Heal" },
    { AURA_TYPE_MOD_ATTACKSPEED, "Mod Attackspeed" },
    { AURA_TYPE_MOD_THREAT, "Mod Threat" },
    { AURA_TYPE_MOD_TAUNT, "Mod Taunt" },
    { AURA_TYPE_MOD_STUN, "Mod Stun" },
    { AURA_TYPE_MOD_DAMAGE_DONE, "Mod Damage Done" },
    { AURA_TYPE_MOD_DAMAGE_TAKEN, "Mod Damage Taken" },
    { AURA_TYPE_DAMAGE_SHIELD, "Damage Shield" },
    { AURA_TYPE_MOD_STEALTH, "Mod Stealth" },
    { AURA_TYPE_MOD_STEALTH_DETECT, "Mod Stealth Detect" },
    { AURA_TYPE_MOD_INVISIBILITY, "Mod Invisibility" },
    { AURA_TYPE_MOD_INVISIBILITY_DETECTION, "Mod Invisibility Detection" },
    { AURA_TYPE_OBS_MOD_HEALTH, "Obs Mod Health" },
    { AURA_TYPE_OBS_MOD_MANA, "Obs Mod Mana" },
    { AURA_TYPE_MOD_RESISTANCE, "Mod Resistance" },
    { AURA_TYPE_PERIODIC_TRIGGER_SPELL, "Periodic Trigger Spell" },
    { AURA_TYPE_PERIODIC_ENERGIZE, "Periodic Energize" },
    { AURA_TYPE_MOD_PACIFY, "Mod Pacify" },
    { AURA_TYPE_MOD_ROOT, "Mod Root" },
    { AURA_TYPE_MOD_SILENCE, "Mod Silence" },
    { AURA_TYPE_REFLECT_SPELLS, "Reflect Spells" },
    { AURA_TYPE_MOD_STAT, "Mod Stat" },
    { AURA_TYPE_MOD_SKILL, "Mod Skill" },
    { AURA_TYPE_MOD_INCREASE_SPEED, "Mod Increase Speed" },
    { AURA_TYPE_MOD_INCREASE_MOUNTED_SPEED, "Mod Increase Mounted Speed" },
    { AURA_TYPE_MOD_DECREASE_SPEED, "Mod Decrease Speed" },
    { AURA_TYPE_MOD_INCREASE_HEALTH, "Mod Increase Health" },
    { AURA_TYPE_MOD_INCREASE_ENERGY, "Mod Increase Energy" },
    { AURA_TYPE_MOD_SHAPESHIFT, "Mod Shapeshift" },
    { AURA_TYPE_EFFECT_IMMUNITY, "Effect Immunity" },
    { AURA_TYPE_STATE_IMMUNITY, "State Immunity" },
    { AURA_TYPE_SCHOOL_IMMUNITY, "School Immunity" },
    { AURA_TYPE_DAMAGE_IMMUNITY, "Damage Immunity" },
    { AURA_TYPE_DISPEL_IMMUNITY, "Dispel Immunity" },
    { AURA_TYPE_PROC_TRIGGER_SPELL, "Proc Trigger Spell" },
    { AURA_TYPE_PROC_TRIGGER_DAMAGE, "Proc Trigger Damage" },
    { AURA_TYPE_TRACK_CREATURES, "Track Creatures" },
    { AURA_TYPE_TRACK_RESOURCES, "Track Resources" },
    { AURA_TYPE_UNKNOWN46, "Unknown46" },
    { AURA_TYPE_MOD_PARRY_PERCENT, "Mod Parry Percent" },
    { AURA_TYPE_UNKNOWN48, "Unknown48" },
    { AURA_TYPE_MOD_DODGE_PERCENT, "Mod Dodge Percent" },
    { AURA_TYPE_MOD_BLOCK_SKILL, "Mod Block Skill" },
    { AURA_TYPE_MOD_BLOCK_PERCENT, "Mod Block Percent" },
    { AURA_TYPE_MOD_CRIT_PERCENT, "Mod Crit Percent" },
    { AURA_TYPE_PERIODIC_LEECH, "Periodic Leech" },
    { AURA_TYPE_MOD_HIT_CHANCE, "Mod Hit Chance" },
    { AURA_TYPE_MOD_SPELL_HIT_CHANCE, "Mod Spell Hit Chance" },
    { AURA_TYPE_TRANSFORM, "Transform" },
    { AURA_TYPE_MOD_SPELL_CRIT_CHANCE, "Mod Spell Crit Chance" },
    { AURA_TYPE_MOD_INCREASE_SWIM_SPEED, "Mod Increase Swim Speed" },
    { AURA_TYPE_MOD_DAMAGE_DONE_CREATURE, "Mod Damage Done Creature" },
    { AURA_TYPE_MOD_PACIFY_SILENCE, "Mod Pacify Silence" },
    { AURA_TYPE_MOD_SCALE, "Mod Scale" },
    { AURA_TYPE_PERIODIC_HEALTH_FUNNEL, "Periodic Health Funnel" },
    { AURA_TYPE_PERIODIC_MANA_FUNNEL, "Periodic Mana Funnel" },
    { AURA_TYPE_PERIODIC_MANA_LEECH, "Periodic Mana Leech" },
    { AURA_TYPE_MOD_CASTING_SPEED_NOT_STACK, "Mod Casting Speed Not Stack" },
    { AURA_TYPE_FEIGN_DEATH, "Feign Death" },
    { AURA_TYPE_MOD_DISARM, "Mod Disarm" },
    { AURA_TYPE_MOD_STALKED, "Mod Stalked" },
    { AURA_TYPE_SCHOOL_ABSORB, "School Absorb" },
    { AURA_TYPE_EXTRA_ATTACKS, "Extra Attacks" },
    { AURA_TYPE_MOD_SPELL_CRIT_CHANCE_SCHOOL, "Mod Spell Crit Chance School" },
    { AURA_TYPE_MOD_POWER_COST_SCHOOL_PCT, "Mod Power Cost School Pct" },
    { AURA_TYPE_MOD_POWER_COST_SCHOOL, "Mod Power Cost School" },
    { AURA_TYPE_REFLECT_SPELLS_SCHOOL, "Reflect Spells School" },
    { AURA_TYPE_MOD_LANGUAGE, "Mod Language" },
    { AURA_TYPE_FAR_SIGHT, "Far Sight" },
    { AURA_TYPE_MECHANIC_IMMUNITY, "Mechanic Immunity" },
    { AURA_TYPE_MOUNTED, "Mounted" },
    { AURA_TYPE_MOD_DAMAGE_PERCENT_DONE, "Mod Damage Percent Done" },
    { AURA_TYPE_MOD_PERCENT_STAT, "Mod Percent Stat" },
    { AURA_TYPE_SPLIT_DAMAGE_PCT, "Split Damage Pct" },
    { AURA_TYPE_WATER_BREATHING, "Water Breathing" },
    { AURA_TYPE_MOD_BASE_RESISTANCE, "Mod Base Resistance" },
    { AURA_TYPE_MOD_REGEN, "Mod Regen" },
    { AURA_TYPE_MOD_POWER_REGEN, "Mod Power Regen" },
    { AURA_TYPE_CHANNEL_DEATH_ITEM, "Channel Death Item" },
    { AURA_TYPE_MOD_DAMAGE_PERCENT_TAKEN, "Mod Damage Percent Taken" },
    { AURA_TYPE_MOD_HEALTH_REGEN_PERCENT, "Mod Health Regen Percent" },
    { AURA_TYPE_PERIODIC_DAMAGE_PERCENT, "Periodic Damage Percent" },
    { AURA_TYPE_MOD_RESIST_CHANCE, "Mod Resist Chance" },
    { AURA_TYPE_MOD_DETECT_RANGE, "Mod Detect Range" },
    { AURA_TYPE_PREVENTS_FLEEING, "Prevents Fleeing" },
    { AURA_TYPE_MOD_UNATTACKABLE, "Mod Unattackable" },
    { AURA_TYPE_INTERRUPT_REGEN, "Interrupt Regen" },
    { AURA_TYPE_GHOST, "Ghost" },
    { AURA_TYPE_SPELL_MAGNET, "Spell Magnet" },
    { AURA_TYPE_MANA_SHIELD, "Mana Shield" },
    { AURA_TYPE_MOD_SKILL_TALENT, "Mod Skill Talent" },
    { AURA_TYPE_MOD_ATTACK_POWER, "Mod Attack Power" },
    { AURA_TYPE_AURAS_VISIBLE, "Auras Visible" },
    { AURA_TYPE_MOD_RESISTANCE_PCT, "Mod Resistance Pct" },
    { AURA_TYPE_MOD_MELEE_ATTACK_POWER_VERSUS, "Mod Melee Attack Power Versus" },
    { AURA_TYPE_MOD_TOTAL_THREAT, "Mod Total Threat" },
    { AURA_TYPE_WATER_WALK, "Water Walk" },
    { AURA_TYPE_FEATHER_FALL, "Feather Fall" },
    { AURA_TYPE_HOVER, "Hover" },
    { AURA_TYPE_ADD_FLAT_MODIFIER, "Add Flat Modifier" },
    { AURA_TYPE_ADD_PCT_MODIFIER, "Add Pct Modifier" },
    { AURA_TYPE_ADD_TARGET_TRIGGER, "Add Target Trigger" },
    { AURA_TYPE_MOD_POWER_REGEN_PERCENT, "Mod Power Regen Percent" },
    { AURA_TYPE_ADD_CASTER_HIT_TRIGGER, "Add Caster Hit Trigger" },
    { AURA_TYPE_OVERRIDE_CLASS_SCRIPTS, "Override Class Scripts" },
    { AURA_TYPE_MOD_RANGED_DAMAGE_TAKEN, "Mod Ranged Damage Taken" },
    { AURA_TYPE_MOD_RANGED_DAMAGE_TAKEN_PCT, "Mod Ranged Damage Taken Pct" },
    { AURA_TYPE_MOD_HEALING, "Mod Healing" },
    { AURA_TYPE_MOD_REGEN_DURING_COMBAT, "Mod Regen During Combat" },
    { AURA_TYPE_MOD_MECHANIC_RESISTANCE, "Mod Mechanic Resistance" },
    { AURA_TYPE_MOD_HEALING_PCT, "Mod Healing Pct" },
    { AURA_TYPE_SHARE_PET_TRACKING, "Share Pet Tracking" },
    { AURA_TYPE_UNTRACKABLE, "Untrackable" },
    { AURA_TYPE_EMPATHY, "Empathy" },
    { AURA_TYPE_MOD_OFFHAND_DAMAGE_PCT, "Mod Offhand Damage Pct" },
    { AURA_TYPE_MOD_TARGET_RESISTANCE, "Mod Target Resistance" },
    { AURA_TYPE_MOD_RANGED_ATTACK_POWER, "Mod Ranged Attack Power" },
    { AURA_TYPE_MOD_MELEE_DAMAGE_TAKEN, "Mod Melee Damage Taken" },
    { AURA_TYPE_MOD_MELEE_DAMAGE_TAKEN_PCT, "Mod Melee Damage Taken Pct" },
    { AURA_TYPE_RANGED_ATTACK_POWER_ATTACKER_BONUS, "Ranged Attack Power Attacker Bonus" },
    { AURA_TYPE_MOD_POSSESS_PET, "Mod Possess Pet" },
    { AURA_TYPE_MOD_SPEED_ALWAYS, "Mod Speed Always" },
    { AURA_TYPE_MOD_MOUNTED_SPEED_ALWAYS, "Mod Mounted Speed Always" },
    { AURA_TYPE_MOD_RANGED_ATTACK_POWER_VERSUS, "Mod Ranged Attack Power Versus" },
    { AURA_TYPE_MOD_INCREASE_ENERGY_PERCENT, "Mod Increase Energy Percent" },
    { AURA_TYPE_MOD_INCREASE_HEALTH_PERCENT, "Mod Increase Health Percent" },
    { AURA_TYPE_MOD_MANA_REGEN_INTERRUPT, "Mod Mana Regen Interrupt" },
    { AURA_TYPE_MOD_HEALING_DONE, "Mod Healing Done" },
    { AURA_TYPE_MOD_HEALING_DONE_PERCENT, "Mod Healing Done Percent" },
    { AURA_TYPE_MOD_TOTAL_STAT_PERCENTAGE, "Mod Total Stat Percentage" },
    { AURA_TYPE_MOD_MELEE_HASTE, "Mod Melee Haste" },
    { AURA_TYPE_FORCE_REACTION, "Force Reaction" },
    { AURA_TYPE_MOD_RANGED_HASTE, "Mod Ranged Haste" },
    { AURA_TYPE_MOD_RANGED_AMMO_HASTE, "Mod Ranged Ammo Haste" },
    { AURA_TYPE_MOD_BASE_RESISTANCE_PCT, "Mod Base Resistance Pct" },
    { AURA_TYPE_MOD_RESISTANCE_EXCLUSIVE, "Mod Resistance Exclusive" },
    { AURA_TYPE_SAFE_FALL, "Safe Fall" },
    { AURA_TYPE_CHARISMA, "Charisma" },
    { AURA_TYPE_PERSUADED, "Persuaded" },
    { AURA_TYPE_MECHANIC_IMMUNITY_MASK, "Mechanic Immunity Mask" },
    { AURA_TYPE_RETAIN_COMBO_POINTS, "Retain Combo Points" },
    { AURA_TYPE_RESIST_PUSHBACK, "Resist Pushback" },
    { AURA_TYPE_MOD_SHIELD_BLOCKVALUE_PCT, "Mod Shield Blockvalue Pct" },
    { AURA_TYPE_TRACK_STEALTHED, "Track Stealthed" },
    { AURA_TYPE_MOD_DETECTED_RANGE, "Mod Detected Range" },
    { AURA_TYPE_SPLIT_DAMAGE_FLAT, "Split Damage Flat" },
    { AURA_TYPE_MOD_STEALTH_LEVEL, "Mod Stealth Level" },
    { AURA_TYPE_MOD_WATER_BREATHING, "Mod Water Breathing" },
    { AURA_TYPE_MOD_REPUTATION_GAIN, "Mod Reputation Gain" },
    { AURA_TYPE_PET_DAMAGE_MULTI, "Pet Damage Multi" },
    { AURA_TYPE_MOD_SHIELD_BLOCKVALUE, "Mod Shield Blockvalue" },
    { AURA_TYPE_NO_PVP_CREDIT, "No Pvp Credit" },
    { AURA_TYPE_MOD_AOE_AVOIDANCE, "Mod Aoe Avoidance" },
    { AURA_TYPE_MOD_HEALTH_REGEN_IN_COMBAT, "Mod Health Regen In Combat" },
    { AURA_TYPE_POWER_BURN_MANA, "Power Burn Mana" },
    { AURA_TYPE_MOD_CRIT_DAMAGE_BONUS, "Mod Crit Damage Bonus" },
    { AURA_TYPE_UNKNOWN164, "Unknown164" },
    { AURA_TYPE_MELEE_ATTACK_POWER_ATTACKER_BONUS, "Melee Attack Power Attacker Bonus" },
    { AURA_TYPE_MOD_ATTACK_POWER_PCT, "Mod Attack Power Pct" },
    { AURA_TYPE_MOD_RANGED_ATTACK_POWER_PCT, "Mod Ranged Attack Power Pct" },
    { AURA_TYPE_MOD_DAMAGE_DONE_VERSUS, "Mod Damage Done Versus" },
    { AURA_TYPE_MOD_CRIT_PERCENT_VERSUS, "Mod Crit Percent Versus" },
    { AURA_TYPE_DETECT_AMORE, "Detect Amore" },
    { AURA_TYPE_MOD_SPEED_NOT_STACK, "Mod Speed Not Stack" },
    { AURA_TYPE_MOD_MOUNTED_SPEED_NOT_STACK, "Mod Mounted Speed Not Stack" },
    { AURA_TYPE_ALLOW_CHAMPION_SPELLS, "Allow Champion Spells" },
    { AURA_TYPE_MOD_SPELL_DAMAGE_OF_STAT_PERCENT, "Mod Spell Damage Of Stat Percent" },
    { AURA_TYPE_MOD_SPELL_HEALING_OF_STAT_PERCENT, "Mod Spell Healing Of Stat Percent" },
    { AURA_TYPE_SPIRIT_OF_REDEMPTION, "Spirit Of Redemption" },
    { AURA_TYPE_AOE_CHARM, "Aoe Charm" },
    { AURA_TYPE_MOD_DEBUFF_RESISTANCE, "Mod Debuff Resistance" },
    { AURA_TYPE_MOD_ATTACKER_SPELL_CRIT_CHANCE, "Mod Attacker Spell Crit Chance" },
    { AURA_TYPE_MOD_FLAT_SPELL_DAMAGE_VERSUS, "Mod Flat Spell Damage Versus" },
    { AURA_TYPE_MOD_FLAT_SPELL_CRIT_DAMAGE_VERSUS, "Mod Flat Spell Crit Damage Versus" },
    { AURA_TYPE_MOD_RESISTANCE_OF_STAT_PERCENT, "Mod Resistance Of Stat Percent" },
    { AURA_TYPE_MOD_CRITICAL_THREAT, "Mod Critical Threat" },
    { AURA_TYPE_MOD_ATTACKER_MELEE_HIT_CHANCE, "Mod Attacker Melee Hit Chance" },
    { AURA_TYPE_MOD_ATTACKER_RANGED_HIT_CHANCE, "Mod Attacker Ranged Hit Chance" },
    { AURA_TYPE_MOD_ATTACKER_SPELL_HIT_CHANCE, "Mod Attacker Spell Hit Chance" },
    { AURA_TYPE_MOD_ATTACKER_MELEE_CRIT_CHANCE, "Mod Attacker Melee Crit Chance" },
    { AURA_TYPE_MOD_ATTACKER_RANGED_CRIT_CHANCE, "Mod Attacker Ranged Crit Chance" },
    { AURA_TYPE_MOD_RATING, "Mod Rating" },
    { AURA_TYPE_MOD_FACTION_REPUTATION_GAIN, "Mod Faction Reputation Gain" },
    { AURA_TYPE_USE_NORMAL_MOVEMENT_SPEED, "Use Normal Movement Speed" },
    { 0, NULL }
};

typedef enum {
    SPELL_SCHOOL_NORMAL = 0x0,
    SPELL_SCHOOL_HOLY = 0x1,
    SPELL_SCHOOL_FIRE = 0x2,
    SPELL_SCHOOL_NATURE = 0x3,
    SPELL_SCHOOL_FROST = 0x4,
    SPELL_SCHOOL_SHADOW = 0x5,
    SPELL_SCHOOL_ARCANE = 0x6,
} e_spell_school;
static const value_string e_spell_school_strings[] =  {
    { SPELL_SCHOOL_NORMAL, "Normal" },
    { SPELL_SCHOOL_HOLY, "Holy" },
    { SPELL_SCHOOL_FIRE, "Fire" },
    { SPELL_SCHOOL_NATURE, "Nature" },
    { SPELL_SCHOOL_FROST, "Frost" },
    { SPELL_SCHOOL_SHADOW, "Shadow" },
    { SPELL_SCHOOL_ARCANE, "Arcane" },
    { 0, NULL }
};

typedef enum {
    PVP_RANK_NO_RANK = 0x00,
    PVP_RANK_PARIAH = 0x01,
    PVP_RANK_OUTLAW = 0x02,
    PVP_RANK_EXILED = 0x03,
    PVP_RANK_DISHONORED = 0x04,
    PVP_RANK_RANK1 = 0x05,
    PVP_RANK_RANK2 = 0x06,
    PVP_RANK_RANK3 = 0x07,
    PVP_RANK_RANK4 = 0x08,
    PVP_RANK_RANK5 = 0x09,
    PVP_RANK_RANK6 = 0x0A,
    PVP_RANK_RANK7 = 0x0B,
    PVP_RANK_RANK8 = 0x0C,
    PVP_RANK_RANK9 = 0x0D,
    PVP_RANK_RANK10 = 0x0E,
    PVP_RANK_RANK11 = 0x0F,
    PVP_RANK_RANK12 = 0x10,
    PVP_RANK_RANK13 = 0x11,
    PVP_RANK_RANK14 = 0x12,
    PVP_RANK_FACTION_LEADER = 0x13,
} e_pvp_rank;
static const value_string e_pvp_rank_strings[] =  {
    { PVP_RANK_NO_RANK, "No Rank" },
    { PVP_RANK_PARIAH, "Pariah" },
    { PVP_RANK_OUTLAW, "Outlaw" },
    { PVP_RANK_EXILED, "Exiled" },
    { PVP_RANK_DISHONORED, "Dishonored" },
    { PVP_RANK_RANK1, "Rank1" },
    { PVP_RANK_RANK2, "Rank2" },
    { PVP_RANK_RANK3, "Rank3" },
    { PVP_RANK_RANK4, "Rank4" },
    { PVP_RANK_RANK5, "Rank5" },
    { PVP_RANK_RANK6, "Rank6" },
    { PVP_RANK_RANK7, "Rank7" },
    { PVP_RANK_RANK8, "Rank8" },
    { PVP_RANK_RANK9, "Rank9" },
    { PVP_RANK_RANK10, "Rank10" },
    { PVP_RANK_RANK11, "Rank11" },
    { PVP_RANK_RANK12, "Rank12" },
    { PVP_RANK_RANK13, "Rank13" },
    { PVP_RANK_RANK14, "Rank14" },
    { PVP_RANK_FACTION_LEADER, "Faction Leader" },
    { 0, NULL }
};

typedef enum {
    RACE_HUMAN = 0x1,
    RACE_ORC = 0x2,
    RACE_DWARF = 0x3,
    RACE_NIGHT_ELF = 0x4,
    RACE_UNDEAD = 0x5,
    RACE_TAUREN = 0x6,
    RACE_GNOME = 0x7,
    RACE_TROLL = 0x8,
    RACE_GOBLIN = 0x9,
} e_race;
static const value_string e_race_strings[] =  {
    { RACE_HUMAN, "Human" },
    { RACE_ORC, "Orc" },
    { RACE_DWARF, "Dwarf" },
    { RACE_NIGHT_ELF, "Night Elf" },
    { RACE_UNDEAD, "Undead" },
    { RACE_TAUREN, "Tauren" },
    { RACE_GNOME, "Gnome" },
    { RACE_TROLL, "Troll" },
    { RACE_GOBLIN, "Goblin" },
    { 0, NULL }
};

typedef enum {
    CLASS_WARRIOR = 0x1,
    CLASS_PALADIN = 0x2,
    CLASS_HUNTER = 0x3,
    CLASS_ROGUE = 0x4,
    CLASS_PRIEST = 0x5,
    CLASS_SHAMAN = 0x7,
    CLASS_MAGE = 0x8,
    CLASS_WARLOCK = 0x9,
    CLASS_DRUID = 0xB,
} e_class;
static const value_string e_class_strings[] =  {
    { CLASS_WARRIOR, "Warrior" },
    { CLASS_PALADIN, "Paladin" },
    { CLASS_HUNTER, "Hunter" },
    { CLASS_ROGUE, "Rogue" },
    { CLASS_PRIEST, "Priest" },
    { CLASS_SHAMAN, "Shaman" },
    { CLASS_MAGE, "Mage" },
    { CLASS_WARLOCK, "Warlock" },
    { CLASS_DRUID, "Druid" },
    { 0, NULL }
};

typedef enum {
    GENDER_MALE = 0x0,
    GENDER_FEMALE = 0x1,
    GENDER_NONE = 0x2,
} e_gender;
static const value_string e_gender_strings[] =  {
    { GENDER_MALE, "Male" },
    { GENDER_FEMALE, "Female" },
    { GENDER_NONE, "None" },
    { 0, NULL }
};

typedef enum {
    AREA_NONE = 0x000,
    AREA_DUN_MOROGH = 0x001,
    AREA_LONGSHORE = 0x002,
    AREA_BADLANDS = 0x003,
    AREA_BLASTED_LANDS = 0x004,
    AREA_BLACKWATER_COVE = 0x007,
    AREA_SWAMP_OF_SORROWS = 0x008,
    AREA_NORTHSHIRE_VALLEY = 0x009,
    AREA_DUSKWOOD = 0x00A,
    AREA_WETLANDS = 0x00B,
    AREA_ELWYNN_FOREST = 0x00C,
    AREA_THE_WORLD_TREE = 0x00D,
    AREA_DUROTAR = 0x00E,
    AREA_DUSTWALLOW_MARSH = 0x00F,
    AREA_AZSHARA = 0x010,
    AREA_THE_BARRENS = 0x011,
    AREA_CRYSTAL_LAKE = 0x012,
    AREA_ZUL_GURUB0 = 0x013,
    AREA_MOONBROOK = 0x014,
    AREA_KUL_TIRAS = 0x015,
    AREA_PROGRAMMER_ISLE = 0x016,
    AREA_NORTHSHIRE_RIVER = 0x017,
    AREA_NORTHSHIRE_ABBEY = 0x018,
    AREA_BLACKROCK_MOUNTAIN0 = 0x019,
    AREA_LIGHTHOUSE = 0x01A,
    AREA_WESTERN_PLAGUELANDS = 0x01C,
    AREA_NINE = 0x01E,
    AREA_THE_CEMETARY = 0x020,
    AREA_STRANGLETHORN_VALE = 0x021,
    AREA_ECHO_RIDGE_MINE = 0x022,
    AREA_BOOTY_BAY = 0x023,
    AREA_ALTERAC_MOUNTAINS = 0x024,
    AREA_LAKE_NAZFERITI = 0x025,
    AREA_LOCH_MODAN = 0x026,
    AREA_WESTFALL0 = 0x028,
    AREA_DEADWIND_PASS = 0x029,
    AREA_DARKSHIRE = 0x02A,
    AREA_WILD_SHORE = 0x02B,
    AREA_REDRIDGE_MOUNTAINS = 0x02C,
    AREA_ARATHI_HIGHLANDS = 0x02D,
    AREA_BURNING_STEPPES = 0x02E,
    AREA_THE_HINTERLANDS = 0x02F,
    AREA_DEAD_MANS_HOLE = 0x031,
    AREA_SEARING_GORGE = 0x033,
    AREA_THIEVES_CAMP = 0x035,
    AREA_JASPERLODE_MINE = 0x036,
    AREA_VALLEY_OF_HEROES_UNUSED = 0x037,
    AREA_HEROES_VIGIL = 0x038,
    AREA_FARGODEEP_MINE = 0x039,
    AREA_NORTHSHIRE_VINEYARDS = 0x03B,
    AREA_FORESTS_EDGE = 0x03C,
    AREA_THUNDER_FALLS = 0x03D,
    AREA_BRACKWELL_PUMPKIN_PATCH = 0x03E,
    AREA_THE_STONEFIELD_FARM = 0x03F,
    AREA_THE_MACLURE_VINEYARDS = 0x040,
    AREA_ON_MAP_DUNGEON0 = 0x041,
    AREA_ON_MAP_DUNGEON1 = 0x042,
    AREA_ON_MAP_DUNGEON2 = 0x043,
    AREA_LAKE_EVERSTILL = 0x044,
    AREA_LAKESHIRE = 0x045,
    AREA_STONEWATCH = 0x046,
    AREA_STONEWATCH_FALLS = 0x047,
    AREA_THE_DARK_PORTAL = 0x048,
    AREA_THE_TAINTED_SCAR = 0x049,
    AREA_POOL_OF_TEARS = 0x04A,
    AREA_STONARD = 0x04B,
    AREA_FALLOW_SANCTUARY = 0x04C,
    AREA_ANVILMAR = 0x04D,
    AREA_STORMWIND_MOUNTAINS = 0x050,
    AREA_JEFF_NE_QUADRANT_CHANGED = 0x051,
    AREA_JEFF_NW_QUADRANT = 0x052,
    AREA_JEFF_SE_QUADRANT = 0x053,
    AREA_JEFF_SW_QUADRANT = 0x054,
    AREA_TIRISFAL_GLADES = 0x055,
    AREA_STONE_CAIRN_LAKE = 0x056,
    AREA_GOLDSHIRE = 0x057,
    AREA_EASTVALE_LOGGING_CAMP = 0x058,
    AREA_MIRROR_LAKE_ORCHARD = 0x059,
    AREA_TOWER_OF_AZORA = 0x05B,
    AREA_MIRROR_LAKE = 0x05C,
    AREA_VUL_GOL_OGRE_MOUND = 0x05D,
    AREA_RAVEN_HILL = 0x05E,
    AREA_REDRIDGE_CANYONS = 0x05F,
    AREA_TOWER_OF_ILGALAR = 0x060,
    AREA_ALTHERS_MILL = 0x061,
    AREA_RETHBAN_CAVERNS = 0x062,
    AREA_REBEL_CAMP = 0x063,
    AREA_NESINGWARYS_EXPEDITION = 0x064,
    AREA_KURZENS_COMPOUND = 0x065,
    AREA_RUINS_OF_ZUL_KUNDA = 0x066,
    AREA_RUINS_OF_ZUL_MAMWE = 0x067,
    AREA_THE_VILE_REEF = 0x068,
    AREA_MOSH_OGG_OGRE_MOUND = 0x069,
    AREA_THE_STOCKPILE = 0x06A,
    AREA_SALDEANS_FARM = 0x06B,
    AREA_SENTINEL_HILL = 0x06C,
    AREA_FURLBROWS_PUMPKIN_FARM = 0x06D,
    AREA_JANGOLODE_MINE = 0x06F,
    AREA_GOLD_COAST_QUARRY = 0x071,
    AREA_WESTFALL_LIGHTHOUSE = 0x073,
    AREA_MISTY_VALLEY = 0x074,
    AREA_GROM_GOL_BASE_CAMP = 0x075,
    AREA_WHELGARS_EXCAVATION_SITE = 0x076,
    AREA_WESTBROOK_GARRISON = 0x078,
    AREA_TRANQUIL_GARDENS_CEMETERY = 0x079,
    AREA_ZUULDAIA_RUINS = 0x07A,
    AREA_BAL_LAL_RUINS = 0x07B,
    AREA_KAL_AI_RUINS = 0x07D,
    AREA_TKASHI_RUINS = 0x07E,
    AREA_BALIA_MAH_RUINS = 0x07F,
    AREA_ZIATAJAI_RUINS = 0x080,
    AREA_MIZJAH_RUINS = 0x081,
    AREA_SILVERPINE_FOREST = 0x082,
    AREA_KHARANOS = 0x083,
    AREA_COLDRIDGE_VALLEY = 0x084,
    AREA_GNOMEREGAN0 = 0x085,
    AREA_GOL_BOLAR_QUARRY = 0x086,
    AREA_FROSTMANE_HOLD = 0x087,
    AREA_THE_GRIZZLED_DEN = 0x088,
    AREA_BREWNALL_VILLAGE = 0x089,
    AREA_MISTY_PINE_REFUGE = 0x08A,
    AREA_EASTERN_PLAGUELANDS = 0x08B,
    AREA_TELDRASSIL = 0x08D,
    AREA_IRONBANDS_EXCAVATION_SITE = 0x08E,
    AREA_MO_GROSH_STRONGHOLD = 0x08F,
    AREA_THELSAMAR = 0x090,
    AREA_ALGAZ_GATE = 0x091,
    AREA_STONEWROUGHT_DAM = 0x092,
    AREA_THE_FARSTRIDER_LODGE = 0x093,
    AREA_DARKSHORE = 0x094,
    AREA_SILVER_STREAM_MINE = 0x095,
    AREA_MENETHIL_HARBOR = 0x096,
    AREA_DESIGNER_ISLAND = 0x097,
    AREA_THE_BULWARK0 = 0x098,
    AREA_RUINS_OF_LORDAERON = 0x099,
    AREA_DEATHKNELL = 0x09A,
    AREA_NIGHT_WEBS_HOLLOW = 0x09B,
    AREA_SOLLIDEN_FARMSTEAD = 0x09C,
    AREA_AGAMAND_MILLS = 0x09D,
    AREA_AGAMAND_FAMILY_CRYPT = 0x09E,
    AREA_BRILL = 0x09F,
    AREA_WHISPERING_GARDENS = 0x0A0,
    AREA_TERRACE_OF_REPOSE = 0x0A1,
    AREA_BRIGHTWATER_LAKE = 0x0A2,
    AREA_GUNTHERS_RETREAT = 0x0A3,
    AREA_GARRENS_HAUNT = 0x0A4,
    AREA_BALNIR_FARMSTEAD = 0x0A5,
    AREA_COLD_HEARTH_MANOR = 0x0A6,
    AREA_CRUSADER_OUTPOST = 0x0A7,
    AREA_THE_NORTH_COAST = 0x0A8,
    AREA_WHISPERING_SHORE = 0x0A9,
    AREA_LORDAMERE_LAKE0 = 0x0AA,
    AREA_FENRIS_ISLE = 0x0AC,
    AREA_FAOLS_REST = 0x0AD,
    AREA_DOLANAAR = 0x0BA,
    AREA_DARNASSUS_UNUSED = 0x0BB,
    AREA_SHADOWGLEN = 0x0BC,
    AREA_STEELGRILLS_DEPOT = 0x0BD,
    AREA_HEARTHGLEN = 0x0BE,
    AREA_NORTHRIDGE_LUMBER_CAMP = 0x0C0,
    AREA_RUINS_OF_ANDORHAL = 0x0C1,
    AREA_SCHOOL_OF_NECROMANCY = 0x0C3,
    AREA_UTHERS_TOMB = 0x0C4,
    AREA_SORROW_HILL = 0x0C5,
    AREA_THE_WEEPING_CAVE = 0x0C6,
    AREA_FELSTONE_FIELD = 0x0C7,
    AREA_DALSONS_TEARS = 0x0C8,
    AREA_GAHRRONS_WITHERING = 0x0C9,
    AREA_THE_WRITHING_HAUNT = 0x0CA,
    AREA_MARDENHOLDE_KEEP = 0x0CB,
    AREA_PYREWOOD_VILLAGE = 0x0CC,
    AREA_DUN_MODR = 0x0CD,
    AREA_WESTFALL1 = 0x0CE,
    AREA_THE_GREAT_SEA0 = 0x0CF,
    AREA_UNUSED_IRONCLADCOVE = 0x0D0,
    AREA_SHADOWFANG_KEEP0 = 0x0D1,
    AREA_ON_MAP_DUNGEON3 = 0x0D2,
    AREA_ICEFLOW_LAKE = 0x0D3,
    AREA_HELMS_BED_LAKE = 0x0D4,
    AREA_DEEP_ELEM_MINE = 0x0D5,
    AREA_THE_GREAT_SEA1 = 0x0D6,
    AREA_MULGORE = 0x0D7,
    AREA_ALEXSTON_FARMSTEAD = 0x0DB,
    AREA_RED_CLOUD_MESA = 0x0DC,
    AREA_CAMP_NARACHE = 0x0DD,
    AREA_BLOODHOOF_VILLAGE = 0x0DE,
    AREA_STONEBULL_LAKE = 0x0DF,
    AREA_RAVAGED_CARAVAN = 0x0E0,
    AREA_RED_ROCKS = 0x0E1,
    AREA_THE_SKITTERING_DARK = 0x0E2,
    AREA_VALGANS_FIELD = 0x0E3,
    AREA_THE_SEPULCHER = 0x0E4,
    AREA_OLSENS_FARTHING = 0x0E5,
    AREA_THE_GREYMANE_WALL = 0x0E6,
    AREA_BERENS_PERIL = 0x0E7,
    AREA_THE_DAWNING_ISLES = 0x0E8,
    AREA_AMBERMILL = 0x0E9,
    AREA_FENRIS_KEEP = 0x0EB,
    AREA_SHADOWFANG_KEEP1 = 0x0EC,
    AREA_THE_DECREPIT_FERRY = 0x0ED,
    AREA_MALDENS_ORCHARD = 0x0EE,
    AREA_THE_IVAR_PATCH = 0x0EF,
    AREA_THE_DEAD_FIELD = 0x0F0,
    AREA_THE_ROTTING_ORCHARD = 0x0F1,
    AREA_BRIGHTWOOD_GROVE = 0x0F2,
    AREA_FORLORN_ROWE = 0x0F3,
    AREA_THE_WHIPPLE_ESTATE = 0x0F4,
    AREA_THE_YORGEN_FARMSTEAD = 0x0F5,
    AREA_THE_CAULDRON = 0x0F6,
    AREA_GRIMESILT_DIG_SITE = 0x0F7,
    AREA_DREADMAUL_ROCK = 0x0F9,
    AREA_RUINS_OF_THAURISSAN = 0x0FA,
    AREA_FLAME_CREST = 0x0FB,
    AREA_BLACKROCK_STRONGHOLD = 0x0FC,
    AREA_THE_PILLAR_OF_ASH = 0x0FD,
    AREA_BLACKROCK_MOUNTAIN1 = 0x0FE,
    AREA_ALTAR_OF_STORMS0 = 0x0FF,
    AREA_ALDRASSIL = 0x100,
    AREA_SHADOWTHREAD_CAVE = 0x101,
    AREA_FEL_ROCK = 0x102,
    AREA_LAKE_AL_AMETH = 0x103,
    AREA_STARBREEZE_VILLAGE = 0x104,
    AREA_GNARLPINE_HOLD = 0x105,
    AREA_BAN_ETHIL_BARROW_DEN = 0x106,
    AREA_THE_CLEFT = 0x107,
    AREA_THE_ORACLE_GLADE = 0x108,
    AREA_WELLSPRING_RIVER = 0x109,
    AREA_WELLSPRING_LAKE = 0x10A,
    AREA_HILLSBRAD_FOOTHILLS = 0x10B,
    AREA_AZSHARA_CRATER = 0x10C,
    AREA_DUN_ALGAZ0 = 0x10D,
    AREA_SOUTHSHORE0 = 0x10F,
    AREA_TARREN_MILL0 = 0x110,
    AREA_DURNHOLDE_KEEP0 = 0x113,
    AREA_UNUSED_STONEWROUGHT_PASS = 0x114,
    AREA_THE_FOOTHILL_CAVERNS = 0x115,
    AREA_LORDAMERE_INTERNMENT_CAMP = 0x116,
    AREA_DALARAN = 0x117,
    AREA_STRAHNBRAD = 0x118,
    AREA_RUINS_OF_ALTERAC = 0x119,
    AREA_CRUSHRIDGE_HOLD = 0x11A,
    AREA_SLAUGHTER_HOLLOW = 0x11B,
    AREA_THE_UPLANDS = 0x11C,
    AREA_SOUTHPOINT_TOWER0 = 0x11D,
    AREA_HILLSBRAD_FIELDS0 = 0x11E,
    AREA_HILLSBRAD = 0x11F,
    AREA_AZURELODE_MINE0 = 0x120,
    AREA_NETHANDER_STEAD0 = 0x121,
    AREA_DUN_GAROK0 = 0x122,
    AREA_THORADINS_WALL0 = 0x125,
    AREA_EASTERN_STRAND0 = 0x126,
    AREA_WESTERN_STRAND0 = 0x127,
    AREA_SOUTH_SEAS_UNUSED = 0x128,
    AREA_JAGUERO_ISLE = 0x129,
    AREA_BARADIN_BAY = 0x12A,
    AREA_MENETHIL_BAY = 0x12B,
    AREA_MISTY_REED_STRAND = 0x12C,
    AREA_THE_SAVAGE_COAST = 0x12D,
    AREA_THE_CRYSTAL_SHORE = 0x12E,
    AREA_SHELL_BEACH = 0x12F,
    AREA_NORTH_TIDES_RUN = 0x131,
    AREA_SOUTH_TIDES_RUN = 0x132,
    AREA_THE_OVERLOOK_CLIFFS = 0x133,
    AREA_THE_FORBIDDING_SEA0 = 0x134,
    AREA_IRONBEARDS_TOMB = 0x135,
    AREA_CRYSTALVEIN_MINE = 0x136,
    AREA_RUINS_OF_ABORAZ = 0x137,
    AREA_JANEIROS_POINT = 0x138,
    AREA_NORTHFOLD_MANOR = 0x139,
    AREA_GO_SHEK_FARM = 0x13A,
    AREA_DABYRIES_FARMSTEAD = 0x13B,
    AREA_BOULDERFIST_HALL = 0x13C,
    AREA_WITHERBARK_VILLAGE = 0x13D,
    AREA_DRYWHISKER_GORGE = 0x13E,
    AREA_REFUGE_POINTE = 0x140,
    AREA_HAMMERFALL = 0x141,
    AREA_BLACKWATER_SHIPWRECKS = 0x142,
    AREA_O_BREENS_CAMP = 0x143,
    AREA_STROMGARDE_KEEP = 0x144,
    AREA_THE_TOWER_OF_ARATHOR = 0x145,
    AREA_THE_SANCTUM = 0x146,
    AREA_FALDIRS_COVE = 0x147,
    AREA_THE_DROWNED_REEF = 0x148,
    AREA_THANDOL_SPAN0 = 0x14A,
    AREA_ASHENVALE = 0x14B,
    AREA_THE_GREAT_SEA2 = 0x14C,
    AREA_CIRCLE_OF_EAST_BINDING = 0x14D,
    AREA_CIRCLE_OF_WEST_BINDING = 0x14E,
    AREA_CIRCLE_OF_INNER_BINDING = 0x14F,
    AREA_CIRCLE_OF_OUTER_BINDING = 0x150,
    AREA_APOCRYPHANS_REST = 0x151,
    AREA_ANGOR_FORTRESS = 0x152,
    AREA_LETHLOR_RAVINE = 0x153,
    AREA_KARGATH = 0x154,
    AREA_CAMP_KOSH = 0x155,
    AREA_CAMP_BOFF = 0x156,
    AREA_CAMP_WURG = 0x157,
    AREA_CAMP_CAGG = 0x158,
    AREA_AGMONDS_END = 0x159,
    AREA_HAMMERTOES_DIGSITE = 0x15A,
    AREA_DUSTBELCH_GROTTO = 0x15B,
    AREA_AERIE_PEAK = 0x15C,
    AREA_WILDHAMMER_KEEP = 0x15D,
    AREA_QUEL_DANIL_LODGE = 0x15E,
    AREA_SKULK_ROCK = 0x15F,
    AREA_ZUN_WATHA = 0x160,
    AREA_SHADRA_ALOR = 0x161,
    AREA_JINTHA_ALOR = 0x162,
    AREA_THE_ALTAR_OF_ZUL = 0x163,
    AREA_SERADANE = 0x164,
    AREA_FERALAS = 0x165,
    AREA_BRAMBLEBLADE_RAVINE = 0x166,
    AREA_BAEL_MODAN = 0x167,
    AREA_THE_VENTURE_CO_MINE = 0x168,
    AREA_FELWOOD = 0x169,
    AREA_RAZOR_HILL = 0x16A,
    AREA_VALLEY_OF_TRIALS = 0x16B,
    AREA_THE_DEN = 0x16C,
    AREA_BURNING_BLADE_COVEN = 0x16D,
    AREA_KOLKAR_CRAG = 0x16E,
    AREA_SEN_JIN_VILLAGE = 0x16F,
    AREA_ECHO_ISLES = 0x170,
    AREA_THUNDER_RIDGE = 0x171,
    AREA_DRYGULCH_RAVINE = 0x172,
    AREA_DUSTWIND_CAVE = 0x173,
    AREA_TIRAGARDE_KEEP = 0x174,
    AREA_SCUTTLE_COAST = 0x175,
    AREA_BLADEFIST_BAY = 0x176,
    AREA_DEADEYE_SHORE = 0x177,
    AREA_SOUTHFURY_RIVER0 = 0x179,
    AREA_CAMP_TAURAJO = 0x17A,
    AREA_FAR_WATCH_POST = 0x17B,
    AREA_THE_CROSSROADS = 0x17C,
    AREA_BOULDER_LODE_MINE = 0x17D,
    AREA_THE_SLUDGE_FEN = 0x17E,
    AREA_THE_DRY_HILLS = 0x17F,
    AREA_DREADMIST_PEAK = 0x180,
    AREA_NORTHWATCH_HOLD = 0x181,
    AREA_THE_FORGOTTEN_POOLS = 0x182,
    AREA_LUSHWATER_OASIS = 0x183,
    AREA_THE_STAGNANT_OASIS = 0x184,
    AREA_FIELD_OF_GIANTS = 0x186,
    AREA_THE_MERCHANT_COAST = 0x187,
    AREA_RATCHET = 0x188,
    AREA_DARKSPEAR_STRAND = 0x189,
    AREA_DARROWMERE_LAKE_UNUSED = 0x18A,
    AREA_CAER_DARROW_UNUSED = 0x18B,
    AREA_WINTERHOOF_WATER_WELL = 0x18C,
    AREA_THUNDERHORN_WATER_WELL = 0x18D,
    AREA_WILDMANE_WATER_WELL = 0x18E,
    AREA_SKYLINE_RIDGE = 0x18F,
    AREA_THOUSAND_NEEDLES = 0x190,
    AREA_THE_TIDUS_STAIR = 0x191,
    AREA_SHADY_REST_INN = 0x193,
    AREA_BAEL_DUN_DIGSITE = 0x194,
    AREA_DESOLACE = 0x195,
    AREA_STONETALON_MOUNTAINS = 0x196,
    AREA_ORGRIMMAR_UNUSED = 0x197,
    AREA_GILLIJIMS_ISLE = 0x198,
    AREA_ISLAND_OF_DOCTOR_LAPIDIS = 0x199,
    AREA_RAZORWIND_CANYON = 0x19A,
    AREA_BATHRANS_HAUNT = 0x19B,
    AREA_THE_RUINS_OF_ORDIL_ARAN = 0x19C,
    AREA_MAESTRAS_POST = 0x19D,
    AREA_THE_ZORAM_STRAND = 0x19E,
    AREA_ASTRANAAR = 0x19F,
    AREA_THE_SHRINE_OF_AESSINA = 0x1A0,
    AREA_FIRE_SCAR_SHRINE = 0x1A1,
    AREA_THE_RUINS_OF_STARDUST = 0x1A2,
    AREA_THE_HOWLING_VALE = 0x1A3,
    AREA_SILVERWIND_REFUGE = 0x1A4,
    AREA_MYSTRAL_LAKE = 0x1A5,
    AREA_FALLEN_SKY_LAKE = 0x1A6,
    AREA_IRIS_LAKE = 0x1A8,
    AREA_MOONWELL = 0x1A9,
    AREA_RAYNEWOOD_RETREAT = 0x1AA,
    AREA_THE_SHADY_NOOK = 0x1AB,
    AREA_NIGHT_RUN = 0x1AC,
    AREA_XAVIAN = 0x1AD,
    AREA_SATYRNAAR = 0x1AE,
    AREA_SPLINTERTREE_POST = 0x1AF,
    AREA_THE_DOR_DANIL_BARROW_DEN = 0x1B0,
    AREA_FALFARREN_RIVER = 0x1B1,
    AREA_FELFIRE_HILL = 0x1B2,
    AREA_DEMON_FALL_CANYON = 0x1B3,
    AREA_DEMON_FALL_RIDGE = 0x1B4,
    AREA_WARSONG_LUMBER_CAMP = 0x1B5,
    AREA_BOUGH_SHADOW = 0x1B6,
    AREA_THE_SHIMMERING_FLATS = 0x1B7,
    AREA_TANARIS = 0x1B8,
    AREA_LAKE_FALATHIM = 0x1B9,
    AREA_AUBERDINE = 0x1BA,
    AREA_RUINS_OF_MATHYSTRA = 0x1BB,
    AREA_TOWER_OF_ALTHALAXX = 0x1BC,
    AREA_CLIFFSPRING_FALLS = 0x1BD,
    AREA_BASHAL_ARAN = 0x1BE,
    AREA_AMETH_ARAN = 0x1BF,
    AREA_GROVE_OF_THE_ANCIENTS = 0x1C0,
    AREA_THE_MASTERS_GLAIVE = 0x1C1,
    AREA_REMTRAVELS_EXCAVATION = 0x1C2,
    AREA_MISTS_EDGE = 0x1C4,
    AREA_THE_LONG_WASH = 0x1C5,
    AREA_WILDBEND_RIVER = 0x1C6,
    AREA_BLACKWOOD_DEN = 0x1C7,
    AREA_CLIFFSPRING_RIVER = 0x1C8,
    AREA_THE_VEILED_SEA0 = 0x1C9,
    AREA_GOLD_ROAD = 0x1CA,
    AREA_SCARLET_WATCH_POST = 0x1CB,
    AREA_SUN_ROCK_RETREAT = 0x1CC,
    AREA_WINDSHEAR_CRAG = 0x1CD,
    AREA_CRAGPOOL_LAKE = 0x1CF,
    AREA_MIRKFALLON_LAKE = 0x1D0,
    AREA_THE_CHARRED_VALE = 0x1D1,
    AREA_VALLEY_OF_THE_BLOODFURIES = 0x1D2,
    AREA_STONETALON_PEAK = 0x1D3,
    AREA_THE_TALON_DEN = 0x1D4,
    AREA_GREATWOOD_VALE = 0x1D5,
    AREA_THUNDER_BLUFF_UNUSED = 0x1D6,
    AREA_BRAVE_WIND_MESA = 0x1D7,
    AREA_FIRE_STONE_MESA = 0x1D8,
    AREA_MANTLE_ROCK = 0x1D9,
    AREA_HUNTER_RISE_UNUSED = 0x1DA,
    AREA_SPIRIT_RISE_UNUSED = 0x1DB,
    AREA_ELDER_RISE_UNUSED = 0x1DC,
    AREA_RUINS_OF_JUBUWAL = 0x1DD,
    AREA_POOLS_OF_ARLITHRIEN = 0x1DE,
    AREA_THE_RUSTMAUL_DIG_SITE = 0x1DF,
    AREA_CAMP_E_THOK = 0x1E0,
    AREA_SPLITHOOF_CRAG = 0x1E1,
    AREA_HIGHPERCH = 0x1E2,
    AREA_THE_SCREECHING_CANYON = 0x1E3,
    AREA_FREEWIND_POST = 0x1E4,
    AREA_THE_GREAT_LIFT0 = 0x1E5,
    AREA_GALAK_HOLD = 0x1E6,
    AREA_ROGUEFEATHER_DEN = 0x1E7,
    AREA_THE_WEATHERED_NOOK = 0x1E8,
    AREA_THALANAAR = 0x1E9,
    AREA_UN_GORO_CRATER = 0x1EA,
    AREA_RAZORFEN_KRAUL0 = 0x1EB,
    AREA_RAVEN_HILL_CEMETERY = 0x1EC,
    AREA_MOONGLADE = 0x1ED,
    AREA_DELETE_ME0 = 0x1EF,
    AREA_BRACKENWALL_VILLAGE = 0x1F0,
    AREA_SWAMPLIGHT_MANOR = 0x1F1,
    AREA_BLOODFEN_BURROW = 0x1F2,
    AREA_DARKMIST_CAVERN = 0x1F3,
    AREA_MOGGLE_POINT = 0x1F4,
    AREA_BEEZILS_WRECK = 0x1F5,
    AREA_WITCH_HILL = 0x1F6,
    AREA_SENTRY_POINT = 0x1F7,
    AREA_NORTH_POINT_TOWER = 0x1F8,
    AREA_WEST_POINT_TOWER = 0x1F9,
    AREA_LOST_POINT = 0x1FA,
    AREA_BLUEFEN = 0x1FB,
    AREA_STONEMAUL_RUINS = 0x1FC,
    AREA_THE_DEN_OF_FLAME = 0x1FD,
    AREA_THE_DRAGONMURK = 0x1FE,
    AREA_WYRMBOG = 0x1FF,
    AREA_ONYXIAS_LAIR_UNUSED = 0x200,
    AREA_THERAMORE_ISLE = 0x201,
    AREA_FOOTHOLD_CITADEL = 0x202,
    AREA_IRONCLAD_PRISON = 0x203,
    AREA_DUSTWALLOW_BAY = 0x204,
    AREA_TIDEFURY_COVE = 0x205,
    AREA_DREADMURK_SHORE = 0x206,
    AREA_ADDLES_STEAD = 0x218,
    AREA_FIRE_PLUME_RIDGE = 0x219,
    AREA_LAKKARI_TAR_PITS = 0x21A,
    AREA_TERROR_RUN = 0x21B,
    AREA_THE_SLITHERING_SCAR = 0x21C,
    AREA_MARSHALS_REFUGE = 0x21D,
    AREA_FUNGAL_ROCK = 0x21E,
    AREA_GOLAKKA_HOT_SPRINGS = 0x21F,
    AREA_THE_LOCH = 0x22C,
    AREA_BEGGARS_HAUNT = 0x240,
    AREA_KODO_GRAVEYARD = 0x254,
    AREA_GHOST_WALKER_POST = 0x255,
    AREA_SAR_THERIS_STRAND = 0x256,
    AREA_THUNDER_AXE_FORTRESS = 0x257,
    AREA_BOLGANS_HOLE = 0x258,
    AREA_MANNOROC_COVEN = 0x25A,
    AREA_SARGERON = 0x25B,
    AREA_MAGRAM_VILLAGE = 0x25C,
    AREA_GELKIS_VILLAGE = 0x25E,
    AREA_VALLEY_OF_SPEARS = 0x25F,
    AREA_NIJELS_POINT = 0x260,
    AREA_KOLKAR_VILLAGE = 0x261,
    AREA_HYJAL = 0x268,
    AREA_WINTERSPRING = 0x26A,
    AREA_BLACKWOLF_RIVER = 0x27C,
    AREA_KODO_ROCK = 0x27D,
    AREA_HIDDEN_PATH = 0x27E,
    AREA_SPIRIT_ROCK = 0x27F,
    AREA_SHRINE_OF_THE_DORMANT_FLAME = 0x280,
    AREA_LAKE_ELUNE_ARA = 0x290,
    AREA_THE_HARBORAGE = 0x291,
    AREA_OUTLAND = 0x2A4,
    AREA_CRAFTSMENS_TERRACE_UNUSED = 0x2B8,
    AREA_TRADESMENS_TERRACE_UNUSED = 0x2B9,
    AREA_THE_TEMPLE_GARDENS_UNUSED = 0x2BA,
    AREA_TEMPLE_OF_ELUNE_UNUSED = 0x2BB,
    AREA_CENARION_ENCLAVE_UNUSED = 0x2BC,
    AREA_WARRIORS_TERRACE_UNUSED = 0x2BD,
    AREA_RUT_THERAN_VILLAGE = 0x2BE,
    AREA_IRONBANDS_COMPOUND = 0x2CC,
    AREA_THE_STOCKADE = 0x2CD,
    AREA_WAILING_CAVERNS = 0x2CE,
    AREA_BLACKFATHOM_DEEPS0 = 0x2CF,
    AREA_FRAY_ISLAND = 0x2D0,
    AREA_GNOMEREGAN1 = 0x2D1,
    AREA_RAZORFEN_DOWNS0 = 0x2D2,
    AREA_BAN_ETHIL_HOLLOW = 0x2E0,
    AREA_SCARLET_MONASTERY = 0x31C,
    AREA_JERODS_LANDING = 0x31D,
    AREA_RIDGEPOINT_TOWER = 0x31E,
    AREA_THE_DARKENED_BANK = 0x31F,
    AREA_COLDRIDGE_PASS = 0x320,
    AREA_CHILL_BREEZE_VALLEY = 0x321,
    AREA_SHIMMER_RIDGE = 0x322,
    AREA_AMBERSTILL_RANCH = 0x323,
    AREA_THE_TUNDRID_HILLS = 0x324,
    AREA_SOUTH_GATE_PASS0 = 0x325,
    AREA_SOUTH_GATE_OUTPOST = 0x326,
    AREA_NORTH_GATE_PASS0 = 0x327,
    AREA_NORTH_GATE_OUTPOST = 0x328,
    AREA_GATES_OF_IRONFORGE = 0x329,
    AREA_STILLWATER_POND = 0x32A,
    AREA_NIGHTMARE_VALE = 0x32B,
    AREA_VENOMWEB_VALE = 0x32C,
    AREA_THE_BULWARK1 = 0x32D,
    AREA_SOUTHFURY_RIVER1 = 0x32E,
    AREA_SOUTHFURY_RIVER2 = 0x32F,
    AREA_RAZORMANE_GROUNDS = 0x330,
    AREA_SKULL_ROCK = 0x331,
    AREA_PALEMANE_ROCK = 0x332,
    AREA_WINDFURY_RIDGE = 0x333,
    AREA_THE_GOLDEN_PLAINS = 0x334,
    AREA_THE_ROLLING_PLAINS = 0x335,
    AREA_DUN_ALGAZ1 = 0x344,
    AREA_DUN_ALGAZ2 = 0x345,
    AREA_NORTH_GATE_PASS1 = 0x346,
    AREA_SOUTH_GATE_PASS1 = 0x347,
    AREA_TWILIGHT_GROVE = 0x358,
    AREA_GM_ISLAND = 0x36C,
    AREA_DELETE_ME1 = 0x36D,
    AREA_SOUTHFURY_RIVER3 = 0x36E,
    AREA_SOUTHFURY_RIVER4 = 0x36F,
    AREA_THANDOL_SPAN1 = 0x370,
    AREA_THANDOL_SPAN2 = 0x371,
    AREA_PURGATION_ISLE = 0x380,
    AREA_THE_JANSEN_STEAD = 0x394,
    AREA_THE_DEAD_ACRE = 0x395,
    AREA_THE_MOLSEN_FARM = 0x396,
    AREA_STENDELS_POND = 0x397,
    AREA_THE_DAGGER_HILLS = 0x398,
    AREA_DEMONTS_PLACE = 0x399,
    AREA_THE_DUST_PLAINS = 0x39A,
    AREA_STONESPLINTER_VALLEY = 0x39B,
    AREA_VALLEY_OF_KINGS = 0x39C,
    AREA_ALGAZ_STATION = 0x39D,
    AREA_BUCKLEBREE_FARM = 0x39E,
    AREA_THE_SHINING_STRAND = 0x39F,
    AREA_NORTH_TIDES_HOLLOW = 0x3A0,
    AREA_GRIZZLEPAW_RIDGE = 0x3A8,
    AREA_THE_VERDANT_FIELDS = 0x3BC,
    AREA_GADGETZAN = 0x3D0,
    AREA_STEAMWHEEDLE_PORT = 0x3D1,
    AREA_ZUL_FARRAK0 = 0x3D2,
    AREA_SANDSORROW_WATCH = 0x3D3,
    AREA_THISTLESHRUB_VALLEY = 0x3D4,
    AREA_THE_GAPING_CHASM = 0x3D5,
    AREA_THE_NOXIOUS_LAIR = 0x3D6,
    AREA_DUNEMAUL_COMPOUND = 0x3D7,
    AREA_EASTMOON_RUINS = 0x3D8,
    AREA_WATERSPRING_FIELD = 0x3D9,
    AREA_ZALASHJIS_DEN = 0x3DA,
    AREA_LANDS_END_BEACH = 0x3DB,
    AREA_WAVESTRIDER_BEACH = 0x3DC,
    AREA_ULDUM = 0x3DD,
    AREA_VALLEY_OF_THE_WATCHERS = 0x3DE,
    AREA_GUNSTANS_POST = 0x3DF,
    AREA_SOUTHMOON_RUINS = 0x3E0,
    AREA_RENDERS_CAMP = 0x3E4,
    AREA_RENDERS_VALLEY = 0x3E5,
    AREA_RENDERS_ROCK = 0x3E6,
    AREA_STONEWATCH_TOWER = 0x3E7,
    AREA_GALARDELL_VALLEY = 0x3E8,
    AREA_LAKERIDGE_HIGHWAY = 0x3E9,
    AREA_THREE_CORNERS = 0x3EA,
    AREA_DIREFORGE_HILL = 0x3F8,
    AREA_RAPTOR_RIDGE = 0x3F9,
    AREA_BLACK_CHANNEL_MARSH = 0x3FA,
    AREA_THE_GREEN_BELT0 = 0x3FB,
    AREA_MOSSHIDE_FEN = 0x3FC,
    AREA_THELGEN_ROCK = 0x3FD,
    AREA_BLUEGILL_MARSH = 0x3FE,
    AREA_SALTSPRAY_GLEN = 0x3FF,
    AREA_SUNDOWN_MARSH = 0x400,
    AREA_THE_GREEN_BELT1 = 0x401,
    AREA_ANGERFANG_ENCAMPMENT = 0x40C,
    AREA_GRIM_BATOL = 0x40D,
    AREA_DRAGONMAW_GATES = 0x40E,
    AREA_THE_LOST_FLEET = 0x40F,
    AREA_DARROW_HILL0 = 0x420,
    AREA_THORADINS_WALL1 = 0x421,
    AREA_WEBWINDER_PATH = 0x434,
    AREA_THE_HUSHED_BANK = 0x449,
    AREA_MANOR_MISTMANTLE = 0x44A,
    AREA_CAMP_MOJACHE = 0x44B,
    AREA_GRIMTOTEM_COMPOUND = 0x44C,
    AREA_THE_WRITHING_DEEP = 0x44D,
    AREA_WILDWIND_LAKE = 0x44E,
    AREA_GORDUNNI_OUTPOST = 0x44F,
    AREA_MOK_GORDUN = 0x450,
    AREA_FERAL_SCAR_VALE = 0x451,
    AREA_FRAYFEATHER_HIGHLANDS = 0x452,
    AREA_IDLEWIND_LAKE = 0x453,
    AREA_THE_FORGOTTEN_COAST = 0x454,
    AREA_EAST_PILLAR = 0x455,
    AREA_WEST_PILLAR = 0x456,
    AREA_DREAM_BOUGH = 0x457,
    AREA_JADEMIR_LAKE = 0x458,
    AREA_ONEIROS = 0x459,
    AREA_RUINS_OF_RAVENWIND = 0x45A,
    AREA_RAGE_SCAR_HOLD = 0x45B,
    AREA_FEATHERMOON_STRONGHOLD = 0x45C,
    AREA_RUINS_OF_SOLARSAL = 0x45D,
    AREA_LOWER_WILDS_UNUSED = 0x45E,
    AREA_THE_TWIN_COLOSSALS = 0x45F,
    AREA_SARDOR_ISLE = 0x460,
    AREA_ISLE_OF_DREAD = 0x461,
    AREA_HIGH_WILDERNESS = 0x470,
    AREA_LOWER_WILDS = 0x471,
    AREA_SOUTHERN_BARRENS = 0x484,
    AREA_SOUTHERN_GOLD_ROAD = 0x485,
    AREA_ZUL_FARRAK1 = 0x498,
    AREA_UNUSED_ALCAZ_ISLAND = 0x4AC,
    AREA_TIMBERMAW_HOLD0 = 0x4C0,
    AREA_VANNDIR_ENCAMPMENT = 0x4C1,
    AREA_TEST_AZSHARA = 0x4C2,
    AREA_LEGASH_ENCAMPMENT = 0x4C3,
    AREA_THALASSIAN_BASE_CAMP = 0x4C4,
    AREA_RUINS_OF_ELDARATH = 0x4C5,
    AREA_HETAERAS_CLUTCH = 0x4C6,
    AREA_TEMPLE_OF_ZIN_MALOR = 0x4C7,
    AREA_BEARS_HEAD = 0x4C8,
    AREA_URSOLAN = 0x4C9,
    AREA_TEMPLE_OF_ARKKORAN = 0x4CA,
    AREA_BAY_OF_STORMS = 0x4CB,
    AREA_THE_SHATTERED_STRAND = 0x4CC,
    AREA_TOWER_OF_ELDARA = 0x4CD,
    AREA_JAGGED_REEF = 0x4CE,
    AREA_SOUTHRIDGE_BEACH = 0x4CF,
    AREA_RAVENCREST_MONUMENT = 0x4D0,
    AREA_FORLORN_RIDGE = 0x4D1,
    AREA_LAKE_MENNAR = 0x4D2,
    AREA_SHADOWSONG_SHRINE = 0x4D3,
    AREA_HALDARR_ENCAMPMENT = 0x4D4,
    AREA_VALORMOK = 0x4D5,
    AREA_THE_RUINED_REACHES = 0x4E8,
    AREA_THE_TALONDEEP_PATH0 = 0x4FC,
    AREA_THE_TALONDEEP_PATH1 = 0x4FD,
    AREA_ROCKTUSK_FARM = 0x510,
    AREA_JAGGEDSWINE_FARM = 0x511,
    AREA_RAZORFEN_DOWNS1 = 0x524,
    AREA_LOST_RIGGER_COVE = 0x538,
    AREA_ULDAMAN0 = 0x539,
    AREA_LORDAMERE_LAKE1 = 0x53A,
    AREA_LORDAMERE_LAKE2 = 0x53B,
    AREA_GALLOWS_CORNER = 0x54D,
    AREA_SILITHUS = 0x561,
    AREA_EMERALD_FOREST = 0x575,
    AREA_SUNKEN_TEMPLE = 0x589,
    AREA_DREADMAUL_HOLD = 0x59D,
    AREA_NETHERGARDE_KEEP = 0x59E,
    AREA_DREADMAUL_POST = 0x59F,
    AREA_SERPENTS_COIL = 0x5A0,
    AREA_ALTAR_OF_STORMS1 = 0x5A1,
    AREA_FIREWATCH_RIDGE = 0x5A2,
    AREA_THE_SLAG_PIT = 0x5A3,
    AREA_THE_SEA_OF_CINDERS = 0x5A4,
    AREA_BLACKROCK_MOUNTAIN2 = 0x5A5,
    AREA_THORIUM_POINT = 0x5A6,
    AREA_GARRISON_ARMORY = 0x5B1,
    AREA_THE_TEMPLE_OF_ATAL_HAKKAR = 0x5C5,
    AREA_UNDERCITY = 0x5D9,
    AREA_ULDAMAN1 = 0x5ED,
    AREA_NOT_USED_DEADMINES = 0x5EE,
    AREA_STORMWIND_CITY = 0x5EF,
    AREA_IRONFORGE = 0x601,
    AREA_SPLITHOOF_HOLD = 0x615,
    AREA_THE_CAPE_OF_STRANGLETHORN = 0x629,
    AREA_SOUTHERN_SAVAGE_COAST = 0x62A,
    AREA_UNUSED_THE_DEADMINES_002 = 0x62B,
    AREA_UNUSED_IRONCLAD_COVE_003 = 0x62C,
    AREA_THE_DEADMINES = 0x62D,
    AREA_IRONCLAD_COVE = 0x62E,
    AREA_BLACKROCK_SPIRE = 0x62F,
    AREA_BLACKROCK_DEPTHS = 0x630,
    AREA_RAPTOR_GROUNDS_UNUSED = 0x63D,
    AREA_GROL_DOM_FARM_UNUSED = 0x63E,
    AREA_MORSHAN_BASE_CAMP = 0x63F,
    AREA_HONORS_STAND_UNUSED = 0x640,
    AREA_BLACKTHORN_RIDGE_UNUSED = 0x641,
    AREA_BRAMBLESCAR_UNUSED = 0x642,
    AREA_AGAMA_GOR_UNUSED = 0x643,
    AREA_VALLEY_OF_HEROES = 0x651,
    AREA_ORGRIMMAR = 0x665,
    AREA_THUNDER_BLUFF = 0x666,
    AREA_ELDER_RISE = 0x667,
    AREA_SPIRIT_RISE = 0x668,
    AREA_HUNTER_RISE = 0x669,
    AREA_DARNASSUS = 0x679,
    AREA_CENARION_ENCLAVE = 0x67A,
    AREA_CRAFTSMENS_TERRACE = 0x67B,
    AREA_WARRIORS_TERRACE = 0x67C,
    AREA_THE_TEMPLE_GARDENS = 0x67D,
    AREA_TRADESMENS_TERRACE = 0x67E,
    AREA_GAVINS_NAZE = 0x68D,
    AREA_SOFERAS_NAZE = 0x68E,
    AREA_CORRAHNS_DAGGER = 0x68F,
    AREA_THE_HEADLAND = 0x690,
    AREA_MISTY_SHORE = 0x691,
    AREA_DANDREDS_FOLD = 0x692,
    AREA_GROWLESS_CAVE = 0x693,
    AREA_CHILLWIND_POINT = 0x694,
    AREA_RAPTOR_GROUNDS = 0x6A1,
    AREA_BRAMBLESCAR = 0x6A2,
    AREA_THORN_HILL = 0x6A3,
    AREA_AGAMA_GOR = 0x6A4,
    AREA_BLACKTHORN_RIDGE = 0x6A5,
    AREA_HONORS_STAND = 0x6A6,
    AREA_THE_MORSHAN_RAMPART = 0x6A7,
    AREA_GROL_DOM_FARM = 0x6A8,
    AREA_RAZORFEN_KRAUL1 = 0x6B5,
    AREA_THE_GREAT_LIFT1 = 0x6B6,
    AREA_MISTVALE_VALLEY = 0x6C9,
    AREA_NEK_MANI_WELLSPRING = 0x6CA,
    AREA_BLOODSAIL_COMPOUND = 0x6CB,
    AREA_VENTURE_CO_BASE_CAMP = 0x6CC,
    AREA_GURUBASHI_ARENA = 0x6CD,
    AREA_SPIRIT_DEN = 0x6CE,
    AREA_THE_CRIMSON_VEIL = 0x6DD,
    AREA_THE_RIPTIDE = 0x6DE,
    AREA_THE_DAMSELS_LUCK = 0x6DF,
    AREA_VENTURE_CO_OPERATIONS_CENTER = 0x6E0,
    AREA_DEADWOOD_VILLAGE = 0x6E1,
    AREA_FELPAW_VILLAGE = 0x6E2,
    AREA_JAEDENAR = 0x6E3,
    AREA_BLOODVENOM_RIVER = 0x6E4,
    AREA_BLOODVENOM_FALLS = 0x6E5,
    AREA_SHATTER_SCAR_VALE = 0x6E6,
    AREA_IRONTREE_WOODS = 0x6E7,
    AREA_IRONTREE_CAVERN = 0x6E8,
    AREA_TIMBERMAW_HOLD1 = 0x6E9,
    AREA_SHADOW_HOLD = 0x6EA,
    AREA_SHRINE_OF_THE_DECEIVER = 0x6EB,
    AREA_ITHARIUSS_CAVE = 0x6F1,
    AREA_SORROWMURK = 0x6F2,
    AREA_DRAENIL_DUR_VILLAGE = 0x6F3,
    AREA_SPLINTERSPEAR_JUNCTION = 0x6F4,
    AREA_STAGALBOG = 0x705,
    AREA_THE_SHIFTING_MIRE = 0x706,
    AREA_STAGALBOG_CAVE = 0x719,
    AREA_WITHERBARK_CAVERNS = 0x72D,
    AREA_THORADINS_WALL2 = 0x741,
    AREA_BOULDER_GOR = 0x742,
    AREA_VALLEY_OF_FANGS = 0x755,
    AREA_THE_DUSTBOWL = 0x756,
    AREA_MIRAGE_FLATS = 0x757,
    AREA_FEATHERBEARDS_HOVEL = 0x758,
    AREA_SHINDIGGERS_CAMP = 0x759,
    AREA_PLAGUEMIST_RAVINE = 0x75A,
    AREA_VALORWIND_LAKE = 0x75B,
    AREA_AGOL_WATHA = 0x75C,
    AREA_HIRI_WATHA = 0x75D,
    AREA_THE_CREEPING_RUIN = 0x75E,
    AREA_BOGENS_LEDGE = 0x75F,
    AREA_THE_MAKERS_TERRACE = 0x769,
    AREA_DUSTWIND_GULCH = 0x76A,
    AREA_SHAOL_WATHA = 0x77D,
    AREA_NOONSHADE_RUINS = 0x791,
    AREA_BROKEN_PILLAR = 0x792,
    AREA_ABYSSAL_SANDS = 0x793,
    AREA_SOUTHBREAK_SHORE = 0x794,
    AREA_CAVERNS_OF_TIME0 = 0x795,
    AREA_THE_MARSHLANDS = 0x796,
    AREA_IRONSTONE_PLATEAU = 0x797,
    AREA_BLACKCHAR_CAVE = 0x7A5,
    AREA_TANNER_CAMP = 0x7A6,
    AREA_DUSTFIRE_VALLEY = 0x7A7,
    AREA_ZUL_GURUB1 = 0x7B9,
    AREA_MISTY_REED_POST = 0x7BA,
    AREA_BLOODVENOM_POST = 0x7CD,
    AREA_TALONBRANCH_GLADE = 0x7CE,
    AREA_STRATHOLME0 = 0x7E1,
    AREA_UNUSED_SHADOWFANG_KEEP_003 = 0x7F5,
    AREA_SCHOLOMANCE = 0x809,
    AREA_TWILIGHT_VALE = 0x81D,
    AREA_TWILIGHT_SHORE = 0x81E,
    AREA_ALCAZ_ISLAND = 0x81F,
    AREA_DARKCLOUD_PINNACLE = 0x831,
    AREA_DAWNING_WOOD_CATACOMBS = 0x832,
    AREA_STONEWATCH_KEEP = 0x833,
    AREA_MARAUDON = 0x834,
    AREA_STOUTLAGER_INN = 0x835,
    AREA_THUNDERBREW_DISTILLERY = 0x836,
    AREA_MENETHIL_KEEP = 0x837,
    AREA_DEEPWATER_TAVERN = 0x838,
    AREA_SHADOW_GRAVE = 0x845,
    AREA_BRILL_TOWN_HALL = 0x846,
    AREA_GALLOWS_END_TAVERN = 0x847,
    AREA_THE_POOLS_OF_VISION_UNUSED = 0x859,
    AREA_DREADMIST_DEN = 0x85A,
    AREA_BAEL_DUN_KEEP = 0x86D,
    AREA_EMBERSTRIFES_DEN = 0x86E,
    AREA_ONYXIAS_LAIR = 0x86F,
    AREA_WINDSHEAR_MINE = 0x870,
    AREA_ROLANDS_DOOM = 0x871,
    AREA_BATTLE_RING = 0x881,
    AREA_THE_POOLS_OF_VISION = 0x895,
    AREA_SHADOWBREAK_RAVINE = 0x896,
    AREA_BROKEN_SPEAR_VILLAGE = 0x8A9,
    AREA_WHITEREACH_POST = 0x8BD,
    AREA_GORNIA = 0x8BE,
    AREA_ZANES_EYE_CRATER = 0x8BF,
    AREA_MIRAGE_RACEWAY = 0x8C0,
    AREA_FROSTSABER_ROCK = 0x8C1,
    AREA_THE_HIDDEN_GROVE = 0x8C2,
    AREA_TIMBERMAW_POST = 0x8C3,
    AREA_WINTERFALL_VILLAGE = 0x8C4,
    AREA_MAZTHORIL = 0x8C5,
    AREA_FROSTFIRE_HOT_SPRINGS = 0x8C6,
    AREA_ICE_THISTLE_HILLS = 0x8C7,
    AREA_DUN_MANDARR = 0x8C8,
    AREA_FROSTWHISPER_GORGE = 0x8C9,
    AREA_OWL_WING_THICKET = 0x8CA,
    AREA_LAKE_KEL_THERIL = 0x8CB,
    AREA_THE_RUINS_OF_KEL_THERIL = 0x8CC,
    AREA_STARFALL_VILLAGE = 0x8CD,
    AREA_BAN_THALLOW_BARROW_DEN = 0x8CE,
    AREA_EVERLOOK = 0x8CF,
    AREA_DARKWHISPER_GORGE = 0x8D0,
    AREA_DEEPRUN_TRAM = 0x8D1,
    AREA_THE_FUNGAL_VALE = 0x8D2,
    AREA_UNUSED_THE_MARRIS_STEAD = 0x8D3,
    AREA_THE_MARRIS_STEAD = 0x8D4,
    AREA_THE_UNDERCROFT = 0x8D5,
    AREA_DARROWSHIRE = 0x8D6,
    AREA_CROWN_GUARD_TOWER = 0x8D7,
    AREA_CORINS_CROSSING = 0x8D8,
    AREA_SCARLET_BASE_CAMP = 0x8D9,
    AREA_TYRS_HAND = 0x8DA,
    AREA_THE_SCARLET_BASILICA = 0x8DB,
    AREA_LIGHTS_HOPE_CHAPEL = 0x8DC,
    AREA_BROWMAN_MILL = 0x8DD,
    AREA_THE_NOXIOUS_GLADE = 0x8DE,
    AREA_EASTWALL_TOWER = 0x8DF,
    AREA_NORTHDALE = 0x8E0,
    AREA_ZUL_MASHAR = 0x8E1,
    AREA_MAZRA_ALOR = 0x8E2,
    AREA_NORTHPASS_TOWER = 0x8E3,
    AREA_QUEL_LITHIEN_LODGE = 0x8E4,
    AREA_PLAGUEWOOD = 0x8E5,
    AREA_SCOURGEHOLD = 0x8E6,
    AREA_STRATHOLME1 = 0x8E7,
    AREA_UNUSED_STRATHOLME = 0x8E8,
    AREA_DARROWMERE_LAKE0 = 0x8F9,
    AREA_CAER_DARROW = 0x8FA,
    AREA_DARROWMERE_LAKE1 = 0x8FB,
    AREA_CAVERNS_OF_TIME1 = 0x8FC,
    AREA_THISTLEFUR_VILLAGE = 0x8FD,
    AREA_THE_QUAGMIRE = 0x8FE,
    AREA_WINDBREAK_CANYON = 0x8FF,
    AREA_SOUTH_SEAS0 = 0x90D,
    AREA_THE_GREAT_SEA3 = 0x90E,
    AREA_THE_GREAT_SEA4 = 0x90F,
    AREA_THE_GREAT_SEA5 = 0x910,
    AREA_THE_GREAT_SEA6 = 0x911,
    AREA_THE_VEILED_SEA1 = 0x912,
    AREA_THE_VEILED_SEA2 = 0x913,
    AREA_THE_VEILED_SEA3 = 0x914,
    AREA_THE_VEILED_SEA4 = 0x915,
    AREA_THE_VEILED_SEA5 = 0x916,
    AREA_RAZOR_HILL_BARRACKS = 0x921,
    AREA_SOUTH_SEAS1 = 0x922,
    AREA_THE_GREAT_SEA7 = 0x923,
    AREA_BLOODTOOTH_CAMP = 0x935,
    AREA_FOREST_SONG = 0x936,
    AREA_GREENPAW_VILLAGE = 0x937,
    AREA_SILVERWING_OUTPOST = 0x938,
    AREA_NIGHTHAVEN = 0x939,
    AREA_SHRINE_OF_REMULOS = 0x93A,
    AREA_STORMRAGE_BARROW_DENS = 0x93B,
    AREA_THE_GREAT_SEA8 = 0x93C,
    AREA_THE_GREAT_SEA9 = 0x93D,
    AREA_THE_BLACK_MORASS = 0x93E,
    AREA_OLD_HILLSBRAD_FOOTHILLS = 0x93F,
    AREA_TARREN_MILL1 = 0x940,
    AREA_SOUTHSHORE1 = 0x941,
    AREA_DURNHOLDE_KEEP1 = 0x942,
    AREA_DUN_GAROK1 = 0x943,
    AREA_HILLSBRAD_FIELDS1 = 0x944,
    AREA_EASTERN_STRAND1 = 0x945,
    AREA_NETHANDER_STEAD1 = 0x946,
    AREA_DARROW_HILL1 = 0x947,
    AREA_SOUTHPOINT_TOWER1 = 0x948,
    AREA_THORADINS_WALL3 = 0x949,
    AREA_WESTERN_STRAND1 = 0x94A,
    AREA_AZURELODE_MINE1 = 0x94B,
    AREA_THE_GREAT_SEA10 = 0x95D,
    AREA_THE_GREAT_SEA11 = 0x95E,
    AREA_THE_GREAT_SEA12 = 0x95F,
    AREA_THE_FORBIDDING_SEA1 = 0x960,
    AREA_THE_FORBIDDING_SEA2 = 0x961,
    AREA_THE_FORBIDDING_SEA3 = 0x962,
    AREA_THE_FORBIDDING_SEA4 = 0x963,
    AREA_TETHRIS_ARAN = 0x964,
    AREA_ETHEL_RETHOR = 0x965,
    AREA_RANAZJAR_ISLE = 0x966,
    AREA_KORMEKS_HUT = 0x967,
    AREA_SHADOWPREY_VILLAGE = 0x968,
    AREA_BLACKROCK_PASS = 0x971,
    AREA_MORGANS_VIGIL = 0x972,
    AREA_SLITHER_ROCK = 0x973,
    AREA_TERROR_WING_PATH = 0x974,
    AREA_DRACO_DAR = 0x975,
    AREA_RAGEFIRE_CHASM = 0x985,
    AREA_NIGHTSONG_WOODS = 0x999,
    AREA_THE_VEILED_SEA6 = 0x9AD,
    AREA_MORLOS_ARAN = 0x9AE,
    AREA_EMERALD_SANCTUARY = 0x9AF,
    AREA_JADEFIRE_GLEN = 0x9B0,
    AREA_RUINS_OF_CONSTELLAS = 0x9B1,
    AREA_BITTER_REACHES = 0x9C1,
    AREA_RISE_OF_THE_DEFILER = 0x9D5,
    AREA_LARISS_PAVILION = 0x9D6,
    AREA_WOODPAW_HILLS = 0x9D7,
    AREA_WOODPAW_DEN = 0x9D8,
    AREA_VERDANTIS_RIVER = 0x9D9,
    AREA_RUINS_OF_ISILDIEN = 0x9DA,
    AREA_GRIMTOTEM_POST = 0x9E9,
    AREA_CAMP_APARAJE = 0x9EA,
    AREA_MALAKA_JIN = 0x9EB,
    AREA_BOULDERSLIDE_RAVINE = 0x9EC,
    AREA_SISHIR_CANYON = 0x9ED,
    AREA_DIRE_MAUL0 = 0x9FD,
    AREA_DEADWIND_RAVINE = 0x9FE,
    AREA_DIAMONDHEAD_RIVER = 0x9FF,
    AREA_ARIDENS_CAMP = 0xA00,
    AREA_THE_VICE = 0xA01,
    AREA_KARAZHAN = 0xA02,
    AREA_MORGANS_PLOT = 0xA03,
    AREA_DIRE_MAUL1 = 0xA11,
    AREA_ALTERAC_VALLEY0 = 0xA25,
    AREA_SCRABBLESCREWS_CAMP = 0xA39,
    AREA_JADEFIRE_RUN = 0xA3A,
    AREA_THONDRORIL_RIVER0 = 0xA3B,
    AREA_THONDRORIL_RIVER1 = 0xA3C,
    AREA_LAKE_MERELDAR = 0xA3D,
    AREA_PESTILENT_SCAR = 0xA3E,
    AREA_THE_INFECTIS_SCAR = 0xA3F,
    AREA_BLACKWOOD_LAKE = 0xA40,
    AREA_EASTWALL_GATE = 0xA41,
    AREA_TERRORWEB_TUNNEL = 0xA42,
    AREA_TERRORDALE = 0xA43,
    AREA_KARGATHIA_KEEP = 0xA4D,
    AREA_VALLEY_OF_BONES = 0xA61,
    AREA_BLACKWING_LAIR = 0xA75,
    AREA_DEADMANS_CROSSING = 0xA89,
    AREA_MOLTEN_CORE = 0xA9D,
    AREA_THE_SCARAB_WALL = 0xAB1,
    AREA_SOUTHWIND_VILLAGE = 0xAB2,
    AREA_TWILIGHT_BASE_CAMP = 0xAB3,
    AREA_THE_CRYSTAL_VALE = 0xAB4,
    AREA_THE_SCARAB_DAIS = 0xAB5,
    AREA_HIVE_ASHI = 0xAB6,
    AREA_HIVE_ZORA = 0xAB7,
    AREA_HIVE_REGAL = 0xAB8,
    AREA_SHRINE_OF_THE_FALLEN_WARRIOR = 0xAC5,
    AREA_UNUSED_ALTERAC_VALLEY = 0xAD9,
    AREA_BLACKFATHOM_DEEPS1 = 0xAED,
    AREA_ON_MAP_DUNGEON4 = 0xB01,
    AREA_THE_MASTERS_CELLAR = 0xB15,
    AREA_STONEWROUGHT_PASS = 0xB16,
    AREA_ALTERAC_VALLEY1 = 0xB17,
    AREA_THE_RUMBLE_CAGE = 0xB29,
    AREA_CHUNK_TEST = 0xB3D,
    AREA_ZORAMGAR_OUTPOST = 0xB51,
    AREA_HALL_OF_LEGENDS = 0xB65,
    AREA_CHAMPIONS_HALL = 0xB66,
    AREA_GROSH_GOK_COMPOUND = 0xB79,
    AREA_SLEEPING_GORGE = 0xB7A,
    AREA_IRONDEEP_MINE = 0xB8D,
    AREA_STONEHEARTH_OUTPOST = 0xB8E,
    AREA_DUN_BALDAR = 0xB8F,
    AREA_ICEWING_PASS = 0xB90,
    AREA_FROSTWOLF_VILLAGE = 0xB91,
    AREA_TOWER_POINT = 0xB92,
    AREA_COLDTOOTH_MINE = 0xB93,
    AREA_WINTERAX_HOLD = 0xB94,
    AREA_ICEBLOOD_GARRISON = 0xBA1,
    AREA_FROSTWOLF_KEEP = 0xBA2,
    AREA_TOR_KREN_FARM = 0xBA3,
    AREA_FROST_DAGGER_PASS = 0xBC9,
    AREA_IRONSTONE_CAMP = 0xBDD,
    AREA_WEAZELS_CRATER = 0xBDE,
    AREA_TAHONDA_RUINS = 0xBDF,
    AREA_FIELD_OF_STRIFE = 0xBF1,
    AREA_ICEWING_CAVERN = 0xBF2,
    AREA_VALORS_REST = 0xC05,
    AREA_THE_SWARMING_PILLAR = 0xC19,
    AREA_TWILIGHT_POST = 0xC1A,
    AREA_TWILIGHT_OUTPOST = 0xC1B,
    AREA_RAVAGED_TWILIGHT_CAMP = 0xC1C,
    AREA_SHALZARUS_LAIR = 0xC2D,
    AREA_TALRENDIS_POINT = 0xC41,
    AREA_RETHRESS_SANCTUM = 0xC42,
    AREA_MOON_HORROR_DEN = 0xC43,
    AREA_SCALEBEARDS_CAVE = 0xC44,
    AREA_BOULDERSLIDE_CAVERN = 0xC55,
    AREA_WARSONG_LABOR_CAMP = 0xC69,
    AREA_CHILLWIND_CAMP = 0xC7D,
    AREA_THE_MAUL = 0xC91,
    AREA_THE_MAUL_UNUSED = 0xCA5,
    AREA_BONES_OF_GRAKKAROND = 0xCB9,
    AREA_WARSONG_GULCH = 0xCCD,
    AREA_FROSTWOLF_GRAVEYARD = 0xCE1,
    AREA_FROSTWOLF_PASS = 0xCE2,
    AREA_DUN_BALDAR_PASS = 0xCE3,
    AREA_ICEBLOOD_GRAVEYARD = 0xCE4,
    AREA_SNOWFALL_GRAVEYARD = 0xCE5,
    AREA_STONEHEARTH_GRAVEYARD = 0xCE6,
    AREA_STORMPIKE_GRAVEYARD = 0xCE7,
    AREA_ICEWING_BUNKER = 0xCE8,
    AREA_STONEHEARTH_BUNKER = 0xCE9,
    AREA_WILDPAW_RIDGE = 0xCEA,
    AREA_REVANTUSK_VILLAGE = 0xCF5,
    AREA_ROCK_OF_DUROTAN = 0xCF6,
    AREA_SILVERWING_GROVE = 0xCF7,
    AREA_WARSONG_LUMBER_MILL = 0xCF8,
    AREA_SILVERWING_HOLD = 0xCF9,
    AREA_WILDPAW_CAVERN = 0xD09,
    AREA_THE_VEILED_CLEFT = 0xD0A,
    AREA_YOJAMBA_ISLE = 0xD1D,
    AREA_ARATHI_BASIN = 0xD1E,
    AREA_THE_COIL = 0xD31,
    AREA_ALTAR_OF_HIR_EEK = 0xD32,
    AREA_SHADRA_ZAAR = 0xD33,
    AREA_HAKKARI_GROUNDS = 0xD34,
    AREA_NAZE_OF_SHIRVALLAH = 0xD35,
    AREA_TEMPLE_OF_BETHEKK = 0xD36,
    AREA_THE_BLOODFIRE_PIT = 0xD37,
    AREA_ALTAR_OF_THE_BLOOD_GOD = 0xD38,
    AREA_ZANZAS_RISE = 0xD45,
    AREA_EDGE_OF_MADNESS = 0xD46,
    AREA_TROLLBANE_HALL = 0xD59,
    AREA_DEFILERS_DEN = 0xD5A,
    AREA_PAGLES_POINTE = 0xD5B,
    AREA_FARM = 0xD5C,
    AREA_BLACKSMITH = 0xD5D,
    AREA_LUMBER_MILL = 0xD5E,
    AREA_GOLD_MINE = 0xD5F,
    AREA_STABLES = 0xD60,
    AREA_CENARION_HOLD = 0xD61,
    AREA_STAGHELM_POINT = 0xD62,
    AREA_BRONZEBEARD_ENCAMPMENT = 0xD63,
    AREA_AHN_QIRAJ = 0xD64,
    AREA_RUINS_OF_AHN_QIRAJ0 = 0xD65,
    AREA_TWILIGHTS_RUN = 0xD76,
    AREA_ORTELLS_HIDEOUT = 0xD77,
    AREA_SCARAB_TERRACE = 0xD78,
    AREA_GENERALS_TERRACE = 0xD79,
    AREA_THE_RESERVOIR = 0xD7A,
    AREA_THE_HATCHERY = 0xD7B,
    AREA_THE_COMB = 0xD7C,
    AREA_WATCHERS_TERRACE = 0xD7D,
    AREA_RUINS_OF_AHN_QIRAJ1 = 0xD7E,
    AREA_NAXXRAMAS = 0xD80,
    AREA_CITY = 0xD83,
    AREA_GATES_OF_AHN_QIRAJ = 0xD96,
    AREA_RAVENHOLDT_MANOR = 0xD9E,
} e_area;
static const value_string e_area_strings[] =  {
    { AREA_NONE, "None" },
    { AREA_DUN_MOROGH, "Dun Morogh" },
    { AREA_LONGSHORE, "Longshore" },
    { AREA_BADLANDS, "Badlands" },
    { AREA_BLASTED_LANDS, "Blasted Lands" },
    { AREA_BLACKWATER_COVE, "Blackwater Cove" },
    { AREA_SWAMP_OF_SORROWS, "Swamp Of Sorrows" },
    { AREA_NORTHSHIRE_VALLEY, "Northshire Valley" },
    { AREA_DUSKWOOD, "Duskwood" },
    { AREA_WETLANDS, "Wetlands" },
    { AREA_ELWYNN_FOREST, "Elwynn Forest" },
    { AREA_THE_WORLD_TREE, "The World Tree" },
    { AREA_DUROTAR, "Durotar" },
    { AREA_DUSTWALLOW_MARSH, "Dustwallow Marsh" },
    { AREA_AZSHARA, "Azshara" },
    { AREA_THE_BARRENS, "The Barrens" },
    { AREA_CRYSTAL_LAKE, "Crystal Lake" },
    { AREA_ZUL_GURUB0, "Zul Gurub0" },
    { AREA_MOONBROOK, "Moonbrook" },
    { AREA_KUL_TIRAS, "Kul Tiras" },
    { AREA_PROGRAMMER_ISLE, "Programmer Isle" },
    { AREA_NORTHSHIRE_RIVER, "Northshire River" },
    { AREA_NORTHSHIRE_ABBEY, "Northshire Abbey" },
    { AREA_BLACKROCK_MOUNTAIN0, "Blackrock Mountain0" },
    { AREA_LIGHTHOUSE, "Lighthouse" },
    { AREA_WESTERN_PLAGUELANDS, "Western Plaguelands" },
    { AREA_NINE, "Nine" },
    { AREA_THE_CEMETARY, "The Cemetary" },
    { AREA_STRANGLETHORN_VALE, "Stranglethorn Vale" },
    { AREA_ECHO_RIDGE_MINE, "Echo Ridge Mine" },
    { AREA_BOOTY_BAY, "Booty Bay" },
    { AREA_ALTERAC_MOUNTAINS, "Alterac Mountains" },
    { AREA_LAKE_NAZFERITI, "Lake Nazferiti" },
    { AREA_LOCH_MODAN, "Loch Modan" },
    { AREA_WESTFALL0, "Westfall0" },
    { AREA_DEADWIND_PASS, "Deadwind Pass" },
    { AREA_DARKSHIRE, "Darkshire" },
    { AREA_WILD_SHORE, "Wild Shore" },
    { AREA_REDRIDGE_MOUNTAINS, "Redridge Mountains" },
    { AREA_ARATHI_HIGHLANDS, "Arathi Highlands" },
    { AREA_BURNING_STEPPES, "Burning Steppes" },
    { AREA_THE_HINTERLANDS, "The Hinterlands" },
    { AREA_DEAD_MANS_HOLE, "Dead Mans Hole" },
    { AREA_SEARING_GORGE, "Searing Gorge" },
    { AREA_THIEVES_CAMP, "Thieves Camp" },
    { AREA_JASPERLODE_MINE, "Jasperlode Mine" },
    { AREA_VALLEY_OF_HEROES_UNUSED, "Valley Of Heroes Unused" },
    { AREA_HEROES_VIGIL, "Heroes Vigil" },
    { AREA_FARGODEEP_MINE, "Fargodeep Mine" },
    { AREA_NORTHSHIRE_VINEYARDS, "Northshire Vineyards" },
    { AREA_FORESTS_EDGE, "Forests Edge" },
    { AREA_THUNDER_FALLS, "Thunder Falls" },
    { AREA_BRACKWELL_PUMPKIN_PATCH, "Brackwell Pumpkin Patch" },
    { AREA_THE_STONEFIELD_FARM, "The Stonefield Farm" },
    { AREA_THE_MACLURE_VINEYARDS, "The Maclure Vineyards" },
    { AREA_ON_MAP_DUNGEON0, "On Map Dungeon0" },
    { AREA_ON_MAP_DUNGEON1, "On Map Dungeon1" },
    { AREA_ON_MAP_DUNGEON2, "On Map Dungeon2" },
    { AREA_LAKE_EVERSTILL, "Lake Everstill" },
    { AREA_LAKESHIRE, "Lakeshire" },
    { AREA_STONEWATCH, "Stonewatch" },
    { AREA_STONEWATCH_FALLS, "Stonewatch Falls" },
    { AREA_THE_DARK_PORTAL, "The Dark Portal" },
    { AREA_THE_TAINTED_SCAR, "The Tainted Scar" },
    { AREA_POOL_OF_TEARS, "Pool Of Tears" },
    { AREA_STONARD, "Stonard" },
    { AREA_FALLOW_SANCTUARY, "Fallow Sanctuary" },
    { AREA_ANVILMAR, "Anvilmar" },
    { AREA_STORMWIND_MOUNTAINS, "Stormwind Mountains" },
    { AREA_JEFF_NE_QUADRANT_CHANGED, "Jeff Ne Quadrant Changed" },
    { AREA_JEFF_NW_QUADRANT, "Jeff Nw Quadrant" },
    { AREA_JEFF_SE_QUADRANT, "Jeff Se Quadrant" },
    { AREA_JEFF_SW_QUADRANT, "Jeff Sw Quadrant" },
    { AREA_TIRISFAL_GLADES, "Tirisfal Glades" },
    { AREA_STONE_CAIRN_LAKE, "Stone Cairn Lake" },
    { AREA_GOLDSHIRE, "Goldshire" },
    { AREA_EASTVALE_LOGGING_CAMP, "Eastvale Logging Camp" },
    { AREA_MIRROR_LAKE_ORCHARD, "Mirror Lake Orchard" },
    { AREA_TOWER_OF_AZORA, "Tower Of Azora" },
    { AREA_MIRROR_LAKE, "Mirror Lake" },
    { AREA_VUL_GOL_OGRE_MOUND, "Vul Gol Ogre Mound" },
    { AREA_RAVEN_HILL, "Raven Hill" },
    { AREA_REDRIDGE_CANYONS, "Redridge Canyons" },
    { AREA_TOWER_OF_ILGALAR, "Tower Of Ilgalar" },
    { AREA_ALTHERS_MILL, "Althers Mill" },
    { AREA_RETHBAN_CAVERNS, "Rethban Caverns" },
    { AREA_REBEL_CAMP, "Rebel Camp" },
    { AREA_NESINGWARYS_EXPEDITION, "Nesingwarys Expedition" },
    { AREA_KURZENS_COMPOUND, "Kurzens Compound" },
    { AREA_RUINS_OF_ZUL_KUNDA, "Ruins Of Zul Kunda" },
    { AREA_RUINS_OF_ZUL_MAMWE, "Ruins Of Zul Mamwe" },
    { AREA_THE_VILE_REEF, "The Vile Reef" },
    { AREA_MOSH_OGG_OGRE_MOUND, "Mosh Ogg Ogre Mound" },
    { AREA_THE_STOCKPILE, "The Stockpile" },
    { AREA_SALDEANS_FARM, "Saldeans Farm" },
    { AREA_SENTINEL_HILL, "Sentinel Hill" },
    { AREA_FURLBROWS_PUMPKIN_FARM, "Furlbrows Pumpkin Farm" },
    { AREA_JANGOLODE_MINE, "Jangolode Mine" },
    { AREA_GOLD_COAST_QUARRY, "Gold Coast Quarry" },
    { AREA_WESTFALL_LIGHTHOUSE, "Westfall Lighthouse" },
    { AREA_MISTY_VALLEY, "Misty Valley" },
    { AREA_GROM_GOL_BASE_CAMP, "Grom Gol Base Camp" },
    { AREA_WHELGARS_EXCAVATION_SITE, "Whelgars Excavation Site" },
    { AREA_WESTBROOK_GARRISON, "Westbrook Garrison" },
    { AREA_TRANQUIL_GARDENS_CEMETERY, "Tranquil Gardens Cemetery" },
    { AREA_ZUULDAIA_RUINS, "Zuuldaia Ruins" },
    { AREA_BAL_LAL_RUINS, "Bal Lal Ruins" },
    { AREA_KAL_AI_RUINS, "Kal Ai Ruins" },
    { AREA_TKASHI_RUINS, "Tkashi Ruins" },
    { AREA_BALIA_MAH_RUINS, "Balia Mah Ruins" },
    { AREA_ZIATAJAI_RUINS, "Ziatajai Ruins" },
    { AREA_MIZJAH_RUINS, "Mizjah Ruins" },
    { AREA_SILVERPINE_FOREST, "Silverpine Forest" },
    { AREA_KHARANOS, "Kharanos" },
    { AREA_COLDRIDGE_VALLEY, "Coldridge Valley" },
    { AREA_GNOMEREGAN0, "Gnomeregan0" },
    { AREA_GOL_BOLAR_QUARRY, "Gol Bolar Quarry" },
    { AREA_FROSTMANE_HOLD, "Frostmane Hold" },
    { AREA_THE_GRIZZLED_DEN, "The Grizzled Den" },
    { AREA_BREWNALL_VILLAGE, "Brewnall Village" },
    { AREA_MISTY_PINE_REFUGE, "Misty Pine Refuge" },
    { AREA_EASTERN_PLAGUELANDS, "Eastern Plaguelands" },
    { AREA_TELDRASSIL, "Teldrassil" },
    { AREA_IRONBANDS_EXCAVATION_SITE, "Ironbands Excavation Site" },
    { AREA_MO_GROSH_STRONGHOLD, "Mo Grosh Stronghold" },
    { AREA_THELSAMAR, "Thelsamar" },
    { AREA_ALGAZ_GATE, "Algaz Gate" },
    { AREA_STONEWROUGHT_DAM, "Stonewrought Dam" },
    { AREA_THE_FARSTRIDER_LODGE, "The Farstrider Lodge" },
    { AREA_DARKSHORE, "Darkshore" },
    { AREA_SILVER_STREAM_MINE, "Silver Stream Mine" },
    { AREA_MENETHIL_HARBOR, "Menethil Harbor" },
    { AREA_DESIGNER_ISLAND, "Designer Island" },
    { AREA_THE_BULWARK0, "The Bulwark0" },
    { AREA_RUINS_OF_LORDAERON, "Ruins Of Lordaeron" },
    { AREA_DEATHKNELL, "Deathknell" },
    { AREA_NIGHT_WEBS_HOLLOW, "Night Webs Hollow" },
    { AREA_SOLLIDEN_FARMSTEAD, "Solliden Farmstead" },
    { AREA_AGAMAND_MILLS, "Agamand Mills" },
    { AREA_AGAMAND_FAMILY_CRYPT, "Agamand Family Crypt" },
    { AREA_BRILL, "Brill" },
    { AREA_WHISPERING_GARDENS, "Whispering Gardens" },
    { AREA_TERRACE_OF_REPOSE, "Terrace Of Repose" },
    { AREA_BRIGHTWATER_LAKE, "Brightwater Lake" },
    { AREA_GUNTHERS_RETREAT, "Gunthers Retreat" },
    { AREA_GARRENS_HAUNT, "Garrens Haunt" },
    { AREA_BALNIR_FARMSTEAD, "Balnir Farmstead" },
    { AREA_COLD_HEARTH_MANOR, "Cold Hearth Manor" },
    { AREA_CRUSADER_OUTPOST, "Crusader Outpost" },
    { AREA_THE_NORTH_COAST, "The North Coast" },
    { AREA_WHISPERING_SHORE, "Whispering Shore" },
    { AREA_LORDAMERE_LAKE0, "Lordamere Lake0" },
    { AREA_FENRIS_ISLE, "Fenris Isle" },
    { AREA_FAOLS_REST, "Faols Rest" },
    { AREA_DOLANAAR, "Dolanaar" },
    { AREA_DARNASSUS_UNUSED, "Darnassus Unused" },
    { AREA_SHADOWGLEN, "Shadowglen" },
    { AREA_STEELGRILLS_DEPOT, "Steelgrills Depot" },
    { AREA_HEARTHGLEN, "Hearthglen" },
    { AREA_NORTHRIDGE_LUMBER_CAMP, "Northridge Lumber Camp" },
    { AREA_RUINS_OF_ANDORHAL, "Ruins Of Andorhal" },
    { AREA_SCHOOL_OF_NECROMANCY, "School Of Necromancy" },
    { AREA_UTHERS_TOMB, "Uthers Tomb" },
    { AREA_SORROW_HILL, "Sorrow Hill" },
    { AREA_THE_WEEPING_CAVE, "The Weeping Cave" },
    { AREA_FELSTONE_FIELD, "Felstone Field" },
    { AREA_DALSONS_TEARS, "Dalsons Tears" },
    { AREA_GAHRRONS_WITHERING, "Gahrrons Withering" },
    { AREA_THE_WRITHING_HAUNT, "The Writhing Haunt" },
    { AREA_MARDENHOLDE_KEEP, "Mardenholde Keep" },
    { AREA_PYREWOOD_VILLAGE, "Pyrewood Village" },
    { AREA_DUN_MODR, "Dun Modr" },
    { AREA_WESTFALL1, "Westfall1" },
    { AREA_THE_GREAT_SEA0, "The Great Sea0" },
    { AREA_UNUSED_IRONCLADCOVE, "Unused Ironcladcove" },
    { AREA_SHADOWFANG_KEEP0, "Shadowfang Keep0" },
    { AREA_ON_MAP_DUNGEON3, "On Map Dungeon3" },
    { AREA_ICEFLOW_LAKE, "Iceflow Lake" },
    { AREA_HELMS_BED_LAKE, "Helms Bed Lake" },
    { AREA_DEEP_ELEM_MINE, "Deep Elem Mine" },
    { AREA_THE_GREAT_SEA1, "The Great Sea1" },
    { AREA_MULGORE, "Mulgore" },
    { AREA_ALEXSTON_FARMSTEAD, "Alexston Farmstead" },
    { AREA_RED_CLOUD_MESA, "Red Cloud Mesa" },
    { AREA_CAMP_NARACHE, "Camp Narache" },
    { AREA_BLOODHOOF_VILLAGE, "Bloodhoof Village" },
    { AREA_STONEBULL_LAKE, "Stonebull Lake" },
    { AREA_RAVAGED_CARAVAN, "Ravaged Caravan" },
    { AREA_RED_ROCKS, "Red Rocks" },
    { AREA_THE_SKITTERING_DARK, "The Skittering Dark" },
    { AREA_VALGANS_FIELD, "Valgans Field" },
    { AREA_THE_SEPULCHER, "The Sepulcher" },
    { AREA_OLSENS_FARTHING, "Olsens Farthing" },
    { AREA_THE_GREYMANE_WALL, "The Greymane Wall" },
    { AREA_BERENS_PERIL, "Berens Peril" },
    { AREA_THE_DAWNING_ISLES, "The Dawning Isles" },
    { AREA_AMBERMILL, "Ambermill" },
    { AREA_FENRIS_KEEP, "Fenris Keep" },
    { AREA_SHADOWFANG_KEEP1, "Shadowfang Keep1" },
    { AREA_THE_DECREPIT_FERRY, "The Decrepit Ferry" },
    { AREA_MALDENS_ORCHARD, "Maldens Orchard" },
    { AREA_THE_IVAR_PATCH, "The Ivar Patch" },
    { AREA_THE_DEAD_FIELD, "The Dead Field" },
    { AREA_THE_ROTTING_ORCHARD, "The Rotting Orchard" },
    { AREA_BRIGHTWOOD_GROVE, "Brightwood Grove" },
    { AREA_FORLORN_ROWE, "Forlorn Rowe" },
    { AREA_THE_WHIPPLE_ESTATE, "The Whipple Estate" },
    { AREA_THE_YORGEN_FARMSTEAD, "The Yorgen Farmstead" },
    { AREA_THE_CAULDRON, "The Cauldron" },
    { AREA_GRIMESILT_DIG_SITE, "Grimesilt Dig Site" },
    { AREA_DREADMAUL_ROCK, "Dreadmaul Rock" },
    { AREA_RUINS_OF_THAURISSAN, "Ruins Of Thaurissan" },
    { AREA_FLAME_CREST, "Flame Crest" },
    { AREA_BLACKROCK_STRONGHOLD, "Blackrock Stronghold" },
    { AREA_THE_PILLAR_OF_ASH, "The Pillar Of Ash" },
    { AREA_BLACKROCK_MOUNTAIN1, "Blackrock Mountain1" },
    { AREA_ALTAR_OF_STORMS0, "Altar Of Storms0" },
    { AREA_ALDRASSIL, "Aldrassil" },
    { AREA_SHADOWTHREAD_CAVE, "Shadowthread Cave" },
    { AREA_FEL_ROCK, "Fel Rock" },
    { AREA_LAKE_AL_AMETH, "Lake Al Ameth" },
    { AREA_STARBREEZE_VILLAGE, "Starbreeze Village" },
    { AREA_GNARLPINE_HOLD, "Gnarlpine Hold" },
    { AREA_BAN_ETHIL_BARROW_DEN, "Ban Ethil Barrow Den" },
    { AREA_THE_CLEFT, "The Cleft" },
    { AREA_THE_ORACLE_GLADE, "The Oracle Glade" },
    { AREA_WELLSPRING_RIVER, "Wellspring River" },
    { AREA_WELLSPRING_LAKE, "Wellspring Lake" },
    { AREA_HILLSBRAD_FOOTHILLS, "Hillsbrad Foothills" },
    { AREA_AZSHARA_CRATER, "Azshara Crater" },
    { AREA_DUN_ALGAZ0, "Dun Algaz0" },
    { AREA_SOUTHSHORE0, "Southshore0" },
    { AREA_TARREN_MILL0, "Tarren Mill0" },
    { AREA_DURNHOLDE_KEEP0, "Durnholde Keep0" },
    { AREA_UNUSED_STONEWROUGHT_PASS, "Unused Stonewrought Pass" },
    { AREA_THE_FOOTHILL_CAVERNS, "The Foothill Caverns" },
    { AREA_LORDAMERE_INTERNMENT_CAMP, "Lordamere Internment Camp" },
    { AREA_DALARAN, "Dalaran" },
    { AREA_STRAHNBRAD, "Strahnbrad" },
    { AREA_RUINS_OF_ALTERAC, "Ruins Of Alterac" },
    { AREA_CRUSHRIDGE_HOLD, "Crushridge Hold" },
    { AREA_SLAUGHTER_HOLLOW, "Slaughter Hollow" },
    { AREA_THE_UPLANDS, "The Uplands" },
    { AREA_SOUTHPOINT_TOWER0, "Southpoint Tower0" },
    { AREA_HILLSBRAD_FIELDS0, "Hillsbrad Fields0" },
    { AREA_HILLSBRAD, "Hillsbrad" },
    { AREA_AZURELODE_MINE0, "Azurelode Mine0" },
    { AREA_NETHANDER_STEAD0, "Nethander Stead0" },
    { AREA_DUN_GAROK0, "Dun Garok0" },
    { AREA_THORADINS_WALL0, "Thoradins Wall0" },
    { AREA_EASTERN_STRAND0, "Eastern Strand0" },
    { AREA_WESTERN_STRAND0, "Western Strand0" },
    { AREA_SOUTH_SEAS_UNUSED, "South Seas Unused" },
    { AREA_JAGUERO_ISLE, "Jaguero Isle" },
    { AREA_BARADIN_BAY, "Baradin Bay" },
    { AREA_MENETHIL_BAY, "Menethil Bay" },
    { AREA_MISTY_REED_STRAND, "Misty Reed Strand" },
    { AREA_THE_SAVAGE_COAST, "The Savage Coast" },
    { AREA_THE_CRYSTAL_SHORE, "The Crystal Shore" },
    { AREA_SHELL_BEACH, "Shell Beach" },
    { AREA_NORTH_TIDES_RUN, "North Tides Run" },
    { AREA_SOUTH_TIDES_RUN, "South Tides Run" },
    { AREA_THE_OVERLOOK_CLIFFS, "The Overlook Cliffs" },
    { AREA_THE_FORBIDDING_SEA0, "The Forbidding Sea0" },
    { AREA_IRONBEARDS_TOMB, "Ironbeards Tomb" },
    { AREA_CRYSTALVEIN_MINE, "Crystalvein Mine" },
    { AREA_RUINS_OF_ABORAZ, "Ruins Of Aboraz" },
    { AREA_JANEIROS_POINT, "Janeiros Point" },
    { AREA_NORTHFOLD_MANOR, "Northfold Manor" },
    { AREA_GO_SHEK_FARM, "Go Shek Farm" },
    { AREA_DABYRIES_FARMSTEAD, "Dabyries Farmstead" },
    { AREA_BOULDERFIST_HALL, "Boulderfist Hall" },
    { AREA_WITHERBARK_VILLAGE, "Witherbark Village" },
    { AREA_DRYWHISKER_GORGE, "Drywhisker Gorge" },
    { AREA_REFUGE_POINTE, "Refuge Pointe" },
    { AREA_HAMMERFALL, "Hammerfall" },
    { AREA_BLACKWATER_SHIPWRECKS, "Blackwater Shipwrecks" },
    { AREA_O_BREENS_CAMP, "O Breens Camp" },
    { AREA_STROMGARDE_KEEP, "Stromgarde Keep" },
    { AREA_THE_TOWER_OF_ARATHOR, "The Tower Of Arathor" },
    { AREA_THE_SANCTUM, "The Sanctum" },
    { AREA_FALDIRS_COVE, "Faldirs Cove" },
    { AREA_THE_DROWNED_REEF, "The Drowned Reef" },
    { AREA_THANDOL_SPAN0, "Thandol Span0" },
    { AREA_ASHENVALE, "Ashenvale" },
    { AREA_THE_GREAT_SEA2, "The Great Sea2" },
    { AREA_CIRCLE_OF_EAST_BINDING, "Circle Of East Binding" },
    { AREA_CIRCLE_OF_WEST_BINDING, "Circle Of West Binding" },
    { AREA_CIRCLE_OF_INNER_BINDING, "Circle Of Inner Binding" },
    { AREA_CIRCLE_OF_OUTER_BINDING, "Circle Of Outer Binding" },
    { AREA_APOCRYPHANS_REST, "Apocryphans Rest" },
    { AREA_ANGOR_FORTRESS, "Angor Fortress" },
    { AREA_LETHLOR_RAVINE, "Lethlor Ravine" },
    { AREA_KARGATH, "Kargath" },
    { AREA_CAMP_KOSH, "Camp Kosh" },
    { AREA_CAMP_BOFF, "Camp Boff" },
    { AREA_CAMP_WURG, "Camp Wurg" },
    { AREA_CAMP_CAGG, "Camp Cagg" },
    { AREA_AGMONDS_END, "Agmonds End" },
    { AREA_HAMMERTOES_DIGSITE, "Hammertoes Digsite" },
    { AREA_DUSTBELCH_GROTTO, "Dustbelch Grotto" },
    { AREA_AERIE_PEAK, "Aerie Peak" },
    { AREA_WILDHAMMER_KEEP, "Wildhammer Keep" },
    { AREA_QUEL_DANIL_LODGE, "Quel Danil Lodge" },
    { AREA_SKULK_ROCK, "Skulk Rock" },
    { AREA_ZUN_WATHA, "Zun Watha" },
    { AREA_SHADRA_ALOR, "Shadra Alor" },
    { AREA_JINTHA_ALOR, "Jintha Alor" },
    { AREA_THE_ALTAR_OF_ZUL, "The Altar Of Zul" },
    { AREA_SERADANE, "Seradane" },
    { AREA_FERALAS, "Feralas" },
    { AREA_BRAMBLEBLADE_RAVINE, "Brambleblade Ravine" },
    { AREA_BAEL_MODAN, "Bael Modan" },
    { AREA_THE_VENTURE_CO_MINE, "The Venture Co Mine" },
    { AREA_FELWOOD, "Felwood" },
    { AREA_RAZOR_HILL, "Razor Hill" },
    { AREA_VALLEY_OF_TRIALS, "Valley Of Trials" },
    { AREA_THE_DEN, "The Den" },
    { AREA_BURNING_BLADE_COVEN, "Burning Blade Coven" },
    { AREA_KOLKAR_CRAG, "Kolkar Crag" },
    { AREA_SEN_JIN_VILLAGE, "Sen Jin Village" },
    { AREA_ECHO_ISLES, "Echo Isles" },
    { AREA_THUNDER_RIDGE, "Thunder Ridge" },
    { AREA_DRYGULCH_RAVINE, "Drygulch Ravine" },
    { AREA_DUSTWIND_CAVE, "Dustwind Cave" },
    { AREA_TIRAGARDE_KEEP, "Tiragarde Keep" },
    { AREA_SCUTTLE_COAST, "Scuttle Coast" },
    { AREA_BLADEFIST_BAY, "Bladefist Bay" },
    { AREA_DEADEYE_SHORE, "Deadeye Shore" },
    { AREA_SOUTHFURY_RIVER0, "Southfury River0" },
    { AREA_CAMP_TAURAJO, "Camp Taurajo" },
    { AREA_FAR_WATCH_POST, "Far Watch Post" },
    { AREA_THE_CROSSROADS, "The Crossroads" },
    { AREA_BOULDER_LODE_MINE, "Boulder Lode Mine" },
    { AREA_THE_SLUDGE_FEN, "The Sludge Fen" },
    { AREA_THE_DRY_HILLS, "The Dry Hills" },
    { AREA_DREADMIST_PEAK, "Dreadmist Peak" },
    { AREA_NORTHWATCH_HOLD, "Northwatch Hold" },
    { AREA_THE_FORGOTTEN_POOLS, "The Forgotten Pools" },
    { AREA_LUSHWATER_OASIS, "Lushwater Oasis" },
    { AREA_THE_STAGNANT_OASIS, "The Stagnant Oasis" },
    { AREA_FIELD_OF_GIANTS, "Field Of Giants" },
    { AREA_THE_MERCHANT_COAST, "The Merchant Coast" },
    { AREA_RATCHET, "Ratchet" },
    { AREA_DARKSPEAR_STRAND, "Darkspear Strand" },
    { AREA_DARROWMERE_LAKE_UNUSED, "Darrowmere Lake Unused" },
    { AREA_CAER_DARROW_UNUSED, "Caer Darrow Unused" },
    { AREA_WINTERHOOF_WATER_WELL, "Winterhoof Water Well" },
    { AREA_THUNDERHORN_WATER_WELL, "Thunderhorn Water Well" },
    { AREA_WILDMANE_WATER_WELL, "Wildmane Water Well" },
    { AREA_SKYLINE_RIDGE, "Skyline Ridge" },
    { AREA_THOUSAND_NEEDLES, "Thousand Needles" },
    { AREA_THE_TIDUS_STAIR, "The Tidus Stair" },
    { AREA_SHADY_REST_INN, "Shady Rest Inn" },
    { AREA_BAEL_DUN_DIGSITE, "Bael Dun Digsite" },
    { AREA_DESOLACE, "Desolace" },
    { AREA_STONETALON_MOUNTAINS, "Stonetalon Mountains" },
    { AREA_ORGRIMMAR_UNUSED, "Orgrimmar Unused" },
    { AREA_GILLIJIMS_ISLE, "Gillijims Isle" },
    { AREA_ISLAND_OF_DOCTOR_LAPIDIS, "Island Of Doctor Lapidis" },
    { AREA_RAZORWIND_CANYON, "Razorwind Canyon" },
    { AREA_BATHRANS_HAUNT, "Bathrans Haunt" },
    { AREA_THE_RUINS_OF_ORDIL_ARAN, "The Ruins Of Ordil Aran" },
    { AREA_MAESTRAS_POST, "Maestras Post" },
    { AREA_THE_ZORAM_STRAND, "The Zoram Strand" },
    { AREA_ASTRANAAR, "Astranaar" },
    { AREA_THE_SHRINE_OF_AESSINA, "The Shrine Of Aessina" },
    { AREA_FIRE_SCAR_SHRINE, "Fire Scar Shrine" },
    { AREA_THE_RUINS_OF_STARDUST, "The Ruins Of Stardust" },
    { AREA_THE_HOWLING_VALE, "The Howling Vale" },
    { AREA_SILVERWIND_REFUGE, "Silverwind Refuge" },
    { AREA_MYSTRAL_LAKE, "Mystral Lake" },
    { AREA_FALLEN_SKY_LAKE, "Fallen Sky Lake" },
    { AREA_IRIS_LAKE, "Iris Lake" },
    { AREA_MOONWELL, "Moonwell" },
    { AREA_RAYNEWOOD_RETREAT, "Raynewood Retreat" },
    { AREA_THE_SHADY_NOOK, "The Shady Nook" },
    { AREA_NIGHT_RUN, "Night Run" },
    { AREA_XAVIAN, "Xavian" },
    { AREA_SATYRNAAR, "Satyrnaar" },
    { AREA_SPLINTERTREE_POST, "Splintertree Post" },
    { AREA_THE_DOR_DANIL_BARROW_DEN, "The Dor Danil Barrow Den" },
    { AREA_FALFARREN_RIVER, "Falfarren River" },
    { AREA_FELFIRE_HILL, "Felfire Hill" },
    { AREA_DEMON_FALL_CANYON, "Demon Fall Canyon" },
    { AREA_DEMON_FALL_RIDGE, "Demon Fall Ridge" },
    { AREA_WARSONG_LUMBER_CAMP, "Warsong Lumber Camp" },
    { AREA_BOUGH_SHADOW, "Bough Shadow" },
    { AREA_THE_SHIMMERING_FLATS, "The Shimmering Flats" },
    { AREA_TANARIS, "Tanaris" },
    { AREA_LAKE_FALATHIM, "Lake Falathim" },
    { AREA_AUBERDINE, "Auberdine" },
    { AREA_RUINS_OF_MATHYSTRA, "Ruins Of Mathystra" },
    { AREA_TOWER_OF_ALTHALAXX, "Tower Of Althalaxx" },
    { AREA_CLIFFSPRING_FALLS, "Cliffspring Falls" },
    { AREA_BASHAL_ARAN, "Bashal Aran" },
    { AREA_AMETH_ARAN, "Ameth Aran" },
    { AREA_GROVE_OF_THE_ANCIENTS, "Grove Of The Ancients" },
    { AREA_THE_MASTERS_GLAIVE, "The Masters Glaive" },
    { AREA_REMTRAVELS_EXCAVATION, "Remtravels Excavation" },
    { AREA_MISTS_EDGE, "Mists Edge" },
    { AREA_THE_LONG_WASH, "The Long Wash" },
    { AREA_WILDBEND_RIVER, "Wildbend River" },
    { AREA_BLACKWOOD_DEN, "Blackwood Den" },
    { AREA_CLIFFSPRING_RIVER, "Cliffspring River" },
    { AREA_THE_VEILED_SEA0, "The Veiled Sea0" },
    { AREA_GOLD_ROAD, "Gold Road" },
    { AREA_SCARLET_WATCH_POST, "Scarlet Watch Post" },
    { AREA_SUN_ROCK_RETREAT, "Sun Rock Retreat" },
    { AREA_WINDSHEAR_CRAG, "Windshear Crag" },
    { AREA_CRAGPOOL_LAKE, "Cragpool Lake" },
    { AREA_MIRKFALLON_LAKE, "Mirkfallon Lake" },
    { AREA_THE_CHARRED_VALE, "The Charred Vale" },
    { AREA_VALLEY_OF_THE_BLOODFURIES, "Valley Of The Bloodfuries" },
    { AREA_STONETALON_PEAK, "Stonetalon Peak" },
    { AREA_THE_TALON_DEN, "The Talon Den" },
    { AREA_GREATWOOD_VALE, "Greatwood Vale" },
    { AREA_THUNDER_BLUFF_UNUSED, "Thunder Bluff Unused" },
    { AREA_BRAVE_WIND_MESA, "Brave Wind Mesa" },
    { AREA_FIRE_STONE_MESA, "Fire Stone Mesa" },
    { AREA_MANTLE_ROCK, "Mantle Rock" },
    { AREA_HUNTER_RISE_UNUSED, "Hunter Rise Unused" },
    { AREA_SPIRIT_RISE_UNUSED, "Spirit Rise Unused" },
    { AREA_ELDER_RISE_UNUSED, "Elder Rise Unused" },
    { AREA_RUINS_OF_JUBUWAL, "Ruins Of Jubuwal" },
    { AREA_POOLS_OF_ARLITHRIEN, "Pools Of Arlithrien" },
    { AREA_THE_RUSTMAUL_DIG_SITE, "The Rustmaul Dig Site" },
    { AREA_CAMP_E_THOK, "Camp E Thok" },
    { AREA_SPLITHOOF_CRAG, "Splithoof Crag" },
    { AREA_HIGHPERCH, "Highperch" },
    { AREA_THE_SCREECHING_CANYON, "The Screeching Canyon" },
    { AREA_FREEWIND_POST, "Freewind Post" },
    { AREA_THE_GREAT_LIFT0, "The Great Lift0" },
    { AREA_GALAK_HOLD, "Galak Hold" },
    { AREA_ROGUEFEATHER_DEN, "Roguefeather Den" },
    { AREA_THE_WEATHERED_NOOK, "The Weathered Nook" },
    { AREA_THALANAAR, "Thalanaar" },
    { AREA_UN_GORO_CRATER, "Un Goro Crater" },
    { AREA_RAZORFEN_KRAUL0, "Razorfen Kraul0" },
    { AREA_RAVEN_HILL_CEMETERY, "Raven Hill Cemetery" },
    { AREA_MOONGLADE, "Moonglade" },
    { AREA_DELETE_ME0, "Delete Me0" },
    { AREA_BRACKENWALL_VILLAGE, "Brackenwall Village" },
    { AREA_SWAMPLIGHT_MANOR, "Swamplight Manor" },
    { AREA_BLOODFEN_BURROW, "Bloodfen Burrow" },
    { AREA_DARKMIST_CAVERN, "Darkmist Cavern" },
    { AREA_MOGGLE_POINT, "Moggle Point" },
    { AREA_BEEZILS_WRECK, "Beezils Wreck" },
    { AREA_WITCH_HILL, "Witch Hill" },
    { AREA_SENTRY_POINT, "Sentry Point" },
    { AREA_NORTH_POINT_TOWER, "North Point Tower" },
    { AREA_WEST_POINT_TOWER, "West Point Tower" },
    { AREA_LOST_POINT, "Lost Point" },
    { AREA_BLUEFEN, "Bluefen" },
    { AREA_STONEMAUL_RUINS, "Stonemaul Ruins" },
    { AREA_THE_DEN_OF_FLAME, "The Den Of Flame" },
    { AREA_THE_DRAGONMURK, "The Dragonmurk" },
    { AREA_WYRMBOG, "Wyrmbog" },
    { AREA_ONYXIAS_LAIR_UNUSED, "Onyxias Lair Unused" },
    { AREA_THERAMORE_ISLE, "Theramore Isle" },
    { AREA_FOOTHOLD_CITADEL, "Foothold Citadel" },
    { AREA_IRONCLAD_PRISON, "Ironclad Prison" },
    { AREA_DUSTWALLOW_BAY, "Dustwallow Bay" },
    { AREA_TIDEFURY_COVE, "Tidefury Cove" },
    { AREA_DREADMURK_SHORE, "Dreadmurk Shore" },
    { AREA_ADDLES_STEAD, "Addles Stead" },
    { AREA_FIRE_PLUME_RIDGE, "Fire Plume Ridge" },
    { AREA_LAKKARI_TAR_PITS, "Lakkari Tar Pits" },
    { AREA_TERROR_RUN, "Terror Run" },
    { AREA_THE_SLITHERING_SCAR, "The Slithering Scar" },
    { AREA_MARSHALS_REFUGE, "Marshals Refuge" },
    { AREA_FUNGAL_ROCK, "Fungal Rock" },
    { AREA_GOLAKKA_HOT_SPRINGS, "Golakka Hot Springs" },
    { AREA_THE_LOCH, "The Loch" },
    { AREA_BEGGARS_HAUNT, "Beggars Haunt" },
    { AREA_KODO_GRAVEYARD, "Kodo Graveyard" },
    { AREA_GHOST_WALKER_POST, "Ghost Walker Post" },
    { AREA_SAR_THERIS_STRAND, "Sar Theris Strand" },
    { AREA_THUNDER_AXE_FORTRESS, "Thunder Axe Fortress" },
    { AREA_BOLGANS_HOLE, "Bolgans Hole" },
    { AREA_MANNOROC_COVEN, "Mannoroc Coven" },
    { AREA_SARGERON, "Sargeron" },
    { AREA_MAGRAM_VILLAGE, "Magram Village" },
    { AREA_GELKIS_VILLAGE, "Gelkis Village" },
    { AREA_VALLEY_OF_SPEARS, "Valley Of Spears" },
    { AREA_NIJELS_POINT, "Nijels Point" },
    { AREA_KOLKAR_VILLAGE, "Kolkar Village" },
    { AREA_HYJAL, "Hyjal" },
    { AREA_WINTERSPRING, "Winterspring" },
    { AREA_BLACKWOLF_RIVER, "Blackwolf River" },
    { AREA_KODO_ROCK, "Kodo Rock" },
    { AREA_HIDDEN_PATH, "Hidden Path" },
    { AREA_SPIRIT_ROCK, "Spirit Rock" },
    { AREA_SHRINE_OF_THE_DORMANT_FLAME, "Shrine Of The Dormant Flame" },
    { AREA_LAKE_ELUNE_ARA, "Lake Elune Ara" },
    { AREA_THE_HARBORAGE, "The Harborage" },
    { AREA_OUTLAND, "Outland" },
    { AREA_CRAFTSMENS_TERRACE_UNUSED, "Craftsmens Terrace Unused" },
    { AREA_TRADESMENS_TERRACE_UNUSED, "Tradesmens Terrace Unused" },
    { AREA_THE_TEMPLE_GARDENS_UNUSED, "The Temple Gardens Unused" },
    { AREA_TEMPLE_OF_ELUNE_UNUSED, "Temple Of Elune Unused" },
    { AREA_CENARION_ENCLAVE_UNUSED, "Cenarion Enclave Unused" },
    { AREA_WARRIORS_TERRACE_UNUSED, "Warriors Terrace Unused" },
    { AREA_RUT_THERAN_VILLAGE, "Rut Theran Village" },
    { AREA_IRONBANDS_COMPOUND, "Ironbands Compound" },
    { AREA_THE_STOCKADE, "The Stockade" },
    { AREA_WAILING_CAVERNS, "Wailing Caverns" },
    { AREA_BLACKFATHOM_DEEPS0, "Blackfathom Deeps0" },
    { AREA_FRAY_ISLAND, "Fray Island" },
    { AREA_GNOMEREGAN1, "Gnomeregan1" },
    { AREA_RAZORFEN_DOWNS0, "Razorfen Downs0" },
    { AREA_BAN_ETHIL_HOLLOW, "Ban Ethil Hollow" },
    { AREA_SCARLET_MONASTERY, "Scarlet Monastery" },
    { AREA_JERODS_LANDING, "Jerods Landing" },
    { AREA_RIDGEPOINT_TOWER, "Ridgepoint Tower" },
    { AREA_THE_DARKENED_BANK, "The Darkened Bank" },
    { AREA_COLDRIDGE_PASS, "Coldridge Pass" },
    { AREA_CHILL_BREEZE_VALLEY, "Chill Breeze Valley" },
    { AREA_SHIMMER_RIDGE, "Shimmer Ridge" },
    { AREA_AMBERSTILL_RANCH, "Amberstill Ranch" },
    { AREA_THE_TUNDRID_HILLS, "The Tundrid Hills" },
    { AREA_SOUTH_GATE_PASS0, "South Gate Pass0" },
    { AREA_SOUTH_GATE_OUTPOST, "South Gate Outpost" },
    { AREA_NORTH_GATE_PASS0, "North Gate Pass0" },
    { AREA_NORTH_GATE_OUTPOST, "North Gate Outpost" },
    { AREA_GATES_OF_IRONFORGE, "Gates Of Ironforge" },
    { AREA_STILLWATER_POND, "Stillwater Pond" },
    { AREA_NIGHTMARE_VALE, "Nightmare Vale" },
    { AREA_VENOMWEB_VALE, "Venomweb Vale" },
    { AREA_THE_BULWARK1, "The Bulwark1" },
    { AREA_SOUTHFURY_RIVER1, "Southfury River1" },
    { AREA_SOUTHFURY_RIVER2, "Southfury River2" },
    { AREA_RAZORMANE_GROUNDS, "Razormane Grounds" },
    { AREA_SKULL_ROCK, "Skull Rock" },
    { AREA_PALEMANE_ROCK, "Palemane Rock" },
    { AREA_WINDFURY_RIDGE, "Windfury Ridge" },
    { AREA_THE_GOLDEN_PLAINS, "The Golden Plains" },
    { AREA_THE_ROLLING_PLAINS, "The Rolling Plains" },
    { AREA_DUN_ALGAZ1, "Dun Algaz1" },
    { AREA_DUN_ALGAZ2, "Dun Algaz2" },
    { AREA_NORTH_GATE_PASS1, "North Gate Pass1" },
    { AREA_SOUTH_GATE_PASS1, "South Gate Pass1" },
    { AREA_TWILIGHT_GROVE, "Twilight Grove" },
    { AREA_GM_ISLAND, "Gm Island" },
    { AREA_DELETE_ME1, "Delete Me1" },
    { AREA_SOUTHFURY_RIVER3, "Southfury River3" },
    { AREA_SOUTHFURY_RIVER4, "Southfury River4" },
    { AREA_THANDOL_SPAN1, "Thandol Span1" },
    { AREA_THANDOL_SPAN2, "Thandol Span2" },
    { AREA_PURGATION_ISLE, "Purgation Isle" },
    { AREA_THE_JANSEN_STEAD, "The Jansen Stead" },
    { AREA_THE_DEAD_ACRE, "The Dead Acre" },
    { AREA_THE_MOLSEN_FARM, "The Molsen Farm" },
    { AREA_STENDELS_POND, "Stendels Pond" },
    { AREA_THE_DAGGER_HILLS, "The Dagger Hills" },
    { AREA_DEMONTS_PLACE, "Demonts Place" },
    { AREA_THE_DUST_PLAINS, "The Dust Plains" },
    { AREA_STONESPLINTER_VALLEY, "Stonesplinter Valley" },
    { AREA_VALLEY_OF_KINGS, "Valley Of Kings" },
    { AREA_ALGAZ_STATION, "Algaz Station" },
    { AREA_BUCKLEBREE_FARM, "Bucklebree Farm" },
    { AREA_THE_SHINING_STRAND, "The Shining Strand" },
    { AREA_NORTH_TIDES_HOLLOW, "North Tides Hollow" },
    { AREA_GRIZZLEPAW_RIDGE, "Grizzlepaw Ridge" },
    { AREA_THE_VERDANT_FIELDS, "The Verdant Fields" },
    { AREA_GADGETZAN, "Gadgetzan" },
    { AREA_STEAMWHEEDLE_PORT, "Steamwheedle Port" },
    { AREA_ZUL_FARRAK0, "Zul Farrak0" },
    { AREA_SANDSORROW_WATCH, "Sandsorrow Watch" },
    { AREA_THISTLESHRUB_VALLEY, "Thistleshrub Valley" },
    { AREA_THE_GAPING_CHASM, "The Gaping Chasm" },
    { AREA_THE_NOXIOUS_LAIR, "The Noxious Lair" },
    { AREA_DUNEMAUL_COMPOUND, "Dunemaul Compound" },
    { AREA_EASTMOON_RUINS, "Eastmoon Ruins" },
    { AREA_WATERSPRING_FIELD, "Waterspring Field" },
    { AREA_ZALASHJIS_DEN, "Zalashjis Den" },
    { AREA_LANDS_END_BEACH, "Lands End Beach" },
    { AREA_WAVESTRIDER_BEACH, "Wavestrider Beach" },
    { AREA_ULDUM, "Uldum" },
    { AREA_VALLEY_OF_THE_WATCHERS, "Valley Of The Watchers" },
    { AREA_GUNSTANS_POST, "Gunstans Post" },
    { AREA_SOUTHMOON_RUINS, "Southmoon Ruins" },
    { AREA_RENDERS_CAMP, "Renders Camp" },
    { AREA_RENDERS_VALLEY, "Renders Valley" },
    { AREA_RENDERS_ROCK, "Renders Rock" },
    { AREA_STONEWATCH_TOWER, "Stonewatch Tower" },
    { AREA_GALARDELL_VALLEY, "Galardell Valley" },
    { AREA_LAKERIDGE_HIGHWAY, "Lakeridge Highway" },
    { AREA_THREE_CORNERS, "Three Corners" },
    { AREA_DIREFORGE_HILL, "Direforge Hill" },
    { AREA_RAPTOR_RIDGE, "Raptor Ridge" },
    { AREA_BLACK_CHANNEL_MARSH, "Black Channel Marsh" },
    { AREA_THE_GREEN_BELT0, "The Green Belt0" },
    { AREA_MOSSHIDE_FEN, "Mosshide Fen" },
    { AREA_THELGEN_ROCK, "Thelgen Rock" },
    { AREA_BLUEGILL_MARSH, "Bluegill Marsh" },
    { AREA_SALTSPRAY_GLEN, "Saltspray Glen" },
    { AREA_SUNDOWN_MARSH, "Sundown Marsh" },
    { AREA_THE_GREEN_BELT1, "The Green Belt1" },
    { AREA_ANGERFANG_ENCAMPMENT, "Angerfang Encampment" },
    { AREA_GRIM_BATOL, "Grim Batol" },
    { AREA_DRAGONMAW_GATES, "Dragonmaw Gates" },
    { AREA_THE_LOST_FLEET, "The Lost Fleet" },
    { AREA_DARROW_HILL0, "Darrow Hill0" },
    { AREA_THORADINS_WALL1, "Thoradins Wall1" },
    { AREA_WEBWINDER_PATH, "Webwinder Path" },
    { AREA_THE_HUSHED_BANK, "The Hushed Bank" },
    { AREA_MANOR_MISTMANTLE, "Manor Mistmantle" },
    { AREA_CAMP_MOJACHE, "Camp Mojache" },
    { AREA_GRIMTOTEM_COMPOUND, "Grimtotem Compound" },
    { AREA_THE_WRITHING_DEEP, "The Writhing Deep" },
    { AREA_WILDWIND_LAKE, "Wildwind Lake" },
    { AREA_GORDUNNI_OUTPOST, "Gordunni Outpost" },
    { AREA_MOK_GORDUN, "Mok Gordun" },
    { AREA_FERAL_SCAR_VALE, "Feral Scar Vale" },
    { AREA_FRAYFEATHER_HIGHLANDS, "Frayfeather Highlands" },
    { AREA_IDLEWIND_LAKE, "Idlewind Lake" },
    { AREA_THE_FORGOTTEN_COAST, "The Forgotten Coast" },
    { AREA_EAST_PILLAR, "East Pillar" },
    { AREA_WEST_PILLAR, "West Pillar" },
    { AREA_DREAM_BOUGH, "Dream Bough" },
    { AREA_JADEMIR_LAKE, "Jademir Lake" },
    { AREA_ONEIROS, "Oneiros" },
    { AREA_RUINS_OF_RAVENWIND, "Ruins Of Ravenwind" },
    { AREA_RAGE_SCAR_HOLD, "Rage Scar Hold" },
    { AREA_FEATHERMOON_STRONGHOLD, "Feathermoon Stronghold" },
    { AREA_RUINS_OF_SOLARSAL, "Ruins Of Solarsal" },
    { AREA_LOWER_WILDS_UNUSED, "Lower Wilds Unused" },
    { AREA_THE_TWIN_COLOSSALS, "The Twin Colossals" },
    { AREA_SARDOR_ISLE, "Sardor Isle" },
    { AREA_ISLE_OF_DREAD, "Isle Of Dread" },
    { AREA_HIGH_WILDERNESS, "High Wilderness" },
    { AREA_LOWER_WILDS, "Lower Wilds" },
    { AREA_SOUTHERN_BARRENS, "Southern Barrens" },
    { AREA_SOUTHERN_GOLD_ROAD, "Southern Gold Road" },
    { AREA_ZUL_FARRAK1, "Zul Farrak1" },
    { AREA_UNUSED_ALCAZ_ISLAND, "Unused Alcaz Island" },
    { AREA_TIMBERMAW_HOLD0, "Timbermaw Hold0" },
    { AREA_VANNDIR_ENCAMPMENT, "Vanndir Encampment" },
    { AREA_TEST_AZSHARA, "Test Azshara" },
    { AREA_LEGASH_ENCAMPMENT, "Legash Encampment" },
    { AREA_THALASSIAN_BASE_CAMP, "Thalassian Base Camp" },
    { AREA_RUINS_OF_ELDARATH, "Ruins Of Eldarath" },
    { AREA_HETAERAS_CLUTCH, "Hetaeras Clutch" },
    { AREA_TEMPLE_OF_ZIN_MALOR, "Temple Of Zin Malor" },
    { AREA_BEARS_HEAD, "Bears Head" },
    { AREA_URSOLAN, "Ursolan" },
    { AREA_TEMPLE_OF_ARKKORAN, "Temple Of Arkkoran" },
    { AREA_BAY_OF_STORMS, "Bay Of Storms" },
    { AREA_THE_SHATTERED_STRAND, "The Shattered Strand" },
    { AREA_TOWER_OF_ELDARA, "Tower Of Eldara" },
    { AREA_JAGGED_REEF, "Jagged Reef" },
    { AREA_SOUTHRIDGE_BEACH, "Southridge Beach" },
    { AREA_RAVENCREST_MONUMENT, "Ravencrest Monument" },
    { AREA_FORLORN_RIDGE, "Forlorn Ridge" },
    { AREA_LAKE_MENNAR, "Lake Mennar" },
    { AREA_SHADOWSONG_SHRINE, "Shadowsong Shrine" },
    { AREA_HALDARR_ENCAMPMENT, "Haldarr Encampment" },
    { AREA_VALORMOK, "Valormok" },
    { AREA_THE_RUINED_REACHES, "The Ruined Reaches" },
    { AREA_THE_TALONDEEP_PATH0, "The Talondeep Path0" },
    { AREA_THE_TALONDEEP_PATH1, "The Talondeep Path1" },
    { AREA_ROCKTUSK_FARM, "Rocktusk Farm" },
    { AREA_JAGGEDSWINE_FARM, "Jaggedswine Farm" },
    { AREA_RAZORFEN_DOWNS1, "Razorfen Downs1" },
    { AREA_LOST_RIGGER_COVE, "Lost Rigger Cove" },
    { AREA_ULDAMAN0, "Uldaman0" },
    { AREA_LORDAMERE_LAKE1, "Lordamere Lake1" },
    { AREA_LORDAMERE_LAKE2, "Lordamere Lake2" },
    { AREA_GALLOWS_CORNER, "Gallows Corner" },
    { AREA_SILITHUS, "Silithus" },
    { AREA_EMERALD_FOREST, "Emerald Forest" },
    { AREA_SUNKEN_TEMPLE, "Sunken Temple" },
    { AREA_DREADMAUL_HOLD, "Dreadmaul Hold" },
    { AREA_NETHERGARDE_KEEP, "Nethergarde Keep" },
    { AREA_DREADMAUL_POST, "Dreadmaul Post" },
    { AREA_SERPENTS_COIL, "Serpents Coil" },
    { AREA_ALTAR_OF_STORMS1, "Altar Of Storms1" },
    { AREA_FIREWATCH_RIDGE, "Firewatch Ridge" },
    { AREA_THE_SLAG_PIT, "The Slag Pit" },
    { AREA_THE_SEA_OF_CINDERS, "The Sea Of Cinders" },
    { AREA_BLACKROCK_MOUNTAIN2, "Blackrock Mountain2" },
    { AREA_THORIUM_POINT, "Thorium Point" },
    { AREA_GARRISON_ARMORY, "Garrison Armory" },
    { AREA_THE_TEMPLE_OF_ATAL_HAKKAR, "The Temple Of Atal Hakkar" },
    { AREA_UNDERCITY, "Undercity" },
    { AREA_ULDAMAN1, "Uldaman1" },
    { AREA_NOT_USED_DEADMINES, "Not Used Deadmines" },
    { AREA_STORMWIND_CITY, "Stormwind City" },
    { AREA_IRONFORGE, "Ironforge" },
    { AREA_SPLITHOOF_HOLD, "Splithoof Hold" },
    { AREA_THE_CAPE_OF_STRANGLETHORN, "The Cape Of Stranglethorn" },
    { AREA_SOUTHERN_SAVAGE_COAST, "Southern Savage Coast" },
    { AREA_UNUSED_THE_DEADMINES_002, "Unused The Deadmines 002" },
    { AREA_UNUSED_IRONCLAD_COVE_003, "Unused Ironclad Cove 003" },
    { AREA_THE_DEADMINES, "The Deadmines" },
    { AREA_IRONCLAD_COVE, "Ironclad Cove" },
    { AREA_BLACKROCK_SPIRE, "Blackrock Spire" },
    { AREA_BLACKROCK_DEPTHS, "Blackrock Depths" },
    { AREA_RAPTOR_GROUNDS_UNUSED, "Raptor Grounds Unused" },
    { AREA_GROL_DOM_FARM_UNUSED, "Grol Dom Farm Unused" },
    { AREA_MORSHAN_BASE_CAMP, "Morshan Base Camp" },
    { AREA_HONORS_STAND_UNUSED, "Honors Stand Unused" },
    { AREA_BLACKTHORN_RIDGE_UNUSED, "Blackthorn Ridge Unused" },
    { AREA_BRAMBLESCAR_UNUSED, "Bramblescar Unused" },
    { AREA_AGAMA_GOR_UNUSED, "Agama Gor Unused" },
    { AREA_VALLEY_OF_HEROES, "Valley Of Heroes" },
    { AREA_ORGRIMMAR, "Orgrimmar" },
    { AREA_THUNDER_BLUFF, "Thunder Bluff" },
    { AREA_ELDER_RISE, "Elder Rise" },
    { AREA_SPIRIT_RISE, "Spirit Rise" },
    { AREA_HUNTER_RISE, "Hunter Rise" },
    { AREA_DARNASSUS, "Darnassus" },
    { AREA_CENARION_ENCLAVE, "Cenarion Enclave" },
    { AREA_CRAFTSMENS_TERRACE, "Craftsmens Terrace" },
    { AREA_WARRIORS_TERRACE, "Warriors Terrace" },
    { AREA_THE_TEMPLE_GARDENS, "The Temple Gardens" },
    { AREA_TRADESMENS_TERRACE, "Tradesmens Terrace" },
    { AREA_GAVINS_NAZE, "Gavins Naze" },
    { AREA_SOFERAS_NAZE, "Soferas Naze" },
    { AREA_CORRAHNS_DAGGER, "Corrahns Dagger" },
    { AREA_THE_HEADLAND, "The Headland" },
    { AREA_MISTY_SHORE, "Misty Shore" },
    { AREA_DANDREDS_FOLD, "Dandreds Fold" },
    { AREA_GROWLESS_CAVE, "Growless Cave" },
    { AREA_CHILLWIND_POINT, "Chillwind Point" },
    { AREA_RAPTOR_GROUNDS, "Raptor Grounds" },
    { AREA_BRAMBLESCAR, "Bramblescar" },
    { AREA_THORN_HILL, "Thorn Hill" },
    { AREA_AGAMA_GOR, "Agama Gor" },
    { AREA_BLACKTHORN_RIDGE, "Blackthorn Ridge" },
    { AREA_HONORS_STAND, "Honors Stand" },
    { AREA_THE_MORSHAN_RAMPART, "The Morshan Rampart" },
    { AREA_GROL_DOM_FARM, "Grol Dom Farm" },
    { AREA_RAZORFEN_KRAUL1, "Razorfen Kraul1" },
    { AREA_THE_GREAT_LIFT1, "The Great Lift1" },
    { AREA_MISTVALE_VALLEY, "Mistvale Valley" },
    { AREA_NEK_MANI_WELLSPRING, "Nek Mani Wellspring" },
    { AREA_BLOODSAIL_COMPOUND, "Bloodsail Compound" },
    { AREA_VENTURE_CO_BASE_CAMP, "Venture Co Base Camp" },
    { AREA_GURUBASHI_ARENA, "Gurubashi Arena" },
    { AREA_SPIRIT_DEN, "Spirit Den" },
    { AREA_THE_CRIMSON_VEIL, "The Crimson Veil" },
    { AREA_THE_RIPTIDE, "The Riptide" },
    { AREA_THE_DAMSELS_LUCK, "The Damsels Luck" },
    { AREA_VENTURE_CO_OPERATIONS_CENTER, "Venture Co Operations Center" },
    { AREA_DEADWOOD_VILLAGE, "Deadwood Village" },
    { AREA_FELPAW_VILLAGE, "Felpaw Village" },
    { AREA_JAEDENAR, "Jaedenar" },
    { AREA_BLOODVENOM_RIVER, "Bloodvenom River" },
    { AREA_BLOODVENOM_FALLS, "Bloodvenom Falls" },
    { AREA_SHATTER_SCAR_VALE, "Shatter Scar Vale" },
    { AREA_IRONTREE_WOODS, "Irontree Woods" },
    { AREA_IRONTREE_CAVERN, "Irontree Cavern" },
    { AREA_TIMBERMAW_HOLD1, "Timbermaw Hold1" },
    { AREA_SHADOW_HOLD, "Shadow Hold" },
    { AREA_SHRINE_OF_THE_DECEIVER, "Shrine Of The Deceiver" },
    { AREA_ITHARIUSS_CAVE, "Ithariuss Cave" },
    { AREA_SORROWMURK, "Sorrowmurk" },
    { AREA_DRAENIL_DUR_VILLAGE, "Draenil Dur Village" },
    { AREA_SPLINTERSPEAR_JUNCTION, "Splinterspear Junction" },
    { AREA_STAGALBOG, "Stagalbog" },
    { AREA_THE_SHIFTING_MIRE, "The Shifting Mire" },
    { AREA_STAGALBOG_CAVE, "Stagalbog Cave" },
    { AREA_WITHERBARK_CAVERNS, "Witherbark Caverns" },
    { AREA_THORADINS_WALL2, "Thoradins Wall2" },
    { AREA_BOULDER_GOR, "Boulder Gor" },
    { AREA_VALLEY_OF_FANGS, "Valley Of Fangs" },
    { AREA_THE_DUSTBOWL, "The Dustbowl" },
    { AREA_MIRAGE_FLATS, "Mirage Flats" },
    { AREA_FEATHERBEARDS_HOVEL, "Featherbeards Hovel" },
    { AREA_SHINDIGGERS_CAMP, "Shindiggers Camp" },
    { AREA_PLAGUEMIST_RAVINE, "Plaguemist Ravine" },
    { AREA_VALORWIND_LAKE, "Valorwind Lake" },
    { AREA_AGOL_WATHA, "Agol Watha" },
    { AREA_HIRI_WATHA, "Hiri Watha" },
    { AREA_THE_CREEPING_RUIN, "The Creeping Ruin" },
    { AREA_BOGENS_LEDGE, "Bogens Ledge" },
    { AREA_THE_MAKERS_TERRACE, "The Makers Terrace" },
    { AREA_DUSTWIND_GULCH, "Dustwind Gulch" },
    { AREA_SHAOL_WATHA, "Shaol Watha" },
    { AREA_NOONSHADE_RUINS, "Noonshade Ruins" },
    { AREA_BROKEN_PILLAR, "Broken Pillar" },
    { AREA_ABYSSAL_SANDS, "Abyssal Sands" },
    { AREA_SOUTHBREAK_SHORE, "Southbreak Shore" },
    { AREA_CAVERNS_OF_TIME0, "Caverns Of Time0" },
    { AREA_THE_MARSHLANDS, "The Marshlands" },
    { AREA_IRONSTONE_PLATEAU, "Ironstone Plateau" },
    { AREA_BLACKCHAR_CAVE, "Blackchar Cave" },
    { AREA_TANNER_CAMP, "Tanner Camp" },
    { AREA_DUSTFIRE_VALLEY, "Dustfire Valley" },
    { AREA_ZUL_GURUB1, "Zul Gurub1" },
    { AREA_MISTY_REED_POST, "Misty Reed Post" },
    { AREA_BLOODVENOM_POST, "Bloodvenom Post" },
    { AREA_TALONBRANCH_GLADE, "Talonbranch Glade" },
    { AREA_STRATHOLME0, "Stratholme0" },
    { AREA_UNUSED_SHADOWFANG_KEEP_003, "Unused Shadowfang Keep 003" },
    { AREA_SCHOLOMANCE, "Scholomance" },
    { AREA_TWILIGHT_VALE, "Twilight Vale" },
    { AREA_TWILIGHT_SHORE, "Twilight Shore" },
    { AREA_ALCAZ_ISLAND, "Alcaz Island" },
    { AREA_DARKCLOUD_PINNACLE, "Darkcloud Pinnacle" },
    { AREA_DAWNING_WOOD_CATACOMBS, "Dawning Wood Catacombs" },
    { AREA_STONEWATCH_KEEP, "Stonewatch Keep" },
    { AREA_MARAUDON, "Maraudon" },
    { AREA_STOUTLAGER_INN, "Stoutlager Inn" },
    { AREA_THUNDERBREW_DISTILLERY, "Thunderbrew Distillery" },
    { AREA_MENETHIL_KEEP, "Menethil Keep" },
    { AREA_DEEPWATER_TAVERN, "Deepwater Tavern" },
    { AREA_SHADOW_GRAVE, "Shadow Grave" },
    { AREA_BRILL_TOWN_HALL, "Brill Town Hall" },
    { AREA_GALLOWS_END_TAVERN, "Gallows End Tavern" },
    { AREA_THE_POOLS_OF_VISION_UNUSED, "The Pools Of Vision Unused" },
    { AREA_DREADMIST_DEN, "Dreadmist Den" },
    { AREA_BAEL_DUN_KEEP, "Bael Dun Keep" },
    { AREA_EMBERSTRIFES_DEN, "Emberstrifes Den" },
    { AREA_ONYXIAS_LAIR, "Onyxias Lair" },
    { AREA_WINDSHEAR_MINE, "Windshear Mine" },
    { AREA_ROLANDS_DOOM, "Rolands Doom" },
    { AREA_BATTLE_RING, "Battle Ring" },
    { AREA_THE_POOLS_OF_VISION, "The Pools Of Vision" },
    { AREA_SHADOWBREAK_RAVINE, "Shadowbreak Ravine" },
    { AREA_BROKEN_SPEAR_VILLAGE, "Broken Spear Village" },
    { AREA_WHITEREACH_POST, "Whitereach Post" },
    { AREA_GORNIA, "Gornia" },
    { AREA_ZANES_EYE_CRATER, "Zanes Eye Crater" },
    { AREA_MIRAGE_RACEWAY, "Mirage Raceway" },
    { AREA_FROSTSABER_ROCK, "Frostsaber Rock" },
    { AREA_THE_HIDDEN_GROVE, "The Hidden Grove" },
    { AREA_TIMBERMAW_POST, "Timbermaw Post" },
    { AREA_WINTERFALL_VILLAGE, "Winterfall Village" },
    { AREA_MAZTHORIL, "Mazthoril" },
    { AREA_FROSTFIRE_HOT_SPRINGS, "Frostfire Hot Springs" },
    { AREA_ICE_THISTLE_HILLS, "Ice Thistle Hills" },
    { AREA_DUN_MANDARR, "Dun Mandarr" },
    { AREA_FROSTWHISPER_GORGE, "Frostwhisper Gorge" },
    { AREA_OWL_WING_THICKET, "Owl Wing Thicket" },
    { AREA_LAKE_KEL_THERIL, "Lake Kel Theril" },
    { AREA_THE_RUINS_OF_KEL_THERIL, "The Ruins Of Kel Theril" },
    { AREA_STARFALL_VILLAGE, "Starfall Village" },
    { AREA_BAN_THALLOW_BARROW_DEN, "Ban Thallow Barrow Den" },
    { AREA_EVERLOOK, "Everlook" },
    { AREA_DARKWHISPER_GORGE, "Darkwhisper Gorge" },
    { AREA_DEEPRUN_TRAM, "Deeprun Tram" },
    { AREA_THE_FUNGAL_VALE, "The Fungal Vale" },
    { AREA_UNUSED_THE_MARRIS_STEAD, "Unused The Marris Stead" },
    { AREA_THE_MARRIS_STEAD, "The Marris Stead" },
    { AREA_THE_UNDERCROFT, "The Undercroft" },
    { AREA_DARROWSHIRE, "Darrowshire" },
    { AREA_CROWN_GUARD_TOWER, "Crown Guard Tower" },
    { AREA_CORINS_CROSSING, "Corins Crossing" },
    { AREA_SCARLET_BASE_CAMP, "Scarlet Base Camp" },
    { AREA_TYRS_HAND, "Tyrs Hand" },
    { AREA_THE_SCARLET_BASILICA, "The Scarlet Basilica" },
    { AREA_LIGHTS_HOPE_CHAPEL, "Lights Hope Chapel" },
    { AREA_BROWMAN_MILL, "Browman Mill" },
    { AREA_THE_NOXIOUS_GLADE, "The Noxious Glade" },
    { AREA_EASTWALL_TOWER, "Eastwall Tower" },
    { AREA_NORTHDALE, "Northdale" },
    { AREA_ZUL_MASHAR, "Zul Mashar" },
    { AREA_MAZRA_ALOR, "Mazra Alor" },
    { AREA_NORTHPASS_TOWER, "Northpass Tower" },
    { AREA_QUEL_LITHIEN_LODGE, "Quel Lithien Lodge" },
    { AREA_PLAGUEWOOD, "Plaguewood" },
    { AREA_SCOURGEHOLD, "Scourgehold" },
    { AREA_STRATHOLME1, "Stratholme1" },
    { AREA_UNUSED_STRATHOLME, "Unused Stratholme" },
    { AREA_DARROWMERE_LAKE0, "Darrowmere Lake0" },
    { AREA_CAER_DARROW, "Caer Darrow" },
    { AREA_DARROWMERE_LAKE1, "Darrowmere Lake1" },
    { AREA_CAVERNS_OF_TIME1, "Caverns Of Time1" },
    { AREA_THISTLEFUR_VILLAGE, "Thistlefur Village" },
    { AREA_THE_QUAGMIRE, "The Quagmire" },
    { AREA_WINDBREAK_CANYON, "Windbreak Canyon" },
    { AREA_SOUTH_SEAS0, "South Seas0" },
    { AREA_THE_GREAT_SEA3, "The Great Sea3" },
    { AREA_THE_GREAT_SEA4, "The Great Sea4" },
    { AREA_THE_GREAT_SEA5, "The Great Sea5" },
    { AREA_THE_GREAT_SEA6, "The Great Sea6" },
    { AREA_THE_VEILED_SEA1, "The Veiled Sea1" },
    { AREA_THE_VEILED_SEA2, "The Veiled Sea2" },
    { AREA_THE_VEILED_SEA3, "The Veiled Sea3" },
    { AREA_THE_VEILED_SEA4, "The Veiled Sea4" },
    { AREA_THE_VEILED_SEA5, "The Veiled Sea5" },
    { AREA_RAZOR_HILL_BARRACKS, "Razor Hill Barracks" },
    { AREA_SOUTH_SEAS1, "South Seas1" },
    { AREA_THE_GREAT_SEA7, "The Great Sea7" },
    { AREA_BLOODTOOTH_CAMP, "Bloodtooth Camp" },
    { AREA_FOREST_SONG, "Forest Song" },
    { AREA_GREENPAW_VILLAGE, "Greenpaw Village" },
    { AREA_SILVERWING_OUTPOST, "Silverwing Outpost" },
    { AREA_NIGHTHAVEN, "Nighthaven" },
    { AREA_SHRINE_OF_REMULOS, "Shrine Of Remulos" },
    { AREA_STORMRAGE_BARROW_DENS, "Stormrage Barrow Dens" },
    { AREA_THE_GREAT_SEA8, "The Great Sea8" },
    { AREA_THE_GREAT_SEA9, "The Great Sea9" },
    { AREA_THE_BLACK_MORASS, "The Black Morass" },
    { AREA_OLD_HILLSBRAD_FOOTHILLS, "Old Hillsbrad Foothills" },
    { AREA_TARREN_MILL1, "Tarren Mill1" },
    { AREA_SOUTHSHORE1, "Southshore1" },
    { AREA_DURNHOLDE_KEEP1, "Durnholde Keep1" },
    { AREA_DUN_GAROK1, "Dun Garok1" },
    { AREA_HILLSBRAD_FIELDS1, "Hillsbrad Fields1" },
    { AREA_EASTERN_STRAND1, "Eastern Strand1" },
    { AREA_NETHANDER_STEAD1, "Nethander Stead1" },
    { AREA_DARROW_HILL1, "Darrow Hill1" },
    { AREA_SOUTHPOINT_TOWER1, "Southpoint Tower1" },
    { AREA_THORADINS_WALL3, "Thoradins Wall3" },
    { AREA_WESTERN_STRAND1, "Western Strand1" },
    { AREA_AZURELODE_MINE1, "Azurelode Mine1" },
    { AREA_THE_GREAT_SEA10, "The Great Sea10" },
    { AREA_THE_GREAT_SEA11, "The Great Sea11" },
    { AREA_THE_GREAT_SEA12, "The Great Sea12" },
    { AREA_THE_FORBIDDING_SEA1, "The Forbidding Sea1" },
    { AREA_THE_FORBIDDING_SEA2, "The Forbidding Sea2" },
    { AREA_THE_FORBIDDING_SEA3, "The Forbidding Sea3" },
    { AREA_THE_FORBIDDING_SEA4, "The Forbidding Sea4" },
    { AREA_TETHRIS_ARAN, "Tethris Aran" },
    { AREA_ETHEL_RETHOR, "Ethel Rethor" },
    { AREA_RANAZJAR_ISLE, "Ranazjar Isle" },
    { AREA_KORMEKS_HUT, "Kormeks Hut" },
    { AREA_SHADOWPREY_VILLAGE, "Shadowprey Village" },
    { AREA_BLACKROCK_PASS, "Blackrock Pass" },
    { AREA_MORGANS_VIGIL, "Morgans Vigil" },
    { AREA_SLITHER_ROCK, "Slither Rock" },
    { AREA_TERROR_WING_PATH, "Terror Wing Path" },
    { AREA_DRACO_DAR, "Draco Dar" },
    { AREA_RAGEFIRE_CHASM, "Ragefire Chasm" },
    { AREA_NIGHTSONG_WOODS, "Nightsong Woods" },
    { AREA_THE_VEILED_SEA6, "The Veiled Sea6" },
    { AREA_MORLOS_ARAN, "Morlos Aran" },
    { AREA_EMERALD_SANCTUARY, "Emerald Sanctuary" },
    { AREA_JADEFIRE_GLEN, "Jadefire Glen" },
    { AREA_RUINS_OF_CONSTELLAS, "Ruins Of Constellas" },
    { AREA_BITTER_REACHES, "Bitter Reaches" },
    { AREA_RISE_OF_THE_DEFILER, "Rise Of The Defiler" },
    { AREA_LARISS_PAVILION, "Lariss Pavilion" },
    { AREA_WOODPAW_HILLS, "Woodpaw Hills" },
    { AREA_WOODPAW_DEN, "Woodpaw Den" },
    { AREA_VERDANTIS_RIVER, "Verdantis River" },
    { AREA_RUINS_OF_ISILDIEN, "Ruins Of Isildien" },
    { AREA_GRIMTOTEM_POST, "Grimtotem Post" },
    { AREA_CAMP_APARAJE, "Camp Aparaje" },
    { AREA_MALAKA_JIN, "Malaka Jin" },
    { AREA_BOULDERSLIDE_RAVINE, "Boulderslide Ravine" },
    { AREA_SISHIR_CANYON, "Sishir Canyon" },
    { AREA_DIRE_MAUL0, "Dire Maul0" },
    { AREA_DEADWIND_RAVINE, "Deadwind Ravine" },
    { AREA_DIAMONDHEAD_RIVER, "Diamondhead River" },
    { AREA_ARIDENS_CAMP, "Aridens Camp" },
    { AREA_THE_VICE, "The Vice" },
    { AREA_KARAZHAN, "Karazhan" },
    { AREA_MORGANS_PLOT, "Morgans Plot" },
    { AREA_DIRE_MAUL1, "Dire Maul1" },
    { AREA_ALTERAC_VALLEY0, "Alterac Valley0" },
    { AREA_SCRABBLESCREWS_CAMP, "Scrabblescrews Camp" },
    { AREA_JADEFIRE_RUN, "Jadefire Run" },
    { AREA_THONDRORIL_RIVER0, "Thondroril River0" },
    { AREA_THONDRORIL_RIVER1, "Thondroril River1" },
    { AREA_LAKE_MERELDAR, "Lake Mereldar" },
    { AREA_PESTILENT_SCAR, "Pestilent Scar" },
    { AREA_THE_INFECTIS_SCAR, "The Infectis Scar" },
    { AREA_BLACKWOOD_LAKE, "Blackwood Lake" },
    { AREA_EASTWALL_GATE, "Eastwall Gate" },
    { AREA_TERRORWEB_TUNNEL, "Terrorweb Tunnel" },
    { AREA_TERRORDALE, "Terrordale" },
    { AREA_KARGATHIA_KEEP, "Kargathia Keep" },
    { AREA_VALLEY_OF_BONES, "Valley Of Bones" },
    { AREA_BLACKWING_LAIR, "Blackwing Lair" },
    { AREA_DEADMANS_CROSSING, "Deadmans Crossing" },
    { AREA_MOLTEN_CORE, "Molten Core" },
    { AREA_THE_SCARAB_WALL, "The Scarab Wall" },
    { AREA_SOUTHWIND_VILLAGE, "Southwind Village" },
    { AREA_TWILIGHT_BASE_CAMP, "Twilight Base Camp" },
    { AREA_THE_CRYSTAL_VALE, "The Crystal Vale" },
    { AREA_THE_SCARAB_DAIS, "The Scarab Dais" },
    { AREA_HIVE_ASHI, "Hive Ashi" },
    { AREA_HIVE_ZORA, "Hive Zora" },
    { AREA_HIVE_REGAL, "Hive Regal" },
    { AREA_SHRINE_OF_THE_FALLEN_WARRIOR, "Shrine Of The Fallen Warrior" },
    { AREA_UNUSED_ALTERAC_VALLEY, "Unused Alterac Valley" },
    { AREA_BLACKFATHOM_DEEPS1, "Blackfathom Deeps1" },
    { AREA_ON_MAP_DUNGEON4, "On Map Dungeon4" },
    { AREA_THE_MASTERS_CELLAR, "The Masters Cellar" },
    { AREA_STONEWROUGHT_PASS, "Stonewrought Pass" },
    { AREA_ALTERAC_VALLEY1, "Alterac Valley1" },
    { AREA_THE_RUMBLE_CAGE, "The Rumble Cage" },
    { AREA_CHUNK_TEST, "Chunk Test" },
    { AREA_ZORAMGAR_OUTPOST, "Zoramgar Outpost" },
    { AREA_HALL_OF_LEGENDS, "Hall Of Legends" },
    { AREA_CHAMPIONS_HALL, "Champions Hall" },
    { AREA_GROSH_GOK_COMPOUND, "Grosh Gok Compound" },
    { AREA_SLEEPING_GORGE, "Sleeping Gorge" },
    { AREA_IRONDEEP_MINE, "Irondeep Mine" },
    { AREA_STONEHEARTH_OUTPOST, "Stonehearth Outpost" },
    { AREA_DUN_BALDAR, "Dun Baldar" },
    { AREA_ICEWING_PASS, "Icewing Pass" },
    { AREA_FROSTWOLF_VILLAGE, "Frostwolf Village" },
    { AREA_TOWER_POINT, "Tower Point" },
    { AREA_COLDTOOTH_MINE, "Coldtooth Mine" },
    { AREA_WINTERAX_HOLD, "Winterax Hold" },
    { AREA_ICEBLOOD_GARRISON, "Iceblood Garrison" },
    { AREA_FROSTWOLF_KEEP, "Frostwolf Keep" },
    { AREA_TOR_KREN_FARM, "Tor Kren Farm" },
    { AREA_FROST_DAGGER_PASS, "Frost Dagger Pass" },
    { AREA_IRONSTONE_CAMP, "Ironstone Camp" },
    { AREA_WEAZELS_CRATER, "Weazels Crater" },
    { AREA_TAHONDA_RUINS, "Tahonda Ruins" },
    { AREA_FIELD_OF_STRIFE, "Field Of Strife" },
    { AREA_ICEWING_CAVERN, "Icewing Cavern" },
    { AREA_VALORS_REST, "Valors Rest" },
    { AREA_THE_SWARMING_PILLAR, "The Swarming Pillar" },
    { AREA_TWILIGHT_POST, "Twilight Post" },
    { AREA_TWILIGHT_OUTPOST, "Twilight Outpost" },
    { AREA_RAVAGED_TWILIGHT_CAMP, "Ravaged Twilight Camp" },
    { AREA_SHALZARUS_LAIR, "Shalzarus Lair" },
    { AREA_TALRENDIS_POINT, "Talrendis Point" },
    { AREA_RETHRESS_SANCTUM, "Rethress Sanctum" },
    { AREA_MOON_HORROR_DEN, "Moon Horror Den" },
    { AREA_SCALEBEARDS_CAVE, "Scalebeards Cave" },
    { AREA_BOULDERSLIDE_CAVERN, "Boulderslide Cavern" },
    { AREA_WARSONG_LABOR_CAMP, "Warsong Labor Camp" },
    { AREA_CHILLWIND_CAMP, "Chillwind Camp" },
    { AREA_THE_MAUL, "The Maul" },
    { AREA_THE_MAUL_UNUSED, "The Maul Unused" },
    { AREA_BONES_OF_GRAKKAROND, "Bones Of Grakkarond" },
    { AREA_WARSONG_GULCH, "Warsong Gulch" },
    { AREA_FROSTWOLF_GRAVEYARD, "Frostwolf Graveyard" },
    { AREA_FROSTWOLF_PASS, "Frostwolf Pass" },
    { AREA_DUN_BALDAR_PASS, "Dun Baldar Pass" },
    { AREA_ICEBLOOD_GRAVEYARD, "Iceblood Graveyard" },
    { AREA_SNOWFALL_GRAVEYARD, "Snowfall Graveyard" },
    { AREA_STONEHEARTH_GRAVEYARD, "Stonehearth Graveyard" },
    { AREA_STORMPIKE_GRAVEYARD, "Stormpike Graveyard" },
    { AREA_ICEWING_BUNKER, "Icewing Bunker" },
    { AREA_STONEHEARTH_BUNKER, "Stonehearth Bunker" },
    { AREA_WILDPAW_RIDGE, "Wildpaw Ridge" },
    { AREA_REVANTUSK_VILLAGE, "Revantusk Village" },
    { AREA_ROCK_OF_DUROTAN, "Rock Of Durotan" },
    { AREA_SILVERWING_GROVE, "Silverwing Grove" },
    { AREA_WARSONG_LUMBER_MILL, "Warsong Lumber Mill" },
    { AREA_SILVERWING_HOLD, "Silverwing Hold" },
    { AREA_WILDPAW_CAVERN, "Wildpaw Cavern" },
    { AREA_THE_VEILED_CLEFT, "The Veiled Cleft" },
    { AREA_YOJAMBA_ISLE, "Yojamba Isle" },
    { AREA_ARATHI_BASIN, "Arathi Basin" },
    { AREA_THE_COIL, "The Coil" },
    { AREA_ALTAR_OF_HIR_EEK, "Altar Of Hir Eek" },
    { AREA_SHADRA_ZAAR, "Shadra Zaar" },
    { AREA_HAKKARI_GROUNDS, "Hakkari Grounds" },
    { AREA_NAZE_OF_SHIRVALLAH, "Naze Of Shirvallah" },
    { AREA_TEMPLE_OF_BETHEKK, "Temple Of Bethekk" },
    { AREA_THE_BLOODFIRE_PIT, "The Bloodfire Pit" },
    { AREA_ALTAR_OF_THE_BLOOD_GOD, "Altar Of The Blood God" },
    { AREA_ZANZAS_RISE, "Zanzas Rise" },
    { AREA_EDGE_OF_MADNESS, "Edge Of Madness" },
    { AREA_TROLLBANE_HALL, "Trollbane Hall" },
    { AREA_DEFILERS_DEN, "Defilers Den" },
    { AREA_PAGLES_POINTE, "Pagles Pointe" },
    { AREA_FARM, "Farm" },
    { AREA_BLACKSMITH, "Blacksmith" },
    { AREA_LUMBER_MILL, "Lumber Mill" },
    { AREA_GOLD_MINE, "Gold Mine" },
    { AREA_STABLES, "Stables" },
    { AREA_CENARION_HOLD, "Cenarion Hold" },
    { AREA_STAGHELM_POINT, "Staghelm Point" },
    { AREA_BRONZEBEARD_ENCAMPMENT, "Bronzebeard Encampment" },
    { AREA_AHN_QIRAJ, "Ahn Qiraj" },
    { AREA_RUINS_OF_AHN_QIRAJ0, "Ruins Of Ahn Qiraj0" },
    { AREA_TWILIGHTS_RUN, "Twilights Run" },
    { AREA_ORTELLS_HIDEOUT, "Ortells Hideout" },
    { AREA_SCARAB_TERRACE, "Scarab Terrace" },
    { AREA_GENERALS_TERRACE, "Generals Terrace" },
    { AREA_THE_RESERVOIR, "The Reservoir" },
    { AREA_THE_HATCHERY, "The Hatchery" },
    { AREA_THE_COMB, "The Comb" },
    { AREA_WATCHERS_TERRACE, "Watchers Terrace" },
    { AREA_RUINS_OF_AHN_QIRAJ1, "Ruins Of Ahn Qiraj1" },
    { AREA_NAXXRAMAS, "Naxxramas" },
    { AREA_CITY, "City" },
    { AREA_GATES_OF_AHN_QIRAJ, "Gates Of Ahn Qiraj" },
    { AREA_RAVENHOLDT_MANOR, "Ravenholdt Manor" },
    { 0, NULL }
};

typedef enum {
    MAP_EASTERN_KINGDOMS = 0x000,
    MAP_KALIMDOR = 0x001,
    MAP_TESTING = 0x00D,
    MAP_SCOTT_TEST = 0x019,
    MAP_CASH_TEST = 0x01D,
    MAP_ALTERAC_VALLEY = 0x01E,
    MAP_SHADOWFANG_KEEP = 0x021,
    MAP_STORMWIND_STOCKADE = 0x022,
    MAP_STORMWIND_PRISON = 0x023,
    MAP_DEADMINES = 0x024,
    MAP_AZSHARA_CRATER = 0x025,
    MAP_COLLINS_TEST = 0x02A,
    MAP_WAILING_CAVERNS = 0x02B,
    MAP_MONASTERY_UNUSED = 0x02C,
    MAP_RAZORFEN_KRAUL = 0x02F,
    MAP_BLACKFATHOM_DEEPS = 0x030,
    MAP_ULDAMAN = 0x046,
    MAP_GNOMEREGAN = 0x05A,
    MAP_SUNKEN_TEMPLE = 0x06D,
    MAP_RAZORFEN_DOWNS = 0x081,
    MAP_EMERALD_DREAM = 0x0A9,
    MAP_SCARLET_MONASTERY = 0x0BD,
    MAP_ZUL_FARRAK = 0x0D1,
    MAP_BLACKROCK_SPIRE = 0x0E5,
    MAP_BLACKROCK_DEPTHS = 0x0E6,
    MAP_ONYXIAS_LAIR = 0x0F9,
    MAP_OPENING_OF_THE_DARK_PORTAL = 0x10D,
    MAP_SCHOLOMANCE = 0x121,
    MAP_ZUL_GURUB = 0x135,
    MAP_STRATHOLME = 0x149,
    MAP_MARAUDON = 0x15D,
    MAP_DEEPRUN_TRAM = 0x171,
    MAP_RAGEFIRE_CHASM = 0x185,
    MAP_MOLTEN_CORE = 0x199,
    MAP_DIRE_MAUL = 0x1AD,
    MAP_ALLIANCE_PVP_BARRACKS = 0x1C1,
    MAP_HORDE_PVP_BARRACKS = 0x1C2,
    MAP_DEVELOPMENT_LAND = 0x1C3,
    MAP_BLACKWING_LAIR = 0x1D5,
    MAP_WARSONG_GULCH = 0x1E9,
    MAP_RUINS_OF_AHN_QIRAJ = 0x1FD,
    MAP_ARATHI_BASIN = 0x211,
    MAP_AHN_QIRAJ_TEMPLE = 0x213,
    MAP_NAXXRAMAS = 0x215,
} e_map;
static const value_string e_map_strings[] =  {
    { MAP_EASTERN_KINGDOMS, "Eastern Kingdoms" },
    { MAP_KALIMDOR, "Kalimdor" },
    { MAP_TESTING, "Testing" },
    { MAP_SCOTT_TEST, "Scott Test" },
    { MAP_CASH_TEST, "Cash Test" },
    { MAP_ALTERAC_VALLEY, "Alterac Valley" },
    { MAP_SHADOWFANG_KEEP, "Shadowfang Keep" },
    { MAP_STORMWIND_STOCKADE, "Stormwind Stockade" },
    { MAP_STORMWIND_PRISON, "Stormwind Prison" },
    { MAP_DEADMINES, "Deadmines" },
    { MAP_AZSHARA_CRATER, "Azshara Crater" },
    { MAP_COLLINS_TEST, "Collins Test" },
    { MAP_WAILING_CAVERNS, "Wailing Caverns" },
    { MAP_MONASTERY_UNUSED, "Monastery Unused" },
    { MAP_RAZORFEN_KRAUL, "Razorfen Kraul" },
    { MAP_BLACKFATHOM_DEEPS, "Blackfathom Deeps" },
    { MAP_ULDAMAN, "Uldaman" },
    { MAP_GNOMEREGAN, "Gnomeregan" },
    { MAP_SUNKEN_TEMPLE, "Sunken Temple" },
    { MAP_RAZORFEN_DOWNS, "Razorfen Downs" },
    { MAP_EMERALD_DREAM, "Emerald Dream" },
    { MAP_SCARLET_MONASTERY, "Scarlet Monastery" },
    { MAP_ZUL_FARRAK, "Zul Farrak" },
    { MAP_BLACKROCK_SPIRE, "Blackrock Spire" },
    { MAP_BLACKROCK_DEPTHS, "Blackrock Depths" },
    { MAP_ONYXIAS_LAIR, "Onyxias Lair" },
    { MAP_OPENING_OF_THE_DARK_PORTAL, "Opening Of The Dark Portal" },
    { MAP_SCHOLOMANCE, "Scholomance" },
    { MAP_ZUL_GURUB, "Zul Gurub" },
    { MAP_STRATHOLME, "Stratholme" },
    { MAP_MARAUDON, "Maraudon" },
    { MAP_DEEPRUN_TRAM, "Deeprun Tram" },
    { MAP_RAGEFIRE_CHASM, "Ragefire Chasm" },
    { MAP_MOLTEN_CORE, "Molten Core" },
    { MAP_DIRE_MAUL, "Dire Maul" },
    { MAP_ALLIANCE_PVP_BARRACKS, "Alliance Pvp Barracks" },
    { MAP_HORDE_PVP_BARRACKS, "Horde Pvp Barracks" },
    { MAP_DEVELOPMENT_LAND, "Development Land" },
    { MAP_BLACKWING_LAIR, "Blackwing Lair" },
    { MAP_WARSONG_GULCH, "Warsong Gulch" },
    { MAP_RUINS_OF_AHN_QIRAJ, "Ruins Of Ahn Qiraj" },
    { MAP_ARATHI_BASIN, "Arathi Basin" },
    { MAP_AHN_QIRAJ_TEMPLE, "Ahn Qiraj Temple" },
    { MAP_NAXXRAMAS, "Naxxramas" },
    { 0, NULL }
};

typedef enum {
    CREATURE_FAMILY_NONE = 0x00,
    CREATURE_FAMILY_WOLF = 0x01,
    CREATURE_FAMILY_CAT = 0x02,
    CREATURE_FAMILY_SPIDER = 0x03,
    CREATURE_FAMILY_BEAR = 0x04,
    CREATURE_FAMILY_BOAR = 0x05,
    CREATURE_FAMILY_CROCOLISK = 0x06,
    CREATURE_FAMILY_CARRION_BIRD = 0x07,
    CREATURE_FAMILY_CRAB = 0x08,
    CREATURE_FAMILY_GORILLA = 0x09,
    CREATURE_FAMILY_RAPTOR = 0x0B,
    CREATURE_FAMILY_TALLSTRIDER = 0x0C,
    CREATURE_FAMILY_FELHUNTER = 0x0F,
    CREATURE_FAMILY_VOIDWALKER = 0x10,
    CREATURE_FAMILY_SUCCUBUS = 0x11,
    CREATURE_FAMILY_DOOMGUARD = 0x13,
    CREATURE_FAMILY_SCORPID = 0x14,
    CREATURE_FAMILY_TURTLE = 0x15,
    CREATURE_FAMILY_IMP = 0x17,
    CREATURE_FAMILY_BAT = 0x18,
    CREATURE_FAMILY_HYENA = 0x19,
    CREATURE_FAMILY_OWL = 0x1A,
    CREATURE_FAMILY_WIND_SERPENT = 0x1B,
    CREATURE_FAMILY_REMOTE_CONTROL = 0x1C,
} e_creature_family;
static const value_string e_creature_family_strings[] =  {
    { CREATURE_FAMILY_NONE, "None" },
    { CREATURE_FAMILY_WOLF, "Wolf" },
    { CREATURE_FAMILY_CAT, "Cat" },
    { CREATURE_FAMILY_SPIDER, "Spider" },
    { CREATURE_FAMILY_BEAR, "Bear" },
    { CREATURE_FAMILY_BOAR, "Boar" },
    { CREATURE_FAMILY_CROCOLISK, "Crocolisk" },
    { CREATURE_FAMILY_CARRION_BIRD, "Carrion Bird" },
    { CREATURE_FAMILY_CRAB, "Crab" },
    { CREATURE_FAMILY_GORILLA, "Gorilla" },
    { CREATURE_FAMILY_RAPTOR, "Raptor" },
    { CREATURE_FAMILY_TALLSTRIDER, "Tallstrider" },
    { CREATURE_FAMILY_FELHUNTER, "Felhunter" },
    { CREATURE_FAMILY_VOIDWALKER, "Voidwalker" },
    { CREATURE_FAMILY_SUCCUBUS, "Succubus" },
    { CREATURE_FAMILY_DOOMGUARD, "Doomguard" },
    { CREATURE_FAMILY_SCORPID, "Scorpid" },
    { CREATURE_FAMILY_TURTLE, "Turtle" },
    { CREATURE_FAMILY_IMP, "Imp" },
    { CREATURE_FAMILY_BAT, "Bat" },
    { CREATURE_FAMILY_HYENA, "Hyena" },
    { CREATURE_FAMILY_OWL, "Owl" },
    { CREATURE_FAMILY_WIND_SERPENT, "Wind Serpent" },
    { CREATURE_FAMILY_REMOTE_CONTROL, "Remote Control" },
    { 0, NULL }
};

typedef enum {
    INVENTORY_TYPE_NON_EQUIP = 0x00,
    INVENTORY_TYPE_HEAD = 0x01,
    INVENTORY_TYPE_NECK = 0x02,
    INVENTORY_TYPE_SHOULDERS = 0x03,
    INVENTORY_TYPE_BODY = 0x04,
    INVENTORY_TYPE_CHEST = 0x05,
    INVENTORY_TYPE_WAIST = 0x06,
    INVENTORY_TYPE_LEGS = 0x07,
    INVENTORY_TYPE_FEET = 0x08,
    INVENTORY_TYPE_WRISTS = 0x09,
    INVENTORY_TYPE_HANDS = 0x0A,
    INVENTORY_TYPE_FINGER = 0x0B,
    INVENTORY_TYPE_TRINKET = 0x0C,
    INVENTORY_TYPE_WEAPON = 0x0D,
    INVENTORY_TYPE_SHIELD = 0x0E,
    INVENTORY_TYPE_RANGED = 0x0F,
    INVENTORY_TYPE_CLOAK = 0x10,
    INVENTORY_TYPE_TWO_HANDED_WEAPON = 0x11,
    INVENTORY_TYPE_BAG = 0x12,
    INVENTORY_TYPE_TABARD = 0x13,
    INVENTORY_TYPE_ROBE = 0x14,
    INVENTORY_TYPE_WEAPON_MAIN_HAND = 0x15,
    INVENTORY_TYPE_WEAPON_OFF_HAND = 0x16,
    INVENTORY_TYPE_HOLDABLE = 0x17,
    INVENTORY_TYPE_AMMO = 0x18,
    INVENTORY_TYPE_THROWN = 0x19,
    INVENTORY_TYPE_RANGED_RIGHT = 0x1A,
    INVENTORY_TYPE_QUIVER = 0x1B,
    INVENTORY_TYPE_RELIC = 0x1C,
} e_inventory_type;
static const value_string e_inventory_type_strings[] =  {
    { INVENTORY_TYPE_NON_EQUIP, "Non Equip" },
    { INVENTORY_TYPE_HEAD, "Head" },
    { INVENTORY_TYPE_NECK, "Neck" },
    { INVENTORY_TYPE_SHOULDERS, "Shoulders" },
    { INVENTORY_TYPE_BODY, "Body" },
    { INVENTORY_TYPE_CHEST, "Chest" },
    { INVENTORY_TYPE_WAIST, "Waist" },
    { INVENTORY_TYPE_LEGS, "Legs" },
    { INVENTORY_TYPE_FEET, "Feet" },
    { INVENTORY_TYPE_WRISTS, "Wrists" },
    { INVENTORY_TYPE_HANDS, "Hands" },
    { INVENTORY_TYPE_FINGER, "Finger" },
    { INVENTORY_TYPE_TRINKET, "Trinket" },
    { INVENTORY_TYPE_WEAPON, "Weapon" },
    { INVENTORY_TYPE_SHIELD, "Shield" },
    { INVENTORY_TYPE_RANGED, "Ranged" },
    { INVENTORY_TYPE_CLOAK, "Cloak" },
    { INVENTORY_TYPE_TWO_HANDED_WEAPON, "Two Handed Weapon" },
    { INVENTORY_TYPE_BAG, "Bag" },
    { INVENTORY_TYPE_TABARD, "Tabard" },
    { INVENTORY_TYPE_ROBE, "Robe" },
    { INVENTORY_TYPE_WEAPON_MAIN_HAND, "Weapon Main Hand" },
    { INVENTORY_TYPE_WEAPON_OFF_HAND, "Weapon Off Hand" },
    { INVENTORY_TYPE_HOLDABLE, "Holdable" },
    { INVENTORY_TYPE_AMMO, "Ammo" },
    { INVENTORY_TYPE_THROWN, "Thrown" },
    { INVENTORY_TYPE_RANGED_RIGHT, "Ranged Right" },
    { INVENTORY_TYPE_QUIVER, "Quiver" },
    { INVENTORY_TYPE_RELIC, "Relic" },
    { 0, NULL }
};

typedef enum {
    COMPRESSED_MOVE_OPCODE_SMSG_MONSTER_MOVE = 0x0DD,
    COMPRESSED_MOVE_OPCODE_SMSG_MONSTER_MOVE_TRANSPORT = 0x2AE,
    COMPRESSED_MOVE_OPCODE_SMSG_SPLINE_SET_RUN_SPEED = 0x2FE,
    COMPRESSED_MOVE_OPCODE_SMSG_SPLINE_MOVE_UNROOT = 0x304,
    COMPRESSED_MOVE_OPCODE_SMSG_SPLINE_MOVE_SET_RUN_MODE = 0x30D,
    COMPRESSED_MOVE_OPCODE_SMSG_SPLINE_MOVE_SET_WALK_MODE = 0x30E,
} e_compressed_move_opcode;
static const value_string e_compressed_move_opcode_strings[] =  {
    { COMPRESSED_MOVE_OPCODE_SMSG_MONSTER_MOVE, "Smsg Monster Move" },
    { COMPRESSED_MOVE_OPCODE_SMSG_MONSTER_MOVE_TRANSPORT, "Smsg Monster Move Transport" },
    { COMPRESSED_MOVE_OPCODE_SMSG_SPLINE_SET_RUN_SPEED, "Smsg Spline Set Run Speed" },
    { COMPRESSED_MOVE_OPCODE_SMSG_SPLINE_MOVE_UNROOT, "Smsg Spline Move Unroot" },
    { COMPRESSED_MOVE_OPCODE_SMSG_SPLINE_MOVE_SET_RUN_MODE, "Smsg Spline Move Set Run Mode" },
    { COMPRESSED_MOVE_OPCODE_SMSG_SPLINE_MOVE_SET_WALK_MODE, "Smsg Spline Move Set Walk Mode" },
    { 0, NULL }
};

typedef enum {
    FACTION_NONE = 0x000,
    FACTION_PLAYER_HUMAN = 0x001,
    FACTION_PLAYER_ORC = 0x002,
    FACTION_PLAYER_DWARF = 0x003,
    FACTION_PLAYER_NIGHT_ELF = 0x004,
    FACTION_PLAYER_UNDEAD = 0x005,
    FACTION_PLAYER_TAUREN = 0x006,
    FACTION_CREATURE = 0x007,
    FACTION_PLAYER_GNOME = 0x008,
    FACTION_PLAYER_TROLL = 0x009,
    FACTION_MONSTER = 0x00E,
    FACTION_DEFIAS_BROTHERHOOD = 0x00F,
    FACTION_GNOLL_RIVERPAW = 0x010,
    FACTION_GNOLL_REDRIDGE = 0x011,
    FACTION_GNOLL_SHADOWHIDE = 0x012,
    FACTION_MURLOC = 0x013,
    FACTION_UNDEAD_SCOURGE = 0x014,
    FACTION_BOOTY_BAY = 0x015,
    FACTION_BEAST_SPIDER = 0x016,
    FACTION_BEAST_BOAR = 0x017,
    FACTION_WORGEN = 0x018,
    FACTION_KOBOLD = 0x019,
    FACTION_TROLL_BLOODSCALP = 0x01A,
    FACTION_TROLL_SKULLSPLITTER = 0x01B,
    FACTION_PREY = 0x01C,
    FACTION_BEAST_WOLF = 0x01D,
    FACTION_DEFIAS_BROTHERHOOD_TRAITOR = 0x01E,
    FACTION_FRIENDLY = 0x01F,
    FACTION_TROGG = 0x020,
    FACTION_TROLL_FROSTMANE = 0x021,
    FACTION_ORC_BLACKROCK = 0x022,
    FACTION_VILLIAN = 0x023,
    FACTION_VICTIM = 0x024,
    FACTION_BEAST_BEAR = 0x025,
    FACTION_OGRE = 0x026,
    FACTION_KURZENS_MERCENARIES = 0x027,
    FACTION_ESCORTEE = 0x028,
    FACTION_VENTURE_COMPANY = 0x029,
    FACTION_BEAST_RAPTOR = 0x02A,
    FACTION_BASILISK = 0x02B,
    FACTION_DRAGONFLIGHT_GREEN = 0x02C,
    FACTION_LOST_ONES = 0x02D,
    FACTION_BLACKSMITHING_ARMORSMITHING = 0x02E,
    FACTION_IRONFORGE = 0x02F,
    FACTION_DARK_IRON_DWARVES = 0x030,
    FACTION_HUMAN_NIGHT_WATCH = 0x031,
    FACTION_DRAGONFLIGHT_RED = 0x032,
    FACTION_GNOLL_MOSSHIDE = 0x033,
    FACTION_ORC_DRAGONMAW = 0x034,
    FACTION_GNOME_LEPER = 0x035,
    FACTION_GNOMEREGAN_EXILES = 0x036,
    FACTION_LEOPARD = 0x037,
    FACTION_SCARLET_CRUSADE = 0x038,
    FACTION_GNOLL_ROTHIDE = 0x039,
    FACTION_BEAST_GORILLA = 0x03A,
    FACTION_THORIUM_BROTHERHOOD = 0x03B,
    FACTION_NAGA = 0x03C,
    FACTION_DALARAN = 0x03D,
    FACTION_FORLORN_SPIRIT = 0x03E,
    FACTION_DARKHOWL = 0x03F,
    FACTION_GRELL = 0x040,
    FACTION_FURBOLG = 0x041,
    FACTION_HORDE_GENERIC = 0x042,
    FACTION_HORDE = 0x043,
    FACTION_UNDERCITY = 0x044,
    FACTION_DARNASSUS = 0x045,
    FACTION_SYNDICATE = 0x046,
    FACTION_HILLSBRAD_MILITIA = 0x047,
    FACTION_STORMWIND = 0x048,
    FACTION_DEMON = 0x049,
    FACTION_ELEMENTAL = 0x04A,
    FACTION_SPIRIT = 0x04B,
    FACTION_ORGRIMMAR = 0x04C,
    FACTION_TREASURE = 0x04D,
    FACTION_GNOLL_MUDSNOUT = 0x04E,
    FACTION_HILLSBRAD_SOUTHSHORE_MAYOR = 0x04F,
    FACTION_DRAGONFLIGHT_BLACK = 0x050,
    FACTION_THUNDER_BLUFF = 0x051,
    FACTION_TROLL_WITHERBARK = 0x052,
    FACTION_LEATHERWORKING_ELEMENTAL = 0x053,
    FACTION_QUILBOAR_RAZORMANE = 0x054,
    FACTION_QUILBOAR_BRISTLEBACK = 0x055,
    FACTION_LEATHERWORKING_DRAGONSCALE = 0x056,
    FACTION_BLOODSAIL_BUCCANEERS = 0x057,
    FACTION_BLACKFATHOM = 0x058,
    FACTION_MAKRURA = 0x059,
    FACTION_CENTAUR_KOLKAR = 0x05A,
    FACTION_CENTAUR_GALAK = 0x05B,
    FACTION_GELKIS_CLAN_CENTAUR = 0x05C,
    FACTION_MAGRAM_CLAN_CENTAUR = 0x05D,
    FACTION_MARAUDINE = 0x05E,
    FACTION_THERAMORE = 0x06C,
    FACTION_QUILBOAR_RAZORFEN = 0x06D,
    FACTION_QUILBOAR_RAZORMANE_2 = 0x06E,
    FACTION_QUILBOAR_DEATHSHEAD = 0x06F,
    FACTION_ENEMY = 0x080,
    FACTION_AMBIENT = 0x094,
    FACTION_NETHERGARDE_CARAVAN = 0x0A8,
    FACTION_STEAMWHEEDLE_CARTEL = 0x0A9,
    FACTION_ALLIANCE_GENERIC = 0x0BD,
    FACTION_NETHERGARDE = 0x0D1,
    FACTION_WAILING_CAVERNS = 0x0E5,
    FACTION_SILITHID = 0x0F9,
    FACTION_SILVERMOON_REMNANT = 0x10D,
    FACTION_ZANDALAR_TRIBE = 0x10E,
    FACTION_BLACKSMITHING_WEAPONSMITHING = 0x121,
    FACTION_SCORPID = 0x135,
    FACTION_BEAST_BAT = 0x136,
    FACTION_TITAN = 0x137,
    FACTION_TASKMASTER_FIZZULE = 0x149,
    FACTION_RAVENHOLDT = 0x15D,
    FACTION_GADGETZAN = 0x171,
    FACTION_GNOMEREGAN_BUG = 0x185,
    FACTION_HARPY = 0x199,
    FACTION_BURNING_BLADE = 0x1AD,
    FACTION_SHADOWSILK_POACHER = 0x1C1,
    FACTION_SEARING_SPIDER = 0x1C2,
    FACTION_ALLIANCE = 0x1D5,
    FACTION_RATCHET = 0x1D6,
    FACTION_WILDHAMMER_CLAN = 0x1D7,
    FACTION_GOBLIN_DARK_IRON_BAR_PATRON = 0x1E9,
    FACTION_THE_LEAGUE_OF_ARATHOR = 0x1FD,
    FACTION_THE_DEFILERS = 0x1FE,
    FACTION_GIANT = 0x1FF,
    FACTION_ARGENT_DAWN = 0x211,
    FACTION_DARKSPEAR_TROLLS = 0x212,
    FACTION_DRAGONFLIGHT_BRONZE = 0x213,
    FACTION_DRAGONFLIGHT_BLUE = 0x214,
    FACTION_LEATHERWORKING_TRIBAL = 0x225,
    FACTION_ENGINEERING_GOBLIN = 0x226,
    FACTION_ENGINEERING_GNOME = 0x227,
    FACTION_BLACKSMITHING_HAMMERSMITHING = 0x239,
    FACTION_BLACKSMITHING_AXESMITHING = 0x23A,
    FACTION_BLACKSMITHING_SWORDSMITHING = 0x23B,
    FACTION_TROLL_VILEBRANCH = 0x23C,
    FACTION_SOUTHSEA_FREEBOOTERS = 0x23D,
    FACTION_CAER_DARROW = 0x23E,
    FACTION_FURBOLG_UNCORRUPTED = 0x23F,
    FACTION_TIMBERMAW_HOLD = 0x240,
    FACTION_EVERLOOK = 0x241,
    FACTION_WINTERSABER_TRAINERS = 0x24D,
    FACTION_CENARION_CIRCLE = 0x261,
    FACTION_SHATTERSPEAR_TROLLS = 0x275,
    FACTION_RAVASAUR_TRAINERS = 0x276,
    FACTION_MAJORDOMO_EXECUTUS = 0x289,
    FACTION_BEAST_CARRION_BIRD = 0x29D,
    FACTION_BEAST_CAT = 0x29E,
    FACTION_BEAST_CRAB = 0x29F,
    FACTION_BEAST_CROCILISK = 0x2A0,
    FACTION_BEAST_HYENA = 0x2A1,
    FACTION_BEAST_OWL = 0x2A2,
    FACTION_BEAST_SCORPID = 0x2A3,
    FACTION_BEAST_TALLSTRIDER = 0x2A4,
    FACTION_BEAST_TURTLE = 0x2A5,
    FACTION_BEAST_WIND_SERPENT = 0x2A6,
    FACTION_TRAINING_DUMMY = 0x2A7,
    FACTION_DRAGONFLIGHT_BLACK_BAIT = 0x2B1,
    FACTION_BATTLEGROUND_NEUTRAL = 0x2C5,
    FACTION_FROSTWOLF_CLAN = 0x2D9,
    FACTION_STORMPIKE_GUARD = 0x2DA,
    FACTION_HYDRAXIAN_WATERLORDS = 0x2ED,
    FACTION_SULFURON_FIRELORDS = 0x2EE,
    FACTION_GIZLOCKS_DUMMY = 0x301,
    FACTION_GIZLOCKS_CHARM = 0x302,
    FACTION_GIZLOCK = 0x303,
    FACTION_MORO_GAI = 0x315,
    FACTION_SPIRIT_GUIDE_ALLIANCE = 0x316,
    FACTION_SHEN_DRALAR = 0x329,
    FACTION_OGRE_CAPTAIN_KROMCRUSH = 0x33D,
    FACTION_SPIRIT_GUIDE_HORDE = 0x351,
    FACTION_JAEDENAR = 0x365,
    FACTION_WARSONG_OUTRIDERS = 0x379,
    FACTION_SILVERWING_SENTINELS = 0x37A,
    FACTION_ALLIANCE_FORCES = 0x37B,
    FACTION_HORDE_FORCES = 0x37C,
    FACTION_REVANTUSK_TROLLS = 0x37D,
    FACTION_DARKMOON_FAIRE = 0x38D,
    FACTION_BROOD_OF_NOZDORMU = 0x38E,
    FACTION_MIGHT_OF_KALIMDOR = 0x390,
    FACTION_ARMIES_OF_C_THUN = 0x393,
    FACTION_SILITHID_ATTACKERS = 0x394,
    FACTION_THE_IRONFORGE_BRIGADE = 0x395,
    FACTION_RC_ENEMIES = 0x396,
    FACTION_RC_OBJECTS = 0x397,
    FACTION_RED = 0x398,
    FACTION_BLUE = 0x399,
    FACTION_SCOURGE_INVADERS = 0x3A0,
    FACTION_TEST_FACTION_NOT_A_REAL_FACTION = 0x3A3,
    FACTION_TOWOW_FLAG = 0x3B6,
    FACTION_TOWOW_FLAG_TRIGGER_ALLIANCE_DND = 0x3B7,
    FACTION_TOWOW_FLAG_TRIGGER_HORDE_DND = 0x3BA,
} e_faction;
static const value_string e_faction_strings[] =  {
    { FACTION_NONE, "None" },
    { FACTION_PLAYER_HUMAN, "Player Human" },
    { FACTION_PLAYER_ORC, "Player Orc" },
    { FACTION_PLAYER_DWARF, "Player Dwarf" },
    { FACTION_PLAYER_NIGHT_ELF, "Player Night Elf" },
    { FACTION_PLAYER_UNDEAD, "Player Undead" },
    { FACTION_PLAYER_TAUREN, "Player Tauren" },
    { FACTION_CREATURE, "Creature" },
    { FACTION_PLAYER_GNOME, "Player Gnome" },
    { FACTION_PLAYER_TROLL, "Player Troll" },
    { FACTION_MONSTER, "Monster" },
    { FACTION_DEFIAS_BROTHERHOOD, "Defias Brotherhood" },
    { FACTION_GNOLL_RIVERPAW, "Gnoll Riverpaw" },
    { FACTION_GNOLL_REDRIDGE, "Gnoll Redridge" },
    { FACTION_GNOLL_SHADOWHIDE, "Gnoll Shadowhide" },
    { FACTION_MURLOC, "Murloc" },
    { FACTION_UNDEAD_SCOURGE, "Undead Scourge" },
    { FACTION_BOOTY_BAY, "Booty Bay" },
    { FACTION_BEAST_SPIDER, "Beast Spider" },
    { FACTION_BEAST_BOAR, "Beast Boar" },
    { FACTION_WORGEN, "Worgen" },
    { FACTION_KOBOLD, "Kobold" },
    { FACTION_TROLL_BLOODSCALP, "Troll Bloodscalp" },
    { FACTION_TROLL_SKULLSPLITTER, "Troll Skullsplitter" },
    { FACTION_PREY, "Prey" },
    { FACTION_BEAST_WOLF, "Beast Wolf" },
    { FACTION_DEFIAS_BROTHERHOOD_TRAITOR, "Defias Brotherhood Traitor" },
    { FACTION_FRIENDLY, "Friendly" },
    { FACTION_TROGG, "Trogg" },
    { FACTION_TROLL_FROSTMANE, "Troll Frostmane" },
    { FACTION_ORC_BLACKROCK, "Orc Blackrock" },
    { FACTION_VILLIAN, "Villian" },
    { FACTION_VICTIM, "Victim" },
    { FACTION_BEAST_BEAR, "Beast Bear" },
    { FACTION_OGRE, "Ogre" },
    { FACTION_KURZENS_MERCENARIES, "Kurzens Mercenaries" },
    { FACTION_ESCORTEE, "Escortee" },
    { FACTION_VENTURE_COMPANY, "Venture Company" },
    { FACTION_BEAST_RAPTOR, "Beast Raptor" },
    { FACTION_BASILISK, "Basilisk" },
    { FACTION_DRAGONFLIGHT_GREEN, "Dragonflight Green" },
    { FACTION_LOST_ONES, "Lost Ones" },
    { FACTION_BLACKSMITHING_ARMORSMITHING, "Blacksmithing Armorsmithing" },
    { FACTION_IRONFORGE, "Ironforge" },
    { FACTION_DARK_IRON_DWARVES, "Dark Iron Dwarves" },
    { FACTION_HUMAN_NIGHT_WATCH, "Human Night Watch" },
    { FACTION_DRAGONFLIGHT_RED, "Dragonflight Red" },
    { FACTION_GNOLL_MOSSHIDE, "Gnoll Mosshide" },
    { FACTION_ORC_DRAGONMAW, "Orc Dragonmaw" },
    { FACTION_GNOME_LEPER, "Gnome Leper" },
    { FACTION_GNOMEREGAN_EXILES, "Gnomeregan Exiles" },
    { FACTION_LEOPARD, "Leopard" },
    { FACTION_SCARLET_CRUSADE, "Scarlet Crusade" },
    { FACTION_GNOLL_ROTHIDE, "Gnoll Rothide" },
    { FACTION_BEAST_GORILLA, "Beast Gorilla" },
    { FACTION_THORIUM_BROTHERHOOD, "Thorium Brotherhood" },
    { FACTION_NAGA, "Naga" },
    { FACTION_DALARAN, "Dalaran" },
    { FACTION_FORLORN_SPIRIT, "Forlorn Spirit" },
    { FACTION_DARKHOWL, "Darkhowl" },
    { FACTION_GRELL, "Grell" },
    { FACTION_FURBOLG, "Furbolg" },
    { FACTION_HORDE_GENERIC, "Horde Generic" },
    { FACTION_HORDE, "Horde" },
    { FACTION_UNDERCITY, "Undercity" },
    { FACTION_DARNASSUS, "Darnassus" },
    { FACTION_SYNDICATE, "Syndicate" },
    { FACTION_HILLSBRAD_MILITIA, "Hillsbrad Militia" },
    { FACTION_STORMWIND, "Stormwind" },
    { FACTION_DEMON, "Demon" },
    { FACTION_ELEMENTAL, "Elemental" },
    { FACTION_SPIRIT, "Spirit" },
    { FACTION_ORGRIMMAR, "Orgrimmar" },
    { FACTION_TREASURE, "Treasure" },
    { FACTION_GNOLL_MUDSNOUT, "Gnoll Mudsnout" },
    { FACTION_HILLSBRAD_SOUTHSHORE_MAYOR, "Hillsbrad Southshore Mayor" },
    { FACTION_DRAGONFLIGHT_BLACK, "Dragonflight Black" },
    { FACTION_THUNDER_BLUFF, "Thunder Bluff" },
    { FACTION_TROLL_WITHERBARK, "Troll Witherbark" },
    { FACTION_LEATHERWORKING_ELEMENTAL, "Leatherworking Elemental" },
    { FACTION_QUILBOAR_RAZORMANE, "Quilboar Razormane" },
    { FACTION_QUILBOAR_BRISTLEBACK, "Quilboar Bristleback" },
    { FACTION_LEATHERWORKING_DRAGONSCALE, "Leatherworking Dragonscale" },
    { FACTION_BLOODSAIL_BUCCANEERS, "Bloodsail Buccaneers" },
    { FACTION_BLACKFATHOM, "Blackfathom" },
    { FACTION_MAKRURA, "Makrura" },
    { FACTION_CENTAUR_KOLKAR, "Centaur Kolkar" },
    { FACTION_CENTAUR_GALAK, "Centaur Galak" },
    { FACTION_GELKIS_CLAN_CENTAUR, "Gelkis Clan Centaur" },
    { FACTION_MAGRAM_CLAN_CENTAUR, "Magram Clan Centaur" },
    { FACTION_MARAUDINE, "Maraudine" },
    { FACTION_THERAMORE, "Theramore" },
    { FACTION_QUILBOAR_RAZORFEN, "Quilboar Razorfen" },
    { FACTION_QUILBOAR_RAZORMANE_2, "Quilboar Razormane 2" },
    { FACTION_QUILBOAR_DEATHSHEAD, "Quilboar Deathshead" },
    { FACTION_ENEMY, "Enemy" },
    { FACTION_AMBIENT, "Ambient" },
    { FACTION_NETHERGARDE_CARAVAN, "Nethergarde Caravan" },
    { FACTION_STEAMWHEEDLE_CARTEL, "Steamwheedle Cartel" },
    { FACTION_ALLIANCE_GENERIC, "Alliance Generic" },
    { FACTION_NETHERGARDE, "Nethergarde" },
    { FACTION_WAILING_CAVERNS, "Wailing Caverns" },
    { FACTION_SILITHID, "Silithid" },
    { FACTION_SILVERMOON_REMNANT, "Silvermoon Remnant" },
    { FACTION_ZANDALAR_TRIBE, "Zandalar Tribe" },
    { FACTION_BLACKSMITHING_WEAPONSMITHING, "Blacksmithing Weaponsmithing" },
    { FACTION_SCORPID, "Scorpid" },
    { FACTION_BEAST_BAT, "Beast Bat" },
    { FACTION_TITAN, "Titan" },
    { FACTION_TASKMASTER_FIZZULE, "Taskmaster Fizzule" },
    { FACTION_RAVENHOLDT, "Ravenholdt" },
    { FACTION_GADGETZAN, "Gadgetzan" },
    { FACTION_GNOMEREGAN_BUG, "Gnomeregan Bug" },
    { FACTION_HARPY, "Harpy" },
    { FACTION_BURNING_BLADE, "Burning Blade" },
    { FACTION_SHADOWSILK_POACHER, "Shadowsilk Poacher" },
    { FACTION_SEARING_SPIDER, "Searing Spider" },
    { FACTION_ALLIANCE, "Alliance" },
    { FACTION_RATCHET, "Ratchet" },
    { FACTION_WILDHAMMER_CLAN, "Wildhammer Clan" },
    { FACTION_GOBLIN_DARK_IRON_BAR_PATRON, "Goblin Dark Iron Bar Patron" },
    { FACTION_THE_LEAGUE_OF_ARATHOR, "The League Of Arathor" },
    { FACTION_THE_DEFILERS, "The Defilers" },
    { FACTION_GIANT, "Giant" },
    { FACTION_ARGENT_DAWN, "Argent Dawn" },
    { FACTION_DARKSPEAR_TROLLS, "Darkspear Trolls" },
    { FACTION_DRAGONFLIGHT_BRONZE, "Dragonflight Bronze" },
    { FACTION_DRAGONFLIGHT_BLUE, "Dragonflight Blue" },
    { FACTION_LEATHERWORKING_TRIBAL, "Leatherworking Tribal" },
    { FACTION_ENGINEERING_GOBLIN, "Engineering Goblin" },
    { FACTION_ENGINEERING_GNOME, "Engineering Gnome" },
    { FACTION_BLACKSMITHING_HAMMERSMITHING, "Blacksmithing Hammersmithing" },
    { FACTION_BLACKSMITHING_AXESMITHING, "Blacksmithing Axesmithing" },
    { FACTION_BLACKSMITHING_SWORDSMITHING, "Blacksmithing Swordsmithing" },
    { FACTION_TROLL_VILEBRANCH, "Troll Vilebranch" },
    { FACTION_SOUTHSEA_FREEBOOTERS, "Southsea Freebooters" },
    { FACTION_CAER_DARROW, "Caer Darrow" },
    { FACTION_FURBOLG_UNCORRUPTED, "Furbolg Uncorrupted" },
    { FACTION_TIMBERMAW_HOLD, "Timbermaw Hold" },
    { FACTION_EVERLOOK, "Everlook" },
    { FACTION_WINTERSABER_TRAINERS, "Wintersaber Trainers" },
    { FACTION_CENARION_CIRCLE, "Cenarion Circle" },
    { FACTION_SHATTERSPEAR_TROLLS, "Shatterspear Trolls" },
    { FACTION_RAVASAUR_TRAINERS, "Ravasaur Trainers" },
    { FACTION_MAJORDOMO_EXECUTUS, "Majordomo Executus" },
    { FACTION_BEAST_CARRION_BIRD, "Beast Carrion Bird" },
    { FACTION_BEAST_CAT, "Beast Cat" },
    { FACTION_BEAST_CRAB, "Beast Crab" },
    { FACTION_BEAST_CROCILISK, "Beast Crocilisk" },
    { FACTION_BEAST_HYENA, "Beast Hyena" },
    { FACTION_BEAST_OWL, "Beast Owl" },
    { FACTION_BEAST_SCORPID, "Beast Scorpid" },
    { FACTION_BEAST_TALLSTRIDER, "Beast Tallstrider" },
    { FACTION_BEAST_TURTLE, "Beast Turtle" },
    { FACTION_BEAST_WIND_SERPENT, "Beast Wind Serpent" },
    { FACTION_TRAINING_DUMMY, "Training Dummy" },
    { FACTION_DRAGONFLIGHT_BLACK_BAIT, "Dragonflight Black Bait" },
    { FACTION_BATTLEGROUND_NEUTRAL, "Battleground Neutral" },
    { FACTION_FROSTWOLF_CLAN, "Frostwolf Clan" },
    { FACTION_STORMPIKE_GUARD, "Stormpike Guard" },
    { FACTION_HYDRAXIAN_WATERLORDS, "Hydraxian Waterlords" },
    { FACTION_SULFURON_FIRELORDS, "Sulfuron Firelords" },
    { FACTION_GIZLOCKS_DUMMY, "Gizlocks Dummy" },
    { FACTION_GIZLOCKS_CHARM, "Gizlocks Charm" },
    { FACTION_GIZLOCK, "Gizlock" },
    { FACTION_MORO_GAI, "Moro Gai" },
    { FACTION_SPIRIT_GUIDE_ALLIANCE, "Spirit Guide Alliance" },
    { FACTION_SHEN_DRALAR, "Shen Dralar" },
    { FACTION_OGRE_CAPTAIN_KROMCRUSH, "Ogre Captain Kromcrush" },
    { FACTION_SPIRIT_GUIDE_HORDE, "Spirit Guide Horde" },
    { FACTION_JAEDENAR, "Jaedenar" },
    { FACTION_WARSONG_OUTRIDERS, "Warsong Outriders" },
    { FACTION_SILVERWING_SENTINELS, "Silverwing Sentinels" },
    { FACTION_ALLIANCE_FORCES, "Alliance Forces" },
    { FACTION_HORDE_FORCES, "Horde Forces" },
    { FACTION_REVANTUSK_TROLLS, "Revantusk Trolls" },
    { FACTION_DARKMOON_FAIRE, "Darkmoon Faire" },
    { FACTION_BROOD_OF_NOZDORMU, "Brood Of Nozdormu" },
    { FACTION_MIGHT_OF_KALIMDOR, "Might Of Kalimdor" },
    { FACTION_ARMIES_OF_C_THUN, "Armies Of C Thun" },
    { FACTION_SILITHID_ATTACKERS, "Silithid Attackers" },
    { FACTION_THE_IRONFORGE_BRIGADE, "The Ironforge Brigade" },
    { FACTION_RC_ENEMIES, "Rc Enemies" },
    { FACTION_RC_OBJECTS, "Rc Objects" },
    { FACTION_RED, "Red" },
    { FACTION_BLUE, "Blue" },
    { FACTION_SCOURGE_INVADERS, "Scourge Invaders" },
    { FACTION_TEST_FACTION_NOT_A_REAL_FACTION, "Test Faction Not A Real Faction" },
    { FACTION_TOWOW_FLAG, "Towow Flag" },
    { FACTION_TOWOW_FLAG_TRIGGER_ALLIANCE_DND, "Towow Flag Trigger Alliance Dnd" },
    { FACTION_TOWOW_FLAG_TRIGGER_HORDE_DND, "Towow Flag Trigger Horde Dnd" },
    { 0, NULL }
};

typedef enum {
    FRIEND_STATUS_OFFLINE = 0x0,
    FRIEND_STATUS_ONLINE = 0x1,
    FRIEND_STATUS_AFK = 0x2,
    FRIEND_STATUS_UNKNOWN3 = 0x3,
    FRIEND_STATUS_DND = 0x4,
} e_friend_status;
static const value_string e_friend_status_strings[] =  {
    { FRIEND_STATUS_OFFLINE, "Offline" },
    { FRIEND_STATUS_ONLINE, "Online" },
    { FRIEND_STATUS_AFK, "Afk" },
    { FRIEND_STATUS_UNKNOWN3, "Unknown3" },
    { FRIEND_STATUS_DND, "Dnd" },
    { 0, NULL }
};

typedef enum {
    GUILD_MEMBER_STATUS_OFFLINE = 0x0,
    GUILD_MEMBER_STATUS_ONLINE = 0x1,
} e_guild_member_status;
static const value_string e_guild_member_status_strings[] =  {
    { GUILD_MEMBER_STATUS_OFFLINE, "Offline" },
    { GUILD_MEMBER_STATUS_ONLINE, "Online" },
    { 0, NULL }
};

typedef enum {
    SPELL_TRIGGER_TYPE_ON_USE = 0x0,
    SPELL_TRIGGER_TYPE_ON_EQUIP = 0x1,
    SPELL_TRIGGER_TYPE_CHANCE_ON_HIT = 0x2,
} e_spell_trigger_type;
static const value_string e_spell_trigger_type_strings[] =  {
    { SPELL_TRIGGER_TYPE_ON_USE, "On Use" },
    { SPELL_TRIGGER_TYPE_ON_EQUIP, "On Equip" },
    { SPELL_TRIGGER_TYPE_CHANCE_ON_HIT, "Chance On Hit" },
    { 0, NULL }
};

typedef enum {
    ITEM_STAT_TYPE_MANA = 0x0,
    ITEM_STAT_TYPE_HEALTH = 0x1,
    ITEM_STAT_TYPE_AGILITY = 0x3,
    ITEM_STAT_TYPE_STRENGTH = 0x4,
    ITEM_STAT_TYPE_INTELLECT = 0x5,
    ITEM_STAT_TYPE_SPIRIT = 0x6,
    ITEM_STAT_TYPE_STAMINA = 0x7,
} e_item_stat_type;
static const value_string e_item_stat_type_strings[] =  {
    { ITEM_STAT_TYPE_MANA, "Mana" },
    { ITEM_STAT_TYPE_HEALTH, "Health" },
    { ITEM_STAT_TYPE_AGILITY, "Agility" },
    { ITEM_STAT_TYPE_STRENGTH, "Strength" },
    { ITEM_STAT_TYPE_INTELLECT, "Intellect" },
    { ITEM_STAT_TYPE_SPIRIT, "Spirit" },
    { ITEM_STAT_TYPE_STAMINA, "Stamina" },
    { 0, NULL }
};

typedef enum {
    LOOT_SLOT_TYPE_TYPE_ALLOW_LOOT = 0x0,
    LOOT_SLOT_TYPE_TYPE_ROLL_ONGOING = 0x1,
    LOOT_SLOT_TYPE_TYPE_MASTER = 0x2,
    LOOT_SLOT_TYPE_TYPE_LOCKED = 0x3,
    LOOT_SLOT_TYPE_TYPE_OWNER = 0x4,
} e_loot_slot_type;
static const value_string e_loot_slot_type_strings[] =  {
    { LOOT_SLOT_TYPE_TYPE_ALLOW_LOOT, "Type Allow Loot" },
    { LOOT_SLOT_TYPE_TYPE_ROLL_ONGOING, "Type Roll Ongoing" },
    { LOOT_SLOT_TYPE_TYPE_MASTER, "Type Master" },
    { LOOT_SLOT_TYPE_TYPE_LOCKED, "Type Locked" },
    { LOOT_SLOT_TYPE_TYPE_OWNER, "Type Owner" },
    { 0, NULL }
};

typedef enum {
    MAIL_TYPE_NORMAL = 0x0,
    MAIL_TYPE_AUCTION = 0x2,
    MAIL_TYPE_CREATURE = 0x3,
    MAIL_TYPE_GAMEOBJECT = 0x4,
    MAIL_TYPE_ITEM = 0x5,
} e_mail_type;
static const value_string e_mail_type_strings[] =  {
    { MAIL_TYPE_NORMAL, "Normal" },
    { MAIL_TYPE_AUCTION, "Auction" },
    { MAIL_TYPE_CREATURE, "Creature" },
    { MAIL_TYPE_GAMEOBJECT, "Gameobject" },
    { MAIL_TYPE_ITEM, "Item" },
    { 0, NULL }
};

typedef enum {
    MONSTER_MOVE_TYPE_NORMAL = 0x0,
    MONSTER_MOVE_TYPE_STOP = 0x1,
    MONSTER_MOVE_TYPE_FACING_SPOT = 0x2,
    MONSTER_MOVE_TYPE_FACING_TARGET = 0x3,
    MONSTER_MOVE_TYPE_FACING_ANGLE = 0x4,
} e_monster_move_type;
static const value_string e_monster_move_type_strings[] =  {
    { MONSTER_MOVE_TYPE_NORMAL, "Normal" },
    { MONSTER_MOVE_TYPE_STOP, "Stop" },
    { MONSTER_MOVE_TYPE_FACING_SPOT, "Facing Spot" },
    { MONSTER_MOVE_TYPE_FACING_TARGET, "Facing Target" },
    { MONSTER_MOVE_TYPE_FACING_ANGLE, "Facing Angle" },
    { 0, NULL }
};

typedef enum {
    LANGUAGE_UNIVERSAL = 0x00000000,
    LANGUAGE_ORCISH = 0x00000001,
    LANGUAGE_DARNASSIAN = 0x00000002,
    LANGUAGE_TAURAHE = 0x00000003,
    LANGUAGE_DWARVISH = 0x00000006,
    LANGUAGE_COMMON = 0x00000007,
    LANGUAGE_DEMONIC = 0x00000008,
    LANGUAGE_TITAN = 0x00000009,
    LANGUAGE_THALASSIAN = 0x0000000A,
    LANGUAGE_DRACONIC = 0x0000000B,
    LANGUAGE_KALIMAG = 0x0000000C,
    LANGUAGE_GNOMISH = 0x0000000D,
    LANGUAGE_TROLL = 0x0000000E,
    LANGUAGE_GUTTERSPEAK = 0x00000021,
    LANGUAGE_ADDON = -0x0000001,
} e_language;
static const value_string e_language_strings[] =  {
    { LANGUAGE_UNIVERSAL, "Universal" },
    { LANGUAGE_ORCISH, "Orcish" },
    { LANGUAGE_DARNASSIAN, "Darnassian" },
    { LANGUAGE_TAURAHE, "Taurahe" },
    { LANGUAGE_DWARVISH, "Dwarvish" },
    { LANGUAGE_COMMON, "Common" },
    { LANGUAGE_DEMONIC, "Demonic" },
    { LANGUAGE_TITAN, "Titan" },
    { LANGUAGE_THALASSIAN, "Thalassian" },
    { LANGUAGE_DRACONIC, "Draconic" },
    { LANGUAGE_KALIMAG, "Kalimag" },
    { LANGUAGE_GNOMISH, "Gnomish" },
    { LANGUAGE_TROLL, "Troll" },
    { LANGUAGE_GUTTERSPEAK, "Gutterspeak" },
    { LANGUAGE_ADDON, "Addon" },
    { 0, NULL }
};

typedef enum {
    UPDATE_TYPE_VALUES = 0x0,
    UPDATE_TYPE_MOVEMENT = 0x1,
    UPDATE_TYPE_CREATE_OBJECT = 0x2,
    UPDATE_TYPE_CREATE_OBJECT2 = 0x3,
    UPDATE_TYPE_OUT_OF_RANGE_OBJECTS = 0x4,
    UPDATE_TYPE_NEAR_OBJECTS = 0x5,
} e_update_type;
static const value_string e_update_type_strings[] =  {
    { UPDATE_TYPE_VALUES, "Values" },
    { UPDATE_TYPE_MOVEMENT, "Movement" },
    { UPDATE_TYPE_CREATE_OBJECT, "Create Object" },
    { UPDATE_TYPE_CREATE_OBJECT2, "Create Object2" },
    { UPDATE_TYPE_OUT_OF_RANGE_OBJECTS, "Out Of Range Objects" },
    { UPDATE_TYPE_NEAR_OBJECTS, "Near Objects" },
    { 0, NULL }
};

typedef enum {
    OBJECT_TYPE_OBJECT = 0x0,
    OBJECT_TYPE_ITEM = 0x1,
    OBJECT_TYPE_CONTAINER = 0x2,
    OBJECT_TYPE_UNIT = 0x3,
    OBJECT_TYPE_PLAYER = 0x4,
    OBJECT_TYPE_GAME_OBJECT = 0x5,
    OBJECT_TYPE_DYNAMIC_OBJECT = 0x6,
    OBJECT_TYPE_CORPSE = 0x7,
} e_object_type;
static const value_string e_object_type_strings[] =  {
    { OBJECT_TYPE_OBJECT, "Object" },
    { OBJECT_TYPE_ITEM, "Item" },
    { OBJECT_TYPE_CONTAINER, "Container" },
    { OBJECT_TYPE_UNIT, "Unit" },
    { OBJECT_TYPE_PLAYER, "Player" },
    { OBJECT_TYPE_GAME_OBJECT, "Game Object" },
    { OBJECT_TYPE_DYNAMIC_OBJECT, "Dynamic Object" },
    { OBJECT_TYPE_CORPSE, "Corpse" },
    { 0, NULL }
};

typedef enum {
    RAID_TARGET_INDEX_UNKNOWN0 = 0x00,
    RAID_TARGET_INDEX_UNKNOWN1 = 0x01,
    RAID_TARGET_INDEX_UNKNOWN2 = 0x02,
    RAID_TARGET_INDEX_UNKNOWN3 = 0x03,
    RAID_TARGET_INDEX_UNKNOWN4 = 0x04,
    RAID_TARGET_INDEX_UNKNOWN5 = 0x05,
    RAID_TARGET_INDEX_UNKNOWN6 = 0x06,
    RAID_TARGET_INDEX_UNKNOWN7 = 0x07,
    RAID_TARGET_INDEX_UNKNOWN8 = 0x08,
    RAID_TARGET_INDEX_REQUEST_ICONS = 0xFF,
} e_raid_target_index;
static const value_string e_raid_target_index_strings[] =  {
    { RAID_TARGET_INDEX_UNKNOWN0, "Unknown0" },
    { RAID_TARGET_INDEX_UNKNOWN1, "Unknown1" },
    { RAID_TARGET_INDEX_UNKNOWN2, "Unknown2" },
    { RAID_TARGET_INDEX_UNKNOWN3, "Unknown3" },
    { RAID_TARGET_INDEX_UNKNOWN4, "Unknown4" },
    { RAID_TARGET_INDEX_UNKNOWN5, "Unknown5" },
    { RAID_TARGET_INDEX_UNKNOWN6, "Unknown6" },
    { RAID_TARGET_INDEX_UNKNOWN7, "Unknown7" },
    { RAID_TARGET_INDEX_UNKNOWN8, "Unknown8" },
    { RAID_TARGET_INDEX_REQUEST_ICONS, "Request Icons" },
    { 0, NULL }
};

typedef enum {
    SPELL_EFFECT_NONE = 0x00,
    SPELL_EFFECT_INSTAKILL = 0x01,
    SPELL_EFFECT_SCHOOL_DAMAGE = 0x02,
    SPELL_EFFECT_DUMMY = 0x03,
    SPELL_EFFECT_PORTAL_TELEPORT = 0x04,
    SPELL_EFFECT_TELEPORT_UNITS = 0x05,
    SPELL_EFFECT_APPLY_AURA = 0x06,
    SPELL_EFFECT_ENVIRONMENTAL_DAMAGE = 0x07,
    SPELL_EFFECT_POWER_DRAIN = 0x08,
    SPELL_EFFECT_HEALTH_LEECH = 0x09,
    SPELL_EFFECT_HEAL = 0x0A,
    SPELL_EFFECT_BIND = 0x0B,
    SPELL_EFFECT_PORTAL = 0x0C,
    SPELL_EFFECT_RITUAL_BASE = 0x0D,
    SPELL_EFFECT_RITUAL_SPECIALIZE = 0x0E,
    SPELL_EFFECT_RITUAL_ACTIVATE_PORTAL = 0x0F,
    SPELL_EFFECT_QUEST_COMPLETE = 0x10,
    SPELL_EFFECT_WEAPON_DAMAGE_NOSCHOOL = 0x11,
    SPELL_EFFECT_RESURRECT = 0x12,
    SPELL_EFFECT_ADD_EXTRA_ATTACKS = 0x13,
    SPELL_EFFECT_DODGE = 0x14,
    SPELL_EFFECT_EVADE = 0x15,
    SPELL_EFFECT_PARRY = 0x16,
    SPELL_EFFECT_BLOCK = 0x17,
    SPELL_EFFECT_CREATE_ITEM = 0x18,
    SPELL_EFFECT_WEAPON = 0x19,
    SPELL_EFFECT_DEFENSE = 0x1A,
    SPELL_EFFECT_PERSISTENT_AREA_AURA = 0x1B,
    SPELL_EFFECT_SUMMON = 0x1C,
    SPELL_EFFECT_LEAP = 0x1D,
    SPELL_EFFECT_ENERGIZE = 0x1E,
    SPELL_EFFECT_WEAPON_PERCENT_DAMAGE = 0x1F,
    SPELL_EFFECT_TRIGGER_MISSILE = 0x20,
    SPELL_EFFECT_OPEN_LOCK = 0x21,
    SPELL_EFFECT_SUMMON_CHANGE_ITEM = 0x22,
    SPELL_EFFECT_APPLY_AREA_AURA_PARTY = 0x23,
    SPELL_EFFECT_LEARN_SPELL = 0x24,
    SPELL_EFFECT_SPELL_DEFENSE = 0x25,
    SPELL_EFFECT_DISPEL = 0x26,
    SPELL_EFFECT_LANGUAGE = 0x27,
    SPELL_EFFECT_DUAL_WIELD = 0x28,
    SPELL_EFFECT_SUMMON_WILD = 0x29,
    SPELL_EFFECT_SUMMON_GUARDIAN = 0x2A,
    SPELL_EFFECT_TELEPORT_UNITS_FACE_CASTER = 0x2B,
    SPELL_EFFECT_SKILL_STEP = 0x2C,
    SPELL_EFFECT_ADD_HONOR = 0x2D,
    SPELL_EFFECT_SPAWN = 0x2E,
    SPELL_EFFECT_TRADE_SKILL = 0x2F,
    SPELL_EFFECT_STEALTH = 0x30,
    SPELL_EFFECT_DETECT = 0x31,
    SPELL_EFFECT_TRANS_DOOR = 0x32,
    SPELL_EFFECT_FORCE_CRITICAL_HIT = 0x33,
    SPELL_EFFECT_GUARANTEE_HIT = 0x34,
    SPELL_EFFECT_ENCHANT_ITEM = 0x35,
    SPELL_EFFECT_ENCHANT_ITEM_TEMPORARY = 0x36,
    SPELL_EFFECT_TAMECREATURE = 0x37,
    SPELL_EFFECT_SUMMON_PET = 0x38,
    SPELL_EFFECT_LEARN_PET_SPELL = 0x39,
    SPELL_EFFECT_WEAPON_DAMAGE = 0x3A,
    SPELL_EFFECT_OPEN_LOCK_ITEM = 0x3B,
    SPELL_EFFECT_PROFICIENCY = 0x3C,
    SPELL_EFFECT_SEND_EVENT = 0x3D,
    SPELL_EFFECT_POWER_BURN = 0x3E,
    SPELL_EFFECT_THREAT = 0x3F,
    SPELL_EFFECT_TRIGGER_SPELL = 0x40,
    SPELL_EFFECT_HEALTH_FUNNEL = 0x41,
    SPELL_EFFECT_POWER_FUNNEL = 0x42,
    SPELL_EFFECT_HEAL_MAX_HEALTH = 0x43,
    SPELL_EFFECT_INTERRUPT_CAST = 0x44,
    SPELL_EFFECT_DISTRACT = 0x45,
    SPELL_EFFECT_PULL = 0x46,
    SPELL_EFFECT_PICKPOCKET = 0x47,
    SPELL_EFFECT_ADD_FARSIGHT = 0x48,
    SPELL_EFFECT_SUMMON_POSSESSED = 0x49,
    SPELL_EFFECT_SUMMON_TOTEM = 0x4A,
    SPELL_EFFECT_HEAL_MECHANICAL = 0x4B,
    SPELL_EFFECT_SUMMON_OBJECT_WILD = 0x4C,
    SPELL_EFFECT_SCRIPT_EFFECT = 0x4D,
    SPELL_EFFECT_ATTACK = 0x4E,
    SPELL_EFFECT_SANCTUARY = 0x4F,
    SPELL_EFFECT_ADD_COMBO_POINTS = 0x50,
    SPELL_EFFECT_CREATE_HOUSE = 0x51,
    SPELL_EFFECT_BIND_SIGHT = 0x52,
    SPELL_EFFECT_DUEL = 0x53,
    SPELL_EFFECT_STUCK = 0x54,
    SPELL_EFFECT_SUMMON_PLAYER = 0x55,
    SPELL_EFFECT_ACTIVATE_OBJECT = 0x56,
    SPELL_EFFECT_SUMMON_TOTEM_SLOT1 = 0x57,
    SPELL_EFFECT_SUMMON_TOTEM_SLOT2 = 0x58,
    SPELL_EFFECT_SUMMON_TOTEM_SLOT3 = 0x59,
    SPELL_EFFECT_SUMMON_TOTEM_SLOT4 = 0x5A,
    SPELL_EFFECT_THREAT_ALL = 0x5B,
    SPELL_EFFECT_ENCHANT_HELD_ITEM = 0x5C,
    SPELL_EFFECT_SUMMON_PHANTASM = 0x5D,
    SPELL_EFFECT_SELF_RESURRECT = 0x5E,
    SPELL_EFFECT_SKINNING = 0x5F,
    SPELL_EFFECT_CHARGE = 0x60,
    SPELL_EFFECT_SUMMON_CRITTER = 0x61,
    SPELL_EFFECT_KNOCK_BACK = 0x62,
    SPELL_EFFECT_DISENCHANT = 0x63,
    SPELL_EFFECT_INEBRIATE = 0x64,
    SPELL_EFFECT_FEED_PET = 0x65,
    SPELL_EFFECT_DISMISS_PET = 0x66,
    SPELL_EFFECT_REPUTATION = 0x67,
    SPELL_EFFECT_SUMMON_OBJECT_SLOT1 = 0x68,
    SPELL_EFFECT_SUMMON_OBJECT_SLOT2 = 0x69,
    SPELL_EFFECT_SUMMON_OBJECT_SLOT3 = 0x6A,
    SPELL_EFFECT_SUMMON_OBJECT_SLOT4 = 0x6B,
    SPELL_EFFECT_DISPEL_MECHANIC = 0x6C,
    SPELL_EFFECT_SUMMON_DEAD_PET = 0x6D,
    SPELL_EFFECT_DESTROY_ALL_TOTEMS = 0x6E,
    SPELL_EFFECT_DURABILITY_DAMAGE = 0x6F,
    SPELL_EFFECT_SUMMON_DEMON = 0x70,
    SPELL_EFFECT_RESURRECT_NEW = 0x71,
    SPELL_EFFECT_ATTACK_ME = 0x72,
    SPELL_EFFECT_DURABILITY_DAMAGE_PCT = 0x73,
    SPELL_EFFECT_SKIN_PLAYER_CORPSE = 0x74,
    SPELL_EFFECT_SPIRIT_HEAL = 0x75,
    SPELL_EFFECT_SKILL = 0x76,
    SPELL_EFFECT_APPLY_AREA_AURA_PET = 0x77,
    SPELL_EFFECT_TELEPORT_GRAVEYARD = 0x78,
    SPELL_EFFECT_NORMALIZED_WEAPON_DMG = 0x79,
    SPELL_EFFECT_UNKNOWN122 = 0x7A,
    SPELL_EFFECT_SEND_TAXI = 0x7B,
    SPELL_EFFECT_PLAYER_PULL = 0x7C,
    SPELL_EFFECT_MODIFY_THREAT_PERCENT = 0x7D,
    SPELL_EFFECT_UNKNOWN126 = 0x7E,
    SPELL_EFFECT_UNKNOWN127 = 0x7F,
} e_spell_effect;
static const value_string e_spell_effect_strings[] =  {
    { SPELL_EFFECT_NONE, "None" },
    { SPELL_EFFECT_INSTAKILL, "Instakill" },
    { SPELL_EFFECT_SCHOOL_DAMAGE, "School Damage" },
    { SPELL_EFFECT_DUMMY, "Dummy" },
    { SPELL_EFFECT_PORTAL_TELEPORT, "Portal Teleport" },
    { SPELL_EFFECT_TELEPORT_UNITS, "Teleport Units" },
    { SPELL_EFFECT_APPLY_AURA, "Apply Aura" },
    { SPELL_EFFECT_ENVIRONMENTAL_DAMAGE, "Environmental Damage" },
    { SPELL_EFFECT_POWER_DRAIN, "Power Drain" },
    { SPELL_EFFECT_HEALTH_LEECH, "Health Leech" },
    { SPELL_EFFECT_HEAL, "Heal" },
    { SPELL_EFFECT_BIND, "Bind" },
    { SPELL_EFFECT_PORTAL, "Portal" },
    { SPELL_EFFECT_RITUAL_BASE, "Ritual Base" },
    { SPELL_EFFECT_RITUAL_SPECIALIZE, "Ritual Specialize" },
    { SPELL_EFFECT_RITUAL_ACTIVATE_PORTAL, "Ritual Activate Portal" },
    { SPELL_EFFECT_QUEST_COMPLETE, "Quest Complete" },
    { SPELL_EFFECT_WEAPON_DAMAGE_NOSCHOOL, "Weapon Damage Noschool" },
    { SPELL_EFFECT_RESURRECT, "Resurrect" },
    { SPELL_EFFECT_ADD_EXTRA_ATTACKS, "Add Extra Attacks" },
    { SPELL_EFFECT_DODGE, "Dodge" },
    { SPELL_EFFECT_EVADE, "Evade" },
    { SPELL_EFFECT_PARRY, "Parry" },
    { SPELL_EFFECT_BLOCK, "Block" },
    { SPELL_EFFECT_CREATE_ITEM, "Create Item" },
    { SPELL_EFFECT_WEAPON, "Weapon" },
    { SPELL_EFFECT_DEFENSE, "Defense" },
    { SPELL_EFFECT_PERSISTENT_AREA_AURA, "Persistent Area Aura" },
    { SPELL_EFFECT_SUMMON, "Summon" },
    { SPELL_EFFECT_LEAP, "Leap" },
    { SPELL_EFFECT_ENERGIZE, "Energize" },
    { SPELL_EFFECT_WEAPON_PERCENT_DAMAGE, "Weapon Percent Damage" },
    { SPELL_EFFECT_TRIGGER_MISSILE, "Trigger Missile" },
    { SPELL_EFFECT_OPEN_LOCK, "Open Lock" },
    { SPELL_EFFECT_SUMMON_CHANGE_ITEM, "Summon Change Item" },
    { SPELL_EFFECT_APPLY_AREA_AURA_PARTY, "Apply Area Aura Party" },
    { SPELL_EFFECT_LEARN_SPELL, "Learn Spell" },
    { SPELL_EFFECT_SPELL_DEFENSE, "Spell Defense" },
    { SPELL_EFFECT_DISPEL, "Dispel" },
    { SPELL_EFFECT_LANGUAGE, "Language" },
    { SPELL_EFFECT_DUAL_WIELD, "Dual Wield" },
    { SPELL_EFFECT_SUMMON_WILD, "Summon Wild" },
    { SPELL_EFFECT_SUMMON_GUARDIAN, "Summon Guardian" },
    { SPELL_EFFECT_TELEPORT_UNITS_FACE_CASTER, "Teleport Units Face Caster" },
    { SPELL_EFFECT_SKILL_STEP, "Skill Step" },
    { SPELL_EFFECT_ADD_HONOR, "Add Honor" },
    { SPELL_EFFECT_SPAWN, "Spawn" },
    { SPELL_EFFECT_TRADE_SKILL, "Trade Skill" },
    { SPELL_EFFECT_STEALTH, "Stealth" },
    { SPELL_EFFECT_DETECT, "Detect" },
    { SPELL_EFFECT_TRANS_DOOR, "Trans Door" },
    { SPELL_EFFECT_FORCE_CRITICAL_HIT, "Force Critical Hit" },
    { SPELL_EFFECT_GUARANTEE_HIT, "Guarantee Hit" },
    { SPELL_EFFECT_ENCHANT_ITEM, "Enchant Item" },
    { SPELL_EFFECT_ENCHANT_ITEM_TEMPORARY, "Enchant Item Temporary" },
    { SPELL_EFFECT_TAMECREATURE, "Tamecreature" },
    { SPELL_EFFECT_SUMMON_PET, "Summon Pet" },
    { SPELL_EFFECT_LEARN_PET_SPELL, "Learn Pet Spell" },
    { SPELL_EFFECT_WEAPON_DAMAGE, "Weapon Damage" },
    { SPELL_EFFECT_OPEN_LOCK_ITEM, "Open Lock Item" },
    { SPELL_EFFECT_PROFICIENCY, "Proficiency" },
    { SPELL_EFFECT_SEND_EVENT, "Send Event" },
    { SPELL_EFFECT_POWER_BURN, "Power Burn" },
    { SPELL_EFFECT_THREAT, "Threat" },
    { SPELL_EFFECT_TRIGGER_SPELL, "Trigger Spell" },
    { SPELL_EFFECT_HEALTH_FUNNEL, "Health Funnel" },
    { SPELL_EFFECT_POWER_FUNNEL, "Power Funnel" },
    { SPELL_EFFECT_HEAL_MAX_HEALTH, "Heal Max Health" },
    { SPELL_EFFECT_INTERRUPT_CAST, "Interrupt Cast" },
    { SPELL_EFFECT_DISTRACT, "Distract" },
    { SPELL_EFFECT_PULL, "Pull" },
    { SPELL_EFFECT_PICKPOCKET, "Pickpocket" },
    { SPELL_EFFECT_ADD_FARSIGHT, "Add Farsight" },
    { SPELL_EFFECT_SUMMON_POSSESSED, "Summon Possessed" },
    { SPELL_EFFECT_SUMMON_TOTEM, "Summon Totem" },
    { SPELL_EFFECT_HEAL_MECHANICAL, "Heal Mechanical" },
    { SPELL_EFFECT_SUMMON_OBJECT_WILD, "Summon Object Wild" },
    { SPELL_EFFECT_SCRIPT_EFFECT, "Script Effect" },
    { SPELL_EFFECT_ATTACK, "Attack" },
    { SPELL_EFFECT_SANCTUARY, "Sanctuary" },
    { SPELL_EFFECT_ADD_COMBO_POINTS, "Add Combo Points" },
    { SPELL_EFFECT_CREATE_HOUSE, "Create House" },
    { SPELL_EFFECT_BIND_SIGHT, "Bind Sight" },
    { SPELL_EFFECT_DUEL, "Duel" },
    { SPELL_EFFECT_STUCK, "Stuck" },
    { SPELL_EFFECT_SUMMON_PLAYER, "Summon Player" },
    { SPELL_EFFECT_ACTIVATE_OBJECT, "Activate Object" },
    { SPELL_EFFECT_SUMMON_TOTEM_SLOT1, "Summon Totem Slot1" },
    { SPELL_EFFECT_SUMMON_TOTEM_SLOT2, "Summon Totem Slot2" },
    { SPELL_EFFECT_SUMMON_TOTEM_SLOT3, "Summon Totem Slot3" },
    { SPELL_EFFECT_SUMMON_TOTEM_SLOT4, "Summon Totem Slot4" },
    { SPELL_EFFECT_THREAT_ALL, "Threat All" },
    { SPELL_EFFECT_ENCHANT_HELD_ITEM, "Enchant Held Item" },
    { SPELL_EFFECT_SUMMON_PHANTASM, "Summon Phantasm" },
    { SPELL_EFFECT_SELF_RESURRECT, "Self Resurrect" },
    { SPELL_EFFECT_SKINNING, "Skinning" },
    { SPELL_EFFECT_CHARGE, "Charge" },
    { SPELL_EFFECT_SUMMON_CRITTER, "Summon Critter" },
    { SPELL_EFFECT_KNOCK_BACK, "Knock Back" },
    { SPELL_EFFECT_DISENCHANT, "Disenchant" },
    { SPELL_EFFECT_INEBRIATE, "Inebriate" },
    { SPELL_EFFECT_FEED_PET, "Feed Pet" },
    { SPELL_EFFECT_DISMISS_PET, "Dismiss Pet" },
    { SPELL_EFFECT_REPUTATION, "Reputation" },
    { SPELL_EFFECT_SUMMON_OBJECT_SLOT1, "Summon Object Slot1" },
    { SPELL_EFFECT_SUMMON_OBJECT_SLOT2, "Summon Object Slot2" },
    { SPELL_EFFECT_SUMMON_OBJECT_SLOT3, "Summon Object Slot3" },
    { SPELL_EFFECT_SUMMON_OBJECT_SLOT4, "Summon Object Slot4" },
    { SPELL_EFFECT_DISPEL_MECHANIC, "Dispel Mechanic" },
    { SPELL_EFFECT_SUMMON_DEAD_PET, "Summon Dead Pet" },
    { SPELL_EFFECT_DESTROY_ALL_TOTEMS, "Destroy All Totems" },
    { SPELL_EFFECT_DURABILITY_DAMAGE, "Durability Damage" },
    { SPELL_EFFECT_SUMMON_DEMON, "Summon Demon" },
    { SPELL_EFFECT_RESURRECT_NEW, "Resurrect New" },
    { SPELL_EFFECT_ATTACK_ME, "Attack Me" },
    { SPELL_EFFECT_DURABILITY_DAMAGE_PCT, "Durability Damage Pct" },
    { SPELL_EFFECT_SKIN_PLAYER_CORPSE, "Skin Player Corpse" },
    { SPELL_EFFECT_SPIRIT_HEAL, "Spirit Heal" },
    { SPELL_EFFECT_SKILL, "Skill" },
    { SPELL_EFFECT_APPLY_AREA_AURA_PET, "Apply Area Aura Pet" },
    { SPELL_EFFECT_TELEPORT_GRAVEYARD, "Teleport Graveyard" },
    { SPELL_EFFECT_NORMALIZED_WEAPON_DMG, "Normalized Weapon Dmg" },
    { SPELL_EFFECT_UNKNOWN122, "Unknown122" },
    { SPELL_EFFECT_SEND_TAXI, "Send Taxi" },
    { SPELL_EFFECT_PLAYER_PULL, "Player Pull" },
    { SPELL_EFFECT_MODIFY_THREAT_PERCENT, "Modify Threat Percent" },
    { SPELL_EFFECT_UNKNOWN126, "Unknown126" },
    { SPELL_EFFECT_UNKNOWN127, "Unknown127" },
    { 0, NULL }
};

typedef enum {
    POWER_MANA = 0x00,
    POWER_RAGE = 0x01,
    POWER_FOCUS = 0x02,
    POWER_ENERGY = 0x03,
    POWER_HAPPINESS = 0x04,
    POWER_HEALTH = 0xFE,
} e_power;
static const value_string e_power_strings[] =  {
    { POWER_MANA, "Mana" },
    { POWER_RAGE, "Rage" },
    { POWER_FOCUS, "Focus" },
    { POWER_ENERGY, "Energy" },
    { POWER_HAPPINESS, "Happiness" },
    { POWER_HEALTH, "Health" },
    { 0, NULL }
};

typedef enum {
    SPELL_MISS_INFO_NONE = 0x0,
    SPELL_MISS_INFO_MISS = 0x1,
    SPELL_MISS_INFO_RESIST = 0x2,
    SPELL_MISS_INFO_DODGE = 0x3,
    SPELL_MISS_INFO_PARRY = 0x4,
    SPELL_MISS_INFO_BLOCK = 0x5,
    SPELL_MISS_INFO_EVADE = 0x6,
    SPELL_MISS_INFO_IMMUNE = 0x7,
    SPELL_MISS_INFO_IMMUNE2 = 0x8,
    SPELL_MISS_INFO_DEFLECT = 0x9,
    SPELL_MISS_INFO_ABSORB = 0xA,
    SPELL_MISS_INFO_REFLECT = 0xB,
} e_spell_miss_info;
static const value_string e_spell_miss_info_strings[] =  {
    { SPELL_MISS_INFO_NONE, "None" },
    { SPELL_MISS_INFO_MISS, "Miss" },
    { SPELL_MISS_INFO_RESIST, "Resist" },
    { SPELL_MISS_INFO_DODGE, "Dodge" },
    { SPELL_MISS_INFO_PARRY, "Parry" },
    { SPELL_MISS_INFO_BLOCK, "Block" },
    { SPELL_MISS_INFO_EVADE, "Evade" },
    { SPELL_MISS_INFO_IMMUNE, "Immune" },
    { SPELL_MISS_INFO_IMMUNE2, "Immune2" },
    { SPELL_MISS_INFO_DEFLECT, "Deflect" },
    { SPELL_MISS_INFO_ABSORB, "Absorb" },
    { SPELL_MISS_INFO_REFLECT, "Reflect" },
    { 0, NULL }
};

typedef enum {
    TRAINER_SPELL_STATE_GREEN = 0x0,
    TRAINER_SPELL_STATE_RED = 0x1,
    TRAINER_SPELL_STATE_GRAY = 0x2,
} e_trainer_spell_state;
static const value_string e_trainer_spell_state_strings[] =  {
    { TRAINER_SPELL_STATE_GREEN, "Green" },
    { TRAINER_SPELL_STATE_RED, "Red" },
    { TRAINER_SPELL_STATE_GRAY, "Gray" },
    { 0, NULL }
};

typedef enum {
    SKILL_NONE = 0x000,
    SKILL_FROST = 0x006,
    SKILL_FIRE = 0x008,
    SKILL_ARMS = 0x01A,
    SKILL_COMBAT = 0x026,
    SKILL_SUBTLETY = 0x027,
    SKILL_POISONS = 0x028,
    SKILL_SWORDS = 0x02B,
    SKILL_AXES = 0x02C,
    SKILL_BOWS = 0x02D,
    SKILL_GUNS = 0x02E,
    SKILL_BEAST_MASTERY = 0x032,
    SKILL_SURVIVAL = 0x033,
    SKILL_MACES = 0x036,
    SKILL_TWO_HANDED_SWORDS = 0x037,
    SKILL_HOLY = 0x038,
    SKILL_SHADOW = 0x04E,
    SKILL_DEFENSE = 0x05F,
    SKILL_LANGUAGE_COMMON = 0x062,
    SKILL_RACIAL_DWARVEN = 0x065,
    SKILL_LANGUAGE_ORCISH = 0x06D,
    SKILL_LANGUAGE_DWARVEN = 0x06F,
    SKILL_LANGUAGE_DARNASSIAN = 0x071,
    SKILL_LANGUAGE_TAURAHE = 0x073,
    SKILL_DUAL_WIELD = 0x076,
    SKILL_RACIAL_TAUREN = 0x07C,
    SKILL_ORC_RACIAL = 0x07D,
    SKILL_RACIAL_NIGHT_ELF = 0x07E,
    SKILL_FIRST_AID = 0x081,
    SKILL_FERAL_COMBAT = 0x086,
    SKILL_STAVES = 0x088,
    SKILL_LANGUAGE_THALASSIAN = 0x089,
    SKILL_LANGUAGE_DRACONIC = 0x08A,
    SKILL_LANGUAGE_DEMON_TONGUE = 0x08B,
    SKILL_LANGUAGE_TITAN = 0x08C,
    SKILL_LANGUAGE_OLD_TONGUE = 0x08D,
    SKILL_SURVIVAL2 = 0x08E,
    SKILL_RIDING_HORSE = 0x094,
    SKILL_RIDING_WOLF = 0x095,
    SKILL_RIDING_TIGER = 0x096,
    SKILL_RIDING_RAM = 0x098,
    SKILL_SWIMMING = 0x09B,
    SKILL_TWO_HANDED_MACES = 0x0A0,
    SKILL_UNARMED = 0x0A2,
    SKILL_MARKSMANSHIP = 0x0A3,
    SKILL_BLACKSMITHING = 0x0A4,
    SKILL_LEATHERWORKING = 0x0A5,
    SKILL_ALCHEMY = 0x0AB,
    SKILL_TWO_HANDED_AXES = 0x0AC,
    SKILL_DAGGERS = 0x0AD,
    SKILL_THROWN = 0x0B0,
    SKILL_HERBALISM = 0x0B6,
    SKILL_GENERIC_DND = 0x0B7,
    SKILL_RETRIBUTION = 0x0B8,
    SKILL_COOKING = 0x0B9,
    SKILL_MINING = 0x0BA,
    SKILL_PET_IMP = 0x0BC,
    SKILL_PET_FELHUNTER = 0x0BD,
    SKILL_TAILORING = 0x0C5,
    SKILL_ENGINEERING = 0x0CA,
    SKILL_PET_SPIDER = 0x0CB,
    SKILL_PET_VOIDWALKER = 0x0CC,
    SKILL_PET_SUCCUBUS = 0x0CD,
    SKILL_PET_INFERNAL = 0x0CE,
    SKILL_PET_DOOMGUARD = 0x0CF,
    SKILL_PET_WOLF = 0x0D0,
    SKILL_PET_CAT = 0x0D1,
    SKILL_PET_BEAR = 0x0D2,
    SKILL_PET_BOAR = 0x0D3,
    SKILL_PET_CROCILISK = 0x0D4,
    SKILL_PET_CARRION_BIRD = 0x0D5,
    SKILL_PET_CRAB = 0x0D6,
    SKILL_PET_GORILLA = 0x0D7,
    SKILL_PET_RAPTOR = 0x0D9,
    SKILL_PET_TALLSTRIDER = 0x0DA,
    SKILL_RACIAL_UNDED = 0x0DC,
    SKILL_CROSSBOWS = 0x0E2,
    SKILL_WANDS = 0x0E4,
    SKILL_POLEARMS = 0x0E5,
    SKILL_PET_SCORPID = 0x0EC,
    SKILL_ARCANE = 0x0ED,
    SKILL_PET_TURTLE = 0x0FB,
    SKILL_ASSASSINATION = 0x0FD,
    SKILL_FURY = 0x100,
    SKILL_PROTECTION = 0x101,
    SKILL_BEAST_TRAINING = 0x105,
    SKILL_PROTECTION2 = 0x10B,
    SKILL_PET_TALENTS = 0x10E,
    SKILL_PLATE_MAIL = 0x125,
    SKILL_LANGUAGE_GNOMISH = 0x139,
    SKILL_LANGUAGE_TROLL = 0x13B,
    SKILL_ENCHANTING = 0x14D,
    SKILL_DEMONOLOGY = 0x162,
    SKILL_AFFLICTION = 0x163,
    SKILL_FISHING = 0x164,
    SKILL_ENHANCEMENT = 0x175,
    SKILL_RESTORATION = 0x176,
    SKILL_ELEMENTAL_COMBAT = 0x177,
    SKILL_SKINNING = 0x189,
    SKILL_MAIL = 0x19D,
    SKILL_LEATHER = 0x19E,
    SKILL_CLOTH = 0x19F,
    SKILL_SHIELD = 0x1B1,
    SKILL_FIST_WEAPONS = 0x1D9,
    SKILL_RIDING_RAPTOR = 0x215,
    SKILL_RIDING_MECHANOSTRIDER = 0x229,
    SKILL_RIDING_UNDEAD_HORSE = 0x22A,
    SKILL_RESTORATION2 = 0x23D,
    SKILL_BALANCE = 0x23E,
    SKILL_DESTRUCTION = 0x251,
    SKILL_HOLY2 = 0x252,
    SKILL_DISCIPLINE = 0x265,
    SKILL_LOCKPICKING = 0x279,
    SKILL_PET_BAT = 0x28D,
    SKILL_PET_HYENA = 0x28E,
    SKILL_PET_OWL = 0x28F,
    SKILL_PET_WIND_SERPENT = 0x290,
    SKILL_LANGUAGE_GUTTERSPEAK = 0x2A1,
    SKILL_RIDING_KODO = 0x2C9,
    SKILL_RACIAL_TROLL = 0x2DD,
    SKILL_RACIAL_GNOME = 0x2F1,
    SKILL_RACIAL_HUMAN = 0x2F2,
    SKILL_PET_EVENT_RC = 0x2F6,
    SKILL_RIDING = 0x2FA,
} e_skill;
static const value_string e_skill_strings[] =  {
    { SKILL_NONE, "None" },
    { SKILL_FROST, "Frost" },
    { SKILL_FIRE, "Fire" },
    { SKILL_ARMS, "Arms" },
    { SKILL_COMBAT, "Combat" },
    { SKILL_SUBTLETY, "Subtlety" },
    { SKILL_POISONS, "Poisons" },
    { SKILL_SWORDS, "Swords" },
    { SKILL_AXES, "Axes" },
    { SKILL_BOWS, "Bows" },
    { SKILL_GUNS, "Guns" },
    { SKILL_BEAST_MASTERY, "Beast Mastery" },
    { SKILL_SURVIVAL, "Survival" },
    { SKILL_MACES, "Maces" },
    { SKILL_TWO_HANDED_SWORDS, "Two Handed Swords" },
    { SKILL_HOLY, "Holy" },
    { SKILL_SHADOW, "Shadow" },
    { SKILL_DEFENSE, "Defense" },
    { SKILL_LANGUAGE_COMMON, "Language Common" },
    { SKILL_RACIAL_DWARVEN, "Racial Dwarven" },
    { SKILL_LANGUAGE_ORCISH, "Language Orcish" },
    { SKILL_LANGUAGE_DWARVEN, "Language Dwarven" },
    { SKILL_LANGUAGE_DARNASSIAN, "Language Darnassian" },
    { SKILL_LANGUAGE_TAURAHE, "Language Taurahe" },
    { SKILL_DUAL_WIELD, "Dual Wield" },
    { SKILL_RACIAL_TAUREN, "Racial Tauren" },
    { SKILL_ORC_RACIAL, "Orc Racial" },
    { SKILL_RACIAL_NIGHT_ELF, "Racial Night Elf" },
    { SKILL_FIRST_AID, "First Aid" },
    { SKILL_FERAL_COMBAT, "Feral Combat" },
    { SKILL_STAVES, "Staves" },
    { SKILL_LANGUAGE_THALASSIAN, "Language Thalassian" },
    { SKILL_LANGUAGE_DRACONIC, "Language Draconic" },
    { SKILL_LANGUAGE_DEMON_TONGUE, "Language Demon Tongue" },
    { SKILL_LANGUAGE_TITAN, "Language Titan" },
    { SKILL_LANGUAGE_OLD_TONGUE, "Language Old Tongue" },
    { SKILL_SURVIVAL2, "Survival2" },
    { SKILL_RIDING_HORSE, "Riding Horse" },
    { SKILL_RIDING_WOLF, "Riding Wolf" },
    { SKILL_RIDING_TIGER, "Riding Tiger" },
    { SKILL_RIDING_RAM, "Riding Ram" },
    { SKILL_SWIMMING, "Swimming" },
    { SKILL_TWO_HANDED_MACES, "Two Handed Maces" },
    { SKILL_UNARMED, "Unarmed" },
    { SKILL_MARKSMANSHIP, "Marksmanship" },
    { SKILL_BLACKSMITHING, "Blacksmithing" },
    { SKILL_LEATHERWORKING, "Leatherworking" },
    { SKILL_ALCHEMY, "Alchemy" },
    { SKILL_TWO_HANDED_AXES, "Two Handed Axes" },
    { SKILL_DAGGERS, "Daggers" },
    { SKILL_THROWN, "Thrown" },
    { SKILL_HERBALISM, "Herbalism" },
    { SKILL_GENERIC_DND, "Generic Dnd" },
    { SKILL_RETRIBUTION, "Retribution" },
    { SKILL_COOKING, "Cooking" },
    { SKILL_MINING, "Mining" },
    { SKILL_PET_IMP, "Pet Imp" },
    { SKILL_PET_FELHUNTER, "Pet Felhunter" },
    { SKILL_TAILORING, "Tailoring" },
    { SKILL_ENGINEERING, "Engineering" },
    { SKILL_PET_SPIDER, "Pet Spider" },
    { SKILL_PET_VOIDWALKER, "Pet Voidwalker" },
    { SKILL_PET_SUCCUBUS, "Pet Succubus" },
    { SKILL_PET_INFERNAL, "Pet Infernal" },
    { SKILL_PET_DOOMGUARD, "Pet Doomguard" },
    { SKILL_PET_WOLF, "Pet Wolf" },
    { SKILL_PET_CAT, "Pet Cat" },
    { SKILL_PET_BEAR, "Pet Bear" },
    { SKILL_PET_BOAR, "Pet Boar" },
    { SKILL_PET_CROCILISK, "Pet Crocilisk" },
    { SKILL_PET_CARRION_BIRD, "Pet Carrion Bird" },
    { SKILL_PET_CRAB, "Pet Crab" },
    { SKILL_PET_GORILLA, "Pet Gorilla" },
    { SKILL_PET_RAPTOR, "Pet Raptor" },
    { SKILL_PET_TALLSTRIDER, "Pet Tallstrider" },
    { SKILL_RACIAL_UNDED, "Racial Unded" },
    { SKILL_CROSSBOWS, "Crossbows" },
    { SKILL_WANDS, "Wands" },
    { SKILL_POLEARMS, "Polearms" },
    { SKILL_PET_SCORPID, "Pet Scorpid" },
    { SKILL_ARCANE, "Arcane" },
    { SKILL_PET_TURTLE, "Pet Turtle" },
    { SKILL_ASSASSINATION, "Assassination" },
    { SKILL_FURY, "Fury" },
    { SKILL_PROTECTION, "Protection" },
    { SKILL_BEAST_TRAINING, "Beast Training" },
    { SKILL_PROTECTION2, "Protection2" },
    { SKILL_PET_TALENTS, "Pet Talents" },
    { SKILL_PLATE_MAIL, "Plate Mail" },
    { SKILL_LANGUAGE_GNOMISH, "Language Gnomish" },
    { SKILL_LANGUAGE_TROLL, "Language Troll" },
    { SKILL_ENCHANTING, "Enchanting" },
    { SKILL_DEMONOLOGY, "Demonology" },
    { SKILL_AFFLICTION, "Affliction" },
    { SKILL_FISHING, "Fishing" },
    { SKILL_ENHANCEMENT, "Enhancement" },
    { SKILL_RESTORATION, "Restoration" },
    { SKILL_ELEMENTAL_COMBAT, "Elemental Combat" },
    { SKILL_SKINNING, "Skinning" },
    { SKILL_MAIL, "Mail" },
    { SKILL_LEATHER, "Leather" },
    { SKILL_CLOTH, "Cloth" },
    { SKILL_SHIELD, "Shield" },
    { SKILL_FIST_WEAPONS, "Fist Weapons" },
    { SKILL_RIDING_RAPTOR, "Riding Raptor" },
    { SKILL_RIDING_MECHANOSTRIDER, "Riding Mechanostrider" },
    { SKILL_RIDING_UNDEAD_HORSE, "Riding Undead Horse" },
    { SKILL_RESTORATION2, "Restoration2" },
    { SKILL_BALANCE, "Balance" },
    { SKILL_DESTRUCTION, "Destruction" },
    { SKILL_HOLY2, "Holy2" },
    { SKILL_DISCIPLINE, "Discipline" },
    { SKILL_LOCKPICKING, "Lockpicking" },
    { SKILL_PET_BAT, "Pet Bat" },
    { SKILL_PET_HYENA, "Pet Hyena" },
    { SKILL_PET_OWL, "Pet Owl" },
    { SKILL_PET_WIND_SERPENT, "Pet Wind Serpent" },
    { SKILL_LANGUAGE_GUTTERSPEAK, "Language Gutterspeak" },
    { SKILL_RIDING_KODO, "Riding Kodo" },
    { SKILL_RACIAL_TROLL, "Racial Troll" },
    { SKILL_RACIAL_GNOME, "Racial Gnome" },
    { SKILL_RACIAL_HUMAN, "Racial Human" },
    { SKILL_PET_EVENT_RC, "Pet Event Rc" },
    { SKILL_RIDING, "Riding" },
    { 0, NULL }
};

typedef enum {
    ITEM_QUALITY_POOR = 0x0,
    ITEM_QUALITY_NORMAL = 0x1,
    ITEM_QUALITY_UNCOMMON = 0x2,
    ITEM_QUALITY_RARE = 0x3,
    ITEM_QUALITY_EPIC = 0x4,
    ITEM_QUALITY_LEGENDARY = 0x5,
    ITEM_QUALITY_ARTIFACT = 0x6,
} e_item_quality;
static const value_string e_item_quality_strings[] =  {
    { ITEM_QUALITY_POOR, "Poor" },
    { ITEM_QUALITY_NORMAL, "Normal" },
    { ITEM_QUALITY_UNCOMMON, "Uncommon" },
    { ITEM_QUALITY_RARE, "Rare" },
    { ITEM_QUALITY_EPIC, "Epic" },
    { ITEM_QUALITY_LEGENDARY, "Legendary" },
    { ITEM_QUALITY_ARTIFACT, "Artifact" },
    { 0, NULL }
};

typedef enum {
    BATTLEFIELD_PORT_ACTION_LEAVE_QUEUE = 0x0,
    BATTLEFIELD_PORT_ACTION_ENTER_BATTLE = 0x1,
} e_battlefield_port_action;
static const value_string e_battlefield_port_action_strings[] =  {
    { BATTLEFIELD_PORT_ACTION_LEAVE_QUEUE, "Leave Queue" },
    { BATTLEFIELD_PORT_ACTION_ENTER_BATTLE, "Enter Battle" },
    { 0, NULL }
};

typedef enum {
    BUYBACK_SLOT_SLOT1 = 0x45,
    BUYBACK_SLOT_SLOT2 = 0x46,
    BUYBACK_SLOT_SLOT3 = 0x47,
    BUYBACK_SLOT_SLOT4 = 0x48,
    BUYBACK_SLOT_SLOT5 = 0x49,
    BUYBACK_SLOT_SLOT6 = 0x4A,
    BUYBACK_SLOT_SLOT7 = 0x4B,
    BUYBACK_SLOT_SLOT8 = 0x4C,
    BUYBACK_SLOT_SLOT9 = 0x4D,
    BUYBACK_SLOT_SLOT10 = 0x4E,
    BUYBACK_SLOT_SLOT11 = 0x4F,
    BUYBACK_SLOT_SLOT12 = 0x50,
    BUYBACK_SLOT_SLOT13 = 0x51,
} e_buyback_slot;
static const value_string e_buyback_slot_strings[] =  {
    { BUYBACK_SLOT_SLOT1, "Slot1" },
    { BUYBACK_SLOT_SLOT2, "Slot2" },
    { BUYBACK_SLOT_SLOT3, "Slot3" },
    { BUYBACK_SLOT_SLOT4, "Slot4" },
    { BUYBACK_SLOT_SLOT5, "Slot5" },
    { BUYBACK_SLOT_SLOT6, "Slot6" },
    { BUYBACK_SLOT_SLOT7, "Slot7" },
    { BUYBACK_SLOT_SLOT8, "Slot8" },
    { BUYBACK_SLOT_SLOT9, "Slot9" },
    { BUYBACK_SLOT_SLOT10, "Slot10" },
    { BUYBACK_SLOT_SLOT11, "Slot11" },
    { BUYBACK_SLOT_SLOT12, "Slot12" },
    { BUYBACK_SLOT_SLOT13, "Slot13" },
    { 0, NULL }
};

typedef enum {
    EMOTE_ONESHOT_NONE = 0x000,
    EMOTE_ONESHOT_TALK = 0x001,
    EMOTE_ONESHOT_BOW = 0x002,
    EMOTE_ONESHOT_WAVE = 0x003,
    EMOTE_ONESHOT_CHEER = 0x004,
    EMOTE_ONESHOT_EXCLAMATION = 0x005,
    EMOTE_ONESHOT_QUESTION = 0x006,
    EMOTE_ONESHOT_EAT = 0x007,
    EMOTE_STATE_DANCE = 0x00A,
    EMOTE_ONESHOT_LAUGH = 0x00B,
    EMOTE_STATE_SLEEP = 0x00C,
    EMOTE_STATE_SIT = 0x00D,
    EMOTE_ONESHOT_RUDE = 0x00E,
    EMOTE_ONESHOT_ROAR = 0x00F,
    EMOTE_ONESHOT_KNEEL = 0x010,
    EMOTE_ONESHOT_KISS = 0x011,
    EMOTE_ONESHOT_CRY = 0x012,
    EMOTE_ONESHOT_CHICKEN = 0x013,
    EMOTE_ONESHOT_BEG = 0x014,
    EMOTE_ONESHOT_APPLAUD = 0x015,
    EMOTE_ONESHOT_SHOUT = 0x016,
    EMOTE_ONESHOT_FLEX = 0x017,
    EMOTE_ONESHOT_SHY = 0x018,
    EMOTE_ONESHOT_POINT = 0x019,
    EMOTE_STATE_STAND = 0x01A,
    EMOTE_STATE_READYUNARMED = 0x01B,
    EMOTE_STATE_WORK_SHEATHED = 0x01C,
    EMOTE_STATE_POINT = 0x01D,
    EMOTE_STATE_NONE = 0x01E,
    EMOTE_ONESHOT_WOUND = 0x021,
    EMOTE_ONESHOT_WOUNDCRITICAL = 0x022,
    EMOTE_ONESHOT_ATTACKUNARMED = 0x023,
    EMOTE_ONESHOT_ATTACK1H = 0x024,
    EMOTE_ONESHOT_ATTACK2HTIGHT = 0x025,
    EMOTE_ONESHOT_ATTACK2HLOOSE = 0x026,
    EMOTE_ONESHOT_PARRYUNARMED = 0x027,
    EMOTE_ONESHOT_PARRYSHIELD = 0x02B,
    EMOTE_ONESHOT_READYUNARMED = 0x02C,
    EMOTE_ONESHOT_READY1H = 0x02D,
    EMOTE_ONESHOT_READYBOW = 0x030,
    EMOTE_ONESHOT_SPELLPRECAST = 0x032,
    EMOTE_ONESHOT_SPELLCAST = 0x033,
    EMOTE_ONESHOT_BATTLEROAR = 0x035,
    EMOTE_ONESHOT_SPECIALATTACK1H = 0x036,
    EMOTE_ONESHOT_KICK = 0x03C,
    EMOTE_ONESHOT_ATTACKTHROWN = 0x03D,
    EMOTE_STATE_STUN = 0x040,
    EMOTE_STATE_DEAD = 0x041,
    EMOTE_ONESHOT_SALUTE = 0x042,
    EMOTE_STATE_KNEEL = 0x044,
    EMOTE_STATE_USESTANDING = 0x045,
    EMOTE_ONESHOT_WAVE_NOSHEATHE = 0x046,
    EMOTE_ONESHOT_CHEER_NOSHEATHE = 0x047,
    EMOTE_ONESHOT_EAT_NOSHEATHE = 0x05C,
    EMOTE_STATE_STUN_NOSHEATHE = 0x05D,
    EMOTE_ONESHOT_DANCE = 0x05E,
    EMOTE_ONESHOT_SALUTE_NOSHEATH = 0x071,
    EMOTE_STATE_USESTANDING_NOSHEATHE = 0x085,
    EMOTE_ONESHOT_LAUGH_NOSHEATHE = 0x099,
    EMOTE_STATE_WORK = 0x0AD,
    EMOTE_STATE_SPELLPRECAST = 0x0C1,
    EMOTE_ONESHOT_READYRIFLE = 0x0D5,
    EMOTE_STATE_READYRIFLE = 0x0D6,
    EMOTE_STATE_WORK_MINING = 0x0E9,
    EMOTE_STATE_WORK_CHOPWOOD = 0x0EA,
    EMOTE_STATE_APPLAUD = 0x0FD,
    EMOTE_ONESHOT_LIFTOFF = 0x0FE,
    EMOTE_ONESHOT_YES = 0x111,
    EMOTE_ONESHOT_NO = 0x112,
    EMOTE_ONESHOT_TRAIN = 0x113,
    EMOTE_ONESHOT_LAND = 0x125,
    EMOTE_STATE_AT_EASE = 0x139,
    EMOTE_STATE_READY1H = 0x14D,
    EMOTE_STATE_SPELLKNEELSTART = 0x161,
    EMOTE_STATE_SUBMERGED = 0x175,
    EMOTE_ONESHOT_SUBMERGE = 0x176,
    EMOTE_STATE_READY2H = 0x177,
    EMOTE_STATE_READYBOW = 0x178,
    EMOTE_ONESHOT_MOUNTSPECIAL = 0x179,
    EMOTE_STATE_TALK = 0x17A,
    EMOTE_STATE_FISHING = 0x17B,
    EMOTE_ONESHOT_FISHING = 0x17C,
    EMOTE_ONESHOT_LOOT = 0x17D,
    EMOTE_STATE_WHIRLWIND = 0x17E,
    EMOTE_STATE_DROWNED = 0x17F,
    EMOTE_STATE_HOLD_BOW = 0x180,
    EMOTE_STATE_HOLD_RIFLE = 0x181,
    EMOTE_STATE_HOLD_THROWN = 0x182,
    EMOTE_ONESHOT_DROWN = 0x183,
    EMOTE_ONESHOT_STOMP = 0x184,
    EMOTE_ONESHOT_ATTACKOFF = 0x185,
    EMOTE_ONESHOT_ATTACKOFFPIERCE = 0x186,
    EMOTE_STATE_ROAR = 0x187,
    EMOTE_STATE_LAUGH = 0x188,
    EMOTE_ONESHOT_CREATURE_SPECIAL = 0x189,
    EMOTE_ONESHOT_JUMPLANDRUN = 0x18A,
    EMOTE_ONESHOT_JUMPEND = 0x18B,
    EMOTE_ONESHOT_TALK_NOSHEATHE = 0x18C,
    EMOTE_ONESHOT_POINT_NOSHEATHE = 0x18D,
    EMOTE_STATE_CANNIBALIZE = 0x18E,
    EMOTE_ONESHOT_JUMPSTART = 0x18F,
    EMOTE_STATE_DANCESPECIAL = 0x190,
    EMOTE_ONESHOT_DANCESPECIAL = 0x191,
    EMOTE_ONESHOT_CUSTOMSPELL01 = 0x192,
    EMOTE_ONESHOT_CUSTOMSPELL02 = 0x193,
    EMOTE_ONESHOT_CUSTOMSPELL03 = 0x194,
    EMOTE_ONESHOT_CUSTOMSPELL04 = 0x195,
    EMOTE_ONESHOT_CUSTOMSPELL05 = 0x196,
    EMOTE_ONESHOT_CUSTOMSPELL06 = 0x197,
    EMOTE_ONESHOT_CUSTOMSPELL07 = 0x198,
    EMOTE_ONESHOT_CUSTOMSPELL08 = 0x199,
    EMOTE_ONESHOT_CUSTOMSPELL09 = 0x19A,
    EMOTE_ONESHOT_CUSTOMSPELL10 = 0x19B,
    EMOTE_STATE_EXCLAIM = 0x19C,
    EMOTE_STATE_SIT_CHAIR_MED = 0x19F,
    EMOTE_STATE_SPELLEFFECT_HOLD = 0x1A6,
    EMOTE_STATE_EAT_NO_SHEATHE = 0x1A7,
} e_emote;
static const value_string e_emote_strings[] =  {
    { EMOTE_ONESHOT_NONE, "Oneshot None" },
    { EMOTE_ONESHOT_TALK, "Oneshot Talk" },
    { EMOTE_ONESHOT_BOW, "Oneshot Bow" },
    { EMOTE_ONESHOT_WAVE, "Oneshot Wave" },
    { EMOTE_ONESHOT_CHEER, "Oneshot Cheer" },
    { EMOTE_ONESHOT_EXCLAMATION, "Oneshot Exclamation" },
    { EMOTE_ONESHOT_QUESTION, "Oneshot Question" },
    { EMOTE_ONESHOT_EAT, "Oneshot Eat" },
    { EMOTE_STATE_DANCE, "State Dance" },
    { EMOTE_ONESHOT_LAUGH, "Oneshot Laugh" },
    { EMOTE_STATE_SLEEP, "State Sleep" },
    { EMOTE_STATE_SIT, "State Sit" },
    { EMOTE_ONESHOT_RUDE, "Oneshot Rude" },
    { EMOTE_ONESHOT_ROAR, "Oneshot Roar" },
    { EMOTE_ONESHOT_KNEEL, "Oneshot Kneel" },
    { EMOTE_ONESHOT_KISS, "Oneshot Kiss" },
    { EMOTE_ONESHOT_CRY, "Oneshot Cry" },
    { EMOTE_ONESHOT_CHICKEN, "Oneshot Chicken" },
    { EMOTE_ONESHOT_BEG, "Oneshot Beg" },
    { EMOTE_ONESHOT_APPLAUD, "Oneshot Applaud" },
    { EMOTE_ONESHOT_SHOUT, "Oneshot Shout" },
    { EMOTE_ONESHOT_FLEX, "Oneshot Flex" },
    { EMOTE_ONESHOT_SHY, "Oneshot Shy" },
    { EMOTE_ONESHOT_POINT, "Oneshot Point" },
    { EMOTE_STATE_STAND, "State Stand" },
    { EMOTE_STATE_READYUNARMED, "State Readyunarmed" },
    { EMOTE_STATE_WORK_SHEATHED, "State Work Sheathed" },
    { EMOTE_STATE_POINT, "State Point" },
    { EMOTE_STATE_NONE, "State None" },
    { EMOTE_ONESHOT_WOUND, "Oneshot Wound" },
    { EMOTE_ONESHOT_WOUNDCRITICAL, "Oneshot Woundcritical" },
    { EMOTE_ONESHOT_ATTACKUNARMED, "Oneshot Attackunarmed" },
    { EMOTE_ONESHOT_ATTACK1H, "Oneshot Attack1h" },
    { EMOTE_ONESHOT_ATTACK2HTIGHT, "Oneshot Attack2htight" },
    { EMOTE_ONESHOT_ATTACK2HLOOSE, "Oneshot Attack2hloose" },
    { EMOTE_ONESHOT_PARRYUNARMED, "Oneshot Parryunarmed" },
    { EMOTE_ONESHOT_PARRYSHIELD, "Oneshot Parryshield" },
    { EMOTE_ONESHOT_READYUNARMED, "Oneshot Readyunarmed" },
    { EMOTE_ONESHOT_READY1H, "Oneshot Ready1h" },
    { EMOTE_ONESHOT_READYBOW, "Oneshot Readybow" },
    { EMOTE_ONESHOT_SPELLPRECAST, "Oneshot Spellprecast" },
    { EMOTE_ONESHOT_SPELLCAST, "Oneshot Spellcast" },
    { EMOTE_ONESHOT_BATTLEROAR, "Oneshot Battleroar" },
    { EMOTE_ONESHOT_SPECIALATTACK1H, "Oneshot Specialattack1h" },
    { EMOTE_ONESHOT_KICK, "Oneshot Kick" },
    { EMOTE_ONESHOT_ATTACKTHROWN, "Oneshot Attackthrown" },
    { EMOTE_STATE_STUN, "State Stun" },
    { EMOTE_STATE_DEAD, "State Dead" },
    { EMOTE_ONESHOT_SALUTE, "Oneshot Salute" },
    { EMOTE_STATE_KNEEL, "State Kneel" },
    { EMOTE_STATE_USESTANDING, "State Usestanding" },
    { EMOTE_ONESHOT_WAVE_NOSHEATHE, "Oneshot Wave Nosheathe" },
    { EMOTE_ONESHOT_CHEER_NOSHEATHE, "Oneshot Cheer Nosheathe" },
    { EMOTE_ONESHOT_EAT_NOSHEATHE, "Oneshot Eat Nosheathe" },
    { EMOTE_STATE_STUN_NOSHEATHE, "State Stun Nosheathe" },
    { EMOTE_ONESHOT_DANCE, "Oneshot Dance" },
    { EMOTE_ONESHOT_SALUTE_NOSHEATH, "Oneshot Salute Nosheath" },
    { EMOTE_STATE_USESTANDING_NOSHEATHE, "State Usestanding Nosheathe" },
    { EMOTE_ONESHOT_LAUGH_NOSHEATHE, "Oneshot Laugh Nosheathe" },
    { EMOTE_STATE_WORK, "State Work" },
    { EMOTE_STATE_SPELLPRECAST, "State Spellprecast" },
    { EMOTE_ONESHOT_READYRIFLE, "Oneshot Readyrifle" },
    { EMOTE_STATE_READYRIFLE, "State Readyrifle" },
    { EMOTE_STATE_WORK_MINING, "State Work Mining" },
    { EMOTE_STATE_WORK_CHOPWOOD, "State Work Chopwood" },
    { EMOTE_STATE_APPLAUD, "State Applaud" },
    { EMOTE_ONESHOT_LIFTOFF, "Oneshot Liftoff" },
    { EMOTE_ONESHOT_YES, "Oneshot Yes" },
    { EMOTE_ONESHOT_NO, "Oneshot No" },
    { EMOTE_ONESHOT_TRAIN, "Oneshot Train" },
    { EMOTE_ONESHOT_LAND, "Oneshot Land" },
    { EMOTE_STATE_AT_EASE, "State At Ease" },
    { EMOTE_STATE_READY1H, "State Ready1h" },
    { EMOTE_STATE_SPELLKNEELSTART, "State Spellkneelstart" },
    { EMOTE_STATE_SUBMERGED, "State Submerged" },
    { EMOTE_ONESHOT_SUBMERGE, "Oneshot Submerge" },
    { EMOTE_STATE_READY2H, "State Ready2h" },
    { EMOTE_STATE_READYBOW, "State Readybow" },
    { EMOTE_ONESHOT_MOUNTSPECIAL, "Oneshot Mountspecial" },
    { EMOTE_STATE_TALK, "State Talk" },
    { EMOTE_STATE_FISHING, "State Fishing" },
    { EMOTE_ONESHOT_FISHING, "Oneshot Fishing" },
    { EMOTE_ONESHOT_LOOT, "Oneshot Loot" },
    { EMOTE_STATE_WHIRLWIND, "State Whirlwind" },
    { EMOTE_STATE_DROWNED, "State Drowned" },
    { EMOTE_STATE_HOLD_BOW, "State Hold Bow" },
    { EMOTE_STATE_HOLD_RIFLE, "State Hold Rifle" },
    { EMOTE_STATE_HOLD_THROWN, "State Hold Thrown" },
    { EMOTE_ONESHOT_DROWN, "Oneshot Drown" },
    { EMOTE_ONESHOT_STOMP, "Oneshot Stomp" },
    { EMOTE_ONESHOT_ATTACKOFF, "Oneshot Attackoff" },
    { EMOTE_ONESHOT_ATTACKOFFPIERCE, "Oneshot Attackoffpierce" },
    { EMOTE_STATE_ROAR, "State Roar" },
    { EMOTE_STATE_LAUGH, "State Laugh" },
    { EMOTE_ONESHOT_CREATURE_SPECIAL, "Oneshot Creature Special" },
    { EMOTE_ONESHOT_JUMPLANDRUN, "Oneshot Jumplandrun" },
    { EMOTE_ONESHOT_JUMPEND, "Oneshot Jumpend" },
    { EMOTE_ONESHOT_TALK_NOSHEATHE, "Oneshot Talk Nosheathe" },
    { EMOTE_ONESHOT_POINT_NOSHEATHE, "Oneshot Point Nosheathe" },
    { EMOTE_STATE_CANNIBALIZE, "State Cannibalize" },
    { EMOTE_ONESHOT_JUMPSTART, "Oneshot Jumpstart" },
    { EMOTE_STATE_DANCESPECIAL, "State Dancespecial" },
    { EMOTE_ONESHOT_DANCESPECIAL, "Oneshot Dancespecial" },
    { EMOTE_ONESHOT_CUSTOMSPELL01, "Oneshot Customspell01" },
    { EMOTE_ONESHOT_CUSTOMSPELL02, "Oneshot Customspell02" },
    { EMOTE_ONESHOT_CUSTOMSPELL03, "Oneshot Customspell03" },
    { EMOTE_ONESHOT_CUSTOMSPELL04, "Oneshot Customspell04" },
    { EMOTE_ONESHOT_CUSTOMSPELL05, "Oneshot Customspell05" },
    { EMOTE_ONESHOT_CUSTOMSPELL06, "Oneshot Customspell06" },
    { EMOTE_ONESHOT_CUSTOMSPELL07, "Oneshot Customspell07" },
    { EMOTE_ONESHOT_CUSTOMSPELL08, "Oneshot Customspell08" },
    { EMOTE_ONESHOT_CUSTOMSPELL09, "Oneshot Customspell09" },
    { EMOTE_ONESHOT_CUSTOMSPELL10, "Oneshot Customspell10" },
    { EMOTE_STATE_EXCLAIM, "State Exclaim" },
    { EMOTE_STATE_SIT_CHAIR_MED, "State Sit Chair Med" },
    { EMOTE_STATE_SPELLEFFECT_HOLD, "State Spelleffect Hold" },
    { EMOTE_STATE_EAT_NO_SHEATHE, "State Eat No Sheathe" },
    { 0, NULL }
};

typedef enum {
    FAR_SIGHT_OPERATION_REMOVE = 0x0,
    FAR_SIGHT_OPERATION_ADD = 0x1,
} e_far_sight_operation;
static const value_string e_far_sight_operation_strings[] =  {
    { FAR_SIGHT_OPERATION_REMOVE, "Remove" },
    { FAR_SIGHT_OPERATION_ADD, "Add" },
    { 0, NULL }
};

typedef enum {
    GM_TICKET_TYPE_STUCK = 0x1,
    GM_TICKET_TYPE_BEHAVIOR_HARASSMENT = 0x2,
    GM_TICKET_TYPE_GUILD = 0x3,
    GM_TICKET_TYPE_ITEM = 0x4,
    GM_TICKET_TYPE_ENVIRONMENTAL = 0x5,
    GM_TICKET_TYPE_NONQUEST_CREEP = 0x6,
    GM_TICKET_TYPE_QUEST_QUESTNPC = 0x7,
    GM_TICKET_TYPE_TECHNICAL = 0x8,
    GM_TICKET_TYPE_ACCOUNT_BILLING = 0x9,
    GM_TICKET_TYPE_CHARACTER = 0xA,
} e_gm_ticket_type;
static const value_string e_gm_ticket_type_strings[] =  {
    { GM_TICKET_TYPE_STUCK, "Stuck" },
    { GM_TICKET_TYPE_BEHAVIOR_HARASSMENT, "Behavior Harassment" },
    { GM_TICKET_TYPE_GUILD, "Guild" },
    { GM_TICKET_TYPE_ITEM, "Item" },
    { GM_TICKET_TYPE_ENVIRONMENTAL, "Environmental" },
    { GM_TICKET_TYPE_NONQUEST_CREEP, "Nonquest Creep" },
    { GM_TICKET_TYPE_QUEST_QUESTNPC, "Quest Questnpc" },
    { GM_TICKET_TYPE_TECHNICAL, "Technical" },
    { GM_TICKET_TYPE_ACCOUNT_BILLING, "Account Billing" },
    { GM_TICKET_TYPE_CHARACTER, "Character" },
    { 0, NULL }
};

typedef enum {
    TALENT_BURNING_SOUL = 0x017,
    TALENT_IMPROVED_FIRE_WARD = 0x018,
    TALENT_IMPROVED_SCORCH = 0x019,
    TALENT_IMPROVED_FIREBALL = 0x01A,
    TALENT_IMPROVED_FIRE_BLAST = 0x01B,
    TALENT_FLAME_THROWING = 0x01C,
    TALENT_PYROBLAST = 0x01D,
    TALENT_IMPACT = 0x01E,
    TALENT_IMPROVED_FLAMESTRIKE = 0x01F,
    TALENT_BLAST_WAVE = 0x020,
    TALENT_CRITICAL_MASS = 0x021,
    TALENT_IGNITE = 0x022,
    TALENT_FIRE_POWER = 0x023,
    TALENT_COMBUSTION = 0x024,
    TALENT_IMPROVED_FROSTBOLT = 0x025,
    TALENT_FROSTBITE = 0x026,
    TALENT_PIERCING_ICE = 0x03D,
    TALENT_IMPROVED_FROST_NOVA = 0x03E,
    TALENT_IMPROVED_BLIZZARD = 0x03F,
    TALENT_IMPROVED_CONE_OF_COLD = 0x040,
    TALENT_PERMAFROST = 0x041,
    TALENT_FROST_CHANNELING = 0x042,
    TALENT_SHATTER = 0x043,
    TALENT_WINTERS_CHILL = 0x044,
    TALENT_COLD_SNAP = 0x045,
    TALENT_FROST_WARDING = 0x046,
    TALENT_ICE_BARRIER = 0x047,
    TALENT_ICE_BLOCK = 0x048,
    TALENT_ICE_SHARDS = 0x049,
    TALENT_ARCANE_SUBTLETY = 0x04A,
    TALENT_ARCANE_CONCENTRATION = 0x04B,
    TALENT_ARCANE_FOCUS = 0x04C,
    TALENT_ARCANE_MIND = 0x04D,
    TALENT_MAGE_WAND_SPECIALIZATION = 0x04E,
    TALENT_IMPROVED_ARCANE_MISSILES = 0x050,
    TALENT_IMPROVED_ARCANE_EXPLOSION = 0x051,
    TALENT_MAGIC_ATTUNEMENT = 0x052,
    TALENT_IMPROVED_MANA_SHIELD = 0x053,
    TALENT_ARCANE_RESILIENCE = 0x055,
    TALENT_PRESENCE_OF_MIND = 0x056,
    TALENT_ARCANE_POWER = 0x057,
    TALENT_IMPROVED_COUNTERSPELL = 0x058,
    TALENT_DEEP_WOUNDS = 0x079,
    TALENT_WARRIOR_SWORD_SPECIALIZATION = 0x07B,
    TALENT_IMPROVED_HEROIC_STRIKE = 0x07C,
    TALENT_WARRIOR_MACE_SPECIALIZATION = 0x07D,
    TALENT_IMPROVED_CHARGE = 0x07E,
    TALENT_IMPROVED_REND = 0x07F,
    TALENT_IMPROVED_THUNDER_CLAP = 0x080,
    TALENT_IMPROVED_HAMSTRING = 0x081,
    TALENT_WARRIOR_DEFLECTION = 0x082,
    TALENT_IMPROVED_OVERPOWER = 0x083,
    TALENT_AXE_SPECIALIZATION = 0x084,
    TALENT_SWEEPING_STRIKES = 0x085,
    TALENT_POLEARM_SPECIALIZATION = 0x086,
    TALENT_MORTAL_STRIKE = 0x087,
    TALENT_WARRIOR_TWO_HANDED_WEAPON_SPECIALIZATION = 0x088,
    TALENT_ANGER_MANAGEMENT = 0x089,
    TALENT_WARRIOR_ANTICIPATION = 0x08A,
    TALENT_WARRIOR_TOUGHNESS = 0x08C,
    TALENT_IRON_WILL = 0x08D,
    TALENT_IMPROVED_BLOODRAGE = 0x08E,
    TALENT_IMPROVED_TAUNT = 0x08F,
    TALENT_DEFIANCE = 0x090,
    TALENT_IMPROVED_SHIELD_BLOCK = 0x091,
    TALENT_IMPROVED_SUNDER_ARMOR = 0x092,
    TALENT_IMPROVED_REVENGE = 0x093,
    TALENT_SHIELD_SLAM = 0x094,
    TALENT_IMPROVED_SHIELD_BASH = 0x095,
    TALENT_IMPROVED_SHIELD_WALL = 0x096,
    TALENT_IMPROVED_DISARM = 0x097,
    TALENT_CONCUSSION_BLOW = 0x098,
    TALENT_LAST_STAND = 0x099,
    TALENT_IMPROVED_BATTLE_SHOUT = 0x09A,
    TALENT_ENRAGE = 0x09B,
    TALENT_WARRIOR_FLURRY = 0x09C,
    TALENT_CRUELTY = 0x09D,
    TALENT_BOOMING_VOICE = 0x09E,
    TALENT_UNBRIDLED_WRATH = 0x09F,
    TALENT_PIERCING_HOWL = 0x0A0,
    TALENT_IMPROVED_DEMORALIZING_SHOUT = 0x0A1,
    TALENT_DEATH_WISH = 0x0A5,
    TALENT_IMPROVED_CLEAVE = 0x0A6,
    TALENT_BLOODTHIRST = 0x0A7,
    TALENT_IMPROVED_SLAM = 0x0A8,
    TALENT_ROGUE_PRECISION = 0x0B5,
    TALENT_DAGGER_SPECIALIZATION = 0x0B6,
    TALENT_FIST_WEAPON_SPECIALIZATION = 0x0B7,
    TALENT_ROGUE_MACE_SPECIALIZATION = 0x0B8,
    TALENT_ROGUE_LIGHTNING_REFLEXES = 0x0BA,
    TALENT_ROGUE_DEFLECTION = 0x0BB,
    TALENT_IMPROVED_SINISTER_STRIKE = 0x0C9,
    TALENT_IMPROVED_BACKSTAB = 0x0CA,
    TALENT_IMPROVED_GOUGE = 0x0CB,
    TALENT_ENDURANCE = 0x0CC,
    TALENT_ADRENALINE_RUSH = 0x0CD,
    TALENT_IMPROVED_KICK = 0x0CE,
    TALENT_ROGUE_DUAL_WIELD_SPECIALIZATION = 0x0DD,
    TALENT_IMPROVED_SPRINT = 0x0DE,
    TALENT_BLADE_FLURRY = 0x0DF,
    TALENT_MASTER_OF_DECEPTION = 0x0F1,
    TALENT_ROGUE_SWORD_SPECIALIZATION = 0x0F2,
    TALENT_CAMOUFLAGE = 0x0F4,
    TALENT_INITIATIVE = 0x0F5,
    TALENT_SETUP = 0x0F6,
    TALENT_ELUSIVENESS = 0x0F7,
    TALENT_OPPORTUNITY = 0x105,
    TALENT_IMPROVED_SAP = 0x106,
    TALENT_IMPROVED_AMBUSH = 0x107,
    TALENT_DIRTY_DEEDS = 0x109,
    TALENT_IMPROVED_POISONS = 0x10C,
    TALENT_LETHALITY = 0x10D,
    TALENT_MALICE = 0x10E,
    TALENT_REMORSELESS_ATTACKS = 0x110,
    TALENT_RUTHLESSNESS = 0x111,
    TALENT_MURDER = 0x112,
    TALENT_IMPROVED_EVISCERATE = 0x114,
    TALENT_IMPROVED_SLICE_AND_DICE = 0x115,
    TALENT_IMPROVED_EXPOSE_ARMOR = 0x116,
    TALENT_IMPROVED_KIDNEY_SHOT = 0x117,
    TALENT_COLD_BLOOD = 0x118,
    TALENT_RELENTLESS_STRIKES = 0x119,
    TALENT_SEAL_FATE = 0x11B,
    TALENT_PREPARATION = 0x11C,
    TALENT_RIPOSTE = 0x12D,
    TALENT_GHOSTLY_STRIKE = 0x12F,
    TALENT_MARTYRDOM = 0x141,
    TALENT_POWER_INFUSION = 0x142,
    TALENT_MENTAL_AGILITY = 0x155,
    TALENT_UNBREAKABLE_WILL = 0x156,
    TALENT_IMPROVED_POWER_WORD_SHIELD = 0x157,
    TALENT_IMPROVED_POWER_WORD_FORTITUDE = 0x158,
    TALENT_PRIEST_WAND_SPECIALIZATION = 0x159,
    TALENT_IMPROVED_INNER_FIRE = 0x15A,
    TALENT_MEDITATION = 0x15B,
    TALENT_INNER_FOCUS = 0x15C,
    TALENT_IMPROVED_MANA_BURN = 0x15E,
    TALENT_DIVINE_SPIRIT = 0x15F,
    TALENT_SILENT_RESOLVE = 0x160,
    TALENT_INSPIRATION = 0x169,
    TALENT_PREMEDITATION = 0x17D,
    TALENT_VIGOR = 0x17E,
    TALENT_HOLY_SPECIALIZATION = 0x191,
    TALENT_SPIRITUAL_GUIDANCE = 0x192,
    TALENT_SEARING_LIGHT = 0x193,
    TALENT_SPIRITUAL_HEALING = 0x194,
    TALENT_IMPROVED_RENEW = 0x196,
    TALENT_IMPROVED_HEALING = 0x198,
    TALENT_PRIEST_HEALING_FOCUS = 0x19A,
    TALENT_SPELL_WARDING = 0x19B,
    TALENT_IMPROVED_PRAYER_OF_HEALING = 0x19D,
    TALENT_ARCANE_INSTABILITY = 0x1A5,
    TALENT_HOLY_NOVA = 0x1BA,
    TALENT_SHADOW_WEAVING = 0x1CD,
    TALENT_DARKNESS = 0x1CE,
    TALENT_SHADOW_FOCUS = 0x1CF,
    TALENT_BLACKOUT = 0x1D0,
    TALENT_SPIRIT_TAP = 0x1D1,
    TALENT_SHADOW_AFFINITY = 0x1D2,
    TALENT_IMPROVED_MIND_BLAST = 0x1E1,
    TALENT_IMPROVED_SHADOW_WORD_PAIN = 0x1E2,
    TALENT_IMPROVED_FADE = 0x1E3,
    TALENT_VAMPIRIC_EMBRACE = 0x1E4,
    TALENT_MIND_FLAY = 0x1F5,
    TALENT_SHADOWFORM = 0x209,
    TALENT_SILENCE = 0x21D,
    TALENT_IMPROVED_PSYCHIC_SCREAM = 0x21E,
    TALENT_CALL_OF_FLAME = 0x231,
    TALENT_CALL_OF_THUNDER = 0x232,
    TALENT_CONCUSSION = 0x233,
    TALENT_CONVECTION = 0x234,
    TALENT_ELEMENTAL_FURY = 0x235,
    TALENT_IMPROVED_FIRE_TOTEMS = 0x237,
    TALENT_EARTHS_GRASP = 0x23C,
    TALENT_ELEMENTAL_MASTERY = 0x23D,
    TALENT_ELEMENTAL_FOCUS = 0x23E,
    TALENT_REVERBERATION = 0x23F,
    TALENT_ANCESTRAL_HEALING = 0x245,
    TALENT_TOTEMIC_MASTERY = 0x246,
    TALENT_NATURES_GUIDANCE = 0x247,
    TALENT_IMPROVED_HEALING_WAVE = 0x24A,
    TALENT_SHAMAN_HEALING_FOCUS = 0x24B,
    TALENT_RESTORATIVE_TOTEMS = 0x24C,
    TALENT_IMPROVED_REINCARNATION = 0x24D,
    TALENT_MANA_TIDE_TOTEM = 0x24E,
    TALENT_SHAMAN_NATURES_SWIFTNESS = 0x24F,
    TALENT_PURIFICATION = 0x250,
    TALENT_TIDAL_FOCUS = 0x251,
    TALENT_TIDAL_MASTERY = 0x252,
    TALENT_TOTEMIC_FOCUS = 0x253,
    TALENT_SHAMAN_ANTICIPATION = 0x259,
    TALENT_SHAMAN_FLURRY = 0x25A,
    TALENT_IMPROVED_GHOST_WOLF = 0x25D,
    TALENT_IMPROVED_LIGHTNING_SHIELD = 0x25F,
    TALENT_GUARDIAN_TOTEMS = 0x261,
    TALENT_ENHANCING_TOTEMS = 0x262,
    TALENT_ELEMENTAL_WEAPONS = 0x263,
    TALENT_SHAMAN_SHIELD_SPECIALIZATION = 0x264,
    TALENT_THUNDERING_STRIKES = 0x265,
    TALENT_ANCESTRAL_KNOWLEDGE = 0x266,
    TALENT_SHAMAN_TOUGHNESS = 0x267,
    TALENT_PARRY = 0x268,
    TALENT_TWO_HANDED_AXES_AND_MACES = 0x269,
    TALENT_TACTICAL_MASTERY = 0x281,
    TALENT_BLOOD_CRAZE = 0x295,
    TALENT_IMPALE = 0x296,
    TALENT_HEMORRHAGE = 0x2A9,
    TALENT_VILE_POISONS = 0x2AA,
    TALENT_WARRIOR_ONE_HANDED_WEAPON_SPECIALIZATION = 0x2BE,
    TALENT_LIGHTNING_MASTERY = 0x2D1,
    TALENT_ARCTIC_REACH = 0x2E5,
    TALENT_NATURES_GRASP = 0x2F9,
    TALENT_IMPROVED_WRATH = 0x2FA,
    TALENT_IMPROVED_MOONFIRE = 0x2FB,
    TALENT_NATURES_REACH = 0x2FC,
    TALENT_NATURAL_SHAPESHIFTER = 0x30D,
    TALENT_IMPROVED_THORNS = 0x30E,
    TALENT_MOONGLOW = 0x30F,
    TALENT_IMPROVED_STARFIRE = 0x310,
    TALENT_IMPROVED_ENTANGLING_ROOTS = 0x313,
    TALENT_OMEN_OF_CLARITY = 0x314,
    TALENT_NATURES_GRACE = 0x315,
    TALENT_MOONFURY = 0x316,
    TALENT_NATURAL_WEAPONS = 0x317,
    TALENT_DRUID_VENGEANCE = 0x318,
    TALENT_MOONKIN_FORM = 0x319,
    TALENT_DRUID_THICK_HIDE = 0x31A,
    TALENT_FERAL_AGGRESSION = 0x31B,
    TALENT_DRUID_FEROCITY = 0x31C,
    TALENT_BRUTAL_IMPACT = 0x31D,
    TALENT_SHARPENED_CLAWS = 0x31E,
    TALENT_FERAL_INSTINCT = 0x31F,
    TALENT_BLOOD_FRENZY = 0x320,
    TALENT_PRIMAL_FURY = 0x321,
    TALENT_IMPROVED_SHRED = 0x322,
    TALENT_PREDATORY_STRIKES = 0x323,
    TALENT_FERAL_CHARGE = 0x324,
    TALENT_SAVAGE_FURY = 0x325,
    TALENT_FELINE_SWIFTNESS = 0x327,
    TALENT_HEART_OF_THE_WILD = 0x328,
    TALENT_LEADER_OF_THE_PACK = 0x329,
    TALENT_IMPROVED_MARK_OF_THE_WILD = 0x335,
    TALENT_FUROR = 0x336,
    TALENT_NATURES_FOCUS = 0x337,
    TALENT_IMPROVED_HEALING_TOUCH = 0x338,
    TALENT_IMPROVED_REGROWTH = 0x339,
    TALENT_IMPROVED_ENRAGE = 0x33A,
    TALENT_INSECT_SWARM = 0x33B,
    TALENT_GIFT_OF_NATURE = 0x33C,
    TALENT_REFLECTION = 0x33D,
    TALENT_IMPROVED_REJUVENATION = 0x33E,
    TALENT_DRUID_NATURES_SWIFTNESS = 0x33F,
    TALENT_SUBTLETY = 0x349,
    TALENT_IMPROVED_TRANQUILITY = 0x34A,
    TALENT_TRANQUIL_SPIRIT = 0x34B,
    TALENT_SWIFTMEND = 0x34C,
    TALENT_SHADOW_REACH = 0x371,
    TALENT_STORMSTRIKE = 0x385,
    TALENT_IMPROVED_NATURES_GRASP = 0x399,
    TALENT_CATACLYSM = 0x3AD,
    TALENT_BANE = 0x3AF,
    TALENT_IMPROVED_SHADOW_BOLT = 0x3B0,
    TALENT_IMPROVED_IMMOLATE = 0x3C1,
    TALENT_SHADOWBURN = 0x3C3,
    TALENT_DESTRUCTIVE_REACH = 0x3C4,
    TALENT_IMPROVED_SEARING_PAIN = 0x3C5,
    TALENT_EMBERSTORM = 0x3C6,
    TALENT_RUIN = 0x3C7,
    TALENT_CONFLAGRATE = 0x3C8,
    TALENT_DEVASTATION = 0x3D5,
    TALENT_AFTERMATH = 0x3D6,
    TALENT_IMPROVED_FIREBOLT = 0x3D7,
    TALENT_IMPROVED_LASH_OF_PAIN = 0x3D8,
    TALENT_INTENSITY = 0x3D9,
    TALENT_PYROCLASM = 0x3DA,
    TALENT_FEL_CONCENTRATION = 0x3E9,
    TALENT_NIGHTFALL = 0x3EA,
    TALENT_IMPROVED_CORRUPTION = 0x3EB,
    TALENT_IMPROVED_DRAIN_LIFE = 0x3EC,
    TALENT_SUPPRESSION = 0x3ED,
    TALENT_IMPROVED_CURSE_OF_WEAKNESS = 0x3EE,
    TALENT_IMPROVED_LIFE_TAP = 0x3EF,
    TALENT_GRIM_REACH = 0x3FD,
    TALENT_DARK_PACT = 0x3FE,
    TALENT_SIPHON_LIFE = 0x411,
    TALENT_SHADOW_MASTERY = 0x412,
    TALENT_AMPLIFY_CURSE = 0x425,
    TALENT_CURSE_OF_EXHAUSTION = 0x439,
    TALENT_IMPROVED_CURSE_OF_EXHAUSTION = 0x43A,
    TALENT_IMPROVED_DRAIN_SOUL = 0x44D,
    TALENT_IMPROVED_DRAIN_MANA = 0x461,
    TALENT_AGGRESSION = 0x462,
    TALENT_SERRATED_BLADES = 0x463,
    TALENT_INCINERATE = 0x475,
    TALENT_ARCANE_MEDITATION = 0x476,
    TALENT_FAERIE_FIRE_FERAL = 0x48A,
    TALENT_DIVINE_FURY = 0x49D,
    TALENT_MENTAL_STRENGTH = 0x4B1,
    TALENT_FORCE_OF_WILL = 0x4B2,
    TALENT_IMPROVED_HEALTHSTONE = 0x4C5,
    TALENT_IMPROVED_IMP = 0x4C6,
    TALENT_DEMONIC_EMBRACE = 0x4C7,
    TALENT_IMPROVED_HEALTH_FUNNEL = 0x4C8,
    TALENT_IMPROVED_VOIDWALKER = 0x4C9,
    TALENT_FEL_DOMINATION = 0x4CA,
    TALENT_MASTER_SUMMONER = 0x4CB,
    TALENT_FEL_STAMINA = 0x4D9,
    TALENT_FEL_INTELLECT = 0x4DA,
    TALENT_IMPROVED_SUCCUBUS = 0x4DB,
    TALENT_MASTER_DEMONOLOGIST = 0x4DC,
    TALENT_IMPROVED_FIRESTONE = 0x4ED,
    TALENT_UNHOLY_POWER = 0x4EE,
    TALENT_IMPROVED_SPELLSTONE = 0x4EF,
    TALENT_DEMONIC_SACRIFICE = 0x501,
    TALENT_SOUL_LINK = 0x502,
    TALENT_IMPROVED_ENSLAVE_DEMON = 0x503,
    TALENT_IMPROVED_CURSE_OF_AGONY = 0x504,
    TALENT_HUMANOID_SLAYING = 0x515,
    TALENT_HUNTER_LIGHTNING_REFLEXES = 0x517,
    TALENT_ENTRAPMENT = 0x518,
    TALENT_IMPROVED_WING_CLIP = 0x519,
    TALENT_CLEVER_TRAPS = 0x51A,
    TALENT_DETERRENCE = 0x51C,
    TALENT_IMPROVED_FEIGN_DEATH = 0x51D,
    TALENT_SUREFOOTED = 0x51E,
    TALENT_HUNTER_DEFLECTION = 0x51F,
    TALENT_COUNTERATTACK = 0x520,
    TALENT_KILLER_INSTINCT = 0x529,
    TALENT_TRAP_MASTERY = 0x52A,
    TALENT_WYVERN_STING = 0x52D,
    TALENT_IMPROVED_CONCUSSIVE_SHOT = 0x53D,
    TALENT_EFFICIENCY = 0x53E,
    TALENT_IMPROVED_HUNTERS_MARK = 0x53F,
    TALENT_LETHAL_SHOTS = 0x540,
    TALENT_AIMED_SHOT = 0x541,
    TALENT_IMPROVED_ARCANE_SHOT = 0x542,
    TALENT_BARRAGE = 0x543,
    TALENT_IMPROVED_SERPENT_STING = 0x544,
    TALENT_MORTAL_SHOTS = 0x545,
    TALENT_IMPROVED_SCORPID_STING = 0x547,
    TALENT_HAWK_EYE = 0x548,
    TALENT_SCATTER_SHOT = 0x549,
    TALENT_TRUESHOT_AURA = 0x551,
    TALENT_RANGED_WEAPON_SPECIALIZATION = 0x552,
    TALENT_IMPROVED_ASPECT_OF_THE_MONKEY = 0x565,
    TALENT_IMPROVED_ASPECT_OF_THE_HAWK = 0x566,
    TALENT_PATHFINDING = 0x568,
    TALENT_IMPROVED_MEND_PET = 0x569,
    TALENT_BESTIAL_WRATH = 0x56A,
    TALENT_INTIMIDATION = 0x56B,
    TALENT_SPIRIT_BOND = 0x56C,
    TALENT_ENDURANCE_TRAINING = 0x56D,
    TALENT_BESTIAL_DISCIPLINE = 0x56E,
    TALENT_BESTIAL_SWIFTNESS = 0x56F,
    TALENT_HUNTER_FEROCITY = 0x571,
    TALENT_HUNTER_THICK_HIDE = 0x573,
    TALENT_UNLEASHED_FURY = 0x574,
    TALENT_FRENZY = 0x575,
    TALENT_IMPROVED_BLESSING_OF_MIGHT = 0x579,
    TALENT_PALADIN_VENGEANCE = 0x57A,
    TALENT_PALADIN_DEFLECTION = 0x57B,
    TALENT_IMPROVED_RETRIBUTION_AURA = 0x57D,
    TALENT_BENEDICTION = 0x57F,
    TALENT_SANCTITY_AURA = 0x581,
    TALENT_PALADIN_TWO_HANDED_WEAPON_SPECIALIZATION = 0x582,
    TALENT_CONVICTION = 0x583,
    TALENT_REDOUBT = 0x58D,
    TALENT_IMPROVED_DEVOTION_AURA = 0x58E,
    TALENT_PALADIN_TOUGHNESS = 0x58F,
    TALENT_PALADIN_SHIELD_SPECIALIZATION = 0x590,
    TALENT_GUARDIANS_FAVOR = 0x591,
    TALENT_RECKONING = 0x592,
    TALENT_PALADIN_ONE_HANDED_WEAPON_SPECIALIZATION = 0x595,
    TALENT_HOLY_SHIELD = 0x596,
    TALENT_BLESSING_OF_SANCTUARY = 0x597,
    TALENT_SPIRITUAL_FOCUS = 0x598,
    TALENT_DIVINE_FAVOR = 0x599,
    TALENT_CONSECRATION = 0x59B,
    TALENT_REPENTANCE = 0x5A1,
    TALENT_BLESSING_OF_KINGS = 0x5A2,
    TALENT_IMPROVED_LAY_ON_HANDS = 0x5A3,
    TALENT_HEALING_LIGHT = 0x5A4,
    TALENT_IMPROVED_BLESSING_OF_WISDOM = 0x5A6,
    TALENT_DIVINE_INTELLECT = 0x5A9,
    TALENT_DIVINE_STRENGTH = 0x5AA,
    TALENT_ILLUMINATION = 0x5B5,
    TALENT_IMPROVED_SEAL_OF_RIGHTEOUSNESS = 0x5B7,
    TALENT_IMPROVED_SEAL_OF_THE_CRUSADER = 0x5B8,
    TALENT_LASTING_JUDGEMENT = 0x5B9,
    TALENT_SEAL_OF_COMMAND = 0x5C9,
    TALENT_IMPROVED_RIGHTEOUS_FURY = 0x5DD,
    TALENT_HOLY_SHOCK = 0x5DE,
    TALENT_IMPROVED_HAMMER_OF_JUSTICE = 0x5F1,
    TALENT_IMPROVED_BERSERKER_RAGE = 0x605,
    TALENT_IMPROVED_EXECUTE = 0x606,
    TALENT_IMPROVED_INTERCEPT = 0x607,
    TALENT_SPIRIT_OF_REDEMPTION = 0x619,
    TALENT_WARRIOR_DUAL_WIELD_SPECIALIZATION = 0x62D,
    TALENT_WARRIOR_SHIELD_SPECIALIZATION = 0x641,
    TALENT_SAVAGE_STRIKES = 0x655,
    TALENT_SURVIVALIST = 0x656,
    TALENT_MONSTER_SLAYING = 0x657,
    TALENT_IMPROVED_EYES_OF_THE_BEAST = 0x658,
    TALENT_IMPROVED_REVIVE_PET = 0x659,
    TALENT_IMPROVED_CONCENTRATION_AURA = 0x65A,
    TALENT_HOLY_POWER = 0x65B,
    TALENT_UNYIELDING_FAITH = 0x65C,
    TALENT_PALADIN_ANTICIPATION = 0x65D,
    TALENT_PALADIN_PRECISION = 0x65E,
    TALENT_IMPROVED_JUDGEMENT = 0x65F,
    TALENT_EYE_FOR_AN_EYE = 0x660,
    TALENT_VINDICATION = 0x661,
    TALENT_PURSUIT_OF_JUSTICE = 0x662,
    TALENT_HOLY_REACH = 0x663,
    TALENT_BLESSED_RECOVERY = 0x664,
    TALENT_LIGHTWELL = 0x665,
    TALENT_IMPROVED_VAMPIRIC_EMBRACE = 0x666,
    TALENT_MASTER_OF_ELEMENTS = 0x667,
    TALENT_ELEMENTAL_WARDING = 0x668,
    TALENT_STORM_REACH = 0x669,
    TALENT_EYE_OF_THE_STORM = 0x66A,
    TALENT_WEAPON_MASTERY = 0x66B,
    TALENT_ELEMENTAL_DEVASTATION = 0x66D,
    TALENT_HEALING_GRACE = 0x66E,
    TALENT_IMPROVED_WEAPON_TOTEMS = 0x66F,
    TALENT_HEALING_WAY = 0x670,
    TALENT_ELEMENTAL_PRECISION = 0x671,
    TALENT_MAGIC_ABSORPTION = 0x672,
    TALENT_SLEIGHT_OF_HAND = 0x6A4,
    TALENT_HEIGHTENED_SENSES = 0x6A5,
    TALENT_DEADLINESS = 0x6A6,
    TALENT_WEAPON_EXPERTISE = 0x6A7,
} e_talent;
static const value_string e_talent_strings[] =  {
    { TALENT_BURNING_SOUL, "Burning Soul" },
    { TALENT_IMPROVED_FIRE_WARD, "Improved Fire Ward" },
    { TALENT_IMPROVED_SCORCH, "Improved Scorch" },
    { TALENT_IMPROVED_FIREBALL, "Improved Fireball" },
    { TALENT_IMPROVED_FIRE_BLAST, "Improved Fire Blast" },
    { TALENT_FLAME_THROWING, "Flame Throwing" },
    { TALENT_PYROBLAST, "Pyroblast" },
    { TALENT_IMPACT, "Impact" },
    { TALENT_IMPROVED_FLAMESTRIKE, "Improved Flamestrike" },
    { TALENT_BLAST_WAVE, "Blast Wave" },
    { TALENT_CRITICAL_MASS, "Critical Mass" },
    { TALENT_IGNITE, "Ignite" },
    { TALENT_FIRE_POWER, "Fire Power" },
    { TALENT_COMBUSTION, "Combustion" },
    { TALENT_IMPROVED_FROSTBOLT, "Improved Frostbolt" },
    { TALENT_FROSTBITE, "Frostbite" },
    { TALENT_PIERCING_ICE, "Piercing Ice" },
    { TALENT_IMPROVED_FROST_NOVA, "Improved Frost Nova" },
    { TALENT_IMPROVED_BLIZZARD, "Improved Blizzard" },
    { TALENT_IMPROVED_CONE_OF_COLD, "Improved Cone Of Cold" },
    { TALENT_PERMAFROST, "Permafrost" },
    { TALENT_FROST_CHANNELING, "Frost Channeling" },
    { TALENT_SHATTER, "Shatter" },
    { TALENT_WINTERS_CHILL, "Winters Chill" },
    { TALENT_COLD_SNAP, "Cold Snap" },
    { TALENT_FROST_WARDING, "Frost Warding" },
    { TALENT_ICE_BARRIER, "Ice Barrier" },
    { TALENT_ICE_BLOCK, "Ice Block" },
    { TALENT_ICE_SHARDS, "Ice Shards" },
    { TALENT_ARCANE_SUBTLETY, "Arcane Subtlety" },
    { TALENT_ARCANE_CONCENTRATION, "Arcane Concentration" },
    { TALENT_ARCANE_FOCUS, "Arcane Focus" },
    { TALENT_ARCANE_MIND, "Arcane Mind" },
    { TALENT_MAGE_WAND_SPECIALIZATION, "Mage Wand Specialization" },
    { TALENT_IMPROVED_ARCANE_MISSILES, "Improved Arcane Missiles" },
    { TALENT_IMPROVED_ARCANE_EXPLOSION, "Improved Arcane Explosion" },
    { TALENT_MAGIC_ATTUNEMENT, "Magic Attunement" },
    { TALENT_IMPROVED_MANA_SHIELD, "Improved Mana Shield" },
    { TALENT_ARCANE_RESILIENCE, "Arcane Resilience" },
    { TALENT_PRESENCE_OF_MIND, "Presence Of Mind" },
    { TALENT_ARCANE_POWER, "Arcane Power" },
    { TALENT_IMPROVED_COUNTERSPELL, "Improved Counterspell" },
    { TALENT_DEEP_WOUNDS, "Deep Wounds" },
    { TALENT_WARRIOR_SWORD_SPECIALIZATION, "Warrior Sword Specialization" },
    { TALENT_IMPROVED_HEROIC_STRIKE, "Improved Heroic Strike" },
    { TALENT_WARRIOR_MACE_SPECIALIZATION, "Warrior Mace Specialization" },
    { TALENT_IMPROVED_CHARGE, "Improved Charge" },
    { TALENT_IMPROVED_REND, "Improved Rend" },
    { TALENT_IMPROVED_THUNDER_CLAP, "Improved Thunder Clap" },
    { TALENT_IMPROVED_HAMSTRING, "Improved Hamstring" },
    { TALENT_WARRIOR_DEFLECTION, "Warrior Deflection" },
    { TALENT_IMPROVED_OVERPOWER, "Improved Overpower" },
    { TALENT_AXE_SPECIALIZATION, "Axe Specialization" },
    { TALENT_SWEEPING_STRIKES, "Sweeping Strikes" },
    { TALENT_POLEARM_SPECIALIZATION, "Polearm Specialization" },
    { TALENT_MORTAL_STRIKE, "Mortal Strike" },
    { TALENT_WARRIOR_TWO_HANDED_WEAPON_SPECIALIZATION, "Warrior Two Handed Weapon Specialization" },
    { TALENT_ANGER_MANAGEMENT, "Anger Management" },
    { TALENT_WARRIOR_ANTICIPATION, "Warrior Anticipation" },
    { TALENT_WARRIOR_TOUGHNESS, "Warrior Toughness" },
    { TALENT_IRON_WILL, "Iron Will" },
    { TALENT_IMPROVED_BLOODRAGE, "Improved Bloodrage" },
    { TALENT_IMPROVED_TAUNT, "Improved Taunt" },
    { TALENT_DEFIANCE, "Defiance" },
    { TALENT_IMPROVED_SHIELD_BLOCK, "Improved Shield Block" },
    { TALENT_IMPROVED_SUNDER_ARMOR, "Improved Sunder Armor" },
    { TALENT_IMPROVED_REVENGE, "Improved Revenge" },
    { TALENT_SHIELD_SLAM, "Shield Slam" },
    { TALENT_IMPROVED_SHIELD_BASH, "Improved Shield Bash" },
    { TALENT_IMPROVED_SHIELD_WALL, "Improved Shield Wall" },
    { TALENT_IMPROVED_DISARM, "Improved Disarm" },
    { TALENT_CONCUSSION_BLOW, "Concussion Blow" },
    { TALENT_LAST_STAND, "Last Stand" },
    { TALENT_IMPROVED_BATTLE_SHOUT, "Improved Battle Shout" },
    { TALENT_ENRAGE, "Enrage" },
    { TALENT_WARRIOR_FLURRY, "Warrior Flurry" },
    { TALENT_CRUELTY, "Cruelty" },
    { TALENT_BOOMING_VOICE, "Booming Voice" },
    { TALENT_UNBRIDLED_WRATH, "Unbridled Wrath" },
    { TALENT_PIERCING_HOWL, "Piercing Howl" },
    { TALENT_IMPROVED_DEMORALIZING_SHOUT, "Improved Demoralizing Shout" },
    { TALENT_DEATH_WISH, "Death Wish" },
    { TALENT_IMPROVED_CLEAVE, "Improved Cleave" },
    { TALENT_BLOODTHIRST, "Bloodthirst" },
    { TALENT_IMPROVED_SLAM, "Improved Slam" },
    { TALENT_ROGUE_PRECISION, "Rogue Precision" },
    { TALENT_DAGGER_SPECIALIZATION, "Dagger Specialization" },
    { TALENT_FIST_WEAPON_SPECIALIZATION, "Fist Weapon Specialization" },
    { TALENT_ROGUE_MACE_SPECIALIZATION, "Rogue Mace Specialization" },
    { TALENT_ROGUE_LIGHTNING_REFLEXES, "Rogue Lightning Reflexes" },
    { TALENT_ROGUE_DEFLECTION, "Rogue Deflection" },
    { TALENT_IMPROVED_SINISTER_STRIKE, "Improved Sinister Strike" },
    { TALENT_IMPROVED_BACKSTAB, "Improved Backstab" },
    { TALENT_IMPROVED_GOUGE, "Improved Gouge" },
    { TALENT_ENDURANCE, "Endurance" },
    { TALENT_ADRENALINE_RUSH, "Adrenaline Rush" },
    { TALENT_IMPROVED_KICK, "Improved Kick" },
    { TALENT_ROGUE_DUAL_WIELD_SPECIALIZATION, "Rogue Dual Wield Specialization" },
    { TALENT_IMPROVED_SPRINT, "Improved Sprint" },
    { TALENT_BLADE_FLURRY, "Blade Flurry" },
    { TALENT_MASTER_OF_DECEPTION, "Master Of Deception" },
    { TALENT_ROGUE_SWORD_SPECIALIZATION, "Rogue Sword Specialization" },
    { TALENT_CAMOUFLAGE, "Camouflage" },
    { TALENT_INITIATIVE, "Initiative" },
    { TALENT_SETUP, "Setup" },
    { TALENT_ELUSIVENESS, "Elusiveness" },
    { TALENT_OPPORTUNITY, "Opportunity" },
    { TALENT_IMPROVED_SAP, "Improved Sap" },
    { TALENT_IMPROVED_AMBUSH, "Improved Ambush" },
    { TALENT_DIRTY_DEEDS, "Dirty Deeds" },
    { TALENT_IMPROVED_POISONS, "Improved Poisons" },
    { TALENT_LETHALITY, "Lethality" },
    { TALENT_MALICE, "Malice" },
    { TALENT_REMORSELESS_ATTACKS, "Remorseless Attacks" },
    { TALENT_RUTHLESSNESS, "Ruthlessness" },
    { TALENT_MURDER, "Murder" },
    { TALENT_IMPROVED_EVISCERATE, "Improved Eviscerate" },
    { TALENT_IMPROVED_SLICE_AND_DICE, "Improved Slice And Dice" },
    { TALENT_IMPROVED_EXPOSE_ARMOR, "Improved Expose Armor" },
    { TALENT_IMPROVED_KIDNEY_SHOT, "Improved Kidney Shot" },
    { TALENT_COLD_BLOOD, "Cold Blood" },
    { TALENT_RELENTLESS_STRIKES, "Relentless Strikes" },
    { TALENT_SEAL_FATE, "Seal Fate" },
    { TALENT_PREPARATION, "Preparation" },
    { TALENT_RIPOSTE, "Riposte" },
    { TALENT_GHOSTLY_STRIKE, "Ghostly Strike" },
    { TALENT_MARTYRDOM, "Martyrdom" },
    { TALENT_POWER_INFUSION, "Power Infusion" },
    { TALENT_MENTAL_AGILITY, "Mental Agility" },
    { TALENT_UNBREAKABLE_WILL, "Unbreakable Will" },
    { TALENT_IMPROVED_POWER_WORD_SHIELD, "Improved Power Word Shield" },
    { TALENT_IMPROVED_POWER_WORD_FORTITUDE, "Improved Power Word Fortitude" },
    { TALENT_PRIEST_WAND_SPECIALIZATION, "Priest Wand Specialization" },
    { TALENT_IMPROVED_INNER_FIRE, "Improved Inner Fire" },
    { TALENT_MEDITATION, "Meditation" },
    { TALENT_INNER_FOCUS, "Inner Focus" },
    { TALENT_IMPROVED_MANA_BURN, "Improved Mana Burn" },
    { TALENT_DIVINE_SPIRIT, "Divine Spirit" },
    { TALENT_SILENT_RESOLVE, "Silent Resolve" },
    { TALENT_INSPIRATION, "Inspiration" },
    { TALENT_PREMEDITATION, "Premeditation" },
    { TALENT_VIGOR, "Vigor" },
    { TALENT_HOLY_SPECIALIZATION, "Holy Specialization" },
    { TALENT_SPIRITUAL_GUIDANCE, "Spiritual Guidance" },
    { TALENT_SEARING_LIGHT, "Searing Light" },
    { TALENT_SPIRITUAL_HEALING, "Spiritual Healing" },
    { TALENT_IMPROVED_RENEW, "Improved Renew" },
    { TALENT_IMPROVED_HEALING, "Improved Healing" },
    { TALENT_PRIEST_HEALING_FOCUS, "Priest Healing Focus" },
    { TALENT_SPELL_WARDING, "Spell Warding" },
    { TALENT_IMPROVED_PRAYER_OF_HEALING, "Improved Prayer Of Healing" },
    { TALENT_ARCANE_INSTABILITY, "Arcane Instability" },
    { TALENT_HOLY_NOVA, "Holy Nova" },
    { TALENT_SHADOW_WEAVING, "Shadow Weaving" },
    { TALENT_DARKNESS, "Darkness" },
    { TALENT_SHADOW_FOCUS, "Shadow Focus" },
    { TALENT_BLACKOUT, "Blackout" },
    { TALENT_SPIRIT_TAP, "Spirit Tap" },
    { TALENT_SHADOW_AFFINITY, "Shadow Affinity" },
    { TALENT_IMPROVED_MIND_BLAST, "Improved Mind Blast" },
    { TALENT_IMPROVED_SHADOW_WORD_PAIN, "Improved Shadow Word Pain" },
    { TALENT_IMPROVED_FADE, "Improved Fade" },
    { TALENT_VAMPIRIC_EMBRACE, "Vampiric Embrace" },
    { TALENT_MIND_FLAY, "Mind Flay" },
    { TALENT_SHADOWFORM, "Shadowform" },
    { TALENT_SILENCE, "Silence" },
    { TALENT_IMPROVED_PSYCHIC_SCREAM, "Improved Psychic Scream" },
    { TALENT_CALL_OF_FLAME, "Call Of Flame" },
    { TALENT_CALL_OF_THUNDER, "Call Of Thunder" },
    { TALENT_CONCUSSION, "Concussion" },
    { TALENT_CONVECTION, "Convection" },
    { TALENT_ELEMENTAL_FURY, "Elemental Fury" },
    { TALENT_IMPROVED_FIRE_TOTEMS, "Improved Fire Totems" },
    { TALENT_EARTHS_GRASP, "Earths Grasp" },
    { TALENT_ELEMENTAL_MASTERY, "Elemental Mastery" },
    { TALENT_ELEMENTAL_FOCUS, "Elemental Focus" },
    { TALENT_REVERBERATION, "Reverberation" },
    { TALENT_ANCESTRAL_HEALING, "Ancestral Healing" },
    { TALENT_TOTEMIC_MASTERY, "Totemic Mastery" },
    { TALENT_NATURES_GUIDANCE, "Natures Guidance" },
    { TALENT_IMPROVED_HEALING_WAVE, "Improved Healing Wave" },
    { TALENT_SHAMAN_HEALING_FOCUS, "Shaman Healing Focus" },
    { TALENT_RESTORATIVE_TOTEMS, "Restorative Totems" },
    { TALENT_IMPROVED_REINCARNATION, "Improved Reincarnation" },
    { TALENT_MANA_TIDE_TOTEM, "Mana Tide Totem" },
    { TALENT_SHAMAN_NATURES_SWIFTNESS, "Shaman Natures Swiftness" },
    { TALENT_PURIFICATION, "Purification" },
    { TALENT_TIDAL_FOCUS, "Tidal Focus" },
    { TALENT_TIDAL_MASTERY, "Tidal Mastery" },
    { TALENT_TOTEMIC_FOCUS, "Totemic Focus" },
    { TALENT_SHAMAN_ANTICIPATION, "Shaman Anticipation" },
    { TALENT_SHAMAN_FLURRY, "Shaman Flurry" },
    { TALENT_IMPROVED_GHOST_WOLF, "Improved Ghost Wolf" },
    { TALENT_IMPROVED_LIGHTNING_SHIELD, "Improved Lightning Shield" },
    { TALENT_GUARDIAN_TOTEMS, "Guardian Totems" },
    { TALENT_ENHANCING_TOTEMS, "Enhancing Totems" },
    { TALENT_ELEMENTAL_WEAPONS, "Elemental Weapons" },
    { TALENT_SHAMAN_SHIELD_SPECIALIZATION, "Shaman Shield Specialization" },
    { TALENT_THUNDERING_STRIKES, "Thundering Strikes" },
    { TALENT_ANCESTRAL_KNOWLEDGE, "Ancestral Knowledge" },
    { TALENT_SHAMAN_TOUGHNESS, "Shaman Toughness" },
    { TALENT_PARRY, "Parry" },
    { TALENT_TWO_HANDED_AXES_AND_MACES, "Two Handed Axes And Maces" },
    { TALENT_TACTICAL_MASTERY, "Tactical Mastery" },
    { TALENT_BLOOD_CRAZE, "Blood Craze" },
    { TALENT_IMPALE, "Impale" },
    { TALENT_HEMORRHAGE, "Hemorrhage" },
    { TALENT_VILE_POISONS, "Vile Poisons" },
    { TALENT_WARRIOR_ONE_HANDED_WEAPON_SPECIALIZATION, "Warrior One Handed Weapon Specialization" },
    { TALENT_LIGHTNING_MASTERY, "Lightning Mastery" },
    { TALENT_ARCTIC_REACH, "Arctic Reach" },
    { TALENT_NATURES_GRASP, "Natures Grasp" },
    { TALENT_IMPROVED_WRATH, "Improved Wrath" },
    { TALENT_IMPROVED_MOONFIRE, "Improved Moonfire" },
    { TALENT_NATURES_REACH, "Natures Reach" },
    { TALENT_NATURAL_SHAPESHIFTER, "Natural Shapeshifter" },
    { TALENT_IMPROVED_THORNS, "Improved Thorns" },
    { TALENT_MOONGLOW, "Moonglow" },
    { TALENT_IMPROVED_STARFIRE, "Improved Starfire" },
    { TALENT_IMPROVED_ENTANGLING_ROOTS, "Improved Entangling Roots" },
    { TALENT_OMEN_OF_CLARITY, "Omen Of Clarity" },
    { TALENT_NATURES_GRACE, "Natures Grace" },
    { TALENT_MOONFURY, "Moonfury" },
    { TALENT_NATURAL_WEAPONS, "Natural Weapons" },
    { TALENT_DRUID_VENGEANCE, "Druid Vengeance" },
    { TALENT_MOONKIN_FORM, "Moonkin Form" },
    { TALENT_DRUID_THICK_HIDE, "Druid Thick Hide" },
    { TALENT_FERAL_AGGRESSION, "Feral Aggression" },
    { TALENT_DRUID_FEROCITY, "Druid Ferocity" },
    { TALENT_BRUTAL_IMPACT, "Brutal Impact" },
    { TALENT_SHARPENED_CLAWS, "Sharpened Claws" },
    { TALENT_FERAL_INSTINCT, "Feral Instinct" },
    { TALENT_BLOOD_FRENZY, "Blood Frenzy" },
    { TALENT_PRIMAL_FURY, "Primal Fury" },
    { TALENT_IMPROVED_SHRED, "Improved Shred" },
    { TALENT_PREDATORY_STRIKES, "Predatory Strikes" },
    { TALENT_FERAL_CHARGE, "Feral Charge" },
    { TALENT_SAVAGE_FURY, "Savage Fury" },
    { TALENT_FELINE_SWIFTNESS, "Feline Swiftness" },
    { TALENT_HEART_OF_THE_WILD, "Heart Of The Wild" },
    { TALENT_LEADER_OF_THE_PACK, "Leader Of The Pack" },
    { TALENT_IMPROVED_MARK_OF_THE_WILD, "Improved Mark Of The Wild" },
    { TALENT_FUROR, "Furor" },
    { TALENT_NATURES_FOCUS, "Natures Focus" },
    { TALENT_IMPROVED_HEALING_TOUCH, "Improved Healing Touch" },
    { TALENT_IMPROVED_REGROWTH, "Improved Regrowth" },
    { TALENT_IMPROVED_ENRAGE, "Improved Enrage" },
    { TALENT_INSECT_SWARM, "Insect Swarm" },
    { TALENT_GIFT_OF_NATURE, "Gift Of Nature" },
    { TALENT_REFLECTION, "Reflection" },
    { TALENT_IMPROVED_REJUVENATION, "Improved Rejuvenation" },
    { TALENT_DRUID_NATURES_SWIFTNESS, "Druid Natures Swiftness" },
    { TALENT_SUBTLETY, "Subtlety" },
    { TALENT_IMPROVED_TRANQUILITY, "Improved Tranquility" },
    { TALENT_TRANQUIL_SPIRIT, "Tranquil Spirit" },
    { TALENT_SWIFTMEND, "Swiftmend" },
    { TALENT_SHADOW_REACH, "Shadow Reach" },
    { TALENT_STORMSTRIKE, "Stormstrike" },
    { TALENT_IMPROVED_NATURES_GRASP, "Improved Natures Grasp" },
    { TALENT_CATACLYSM, "Cataclysm" },
    { TALENT_BANE, "Bane" },
    { TALENT_IMPROVED_SHADOW_BOLT, "Improved Shadow Bolt" },
    { TALENT_IMPROVED_IMMOLATE, "Improved Immolate" },
    { TALENT_SHADOWBURN, "Shadowburn" },
    { TALENT_DESTRUCTIVE_REACH, "Destructive Reach" },
    { TALENT_IMPROVED_SEARING_PAIN, "Improved Searing Pain" },
    { TALENT_EMBERSTORM, "Emberstorm" },
    { TALENT_RUIN, "Ruin" },
    { TALENT_CONFLAGRATE, "Conflagrate" },
    { TALENT_DEVASTATION, "Devastation" },
    { TALENT_AFTERMATH, "Aftermath" },
    { TALENT_IMPROVED_FIREBOLT, "Improved Firebolt" },
    { TALENT_IMPROVED_LASH_OF_PAIN, "Improved Lash Of Pain" },
    { TALENT_INTENSITY, "Intensity" },
    { TALENT_PYROCLASM, "Pyroclasm" },
    { TALENT_FEL_CONCENTRATION, "Fel Concentration" },
    { TALENT_NIGHTFALL, "Nightfall" },
    { TALENT_IMPROVED_CORRUPTION, "Improved Corruption" },
    { TALENT_IMPROVED_DRAIN_LIFE, "Improved Drain Life" },
    { TALENT_SUPPRESSION, "Suppression" },
    { TALENT_IMPROVED_CURSE_OF_WEAKNESS, "Improved Curse Of Weakness" },
    { TALENT_IMPROVED_LIFE_TAP, "Improved Life Tap" },
    { TALENT_GRIM_REACH, "Grim Reach" },
    { TALENT_DARK_PACT, "Dark Pact" },
    { TALENT_SIPHON_LIFE, "Siphon Life" },
    { TALENT_SHADOW_MASTERY, "Shadow Mastery" },
    { TALENT_AMPLIFY_CURSE, "Amplify Curse" },
    { TALENT_CURSE_OF_EXHAUSTION, "Curse Of Exhaustion" },
    { TALENT_IMPROVED_CURSE_OF_EXHAUSTION, "Improved Curse Of Exhaustion" },
    { TALENT_IMPROVED_DRAIN_SOUL, "Improved Drain Soul" },
    { TALENT_IMPROVED_DRAIN_MANA, "Improved Drain Mana" },
    { TALENT_AGGRESSION, "Aggression" },
    { TALENT_SERRATED_BLADES, "Serrated Blades" },
    { TALENT_INCINERATE, "Incinerate" },
    { TALENT_ARCANE_MEDITATION, "Arcane Meditation" },
    { TALENT_FAERIE_FIRE_FERAL, "Faerie Fire Feral" },
    { TALENT_DIVINE_FURY, "Divine Fury" },
    { TALENT_MENTAL_STRENGTH, "Mental Strength" },
    { TALENT_FORCE_OF_WILL, "Force Of Will" },
    { TALENT_IMPROVED_HEALTHSTONE, "Improved Healthstone" },
    { TALENT_IMPROVED_IMP, "Improved Imp" },
    { TALENT_DEMONIC_EMBRACE, "Demonic Embrace" },
    { TALENT_IMPROVED_HEALTH_FUNNEL, "Improved Health Funnel" },
    { TALENT_IMPROVED_VOIDWALKER, "Improved Voidwalker" },
    { TALENT_FEL_DOMINATION, "Fel Domination" },
    { TALENT_MASTER_SUMMONER, "Master Summoner" },
    { TALENT_FEL_STAMINA, "Fel Stamina" },
    { TALENT_FEL_INTELLECT, "Fel Intellect" },
    { TALENT_IMPROVED_SUCCUBUS, "Improved Succubus" },
    { TALENT_MASTER_DEMONOLOGIST, "Master Demonologist" },
    { TALENT_IMPROVED_FIRESTONE, "Improved Firestone" },
    { TALENT_UNHOLY_POWER, "Unholy Power" },
    { TALENT_IMPROVED_SPELLSTONE, "Improved Spellstone" },
    { TALENT_DEMONIC_SACRIFICE, "Demonic Sacrifice" },
    { TALENT_SOUL_LINK, "Soul Link" },
    { TALENT_IMPROVED_ENSLAVE_DEMON, "Improved Enslave Demon" },
    { TALENT_IMPROVED_CURSE_OF_AGONY, "Improved Curse Of Agony" },
    { TALENT_HUMANOID_SLAYING, "Humanoid Slaying" },
    { TALENT_HUNTER_LIGHTNING_REFLEXES, "Hunter Lightning Reflexes" },
    { TALENT_ENTRAPMENT, "Entrapment" },
    { TALENT_IMPROVED_WING_CLIP, "Improved Wing Clip" },
    { TALENT_CLEVER_TRAPS, "Clever Traps" },
    { TALENT_DETERRENCE, "Deterrence" },
    { TALENT_IMPROVED_FEIGN_DEATH, "Improved Feign Death" },
    { TALENT_SUREFOOTED, "Surefooted" },
    { TALENT_HUNTER_DEFLECTION, "Hunter Deflection" },
    { TALENT_COUNTERATTACK, "Counterattack" },
    { TALENT_KILLER_INSTINCT, "Killer Instinct" },
    { TALENT_TRAP_MASTERY, "Trap Mastery" },
    { TALENT_WYVERN_STING, "Wyvern Sting" },
    { TALENT_IMPROVED_CONCUSSIVE_SHOT, "Improved Concussive Shot" },
    { TALENT_EFFICIENCY, "Efficiency" },
    { TALENT_IMPROVED_HUNTERS_MARK, "Improved Hunters Mark" },
    { TALENT_LETHAL_SHOTS, "Lethal Shots" },
    { TALENT_AIMED_SHOT, "Aimed Shot" },
    { TALENT_IMPROVED_ARCANE_SHOT, "Improved Arcane Shot" },
    { TALENT_BARRAGE, "Barrage" },
    { TALENT_IMPROVED_SERPENT_STING, "Improved Serpent Sting" },
    { TALENT_MORTAL_SHOTS, "Mortal Shots" },
    { TALENT_IMPROVED_SCORPID_STING, "Improved Scorpid Sting" },
    { TALENT_HAWK_EYE, "Hawk Eye" },
    { TALENT_SCATTER_SHOT, "Scatter Shot" },
    { TALENT_TRUESHOT_AURA, "Trueshot Aura" },
    { TALENT_RANGED_WEAPON_SPECIALIZATION, "Ranged Weapon Specialization" },
    { TALENT_IMPROVED_ASPECT_OF_THE_MONKEY, "Improved Aspect Of The Monkey" },
    { TALENT_IMPROVED_ASPECT_OF_THE_HAWK, "Improved Aspect Of The Hawk" },
    { TALENT_PATHFINDING, "Pathfinding" },
    { TALENT_IMPROVED_MEND_PET, "Improved Mend Pet" },
    { TALENT_BESTIAL_WRATH, "Bestial Wrath" },
    { TALENT_INTIMIDATION, "Intimidation" },
    { TALENT_SPIRIT_BOND, "Spirit Bond" },
    { TALENT_ENDURANCE_TRAINING, "Endurance Training" },
    { TALENT_BESTIAL_DISCIPLINE, "Bestial Discipline" },
    { TALENT_BESTIAL_SWIFTNESS, "Bestial Swiftness" },
    { TALENT_HUNTER_FEROCITY, "Hunter Ferocity" },
    { TALENT_HUNTER_THICK_HIDE, "Hunter Thick Hide" },
    { TALENT_UNLEASHED_FURY, "Unleashed Fury" },
    { TALENT_FRENZY, "Frenzy" },
    { TALENT_IMPROVED_BLESSING_OF_MIGHT, "Improved Blessing Of Might" },
    { TALENT_PALADIN_VENGEANCE, "Paladin Vengeance" },
    { TALENT_PALADIN_DEFLECTION, "Paladin Deflection" },
    { TALENT_IMPROVED_RETRIBUTION_AURA, "Improved Retribution Aura" },
    { TALENT_BENEDICTION, "Benediction" },
    { TALENT_SANCTITY_AURA, "Sanctity Aura" },
    { TALENT_PALADIN_TWO_HANDED_WEAPON_SPECIALIZATION, "Paladin Two Handed Weapon Specialization" },
    { TALENT_CONVICTION, "Conviction" },
    { TALENT_REDOUBT, "Redoubt" },
    { TALENT_IMPROVED_DEVOTION_AURA, "Improved Devotion Aura" },
    { TALENT_PALADIN_TOUGHNESS, "Paladin Toughness" },
    { TALENT_PALADIN_SHIELD_SPECIALIZATION, "Paladin Shield Specialization" },
    { TALENT_GUARDIANS_FAVOR, "Guardians Favor" },
    { TALENT_RECKONING, "Reckoning" },
    { TALENT_PALADIN_ONE_HANDED_WEAPON_SPECIALIZATION, "Paladin One Handed Weapon Specialization" },
    { TALENT_HOLY_SHIELD, "Holy Shield" },
    { TALENT_BLESSING_OF_SANCTUARY, "Blessing Of Sanctuary" },
    { TALENT_SPIRITUAL_FOCUS, "Spiritual Focus" },
    { TALENT_DIVINE_FAVOR, "Divine Favor" },
    { TALENT_CONSECRATION, "Consecration" },
    { TALENT_REPENTANCE, "Repentance" },
    { TALENT_BLESSING_OF_KINGS, "Blessing Of Kings" },
    { TALENT_IMPROVED_LAY_ON_HANDS, "Improved Lay On Hands" },
    { TALENT_HEALING_LIGHT, "Healing Light" },
    { TALENT_IMPROVED_BLESSING_OF_WISDOM, "Improved Blessing Of Wisdom" },
    { TALENT_DIVINE_INTELLECT, "Divine Intellect" },
    { TALENT_DIVINE_STRENGTH, "Divine Strength" },
    { TALENT_ILLUMINATION, "Illumination" },
    { TALENT_IMPROVED_SEAL_OF_RIGHTEOUSNESS, "Improved Seal Of Righteousness" },
    { TALENT_IMPROVED_SEAL_OF_THE_CRUSADER, "Improved Seal Of The Crusader" },
    { TALENT_LASTING_JUDGEMENT, "Lasting Judgement" },
    { TALENT_SEAL_OF_COMMAND, "Seal Of Command" },
    { TALENT_IMPROVED_RIGHTEOUS_FURY, "Improved Righteous Fury" },
    { TALENT_HOLY_SHOCK, "Holy Shock" },
    { TALENT_IMPROVED_HAMMER_OF_JUSTICE, "Improved Hammer Of Justice" },
    { TALENT_IMPROVED_BERSERKER_RAGE, "Improved Berserker Rage" },
    { TALENT_IMPROVED_EXECUTE, "Improved Execute" },
    { TALENT_IMPROVED_INTERCEPT, "Improved Intercept" },
    { TALENT_SPIRIT_OF_REDEMPTION, "Spirit Of Redemption" },
    { TALENT_WARRIOR_DUAL_WIELD_SPECIALIZATION, "Warrior Dual Wield Specialization" },
    { TALENT_WARRIOR_SHIELD_SPECIALIZATION, "Warrior Shield Specialization" },
    { TALENT_SAVAGE_STRIKES, "Savage Strikes" },
    { TALENT_SURVIVALIST, "Survivalist" },
    { TALENT_MONSTER_SLAYING, "Monster Slaying" },
    { TALENT_IMPROVED_EYES_OF_THE_BEAST, "Improved Eyes Of The Beast" },
    { TALENT_IMPROVED_REVIVE_PET, "Improved Revive Pet" },
    { TALENT_IMPROVED_CONCENTRATION_AURA, "Improved Concentration Aura" },
    { TALENT_HOLY_POWER, "Holy Power" },
    { TALENT_UNYIELDING_FAITH, "Unyielding Faith" },
    { TALENT_PALADIN_ANTICIPATION, "Paladin Anticipation" },
    { TALENT_PALADIN_PRECISION, "Paladin Precision" },
    { TALENT_IMPROVED_JUDGEMENT, "Improved Judgement" },
    { TALENT_EYE_FOR_AN_EYE, "Eye For An Eye" },
    { TALENT_VINDICATION, "Vindication" },
    { TALENT_PURSUIT_OF_JUSTICE, "Pursuit Of Justice" },
    { TALENT_HOLY_REACH, "Holy Reach" },
    { TALENT_BLESSED_RECOVERY, "Blessed Recovery" },
    { TALENT_LIGHTWELL, "Lightwell" },
    { TALENT_IMPROVED_VAMPIRIC_EMBRACE, "Improved Vampiric Embrace" },
    { TALENT_MASTER_OF_ELEMENTS, "Master Of Elements" },
    { TALENT_ELEMENTAL_WARDING, "Elemental Warding" },
    { TALENT_STORM_REACH, "Storm Reach" },
    { TALENT_EYE_OF_THE_STORM, "Eye Of The Storm" },
    { TALENT_WEAPON_MASTERY, "Weapon Mastery" },
    { TALENT_ELEMENTAL_DEVASTATION, "Elemental Devastation" },
    { TALENT_HEALING_GRACE, "Healing Grace" },
    { TALENT_IMPROVED_WEAPON_TOTEMS, "Improved Weapon Totems" },
    { TALENT_HEALING_WAY, "Healing Way" },
    { TALENT_ELEMENTAL_PRECISION, "Elemental Precision" },
    { TALENT_MAGIC_ABSORPTION, "Magic Absorption" },
    { TALENT_SLEIGHT_OF_HAND, "Sleight Of Hand" },
    { TALENT_HEIGHTENED_SENSES, "Heightened Senses" },
    { TALENT_DEADLINESS, "Deadliness" },
    { TALENT_WEAPON_EXPERTISE, "Weapon Expertise" },
    { 0, NULL }
};

typedef enum {
    GROUP_LOOT_SETTING_FREE_FOR_ALL = 0x0,
    GROUP_LOOT_SETTING_ROUND_ROBIN = 0x1,
    GROUP_LOOT_SETTING_MASTER_LOOT = 0x2,
    GROUP_LOOT_SETTING_GROUP_LOOT = 0x3,
    GROUP_LOOT_SETTING_NEED_BEFORE_GREED = 0x4,
} e_group_loot_setting;
static const value_string e_group_loot_setting_strings[] =  {
    { GROUP_LOOT_SETTING_FREE_FOR_ALL, "Free For All" },
    { GROUP_LOOT_SETTING_ROUND_ROBIN, "Round Robin" },
    { GROUP_LOOT_SETTING_MASTER_LOOT, "Master Loot" },
    { GROUP_LOOT_SETTING_GROUP_LOOT, "Group Loot" },
    { GROUP_LOOT_SETTING_NEED_BEFORE_GREED, "Need Before Greed" },
    { 0, NULL }
};

typedef enum {
    ROLL_VOTE_PASS = 0x0,
    ROLL_VOTE_NEED = 0x1,
    ROLL_VOTE_GREED = 0x2,
} e_roll_vote;
static const value_string e_roll_vote_strings[] =  {
    { ROLL_VOTE_PASS, "Pass" },
    { ROLL_VOTE_NEED, "Need" },
    { ROLL_VOTE_GREED, "Greed" },
    { 0, NULL }
};

typedef enum {
    CHAT_TYPE_SAY = 0x00,
    CHAT_TYPE_PARTY = 0x01,
    CHAT_TYPE_RAID = 0x02,
    CHAT_TYPE_GUILD = 0x03,
    CHAT_TYPE_OFFICER = 0x04,
    CHAT_TYPE_YELL = 0x05,
    CHAT_TYPE_WHISPER = 0x06,
    CHAT_TYPE_WHISPER_INFORM = 0x07,
    CHAT_TYPE_EMOTE = 0x08,
    CHAT_TYPE_TEXT_EMOTE = 0x09,
    CHAT_TYPE_SYSTEM = 0x0A,
    CHAT_TYPE_MONSTER_SAY = 0x0B,
    CHAT_TYPE_MONSTER_YELL = 0x0C,
    CHAT_TYPE_MONSTER_EMOTE = 0x0D,
    CHAT_TYPE_CHANNEL = 0x0E,
    CHAT_TYPE_CHANNEL_JOIN = 0x0F,
    CHAT_TYPE_CHANNEL_LEAVE = 0x10,
    CHAT_TYPE_CHANNEL_LIST = 0x11,
    CHAT_TYPE_CHANNEL_NOTICE = 0x12,
    CHAT_TYPE_CHANNEL_NOTICE_USER = 0x13,
    CHAT_TYPE_AFK = 0x14,
    CHAT_TYPE_DND = 0x15,
    CHAT_TYPE_IGNORED = 0x16,
    CHAT_TYPE_SKILL = 0x17,
    CHAT_TYPE_LOOT = 0x18,
    CHAT_TYPE_MONSTER_WHISPER = 0x1A,
    CHAT_TYPE_BG_SYSTEM_NEUTRAL = 0x52,
    CHAT_TYPE_BG_SYSTEM_ALLIANCE = 0x53,
    CHAT_TYPE_BG_SYSTEM_HORDE = 0x54,
    CHAT_TYPE_RAID_LEADER = 0x57,
    CHAT_TYPE_RAID_WARNING = 0x58,
    CHAT_TYPE_RAID_BOSS_WHISPER = 0x59,
    CHAT_TYPE_RAID_BOSS_EMOTE = 0x5A,
    CHAT_TYPE_BATTLEGROUND = 0x5C,
    CHAT_TYPE_BATTLEGROUND_LEADER = 0x5D,
} e_chat_type;
static const value_string e_chat_type_strings[] =  {
    { CHAT_TYPE_SAY, "Say" },
    { CHAT_TYPE_PARTY, "Party" },
    { CHAT_TYPE_RAID, "Raid" },
    { CHAT_TYPE_GUILD, "Guild" },
    { CHAT_TYPE_OFFICER, "Officer" },
    { CHAT_TYPE_YELL, "Yell" },
    { CHAT_TYPE_WHISPER, "Whisper" },
    { CHAT_TYPE_WHISPER_INFORM, "Whisper Inform" },
    { CHAT_TYPE_EMOTE, "Emote" },
    { CHAT_TYPE_TEXT_EMOTE, "Text Emote" },
    { CHAT_TYPE_SYSTEM, "System" },
    { CHAT_TYPE_MONSTER_SAY, "Monster Say" },
    { CHAT_TYPE_MONSTER_YELL, "Monster Yell" },
    { CHAT_TYPE_MONSTER_EMOTE, "Monster Emote" },
    { CHAT_TYPE_CHANNEL, "Channel" },
    { CHAT_TYPE_CHANNEL_JOIN, "Channel Join" },
    { CHAT_TYPE_CHANNEL_LEAVE, "Channel Leave" },
    { CHAT_TYPE_CHANNEL_LIST, "Channel List" },
    { CHAT_TYPE_CHANNEL_NOTICE, "Channel Notice" },
    { CHAT_TYPE_CHANNEL_NOTICE_USER, "Channel Notice User" },
    { CHAT_TYPE_AFK, "Afk" },
    { CHAT_TYPE_DND, "Dnd" },
    { CHAT_TYPE_IGNORED, "Ignored" },
    { CHAT_TYPE_SKILL, "Skill" },
    { CHAT_TYPE_LOOT, "Loot" },
    { CHAT_TYPE_MONSTER_WHISPER, "Monster Whisper" },
    { CHAT_TYPE_BG_SYSTEM_NEUTRAL, "Bg System Neutral" },
    { CHAT_TYPE_BG_SYSTEM_ALLIANCE, "Bg System Alliance" },
    { CHAT_TYPE_BG_SYSTEM_HORDE, "Bg System Horde" },
    { CHAT_TYPE_RAID_LEADER, "Raid Leader" },
    { CHAT_TYPE_RAID_WARNING, "Raid Warning" },
    { CHAT_TYPE_RAID_BOSS_WHISPER, "Raid Boss Whisper" },
    { CHAT_TYPE_RAID_BOSS_EMOTE, "Raid Boss Emote" },
    { CHAT_TYPE_BATTLEGROUND, "Battleground" },
    { CHAT_TYPE_BATTLEGROUND_LEADER, "Battleground Leader" },
    { 0, NULL }
};

typedef enum {
    SHEATH_STATE_UNARMED = 0x0,
    SHEATH_STATE_MELEE = 0x1,
    SHEATH_STATE_RANGED = 0x2,
} e_sheath_state;
static const value_string e_sheath_state_strings[] =  {
    { SHEATH_STATE_UNARMED, "Unarmed" },
    { SHEATH_STATE_MELEE, "Melee" },
    { SHEATH_STATE_RANGED, "Ranged" },
    { 0, NULL }
};

typedef enum {
    UNIT_STAND_STATE_STAND = 0x0,
    UNIT_STAND_STATE_SIT = 0x1,
    UNIT_STAND_STATE_SIT_CHAIR = 0x2,
    UNIT_STAND_STATE_SLEEP = 0x3,
    UNIT_STAND_STATE_SIT_LOW_CHAIR = 0x4,
    UNIT_STAND_STATE_SIT_MEDIUM_CHAIR = 0x5,
    UNIT_STAND_STATE_SIT_HIGH_CHAIR = 0x6,
    UNIT_STAND_STATE_DEAD = 0x7,
    UNIT_STAND_STATE_KNEEL = 0x8,
    UNIT_STAND_STATE_CUSTOM = 0x9,
} e_unit_stand_state;
static const value_string e_unit_stand_state_strings[] =  {
    { UNIT_STAND_STATE_STAND, "Stand" },
    { UNIT_STAND_STATE_SIT, "Sit" },
    { UNIT_STAND_STATE_SIT_CHAIR, "Sit Chair" },
    { UNIT_STAND_STATE_SLEEP, "Sleep" },
    { UNIT_STAND_STATE_SIT_LOW_CHAIR, "Sit Low Chair" },
    { UNIT_STAND_STATE_SIT_MEDIUM_CHAIR, "Sit Medium Chair" },
    { UNIT_STAND_STATE_SIT_HIGH_CHAIR, "Sit High Chair" },
    { UNIT_STAND_STATE_DEAD, "Dead" },
    { UNIT_STAND_STATE_KNEEL, "Kneel" },
    { UNIT_STAND_STATE_CUSTOM, "Custom" },
    { 0, NULL }
};

typedef enum {
    ITEM_SLOT_HEAD = 0x00,
    ITEM_SLOT_NECK = 0x01,
    ITEM_SLOT_SHOULDERS = 0x02,
    ITEM_SLOT_SHIRT = 0x03,
    ITEM_SLOT_CHEST = 0x04,
    ITEM_SLOT_WAIST = 0x05,
    ITEM_SLOT_LEGS = 0x06,
    ITEM_SLOT_BOOTS = 0x07,
    ITEM_SLOT_WRIST = 0x08,
    ITEM_SLOT_HANDS = 0x09,
    ITEM_SLOT_RING1 = 0x0A,
    ITEM_SLOT_RING2 = 0x0B,
    ITEM_SLOT_TRINKET1 = 0x0C,
    ITEM_SLOT_TRINKET2 = 0x0D,
    ITEM_SLOT_BACK = 0x0E,
    ITEM_SLOT_MAIN_HAND = 0x0F,
    ITEM_SLOT_OFF_HAND = 0x10,
    ITEM_SLOT_RANGED_OR_RELIC = 0x11,
    ITEM_SLOT_TABARD = 0x12,
    ITEM_SLOT_BAG1 = 0x13,
    ITEM_SLOT_BAG2 = 0x14,
    ITEM_SLOT_BAG3 = 0x15,
    ITEM_SLOT_BAG4 = 0x16,
    ITEM_SLOT_INVENTORY_0 = 0x17,
    ITEM_SLOT_INVENTORY_1 = 0x18,
    ITEM_SLOT_INVENTORY_2 = 0x19,
    ITEM_SLOT_INVENTORY_3 = 0x1A,
    ITEM_SLOT_INVENTORY_4 = 0x1B,
    ITEM_SLOT_INVENTORY_5 = 0x1C,
    ITEM_SLOT_INVENTORY_6 = 0x1D,
    ITEM_SLOT_INVENTORY_7 = 0x1E,
    ITEM_SLOT_INVENTORY_8 = 0x1F,
    ITEM_SLOT_INVENTORY_9 = 0x20,
    ITEM_SLOT_INVENTORY_10 = 0x21,
    ITEM_SLOT_INVENTORY_11 = 0x22,
    ITEM_SLOT_INVENTORY_12 = 0x23,
    ITEM_SLOT_INVENTORY_13 = 0x24,
    ITEM_SLOT_INVENTORY_14 = 0x25,
    ITEM_SLOT_INVENTORY_15 = 0x26,
    ITEM_SLOT_BANK_1 = 0x27,
    ITEM_SLOT_BANK_2 = 0x28,
    ITEM_SLOT_BANK_3 = 0x29,
    ITEM_SLOT_BANK_4 = 0x2A,
    ITEM_SLOT_BANK_5 = 0x2B,
    ITEM_SLOT_BANK_6 = 0x2C,
    ITEM_SLOT_BANK_7 = 0x2D,
    ITEM_SLOT_BANK_8 = 0x2E,
    ITEM_SLOT_BANK_9 = 0x2F,
    ITEM_SLOT_BANK_10 = 0x30,
    ITEM_SLOT_BANK_11 = 0x31,
    ITEM_SLOT_BANK_12 = 0x32,
    ITEM_SLOT_BANK_13 = 0x33,
    ITEM_SLOT_BANK_14 = 0x34,
    ITEM_SLOT_BANK_15 = 0x35,
    ITEM_SLOT_BANK_16 = 0x36,
    ITEM_SLOT_BANK_17 = 0x37,
    ITEM_SLOT_BANK_18 = 0x38,
    ITEM_SLOT_BANK_19 = 0x39,
    ITEM_SLOT_BANK_20 = 0x3A,
    ITEM_SLOT_BANK_21 = 0x3B,
    ITEM_SLOT_BANK_22 = 0x3C,
    ITEM_SLOT_BANK_23 = 0x3D,
    ITEM_SLOT_BANK_24 = 0x3E,
    ITEM_SLOT_BANK_BAG_SLOT_1 = 0x3F,
    ITEM_SLOT_BANK_BAG_SLOT_2 = 0x40,
    ITEM_SLOT_BANK_BAG_SLOT_3 = 0x41,
    ITEM_SLOT_BANK_BAG_SLOT_4 = 0x42,
    ITEM_SLOT_BANK_BAG_SLOT_5 = 0x43,
    ITEM_SLOT_BANK_BAG_SLOT_6 = 0x44,
    ITEM_SLOT_VENDOR_BUYBACK_1 = 0x45,
    ITEM_SLOT_VENDOR_BUYBACK_2 = 0x46,
    ITEM_SLOT_VENDOR_BUYBACK_3 = 0x47,
    ITEM_SLOT_VENDOR_BUYBACK_4 = 0x48,
    ITEM_SLOT_VENDOR_BUYBACK_5 = 0x49,
    ITEM_SLOT_VENDOR_BUYBACK_6 = 0x4A,
    ITEM_SLOT_VENDOR_BUYBACK_7 = 0x4B,
    ITEM_SLOT_VENDOR_BUYBACK_8 = 0x4C,
    ITEM_SLOT_VENDOR_BUYBACK_9 = 0x4D,
    ITEM_SLOT_VENDOR_BUYBACK_10 = 0x4E,
    ITEM_SLOT_VENDOR_BUYBACK_11 = 0x4F,
    ITEM_SLOT_VENDOR_BUYBACK_12 = 0x50,
    ITEM_SLOT_KEYRING_1 = 0x51,
    ITEM_SLOT_KEYRING_2 = 0x52,
    ITEM_SLOT_KEYRING_3 = 0x53,
    ITEM_SLOT_KEYRING_4 = 0x54,
    ITEM_SLOT_KEYRING_5 = 0x55,
    ITEM_SLOT_KEYRING_6 = 0x56,
    ITEM_SLOT_KEYRING_7 = 0x57,
    ITEM_SLOT_KEYRING_8 = 0x58,
    ITEM_SLOT_KEYRING_9 = 0x59,
    ITEM_SLOT_KEYRING_10 = 0x5A,
    ITEM_SLOT_KEYRING_11 = 0x5B,
    ITEM_SLOT_KEYRING_12 = 0x5C,
    ITEM_SLOT_KEYRING_13 = 0x5D,
    ITEM_SLOT_KEYRING_14 = 0x5E,
    ITEM_SLOT_KEYRING_15 = 0x5F,
    ITEM_SLOT_KEYRING_16 = 0x60,
    ITEM_SLOT_KEYRING_17 = 0x61,
    ITEM_SLOT_KEYRING_18 = 0x62,
    ITEM_SLOT_KEYRING_19 = 0x63,
    ITEM_SLOT_KEYRING_20 = 0x64,
    ITEM_SLOT_KEYRING_21 = 0x65,
    ITEM_SLOT_KEYRING_22 = 0x66,
    ITEM_SLOT_KEYRING_23 = 0x67,
    ITEM_SLOT_KEYRING_24 = 0x68,
    ITEM_SLOT_KEYRING_25 = 0x69,
    ITEM_SLOT_KEYRING_26 = 0x6A,
    ITEM_SLOT_KEYRING_27 = 0x6B,
    ITEM_SLOT_KEYRING_28 = 0x6C,
    ITEM_SLOT_KEYRING_29 = 0x6D,
    ITEM_SLOT_KEYRING_30 = 0x6E,
    ITEM_SLOT_KEYRING_31 = 0x6F,
    ITEM_SLOT_KEYRING_32 = 0x70,
} e_item_slot;
static const value_string e_item_slot_strings[] =  {
    { ITEM_SLOT_HEAD, "Head" },
    { ITEM_SLOT_NECK, "Neck" },
    { ITEM_SLOT_SHOULDERS, "Shoulders" },
    { ITEM_SLOT_SHIRT, "Shirt" },
    { ITEM_SLOT_CHEST, "Chest" },
    { ITEM_SLOT_WAIST, "Waist" },
    { ITEM_SLOT_LEGS, "Legs" },
    { ITEM_SLOT_BOOTS, "Boots" },
    { ITEM_SLOT_WRIST, "Wrist" },
    { ITEM_SLOT_HANDS, "Hands" },
    { ITEM_SLOT_RING1, "Ring1" },
    { ITEM_SLOT_RING2, "Ring2" },
    { ITEM_SLOT_TRINKET1, "Trinket1" },
    { ITEM_SLOT_TRINKET2, "Trinket2" },
    { ITEM_SLOT_BACK, "Back" },
    { ITEM_SLOT_MAIN_HAND, "Main Hand" },
    { ITEM_SLOT_OFF_HAND, "Off Hand" },
    { ITEM_SLOT_RANGED_OR_RELIC, "Ranged Or Relic" },
    { ITEM_SLOT_TABARD, "Tabard" },
    { ITEM_SLOT_BAG1, "Bag1" },
    { ITEM_SLOT_BAG2, "Bag2" },
    { ITEM_SLOT_BAG3, "Bag3" },
    { ITEM_SLOT_BAG4, "Bag4" },
    { ITEM_SLOT_INVENTORY_0, "Inventory 0" },
    { ITEM_SLOT_INVENTORY_1, "Inventory 1" },
    { ITEM_SLOT_INVENTORY_2, "Inventory 2" },
    { ITEM_SLOT_INVENTORY_3, "Inventory 3" },
    { ITEM_SLOT_INVENTORY_4, "Inventory 4" },
    { ITEM_SLOT_INVENTORY_5, "Inventory 5" },
    { ITEM_SLOT_INVENTORY_6, "Inventory 6" },
    { ITEM_SLOT_INVENTORY_7, "Inventory 7" },
    { ITEM_SLOT_INVENTORY_8, "Inventory 8" },
    { ITEM_SLOT_INVENTORY_9, "Inventory 9" },
    { ITEM_SLOT_INVENTORY_10, "Inventory 10" },
    { ITEM_SLOT_INVENTORY_11, "Inventory 11" },
    { ITEM_SLOT_INVENTORY_12, "Inventory 12" },
    { ITEM_SLOT_INVENTORY_13, "Inventory 13" },
    { ITEM_SLOT_INVENTORY_14, "Inventory 14" },
    { ITEM_SLOT_INVENTORY_15, "Inventory 15" },
    { ITEM_SLOT_BANK_1, "Bank 1" },
    { ITEM_SLOT_BANK_2, "Bank 2" },
    { ITEM_SLOT_BANK_3, "Bank 3" },
    { ITEM_SLOT_BANK_4, "Bank 4" },
    { ITEM_SLOT_BANK_5, "Bank 5" },
    { ITEM_SLOT_BANK_6, "Bank 6" },
    { ITEM_SLOT_BANK_7, "Bank 7" },
    { ITEM_SLOT_BANK_8, "Bank 8" },
    { ITEM_SLOT_BANK_9, "Bank 9" },
    { ITEM_SLOT_BANK_10, "Bank 10" },
    { ITEM_SLOT_BANK_11, "Bank 11" },
    { ITEM_SLOT_BANK_12, "Bank 12" },
    { ITEM_SLOT_BANK_13, "Bank 13" },
    { ITEM_SLOT_BANK_14, "Bank 14" },
    { ITEM_SLOT_BANK_15, "Bank 15" },
    { ITEM_SLOT_BANK_16, "Bank 16" },
    { ITEM_SLOT_BANK_17, "Bank 17" },
    { ITEM_SLOT_BANK_18, "Bank 18" },
    { ITEM_SLOT_BANK_19, "Bank 19" },
    { ITEM_SLOT_BANK_20, "Bank 20" },
    { ITEM_SLOT_BANK_21, "Bank 21" },
    { ITEM_SLOT_BANK_22, "Bank 22" },
    { ITEM_SLOT_BANK_23, "Bank 23" },
    { ITEM_SLOT_BANK_24, "Bank 24" },
    { ITEM_SLOT_BANK_BAG_SLOT_1, "Bank Bag Slot 1" },
    { ITEM_SLOT_BANK_BAG_SLOT_2, "Bank Bag Slot 2" },
    { ITEM_SLOT_BANK_BAG_SLOT_3, "Bank Bag Slot 3" },
    { ITEM_SLOT_BANK_BAG_SLOT_4, "Bank Bag Slot 4" },
    { ITEM_SLOT_BANK_BAG_SLOT_5, "Bank Bag Slot 5" },
    { ITEM_SLOT_BANK_BAG_SLOT_6, "Bank Bag Slot 6" },
    { ITEM_SLOT_VENDOR_BUYBACK_1, "Vendor Buyback 1" },
    { ITEM_SLOT_VENDOR_BUYBACK_2, "Vendor Buyback 2" },
    { ITEM_SLOT_VENDOR_BUYBACK_3, "Vendor Buyback 3" },
    { ITEM_SLOT_VENDOR_BUYBACK_4, "Vendor Buyback 4" },
    { ITEM_SLOT_VENDOR_BUYBACK_5, "Vendor Buyback 5" },
    { ITEM_SLOT_VENDOR_BUYBACK_6, "Vendor Buyback 6" },
    { ITEM_SLOT_VENDOR_BUYBACK_7, "Vendor Buyback 7" },
    { ITEM_SLOT_VENDOR_BUYBACK_8, "Vendor Buyback 8" },
    { ITEM_SLOT_VENDOR_BUYBACK_9, "Vendor Buyback 9" },
    { ITEM_SLOT_VENDOR_BUYBACK_10, "Vendor Buyback 10" },
    { ITEM_SLOT_VENDOR_BUYBACK_11, "Vendor Buyback 11" },
    { ITEM_SLOT_VENDOR_BUYBACK_12, "Vendor Buyback 12" },
    { ITEM_SLOT_KEYRING_1, "Keyring 1" },
    { ITEM_SLOT_KEYRING_2, "Keyring 2" },
    { ITEM_SLOT_KEYRING_3, "Keyring 3" },
    { ITEM_SLOT_KEYRING_4, "Keyring 4" },
    { ITEM_SLOT_KEYRING_5, "Keyring 5" },
    { ITEM_SLOT_KEYRING_6, "Keyring 6" },
    { ITEM_SLOT_KEYRING_7, "Keyring 7" },
    { ITEM_SLOT_KEYRING_8, "Keyring 8" },
    { ITEM_SLOT_KEYRING_9, "Keyring 9" },
    { ITEM_SLOT_KEYRING_10, "Keyring 10" },
    { ITEM_SLOT_KEYRING_11, "Keyring 11" },
    { ITEM_SLOT_KEYRING_12, "Keyring 12" },
    { ITEM_SLOT_KEYRING_13, "Keyring 13" },
    { ITEM_SLOT_KEYRING_14, "Keyring 14" },
    { ITEM_SLOT_KEYRING_15, "Keyring 15" },
    { ITEM_SLOT_KEYRING_16, "Keyring 16" },
    { ITEM_SLOT_KEYRING_17, "Keyring 17" },
    { ITEM_SLOT_KEYRING_18, "Keyring 18" },
    { ITEM_SLOT_KEYRING_19, "Keyring 19" },
    { ITEM_SLOT_KEYRING_20, "Keyring 20" },
    { ITEM_SLOT_KEYRING_21, "Keyring 21" },
    { ITEM_SLOT_KEYRING_22, "Keyring 22" },
    { ITEM_SLOT_KEYRING_23, "Keyring 23" },
    { ITEM_SLOT_KEYRING_24, "Keyring 24" },
    { ITEM_SLOT_KEYRING_25, "Keyring 25" },
    { ITEM_SLOT_KEYRING_26, "Keyring 26" },
    { ITEM_SLOT_KEYRING_27, "Keyring 27" },
    { ITEM_SLOT_KEYRING_28, "Keyring 28" },
    { ITEM_SLOT_KEYRING_29, "Keyring 29" },
    { ITEM_SLOT_KEYRING_30, "Keyring 30" },
    { ITEM_SLOT_KEYRING_31, "Keyring 31" },
    { ITEM_SLOT_KEYRING_32, "Keyring 32" },
    { 0, NULL }
};

typedef enum {
    TEXT_EMOTE_AGREE = 0x001,
    TEXT_EMOTE_AMAZE = 0x002,
    TEXT_EMOTE_ANGRY = 0x003,
    TEXT_EMOTE_APOLOGIZE = 0x004,
    TEXT_EMOTE_APPLAUD = 0x005,
    TEXT_EMOTE_BASHFUL = 0x006,
    TEXT_EMOTE_BECKON = 0x007,
    TEXT_EMOTE_BEG = 0x008,
    TEXT_EMOTE_BITE = 0x009,
    TEXT_EMOTE_BLEED = 0x00A,
    TEXT_EMOTE_BLINK = 0x00B,
    TEXT_EMOTE_BLUSH = 0x00C,
    TEXT_EMOTE_BONK = 0x00D,
    TEXT_EMOTE_BORED = 0x00E,
    TEXT_EMOTE_BOUNCE = 0x00F,
    TEXT_EMOTE_BRB = 0x010,
    TEXT_EMOTE_BOW = 0x011,
    TEXT_EMOTE_BURP = 0x012,
    TEXT_EMOTE_BYE = 0x013,
    TEXT_EMOTE_CACKLE = 0x014,
    TEXT_EMOTE_CHEER = 0x015,
    TEXT_EMOTE_CHICKEN = 0x016,
    TEXT_EMOTE_CHUCKLE = 0x017,
    TEXT_EMOTE_CLAP = 0x018,
    TEXT_EMOTE_CONFUSED = 0x019,
    TEXT_EMOTE_CONGRATULATE = 0x01A,
    TEXT_EMOTE_COUGH = 0x01B,
    TEXT_EMOTE_COWER = 0x01C,
    TEXT_EMOTE_CRACK = 0x01D,
    TEXT_EMOTE_CRINGE = 0x01E,
    TEXT_EMOTE_CRY = 0x01F,
    TEXT_EMOTE_CURIOUS = 0x020,
    TEXT_EMOTE_CURTSEY = 0x021,
    TEXT_EMOTE_DANCE = 0x022,
    TEXT_EMOTE_DRINK = 0x023,
    TEXT_EMOTE_DROOL = 0x024,
    TEXT_EMOTE_EAT = 0x025,
    TEXT_EMOTE_EYE = 0x026,
    TEXT_EMOTE_FART = 0x027,
    TEXT_EMOTE_FIDGET = 0x028,
    TEXT_EMOTE_FLEX = 0x029,
    TEXT_EMOTE_FROWN = 0x02A,
    TEXT_EMOTE_GASP = 0x02B,
    TEXT_EMOTE_GAZE = 0x02C,
    TEXT_EMOTE_GIGGLE = 0x02D,
    TEXT_EMOTE_GLARE = 0x02E,
    TEXT_EMOTE_GLOAT = 0x02F,
    TEXT_EMOTE_GREET = 0x030,
    TEXT_EMOTE_GRIN = 0x031,
    TEXT_EMOTE_GROAN = 0x032,
    TEXT_EMOTE_GROVEL = 0x033,
    TEXT_EMOTE_GUFFAW = 0x034,
    TEXT_EMOTE_HAIL = 0x035,
    TEXT_EMOTE_HAPPY = 0x036,
    TEXT_EMOTE_HELLO = 0x037,
    TEXT_EMOTE_HUG = 0x038,
    TEXT_EMOTE_HUNGRY = 0x039,
    TEXT_EMOTE_KISS = 0x03A,
    TEXT_EMOTE_KNEEL = 0x03B,
    TEXT_EMOTE_LAUGH = 0x03C,
    TEXT_EMOTE_LAYDOWN = 0x03D,
    TEXT_EMOTE_MASSAGE = 0x03E,
    TEXT_EMOTE_MOAN = 0x03F,
    TEXT_EMOTE_MOON = 0x040,
    TEXT_EMOTE_MOURN = 0x041,
    TEXT_EMOTE_NO = 0x042,
    TEXT_EMOTE_NOD = 0x043,
    TEXT_EMOTE_NOSE_PICK = 0x044,
    TEXT_EMOTE_PANIC = 0x045,
    TEXT_EMOTE_PEER = 0x046,
    TEXT_EMOTE_PLEAD = 0x047,
    TEXT_EMOTE_POINT = 0x048,
    TEXT_EMOTE_POKE = 0x049,
    TEXT_EMOTE_PRAY = 0x04A,
    TEXT_EMOTE_ROAR = 0x04B,
    TEXT_EMOTE_ROFL = 0x04C,
    TEXT_EMOTE_RUDE = 0x04D,
    TEXT_EMOTE_SALUTE = 0x04E,
    TEXT_EMOTE_SCRATCH = 0x04F,
    TEXT_EMOTE_SEXY = 0x050,
    TEXT_EMOTE_SHAKE = 0x051,
    TEXT_EMOTE_SHOUT = 0x052,
    TEXT_EMOTE_SHRUG = 0x053,
    TEXT_EMOTE_SHY = 0x054,
    TEXT_EMOTE_SIGH = 0x055,
    TEXT_EMOTE_SIT = 0x056,
    TEXT_EMOTE_SLEEP = 0x057,
    TEXT_EMOTE_SNARL = 0x058,
    TEXT_EMOTE_SPIT = 0x059,
    TEXT_EMOTE_STARE = 0x05A,
    TEXT_EMOTE_SURPRISED = 0x05B,
    TEXT_EMOTE_SURRENDER = 0x05C,
    TEXT_EMOTE_TALK = 0x05D,
    TEXT_EMOTE_TALK_EX = 0x05E,
    TEXT_EMOTE_TALK_Q = 0x05F,
    TEXT_EMOTE_TAP = 0x060,
    TEXT_EMOTE_THANK = 0x061,
    TEXT_EMOTE_THREATEN = 0x062,
    TEXT_EMOTE_TIRED = 0x063,
    TEXT_EMOTE_VICTORY = 0x064,
    TEXT_EMOTE_WAVE = 0x065,
    TEXT_EMOTE_WELCOME = 0x066,
    TEXT_EMOTE_WHINE = 0x067,
    TEXT_EMOTE_WHISTLE = 0x068,
    TEXT_EMOTE_WORK = 0x069,
    TEXT_EMOTE_YAWN = 0x06A,
    TEXT_EMOTE_BOGGLE = 0x06B,
    TEXT_EMOTE_CALM = 0x06C,
    TEXT_EMOTE_COLD = 0x06D,
    TEXT_EMOTE_COMFORT = 0x06E,
    TEXT_EMOTE_CUDDLE = 0x06F,
    TEXT_EMOTE_DUCK = 0x070,
    TEXT_EMOTE_INSULT = 0x071,
    TEXT_EMOTE_INTRODUCE = 0x072,
    TEXT_EMOTE_JK = 0x073,
    TEXT_EMOTE_LICK = 0x074,
    TEXT_EMOTE_LISTEN = 0x075,
    TEXT_EMOTE_LOST = 0x076,
    TEXT_EMOTE_MOCK = 0x077,
    TEXT_EMOTE_PONDER = 0x078,
    TEXT_EMOTE_POUNCE = 0x079,
    TEXT_EMOTE_PRAISE = 0x07A,
    TEXT_EMOTE_PURR = 0x07B,
    TEXT_EMOTE_PUZZLE = 0x07C,
    TEXT_EMOTE_RAISE = 0x07D,
    TEXT_EMOTE_READY = 0x07E,
    TEXT_EMOTE_SHIMMY = 0x07F,
    TEXT_EMOTE_SHIVER = 0x080,
    TEXT_EMOTE_SHOO = 0x081,
    TEXT_EMOTE_SLAP = 0x082,
    TEXT_EMOTE_SMIRK = 0x083,
    TEXT_EMOTE_SNIFF = 0x084,
    TEXT_EMOTE_SNUB = 0x085,
    TEXT_EMOTE_SOOTHE = 0x086,
    TEXT_EMOTE_STINK = 0x087,
    TEXT_EMOTE_TAUNT = 0x088,
    TEXT_EMOTE_TEASE = 0x089,
    TEXT_EMOTE_THIRSTY = 0x08A,
    TEXT_EMOTE_VETO = 0x08B,
    TEXT_EMOTE_SNICKER = 0x08C,
    TEXT_EMOTE_STAND = 0x08D,
    TEXT_EMOTE_TICKLE = 0x08E,
    TEXT_EMOTE_VIOLIN = 0x08F,
    TEXT_EMOTE_SMILE = 0x0A3,
    TEXT_EMOTE_RASP = 0x0B7,
    TEXT_EMOTE_PITY = 0x0CB,
    TEXT_EMOTE_GROWL = 0x0CC,
    TEXT_EMOTE_BARK = 0x0CD,
    TEXT_EMOTE_SCARED = 0x0DF,
    TEXT_EMOTE_FLOP = 0x0E0,
    TEXT_EMOTE_LOVE = 0x0E1,
    TEXT_EMOTE_MOO = 0x0E2,
    TEXT_EMOTE_COMMEND = 0x0F3,
    TEXT_EMOTE_TRAIN = 0x108,
    TEXT_EMOTE_HELPME = 0x12F,
    TEXT_EMOTE_INCOMING = 0x130,
    TEXT_EMOTE_CHARGE = 0x131,
    TEXT_EMOTE_FLEE = 0x132,
    TEXT_EMOTE_ATTACK_MY_TARGET = 0x133,
    TEXT_EMOTE_OOM = 0x143,
    TEXT_EMOTE_FOLLOW = 0x144,
    TEXT_EMOTE_WAIT = 0x145,
    TEXT_EMOTE_HEAL_ME = 0x146,
    TEXT_EMOTE_OPEN_FIRE = 0x147,
    TEXT_EMOTE_FLIRT = 0x148,
    TEXT_EMOTE_JOKE = 0x149,
    TEXT_EMOTE_GOLF_CLAP = 0x157,
    TEXT_EMOTE_WINK = 0x16B,
    TEXT_EMOTE_PAT = 0x16C,
} e_text_emote;
static const value_string e_text_emote_strings[] =  {
    { TEXT_EMOTE_AGREE, "Agree" },
    { TEXT_EMOTE_AMAZE, "Amaze" },
    { TEXT_EMOTE_ANGRY, "Angry" },
    { TEXT_EMOTE_APOLOGIZE, "Apologize" },
    { TEXT_EMOTE_APPLAUD, "Applaud" },
    { TEXT_EMOTE_BASHFUL, "Bashful" },
    { TEXT_EMOTE_BECKON, "Beckon" },
    { TEXT_EMOTE_BEG, "Beg" },
    { TEXT_EMOTE_BITE, "Bite" },
    { TEXT_EMOTE_BLEED, "Bleed" },
    { TEXT_EMOTE_BLINK, "Blink" },
    { TEXT_EMOTE_BLUSH, "Blush" },
    { TEXT_EMOTE_BONK, "Bonk" },
    { TEXT_EMOTE_BORED, "Bored" },
    { TEXT_EMOTE_BOUNCE, "Bounce" },
    { TEXT_EMOTE_BRB, "Brb" },
    { TEXT_EMOTE_BOW, "Bow" },
    { TEXT_EMOTE_BURP, "Burp" },
    { TEXT_EMOTE_BYE, "Bye" },
    { TEXT_EMOTE_CACKLE, "Cackle" },
    { TEXT_EMOTE_CHEER, "Cheer" },
    { TEXT_EMOTE_CHICKEN, "Chicken" },
    { TEXT_EMOTE_CHUCKLE, "Chuckle" },
    { TEXT_EMOTE_CLAP, "Clap" },
    { TEXT_EMOTE_CONFUSED, "Confused" },
    { TEXT_EMOTE_CONGRATULATE, "Congratulate" },
    { TEXT_EMOTE_COUGH, "Cough" },
    { TEXT_EMOTE_COWER, "Cower" },
    { TEXT_EMOTE_CRACK, "Crack" },
    { TEXT_EMOTE_CRINGE, "Cringe" },
    { TEXT_EMOTE_CRY, "Cry" },
    { TEXT_EMOTE_CURIOUS, "Curious" },
    { TEXT_EMOTE_CURTSEY, "Curtsey" },
    { TEXT_EMOTE_DANCE, "Dance" },
    { TEXT_EMOTE_DRINK, "Drink" },
    { TEXT_EMOTE_DROOL, "Drool" },
    { TEXT_EMOTE_EAT, "Eat" },
    { TEXT_EMOTE_EYE, "Eye" },
    { TEXT_EMOTE_FART, "Fart" },
    { TEXT_EMOTE_FIDGET, "Fidget" },
    { TEXT_EMOTE_FLEX, "Flex" },
    { TEXT_EMOTE_FROWN, "Frown" },
    { TEXT_EMOTE_GASP, "Gasp" },
    { TEXT_EMOTE_GAZE, "Gaze" },
    { TEXT_EMOTE_GIGGLE, "Giggle" },
    { TEXT_EMOTE_GLARE, "Glare" },
    { TEXT_EMOTE_GLOAT, "Gloat" },
    { TEXT_EMOTE_GREET, "Greet" },
    { TEXT_EMOTE_GRIN, "Grin" },
    { TEXT_EMOTE_GROAN, "Groan" },
    { TEXT_EMOTE_GROVEL, "Grovel" },
    { TEXT_EMOTE_GUFFAW, "Guffaw" },
    { TEXT_EMOTE_HAIL, "Hail" },
    { TEXT_EMOTE_HAPPY, "Happy" },
    { TEXT_EMOTE_HELLO, "Hello" },
    { TEXT_EMOTE_HUG, "Hug" },
    { TEXT_EMOTE_HUNGRY, "Hungry" },
    { TEXT_EMOTE_KISS, "Kiss" },
    { TEXT_EMOTE_KNEEL, "Kneel" },
    { TEXT_EMOTE_LAUGH, "Laugh" },
    { TEXT_EMOTE_LAYDOWN, "Laydown" },
    { TEXT_EMOTE_MASSAGE, "Massage" },
    { TEXT_EMOTE_MOAN, "Moan" },
    { TEXT_EMOTE_MOON, "Moon" },
    { TEXT_EMOTE_MOURN, "Mourn" },
    { TEXT_EMOTE_NO, "No" },
    { TEXT_EMOTE_NOD, "Nod" },
    { TEXT_EMOTE_NOSE_PICK, "Nose Pick" },
    { TEXT_EMOTE_PANIC, "Panic" },
    { TEXT_EMOTE_PEER, "Peer" },
    { TEXT_EMOTE_PLEAD, "Plead" },
    { TEXT_EMOTE_POINT, "Point" },
    { TEXT_EMOTE_POKE, "Poke" },
    { TEXT_EMOTE_PRAY, "Pray" },
    { TEXT_EMOTE_ROAR, "Roar" },
    { TEXT_EMOTE_ROFL, "Rofl" },
    { TEXT_EMOTE_RUDE, "Rude" },
    { TEXT_EMOTE_SALUTE, "Salute" },
    { TEXT_EMOTE_SCRATCH, "Scratch" },
    { TEXT_EMOTE_SEXY, "Sexy" },
    { TEXT_EMOTE_SHAKE, "Shake" },
    { TEXT_EMOTE_SHOUT, "Shout" },
    { TEXT_EMOTE_SHRUG, "Shrug" },
    { TEXT_EMOTE_SHY, "Shy" },
    { TEXT_EMOTE_SIGH, "Sigh" },
    { TEXT_EMOTE_SIT, "Sit" },
    { TEXT_EMOTE_SLEEP, "Sleep" },
    { TEXT_EMOTE_SNARL, "Snarl" },
    { TEXT_EMOTE_SPIT, "Spit" },
    { TEXT_EMOTE_STARE, "Stare" },
    { TEXT_EMOTE_SURPRISED, "Surprised" },
    { TEXT_EMOTE_SURRENDER, "Surrender" },
    { TEXT_EMOTE_TALK, "Talk" },
    { TEXT_EMOTE_TALK_EX, "Talk Ex" },
    { TEXT_EMOTE_TALK_Q, "Talk Q" },
    { TEXT_EMOTE_TAP, "Tap" },
    { TEXT_EMOTE_THANK, "Thank" },
    { TEXT_EMOTE_THREATEN, "Threaten" },
    { TEXT_EMOTE_TIRED, "Tired" },
    { TEXT_EMOTE_VICTORY, "Victory" },
    { TEXT_EMOTE_WAVE, "Wave" },
    { TEXT_EMOTE_WELCOME, "Welcome" },
    { TEXT_EMOTE_WHINE, "Whine" },
    { TEXT_EMOTE_WHISTLE, "Whistle" },
    { TEXT_EMOTE_WORK, "Work" },
    { TEXT_EMOTE_YAWN, "Yawn" },
    { TEXT_EMOTE_BOGGLE, "Boggle" },
    { TEXT_EMOTE_CALM, "Calm" },
    { TEXT_EMOTE_COLD, "Cold" },
    { TEXT_EMOTE_COMFORT, "Comfort" },
    { TEXT_EMOTE_CUDDLE, "Cuddle" },
    { TEXT_EMOTE_DUCK, "Duck" },
    { TEXT_EMOTE_INSULT, "Insult" },
    { TEXT_EMOTE_INTRODUCE, "Introduce" },
    { TEXT_EMOTE_JK, "Jk" },
    { TEXT_EMOTE_LICK, "Lick" },
    { TEXT_EMOTE_LISTEN, "Listen" },
    { TEXT_EMOTE_LOST, "Lost" },
    { TEXT_EMOTE_MOCK, "Mock" },
    { TEXT_EMOTE_PONDER, "Ponder" },
    { TEXT_EMOTE_POUNCE, "Pounce" },
    { TEXT_EMOTE_PRAISE, "Praise" },
    { TEXT_EMOTE_PURR, "Purr" },
    { TEXT_EMOTE_PUZZLE, "Puzzle" },
    { TEXT_EMOTE_RAISE, "Raise" },
    { TEXT_EMOTE_READY, "Ready" },
    { TEXT_EMOTE_SHIMMY, "Shimmy" },
    { TEXT_EMOTE_SHIVER, "Shiver" },
    { TEXT_EMOTE_SHOO, "Shoo" },
    { TEXT_EMOTE_SLAP, "Slap" },
    { TEXT_EMOTE_SMIRK, "Smirk" },
    { TEXT_EMOTE_SNIFF, "Sniff" },
    { TEXT_EMOTE_SNUB, "Snub" },
    { TEXT_EMOTE_SOOTHE, "Soothe" },
    { TEXT_EMOTE_STINK, "Stink" },
    { TEXT_EMOTE_TAUNT, "Taunt" },
    { TEXT_EMOTE_TEASE, "Tease" },
    { TEXT_EMOTE_THIRSTY, "Thirsty" },
    { TEXT_EMOTE_VETO, "Veto" },
    { TEXT_EMOTE_SNICKER, "Snicker" },
    { TEXT_EMOTE_STAND, "Stand" },
    { TEXT_EMOTE_TICKLE, "Tickle" },
    { TEXT_EMOTE_VIOLIN, "Violin" },
    { TEXT_EMOTE_SMILE, "Smile" },
    { TEXT_EMOTE_RASP, "Rasp" },
    { TEXT_EMOTE_PITY, "Pity" },
    { TEXT_EMOTE_GROWL, "Growl" },
    { TEXT_EMOTE_BARK, "Bark" },
    { TEXT_EMOTE_SCARED, "Scared" },
    { TEXT_EMOTE_FLOP, "Flop" },
    { TEXT_EMOTE_LOVE, "Love" },
    { TEXT_EMOTE_MOO, "Moo" },
    { TEXT_EMOTE_COMMEND, "Commend" },
    { TEXT_EMOTE_TRAIN, "Train" },
    { TEXT_EMOTE_HELPME, "Helpme" },
    { TEXT_EMOTE_INCOMING, "Incoming" },
    { TEXT_EMOTE_CHARGE, "Charge" },
    { TEXT_EMOTE_FLEE, "Flee" },
    { TEXT_EMOTE_ATTACK_MY_TARGET, "Attack My Target" },
    { TEXT_EMOTE_OOM, "Oom" },
    { TEXT_EMOTE_FOLLOW, "Follow" },
    { TEXT_EMOTE_WAIT, "Wait" },
    { TEXT_EMOTE_HEAL_ME, "Heal Me" },
    { TEXT_EMOTE_OPEN_FIRE, "Open Fire" },
    { TEXT_EMOTE_FLIRT, "Flirt" },
    { TEXT_EMOTE_JOKE, "Joke" },
    { TEXT_EMOTE_GOLF_CLAP, "Golf Clap" },
    { TEXT_EMOTE_WINK, "Wink" },
    { TEXT_EMOTE_PAT, "Pat" },
    { 0, NULL }
};

typedef enum {
    ACCOUNT_DATA_TYPE_GLOBAL_CONFIG_CACHE = 0x0,
    ACCOUNT_DATA_TYPE_PER_CHARACTER_CONFIG_CACHE = 0x1,
    ACCOUNT_DATA_TYPE_GLOBAL_BINDINGS_CACHE = 0x2,
    ACCOUNT_DATA_TYPE_PER_CHARACTER_BINDINGS_CACHE = 0x3,
    ACCOUNT_DATA_TYPE_GLOBAL_MACROS_CACHE = 0x4,
    ACCOUNT_DATA_TYPE_PER_CHARACTER_MACROS_CACHE = 0x5,
    ACCOUNT_DATA_TYPE_PER_CHARACTER_LAYOUT_CACHE = 0x6,
    ACCOUNT_DATA_TYPE_PER_CHARACTER_CHAT_CACHE = 0x7,
    ACCOUNT_DATA_TYPE_NUM_ACCOUNT_DATA_TYPES = 0x8,
} e_account_data_type;
static const value_string e_account_data_type_strings[] =  {
    { ACCOUNT_DATA_TYPE_GLOBAL_CONFIG_CACHE, "Global Config Cache" },
    { ACCOUNT_DATA_TYPE_PER_CHARACTER_CONFIG_CACHE, "Per Character Config Cache" },
    { ACCOUNT_DATA_TYPE_GLOBAL_BINDINGS_CACHE, "Global Bindings Cache" },
    { ACCOUNT_DATA_TYPE_PER_CHARACTER_BINDINGS_CACHE, "Per Character Bindings Cache" },
    { ACCOUNT_DATA_TYPE_GLOBAL_MACROS_CACHE, "Global Macros Cache" },
    { ACCOUNT_DATA_TYPE_PER_CHARACTER_MACROS_CACHE, "Per Character Macros Cache" },
    { ACCOUNT_DATA_TYPE_PER_CHARACTER_LAYOUT_CACHE, "Per Character Layout Cache" },
    { ACCOUNT_DATA_TYPE_PER_CHARACTER_CHAT_CACHE, "Per Character Chat Cache" },
    { ACCOUNT_DATA_TYPE_NUM_ACCOUNT_DATA_TYPES, "Num Account Data Types" },
    { 0, NULL }
};

typedef enum {
    AUCTION_HOUSE_STORMWIND = 0x1,
    AUCTION_HOUSE_ALLIANCE = 0x2,
    AUCTION_HOUSE_DARNASSUS = 0x3,
    AUCTION_HOUSE_UNDERCITY = 0x4,
    AUCTION_HOUSE_THUNDER_BLUFF = 0x5,
    AUCTION_HOUSE_HORDE = 0x6,
    AUCTION_HOUSE_GOBLIN = 0x7,
} e_auction_house;
static const value_string e_auction_house_strings[] =  {
    { AUCTION_HOUSE_STORMWIND, "Stormwind" },
    { AUCTION_HOUSE_ALLIANCE, "Alliance" },
    { AUCTION_HOUSE_DARNASSUS, "Darnassus" },
    { AUCTION_HOUSE_UNDERCITY, "Undercity" },
    { AUCTION_HOUSE_THUNDER_BLUFF, "Thunder Bluff" },
    { AUCTION_HOUSE_HORDE, "Horde" },
    { AUCTION_HOUSE_GOBLIN, "Goblin" },
    { 0, NULL }
};

typedef enum {
    CORPSE_QUERY_RESULT_NOT_FOUND = 0x0,
    CORPSE_QUERY_RESULT_FOUND = 0x1,
} e_corpse_query_result;
static const value_string e_corpse_query_result_strings[] =  {
    { CORPSE_QUERY_RESULT_NOT_FOUND, "Not Found" },
    { CORPSE_QUERY_RESULT_FOUND, "Found" },
    { 0, NULL }
};

typedef enum {
    BATTLEGROUND_END_STATUS_NOT_ENDED = 0x0,
    BATTLEGROUND_END_STATUS_ENDED = 0x1,
} e_battleground_end_status;
static const value_string e_battleground_end_status_strings[] =  {
    { BATTLEGROUND_END_STATUS_NOT_ENDED, "Not Ended" },
    { BATTLEGROUND_END_STATUS_ENDED, "Ended" },
    { 0, NULL }
};

typedef enum {
    BATTLEGROUND_WINNER_HORDE = 0x0,
    BATTLEGROUND_WINNER_ALLIANCE = 0x1,
    BATTLEGROUND_WINNER_NONE = 0x2,
} e_battleground_winner;
static const value_string e_battleground_winner_strings[] =  {
    { BATTLEGROUND_WINNER_HORDE, "Horde" },
    { BATTLEGROUND_WINNER_ALLIANCE, "Alliance" },
    { BATTLEGROUND_WINNER_NONE, "None" },
    { 0, NULL }
};

typedef enum {
    QUEST_PARTY_MESSAGE_SHARING_QUEST = 0x0,
    QUEST_PARTY_MESSAGE_CANT_TAKE_QUEST = 0x1,
    QUEST_PARTY_MESSAGE_ACCEPT_QUEST = 0x2,
    QUEST_PARTY_MESSAGE_DECLINE_QUEST = 0x3,
    QUEST_PARTY_MESSAGE_TOO_FAR = 0x4,
    QUEST_PARTY_MESSAGE_BUSY = 0x5,
    QUEST_PARTY_MESSAGE_LOG_FULL = 0x6,
    QUEST_PARTY_MESSAGE_HAVE_QUEST = 0x7,
    QUEST_PARTY_MESSAGE_FINISH_QUEST = 0x8,
} e_quest_party_message;
static const value_string e_quest_party_message_strings[] =  {
    { QUEST_PARTY_MESSAGE_SHARING_QUEST, "Sharing Quest" },
    { QUEST_PARTY_MESSAGE_CANT_TAKE_QUEST, "Cant Take Quest" },
    { QUEST_PARTY_MESSAGE_ACCEPT_QUEST, "Accept Quest" },
    { QUEST_PARTY_MESSAGE_DECLINE_QUEST, "Decline Quest" },
    { QUEST_PARTY_MESSAGE_TOO_FAR, "Too Far" },
    { QUEST_PARTY_MESSAGE_BUSY, "Busy" },
    { QUEST_PARTY_MESSAGE_LOG_FULL, "Log Full" },
    { QUEST_PARTY_MESSAGE_HAVE_QUEST, "Have Quest" },
    { QUEST_PARTY_MESSAGE_FINISH_QUEST, "Finish Quest" },
    { 0, NULL }
};

typedef enum {
    RAID_TARGET_UPDATE_TYPE_PARTIAL = 0x0,
    RAID_TARGET_UPDATE_TYPE_FULL = 0x1,
} e_raid_target_update_type;
static const value_string e_raid_target_update_type_strings[] =  {
    { RAID_TARGET_UPDATE_TYPE_PARTIAL, "Partial" },
    { RAID_TARGET_UPDATE_TYPE_FULL, "Full" },
    { 0, NULL }
};

typedef enum {
    GUILD_EMBLEM_RESULT_SUCCESS = 0x0,
    GUILD_EMBLEM_RESULT_INVALID_TABARD_COLORS = 0x1,
    GUILD_EMBLEM_RESULT_NO_GUILD = 0x2,
    GUILD_EMBLEM_RESULT_NOT_GUILD_MASTER = 0x3,
    GUILD_EMBLEM_RESULT_NOT_ENOUGH_MONEY = 0x4,
    GUILD_EMBLEM_RESULT_NO_MESSAGE = 0x5,
} e_guild_emblem_result;
static const value_string e_guild_emblem_result_strings[] =  {
    { GUILD_EMBLEM_RESULT_SUCCESS, "Success" },
    { GUILD_EMBLEM_RESULT_INVALID_TABARD_COLORS, "Invalid Tabard Colors" },
    { GUILD_EMBLEM_RESULT_NO_GUILD, "No Guild" },
    { GUILD_EMBLEM_RESULT_NOT_GUILD_MASTER, "Not Guild Master" },
    { GUILD_EMBLEM_RESULT_NOT_ENOUGH_MONEY, "Not Enough Money" },
    { GUILD_EMBLEM_RESULT_NO_MESSAGE, "No Message" },
    { 0, NULL }
};

typedef enum {
    ACTIVATE_TAXI_REPLY_OK = 0x0,
    ACTIVATE_TAXI_REPLY_UNSPECIFIED_SERVER_ERROR = 0x1,
    ACTIVATE_TAXI_REPLY_NO_SUCH_PATH = 0x2,
    ACTIVATE_TAXI_REPLY_NOT_ENOUGH_MONEY = 0x3,
    ACTIVATE_TAXI_REPLY_TOO_FAR_AWAY = 0x4,
    ACTIVATE_TAXI_REPLY_NO_VENDOR_NEARBY = 0x5,
    ACTIVATE_TAXI_REPLY_NOT_VISITED = 0x6,
    ACTIVATE_TAXI_REPLY_PLAYER_BUSY = 0x7,
    ACTIVATE_TAXI_REPLY_PLAYER_ALREADY_MOUNTED = 0x8,
    ACTIVATE_TAXI_REPLY_PLAYER_SHAPE_SHIFTED = 0x9,
    ACTIVATE_TAXI_REPLY_PLAYER_MOVING = 0xA,
    ACTIVATE_TAXI_REPLY_SAME_NODE = 0xB,
    ACTIVATE_TAXI_REPLY_NOT_STANDING = 0xC,
} e_activate_taxi_reply;
static const value_string e_activate_taxi_reply_strings[] =  {
    { ACTIVATE_TAXI_REPLY_OK, "Ok" },
    { ACTIVATE_TAXI_REPLY_UNSPECIFIED_SERVER_ERROR, "Unspecified Server Error" },
    { ACTIVATE_TAXI_REPLY_NO_SUCH_PATH, "No Such Path" },
    { ACTIVATE_TAXI_REPLY_NOT_ENOUGH_MONEY, "Not Enough Money" },
    { ACTIVATE_TAXI_REPLY_TOO_FAR_AWAY, "Too Far Away" },
    { ACTIVATE_TAXI_REPLY_NO_VENDOR_NEARBY, "No Vendor Nearby" },
    { ACTIVATE_TAXI_REPLY_NOT_VISITED, "Not Visited" },
    { ACTIVATE_TAXI_REPLY_PLAYER_BUSY, "Player Busy" },
    { ACTIVATE_TAXI_REPLY_PLAYER_ALREADY_MOUNTED, "Player Already Mounted" },
    { ACTIVATE_TAXI_REPLY_PLAYER_SHAPE_SHIFTED, "Player Shape Shifted" },
    { ACTIVATE_TAXI_REPLY_PLAYER_MOVING, "Player Moving" },
    { ACTIVATE_TAXI_REPLY_SAME_NODE, "Same Node" },
    { ACTIVATE_TAXI_REPLY_NOT_STANDING, "Not Standing" },
    { 0, NULL }
};

typedef enum {
    AI_REACTION_ALERT = 0x0,
    AI_REACTION_FRIENDLY = 0x1,
    AI_REACTION_HOSTILE = 0x2,
    AI_REACTION_AFRAID = 0x3,
    AI_REACTION_DESTROY = 0x4,
} e_ai_reaction;
static const value_string e_ai_reaction_strings[] =  {
    { AI_REACTION_ALERT, "Alert" },
    { AI_REACTION_FRIENDLY, "Friendly" },
    { AI_REACTION_HOSTILE, "Hostile" },
    { AI_REACTION_AFRAID, "Afraid" },
    { AI_REACTION_DESTROY, "Destroy" },
    { 0, NULL }
};

typedef enum {
    HIT_INFO_NORMAL_SWING = 0x00000,
    HIT_INFO_UNK1 = 0x00001,
    HIT_INFO_AFFECTS_VICTIM = 0x00002,
    HIT_INFO_LEFT_SWING = 0x00004,
    HIT_INFO_EARLY_CRITICAL_HIT = 0x00008,
    HIT_INFO_MISS = 0x00010,
    HIT_INFO_ABSORB = 0x00020,
    HIT_INFO_RESIST = 0x00040,
    HIT_INFO_CRITICAL_HIT = 0x00080,
    HIT_INFO_UNK9 = 0x00100,
    HIT_INFO_UNK10 = 0x02000,
    HIT_INFO_GLANCING = 0x04000,
    HIT_INFO_CRUSHING = 0x08000,
    HIT_INFO_NO_ACTION = 0x10000,
    HIT_INFO_SWING_NO_HIT_SOUND = 0x80000,
} e_hit_info;
static const value_string e_hit_info_strings[] =  {
    { HIT_INFO_NORMAL_SWING, "Normal Swing" },
    { HIT_INFO_UNK1, "Unk1" },
    { HIT_INFO_AFFECTS_VICTIM, "Affects Victim" },
    { HIT_INFO_LEFT_SWING, "Left Swing" },
    { HIT_INFO_EARLY_CRITICAL_HIT, "Early Critical Hit" },
    { HIT_INFO_MISS, "Miss" },
    { HIT_INFO_ABSORB, "Absorb" },
    { HIT_INFO_RESIST, "Resist" },
    { HIT_INFO_CRITICAL_HIT, "Critical Hit" },
    { HIT_INFO_UNK9, "Unk9" },
    { HIT_INFO_UNK10, "Unk10" },
    { HIT_INFO_GLANCING, "Glancing" },
    { HIT_INFO_CRUSHING, "Crushing" },
    { HIT_INFO_NO_ACTION, "No Action" },
    { HIT_INFO_SWING_NO_HIT_SOUND, "Swing No Hit Sound" },
    { 0, NULL }
};

typedef enum {
    AUCTION_COMMAND_ACTION_STARTED = 0x0,
    AUCTION_COMMAND_ACTION_REMOVED = 0x1,
    AUCTION_COMMAND_ACTION_BID_PLACED = 0x2,
} e_auction_command_action;
static const value_string e_auction_command_action_strings[] =  {
    { AUCTION_COMMAND_ACTION_STARTED, "Started" },
    { AUCTION_COMMAND_ACTION_REMOVED, "Removed" },
    { AUCTION_COMMAND_ACTION_BID_PLACED, "Bid Placed" },
    { 0, NULL }
};

typedef enum {
    AUCTION_COMMAND_RESULT_OK = 0x0,
    AUCTION_COMMAND_RESULT_ERR_INVENTORY = 0x1,
    AUCTION_COMMAND_RESULT_ERR_DATABASE = 0x2,
    AUCTION_COMMAND_RESULT_ERR_NOT_ENOUGH_MONEY = 0x3,
    AUCTION_COMMAND_RESULT_ERR_ITEM_NOT_FOUND = 0x4,
    AUCTION_COMMAND_RESULT_ERR_HIGHER_BID = 0x5,
    AUCTION_COMMAND_RESULT_ERR_BID_INCREMENT = 0x7,
    AUCTION_COMMAND_RESULT_ERR_BID_OWN = 0xA,
    AUCTION_COMMAND_RESULT_ERR_RESTRICTED_ACCOUNT = 0xD,
} e_auction_command_result;
static const value_string e_auction_command_result_strings[] =  {
    { AUCTION_COMMAND_RESULT_OK, "Ok" },
    { AUCTION_COMMAND_RESULT_ERR_INVENTORY, "Err Inventory" },
    { AUCTION_COMMAND_RESULT_ERR_DATABASE, "Err Database" },
    { AUCTION_COMMAND_RESULT_ERR_NOT_ENOUGH_MONEY, "Err Not Enough Money" },
    { AUCTION_COMMAND_RESULT_ERR_ITEM_NOT_FOUND, "Err Item Not Found" },
    { AUCTION_COMMAND_RESULT_ERR_HIGHER_BID, "Err Higher Bid" },
    { AUCTION_COMMAND_RESULT_ERR_BID_INCREMENT, "Err Bid Increment" },
    { AUCTION_COMMAND_RESULT_ERR_BID_OWN, "Err Bid Own" },
    { AUCTION_COMMAND_RESULT_ERR_RESTRICTED_ACCOUNT, "Err Restricted Account" },
    { 0, NULL }
};

typedef enum {
    INVENTORY_RESULT_OK = 0x00,
    INVENTORY_RESULT_CANT_EQUIP_LEVEL_I = 0x01,
    INVENTORY_RESULT_CANT_EQUIP_SKILL = 0x02,
    INVENTORY_RESULT_ITEM_DOESNT_GO_TO_SLOT = 0x03,
    INVENTORY_RESULT_BAG_FULL = 0x04,
    INVENTORY_RESULT_NONEMPTY_BAG_OVER_OTHER_BAG = 0x05,
    INVENTORY_RESULT_CANT_TRADE_EQUIP_BAGS = 0x06,
    INVENTORY_RESULT_ONLY_AMMO_CAN_GO_HERE = 0x07,
    INVENTORY_RESULT_NO_REQUIRED_PROFICIENCY = 0x08,
    INVENTORY_RESULT_NO_EQUIPMENT_SLOT_AVAILABLE = 0x09,
    INVENTORY_RESULT_YOU_CAN_NEVER_USE_THAT_ITEM = 0x0A,
    INVENTORY_RESULT_YOU_CAN_NEVER_USE_THAT_ITEM2 = 0x0B,
    INVENTORY_RESULT_NO_EQUIPMENT_SLOT_AVAILABLE2 = 0x0C,
    INVENTORY_RESULT_CANT_EQUIP_WITH_TWOHANDED = 0x0D,
    INVENTORY_RESULT_CANT_DUAL_WIELD = 0x0E,
    INVENTORY_RESULT_ITEM_DOESNT_GO_INTO_BAG = 0x0F,
    INVENTORY_RESULT_ITEM_DOESNT_GO_INTO_BAG2 = 0x10,
    INVENTORY_RESULT_CANT_CARRY_MORE_OF_THIS = 0x11,
    INVENTORY_RESULT_NO_EQUIPMENT_SLOT_AVAILABLE3 = 0x12,
    INVENTORY_RESULT_ITEM_CANT_STACK = 0x13,
    INVENTORY_RESULT_ITEM_CANT_BE_EQUIPPED = 0x14,
    INVENTORY_RESULT_ITEMS_CANT_BE_SWAPPED = 0x15,
    INVENTORY_RESULT_SLOT_IS_EMPTY = 0x16,
    INVENTORY_RESULT_ITEM_NOT_FOUND = 0x17,
    INVENTORY_RESULT_CANT_DROP_SOULBOUND = 0x18,
    INVENTORY_RESULT_OUT_OF_RANGE = 0x19,
    INVENTORY_RESULT_TRIED_TO_SPLIT_MORE_THAN_COUNT = 0x1A,
    INVENTORY_RESULT_COULDNT_SPLIT_ITEMS = 0x1B,
    INVENTORY_RESULT_MISSING_REAGENT = 0x1C,
    INVENTORY_RESULT_NOT_ENOUGH_MONEY = 0x1D,
    INVENTORY_RESULT_NOT_A_BAG = 0x1E,
    INVENTORY_RESULT_CAN_ONLY_DO_WITH_EMPTY_BAGS = 0x1F,
    INVENTORY_RESULT_DONT_OWN_THAT_ITEM = 0x20,
    INVENTORY_RESULT_CAN_EQUIP_ONLY1_QUIVER = 0x21,
    INVENTORY_RESULT_MUST_PURCHASE_THAT_BAG_SLOT = 0x22,
    INVENTORY_RESULT_TOO_FAR_AWAY_FROM_BANK = 0x23,
    INVENTORY_RESULT_ITEM_LOCKED = 0x24,
    INVENTORY_RESULT_YOU_ARE_STUNNED = 0x25,
    INVENTORY_RESULT_YOU_ARE_DEAD = 0x26,
    INVENTORY_RESULT_CANT_DO_RIGHT_NOW = 0x27,
    INVENTORY_RESULT_INT_BAG_ERROR = 0x28,
    INVENTORY_RESULT_CAN_EQUIP_ONLY1_BOLT = 0x29,
    INVENTORY_RESULT_CAN_EQUIP_ONLY1_AMMOPOUCH = 0x2A,
    INVENTORY_RESULT_STACKABLE_CANT_BE_WRAPPED = 0x2B,
    INVENTORY_RESULT_EQUIPPED_CANT_BE_WRAPPED = 0x2C,
    INVENTORY_RESULT_WRAPPED_CANT_BE_WRAPPED = 0x2D,
    INVENTORY_RESULT_BOUND_CANT_BE_WRAPPED = 0x2E,
    INVENTORY_RESULT_UNIQUE_CANT_BE_WRAPPED = 0x2F,
    INVENTORY_RESULT_BAGS_CANT_BE_WRAPPED = 0x30,
    INVENTORY_RESULT_ALREADY_LOOTED = 0x31,
    INVENTORY_RESULT_INVENTORY_FULL = 0x32,
    INVENTORY_RESULT_BANK_FULL = 0x33,
    INVENTORY_RESULT_ITEM_IS_CURRENTLY_SOLD_OUT = 0x34,
    INVENTORY_RESULT_BAG_FULL3 = 0x35,
    INVENTORY_RESULT_ITEM_NOT_FOUND2 = 0x36,
    INVENTORY_RESULT_ITEM_CANT_STACK2 = 0x37,
    INVENTORY_RESULT_BAG_FULL4 = 0x38,
    INVENTORY_RESULT_ITEM_SOLD_OUT = 0x39,
    INVENTORY_RESULT_OBJECT_IS_BUSY = 0x3A,
    INVENTORY_RESULT_NONE = 0x3B,
    INVENTORY_RESULT_NOT_IN_COMBAT = 0x3C,
    INVENTORY_RESULT_NOT_WHILE_DISARMED = 0x3D,
    INVENTORY_RESULT_BAG_FULL6 = 0x3E,
    INVENTORY_RESULT_CANT_EQUIP_RANK = 0x3F,
    INVENTORY_RESULT_CANT_EQUIP_REPUTATION = 0x40,
    INVENTORY_RESULT_TOO_MANY_SPECIAL_BAGS = 0x41,
    INVENTORY_RESULT_LOOT_CANT_LOOT_THAT_NOW = 0x42,
} e_inventory_result;
static const value_string e_inventory_result_strings[] =  {
    { INVENTORY_RESULT_OK, "Ok" },
    { INVENTORY_RESULT_CANT_EQUIP_LEVEL_I, "Cant Equip Level I" },
    { INVENTORY_RESULT_CANT_EQUIP_SKILL, "Cant Equip Skill" },
    { INVENTORY_RESULT_ITEM_DOESNT_GO_TO_SLOT, "Item Doesn't Go To Slot" },
    { INVENTORY_RESULT_BAG_FULL, "Bag Full" },
    { INVENTORY_RESULT_NONEMPTY_BAG_OVER_OTHER_BAG, "Nonempty Bag Over Other Bag" },
    { INVENTORY_RESULT_CANT_TRADE_EQUIP_BAGS, "Cant Trade Equip Bags" },
    { INVENTORY_RESULT_ONLY_AMMO_CAN_GO_HERE, "Only Ammo Can Go Here" },
    { INVENTORY_RESULT_NO_REQUIRED_PROFICIENCY, "No Required Proficiency" },
    { INVENTORY_RESULT_NO_EQUIPMENT_SLOT_AVAILABLE, "No Equipment Slot Available" },
    { INVENTORY_RESULT_YOU_CAN_NEVER_USE_THAT_ITEM, "You Can Never Use That Item" },
    { INVENTORY_RESULT_YOU_CAN_NEVER_USE_THAT_ITEM2, "You Can Never Use That Item2" },
    { INVENTORY_RESULT_NO_EQUIPMENT_SLOT_AVAILABLE2, "No Equipment Slot Available2" },
    { INVENTORY_RESULT_CANT_EQUIP_WITH_TWOHANDED, "Cant Equip With Twohanded" },
    { INVENTORY_RESULT_CANT_DUAL_WIELD, "Cant Dual Wield" },
    { INVENTORY_RESULT_ITEM_DOESNT_GO_INTO_BAG, "Item Doesn't Go Into Bag" },
    { INVENTORY_RESULT_ITEM_DOESNT_GO_INTO_BAG2, "Item Doesn't Go Into Bag2" },
    { INVENTORY_RESULT_CANT_CARRY_MORE_OF_THIS, "Cant Carry More Of This" },
    { INVENTORY_RESULT_NO_EQUIPMENT_SLOT_AVAILABLE3, "No Equipment Slot Available3" },
    { INVENTORY_RESULT_ITEM_CANT_STACK, "Item Cant Stack" },
    { INVENTORY_RESULT_ITEM_CANT_BE_EQUIPPED, "Item Cant Be Equipped" },
    { INVENTORY_RESULT_ITEMS_CANT_BE_SWAPPED, "Items Cant Be Swapped" },
    { INVENTORY_RESULT_SLOT_IS_EMPTY, "Slot Is Empty" },
    { INVENTORY_RESULT_ITEM_NOT_FOUND, "Item Not Found" },
    { INVENTORY_RESULT_CANT_DROP_SOULBOUND, "Cant Drop Soulbound" },
    { INVENTORY_RESULT_OUT_OF_RANGE, "Out Of Range" },
    { INVENTORY_RESULT_TRIED_TO_SPLIT_MORE_THAN_COUNT, "Tried To Split More Than Count" },
    { INVENTORY_RESULT_COULDNT_SPLIT_ITEMS, "Couldnt Split Items" },
    { INVENTORY_RESULT_MISSING_REAGENT, "Missing Reagent" },
    { INVENTORY_RESULT_NOT_ENOUGH_MONEY, "Not Enough Money" },
    { INVENTORY_RESULT_NOT_A_BAG, "Not A Bag" },
    { INVENTORY_RESULT_CAN_ONLY_DO_WITH_EMPTY_BAGS, "Can Only Do With Empty Bags" },
    { INVENTORY_RESULT_DONT_OWN_THAT_ITEM, "Dont Own That Item" },
    { INVENTORY_RESULT_CAN_EQUIP_ONLY1_QUIVER, "Can Equip Only1 Quiver" },
    { INVENTORY_RESULT_MUST_PURCHASE_THAT_BAG_SLOT, "Must Purchase That Bag Slot" },
    { INVENTORY_RESULT_TOO_FAR_AWAY_FROM_BANK, "Too Far Away From Bank" },
    { INVENTORY_RESULT_ITEM_LOCKED, "Item Locked" },
    { INVENTORY_RESULT_YOU_ARE_STUNNED, "You Are Stunned" },
    { INVENTORY_RESULT_YOU_ARE_DEAD, "You Are Dead" },
    { INVENTORY_RESULT_CANT_DO_RIGHT_NOW, "Cant Do Right Now" },
    { INVENTORY_RESULT_INT_BAG_ERROR, "Int Bag Error" },
    { INVENTORY_RESULT_CAN_EQUIP_ONLY1_BOLT, "Can Equip Only1 Bolt" },
    { INVENTORY_RESULT_CAN_EQUIP_ONLY1_AMMOPOUCH, "Can Equip Only1 Ammopouch" },
    { INVENTORY_RESULT_STACKABLE_CANT_BE_WRAPPED, "Stackable Cant Be Wrapped" },
    { INVENTORY_RESULT_EQUIPPED_CANT_BE_WRAPPED, "Equipped Cant Be Wrapped" },
    { INVENTORY_RESULT_WRAPPED_CANT_BE_WRAPPED, "Wrapped Cant Be Wrapped" },
    { INVENTORY_RESULT_BOUND_CANT_BE_WRAPPED, "Bound Cant Be Wrapped" },
    { INVENTORY_RESULT_UNIQUE_CANT_BE_WRAPPED, "Unique Cant Be Wrapped" },
    { INVENTORY_RESULT_BAGS_CANT_BE_WRAPPED, "Bags Cant Be Wrapped" },
    { INVENTORY_RESULT_ALREADY_LOOTED, "Already Looted" },
    { INVENTORY_RESULT_INVENTORY_FULL, "Inventory Full" },
    { INVENTORY_RESULT_BANK_FULL, "Bank Full" },
    { INVENTORY_RESULT_ITEM_IS_CURRENTLY_SOLD_OUT, "Item Is Currently Sold Out" },
    { INVENTORY_RESULT_BAG_FULL3, "Bag Full3" },
    { INVENTORY_RESULT_ITEM_NOT_FOUND2, "Item Not Found2" },
    { INVENTORY_RESULT_ITEM_CANT_STACK2, "Item Cant Stack2" },
    { INVENTORY_RESULT_BAG_FULL4, "Bag Full4" },
    { INVENTORY_RESULT_ITEM_SOLD_OUT, "Item Sold Out" },
    { INVENTORY_RESULT_OBJECT_IS_BUSY, "Object Is Busy" },
    { INVENTORY_RESULT_NONE, "None" },
    { INVENTORY_RESULT_NOT_IN_COMBAT, "Not In Combat" },
    { INVENTORY_RESULT_NOT_WHILE_DISARMED, "Not While Disarmed" },
    { INVENTORY_RESULT_BAG_FULL6, "Bag Full6" },
    { INVENTORY_RESULT_CANT_EQUIP_RANK, "Cant Equip Rank" },
    { INVENTORY_RESULT_CANT_EQUIP_REPUTATION, "Cant Equip Reputation" },
    { INVENTORY_RESULT_TOO_MANY_SPECIAL_BAGS, "Too Many Special Bags" },
    { INVENTORY_RESULT_LOOT_CANT_LOOT_THAT_NOW, "Loot Cant Loot That Now" },
    { 0, NULL }
};

typedef enum {
    AUCTION_COMMAND_RESULT_TWO_OK = 0x0,
    AUCTION_COMMAND_RESULT_TWO_ERR_INVENTORY = 0x1,
    AUCTION_COMMAND_RESULT_TWO_ERR_DATABASE = 0x2,
    AUCTION_COMMAND_RESULT_TWO_ERR_NOT_ENOUGH_MONEY = 0x3,
    AUCTION_COMMAND_RESULT_TWO_ERR_ITEM_NOT_FOUND = 0x4,
    AUCTION_COMMAND_RESULT_TWO_ERR_HIGHER_BID = 0x5,
    AUCTION_COMMAND_RESULT_TWO_ERR_BID_INCREMENT = 0x7,
    AUCTION_COMMAND_RESULT_TWO_ERR_BID_OWN = 0xA,
    AUCTION_COMMAND_RESULT_TWO_ERR_RESTRICTED_ACCOUNT = 0xD,
} e_auction_command_result_two;
static const value_string e_auction_command_result_two_strings[] =  {
    { AUCTION_COMMAND_RESULT_TWO_OK, "Ok" },
    { AUCTION_COMMAND_RESULT_TWO_ERR_INVENTORY, "Err Inventory" },
    { AUCTION_COMMAND_RESULT_TWO_ERR_DATABASE, "Err Database" },
    { AUCTION_COMMAND_RESULT_TWO_ERR_NOT_ENOUGH_MONEY, "Err Not Enough Money" },
    { AUCTION_COMMAND_RESULT_TWO_ERR_ITEM_NOT_FOUND, "Err Item Not Found" },
    { AUCTION_COMMAND_RESULT_TWO_ERR_HIGHER_BID, "Err Higher Bid" },
    { AUCTION_COMMAND_RESULT_TWO_ERR_BID_INCREMENT, "Err Bid Increment" },
    { AUCTION_COMMAND_RESULT_TWO_ERR_BID_OWN, "Err Bid Own" },
    { AUCTION_COMMAND_RESULT_TWO_ERR_RESTRICTED_ACCOUNT, "Err Restricted Account" },
    { 0, NULL }
};

typedef enum {
    WORLD_RESULT_RESPONSE_SUCCESS = 0x00,
    WORLD_RESULT_RESPONSE_FAILURE = 0x01,
    WORLD_RESULT_RESPONSE_CANCELLED = 0x02,
    WORLD_RESULT_RESPONSE_DISCONNECTED = 0x03,
    WORLD_RESULT_RESPONSE_FAILED_TO_CONNECT = 0x04,
    WORLD_RESULT_RESPONSE_CONNECTED = 0x05,
    WORLD_RESULT_RESPONSE_VERSION_MISMATCH = 0x06,
    WORLD_RESULT_CSTATUS_CONNECTING = 0x07,
    WORLD_RESULT_CSTATUS_NEGOTIATING_SECURITY = 0x08,
    WORLD_RESULT_CSTATUS_NEGOTIATION_COMPLETE = 0x09,
    WORLD_RESULT_CSTATUS_NEGOTIATION_FAILED = 0x0A,
    WORLD_RESULT_CSTATUS_AUTHENTICATING = 0x0B,
    WORLD_RESULT_AUTH_OK = 0x0C,
    WORLD_RESULT_AUTH_FAILED = 0x0D,
    WORLD_RESULT_AUTH_REJECT = 0x0E,
    WORLD_RESULT_AUTH_BAD_SERVER_PROOF = 0x0F,
    WORLD_RESULT_AUTH_UNAVAILABLE = 0x10,
    WORLD_RESULT_AUTH_SYSTEM_ERROR = 0x11,
    WORLD_RESULT_AUTH_BILLING_ERROR = 0x12,
    WORLD_RESULT_AUTH_BILLING_EXPIRED = 0x13,
    WORLD_RESULT_AUTH_VERSION_MISMATCH = 0x14,
    WORLD_RESULT_AUTH_UNKNOWN_ACCOUNT = 0x15,
    WORLD_RESULT_AUTH_INCORRECT_PASSWORD = 0x16,
    WORLD_RESULT_AUTH_SESSION_EXPIRED = 0x17,
    WORLD_RESULT_AUTH_SERVER_SHUTTING_DOWN = 0x18,
    WORLD_RESULT_AUTH_ALREADY_LOGGING_IN = 0x19,
    WORLD_RESULT_AUTH_LOGIN_SERVER_NOT_FOUND = 0x1A,
    WORLD_RESULT_AUTH_WAIT_QUEUE = 0x1B,
    WORLD_RESULT_AUTH_BANNED = 0x1C,
    WORLD_RESULT_AUTH_ALREADY_ONLINE = 0x1D,
    WORLD_RESULT_AUTH_NO_TIME = 0x1E,
    WORLD_RESULT_AUTH_DB_BUSY = 0x1F,
    WORLD_RESULT_AUTH_SUSPENDED = 0x20,
    WORLD_RESULT_AUTH_PARENTAL_CONTROL = 0x21,
    WORLD_RESULT_REALM_LIST_IN_PROGRESS = 0x22,
    WORLD_RESULT_REALM_LIST_SUCCESS = 0x23,
    WORLD_RESULT_REALM_LIST_FAILED = 0x24,
    WORLD_RESULT_REALM_LIST_INVALID = 0x25,
    WORLD_RESULT_REALM_LIST_REALM_NOT_FOUND = 0x26,
    WORLD_RESULT_ACCOUNT_CREATE_IN_PROGRESS = 0x27,
    WORLD_RESULT_ACCOUNT_CREATE_SUCCESS = 0x28,
    WORLD_RESULT_ACCOUNT_CREATE_FAILED = 0x29,
    WORLD_RESULT_CHAR_LIST_RETRIEVING = 0x2A,
    WORLD_RESULT_CHAR_LIST_RETRIEVED = 0x2B,
    WORLD_RESULT_CHAR_LIST_FAILED = 0x2C,
    WORLD_RESULT_CHAR_CREATE_IN_PROGRESS = 0x2D,
    WORLD_RESULT_CHAR_CREATE_SUCCESS = 0x2E,
    WORLD_RESULT_CHAR_CREATE_ERROR = 0x2F,
    WORLD_RESULT_CHAR_CREATE_FAILED = 0x30,
    WORLD_RESULT_CHAR_CREATE_NAME_IN_USE = 0x31,
    WORLD_RESULT_CHAR_CREATE_DISABLED = 0x32,
    WORLD_RESULT_CHAR_CREATE_PVP_TEAMS_VIOLATION = 0x33,
    WORLD_RESULT_CHAR_CREATE_SERVER_LIMIT = 0x34,
    WORLD_RESULT_CHAR_CREATE_ACCOUNT_LIMIT = 0x35,
    WORLD_RESULT_CHAR_CREATE_SERVER_QUEUE = 0x36,
    WORLD_RESULT_CHAR_CREATE_ONLY_EXISTING = 0x37,
    WORLD_RESULT_CHAR_DELETE_IN_PROGRESS = 0x38,
    WORLD_RESULT_CHAR_DELETE_SUCCESS = 0x39,
    WORLD_RESULT_CHAR_DELETE_FAILED = 0x3A,
    WORLD_RESULT_CHAR_DELETE_FAILED_LOCKED_FOR_TRANSFER = 0x3B,
    WORLD_RESULT_CHAR_LOGIN_IN_PROGRESS = 0x3C,
    WORLD_RESULT_CHAR_LOGIN_SUCCESS = 0x3D,
    WORLD_RESULT_CHAR_LOGIN_NO_WORLD = 0x3E,
    WORLD_RESULT_CHAR_LOGIN_DUPLICATE_CHARACTER = 0x3F,
    WORLD_RESULT_CHAR_LOGIN_NO_INSTANCES = 0x40,
    WORLD_RESULT_CHAR_LOGIN_FAILED = 0x41,
    WORLD_RESULT_CHAR_LOGIN_DISABLED = 0x42,
    WORLD_RESULT_CHAR_LOGIN_NO_CHARACTER = 0x43,
    WORLD_RESULT_CHAR_LOGIN_LOCKED_FOR_TRANSFER = 0x44,
    WORLD_RESULT_CHAR_NAME_NO_NAME = 0x45,
    WORLD_RESULT_CHAR_NAME_TOO_SHORT = 0x46,
    WORLD_RESULT_CHAR_NAME_TOO_LONG = 0x47,
    WORLD_RESULT_CHAR_NAME_ONLY_LETTERS = 0x48,
    WORLD_RESULT_CHAR_NAME_MIXED_LANGUAGES = 0x49,
    WORLD_RESULT_CHAR_NAME_PROFANE = 0x4A,
    WORLD_RESULT_CHAR_NAME_RESERVED = 0x4B,
    WORLD_RESULT_CHAR_NAME_INVALID_APOSTROPHE = 0x4C,
    WORLD_RESULT_CHAR_NAME_MULTIPLE_APOSTROPHES = 0x4D,
    WORLD_RESULT_CHAR_NAME_THREE_CONSECUTIVE = 0x4E,
    WORLD_RESULT_CHAR_NAME_INVALID_SPACE = 0x4F,
    WORLD_RESULT_CHAR_NAME_SUCCESS = 0x50,
    WORLD_RESULT_CHAR_NAME_FAILURE = 0x51,
} e_world_result;
static const value_string e_world_result_strings[] =  {
    { WORLD_RESULT_RESPONSE_SUCCESS, "Response Success" },
    { WORLD_RESULT_RESPONSE_FAILURE, "Response Failure" },
    { WORLD_RESULT_RESPONSE_CANCELLED, "Response Cancelled" },
    { WORLD_RESULT_RESPONSE_DISCONNECTED, "Response Disconnected" },
    { WORLD_RESULT_RESPONSE_FAILED_TO_CONNECT, "Response Failed To Connect" },
    { WORLD_RESULT_RESPONSE_CONNECTED, "Response Connected" },
    { WORLD_RESULT_RESPONSE_VERSION_MISMATCH, "Response Version Mismatch" },
    { WORLD_RESULT_CSTATUS_CONNECTING, "Cstatus Connecting" },
    { WORLD_RESULT_CSTATUS_NEGOTIATING_SECURITY, "Cstatus Negotiating Security" },
    { WORLD_RESULT_CSTATUS_NEGOTIATION_COMPLETE, "Cstatus Negotiation Complete" },
    { WORLD_RESULT_CSTATUS_NEGOTIATION_FAILED, "Cstatus Negotiation Failed" },
    { WORLD_RESULT_CSTATUS_AUTHENTICATING, "Cstatus Authenticating" },
    { WORLD_RESULT_AUTH_OK, "Auth Ok" },
    { WORLD_RESULT_AUTH_FAILED, "Auth Failed" },
    { WORLD_RESULT_AUTH_REJECT, "Auth Reject" },
    { WORLD_RESULT_AUTH_BAD_SERVER_PROOF, "Auth Bad Server Proof" },
    { WORLD_RESULT_AUTH_UNAVAILABLE, "Auth Unavailable" },
    { WORLD_RESULT_AUTH_SYSTEM_ERROR, "Auth System Error" },
    { WORLD_RESULT_AUTH_BILLING_ERROR, "Auth Billing Error" },
    { WORLD_RESULT_AUTH_BILLING_EXPIRED, "Auth Billing Expired" },
    { WORLD_RESULT_AUTH_VERSION_MISMATCH, "Auth Version Mismatch" },
    { WORLD_RESULT_AUTH_UNKNOWN_ACCOUNT, "Auth Unknown Account" },
    { WORLD_RESULT_AUTH_INCORRECT_PASSWORD, "Auth Incorrect Password" },
    { WORLD_RESULT_AUTH_SESSION_EXPIRED, "Auth Session Expired" },
    { WORLD_RESULT_AUTH_SERVER_SHUTTING_DOWN, "Auth Server Shutting Down" },
    { WORLD_RESULT_AUTH_ALREADY_LOGGING_IN, "Auth Already Logging In" },
    { WORLD_RESULT_AUTH_LOGIN_SERVER_NOT_FOUND, "Auth Login Server Not Found" },
    { WORLD_RESULT_AUTH_WAIT_QUEUE, "Auth Wait Queue" },
    { WORLD_RESULT_AUTH_BANNED, "Auth Banned" },
    { WORLD_RESULT_AUTH_ALREADY_ONLINE, "Auth Already Online" },
    { WORLD_RESULT_AUTH_NO_TIME, "Auth No Time" },
    { WORLD_RESULT_AUTH_DB_BUSY, "Auth Db Busy" },
    { WORLD_RESULT_AUTH_SUSPENDED, "Auth Suspended" },
    { WORLD_RESULT_AUTH_PARENTAL_CONTROL, "Auth Parental Control" },
    { WORLD_RESULT_REALM_LIST_IN_PROGRESS, "Realm List In Progress" },
    { WORLD_RESULT_REALM_LIST_SUCCESS, "Realm List Success" },
    { WORLD_RESULT_REALM_LIST_FAILED, "Realm List Failed" },
    { WORLD_RESULT_REALM_LIST_INVALID, "Realm List Invalid" },
    { WORLD_RESULT_REALM_LIST_REALM_NOT_FOUND, "Realm List Realm Not Found" },
    { WORLD_RESULT_ACCOUNT_CREATE_IN_PROGRESS, "Account Create In Progress" },
    { WORLD_RESULT_ACCOUNT_CREATE_SUCCESS, "Account Create Success" },
    { WORLD_RESULT_ACCOUNT_CREATE_FAILED, "Account Create Failed" },
    { WORLD_RESULT_CHAR_LIST_RETRIEVING, "Char List Retrieving" },
    { WORLD_RESULT_CHAR_LIST_RETRIEVED, "Char List Retrieved" },
    { WORLD_RESULT_CHAR_LIST_FAILED, "Char List Failed" },
    { WORLD_RESULT_CHAR_CREATE_IN_PROGRESS, "Char Create In Progress" },
    { WORLD_RESULT_CHAR_CREATE_SUCCESS, "Char Create Success" },
    { WORLD_RESULT_CHAR_CREATE_ERROR, "Char Create Error" },
    { WORLD_RESULT_CHAR_CREATE_FAILED, "Char Create Failed" },
    { WORLD_RESULT_CHAR_CREATE_NAME_IN_USE, "Char Create Name In Use" },
    { WORLD_RESULT_CHAR_CREATE_DISABLED, "Char Create Disabled" },
    { WORLD_RESULT_CHAR_CREATE_PVP_TEAMS_VIOLATION, "Char Create Pvp Teams Violation" },
    { WORLD_RESULT_CHAR_CREATE_SERVER_LIMIT, "Char Create Server Limit" },
    { WORLD_RESULT_CHAR_CREATE_ACCOUNT_LIMIT, "Char Create Account Limit" },
    { WORLD_RESULT_CHAR_CREATE_SERVER_QUEUE, "Char Create Server Queue" },
    { WORLD_RESULT_CHAR_CREATE_ONLY_EXISTING, "Char Create Only Existing" },
    { WORLD_RESULT_CHAR_DELETE_IN_PROGRESS, "Char Delete In Progress" },
    { WORLD_RESULT_CHAR_DELETE_SUCCESS, "Char Delete Success" },
    { WORLD_RESULT_CHAR_DELETE_FAILED, "Char Delete Failed" },
    { WORLD_RESULT_CHAR_DELETE_FAILED_LOCKED_FOR_TRANSFER, "Char Delete Failed Locked For Transfer" },
    { WORLD_RESULT_CHAR_LOGIN_IN_PROGRESS, "Char Login In Progress" },
    { WORLD_RESULT_CHAR_LOGIN_SUCCESS, "Char Login Success" },
    { WORLD_RESULT_CHAR_LOGIN_NO_WORLD, "Char Login No World" },
    { WORLD_RESULT_CHAR_LOGIN_DUPLICATE_CHARACTER, "Char Login Duplicate Character" },
    { WORLD_RESULT_CHAR_LOGIN_NO_INSTANCES, "Char Login No Instances" },
    { WORLD_RESULT_CHAR_LOGIN_FAILED, "Char Login Failed" },
    { WORLD_RESULT_CHAR_LOGIN_DISABLED, "Char Login Disabled" },
    { WORLD_RESULT_CHAR_LOGIN_NO_CHARACTER, "Char Login No Character" },
    { WORLD_RESULT_CHAR_LOGIN_LOCKED_FOR_TRANSFER, "Char Login Locked For Transfer" },
    { WORLD_RESULT_CHAR_NAME_NO_NAME, "Char Name No Name" },
    { WORLD_RESULT_CHAR_NAME_TOO_SHORT, "Char Name Too Short" },
    { WORLD_RESULT_CHAR_NAME_TOO_LONG, "Char Name Too Long" },
    { WORLD_RESULT_CHAR_NAME_ONLY_LETTERS, "Char Name Only Letters" },
    { WORLD_RESULT_CHAR_NAME_MIXED_LANGUAGES, "Char Name Mixed Languages" },
    { WORLD_RESULT_CHAR_NAME_PROFANE, "Char Name Profane" },
    { WORLD_RESULT_CHAR_NAME_RESERVED, "Char Name Reserved" },
    { WORLD_RESULT_CHAR_NAME_INVALID_APOSTROPHE, "Char Name Invalid Apostrophe" },
    { WORLD_RESULT_CHAR_NAME_MULTIPLE_APOSTROPHES, "Char Name Multiple Apostrophes" },
    { WORLD_RESULT_CHAR_NAME_THREE_CONSECUTIVE, "Char Name Three Consecutive" },
    { WORLD_RESULT_CHAR_NAME_INVALID_SPACE, "Char Name Invalid Space" },
    { WORLD_RESULT_CHAR_NAME_SUCCESS, "Char Name Success" },
    { WORLD_RESULT_CHAR_NAME_FAILURE, "Char Name Failure" },
    { 0, NULL }
};

typedef enum {
    BATTLEGROUND_BRACKET_TENS = 0x0,
    BATTLEGROUND_BRACKET_TWENTIES = 0x1,
    BATTLEGROUND_BRACKET_THIRTIES = 0x2,
    BATTLEGROUND_BRACKET_FORTIES = 0x3,
    BATTLEGROUND_BRACKET_FIFTIES = 0x4,
    BATTLEGROUND_BRACKET_SIXTY = 0x5,
} e_battleground_bracket;
static const value_string e_battleground_bracket_strings[] =  {
    { BATTLEGROUND_BRACKET_TENS, "Tens" },
    { BATTLEGROUND_BRACKET_TWENTIES, "Twenties" },
    { BATTLEGROUND_BRACKET_THIRTIES, "Thirties" },
    { BATTLEGROUND_BRACKET_FORTIES, "Forties" },
    { BATTLEGROUND_BRACKET_FIFTIES, "Fifties" },
    { BATTLEGROUND_BRACKET_SIXTY, "Sixty" },
    { 0, NULL }
};

typedef enum {
    STATUS_ID_NONE = 0x0,
    STATUS_ID_WAIT_QUEUE = 0x1,
    STATUS_ID_WAIT_JOIN = 0x2,
    STATUS_ID_IN_PROGRESS = 0x3,
    STATUS_ID_WAIT_LEAVE = 0x4,
} e_status_id;
static const value_string e_status_id_strings[] =  {
    { STATUS_ID_NONE, "None" },
    { STATUS_ID_WAIT_QUEUE, "Wait Queue" },
    { STATUS_ID_WAIT_JOIN, "Wait Join" },
    { STATUS_ID_IN_PROGRESS, "In Progress" },
    { STATUS_ID_WAIT_LEAVE, "Wait Leave" },
    { 0, NULL }
};

typedef enum {
    BUY_BANK_SLOT_RESULT_FAILED_TOO_MANY = 0x0,
    BUY_BANK_SLOT_RESULT_INSUFFICIENT_FUNDS = 0x1,
    BUY_BANK_SLOT_RESULT_NOT_BANKER = 0x2,
    BUY_BANK_SLOT_RESULT_OK = 0x3,
} e_buy_bank_slot_result;
static const value_string e_buy_bank_slot_result_strings[] =  {
    { BUY_BANK_SLOT_RESULT_FAILED_TOO_MANY, "Failed Too Many" },
    { BUY_BANK_SLOT_RESULT_INSUFFICIENT_FUNDS, "Insufficient Funds" },
    { BUY_BANK_SLOT_RESULT_NOT_BANKER, "Not Banker" },
    { BUY_BANK_SLOT_RESULT_OK, "Ok" },
    { 0, NULL }
};

typedef enum {
    BUY_RESULT_CANT_FIND_ITEM = 0x0,
    BUY_RESULT_ITEM_ALREADY_SOLD = 0x1,
    BUY_RESULT_NOT_ENOUGH_MONEY = 0x2,
    BUY_RESULT_SELLER_DONT_LIKE_YOU = 0x4,
    BUY_RESULT_DISTANCE_TOO_FAR = 0x5,
    BUY_RESULT_ITEM_SOLD_OUT = 0x7,
    BUY_RESULT_CANT_CARRY_MORE = 0x8,
    BUY_RESULT_RANK_REQUIRE = 0xB,
    BUY_RESULT_REPUTATION_REQUIRE = 0xC,
} e_buy_result;
static const value_string e_buy_result_strings[] =  {
    { BUY_RESULT_CANT_FIND_ITEM, "Cant Find Item" },
    { BUY_RESULT_ITEM_ALREADY_SOLD, "Item Already Sold" },
    { BUY_RESULT_NOT_ENOUGH_MONEY, "Not Enough Money" },
    { BUY_RESULT_SELLER_DONT_LIKE_YOU, "Seller Dont Like You" },
    { BUY_RESULT_DISTANCE_TOO_FAR, "Distance Too Far" },
    { BUY_RESULT_ITEM_SOLD_OUT, "Item Sold Out" },
    { BUY_RESULT_CANT_CARRY_MORE, "Cant Carry More" },
    { BUY_RESULT_RANK_REQUIRE, "Rank Require" },
    { BUY_RESULT_REPUTATION_REQUIRE, "Reputation Require" },
    { 0, NULL }
};

typedef enum {
    SIMPLE_SPELL_CAST_RESULT_SUCCESS = 0x0,
    SIMPLE_SPELL_CAST_RESULT_FAILURE = 0x2,
} e_simple_spell_cast_result;
static const value_string e_simple_spell_cast_result_strings[] =  {
    { SIMPLE_SPELL_CAST_RESULT_SUCCESS, "Success" },
    { SIMPLE_SPELL_CAST_RESULT_FAILURE, "Failure" },
    { 0, NULL }
};

typedef enum {
    CAST_FAILURE_REASON_AFFECTING_COMBAT = 0x00,
    CAST_FAILURE_REASON_ALREADY_AT_FULL_HEALTH = 0x01,
    CAST_FAILURE_REASON_ALREADY_AT_FULL_POWER = 0x02,
    CAST_FAILURE_REASON_ALREADY_BEING_TAMED = 0x03,
    CAST_FAILURE_REASON_ALREADY_HAVE_CHARM = 0x04,
    CAST_FAILURE_REASON_ALREADY_HAVE_SUMMON = 0x05,
    CAST_FAILURE_REASON_ALREADY_OPEN = 0x06,
    CAST_FAILURE_REASON_AURA_BOUNCED = 0x07,
    CAST_FAILURE_REASON_AUTOTRACK_INTERRUPTED = 0x08,
    CAST_FAILURE_REASON_BAD_IMPLICIT_TARGETS = 0x09,
    CAST_FAILURE_REASON_BAD_TARGETS = 0x0A,
    CAST_FAILURE_REASON_CANT_BE_CHARMED = 0x0B,
    CAST_FAILURE_REASON_CANT_BE_DISENCHANTED = 0x0C,
    CAST_FAILURE_REASON_CANT_BE_PROSPECTED = 0x0D,
    CAST_FAILURE_REASON_CANT_CAST_ON_TAPPED = 0x0E,
    CAST_FAILURE_REASON_CANT_DUEL_WHILE_INVISIBLE = 0x0F,
    CAST_FAILURE_REASON_CANT_DUEL_WHILE_STEALTHED = 0x10,
    CAST_FAILURE_REASON_CANT_STEALTH = 0x11,
    CAST_FAILURE_REASON_CASTER_AURASTATE = 0x12,
    CAST_FAILURE_REASON_CASTER_DEAD = 0x13,
    CAST_FAILURE_REASON_CHARMED = 0x14,
    CAST_FAILURE_REASON_CHEST_IN_USE = 0x15,
    CAST_FAILURE_REASON_CONFUSED = 0x16,
    CAST_FAILURE_REASON_DONT_REPORT = 0x17,
    CAST_FAILURE_REASON_EQUIPPED_ITEM = 0x18,
    CAST_FAILURE_REASON_EQUIPPED_ITEM_CLASS = 0x19,
    CAST_FAILURE_REASON_EQUIPPED_ITEM_CLASS_MAINHAND = 0x1A,
    CAST_FAILURE_REASON_EQUIPPED_ITEM_CLASS_OFFHAND = 0x1B,
    CAST_FAILURE_REASON_ERROR = 0x1C,
    CAST_FAILURE_REASON_FIZZLE = 0x1D,
    CAST_FAILURE_REASON_FLEEING = 0x1E,
    CAST_FAILURE_REASON_FOOD_LOWLEVEL = 0x1F,
    CAST_FAILURE_REASON_HIGHLEVEL = 0x20,
    CAST_FAILURE_REASON_HUNGER_SATIATED = 0x21,
    CAST_FAILURE_REASON_IMMUNE = 0x22,
    CAST_FAILURE_REASON_INTERRUPTED = 0x23,
    CAST_FAILURE_REASON_INTERRUPTED_COMBAT = 0x24,
    CAST_FAILURE_REASON_ITEM_ALREADY_ENCHANTED = 0x25,
    CAST_FAILURE_REASON_ITEM_GONE = 0x26,
    CAST_FAILURE_REASON_ITEM_NOT_FOUND = 0x27,
    CAST_FAILURE_REASON_ITEM_NOT_READY = 0x28,
    CAST_FAILURE_REASON_LEVEL_REQUIREMENT = 0x29,
    CAST_FAILURE_REASON_LINE_OF_SIGHT = 0x2A,
    CAST_FAILURE_REASON_LOWLEVEL = 0x2B,
    CAST_FAILURE_REASON_LOW_CASTLEVEL = 0x2C,
    CAST_FAILURE_REASON_MAINHAND_EMPTY = 0x2D,
    CAST_FAILURE_REASON_MOVING = 0x2E,
    CAST_FAILURE_REASON_NEED_AMMO = 0x2F,
    CAST_FAILURE_REASON_NEED_AMMO_POUCH = 0x30,
    CAST_FAILURE_REASON_NEED_EXOTIC_AMMO = 0x31,
    CAST_FAILURE_REASON_NOPATH = 0x32,
    CAST_FAILURE_REASON_NOT_BEHIND = 0x33,
    CAST_FAILURE_REASON_NOT_FISHABLE = 0x34,
    CAST_FAILURE_REASON_NOT_HERE = 0x35,
    CAST_FAILURE_REASON_NOT_INFRONT = 0x36,
    CAST_FAILURE_REASON_NOT_IN_CONTROL = 0x37,
    CAST_FAILURE_REASON_NOT_KNOWN = 0x38,
    CAST_FAILURE_REASON_NOT_MOUNTED = 0x39,
    CAST_FAILURE_REASON_NOT_ON_TAXI = 0x3A,
    CAST_FAILURE_REASON_NOT_ON_TRANSPORT = 0x3B,
    CAST_FAILURE_REASON_NOT_READY = 0x3C,
    CAST_FAILURE_REASON_NOT_SHAPESHIFT = 0x3D,
    CAST_FAILURE_REASON_NOT_STANDING = 0x3E,
    CAST_FAILURE_REASON_NOT_TRADEABLE = 0x3F,
    CAST_FAILURE_REASON_NOT_TRADING = 0x40,
    CAST_FAILURE_REASON_NOT_UNSHEATHED = 0x41,
    CAST_FAILURE_REASON_NOT_WHILE_GHOST = 0x42,
    CAST_FAILURE_REASON_NO_AMMO = 0x43,
    CAST_FAILURE_REASON_NO_CHARGES_REMAIN = 0x44,
    CAST_FAILURE_REASON_NO_CHAMPION = 0x45,
    CAST_FAILURE_REASON_NO_COMBO_POINTS = 0x46,
    CAST_FAILURE_REASON_NO_DUELING = 0x47,
    CAST_FAILURE_REASON_NO_ENDURANCE = 0x48,
    CAST_FAILURE_REASON_NO_FISH = 0x49,
    CAST_FAILURE_REASON_NO_ITEMS_WHILE_SHAPESHIFTED = 0x4A,
    CAST_FAILURE_REASON_NO_MOUNTS_ALLOWED = 0x4B,
    CAST_FAILURE_REASON_NO_PET = 0x4C,
    CAST_FAILURE_REASON_NO_POWER = 0x4D,
    CAST_FAILURE_REASON_NOTHING_TO_DISPEL = 0x4E,
    CAST_FAILURE_REASON_NOTHING_TO_STEAL = 0x4F,
    CAST_FAILURE_REASON_ONLY_ABOVEWATER = 0x50,
    CAST_FAILURE_REASON_ONLY_DAYTIME = 0x51,
    CAST_FAILURE_REASON_ONLY_INDOORS = 0x52,
    CAST_FAILURE_REASON_ONLY_MOUNTED = 0x53,
    CAST_FAILURE_REASON_ONLY_NIGHTTIME = 0x54,
    CAST_FAILURE_REASON_ONLY_OUTDOORS = 0x55,
    CAST_FAILURE_REASON_ONLY_SHAPESHIFT = 0x56,
    CAST_FAILURE_REASON_ONLY_STEALTHED = 0x57,
    CAST_FAILURE_REASON_ONLY_UNDERWATER = 0x58,
    CAST_FAILURE_REASON_OUT_OF_RANGE = 0x59,
    CAST_FAILURE_REASON_PACIFIED = 0x5A,
    CAST_FAILURE_REASON_POSSESSED = 0x5B,
    CAST_FAILURE_REASON_REAGENTS = 0x5C,
    CAST_FAILURE_REASON_REQUIRES_AREA = 0x5D,
    CAST_FAILURE_REASON_REQUIRES_SPELL_FOCUS = 0x5E,
    CAST_FAILURE_REASON_ROOTED = 0x5F,
    CAST_FAILURE_REASON_SILENCED = 0x60,
    CAST_FAILURE_REASON_SPELL_IN_PROGRESS = 0x61,
    CAST_FAILURE_REASON_SPELL_LEARNED = 0x62,
    CAST_FAILURE_REASON_SPELL_UNAVAILABLE = 0x63,
    CAST_FAILURE_REASON_STUNNED = 0x64,
    CAST_FAILURE_REASON_TARGETS_DEAD = 0x65,
    CAST_FAILURE_REASON_TARGET_AFFECTING_COMBAT = 0x66,
    CAST_FAILURE_REASON_TARGET_AURASTATE = 0x67,
    CAST_FAILURE_REASON_TARGET_DUELING = 0x68,
    CAST_FAILURE_REASON_TARGET_ENEMY = 0x69,
    CAST_FAILURE_REASON_TARGET_ENRAGED = 0x6A,
    CAST_FAILURE_REASON_TARGET_FRIENDLY = 0x6B,
    CAST_FAILURE_REASON_TARGET_IN_COMBAT = 0x6C,
    CAST_FAILURE_REASON_TARGET_IS_PLAYER = 0x6D,
    CAST_FAILURE_REASON_TARGET_NOT_DEAD = 0x6E,
    CAST_FAILURE_REASON_TARGET_NOT_IN_PARTY = 0x6F,
    CAST_FAILURE_REASON_TARGET_NOT_LOOTED = 0x70,
    CAST_FAILURE_REASON_TARGET_NOT_PLAYER = 0x71,
    CAST_FAILURE_REASON_TARGET_NO_POCKETS = 0x72,
    CAST_FAILURE_REASON_TARGET_NO_WEAPONS = 0x73,
    CAST_FAILURE_REASON_TARGET_UNSKINNABLE = 0x74,
    CAST_FAILURE_REASON_THIRST_SATIATED = 0x75,
    CAST_FAILURE_REASON_TOO_CLOSE = 0x76,
    CAST_FAILURE_REASON_TOO_MANY_OF_ITEM = 0x77,
    CAST_FAILURE_REASON_TOTEMS = 0x78,
    CAST_FAILURE_REASON_TRAINING_POINTS = 0x79,
    CAST_FAILURE_REASON_TRY_AGAIN = 0x7A,
    CAST_FAILURE_REASON_UNIT_NOT_BEHIND = 0x7B,
    CAST_FAILURE_REASON_UNIT_NOT_INFRONT = 0x7C,
    CAST_FAILURE_REASON_WRONG_PET_FOOD = 0x7D,
    CAST_FAILURE_REASON_NOT_WHILE_FATIGUED = 0x7E,
    CAST_FAILURE_REASON_TARGET_NOT_IN_INSTANCE = 0x7F,
    CAST_FAILURE_REASON_NOT_WHILE_TRADING = 0x80,
    CAST_FAILURE_REASON_TARGET_NOT_IN_RAID = 0x81,
    CAST_FAILURE_REASON_DISENCHANT_WHILE_LOOTING = 0x82,
    CAST_FAILURE_REASON_PROSPECT_WHILE_LOOTING = 0x83,
    CAST_FAILURE_REASON_PROSPECT_NEED_MORE = 0x84,
    CAST_FAILURE_REASON_TARGET_FREEFORALL = 0x85,
    CAST_FAILURE_REASON_NO_EDIBLE_CORPSES = 0x86,
    CAST_FAILURE_REASON_ONLY_BATTLEGROUNDS = 0x87,
    CAST_FAILURE_REASON_TARGET_NOT_GHOST = 0x88,
    CAST_FAILURE_REASON_TOO_MANY_SKILLS = 0x89,
    CAST_FAILURE_REASON_TRANSFORM_UNUSABLE = 0x8A,
    CAST_FAILURE_REASON_WRONG_WEATHER = 0x8B,
    CAST_FAILURE_REASON_DAMAGE_IMMUNE = 0x8C,
    CAST_FAILURE_REASON_PREVENTED_BY_MECHANIC = 0x8D,
    CAST_FAILURE_REASON_PLAY_TIME = 0x8E,
    CAST_FAILURE_REASON_REPUTATION = 0x8F,
    CAST_FAILURE_REASON_MIN_SKILL = 0x90,
    CAST_FAILURE_REASON_UNKNOWN = 0x91,
} e_cast_failure_reason;
static const value_string e_cast_failure_reason_strings[] =  {
    { CAST_FAILURE_REASON_AFFECTING_COMBAT, "Affecting Combat" },
    { CAST_FAILURE_REASON_ALREADY_AT_FULL_HEALTH, "Already At Full Health" },
    { CAST_FAILURE_REASON_ALREADY_AT_FULL_POWER, "Already At Full Power" },
    { CAST_FAILURE_REASON_ALREADY_BEING_TAMED, "Already Being Tamed" },
    { CAST_FAILURE_REASON_ALREADY_HAVE_CHARM, "Already Have Charm" },
    { CAST_FAILURE_REASON_ALREADY_HAVE_SUMMON, "Already Have Summon" },
    { CAST_FAILURE_REASON_ALREADY_OPEN, "Already Open" },
    { CAST_FAILURE_REASON_AURA_BOUNCED, "Aura Bounced" },
    { CAST_FAILURE_REASON_AUTOTRACK_INTERRUPTED, "Autotrack Interrupted" },
    { CAST_FAILURE_REASON_BAD_IMPLICIT_TARGETS, "Bad Implicit Targets" },
    { CAST_FAILURE_REASON_BAD_TARGETS, "Bad Targets" },
    { CAST_FAILURE_REASON_CANT_BE_CHARMED, "Cant Be Charmed" },
    { CAST_FAILURE_REASON_CANT_BE_DISENCHANTED, "Cant Be Disenchanted" },
    { CAST_FAILURE_REASON_CANT_BE_PROSPECTED, "Cant Be Prospected" },
    { CAST_FAILURE_REASON_CANT_CAST_ON_TAPPED, "Cant Cast On Tapped" },
    { CAST_FAILURE_REASON_CANT_DUEL_WHILE_INVISIBLE, "Cant Duel While Invisible" },
    { CAST_FAILURE_REASON_CANT_DUEL_WHILE_STEALTHED, "Cant Duel While Stealthed" },
    { CAST_FAILURE_REASON_CANT_STEALTH, "Cant Stealth" },
    { CAST_FAILURE_REASON_CASTER_AURASTATE, "Caster Aurastate" },
    { CAST_FAILURE_REASON_CASTER_DEAD, "Caster Dead" },
    { CAST_FAILURE_REASON_CHARMED, "Charmed" },
    { CAST_FAILURE_REASON_CHEST_IN_USE, "Chest In Use" },
    { CAST_FAILURE_REASON_CONFUSED, "Confused" },
    { CAST_FAILURE_REASON_DONT_REPORT, "Dont Report" },
    { CAST_FAILURE_REASON_EQUIPPED_ITEM, "Equipped Item" },
    { CAST_FAILURE_REASON_EQUIPPED_ITEM_CLASS, "Equipped Item Class" },
    { CAST_FAILURE_REASON_EQUIPPED_ITEM_CLASS_MAINHAND, "Equipped Item Class Mainhand" },
    { CAST_FAILURE_REASON_EQUIPPED_ITEM_CLASS_OFFHAND, "Equipped Item Class Offhand" },
    { CAST_FAILURE_REASON_ERROR, "Error" },
    { CAST_FAILURE_REASON_FIZZLE, "Fizzle" },
    { CAST_FAILURE_REASON_FLEEING, "Fleeing" },
    { CAST_FAILURE_REASON_FOOD_LOWLEVEL, "Food Lowlevel" },
    { CAST_FAILURE_REASON_HIGHLEVEL, "Highlevel" },
    { CAST_FAILURE_REASON_HUNGER_SATIATED, "Hunger Satiated" },
    { CAST_FAILURE_REASON_IMMUNE, "Immune" },
    { CAST_FAILURE_REASON_INTERRUPTED, "Interrupted" },
    { CAST_FAILURE_REASON_INTERRUPTED_COMBAT, "Interrupted Combat" },
    { CAST_FAILURE_REASON_ITEM_ALREADY_ENCHANTED, "Item Already Enchanted" },
    { CAST_FAILURE_REASON_ITEM_GONE, "Item Gone" },
    { CAST_FAILURE_REASON_ITEM_NOT_FOUND, "Item Not Found" },
    { CAST_FAILURE_REASON_ITEM_NOT_READY, "Item Not Ready" },
    { CAST_FAILURE_REASON_LEVEL_REQUIREMENT, "Level Requirement" },
    { CAST_FAILURE_REASON_LINE_OF_SIGHT, "Line Of Sight" },
    { CAST_FAILURE_REASON_LOWLEVEL, "Lowlevel" },
    { CAST_FAILURE_REASON_LOW_CASTLEVEL, "Low Castlevel" },
    { CAST_FAILURE_REASON_MAINHAND_EMPTY, "Mainhand Empty" },
    { CAST_FAILURE_REASON_MOVING, "Moving" },
    { CAST_FAILURE_REASON_NEED_AMMO, "Need Ammo" },
    { CAST_FAILURE_REASON_NEED_AMMO_POUCH, "Need Ammo Pouch" },
    { CAST_FAILURE_REASON_NEED_EXOTIC_AMMO, "Need Exotic Ammo" },
    { CAST_FAILURE_REASON_NOPATH, "Nopath" },
    { CAST_FAILURE_REASON_NOT_BEHIND, "Not Behind" },
    { CAST_FAILURE_REASON_NOT_FISHABLE, "Not Fishable" },
    { CAST_FAILURE_REASON_NOT_HERE, "Not Here" },
    { CAST_FAILURE_REASON_NOT_INFRONT, "Not Infront" },
    { CAST_FAILURE_REASON_NOT_IN_CONTROL, "Not In Control" },
    { CAST_FAILURE_REASON_NOT_KNOWN, "Not Known" },
    { CAST_FAILURE_REASON_NOT_MOUNTED, "Not Mounted" },
    { CAST_FAILURE_REASON_NOT_ON_TAXI, "Not On Taxi" },
    { CAST_FAILURE_REASON_NOT_ON_TRANSPORT, "Not On Transport" },
    { CAST_FAILURE_REASON_NOT_READY, "Not Ready" },
    { CAST_FAILURE_REASON_NOT_SHAPESHIFT, "Not Shapeshift" },
    { CAST_FAILURE_REASON_NOT_STANDING, "Not Standing" },
    { CAST_FAILURE_REASON_NOT_TRADEABLE, "Not Tradeable" },
    { CAST_FAILURE_REASON_NOT_TRADING, "Not Trading" },
    { CAST_FAILURE_REASON_NOT_UNSHEATHED, "Not Unsheathed" },
    { CAST_FAILURE_REASON_NOT_WHILE_GHOST, "Not While Ghost" },
    { CAST_FAILURE_REASON_NO_AMMO, "No Ammo" },
    { CAST_FAILURE_REASON_NO_CHARGES_REMAIN, "No Charges Remain" },
    { CAST_FAILURE_REASON_NO_CHAMPION, "No Champion" },
    { CAST_FAILURE_REASON_NO_COMBO_POINTS, "No Combo Points" },
    { CAST_FAILURE_REASON_NO_DUELING, "No Dueling" },
    { CAST_FAILURE_REASON_NO_ENDURANCE, "No Endurance" },
    { CAST_FAILURE_REASON_NO_FISH, "No Fish" },
    { CAST_FAILURE_REASON_NO_ITEMS_WHILE_SHAPESHIFTED, "No Items While Shapeshifted" },
    { CAST_FAILURE_REASON_NO_MOUNTS_ALLOWED, "No Mounts Allowed" },
    { CAST_FAILURE_REASON_NO_PET, "No Pet" },
    { CAST_FAILURE_REASON_NO_POWER, "No Power" },
    { CAST_FAILURE_REASON_NOTHING_TO_DISPEL, "Nothing To Dispel" },
    { CAST_FAILURE_REASON_NOTHING_TO_STEAL, "Nothing To Steal" },
    { CAST_FAILURE_REASON_ONLY_ABOVEWATER, "Only Abovewater" },
    { CAST_FAILURE_REASON_ONLY_DAYTIME, "Only Daytime" },
    { CAST_FAILURE_REASON_ONLY_INDOORS, "Only Indoors" },
    { CAST_FAILURE_REASON_ONLY_MOUNTED, "Only Mounted" },
    { CAST_FAILURE_REASON_ONLY_NIGHTTIME, "Only Nighttime" },
    { CAST_FAILURE_REASON_ONLY_OUTDOORS, "Only Outdoors" },
    { CAST_FAILURE_REASON_ONLY_SHAPESHIFT, "Only Shapeshift" },
    { CAST_FAILURE_REASON_ONLY_STEALTHED, "Only Stealthed" },
    { CAST_FAILURE_REASON_ONLY_UNDERWATER, "Only Underwater" },
    { CAST_FAILURE_REASON_OUT_OF_RANGE, "Out Of Range" },
    { CAST_FAILURE_REASON_PACIFIED, "Pacified" },
    { CAST_FAILURE_REASON_POSSESSED, "Possessed" },
    { CAST_FAILURE_REASON_REAGENTS, "Reagents" },
    { CAST_FAILURE_REASON_REQUIRES_AREA, "Requires Area" },
    { CAST_FAILURE_REASON_REQUIRES_SPELL_FOCUS, "Requires Spell Focus" },
    { CAST_FAILURE_REASON_ROOTED, "Rooted" },
    { CAST_FAILURE_REASON_SILENCED, "Silenced" },
    { CAST_FAILURE_REASON_SPELL_IN_PROGRESS, "Spell In Progress" },
    { CAST_FAILURE_REASON_SPELL_LEARNED, "Spell Learned" },
    { CAST_FAILURE_REASON_SPELL_UNAVAILABLE, "Spell Unavailable" },
    { CAST_FAILURE_REASON_STUNNED, "Stunned" },
    { CAST_FAILURE_REASON_TARGETS_DEAD, "Targets Dead" },
    { CAST_FAILURE_REASON_TARGET_AFFECTING_COMBAT, "Target Affecting Combat" },
    { CAST_FAILURE_REASON_TARGET_AURASTATE, "Target Aurastate" },
    { CAST_FAILURE_REASON_TARGET_DUELING, "Target Dueling" },
    { CAST_FAILURE_REASON_TARGET_ENEMY, "Target Enemy" },
    { CAST_FAILURE_REASON_TARGET_ENRAGED, "Target Enraged" },
    { CAST_FAILURE_REASON_TARGET_FRIENDLY, "Target Friendly" },
    { CAST_FAILURE_REASON_TARGET_IN_COMBAT, "Target In Combat" },
    { CAST_FAILURE_REASON_TARGET_IS_PLAYER, "Target Is Player" },
    { CAST_FAILURE_REASON_TARGET_NOT_DEAD, "Target Not Dead" },
    { CAST_FAILURE_REASON_TARGET_NOT_IN_PARTY, "Target Not In Party" },
    { CAST_FAILURE_REASON_TARGET_NOT_LOOTED, "Target Not Looted" },
    { CAST_FAILURE_REASON_TARGET_NOT_PLAYER, "Target Not Player" },
    { CAST_FAILURE_REASON_TARGET_NO_POCKETS, "Target No Pockets" },
    { CAST_FAILURE_REASON_TARGET_NO_WEAPONS, "Target No Weapons" },
    { CAST_FAILURE_REASON_TARGET_UNSKINNABLE, "Target Unskinnable" },
    { CAST_FAILURE_REASON_THIRST_SATIATED, "Thirst Satiated" },
    { CAST_FAILURE_REASON_TOO_CLOSE, "Too Close" },
    { CAST_FAILURE_REASON_TOO_MANY_OF_ITEM, "Too Many Of Item" },
    { CAST_FAILURE_REASON_TOTEMS, "Totems" },
    { CAST_FAILURE_REASON_TRAINING_POINTS, "Training Points" },
    { CAST_FAILURE_REASON_TRY_AGAIN, "Try Again" },
    { CAST_FAILURE_REASON_UNIT_NOT_BEHIND, "Unit Not Behind" },
    { CAST_FAILURE_REASON_UNIT_NOT_INFRONT, "Unit Not Infront" },
    { CAST_FAILURE_REASON_WRONG_PET_FOOD, "Wrong Pet Food" },
    { CAST_FAILURE_REASON_NOT_WHILE_FATIGUED, "Not While Fatigued" },
    { CAST_FAILURE_REASON_TARGET_NOT_IN_INSTANCE, "Target Not In Instance" },
    { CAST_FAILURE_REASON_NOT_WHILE_TRADING, "Not While Trading" },
    { CAST_FAILURE_REASON_TARGET_NOT_IN_RAID, "Target Not In Raid" },
    { CAST_FAILURE_REASON_DISENCHANT_WHILE_LOOTING, "Disenchant While Looting" },
    { CAST_FAILURE_REASON_PROSPECT_WHILE_LOOTING, "Prospect While Looting" },
    { CAST_FAILURE_REASON_PROSPECT_NEED_MORE, "Prospect Need More" },
    { CAST_FAILURE_REASON_TARGET_FREEFORALL, "Target Freeforall" },
    { CAST_FAILURE_REASON_NO_EDIBLE_CORPSES, "No Edible Corpses" },
    { CAST_FAILURE_REASON_ONLY_BATTLEGROUNDS, "Only Battlegrounds" },
    { CAST_FAILURE_REASON_TARGET_NOT_GHOST, "Target Not Ghost" },
    { CAST_FAILURE_REASON_TOO_MANY_SKILLS, "Too Many Skills" },
    { CAST_FAILURE_REASON_TRANSFORM_UNUSABLE, "Transform Unusable" },
    { CAST_FAILURE_REASON_WRONG_WEATHER, "Wrong Weather" },
    { CAST_FAILURE_REASON_DAMAGE_IMMUNE, "Damage Immune" },
    { CAST_FAILURE_REASON_PREVENTED_BY_MECHANIC, "Prevented By Mechanic" },
    { CAST_FAILURE_REASON_PLAY_TIME, "Play Time" },
    { CAST_FAILURE_REASON_REPUTATION, "Reputation" },
    { CAST_FAILURE_REASON_MIN_SKILL, "Min Skill" },
    { CAST_FAILURE_REASON_UNKNOWN, "Unknown" },
    { 0, NULL }
};

typedef enum {
    CHAT_NOTIFY_JOINED_NOTICE = 0x00,
    CHAT_NOTIFY_LEFT_NOTICE = 0x01,
    CHAT_NOTIFY_YOU_JOINED_NOTICE = 0x02,
    CHAT_NOTIFY_YOU_LEFT_NOTICE = 0x03,
    CHAT_NOTIFY_WRONG_PASSWORD_NOTICE = 0x04,
    CHAT_NOTIFY_NOT_MEMBER_NOTICE = 0x05,
    CHAT_NOTIFY_NOT_MODERATOR_NOTICE = 0x06,
    CHAT_NOTIFY_PASSWORD_CHANGED_NOTICE = 0x07,
    CHAT_NOTIFY_OWNER_CHANGED_NOTICE = 0x08,
    CHAT_NOTIFY_PLAYER_NOT_FOUND_NOTICE = 0x09,
    CHAT_NOTIFY_NOT_OWNER_NOTICE = 0x0A,
    CHAT_NOTIFY_CHANNEL_OWNER_NOTICE = 0x0B,
    CHAT_NOTIFY_MODE_CHANGE_NOTICE = 0x0C,
    CHAT_NOTIFY_ANNOUNCEMENTS_ON_NOTICE = 0x0D,
    CHAT_NOTIFY_ANNOUNCEMENTS_OFF_NOTICE = 0x0E,
    CHAT_NOTIFY_MODERATION_ON_NOTICE = 0x0F,
    CHAT_NOTIFY_MODERATION_OFF_NOTICE = 0x10,
    CHAT_NOTIFY_MUTED_NOTICE = 0x11,
    CHAT_NOTIFY_PLAYER_KICKED_NOTICE = 0x12,
    CHAT_NOTIFY_BANNED_NOTICE = 0x13,
    CHAT_NOTIFY_PLAYER_BANNED_NOTICE = 0x14,
    CHAT_NOTIFY_PLAYER_UNBANNED_NOTICE = 0x15,
    CHAT_NOTIFY_PLAYER_NOT_BANNED_NOTICE = 0x16,
    CHAT_NOTIFY_PLAYER_ALREADY_MEMBER_NOTICE = 0x17,
    CHAT_NOTIFY_INVITE_NOTICE = 0x18,
    CHAT_NOTIFY_INVITE_WRONG_FACTION_NOTICE = 0x19,
    CHAT_NOTIFY_WRONG_FACTION_NOTICE = 0x1A,
    CHAT_NOTIFY_INVALID_NAME_NOTICE = 0x1B,
    CHAT_NOTIFY_NOT_MODERATED_NOTICE = 0x1C,
    CHAT_NOTIFY_PLAYER_INVITED_NOTICE = 0x1D,
    CHAT_NOTIFY_PLAYER_INVITE_BANNED_NOTICE = 0x1E,
    CHAT_NOTIFY_THROTTLED_NOTICE = 0x1F,
} e_chat_notify;
static const value_string e_chat_notify_strings[] =  {
    { CHAT_NOTIFY_JOINED_NOTICE, "Joined Notice" },
    { CHAT_NOTIFY_LEFT_NOTICE, "Left Notice" },
    { CHAT_NOTIFY_YOU_JOINED_NOTICE, "You Joined Notice" },
    { CHAT_NOTIFY_YOU_LEFT_NOTICE, "You Left Notice" },
    { CHAT_NOTIFY_WRONG_PASSWORD_NOTICE, "Wrong Password Notice" },
    { CHAT_NOTIFY_NOT_MEMBER_NOTICE, "Not Member Notice" },
    { CHAT_NOTIFY_NOT_MODERATOR_NOTICE, "Not Moderator Notice" },
    { CHAT_NOTIFY_PASSWORD_CHANGED_NOTICE, "Password Changed Notice" },
    { CHAT_NOTIFY_OWNER_CHANGED_NOTICE, "Owner Changed Notice" },
    { CHAT_NOTIFY_PLAYER_NOT_FOUND_NOTICE, "Player Not Found Notice" },
    { CHAT_NOTIFY_NOT_OWNER_NOTICE, "Not Owner Notice" },
    { CHAT_NOTIFY_CHANNEL_OWNER_NOTICE, "Channel Owner Notice" },
    { CHAT_NOTIFY_MODE_CHANGE_NOTICE, "Mode Change Notice" },
    { CHAT_NOTIFY_ANNOUNCEMENTS_ON_NOTICE, "Announcements On Notice" },
    { CHAT_NOTIFY_ANNOUNCEMENTS_OFF_NOTICE, "Announcements Off Notice" },
    { CHAT_NOTIFY_MODERATION_ON_NOTICE, "Moderation On Notice" },
    { CHAT_NOTIFY_MODERATION_OFF_NOTICE, "Moderation Off Notice" },
    { CHAT_NOTIFY_MUTED_NOTICE, "Muted Notice" },
    { CHAT_NOTIFY_PLAYER_KICKED_NOTICE, "Player Kicked Notice" },
    { CHAT_NOTIFY_BANNED_NOTICE, "Banned Notice" },
    { CHAT_NOTIFY_PLAYER_BANNED_NOTICE, "Player Banned Notice" },
    { CHAT_NOTIFY_PLAYER_UNBANNED_NOTICE, "Player Unbanned Notice" },
    { CHAT_NOTIFY_PLAYER_NOT_BANNED_NOTICE, "Player Not Banned Notice" },
    { CHAT_NOTIFY_PLAYER_ALREADY_MEMBER_NOTICE, "Player Already Member Notice" },
    { CHAT_NOTIFY_INVITE_NOTICE, "Invite Notice" },
    { CHAT_NOTIFY_INVITE_WRONG_FACTION_NOTICE, "Invite Wrong Faction Notice" },
    { CHAT_NOTIFY_WRONG_FACTION_NOTICE, "Wrong Faction Notice" },
    { CHAT_NOTIFY_INVALID_NAME_NOTICE, "Invalid Name Notice" },
    { CHAT_NOTIFY_NOT_MODERATED_NOTICE, "Not Moderated Notice" },
    { CHAT_NOTIFY_PLAYER_INVITED_NOTICE, "Player Invited Notice" },
    { CHAT_NOTIFY_PLAYER_INVITE_BANNED_NOTICE, "Player Invite Banned Notice" },
    { CHAT_NOTIFY_THROTTLED_NOTICE, "Throttled Notice" },
    { 0, NULL }
};

typedef enum {
    DISMOUNT_RESULT_NOT_MOUNTED = 0x1,
    DISMOUNT_RESULT_OK = 0x3,
} e_dismount_result;
static const value_string e_dismount_result_strings[] =  {
    { DISMOUNT_RESULT_NOT_MOUNTED, "Not Mounted" },
    { DISMOUNT_RESULT_OK, "Ok" },
    { 0, NULL }
};

typedef enum {
    DUEL_WINNER_REASON_WON = 0x0,
    DUEL_WINNER_REASON_FLED = 0x1,
} e_duel_winner_reason;
static const value_string e_duel_winner_reason_strings[] =  {
    { DUEL_WINNER_REASON_WON, "Won" },
    { DUEL_WINNER_REASON_FLED, "Fled" },
    { 0, NULL }
};

typedef enum {
    ENVIRONMENTAL_DAMAGE_TYPE_EXHAUSTED = 0x0,
    ENVIRONMENTAL_DAMAGE_TYPE_DROWNING = 0x1,
    ENVIRONMENTAL_DAMAGE_TYPE_FALL = 0x2,
    ENVIRONMENTAL_DAMAGE_TYPE_LAVA = 0x3,
    ENVIRONMENTAL_DAMAGE_TYPE_SLIME = 0x4,
    ENVIRONMENTAL_DAMAGE_TYPE_FIRE = 0x5,
} e_environmental_damage_type;
static const value_string e_environmental_damage_type_strings[] =  {
    { ENVIRONMENTAL_DAMAGE_TYPE_EXHAUSTED, "Exhausted" },
    { ENVIRONMENTAL_DAMAGE_TYPE_DROWNING, "Drowning" },
    { ENVIRONMENTAL_DAMAGE_TYPE_FALL, "Fall" },
    { ENVIRONMENTAL_DAMAGE_TYPE_LAVA, "Lava" },
    { ENVIRONMENTAL_DAMAGE_TYPE_SLIME, "Slime" },
    { ENVIRONMENTAL_DAMAGE_TYPE_FIRE, "Fire" },
    { 0, NULL }
};

typedef enum {
    FRIEND_RESULT_DB_ERROR = 0x00,
    FRIEND_RESULT_LIST_FULL = 0x01,
    FRIEND_RESULT_ONLINE = 0x02,
    FRIEND_RESULT_OFFLINE = 0x03,
    FRIEND_RESULT_NOT_FOUND = 0x04,
    FRIEND_RESULT_REMOVED = 0x05,
    FRIEND_RESULT_ADDED_ONLINE = 0x06,
    FRIEND_RESULT_ADDED_OFFLINE = 0x07,
    FRIEND_RESULT_ALREADY = 0x08,
    FRIEND_RESULT_SELF = 0x09,
    FRIEND_RESULT_ENEMY = 0x0A,
    FRIEND_RESULT_IGNORE_FULL = 0x0B,
    FRIEND_RESULT_IGNORE_SELF = 0x0C,
    FRIEND_RESULT_IGNORE_NOT_FOUND = 0x0D,
    FRIEND_RESULT_IGNORE_ALREADY = 0x0E,
    FRIEND_RESULT_IGNORE_ADDED = 0x0F,
    FRIEND_RESULT_IGNORE_REMOVED = 0x10,
    FRIEND_RESULT_IGNORE_AMBIGUOUS = 0x11,
    FRIEND_RESULT_MUTE_FULL = 0x12,
    FRIEND_RESULT_MUTE_SELF = 0x13,
    FRIEND_RESULT_MUTE_NOT_FOUND = 0x14,
    FRIEND_RESULT_MUTE_ALREADY = 0x15,
    FRIEND_RESULT_MUTE_ADDED = 0x16,
    FRIEND_RESULT_MUTE_REMOVED = 0x17,
    FRIEND_RESULT_MUTE_AMBIGUOUS = 0x18,
    FRIEND_RESULT_UNKNOWN19 = 0x19,
    FRIEND_RESULT_UNKNOWN20 = 0x1A,
} e_friend_result;
static const value_string e_friend_result_strings[] =  {
    { FRIEND_RESULT_DB_ERROR, "Db Error" },
    { FRIEND_RESULT_LIST_FULL, "List Full" },
    { FRIEND_RESULT_ONLINE, "Online" },
    { FRIEND_RESULT_OFFLINE, "Offline" },
    { FRIEND_RESULT_NOT_FOUND, "Not Found" },
    { FRIEND_RESULT_REMOVED, "Removed" },
    { FRIEND_RESULT_ADDED_ONLINE, "Added Online" },
    { FRIEND_RESULT_ADDED_OFFLINE, "Added Offline" },
    { FRIEND_RESULT_ALREADY, "Already" },
    { FRIEND_RESULT_SELF, "Self" },
    { FRIEND_RESULT_ENEMY, "Enemy" },
    { FRIEND_RESULT_IGNORE_FULL, "Ignore Full" },
    { FRIEND_RESULT_IGNORE_SELF, "Ignore Self" },
    { FRIEND_RESULT_IGNORE_NOT_FOUND, "Ignore Not Found" },
    { FRIEND_RESULT_IGNORE_ALREADY, "Ignore Already" },
    { FRIEND_RESULT_IGNORE_ADDED, "Ignore Added" },
    { FRIEND_RESULT_IGNORE_REMOVED, "Ignore Removed" },
    { FRIEND_RESULT_IGNORE_AMBIGUOUS, "Ignore Ambiguous" },
    { FRIEND_RESULT_MUTE_FULL, "Mute Full" },
    { FRIEND_RESULT_MUTE_SELF, "Mute Self" },
    { FRIEND_RESULT_MUTE_NOT_FOUND, "Mute Not Found" },
    { FRIEND_RESULT_MUTE_ALREADY, "Mute Already" },
    { FRIEND_RESULT_MUTE_ADDED, "Mute Added" },
    { FRIEND_RESULT_MUTE_REMOVED, "Mute Removed" },
    { FRIEND_RESULT_MUTE_AMBIGUOUS, "Mute Ambiguous" },
    { FRIEND_RESULT_UNKNOWN19, "Unknown19" },
    { FRIEND_RESULT_UNKNOWN20, "Unknown20" },
    { 0, NULL }
};

typedef enum {
    GM_TICKET_RESPONSE_NOT_EXIST = 0x0,
    GM_TICKET_RESPONSE_ALREADY_EXIST = 0x1,
    GM_TICKET_RESPONSE_CREATE_SUCCESS = 0x2,
    GM_TICKET_RESPONSE_CREATE_ERROR = 0x3,
    GM_TICKET_RESPONSE_UPDATE_SUCCESS = 0x4,
    GM_TICKET_RESPONSE_UPDATE_ERROR = 0x5,
    GM_TICKET_RESPONSE_TICKET_DELETED = 0x9,
} e_gm_ticket_response;
static const value_string e_gm_ticket_response_strings[] =  {
    { GM_TICKET_RESPONSE_NOT_EXIST, "Not Exist" },
    { GM_TICKET_RESPONSE_ALREADY_EXIST, "Already Exist" },
    { GM_TICKET_RESPONSE_CREATE_SUCCESS, "Create Success" },
    { GM_TICKET_RESPONSE_CREATE_ERROR, "Create Error" },
    { GM_TICKET_RESPONSE_UPDATE_SUCCESS, "Update Success" },
    { GM_TICKET_RESPONSE_UPDATE_ERROR, "Update Error" },
    { GM_TICKET_RESPONSE_TICKET_DELETED, "Ticket Deleted" },
    { 0, NULL }
};

typedef enum {
    GM_TICKET_STATUS_DB_ERROR = 0x0,
    GM_TICKET_STATUS_HAS_TEXT = 0x6,
    GM_TICKET_STATUS_DEFAULT = 0xA,
} e_gm_ticket_status;
static const value_string e_gm_ticket_status_strings[] =  {
    { GM_TICKET_STATUS_DB_ERROR, "Db Error" },
    { GM_TICKET_STATUS_HAS_TEXT, "Has Text" },
    { GM_TICKET_STATUS_DEFAULT, "Default" },
    { 0, NULL }
};

typedef enum {
    GM_TICKET_ESCALATION_STATUS_GMTICKET_ASSIGNEDTOGM_STATUS_NOT_ASSIGNED = 0x0,
    GM_TICKET_ESCALATION_STATUS_GMTICKET_ASSIGNEDTOGM_STATUS_ASSIGNED = 0x1,
    GM_TICKET_ESCALATION_STATUS_GMTICKET_ASSIGNEDTOGM_STATUS_ESCALATED = 0x2,
} e_gm_ticket_escalation_status;
static const value_string e_gm_ticket_escalation_status_strings[] =  {
    { GM_TICKET_ESCALATION_STATUS_GMTICKET_ASSIGNEDTOGM_STATUS_NOT_ASSIGNED, "Gmticket Assignedtogm Status Not Assigned" },
    { GM_TICKET_ESCALATION_STATUS_GMTICKET_ASSIGNEDTOGM_STATUS_ASSIGNED, "Gmticket Assignedtogm Status Assigned" },
    { GM_TICKET_ESCALATION_STATUS_GMTICKET_ASSIGNEDTOGM_STATUS_ESCALATED, "Gmticket Assignedtogm Status Escalated" },
    { 0, NULL }
};

typedef enum {
    GM_TICKET_QUEUE_STATUS_ENABLED = 0x1,
    GM_TICKET_QUEUE_STATUS_DISABLED = 0x0,
} e_gm_ticket_queue_status;
static const value_string e_gm_ticket_queue_status_strings[] =  {
    { GM_TICKET_QUEUE_STATUS_ENABLED, "Enabled" },
    { GM_TICKET_QUEUE_STATUS_DISABLED, "Disabled" },
    { 0, NULL }
};

typedef enum {
    GM_TICKET_STATUS_RESPONSE_UPDATED = 0x1,
    GM_TICKET_STATUS_RESPONSE_CLOSED = 0x2,
    GM_TICKET_STATUS_RESPONSE_SURVEY = 0x3,
} e_gm_ticket_status_response;
static const value_string e_gm_ticket_status_response_strings[] =  {
    { GM_TICKET_STATUS_RESPONSE_UPDATED, "Updated" },
    { GM_TICKET_STATUS_RESPONSE_CLOSED, "Closed" },
    { GM_TICKET_STATUS_RESPONSE_SURVEY, "Survey" },
    { 0, NULL }
};

typedef enum {
    BG_TYPE_ID_NOT_ELIGIBLE = 0x00000000,
    BG_TYPE_ID_QUEUED_FOR_AV = 0x00000001,
    BG_TYPE_ID_QUEUED_FOR_WSG = 0x00000002,
    BG_TYPE_ID_QUEUED_FOR_AB = 0x00000003,
    BG_TYPE_ID_REMOVE_FROM_QUEUE = -0x0000002,
} e_bg_type_id;
static const value_string e_bg_type_id_strings[] =  {
    { BG_TYPE_ID_NOT_ELIGIBLE, "Not Eligible" },
    { BG_TYPE_ID_QUEUED_FOR_AV, "Queued For Av" },
    { BG_TYPE_ID_QUEUED_FOR_WSG, "Queued For Wsg" },
    { BG_TYPE_ID_QUEUED_FOR_AB, "Queued For Ab" },
    { BG_TYPE_ID_REMOVE_FROM_QUEUE, "Remove From Queue" },
    { 0, NULL }
};

typedef enum {
    GROUP_TYPE_NORMAL = 0x0,
    GROUP_TYPE_RAID = 0x1,
} e_group_type;
static const value_string e_group_type_strings[] =  {
    { GROUP_TYPE_NORMAL, "Normal" },
    { GROUP_TYPE_RAID, "Raid" },
    { 0, NULL }
};

typedef enum {
    GUILD_COMMAND_CREATE = 0x00,
    GUILD_COMMAND_INVITE = 0x01,
    GUILD_COMMAND_QUIT = 0x03,
    GUILD_COMMAND_FOUNDER = 0x0E,
    GUILD_COMMAND_UNKNOWN19 = 0x13,
    GUILD_COMMAND_UNKNOWN20 = 0x14,
} e_guild_command;
static const value_string e_guild_command_strings[] =  {
    { GUILD_COMMAND_CREATE, "Create" },
    { GUILD_COMMAND_INVITE, "Invite" },
    { GUILD_COMMAND_QUIT, "Quit" },
    { GUILD_COMMAND_FOUNDER, "Founder" },
    { GUILD_COMMAND_UNKNOWN19, "Unknown19" },
    { GUILD_COMMAND_UNKNOWN20, "Unknown20" },
    { 0, NULL }
};

typedef enum {
    GUILD_COMMAND_RESULT_PLAYER_NO_MORE_IN_GUILD = 0x0,
    GUILD_COMMAND_RESULT_GUILD_INTERNAL = 0x1,
    GUILD_COMMAND_RESULT_ALREADY_IN_GUILD = 0x2,
    GUILD_COMMAND_RESULT_ALREADY_IN_GUILD_S = 0x3,
    GUILD_COMMAND_RESULT_INVITED_TO_GUILD = 0x4,
    GUILD_COMMAND_RESULT_ALREADY_INVITED_TO_GUILD_S = 0x5,
    GUILD_COMMAND_RESULT_GUILD_NAME_INVALID = 0x6,
    GUILD_COMMAND_RESULT_GUILD_NAME_EXISTS_S = 0x7,
    GUILD_COMMAND_RESULT_GUILD_PERMISSIONS_OR_LEADER = 0x8,
    GUILD_COMMAND_RESULT_GUILD_PLAYER_NOT_IN_GUILD = 0x9,
    GUILD_COMMAND_RESULT_GUILD_PLAYER_NOT_IN_GUILD_S = 0xA,
    GUILD_COMMAND_RESULT_GUILD_PLAYER_NOT_FOUND_S = 0xB,
    GUILD_COMMAND_RESULT_GUILD_NOT_ALLIED = 0xC,
    GUILD_COMMAND_RESULT_GUILD_RANK_TOO_HIGH_S = 0xD,
    GUILD_COMMAND_RESULT_GUILD_RANK_TOO_LOW_S = 0xE,
} e_guild_command_result;
static const value_string e_guild_command_result_strings[] =  {
    { GUILD_COMMAND_RESULT_PLAYER_NO_MORE_IN_GUILD, "Player No More In Guild" },
    { GUILD_COMMAND_RESULT_GUILD_INTERNAL, "Guild Internal" },
    { GUILD_COMMAND_RESULT_ALREADY_IN_GUILD, "Already In Guild" },
    { GUILD_COMMAND_RESULT_ALREADY_IN_GUILD_S, "Already In Guild S" },
    { GUILD_COMMAND_RESULT_INVITED_TO_GUILD, "Invited To Guild" },
    { GUILD_COMMAND_RESULT_ALREADY_INVITED_TO_GUILD_S, "Already Invited To Guild S" },
    { GUILD_COMMAND_RESULT_GUILD_NAME_INVALID, "Guild Name Invalid" },
    { GUILD_COMMAND_RESULT_GUILD_NAME_EXISTS_S, "Guild Name Exists S" },
    { GUILD_COMMAND_RESULT_GUILD_PERMISSIONS_OR_LEADER, "Guild Permissions Or Leader" },
    { GUILD_COMMAND_RESULT_GUILD_PLAYER_NOT_IN_GUILD, "Guild Player Not In Guild" },
    { GUILD_COMMAND_RESULT_GUILD_PLAYER_NOT_IN_GUILD_S, "Guild Player Not In Guild S" },
    { GUILD_COMMAND_RESULT_GUILD_PLAYER_NOT_FOUND_S, "Guild Player Not Found S" },
    { GUILD_COMMAND_RESULT_GUILD_NOT_ALLIED, "Guild Not Allied" },
    { GUILD_COMMAND_RESULT_GUILD_RANK_TOO_HIGH_S, "Guild Rank Too High S" },
    { GUILD_COMMAND_RESULT_GUILD_RANK_TOO_LOW_S, "Guild Rank Too Low S" },
    { 0, NULL }
};

typedef enum {
    GUILD_EVENT_PROMOTION = 0x0,
    GUILD_EVENT_DEMOTION = 0x1,
    GUILD_EVENT_MOTD = 0x2,
    GUILD_EVENT_JOINED = 0x3,
    GUILD_EVENT_LEFT = 0x4,
    GUILD_EVENT_REMOVED = 0x5,
    GUILD_EVENT_LEADER_IS = 0x6,
    GUILD_EVENT_LEADER_CHANGED = 0x7,
    GUILD_EVENT_DISBANDED = 0x8,
    GUILD_EVENT_TABARD_CHANGED = 0x9,
    GUILD_EVENT_UNKNOWN10 = 0xA,
    GUILD_EVENT_ROSTER_UPDATE = 0xB,
    GUILD_EVENT_SIGNED_ON = 0xC,
    GUILD_EVENT_SIGNED_OFF = 0xD,
} e_guild_event;
static const value_string e_guild_event_strings[] =  {
    { GUILD_EVENT_PROMOTION, "Promotion" },
    { GUILD_EVENT_DEMOTION, "Demotion" },
    { GUILD_EVENT_MOTD, "Motd" },
    { GUILD_EVENT_JOINED, "Joined" },
    { GUILD_EVENT_LEFT, "Left" },
    { GUILD_EVENT_REMOVED, "Removed" },
    { GUILD_EVENT_LEADER_IS, "Leader Is" },
    { GUILD_EVENT_LEADER_CHANGED, "Leader Changed" },
    { GUILD_EVENT_DISBANDED, "Disbanded" },
    { GUILD_EVENT_TABARD_CHANGED, "Tabard Changed" },
    { GUILD_EVENT_UNKNOWN10, "Unknown10" },
    { GUILD_EVENT_ROSTER_UPDATE, "Roster Update" },
    { GUILD_EVENT_SIGNED_ON, "Signed On" },
    { GUILD_EVENT_SIGNED_OFF, "Signed Off" },
    { 0, NULL }
};

typedef enum {
    INSTANCE_RESET_FAILED_REASON_GENERAL = 0x0,
    INSTANCE_RESET_FAILED_REASON_OFFLINE = 0x1,
    INSTANCE_RESET_FAILED_REASON_ZONING = 0x2,
    INSTANCE_RESET_FAILED_REASON_SILENTLY = 0x3,
} e_instance_reset_failed_reason;
static const value_string e_instance_reset_failed_reason_strings[] =  {
    { INSTANCE_RESET_FAILED_REASON_GENERAL, "General" },
    { INSTANCE_RESET_FAILED_REASON_OFFLINE, "Offline" },
    { INSTANCE_RESET_FAILED_REASON_ZONING, "Zoning" },
    { INSTANCE_RESET_FAILED_REASON_SILENTLY, "Silently" },
    { 0, NULL }
};

typedef enum {
    NEW_ITEM_SOURCE_LOOTED = 0x0,
    NEW_ITEM_SOURCE_FROM_NPC = 0x1,
} e_new_item_source;
static const value_string e_new_item_source_strings[] =  {
    { NEW_ITEM_SOURCE_LOOTED, "Looted" },
    { NEW_ITEM_SOURCE_FROM_NPC, "From Npc" },
    { 0, NULL }
};

typedef enum {
    NEW_ITEM_CREATION_TYPE_RECEIVED = 0x0,
    NEW_ITEM_CREATION_TYPE_CREATED = 0x1,
} e_new_item_creation_type;
static const value_string e_new_item_creation_type_strings[] =  {
    { NEW_ITEM_CREATION_TYPE_RECEIVED, "Received" },
    { NEW_ITEM_CREATION_TYPE_CREATED, "Created" },
    { 0, NULL }
};

typedef enum {
    NEW_ITEM_CHAT_ALERT_DO_NOT_SHOW = 0x0,
    NEW_ITEM_CHAT_ALERT_SHOW = 0x1,
} e_new_item_chat_alert;
static const value_string e_new_item_chat_alert_strings[] =  {
    { NEW_ITEM_CHAT_ALERT_DO_NOT_SHOW, "Do Not Show" },
    { NEW_ITEM_CHAT_ALERT_SHOW, "Show" },
    { 0, NULL }
};

typedef enum {
    ITEM_CLASS_AND_SUB_CLASS_CONSUMABLE = 0x0000000000,
    ITEM_CLASS_AND_SUB_CLASS_CHEESE_OR_BREAD_OBSOLETE = 0x0000000000,
    ITEM_CLASS_AND_SUB_CLASS_LIQUID_OBSOLETE = 0x0000000000,
    ITEM_CLASS_AND_SUB_CLASS_BAG = 0x0000000001,
    ITEM_CLASS_AND_SUB_CLASS_SOUL_BAG = 0x0000000001,
    ITEM_CLASS_AND_SUB_CLASS_HERB_BAG = 0x0000000001,
    ITEM_CLASS_AND_SUB_CLASS_ENCHANTING_BAG = 0x0000000001,
    ITEM_CLASS_AND_SUB_CLASS_ENGINEERING_BAG = 0x0000000001,
    ITEM_CLASS_AND_SUB_CLASS_ONE_HANDED_AXE = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_TWO_HANDED_AXE = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_BOW = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_GUN = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_ONE_HANDED_MACE = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_TWO_HANDED_MACE = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_POLEARM = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_ONE_HANDED_SWORD = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_TWO_HANDED_SWORD = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_OBSOLETE_WEAPON = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_STAFF = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_ONE_HANDED_EXOTIC = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_TWO_HANDED_EXOTIC = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_FIST_WEAPON = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_MISCELLANEOUS_WEAPON = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_DAGGER = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_THROWN = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_SPEAR = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_CROSSBOW = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_WAND = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_FISHING_POLE = 0x0000000002,
    ITEM_CLASS_AND_SUB_CLASS_JEWELRY_OBSOLETE = 0x0000000003,
    ITEM_CLASS_AND_SUB_CLASS_MISCELLANEOUS_ARMOR = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_CLOTH_ARMOR = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_LEATHER_ARMOR = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_MAIL_ARMOR = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_PLATE_ARMOR = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_BUCKLER_OBSOLETE = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_SHIELD = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_LIBRAM = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_IDOL = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_TOTEM = 0x0000000004,
    ITEM_CLASS_AND_SUB_CLASS_REAGENT = 0x0000000005,
    ITEM_CLASS_AND_SUB_CLASS_WAND_OBSOLETE = 0x0000000006,
    ITEM_CLASS_AND_SUB_CLASS_BOLT_OBSOLETE = 0x0000000006,
    ITEM_CLASS_AND_SUB_CLASS_ARROW = 0x0000000006,
    ITEM_CLASS_AND_SUB_CLASS_BULLET = 0x0000000006,
    ITEM_CLASS_AND_SUB_CLASS_THROWN_OBSOLETE = 0x0000000006,
    ITEM_CLASS_AND_SUB_CLASS_TRADE_GOOD = 0x0000000007,
    ITEM_CLASS_AND_SUB_CLASS_PART_TRADE_GOOD = 0x0000000007,
    ITEM_CLASS_AND_SUB_CLASS_EXPLOSIVE_TRADE_GOOD = 0x0000000007,
    ITEM_CLASS_AND_SUB_CLASS_DEVICE_TRADE_GOOD = 0x0000000007,
    ITEM_CLASS_AND_SUB_CLASS_GENERIC_OBSOLETE = 0x0000000008,
    ITEM_CLASS_AND_SUB_CLASS_BOOK = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_LEATHERWORKING_RECIPE = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_TAILORING_RECIPE = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_ENGINEERING_RECIPE = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_BLACKSMITHING_RECIPE = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_COOKING_RECIPE = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_ALCHEMY_RECIPE = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_FIRST_AID_RECIPE = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_ENCHANTING_RECIPE = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_FISHING_RECIPE = 0x0000000009,
    ITEM_CLASS_AND_SUB_CLASS_MONEY_OBSOLETE = 0x000000000A,
    ITEM_CLASS_AND_SUB_CLASS_QUIVER_OBSOLETE = 0x000000000B,
    ITEM_CLASS_AND_SUB_CLASS_QUIVER_OBSOLETE1 = 0x000000000B,
    ITEM_CLASS_AND_SUB_CLASS_QUIVER = 0x000000000B,
    ITEM_CLASS_AND_SUB_CLASS_AMMO_POUCH = 0x000000000B,
    ITEM_CLASS_AND_SUB_CLASS_QUEST = 0x000000000C,
    ITEM_CLASS_AND_SUB_CLASS_KEY = 0x000000000D,
    ITEM_CLASS_AND_SUB_CLASS_LOCKPICK = 0x000000000D,
    ITEM_CLASS_AND_SUB_CLASS_PERMANENT = 0x000000000E,
    ITEM_CLASS_AND_SUB_CLASS_JUNK = 0x000000000F,
} e_item_class_and_sub_class;
static const val64_string e_item_class_and_sub_class_strings[] =  {
    { ITEM_CLASS_AND_SUB_CLASS_CONSUMABLE, "Consumable" },
    { ITEM_CLASS_AND_SUB_CLASS_CHEESE_OR_BREAD_OBSOLETE, "Cheese Or Bread Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_LIQUID_OBSOLETE, "Liquid Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_BAG, "Bag" },
    { ITEM_CLASS_AND_SUB_CLASS_SOUL_BAG, "Soul Bag" },
    { ITEM_CLASS_AND_SUB_CLASS_HERB_BAG, "Herb Bag" },
    { ITEM_CLASS_AND_SUB_CLASS_ENCHANTING_BAG, "Enchanting Bag" },
    { ITEM_CLASS_AND_SUB_CLASS_ENGINEERING_BAG, "Engineering Bag" },
    { ITEM_CLASS_AND_SUB_CLASS_ONE_HANDED_AXE, "One Handed Axe" },
    { ITEM_CLASS_AND_SUB_CLASS_TWO_HANDED_AXE, "Two Handed Axe" },
    { ITEM_CLASS_AND_SUB_CLASS_BOW, "Bow" },
    { ITEM_CLASS_AND_SUB_CLASS_GUN, "Gun" },
    { ITEM_CLASS_AND_SUB_CLASS_ONE_HANDED_MACE, "One Handed Mace" },
    { ITEM_CLASS_AND_SUB_CLASS_TWO_HANDED_MACE, "Two Handed Mace" },
    { ITEM_CLASS_AND_SUB_CLASS_POLEARM, "Polearm" },
    { ITEM_CLASS_AND_SUB_CLASS_ONE_HANDED_SWORD, "One Handed Sword" },
    { ITEM_CLASS_AND_SUB_CLASS_TWO_HANDED_SWORD, "Two Handed Sword" },
    { ITEM_CLASS_AND_SUB_CLASS_OBSOLETE_WEAPON, "Obsolete Weapon" },
    { ITEM_CLASS_AND_SUB_CLASS_STAFF, "Staff" },
    { ITEM_CLASS_AND_SUB_CLASS_ONE_HANDED_EXOTIC, "One Handed Exotic" },
    { ITEM_CLASS_AND_SUB_CLASS_TWO_HANDED_EXOTIC, "Two Handed Exotic" },
    { ITEM_CLASS_AND_SUB_CLASS_FIST_WEAPON, "Fist Weapon" },
    { ITEM_CLASS_AND_SUB_CLASS_MISCELLANEOUS_WEAPON, "Miscellaneous Weapon" },
    { ITEM_CLASS_AND_SUB_CLASS_DAGGER, "Dagger" },
    { ITEM_CLASS_AND_SUB_CLASS_THROWN, "Thrown" },
    { ITEM_CLASS_AND_SUB_CLASS_SPEAR, "Spear" },
    { ITEM_CLASS_AND_SUB_CLASS_CROSSBOW, "Crossbow" },
    { ITEM_CLASS_AND_SUB_CLASS_WAND, "Wand" },
    { ITEM_CLASS_AND_SUB_CLASS_FISHING_POLE, "Fishing Pole" },
    { ITEM_CLASS_AND_SUB_CLASS_JEWELRY_OBSOLETE, "Jewelry Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_MISCELLANEOUS_ARMOR, "Miscellaneous Armor" },
    { ITEM_CLASS_AND_SUB_CLASS_CLOTH_ARMOR, "Cloth Armor" },
    { ITEM_CLASS_AND_SUB_CLASS_LEATHER_ARMOR, "Leather Armor" },
    { ITEM_CLASS_AND_SUB_CLASS_MAIL_ARMOR, "Mail Armor" },
    { ITEM_CLASS_AND_SUB_CLASS_PLATE_ARMOR, "Plate Armor" },
    { ITEM_CLASS_AND_SUB_CLASS_BUCKLER_OBSOLETE, "Buckler Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_SHIELD, "Shield" },
    { ITEM_CLASS_AND_SUB_CLASS_LIBRAM, "Libram" },
    { ITEM_CLASS_AND_SUB_CLASS_IDOL, "Idol" },
    { ITEM_CLASS_AND_SUB_CLASS_TOTEM, "Totem" },
    { ITEM_CLASS_AND_SUB_CLASS_REAGENT, "Reagent" },
    { ITEM_CLASS_AND_SUB_CLASS_WAND_OBSOLETE, "Wand Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_BOLT_OBSOLETE, "Bolt Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_ARROW, "Arrow" },
    { ITEM_CLASS_AND_SUB_CLASS_BULLET, "Bullet" },
    { ITEM_CLASS_AND_SUB_CLASS_THROWN_OBSOLETE, "Thrown Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_TRADE_GOOD, "Trade Good" },
    { ITEM_CLASS_AND_SUB_CLASS_PART_TRADE_GOOD, "Part Trade Good" },
    { ITEM_CLASS_AND_SUB_CLASS_EXPLOSIVE_TRADE_GOOD, "Explosive Trade Good" },
    { ITEM_CLASS_AND_SUB_CLASS_DEVICE_TRADE_GOOD, "Device Trade Good" },
    { ITEM_CLASS_AND_SUB_CLASS_GENERIC_OBSOLETE, "Generic Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_BOOK, "Book" },
    { ITEM_CLASS_AND_SUB_CLASS_LEATHERWORKING_RECIPE, "Leatherworking Recipe" },
    { ITEM_CLASS_AND_SUB_CLASS_TAILORING_RECIPE, "Tailoring Recipe" },
    { ITEM_CLASS_AND_SUB_CLASS_ENGINEERING_RECIPE, "Engineering Recipe" },
    { ITEM_CLASS_AND_SUB_CLASS_BLACKSMITHING_RECIPE, "Blacksmithing Recipe" },
    { ITEM_CLASS_AND_SUB_CLASS_COOKING_RECIPE, "Cooking Recipe" },
    { ITEM_CLASS_AND_SUB_CLASS_ALCHEMY_RECIPE, "Alchemy Recipe" },
    { ITEM_CLASS_AND_SUB_CLASS_FIRST_AID_RECIPE, "First Aid Recipe" },
    { ITEM_CLASS_AND_SUB_CLASS_ENCHANTING_RECIPE, "Enchanting Recipe" },
    { ITEM_CLASS_AND_SUB_CLASS_FISHING_RECIPE, "Fishing Recipe" },
    { ITEM_CLASS_AND_SUB_CLASS_MONEY_OBSOLETE, "Money Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_QUIVER_OBSOLETE, "Quiver Obsolete" },
    { ITEM_CLASS_AND_SUB_CLASS_QUIVER_OBSOLETE1, "Quiver Obsolete1" },
    { ITEM_CLASS_AND_SUB_CLASS_QUIVER, "Quiver" },
    { ITEM_CLASS_AND_SUB_CLASS_AMMO_POUCH, "Ammo Pouch" },
    { ITEM_CLASS_AND_SUB_CLASS_QUEST, "Quest" },
    { ITEM_CLASS_AND_SUB_CLASS_KEY, "Key" },
    { ITEM_CLASS_AND_SUB_CLASS_LOCKPICK, "Lockpick" },
    { ITEM_CLASS_AND_SUB_CLASS_PERMANENT, "Permanent" },
    { ITEM_CLASS_AND_SUB_CLASS_JUNK, "Junk" },
    { 0, NULL }
};

typedef enum {
    BONDING_NO_BIND = 0x0,
    BONDING_PICK_UP = 0x1,
    BONDING_EQUIP = 0x2,
    BONDING_USE = 0x3,
    BONDING_QUEST_ITEM = 0x4,
    BONDING_QUEST_ITEM1 = 0x5,
} e_bonding;
static const value_string e_bonding_strings[] =  {
    { BONDING_NO_BIND, "No Bind" },
    { BONDING_PICK_UP, "Pick Up" },
    { BONDING_EQUIP, "Equip" },
    { BONDING_USE, "Use" },
    { BONDING_QUEST_ITEM, "Quest Item" },
    { BONDING_QUEST_ITEM1, "Quest Item1" },
    { 0, NULL }
};

typedef enum {
    PAGE_TEXT_MATERIAL_NONE = 0x0,
    PAGE_TEXT_MATERIAL_PARCHMENT = 0x1,
    PAGE_TEXT_MATERIAL_STONE = 0x2,
    PAGE_TEXT_MATERIAL_MARBLE = 0x3,
    PAGE_TEXT_MATERIAL_SILVER = 0x4,
    PAGE_TEXT_MATERIAL_BRONZE = 0x5,
    PAGE_TEXT_MATERIAL_VALENTINE = 0x6,
} e_page_text_material;
static const value_string e_page_text_material_strings[] =  {
    { PAGE_TEXT_MATERIAL_NONE, "None" },
    { PAGE_TEXT_MATERIAL_PARCHMENT, "Parchment" },
    { PAGE_TEXT_MATERIAL_STONE, "Stone" },
    { PAGE_TEXT_MATERIAL_MARBLE, "Marble" },
    { PAGE_TEXT_MATERIAL_SILVER, "Silver" },
    { PAGE_TEXT_MATERIAL_BRONZE, "Bronze" },
    { PAGE_TEXT_MATERIAL_VALENTINE, "Valentine" },
    { 0, NULL }
};

typedef enum {
    SHEATHE_TYPE_NONE = 0x0,
    SHEATHE_TYPE_MAIN_HAND = 0x1,
    SHEATHE_TYPE_OFF_HAND = 0x2,
    SHEATHE_TYPE_LARGE_WEAPON_LEFT = 0x3,
    SHEATHE_TYPE_LARGE_WEAPON_RIGHT = 0x4,
    SHEATHE_TYPE_HIP_WEAPON_LEFT = 0x5,
    SHEATHE_TYPE_HIP_WEAPON_RIGHT = 0x6,
    SHEATHE_TYPE_SHIELD = 0x7,
} e_sheathe_type;
static const value_string e_sheathe_type_strings[] =  {
    { SHEATHE_TYPE_NONE, "None" },
    { SHEATHE_TYPE_MAIN_HAND, "Main Hand" },
    { SHEATHE_TYPE_OFF_HAND, "Off Hand" },
    { SHEATHE_TYPE_LARGE_WEAPON_LEFT, "Large Weapon Left" },
    { SHEATHE_TYPE_LARGE_WEAPON_RIGHT, "Large Weapon Right" },
    { SHEATHE_TYPE_HIP_WEAPON_LEFT, "Hip Weapon Left" },
    { SHEATHE_TYPE_HIP_WEAPON_RIGHT, "Hip Weapon Right" },
    { SHEATHE_TYPE_SHIELD, "Shield" },
    { 0, NULL }
};

typedef enum {
    ITEM_SET_NONE = 0x000,
    ITEM_SET_THE_GLADIATOR = 0x001,
    ITEM_SET_DAL_RENDS_ARMS = 0x029,
    ITEM_SET_SPIDERS_KISS = 0x041,
    ITEM_SET_THE_POSTMASTER = 0x051,
    ITEM_SET_CADAVEROUS_GARB = 0x079,
    ITEM_SET_NECROPILE_RAIMENT = 0x07A,
    ITEM_SET_BLOODMAIL_REGALIA = 0x07B,
    ITEM_SET_DEATHBONE_GUARDIAN = 0x07C,
    ITEM_SET_VOLCANIC_ARMOR = 0x08D,
    ITEM_SET_STORMSHROUD_ARMOR = 0x08E,
    ITEM_SET_DEVILSAUR_ARMOR = 0x08F,
    ITEM_SET_IRONFEATHER_ARMOR = 0x090,
    ITEM_SET_DEFIAS_LEATHER = 0x0A1,
    ITEM_SET_EMBRACE_OF_THE_VIPER = 0x0A2,
    ITEM_SET_CHAIN_OF_THE_SCARLET_CRUSADE = 0x0A3,
    ITEM_SET_MAGISTERS_REGALIA = 0x0B5,
    ITEM_SET_VESTMENTS_OF_THE_DEVOUT = 0x0B6,
    ITEM_SET_DREADMIST_RAIMENT = 0x0B7,
    ITEM_SET_SHADOWCRAFT_ARMOR = 0x0B8,
    ITEM_SET_WILDHEART_RAIMENT = 0x0B9,
    ITEM_SET_BEASTSTALKER_ARMOR = 0x0BA,
    ITEM_SET_THE_ELEMENTS = 0x0BB,
    ITEM_SET_LIGHTFORGE_ARMOR = 0x0BC,
    ITEM_SET_BATTLEGEAR_OF_VALOR = 0x0BD,
    ITEM_SET_ARCANIST_REGALIA = 0x0C9,
    ITEM_SET_VESTMENTS_OF_PROPHECY = 0x0CA,
    ITEM_SET_FELHEART_RAIMENT = 0x0CB,
    ITEM_SET_NIGHTSLAYER_ARMOR = 0x0CC,
    ITEM_SET_CENARION_RAIMENT = 0x0CD,
    ITEM_SET_GIANTSTALKER_ARMOR = 0x0CE,
    ITEM_SET_THE_EARTHFURY = 0x0CF,
    ITEM_SET_LAWBRINGER_ARMOR = 0x0D0,
    ITEM_SET_BATTLEGEAR_OF_MIGHT = 0x0D1,
    ITEM_SET_NETHERWIND_REGALIA = 0x0D2,
    ITEM_SET_VESTMENTS_OF_TRANSCENDENCE = 0x0D3,
    ITEM_SET_NEMESIS_RAIMENT = 0x0D4,
    ITEM_SET_BLOODFANG_ARMOR = 0x0D5,
    ITEM_SET_STORMRAGE_RAIMENT = 0x0D6,
    ITEM_SET_DRAGONSTALKER_ARMOR = 0x0D7,
    ITEM_SET_THE_TEN_STORMS = 0x0D8,
    ITEM_SET_JUDGEMENT_ARMOR = 0x0D9,
    ITEM_SET_BATTLEGEAR_OF_WRATH = 0x0DA,
    ITEM_SET_GARB_OF_THERO_SHAN = 0x0DD,
    ITEM_SET_SHARD_OF_THE_GODS = 0x0F1,
    ITEM_SET_SPIRIT_OF_ESKHANDAR = 0x105,
    ITEM_SET_CHAMPIONS_BATTLEGEAR = 0x119,
    ITEM_SET_LIEUTENANT_COMMANDERS_BATTLEGEAR = 0x11A,
    ITEM_SET_CHAMPIONS_EARTHSHAKER = 0x12D,
    ITEM_SET_IMPERIAL_PLATE = 0x141,
    ITEM_SET_CHAMPIONS_REGALIA = 0x155,
    ITEM_SET_CHAMPIONS_RAIMENT = 0x156,
    ITEM_SET_LIEUTENANT_COMMANDERS_REGALIA = 0x157,
    ITEM_SET_LIEUTENANT_COMMANDERS_RAIMENT = 0x158,
    ITEM_SET_CHAMPIONS_THREADS = 0x159,
    ITEM_SET_LIEUTENANT_COMMANDERS_THREADS = 0x15A,
    ITEM_SET_CHAMPIONS_VESTMENTS = 0x15B,
    ITEM_SET_LIEUTENANT_COMMANDERS_VESTMENTS = 0x15C,
    ITEM_SET_CHAMPIONS_PURSUIT = 0x169,
    ITEM_SET_LIEUTENANT_COMMANDERS_PURSUIT = 0x16A,
    ITEM_SET_LIEUTENANT_COMMANDERS_SANCTUARY = 0x17D,
    ITEM_SET_CHAMPIONS_SANCTUARY = 0x17E,
    ITEM_SET_WARLORDS_BATTLEGEAR = 0x17F,
    ITEM_SET_FIELD_MARSHALS_BATTLEGEAR = 0x180,
    ITEM_SET_WARLORDS_EARTHSHAKER = 0x182,
    ITEM_SET_WARLORDS_REGALIA = 0x183,
    ITEM_SET_FIELD_MARSHALS_REGALIA = 0x184,
    ITEM_SET_FIELD_MARSHALS_RAIMENT = 0x185,
    ITEM_SET_WARLORDS_RAIMENT = 0x186,
    ITEM_SET_WARLORDS_THREADS = 0x187,
    ITEM_SET_FIELD_MARSHALS_THREADS = 0x188,
    ITEM_SET_WARLORDS_VESTMENTS = 0x189,
    ITEM_SET_FIELD_MARSHALS_VESTMENTS = 0x18A,
    ITEM_SET_FIELD_MARSHALS_PURSUIT = 0x18B,
    ITEM_SET_WARLORDS_PURSUIT = 0x18C,
    ITEM_SET_FIELD_MARSHALS_SANCTUARY = 0x18D,
    ITEM_SET_WARLORDS_SANCTUARY = 0x18E,
    ITEM_SET_LIEUTENANT_COMMANDERS_AEGIS = 0x191,
    ITEM_SET_FIELD_MARSHALS_AEGIS = 0x192,
    ITEM_SET_BLOODVINE_GARB = 0x1A5,
    ITEM_SET_PRIMAL_BATSKIN = 0x1B9,
    ITEM_SET_BLOOD_TIGER_HARNESS = 0x1BA,
    ITEM_SET_BLOODSOUL_EMBRACE = 0x1BB,
    ITEM_SET_THE_DARKSOUL = 0x1BC,
    ITEM_SET_THE_TWIN_BLADES_OF_HAKKARI = 0x1CD,
    ITEM_SET_ZANZILS_CONCENTRATION = 0x1CE,
    ITEM_SET_PRIMAL_BLESSING = 0x1CF,
    ITEM_SET_OVERLORDS_RESOLUTION = 0x1D0,
    ITEM_SET_PRAYER_OF_THE_PRIMAL = 0x1D1,
    ITEM_SET_MAJOR_MOJO_INFUSION = 0x1D2,
    ITEM_SET_THE_HIGHLANDERS_RESOLUTION = 0x1D3,
    ITEM_SET_THE_HIGHLANDERS_RESOLVE = 0x1D4,
    ITEM_SET_THE_HIGHLANDERS_DETERMINATION = 0x1D5,
    ITEM_SET_THE_HIGHLANDERS_FORTITUDE = 0x1D6,
    ITEM_SET_THE_HIGHLANDERS_PURPOSE = 0x1D7,
    ITEM_SET_THE_HIGHLANDERS_WILL = 0x1D8,
    ITEM_SET_THE_HIGHLANDERS_INTENT = 0x1D9,
    ITEM_SET_VINDICATORS_BATTLEGEAR = 0x1DA,
    ITEM_SET_FREETHINKERS_ARMOR = 0x1DB,
    ITEM_SET_AUGURS_REGALIA = 0x1DC,
    ITEM_SET_PREDATORS_ARMOR = 0x1DD,
    ITEM_SET_MADCAPS_OUTFIT = 0x1DE,
    ITEM_SET_HARUSPEXS_GARB = 0x1DF,
    ITEM_SET_CONFESSORS_RAIMENT = 0x1E0,
    ITEM_SET_DEMONIACS_THREADS = 0x1E1,
    ITEM_SET_ILLUSIONISTS_ATTIRE = 0x1E2,
    ITEM_SET_THE_DEFILERS_DETERMINATION = 0x1E3,
    ITEM_SET_THE_DEFILERS_FORTITUDE = 0x1E4,
    ITEM_SET_THE_DEFILERS_INTENT = 0x1E5,
    ITEM_SET_THE_DEFILERS_PURPOSE = 0x1E6,
    ITEM_SET_THE_DEFILERS_RESOLUTION = 0x1E7,
    ITEM_SET_THE_DEFILERS_WILL = 0x1E8,
    ITEM_SET_BLACK_DRAGON_MAIL = 0x1E9,
    ITEM_SET_GREEN_DRAGON_MAIL = 0x1EA,
    ITEM_SET_BLUE_DRAGON_MAIL = 0x1EB,
    ITEM_SET_TWILIGHT_TRAPPINGS = 0x1EC,
    ITEM_SET_GENESIS_RAIMENT = 0x1ED,
    ITEM_SET_SYMBOLS_OF_UNENDING_LIFE = 0x1EE,
    ITEM_SET_BATTLEGEAR_OF_UNYIELDING_STRENGTH = 0x1EF,
    ITEM_SET_CONQUERORS_BATTLEGEAR = 0x1F0,
    ITEM_SET_DEATHDEALERS_EMBRACE = 0x1F1,
    ITEM_SET_EMBLEMS_OF_VEILED_SHADOWS = 0x1F2,
    ITEM_SET_DOOMCALLERS_ATTIRE = 0x1F3,
    ITEM_SET_IMPLEMENTS_OF_UNSPOKEN_NAMES = 0x1F4,
    ITEM_SET_STORMCALLERS_GARB = 0x1F5,
    ITEM_SET_GIFT_OF_THE_GATHERING_STORM = 0x1F6,
    ITEM_SET_ENIGMA_VESTMENTS = 0x1F7,
    ITEM_SET_TRAPPINGS_OF_VAULTED_SECRETS = 0x1F8,
    ITEM_SET_AVENGERS_BATTLEGEAR = 0x1F9,
    ITEM_SET_BATTLEGEAR_OF_ETERNAL_JUSTICE = 0x1FA,
    ITEM_SET_GARMENTS_OF_THE_ORACLE = 0x1FB,
    ITEM_SET_FINERY_OF_INFINITE_WISDOM = 0x1FC,
    ITEM_SET_STRIKERS_GARB = 0x1FD,
    ITEM_SET_TRAPPINGS_OF_THE_UNSEEN_PATH = 0x1FE,
    ITEM_SET_BATTLEGEAR_OF_HEROISM = 0x1FF,
    ITEM_SET_DARKMANTLE_ARMOR = 0x200,
    ITEM_SET_FERALHEART_RAIMENT = 0x201,
    ITEM_SET_VESTMENTS_OF_THE_VIRTUOUS = 0x202,
    ITEM_SET_BEASTMASTER_ARMOR = 0x203,
    ITEM_SET_SOULFORGE_ARMOR = 0x204,
    ITEM_SET_SORCERERS_REGALIA = 0x205,
    ITEM_SET_DEATHMIST_RAIMENT = 0x206,
    ITEM_SET_THE_FIVE_THUNDERS = 0x207,
    ITEM_SET_IRONWEAVE_BATTLESUIT = 0x208,
    ITEM_SET_DREAMWALKER_RAIMENT = 0x209,
    ITEM_SET_CHAMPIONS_GUARD = 0x20A,
    ITEM_SET_DREADNAUGHTS_BATTLEGEAR = 0x20B,
    ITEM_SET_BONESCYTHE_ARMOR = 0x20C,
    ITEM_SET_VESTMENTS_OF_FAITH = 0x20D,
    ITEM_SET_FROSTFIRE_REGALIA = 0x20E,
    ITEM_SET_THE_EARTHSHATTERER = 0x20F,
    ITEM_SET_REDEMPTION_ARMOR = 0x210,
    ITEM_SET_PLAGUEHEART_RAIMENT = 0x211,
    ITEM_SET_CRYPTSTALKER_ARMOR = 0x212,
    ITEM_SET_BATTLEGEAR_OF_UNDEAD_SLAYING = 0x215,
    ITEM_SET_UNDEAD_SLAYERS_ARMOR = 0x216,
    ITEM_SET_GARB_OF_THE_UNDEAD_SLAYER = 0x217,
    ITEM_SET_REGALIA_OF_UNDEAD_CLEANSING = 0x218,
    ITEM_SET_CHAMPIONS_BATTLEARMOR = 0x219,
    ITEM_SET_CHAMPIONS_STORMCALLER = 0x21A,
    ITEM_SET_CHAMPIONS_REFUGE = 0x21B,
    ITEM_SET_CHAMPIONS_INVESTITURE = 0x21C,
    ITEM_SET_CHAMPIONS_DREADGEAR = 0x21D,
    ITEM_SET_CHAMPIONS_ARCANUM = 0x21E,
    ITEM_SET_CHAMPIONS_PURSUANCE = 0x21F,
    ITEM_SET_LIEUTENANT_COMMANDERS_REDOUBT = 0x220,
    ITEM_SET_LIEUTENANT_COMMANDERS_BATTLEARMOR = 0x221,
    ITEM_SET_LIEUTENANT_COMMANDERS_ARCANUM = 0x222,
    ITEM_SET_LIEUTENANT_COMMANDERS_DREADGEAR = 0x223,
    ITEM_SET_LIEUTENANT_COMMANDERS_GUARD = 0x224,
    ITEM_SET_LIEUTENANT_COMMANDERS_INVESTITURE = 0x225,
    ITEM_SET_LIEUTENANT_COMMANDERS_PURSUANCE = 0x226,
    ITEM_SET_LIEUTENANT_COMMANDERS_REFUGE = 0x227,
} e_item_set;
static const value_string e_item_set_strings[] =  {
    { ITEM_SET_NONE, "None" },
    { ITEM_SET_THE_GLADIATOR, "The Gladiator" },
    { ITEM_SET_DAL_RENDS_ARMS, "Dal Rends Arms" },
    { ITEM_SET_SPIDERS_KISS, "Spiders Kiss" },
    { ITEM_SET_THE_POSTMASTER, "The Postmaster" },
    { ITEM_SET_CADAVEROUS_GARB, "Cadaverous Garb" },
    { ITEM_SET_NECROPILE_RAIMENT, "Necropile Raiment" },
    { ITEM_SET_BLOODMAIL_REGALIA, "Bloodmail Regalia" },
    { ITEM_SET_DEATHBONE_GUARDIAN, "Deathbone Guardian" },
    { ITEM_SET_VOLCANIC_ARMOR, "Volcanic Armor" },
    { ITEM_SET_STORMSHROUD_ARMOR, "Stormshroud Armor" },
    { ITEM_SET_DEVILSAUR_ARMOR, "Devilsaur Armor" },
    { ITEM_SET_IRONFEATHER_ARMOR, "Ironfeather Armor" },
    { ITEM_SET_DEFIAS_LEATHER, "Defias Leather" },
    { ITEM_SET_EMBRACE_OF_THE_VIPER, "Embrace Of The Viper" },
    { ITEM_SET_CHAIN_OF_THE_SCARLET_CRUSADE, "Chain Of The Scarlet Crusade" },
    { ITEM_SET_MAGISTERS_REGALIA, "Magisters Regalia" },
    { ITEM_SET_VESTMENTS_OF_THE_DEVOUT, "Vestments Of The Devout" },
    { ITEM_SET_DREADMIST_RAIMENT, "Dreadmist Raiment" },
    { ITEM_SET_SHADOWCRAFT_ARMOR, "Shadowcraft Armor" },
    { ITEM_SET_WILDHEART_RAIMENT, "Wildheart Raiment" },
    { ITEM_SET_BEASTSTALKER_ARMOR, "Beaststalker Armor" },
    { ITEM_SET_THE_ELEMENTS, "The Elements" },
    { ITEM_SET_LIGHTFORGE_ARMOR, "Lightforge Armor" },
    { ITEM_SET_BATTLEGEAR_OF_VALOR, "Battlegear Of Valor" },
    { ITEM_SET_ARCANIST_REGALIA, "Arcanist Regalia" },
    { ITEM_SET_VESTMENTS_OF_PROPHECY, "Vestments Of Prophecy" },
    { ITEM_SET_FELHEART_RAIMENT, "Felheart Raiment" },
    { ITEM_SET_NIGHTSLAYER_ARMOR, "Nightslayer Armor" },
    { ITEM_SET_CENARION_RAIMENT, "Cenarion Raiment" },
    { ITEM_SET_GIANTSTALKER_ARMOR, "Giantstalker Armor" },
    { ITEM_SET_THE_EARTHFURY, "The Earthfury" },
    { ITEM_SET_LAWBRINGER_ARMOR, "Lawbringer Armor" },
    { ITEM_SET_BATTLEGEAR_OF_MIGHT, "Battlegear Of Might" },
    { ITEM_SET_NETHERWIND_REGALIA, "Netherwind Regalia" },
    { ITEM_SET_VESTMENTS_OF_TRANSCENDENCE, "Vestments Of Transcendence" },
    { ITEM_SET_NEMESIS_RAIMENT, "Nemesis Raiment" },
    { ITEM_SET_BLOODFANG_ARMOR, "Bloodfang Armor" },
    { ITEM_SET_STORMRAGE_RAIMENT, "Stormrage Raiment" },
    { ITEM_SET_DRAGONSTALKER_ARMOR, "Dragonstalker Armor" },
    { ITEM_SET_THE_TEN_STORMS, "The Ten Storms" },
    { ITEM_SET_JUDGEMENT_ARMOR, "Judgement Armor" },
    { ITEM_SET_BATTLEGEAR_OF_WRATH, "Battlegear Of Wrath" },
    { ITEM_SET_GARB_OF_THERO_SHAN, "Garb Of Thero Shan" },
    { ITEM_SET_SHARD_OF_THE_GODS, "Shard Of The Gods" },
    { ITEM_SET_SPIRIT_OF_ESKHANDAR, "Spirit Of Eskhandar" },
    { ITEM_SET_CHAMPIONS_BATTLEGEAR, "Champions Battlegear" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_BATTLEGEAR, "Lieutenant Commanders Battlegear" },
    { ITEM_SET_CHAMPIONS_EARTHSHAKER, "Champions Earthshaker" },
    { ITEM_SET_IMPERIAL_PLATE, "Imperial Plate" },
    { ITEM_SET_CHAMPIONS_REGALIA, "Champions Regalia" },
    { ITEM_SET_CHAMPIONS_RAIMENT, "Champions Raiment" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_REGALIA, "Lieutenant Commanders Regalia" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_RAIMENT, "Lieutenant Commanders Raiment" },
    { ITEM_SET_CHAMPIONS_THREADS, "Champions Threads" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_THREADS, "Lieutenant Commanders Threads" },
    { ITEM_SET_CHAMPIONS_VESTMENTS, "Champions Vestments" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_VESTMENTS, "Lieutenant Commanders Vestments" },
    { ITEM_SET_CHAMPIONS_PURSUIT, "Champions Pursuit" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_PURSUIT, "Lieutenant Commanders Pursuit" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_SANCTUARY, "Lieutenant Commanders Sanctuary" },
    { ITEM_SET_CHAMPIONS_SANCTUARY, "Champions Sanctuary" },
    { ITEM_SET_WARLORDS_BATTLEGEAR, "Warlords Battlegear" },
    { ITEM_SET_FIELD_MARSHALS_BATTLEGEAR, "Field Marshals Battlegear" },
    { ITEM_SET_WARLORDS_EARTHSHAKER, "Warlords Earthshaker" },
    { ITEM_SET_WARLORDS_REGALIA, "Warlords Regalia" },
    { ITEM_SET_FIELD_MARSHALS_REGALIA, "Field Marshals Regalia" },
    { ITEM_SET_FIELD_MARSHALS_RAIMENT, "Field Marshals Raiment" },
    { ITEM_SET_WARLORDS_RAIMENT, "Warlords Raiment" },
    { ITEM_SET_WARLORDS_THREADS, "Warlords Threads" },
    { ITEM_SET_FIELD_MARSHALS_THREADS, "Field Marshals Threads" },
    { ITEM_SET_WARLORDS_VESTMENTS, "Warlords Vestments" },
    { ITEM_SET_FIELD_MARSHALS_VESTMENTS, "Field Marshals Vestments" },
    { ITEM_SET_FIELD_MARSHALS_PURSUIT, "Field Marshals Pursuit" },
    { ITEM_SET_WARLORDS_PURSUIT, "Warlords Pursuit" },
    { ITEM_SET_FIELD_MARSHALS_SANCTUARY, "Field Marshals Sanctuary" },
    { ITEM_SET_WARLORDS_SANCTUARY, "Warlords Sanctuary" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_AEGIS, "Lieutenant Commanders Aegis" },
    { ITEM_SET_FIELD_MARSHALS_AEGIS, "Field Marshals Aegis" },
    { ITEM_SET_BLOODVINE_GARB, "Bloodvine Garb" },
    { ITEM_SET_PRIMAL_BATSKIN, "Primal Batskin" },
    { ITEM_SET_BLOOD_TIGER_HARNESS, "Blood Tiger Harness" },
    { ITEM_SET_BLOODSOUL_EMBRACE, "Bloodsoul Embrace" },
    { ITEM_SET_THE_DARKSOUL, "The Darksoul" },
    { ITEM_SET_THE_TWIN_BLADES_OF_HAKKARI, "The Twin Blades Of Hakkari" },
    { ITEM_SET_ZANZILS_CONCENTRATION, "Zanzils Concentration" },
    { ITEM_SET_PRIMAL_BLESSING, "Primal Blessing" },
    { ITEM_SET_OVERLORDS_RESOLUTION, "Overlords Resolution" },
    { ITEM_SET_PRAYER_OF_THE_PRIMAL, "Prayer Of The Primal" },
    { ITEM_SET_MAJOR_MOJO_INFUSION, "Major Mojo Infusion" },
    { ITEM_SET_THE_HIGHLANDERS_RESOLUTION, "The Highlanders Resolution" },
    { ITEM_SET_THE_HIGHLANDERS_RESOLVE, "The Highlanders Resolve" },
    { ITEM_SET_THE_HIGHLANDERS_DETERMINATION, "The Highlanders Determination" },
    { ITEM_SET_THE_HIGHLANDERS_FORTITUDE, "The Highlanders Fortitude" },
    { ITEM_SET_THE_HIGHLANDERS_PURPOSE, "The Highlanders Purpose" },
    { ITEM_SET_THE_HIGHLANDERS_WILL, "The Highlanders Will" },
    { ITEM_SET_THE_HIGHLANDERS_INTENT, "The Highlanders Intent" },
    { ITEM_SET_VINDICATORS_BATTLEGEAR, "Vindicators Battlegear" },
    { ITEM_SET_FREETHINKERS_ARMOR, "Freethinkers Armor" },
    { ITEM_SET_AUGURS_REGALIA, "Augurs Regalia" },
    { ITEM_SET_PREDATORS_ARMOR, "Predators Armor" },
    { ITEM_SET_MADCAPS_OUTFIT, "Madcaps Outfit" },
    { ITEM_SET_HARUSPEXS_GARB, "Haruspexs Garb" },
    { ITEM_SET_CONFESSORS_RAIMENT, "Confessors Raiment" },
    { ITEM_SET_DEMONIACS_THREADS, "Demoniacs Threads" },
    { ITEM_SET_ILLUSIONISTS_ATTIRE, "Illusionists Attire" },
    { ITEM_SET_THE_DEFILERS_DETERMINATION, "The Defilers Determination" },
    { ITEM_SET_THE_DEFILERS_FORTITUDE, "The Defilers Fortitude" },
    { ITEM_SET_THE_DEFILERS_INTENT, "The Defilers Intent" },
    { ITEM_SET_THE_DEFILERS_PURPOSE, "The Defilers Purpose" },
    { ITEM_SET_THE_DEFILERS_RESOLUTION, "The Defilers Resolution" },
    { ITEM_SET_THE_DEFILERS_WILL, "The Defilers Will" },
    { ITEM_SET_BLACK_DRAGON_MAIL, "Black Dragon Mail" },
    { ITEM_SET_GREEN_DRAGON_MAIL, "Green Dragon Mail" },
    { ITEM_SET_BLUE_DRAGON_MAIL, "Blue Dragon Mail" },
    { ITEM_SET_TWILIGHT_TRAPPINGS, "Twilight Trappings" },
    { ITEM_SET_GENESIS_RAIMENT, "Genesis Raiment" },
    { ITEM_SET_SYMBOLS_OF_UNENDING_LIFE, "Symbols Of Unending Life" },
    { ITEM_SET_BATTLEGEAR_OF_UNYIELDING_STRENGTH, "Battlegear Of Unyielding Strength" },
    { ITEM_SET_CONQUERORS_BATTLEGEAR, "Conquerors Battlegear" },
    { ITEM_SET_DEATHDEALERS_EMBRACE, "Deathdealers Embrace" },
    { ITEM_SET_EMBLEMS_OF_VEILED_SHADOWS, "Emblems Of Veiled Shadows" },
    { ITEM_SET_DOOMCALLERS_ATTIRE, "Doomcallers Attire" },
    { ITEM_SET_IMPLEMENTS_OF_UNSPOKEN_NAMES, "Implements Of Unspoken Names" },
    { ITEM_SET_STORMCALLERS_GARB, "Stormcallers Garb" },
    { ITEM_SET_GIFT_OF_THE_GATHERING_STORM, "Gift Of The Gathering Storm" },
    { ITEM_SET_ENIGMA_VESTMENTS, "Enigma Vestments" },
    { ITEM_SET_TRAPPINGS_OF_VAULTED_SECRETS, "Trappings Of Vaulted Secrets" },
    { ITEM_SET_AVENGERS_BATTLEGEAR, "Avengers Battlegear" },
    { ITEM_SET_BATTLEGEAR_OF_ETERNAL_JUSTICE, "Battlegear Of Eternal Justice" },
    { ITEM_SET_GARMENTS_OF_THE_ORACLE, "Garments Of The Oracle" },
    { ITEM_SET_FINERY_OF_INFINITE_WISDOM, "Finery Of Infinite Wisdom" },
    { ITEM_SET_STRIKERS_GARB, "Strikers Garb" },
    { ITEM_SET_TRAPPINGS_OF_THE_UNSEEN_PATH, "Trappings Of The Unseen Path" },
    { ITEM_SET_BATTLEGEAR_OF_HEROISM, "Battlegear Of Heroism" },
    { ITEM_SET_DARKMANTLE_ARMOR, "Darkmantle Armor" },
    { ITEM_SET_FERALHEART_RAIMENT, "Feralheart Raiment" },
    { ITEM_SET_VESTMENTS_OF_THE_VIRTUOUS, "Vestments Of The Virtuous" },
    { ITEM_SET_BEASTMASTER_ARMOR, "Beastmaster Armor" },
    { ITEM_SET_SOULFORGE_ARMOR, "Soulforge Armor" },
    { ITEM_SET_SORCERERS_REGALIA, "Sorcerers Regalia" },
    { ITEM_SET_DEATHMIST_RAIMENT, "Deathmist Raiment" },
    { ITEM_SET_THE_FIVE_THUNDERS, "The Five Thunders" },
    { ITEM_SET_IRONWEAVE_BATTLESUIT, "Ironweave Battlesuit" },
    { ITEM_SET_DREAMWALKER_RAIMENT, "Dreamwalker Raiment" },
    { ITEM_SET_CHAMPIONS_GUARD, "Champions Guard" },
    { ITEM_SET_DREADNAUGHTS_BATTLEGEAR, "Dreadnaughts Battlegear" },
    { ITEM_SET_BONESCYTHE_ARMOR, "Bonescythe Armor" },
    { ITEM_SET_VESTMENTS_OF_FAITH, "Vestments Of Faith" },
    { ITEM_SET_FROSTFIRE_REGALIA, "Frostfire Regalia" },
    { ITEM_SET_THE_EARTHSHATTERER, "The Earthshatterer" },
    { ITEM_SET_REDEMPTION_ARMOR, "Redemption Armor" },
    { ITEM_SET_PLAGUEHEART_RAIMENT, "Plagueheart Raiment" },
    { ITEM_SET_CRYPTSTALKER_ARMOR, "Cryptstalker Armor" },
    { ITEM_SET_BATTLEGEAR_OF_UNDEAD_SLAYING, "Battlegear Of Undead Slaying" },
    { ITEM_SET_UNDEAD_SLAYERS_ARMOR, "Undead Slayers Armor" },
    { ITEM_SET_GARB_OF_THE_UNDEAD_SLAYER, "Garb Of The Undead Slayer" },
    { ITEM_SET_REGALIA_OF_UNDEAD_CLEANSING, "Regalia Of Undead Cleansing" },
    { ITEM_SET_CHAMPIONS_BATTLEARMOR, "Champions Battlearmor" },
    { ITEM_SET_CHAMPIONS_STORMCALLER, "Champions Stormcaller" },
    { ITEM_SET_CHAMPIONS_REFUGE, "Champions Refuge" },
    { ITEM_SET_CHAMPIONS_INVESTITURE, "Champions Investiture" },
    { ITEM_SET_CHAMPIONS_DREADGEAR, "Champions Dreadgear" },
    { ITEM_SET_CHAMPIONS_ARCANUM, "Champions Arcanum" },
    { ITEM_SET_CHAMPIONS_PURSUANCE, "Champions Pursuance" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_REDOUBT, "Lieutenant Commanders Redoubt" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_BATTLEARMOR, "Lieutenant Commanders Battlearmor" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_ARCANUM, "Lieutenant Commanders Arcanum" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_DREADGEAR, "Lieutenant Commanders Dreadgear" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_GUARD, "Lieutenant Commanders Guard" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_INVESTITURE, "Lieutenant Commanders Investiture" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_PURSUANCE, "Lieutenant Commanders Pursuance" },
    { ITEM_SET_LIEUTENANT_COMMANDERS_REFUGE, "Lieutenant Commanders Refuge" },
    { 0, NULL }
};

typedef enum {
    BAG_FAMILY_NONE = 0x0,
    BAG_FAMILY_ARROWS = 0x1,
    BAG_FAMILY_BULLETS = 0x2,
    BAG_FAMILY_SOUL_SHARDS = 0x3,
    BAG_FAMILY_UNKNOWN4 = 0x4,
    BAG_FAMILY_UNKNOWN5 = 0x5,
    BAG_FAMILY_HERBS = 0x6,
    BAG_FAMILY_ENCHANTING_SUPPLIES = 0x7,
    BAG_FAMILY_ENGINEERING_SUPPLIES = 0x8,
    BAG_FAMILY_KEYS = 0x9,
} e_bag_family;
static const value_string e_bag_family_strings[] =  {
    { BAG_FAMILY_NONE, "None" },
    { BAG_FAMILY_ARROWS, "Arrows" },
    { BAG_FAMILY_BULLETS, "Bullets" },
    { BAG_FAMILY_SOUL_SHARDS, "Soul Shards" },
    { BAG_FAMILY_UNKNOWN4, "Unknown4" },
    { BAG_FAMILY_UNKNOWN5, "Unknown5" },
    { BAG_FAMILY_HERBS, "Herbs" },
    { BAG_FAMILY_ENCHANTING_SUPPLIES, "Enchanting Supplies" },
    { BAG_FAMILY_ENGINEERING_SUPPLIES, "Engineering Supplies" },
    { BAG_FAMILY_KEYS, "Keys" },
    { 0, NULL }
};

typedef enum {
    LOGOUT_RESULT_SUCCESS = 0x0,
    LOGOUT_RESULT_FAILURE_IN_COMBAT = 0x1,
    LOGOUT_RESULT_FAILURE_FROZEN_BY_GM = 0x2,
    LOGOUT_RESULT_FAILURE_JUMPING_OR_FALLING = 0x3,
} e_logout_result;
static const value_string e_logout_result_strings[] =  {
    { LOGOUT_RESULT_SUCCESS, "Success" },
    { LOGOUT_RESULT_FAILURE_IN_COMBAT, "Failure In Combat" },
    { LOGOUT_RESULT_FAILURE_FROZEN_BY_GM, "Failure Frozen By Gm" },
    { LOGOUT_RESULT_FAILURE_JUMPING_OR_FALLING, "Failure Jumping Or Falling" },
    { 0, NULL }
};

typedef enum {
    LOGOUT_SPEED_DELAYED = 0x0,
    LOGOUT_SPEED_INSTANT = 0x1,
} e_logout_speed;
static const value_string e_logout_speed_strings[] =  {
    { LOGOUT_SPEED_DELAYED, "Delayed" },
    { LOGOUT_SPEED_INSTANT, "Instant" },
    { 0, NULL }
};

typedef enum {
    EXPERIENCE_AWARD_TYPE_KILL = 0x0,
    EXPERIENCE_AWARD_TYPE_NON_KILL = 0x1,
} e_experience_award_type;
static const value_string e_experience_award_type_strings[] =  {
    { EXPERIENCE_AWARD_TYPE_KILL, "Kill" },
    { EXPERIENCE_AWARD_TYPE_NON_KILL, "Non Kill" },
    { 0, NULL }
};

typedef enum {
    LOOT_METHOD_ERROR = 0x00,
    LOOT_METHOD_CORPSE = 0x01,
    LOOT_METHOD_PICKPOCKETING = 0x02,
    LOOT_METHOD_FISHING = 0x03,
    LOOT_METHOD_DISENCHANTING = 0x04,
    LOOT_METHOD_SKINNING = 0x06,
    LOOT_METHOD_FISHINGHOLE = 0x14,
    LOOT_METHOD_FISHING_FAIL = 0x15,
    LOOT_METHOD_INSIGNIA = 0x16,
} e_loot_method;
static const value_string e_loot_method_strings[] =  {
    { LOOT_METHOD_ERROR, "Error" },
    { LOOT_METHOD_CORPSE, "Corpse" },
    { LOOT_METHOD_PICKPOCKETING, "Pickpocketing" },
    { LOOT_METHOD_FISHING, "Fishing" },
    { LOOT_METHOD_DISENCHANTING, "Disenchanting" },
    { LOOT_METHOD_SKINNING, "Skinning" },
    { LOOT_METHOD_FISHINGHOLE, "Fishinghole" },
    { LOOT_METHOD_FISHING_FAIL, "Fishing Fail" },
    { LOOT_METHOD_INSIGNIA, "Insignia" },
    { 0, NULL }
};

typedef enum {
    LOOT_METHOD_ERROR_DIDNT_KILL = 0x00,
    LOOT_METHOD_ERROR_TOO_FAR = 0x04,
    LOOT_METHOD_ERROR_BAD_FACING = 0x05,
    LOOT_METHOD_ERROR_LOCKED = 0x06,
    LOOT_METHOD_ERROR_NOTSTANDING = 0x08,
    LOOT_METHOD_ERROR_STUNNED = 0x09,
    LOOT_METHOD_ERROR_PLAYER_NOT_FOUND = 0x0A,
    LOOT_METHOD_ERROR_PLAY_TIME_EXCEEDED = 0x0B,
    LOOT_METHOD_ERROR_MASTER_INV_FULL = 0x0C,
    LOOT_METHOD_ERROR_MASTER_UNIQUE_ITEM = 0x0D,
    LOOT_METHOD_ERROR_MASTER_OTHER = 0x0E,
    LOOT_METHOD_ERROR_ALREADY_PICKPOCKETED = 0x0F,
    LOOT_METHOD_ERROR_NOT_WHILE_SHAPESHIFTED = 0x10,
} e_loot_method_error;
static const value_string e_loot_method_error_strings[] =  {
    { LOOT_METHOD_ERROR_DIDNT_KILL, "Didn't Kill" },
    { LOOT_METHOD_ERROR_TOO_FAR, "Too Far" },
    { LOOT_METHOD_ERROR_BAD_FACING, "Bad Facing" },
    { LOOT_METHOD_ERROR_LOCKED, "Locked" },
    { LOOT_METHOD_ERROR_NOTSTANDING, "Notstanding" },
    { LOOT_METHOD_ERROR_STUNNED, "Stunned" },
    { LOOT_METHOD_ERROR_PLAYER_NOT_FOUND, "Player Not Found" },
    { LOOT_METHOD_ERROR_PLAY_TIME_EXCEEDED, "Play Time Exceeded" },
    { LOOT_METHOD_ERROR_MASTER_INV_FULL, "Master Inv Full" },
    { LOOT_METHOD_ERROR_MASTER_UNIQUE_ITEM, "Master Unique Item" },
    { LOOT_METHOD_ERROR_MASTER_OTHER, "Master Other" },
    { LOOT_METHOD_ERROR_ALREADY_PICKPOCKETED, "Already Pickpocketed" },
    { LOOT_METHOD_ERROR_NOT_WHILE_SHAPESHIFTED, "Not While Shapeshifted" },
    { 0, NULL }
};

typedef enum {
    MEETING_STONE_FAILURE_MEETINGSTONE_FAIL_PARTYLEADER = 0x1,
    MEETING_STONE_FAILURE_MEETINGSTONE_FAIL_FULL_GROUP = 0x2,
    MEETING_STONE_FAILURE_MEETINGSTONE_FAIL_RAID_GROUP = 0x3,
} e_meeting_stone_failure;
static const value_string e_meeting_stone_failure_strings[] =  {
    { MEETING_STONE_FAILURE_MEETINGSTONE_FAIL_PARTYLEADER, "Meetingstone Fail Partyleader" },
    { MEETING_STONE_FAILURE_MEETINGSTONE_FAIL_FULL_GROUP, "Meetingstone Fail Full Group" },
    { MEETING_STONE_FAILURE_MEETINGSTONE_FAIL_RAID_GROUP, "Meetingstone Fail Raid Group" },
    { 0, NULL }
};

typedef enum {
    MEETING_STONE_STATUS_LEAVE_QUEUE = 0x0,
    MEETING_STONE_STATUS_JOINED_QUEUE = 0x1,
    MEETING_STONE_STATUS_PARTY_MEMBER_LEFT_LFG = 0x2,
    MEETING_STONE_STATUS_PARTY_MEMBER_REMOVED_PARTY_REMOVED = 0x3,
    MEETING_STONE_STATUS_LOOKING_FOR_NEW_PARTY_IN_QUEUE = 0x4,
    MEETING_STONE_STATUS_NONE = 0x5,
} e_meeting_stone_status;
static const value_string e_meeting_stone_status_strings[] =  {
    { MEETING_STONE_STATUS_LEAVE_QUEUE, "Leave Queue" },
    { MEETING_STONE_STATUS_JOINED_QUEUE, "Joined Queue" },
    { MEETING_STONE_STATUS_PARTY_MEMBER_LEFT_LFG, "Party Member Left Lfg" },
    { MEETING_STONE_STATUS_PARTY_MEMBER_REMOVED_PARTY_REMOVED, "Party Member Removed Party Removed" },
    { MEETING_STONE_STATUS_LOOKING_FOR_NEW_PARTY_IN_QUEUE, "Looking For New Party In Queue" },
    { MEETING_STONE_STATUS_NONE, "None" },
    { 0, NULL }
};

typedef enum {
    PLAYER_CHAT_TAG_NONE = 0x0,
    PLAYER_CHAT_TAG_AFK = 0x1,
    PLAYER_CHAT_TAG_DND = 0x2,
    PLAYER_CHAT_TAG_GM = 0x3,
} e_player_chat_tag;
static const value_string e_player_chat_tag_strings[] =  {
    { PLAYER_CHAT_TAG_NONE, "None" },
    { PLAYER_CHAT_TAG_AFK, "Afk" },
    { PLAYER_CHAT_TAG_DND, "Dnd" },
    { PLAYER_CHAT_TAG_GM, "Gm" },
    { 0, NULL }
};

typedef enum {
    MOUNT_RESULT_INVALID_MOUNTEE = 0x0,
    MOUNT_RESULT_TOO_FAR_AWAY = 0x1,
    MOUNT_RESULT_ALREADY_MOUNTED = 0x2,
    MOUNT_RESULT_NOT_MOUNTABLE = 0x3,
    MOUNT_RESULT_NOT_YOUR_PET = 0x4,
    MOUNT_RESULT_OTHER = 0x5,
    MOUNT_RESULT_LOOTING = 0x6,
    MOUNT_RESULT_RACE_CANT_MOUNT = 0x7,
    MOUNT_RESULT_SHAPESHIFTED = 0x8,
    MOUNT_RESULT_FORCED_DISMOUNT = 0x9,
    MOUNT_RESULT_OK = 0xA,
} e_mount_result;
static const value_string e_mount_result_strings[] =  {
    { MOUNT_RESULT_INVALID_MOUNTEE, "Invalid Mountee" },
    { MOUNT_RESULT_TOO_FAR_AWAY, "Too Far Away" },
    { MOUNT_RESULT_ALREADY_MOUNTED, "Already Mounted" },
    { MOUNT_RESULT_NOT_MOUNTABLE, "Not Mountable" },
    { MOUNT_RESULT_NOT_YOUR_PET, "Not Your Pet" },
    { MOUNT_RESULT_OTHER, "Other" },
    { MOUNT_RESULT_LOOTING, "Looting" },
    { MOUNT_RESULT_RACE_CANT_MOUNT, "Race Cant Mount" },
    { MOUNT_RESULT_SHAPESHIFTED, "Shapeshifted" },
    { MOUNT_RESULT_FORCED_DISMOUNT, "Forced Dismount" },
    { MOUNT_RESULT_OK, "Ok" },
    { 0, NULL }
};

typedef enum {
    PARTY_OPERATION_INVITE = 0x0,
    PARTY_OPERATION_LEAVE = 0x2,
} e_party_operation;
static const value_string e_party_operation_strings[] =  {
    { PARTY_OPERATION_INVITE, "Invite" },
    { PARTY_OPERATION_LEAVE, "Leave" },
    { 0, NULL }
};

typedef enum {
    PARTY_RESULT_SUCCESS = 0x0,
    PARTY_RESULT_BAD_PLAYER_NAME = 0x1,
    PARTY_RESULT_TARGET_NOT_IN_GROUP = 0x2,
    PARTY_RESULT_GROUP_FULL = 0x3,
    PARTY_RESULT_ALREADY_IN_GROUP = 0x4,
    PARTY_RESULT_NOT_IN_GROUP = 0x5,
    PARTY_RESULT_NOT_LEADER = 0x6,
    PARTY_RESULT_PLAYER_WRONG_FACTION = 0x7,
    PARTY_RESULT_IGNORING_YOU = 0x8,
} e_party_result;
static const value_string e_party_result_strings[] =  {
    { PARTY_RESULT_SUCCESS, "Success" },
    { PARTY_RESULT_BAD_PLAYER_NAME, "Bad Player Name" },
    { PARTY_RESULT_TARGET_NOT_IN_GROUP, "Target Not In Group" },
    { PARTY_RESULT_GROUP_FULL, "Group Full" },
    { PARTY_RESULT_ALREADY_IN_GROUP, "Already In Group" },
    { PARTY_RESULT_NOT_IN_GROUP, "Not In Group" },
    { PARTY_RESULT_NOT_LEADER, "Not Leader" },
    { PARTY_RESULT_PLAYER_WRONG_FACTION, "Player Wrong Faction" },
    { PARTY_RESULT_IGNORING_YOU, "Ignoring You" },
    { 0, NULL }
};

typedef enum {
    TIMER_TYPE_FATIGUE = 0x0,
    TIMER_TYPE_BREATH = 0x1,
    TIMER_TYPE_FEIGN_DEATH = 0x2,
    TIMER_TYPE_ENVIRONMENTAL = 0x3,
} e_timer_type;
static const value_string e_timer_type_strings[] =  {
    { TIMER_TYPE_FATIGUE, "Fatigue" },
    { TIMER_TYPE_BREATH, "Breath" },
    { TIMER_TYPE_FEIGN_DEATH, "Feign Death" },
    { TIMER_TYPE_ENVIRONMENTAL, "Environmental" },
    { 0, NULL }
};

typedef enum {
    PETITION_RESULT_OK = 0x0,
    PETITION_RESULT_ALREADY_SIGNED = 0x1,
    PETITION_RESULT_ALREADY_IN_GUILD = 0x2,
    PETITION_RESULT_CANT_SIGN_OWN = 0x3,
    PETITION_RESULT_NEED_MORE = 0x4,
    PETITION_RESULT_NOT_SERVER = 0x5,
} e_petition_result;
static const value_string e_petition_result_strings[] =  {
    { PETITION_RESULT_OK, "Ok" },
    { PETITION_RESULT_ALREADY_SIGNED, "Already Signed" },
    { PETITION_RESULT_ALREADY_IN_GUILD, "Already In Guild" },
    { PETITION_RESULT_CANT_SIGN_OWN, "Cant Sign Own" },
    { PETITION_RESULT_NEED_MORE, "Need More" },
    { PETITION_RESULT_NOT_SERVER, "Not Server" },
    { 0, NULL }
};

typedef enum {
    PET_FEEDBACK_PET_DEAD = 0x1,
    PET_FEEDBACK_NOTHING_TO_ATTACK = 0x2,
    PET_FEEDBACK_CANT_ATTACK_TARGET = 0x3,
    PET_FEEDBACK_NO_PATH_TO = 0x4,
} e_pet_feedback;
static const value_string e_pet_feedback_strings[] =  {
    { PET_FEEDBACK_PET_DEAD, "Pet Dead" },
    { PET_FEEDBACK_NOTHING_TO_ATTACK, "Nothing To Attack" },
    { PET_FEEDBACK_CANT_ATTACK_TARGET, "Cant Attack Target" },
    { PET_FEEDBACK_NO_PATH_TO, "No Path To" },
    { 0, NULL }
};

typedef enum {
    PET_TALK_REASON_SPECIAL_SPELL = 0x0,
    PET_TALK_REASON_ATTACK = 0x1,
} e_pet_talk_reason;
static const value_string e_pet_talk_reason_strings[] =  {
    { PET_TALK_REASON_SPECIAL_SPELL, "Special Spell" },
    { PET_TALK_REASON_ATTACK, "Attack" },
    { 0, NULL }
};

typedef enum {
    SPELL_CAST_RESULT_AFFECTING_COMBAT = 0x00,
    SPELL_CAST_RESULT_ALREADY_AT_FULL_HEALTH = 0x01,
    SPELL_CAST_RESULT_ALREADY_AT_FULL_MANA = 0x02,
    SPELL_CAST_RESULT_ALREADY_BEING_TAMED = 0x03,
    SPELL_CAST_RESULT_ALREADY_HAVE_CHARM = 0x04,
    SPELL_CAST_RESULT_ALREADY_HAVE_SUMMON = 0x05,
    SPELL_CAST_RESULT_ALREADY_OPEN = 0x06,
    SPELL_CAST_RESULT_MORE_POWERFUL_SPELL_ACTIVE = 0x07,
    SPELL_CAST_RESULT_BAD_IMPLICIT_TARGETS = 0x09,
    SPELL_CAST_RESULT_BAD_TARGETS = 0x0A,
    SPELL_CAST_RESULT_CANT_BE_CHARMED = 0x0B,
    SPELL_CAST_RESULT_CANT_BE_DISENCHANTED = 0x0C,
    SPELL_CAST_RESULT_CANT_BE_PROSPECTED = 0x0D,
    SPELL_CAST_RESULT_CANT_CAST_ON_TAPPED = 0x0E,
    SPELL_CAST_RESULT_CANT_DUEL_WHILE_INVISIBLE = 0x0F,
    SPELL_CAST_RESULT_CANT_DUEL_WHILE_STEALTHED = 0x10,
    SPELL_CAST_RESULT_CANT_TOO_CLOSE_TO_ENEMY = 0x11,
    SPELL_CAST_RESULT_CANT_DO_THAT_YET = 0x12,
    SPELL_CAST_RESULT_CASTER_DEAD = 0x13,
    SPELL_CAST_RESULT_CHARMED = 0x14,
    SPELL_CAST_RESULT_CHEST_IN_USE = 0x15,
    SPELL_CAST_RESULT_CONFUSED = 0x16,
    SPELL_CAST_RESULT_DONT_REPORT = 0x17,
    SPELL_CAST_RESULT_EQUIPPED_ITEM = 0x18,
    SPELL_CAST_RESULT_EQUIPPED_ITEM_CLASS = 0x19,
    SPELL_CAST_RESULT_EQUIPPED_ITEM_CLASS_MAINHAND = 0x1A,
    SPELL_CAST_RESULT_EQUIPPED_ITEM_CLASS_OFFHAND = 0x1B,
    SPELL_CAST_RESULT_ERROR = 0x1C,
    SPELL_CAST_RESULT_FIZZLE = 0x1D,
    SPELL_CAST_RESULT_FLEEING = 0x1E,
    SPELL_CAST_RESULT_FOOD_LOWLEVEL = 0x1F,
    SPELL_CAST_RESULT_HIGHLEVEL = 0x20,
    SPELL_CAST_RESULT_IMMUNE = 0x22,
    SPELL_CAST_RESULT_INTERRUPTED = 0x23,
    SPELL_CAST_RESULT_INTERRUPTED_COMBAT = 0x24,
    SPELL_CAST_RESULT_ITEM_ALREADY_ENCHANTED = 0x25,
    SPELL_CAST_RESULT_ITEM_GONE = 0x26,
    SPELL_CAST_RESULT_ENCHANT_NOT_EXISTING_ITEM = 0x27,
    SPELL_CAST_RESULT_ITEM_NOT_READY = 0x28,
    SPELL_CAST_RESULT_LEVEL_REQUIREMENT = 0x29,
    SPELL_CAST_RESULT_LINE_OF_SIGHT = 0x2A,
    SPELL_CAST_RESULT_LOWLEVEL = 0x2B,
    SPELL_CAST_RESULT_SKILL_NOT_HIGH_ENOUGH = 0x2C,
    SPELL_CAST_RESULT_MAINHAND_EMPTY = 0x2D,
    SPELL_CAST_RESULT_MOVING = 0x2E,
    SPELL_CAST_RESULT_NEED_AMMO = 0x2F,
    SPELL_CAST_RESULT_NEED_REQUIRES_SOMETHING = 0x30,
    SPELL_CAST_RESULT_NEED_EXOTIC_AMMO = 0x31,
    SPELL_CAST_RESULT_NOPATH = 0x32,
    SPELL_CAST_RESULT_NOT_BEHIND = 0x33,
    SPELL_CAST_RESULT_NOT_FISHABLE = 0x34,
    SPELL_CAST_RESULT_NOT_HERE = 0x35,
    SPELL_CAST_RESULT_NOT_INFRONT = 0x36,
    SPELL_CAST_RESULT_NOT_IN_CONTROL = 0x37,
    SPELL_CAST_RESULT_NOT_KNOWN = 0x38,
    SPELL_CAST_RESULT_NOT_MOUNTED = 0x39,
    SPELL_CAST_RESULT_NOT_ON_TAXI = 0x3A,
    SPELL_CAST_RESULT_NOT_ON_TRANSPORT = 0x3B,
    SPELL_CAST_RESULT_NOT_READY = 0x3C,
    SPELL_CAST_RESULT_NOT_SHAPESHIFT = 0x3D,
    SPELL_CAST_RESULT_NOT_STANDING = 0x3E,
    SPELL_CAST_RESULT_NOT_TRADEABLE = 0x3F,
    SPELL_CAST_RESULT_NOT_TRADING = 0x40,
    SPELL_CAST_RESULT_NOT_UNSHEATHED = 0x41,
    SPELL_CAST_RESULT_NOT_WHILE_GHOST = 0x42,
    SPELL_CAST_RESULT_NO_AMMO = 0x43,
    SPELL_CAST_RESULT_NO_CHARGES_REMAIN = 0x44,
    SPELL_CAST_RESULT_NO_CHAMPION = 0x45,
    SPELL_CAST_RESULT_NO_COMBO_POINTS = 0x46,
    SPELL_CAST_RESULT_NO_DUELING = 0x47,
    SPELL_CAST_RESULT_NO_ENDURANCE = 0x48,
    SPELL_CAST_RESULT_NO_FISH = 0x49,
    SPELL_CAST_RESULT_NO_ITEMS_WHILE_SHAPESHIFTED = 0x4A,
    SPELL_CAST_RESULT_NO_MOUNTS_ALLOWED = 0x4B,
    SPELL_CAST_RESULT_NO_PET = 0x4C,
    SPELL_CAST_RESULT_NO_POWER = 0x4D,
    SPELL_CAST_RESULT_NOTHING_TO_DISPEL = 0x4E,
    SPELL_CAST_RESULT_NOTHING_TO_STEAL = 0x4F,
    SPELL_CAST_RESULT_ONLY_ABOVEWATER = 0x50,
    SPELL_CAST_RESULT_ONLY_DAYTIME = 0x51,
    SPELL_CAST_RESULT_ONLY_INDOORS = 0x52,
    SPELL_CAST_RESULT_ONLY_MOUNTED = 0x53,
    SPELL_CAST_RESULT_ONLY_NIGHTTIME = 0x54,
    SPELL_CAST_RESULT_ONLY_OUTDOORS = 0x55,
    SPELL_CAST_RESULT_ONLY_SHAPESHIFT = 0x56,
    SPELL_CAST_RESULT_ONLY_STEALTHED = 0x57,
    SPELL_CAST_RESULT_ONLY_UNDERWATER = 0x58,
    SPELL_CAST_RESULT_OUT_OF_RANGE = 0x59,
    SPELL_CAST_RESULT_PACIFIED = 0x5A,
    SPELL_CAST_RESULT_POSSESSED = 0x5B,
    SPELL_CAST_RESULT_REQUIRES_AREA = 0x5D,
    SPELL_CAST_RESULT_REQUIRES_SPELL_FOCUS = 0x5E,
    SPELL_CAST_RESULT_ROOTED = 0x5F,
    SPELL_CAST_RESULT_SILENCED = 0x60,
    SPELL_CAST_RESULT_SPELL_IN_PROGRESS = 0x61,
    SPELL_CAST_RESULT_SPELL_LEARNED = 0x62,
    SPELL_CAST_RESULT_SPELL_UNAVAILABLE = 0x63,
    SPELL_CAST_RESULT_STUNNED = 0x64,
    SPELL_CAST_RESULT_TARGETS_DEAD = 0x65,
    SPELL_CAST_RESULT_TARGET_AFFECTING_COMBAT = 0x66,
    SPELL_CAST_RESULT_TARGET_AURASTATE = 0x67,
    SPELL_CAST_RESULT_TARGET_DUELING = 0x68,
    SPELL_CAST_RESULT_TARGET_ENEMY = 0x69,
    SPELL_CAST_RESULT_TARGET_ENRAGED = 0x6A,
    SPELL_CAST_RESULT_TARGET_FRIENDLY = 0x6B,
    SPELL_CAST_RESULT_TARGET_IN_COMBAT = 0x6C,
    SPELL_CAST_RESULT_TARGET_IS_PLAYER = 0x6D,
    SPELL_CAST_RESULT_TARGET_NOT_DEAD = 0x6E,
    SPELL_CAST_RESULT_TARGET_NOT_IN_PARTY = 0x6F,
    SPELL_CAST_RESULT_TARGET_NOT_LOOTED = 0x70,
    SPELL_CAST_RESULT_TARGET_NOT_PLAYER = 0x71,
    SPELL_CAST_RESULT_TARGET_NO_POCKETS = 0x72,
    SPELL_CAST_RESULT_TARGET_NO_WEAPONS = 0x73,
    SPELL_CAST_RESULT_TARGET_UNSKINNABLE = 0x74,
    SPELL_CAST_RESULT_THIRST_SATIATED = 0x75,
    SPELL_CAST_RESULT_TOO_CLOSE = 0x76,
    SPELL_CAST_RESULT_TOO_MANY_OF_ITEM = 0x77,
    SPELL_CAST_RESULT_TRAINING_POINTS = 0x79,
    SPELL_CAST_RESULT_TRY_AGAIN = 0x7A,
    SPELL_CAST_RESULT_UNIT_NOT_BEHIND = 0x7B,
    SPELL_CAST_RESULT_UNIT_NOT_INFRONT = 0x7C,
    SPELL_CAST_RESULT_WRONG_PET_FOOD = 0x7D,
    SPELL_CAST_RESULT_NOT_WHILE_FATIGUED = 0x7E,
    SPELL_CAST_RESULT_TARGET_NOT_IN_INSTANCE = 0x7F,
    SPELL_CAST_RESULT_NOT_WHILE_TRADING = 0x80,
    SPELL_CAST_RESULT_TARGET_NOT_IN_RAID = 0x81,
    SPELL_CAST_RESULT_DISENCHANT_WHILE_LOOTING = 0x82,
    SPELL_CAST_RESULT_PROSPECT_WHILE_LOOTING = 0x83,
    SPELL_CAST_RESULT_TARGET_FREEFORALL = 0x85,
    SPELL_CAST_RESULT_NO_EDIBLE_CORPSES = 0x86,
    SPELL_CAST_RESULT_ONLY_BATTLEGROUNDS = 0x87,
    SPELL_CAST_RESULT_TARGET_NOT_GHOST = 0x88,
    SPELL_CAST_RESULT_TOO_MANY_SKILLS = 0x89,
    SPELL_CAST_RESULT_CANT_USE_NEW_ITEM = 0x8A,
    SPELL_CAST_RESULT_WRONG_WEATHER = 0x8B,
    SPELL_CAST_RESULT_DAMAGE_IMMUNE = 0x8C,
    SPELL_CAST_RESULT_PREVENTED_BY_MECHANIC = 0x8D,
    SPELL_CAST_RESULT_PLAY_TIME = 0x8E,
    SPELL_CAST_RESULT_REPUTATION = 0x8F,
    SPELL_CAST_RESULT_MIN_SKILL = 0x90,
    SPELL_CAST_RESULT_UNKNOWN = 0x91,
} e_spell_cast_result;
static const value_string e_spell_cast_result_strings[] =  {
    { SPELL_CAST_RESULT_AFFECTING_COMBAT, "Affecting Combat" },
    { SPELL_CAST_RESULT_ALREADY_AT_FULL_HEALTH, "Already At Full Health" },
    { SPELL_CAST_RESULT_ALREADY_AT_FULL_MANA, "Already At Full Mana" },
    { SPELL_CAST_RESULT_ALREADY_BEING_TAMED, "Already Being Tamed" },
    { SPELL_CAST_RESULT_ALREADY_HAVE_CHARM, "Already Have Charm" },
    { SPELL_CAST_RESULT_ALREADY_HAVE_SUMMON, "Already Have Summon" },
    { SPELL_CAST_RESULT_ALREADY_OPEN, "Already Open" },
    { SPELL_CAST_RESULT_MORE_POWERFUL_SPELL_ACTIVE, "More Powerful Spell Active" },
    { SPELL_CAST_RESULT_BAD_IMPLICIT_TARGETS, "Bad Implicit Targets" },
    { SPELL_CAST_RESULT_BAD_TARGETS, "Bad Targets" },
    { SPELL_CAST_RESULT_CANT_BE_CHARMED, "Cant Be Charmed" },
    { SPELL_CAST_RESULT_CANT_BE_DISENCHANTED, "Cant Be Disenchanted" },
    { SPELL_CAST_RESULT_CANT_BE_PROSPECTED, "Cant Be Prospected" },
    { SPELL_CAST_RESULT_CANT_CAST_ON_TAPPED, "Cant Cast On Tapped" },
    { SPELL_CAST_RESULT_CANT_DUEL_WHILE_INVISIBLE, "Cant Duel While Invisible" },
    { SPELL_CAST_RESULT_CANT_DUEL_WHILE_STEALTHED, "Cant Duel While Stealthed" },
    { SPELL_CAST_RESULT_CANT_TOO_CLOSE_TO_ENEMY, "Cant Too Close To Enemy" },
    { SPELL_CAST_RESULT_CANT_DO_THAT_YET, "Cant Do That Yet" },
    { SPELL_CAST_RESULT_CASTER_DEAD, "Caster Dead" },
    { SPELL_CAST_RESULT_CHARMED, "Charmed" },
    { SPELL_CAST_RESULT_CHEST_IN_USE, "Chest In Use" },
    { SPELL_CAST_RESULT_CONFUSED, "Confused" },
    { SPELL_CAST_RESULT_DONT_REPORT, "Dont Report" },
    { SPELL_CAST_RESULT_EQUIPPED_ITEM, "Equipped Item" },
    { SPELL_CAST_RESULT_EQUIPPED_ITEM_CLASS, "Equipped Item Class" },
    { SPELL_CAST_RESULT_EQUIPPED_ITEM_CLASS_MAINHAND, "Equipped Item Class Mainhand" },
    { SPELL_CAST_RESULT_EQUIPPED_ITEM_CLASS_OFFHAND, "Equipped Item Class Offhand" },
    { SPELL_CAST_RESULT_ERROR, "Error" },
    { SPELL_CAST_RESULT_FIZZLE, "Fizzle" },
    { SPELL_CAST_RESULT_FLEEING, "Fleeing" },
    { SPELL_CAST_RESULT_FOOD_LOWLEVEL, "Food Lowlevel" },
    { SPELL_CAST_RESULT_HIGHLEVEL, "Highlevel" },
    { SPELL_CAST_RESULT_IMMUNE, "Immune" },
    { SPELL_CAST_RESULT_INTERRUPTED, "Interrupted" },
    { SPELL_CAST_RESULT_INTERRUPTED_COMBAT, "Interrupted Combat" },
    { SPELL_CAST_RESULT_ITEM_ALREADY_ENCHANTED, "Item Already Enchanted" },
    { SPELL_CAST_RESULT_ITEM_GONE, "Item Gone" },
    { SPELL_CAST_RESULT_ENCHANT_NOT_EXISTING_ITEM, "Enchant Not Existing Item" },
    { SPELL_CAST_RESULT_ITEM_NOT_READY, "Item Not Ready" },
    { SPELL_CAST_RESULT_LEVEL_REQUIREMENT, "Level Requirement" },
    { SPELL_CAST_RESULT_LINE_OF_SIGHT, "Line Of Sight" },
    { SPELL_CAST_RESULT_LOWLEVEL, "Lowlevel" },
    { SPELL_CAST_RESULT_SKILL_NOT_HIGH_ENOUGH, "Skill Not High Enough" },
    { SPELL_CAST_RESULT_MAINHAND_EMPTY, "Mainhand Empty" },
    { SPELL_CAST_RESULT_MOVING, "Moving" },
    { SPELL_CAST_RESULT_NEED_AMMO, "Need Ammo" },
    { SPELL_CAST_RESULT_NEED_REQUIRES_SOMETHING, "Need Requires Something" },
    { SPELL_CAST_RESULT_NEED_EXOTIC_AMMO, "Need Exotic Ammo" },
    { SPELL_CAST_RESULT_NOPATH, "Nopath" },
    { SPELL_CAST_RESULT_NOT_BEHIND, "Not Behind" },
    { SPELL_CAST_RESULT_NOT_FISHABLE, "Not Fishable" },
    { SPELL_CAST_RESULT_NOT_HERE, "Not Here" },
    { SPELL_CAST_RESULT_NOT_INFRONT, "Not Infront" },
    { SPELL_CAST_RESULT_NOT_IN_CONTROL, "Not In Control" },
    { SPELL_CAST_RESULT_NOT_KNOWN, "Not Known" },
    { SPELL_CAST_RESULT_NOT_MOUNTED, "Not Mounted" },
    { SPELL_CAST_RESULT_NOT_ON_TAXI, "Not On Taxi" },
    { SPELL_CAST_RESULT_NOT_ON_TRANSPORT, "Not On Transport" },
    { SPELL_CAST_RESULT_NOT_READY, "Not Ready" },
    { SPELL_CAST_RESULT_NOT_SHAPESHIFT, "Not Shapeshift" },
    { SPELL_CAST_RESULT_NOT_STANDING, "Not Standing" },
    { SPELL_CAST_RESULT_NOT_TRADEABLE, "Not Tradeable" },
    { SPELL_CAST_RESULT_NOT_TRADING, "Not Trading" },
    { SPELL_CAST_RESULT_NOT_UNSHEATHED, "Not Unsheathed" },
    { SPELL_CAST_RESULT_NOT_WHILE_GHOST, "Not While Ghost" },
    { SPELL_CAST_RESULT_NO_AMMO, "No Ammo" },
    { SPELL_CAST_RESULT_NO_CHARGES_REMAIN, "No Charges Remain" },
    { SPELL_CAST_RESULT_NO_CHAMPION, "No Champion" },
    { SPELL_CAST_RESULT_NO_COMBO_POINTS, "No Combo Points" },
    { SPELL_CAST_RESULT_NO_DUELING, "No Dueling" },
    { SPELL_CAST_RESULT_NO_ENDURANCE, "No Endurance" },
    { SPELL_CAST_RESULT_NO_FISH, "No Fish" },
    { SPELL_CAST_RESULT_NO_ITEMS_WHILE_SHAPESHIFTED, "No Items While Shapeshifted" },
    { SPELL_CAST_RESULT_NO_MOUNTS_ALLOWED, "No Mounts Allowed" },
    { SPELL_CAST_RESULT_NO_PET, "No Pet" },
    { SPELL_CAST_RESULT_NO_POWER, "No Power" },
    { SPELL_CAST_RESULT_NOTHING_TO_DISPEL, "Nothing To Dispel" },
    { SPELL_CAST_RESULT_NOTHING_TO_STEAL, "Nothing To Steal" },
    { SPELL_CAST_RESULT_ONLY_ABOVEWATER, "Only Abovewater" },
    { SPELL_CAST_RESULT_ONLY_DAYTIME, "Only Daytime" },
    { SPELL_CAST_RESULT_ONLY_INDOORS, "Only Indoors" },
    { SPELL_CAST_RESULT_ONLY_MOUNTED, "Only Mounted" },
    { SPELL_CAST_RESULT_ONLY_NIGHTTIME, "Only Nighttime" },
    { SPELL_CAST_RESULT_ONLY_OUTDOORS, "Only Outdoors" },
    { SPELL_CAST_RESULT_ONLY_SHAPESHIFT, "Only Shapeshift" },
    { SPELL_CAST_RESULT_ONLY_STEALTHED, "Only Stealthed" },
    { SPELL_CAST_RESULT_ONLY_UNDERWATER, "Only Underwater" },
    { SPELL_CAST_RESULT_OUT_OF_RANGE, "Out Of Range" },
    { SPELL_CAST_RESULT_PACIFIED, "Pacified" },
    { SPELL_CAST_RESULT_POSSESSED, "Possessed" },
    { SPELL_CAST_RESULT_REQUIRES_AREA, "Requires Area" },
    { SPELL_CAST_RESULT_REQUIRES_SPELL_FOCUS, "Requires Spell Focus" },
    { SPELL_CAST_RESULT_ROOTED, "Rooted" },
    { SPELL_CAST_RESULT_SILENCED, "Silenced" },
    { SPELL_CAST_RESULT_SPELL_IN_PROGRESS, "Spell In Progress" },
    { SPELL_CAST_RESULT_SPELL_LEARNED, "Spell Learned" },
    { SPELL_CAST_RESULT_SPELL_UNAVAILABLE, "Spell Unavailable" },
    { SPELL_CAST_RESULT_STUNNED, "Stunned" },
    { SPELL_CAST_RESULT_TARGETS_DEAD, "Targets Dead" },
    { SPELL_CAST_RESULT_TARGET_AFFECTING_COMBAT, "Target Affecting Combat" },
    { SPELL_CAST_RESULT_TARGET_AURASTATE, "Target Aurastate" },
    { SPELL_CAST_RESULT_TARGET_DUELING, "Target Dueling" },
    { SPELL_CAST_RESULT_TARGET_ENEMY, "Target Enemy" },
    { SPELL_CAST_RESULT_TARGET_ENRAGED, "Target Enraged" },
    { SPELL_CAST_RESULT_TARGET_FRIENDLY, "Target Friendly" },
    { SPELL_CAST_RESULT_TARGET_IN_COMBAT, "Target In Combat" },
    { SPELL_CAST_RESULT_TARGET_IS_PLAYER, "Target Is Player" },
    { SPELL_CAST_RESULT_TARGET_NOT_DEAD, "Target Not Dead" },
    { SPELL_CAST_RESULT_TARGET_NOT_IN_PARTY, "Target Not In Party" },
    { SPELL_CAST_RESULT_TARGET_NOT_LOOTED, "Target Not Looted" },
    { SPELL_CAST_RESULT_TARGET_NOT_PLAYER, "Target Not Player" },
    { SPELL_CAST_RESULT_TARGET_NO_POCKETS, "Target No Pockets" },
    { SPELL_CAST_RESULT_TARGET_NO_WEAPONS, "Target No Weapons" },
    { SPELL_CAST_RESULT_TARGET_UNSKINNABLE, "Target Unskinnable" },
    { SPELL_CAST_RESULT_THIRST_SATIATED, "Thirst Satiated" },
    { SPELL_CAST_RESULT_TOO_CLOSE, "Too Close" },
    { SPELL_CAST_RESULT_TOO_MANY_OF_ITEM, "Too Many Of Item" },
    { SPELL_CAST_RESULT_TRAINING_POINTS, "Training Points" },
    { SPELL_CAST_RESULT_TRY_AGAIN, "Try Again" },
    { SPELL_CAST_RESULT_UNIT_NOT_BEHIND, "Unit Not Behind" },
    { SPELL_CAST_RESULT_UNIT_NOT_INFRONT, "Unit Not Infront" },
    { SPELL_CAST_RESULT_WRONG_PET_FOOD, "Wrong Pet Food" },
    { SPELL_CAST_RESULT_NOT_WHILE_FATIGUED, "Not While Fatigued" },
    { SPELL_CAST_RESULT_TARGET_NOT_IN_INSTANCE, "Target Not In Instance" },
    { SPELL_CAST_RESULT_NOT_WHILE_TRADING, "Not While Trading" },
    { SPELL_CAST_RESULT_TARGET_NOT_IN_RAID, "Target Not In Raid" },
    { SPELL_CAST_RESULT_DISENCHANT_WHILE_LOOTING, "Disenchant While Looting" },
    { SPELL_CAST_RESULT_PROSPECT_WHILE_LOOTING, "Prospect While Looting" },
    { SPELL_CAST_RESULT_TARGET_FREEFORALL, "Target Freeforall" },
    { SPELL_CAST_RESULT_NO_EDIBLE_CORPSES, "No Edible Corpses" },
    { SPELL_CAST_RESULT_ONLY_BATTLEGROUNDS, "Only Battlegrounds" },
    { SPELL_CAST_RESULT_TARGET_NOT_GHOST, "Target Not Ghost" },
    { SPELL_CAST_RESULT_TOO_MANY_SKILLS, "Too Many Skills" },
    { SPELL_CAST_RESULT_CANT_USE_NEW_ITEM, "Cant Use New Item" },
    { SPELL_CAST_RESULT_WRONG_WEATHER, "Wrong Weather" },
    { SPELL_CAST_RESULT_DAMAGE_IMMUNE, "Damage Immune" },
    { SPELL_CAST_RESULT_PREVENTED_BY_MECHANIC, "Prevented By Mechanic" },
    { SPELL_CAST_RESULT_PLAY_TIME, "Play Time" },
    { SPELL_CAST_RESULT_REPUTATION, "Reputation" },
    { SPELL_CAST_RESULT_MIN_SKILL, "Min Skill" },
    { SPELL_CAST_RESULT_UNKNOWN, "Unknown" },
    { 0, NULL }
};

typedef enum {
    PET_REACT_STATE_PASSIVE = 0x0,
    PET_REACT_STATE_DEFENSIVE = 0x1,
    PET_REACT_STATE_AGGRESSIVE = 0x2,
} e_pet_react_state;
static const value_string e_pet_react_state_strings[] =  {
    { PET_REACT_STATE_PASSIVE, "Passive" },
    { PET_REACT_STATE_DEFENSIVE, "Defensive" },
    { PET_REACT_STATE_AGGRESSIVE, "Aggressive" },
    { 0, NULL }
};

typedef enum {
    PET_COMMAND_STATE_STAY = 0x0,
    PET_COMMAND_STATE_FOLLOW = 0x1,
    PET_COMMAND_STATE_ATTACK = 0x2,
    PET_COMMAND_STATE_DISMISS = 0x3,
} e_pet_command_state;
static const value_string e_pet_command_state_strings[] =  {
    { PET_COMMAND_STATE_STAY, "Stay" },
    { PET_COMMAND_STATE_FOLLOW, "Follow" },
    { PET_COMMAND_STATE_ATTACK, "Attack" },
    { PET_COMMAND_STATE_DISMISS, "Dismiss" },
    { 0, NULL }
};

typedef enum {
    PET_ENABLED_ENABLED = 0x8,
    PET_ENABLED_DISABLED = 0x0,
} e_pet_enabled;
static const value_string e_pet_enabled_strings[] =  {
    { PET_ENABLED_ENABLED, "Enabled" },
    { PET_ENABLED_DISABLED, "Disabled" },
    { 0, NULL }
};

typedef enum {
    PET_TAME_FAILURE_REASON_INVALID_CREATURE = 0x1,
    PET_TAME_FAILURE_REASON_TOO_MANY = 0x2,
    PET_TAME_FAILURE_REASON_CREATURE_ALREADY_OWNED = 0x3,
    PET_TAME_FAILURE_REASON_NOT_TAMEABLE = 0x4,
    PET_TAME_FAILURE_REASON_ANOTHER_SUMMON_ACTIVE = 0x5,
    PET_TAME_FAILURE_REASON_UNITS_CANT_TAME = 0x6,
    PET_TAME_FAILURE_REASON_NO_PET_AVAILABLE = 0x7,
    PET_TAME_FAILURE_REASON_INTERNAL_ERROR = 0x8,
    PET_TAME_FAILURE_REASON_TOO_HIGH_LEVEL = 0x9,
    PET_TAME_FAILURE_REASON_DEAD = 0xA,
    PET_TAME_FAILURE_REASON_NOT_DEAD = 0xB,
    PET_TAME_FAILURE_REASON_UNKNOWN_ERROR = 0xC,
} e_pet_tame_failure_reason;
static const value_string e_pet_tame_failure_reason_strings[] =  {
    { PET_TAME_FAILURE_REASON_INVALID_CREATURE, "Invalid Creature" },
    { PET_TAME_FAILURE_REASON_TOO_MANY, "Too Many" },
    { PET_TAME_FAILURE_REASON_CREATURE_ALREADY_OWNED, "Creature Already Owned" },
    { PET_TAME_FAILURE_REASON_NOT_TAMEABLE, "Not Tameable" },
    { PET_TAME_FAILURE_REASON_ANOTHER_SUMMON_ACTIVE, "Another Summon Active" },
    { PET_TAME_FAILURE_REASON_UNITS_CANT_TAME, "Units Cant Tame" },
    { PET_TAME_FAILURE_REASON_NO_PET_AVAILABLE, "No Pet Available" },
    { PET_TAME_FAILURE_REASON_INTERNAL_ERROR, "Internal Error" },
    { PET_TAME_FAILURE_REASON_TOO_HIGH_LEVEL, "Too High Level" },
    { PET_TAME_FAILURE_REASON_DEAD, "Dead" },
    { PET_TAME_FAILURE_REASON_NOT_DEAD, "Not Dead" },
    { PET_TAME_FAILURE_REASON_UNKNOWN_ERROR, "Unknown Error" },
    { 0, NULL }
};

typedef enum {
    LOG_FORMAT_DEFAULT = 0x0,
    LOG_FORMAT_DEBUG = 0x1,
} e_log_format;
static const value_string e_log_format_strings[] =  {
    { LOG_FORMAT_DEFAULT, "Default" },
    { LOG_FORMAT_DEBUG, "Debug" },
    { 0, NULL }
};

typedef enum {
    QUEST_FAILED_REASON_DONT_HAVE_REQ = 0x00,
    QUEST_FAILED_REASON_QUEST_FAILED_LOW_LEVEL = 0x01,
    QUEST_FAILED_REASON_QUEST_FAILED_REQS = 0x02,
    QUEST_FAILED_REASON_QUEST_FAILED_INVENTORY_FULL = 0x04,
    QUEST_FAILED_REASON_QUEST_FAILED_WRONG_RACE = 0x06,
    QUEST_FAILED_REASON_QUEST_ONLY_ONE_TIMED = 0x0C,
    QUEST_FAILED_REASON_QUEST_ALREADY_ON = 0x0D,
    QUEST_FAILED_REASON_QUEST_FAILED_DUPLICATE_ITEM = 0x11,
    QUEST_FAILED_REASON_QUEST_FAILED_MISSING_ITEMS = 0x14,
    QUEST_FAILED_REASON_QUEST_FAILED_NOT_ENOUGH_MONEY = 0x16,
} e_quest_failed_reason;
static const value_string e_quest_failed_reason_strings[] =  {
    { QUEST_FAILED_REASON_DONT_HAVE_REQ, "Dont Have Req" },
    { QUEST_FAILED_REASON_QUEST_FAILED_LOW_LEVEL, "Quest Failed Low Level" },
    { QUEST_FAILED_REASON_QUEST_FAILED_REQS, "Quest Failed Reqs" },
    { QUEST_FAILED_REASON_QUEST_FAILED_INVENTORY_FULL, "Quest Failed Inventory Full" },
    { QUEST_FAILED_REASON_QUEST_FAILED_WRONG_RACE, "Quest Failed Wrong Race" },
    { QUEST_FAILED_REASON_QUEST_ONLY_ONE_TIMED, "Quest Only One Timed" },
    { QUEST_FAILED_REASON_QUEST_ALREADY_ON, "Quest Already On" },
    { QUEST_FAILED_REASON_QUEST_FAILED_DUPLICATE_ITEM, "Quest Failed Duplicate Item" },
    { QUEST_FAILED_REASON_QUEST_FAILED_MISSING_ITEMS, "Quest Failed Missing Items" },
    { QUEST_FAILED_REASON_QUEST_FAILED_NOT_ENOUGH_MONEY, "Quest Failed Not Enough Money" },
    { 0, NULL }
};

typedef enum {
    QUEST_COMPLETABLE_NOT_COMPLETABLE = 0x0,
    QUEST_COMPLETABLE_COMPLETEABLE = 0x3,
} e_quest_completable;
static const value_string e_quest_completable_strings[] =  {
    { QUEST_COMPLETABLE_NOT_COMPLETABLE, "Not Completable" },
    { QUEST_COMPLETABLE_COMPLETEABLE, "Completeable" },
    { 0, NULL }
};

typedef enum {
    QUEST_GIVER_STATUS_NONE = 0x0,
    QUEST_GIVER_STATUS_UNAVAILABLE = 0x1,
    QUEST_GIVER_STATUS_CHAT = 0x2,
    QUEST_GIVER_STATUS_INCOMPLETE = 0x3,
    QUEST_GIVER_STATUS_REWARD_REP = 0x4,
    QUEST_GIVER_STATUS_AVAILABLE = 0x5,
    QUEST_GIVER_STATUS_REWARD_OLD = 0x6,
    QUEST_GIVER_STATUS_REWARD2 = 0x7,
} e_quest_giver_status;
static const value_string e_quest_giver_status_strings[] =  {
    { QUEST_GIVER_STATUS_NONE, "None" },
    { QUEST_GIVER_STATUS_UNAVAILABLE, "Unavailable" },
    { QUEST_GIVER_STATUS_CHAT, "Chat" },
    { QUEST_GIVER_STATUS_INCOMPLETE, "Incomplete" },
    { QUEST_GIVER_STATUS_REWARD_REP, "Reward Rep" },
    { QUEST_GIVER_STATUS_AVAILABLE, "Available" },
    { QUEST_GIVER_STATUS_REWARD_OLD, "Reward Old" },
    { QUEST_GIVER_STATUS_REWARD2, "Reward2" },
    { 0, NULL }
};

typedef enum {
    RAID_GROUP_ERROR_REQUIRED = 0x1,
    RAID_GROUP_ERROR_FULL = 0x2,
} e_raid_group_error;
static const value_string e_raid_group_error_strings[] =  {
    { RAID_GROUP_ERROR_REQUIRED, "Required" },
    { RAID_GROUP_ERROR_FULL, "Full" },
    { 0, NULL }
};

typedef enum {
    RAID_INSTANCE_MESSAGE_WARNING_HOURS = 0x1,
    RAID_INSTANCE_MESSAGE_WARNING_MIN = 0x2,
    RAID_INSTANCE_MESSAGE_WARNING_MIN_SOON = 0x3,
    RAID_INSTANCE_MESSAGE_WELCOME = 0x4,
} e_raid_instance_message;
static const value_string e_raid_instance_message_strings[] =  {
    { RAID_INSTANCE_MESSAGE_WARNING_HOURS, "Warning Hours" },
    { RAID_INSTANCE_MESSAGE_WARNING_MIN, "Warning Min" },
    { RAID_INSTANCE_MESSAGE_WARNING_MIN_SOON, "Warning Min Soon" },
    { RAID_INSTANCE_MESSAGE_WELCOME, "Welcome" },
    { 0, NULL }
};

typedef enum {
    SELL_ITEM_RESULT_CANT_FIND_ITEM = 0x1,
    SELL_ITEM_RESULT_CANT_SELL_ITEM = 0x2,
    SELL_ITEM_RESULT_CANT_FIND_VENDOR = 0x3,
    SELL_ITEM_RESULT_YOU_DONT_OWN_THAT_ITEM = 0x4,
    SELL_ITEM_RESULT_UNK = 0x5,
    SELL_ITEM_RESULT_ONLY_EMPTY_BAG = 0x6,
} e_sell_item_result;
static const value_string e_sell_item_result_strings[] =  {
    { SELL_ITEM_RESULT_CANT_FIND_ITEM, "Cant Find Item" },
    { SELL_ITEM_RESULT_CANT_SELL_ITEM, "Cant Sell Item" },
    { SELL_ITEM_RESULT_CANT_FIND_VENDOR, "Cant Find Vendor" },
    { SELL_ITEM_RESULT_YOU_DONT_OWN_THAT_ITEM, "You Dont Own That Item" },
    { SELL_ITEM_RESULT_UNK, "Unk" },
    { SELL_ITEM_RESULT_ONLY_EMPTY_BAG, "Only Empty Bag" },
    { 0, NULL }
};

typedef enum {
    MAIL_ACTION_SEND = 0x0,
    MAIL_ACTION_MONEY_TAKEN = 0x1,
    MAIL_ACTION_ITEM_TAKEN = 0x2,
    MAIL_ACTION_RETURNED_TO_SENDER = 0x3,
    MAIL_ACTION_DELETED = 0x4,
    MAIL_ACTION_MADE_PERMANENT = 0x5,
} e_mail_action;
static const value_string e_mail_action_strings[] =  {
    { MAIL_ACTION_SEND, "Send" },
    { MAIL_ACTION_MONEY_TAKEN, "Money Taken" },
    { MAIL_ACTION_ITEM_TAKEN, "Item Taken" },
    { MAIL_ACTION_RETURNED_TO_SENDER, "Returned To Sender" },
    { MAIL_ACTION_DELETED, "Deleted" },
    { MAIL_ACTION_MADE_PERMANENT, "Made Permanent" },
    { 0, NULL }
};

typedef enum {
    MAIL_RESULT_OK = 0x00,
    MAIL_RESULT_ERR_EQUIP_ERROR = 0x01,
    MAIL_RESULT_ERR_CANNOT_SEND_TO_SELF = 0x02,
    MAIL_RESULT_ERR_NOT_ENOUGH_MONEY = 0x03,
    MAIL_RESULT_ERR_RECIPIENT_NOT_FOUND = 0x04,
    MAIL_RESULT_ERR_NOT_YOUR_TEAM = 0x05,
    MAIL_RESULT_ERR_INTERNAL_ERROR = 0x06,
    MAIL_RESULT_ERR_DISABLED_FOR_TRIAL_ACC = 0x0E,
    MAIL_RESULT_ERR_RECIPIENT_CAP_REACHED = 0x0F,
    MAIL_RESULT_ERR_CANT_SEND_WRAPPED_COD = 0x10,
    MAIL_RESULT_ERR_MAIL_AND_CHAT_SUSPENDED = 0x11,
    MAIL_RESULT_ERR_TOO_MANY_ATTACHMENTS = 0x12,
    MAIL_RESULT_ERR_MAIL_ATTACHMENT_INVALID = 0x13,
} e_mail_result;
static const value_string e_mail_result_strings[] =  {
    { MAIL_RESULT_OK, "Ok" },
    { MAIL_RESULT_ERR_EQUIP_ERROR, "Err Equip Error" },
    { MAIL_RESULT_ERR_CANNOT_SEND_TO_SELF, "Err Cannot Send To Self" },
    { MAIL_RESULT_ERR_NOT_ENOUGH_MONEY, "Err Not Enough Money" },
    { MAIL_RESULT_ERR_RECIPIENT_NOT_FOUND, "Err Recipient Not Found" },
    { MAIL_RESULT_ERR_NOT_YOUR_TEAM, "Err Not Your Team" },
    { MAIL_RESULT_ERR_INTERNAL_ERROR, "Err Internal Error" },
    { MAIL_RESULT_ERR_DISABLED_FOR_TRIAL_ACC, "Err Disabled For Trial Acc" },
    { MAIL_RESULT_ERR_RECIPIENT_CAP_REACHED, "Err Recipient Cap Reached" },
    { MAIL_RESULT_ERR_CANT_SEND_WRAPPED_COD, "Err Cant Send Wrapped Cod" },
    { MAIL_RESULT_ERR_MAIL_AND_CHAT_SUSPENDED, "Err Mail And Chat Suspended" },
    { MAIL_RESULT_ERR_TOO_MANY_ATTACHMENTS, "Err Too Many Attachments" },
    { MAIL_RESULT_ERR_MAIL_ATTACHMENT_INVALID, "Err Mail Attachment Invalid" },
    { 0, NULL }
};

typedef enum {
    MAIL_RESULT_TWO_OK = 0x00,
    MAIL_RESULT_TWO_ERR_EQUIP_ERROR = 0x01,
    MAIL_RESULT_TWO_ERR_CANNOT_SEND_TO_SELF = 0x02,
    MAIL_RESULT_TWO_ERR_NOT_ENOUGH_MONEY = 0x03,
    MAIL_RESULT_TWO_ERR_RECIPIENT_NOT_FOUND = 0x04,
    MAIL_RESULT_TWO_ERR_NOT_YOUR_TEAM = 0x05,
    MAIL_RESULT_TWO_ERR_INTERNAL_ERROR = 0x06,
    MAIL_RESULT_TWO_ERR_DISABLED_FOR_TRIAL_ACC = 0x0E,
    MAIL_RESULT_TWO_ERR_RECIPIENT_CAP_REACHED = 0x0F,
    MAIL_RESULT_TWO_ERR_CANT_SEND_WRAPPED_COD = 0x10,
    MAIL_RESULT_TWO_ERR_MAIL_AND_CHAT_SUSPENDED = 0x11,
    MAIL_RESULT_TWO_ERR_TOO_MANY_ATTACHMENTS = 0x12,
    MAIL_RESULT_TWO_ERR_MAIL_ATTACHMENT_INVALID = 0x13,
} e_mail_result_two;
static const value_string e_mail_result_two_strings[] =  {
    { MAIL_RESULT_TWO_OK, "Ok" },
    { MAIL_RESULT_TWO_ERR_EQUIP_ERROR, "Err Equip Error" },
    { MAIL_RESULT_TWO_ERR_CANNOT_SEND_TO_SELF, "Err Cannot Send To Self" },
    { MAIL_RESULT_TWO_ERR_NOT_ENOUGH_MONEY, "Err Not Enough Money" },
    { MAIL_RESULT_TWO_ERR_RECIPIENT_NOT_FOUND, "Err Recipient Not Found" },
    { MAIL_RESULT_TWO_ERR_NOT_YOUR_TEAM, "Err Not Your Team" },
    { MAIL_RESULT_TWO_ERR_INTERNAL_ERROR, "Err Internal Error" },
    { MAIL_RESULT_TWO_ERR_DISABLED_FOR_TRIAL_ACC, "Err Disabled For Trial Acc" },
    { MAIL_RESULT_TWO_ERR_RECIPIENT_CAP_REACHED, "Err Recipient Cap Reached" },
    { MAIL_RESULT_TWO_ERR_CANT_SEND_WRAPPED_COD, "Err Cant Send Wrapped Cod" },
    { MAIL_RESULT_TWO_ERR_MAIL_AND_CHAT_SUSPENDED, "Err Mail And Chat Suspended" },
    { MAIL_RESULT_TWO_ERR_TOO_MANY_ATTACHMENTS, "Err Too Many Attachments" },
    { MAIL_RESULT_TWO_ERR_MAIL_ATTACHMENT_INVALID, "Err Mail Attachment Invalid" },
    { 0, NULL }
};

typedef enum {
    SERVER_MESSAGE_TYPE_SHUTDOWN_TIME = 0x1,
    SERVER_MESSAGE_TYPE_RESTART_TIME = 0x2,
    SERVER_MESSAGE_TYPE_CUSTOM = 0x3,
    SERVER_MESSAGE_TYPE_SHUTDOWN_CANCELLED = 0x4,
    SERVER_MESSAGE_TYPE_RESTART_CANCELLED = 0x5,
} e_server_message_type;
static const value_string e_server_message_type_strings[] =  {
    { SERVER_MESSAGE_TYPE_SHUTDOWN_TIME, "Shutdown Time" },
    { SERVER_MESSAGE_TYPE_RESTART_TIME, "Restart Time" },
    { SERVER_MESSAGE_TYPE_CUSTOM, "Custom" },
    { SERVER_MESSAGE_TYPE_SHUTDOWN_CANCELLED, "Shutdown Cancelled" },
    { SERVER_MESSAGE_TYPE_RESTART_CANCELLED, "Restart Cancelled" },
    { 0, NULL }
};

typedef enum {
    ITEM_CLASS_CONSUMABLE = 0x0,
    ITEM_CLASS_CONTAINER = 0x1,
    ITEM_CLASS_WEAPON = 0x2,
    ITEM_CLASS_GEM = 0x3,
    ITEM_CLASS_ARMOR = 0x4,
    ITEM_CLASS_REAGENT = 0x5,
    ITEM_CLASS_PROJECTILE = 0x6,
    ITEM_CLASS_TRADE_GOODS = 0x7,
    ITEM_CLASS_GENERIC = 0x8,
    ITEM_CLASS_RECIPE = 0x9,
    ITEM_CLASS_MONEY = 0xA,
    ITEM_CLASS_QUIVER = 0xB,
    ITEM_CLASS_QUEST = 0xC,
    ITEM_CLASS_KEY = 0xD,
    ITEM_CLASS_PERMANENT = 0xE,
    ITEM_CLASS_MISC = 0xF,
} e_item_class;
static const value_string e_item_class_strings[] =  {
    { ITEM_CLASS_CONSUMABLE, "Consumable" },
    { ITEM_CLASS_CONTAINER, "Container" },
    { ITEM_CLASS_WEAPON, "Weapon" },
    { ITEM_CLASS_GEM, "Gem" },
    { ITEM_CLASS_ARMOR, "Armor" },
    { ITEM_CLASS_REAGENT, "Reagent" },
    { ITEM_CLASS_PROJECTILE, "Projectile" },
    { ITEM_CLASS_TRADE_GOODS, "Trade Goods" },
    { ITEM_CLASS_GENERIC, "Generic" },
    { ITEM_CLASS_RECIPE, "Recipe" },
    { ITEM_CLASS_MONEY, "Money" },
    { ITEM_CLASS_QUIVER, "Quiver" },
    { ITEM_CLASS_QUEST, "Quest" },
    { ITEM_CLASS_KEY, "Key" },
    { ITEM_CLASS_PERMANENT, "Permanent" },
    { ITEM_CLASS_MISC, "Misc" },
    { 0, NULL }
};

typedef enum {
    STABLE_RESULT_ERR_MONEY = 0x1,
    STABLE_RESULT_ERR_STABLE = 0x6,
    STABLE_RESULT_SUCCESS_STABLE = 0x8,
    STABLE_RESULT_SUCCESS_UNSTABLE = 0x9,
    STABLE_RESULT_SUCCESS_BUY_SLOT = 0xA,
} e_stable_result;
static const value_string e_stable_result_strings[] =  {
    { STABLE_RESULT_ERR_MONEY, "Err Money" },
    { STABLE_RESULT_ERR_STABLE, "Err Stable" },
    { STABLE_RESULT_SUCCESS_STABLE, "Success Stable" },
    { STABLE_RESULT_SUCCESS_UNSTABLE, "Success Unstable" },
    { STABLE_RESULT_SUCCESS_BUY_SLOT, "Success Buy Slot" },
    { 0, NULL }
};

typedef enum {
    TRADE_STATUS_BUSY = 0x00,
    TRADE_STATUS_BEGIN_TRADE = 0x01,
    TRADE_STATUS_OPEN_WINDOW = 0x02,
    TRADE_STATUS_TRADE_CANCELED = 0x03,
    TRADE_STATUS_TRADE_ACCEPT = 0x04,
    TRADE_STATUS_BUSY_2 = 0x05,
    TRADE_STATUS_NO_TARGET = 0x06,
    TRADE_STATUS_BACK_TO_TRADE = 0x07,
    TRADE_STATUS_TRADE_COMPLETE = 0x08,
    TRADE_STATUS_TRADE_REJECTED = 0x09,
    TRADE_STATUS_TARGET_TO_FAR = 0x0A,
    TRADE_STATUS_WRONG_FACTION = 0x0B,
    TRADE_STATUS_CLOSE_WINDOW = 0x0C,
    TRADE_STATUS_UNKNOWN_13 = 0x0D,
    TRADE_STATUS_IGNORE_YOU = 0x0E,
    TRADE_STATUS_YOU_STUNNED = 0x0F,
    TRADE_STATUS_TARGET_STUNNED = 0x10,
    TRADE_STATUS_YOU_DEAD = 0x11,
    TRADE_STATUS_TARGET_DEAD = 0x12,
    TRADE_STATUS_YOU_LOGOUT = 0x13,
    TRADE_STATUS_TARGET_LOGOUT = 0x14,
    TRADE_STATUS_TRIAL_ACCOUNT = 0x15,
    TRADE_STATUS_ONLY_CONJURED = 0x16,
    TRADE_STATUS_NOT_ON_TAPLIST = 0x17,
} e_trade_status;
static const value_string e_trade_status_strings[] =  {
    { TRADE_STATUS_BUSY, "Busy" },
    { TRADE_STATUS_BEGIN_TRADE, "Begin Trade" },
    { TRADE_STATUS_OPEN_WINDOW, "Open Window" },
    { TRADE_STATUS_TRADE_CANCELED, "Trade Canceled" },
    { TRADE_STATUS_TRADE_ACCEPT, "Trade Accept" },
    { TRADE_STATUS_BUSY_2, "Busy 2" },
    { TRADE_STATUS_NO_TARGET, "No Target" },
    { TRADE_STATUS_BACK_TO_TRADE, "Back To Trade" },
    { TRADE_STATUS_TRADE_COMPLETE, "Trade Complete" },
    { TRADE_STATUS_TRADE_REJECTED, "Trade Rejected" },
    { TRADE_STATUS_TARGET_TO_FAR, "Target To Far" },
    { TRADE_STATUS_WRONG_FACTION, "Wrong Faction" },
    { TRADE_STATUS_CLOSE_WINDOW, "Close Window" },
    { TRADE_STATUS_UNKNOWN_13, "Unknown 13" },
    { TRADE_STATUS_IGNORE_YOU, "Ignore You" },
    { TRADE_STATUS_YOU_STUNNED, "You Stunned" },
    { TRADE_STATUS_TARGET_STUNNED, "Target Stunned" },
    { TRADE_STATUS_YOU_DEAD, "You Dead" },
    { TRADE_STATUS_TARGET_DEAD, "Target Dead" },
    { TRADE_STATUS_YOU_LOGOUT, "You Logout" },
    { TRADE_STATUS_TARGET_LOGOUT, "Target Logout" },
    { TRADE_STATUS_TRIAL_ACCOUNT, "Trial Account" },
    { TRADE_STATUS_ONLY_CONJURED, "Only Conjured" },
    { TRADE_STATUS_NOT_ON_TAPLIST, "Not On Taplist" },
    { 0, NULL }
};

typedef enum {
    TRAINING_FAILURE_REASON_UNAVAILABLE = 0x0,
    TRAINING_FAILURE_REASON_NOT_ENOUGH_MONEY = 0x1,
    TRAINING_FAILURE_REASON_NOT_ENOUGH_SKILL = 0x2,
} e_training_failure_reason;
static const value_string e_training_failure_reason_strings[] =  {
    { TRAINING_FAILURE_REASON_UNAVAILABLE, "Unavailable" },
    { TRAINING_FAILURE_REASON_NOT_ENOUGH_MONEY, "Not Enough Money" },
    { TRAINING_FAILURE_REASON_NOT_ENOUGH_SKILL, "Not Enough Skill" },
    { 0, NULL }
};

typedef enum {
    TRANSFER_ABORT_REASON_NONE = 0x0,
    TRANSFER_ABORT_REASON_IS_FULL = 0x1,
    TRANSFER_ABORT_REASON_NOT_FOUND = 0x2,
    TRANSFER_ABORT_REASON_TOO_MANY_INSTANCES = 0x3,
    TRANSFER_ABORT_REASON_ZONE_IS_IN_COMBAT = 0x5,
} e_transfer_abort_reason;
static const value_string e_transfer_abort_reason_strings[] =  {
    { TRANSFER_ABORT_REASON_NONE, "None" },
    { TRANSFER_ABORT_REASON_IS_FULL, "Is Full" },
    { TRANSFER_ABORT_REASON_NOT_FOUND, "Not Found" },
    { TRANSFER_ABORT_REASON_TOO_MANY_INSTANCES, "Too Many Instances" },
    { TRANSFER_ABORT_REASON_ZONE_IS_IN_COMBAT, "Zone Is In Combat" },
    { 0, NULL }
};

typedef enum {
    CINEMATIC_SEQUENCE_ID_GOBLIN = 0x00,
    CINEMATIC_SEQUENCE_ID_UNDEAD = 0x02,
    CINEMATIC_SEQUENCE_ID_ORC = 0x15,
    CINEMATIC_SEQUENCE_ID_DWARF = 0x29,
    CINEMATIC_SEQUENCE_ID_NIGHT_ELF = 0x3D,
    CINEMATIC_SEQUENCE_ID_HUMAN = 0x51,
    CINEMATIC_SEQUENCE_ID_GNOME = 0x65,
    CINEMATIC_SEQUENCE_ID_TROLL = 0x79,
    CINEMATIC_SEQUENCE_ID_TAUREN = 0x8D,
} e_cinematic_sequence_id;
static const value_string e_cinematic_sequence_id_strings[] =  {
    { CINEMATIC_SEQUENCE_ID_GOBLIN, "Goblin" },
    { CINEMATIC_SEQUENCE_ID_UNDEAD, "Undead" },
    { CINEMATIC_SEQUENCE_ID_ORC, "Orc" },
    { CINEMATIC_SEQUENCE_ID_DWARF, "Dwarf" },
    { CINEMATIC_SEQUENCE_ID_NIGHT_ELF, "Night Elf" },
    { CINEMATIC_SEQUENCE_ID_HUMAN, "Human" },
    { CINEMATIC_SEQUENCE_ID_GNOME, "Gnome" },
    { CINEMATIC_SEQUENCE_ID_TROLL, "Troll" },
    { CINEMATIC_SEQUENCE_ID_TAUREN, "Tauren" },
    { 0, NULL }
};

typedef enum {
    WEATHER_TYPE_FINE = 0x0,
    WEATHER_TYPE_RAIN = 0x1,
    WEATHER_TYPE_SNOW = 0x2,
    WEATHER_TYPE_STORM = 0x3,
} e_weather_type;
static const value_string e_weather_type_strings[] =  {
    { WEATHER_TYPE_FINE, "Fine" },
    { WEATHER_TYPE_RAIN, "Rain" },
    { WEATHER_TYPE_SNOW, "Snow" },
    { WEATHER_TYPE_STORM, "Storm" },
    { 0, NULL }
};

typedef enum {
    WEATHER_CHANGE_TYPE_SMOOTH = 0x0,
    WEATHER_CHANGE_TYPE_INSTANT = 0x1,
} e_weather_change_type;
static const value_string e_weather_change_type_strings[] =  {
    { WEATHER_CHANGE_TYPE_SMOOTH, "Smooth" },
    { WEATHER_CHANGE_TYPE_INSTANT, "Instant" },
    { 0, NULL }
};


typedef enum {
    CHANNEL_MEMBER_FLAGS_NONE = 0x00,
    CHANNEL_MEMBER_FLAGS_OWNER = 0x01,
    CHANNEL_MEMBER_FLAGS_MODERATOR = 0x04,
    CHANNEL_MEMBER_FLAGS_VOICED = 0x08,
    CHANNEL_MEMBER_FLAGS_MUTED = 0x10,
    CHANNEL_MEMBER_FLAGS_CUSTOM = 0x20,
    CHANNEL_MEMBER_FLAGS_MICROPHONE_MUTE = 0x40,
} e_channel_member_flags;

typedef enum {
    CHARACTER_FLAGS_NONE = 0x0000,
    CHARACTER_FLAGS_LOCKED_FOR_TRANSFER = 0x0004,
    CHARACTER_FLAGS_HIDE_HELM = 0x0400,
    CHARACTER_FLAGS_HIDE_CLOAK = 0x0800,
    CHARACTER_FLAGS_GHOST = 0x2000,
    CHARACTER_FLAGS_RENAME = 0x4000,
} e_character_flags;

typedef enum {
    FACTION_FLAG_VISIBLE = 0x01,
    FACTION_FLAG_AT_WAR = 0x02,
    FACTION_FLAG_HIDDEN = 0x04,
    FACTION_FLAG_INVISIBLE_FORCED = 0x08,
    FACTION_FLAG_PEACE_FORCED = 0x10,
    FACTION_FLAG_INACTIVE = 0x20,
    FACTION_FLAG_RIVAL = 0x40,
} e_faction_flag;

typedef enum {
    SPLINE_FLAG_NONE = 0x00000000,
    SPLINE_FLAG_DONE = 0x00000001,
    SPLINE_FLAG_FALLING = 0x00000002,
    SPLINE_FLAG_UNKNOWN3 = 0x00000004,
    SPLINE_FLAG_UNKNOWN4 = 0x00000008,
    SPLINE_FLAG_UNKNOWN5 = 0x00000010,
    SPLINE_FLAG_UNKNOWN6 = 0x00000020,
    SPLINE_FLAG_UNKNOWN7 = 0x00000040,
    SPLINE_FLAG_UNKNOWN8 = 0x00000080,
    SPLINE_FLAG_RUN_MODE = 0x00000100,
    SPLINE_FLAG_FLYING = 0x00000200,
    SPLINE_FLAG_NO_SPLINE = 0x00000400,
    SPLINE_FLAG_UNKNOWN12 = 0x00000800,
    SPLINE_FLAG_UNKNOWN13 = 0x00001000,
    SPLINE_FLAG_UNKNOWN14 = 0x00002000,
    SPLINE_FLAG_UNKNOWN15 = 0x00004000,
    SPLINE_FLAG_UNKNOWN16 = 0x00008000,
    SPLINE_FLAG_FINAL_POINT = 0x00010000,
    SPLINE_FLAG_FINAL_TARGET = 0x00020000,
    SPLINE_FLAG_FINAL_ANGLE = 0x00040000,
    SPLINE_FLAG_UNKNOWN19 = 0x00080000,
    SPLINE_FLAG_CYCLIC = 0x00100000,
    SPLINE_FLAG_ENTER_CYCLE = 0x00200000,
    SPLINE_FLAG_FROZEN = 0x00400000,
    SPLINE_FLAG_UNKNOWN23 = 0x00800000,
    SPLINE_FLAG_UNKNOWN24 = 0x01000000,
    SPLINE_FLAG_UNKNOWN25 = 0x02000000,
    SPLINE_FLAG_UNKNOWN26 = 0x04000000,
    SPLINE_FLAG_UNKNOWN27 = 0x08000000,
    SPLINE_FLAG_UNKNOWN28 = 0x10000000,
    SPLINE_FLAG_UNKNOWN29 = 0x20000000,
    SPLINE_FLAG_UNKNOWN30 = 0x40000000,
    SPLINE_FLAG_UNKNOWN31 = -0x7FFFFFFF,
} e_spline_flag;

typedef enum {
    UPDATE_FLAG_NONE = 0x00,
    UPDATE_FLAG_SELF = 0x01,
    UPDATE_FLAG_TRANSPORT = 0x02,
    UPDATE_FLAG_MELEE_ATTACKING = 0x04,
    UPDATE_FLAG_HIGH_GUID = 0x08,
    UPDATE_FLAG_ALL = 0x10,
    UPDATE_FLAG_LIVING = 0x20,
    UPDATE_FLAG_HAS_POSITION = 0x40,
} e_update_flag;

typedef enum {
    MOVEMENT_FLAGS_NONE = 0x00000000,
    MOVEMENT_FLAGS_FORWARD = 0x00000001,
    MOVEMENT_FLAGS_BACKWARD = 0x00000002,
    MOVEMENT_FLAGS_STRAFE_LEFT = 0x00000004,
    MOVEMENT_FLAGS_STRAFE_RIGHT = 0x00000008,
    MOVEMENT_FLAGS_TURN_LEFT = 0x00000010,
    MOVEMENT_FLAGS_TURN_RIGHT = 0x00000020,
    MOVEMENT_FLAGS_PITCH_UP = 0x00000040,
    MOVEMENT_FLAGS_PITCH_DOWN = 0x00000080,
    MOVEMENT_FLAGS_WALK_MODE = 0x00000100,
    MOVEMENT_FLAGS_ON_TRANSPORT = 0x00000200,
    MOVEMENT_FLAGS_LEVITATING = 0x00000400,
    MOVEMENT_FLAGS_FIXED_Z = 0x00000800,
    MOVEMENT_FLAGS_ROOT = 0x00001000,
    MOVEMENT_FLAGS_JUMPING = 0x00002000,
    MOVEMENT_FLAGS_FALLINGFAR = 0x00004000,
    MOVEMENT_FLAGS_SWIMMING = 0x00200000,
    MOVEMENT_FLAGS_SPLINE_ENABLED = 0x00400000,
    MOVEMENT_FLAGS_CAN_FLY = 0x00800000,
    MOVEMENT_FLAGS_FLYING = 0x01000000,
    MOVEMENT_FLAGS_ONTRANSPORT = 0x02000000,
    MOVEMENT_FLAGS_SPLINE_ELEVATION = 0x04000000,
    MOVEMENT_FLAGS_WATERWALKING = 0x10000000,
    MOVEMENT_FLAGS_SAFE_FALL = 0x20000000,
    MOVEMENT_FLAGS_HOVER = 0x40000000,
} e_movement_flags;

typedef enum {
    SPELL_CAST_TARGET_FLAGS_SELF = 0x0000,
    SPELL_CAST_TARGET_FLAGS_UNUSED1 = 0x0001,
    SPELL_CAST_TARGET_FLAGS_UNIT = 0x0002,
    SPELL_CAST_TARGET_FLAGS_UNUSED2 = 0x0004,
    SPELL_CAST_TARGET_FLAGS_UNUSED3 = 0x0008,
    SPELL_CAST_TARGET_FLAGS_ITEM = 0x0010,
    SPELL_CAST_TARGET_FLAGS_SOURCE_LOCATION = 0x0020,
    SPELL_CAST_TARGET_FLAGS_DEST_LOCATION = 0x0040,
    SPELL_CAST_TARGET_FLAGS_OBJECT_UNK = 0x0080,
    SPELL_CAST_TARGET_FLAGS_UNIT_UNK = 0x0100,
    SPELL_CAST_TARGET_FLAGS_PVP_CORPSE = 0x0200,
    SPELL_CAST_TARGET_FLAGS_UNIT_CORPSE = 0x0400,
    SPELL_CAST_TARGET_FLAGS_GAMEOBJECT = 0x0800,
    SPELL_CAST_TARGET_FLAGS_TRADE_ITEM = 0x1000,
    SPELL_CAST_TARGET_FLAGS_STRING = 0x2000,
    SPELL_CAST_TARGET_FLAGS_UNK1 = 0x4000,
    SPELL_CAST_TARGET_FLAGS_CORPSE = 0x8000,
} e_spell_cast_target_flags;

typedef enum {
    CHANNEL_FLAGS_NONE = 0x00,
    CHANNEL_FLAGS_CUSTOM = 0x01,
    CHANNEL_FLAGS_TRADE = 0x04,
    CHANNEL_FLAGS_NOT_LFG = 0x08,
    CHANNEL_FLAGS_GENERAL = 0x10,
    CHANNEL_FLAGS_CITY = 0x20,
    CHANNEL_FLAGS_LFG = 0x40,
    CHANNEL_FLAGS_VOICE = 0x80,
} e_channel_flags;

typedef enum {
    ITEM_FLAG_NONE = 0x00000,
    ITEM_FLAG_NO_PICKUP = 0x00001,
    ITEM_FLAG_CONJURED = 0x00002,
    ITEM_FLAG_LOOTABLE = 0x00004,
    ITEM_FLAG_DEPRECATED = 0x00010,
    ITEM_FLAG_INDESTRUCTIBLE = 0x00020,
    ITEM_FLAG_PLAYER_CAST = 0x00040,
    ITEM_FLAG_NO_EQUIP_COOLDOWN = 0x00080,
    ITEM_FLAG_INT_BONUS_INSTEAD = 0x00100,
    ITEM_FLAG_WRAPPER = 0x00200,
    ITEM_FLAG_IGNORE_BAG_SPACE = 0x00400,
    ITEM_FLAG_PARTY_LOOT = 0x00800,
    ITEM_FLAG_CHARTER = 0x02000,
    ITEM_FLAG_HAS_TEXT = 0x04000,
    ITEM_FLAG_NO_DISENCHANT = 0x08000,
    ITEM_FLAG_REAL_DURATION = 0x10000,
    ITEM_FLAG_NO_CREATOR = 0x20000,
} e_item_flag;

typedef enum {
    ALLOWED_CLASS_ALL = 0x000,
    ALLOWED_CLASS_WARRIOR = 0x001,
    ALLOWED_CLASS_PALADIN = 0x002,
    ALLOWED_CLASS_HUNTER = 0x004,
    ALLOWED_CLASS_ROGUE = 0x008,
    ALLOWED_CLASS_PRIEST = 0x010,
    ALLOWED_CLASS_SHAMAN = 0x040,
    ALLOWED_CLASS_MAGE = 0x080,
    ALLOWED_CLASS_WARLOCK = 0x100,
    ALLOWED_CLASS_DRUID = 0x400,
} e_allowed_class;

typedef enum {
    ALLOWED_RACE_ALL = 0x000,
    ALLOWED_RACE_HUMAN = 0x001,
    ALLOWED_RACE_ORC = 0x002,
    ALLOWED_RACE_DWARF = 0x004,
    ALLOWED_RACE_NIGHT_ELF = 0x008,
    ALLOWED_RACE_UNDEAD = 0x010,
    ALLOWED_RACE_TAUREN = 0x020,
    ALLOWED_RACE_GNOME = 0x040,
    ALLOWED_RACE_TROLL = 0x080,
    ALLOWED_RACE_GOBLIN = 0x100,
} e_allowed_race;

typedef enum {
    GROUP_UPDATE_FLAGS_NONE = 0x00000000,
    GROUP_UPDATE_FLAGS_STATUS = 0x00000001,
    GROUP_UPDATE_FLAGS_CUR_HP = 0x00000002,
    GROUP_UPDATE_FLAGS_MAX_HP = 0x00000004,
    GROUP_UPDATE_FLAGS_POWER_TYPE = 0x00000008,
    GROUP_UPDATE_FLAGS_CUR_POWER = 0x00000010,
    GROUP_UPDATE_FLAGS_MAX_POWER = 0x00000020,
    GROUP_UPDATE_FLAGS_LEVEL = 0x00000040,
    GROUP_UPDATE_FLAGS_ZONE = 0x00000080,
    GROUP_UPDATE_FLAGS_POSITION = 0x00000100,
    GROUP_UPDATE_FLAGS_AURAS = 0x00000200,
    GROUP_UPDATE_FLAGS_AURAS_2 = 0x00000400,
    GROUP_UPDATE_FLAGS_PET_GUID = 0x00000800,
    GROUP_UPDATE_FLAGS_PET_NAME = 0x00001000,
    GROUP_UPDATE_FLAGS_PET_MODEL_ID = 0x00002000,
    GROUP_UPDATE_FLAGS_PET_CUR_HP = 0x00004000,
    GROUP_UPDATE_FLAGS_PET_MAX_HP = 0x00008000,
    GROUP_UPDATE_FLAGS_PET_POWER_TYPE = 0x00010000,
    GROUP_UPDATE_FLAGS_PET_CUR_POWER = 0x00020000,
    GROUP_UPDATE_FLAGS_PET_MAX_POWER = 0x00040000,
    GROUP_UPDATE_FLAGS_PET_AURAS = 0x00080000,
    GROUP_UPDATE_FLAGS_PET_AURAS_2 = 0x00100000,
    GROUP_UPDATE_FLAGS_MODE_OFFLINE = 0x10000000,
} e_group_update_flags;

typedef enum {
    GROUP_MEMBER_ONLINE_STATUS_OFFLINE = 0x00,
    GROUP_MEMBER_ONLINE_STATUS_ONLINE = 0x01,
    GROUP_MEMBER_ONLINE_STATUS_PVP = 0x02,
    GROUP_MEMBER_ONLINE_STATUS_DEAD = 0x04,
    GROUP_MEMBER_ONLINE_STATUS_GHOST = 0x08,
    GROUP_MEMBER_ONLINE_STATUS_PVP_FFA = 0x10,
    GROUP_MEMBER_ONLINE_STATUS_ZONE_OUT = 0x20,
    GROUP_MEMBER_ONLINE_STATUS_AFK = 0x40,
    GROUP_MEMBER_ONLINE_STATUS_DND = 0x80,
} e_group_member_online_status;

typedef enum {
    CAST_FLAGS_NONE = 0x000,
    CAST_FLAGS_HIDDEN_COMBATLOG = 0x001,
    CAST_FLAGS_UNKNOWN2 = 0x002,
    CAST_FLAGS_UNKNOWN3 = 0x004,
    CAST_FLAGS_UNKNOWN4 = 0x008,
    CAST_FLAGS_UNKNOWN5 = 0x010,
    CAST_FLAGS_AMMO = 0x020,
    CAST_FLAGS_UNKNOWN7 = 0x040,
    CAST_FLAGS_UNKNOWN8 = 0x080,
    CAST_FLAGS_UNKNOWN9 = 0x100,
} e_cast_flags;

/* AUTOGENERATED_END_ENUM */

// All existing opcodes for 1.12.x
typedef enum
{
    MSG_NULL_ACTION                                 = 0x000,
    CMSG_BOOTME                                     = 0x001,
    CMSG_DBLOOKUP                                   = 0x002,
    SMSG_DBLOOKUP                                   = 0x003,
    CMSG_QUERY_OBJECT_POSITION                      = 0x004,
    SMSG_QUERY_OBJECT_POSITION                      = 0x005,
    CMSG_QUERY_OBJECT_ROTATION                      = 0x006,
    SMSG_QUERY_OBJECT_ROTATION                      = 0x007,
    CMSG_WORLD_TELEPORT                             = 0x008,
    CMSG_TELEPORT_TO_UNIT                           = 0x009,
    CMSG_ZONE_MAP                                   = 0x00A,
    SMSG_ZONE_MAP                                   = 0x00B,
    CMSG_DEBUG_CHANGECELLZONE                       = 0x00C,
    CMSG_EMBLAZON_TABARD_OBSOLETE                   = 0x00D,
    CMSG_UNEMBLAZON_TABARD_OBSOLETE                 = 0x00E,
    CMSG_RECHARGE                                   = 0x00F,
    CMSG_LEARN_SPELL                                = 0x010,
    CMSG_CREATEMONSTER                              = 0x011,
    CMSG_DESTROYMONSTER                             = 0x012,
    CMSG_CREATEITEM                                 = 0x013,
    CMSG_CREATEGAMEOBJECT                           = 0x014,
    SMSG_CHECK_FOR_BOTS                             = 0x015,
    CMSG_MAKEMONSTERATTACKGUID                      = 0x016,
    CMSG_BOT_DETECTED2                              = 0x017,
    CMSG_FORCEACTION                                = 0x018,
    CMSG_FORCEACTIONONOTHER                         = 0x019,
    CMSG_FORCEACTIONSHOW                            = 0x01A,
    SMSG_FORCEACTIONSHOW                            = 0x01B,
    CMSG_PETGODMODE                                 = 0x01C,
    SMSG_PETGODMODE                                 = 0x01D,
    SMSG_DEBUGINFOSPELLMISS_OBSOLETE                = 0x01E,
    CMSG_WEATHER_SPEED_CHEAT                        = 0x01F,
    CMSG_UNDRESSPLAYER                              = 0x020,
    CMSG_BEASTMASTER                                = 0x021,
    CMSG_GODMODE                                    = 0x022,
    SMSG_GODMODE                                    = 0x023,
    CMSG_CHEAT_SETMONEY                             = 0x024,
    CMSG_LEVEL_CHEAT                                = 0x025,
    CMSG_PET_LEVEL_CHEAT                            = 0x026,
    CMSG_SET_WORLDSTATE                             = 0x027,
    CMSG_COOLDOWN_CHEAT                             = 0x028,
    CMSG_USE_SKILL_CHEAT                            = 0x029,
    CMSG_FLAG_QUEST                                 = 0x02A,
    CMSG_FLAG_QUEST_FINISH                          = 0x02B,
    CMSG_CLEAR_QUEST                                = 0x02C,
    CMSG_SEND_EVENT                                 = 0x02D,
    CMSG_DEBUG_AISTATE                              = 0x02E,
    SMSG_DEBUG_AISTATE                              = 0x02F,
    CMSG_DISABLE_PVP_CHEAT                          = 0x030,
    CMSG_ADVANCE_SPAWN_TIME                         = 0x031,
    CMSG_PVP_PORT_OBSOLETE                          = 0x032,
    CMSG_AUTH_SRP6_BEGIN                            = 0x033,
    CMSG_AUTH_SRP6_PROOF                            = 0x034,
    CMSG_AUTH_SRP6_RECODE                           = 0x035,
    CMSG_CHAR_CREATE                                = 0x036,
    CMSG_CHAR_ENUM                                  = 0x037,
    CMSG_CHAR_DELETE                                = 0x038,
    SMSG_AUTH_SRP6_RESPONSE                         = 0x039,
    SMSG_CHAR_CREATE                                = 0x03A,
    SMSG_CHAR_ENUM                                  = 0x03B,
    SMSG_CHAR_DELETE                                = 0x03C,
    CMSG_PLAYER_LOGIN                               = 0x03D,
    SMSG_NEW_WORLD                                  = 0x03E,
    SMSG_TRANSFER_PENDING                           = 0x03F,
    SMSG_TRANSFER_ABORTED                           = 0x040,
    SMSG_CHARACTER_LOGIN_FAILED                     = 0x041,
    SMSG_LOGIN_SETTIMESPEED                         = 0x042,
    SMSG_GAMETIME_UPDATE                            = 0x043,
    CMSG_GAMETIME_SET                               = 0x044,
    SMSG_GAMETIME_SET                               = 0x045,
    CMSG_GAMESPEED_SET                              = 0x046,
    SMSG_GAMESPEED_SET                              = 0x047,
    CMSG_SERVERTIME                                 = 0x048,
    SMSG_SERVERTIME                                 = 0x049,
    CMSG_PLAYER_LOGOUT                              = 0x04A,
    CMSG_LOGOUT_REQUEST                             = 0x04B,
    SMSG_LOGOUT_RESPONSE                            = 0x04C,
    SMSG_LOGOUT_COMPLETE                            = 0x04D,
    CMSG_LOGOUT_CANCEL                              = 0x04E,
    SMSG_LOGOUT_CANCEL_ACK                          = 0x04F,
    CMSG_NAME_QUERY                                 = 0x050,
    SMSG_NAME_QUERY_RESPONSE                        = 0x051,
    CMSG_PET_NAME_QUERY                             = 0x052,
    SMSG_PET_NAME_QUERY_RESPONSE                    = 0x053,
    CMSG_GUILD_QUERY                                = 0x054,
    SMSG_GUILD_QUERY_RESPONSE                       = 0x055,
    CMSG_ITEM_QUERY_SINGLE                          = 0x056,
    CMSG_ITEM_QUERY_MULTIPLE                        = 0x057,
    SMSG_ITEM_QUERY_SINGLE_RESPONSE                 = 0x058,
    SMSG_ITEM_QUERY_MULTIPLE_RESPONSE               = 0x059,
    CMSG_PAGE_TEXT_QUERY                            = 0x05A,
    SMSG_PAGE_TEXT_QUERY_RESPONSE                   = 0x05B,
    CMSG_QUEST_QUERY                                = 0x05C,
    SMSG_QUEST_QUERY_RESPONSE                       = 0x05D,
    CMSG_GAMEOBJECT_QUERY                           = 0x05E,
    SMSG_GAMEOBJECT_QUERY_RESPONSE                  = 0x05F,
    CMSG_CREATURE_QUERY                             = 0x060,
    SMSG_CREATURE_QUERY_RESPONSE                    = 0x061,
    CMSG_WHO                                        = 0x062,
    SMSG_WHO                                        = 0x063,
    CMSG_WHOIS                                      = 0x064,
    SMSG_WHOIS                                      = 0x065,
    CMSG_FRIEND_LIST                                = 0x066,
    SMSG_FRIEND_LIST                                = 0x067,
    SMSG_FRIEND_STATUS                              = 0x068,
    CMSG_ADD_FRIEND                                 = 0x069,
    CMSG_DEL_FRIEND                                 = 0x06A,
    SMSG_IGNORE_LIST                                = 0x06B,
    CMSG_ADD_IGNORE                                 = 0x06C,
    CMSG_DEL_IGNORE                                 = 0x06D,
    CMSG_GROUP_INVITE                               = 0x06E,
    SMSG_GROUP_INVITE                               = 0x06F,
    CMSG_GROUP_CANCEL                               = 0x070,
    SMSG_GROUP_CANCEL                               = 0x071,
    CMSG_GROUP_ACCEPT                               = 0x072,
    CMSG_GROUP_DECLINE                              = 0x073,
    SMSG_GROUP_DECLINE                              = 0x074,
    CMSG_GROUP_UNINVITE                             = 0x075,
    CMSG_GROUP_UNINVITE_GUID                        = 0x076,
    SMSG_GROUP_UNINVITE                             = 0x077,
    CMSG_GROUP_SET_LEADER                           = 0x078,
    SMSG_GROUP_SET_LEADER                           = 0x079,
    CMSG_LOOT_METHOD                                = 0x07A,
    CMSG_GROUP_DISBAND                              = 0x07B,
    SMSG_GROUP_DESTROYED                            = 0x07C,
    SMSG_GROUP_LIST                                 = 0x07D,
    SMSG_PARTY_MEMBER_STATS                         = 0x07E,
    SMSG_PARTY_COMMAND_RESULT                       = 0x07F,
    UMSG_UPDATE_GROUP_MEMBERS                       = 0x080,
    CMSG_GUILD_CREATE                               = 0x081,
    CMSG_GUILD_INVITE                               = 0x082,
    SMSG_GUILD_INVITE                               = 0x083,
    CMSG_GUILD_ACCEPT                               = 0x084,
    CMSG_GUILD_DECLINE                              = 0x085,
    SMSG_GUILD_DECLINE                              = 0x086,
    CMSG_GUILD_INFO                                 = 0x087,
    SMSG_GUILD_INFO                                 = 0x088,
    CMSG_GUILD_ROSTER                               = 0x089,
    SMSG_GUILD_ROSTER                               = 0x08A,
    CMSG_GUILD_PROMOTE                              = 0x08B,
    CMSG_GUILD_DEMOTE                               = 0x08C,
    CMSG_GUILD_LEAVE                                = 0x08D,
    CMSG_GUILD_REMOVE                               = 0x08E,
    CMSG_GUILD_DISBAND                              = 0x08F,
    CMSG_GUILD_LEADER                               = 0x090,
    CMSG_GUILD_MOTD                                 = 0x091,
    SMSG_GUILD_EVENT                                = 0x092,
    SMSG_GUILD_COMMAND_RESULT                       = 0x093,
    UMSG_UPDATE_GUILD                               = 0x094,
    CMSG_MESSAGECHAT                                = 0x095,
    SMSG_MESSAGECHAT                                = 0x096,
    CMSG_JOIN_CHANNEL                               = 0x097,
    CMSG_LEAVE_CHANNEL                              = 0x098,
    SMSG_CHANNEL_NOTIFY                             = 0x099,
    CMSG_CHANNEL_LIST                               = 0x09A,
    SMSG_CHANNEL_LIST                               = 0x09B,
    CMSG_CHANNEL_PASSWORD                           = 0x09C,
    CMSG_CHANNEL_SET_OWNER                          = 0x09D,
    CMSG_CHANNEL_OWNER                              = 0x09E,
    CMSG_CHANNEL_MODERATOR                          = 0x09F,
    CMSG_CHANNEL_UNMODERATOR                        = 0x0A0,
    CMSG_CHANNEL_MUTE                               = 0x0A1,
    CMSG_CHANNEL_UNMUTE                             = 0x0A2,
    CMSG_CHANNEL_INVITE                             = 0x0A3,
    CMSG_CHANNEL_KICK                               = 0x0A4,
    CMSG_CHANNEL_BAN                                = 0x0A5,
    CMSG_CHANNEL_UNBAN                              = 0x0A6,
    CMSG_CHANNEL_ANNOUNCEMENTS                      = 0x0A7,
    CMSG_CHANNEL_MODERATE                           = 0x0A8,
    SMSG_UPDATE_OBJECT                              = 0x0A9,
    SMSG_DESTROY_OBJECT                             = 0x0AA,
    CMSG_USE_ITEM                                   = 0x0AB,
    CMSG_OPEN_ITEM                                  = 0x0AC,
    CMSG_READ_ITEM                                  = 0x0AD,
    SMSG_READ_ITEM_OK                               = 0x0AE,
    SMSG_READ_ITEM_FAILED                           = 0x0AF,
    SMSG_ITEM_COOLDOWN                              = 0x0B0,
    CMSG_GAMEOBJ_USE                                = 0x0B1,
    CMSG_GAMEOBJ_CHAIR_USE_OBSOLETE                 = 0x0B2,
    SMSG_GAMEOBJECT_CUSTOM_ANIM                     = 0x0B3,
    CMSG_AREATRIGGER                                = 0x0B4,
    MSG_MOVE_START_FORWARD                          = 0x0B5,
    MSG_MOVE_START_BACKWARD                         = 0x0B6,
    MSG_MOVE_STOP                                   = 0x0B7,
    MSG_MOVE_START_STRAFE_LEFT                      = 0x0B8,
    MSG_MOVE_START_STRAFE_RIGHT                     = 0x0B9,
    MSG_MOVE_STOP_STRAFE                            = 0x0BA,
    MSG_MOVE_JUMP                                   = 0x0BB,
    MSG_MOVE_START_TURN_LEFT                        = 0x0BC,
    MSG_MOVE_START_TURN_RIGHT                       = 0x0BD,
    MSG_MOVE_STOP_TURN                              = 0x0BE,
    MSG_MOVE_START_PITCH_UP                         = 0x0BF,
    MSG_MOVE_START_PITCH_DOWN                       = 0x0C0,
    MSG_MOVE_STOP_PITCH                             = 0x0C1,
    MSG_MOVE_SET_RUN_MODE                           = 0x0C2,
    MSG_MOVE_SET_WALK_MODE                          = 0x0C3,
    MSG_MOVE_TOGGLE_LOGGING                         = 0x0C4,
    MSG_MOVE_TELEPORT                               = 0x0C5,
    MSG_MOVE_TELEPORT_CHEAT                         = 0x0C6,
    MSG_MOVE_TELEPORT_ACK                           = 0x0C7,
    MSG_MOVE_TOGGLE_FALL_LOGGING                    = 0x0C8,
    MSG_MOVE_FALL_LAND                              = 0x0C9,
    MSG_MOVE_START_SWIM                             = 0x0CA,
    MSG_MOVE_STOP_SWIM                              = 0x0CB,
    MSG_MOVE_SET_RUN_SPEED_CHEAT                    = 0x0CC,
    MSG_MOVE_SET_RUN_SPEED                          = 0x0CD,
    MSG_MOVE_SET_RUN_BACK_SPEED_CHEAT               = 0x0CE,
    MSG_MOVE_SET_RUN_BACK_SPEED                     = 0x0CF,
    MSG_MOVE_SET_WALK_SPEED_CHEAT                   = 0x0D0,
    MSG_MOVE_SET_WALK_SPEED                         = 0x0D1,
    MSG_MOVE_SET_SWIM_SPEED_CHEAT                   = 0x0D2,
    MSG_MOVE_SET_SWIM_SPEED                         = 0x0D3,
    MSG_MOVE_SET_SWIM_BACK_SPEED_CHEAT              = 0x0D4,
    MSG_MOVE_SET_SWIM_BACK_SPEED                    = 0x0D5,
    MSG_MOVE_SET_ALL_SPEED_CHEAT                    = 0x0D6,
    MSG_MOVE_SET_TURN_RATE_CHEAT                    = 0x0D7,
    MSG_MOVE_SET_TURN_RATE                          = 0x0D8,
    MSG_MOVE_TOGGLE_COLLISION_CHEAT                 = 0x0D9,
    MSG_MOVE_SET_FACING                             = 0x0DA,
    MSG_MOVE_SET_PITCH                              = 0x0DB,
    MSG_MOVE_WORLDPORT_ACK                          = 0x0DC,
    SMSG_MONSTER_MOVE                               = 0x0DD,
    SMSG_MOVE_WATER_WALK                            = 0x0DE,
    SMSG_MOVE_LAND_WALK                             = 0x0DF,
    MSG_MOVE_SET_RAW_POSITION_ACK                   = 0x0E0,
    CMSG_MOVE_SET_RAW_POSITION                      = 0x0E1,
    SMSG_FORCE_RUN_SPEED_CHANGE                     = 0x0E2,
    CMSG_FORCE_RUN_SPEED_CHANGE_ACK                 = 0x0E3,
    SMSG_FORCE_RUN_BACK_SPEED_CHANGE                = 0x0E4,
    CMSG_FORCE_RUN_BACK_SPEED_CHANGE_ACK            = 0x0E5,
    SMSG_FORCE_SWIM_SPEED_CHANGE                    = 0x0E6,
    CMSG_FORCE_SWIM_SPEED_CHANGE_ACK                = 0x0E7,
    SMSG_FORCE_MOVE_ROOT                            = 0x0E8,
    CMSG_FORCE_MOVE_ROOT_ACK                        = 0x0E9,
    SMSG_FORCE_MOVE_UNROOT                          = 0x0EA,
    CMSG_FORCE_MOVE_UNROOT_ACK                      = 0x0EB,
    MSG_MOVE_ROOT                                   = 0x0EC,
    MSG_MOVE_UNROOT                                 = 0x0ED,
    MSG_MOVE_HEARTBEAT                              = 0x0EE,
    SMSG_MOVE_KNOCK_BACK                            = 0x0EF,
    CMSG_MOVE_KNOCK_BACK_ACK                        = 0x0F0,
    MSG_MOVE_KNOCK_BACK                             = 0x0F1,
    SMSG_MOVE_FEATHER_FALL                          = 0x0F2,
    SMSG_MOVE_NORMAL_FALL                           = 0x0F3,
    SMSG_MOVE_SET_HOVER                             = 0x0F4,
    SMSG_MOVE_UNSET_HOVER                           = 0x0F5,
    CMSG_MOVE_HOVER_ACK                             = 0x0F6,
    MSG_MOVE_HOVER                                  = 0x0F7,
    CMSG_TRIGGER_CINEMATIC_CHEAT                    = 0x0F8,
    CMSG_OPENING_CINEMATIC                          = 0x0F9,
    SMSG_TRIGGER_CINEMATIC                          = 0x0FA,
    CMSG_NEXT_CINEMATIC_CAMERA                      = 0x0FB,
    CMSG_COMPLETE_CINEMATIC                         = 0x0FC,
    SMSG_TUTORIAL_FLAGS                             = 0x0FD,
    CMSG_TUTORIAL_FLAG                              = 0x0FE,
    CMSG_TUTORIAL_CLEAR                             = 0x0FF,
    CMSG_TUTORIAL_RESET                             = 0x100,
    CMSG_STANDSTATECHANGE                           = 0x101,
    CMSG_EMOTE                                      = 0x102,
    SMSG_EMOTE                                      = 0x103,
    CMSG_TEXT_EMOTE                                 = 0x104,
    SMSG_TEXT_EMOTE                                 = 0x105,
    CMSG_AUTOEQUIP_GROUND_ITEM                      = 0x106,
    CMSG_AUTOSTORE_GROUND_ITEM                      = 0x107,
    CMSG_AUTOSTORE_LOOT_ITEM                        = 0x108,
    CMSG_STORE_LOOT_IN_SLOT                         = 0x109,
    CMSG_AUTOEQUIP_ITEM                             = 0x10A,
    CMSG_AUTOSTORE_BAG_ITEM                         = 0x10B,
    CMSG_SWAP_ITEM                                  = 0x10C,
    CMSG_SWAP_INV_ITEM                              = 0x10D,
    CMSG_SPLIT_ITEM                                 = 0x10E,
    CMSG_AUTOEQUIP_ITEM_SLOT                        = 0x10F,
    OBSOLETE_DROP_ITEM                              = 0x110,
    CMSG_DESTROYITEM                                = 0x111,
    SMSG_INVENTORY_CHANGE_FAILURE                   = 0x112,
    SMSG_OPEN_CONTAINER                             = 0x113,
    CMSG_INSPECT                                    = 0x114,
    SMSG_INSPECT                                    = 0x115,
    CMSG_INITIATE_TRADE                             = 0x116,
    CMSG_BEGIN_TRADE                                = 0x117,
    CMSG_BUSY_TRADE                                 = 0x118,
    CMSG_IGNORE_TRADE                               = 0x119,
    CMSG_ACCEPT_TRADE                               = 0x11A,
    CMSG_UNACCEPT_TRADE                             = 0x11B,
    CMSG_CANCEL_TRADE                               = 0x11C,
    CMSG_SET_TRADE_ITEM                             = 0x11D,
    CMSG_CLEAR_TRADE_ITEM                           = 0x11E,
    CMSG_SET_TRADE_GOLD                             = 0x11F,
    SMSG_TRADE_STATUS                               = 0x120,
    SMSG_TRADE_STATUS_EXTENDED                      = 0x121,
    SMSG_INITIALIZE_FACTIONS                        = 0x122,
    SMSG_SET_FACTION_VISIBLE                        = 0x123,
    SMSG_SET_FACTION_STANDING                       = 0x124,
    CMSG_SET_FACTION_ATWAR                          = 0x125,
    CMSG_SET_FACTION_CHEAT                          = 0x126,
    SMSG_SET_PROFICIENCY                            = 0x127,
    CMSG_SET_ACTION_BUTTON                          = 0x128,
    SMSG_ACTION_BUTTONS                             = 0x129,
    SMSG_INITIAL_SPELLS                             = 0x12A,
    SMSG_LEARNED_SPELL                              = 0x12B,
    SMSG_SUPERCEDED_SPELL                           = 0x12C,
    CMSG_NEW_SPELL_SLOT                             = 0x12D,
    CMSG_CAST_SPELL                                 = 0x12E,
    CMSG_CANCEL_CAST                                = 0x12F,
    SMSG_CAST_RESULT                                = 0x130,
    SMSG_SPELL_START                                = 0x131,
    SMSG_SPELL_GO                                   = 0x132,
    SMSG_SPELL_FAILURE                              = 0x133,
    SMSG_SPELL_COOLDOWN                             = 0x134,
    SMSG_COOLDOWN_EVENT                             = 0x135,
    CMSG_CANCEL_AURA                                = 0x136,
    SMSG_UPDATE_AURA_DURATION                       = 0x137,
    SMSG_PET_CAST_FAILED                            = 0x138,
    MSG_CHANNEL_START                               = 0x139,
    MSG_CHANNEL_UPDATE                              = 0x13A,
    CMSG_CANCEL_CHANNELLING                         = 0x13B,
    SMSG_AI_REACTION                                = 0x13C,
    CMSG_SET_SELECTION                              = 0x13D,
    CMSG_SET_TARGET_OBSOLETE                        = 0x13E,
    CMSG_UNUSED                                     = 0x13F,
    CMSG_UNUSED2                                    = 0x140,
    CMSG_ATTACKSWING                                = 0x141,
    CMSG_ATTACKSTOP                                 = 0x142,
    SMSG_ATTACKSTART                                = 0x143,
    SMSG_ATTACKSTOP                                 = 0x144,
    SMSG_ATTACKSWING_NOTINRANGE                     = 0x145,
    SMSG_ATTACKSWING_BADFACING                      = 0x146,
    SMSG_ATTACKSWING_NOTSTANDING                    = 0x147,
    SMSG_ATTACKSWING_DEADTARGET                     = 0x148,
    SMSG_ATTACKSWING_CANT_ATTACK                    = 0x149,
    SMSG_ATTACKERSTATEUPDATE                        = 0x14A,
    SMSG_VICTIMSTATEUPDATE_OBSOLETE                 = 0x14B,
    SMSG_DAMAGE_DONE_OBSOLETE                       = 0x14C,
    SMSG_DAMAGE_TAKEN_OBSOLETE                      = 0x14D,
    SMSG_CANCEL_COMBAT                              = 0x14E,
    SMSG_PLAYER_COMBAT_XP_GAIN_OBSOLETE             = 0x14F,
    SMSG_SPELLHEALLOG                               = 0x150,
    SMSG_SPELLENERGIZELOG                           = 0x151,
    CMSG_SHEATHE_OBSOLETE                           = 0x152,
    CMSG_SAVE_PLAYER                                = 0x153,
    CMSG_SETDEATHBINDPOINT                          = 0x154,
    SMSG_BINDPOINTUPDATE                            = 0x155,
    CMSG_GETDEATHBINDZONE                           = 0x156,
    SMSG_BINDZONEREPLY                              = 0x157,
    SMSG_PLAYERBOUND                                = 0x158,
    SMSG_CLIENT_CONTROL_UPDATE                      = 0x159,
    CMSG_REPOP_REQUEST                              = 0x15A,
    SMSG_RESURRECT_REQUEST                          = 0x15B,
    CMSG_RESURRECT_RESPONSE                         = 0x15C,
    CMSG_LOOT                                       = 0x15D,
    CMSG_LOOT_MONEY                                 = 0x15E,
    CMSG_LOOT_RELEASE                               = 0x15F,
    SMSG_LOOT_RESPONSE                              = 0x160,
    SMSG_LOOT_RELEASE_RESPONSE                      = 0x161,
    SMSG_LOOT_REMOVED                               = 0x162,
    SMSG_LOOT_MONEY_NOTIFY                          = 0x163,
    SMSG_LOOT_ITEM_NOTIFY                           = 0x164,
    SMSG_LOOT_CLEAR_MONEY                           = 0x165,
    SMSG_ITEM_PUSH_RESULT                           = 0x166,
    SMSG_DUEL_REQUESTED                             = 0x167,
    SMSG_DUEL_OUTOFBOUNDS                           = 0x168,
    SMSG_DUEL_INBOUNDS                              = 0x169,
    SMSG_DUEL_COMPLETE                              = 0x16A,
    SMSG_DUEL_WINNER                                = 0x16B,
    CMSG_DUEL_ACCEPTED                              = 0x16C,
    CMSG_DUEL_CANCELLED                             = 0x16D,
    SMSG_MOUNTRESULT                                = 0x16E,
    SMSG_DISMOUNTRESULT                             = 0x16F,
    SMSG_PUREMOUNT_CANCELLED_OBSOLETE               = 0x170,
    CMSG_MOUNTSPECIAL_ANIM                          = 0x171,
    SMSG_MOUNTSPECIAL_ANIM                          = 0x172,
    SMSG_PET_TAME_FAILURE                           = 0x173,
    CMSG_PET_SET_ACTION                             = 0x174,
    CMSG_PET_ACTION                                 = 0x175,
    CMSG_PET_ABANDON                                = 0x176,
    CMSG_PET_RENAME                                 = 0x177,
    SMSG_PET_NAME_INVALID                           = 0x178,
    SMSG_PET_SPELLS                                 = 0x179,
    SMSG_PET_MODE                                   = 0x17A,
    CMSG_GOSSIP_HELLO                               = 0x17B,
    CMSG_GOSSIP_SELECT_OPTION                       = 0x17C,
    SMSG_GOSSIP_MESSAGE                             = 0x17D,
    SMSG_GOSSIP_COMPLETE                            = 0x17E,
    CMSG_NPC_TEXT_QUERY                             = 0x17F,
    SMSG_NPC_TEXT_UPDATE                            = 0x180,
    SMSG_NPC_WONT_TALK                              = 0x181,
    CMSG_QUESTGIVER_STATUS_QUERY                    = 0x182,
    SMSG_QUESTGIVER_STATUS                          = 0x183,
    CMSG_QUESTGIVER_HELLO                           = 0x184,
    SMSG_QUESTGIVER_QUEST_LIST                      = 0x185,
    CMSG_QUESTGIVER_QUERY_QUEST                     = 0x186,
    CMSG_QUESTGIVER_QUEST_AUTOLAUNCH                = 0x187,
    SMSG_QUESTGIVER_QUEST_DETAILS                   = 0x188,
    CMSG_QUESTGIVER_ACCEPT_QUEST                    = 0x189,
    CMSG_QUESTGIVER_COMPLETE_QUEST                  = 0x18A,
    SMSG_QUESTGIVER_REQUEST_ITEMS                   = 0x18B,
    CMSG_QUESTGIVER_REQUEST_REWARD                  = 0x18C,
    SMSG_QUESTGIVER_OFFER_REWARD                    = 0x18D,
    CMSG_QUESTGIVER_CHOOSE_REWARD                   = 0x18E,
    SMSG_QUESTGIVER_QUEST_INVALID                   = 0x18F,
    CMSG_QUESTGIVER_CANCEL                          = 0x190,
    SMSG_QUESTGIVER_QUEST_COMPLETE                  = 0x191,
    SMSG_QUESTGIVER_QUEST_FAILED                    = 0x192,
    CMSG_QUESTLOG_SWAP_QUEST                        = 0x193,
    CMSG_QUESTLOG_REMOVE_QUEST                      = 0x194,
    SMSG_QUESTLOG_FULL                              = 0x195,
    SMSG_QUESTUPDATE_FAILED                         = 0x196,
    SMSG_QUESTUPDATE_FAILEDTIMER                    = 0x197,
    SMSG_QUESTUPDATE_COMPLETE                       = 0x198,
    SMSG_QUESTUPDATE_ADD_KILL                       = 0x199,
    SMSG_QUESTUPDATE_ADD_ITEM                       = 0x19A,
    CMSG_QUEST_CONFIRM_ACCEPT                       = 0x19B,
    SMSG_QUEST_CONFIRM_ACCEPT                       = 0x19C,
    CMSG_PUSHQUESTTOPARTY                           = 0x19D,
    CMSG_LIST_INVENTORY                             = 0x19E,
    SMSG_LIST_INVENTORY                             = 0x19F,
    CMSG_SELL_ITEM                                  = 0x1A0,
    SMSG_SELL_ITEM                                  = 0x1A1,
    CMSG_BUY_ITEM                                   = 0x1A2,
    CMSG_BUY_ITEM_IN_SLOT                           = 0x1A3,
    SMSG_BUY_ITEM                                   = 0x1A4,
    SMSG_BUY_FAILED                                 = 0x1A5,
    CMSG_TAXICLEARALLNODES                          = 0x1A6,
    CMSG_TAXIENABLEALLNODES                         = 0x1A7,
    CMSG_TAXISHOWNODES                              = 0x1A8,
    SMSG_SHOWTAXINODES                              = 0x1A9,
    CMSG_TAXINODE_STATUS_QUERY                      = 0x1AA,
    SMSG_TAXINODE_STATUS                            = 0x1AB,
    CMSG_TAXIQUERYAVAILABLENODES                    = 0x1AC,
    CMSG_ACTIVATETAXI                               = 0x1AD,
    SMSG_ACTIVATETAXIREPLY                          = 0x1AE,
    SMSG_NEW_TAXI_PATH                              = 0x1AF,
    CMSG_TRAINER_LIST                               = 0x1B0,
    SMSG_TRAINER_LIST                               = 0x1B1,
    CMSG_TRAINER_BUY_SPELL                          = 0x1B2,
    SMSG_TRAINER_BUY_SUCCEEDED                      = 0x1B3,
    SMSG_TRAINER_BUY_FAILED                         = 0x1B4,
    CMSG_BINDER_ACTIVATE                            = 0x1B5,
    SMSG_PLAYERBINDERROR                            = 0x1B6,
    CMSG_BANKER_ACTIVATE                            = 0x1B7,
    SMSG_SHOW_BANK                                  = 0x1B8,
    CMSG_BUY_BANK_SLOT                              = 0x1B9,
    SMSG_BUY_BANK_SLOT_RESULT                       = 0x1BA,
    CMSG_PETITION_SHOWLIST                          = 0x1BB,
    SMSG_PETITION_SHOWLIST                          = 0x1BC,
    CMSG_PETITION_BUY                               = 0x1BD,
    CMSG_PETITION_SHOW_SIGNATURES                   = 0x1BE,
    SMSG_PETITION_SHOW_SIGNATURES                   = 0x1BF,
    CMSG_PETITION_SIGN                              = 0x1C0,
    SMSG_PETITION_SIGN_RESULTS                      = 0x1C1,
    MSG_PETITION_DECLINE                            = 0x1C2,
    CMSG_OFFER_PETITION                             = 0x1C3,
    CMSG_TURN_IN_PETITION                           = 0x1C4,
    SMSG_TURN_IN_PETITION_RESULTS                   = 0x1C5,
    CMSG_PETITION_QUERY                             = 0x1C6,
    SMSG_PETITION_QUERY_RESPONSE                    = 0x1C7,
    SMSG_FISH_NOT_HOOKED                            = 0x1C8,
    SMSG_FISH_ESCAPED                               = 0x1C9,
    CMSG_BUG                                        = 0x1CA,
    SMSG_NOTIFICATION                               = 0x1CB,
    CMSG_PLAYED_TIME                                = 0x1CC,
    SMSG_PLAYED_TIME                                = 0x1CD,
    CMSG_QUERY_TIME                                 = 0x1CE,
    SMSG_QUERY_TIME_RESPONSE                        = 0x1CF,
    SMSG_LOG_XPGAIN                                 = 0x1D0,
    SMSG_AURACASTLOG                                = 0x1D1,
    CMSG_RECLAIM_CORPSE                             = 0x1D2,
    CMSG_WRAP_ITEM                                  = 0x1D3,
    SMSG_LEVELUP_INFO                               = 0x1D4,
    MSG_MINIMAP_PING                                = 0x1D5,
    SMSG_RESISTLOG                                  = 0x1D6,
    SMSG_ENCHANTMENTLOG                             = 0x1D7,
    CMSG_SET_SKILL_CHEAT                            = 0x1D8,
    SMSG_START_MIRROR_TIMER                         = 0x1D9,
    SMSG_PAUSE_MIRROR_TIMER                         = 0x1DA,
    SMSG_STOP_MIRROR_TIMER                          = 0x1DB,
    CMSG_PING                                       = 0x1DC,
    SMSG_PONG                                       = 0x1DD,
    SMSG_CLEAR_COOLDOWN                             = 0x1DE,
    SMSG_GAMEOBJECT_PAGETEXT                        = 0x1DF,
    CMSG_SETSHEATHED                                = 0x1E0,
    SMSG_COOLDOWN_CHEAT                             = 0x1E1,
    SMSG_SPELL_DELAYED                              = 0x1E2,
    CMSG_PLAYER_MACRO_OBSOLETE                      = 0x1E3,
    SMSG_PLAYER_MACRO_OBSOLETE                      = 0x1E4,
    CMSG_GHOST                                      = 0x1E5,
    CMSG_GM_INVIS                                   = 0x1E6,
    SMSG_INVALID_PROMOTION_CODE                     = 0x1E7,
    MSG_GM_BIND_OTHER                               = 0x1E8,
    MSG_GM_SUMMON                                   = 0x1E9,
    SMSG_ITEM_TIME_UPDATE                           = 0x1EA,
    SMSG_ITEM_ENCHANT_TIME_UPDATE                   = 0x1EB,
    SMSG_AUTH_CHALLENGE                             = 0x1EC,
    CMSG_AUTH_SESSION                               = 0x1ED,
    SMSG_AUTH_RESPONSE                              = 0x1EE,
    MSG_GM_SHOWLABEL                                = 0x1EF,
    CMSG_PET_CAST_SPELL                             = 0x1F0,
    MSG_SAVE_GUILD_EMBLEM                           = 0x1F1,
    MSG_TABARDVENDOR_ACTIVATE                       = 0x1F2,
    SMSG_PLAY_SPELL_VISUAL                          = 0x1F3,
    CMSG_ZONEUPDATE                                 = 0x1F4,
    SMSG_PARTYKILLLOG                               = 0x1F5,
    SMSG_COMPRESSED_UPDATE_OBJECT                   = 0x1F6,
    SMSG_PLAY_SPELL_IMPACT                          = 0x1F7,
    SMSG_EXPLORATION_EXPERIENCE                     = 0x1F8,
    CMSG_GM_SET_SECURITY_GROUP                      = 0x1F9,
    CMSG_GM_NUKE                                    = 0x1FA,
    MSG_RANDOM_ROLL                                 = 0x1FB,
    SMSG_ENVIRONMENTAL_DAMAGE_LOG                   = 0x1FC,
    CMSG_RWHOIS_OBSOLETE                            = 0x1FD,
    SMSG_RWHOIS                                     = 0x1FE,
    MSG_LOOKING_FOR_GROUP                           = 0x1FF,
    CMSG_SET_LOOKING_FOR_GROUP                      = 0x200,
    CMSG_UNLEARN_SPELL                              = 0x201,
    CMSG_UNLEARN_SKILL                              = 0x202,
    SMSG_REMOVED_SPELL                              = 0x203,
    CMSG_DECHARGE                                   = 0x204,
    CMSG_GMTICKET_CREATE                            = 0x205,
    SMSG_GMTICKET_CREATE                            = 0x206,
    CMSG_GMTICKET_UPDATETEXT                        = 0x207,
    SMSG_GMTICKET_UPDATETEXT                        = 0x208,
    SMSG_ACCOUNT_DATA_TIMES                         = 0x209,
    CMSG_REQUEST_ACCOUNT_DATA                       = 0x20A,
    CMSG_UPDATE_ACCOUNT_DATA                        = 0x20B,
    SMSG_UPDATE_ACCOUNT_DATA                        = 0x20C,
    SMSG_CLEAR_FAR_SIGHT_IMMEDIATE                  = 0x20D,
    SMSG_POWERGAINLOG_OBSOLETE                      = 0x20E,
    CMSG_GM_TEACH                                   = 0x20F,
    CMSG_GM_CREATE_ITEM_TARGET                      = 0x210,
    CMSG_GMTICKET_GETTICKET                         = 0x211,
    SMSG_GMTICKET_GETTICKET                         = 0x212,
    CMSG_UNLEARN_TALENTS                            = 0x213,
    SMSG_GAMEOBJECT_SPAWN_ANIM                      = 0x214,
    SMSG_GAMEOBJECT_DESPAWN_ANIM                    = 0x215,
    MSG_CORPSE_QUERY                                = 0x216,
    CMSG_GMTICKET_DELETETICKET                      = 0x217,
    SMSG_GMTICKET_DELETETICKET                      = 0x218,
    SMSG_CHAT_WRONG_FACTION                         = 0x219,
    CMSG_GMTICKET_SYSTEMSTATUS                      = 0x21A,
    SMSG_GMTICKET_SYSTEMSTATUS                      = 0x21B,
    CMSG_SPIRIT_HEALER_ACTIVATE                     = 0x21C,
    CMSG_SET_STAT_CHEAT                             = 0x21D,
    SMSG_SET_REST_START                             = 0x21E,
    CMSG_SKILL_BUY_STEP                             = 0x21F,
    CMSG_SKILL_BUY_RANK                             = 0x220,
    CMSG_XP_CHEAT                                   = 0x221,
    SMSG_SPIRIT_HEALER_CONFIRM                      = 0x222,
    CMSG_CHARACTER_POINT_CHEAT                      = 0x223,
    SMSG_GOSSIP_POI                                 = 0x224,
    CMSG_CHAT_IGNORED                               = 0x225,
    CMSG_GM_VISION                                  = 0x226,
    CMSG_SERVER_COMMAND                             = 0x227,
    CMSG_GM_SILENCE                                 = 0x228,
    CMSG_GM_REVEALTO                                = 0x229,
    CMSG_GM_RESURRECT                               = 0x22A,
    CMSG_GM_SUMMONMOB                               = 0x22B,
    CMSG_GM_MOVECORPSE                              = 0x22C,
    CMSG_GM_FREEZE                                  = 0x22D,
    CMSG_GM_UBERINVIS                               = 0x22E,
    CMSG_GM_REQUEST_PLAYER_INFO                     = 0x22F,
    SMSG_GM_PLAYER_INFO                             = 0x230,
    CMSG_GUILD_RANK                                 = 0x231,
    CMSG_GUILD_ADD_RANK                             = 0x232,
    CMSG_GUILD_DEL_RANK                             = 0x233,
    CMSG_GUILD_SET_PUBLIC_NOTE                      = 0x234,
    CMSG_GUILD_SET_OFFICER_NOTE                     = 0x235,
    SMSG_LOGIN_VERIFY_WORLD                         = 0x236,
    CMSG_CLEAR_EXPLORATION                          = 0x237,
    CMSG_SEND_MAIL                                  = 0x238,
    SMSG_SEND_MAIL_RESULT                           = 0x239,
    CMSG_GET_MAIL_LIST                              = 0x23A,
    SMSG_MAIL_LIST_RESULT                           = 0x23B,
    CMSG_BATTLEFIELD_LIST                           = 0x23C,
    SMSG_BATTLEFIELD_LIST                           = 0x23D,
    CMSG_BATTLEFIELD_JOIN                           = 0x23E,
    SMSG_BATTLEFIELD_WIN_OBSOLETE                   = 0x23F,
    SMSG_BATTLEFIELD_LOSE_OBSOLETE                  = 0x240,
    CMSG_TAXICLEARNODE                              = 0x241,
    CMSG_TAXIENABLENODE                             = 0x242,
    CMSG_ITEM_TEXT_QUERY                            = 0x243,
    SMSG_ITEM_TEXT_QUERY_RESPONSE                   = 0x244,
    CMSG_MAIL_TAKE_MONEY                            = 0x245,
    CMSG_MAIL_TAKE_ITEM                             = 0x246,
    CMSG_MAIL_MARK_AS_READ                          = 0x247,
    CMSG_MAIL_RETURN_TO_SENDER                      = 0x248,
    CMSG_MAIL_DELETE                                = 0x249,
    CMSG_MAIL_CREATE_TEXT_ITEM                      = 0x24A,
    SMSG_SPELLLOGMISS                               = 0x24B,
    SMSG_SPELLLOGEXECUTE                            = 0x24C,
    SMSG_DEBUGAURAPROC                              = 0x24D,
    SMSG_PERIODICAURALOG                            = 0x24E,
    SMSG_SPELLDAMAGESHIELD                          = 0x24F,
    SMSG_SPELLNONMELEEDAMAGELOG                     = 0x250,
    CMSG_LEARN_TALENT                               = 0x251,
    SMSG_RESURRECT_FAILED                           = 0x252,
    CMSG_TOGGLE_PVP                                 = 0x253,
    SMSG_ZONE_UNDER_ATTACK                          = 0x254,
    MSG_AUCTION_HELLO                               = 0x255,
    CMSG_AUCTION_SELL_ITEM                          = 0x256,
    CMSG_AUCTION_REMOVE_ITEM                        = 0x257,
    CMSG_AUCTION_LIST_ITEMS                         = 0x258,
    CMSG_AUCTION_LIST_OWNER_ITEMS                   = 0x259,
    CMSG_AUCTION_PLACE_BID                          = 0x25A,
    SMSG_AUCTION_COMMAND_RESULT                     = 0x25B,
    SMSG_AUCTION_LIST_RESULT                        = 0x25C,
    SMSG_AUCTION_OWNER_LIST_RESULT                  = 0x25D,
    SMSG_AUCTION_BIDDER_NOTIFICATION                = 0x25E,
    SMSG_AUCTION_OWNER_NOTIFICATION                 = 0x25F,
    SMSG_PROCRESIST                                 = 0x260,
    SMSG_STANDSTATE_CHANGE_FAILURE_OBSOLETE         = 0x261,
    SMSG_DISPEL_FAILED                              = 0x262,
    SMSG_SPELLORDAMAGE_IMMUNE                       = 0x263,
    CMSG_AUCTION_LIST_BIDDER_ITEMS                  = 0x264,
    SMSG_AUCTION_BIDDER_LIST_RESULT                 = 0x265,
    SMSG_SET_FLAT_SPELL_MODIFIER                    = 0x266,
    SMSG_SET_PCT_SPELL_MODIFIER                     = 0x267,
    CMSG_SET_AMMO                                   = 0x268,
    SMSG_CORPSE_RECLAIM_DELAY                       = 0x269,
    CMSG_SET_ACTIVE_MOVER                           = 0x26A,
    CMSG_PET_CANCEL_AURA                            = 0x26B,
    CMSG_PLAYER_AI_CHEAT                            = 0x26C,
    CMSG_CANCEL_AUTO_REPEAT_SPELL                   = 0x26D,
    MSG_GM_ACCOUNT_ONLINE                           = 0x26E,
    MSG_LIST_STABLED_PETS                           = 0x26F,
    CMSG_STABLE_PET                                 = 0x270,
    CMSG_UNSTABLE_PET                               = 0x271,
    CMSG_BUY_STABLE_SLOT                            = 0x272,
    SMSG_STABLE_RESULT                              = 0x273,
    CMSG_STABLE_REVIVE_PET                          = 0x274,
    CMSG_STABLE_SWAP_PET                            = 0x275,
    MSG_QUEST_PUSH_RESULT                           = 0x276,
    SMSG_PLAY_MUSIC                                 = 0x277,
    SMSG_PLAY_OBJECT_SOUND                          = 0x278,
    CMSG_REQUEST_PET_INFO                           = 0x279,
    CMSG_FAR_SIGHT                                  = 0x27A,
    SMSG_SPELLDISPELLOG                             = 0x27B,
    SMSG_DAMAGE_CALC_LOG                            = 0x27C,
    CMSG_ENABLE_DAMAGE_LOG                          = 0x27D,
    CMSG_GROUP_CHANGE_SUB_GROUP                     = 0x27E,
    CMSG_REQUEST_PARTY_MEMBER_STATS                 = 0x27F,
    CMSG_GROUP_SWAP_SUB_GROUP                       = 0x280,
    CMSG_RESET_FACTION_CHEAT                        = 0x281,
    CMSG_AUTOSTORE_BANK_ITEM                        = 0x282,
    CMSG_AUTOBANK_ITEM                              = 0x283,
    MSG_QUERY_NEXT_MAIL_TIME                        = 0x284,
    SMSG_RECEIVED_MAIL                              = 0x285,
    SMSG_RAID_GROUP_ONLY                            = 0x286,
    CMSG_SET_DURABILITY_CHEAT                       = 0x287,
    CMSG_SET_PVP_RANK_CHEAT                         = 0x288,
    CMSG_ADD_PVP_MEDAL_CHEAT                        = 0x289,
    CMSG_DEL_PVP_MEDAL_CHEAT                        = 0x28A,
    CMSG_SET_PVP_TITLE                              = 0x28B,
    SMSG_PVP_CREDIT                                 = 0x28C,
    SMSG_AUCTION_REMOVED_NOTIFICATION               = 0x28D,
    CMSG_GROUP_RAID_CONVERT                         = 0x28E,
    CMSG_GROUP_ASSISTANT_LEADER                     = 0x28F,
    CMSG_BUYBACK_ITEM                               = 0x290,
    SMSG_SERVER_MESSAGE                             = 0x291,
    CMSG_MEETINGSTONE_JOIN                          = 0x292,
    CMSG_MEETINGSTONE_LEAVE                         = 0x293,
    CMSG_MEETINGSTONE_CHEAT                         = 0x294,
    SMSG_MEETINGSTONE_SETQUEUE                      = 0x295,
    CMSG_MEETINGSTONE_INFO                          = 0x296,
    SMSG_MEETINGSTONE_COMPLETE                      = 0x297,
    SMSG_MEETINGSTONE_IN_PROGRESS                   = 0x298,
    SMSG_MEETINGSTONE_MEMBER_ADDED                  = 0x299,
    CMSG_GMTICKETSYSTEM_TOGGLE                      = 0x29A,
    CMSG_CANCEL_GROWTH_AURA                         = 0x29B,
    SMSG_CANCEL_AUTO_REPEAT                         = 0x29C,
    SMSG_STANDSTATE_UPDATE                          = 0x29D,
    SMSG_LOOT_ALL_PASSED                            = 0x29E,
    SMSG_LOOT_ROLL_WON                              = 0x29F,
    CMSG_LOOT_ROLL                                  = 0x2A0,
    SMSG_LOOT_START_ROLL                            = 0x2A1,
    SMSG_LOOT_ROLL                                  = 0x2A2,
    CMSG_LOOT_MASTER_GIVE                           = 0x2A3,
    SMSG_LOOT_MASTER_LIST                           = 0x2A4,
    SMSG_SET_FORCED_REACTIONS                       = 0x2A5,
    SMSG_SPELL_FAILED_OTHER                         = 0x2A6,
    SMSG_GAMEOBJECT_RESET_STATE                     = 0x2A7,
    CMSG_REPAIR_ITEM                                = 0x2A8,
    SMSG_CHAT_PLAYER_NOT_FOUND                      = 0x2A9,
    MSG_TALENT_WIPE_CONFIRM                         = 0x2AA,
    SMSG_SUMMON_REQUEST                             = 0x2AB,
    CMSG_SUMMON_RESPONSE                            = 0x2AC,
    MSG_MOVE_TOGGLE_GRAVITY_CHEAT                   = 0x2AD,
    SMSG_MONSTER_MOVE_TRANSPORT                     = 0x2AE,
    SMSG_PET_BROKEN                                 = 0x2AF,
    MSG_MOVE_FEATHER_FALL                           = 0x2B0,
    MSG_MOVE_WATER_WALK                             = 0x2B1,
    CMSG_SERVER_BROADCAST                           = 0x2B2,
    CMSG_SELF_RES                                   = 0x2B3,
    SMSG_FEIGN_DEATH_RESISTED                       = 0x2B4,
    CMSG_RUN_SCRIPT                                 = 0x2B5,
    SMSG_SCRIPT_MESSAGE                             = 0x2B6,
    SMSG_DUEL_COUNTDOWN                             = 0x2B7,
    SMSG_AREA_TRIGGER_MESSAGE                       = 0x2B8,
    CMSG_TOGGLE_HELM                                = 0x2B9,
    CMSG_TOGGLE_CLOAK                               = 0x2BA,
    SMSG_MEETINGSTONE_JOINFAILED                    = 0x2BB,
    SMSG_PLAYER_SKINNED                             = 0x2BC,
    SMSG_DURABILITY_DAMAGE_DEATH                    = 0x2BD,
    CMSG_SET_EXPLORATION                            = 0x2BE,
    CMSG_SET_ACTIONBAR_TOGGLES                      = 0x2BF,
    UMSG_DELETE_GUILD_CHARTER                       = 0x2C0,
    MSG_PETITION_RENAME                             = 0x2C1,
    SMSG_INIT_WORLD_STATES                          = 0x2C2,
    SMSG_UPDATE_WORLD_STATE                         = 0x2C3,
    CMSG_ITEM_NAME_QUERY                            = 0x2C4,
    SMSG_ITEM_NAME_QUERY_RESPONSE                   = 0x2C5,
    SMSG_PET_ACTION_FEEDBACK                        = 0x2C6,
    CMSG_CHAR_RENAME                                = 0x2C7,
    SMSG_CHAR_RENAME                                = 0x2C8,
    CMSG_MOVE_SPLINE_DONE                           = 0x2C9,
    CMSG_MOVE_FALL_RESET                            = 0x2CA,
    SMSG_INSTANCE_SAVE_CREATED                      = 0x2CB,
    SMSG_RAID_INSTANCE_INFO                         = 0x2CC,
    CMSG_REQUEST_RAID_INFO                          = 0x2CD,
    CMSG_MOVE_TIME_SKIPPED                          = 0x2CE,
    CMSG_MOVE_FEATHER_FALL_ACK                      = 0x2CF,
    CMSG_MOVE_WATER_WALK_ACK                        = 0x2D0,
    CMSG_MOVE_NOT_ACTIVE_MOVER                      = 0x2D1,
    SMSG_PLAY_SOUND                                 = 0x2D2,
    CMSG_BATTLEFIELD_STATUS                         = 0x2D3,
    SMSG_BATTLEFIELD_STATUS                         = 0x2D4,
    CMSG_BATTLEFIELD_PORT                           = 0x2D5,
    MSG_INSPECT_HONOR_STATS                         = 0x2D6,
    CMSG_BATTLEMASTER_HELLO                         = 0x2D7,
    CMSG_MOVE_START_SWIM_CHEAT                      = 0x2D8,
    CMSG_MOVE_STOP_SWIM_CHEAT                       = 0x2D9,
    SMSG_FORCE_WALK_SPEED_CHANGE                    = 0x2DA,
    CMSG_FORCE_WALK_SPEED_CHANGE_ACK                = 0x2DB,
    SMSG_FORCE_SWIM_BACK_SPEED_CHANGE               = 0x2DC,
    CMSG_FORCE_SWIM_BACK_SPEED_CHANGE_ACK           = 0x2DD,
    SMSG_FORCE_TURN_RATE_CHANGE                     = 0x2DE,
    CMSG_FORCE_TURN_RATE_CHANGE_ACK                 = 0x2DF,
    MSG_PVP_LOG_DATA                                = 0x2E0,
    CMSG_LEAVE_BATTLEFIELD                          = 0x2E1,
    CMSG_AREA_SPIRIT_HEALER_QUERY                   = 0x2E2,
    CMSG_AREA_SPIRIT_HEALER_QUEUE                   = 0x2E3,
    SMSG_AREA_SPIRIT_HEALER_TIME                    = 0x2E4,
    CMSG_GM_UNTEACH                                 = 0x2E5,
    SMSG_WARDEN_DATA                                = 0x2E6,
    CMSG_WARDEN_DATA                                = 0x2E7,
    SMSG_GROUP_JOINED_BATTLEGROUND                  = 0x2E8,
    MSG_BATTLEGROUND_PLAYER_POSITIONS               = 0x2E9,
    CMSG_PET_STOP_ATTACK                            = 0x2EA,
    SMSG_BINDER_CONFIRM                             = 0x2EB,
    SMSG_BATTLEGROUND_PLAYER_JOINED                 = 0x2EC,
    SMSG_BATTLEGROUND_PLAYER_LEFT                   = 0x2ED,
    CMSG_BATTLEMASTER_JOIN                          = 0x2EE,
    SMSG_ADDON_INFO                                 = 0x2EF,
    CMSG_PET_UNLEARN                                = 0x2F0,
    SMSG_PET_UNLEARN_CONFIRM                        = 0x2F1,
    SMSG_PARTY_MEMBER_STATS_FULL                    = 0x2F2,
    CMSG_PET_SPELL_AUTOCAST                         = 0x2F3,
    SMSG_WEATHER                                    = 0x2F4,
    SMSG_PLAY_TIME_WARNING                          = 0x2F5,
    SMSG_MINIGAME_SETUP                             = 0x2F6,
    SMSG_MINIGAME_STATE                             = 0x2F7,
    CMSG_MINIGAME_MOVE                              = 0x2F8,
    SMSG_MINIGAME_MOVE_FAILED                       = 0x2F9,
    SMSG_RAID_INSTANCE_MESSAGE                      = 0x2FA,
    SMSG_COMPRESSED_MOVES                           = 0x2FB,
    CMSG_GUILD_INFO_TEXT                            = 0x2FC,
    SMSG_CHAT_RESTRICTED                            = 0x2FD,
    SMSG_SPLINE_SET_RUN_SPEED                       = 0x2FE,
    SMSG_SPLINE_SET_RUN_BACK_SPEED                  = 0x2FF,
    SMSG_SPLINE_SET_SWIM_SPEED                      = 0x300,
    SMSG_SPLINE_SET_WALK_SPEED                      = 0x301,
    SMSG_SPLINE_SET_SWIM_BACK_SPEED                 = 0x302,
    SMSG_SPLINE_SET_TURN_RATE                       = 0x303,
    SMSG_SPLINE_MOVE_UNROOT                         = 0x304,
    SMSG_SPLINE_MOVE_FEATHER_FALL                   = 0x305,
    SMSG_SPLINE_MOVE_NORMAL_FALL                    = 0x306,
    SMSG_SPLINE_MOVE_SET_HOVER                      = 0x307,
    SMSG_SPLINE_MOVE_UNSET_HOVER                    = 0x308,
    SMSG_SPLINE_MOVE_WATER_WALK                     = 0x309,
    SMSG_SPLINE_MOVE_LAND_WALK                      = 0x30A,
    SMSG_SPLINE_MOVE_START_SWIM                     = 0x30B,
    SMSG_SPLINE_MOVE_STOP_SWIM                      = 0x30C,
    SMSG_SPLINE_MOVE_SET_RUN_MODE                   = 0x30D,
    SMSG_SPLINE_MOVE_SET_WALK_MODE                  = 0x30E,
    CMSG_GM_NUKE_ACCOUNT                            = 0x30F,
    MSG_GM_DESTROY_CORPSE                           = 0x310,
    CMSG_GM_DESTROY_ONLINE_CORPSE                   = 0x311,
    CMSG_ACTIVATETAXIEXPRESS                        = 0x312,
    SMSG_SET_FACTION_ATWAR                          = 0x313,
    SMSG_GAMETIMEBIAS_SET                           = 0x314,
    CMSG_DEBUG_ACTIONS_START                        = 0x315,
    CMSG_DEBUG_ACTIONS_STOP                         = 0x316,
    CMSG_SET_FACTION_INACTIVE                       = 0x317,
    CMSG_SET_WATCHED_FACTION                        = 0x318,
    MSG_MOVE_TIME_SKIPPED                           = 0x319,
    SMSG_SPLINE_MOVE_ROOT                           = 0x31A,
    CMSG_SET_EXPLORATION_ALL                        = 0x31B,
    SMSG_INVALIDATE_PLAYER                          = 0x31C,
    CMSG_RESET_INSTANCES                            = 0x31D,
    SMSG_INSTANCE_RESET                             = 0x31E,
    SMSG_INSTANCE_RESET_FAILED                      = 0x31F,
    SMSG_UPDATE_LAST_INSTANCE                       = 0x320,
    MSG_RAID_TARGET_UPDATE                          = 0x321,
    MSG_RAID_READY_CHECK                            = 0x322,
    CMSG_LUA_USAGE                                  = 0x323,
    SMSG_PET_ACTION_SOUND                           = 0x324,
    SMSG_PET_DISMISS_SOUND                          = 0x325,
    SMSG_GHOSTEE_GONE                               = 0x326,
    CMSG_GM_UPDATE_TICKET_STATUS                    = 0x327,
    SMSG_GM_TICKET_STATUS_UPDATE                    = 0x328,
    CMSG_GMSURVEY_SUBMIT                            = 0x32A,
    SMSG_UPDATE_INSTANCE_OWNERSHIP                  = 0x32B,
    CMSG_IGNORE_KNOCKBACK_CHEAT                     = 0x32C,
    SMSG_CHAT_PLAYER_AMBIGUOUS                      = 0x32D,
    MSG_DELAY_GHOST_TELEPORT                        = 0x32E,
    SMSG_SPELLINSTAKILLLOG                          = 0x32F,
    SMSG_SPELL_UPDATE_CHAIN_TARGETS                 = 0x330,
    CMSG_CHAT_FILTERED                              = 0x331,
    SMSG_EXPECTED_SPAM_RECORDS                      = 0x332,
    SMSG_SPELLSTEALLOG                              = 0x333,
    CMSG_LOTTERY_QUERY_OBSOLETE                     = 0x334,
    SMSG_LOTTERY_QUERY_RESULT_OBSOLETE              = 0x335,
    CMSG_BUY_LOTTERY_TICKET_OBSOLETE                = 0x336,
    SMSG_LOTTERY_RESULT_OBSOLETE                    = 0x337,
    SMSG_CHARACTER_PROFILE                          = 0x338,
    SMSG_CHARACTER_PROFILE_REALM_CONNECTED          = 0x339,
    SMSG_DEFENSE_MESSAGE                            = 0x33A,
    MSG_GM_RESETINSTANCELIMIT                       = 0x33C,
    SMSG_MOTD                                       = 0x33D,
    SMSG_MOVE_SET_FLIGHT                            = 0x33E,
    SMSG_MOVE_UNSET_FLIGHT                          = 0x33F,
    CMSG_MOVE_FLIGHT_ACK                            = 0x340,
    MSG_MOVE_START_SWIM_CHEAT                       = 0x341,
    MSG_MOVE_STOP_SWIM_CHEAT                        = 0x342,
    CMSG_CANCEL_MOUNT_AURA                          = 0x375,
    CMSG_CANCEL_TEMP_ENCHANTMENT                    = 0x379,
    CMSG_MAELSTROM_INVALIDATE_CACHE                 = 0x387,
    CMSG_SET_TAXI_BENCHMARK_MODE                    = 0x389,
    CMSG_MOVE_CHNG_TRANSPORT                        = 0x38D,
    MSG_PARTY_ASSIGNMENT                            = 0x38E,
    SMSG_OFFER_PETITION_ERROR                       = 0x38F,
    SMSG_RESET_FAILED_NOTIFY                        = 0x396,
    SMSG_REAL_GROUP_UPDATE                          = 0x397,
    SMSG_INIT_EXTRA_AURA_INFO                       = 0x3A3,
    SMSG_SET_EXTRA_AURA_INFO                        = 0x3A4,
    SMSG_SET_EXTRA_AURA_INFO_NEED_UPDATE            = 0x3A5,
    SMSG_SPELL_CHANCE_PROC_LOG                      = 0x3AA,
    CMSG_MOVE_SET_RUN_SPEED                         = 0x3AB,
    SMSG_DISMOUNT                                   = 0x3AC,
    MSG_RAID_READY_CHECK_CONFIRM                    = 0x3AE,
    SMSG_CLEAR_TARGET                               = 0x3BE,
    CMSG_BOT_DETECTED                               = 0x3BF,
    SMSG_KICK_REASON                                = 0x3C4,
    MSG_RAID_READY_CHECK_FINISHED                   = 0x3C5,
    CMSG_TARGET_CAST                                = 0x3CF,
    CMSG_TARGET_SCRIPT_CAST                         = 0x3D0,
    CMSG_CHANNEL_DISPLAY_LIST                       = 0x3D1,
    CMSG_GET_CHANNEL_MEMBER_COUNT                   = 0x3D3,
    SMSG_CHANNEL_MEMBER_COUNT                       = 0x3D4,
    CMSG_DEBUG_LIST_TARGETS                         = 0x3D7,
    SMSG_DEBUG_LIST_TARGETS                         = 0x3D8,
    CMSG_PARTY_SILENCE                              = 0x3DC,
    CMSG_PARTY_UNSILENCE                            = 0x3DD,
    MSG_NOTIFY_PARTY_SQUELCH                        = 0x3DE,
    SMSG_COMSAT_RECONNECT_TRY                       = 0x3DF,
    SMSG_COMSAT_DISCONNECT                          = 0x3E0,
    SMSG_COMSAT_CONNECT_FAIL                        = 0x3E1,
    CMSG_SET_CHANNEL_WATCH                          = 0x3EE,
    SMSG_USERLIST_ADD                               = 0x3EF,
    SMSG_USERLIST_REMOVE                            = 0x3F0,
    SMSG_USERLIST_UPDATE                            = 0x3F1,
    CMSG_CLEAR_CHANNEL_WATCH                        = 0x3F2,
    SMSG_GOGOGO_OBSOLETE                            = 0x3F4,
    SMSG_ECHO_PARTY_SQUELCH                         = 0x3F5,
    CMSG_SPELLCLICK                                 = 0x3F7,
    SMSG_LOOT_LIST                                  = 0x3F8,
    MSG_GUILD_PERMISSIONS                           = 0x3FC,
    MSG_GUILD_EVENT_LOG_QUERY                       = 0x3FE,
    CMSG_MAELSTROM_RENAME_GUILD                     = 0x3FF,
    CMSG_GET_MIRRORIMAGE_DATA                       = 0x400,
    SMSG_MIRRORIMAGE_DATA                           = 0x401,
    SMSG_FORCE_DISPLAY_UPDATE                       = 0x402,
    SMSG_SPELL_CHANCE_RESIST_PUSHBACK               = 0x403,
    CMSG_IGNORE_DIMINISHING_RETURNS_CHEAT           = 0x404,
    SMSG_IGNORE_DIMINISHING_RETURNS_CHEAT           = 0x405,
    CMSG_KEEP_ALIVE                                 = 0x406,
    SMSG_RAID_READY_CHECK_ERROR                     = 0x407,
    CMSG_OPT_OUT_OF_LOOT                            = 0x408,
    CMSG_SET_GRANTABLE_LEVELS                       = 0x40B,
    CMSG_GRANT_LEVEL                                = 0x40C,
    CMSG_DECLINE_CHANNEL_INVITE                     = 0x40F,
    CMSG_GROUPACTION_THROTTLED                      = 0x410,
    SMSG_OVERRIDE_LIGHT                             = 0x411,
    SMSG_TOTEM_CREATED                              = 0x412,
    CMSG_TOTEM_DESTROYED                            = 0x413,
    CMSG_EXPIRE_RAID_INSTANCE                       = 0x414,
    CMSG_NO_SPELL_VARIANCE                          = 0x415,
    CMSG_QUESTGIVER_STATUS_MULTIPLE_QUERY           = 0x416,
    SMSG_QUESTGIVER_STATUS_MULTIPLE                 = 0x417,
    CMSG_QUERY_SERVER_BUCK_DATA                     = 0x41A,
    CMSG_CLEAR_SERVER_BUCK_DATA                     = 0x41B,
    SMSG_SERVER_BUCK_DATA                           = 0x41C,
    SMSG_SEND_UNLEARN_SPELLS                        = 0x41D,
    SMSG_PROPOSE_LEVEL_GRANT                        = 0x41E,
    CMSG_ACCEPT_LEVEL_GRANT                         = 0x41F,
    SMSG_REFER_A_FRIEND_FAILURE                     = 0x420,
    SMSG_SUMMON_CANCEL                              = 0x423
} world_packets;

static const value_string world_packet_strings[] = {
    { MSG_NULL_ACTION, "MSG_NULL_ACTION" },
    { CMSG_BOOTME, "CMSG_BOOTME" },
    { CMSG_DBLOOKUP, "CMSG_DBLOOKUP" },
    { SMSG_DBLOOKUP, "SMSG_DBLOOKUP" },
    { CMSG_QUERY_OBJECT_POSITION, "CMSG_QUERY_OBJECT_POSITION" },
    { SMSG_QUERY_OBJECT_POSITION, "SMSG_QUERY_OBJECT_POSITION" },
    { CMSG_QUERY_OBJECT_ROTATION, "CMSG_QUERY_OBJECT_ROTATION" },
    { SMSG_QUERY_OBJECT_ROTATION, "SMSG_QUERY_OBJECT_ROTATION" },
    { CMSG_WORLD_TELEPORT, "CMSG_WORLD_TELEPORT" },
    { CMSG_TELEPORT_TO_UNIT, "CMSG_TELEPORT_TO_UNIT" },
    { CMSG_ZONE_MAP, "CMSG_ZONE_MAP" },
    { SMSG_ZONE_MAP, "SMSG_ZONE_MAP" },
    { CMSG_DEBUG_CHANGECELLZONE, "CMSG_DEBUG_CHANGECELLZONE" },
    { CMSG_EMBLAZON_TABARD_OBSOLETE, "CMSG_EMBLAZON_TABARD_OBSOLETE" },
    { CMSG_UNEMBLAZON_TABARD_OBSOLETE, "CMSG_UNEMBLAZON_TABARD_OBSOLETE" },
    { CMSG_RECHARGE, "CMSG_RECHARGE" },
    { CMSG_LEARN_SPELL, "CMSG_LEARN_SPELL" },
    { CMSG_CREATEMONSTER, "CMSG_CREATEMONSTER" },
    { CMSG_DESTROYMONSTER, "CMSG_DESTROYMONSTER" },
    { CMSG_CREATEITEM, "CMSG_CREATEITEM" },
    { CMSG_CREATEGAMEOBJECT, "CMSG_CREATEGAMEOBJECT" },
    { SMSG_CHECK_FOR_BOTS, "SMSG_CHECK_FOR_BOTS" },
    { CMSG_MAKEMONSTERATTACKGUID, "CMSG_MAKEMONSTERATTACKGUID" },
    { CMSG_BOT_DETECTED2, "CMSG_BOT_DETECTED2" },
    { CMSG_FORCEACTION, "CMSG_FORCEACTION" },
    { CMSG_FORCEACTIONONOTHER, "CMSG_FORCEACTIONONOTHER" },
    { CMSG_FORCEACTIONSHOW, "CMSG_FORCEACTIONSHOW" },
    { SMSG_FORCEACTIONSHOW, "SMSG_FORCEACTIONSHOW" },
    { CMSG_PETGODMODE, "CMSG_PETGODMODE" },
    { SMSG_PETGODMODE, "SMSG_PETGODMODE" },
    { SMSG_DEBUGINFOSPELLMISS_OBSOLETE, "SMSG_DEBUGINFOSPELLMISS_OBSOLETE" },
    { CMSG_WEATHER_SPEED_CHEAT, "CMSG_WEATHER_SPEED_CHEAT" },
    { CMSG_UNDRESSPLAYER, "CMSG_UNDRESSPLAYER" },
    { CMSG_BEASTMASTER, "CMSG_BEASTMASTER" },
    { CMSG_GODMODE, "CMSG_GODMODE" },
    { SMSG_GODMODE, "SMSG_GODMODE" },
    { CMSG_CHEAT_SETMONEY, "CMSG_CHEAT_SETMONEY" },
    { CMSG_LEVEL_CHEAT, "CMSG_LEVEL_CHEAT" },
    { CMSG_PET_LEVEL_CHEAT, "CMSG_PET_LEVEL_CHEAT" },
    { CMSG_SET_WORLDSTATE, "CMSG_SET_WORLDSTATE" },
    { CMSG_COOLDOWN_CHEAT, "CMSG_COOLDOWN_CHEAT" },
    { CMSG_USE_SKILL_CHEAT, "CMSG_USE_SKILL_CHEAT" },
    { CMSG_FLAG_QUEST, "CMSG_FLAG_QUEST" },
    { CMSG_FLAG_QUEST_FINISH, "CMSG_FLAG_QUEST_FINISH" },
    { CMSG_CLEAR_QUEST, "CMSG_CLEAR_QUEST" },
    { CMSG_SEND_EVENT, "CMSG_SEND_EVENT" },
    { CMSG_DEBUG_AISTATE, "CMSG_DEBUG_AISTATE" },
    { SMSG_DEBUG_AISTATE, "SMSG_DEBUG_AISTATE" },
    { CMSG_DISABLE_PVP_CHEAT, "CMSG_DISABLE_PVP_CHEAT" },
    { CMSG_ADVANCE_SPAWN_TIME, "CMSG_ADVANCE_SPAWN_TIME" },
    { CMSG_PVP_PORT_OBSOLETE, "CMSG_PVP_PORT_OBSOLETE" },
    { CMSG_AUTH_SRP6_BEGIN, "CMSG_AUTH_SRP6_BEGIN" },
    { CMSG_AUTH_SRP6_PROOF, "CMSG_AUTH_SRP6_PROOF" },
    { CMSG_AUTH_SRP6_RECODE, "CMSG_AUTH_SRP6_RECODE" },
    { CMSG_CHAR_CREATE, "CMSG_CHAR_CREATE" },
    { CMSG_CHAR_ENUM, "CMSG_CHAR_ENUM" },
    { CMSG_CHAR_DELETE, "CMSG_CHAR_DELETE" },
    { SMSG_AUTH_SRP6_RESPONSE, "SMSG_AUTH_SRP6_RESPONSE" },
    { SMSG_CHAR_CREATE, "SMSG_CHAR_CREATE" },
    { SMSG_CHAR_ENUM, "SMSG_CHAR_ENUM" },
    { SMSG_CHAR_DELETE, "SMSG_CHAR_DELETE" },
    { CMSG_PLAYER_LOGIN, "CMSG_PLAYER_LOGIN" },
    { SMSG_NEW_WORLD, "SMSG_NEW_WORLD" },
    { SMSG_TRANSFER_PENDING, "SMSG_TRANSFER_PENDING" },
    { SMSG_TRANSFER_ABORTED, "SMSG_TRANSFER_ABORTED" },
    { SMSG_CHARACTER_LOGIN_FAILED, "SMSG_CHARACTER_LOGIN_FAILED" },
    { SMSG_LOGIN_SETTIMESPEED, "SMSG_LOGIN_SETTIMESPEED" },
    { SMSG_GAMETIME_UPDATE, "SMSG_GAMETIME_UPDATE" },
    { CMSG_GAMETIME_SET, "CMSG_GAMETIME_SET" },
    { SMSG_GAMETIME_SET, "SMSG_GAMETIME_SET" },
    { CMSG_GAMESPEED_SET, "CMSG_GAMESPEED_SET" },
    { SMSG_GAMESPEED_SET, "SMSG_GAMESPEED_SET" },
    { CMSG_SERVERTIME, "CMSG_SERVERTIME" },
    { SMSG_SERVERTIME, "SMSG_SERVERTIME" },
    { CMSG_PLAYER_LOGOUT, "CMSG_PLAYER_LOGOUT" },
    { CMSG_LOGOUT_REQUEST, "CMSG_LOGOUT_REQUEST" },
    { SMSG_LOGOUT_RESPONSE, "SMSG_LOGOUT_RESPONSE" },
    { SMSG_LOGOUT_COMPLETE, "SMSG_LOGOUT_COMPLETE" },
    { CMSG_LOGOUT_CANCEL, "CMSG_LOGOUT_CANCEL" },
    { SMSG_LOGOUT_CANCEL_ACK, "SMSG_LOGOUT_CANCEL_ACK" },
    { CMSG_NAME_QUERY, "CMSG_NAME_QUERY" },
    { SMSG_NAME_QUERY_RESPONSE, "SMSG_NAME_QUERY_RESPONSE" },
    { CMSG_PET_NAME_QUERY, "CMSG_PET_NAME_QUERY" },
    { SMSG_PET_NAME_QUERY_RESPONSE, "SMSG_PET_NAME_QUERY_RESPONSE" },
    { CMSG_GUILD_QUERY, "CMSG_GUILD_QUERY" },
    { SMSG_GUILD_QUERY_RESPONSE, "SMSG_GUILD_QUERY_RESPONSE" },
    { CMSG_ITEM_QUERY_SINGLE, "CMSG_ITEM_QUERY_SINGLE" },
    { CMSG_ITEM_QUERY_MULTIPLE, "CMSG_ITEM_QUERY_MULTIPLE" },
    { SMSG_ITEM_QUERY_SINGLE_RESPONSE, "SMSG_ITEM_QUERY_SINGLE_RESPONSE" },
    { SMSG_ITEM_QUERY_MULTIPLE_RESPONSE, "SMSG_ITEM_QUERY_MULTIPLE_RESPONSE" },
    { CMSG_PAGE_TEXT_QUERY, "CMSG_PAGE_TEXT_QUERY" },
    { SMSG_PAGE_TEXT_QUERY_RESPONSE, "SMSG_PAGE_TEXT_QUERY_RESPONSE" },
    { CMSG_QUEST_QUERY, "CMSG_QUEST_QUERY" },
    { SMSG_QUEST_QUERY_RESPONSE, "SMSG_QUEST_QUERY_RESPONSE" },
    { CMSG_GAMEOBJECT_QUERY, "CMSG_GAMEOBJECT_QUERY" },
    { SMSG_GAMEOBJECT_QUERY_RESPONSE, "SMSG_GAMEOBJECT_QUERY_RESPONSE" },
    { CMSG_CREATURE_QUERY, "CMSG_CREATURE_QUERY" },
    { SMSG_CREATURE_QUERY_RESPONSE, "SMSG_CREATURE_QUERY_RESPONSE" },
    { CMSG_WHO, "CMSG_WHO" },
    { SMSG_WHO, "SMSG_WHO" },
    { CMSG_WHOIS, "CMSG_WHOIS" },
    { SMSG_WHOIS, "SMSG_WHOIS" },
    { CMSG_FRIEND_LIST, "CMSG_FRIEND_LIST" },
    { SMSG_FRIEND_LIST, "SMSG_FRIEND_LIST" },
    { SMSG_FRIEND_STATUS, "SMSG_FRIEND_STATUS" },
    { CMSG_ADD_FRIEND, "CMSG_ADD_FRIEND" },
    { CMSG_DEL_FRIEND, "CMSG_DEL_FRIEND" },
    { SMSG_IGNORE_LIST, "SMSG_IGNORE_LIST" },
    { CMSG_ADD_IGNORE, "CMSG_ADD_IGNORE" },
    { CMSG_DEL_IGNORE, "CMSG_DEL_IGNORE" },
    { CMSG_GROUP_INVITE, "CMSG_GROUP_INVITE" },
    { SMSG_GROUP_INVITE, "SMSG_GROUP_INVITE" },
    { CMSG_GROUP_CANCEL, "CMSG_GROUP_CANCEL" },
    { SMSG_GROUP_CANCEL, "SMSG_GROUP_CANCEL" },
    { CMSG_GROUP_ACCEPT, "CMSG_GROUP_ACCEPT" },
    { CMSG_GROUP_DECLINE, "CMSG_GROUP_DECLINE" },
    { SMSG_GROUP_DECLINE, "SMSG_GROUP_DECLINE" },
    { CMSG_GROUP_UNINVITE, "CMSG_GROUP_UNINVITE" },
    { CMSG_GROUP_UNINVITE_GUID, "CMSG_GROUP_UNINVITE_GUID" },
    { SMSG_GROUP_UNINVITE, "SMSG_GROUP_UNINVITE" },
    { CMSG_GROUP_SET_LEADER, "CMSG_GROUP_SET_LEADER" },
    { SMSG_GROUP_SET_LEADER, "SMSG_GROUP_SET_LEADER" },
    { CMSG_LOOT_METHOD, "CMSG_LOOT_METHOD" },
    { CMSG_GROUP_DISBAND, "CMSG_GROUP_DISBAND" },
    { SMSG_GROUP_DESTROYED, "SMSG_GROUP_DESTROYED" },
    { SMSG_GROUP_LIST, "SMSG_GROUP_LIST" },
    { SMSG_PARTY_MEMBER_STATS, "SMSG_PARTY_MEMBER_STATS" },
    { SMSG_PARTY_COMMAND_RESULT, "SMSG_PARTY_COMMAND_RESULT" },
    { UMSG_UPDATE_GROUP_MEMBERS, "UMSG_UPDATE_GROUP_MEMBERS" },
    { CMSG_GUILD_CREATE, "CMSG_GUILD_CREATE" },
    { CMSG_GUILD_INVITE, "CMSG_GUILD_INVITE" },
    { SMSG_GUILD_INVITE, "SMSG_GUILD_INVITE" },
    { CMSG_GUILD_ACCEPT, "CMSG_GUILD_ACCEPT" },
    { CMSG_GUILD_DECLINE, "CMSG_GUILD_DECLINE" },
    { SMSG_GUILD_DECLINE, "SMSG_GUILD_DECLINE" },
    { CMSG_GUILD_INFO, "CMSG_GUILD_INFO" },
    { SMSG_GUILD_INFO, "SMSG_GUILD_INFO" },
    { CMSG_GUILD_ROSTER, "CMSG_GUILD_ROSTER" },
    { SMSG_GUILD_ROSTER, "SMSG_GUILD_ROSTER" },
    { CMSG_GUILD_PROMOTE, "CMSG_GUILD_PROMOTE" },
    { CMSG_GUILD_DEMOTE, "CMSG_GUILD_DEMOTE" },
    { CMSG_GUILD_LEAVE, "CMSG_GUILD_LEAVE" },
    { CMSG_GUILD_REMOVE, "CMSG_GUILD_REMOVE" },
    { CMSG_GUILD_DISBAND, "CMSG_GUILD_DISBAND" },
    { CMSG_GUILD_LEADER, "CMSG_GUILD_LEADER" },
    { CMSG_GUILD_MOTD, "CMSG_GUILD_MOTD" },
    { SMSG_GUILD_EVENT, "SMSG_GUILD_EVENT" },
    { SMSG_GUILD_COMMAND_RESULT, "SMSG_GUILD_COMMAND_RESULT" },
    { UMSG_UPDATE_GUILD, "UMSG_UPDATE_GUILD" },
    { CMSG_MESSAGECHAT, "CMSG_MESSAGECHAT" },
    { SMSG_MESSAGECHAT, "SMSG_MESSAGECHAT" },
    { CMSG_JOIN_CHANNEL, "CMSG_JOIN_CHANNEL" },
    { CMSG_LEAVE_CHANNEL, "CMSG_LEAVE_CHANNEL" },
    { SMSG_CHANNEL_NOTIFY, "SMSG_CHANNEL_NOTIFY" },
    { CMSG_CHANNEL_LIST, "CMSG_CHANNEL_LIST" },
    { SMSG_CHANNEL_LIST, "SMSG_CHANNEL_LIST" },
    { CMSG_CHANNEL_PASSWORD, "CMSG_CHANNEL_PASSWORD" },
    { CMSG_CHANNEL_SET_OWNER, "CMSG_CHANNEL_SET_OWNER" },
    { CMSG_CHANNEL_OWNER, "CMSG_CHANNEL_OWNER" },
    { CMSG_CHANNEL_MODERATOR, "CMSG_CHANNEL_MODERATOR" },
    { CMSG_CHANNEL_UNMODERATOR, "CMSG_CHANNEL_UNMODERATOR" },
    { CMSG_CHANNEL_MUTE, "CMSG_CHANNEL_MUTE" },
    { CMSG_CHANNEL_UNMUTE, "CMSG_CHANNEL_UNMUTE" },
    { CMSG_CHANNEL_INVITE, "CMSG_CHANNEL_INVITE" },
    { CMSG_CHANNEL_KICK, "CMSG_CHANNEL_KICK" },
    { CMSG_CHANNEL_BAN, "CMSG_CHANNEL_BAN" },
    { CMSG_CHANNEL_UNBAN, "CMSG_CHANNEL_UNBAN" },
    { CMSG_CHANNEL_ANNOUNCEMENTS, "CMSG_CHANNEL_ANNOUNCEMENTS" },
    { CMSG_CHANNEL_MODERATE, "CMSG_CHANNEL_MODERATE" },
    { SMSG_UPDATE_OBJECT, "SMSG_UPDATE_OBJECT" },
    { SMSG_DESTROY_OBJECT, "SMSG_DESTROY_OBJECT" },
    { CMSG_USE_ITEM, "CMSG_USE_ITEM" },
    { CMSG_OPEN_ITEM, "CMSG_OPEN_ITEM" },
    { CMSG_READ_ITEM, "CMSG_READ_ITEM" },
    { SMSG_READ_ITEM_OK, "SMSG_READ_ITEM_OK" },
    { SMSG_READ_ITEM_FAILED, "SMSG_READ_ITEM_FAILED" },
    { SMSG_ITEM_COOLDOWN, "SMSG_ITEM_COOLDOWN" },
    { CMSG_GAMEOBJ_USE, "CMSG_GAMEOBJ_USE" },
    { CMSG_GAMEOBJ_CHAIR_USE_OBSOLETE, "CMSG_GAMEOBJ_CHAIR_USE_OBSOLETE" },
    { SMSG_GAMEOBJECT_CUSTOM_ANIM, "SMSG_GAMEOBJECT_CUSTOM_ANIM" },
    { CMSG_AREATRIGGER, "CMSG_AREATRIGGER" },
    { MSG_MOVE_START_FORWARD, "MSG_MOVE_START_FORWARD" },
    { MSG_MOVE_START_BACKWARD, "MSG_MOVE_START_BACKWARD" },
    { MSG_MOVE_STOP, "MSG_MOVE_STOP" },
    { MSG_MOVE_START_STRAFE_LEFT, "MSG_MOVE_START_STRAFE_LEFT" },
    { MSG_MOVE_START_STRAFE_RIGHT, "MSG_MOVE_START_STRAFE_RIGHT" },
    { MSG_MOVE_STOP_STRAFE, "MSG_MOVE_STOP_STRAFE" },
    { MSG_MOVE_JUMP, "MSG_MOVE_JUMP" },
    { MSG_MOVE_START_TURN_LEFT, "MSG_MOVE_START_TURN_LEFT" },
    { MSG_MOVE_START_TURN_RIGHT, "MSG_MOVE_START_TURN_RIGHT" },
    { MSG_MOVE_STOP_TURN, "MSG_MOVE_STOP_TURN" },
    { MSG_MOVE_START_PITCH_UP, "MSG_MOVE_START_PITCH_UP" },
    { MSG_MOVE_START_PITCH_DOWN, "MSG_MOVE_START_PITCH_DOWN" },
    { MSG_MOVE_STOP_PITCH, "MSG_MOVE_STOP_PITCH" },
    { MSG_MOVE_SET_RUN_MODE, "MSG_MOVE_SET_RUN_MODE" },
    { MSG_MOVE_SET_WALK_MODE, "MSG_MOVE_SET_WALK_MODE" },
    { MSG_MOVE_TOGGLE_LOGGING, "MSG_MOVE_TOGGLE_LOGGING" },
    { MSG_MOVE_TELEPORT, "MSG_MOVE_TELEPORT" },
    { MSG_MOVE_TELEPORT_CHEAT, "MSG_MOVE_TELEPORT_CHEAT" },
    { MSG_MOVE_TELEPORT_ACK, "MSG_MOVE_TELEPORT_ACK" },
    { MSG_MOVE_TOGGLE_FALL_LOGGING, "MSG_MOVE_TOGGLE_FALL_LOGGING" },
    { MSG_MOVE_FALL_LAND, "MSG_MOVE_FALL_LAND" },
    { MSG_MOVE_START_SWIM, "MSG_MOVE_START_SWIM" },
    { MSG_MOVE_STOP_SWIM, "MSG_MOVE_STOP_SWIM" },
    { MSG_MOVE_SET_RUN_SPEED_CHEAT, "MSG_MOVE_SET_RUN_SPEED_CHEAT" },
    { MSG_MOVE_SET_RUN_SPEED, "MSG_MOVE_SET_RUN_SPEED" },
    { MSG_MOVE_SET_RUN_BACK_SPEED_CHEAT, "MSG_MOVE_SET_RUN_BACK_SPEED_CHEAT" },
    { MSG_MOVE_SET_RUN_BACK_SPEED, "MSG_MOVE_SET_RUN_BACK_SPEED" },
    { MSG_MOVE_SET_WALK_SPEED_CHEAT, "MSG_MOVE_SET_WALK_SPEED_CHEAT" },
    { MSG_MOVE_SET_WALK_SPEED, "MSG_MOVE_SET_WALK_SPEED" },
    { MSG_MOVE_SET_SWIM_SPEED_CHEAT, "MSG_MOVE_SET_SWIM_SPEED_CHEAT" },
    { MSG_MOVE_SET_SWIM_SPEED, "MSG_MOVE_SET_SWIM_SPEED" },
    { MSG_MOVE_SET_SWIM_BACK_SPEED_CHEAT, "MSG_MOVE_SET_SWIM_BACK_SPEED_CHEAT" },
    { MSG_MOVE_SET_SWIM_BACK_SPEED, "MSG_MOVE_SET_SWIM_BACK_SPEED" },
    { MSG_MOVE_SET_ALL_SPEED_CHEAT, "MSG_MOVE_SET_ALL_SPEED_CHEAT" },
    { MSG_MOVE_SET_TURN_RATE_CHEAT, "MSG_MOVE_SET_TURN_RATE_CHEAT" },
    { MSG_MOVE_SET_TURN_RATE, "MSG_MOVE_SET_TURN_RATE" },
    { MSG_MOVE_TOGGLE_COLLISION_CHEAT, "MSG_MOVE_TOGGLE_COLLISION_CHEAT" },
    { MSG_MOVE_SET_FACING, "MSG_MOVE_SET_FACING" },
    { MSG_MOVE_SET_PITCH, "MSG_MOVE_SET_PITCH" },
    { MSG_MOVE_WORLDPORT_ACK, "MSG_MOVE_WORLDPORT_ACK" },
    { SMSG_MONSTER_MOVE, "SMSG_MONSTER_MOVE" },
    { SMSG_MOVE_WATER_WALK, "SMSG_MOVE_WATER_WALK" },
    { SMSG_MOVE_LAND_WALK, "SMSG_MOVE_LAND_WALK" },
    { MSG_MOVE_SET_RAW_POSITION_ACK, "MSG_MOVE_SET_RAW_POSITION_ACK" },
    { CMSG_MOVE_SET_RAW_POSITION, "CMSG_MOVE_SET_RAW_POSITION" },
    { SMSG_FORCE_RUN_SPEED_CHANGE, "SMSG_FORCE_RUN_SPEED_CHANGE" },
    { CMSG_FORCE_RUN_SPEED_CHANGE_ACK, "CMSG_FORCE_RUN_SPEED_CHANGE_ACK" },
    { SMSG_FORCE_RUN_BACK_SPEED_CHANGE, "SMSG_FORCE_RUN_BACK_SPEED_CHANGE" },
    { CMSG_FORCE_RUN_BACK_SPEED_CHANGE_ACK, "CMSG_FORCE_RUN_BACK_SPEED_CHANGE_ACK" },
    { SMSG_FORCE_SWIM_SPEED_CHANGE, "SMSG_FORCE_SWIM_SPEED_CHANGE" },
    { CMSG_FORCE_SWIM_SPEED_CHANGE_ACK, "CMSG_FORCE_SWIM_SPEED_CHANGE_ACK" },
    { SMSG_FORCE_MOVE_ROOT, "SMSG_FORCE_MOVE_ROOT" },
    { CMSG_FORCE_MOVE_ROOT_ACK, "CMSG_FORCE_MOVE_ROOT_ACK" },
    { SMSG_FORCE_MOVE_UNROOT, "SMSG_FORCE_MOVE_UNROOT" },
    { CMSG_FORCE_MOVE_UNROOT_ACK, "CMSG_FORCE_MOVE_UNROOT_ACK" },
    { MSG_MOVE_ROOT, "MSG_MOVE_ROOT" },
    { MSG_MOVE_UNROOT, "MSG_MOVE_UNROOT" },
    { MSG_MOVE_HEARTBEAT, "MSG_MOVE_HEARTBEAT" },
    { SMSG_MOVE_KNOCK_BACK, "SMSG_MOVE_KNOCK_BACK" },
    { CMSG_MOVE_KNOCK_BACK_ACK, "CMSG_MOVE_KNOCK_BACK_ACK" },
    { MSG_MOVE_KNOCK_BACK, "MSG_MOVE_KNOCK_BACK" },
    { SMSG_MOVE_FEATHER_FALL, "SMSG_MOVE_FEATHER_FALL" },
    { SMSG_MOVE_NORMAL_FALL, "SMSG_MOVE_NORMAL_FALL" },
    { SMSG_MOVE_SET_HOVER, "SMSG_MOVE_SET_HOVER" },
    { SMSG_MOVE_UNSET_HOVER, "SMSG_MOVE_UNSET_HOVER" },
    { CMSG_MOVE_HOVER_ACK, "CMSG_MOVE_HOVER_ACK" },
    { MSG_MOVE_HOVER, "MSG_MOVE_HOVER" },
    { CMSG_TRIGGER_CINEMATIC_CHEAT, "CMSG_TRIGGER_CINEMATIC_CHEAT" },
    { CMSG_OPENING_CINEMATIC, "CMSG_OPENING_CINEMATIC" },
    { SMSG_TRIGGER_CINEMATIC, "SMSG_TRIGGER_CINEMATIC" },
    { CMSG_NEXT_CINEMATIC_CAMERA, "CMSG_NEXT_CINEMATIC_CAMERA" },
    { CMSG_COMPLETE_CINEMATIC, "CMSG_COMPLETE_CINEMATIC" },
    { SMSG_TUTORIAL_FLAGS, "SMSG_TUTORIAL_FLAGS" },
    { CMSG_TUTORIAL_FLAG, "CMSG_TUTORIAL_FLAG" },
    { CMSG_TUTORIAL_CLEAR, "CMSG_TUTORIAL_CLEAR" },
    { CMSG_TUTORIAL_RESET, "CMSG_TUTORIAL_RESET" },
    { CMSG_STANDSTATECHANGE, "CMSG_STANDSTATECHANGE" },
    { CMSG_EMOTE, "CMSG_EMOTE" },
    { SMSG_EMOTE, "SMSG_EMOTE" },
    { CMSG_TEXT_EMOTE, "CMSG_TEXT_EMOTE" },
    { SMSG_TEXT_EMOTE, "SMSG_TEXT_EMOTE" },
    { CMSG_AUTOEQUIP_GROUND_ITEM, "CMSG_AUTOEQUIP_GROUND_ITEM" },
    { CMSG_AUTOSTORE_GROUND_ITEM, "CMSG_AUTOSTORE_GROUND_ITEM" },
    { CMSG_AUTOSTORE_LOOT_ITEM, "CMSG_AUTOSTORE_LOOT_ITEM" },
    { CMSG_STORE_LOOT_IN_SLOT, "CMSG_STORE_LOOT_IN_SLOT" },
    { CMSG_AUTOEQUIP_ITEM, "CMSG_AUTOEQUIP_ITEM" },
    { CMSG_AUTOSTORE_BAG_ITEM, "CMSG_AUTOSTORE_BAG_ITEM" },
    { CMSG_SWAP_ITEM, "CMSG_SWAP_ITEM" },
    { CMSG_SWAP_INV_ITEM, "CMSG_SWAP_INV_ITEM" },
    { CMSG_SPLIT_ITEM, "CMSG_SPLIT_ITEM" },
    { CMSG_AUTOEQUIP_ITEM_SLOT, "CMSG_AUTOEQUIP_ITEM_SLOT" },
    { OBSOLETE_DROP_ITEM, "OBSOLETE_DROP_ITEM" },
    { CMSG_DESTROYITEM, "CMSG_DESTROYITEM" },
    { SMSG_INVENTORY_CHANGE_FAILURE, "SMSG_INVENTORY_CHANGE_FAILURE" },
    { SMSG_OPEN_CONTAINER, "SMSG_OPEN_CONTAINER" },
    { CMSG_INSPECT, "CMSG_INSPECT" },
    { SMSG_INSPECT, "SMSG_INSPECT" },
    { CMSG_INITIATE_TRADE, "CMSG_INITIATE_TRADE" },
    { CMSG_BEGIN_TRADE, "CMSG_BEGIN_TRADE" },
    { CMSG_BUSY_TRADE, "CMSG_BUSY_TRADE" },
    { CMSG_IGNORE_TRADE, "CMSG_IGNORE_TRADE" },
    { CMSG_ACCEPT_TRADE, "CMSG_ACCEPT_TRADE" },
    { CMSG_UNACCEPT_TRADE, "CMSG_UNACCEPT_TRADE" },
    { CMSG_CANCEL_TRADE, "CMSG_CANCEL_TRADE" },
    { CMSG_SET_TRADE_ITEM, "CMSG_SET_TRADE_ITEM" },
    { CMSG_CLEAR_TRADE_ITEM, "CMSG_CLEAR_TRADE_ITEM" },
    { CMSG_SET_TRADE_GOLD, "CMSG_SET_TRADE_GOLD" },
    { SMSG_TRADE_STATUS, "SMSG_TRADE_STATUS" },
    { SMSG_TRADE_STATUS_EXTENDED, "SMSG_TRADE_STATUS_EXTENDED" },
    { SMSG_INITIALIZE_FACTIONS, "SMSG_INITIALIZE_FACTIONS" },
    { SMSG_SET_FACTION_VISIBLE, "SMSG_SET_FACTION_VISIBLE" },
    { SMSG_SET_FACTION_STANDING, "SMSG_SET_FACTION_STANDING" },
    { CMSG_SET_FACTION_ATWAR, "CMSG_SET_FACTION_ATWAR" },
    { CMSG_SET_FACTION_CHEAT, "CMSG_SET_FACTION_CHEAT" },
    { SMSG_SET_PROFICIENCY, "SMSG_SET_PROFICIENCY" },
    { CMSG_SET_ACTION_BUTTON, "CMSG_SET_ACTION_BUTTON" },
    { SMSG_ACTION_BUTTONS, "SMSG_ACTION_BUTTONS" },
    { SMSG_INITIAL_SPELLS, "SMSG_INITIAL_SPELLS" },
    { SMSG_LEARNED_SPELL, "SMSG_LEARNED_SPELL" },
    { SMSG_SUPERCEDED_SPELL, "SMSG_SUPERCEDED_SPELL" },
    { CMSG_NEW_SPELL_SLOT, "CMSG_NEW_SPELL_SLOT" },
    { CMSG_CAST_SPELL, "CMSG_CAST_SPELL" },
    { CMSG_CANCEL_CAST, "CMSG_CANCEL_CAST" },
    { SMSG_CAST_RESULT, "SMSG_CAST_RESULT" },
    { SMSG_SPELL_START, "SMSG_SPELL_START" },
    { SMSG_SPELL_GO, "SMSG_SPELL_GO" },
    { SMSG_SPELL_FAILURE, "SMSG_SPELL_FAILURE" },
    { SMSG_SPELL_COOLDOWN, "SMSG_SPELL_COOLDOWN" },
    { SMSG_COOLDOWN_EVENT, "SMSG_COOLDOWN_EVENT" },
    { CMSG_CANCEL_AURA, "CMSG_CANCEL_AURA" },
    { SMSG_UPDATE_AURA_DURATION, "SMSG_UPDATE_AURA_DURATION" },
    { SMSG_PET_CAST_FAILED, "SMSG_PET_CAST_FAILED" },
    { MSG_CHANNEL_START, "MSG_CHANNEL_START" },
    { MSG_CHANNEL_UPDATE, "MSG_CHANNEL_UPDATE" },
    { CMSG_CANCEL_CHANNELLING, "CMSG_CANCEL_CHANNELLING" },
    { SMSG_AI_REACTION, "SMSG_AI_REACTION" },
    { CMSG_SET_SELECTION, "CMSG_SET_SELECTION" },
    { CMSG_SET_TARGET_OBSOLETE, "CMSG_SET_TARGET_OBSOLETE" },
    { CMSG_UNUSED, "CMSG_UNUSED" },
    { CMSG_UNUSED2, "CMSG_UNUSED2" },
    { CMSG_ATTACKSWING, "CMSG_ATTACKSWING" },
    { CMSG_ATTACKSTOP, "CMSG_ATTACKSTOP" },
    { SMSG_ATTACKSTART, "SMSG_ATTACKSTART" },
    { SMSG_ATTACKSTOP, "SMSG_ATTACKSTOP" },
    { SMSG_ATTACKSWING_NOTINRANGE, "SMSG_ATTACKSWING_NOTINRANGE" },
    { SMSG_ATTACKSWING_BADFACING, "SMSG_ATTACKSWING_BADFACING" },
    { SMSG_ATTACKSWING_NOTSTANDING, "SMSG_ATTACKSWING_NOTSTANDING" },
    { SMSG_ATTACKSWING_DEADTARGET, "SMSG_ATTACKSWING_DEADTARGET" },
    { SMSG_ATTACKSWING_CANT_ATTACK, "SMSG_ATTACKSWING_CANT_ATTACK" },
    { SMSG_ATTACKERSTATEUPDATE, "SMSG_ATTACKERSTATEUPDATE" },
    { SMSG_VICTIMSTATEUPDATE_OBSOLETE, "SMSG_VICTIMSTATEUPDATE_OBSOLETE" },
    { SMSG_DAMAGE_DONE_OBSOLETE, "SMSG_DAMAGE_DONE_OBSOLETE" },
    { SMSG_DAMAGE_TAKEN_OBSOLETE, "SMSG_DAMAGE_TAKEN_OBSOLETE" },
    { SMSG_CANCEL_COMBAT, "SMSG_CANCEL_COMBAT" },
    { SMSG_PLAYER_COMBAT_XP_GAIN_OBSOLETE, "SMSG_PLAYER_COMBAT_XP_GAIN_OBSOLETE" },
    { SMSG_SPELLHEALLOG, "SMSG_SPELLHEALLOG" },
    { SMSG_SPELLENERGIZELOG, "SMSG_SPELLENERGIZELOG" },
    { CMSG_SHEATHE_OBSOLETE, "CMSG_SHEATHE_OBSOLETE" },
    { CMSG_SAVE_PLAYER, "CMSG_SAVE_PLAYER" },
    { CMSG_SETDEATHBINDPOINT, "CMSG_SETDEATHBINDPOINT" },
    { SMSG_BINDPOINTUPDATE, "SMSG_BINDPOINTUPDATE" },
    { CMSG_GETDEATHBINDZONE, "CMSG_GETDEATHBINDZONE" },
    { SMSG_BINDZONEREPLY, "SMSG_BINDZONEREPLY" },
    { SMSG_PLAYERBOUND, "SMSG_PLAYERBOUND" },
    { SMSG_CLIENT_CONTROL_UPDATE, "SMSG_CLIENT_CONTROL_UPDATE" },
    { CMSG_REPOP_REQUEST, "CMSG_REPOP_REQUEST" },
    { SMSG_RESURRECT_REQUEST, "SMSG_RESURRECT_REQUEST" },
    { CMSG_RESURRECT_RESPONSE, "CMSG_RESURRECT_RESPONSE" },
    { CMSG_LOOT, "CMSG_LOOT" },
    { CMSG_LOOT_MONEY, "CMSG_LOOT_MONEY" },
    { CMSG_LOOT_RELEASE, "CMSG_LOOT_RELEASE" },
    { SMSG_LOOT_RESPONSE, "SMSG_LOOT_RESPONSE" },
    { SMSG_LOOT_RELEASE_RESPONSE, "SMSG_LOOT_RELEASE_RESPONSE" },
    { SMSG_LOOT_REMOVED, "SMSG_LOOT_REMOVED" },
    { SMSG_LOOT_MONEY_NOTIFY, "SMSG_LOOT_MONEY_NOTIFY" },
    { SMSG_LOOT_ITEM_NOTIFY, "SMSG_LOOT_ITEM_NOTIFY" },
    { SMSG_LOOT_CLEAR_MONEY, "SMSG_LOOT_CLEAR_MONEY" },
    { SMSG_ITEM_PUSH_RESULT, "SMSG_ITEM_PUSH_RESULT" },
    { SMSG_DUEL_REQUESTED, "SMSG_DUEL_REQUESTED" },
    { SMSG_DUEL_OUTOFBOUNDS, "SMSG_DUEL_OUTOFBOUNDS" },
    { SMSG_DUEL_INBOUNDS, "SMSG_DUEL_INBOUNDS" },
    { SMSG_DUEL_COMPLETE, "SMSG_DUEL_COMPLETE" },
    { SMSG_DUEL_WINNER, "SMSG_DUEL_WINNER" },
    { CMSG_DUEL_ACCEPTED, "CMSG_DUEL_ACCEPTED" },
    { CMSG_DUEL_CANCELLED, "CMSG_DUEL_CANCELLED" },
    { SMSG_MOUNTRESULT, "SMSG_MOUNTRESULT" },
    { SMSG_DISMOUNTRESULT, "SMSG_DISMOUNTRESULT" },
    { SMSG_PUREMOUNT_CANCELLED_OBSOLETE, "SMSG_PUREMOUNT_CANCELLED_OBSOLETE" },
    { CMSG_MOUNTSPECIAL_ANIM, "CMSG_MOUNTSPECIAL_ANIM" },
    { SMSG_MOUNTSPECIAL_ANIM, "SMSG_MOUNTSPECIAL_ANIM" },
    { SMSG_PET_TAME_FAILURE, "SMSG_PET_TAME_FAILURE" },
    { CMSG_PET_SET_ACTION, "CMSG_PET_SET_ACTION" },
    { CMSG_PET_ACTION, "CMSG_PET_ACTION" },
    { CMSG_PET_ABANDON, "CMSG_PET_ABANDON" },
    { CMSG_PET_RENAME, "CMSG_PET_RENAME" },
    { SMSG_PET_NAME_INVALID, "SMSG_PET_NAME_INVALID" },
    { SMSG_PET_SPELLS, "SMSG_PET_SPELLS" },
    { SMSG_PET_MODE, "SMSG_PET_MODE" },
    { CMSG_GOSSIP_HELLO, "CMSG_GOSSIP_HELLO" },
    { CMSG_GOSSIP_SELECT_OPTION, "CMSG_GOSSIP_SELECT_OPTION" },
    { SMSG_GOSSIP_MESSAGE, "SMSG_GOSSIP_MESSAGE" },
    { SMSG_GOSSIP_COMPLETE, "SMSG_GOSSIP_COMPLETE" },
    { CMSG_NPC_TEXT_QUERY, "CMSG_NPC_TEXT_QUERY" },
    { SMSG_NPC_TEXT_UPDATE, "SMSG_NPC_TEXT_UPDATE" },
    { SMSG_NPC_WONT_TALK, "SMSG_NPC_WONT_TALK" },
    { CMSG_QUESTGIVER_STATUS_QUERY, "CMSG_QUESTGIVER_STATUS_QUERY" },
    { SMSG_QUESTGIVER_STATUS, "SMSG_QUESTGIVER_STATUS" },
    { CMSG_QUESTGIVER_HELLO, "CMSG_QUESTGIVER_HELLO" },
    { SMSG_QUESTGIVER_QUEST_LIST, "SMSG_QUESTGIVER_QUEST_LIST" },
    { CMSG_QUESTGIVER_QUERY_QUEST, "CMSG_QUESTGIVER_QUERY_QUEST" },
    { CMSG_QUESTGIVER_QUEST_AUTOLAUNCH, "CMSG_QUESTGIVER_QUEST_AUTOLAUNCH" },
    { SMSG_QUESTGIVER_QUEST_DETAILS, "SMSG_QUESTGIVER_QUEST_DETAILS" },
    { CMSG_QUESTGIVER_ACCEPT_QUEST, "CMSG_QUESTGIVER_ACCEPT_QUEST" },
    { CMSG_QUESTGIVER_COMPLETE_QUEST, "CMSG_QUESTGIVER_COMPLETE_QUEST" },
    { SMSG_QUESTGIVER_REQUEST_ITEMS, "SMSG_QUESTGIVER_REQUEST_ITEMS" },
    { CMSG_QUESTGIVER_REQUEST_REWARD, "CMSG_QUESTGIVER_REQUEST_REWARD" },
    { SMSG_QUESTGIVER_OFFER_REWARD, "SMSG_QUESTGIVER_OFFER_REWARD" },
    { CMSG_QUESTGIVER_CHOOSE_REWARD, "CMSG_QUESTGIVER_CHOOSE_REWARD" },
    { SMSG_QUESTGIVER_QUEST_INVALID, "SMSG_QUESTGIVER_QUEST_INVALID" },
    { CMSG_QUESTGIVER_CANCEL, "CMSG_QUESTGIVER_CANCEL" },
    { SMSG_QUESTGIVER_QUEST_COMPLETE, "SMSG_QUESTGIVER_QUEST_COMPLETE" },
    { SMSG_QUESTGIVER_QUEST_FAILED, "SMSG_QUESTGIVER_QUEST_FAILED" },
    { CMSG_QUESTLOG_SWAP_QUEST, "CMSG_QUESTLOG_SWAP_QUEST" },
    { CMSG_QUESTLOG_REMOVE_QUEST, "CMSG_QUESTLOG_REMOVE_QUEST" },
    { SMSG_QUESTLOG_FULL, "SMSG_QUESTLOG_FULL" },
    { SMSG_QUESTUPDATE_FAILED, "SMSG_QUESTUPDATE_FAILED" },
    { SMSG_QUESTUPDATE_FAILEDTIMER, "SMSG_QUESTUPDATE_FAILEDTIMER" },
    { SMSG_QUESTUPDATE_COMPLETE, "SMSG_QUESTUPDATE_COMPLETE" },
    { SMSG_QUESTUPDATE_ADD_KILL, "SMSG_QUESTUPDATE_ADD_KILL" },
    { SMSG_QUESTUPDATE_ADD_ITEM, "SMSG_QUESTUPDATE_ADD_ITEM" },
    { CMSG_QUEST_CONFIRM_ACCEPT, "CMSG_QUEST_CONFIRM_ACCEPT" },
    { SMSG_QUEST_CONFIRM_ACCEPT, "SMSG_QUEST_CONFIRM_ACCEPT" },
    { CMSG_PUSHQUESTTOPARTY, "CMSG_PUSHQUESTTOPARTY" },
    { CMSG_LIST_INVENTORY, "CMSG_LIST_INVENTORY" },
    { SMSG_LIST_INVENTORY, "SMSG_LIST_INVENTORY" },
    { CMSG_SELL_ITEM, "CMSG_SELL_ITEM" },
    { SMSG_SELL_ITEM, "SMSG_SELL_ITEM" },
    { CMSG_BUY_ITEM, "CMSG_BUY_ITEM" },
    { CMSG_BUY_ITEM_IN_SLOT, "CMSG_BUY_ITEM_IN_SLOT" },
    { SMSG_BUY_ITEM, "SMSG_BUY_ITEM" },
    { SMSG_BUY_FAILED, "SMSG_BUY_FAILED" },
    { CMSG_TAXICLEARALLNODES, "CMSG_TAXICLEARALLNODES" },
    { CMSG_TAXIENABLEALLNODES, "CMSG_TAXIENABLEALLNODES" },
    { CMSG_TAXISHOWNODES, "CMSG_TAXISHOWNODES" },
    { SMSG_SHOWTAXINODES, "SMSG_SHOWTAXINODES" },
    { CMSG_TAXINODE_STATUS_QUERY, "CMSG_TAXINODE_STATUS_QUERY" },
    { SMSG_TAXINODE_STATUS, "SMSG_TAXINODE_STATUS" },
    { CMSG_TAXIQUERYAVAILABLENODES, "CMSG_TAXIQUERYAVAILABLENODES" },
    { CMSG_ACTIVATETAXI, "CMSG_ACTIVATETAXI" },
    { SMSG_ACTIVATETAXIREPLY, "SMSG_ACTIVATETAXIREPLY" },
    { SMSG_NEW_TAXI_PATH, "SMSG_NEW_TAXI_PATH" },
    { CMSG_TRAINER_LIST, "CMSG_TRAINER_LIST" },
    { SMSG_TRAINER_LIST, "SMSG_TRAINER_LIST" },
    { CMSG_TRAINER_BUY_SPELL, "CMSG_TRAINER_BUY_SPELL" },
    { SMSG_TRAINER_BUY_SUCCEEDED, "SMSG_TRAINER_BUY_SUCCEEDED" },
    { SMSG_TRAINER_BUY_FAILED, "SMSG_TRAINER_BUY_FAILED" },
    { CMSG_BINDER_ACTIVATE, "CMSG_BINDER_ACTIVATE" },
    { SMSG_PLAYERBINDERROR, "SMSG_PLAYERBINDERROR" },
    { CMSG_BANKER_ACTIVATE, "CMSG_BANKER_ACTIVATE" },
    { SMSG_SHOW_BANK, "SMSG_SHOW_BANK" },
    { CMSG_BUY_BANK_SLOT, "CMSG_BUY_BANK_SLOT" },
    { SMSG_BUY_BANK_SLOT_RESULT, "SMSG_BUY_BANK_SLOT_RESULT" },
    { CMSG_PETITION_SHOWLIST, "CMSG_PETITION_SHOWLIST" },
    { SMSG_PETITION_SHOWLIST, "SMSG_PETITION_SHOWLIST" },
    { CMSG_PETITION_BUY, "CMSG_PETITION_BUY" },
    { CMSG_PETITION_SHOW_SIGNATURES, "CMSG_PETITION_SHOW_SIGNATURES" },
    { SMSG_PETITION_SHOW_SIGNATURES, "SMSG_PETITION_SHOW_SIGNATURES" },
    { CMSG_PETITION_SIGN, "CMSG_PETITION_SIGN" },
    { SMSG_PETITION_SIGN_RESULTS, "SMSG_PETITION_SIGN_RESULTS" },
    { MSG_PETITION_DECLINE, "MSG_PETITION_DECLINE" },
    { CMSG_OFFER_PETITION, "CMSG_OFFER_PETITION" },
    { CMSG_TURN_IN_PETITION, "CMSG_TURN_IN_PETITION" },
    { SMSG_TURN_IN_PETITION_RESULTS, "SMSG_TURN_IN_PETITION_RESULTS" },
    { CMSG_PETITION_QUERY, "CMSG_PETITION_QUERY" },
    { SMSG_PETITION_QUERY_RESPONSE, "SMSG_PETITION_QUERY_RESPONSE" },
    { SMSG_FISH_NOT_HOOKED, "SMSG_FISH_NOT_HOOKED" },
    { SMSG_FISH_ESCAPED, "SMSG_FISH_ESCAPED" },
    { CMSG_BUG, "CMSG_BUG" },
    { SMSG_NOTIFICATION, "SMSG_NOTIFICATION" },
    { CMSG_PLAYED_TIME, "CMSG_PLAYED_TIME" },
    { SMSG_PLAYED_TIME, "SMSG_PLAYED_TIME" },
    { CMSG_QUERY_TIME, "CMSG_QUERY_TIME" },
    { SMSG_QUERY_TIME_RESPONSE, "SMSG_QUERY_TIME_RESPONSE" },
    { SMSG_LOG_XPGAIN, "SMSG_LOG_XPGAIN" },
    { SMSG_AURACASTLOG, "SMSG_AURACASTLOG" },
    { CMSG_RECLAIM_CORPSE, "CMSG_RECLAIM_CORPSE" },
    { CMSG_WRAP_ITEM, "CMSG_WRAP_ITEM" },
    { SMSG_LEVELUP_INFO, "SMSG_LEVELUP_INFO" },
    { MSG_MINIMAP_PING, "MSG_MINIMAP_PING" },
    { SMSG_RESISTLOG, "SMSG_RESISTLOG" },
    { SMSG_ENCHANTMENTLOG, "SMSG_ENCHANTMENTLOG" },
    { CMSG_SET_SKILL_CHEAT, "CMSG_SET_SKILL_CHEAT" },
    { SMSG_START_MIRROR_TIMER, "SMSG_START_MIRROR_TIMER" },
    { SMSG_PAUSE_MIRROR_TIMER, "SMSG_PAUSE_MIRROR_TIMER" },
    { SMSG_STOP_MIRROR_TIMER, "SMSG_STOP_MIRROR_TIMER" },
    { CMSG_PING, "CMSG_PING" },
    { SMSG_PONG, "SMSG_PONG" },
    { SMSG_CLEAR_COOLDOWN, "SMSG_CLEAR_COOLDOWN" },
    { SMSG_GAMEOBJECT_PAGETEXT, "SMSG_GAMEOBJECT_PAGETEXT" },
    { CMSG_SETSHEATHED, "CMSG_SETSHEATHED" },
    { SMSG_COOLDOWN_CHEAT, "SMSG_COOLDOWN_CHEAT" },
    { SMSG_SPELL_DELAYED, "SMSG_SPELL_DELAYED" },
    { CMSG_PLAYER_MACRO_OBSOLETE, "CMSG_PLAYER_MACRO_OBSOLETE" },
    { SMSG_PLAYER_MACRO_OBSOLETE, "SMSG_PLAYER_MACRO_OBSOLETE" },
    { CMSG_GHOST, "CMSG_GHOST" },
    { CMSG_GM_INVIS, "CMSG_GM_INVIS" },
    { SMSG_INVALID_PROMOTION_CODE, "SMSG_INVALID_PROMOTION_CODE" },
    { MSG_GM_BIND_OTHER, "MSG_GM_BIND_OTHER" },
    { MSG_GM_SUMMON, "MSG_GM_SUMMON" },
    { SMSG_ITEM_TIME_UPDATE, "SMSG_ITEM_TIME_UPDATE" },
    { SMSG_ITEM_ENCHANT_TIME_UPDATE, "SMSG_ITEM_ENCHANT_TIME_UPDATE" },
    { SMSG_AUTH_CHALLENGE, "SMSG_AUTH_CHALLENGE" },
    { CMSG_AUTH_SESSION, "CMSG_AUTH_SESSION" },
    { SMSG_AUTH_RESPONSE, "SMSG_AUTH_RESPONSE" },
    { MSG_GM_SHOWLABEL, "MSG_GM_SHOWLABEL" },
    { CMSG_PET_CAST_SPELL, "CMSG_PET_CAST_SPELL" },
    { MSG_SAVE_GUILD_EMBLEM, "MSG_SAVE_GUILD_EMBLEM" },
    { MSG_TABARDVENDOR_ACTIVATE, "MSG_TABARDVENDOR_ACTIVATE" },
    { SMSG_PLAY_SPELL_VISUAL, "SMSG_PLAY_SPELL_VISUAL" },
    { CMSG_ZONEUPDATE, "CMSG_ZONEUPDATE" },
    { SMSG_PARTYKILLLOG, "SMSG_PARTYKILLLOG" },
    { SMSG_COMPRESSED_UPDATE_OBJECT, "SMSG_COMPRESSED_UPDATE_OBJECT" },
    { SMSG_PLAY_SPELL_IMPACT, "SMSG_PLAY_SPELL_IMPACT" },
    { SMSG_EXPLORATION_EXPERIENCE, "SMSG_EXPLORATION_EXPERIENCE" },
    { CMSG_GM_SET_SECURITY_GROUP, "CMSG_GM_SET_SECURITY_GROUP" },
    { CMSG_GM_NUKE, "CMSG_GM_NUKE" },
    { MSG_RANDOM_ROLL, "MSG_RANDOM_ROLL" },
    { SMSG_ENVIRONMENTAL_DAMAGE_LOG, "SMSG_ENVIRONMENTAL_DAMAGE_LOG" },
    { CMSG_RWHOIS_OBSOLETE, "CMSG_RWHOIS_OBSOLETE" },
    { SMSG_RWHOIS, "SMSG_RWHOIS" },
    { MSG_LOOKING_FOR_GROUP, "MSG_LOOKING_FOR_GROUP" },
    { CMSG_SET_LOOKING_FOR_GROUP, "CMSG_SET_LOOKING_FOR_GROUP" },
    { CMSG_UNLEARN_SPELL, "CMSG_UNLEARN_SPELL" },
    { CMSG_UNLEARN_SKILL, "CMSG_UNLEARN_SKILL" },
    { SMSG_REMOVED_SPELL, "SMSG_REMOVED_SPELL" },
    { CMSG_DECHARGE, "CMSG_DECHARGE" },
    { CMSG_GMTICKET_CREATE, "CMSG_GMTICKET_CREATE" },
    { SMSG_GMTICKET_CREATE, "SMSG_GMTICKET_CREATE" },
    { CMSG_GMTICKET_UPDATETEXT, "CMSG_GMTICKET_UPDATETEXT" },
    { SMSG_GMTICKET_UPDATETEXT, "SMSG_GMTICKET_UPDATETEXT" },
    { SMSG_ACCOUNT_DATA_TIMES, "SMSG_ACCOUNT_DATA_TIMES" },
    { CMSG_REQUEST_ACCOUNT_DATA, "CMSG_REQUEST_ACCOUNT_DATA" },
    { CMSG_UPDATE_ACCOUNT_DATA, "CMSG_UPDATE_ACCOUNT_DATA" },
    { SMSG_UPDATE_ACCOUNT_DATA, "SMSG_UPDATE_ACCOUNT_DATA" },
    { SMSG_CLEAR_FAR_SIGHT_IMMEDIATE, "SMSG_CLEAR_FAR_SIGHT_IMMEDIATE" },
    { SMSG_POWERGAINLOG_OBSOLETE, "SMSG_POWERGAINLOG_OBSOLETE" },
    { CMSG_GM_TEACH, "CMSG_GM_TEACH" },
    { CMSG_GM_CREATE_ITEM_TARGET, "CMSG_GM_CREATE_ITEM_TARGET" },
    { CMSG_GMTICKET_GETTICKET, "CMSG_GMTICKET_GETTICKET" },
    { SMSG_GMTICKET_GETTICKET, "SMSG_GMTICKET_GETTICKET" },
    { CMSG_UNLEARN_TALENTS, "CMSG_UNLEARN_TALENTS" },
    { SMSG_GAMEOBJECT_SPAWN_ANIM, "SMSG_GAMEOBJECT_SPAWN_ANIM" },
    { SMSG_GAMEOBJECT_DESPAWN_ANIM, "SMSG_GAMEOBJECT_DESPAWN_ANIM" },
    { MSG_CORPSE_QUERY, "MSG_CORPSE_QUERY" },
    { CMSG_GMTICKET_DELETETICKET, "CMSG_GMTICKET_DELETETICKET" },
    { SMSG_GMTICKET_DELETETICKET, "SMSG_GMTICKET_DELETETICKET" },
    { SMSG_CHAT_WRONG_FACTION, "SMSG_CHAT_WRONG_FACTION" },
    { CMSG_GMTICKET_SYSTEMSTATUS, "CMSG_GMTICKET_SYSTEMSTATUS" },
    { SMSG_GMTICKET_SYSTEMSTATUS, "SMSG_GMTICKET_SYSTEMSTATUS" },
    { CMSG_SPIRIT_HEALER_ACTIVATE, "CMSG_SPIRIT_HEALER_ACTIVATE" },
    { CMSG_SET_STAT_CHEAT, "CMSG_SET_STAT_CHEAT" },
    { SMSG_SET_REST_START, "SMSG_SET_REST_START" },
    { CMSG_SKILL_BUY_STEP, "CMSG_SKILL_BUY_STEP" },
    { CMSG_SKILL_BUY_RANK, "CMSG_SKILL_BUY_RANK" },
    { CMSG_XP_CHEAT, "CMSG_XP_CHEAT" },
    { SMSG_SPIRIT_HEALER_CONFIRM, "SMSG_SPIRIT_HEALER_CONFIRM" },
    { CMSG_CHARACTER_POINT_CHEAT, "CMSG_CHARACTER_POINT_CHEAT" },
    { SMSG_GOSSIP_POI, "SMSG_GOSSIP_POI" },
    { CMSG_CHAT_IGNORED, "CMSG_CHAT_IGNORED" },
    { CMSG_GM_VISION, "CMSG_GM_VISION" },
    { CMSG_SERVER_COMMAND, "CMSG_SERVER_COMMAND" },
    { CMSG_GM_SILENCE, "CMSG_GM_SILENCE" },
    { CMSG_GM_REVEALTO, "CMSG_GM_REVEALTO" },
    { CMSG_GM_RESURRECT, "CMSG_GM_RESURRECT" },
    { CMSG_GM_SUMMONMOB, "CMSG_GM_SUMMONMOB" },
    { CMSG_GM_MOVECORPSE, "CMSG_GM_MOVECORPSE" },
    { CMSG_GM_FREEZE, "CMSG_GM_FREEZE" },
    { CMSG_GM_UBERINVIS, "CMSG_GM_UBERINVIS" },
    { CMSG_GM_REQUEST_PLAYER_INFO, "CMSG_GM_REQUEST_PLAYER_INFO" },
    { SMSG_GM_PLAYER_INFO, "SMSG_GM_PLAYER_INFO" },
    { CMSG_GUILD_RANK, "CMSG_GUILD_RANK" },
    { CMSG_GUILD_ADD_RANK, "CMSG_GUILD_ADD_RANK" },
    { CMSG_GUILD_DEL_RANK, "CMSG_GUILD_DEL_RANK" },
    { CMSG_GUILD_SET_PUBLIC_NOTE, "CMSG_GUILD_SET_PUBLIC_NOTE" },
    { CMSG_GUILD_SET_OFFICER_NOTE, "CMSG_GUILD_SET_OFFICER_NOTE" },
    { SMSG_LOGIN_VERIFY_WORLD, "SMSG_LOGIN_VERIFY_WORLD" },
    { CMSG_CLEAR_EXPLORATION, "CMSG_CLEAR_EXPLORATION" },
    { CMSG_SEND_MAIL, "CMSG_SEND_MAIL" },
    { SMSG_SEND_MAIL_RESULT, "SMSG_SEND_MAIL_RESULT" },
    { CMSG_GET_MAIL_LIST, "CMSG_GET_MAIL_LIST" },
    { SMSG_MAIL_LIST_RESULT, "SMSG_MAIL_LIST_RESULT" },
    { CMSG_BATTLEFIELD_LIST, "CMSG_BATTLEFIELD_LIST" },
    { SMSG_BATTLEFIELD_LIST, "SMSG_BATTLEFIELD_LIST" },
    { CMSG_BATTLEFIELD_JOIN, "CMSG_BATTLEFIELD_JOIN" },
    { SMSG_BATTLEFIELD_WIN_OBSOLETE, "SMSG_BATTLEFIELD_WIN_OBSOLETE" },
    { SMSG_BATTLEFIELD_LOSE_OBSOLETE, "SMSG_BATTLEFIELD_LOSE_OBSOLETE" },
    { CMSG_TAXICLEARNODE, "CMSG_TAXICLEARNODE" },
    { CMSG_TAXIENABLENODE, "CMSG_TAXIENABLENODE" },
    { CMSG_ITEM_TEXT_QUERY, "CMSG_ITEM_TEXT_QUERY" },
    { SMSG_ITEM_TEXT_QUERY_RESPONSE, "SMSG_ITEM_TEXT_QUERY_RESPONSE" },
    { CMSG_MAIL_TAKE_MONEY, "CMSG_MAIL_TAKE_MONEY" },
    { CMSG_MAIL_TAKE_ITEM, "CMSG_MAIL_TAKE_ITEM" },
    { CMSG_MAIL_MARK_AS_READ, "CMSG_MAIL_MARK_AS_READ" },
    { CMSG_MAIL_RETURN_TO_SENDER, "CMSG_MAIL_RETURN_TO_SENDER" },
    { CMSG_MAIL_DELETE, "CMSG_MAIL_DELETE" },
    { CMSG_MAIL_CREATE_TEXT_ITEM, "CMSG_MAIL_CREATE_TEXT_ITEM" },
    { SMSG_SPELLLOGMISS, "SMSG_SPELLLOGMISS" },
    { SMSG_SPELLLOGEXECUTE, "SMSG_SPELLLOGEXECUTE" },
    { SMSG_DEBUGAURAPROC, "SMSG_DEBUGAURAPROC" },
    { SMSG_PERIODICAURALOG, "SMSG_PERIODICAURALOG" },
    { SMSG_SPELLDAMAGESHIELD, "SMSG_SPELLDAMAGESHIELD" },
    { SMSG_SPELLNONMELEEDAMAGELOG, "SMSG_SPELLNONMELEEDAMAGELOG" },
    { CMSG_LEARN_TALENT, "CMSG_LEARN_TALENT" },
    { SMSG_RESURRECT_FAILED, "SMSG_RESURRECT_FAILED" },
    { CMSG_TOGGLE_PVP, "CMSG_TOGGLE_PVP" },
    { SMSG_ZONE_UNDER_ATTACK, "SMSG_ZONE_UNDER_ATTACK" },
    { MSG_AUCTION_HELLO, "MSG_AUCTION_HELLO" },
    { CMSG_AUCTION_SELL_ITEM, "CMSG_AUCTION_SELL_ITEM" },
    { CMSG_AUCTION_REMOVE_ITEM, "CMSG_AUCTION_REMOVE_ITEM" },
    { CMSG_AUCTION_LIST_ITEMS, "CMSG_AUCTION_LIST_ITEMS" },
    { CMSG_AUCTION_LIST_OWNER_ITEMS, "CMSG_AUCTION_LIST_OWNER_ITEMS" },
    { CMSG_AUCTION_PLACE_BID, "CMSG_AUCTION_PLACE_BID" },
    { SMSG_AUCTION_COMMAND_RESULT, "SMSG_AUCTION_COMMAND_RESULT" },
    { SMSG_AUCTION_LIST_RESULT, "SMSG_AUCTION_LIST_RESULT" },
    { SMSG_AUCTION_OWNER_LIST_RESULT, "SMSG_AUCTION_OWNER_LIST_RESULT" },
    { SMSG_AUCTION_BIDDER_NOTIFICATION, "SMSG_AUCTION_BIDDER_NOTIFICATION" },
    { SMSG_AUCTION_OWNER_NOTIFICATION, "SMSG_AUCTION_OWNER_NOTIFICATION" },
    { SMSG_PROCRESIST, "SMSG_PROCRESIST" },
    { SMSG_STANDSTATE_CHANGE_FAILURE_OBSOLETE, "SMSG_STANDSTATE_CHANGE_FAILURE_OBSOLETE" },
    { SMSG_DISPEL_FAILED, "SMSG_DISPEL_FAILED" },
    { SMSG_SPELLORDAMAGE_IMMUNE, "SMSG_SPELLORDAMAGE_IMMUNE" },
    { CMSG_AUCTION_LIST_BIDDER_ITEMS, "CMSG_AUCTION_LIST_BIDDER_ITEMS" },
    { SMSG_AUCTION_BIDDER_LIST_RESULT, "SMSG_AUCTION_BIDDER_LIST_RESULT" },
    { SMSG_SET_FLAT_SPELL_MODIFIER, "SMSG_SET_FLAT_SPELL_MODIFIER" },
    { SMSG_SET_PCT_SPELL_MODIFIER, "SMSG_SET_PCT_SPELL_MODIFIER" },
    { CMSG_SET_AMMO, "CMSG_SET_AMMO" },
    { SMSG_CORPSE_RECLAIM_DELAY, "SMSG_CORPSE_RECLAIM_DELAY" },
    { CMSG_SET_ACTIVE_MOVER, "CMSG_SET_ACTIVE_MOVER" },
    { CMSG_PET_CANCEL_AURA, "CMSG_PET_CANCEL_AURA" },
    { CMSG_PLAYER_AI_CHEAT, "CMSG_PLAYER_AI_CHEAT" },
    { CMSG_CANCEL_AUTO_REPEAT_SPELL, "CMSG_CANCEL_AUTO_REPEAT_SPELL" },
    { MSG_GM_ACCOUNT_ONLINE, "MSG_GM_ACCOUNT_ONLINE" },
    { MSG_LIST_STABLED_PETS, "MSG_LIST_STABLED_PETS" },
    { CMSG_STABLE_PET, "CMSG_STABLE_PET" },
    { CMSG_UNSTABLE_PET, "CMSG_UNSTABLE_PET" },
    { CMSG_BUY_STABLE_SLOT, "CMSG_BUY_STABLE_SLOT" },
    { SMSG_STABLE_RESULT, "SMSG_STABLE_RESULT" },
    { CMSG_STABLE_REVIVE_PET, "CMSG_STABLE_REVIVE_PET" },
    { CMSG_STABLE_SWAP_PET, "CMSG_STABLE_SWAP_PET" },
    { MSG_QUEST_PUSH_RESULT, "MSG_QUEST_PUSH_RESULT" },
    { SMSG_PLAY_MUSIC, "SMSG_PLAY_MUSIC" },
    { SMSG_PLAY_OBJECT_SOUND, "SMSG_PLAY_OBJECT_SOUND" },
    { CMSG_REQUEST_PET_INFO, "CMSG_REQUEST_PET_INFO" },
    { CMSG_FAR_SIGHT, "CMSG_FAR_SIGHT" },
    { SMSG_SPELLDISPELLOG, "SMSG_SPELLDISPELLOG" },
    { SMSG_DAMAGE_CALC_LOG, "SMSG_DAMAGE_CALC_LOG" },
    { CMSG_ENABLE_DAMAGE_LOG, "CMSG_ENABLE_DAMAGE_LOG" },
    { CMSG_GROUP_CHANGE_SUB_GROUP, "CMSG_GROUP_CHANGE_SUB_GROUP" },
    { CMSG_REQUEST_PARTY_MEMBER_STATS, "CMSG_REQUEST_PARTY_MEMBER_STATS" },
    { CMSG_GROUP_SWAP_SUB_GROUP, "CMSG_GROUP_SWAP_SUB_GROUP" },
    { CMSG_RESET_FACTION_CHEAT, "CMSG_RESET_FACTION_CHEAT" },
    { CMSG_AUTOSTORE_BANK_ITEM, "CMSG_AUTOSTORE_BANK_ITEM" },
    { CMSG_AUTOBANK_ITEM, "CMSG_AUTOBANK_ITEM" },
    { MSG_QUERY_NEXT_MAIL_TIME, "MSG_QUERY_NEXT_MAIL_TIME" },
    { SMSG_RECEIVED_MAIL, "SMSG_RECEIVED_MAIL" },
    { SMSG_RAID_GROUP_ONLY, "SMSG_RAID_GROUP_ONLY" },
    { CMSG_SET_DURABILITY_CHEAT, "CMSG_SET_DURABILITY_CHEAT" },
    { CMSG_SET_PVP_RANK_CHEAT, "CMSG_SET_PVP_RANK_CHEAT" },
    { CMSG_ADD_PVP_MEDAL_CHEAT, "CMSG_ADD_PVP_MEDAL_CHEAT" },
    { CMSG_DEL_PVP_MEDAL_CHEAT, "CMSG_DEL_PVP_MEDAL_CHEAT" },
    { CMSG_SET_PVP_TITLE, "CMSG_SET_PVP_TITLE" },
    { SMSG_PVP_CREDIT, "SMSG_PVP_CREDIT" },
    { SMSG_AUCTION_REMOVED_NOTIFICATION, "SMSG_AUCTION_REMOVED_NOTIFICATION" },
    { CMSG_GROUP_RAID_CONVERT, "CMSG_GROUP_RAID_CONVERT" },
    { CMSG_GROUP_ASSISTANT_LEADER, "CMSG_GROUP_ASSISTANT_LEADER" },
    { CMSG_BUYBACK_ITEM, "CMSG_BUYBACK_ITEM" },
    { SMSG_SERVER_MESSAGE, "SMSG_SERVER_MESSAGE" },
    { CMSG_MEETINGSTONE_JOIN, "CMSG_MEETINGSTONE_JOIN" },
    { CMSG_MEETINGSTONE_LEAVE, "CMSG_MEETINGSTONE_LEAVE" },
    { CMSG_MEETINGSTONE_CHEAT, "CMSG_MEETINGSTONE_CHEAT" },
    { SMSG_MEETINGSTONE_SETQUEUE, "SMSG_MEETINGSTONE_SETQUEUE" },
    { CMSG_MEETINGSTONE_INFO, "CMSG_MEETINGSTONE_INFO" },
    { SMSG_MEETINGSTONE_COMPLETE, "SMSG_MEETINGSTONE_COMPLETE" },
    { SMSG_MEETINGSTONE_IN_PROGRESS, "SMSG_MEETINGSTONE_IN_PROGRESS" },
    { SMSG_MEETINGSTONE_MEMBER_ADDED, "SMSG_MEETINGSTONE_MEMBER_ADDED" },
    { CMSG_GMTICKETSYSTEM_TOGGLE, "CMSG_GMTICKETSYSTEM_TOGGLE" },
    { CMSG_CANCEL_GROWTH_AURA, "CMSG_CANCEL_GROWTH_AURA" },
    { SMSG_CANCEL_AUTO_REPEAT, "SMSG_CANCEL_AUTO_REPEAT" },
    { SMSG_STANDSTATE_UPDATE, "SMSG_STANDSTATE_UPDATE" },
    { SMSG_LOOT_ALL_PASSED, "SMSG_LOOT_ALL_PASSED" },
    { SMSG_LOOT_ROLL_WON, "SMSG_LOOT_ROLL_WON" },
    { CMSG_LOOT_ROLL, "CMSG_LOOT_ROLL" },
    { SMSG_LOOT_START_ROLL, "SMSG_LOOT_START_ROLL" },
    { SMSG_LOOT_ROLL, "SMSG_LOOT_ROLL" },
    { CMSG_LOOT_MASTER_GIVE, "CMSG_LOOT_MASTER_GIVE" },
    { SMSG_LOOT_MASTER_LIST, "SMSG_LOOT_MASTER_LIST" },
    { SMSG_SET_FORCED_REACTIONS, "SMSG_SET_FORCED_REACTIONS" },
    { SMSG_SPELL_FAILED_OTHER, "SMSG_SPELL_FAILED_OTHER" },
    { SMSG_GAMEOBJECT_RESET_STATE, "SMSG_GAMEOBJECT_RESET_STATE" },
    { CMSG_REPAIR_ITEM, "CMSG_REPAIR_ITEM" },
    { SMSG_CHAT_PLAYER_NOT_FOUND, "SMSG_CHAT_PLAYER_NOT_FOUND" },
    { MSG_TALENT_WIPE_CONFIRM, "MSG_TALENT_WIPE_CONFIRM" },
    { SMSG_SUMMON_REQUEST, "SMSG_SUMMON_REQUEST" },
    { CMSG_SUMMON_RESPONSE, "CMSG_SUMMON_RESPONSE" },
    { MSG_MOVE_TOGGLE_GRAVITY_CHEAT, "MSG_MOVE_TOGGLE_GRAVITY_CHEAT" },
    { SMSG_MONSTER_MOVE_TRANSPORT, "SMSG_MONSTER_MOVE_TRANSPORT" },
    { SMSG_PET_BROKEN, "SMSG_PET_BROKEN" },
    { MSG_MOVE_FEATHER_FALL, "MSG_MOVE_FEATHER_FALL" },
    { MSG_MOVE_WATER_WALK, "MSG_MOVE_WATER_WALK" },
    { CMSG_SERVER_BROADCAST, "CMSG_SERVER_BROADCAST" },
    { CMSG_SELF_RES, "CMSG_SELF_RES" },
    { SMSG_FEIGN_DEATH_RESISTED, "SMSG_FEIGN_DEATH_RESISTED" },
    { CMSG_RUN_SCRIPT, "CMSG_RUN_SCRIPT" },
    { SMSG_SCRIPT_MESSAGE, "SMSG_SCRIPT_MESSAGE" },
    { SMSG_DUEL_COUNTDOWN, "SMSG_DUEL_COUNTDOWN" },
    { SMSG_AREA_TRIGGER_MESSAGE, "SMSG_AREA_TRIGGER_MESSAGE" },
    { CMSG_TOGGLE_HELM, "CMSG_TOGGLE_HELM" },
    { CMSG_TOGGLE_CLOAK, "CMSG_TOGGLE_CLOAK" },
    { SMSG_MEETINGSTONE_JOINFAILED, "SMSG_MEETINGSTONE_JOINFAILED" },
    { SMSG_PLAYER_SKINNED, "SMSG_PLAYER_SKINNED" },
    { SMSG_DURABILITY_DAMAGE_DEATH, "SMSG_DURABILITY_DAMAGE_DEATH" },
    { CMSG_SET_EXPLORATION, "CMSG_SET_EXPLORATION" },
    { CMSG_SET_ACTIONBAR_TOGGLES, "CMSG_SET_ACTIONBAR_TOGGLES" },
    { UMSG_DELETE_GUILD_CHARTER, "UMSG_DELETE_GUILD_CHARTER" },
    { MSG_PETITION_RENAME, "MSG_PETITION_RENAME" },
    { SMSG_INIT_WORLD_STATES, "SMSG_INIT_WORLD_STATES" },
    { SMSG_UPDATE_WORLD_STATE, "SMSG_UPDATE_WORLD_STATE" },
    { CMSG_ITEM_NAME_QUERY, "CMSG_ITEM_NAME_QUERY" },
    { SMSG_ITEM_NAME_QUERY_RESPONSE, "SMSG_ITEM_NAME_QUERY_RESPONSE" },
    { SMSG_PET_ACTION_FEEDBACK, "SMSG_PET_ACTION_FEEDBACK" },
    { CMSG_CHAR_RENAME, "CMSG_CHAR_RENAME" },
    { SMSG_CHAR_RENAME, "SMSG_CHAR_RENAME" },
    { CMSG_MOVE_SPLINE_DONE, "CMSG_MOVE_SPLINE_DONE" },
    { CMSG_MOVE_FALL_RESET, "CMSG_MOVE_FALL_RESET" },
    { SMSG_INSTANCE_SAVE_CREATED, "SMSG_INSTANCE_SAVE_CREATED" },
    { SMSG_RAID_INSTANCE_INFO, "SMSG_RAID_INSTANCE_INFO" },
    { CMSG_REQUEST_RAID_INFO, "CMSG_REQUEST_RAID_INFO" },
    { CMSG_MOVE_TIME_SKIPPED, "CMSG_MOVE_TIME_SKIPPED" },
    { CMSG_MOVE_FEATHER_FALL_ACK, "CMSG_MOVE_FEATHER_FALL_ACK" },
    { CMSG_MOVE_WATER_WALK_ACK, "CMSG_MOVE_WATER_WALK_ACK" },
    { CMSG_MOVE_NOT_ACTIVE_MOVER, "CMSG_MOVE_NOT_ACTIVE_MOVER" },
    { SMSG_PLAY_SOUND, "SMSG_PLAY_SOUND" },
    { CMSG_BATTLEFIELD_STATUS, "CMSG_BATTLEFIELD_STATUS" },
    { SMSG_BATTLEFIELD_STATUS, "SMSG_BATTLEFIELD_STATUS" },
    { CMSG_BATTLEFIELD_PORT, "CMSG_BATTLEFIELD_PORT" },
    { MSG_INSPECT_HONOR_STATS, "MSG_INSPECT_HONOR_STATS" },
    { CMSG_BATTLEMASTER_HELLO, "CMSG_BATTLEMASTER_HELLO" },
    { CMSG_MOVE_START_SWIM_CHEAT, "CMSG_MOVE_START_SWIM_CHEAT" },
    { CMSG_MOVE_STOP_SWIM_CHEAT, "CMSG_MOVE_STOP_SWIM_CHEAT" },
    { SMSG_FORCE_WALK_SPEED_CHANGE, "SMSG_FORCE_WALK_SPEED_CHANGE" },
    { CMSG_FORCE_WALK_SPEED_CHANGE_ACK, "CMSG_FORCE_WALK_SPEED_CHANGE_ACK" },
    { SMSG_FORCE_SWIM_BACK_SPEED_CHANGE, "SMSG_FORCE_SWIM_BACK_SPEED_CHANGE" },
    { CMSG_FORCE_SWIM_BACK_SPEED_CHANGE_ACK, "CMSG_FORCE_SWIM_BACK_SPEED_CHANGE_ACK" },
    { SMSG_FORCE_TURN_RATE_CHANGE, "SMSG_FORCE_TURN_RATE_CHANGE" },
    { CMSG_FORCE_TURN_RATE_CHANGE_ACK, "CMSG_FORCE_TURN_RATE_CHANGE_ACK" },
    { MSG_PVP_LOG_DATA, "MSG_PVP_LOG_DATA" },
    { CMSG_LEAVE_BATTLEFIELD, "CMSG_LEAVE_BATTLEFIELD" },
    { CMSG_AREA_SPIRIT_HEALER_QUERY, "CMSG_AREA_SPIRIT_HEALER_QUERY" },
    { CMSG_AREA_SPIRIT_HEALER_QUEUE, "CMSG_AREA_SPIRIT_HEALER_QUEUE" },
    { SMSG_AREA_SPIRIT_HEALER_TIME, "SMSG_AREA_SPIRIT_HEALER_TIME" },
    { CMSG_GM_UNTEACH, "CMSG_GM_UNTEACH" },
    { SMSG_WARDEN_DATA, "SMSG_WARDEN_DATA" },
    { CMSG_WARDEN_DATA, "CMSG_WARDEN_DATA" },
    { SMSG_GROUP_JOINED_BATTLEGROUND, "SMSG_GROUP_JOINED_BATTLEGROUND" },
    { MSG_BATTLEGROUND_PLAYER_POSITIONS, "MSG_BATTLEGROUND_PLAYER_POSITIONS" },
    { CMSG_PET_STOP_ATTACK, "CMSG_PET_STOP_ATTACK" },
    { SMSG_BINDER_CONFIRM, "SMSG_BINDER_CONFIRM" },
    { SMSG_BATTLEGROUND_PLAYER_JOINED, "SMSG_BATTLEGROUND_PLAYER_JOINED" },
    { SMSG_BATTLEGROUND_PLAYER_LEFT, "SMSG_BATTLEGROUND_PLAYER_LEFT" },
    { CMSG_BATTLEMASTER_JOIN, "CMSG_BATTLEMASTER_JOIN" },
    { SMSG_ADDON_INFO, "SMSG_ADDON_INFO" },
    { CMSG_PET_UNLEARN, "CMSG_PET_UNLEARN" },
    { SMSG_PET_UNLEARN_CONFIRM, "SMSG_PET_UNLEARN_CONFIRM" },
    { SMSG_PARTY_MEMBER_STATS_FULL, "SMSG_PARTY_MEMBER_STATS_FULL" },
    { CMSG_PET_SPELL_AUTOCAST, "CMSG_PET_SPELL_AUTOCAST" },
    { SMSG_WEATHER, "SMSG_WEATHER" },
    { SMSG_PLAY_TIME_WARNING, "SMSG_PLAY_TIME_WARNING" },
    { SMSG_MINIGAME_SETUP, "SMSG_MINIGAME_SETUP" },
    { SMSG_MINIGAME_STATE, "SMSG_MINIGAME_STATE" },
    { CMSG_MINIGAME_MOVE, "CMSG_MINIGAME_MOVE" },
    { SMSG_MINIGAME_MOVE_FAILED, "SMSG_MINIGAME_MOVE_FAILED" },
    { SMSG_RAID_INSTANCE_MESSAGE, "SMSG_RAID_INSTANCE_MESSAGE" },
    { SMSG_COMPRESSED_MOVES, "SMSG_COMPRESSED_MOVES" },
    { CMSG_GUILD_INFO_TEXT, "CMSG_GUILD_INFO_TEXT" },
    { SMSG_CHAT_RESTRICTED, "SMSG_CHAT_RESTRICTED" },
    { SMSG_SPLINE_SET_RUN_SPEED, "SMSG_SPLINE_SET_RUN_SPEED" },
    { SMSG_SPLINE_SET_RUN_BACK_SPEED, "SMSG_SPLINE_SET_RUN_BACK_SPEED" },
    { SMSG_SPLINE_SET_SWIM_SPEED, "SMSG_SPLINE_SET_SWIM_SPEED" },
    { SMSG_SPLINE_SET_WALK_SPEED, "SMSG_SPLINE_SET_WALK_SPEED" },
    { SMSG_SPLINE_SET_SWIM_BACK_SPEED, "SMSG_SPLINE_SET_SWIM_BACK_SPEED" },
    { SMSG_SPLINE_SET_TURN_RATE, "SMSG_SPLINE_SET_TURN_RATE" },
    { SMSG_SPLINE_MOVE_UNROOT, "SMSG_SPLINE_MOVE_UNROOT" },
    { SMSG_SPLINE_MOVE_FEATHER_FALL, "SMSG_SPLINE_MOVE_FEATHER_FALL" },
    { SMSG_SPLINE_MOVE_NORMAL_FALL, "SMSG_SPLINE_MOVE_NORMAL_FALL" },
    { SMSG_SPLINE_MOVE_SET_HOVER, "SMSG_SPLINE_MOVE_SET_HOVER" },
    { SMSG_SPLINE_MOVE_UNSET_HOVER, "SMSG_SPLINE_MOVE_UNSET_HOVER" },
    { SMSG_SPLINE_MOVE_WATER_WALK, "SMSG_SPLINE_MOVE_WATER_WALK" },
    { SMSG_SPLINE_MOVE_LAND_WALK, "SMSG_SPLINE_MOVE_LAND_WALK" },
    { SMSG_SPLINE_MOVE_START_SWIM, "SMSG_SPLINE_MOVE_START_SWIM" },
    { SMSG_SPLINE_MOVE_STOP_SWIM, "SMSG_SPLINE_MOVE_STOP_SWIM" },
    { SMSG_SPLINE_MOVE_SET_RUN_MODE, "SMSG_SPLINE_MOVE_SET_RUN_MODE" },
    { SMSG_SPLINE_MOVE_SET_WALK_MODE, "SMSG_SPLINE_MOVE_SET_WALK_MODE" },
    { CMSG_GM_NUKE_ACCOUNT, "CMSG_GM_NUKE_ACCOUNT" },
    { MSG_GM_DESTROY_CORPSE, "MSG_GM_DESTROY_CORPSE" },
    { CMSG_GM_DESTROY_ONLINE_CORPSE, "CMSG_GM_DESTROY_ONLINE_CORPSE" },
    { CMSG_ACTIVATETAXIEXPRESS, "CMSG_ACTIVATETAXIEXPRESS" },
    { SMSG_SET_FACTION_ATWAR, "SMSG_SET_FACTION_ATWAR" },
    { SMSG_GAMETIMEBIAS_SET, "SMSG_GAMETIMEBIAS_SET" },
    { CMSG_DEBUG_ACTIONS_START, "CMSG_DEBUG_ACTIONS_START" },
    { CMSG_DEBUG_ACTIONS_STOP, "CMSG_DEBUG_ACTIONS_STOP" },
    { CMSG_SET_FACTION_INACTIVE, "CMSG_SET_FACTION_INACTIVE" },
    { CMSG_SET_WATCHED_FACTION, "CMSG_SET_WATCHED_FACTION" },
    { MSG_MOVE_TIME_SKIPPED, "MSG_MOVE_TIME_SKIPPED" },
    { SMSG_SPLINE_MOVE_ROOT, "SMSG_SPLINE_MOVE_ROOT" },
    { CMSG_SET_EXPLORATION_ALL, "CMSG_SET_EXPLORATION_ALL" },
    { SMSG_INVALIDATE_PLAYER, "SMSG_INVALIDATE_PLAYER" },
    { CMSG_RESET_INSTANCES, "CMSG_RESET_INSTANCES" },
    { SMSG_INSTANCE_RESET, "SMSG_INSTANCE_RESET" },
    { SMSG_INSTANCE_RESET_FAILED, "SMSG_INSTANCE_RESET_FAILED" },
    { SMSG_UPDATE_LAST_INSTANCE, "SMSG_UPDATE_LAST_INSTANCE" },
    { MSG_RAID_TARGET_UPDATE, "MSG_RAID_TARGET_UPDATE" },
    { MSG_RAID_READY_CHECK, "MSG_RAID_READY_CHECK" },
    { CMSG_LUA_USAGE, "CMSG_LUA_USAGE" },
    { SMSG_PET_ACTION_SOUND, "SMSG_PET_ACTION_SOUND" },
    { SMSG_PET_DISMISS_SOUND, "SMSG_PET_DISMISS_SOUND" },
    { SMSG_GHOSTEE_GONE, "SMSG_GHOSTEE_GONE" },
    { CMSG_GM_UPDATE_TICKET_STATUS, "CMSG_GM_UPDATE_TICKET_STATUS" },
    { SMSG_GM_TICKET_STATUS_UPDATE, "SMSG_GM_TICKET_STATUS_UPDATE" },
    { CMSG_GMSURVEY_SUBMIT, "CMSG_GMSURVEY_SUBMIT" },
    { SMSG_UPDATE_INSTANCE_OWNERSHIP, "SMSG_UPDATE_INSTANCE_OWNERSHIP" },
    { CMSG_IGNORE_KNOCKBACK_CHEAT, "CMSG_IGNORE_KNOCKBACK_CHEAT" },
    { SMSG_CHAT_PLAYER_AMBIGUOUS, "SMSG_CHAT_PLAYER_AMBIGUOUS" },
    { MSG_DELAY_GHOST_TELEPORT, "MSG_DELAY_GHOST_TELEPORT" },
    { SMSG_SPELLINSTAKILLLOG, "SMSG_SPELLINSTAKILLLOG" },
    { SMSG_SPELL_UPDATE_CHAIN_TARGETS, "SMSG_SPELL_UPDATE_CHAIN_TARGETS" },
    { CMSG_CHAT_FILTERED, "CMSG_CHAT_FILTERED" },
    { SMSG_EXPECTED_SPAM_RECORDS, "SMSG_EXPECTED_SPAM_RECORDS" },
    { SMSG_SPELLSTEALLOG, "SMSG_SPELLSTEALLOG" },
    { CMSG_LOTTERY_QUERY_OBSOLETE, "CMSG_LOTTERY_QUERY_OBSOLETE" },
    { SMSG_LOTTERY_QUERY_RESULT_OBSOLETE, "SMSG_LOTTERY_QUERY_RESULT_OBSOLETE" },
    { CMSG_BUY_LOTTERY_TICKET_OBSOLETE, "CMSG_BUY_LOTTERY_TICKET_OBSOLETE" },
    { SMSG_LOTTERY_RESULT_OBSOLETE, "SMSG_LOTTERY_RESULT_OBSOLETE" },
    { SMSG_CHARACTER_PROFILE, "SMSG_CHARACTER_PROFILE" },
    { SMSG_CHARACTER_PROFILE_REALM_CONNECTED, "SMSG_CHARACTER_PROFILE_REALM_CONNECTED" },
    { SMSG_DEFENSE_MESSAGE, "SMSG_DEFENSE_MESSAGE" },
    { MSG_GM_RESETINSTANCELIMIT, "MSG_GM_RESETINSTANCELIMIT" },
    { SMSG_MOTD, "SMSG_MOTD" },
    { SMSG_MOVE_SET_FLIGHT, "SMSG_MOVE_SET_FLIGHT" },
    { SMSG_MOVE_UNSET_FLIGHT, "SMSG_MOVE_UNSET_FLIGHT" },
    { CMSG_MOVE_FLIGHT_ACK, "CMSG_MOVE_FLIGHT_ACK" },
    { MSG_MOVE_START_SWIM_CHEAT, "MSG_MOVE_START_SWIM_CHEAT" },
    { MSG_MOVE_STOP_SWIM_CHEAT, "MSG_MOVE_STOP_SWIM_CHEAT" },
    { CMSG_CANCEL_MOUNT_AURA, "CMSG_CANCEL_MOUNT_AURA" },
    { CMSG_CANCEL_TEMP_ENCHANTMENT, "CMSG_CANCEL_TEMP_ENCHANTMENT" },
    { CMSG_MAELSTROM_INVALIDATE_CACHE, "CMSG_MAELSTROM_INVALIDATE_CACHE" },
    { CMSG_SET_TAXI_BENCHMARK_MODE, "CMSG_SET_TAXI_BENCHMARK_MODE" },
    { CMSG_MOVE_CHNG_TRANSPORT, "CMSG_MOVE_CHNG_TRANSPORT" },
    { MSG_PARTY_ASSIGNMENT, "MSG_PARTY_ASSIGNMENT" },
    { SMSG_OFFER_PETITION_ERROR, "SMSG_OFFER_PETITION_ERROR" },
    { SMSG_RESET_FAILED_NOTIFY, "SMSG_RESET_FAILED_NOTIFY" },
    { SMSG_REAL_GROUP_UPDATE, "SMSG_REAL_GROUP_UPDATE" },
    { SMSG_INIT_EXTRA_AURA_INFO, "SMSG_INIT_EXTRA_AURA_INFO" },
    { SMSG_SET_EXTRA_AURA_INFO, "SMSG_SET_EXTRA_AURA_INFO" },
    { SMSG_SET_EXTRA_AURA_INFO_NEED_UPDATE, "SMSG_SET_EXTRA_AURA_INFO_NEED_UPDATE" },
    { SMSG_SPELL_CHANCE_PROC_LOG, "SMSG_SPELL_CHANCE_PROC_LOG" },
    { CMSG_MOVE_SET_RUN_SPEED, "CMSG_MOVE_SET_RUN_SPEED" },
    { SMSG_DISMOUNT, "SMSG_DISMOUNT" },
    { MSG_RAID_READY_CHECK_CONFIRM, "MSG_RAID_READY_CHECK_CONFIRM" },
    { SMSG_CLEAR_TARGET, "SMSG_CLEAR_TARGET" },
    { CMSG_BOT_DETECTED, "CMSG_BOT_DETECTED" },
    { SMSG_KICK_REASON, "SMSG_KICK_REASON" },
    { MSG_RAID_READY_CHECK_FINISHED, "MSG_RAID_READY_CHECK_FINISHED" },
    { CMSG_TARGET_CAST, "CMSG_TARGET_CAST" },
    { CMSG_TARGET_SCRIPT_CAST, "CMSG_TARGET_SCRIPT_CAST" },
    { CMSG_CHANNEL_DISPLAY_LIST, "CMSG_CHANNEL_DISPLAY_LIST" },
    { CMSG_GET_CHANNEL_MEMBER_COUNT, "CMSG_GET_CHANNEL_MEMBER_COUNT" },
    { SMSG_CHANNEL_MEMBER_COUNT, "SMSG_CHANNEL_MEMBER_COUNT" },
    { CMSG_DEBUG_LIST_TARGETS, "CMSG_DEBUG_LIST_TARGETS" },
    { SMSG_DEBUG_LIST_TARGETS, "SMSG_DEBUG_LIST_TARGETS" },
    { CMSG_PARTY_SILENCE, "CMSG_PARTY_SILENCE" },
    { CMSG_PARTY_UNSILENCE, "CMSG_PARTY_UNSILENCE" },
    { MSG_NOTIFY_PARTY_SQUELCH, "MSG_NOTIFY_PARTY_SQUELCH" },
    { SMSG_COMSAT_RECONNECT_TRY, "SMSG_COMSAT_RECONNECT_TRY" },
    { SMSG_COMSAT_DISCONNECT, "SMSG_COMSAT_DISCONNECT" },
    { SMSG_COMSAT_CONNECT_FAIL, "SMSG_COMSAT_CONNECT_FAIL" },
    { CMSG_SET_CHANNEL_WATCH, "CMSG_SET_CHANNEL_WATCH" },
    { SMSG_USERLIST_ADD, "SMSG_USERLIST_ADD" },
    { SMSG_USERLIST_REMOVE, "SMSG_USERLIST_REMOVE" },
    { SMSG_USERLIST_UPDATE, "SMSG_USERLIST_UPDATE" },
    { CMSG_CLEAR_CHANNEL_WATCH, "CMSG_CLEAR_CHANNEL_WATCH" },
    { SMSG_GOGOGO_OBSOLETE, "SMSG_GOGOGO_OBSOLETE" },
    { SMSG_ECHO_PARTY_SQUELCH, "SMSG_ECHO_PARTY_SQUELCH" },
    { CMSG_SPELLCLICK, "CMSG_SPELLCLICK" },
    { SMSG_LOOT_LIST, "SMSG_LOOT_LIST" },
    { MSG_GUILD_PERMISSIONS, "MSG_GUILD_PERMISSIONS" },
    { MSG_GUILD_EVENT_LOG_QUERY, "MSG_GUILD_EVENT_LOG_QUERY" },
    { CMSG_MAELSTROM_RENAME_GUILD, "CMSG_MAELSTROM_RENAME_GUILD" },
    { CMSG_GET_MIRRORIMAGE_DATA, "CMSG_GET_MIRRORIMAGE_DATA" },
    { SMSG_MIRRORIMAGE_DATA, "SMSG_MIRRORIMAGE_DATA" },
    { SMSG_FORCE_DISPLAY_UPDATE, "SMSG_FORCE_DISPLAY_UPDATE" },
    { SMSG_SPELL_CHANCE_RESIST_PUSHBACK, "SMSG_SPELL_CHANCE_RESIST_PUSHBACK" },
    { CMSG_IGNORE_DIMINISHING_RETURNS_CHEAT, "CMSG_IGNORE_DIMINISHING_RETURNS_CHEAT" },
    { SMSG_IGNORE_DIMINISHING_RETURNS_CHEAT, "SMSG_IGNORE_DIMINISHING_RETURNS_CHEAT" },
    { CMSG_KEEP_ALIVE, "CMSG_KEEP_ALIVE" },
    { SMSG_RAID_READY_CHECK_ERROR, "SMSG_RAID_READY_CHECK_ERROR" },
    { CMSG_OPT_OUT_OF_LOOT, "CMSG_OPT_OUT_OF_LOOT" },
    { CMSG_SET_GRANTABLE_LEVELS, "CMSG_SET_GRANTABLE_LEVELS" },
    { CMSG_GRANT_LEVEL, "CMSG_GRANT_LEVEL" },
    { CMSG_DECLINE_CHANNEL_INVITE, "CMSG_DECLINE_CHANNEL_INVITE" },
    { CMSG_GROUPACTION_THROTTLED, "CMSG_GROUPACTION_THROTTLED" },
    { SMSG_OVERRIDE_LIGHT, "SMSG_OVERRIDE_LIGHT" },
    { SMSG_TOTEM_CREATED, "SMSG_TOTEM_CREATED" },
    { CMSG_TOTEM_DESTROYED, "CMSG_TOTEM_DESTROYED" },
    { CMSG_EXPIRE_RAID_INSTANCE, "CMSG_EXPIRE_RAID_INSTANCE" },
    { CMSG_NO_SPELL_VARIANCE, "CMSG_NO_SPELL_VARIANCE" },
    { CMSG_QUESTGIVER_STATUS_MULTIPLE_QUERY, "CMSG_QUESTGIVER_STATUS_MULTIPLE_QUERY" },
    { SMSG_QUESTGIVER_STATUS_MULTIPLE, "SMSG_QUESTGIVER_STATUS_MULTIPLE" },
    { CMSG_QUERY_SERVER_BUCK_DATA, "CMSG_QUERY_SERVER_BUCK_DATA" },
    { CMSG_CLEAR_SERVER_BUCK_DATA, "CMSG_CLEAR_SERVER_BUCK_DATA" },
    { SMSG_SERVER_BUCK_DATA, "SMSG_SERVER_BUCK_DATA" },
    { SMSG_SEND_UNLEARN_SPELLS, "SMSG_SEND_UNLEARN_SPELLS" },
    { SMSG_PROPOSE_LEVEL_GRANT, "SMSG_PROPOSE_LEVEL_GRANT" },
    { CMSG_ACCEPT_LEVEL_GRANT, "CMSG_ACCEPT_LEVEL_GRANT" },
    { SMSG_REFER_A_FRIEND_FAILURE, "SMSG_REFER_A_FRIEND_FAILURE" },
    { SMSG_SUMMON_CANCEL, "SMSG_SUMMON_CANCEL" },
    { 0, NULL }
};

/*! Decrypts the header after the session key has been deducted as described in the top level comment. */
static guint8*
get_decrypted_header(const guint8 session_key[WOWW_SESSION_KEY_LENGTH],
                     guint8* idx,
                     guint8* last_encrypted_value,
                     const guint8* header,
                     guint8 header_size) {
    guint8* decrypted_header = wmem_alloc0(wmem_file_scope(), WOWW_HEADER_ARRAY_ALLOC_SIZE);

    for (guint8 i = 0; i < header_size; i++) {

        // x = (E - L) ^ S as described in top level comment
        decrypted_header[i] = (header[i] - *last_encrypted_value) ^ session_key[*idx];

        *last_encrypted_value = header[i];
        *idx = (*idx + 1) % WOWW_SESSION_KEY_LENGTH;
    }

    return decrypted_header;
}

/*! Deduces the session key values as described in the top level comment. */
static void
deduce_header(guint8 session_key[WOWW_SESSION_KEY_LENGTH],
              bool known_indices[WOWW_SESSION_KEY_LENGTH],
              const guint8* header,
              WowwParticipant_t* participant) {
    // Skip size field (2 bytes) and 2 least significant bytes of opcode field
    participant->idx = (participant->idx + 2 + 2) % WOWW_SESSION_KEY_LENGTH;
    // Set last encrypted value to what it's supposed to be
    participant->last_encrypted_value = header[3];

    // 0 ^ (E - L) as described in top level comment
    session_key[participant->idx] = 0 ^ (header[4] - participant->last_encrypted_value);
    known_indices[participant->idx] = true;
    participant->idx = (participant->idx + 1) % WOWW_SESSION_KEY_LENGTH;
    participant->last_encrypted_value = header[4];

    session_key[participant->idx] = 0 ^ (header[5] - participant->last_encrypted_value);
    known_indices[participant->idx] = true;
    participant->idx = (participant->idx + 1) % WOWW_SESSION_KEY_LENGTH;
    participant->last_encrypted_value = header[5];
}

/*! Returns true if all necessary values of the session key are fully known. */
static gboolean
session_key_is_fully_deduced(const bool known_indices[WOWW_SESSION_KEY_LENGTH],
                             guint8 header_length,
                             guint8 start_index) {
    gboolean fully_deduced = true;
    for (guint8 i = 0; i < header_length; i++) {
        if (!known_indices[(start_index + i) % WOWW_SESSION_KEY_LENGTH]) {
            fully_deduced = false;
        }
    }
    return fully_deduced;
}

/*! Returns either a pointer to a valid decrypted header, or NULL if no such header exists yet. */
static WowwDecryptedHeader_t*
handle_packet_header(packet_info* pinfo,
                     tvbuff_t* tvb,
                     WowwParticipant_t* participant,
                     WowwConversation_t* wowwConversation,
                     guint8 headerSize,
                     guint8 index_in_pdu,
                     gint tvb_offset) {
    guint64 key = ((guint64)index_in_pdu << 32) | pinfo->num;

    guint8* decrypted_header = wmem_map_lookup(wowwConversation->decrypted_headers, &key);

    if (decrypted_header) {
        // Header has already been decrypted
        return (WowwDecryptedHeader_t*)decrypted_header;
    }

    if (participant->stopped_at != 0 && participant->stopped_at != key) {
        // We can't continue decrypt further server messages since we
        // don't know the status of the session key index for any message
        // except the last one we couldn't decrypt.
        return NULL;
    }

    // First time we see this header, we need to decrypt it
    guint8* header = wmem_alloc0(wmem_packet_scope(), WOWW_HEADER_ARRAY_ALLOC_SIZE);
    for (int i = 0; i < headerSize; i++) {
        header[i] = tvb_get_guint8(tvb, tvb_offset + i);
    }

    // If we're seeing the first header
    if (!participant->unencrypted_packet_encountered) {
        // Packet is unencrypted, no need to do anything

        // There is only one unencrypted header each for server and client
        participant->unencrypted_packet_encountered = true;

        decrypted_header = wmem_alloc0(wmem_file_scope(), WOWW_HEADER_ARRAY_ALLOC_SIZE);
        memcpy(decrypted_header, header, headerSize);

        guint64* allocated_key = wmem_alloc0(wmem_file_scope(), sizeof(guint64));
        *allocated_key = key;

        wmem_map_insert(wowwConversation->decrypted_headers, allocated_key, decrypted_header);

        return (WowwDecryptedHeader_t*)decrypted_header;
    }

    WowwPreviousValues_t * original_header_values = wmem_map_lookup(wowwConversation->headers_need_decryption, &key);

    if (original_header_values && !session_key_is_fully_deduced(wowwConversation->known_indices, headerSize, original_header_values->idx)) {
        // If we have seen the header before AND
        // we still can't decrypt it
        // there's nothing to do but wait until we get more information
        return NULL;
    }

    if (!original_header_values && !session_key_is_fully_deduced(wowwConversation->known_indices, headerSize, participant->idx)) {
        // If we haven't seen the header before AND
        // we can't decrypt it now
        // we make sure it gets decrypted later
        WowwPreviousValues_t* array_index = wmem_alloc0(wmem_file_scope(), sizeof(WowwPreviousValues_t));
        array_index->idx = participant->idx;
        array_index->last_encrypted_value = participant->last_encrypted_value;

        guint64* allocated_key = wmem_alloc0(wmem_file_scope(), sizeof(guint64));
        *allocated_key = key;

        wmem_map_insert(wowwConversation->headers_need_decryption, allocated_key, array_index);

        // If it's a server header we can use it to deduce the session key
        if (WOWW_CLIENT_TO_SERVER) {
            deduce_header(wowwConversation->session_key, wowwConversation->known_indices, header, participant);
        } else {
            // We don't know if this PDU contains several messages or just one, so we need
            // to stop parsing server messages until we have fully decrypted this one.
            participant->stopped_at = key;
            // Skip the packet, but remember to acknowledge that values changed
            participant->idx = (participant->idx + headerSize) % WOWW_SESSION_KEY_LENGTH;
            participant->last_encrypted_value = header[headerSize - 1];
        }

        return NULL;
    }

    guint8* idx = &participant->idx;
    guint8* last_encrypted_value = &participant->last_encrypted_value;

    // If this is an out of order packet we must use the original state
    if (original_header_values) {
        // We can now (as best as possible) assume that decryption
        // is in the right place.
        participant->stopped_at = 0;
        // We do not care about how these values are mutated since
        // they are never going to be used again.
        idx = &original_header_values->idx;
        last_encrypted_value = &original_header_values->last_encrypted_value;

        // No need to decrypt it again
        wmem_map_remove(wowwConversation->headers_need_decryption, &key);
    }

    decrypted_header = get_decrypted_header(wowwConversation->session_key,
                                            idx,
                                            last_encrypted_value,
                                            header,
                                            headerSize);

    guint64* allocated_key = wmem_alloc0(wmem_file_scope(), sizeof(guint64));
    *allocated_key = key;

    // The header has been fully decrypted, cache it for future use
    wmem_map_insert(wowwConversation->decrypted_headers, allocated_key, decrypted_header);

    return (WowwDecryptedHeader_t*)decrypted_header;
}

static gint32
get_null_terminated_string_length( tvbuff_t* tvb,
                                   gint32 offset)
{
    const gint32 maximum_length = 0x7FFF;
    for (gint32 length = 0; length < maximum_length; length++) {
        guint8 character = tvb_get_guint8(tvb, offset + length);
        if (character == 0) {
            // Include the null character in the length
            return length + 1;
        }
    }

    return 0;
}

static void
add_cstring(ptvcursor_t* ptv, const int* hf) {
    gint32 len = get_null_terminated_string_length(ptvcursor_tvbuff(ptv), ptvcursor_current_offset(ptv));
    ptvcursor_add(ptv, *hf, len, ENC_UTF_8);
}

static void
add_sized_cstring(ptvcursor_t* ptv, const int* hf) {
    ptvcursor_add(ptv, hf_woww_string_length, 4, ENC_LITTLE_ENDIAN);
    gint32 len = get_null_terminated_string_length(ptvcursor_tvbuff(ptv), ptvcursor_current_offset(ptv));
    ptvcursor_add(ptv, *hf, len, ENC_UTF_8);
}

static void
add_aura_mask(ptvcursor_t* ptv) {
    guint32 mask = 0;
    ptvcursor_add_ret_uint(ptv, hf_woww_aura_mask, 4, ENC_LITTLE_ENDIAN, &mask);

    for (gint i = 0; i < 32; ++i) {
        if (mask & (1u << i)) {
          ptvcursor_add(ptv, hf_woww_aura, 4, ENC_LITTLE_ENDIAN);
        }
    }
}

static void
add_monster_move_spline(ptvcursor_t* ptv) {
    guint32 amount_of_splines = 0;
    ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_splines, 4, ENC_LITTLE_ENDIAN, &amount_of_splines);

    if (amount_of_splines >= 1) {
        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Unpacked Spline");
        ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
        ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
        ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
        ptvcursor_pop_subtree(ptv);
    }

    for (guint32 i = 1; i < amount_of_splines; ++i) {
        ptvcursor_add(ptv, hf_woww_packed_spline, 4, ENC_LITTLE_ENDIAN);
    }
}

static void
add_update_mask(ptvcursor_t* ptv, packet_info* pinfo) {
    guint32 amount_of_masks = 0;
    ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_masks, 1, ENC_NA, &amount_of_masks);

    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Update Mask");
    guint32* masks = (guint32*)wmem_alloc0(wmem_packet_scope(), amount_of_masks * 4);
    for (guint32 i = 0; i < amount_of_masks; ++i) {
        ptvcursor_add_ret_uint(ptv, hf_woww_update_mask, 4, ENC_LITTLE_ENDIAN, masks + i);
    }
    ptvcursor_pop_subtree(ptv);

    guint32 amount_of_indices = 0;
    for (guint32 mask = 0; mask < amount_of_masks; ++mask) {
        for (guint bit = 0; bit < 32; ++bit) {
            if (masks[mask] & (1u << bit)) {
                amount_of_indices += 1;
            }
        }
    }

    guint32 indices_length = amount_of_indices * 4;
    guint32* indices = (guint32*)wmem_alloc0(wmem_packet_scope(), indices_length);
    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Update Mask Values");

    tvbuff_t* tvb = ptvcursor_tvbuff(ptv);

    tvbuff_t* next_tvb = tvb_new_child_real_data(tvb, (guint8*)indices, indices_length, indices_length);
    add_new_data_source(pinfo, next_tvb, "Update Mask Indices");

    guint32 indices_index = 0;
    for (guint32 mask = 0; mask < amount_of_masks; ++mask) {
        for (guint32 i = 0; i < 32; ++i) {
            if (masks[mask] & (1u << i)) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Update Mask Pair");

                indices[indices_index] = (32 * mask) + i;
                proto_tree* tree = ptvcursor_tree(ptv);
                proto_tree_add_item(tree, hf_woww_update_mask_index, next_tvb, 4 * indices_index, 4, ENC_LITTLE_ENDIAN);
                indices_index += 1;

                ptvcursor_add(ptv, hf_woww_update_mask_value, 4, ENC_LITTLE_ENDIAN);

                ptvcursor_pop_subtree(ptv);
            }
        }
    }
    ptvcursor_pop_subtree(ptv);
}

static void
add_packed_guid(ptvcursor_t* ptv, packet_info* pinfo) {
    tvbuff_t* tvb = ptvcursor_tvbuff(ptv);
    proto_tree* tree = ptvcursor_tree(ptv);

    guint8 byte_mask = tvb_get_guint8(tvb, ptvcursor_current_offset(ptv));
    ptvcursor_advance(ptv, 1);

    guint8* guid = (guint8*)wmem_alloc(wmem_packet_scope(), 8);

    for (gint i = 0; i < 8; ++i) {
        if (byte_mask & (1 << i)) {
            guid[i] = tvb_get_guint8(tvb, ptvcursor_current_offset(ptv));
            ptvcursor_advance(ptv, 1);
        }
    }

    tvbuff_t* next_tvb = tvb_new_child_real_data(tvb, (guint8*)guid, 8, 8);
    add_new_data_source(pinfo, next_tvb, "Packed Guid");

    proto_tree_add_item(tree, hf_woww_guid, next_tvb, 0, 8, ENC_LITTLE_ENDIAN);
}

static void
add_body_fields(guint32 header_opcode,
                proto_tree* tree,
                tvbuff_t* tvb,
                gint32 offset,
                gint32 offset_packet_end,
                packet_info* pinfo)
{
    guint32 len = 0;
    tvbuff_t* compressed_tvb = NULL;
/* AUTOGENERATED_START_VARIABLES */
    guint32 action = 0;
    guint32 amount_of_auras = 0;
    guint32 amount_of_carriers = 0;
    guint32 amount_of_characters = 0;
    guint32 amount_of_choice_item_rewards = 0;
    guint32 amount_of_cooldowns = 0;
    guint32 amount_of_damages = 0;
    guint32 amount_of_effects = 0;
    guint32 amount_of_emotes = 0;
    guint32 amount_of_entries = 0;
    guint32 amount_of_events = 0;
    guint32 amount_of_extra_fields = 0;
    guint32 amount_of_faction_standings = 0;
    guint32 amount_of_factions = 0;
    guint32 amount_of_friends = 0;
    guint32 amount_of_gossip_items = 0;
    guint32 amount_of_hits = 0;
    guint32 amount_of_ignored = 0;
    guint32 amount_of_item_rewards = 0;
    guint32 amount_of_items = 0;
    guint32 amount_of_mails = 0;
    guint32 amount_of_members = 0;
    guint32 amount_of_misses = 0;
    guint32 amount_of_nodes = 0;
    guint32 amount_of_objects = 0;
    guint32 amount_of_outbid_items = 0;
    guint32 amount_of_petitions = 0;
    guint32 amount_of_pets = 0;
    guint32 amount_of_players = 0;
    guint32 amount_of_quests = 0;
    guint32 amount_of_raid_infos = 0;
    guint32 amount_of_reactions = 0;
    guint32 amount_of_records = 0;
    guint32 amount_of_required_items = 0;
    guint32 amount_of_rights = 0;
    guint32 amount_of_signatures = 0;
    guint32 amount_of_spells = 0;
    guint32 amount_of_states = 0;
    guint32 amount_of_strings = 0;
    guint32 amount_of_targets = 0;
    guint32 amount_of_teammates = 0;
    guint32 amount_of_zones = 0;
    guint32 aura_type = 0;
    guint32 category = 0;
    guint32 chat_type = 0;
    guint32 cooldown_count = 0;
    guint32 count = 0;
    guint32 effect = 0;
    guint32 exp_type = 0;
    guint32 flags = 0;
    guint32 info_block = 0;
    guint32 key_version = 0;
    guint32 listed_players = 0;
    guint32 loot_method = 0;
    guint32 map = 0;
    guint32 mask = 0;
    guint32 message_type = 0;
    guint32 miss_info = 0;
    guint32 move_type = 0;
    guint32 node_count = 0;
    guint32 number_of_battlegrounds = 0;
    guint32 opcode = 0;
    guint32 reason = 0;
    guint32 result = 0;
    guint32 result2 = 0;
    guint32 spell_count = 0;
    guint32 spline_flags = 0;
    guint32 status = 0;
    guint32 status_id = 0;
    guint32 target_flags = 0;
    guint32 target_index = 0;
    guint32 update_flag = 0;
    guint32 update_type = 0;
    guint32 url_info = 0;
/* AUTOGENERATED_END_VARIABLES */
    ptvcursor_t* ptv = ptvcursor_new(wmem_packet_scope(), tree, tvb, offset);
/* AUTOGENERATED_START_PARSER */
    switch (header_opcode) {
        case CMSG_ACCEPT_TRADE:
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_ACTIVATETAXI:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_source_node, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_destination_node, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_ACTIVATETAXIEXPRESS:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_total_cost, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_node_count, 4, ENC_LITTLE_ENDIAN, &node_count);
            for (guint32 i1 = 0; i1 < node_count; ++i1) {
                ptvcursor_add(ptv, hf_woww_nodes, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case CMSG_ADD_FRIEND:
            add_cstring(ptv, &hf_woww_name);
            break;
        case CMSG_ADD_IGNORE:
            add_cstring(ptv, &hf_woww_name);
            break;
        case CMSG_AREATRIGGER:
            ptvcursor_add(ptv, hf_woww_trigger_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AREA_SPIRIT_HEALER_QUERY:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AREA_SPIRIT_HEALER_QUEUE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_ATTACKSWING:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUCTION_LIST_BIDDER_ITEMS:
            ptvcursor_add(ptv, hf_woww_auctioneer, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_start_from_page, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_outbid_items, 4, ENC_LITTLE_ENDIAN, &amount_of_outbid_items);
            for (guint32 i1 = 0; i1 < amount_of_outbid_items; ++i1) {
                ptvcursor_add(ptv, hf_woww_outbid_item_ids, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case CMSG_AUCTION_LIST_ITEMS:
            ptvcursor_add(ptv, hf_woww_auctioneer, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_list_start_item, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_searched_name);
            ptvcursor_add(ptv, hf_woww_minimum_level, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_maximum_level, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auction_slot_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auction_main_category, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auction_sub_category, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_quality, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_usable, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUCTION_LIST_OWNER_ITEMS:
            ptvcursor_add(ptv, hf_woww_auctioneer, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_list_from, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUCTION_PLACE_BID:
            ptvcursor_add(ptv, hf_woww_auctioneer, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auction_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_price, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUCTION_REMOVE_ITEM:
            ptvcursor_add(ptv, hf_woww_auctioneer, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auction_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUCTION_SELL_ITEM:
            ptvcursor_add(ptv, hf_woww_auctioneer, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_starting_bid, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_buyout, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auction_duration_in_minutes, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUTH_SESSION:
            ptvcursor_add(ptv, hf_woww_build, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_server_id, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_username);
            ptvcursor_add(ptv, hf_woww_client_seed, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_client_proof, 20, ENC_NA);
            ptvcursor_add(ptv, hf_woww_decompressed_size, 4, ENC_LITTLE_ENDIAN);
            compressed_tvb = tvb_uncompress_zlib(ptvcursor_tvbuff(ptv), ptvcursor_current_offset(ptv), offset_packet_end - ptvcursor_current_offset(ptv));
            if (compressed_tvb != NULL) {
                ptvcursor_t* old_ptv = ptv;
                ptv = ptvcursor_new(wmem_packet_scope(), tree, compressed_tvb, 0);
                gint compression_end = tvb_reported_length(compressed_tvb);
                while (ptvcursor_current_offset(ptv) < compression_end) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "AddonInfo");
                    add_cstring(ptv, &hf_woww_addon_name);
                    ptvcursor_add(ptv, hf_woww_addon_has_signature, 1, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_addon_crc, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_addon_extra_crc, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                ptvcursor_free(ptv);
                ptv = old_ptv;
                compressed_tvb = NULL;
            }
            break;
        case CMSG_AUTOBANK_ITEM:
            ptvcursor_add(ptv, hf_woww_bag_index, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_slot_index, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUTOEQUIP_ITEM:
            ptvcursor_add(ptv, hf_woww_source_bag, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_source_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUTOEQUIP_ITEM_SLOT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_destination_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUTOSTORE_BAG_ITEM:
            ptvcursor_add(ptv, hf_woww_source_bag, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_source_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_destination_bag, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUTOSTORE_BANK_ITEM:
            ptvcursor_add(ptv, hf_woww_bag_index, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_slot_index, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_AUTOSTORE_LOOT_ITEM:
            ptvcursor_add(ptv, hf_woww_item_slot_int, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BANKER_ACTIVATE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BATTLEFIELD_JOIN:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BATTLEFIELD_LIST:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BATTLEFIELD_PORT:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_battlefield_port_action, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BATTLEMASTER_HELLO:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BATTLEMASTER_JOIN:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_instance_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_join_as_group, 1, ENC_NA);
            break;
        case CMSG_BINDER_ACTIVATE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BUG:
            ptvcursor_add(ptv, hf_woww_suggestion, 4, ENC_LITTLE_ENDIAN);
            add_sized_cstring(ptv, &hf_woww_content);
            add_sized_cstring(ptv, &hf_woww_bug_type);
            break;
        case CMSG_BUYBACK_ITEM:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_buyback_slot, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BUY_BANK_SLOT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BUY_ITEM:
            ptvcursor_add(ptv, hf_woww_vendor, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_amount, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BUY_ITEM_IN_SLOT:
            ptvcursor_add(ptv, hf_woww_vendor, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_bag, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_bag_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_amount, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_BUY_STABLE_SLOT:
            ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_CANCEL_AURA:
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_CANCEL_CAST:
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_CANCEL_CHANNELLING:
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_CAST_SPELL:
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "SpellCastTargets");
            ptvcursor_add_ret_uint(ptv, hf_woww_spell_cast_target_flags, 2, ENC_LITTLE_ENDIAN, &target_flags);
            if (target_flags & SPELL_CAST_TARGET_FLAGS_UNIT) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_GAMEOBJECT) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_OBJECT_UNK) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_TRADE_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_SOURCE_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_DEST_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_STRING) {
                add_cstring(ptv, &hf_woww_target_string);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_PVP_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            ptvcursor_pop_subtree(ptv);
            break;
        case CMSG_CHANNEL_ANNOUNCEMENTS:
            add_cstring(ptv, &hf_woww_channel_name);
            break;
        case CMSG_CHANNEL_BAN:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_CHANNEL_INVITE:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_CHANNEL_KICK:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_CHANNEL_LIST:
            add_cstring(ptv, &hf_woww_channel_name);
            break;
        case CMSG_CHANNEL_MODERATE:
            add_cstring(ptv, &hf_woww_channel_name);
            break;
        case CMSG_CHANNEL_MODERATOR:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_CHANNEL_MUTE:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_CHANNEL_OWNER:
            add_cstring(ptv, &hf_woww_channel_name);
            break;
        case CMSG_CHANNEL_PASSWORD:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_channel_password);
            break;
        case CMSG_CHANNEL_SET_OWNER:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_new_owner);
            break;
        case CMSG_CHANNEL_UNBAN:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_CHANNEL_UNMODERATOR:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_CHANNEL_UNMUTE:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_CHAR_CREATE:
            add_cstring(ptv, &hf_woww_name);
            ptvcursor_add(ptv, hf_woww_race, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_class, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_gender, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_skin_color, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_face, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_hair_style, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_hair_color, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_facial_hair, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_outfit_id, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_CHAR_DELETE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_CHAR_RENAME:
            ptvcursor_add(ptv, hf_woww_character_guid, 8, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_new_name);
            break;
        case CMSG_CHAT_IGNORED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_CLEAR_TRADE_ITEM:
            ptvcursor_add(ptv, hf_woww_trade_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_CREATURE_QUERY:
            ptvcursor_add(ptv, hf_woww_creature, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_DBLOOKUP:
            add_cstring(ptv, &hf_woww_query);
            break;
        case CMSG_DEL_FRIEND:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_DEL_IGNORE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_DESTROYITEM:
            ptvcursor_add(ptv, hf_woww_bag, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_amount, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_DUEL_ACCEPTED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_DUEL_CANCELLED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_EMOTE:
            ptvcursor_add(ptv, hf_woww_emote_enum, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_FAR_SIGHT:
            ptvcursor_add(ptv, hf_woww_far_sight_operation, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_FORCE_MOVE_ROOT_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_movement_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            break;
        case CMSG_FORCE_MOVE_UNROOT_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_movement_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            break;
        case CMSG_FORCE_RUN_BACK_SPEED_CHANGE_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_movement_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_new_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_FORCE_RUN_SPEED_CHANGE_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_new_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_FORCE_SWIM_BACK_SPEED_CHANGE_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_new_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_FORCE_SWIM_SPEED_CHANGE_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_new_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_FORCE_TURN_RATE_CHANGE_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_new_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_FORCE_WALK_SPEED_CHANGE_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_new_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_GAMEOBJECT_QUERY:
            ptvcursor_add(ptv, hf_woww_entry_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_GAMEOBJ_USE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_GET_MAIL_LIST:
            ptvcursor_add(ptv, hf_woww_mailbox, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_GMSURVEY_SUBMIT:
            ptvcursor_add(ptv, hf_woww_survey_id, 4, ENC_LITTLE_ENDIAN);
            for (guint32 i1 = 0; i1 < 10; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "GmSurveyQuestion %i", i1);
                ptvcursor_add(ptv, hf_woww_question_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_answer, 1, ENC_LITTLE_ENDIAN);
                add_cstring(ptv, &hf_woww_comment);
                ptvcursor_pop_subtree(ptv);
            }
            add_cstring(ptv, &hf_woww_answer_comment);
            break;
        case CMSG_GMTICKET_CREATE:
            ptvcursor_add_ret_uint(ptv, hf_woww_gm_ticket_type, 1, ENC_LITTLE_ENDIAN, &category);
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            add_cstring(ptv, &hf_woww_message);
            add_cstring(ptv, &hf_woww_reserved_for_future_use);
            if (category == GM_TICKET_TYPE_BEHAVIOR_HARASSMENT) {
                ptvcursor_add(ptv, hf_woww_chat_data_line_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_decompressed_size, 4, ENC_LITTLE_ENDIAN);
                compressed_tvb = tvb_uncompress_zlib(ptvcursor_tvbuff(ptv), ptvcursor_current_offset(ptv), offset_packet_end - ptvcursor_current_offset(ptv));
                if (compressed_tvb != NULL) {
                    ptvcursor_t* old_ptv = ptv;
                    ptv = ptvcursor_new(wmem_packet_scope(), tree, compressed_tvb, 0);
                    len = offset_packet_end - ptvcursor_current_offset(ptv);
                    ptvcursor_add(ptv, hf_woww_compressed_chat_data, len, ENC_NA);
                    ptvcursor_free(ptv);
                    ptv = old_ptv;
                    compressed_tvb = NULL;
                }
            }
            break;
        case CMSG_GMTICKET_UPDATETEXT:
            ptvcursor_add(ptv, hf_woww_gm_ticket_type, 1, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_message);
            break;
        case CMSG_GOSSIP_HELLO:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_GOSSIP_SELECT_OPTION:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_gossip_list_id, 4, ENC_LITTLE_ENDIAN);
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            if (len > 0) {
                add_cstring(ptv, &hf_woww_code);
            }
            break;
        case CMSG_GROUP_ASSISTANT_LEADER:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_set_assistant, 1, ENC_NA);
            break;
        case CMSG_GROUP_CHANGE_SUB_GROUP:
            add_cstring(ptv, &hf_woww_name);
            ptvcursor_add(ptv, hf_woww_group_number, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_GROUP_INVITE:
            add_cstring(ptv, &hf_woww_name);
            break;
        case CMSG_GROUP_SET_LEADER:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_GROUP_SWAP_SUB_GROUP:
            add_cstring(ptv, &hf_woww_name);
            add_cstring(ptv, &hf_woww_swap_with_name);
            break;
        case CMSG_GROUP_UNINVITE:
            add_cstring(ptv, &hf_woww_name);
            break;
        case CMSG_GROUP_UNINVITE_GUID:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_GUILD_ADD_RANK:
            add_cstring(ptv, &hf_woww_rank_name);
            break;
        case CMSG_GUILD_CREATE:
            add_cstring(ptv, &hf_woww_guild_name);
            break;
        case CMSG_GUILD_DEMOTE:
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_GUILD_INFO_TEXT:
            add_cstring(ptv, &hf_woww_guild_info);
            break;
        case CMSG_GUILD_INVITE:
            add_cstring(ptv, &hf_woww_invited_player);
            break;
        case CMSG_GUILD_LEADER:
            add_cstring(ptv, &hf_woww_new_guild_leader_name);
            break;
        case CMSG_GUILD_MOTD:
            add_cstring(ptv, &hf_woww_message_of_the_day);
            break;
        case CMSG_GUILD_PROMOTE:
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_GUILD_QUERY:
            ptvcursor_add(ptv, hf_woww_guild_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_GUILD_RANK:
            ptvcursor_add(ptv, hf_woww_rank_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_rights, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_rank_name);
            break;
        case CMSG_GUILD_REMOVE:
            add_cstring(ptv, &hf_woww_player_name);
            break;
        case CMSG_GUILD_SET_OFFICER_NOTE:
            add_cstring(ptv, &hf_woww_player_name);
            add_cstring(ptv, &hf_woww_note);
            break;
        case CMSG_GUILD_SET_PUBLIC_NOTE:
            add_cstring(ptv, &hf_woww_player_name);
            add_cstring(ptv, &hf_woww_note);
            break;
        case CMSG_INITIATE_TRADE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_INSPECT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_ITEM_NAME_QUERY:
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_ITEM_QUERY_SINGLE:
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_ITEM_TEXT_QUERY:
            ptvcursor_add(ptv, hf_woww_item_text_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_mail_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_JOIN_CHANNEL:
            add_cstring(ptv, &hf_woww_channel_name);
            add_cstring(ptv, &hf_woww_channel_password);
            break;
        case CMSG_LEARN_TALENT:
            ptvcursor_add(ptv, hf_woww_talent, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_requested_rank, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_LEAVE_BATTLEFIELD:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_LEAVE_CHANNEL:
            add_cstring(ptv, &hf_woww_channel_name);
            break;
        case CMSG_LIST_INVENTORY:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_LOOT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_LOOT_MASTER_GIVE:
            ptvcursor_add(ptv, hf_woww_loot, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_slot_id, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_player, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_LOOT_METHOD:
            ptvcursor_add(ptv, hf_woww_group_loot_setting, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_loot_master, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_quality, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_LOOT_RELEASE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_LOOT_ROLL:
            ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_slot_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_roll_vote, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MAIL_CREATE_TEXT_ITEM:
            ptvcursor_add(ptv, hf_woww_mailbox, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_mail_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_mail_template_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MAIL_DELETE:
            ptvcursor_add(ptv, hf_woww_mailbox_id, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_mail_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MAIL_MARK_AS_READ:
            ptvcursor_add(ptv, hf_woww_mailbox, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_mail_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MAIL_RETURN_TO_SENDER:
            ptvcursor_add(ptv, hf_woww_mailbox_id, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_mail_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MAIL_TAKE_ITEM:
            ptvcursor_add(ptv, hf_woww_mailbox, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_mail_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MAIL_TAKE_MONEY:
            ptvcursor_add(ptv, hf_woww_mailbox, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_mail_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MEETINGSTONE_JOIN:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MESSAGECHAT:
            ptvcursor_add_ret_uint(ptv, hf_woww_chat_type, 4, ENC_LITTLE_ENDIAN, &chat_type);
            ptvcursor_add(ptv, hf_woww_language, 4, ENC_LITTLE_ENDIAN);
            if (chat_type == CHAT_TYPE_WHISPER) {
                add_cstring(ptv, &hf_woww_target_player);
            }
            else if (chat_type == CHAT_TYPE_CHANNEL) {
                add_cstring(ptv, &hf_woww_channel);
            }
            add_cstring(ptv, &hf_woww_message);
            break;
        case CMSG_MOVE_FALL_RESET:
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            break;
        case CMSG_MOVE_FEATHER_FALL_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_movement_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_apply, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MOVE_HOVER_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_is_applied, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MOVE_KNOCK_BACK_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            break;
        case CMSG_MOVE_NOT_ACTIVE_MOVER:
            ptvcursor_add(ptv, hf_woww_old_mover, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            break;
        case CMSG_MOVE_SET_RAW_POSITION:
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MOVE_SPLINE_DONE:
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_movement_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MOVE_TIME_SKIPPED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_lag, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_MOVE_WATER_WALK_ACK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_movement_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_apply, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_NAME_QUERY:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_NPC_TEXT_QUERY:
            ptvcursor_add(ptv, hf_woww_text_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_OFFER_PETITION:
            ptvcursor_add(ptv, hf_woww_petition, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_OPEN_ITEM:
            ptvcursor_add(ptv, hf_woww_bag_index, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PAGE_TEXT_QUERY:
            ptvcursor_add(ptv, hf_woww_page_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PETITION_BUY:
            ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_guid, 8, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_name);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 2, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_index, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PETITION_QUERY:
            ptvcursor_add(ptv, hf_woww_guild_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_petition, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PETITION_SHOWLIST:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PETITION_SHOW_SIGNATURES:
            ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PETITION_SIGN:
            ptvcursor_add(ptv, hf_woww_petition, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PET_ABANDON:
            ptvcursor_add(ptv, hf_woww_pet, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PET_ACTION:
            ptvcursor_add(ptv, hf_woww_pet, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_data, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PET_CANCEL_AURA:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PET_CAST_SPELL:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "SpellCastTargets");
            ptvcursor_add_ret_uint(ptv, hf_woww_spell_cast_target_flags, 2, ENC_LITTLE_ENDIAN, &target_flags);
            if (target_flags & SPELL_CAST_TARGET_FLAGS_UNIT) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_GAMEOBJECT) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_OBJECT_UNK) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_TRADE_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_SOURCE_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_DEST_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_STRING) {
                add_cstring(ptv, &hf_woww_target_string);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_PVP_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            ptvcursor_pop_subtree(ptv);
            break;
        case CMSG_PET_NAME_QUERY:
            ptvcursor_add(ptv, hf_woww_pet_number, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PET_RENAME:
            ptvcursor_add(ptv, hf_woww_pet, 8, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_name);
            break;
        case CMSG_PET_SET_ACTION:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_position, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_data, 4, ENC_LITTLE_ENDIAN);
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            if (len > 0) {
                ptvcursor_add(ptv, hf_woww_position, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_data, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case CMSG_PET_SPELL_AUTOCAST:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_autocast_enabled, 1, ENC_NA);
            break;
        case CMSG_PET_STOP_ATTACK:
            ptvcursor_add(ptv, hf_woww_pet, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PET_UNLEARN:
            ptvcursor_add(ptv, hf_woww_pet, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PING:
            ptvcursor_add(ptv, hf_woww_sequence_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_round_time_in_ms, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PLAYER_LOGIN:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_PUSHQUESTTOPARTY:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUESTGIVER_ACCEPT_QUEST:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUESTGIVER_CHOOSE_REWARD:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_reward, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUESTGIVER_COMPLETE_QUEST:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUESTGIVER_HELLO:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUESTGIVER_QUERY_QUEST:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUESTGIVER_REQUEST_REWARD:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUESTGIVER_STATUS_QUERY:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUESTLOG_REMOVE_QUEST:
            ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUESTLOG_SWAP_QUEST:
            ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUEST_CONFIRM_ACCEPT:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_QUEST_QUERY:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_READ_ITEM:
            ptvcursor_add(ptv, hf_woww_bag_index, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_RECLAIM_CORPSE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_REPAIR_ITEM:
            ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_REQUEST_ACCOUNT_DATA:
            ptvcursor_add(ptv, hf_woww_data_type, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_REQUEST_PARTY_MEMBER_STATS:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_RESURRECT_RESPONSE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_status, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SELL_ITEM:
            ptvcursor_add(ptv, hf_woww_vendor, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_amount, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SEND_MAIL:
            ptvcursor_add(ptv, hf_woww_mailbox, 8, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_receiver);
            add_cstring(ptv, &hf_woww_subject);
            add_cstring(ptv, &hf_woww_body);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_money, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_cash_on_delivery_amount, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SETSHEATHED:
            ptvcursor_add(ptv, hf_woww_sheath_state, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_ACTIONBAR_TOGGLES:
            ptvcursor_add(ptv, hf_woww_action_bar, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_ACTION_BUTTON:
            ptvcursor_add(ptv, hf_woww_button, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_action, 2, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_misc, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_action_type, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_ACTIVE_MOVER:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_AMMO:
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_FACTION_ATWAR:
            ptvcursor_add(ptv, hf_woww_faction, 2, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_faction_flag, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_FACTION_INACTIVE:
            ptvcursor_add(ptv, hf_woww_faction, 2, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_inactive, 1, ENC_NA);
            break;
        case CMSG_SET_SELECTION:
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_TARGET_OBSOLETE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_TRADE_GOLD:
            ptvcursor_add(ptv, hf_woww_gold, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_TRADE_ITEM:
            ptvcursor_add(ptv, hf_woww_trade_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_bag, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SET_WATCHED_FACTION:
            ptvcursor_add(ptv, hf_woww_faction, 2, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SPIRIT_HEALER_ACTIVATE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SPLIT_ITEM:
            ptvcursor_add(ptv, hf_woww_source_bag, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_source_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_destination_bag, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_destination_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_amount, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_STABLE_PET:
            ptvcursor_add(ptv, hf_woww_stable_master, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_STABLE_SWAP_PET:
            ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_pet_slot, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_STANDSTATECHANGE:
            ptvcursor_add(ptv, hf_woww_unit_stand_state, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SUMMON_RESPONSE:
            ptvcursor_add(ptv, hf_woww_summoner, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SWAP_INV_ITEM:
            ptvcursor_add(ptv, hf_woww_item_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_SWAP_ITEM:
            ptvcursor_add(ptv, hf_woww_destination_bag, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_destionation_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_source_bag, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_source_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_TAXINODE_STATUS_QUERY:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_TAXIQUERYAVAILABLENODES:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_TELEPORT_TO_UNIT:
            add_cstring(ptv, &hf_woww_name);
            break;
        case CMSG_TEXT_EMOTE:
            ptvcursor_add(ptv, hf_woww_text_emote, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_emote_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_TOGGLE_PVP:
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            if (len > 0) {
                ptvcursor_add(ptv, hf_woww_enable_pvp, 1, ENC_NA);
            }
            break;
        case CMSG_TRAINER_BUY_SPELL:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_TRAINER_LIST:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_TURN_IN_PETITION:
            ptvcursor_add(ptv, hf_woww_petition, 8, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_TUTORIAL_FLAG:
            ptvcursor_add(ptv, hf_woww_tutorial_flag, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_UNLEARN_SKILL:
            ptvcursor_add(ptv, hf_woww_skill, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_UNSTABLE_PET:
            ptvcursor_add(ptv, hf_woww_stable_master, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_pet_number, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_UPDATE_ACCOUNT_DATA:
            ptvcursor_add(ptv, hf_woww_account_data_type, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_decompressed_size, 4, ENC_LITTLE_ENDIAN);
            compressed_tvb = tvb_uncompress_zlib(ptvcursor_tvbuff(ptv), ptvcursor_current_offset(ptv), offset_packet_end - ptvcursor_current_offset(ptv));
            if (compressed_tvb != NULL) {
                ptvcursor_t* old_ptv = ptv;
                ptv = ptvcursor_new(wmem_packet_scope(), tree, compressed_tvb, 0);
                len = offset_packet_end - ptvcursor_current_offset(ptv);
                ptvcursor_add(ptv, hf_woww_compressed_data, len, ENC_NA);
                ptvcursor_free(ptv);
                ptv = old_ptv;
                compressed_tvb = NULL;
            }
            break;
        case CMSG_USE_ITEM:
            ptvcursor_add(ptv, hf_woww_bag_index, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_bag_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell_index, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "SpellCastTargets");
            ptvcursor_add_ret_uint(ptv, hf_woww_spell_cast_target_flags, 2, ENC_LITTLE_ENDIAN, &target_flags);
            if (target_flags & SPELL_CAST_TARGET_FLAGS_UNIT) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_GAMEOBJECT) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_OBJECT_UNK) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_TRADE_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_SOURCE_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_DEST_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_STRING) {
                add_cstring(ptv, &hf_woww_target_string);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_PVP_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            ptvcursor_pop_subtree(ptv);
            break;
        case CMSG_WARDEN_DATA:
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            ptvcursor_add(ptv, hf_woww_encrypted_data, len, ENC_NA);
            break;
        case CMSG_WHO:
            ptvcursor_add(ptv, hf_woww_minimum_level, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_maximum_level, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_player_name);
            add_cstring(ptv, &hf_woww_guild_name);
            ptvcursor_add(ptv, hf_woww_race_mask, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_class_mask, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_zones, 4, ENC_LITTLE_ENDIAN, &amount_of_zones);
            for (guint32 i1 = 0; i1 < amount_of_zones; ++i1) {
                ptvcursor_add(ptv, hf_woww_zones, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_strings, 4, ENC_LITTLE_ENDIAN, &amount_of_strings);
            for (guint32 i1 = 0; i1 < amount_of_strings; ++i1) {
                add_cstring(ptv, &hf_woww_search_strings);
            }
            break;
        case CMSG_WHOIS:
            add_cstring(ptv, &hf_woww_character_name);
            break;
        case CMSG_WORLD_TELEPORT:
            ptvcursor_add(ptv, hf_woww_time, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_WRAP_ITEM:
            ptvcursor_add(ptv, hf_woww_gift_bag_index, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_gift_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_bag_index, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_slot_int, 1, ENC_LITTLE_ENDIAN);
            break;
        case CMSG_ZONEUPDATE:
            ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            break;
        case MSG_AUCTION_HELLO:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_auctioneer, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_auction_house, 4, ENC_LITTLE_ENDIAN);
            }
            else {
                ptvcursor_add(ptv, hf_woww_auctioneer, 8, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_BATTLEGROUND_PLAYER_POSITIONS:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_teammates, 4, ENC_LITTLE_ENDIAN, &amount_of_teammates);
                for (guint32 i1 = 0; i1 < amount_of_teammates; ++i1) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "BattlegroundPlayerPosition %i", i1);
                    ptvcursor_add(ptv, hf_woww_player, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_position_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_position_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_carriers, 1, ENC_LITTLE_ENDIAN, &amount_of_carriers);
                for (guint32 i1 = 0; i1 < amount_of_carriers; ++i1) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "BattlegroundPlayerPosition %i", i1);
                    ptvcursor_add(ptv, hf_woww_player, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_position_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_position_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
            }
            else {
            }
            break;
        case MSG_CHANNEL_START:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_CHANNEL_UPDATE:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_time, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_CORPSE_QUERY:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add_ret_uint(ptv, hf_woww_corpse_query_result, 1, ENC_LITTLE_ENDIAN, &result);
                if (result == CORPSE_QUERY_RESULT_FOUND) {
                    ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
                }
            }
            else {
            }
            break;
        case MSG_INSPECT_HONOR_STATS:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_pvp_rank, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_today_honorable_and_dishonorable, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_yesterday_honorable, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_unknown_int, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_last_week_honorable, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_unknown_int, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_this_week_honorable, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_unknown_int, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_lifetime_honorable, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_lifetime_dishonorable, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_yesterday_honor, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_last_week_honor, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_this_week_honor, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_pvp_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_rank_progress_bar, 1, ENC_LITTLE_ENDIAN);
            }
            else {
                ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_LIST_STABLED_PETS:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_pets, 1, ENC_LITTLE_ENDIAN, &amount_of_pets);
                ptvcursor_add(ptv, hf_woww_stable_slots, 1, ENC_LITTLE_ENDIAN);
                for (guint32 i1 = 0; i1 < amount_of_pets; ++i1) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "StabledPet %i", i1);
                    ptvcursor_add(ptv, hf_woww_pet_number, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_entry, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_level, 4, ENC_LITTLE_ENDIAN);
                    add_cstring(ptv, &hf_woww_name);
                    ptvcursor_add(ptv, hf_woww_loyalty, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
            }
            else {
                ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_LOOKING_FOR_GROUP:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            }
            else {
            }
            break;
        case MSG_MINIMAP_PING:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_position_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_position_y, 4, ENC_LITTLE_ENDIAN);
            }
            else {
                ptvcursor_add(ptv, hf_woww_position_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_position_y, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_MOVE_FALL_LAND:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_FEATHER_FALL:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_HEARTBEAT:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_JUMP:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_SET_FACING:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_SET_PITCH:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_SET_RUN_MODE:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_SET_WALK_MODE:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_START_BACKWARD:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_START_FORWARD:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_START_PITCH_DOWN:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_START_PITCH_UP:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_START_STRAFE_LEFT:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_START_STRAFE_RIGHT:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_START_SWIM:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_START_TURN_LEFT:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_START_TURN_RIGHT:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_STOP_PITCH:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_STOP_STRAFE:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_STOP_SWIM:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_STOP:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_STOP_TURN:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case MSG_MOVE_TELEPORT_ACK:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add(ptv, hf_woww_movement_counter, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
                ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                    ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                if (flags & MOVEMENT_FLAGS_SWIMMING) {
                    ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                if (flags & MOVEMENT_FLAGS_JUMPING) {
                    ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                }
                if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                    ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            else {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add(ptv, hf_woww_movement_counter, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_time, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_MOVE_TIME_SKIPPED:
            if (WOWW_SERVER_TO_CLIENT) {
                add_packed_guid(ptv, pinfo);
                ptvcursor_add(ptv, hf_woww_time_skipped, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_MOVE_WATER_WALK:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementInfo");
            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TransportInfo");
                add_packed_guid(ptv, pinfo);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
            if (flags & MOVEMENT_FLAGS_JUMPING) {
                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
            }
            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_pop_subtree(ptv);
            break;
        case MSG_PETITION_DECLINE:
            ptvcursor_add(ptv, hf_woww_petition, 8, ENC_LITTLE_ENDIAN);
            break;
        case MSG_PETITION_RENAME:
            ptvcursor_add(ptv, hf_woww_petition, 8, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_new_name);
            break;
        case MSG_PVP_LOG_DATA:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add_ret_uint(ptv, hf_woww_battleground_end_status, 1, ENC_LITTLE_ENDIAN, &status);
                if (status == BATTLEGROUND_END_STATUS_ENDED) {
                    ptvcursor_add(ptv, hf_woww_battleground_winner, 1, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_players, 4, ENC_LITTLE_ENDIAN, &amount_of_players);
                for (guint32 i1 = 0; i1 < amount_of_players; ++i1) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "BattlegroundPlayer %i", i1);
                    ptvcursor_add(ptv, hf_woww_player, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_pvp_rank, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_killing_blows, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_honorable_kills, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_deaths, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_bonus_honor, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_extra_fields, 4, ENC_LITTLE_ENDIAN, &amount_of_extra_fields);
                    for (guint32 i2 = 0; i2 < amount_of_extra_fields; ++i2) {
                        ptvcursor_add(ptv, hf_woww_fields, 4, ENC_LITTLE_ENDIAN);
                    }
                    ptvcursor_pop_subtree(ptv);
                }
            }
            else {
            }
            break;
        case MSG_QUERY_NEXT_MAIL_TIME:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_unread_mails, 4, ENC_LITTLE_ENDIAN);
            }
            else {
            }
            break;
        case MSG_QUEST_PUSH_RESULT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_party_message, 1, ENC_LITTLE_ENDIAN);
            break;
        case MSG_RAID_READY_CHECK:
            if (WOWW_SERVER_TO_CLIENT) {
                len = offset_packet_end - ptvcursor_current_offset(ptv);
                if (len > 0) {
                    ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_state, 1, ENC_LITTLE_ENDIAN);
                }
            }
            else {
                len = offset_packet_end - ptvcursor_current_offset(ptv);
                if (len > 0) {
                    ptvcursor_add(ptv, hf_woww_state, 1, ENC_LITTLE_ENDIAN);
                }
            }
            break;
        case MSG_RAID_TARGET_UPDATE:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add_ret_uint(ptv, hf_woww_raid_target_update_type, 1, ENC_LITTLE_ENDIAN, &update_type);
                if (update_type == RAID_TARGET_UPDATE_TYPE_FULL) {
                    for (guint32 i1 = 0; i1 < 8; ++i1) {
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "RaidTargetUpdate %i", i1);
                        ptvcursor_add(ptv, hf_woww_raid_target_index, 1, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                        ptvcursor_pop_subtree(ptv);
                    }
                }
                else if (update_type == RAID_TARGET_UPDATE_TYPE_PARTIAL) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "RaidTargetUpdate");
                    ptvcursor_add(ptv, hf_woww_raid_target_index, 1, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
            }
            else {
                ptvcursor_add_ret_uint(ptv, hf_woww_raid_target_index, 1, ENC_LITTLE_ENDIAN, &target_index);
                if (target_index != RAID_TARGET_INDEX_REQUEST_ICONS) {
                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                }
            }
            break;
        case MSG_RANDOM_ROLL:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_minimum, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_maximum, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_actual_roll, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            }
            else {
                ptvcursor_add(ptv, hf_woww_minimum, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_maximum, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_SAVE_GUILD_EMBLEM:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_guild_emblem_result, 4, ENC_LITTLE_ENDIAN);
            }
            else {
                ptvcursor_add(ptv, hf_woww_vendor, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_emblem_style, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_emblem_color, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_border_style, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_border_color, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_background_color, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case MSG_TABARDVENDOR_ACTIVATE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case MSG_TALENT_WIPE_CONFIRM:
            if (WOWW_SERVER_TO_CLIENT) {
                ptvcursor_add(ptv, hf_woww_wiping_npc, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cost_in_copper, 4, ENC_LITTLE_ENDIAN);
            }
            else {
                ptvcursor_add(ptv, hf_woww_wiping_npc, 8, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_ACCOUNT_DATA_TIMES:
            for (guint32 i1 = 0; i1 < 32; ++i1) {
                ptvcursor_add(ptv, hf_woww_data, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_ACTION_BUTTONS:
            for (guint32 i1 = 0; i1 < 120; ++i1) {
                ptvcursor_add(ptv, hf_woww_data, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_ACTIVATETAXIREPLY:
            ptvcursor_add(ptv, hf_woww_activate_taxi_reply, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_ADDON_INFO:
            while (ptvcursor_current_offset(ptv) < offset_packet_end) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Addon");
                ptvcursor_add(ptv, hf_woww_addon_type, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add_ret_uint(ptv, hf_woww_info_block, 1, ENC_LITTLE_ENDIAN, &info_block);
                if (info_block == INFO_BLOCK_AVAILABLE) {
                    ptvcursor_add_ret_uint(ptv, hf_woww_key_version, 1, ENC_LITTLE_ENDIAN, &key_version);
                    if (key_version != KEY_VERSION_ZERO) {
                        ptvcursor_add(ptv, hf_woww_public_key, 256, ENC_NA);
                    }
                    ptvcursor_add(ptv, hf_woww_update_available_flag, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add_ret_uint(ptv, hf_woww_url_info, 1, ENC_LITTLE_ENDIAN, &url_info);
                if (url_info == URL_INFO_AVAILABLE) {
                    add_cstring(ptv, &hf_woww_url);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_AI_REACTION:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_ai_reaction, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AREA_SPIRIT_HEALER_TIME:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_next_resurrect_time, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AREA_TRIGGER_MESSAGE:
            add_sized_cstring(ptv, &hf_woww_message);
            break;
        case SMSG_ATTACKERSTATEUPDATE:
            ptvcursor_add(ptv, hf_woww_hit_info, 4, ENC_LITTLE_ENDIAN);
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_total_damage, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_damages, 1, ENC_LITTLE_ENDIAN, &amount_of_damages);
            for (guint32 i1 = 0; i1 < amount_of_damages; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "DamageInfo %i", i1);
                ptvcursor_add(ptv, hf_woww_spell_school_mask, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_damage_float, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_damage_uint, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_absorb, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_resist, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_damage_state, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_blocked_amount, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_ATTACKSTART:
            ptvcursor_add(ptv, hf_woww_attacker, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_victim, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_ATTACKSTOP:
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AUCTION_BIDDER_LIST_RESULT:
            ptvcursor_add_ret_uint(ptv, hf_woww_count, 4, ENC_LITTLE_ENDIAN, &count);
            for (guint32 i1 = 0; i1 < count; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "AuctionListItem %i", i1);
                ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_enchantment, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_suffix_factor, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_charges, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_owner, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_start_bid, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_minimum_bid, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_buyout_amount, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_time_left, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_highest_bidder, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_highest_bid, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_total_amount_of_auctions, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AUCTION_BIDDER_NOTIFICATION:
            ptvcursor_add(ptv, hf_woww_auction_house, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auction_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_bidder, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_won, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_out_bid, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_template, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AUCTION_COMMAND_RESULT:
            ptvcursor_add(ptv, hf_woww_auction_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_auction_command_action, 4, ENC_LITTLE_ENDIAN, &action);
            if (action == AUCTION_COMMAND_ACTION_BID_PLACED) {
                ptvcursor_add_ret_uint(ptv, hf_woww_auction_command_result, 4, ENC_LITTLE_ENDIAN, &result);
                if (result == AUCTION_COMMAND_RESULT_OK) {
                    ptvcursor_add(ptv, hf_woww_auction_outbid, 4, ENC_LITTLE_ENDIAN);
                }
                else if (result == AUCTION_COMMAND_RESULT_ERR_INVENTORY) {
                    ptvcursor_add(ptv, hf_woww_inventory_result, 1, ENC_LITTLE_ENDIAN);
                }
                else if (result == AUCTION_COMMAND_RESULT_ERR_HIGHER_BID) {
                    ptvcursor_add(ptv, hf_woww_higher_bidder, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_new_bid, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_auction_outbid, 4, ENC_LITTLE_ENDIAN);
                }
            }
            else {
                ptvcursor_add_ret_uint(ptv, hf_woww_auction_command_result_two, 4, ENC_LITTLE_ENDIAN, &result2);
                if (result2 == AUCTION_COMMAND_RESULT_TWO_ERR_INVENTORY) {
                    ptvcursor_add(ptv, hf_woww_inventory_result, 1, ENC_LITTLE_ENDIAN);
                }
                else if (result2 == AUCTION_COMMAND_RESULT_TWO_ERR_HIGHER_BID) {
                    ptvcursor_add(ptv, hf_woww_higher_bidder, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_new_bid, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_auction_outbid, 4, ENC_LITTLE_ENDIAN);
                }
            }
            break;
        case SMSG_AUCTION_LIST_RESULT:
            ptvcursor_add_ret_uint(ptv, hf_woww_count, 4, ENC_LITTLE_ENDIAN, &count);
            for (guint32 i1 = 0; i1 < count; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "AuctionListItem %i", i1);
                ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_enchantment, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_suffix_factor, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_charges, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_owner, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_start_bid, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_minimum_bid, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_buyout_amount, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_time_left, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_highest_bidder, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_highest_bid, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_total_amount_of_auctions, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AUCTION_OWNER_LIST_RESULT:
            ptvcursor_add_ret_uint(ptv, hf_woww_count, 4, ENC_LITTLE_ENDIAN, &count);
            for (guint32 i1 = 0; i1 < count; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "AuctionListItem %i", i1);
                ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_enchantment, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_suffix_factor, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_charges, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_owner, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_start_bid, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_minimum_bid, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_buyout_amount, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_time_left, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_highest_bidder, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_highest_bid, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_total_amount_of_auctions, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AUCTION_OWNER_NOTIFICATION:
            ptvcursor_add(ptv, hf_woww_auction_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_bid, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auction_out_bid, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_bidder, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AUCTION_REMOVED_NOTIFICATION:
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_template, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_random_property_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AUTH_CHALLENGE:
            ptvcursor_add(ptv, hf_woww_server_seed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_AUTH_RESPONSE:
            ptvcursor_add_ret_uint(ptv, hf_woww_world_result, 1, ENC_LITTLE_ENDIAN, &result);
            if (result == WORLD_RESULT_AUTH_OK) {
                ptvcursor_add(ptv, hf_woww_billing_time, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_billing_flags, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_billing_rested, 4, ENC_LITTLE_ENDIAN);
            }
            else if (result == WORLD_RESULT_AUTH_WAIT_QUEUE) {
                ptvcursor_add(ptv, hf_woww_queue_position, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_BATTLEFIELD_LIST:
            ptvcursor_add(ptv, hf_woww_battlemaster, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_battleground_bracket, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_number_of_battlegrounds, 4, ENC_LITTLE_ENDIAN, &number_of_battlegrounds);
            for (guint32 i1 = 0; i1 < number_of_battlegrounds; ++i1) {
                ptvcursor_add(ptv, hf_woww_battlegrounds, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_BATTLEFIELD_STATUS:
            ptvcursor_add(ptv, hf_woww_queue_slot, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN, &map);
            if (map != MAP_EASTERN_KINGDOMS) {
                ptvcursor_add(ptv, hf_woww_battleground_bracket, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_client_instance_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_ret_uint(ptv, hf_woww_status_id, 1, ENC_LITTLE_ENDIAN, &status_id);
                if (status_id == STATUS_ID_WAIT_QUEUE) {
                    ptvcursor_add(ptv, hf_woww_average_wait_time_in_ms, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_time_in_queue_in_ms, 4, ENC_LITTLE_ENDIAN);
                }
                else if (status_id == STATUS_ID_WAIT_JOIN) {
                    ptvcursor_add(ptv, hf_woww_time_to_remove_in_queue_in_ms, 4, ENC_LITTLE_ENDIAN);
                }
                else if (status_id == STATUS_ID_IN_PROGRESS) {
                    ptvcursor_add(ptv, hf_woww_time_to_bg_autoleave_in_ms, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_time_to_bg_start_in_ms, 4, ENC_LITTLE_ENDIAN);
                }
            }
            break;
        case SMSG_BATTLEGROUND_PLAYER_JOINED:
            ptvcursor_add(ptv, hf_woww_player, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_BATTLEGROUND_PLAYER_LEFT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_BINDER_CONFIRM:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_BINDPOINTUPDATE:
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_BUY_BANK_SLOT_RESULT:
            ptvcursor_add(ptv, hf_woww_buy_bank_slot_result, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_BUY_FAILED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_buy_result, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_BUY_ITEM:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_vendor_slot, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_amount_for_sale, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_amount_bought, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_CAST_RESULT:
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_simple_spell_cast_result, 1, ENC_LITTLE_ENDIAN, &result);
            if (result != SIMPLE_SPELL_CAST_RESULT_FAILURE) {
                ptvcursor_add_ret_uint(ptv, hf_woww_cast_failure_reason, 1, ENC_LITTLE_ENDIAN, &reason);
                if (reason == CAST_FAILURE_REASON_REQUIRES_SPELL_FOCUS) {
                    ptvcursor_add(ptv, hf_woww_required_spell_focus, 4, ENC_LITTLE_ENDIAN);
                }
                else if (reason == CAST_FAILURE_REASON_REQUIRES_AREA) {
                    ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
                }
                else if (reason == CAST_FAILURE_REASON_EQUIPPED_ITEM_CLASS) {
                    ptvcursor_add(ptv, hf_woww_equipped_item_class, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_equipped_item_subclass_mask, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_equipped_item_inventory_type_mask, 4, ENC_LITTLE_ENDIAN);
                }
            }
            break;
        case SMSG_CHANNEL_LIST:
            add_cstring(ptv, &hf_woww_channel_name);
            ptvcursor_add(ptv, hf_woww_channel_flags, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_members, 4, ENC_LITTLE_ENDIAN, &amount_of_members);
            for (guint32 i1 = 0; i1 < amount_of_members; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "ChannelMember %i", i1);
                ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_channel_member_flags, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_CHANNEL_NOTIFY:
            ptvcursor_add(ptv, hf_woww_chat_notify, 1, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_channel_name);
            break;
        case SMSG_CHARACTER_LOGIN_FAILED:
            ptvcursor_add(ptv, hf_woww_world_result, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_CHAR_CREATE:
            ptvcursor_add(ptv, hf_woww_world_result, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_CHAR_DELETE:
            ptvcursor_add(ptv, hf_woww_world_result, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_CHAR_ENUM:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_characters, 1, ENC_LITTLE_ENDIAN, &amount_of_characters);
            for (guint32 i1 = 0; i1 < amount_of_characters; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Character %i", i1);
                ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                add_cstring(ptv, &hf_woww_name);
                ptvcursor_add(ptv, hf_woww_race, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_class, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_gender, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_skin, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_face, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_hair_style, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_hair_color, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_facial_hair, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_level, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
                ptvcursor_add(ptv, hf_woww_guild_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_character_flags, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_first_login, 1, ENC_NA);
                ptvcursor_add(ptv, hf_woww_pet_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_pet_level, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_creature_family, 4, ENC_LITTLE_ENDIAN);
                for (guint32 i2 = 0; i2 < 19; ++i2) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "CharacterGear %i", i2);
                    ptvcursor_add(ptv, hf_woww_equipment_display_id, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_inventory_type, 1, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                ptvcursor_add(ptv, hf_woww_first_bag_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_first_bag_inventory_id, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_CHAR_RENAME:
            ptvcursor_add_ret_uint(ptv, hf_woww_world_result, 1, ENC_LITTLE_ENDIAN, &result);
            if (result == WORLD_RESULT_RESPONSE_SUCCESS) {
                ptvcursor_add(ptv, hf_woww_character_guid, 8, ENC_LITTLE_ENDIAN);
                add_cstring(ptv, &hf_woww_new_name);
            }
            break;
        case SMSG_CHAT_PLAYER_NOT_FOUND:
            add_cstring(ptv, &hf_woww_name);
            break;
        case SMSG_CLEAR_COOLDOWN:
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_CLIENT_CONTROL_UPDATE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_allow_movement, 1, ENC_NA);
            break;
        case SMSG_COMPRESSED_MOVES:
            ptvcursor_add(ptv, hf_woww_decompressed_size, 4, ENC_LITTLE_ENDIAN);
            compressed_tvb = tvb_uncompress_zlib(ptvcursor_tvbuff(ptv), ptvcursor_current_offset(ptv), offset_packet_end - ptvcursor_current_offset(ptv));
            if (compressed_tvb != NULL) {
                ptvcursor_t* old_ptv = ptv;
                ptv = ptvcursor_new(wmem_packet_scope(), tree, compressed_tvb, 0);
                gint compression_end = tvb_reported_length(compressed_tvb);
                while (ptvcursor_current_offset(ptv) < compression_end) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "CompressedMove");
                    ptvcursor_add(ptv, hf_woww_size_struct, 1, ENC_LITTLE_ENDIAN);
                    ptvcursor_add_ret_uint(ptv, hf_woww_compressed_move_opcode, 2, ENC_LITTLE_ENDIAN, &opcode);
                    add_packed_guid(ptv, pinfo);
                    if (opcode == COMPRESSED_MOVE_OPCODE_SMSG_SPLINE_SET_RUN_SPEED) {
                        ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
                    }
                    else if (opcode == COMPRESSED_MOVE_OPCODE_SMSG_MONSTER_MOVE) {
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MonsterMove");
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                        ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_pop_subtree(ptv);
                        ptvcursor_add(ptv, hf_woww_spline_id, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add_ret_uint(ptv, hf_woww_monster_move_type, 1, ENC_LITTLE_ENDIAN, &move_type);
                        if (move_type == MONSTER_MOVE_TYPE_FACING_TARGET) {
                            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                        }
                        else if (move_type == MONSTER_MOVE_TYPE_FACING_ANGLE) {
                            ptvcursor_add(ptv, hf_woww_angle, 4, ENC_LITTLE_ENDIAN);
                        }
                        else if (move_type == MONSTER_MOVE_TYPE_FACING_SPOT) {
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                        }
                        if (move_type != MONSTER_MOVE_TYPE_STOP) {
                            ptvcursor_add(ptv, hf_woww_spline_flag, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
                            add_monster_move_spline(ptv);
                        }
                        ptvcursor_pop_subtree(ptv);
                    }
                    else if (opcode == COMPRESSED_MOVE_OPCODE_SMSG_MONSTER_MOVE_TRANSPORT) {
                        add_packed_guid(ptv, pinfo);
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MonsterMove");
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                        ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_pop_subtree(ptv);
                        ptvcursor_add(ptv, hf_woww_spline_id, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add_ret_uint(ptv, hf_woww_monster_move_type, 1, ENC_LITTLE_ENDIAN, &move_type);
                        if (move_type == MONSTER_MOVE_TYPE_FACING_TARGET) {
                            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                        }
                        else if (move_type == MONSTER_MOVE_TYPE_FACING_ANGLE) {
                            ptvcursor_add(ptv, hf_woww_angle, 4, ENC_LITTLE_ENDIAN);
                        }
                        else if (move_type == MONSTER_MOVE_TYPE_FACING_SPOT) {
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                        }
                        if (move_type != MONSTER_MOVE_TYPE_STOP) {
                            ptvcursor_add(ptv, hf_woww_spline_flag, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
                            add_monster_move_spline(ptv);
                        }
                        ptvcursor_pop_subtree(ptv);
                    }
                    ptvcursor_pop_subtree(ptv);
                }
                ptvcursor_free(ptv);
                ptv = old_ptv;
                compressed_tvb = NULL;
            }
            break;
        case SMSG_COMPRESSED_UPDATE_OBJECT:
            ptvcursor_add(ptv, hf_woww_decompressed_size, 4, ENC_LITTLE_ENDIAN);
            compressed_tvb = tvb_uncompress_zlib(ptvcursor_tvbuff(ptv), ptvcursor_current_offset(ptv), offset_packet_end - ptvcursor_current_offset(ptv));
            if (compressed_tvb != NULL) {
                ptvcursor_t* old_ptv = ptv;
                ptv = ptvcursor_new(wmem_packet_scope(), tree, compressed_tvb, 0);
                ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_objects, 4, ENC_LITTLE_ENDIAN, &amount_of_objects);
                ptvcursor_add(ptv, hf_woww_has_transport, 1, ENC_LITTLE_ENDIAN);
                for (guint32 i1 = 0; i1 < amount_of_objects; ++i1) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Object %i", i1);
                    ptvcursor_add_ret_uint(ptv, hf_woww_update_type, 1, ENC_LITTLE_ENDIAN, &update_type);
                    if (update_type == UPDATE_TYPE_VALUES) {
                        add_packed_guid(ptv, pinfo);
                        add_update_mask(ptv, pinfo);
                    }
                    else if (update_type == UPDATE_TYPE_MOVEMENT) {
                        add_packed_guid(ptv, pinfo);
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementBlock");
                        ptvcursor_add_ret_uint(ptv, hf_woww_update_flag, 1, ENC_LITTLE_ENDIAN, &update_flag);
                        if (update_flag & UPDATE_FLAG_LIVING) {
                            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                            ptvcursor_add(ptv, hf_woww_living_orientation, 4, ENC_LITTLE_ENDIAN);
                            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                                add_packed_guid(ptv, pinfo);
                                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_pop_subtree(ptv);
                                ptvcursor_add(ptv, hf_woww_transport_orientation, 4, ENC_LITTLE_ENDIAN);
                            }
                            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                            }
                            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                            if (flags & MOVEMENT_FLAGS_JUMPING) {
                                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                            }
                            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                            }
                            ptvcursor_add(ptv, hf_woww_walking_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_running_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_backwards_running_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_swimming_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_backwards_swimming_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_turn_rate, 4, ENC_LITTLE_ENDIAN);
                            if (flags & MOVEMENT_FLAGS_SPLINE_ENABLED) {
                                ptvcursor_add_ret_uint(ptv, hf_woww_spline_flag, 4, ENC_LITTLE_ENDIAN, &spline_flags);
                                if (spline_flags & SPLINE_FLAG_FINAL_ANGLE) {
                                    ptvcursor_add(ptv, hf_woww_angle, 4, ENC_LITTLE_ENDIAN);
                                }
                                else if (spline_flags & SPLINE_FLAG_FINAL_TARGET) {
                                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                                }
                                else if (spline_flags & SPLINE_FLAG_FINAL_POINT) {
                                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_pop_subtree(ptv);
                                }
                                ptvcursor_add(ptv, hf_woww_time_passed, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_nodes, 4, ENC_LITTLE_ENDIAN, &amount_of_nodes);
                                for (guint32 i2 = 0; i2 < amount_of_nodes; ++i2) {
                                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d %i", i2);
                                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_pop_subtree(ptv);
                                }
                                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_pop_subtree(ptv);
                            }
                        }
                        else if (update_flag & UPDATE_FLAG_HAS_POSITION) {
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (update_flag & UPDATE_FLAG_HIGH_GUID) {
                            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (update_flag & UPDATE_FLAG_ALL) {
                            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (update_flag & UPDATE_FLAG_MELEE_ATTACKING) {
                            add_packed_guid(ptv, pinfo);
                        }
                        if (update_flag & UPDATE_FLAG_TRANSPORT) {
                            ptvcursor_add(ptv, hf_woww_transport_progress_in_ms, 4, ENC_LITTLE_ENDIAN);
                        }
                        ptvcursor_pop_subtree(ptv);
                    }
                    else if (update_type == UPDATE_TYPE_CREATE_OBJECT
                     || update_type == UPDATE_TYPE_CREATE_OBJECT2) {
                        add_packed_guid(ptv, pinfo);
                        ptvcursor_add(ptv, hf_woww_object_type, 1, ENC_LITTLE_ENDIAN);
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementBlock");
                        ptvcursor_add_ret_uint(ptv, hf_woww_update_flag, 1, ENC_LITTLE_ENDIAN, &update_flag);
                        if (update_flag & UPDATE_FLAG_LIVING) {
                            ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                            ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                            ptvcursor_add(ptv, hf_woww_living_orientation, 4, ENC_LITTLE_ENDIAN);
                            if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                                add_packed_guid(ptv, pinfo);
                                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_pop_subtree(ptv);
                                ptvcursor_add(ptv, hf_woww_transport_orientation, 4, ENC_LITTLE_ENDIAN);
                            }
                            if (flags & MOVEMENT_FLAGS_SWIMMING) {
                                ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                            }
                            ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                            if (flags & MOVEMENT_FLAGS_JUMPING) {
                                ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                            }
                            if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                                ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                            }
                            ptvcursor_add(ptv, hf_woww_walking_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_running_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_backwards_running_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_swimming_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_backwards_swimming_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_turn_rate, 4, ENC_LITTLE_ENDIAN);
                            if (flags & MOVEMENT_FLAGS_SPLINE_ENABLED) {
                                ptvcursor_add_ret_uint(ptv, hf_woww_spline_flag, 4, ENC_LITTLE_ENDIAN, &spline_flags);
                                if (spline_flags & SPLINE_FLAG_FINAL_ANGLE) {
                                    ptvcursor_add(ptv, hf_woww_angle, 4, ENC_LITTLE_ENDIAN);
                                }
                                else if (spline_flags & SPLINE_FLAG_FINAL_TARGET) {
                                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                                }
                                else if (spline_flags & SPLINE_FLAG_FINAL_POINT) {
                                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_pop_subtree(ptv);
                                }
                                ptvcursor_add(ptv, hf_woww_time_passed, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_nodes, 4, ENC_LITTLE_ENDIAN, &amount_of_nodes);
                                for (guint32 i2 = 0; i2 < amount_of_nodes; ++i2) {
                                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d %i", i2);
                                    ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                    ptvcursor_pop_subtree(ptv);
                                }
                                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_pop_subtree(ptv);
                            }
                        }
                        else if (update_flag & UPDATE_FLAG_HAS_POSITION) {
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (update_flag & UPDATE_FLAG_HIGH_GUID) {
                            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (update_flag & UPDATE_FLAG_ALL) {
                            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (update_flag & UPDATE_FLAG_MELEE_ATTACKING) {
                            add_packed_guid(ptv, pinfo);
                        }
                        if (update_flag & UPDATE_FLAG_TRANSPORT) {
                            ptvcursor_add(ptv, hf_woww_transport_progress_in_ms, 4, ENC_LITTLE_ENDIAN);
                        }
                        ptvcursor_pop_subtree(ptv);
                        add_update_mask(ptv, pinfo);
                    }
                    else if (update_type == UPDATE_TYPE_OUT_OF_RANGE_OBJECTS
                     || update_type == UPDATE_TYPE_NEAR_OBJECTS) {
                        ptvcursor_add_ret_uint(ptv, hf_woww_count, 4, ENC_LITTLE_ENDIAN, &count);
                        for (guint32 i2 = 0; i2 < count; ++i2) {
                            add_packed_guid(ptv, pinfo);
                        }
                    }
                    ptvcursor_pop_subtree(ptv);
                }
                ptvcursor_free(ptv);
                ptv = old_ptv;
                compressed_tvb = NULL;
            }
            break;
        case SMSG_COOLDOWN_EVENT:
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_CORPSE_RECLAIM_DELAY:
            ptvcursor_add(ptv, hf_woww_delay, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_CREATURE_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_creature_entry, 4, ENC_LITTLE_ENDIAN);
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            if (len > 0) {
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_sub_name);
                ptvcursor_add(ptv, hf_woww_type_flags, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_creature_type, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_creature_family, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_creature_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_spell_data_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_civilian, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_racial_leader, 1, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_DEFENSE_MESSAGE:
            ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            add_sized_cstring(ptv, &hf_woww_message);
            break;
        case SMSG_DESTROY_OBJECT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_DISMOUNTRESULT:
            ptvcursor_add(ptv, hf_woww_dismount_result, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_DISPEL_FAILED:
            ptvcursor_add(ptv, hf_woww_caster, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            while (ptvcursor_current_offset(ptv) < offset_packet_end) {
                ptvcursor_add(ptv, hf_woww_spells, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_DUEL_COMPLETE:
            ptvcursor_add(ptv, hf_woww_ended_without_interruption, 1, ENC_NA);
            break;
        case SMSG_DUEL_COUNTDOWN:
            ptvcursor_add(ptv, hf_woww_time, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_DUEL_REQUESTED:
            ptvcursor_add(ptv, hf_woww_initiator, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_DUEL_WINNER:
            ptvcursor_add(ptv, hf_woww_duel_winner_reason, 1, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_opponent_name);
            add_cstring(ptv, &hf_woww_initiator_name);
            break;
        case SMSG_EMOTE:
            ptvcursor_add(ptv, hf_woww_emote_enum, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_ENCHANTMENTLOG:
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_caster, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_show_affiliation, 1, ENC_NA);
            break;
        case SMSG_ENVIRONMENTAL_DAMAGE_LOG:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_environmental_damage_type, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_damage, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_absorb, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_resist, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_EXPECTED_SPAM_RECORDS:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_records, 4, ENC_LITTLE_ENDIAN, &amount_of_records);
            for (guint32 i1 = 0; i1 < amount_of_records; ++i1) {
                add_cstring(ptv, &hf_woww_records);
            }
            break;
        case SMSG_EXPLORATION_EXPERIENCE:
            ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_experience, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_FORCE_MOVE_ROOT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_FORCE_MOVE_UNROOT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_FORCE_RUN_BACK_SPEED_CHANGE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_move_event, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_FORCE_RUN_SPEED_CHANGE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_move_event, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_FORCE_SWIM_BACK_SPEED_CHANGE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_move_event, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_FORCE_SWIM_SPEED_CHANGE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_move_event, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_FORCE_TURN_RATE_CHANGE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_move_event, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_FORCE_WALK_SPEED_CHANGE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_move_event, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_FRIEND_LIST:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_friends, 1, ENC_LITTLE_ENDIAN, &amount_of_friends);
            for (guint32 i1 = 0; i1 < amount_of_friends; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Friend %i", i1);
                ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add_ret_uint(ptv, hf_woww_friend_status, 1, ENC_LITTLE_ENDIAN, &status);
                if (status != FRIEND_STATUS_OFFLINE) {
                    ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_level, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_class, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_FRIEND_STATUS:
            ptvcursor_add(ptv, hf_woww_friend_result, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GAMEOBJECT_CUSTOM_ANIM:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_animation_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GAMEOBJECT_DESPAWN_ANIM:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GAMEOBJECT_PAGETEXT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GAMEOBJECT_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_entry_id, 4, ENC_LITTLE_ENDIAN);
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            if (len > 0) {
                ptvcursor_add(ptv, hf_woww_info_type, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_display_id, 4, ENC_LITTLE_ENDIAN);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                for (guint32 i1 = 0; i1 < 6; ++i1) {
                    ptvcursor_add(ptv, hf_woww_raw_data, 4, ENC_LITTLE_ENDIAN);
                }
            }
            break;
        case SMSG_GAMEOBJECT_RESET_STATE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GAMEOBJECT_SPAWN_ANIM:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GMTICKET_CREATE:
            ptvcursor_add(ptv, hf_woww_gm_ticket_response, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GMTICKET_DELETETICKET:
            ptvcursor_add(ptv, hf_woww_gm_ticket_response, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GMTICKET_GETTICKET:
            ptvcursor_add_ret_uint(ptv, hf_woww_gm_ticket_status, 4, ENC_LITTLE_ENDIAN, &status);
            if (status == GM_TICKET_STATUS_HAS_TEXT) {
                add_cstring(ptv, &hf_woww_text);
                ptvcursor_add(ptv, hf_woww_gm_ticket_type, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_days_since_ticket_creation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_days_since_oldest_ticket_creation, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_days_since_last_updated, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_gm_ticket_escalation_status, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_read_by_gm, 1, ENC_NA);
            }
            break;
        case SMSG_GMTICKET_SYSTEMSTATUS:
            ptvcursor_add(ptv, hf_woww_gm_ticket_queue_status, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GMTICKET_UPDATETEXT:
            ptvcursor_add(ptv, hf_woww_gm_ticket_response, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GM_TICKET_STATUS_UPDATE:
            ptvcursor_add(ptv, hf_woww_gm_ticket_status_response, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GOSSIP_MESSAGE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_title_text_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_gossip_items, 4, ENC_LITTLE_ENDIAN, &amount_of_gossip_items);
            for (guint32 i1 = 0; i1 < amount_of_gossip_items; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "GossipItem %i", i1);
                ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_icon, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_coded, 1, ENC_NA);
                add_cstring(ptv, &hf_woww_message);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_quests, 4, ENC_LITTLE_ENDIAN, &amount_of_quests);
            for (guint32 i1 = 0; i1 < amount_of_quests; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItem %i", i1);
                ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_quest_icon, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_level, 4, ENC_LITTLE_ENDIAN);
                add_cstring(ptv, &hf_woww_title);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_GOSSIP_POI:
            ptvcursor_add(ptv, hf_woww_flags, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector2d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_icon, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_data, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_location_name);
            break;
        case SMSG_GROUP_DECLINE:
            add_cstring(ptv, &hf_woww_name);
            break;
        case SMSG_GROUP_INVITE:
            add_cstring(ptv, &hf_woww_name);
            break;
        case SMSG_GROUP_JOINED_BATTLEGROUND:
            ptvcursor_add(ptv, hf_woww_bg_type_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GROUP_LIST:
            ptvcursor_add(ptv, hf_woww_group_type, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_flags, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_members, 4, ENC_LITTLE_ENDIAN, &amount_of_members);
            for (guint32 i1 = 0; i1 < amount_of_members; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "GroupListMember %i", i1);
                add_cstring(ptv, &hf_woww_name);
                ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_is_online, 1, ENC_NA);
                ptvcursor_add(ptv, hf_woww_flags, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_leader, 8, ENC_LITTLE_ENDIAN);
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            if (len > 0) {
                ptvcursor_add(ptv, hf_woww_group_loot_setting, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_master_loot, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_quality, 1, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_GROUP_SET_LEADER:
            add_cstring(ptv, &hf_woww_name);
            break;
        case SMSG_GUILD_COMMAND_RESULT:
            ptvcursor_add(ptv, hf_woww_guild_command, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_string);
            ptvcursor_add(ptv, hf_woww_guild_command_result, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GUILD_EVENT:
            ptvcursor_add(ptv, hf_woww_guild_event, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_events, 1, ENC_LITTLE_ENDIAN, &amount_of_events);
            for (guint32 i1 = 0; i1 < amount_of_events; ++i1) {
                add_cstring(ptv, &hf_woww_event_descriptions);
            }
            break;
        case SMSG_GUILD_INFO:
            add_cstring(ptv, &hf_woww_guild_name);
            ptvcursor_add(ptv, hf_woww_created_day, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_created_month, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_created_year, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_amount_of_characters_in_guild, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_amount_of_accounts_in_guild, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GUILD_INVITE:
            add_cstring(ptv, &hf_woww_player_name);
            add_cstring(ptv, &hf_woww_guild_name);
            break;
        case SMSG_GUILD_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_name);
            for (guint32 i1 = 0; i1 < 10; ++i1) {
                add_cstring(ptv, &hf_woww_rank_names);
            }
            ptvcursor_add(ptv, hf_woww_emblem_style, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_emblem_color, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_border_style, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_border_color, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_background_color, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_GUILD_ROSTER:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_members, 4, ENC_LITTLE_ENDIAN, &amount_of_members);
            add_cstring(ptv, &hf_woww_motd);
            add_cstring(ptv, &hf_woww_guild_info);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_rights, 4, ENC_LITTLE_ENDIAN, &amount_of_rights);
            for (guint32 i1 = 0; i1 < amount_of_rights; ++i1) {
                ptvcursor_add(ptv, hf_woww_rights, 4, ENC_LITTLE_ENDIAN);
            }
            for (guint32 i1 = 0; i1 < amount_of_members; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "GuildMember %i", i1);
                ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add_ret_uint(ptv, hf_woww_guild_member_status, 1, ENC_LITTLE_ENDIAN, &status);
                add_cstring(ptv, &hf_woww_name);
                ptvcursor_add(ptv, hf_woww_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_level, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_class, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
                if (status == GUILD_MEMBER_STATUS_OFFLINE) {
                    ptvcursor_add(ptv, hf_woww_time_offline, 4, ENC_LITTLE_ENDIAN);
                }
                add_cstring(ptv, &hf_woww_public_note);
                add_cstring(ptv, &hf_woww_officer_note);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_IGNORE_LIST:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_ignored, 1, ENC_LITTLE_ENDIAN, &amount_of_ignored);
            for (guint32 i1 = 0; i1 < amount_of_ignored; ++i1) {
                ptvcursor_add(ptv, hf_woww_ignored, 8, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_INITIALIZE_FACTIONS:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_factions, 4, ENC_LITTLE_ENDIAN, &amount_of_factions);
            for (guint32 i1 = 0; i1 < amount_of_factions; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "FactionInitializer %i", i1);
                ptvcursor_add(ptv, hf_woww_faction_flag, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_standing, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_INITIAL_SPELLS:
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_spell_count, 2, ENC_LITTLE_ENDIAN, &spell_count);
            for (guint32 i1 = 0; i1 < spell_count; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "InitialSpell %i", i1);
                ptvcursor_add(ptv, hf_woww_spell_id, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_unknown_int, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add_ret_uint(ptv, hf_woww_cooldown_count, 2, ENC_LITTLE_ENDIAN, &cooldown_count);
            for (guint32 i1 = 0; i1 < cooldown_count; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "CooldownSpell %i", i1);
                ptvcursor_add(ptv, hf_woww_spell_id, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_id, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_spell_category, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cooldown, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_category_cooldown, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_INIT_WORLD_STATES:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_states, 2, ENC_LITTLE_ENDIAN, &amount_of_states);
            for (guint32 i1 = 0; i1 < amount_of_states; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "WorldState %i", i1);
                ptvcursor_add(ptv, hf_woww_state, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_value, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_INSPECT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_INSTANCE_RESET:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_INSTANCE_RESET_FAILED:
            ptvcursor_add(ptv, hf_woww_instance_reset_failed_reason, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_INSTANCE_SAVE_CREATED:
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_INVALIDATE_PLAYER:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_INVENTORY_CHANGE_FAILURE:
            ptvcursor_add_ret_uint(ptv, hf_woww_inventory_result, 1, ENC_LITTLE_ENDIAN, &result);
            if (result == INVENTORY_RESULT_CANT_EQUIP_LEVEL_I) {
                ptvcursor_add(ptv, hf_woww_required_level, 4, ENC_LITTLE_ENDIAN);
            }
            if (result != INVENTORY_RESULT_OK) {
                ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_bag_type_subclass, 1, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_ITEM_COOLDOWN:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_ITEM_ENCHANT_TIME_UPDATE:
            ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_slot, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_player, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_ITEM_NAME_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_item_name);
            break;
        case SMSG_ITEM_PUSH_RESULT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_new_item_source, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_new_item_creation_type, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_new_item_chat_alert, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_bag_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_slot_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_suffix_factor, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_ITEM_QUERY_SINGLE_RESPONSE:
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            if (len > 0) {
                ptvcursor_add(ptv, hf_woww_item_class_and_sub_class, 8, ENC_LITTLE_ENDIAN);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_name);
                ptvcursor_add(ptv, hf_woww_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_quality, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_flag, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_buy_price, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sell_price, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_inventory_type, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_allowed_class, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_allowed_race, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_level, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_level, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_skill, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_skill_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_spell, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_honor_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_city_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_faction, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_faction_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_max_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_stackable, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_container_slots, 4, ENC_LITTLE_ENDIAN);
                for (guint32 i1 = 0; i1 < 10; ++i1) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "ItemStat %i", i1);
                    ptvcursor_add(ptv, hf_woww_item_stat_type, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_value, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                for (guint32 i1 = 0; i1 < 5; ++i1) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "ItemDamageType %i", i1);
                    ptvcursor_add(ptv, hf_woww_damage_minimum, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_damage_maximum, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_spell_school, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                ptvcursor_add(ptv, hf_woww_armor, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_holy_resistance, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_fire_resistance, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_nature_resistance, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_frost_resistance, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_shadow_resistance, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_arcane_resistance, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_delay, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_ammo_type, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_ranged_range_modification, 4, ENC_LITTLE_ENDIAN);
                for (guint32 i1 = 0; i1 < 5; ++i1) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "ItemSpells %i", i1);
                    ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_spell_trigger_type, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_spell_charges, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_spell_cooldown, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_spell_category, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_spell_category_cooldown, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                ptvcursor_add(ptv, hf_woww_bonding, 4, ENC_LITTLE_ENDIAN);
                add_cstring(ptv, &hf_woww_description);
                ptvcursor_add(ptv, hf_woww_page_text, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_language, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_page_text_material, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_start_quest, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_lock_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_material, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_sheathe_type, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_random_property, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_block, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_set, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_max_durability, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_bag_family, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_ITEM_TEXT_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_item_text_id, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_text);
            break;
        case SMSG_ITEM_TIME_UPDATE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LEARNED_SPELL:
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LEVELUP_INFO:
            ptvcursor_add(ptv, hf_woww_new_level, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_health, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_mana, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_rage, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_focus, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_energy, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_happiness, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_strength, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_agility, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_stamina, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_intellect, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spirit, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LIST_INVENTORY:
            ptvcursor_add(ptv, hf_woww_vendor, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_items, 1, ENC_LITTLE_ENDIAN, &amount_of_items);
            for (guint32 i1 = 0; i1 < amount_of_items; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "ListInventoryItem %i", i1);
                ptvcursor_add(ptv, hf_woww_item_stack_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_max_items, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_price, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_max_durability, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_durability, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_LOGIN_SETTIMESPEED:
            ptvcursor_add(ptv, hf_woww_datetime, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_timescale, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LOGIN_VERIFY_WORLD:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LOGOUT_RESPONSE:
            ptvcursor_add(ptv, hf_woww_logout_result, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_logout_speed, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LOG_XPGAIN:
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_total_exp, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_experience_award_type, 1, ENC_LITTLE_ENDIAN, &exp_type);
            if (exp_type == EXPERIENCE_AWARD_TYPE_NON_KILL) {
                ptvcursor_add(ptv, hf_woww_experience_without_rested, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_exp_group_bonus, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_LOOT_ALL_PASSED:
            ptvcursor_add(ptv, hf_woww_looted_target, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_loot_slot, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_suffix_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LOOT_MASTER_LIST:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_players, 1, ENC_LITTLE_ENDIAN, &amount_of_players);
            for (guint32 i1 = 0; i1 < amount_of_players; ++i1) {
                ptvcursor_add(ptv, hf_woww_guids, 8, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_LOOT_MONEY_NOTIFY:
            ptvcursor_add(ptv, hf_woww_amount, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LOOT_RELEASE_RESPONSE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LOOT_REMOVED:
            ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LOOT_RESPONSE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_loot_method, 1, ENC_LITTLE_ENDIAN, &loot_method);
            if (loot_method == LOOT_METHOD_ERROR) {
                ptvcursor_add(ptv, hf_woww_loot_method_error, 1, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add(ptv, hf_woww_gold, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_items, 1, ENC_LITTLE_ENDIAN, &amount_of_items);
            for (guint32 i1 = 0; i1 < amount_of_items; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "LootItem %i", i1);
                ptvcursor_add(ptv, hf_woww_index, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_loot_slot_type, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_LOOT_ROLL:
            ptvcursor_add(ptv, hf_woww_creature, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_loot_slot, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_player, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_suffix, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_roll_number, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_roll_vote, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LOOT_ROLL_WON:
            ptvcursor_add(ptv, hf_woww_looted_target, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_loot_slot, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_suffix, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_winning_player, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_winning_roll, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_roll_vote, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_LOOT_START_ROLL:
            ptvcursor_add(ptv, hf_woww_creature, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_loot_slot, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_suffix, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_countdown_time, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MAIL_LIST_RESULT:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_mails, 1, ENC_LITTLE_ENDIAN, &amount_of_mails);
            for (guint32 i1 = 0; i1 < amount_of_mails; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Mail %i", i1);
                ptvcursor_add(ptv, hf_woww_message_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add_ret_uint(ptv, hf_woww_mail_type, 1, ENC_LITTLE_ENDIAN, &message_type);
                if (message_type == MAIL_TYPE_NORMAL) {
                    ptvcursor_add(ptv, hf_woww_sender, 8, ENC_LITTLE_ENDIAN);
                }
                else if (message_type == MAIL_TYPE_CREATURE
                 || message_type == MAIL_TYPE_GAMEOBJECT) {
                    ptvcursor_add(ptv, hf_woww_sender_id, 4, ENC_LITTLE_ENDIAN);
                }
                else if (message_type == MAIL_TYPE_AUCTION) {
                    ptvcursor_add(ptv, hf_woww_auction_id, 4, ENC_LITTLE_ENDIAN);
                }
                add_cstring(ptv, &hf_woww_subject);
                ptvcursor_add(ptv, hf_woww_item_text_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_stationery, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_enchant_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_random_property_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_suffix_factor, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_stack_size, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_spell_charges, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_max_durability, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_durability, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_money, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cash_on_delivery_amount, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_checked_timestamp, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_expiration_time, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_mail_template_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_MEETINGSTONE_JOINFAILED:
            ptvcursor_add(ptv, hf_woww_meeting_stone_failure, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MEETINGSTONE_MEMBER_ADDED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MEETINGSTONE_SETQUEUE:
            ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_meeting_stone_status, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MESSAGECHAT:
            ptvcursor_add_ret_uint(ptv, hf_woww_chat_type, 1, ENC_LITTLE_ENDIAN, &chat_type);
            ptvcursor_add(ptv, hf_woww_language, 4, ENC_LITTLE_ENDIAN);
            if (chat_type == CHAT_TYPE_MONSTER_WHISPER
             || chat_type == CHAT_TYPE_RAID_BOSS_EMOTE
             || chat_type == CHAT_TYPE_MONSTER_EMOTE) {
                add_sized_cstring(ptv, &hf_woww_monster_name);
                ptvcursor_add(ptv, hf_woww_monster, 8, ENC_LITTLE_ENDIAN);
            }
            else if (chat_type == CHAT_TYPE_SAY
             || chat_type == CHAT_TYPE_PARTY
             || chat_type == CHAT_TYPE_YELL) {
                ptvcursor_add(ptv, hf_woww_speech_bubble_credit, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_chat_credit, 8, ENC_LITTLE_ENDIAN);
            }
            else if (chat_type == CHAT_TYPE_MONSTER_SAY
             || chat_type == CHAT_TYPE_MONSTER_YELL) {
                ptvcursor_add(ptv, hf_woww_sender, 8, ENC_LITTLE_ENDIAN);
                add_sized_cstring(ptv, &hf_woww_sender_name);
                ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            }
            else if (chat_type == CHAT_TYPE_CHANNEL) {
                add_cstring(ptv, &hf_woww_channel_name);
                ptvcursor_add(ptv, hf_woww_player_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_player, 8, ENC_LITTLE_ENDIAN);
            }
            else {
                ptvcursor_add(ptv, hf_woww_sender, 8, ENC_LITTLE_ENDIAN);
            }
            add_sized_cstring(ptv, &hf_woww_message);
            ptvcursor_add(ptv, hf_woww_player_chat_tag, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MONSTER_MOVE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_spline_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_monster_move_type, 1, ENC_LITTLE_ENDIAN, &move_type);
            if (move_type == MONSTER_MOVE_TYPE_FACING_TARGET) {
                ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            }
            else if (move_type == MONSTER_MOVE_TYPE_FACING_ANGLE) {
                ptvcursor_add(ptv, hf_woww_angle, 4, ENC_LITTLE_ENDIAN);
            }
            else if (move_type == MONSTER_MOVE_TYPE_FACING_SPOT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_spline_flag, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
            add_monster_move_spline(ptv);
            break;
        case SMSG_MONSTER_MOVE_TRANSPORT:
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_spline_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_monster_move_type, 1, ENC_LITTLE_ENDIAN, &move_type);
            if (move_type == MONSTER_MOVE_TYPE_FACING_TARGET) {
                ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            }
            else if (move_type == MONSTER_MOVE_TYPE_FACING_ANGLE) {
                ptvcursor_add(ptv, hf_woww_angle, 4, ENC_LITTLE_ENDIAN);
            }
            else if (move_type == MONSTER_MOVE_TYPE_FACING_SPOT) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_spline_flag, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
            add_monster_move_spline(ptv);
            break;
        case SMSG_MOUNTRESULT:
            ptvcursor_add(ptv, hf_woww_mount_result, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MOUNTSPECIAL_ANIM:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MOVE_FEATHER_FALL:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MOVE_KNOCK_BACK:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_movement_counter, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_v_cos, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_v_sin, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_horizontal_speed, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_vertical_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MOVE_LAND_WALK:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MOVE_NORMAL_FALL:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MOVE_SET_HOVER:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MOVE_UNSET_HOVER:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_MOVE_WATER_WALK:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_counter, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_NAME_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_character_name);
            add_cstring(ptv, &hf_woww_realm_name);
            ptvcursor_add(ptv, hf_woww_race, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_gender, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_class, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_NEW_WORLD:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_NOTIFICATION:
            add_cstring(ptv, &hf_woww_notification);
            break;
        case SMSG_NPC_TEXT_UPDATE:
            ptvcursor_add(ptv, hf_woww_text_id, 4, ENC_LITTLE_ENDIAN);
            for (guint32 i1 = 0; i1 < 8; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "NpcTextUpdate %i", i1);
                ptvcursor_add(ptv, hf_woww_probability, 4, ENC_LITTLE_ENDIAN);
                for (guint32 i2 = 0; i2 < 2; ++i2) {
                    add_cstring(ptv, &hf_woww_texts);
                }
                ptvcursor_add(ptv, hf_woww_language, 4, ENC_LITTLE_ENDIAN);
                for (guint32 i2 = 0; i2 < 3; ++i2) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "NpcTextUpdateEmote %i", i2);
                    ptvcursor_add(ptv, hf_woww_delay, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_emote_int, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_OPEN_CONTAINER:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PAGE_TEXT_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_page_id, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_text);
            ptvcursor_add(ptv, hf_woww_next_page_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PARTYKILLLOG:
            ptvcursor_add(ptv, hf_woww_player_with_killing_blow, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_victim, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PARTY_COMMAND_RESULT:
            ptvcursor_add(ptv, hf_woww_party_operation, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_member);
            ptvcursor_add(ptv, hf_woww_party_result, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PARTY_MEMBER_STATS:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add_ret_uint(ptv, hf_woww_group_update_flags, 4, ENC_LITTLE_ENDIAN, &mask);
            if (mask & GROUP_UPDATE_FLAGS_STATUS) {
                ptvcursor_add(ptv, hf_woww_group_member_online_status, 1, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_CUR_HP) {
                ptvcursor_add(ptv, hf_woww_current_health, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_MAX_HP) {
                ptvcursor_add(ptv, hf_woww_max_health, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_POWER_TYPE) {
                ptvcursor_add(ptv, hf_woww_power, 1, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_CUR_POWER) {
                ptvcursor_add(ptv, hf_woww_current_power, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_MAX_POWER) {
                ptvcursor_add(ptv, hf_woww_max_power, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_LEVEL) {
                ptvcursor_add(ptv, hf_woww_level, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_ZONE) {
                ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_POSITION) {
                ptvcursor_add(ptv, hf_woww_position_x_int, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_position_y_int, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_AURAS) {
                add_aura_mask(ptv);
            }
            if (mask & GROUP_UPDATE_FLAGS_AURAS_2) {
                add_aura_mask(ptv);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_GUID) {
                ptvcursor_add(ptv, hf_woww_pet, 8, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_NAME) {
                add_cstring(ptv, &hf_woww_pet_name);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_MODEL_ID) {
                ptvcursor_add(ptv, hf_woww_pet_display_id, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_CUR_HP) {
                ptvcursor_add(ptv, hf_woww_pet_current_health, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_MAX_HP) {
                ptvcursor_add(ptv, hf_woww_pet_max_health, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_POWER_TYPE) {
                ptvcursor_add(ptv, hf_woww_power, 1, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_CUR_POWER) {
                ptvcursor_add(ptv, hf_woww_pet_current_power, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_MAX_POWER) {
                ptvcursor_add(ptv, hf_woww_pet_max_power, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_AURAS) {
                add_aura_mask(ptv);
            }
            break;
        case SMSG_PARTY_MEMBER_STATS_FULL:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add_ret_uint(ptv, hf_woww_group_update_flags, 4, ENC_LITTLE_ENDIAN, &mask);
            if (mask & GROUP_UPDATE_FLAGS_STATUS) {
                ptvcursor_add(ptv, hf_woww_group_member_online_status, 1, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_CUR_HP) {
                ptvcursor_add(ptv, hf_woww_current_health, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_MAX_HP) {
                ptvcursor_add(ptv, hf_woww_max_health, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_POWER_TYPE) {
                ptvcursor_add(ptv, hf_woww_power, 1, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_CUR_POWER) {
                ptvcursor_add(ptv, hf_woww_current_power, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_MAX_POWER) {
                ptvcursor_add(ptv, hf_woww_max_power, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_LEVEL) {
                ptvcursor_add(ptv, hf_woww_level, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_ZONE) {
                ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_POSITION) {
                ptvcursor_add(ptv, hf_woww_position_x_int, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_position_y_int, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_AURAS) {
                add_aura_mask(ptv);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_GUID) {
                ptvcursor_add(ptv, hf_woww_pet, 8, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_NAME) {
                add_cstring(ptv, &hf_woww_pet_name);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_MODEL_ID) {
                ptvcursor_add(ptv, hf_woww_pet_display_id, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_CUR_HP) {
                ptvcursor_add(ptv, hf_woww_pet_current_health, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_MAX_HP) {
                ptvcursor_add(ptv, hf_woww_pet_max_health, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_POWER_TYPE) {
                ptvcursor_add(ptv, hf_woww_power, 1, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_CUR_POWER) {
                ptvcursor_add(ptv, hf_woww_pet_current_power, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_MAX_POWER) {
                ptvcursor_add(ptv, hf_woww_pet_max_power, 2, ENC_LITTLE_ENDIAN);
            }
            if (mask & GROUP_UPDATE_FLAGS_PET_AURAS) {
                add_aura_mask(ptv);
            }
            break;
        case SMSG_PAUSE_MIRROR_TIMER:
            ptvcursor_add(ptv, hf_woww_timer_type, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_is_frozen, 1, ENC_NA);
            break;
        case SMSG_PERIODICAURALOG:
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_auras, 4, ENC_LITTLE_ENDIAN, &amount_of_auras);
            for (guint32 i1 = 0; i1 < amount_of_auras; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "AuraLog %i", i1);
                ptvcursor_add_ret_uint(ptv, hf_woww_aura_type, 4, ENC_LITTLE_ENDIAN, &aura_type);
                if (aura_type == AURA_TYPE_PERIODIC_DAMAGE
                 || aura_type == AURA_TYPE_PERIODIC_DAMAGE_PERCENT) {
                    ptvcursor_add(ptv, hf_woww_damage, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_spell_school, 1, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_absorbed, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_resisted, 4, ENC_LITTLE_ENDIAN);
                }
                else if (aura_type == AURA_TYPE_PERIODIC_HEAL
                 || aura_type == AURA_TYPE_OBS_MOD_HEALTH) {
                    ptvcursor_add(ptv, hf_woww_damage, 4, ENC_LITTLE_ENDIAN);
                }
                else if (aura_type == AURA_TYPE_OBS_MOD_MANA
                 || aura_type == AURA_TYPE_PERIODIC_ENERGIZE) {
                    ptvcursor_add(ptv, hf_woww_misc_value, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_damage, 4, ENC_LITTLE_ENDIAN);
                }
                else if (aura_type == AURA_TYPE_PERIODIC_MANA_LEECH) {
                    ptvcursor_add(ptv, hf_woww_misc_value, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_damage, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_gain_multiplier, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_PETITION_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_petition_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_charter_owner, 8, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_guild_name);
            add_cstring(ptv, &hf_woww_body_text);
            ptvcursor_add(ptv, hf_woww_unknown_flags, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_minimum_signatures, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_maximum_signatures, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_deadline, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_issue_date, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_allowed_guild_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_allowed_class, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_allowed_race, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_allowed_genders, 2, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_allowed_minimum_level, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_allowed_maximum_level, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_todo_amount_of_signers, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_number_of_choices, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PETITION_SHOWLIST:
            ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_petitions, 1, ENC_LITTLE_ENDIAN, &amount_of_petitions);
            for (guint32 i1 = 0; i1 < amount_of_petitions; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "PetitionShowlist %i", i1);
                ptvcursor_add(ptv, hf_woww_index, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_charter_entry, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_charter_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_guild_charter_cost, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_PETITION_SHOW_SIGNATURES:
            ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_owner, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_petition, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_signatures, 1, ENC_LITTLE_ENDIAN, &amount_of_signatures);
            for (guint32 i1 = 0; i1 < amount_of_signatures; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "PetitionSignature %i", i1);
                ptvcursor_add(ptv, hf_woww_signer, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_PETITION_SIGN_RESULTS:
            ptvcursor_add(ptv, hf_woww_petition, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_owner, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_petition_result, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PET_ACTION_FEEDBACK:
            ptvcursor_add(ptv, hf_woww_pet_feedback, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PET_ACTION_SOUND:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_pet_talk_reason, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PET_CAST_FAILED:
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell_cast_result, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PET_DISMISS_SOUND:
            ptvcursor_add(ptv, hf_woww_sound_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            break;
        case SMSG_PET_MODE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_pet_react_state, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_pet_command_state, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_pet_enabled, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PET_NAME_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_pet_number, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_name);
            ptvcursor_add(ptv, hf_woww_pet_name_timestamp, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PET_SPELLS:
            ptvcursor_add(ptv, hf_woww_pet, 8, ENC_LITTLE_ENDIAN);
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            if (len > 0) {
                ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_pet_react_state, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_pet_command_state, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_pet_enabled, 1, ENC_LITTLE_ENDIAN);
                for (guint32 i1 = 0; i1 < 10; ++i1) {
                    ptvcursor_add(ptv, hf_woww_action_bars, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_spells, 1, ENC_LITTLE_ENDIAN, &amount_of_spells);
                for (guint32 i1 = 0; i1 < amount_of_spells; ++i1) {
                    ptvcursor_add(ptv, hf_woww_spells, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_cooldowns, 1, ENC_LITTLE_ENDIAN, &amount_of_cooldowns);
                for (guint32 i1 = 0; i1 < amount_of_cooldowns; ++i1) {
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "PetSpellCooldown %i", i1);
                    ptvcursor_add(ptv, hf_woww_spell, 2, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_spell_category, 2, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_cooldown, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_category_cooldown, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_pop_subtree(ptv);
                }
            }
            break;
        case SMSG_PET_TAME_FAILURE:
            ptvcursor_add(ptv, hf_woww_pet_tame_failure_reason, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PET_UNLEARN_CONFIRM:
            ptvcursor_add(ptv, hf_woww_pet, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_talent_reset_cost, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PLAYED_TIME:
            ptvcursor_add(ptv, hf_woww_total_played_time, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_level_played_time, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PLAYERBOUND:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PLAYER_SKINNED:
            ptvcursor_add(ptv, hf_woww_spirit_released, 1, ENC_NA);
            break;
        case SMSG_PLAY_MUSIC:
            ptvcursor_add(ptv, hf_woww_sound_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PLAY_OBJECT_SOUND:
            ptvcursor_add(ptv, hf_woww_sound_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PLAY_SOUND:
            ptvcursor_add(ptv, hf_woww_sound_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PLAY_SPELL_IMPACT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell_visual_kit, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PLAY_SPELL_VISUAL:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell_art_kit, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PONG:
            ptvcursor_add(ptv, hf_woww_sequence_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PROCRESIST:
            ptvcursor_add(ptv, hf_woww_caster, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_log_format, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_PVP_CREDIT:
            ptvcursor_add(ptv, hf_woww_honor_points, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_victim, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_pvp_rank, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUERY_TIME_RESPONSE:
            ptvcursor_add(ptv, hf_woww_time, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTGIVER_OFFER_REWARD:
            ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_title);
            add_cstring(ptv, &hf_woww_offer_reward_text);
            ptvcursor_add(ptv, hf_woww_auto_finish, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_emotes, 4, ENC_LITTLE_ENDIAN, &amount_of_emotes);
            for (guint32 i1 = 0; i1 < amount_of_emotes; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "NpcTextUpdateEmote %i", i1);
                ptvcursor_add(ptv, hf_woww_delay, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_emote_int, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_choice_item_rewards, 4, ENC_LITTLE_ENDIAN, &amount_of_choice_item_rewards);
            for (guint32 i1 = 0; i1 < amount_of_choice_item_rewards; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItemRequirement %i", i1);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_item_rewards, 4, ENC_LITTLE_ENDIAN, &amount_of_item_rewards);
            for (guint32 i1 = 0; i1 < amount_of_item_rewards; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItemRequirement %i", i1);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_money_reward, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_reward_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_reward_spell_cast, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTGIVER_QUEST_COMPLETE:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_experience_reward, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_money_reward, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_item_rewards, 4, ENC_LITTLE_ENDIAN, &amount_of_item_rewards);
            for (guint32 i1 = 0; i1 < amount_of_item_rewards; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItemReward %i", i1);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_QUESTGIVER_QUEST_DETAILS:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_title);
            add_cstring(ptv, &hf_woww_details);
            add_cstring(ptv, &hf_woww_objectives);
            ptvcursor_add(ptv, hf_woww_auto_finish, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_choice_item_rewards, 4, ENC_LITTLE_ENDIAN, &amount_of_choice_item_rewards);
            for (guint32 i1 = 0; i1 < amount_of_choice_item_rewards; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItemReward %i", i1);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_item_rewards, 4, ENC_LITTLE_ENDIAN, &amount_of_item_rewards);
            for (guint32 i1 = 0; i1 < amount_of_item_rewards; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItemReward %i", i1);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_money_reward, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_reward_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_emotes, 4, ENC_LITTLE_ENDIAN, &amount_of_emotes);
            for (guint32 i1 = 0; i1 < amount_of_emotes; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestDetailsEmote %i", i1);
                ptvcursor_add(ptv, hf_woww_emote_int, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_emote_delay, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_QUESTGIVER_QUEST_FAILED:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_failed_reason, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTGIVER_QUEST_INVALID:
            ptvcursor_add(ptv, hf_woww_quest_failed_reason, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTGIVER_QUEST_LIST:
            ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_title);
            ptvcursor_add(ptv, hf_woww_emote_delay, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_emote_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_entries, 1, ENC_LITTLE_ENDIAN, &amount_of_entries);
            for (guint32 i1 = 0; i1 < amount_of_entries; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItem %i", i1);
                ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_quest_icon, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_level, 4, ENC_LITTLE_ENDIAN);
                add_cstring(ptv, &hf_woww_title);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_QUESTGIVER_REQUEST_ITEMS:
            ptvcursor_add(ptv, hf_woww_npc, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_title);
            add_cstring(ptv, &hf_woww_request_items_text);
            ptvcursor_add(ptv, hf_woww_emote_delay, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_emote_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auto_finish, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_required_money, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_required_items, 4, ENC_LITTLE_ENDIAN, &amount_of_required_items);
            for (guint32 i1 = 0; i1 < amount_of_required_items; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItemRequirement %i", i1);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_completable, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_flags, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_flags, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTGIVER_STATUS:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_giver_status, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTUPDATE_ADD_ITEM:
            ptvcursor_add(ptv, hf_woww_required_item_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_items_required, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTUPDATE_ADD_KILL:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_creature_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_kill_count, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_required_kill_count, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTUPDATE_COMPLETE:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTUPDATE_FAILED:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUESTUPDATE_FAILEDTIMER:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUEST_CONFIRM_ACCEPT:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_quest_title);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_QUEST_QUERY_RESPONSE:
            ptvcursor_add(ptv, hf_woww_quest_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_method, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_level, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_zone_or_sort, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_type, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_faction, 2, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_reputation_objective_value, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_faction, 2, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_required_opposite_reputation_value, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_next_quest_in_chain, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_money_reward, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_max_level_money_reward, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_reward_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_source_item_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_quest_flags, 4, ENC_LITTLE_ENDIAN);
            for (guint32 i1 = 0; i1 < 4; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItemReward %i", i1);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            for (guint32 i1 = 0; i1 < 6; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestItemReward %i", i1);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add(ptv, hf_woww_point_map_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector2d");
            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            ptvcursor_add(ptv, hf_woww_point_opt, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_title);
            add_cstring(ptv, &hf_woww_objective_text);
            add_cstring(ptv, &hf_woww_details);
            add_cstring(ptv, &hf_woww_end_text);
            for (guint32 i1 = 0; i1 < 4; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "QuestObjective %i", i1);
                ptvcursor_add(ptv, hf_woww_creature_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_kill_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_item_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_item_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            for (guint32 i1 = 0; i1 < 4; ++i1) {
                add_cstring(ptv, &hf_woww_objective_texts);
            }
            break;
        case SMSG_RAID_GROUP_ONLY:
            ptvcursor_add(ptv, hf_woww_homebind_timer, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_raid_group_error, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_RAID_INSTANCE_INFO:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_raid_infos, 4, ENC_LITTLE_ENDIAN, &amount_of_raid_infos);
            for (guint32 i1 = 0; i1 < amount_of_raid_infos; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "RaidInfo %i", i1);
                ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_reset_time, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_instance_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_RAID_INSTANCE_MESSAGE:
            ptvcursor_add(ptv, hf_woww_raid_instance_message, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_time_left, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_READ_ITEM_FAILED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_READ_ITEM_OK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_RECEIVED_MAIL:
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_REMOVED_SPELL:
            ptvcursor_add(ptv, hf_woww_spell, 2, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_RESISTLOG:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_float, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_float, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_RESURRECT_REQUEST:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            add_sized_cstring(ptv, &hf_woww_name);
            ptvcursor_add(ptv, hf_woww_player, 1, ENC_NA);
            break;
        case SMSG_SELL_ITEM:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_sell_item_result, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SEND_MAIL_RESULT:
            ptvcursor_add(ptv, hf_woww_mail_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_mail_action, 4, ENC_LITTLE_ENDIAN, &action);
            if (action == MAIL_ACTION_ITEM_TAKEN) {
                ptvcursor_add_ret_uint(ptv, hf_woww_mail_result, 4, ENC_LITTLE_ENDIAN, &result);
                if (result == MAIL_RESULT_ERR_EQUIP_ERROR) {
                    ptvcursor_add(ptv, hf_woww_equip_error, 4, ENC_LITTLE_ENDIAN);
                }
                else {
                    ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_item_count, 4, ENC_LITTLE_ENDIAN);
                }
            }
            else {
                ptvcursor_add_ret_uint(ptv, hf_woww_mail_result_two, 4, ENC_LITTLE_ENDIAN, &result2);
                if (result2 == MAIL_RESULT_TWO_ERR_EQUIP_ERROR) {
                    ptvcursor_add(ptv, hf_woww_equip_error, 4, ENC_LITTLE_ENDIAN);
                }
            }
            break;
        case SMSG_SERVER_MESSAGE:
            ptvcursor_add(ptv, hf_woww_server_message_type, 4, ENC_LITTLE_ENDIAN);
            add_cstring(ptv, &hf_woww_message);
            break;
        case SMSG_SET_FACTION_STANDING:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_faction_standings, 4, ENC_LITTLE_ENDIAN, &amount_of_faction_standings);
            for (guint32 i1 = 0; i1 < amount_of_faction_standings; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "FactionStanding %i", i1);
                ptvcursor_add(ptv, hf_woww_faction, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_standing, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_SET_FACTION_VISIBLE:
            ptvcursor_add(ptv, hf_woww_faction, 2, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SET_FLAT_SPELL_MODIFIER:
            ptvcursor_add(ptv, hf_woww_eff, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_op, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_value, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SET_FORCED_REACTIONS:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_reactions, 4, ENC_LITTLE_ENDIAN, &amount_of_reactions);
            for (guint32 i1 = 0; i1 < amount_of_reactions; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "ForcedReaction %i", i1);
                ptvcursor_add(ptv, hf_woww_faction, 2, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_reputation_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_SET_PCT_SPELL_MODIFIER:
            ptvcursor_add(ptv, hf_woww_eff, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_op, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_value, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SET_PROFICIENCY:
            ptvcursor_add(ptv, hf_woww_item_class, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_item_sub_class_mask, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SET_REST_START:
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SHOWTAXINODES:
            ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_nearest_node, 4, ENC_LITTLE_ENDIAN);
            while (ptvcursor_current_offset(ptv) < offset_packet_end) {
                ptvcursor_add(ptv, hf_woww_nodes, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_SHOW_BANK:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPELLDAMAGESHIELD:
            ptvcursor_add(ptv, hf_woww_victim, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_caster, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_damage, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell_school, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPELLDISPELLOG:
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_spells, 4, ENC_LITTLE_ENDIAN, &amount_of_spells);
            for (guint32 i1 = 0; i1 < amount_of_spells; ++i1) {
                ptvcursor_add(ptv, hf_woww_spells, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_SPELLENERGIZELOG:
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_power, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_damage, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPELLHEALLOG:
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_damage, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_critical, 1, ENC_NA);
            break;
        case SMSG_SPELLINSTAKILLLOG:
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPELLLOGEXECUTE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_effects, 4, ENC_LITTLE_ENDIAN, &amount_of_effects);
            for (guint32 i1 = 0; i1 < amount_of_effects; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "SpellLog %i", i1);
                ptvcursor_add_ret_uint(ptv, hf_woww_spell_effect, 4, ENC_LITTLE_ENDIAN, &effect);
                ptvcursor_add(ptv, hf_woww_amount_of_logs, 4, ENC_LITTLE_ENDIAN);
                if (effect == SPELL_EFFECT_POWER_DRAIN) {
                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_amount, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_power, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_multiplier, 4, ENC_LITTLE_ENDIAN);
                }
                else if (effect == SPELL_EFFECT_HEAL
                 || effect == SPELL_EFFECT_HEAL_MAX_HEALTH) {
                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_heal_amount, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_heal_critical, 4, ENC_LITTLE_ENDIAN);
                }
                else if (effect == SPELL_EFFECT_ENERGIZE) {
                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_energize_amount, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_energize_power, 4, ENC_LITTLE_ENDIAN);
                }
                else if (effect == SPELL_EFFECT_ADD_EXTRA_ATTACKS) {
                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_extra_attacks, 4, ENC_LITTLE_ENDIAN);
                }
                else if (effect == SPELL_EFFECT_CREATE_ITEM) {
                    ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                }
                else if (effect == SPELL_EFFECT_INTERRUPT_CAST) {
                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_interrupted_spell, 4, ENC_LITTLE_ENDIAN);
                }
                else if (effect == SPELL_EFFECT_DURABILITY_DAMAGE) {
                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_item_to_damage, 4, ENC_LITTLE_ENDIAN);
                    ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                }
                else if (effect == SPELL_EFFECT_FEED_PET) {
                    ptvcursor_add(ptv, hf_woww_feed_pet_item, 4, ENC_LITTLE_ENDIAN);
                }
                else if (effect == SPELL_EFFECT_INSTAKILL
                 || effect == SPELL_EFFECT_RESURRECT
                 || effect == SPELL_EFFECT_DISPEL
                 || effect == SPELL_EFFECT_THREAT
                 || effect == SPELL_EFFECT_DISTRACT
                 || effect == SPELL_EFFECT_SANCTUARY
                 || effect == SPELL_EFFECT_THREAT_ALL
                 || effect == SPELL_EFFECT_DISPEL_MECHANIC
                 || effect == SPELL_EFFECT_RESURRECT_NEW
                 || effect == SPELL_EFFECT_ATTACK_ME
                 || effect == SPELL_EFFECT_SKIN_PLAYER_CORPSE
                 || effect == SPELL_EFFECT_MODIFY_THREAT_PERCENT
                 || effect == SPELL_EFFECT_UNKNOWN126
                 || effect == SPELL_EFFECT_OPEN_LOCK
                 || effect == SPELL_EFFECT_OPEN_LOCK_ITEM
                 || effect == SPELL_EFFECT_DISMISS_PET
                 || effect == SPELL_EFFECT_TRANS_DOOR
                 || effect == SPELL_EFFECT_SUMMON
                 || effect == SPELL_EFFECT_SUMMON_PET
                 || effect == SPELL_EFFECT_SUMMON_WILD
                 || effect == SPELL_EFFECT_SUMMON_GUARDIAN
                 || effect == SPELL_EFFECT_SUMMON_TOTEM_SLOT1
                 || effect == SPELL_EFFECT_SUMMON_TOTEM_SLOT2
                 || effect == SPELL_EFFECT_SUMMON_TOTEM_SLOT3
                 || effect == SPELL_EFFECT_SUMMON_TOTEM_SLOT4
                 || effect == SPELL_EFFECT_SUMMON_POSSESSED
                 || effect == SPELL_EFFECT_SUMMON_TOTEM
                 || effect == SPELL_EFFECT_SUMMON_CRITTER
                 || effect == SPELL_EFFECT_SUMMON_OBJECT_WILD
                 || effect == SPELL_EFFECT_SUMMON_OBJECT_SLOT1
                 || effect == SPELL_EFFECT_SUMMON_OBJECT_SLOT2
                 || effect == SPELL_EFFECT_SUMMON_OBJECT_SLOT3
                 || effect == SPELL_EFFECT_SUMMON_OBJECT_SLOT4
                 || effect == SPELL_EFFECT_SUMMON_DEMON) {
                    ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_SPELLLOGMISS:
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_caster, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_unknown_int, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_targets, 4, ENC_LITTLE_ENDIAN, &amount_of_targets);
            for (guint32 i1 = 0; i1 < amount_of_targets; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "SpellLogMiss %i", i1);
                ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_spell_miss_info, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_SPELLNONMELEEDAMAGELOG:
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_damage, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell_school, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_absorbed_damage, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_resisted, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_periodic_log, 1, ENC_NA);
            ptvcursor_add(ptv, hf_woww_unused, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_blocked, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_hit_info, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_extend_flag, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPELLORDAMAGE_IMMUNE:
            ptvcursor_add(ptv, hf_woww_caster, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_debug_log_format, 1, ENC_NA);
            break;
        case SMSG_SPELL_COOLDOWN:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            while (ptvcursor_current_offset(ptv) < offset_packet_end) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "SpellCooldownStatus");
                ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_cooldown_time, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_SPELL_DELAYED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_delay_time, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPELL_FAILED_OTHER:
            ptvcursor_add(ptv, hf_woww_caster, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPELL_FAILURE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell_cast_result, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPELL_GO:
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_cast_flags, 2, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_hits, 1, ENC_LITTLE_ENDIAN, &amount_of_hits);
            for (guint32 i1 = 0; i1 < amount_of_hits; ++i1) {
                ptvcursor_add(ptv, hf_woww_hits, 8, ENC_LITTLE_ENDIAN);
            }
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_misses, 1, ENC_LITTLE_ENDIAN, &amount_of_misses);
            for (guint32 i1 = 0; i1 < amount_of_misses; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "SpellMiss %i", i1);
                ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add_ret_uint(ptv, hf_woww_spell_miss_info, 1, ENC_LITTLE_ENDIAN, &miss_info);
                ptvcursor_pop_subtree(ptv);
            }
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "SpellCastTargets");
            ptvcursor_add_ret_uint(ptv, hf_woww_spell_cast_target_flags, 2, ENC_LITTLE_ENDIAN, &target_flags);
            if (target_flags & SPELL_CAST_TARGET_FLAGS_UNIT) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_GAMEOBJECT) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_OBJECT_UNK) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_TRADE_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_SOURCE_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_DEST_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_STRING) {
                add_cstring(ptv, &hf_woww_target_string);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_PVP_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            ptvcursor_pop_subtree(ptv);
            if (flags & CAST_FLAGS_AMMO) {
                ptvcursor_add(ptv, hf_woww_ammo_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_ammo_inventory_type, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_SPELL_START:
            add_packed_guid(ptv, pinfo);
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_cast_flags, 2, ENC_LITTLE_ENDIAN, &flags);
            ptvcursor_add(ptv, hf_woww_timer, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "SpellCastTargets");
            ptvcursor_add_ret_uint(ptv, hf_woww_spell_cast_target_flags, 2, ENC_LITTLE_ENDIAN, &target_flags);
            if (target_flags & SPELL_CAST_TARGET_FLAGS_UNIT) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_GAMEOBJECT) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_OBJECT_UNK) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_TRADE_ITEM) {
                add_packed_guid(ptv, pinfo);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_SOURCE_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_DEST_LOCATION) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_STRING) {
                add_cstring(ptv, &hf_woww_target_string);
            }
            if (target_flags & SPELL_CAST_TARGET_FLAGS_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            else if (target_flags & SPELL_CAST_TARGET_FLAGS_PVP_CORPSE) {
                add_packed_guid(ptv, pinfo);
            }
            ptvcursor_pop_subtree(ptv);
            if (flags & CAST_FLAGS_AMMO) {
                ptvcursor_add(ptv, hf_woww_ammo_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_ammo_inventory_type, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_SPELL_UPDATE_CHAIN_TARGETS:
            ptvcursor_add(ptv, hf_woww_caster, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_targets, 4, ENC_LITTLE_ENDIAN, &amount_of_targets);
            for (guint32 i1 = 0; i1 < amount_of_targets; ++i1) {
                ptvcursor_add(ptv, hf_woww_targets, 8, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_SPIRIT_HEALER_CONFIRM:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPLINE_MOVE_FEATHER_FALL:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_LAND_WALK:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_NORMAL_FALL:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_ROOT:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPLINE_MOVE_SET_HOVER:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_SET_RUN_MODE:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_SET_WALK_MODE:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_START_SWIM:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_STOP_SWIM:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_UNROOT:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_UNSET_HOVER:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_MOVE_WATER_WALK:
            add_packed_guid(ptv, pinfo);
            break;
        case SMSG_SPLINE_SET_RUN_BACK_SPEED:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPLINE_SET_RUN_SPEED:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPLINE_SET_SWIM_BACK_SPEED:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPLINE_SET_SWIM_SPEED:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPLINE_SET_TURN_RATE:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SPLINE_SET_WALK_SPEED:
            add_packed_guid(ptv, pinfo);
            ptvcursor_add(ptv, hf_woww_speed, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_STABLE_RESULT:
            ptvcursor_add(ptv, hf_woww_stable_result, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_STANDSTATE_UPDATE:
            ptvcursor_add(ptv, hf_woww_unit_stand_state, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_START_MIRROR_TIMER:
            ptvcursor_add(ptv, hf_woww_timer_type, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_time_remaining, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_scale, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_is_frozen, 1, ENC_NA);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_STOP_MIRROR_TIMER:
            ptvcursor_add(ptv, hf_woww_timer_type, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SUMMON_REQUEST:
            ptvcursor_add(ptv, hf_woww_summoner, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_auto_decline_time, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_SUPERCEDED_SPELL:
            ptvcursor_add(ptv, hf_woww_new_spell_id, 2, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_old_spell_id, 2, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_TAXINODE_STATUS:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_taxi_mask_node_known, 1, ENC_NA);
            break;
        case SMSG_TEXT_EMOTE:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_text_emote, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_emote_int, 4, ENC_LITTLE_ENDIAN);
            add_sized_cstring(ptv, &hf_woww_name);
            break;
        case SMSG_TRADE_STATUS:
            ptvcursor_add_ret_uint(ptv, hf_woww_trade_status, 4, ENC_LITTLE_ENDIAN, &status);
            if (status == TRADE_STATUS_BEGIN_TRADE) {
                ptvcursor_add(ptv, hf_woww_unknown_guid, 8, ENC_LITTLE_ENDIAN);
            }
            else if (status == TRADE_STATUS_CLOSE_WINDOW) {
                ptvcursor_add(ptv, hf_woww_inventory_result, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_target_error, 1, ENC_NA);
                ptvcursor_add(ptv, hf_woww_item_limit_category_id, 4, ENC_LITTLE_ENDIAN);
            }
            else if (status == TRADE_STATUS_ONLY_CONJURED
             || status == TRADE_STATUS_NOT_ON_TAPLIST) {
                ptvcursor_add(ptv, hf_woww_slot, 1, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_TRADE_STATUS_EXTENDED:
            ptvcursor_add(ptv, hf_woww_self_player, 1, ENC_NA);
            ptvcursor_add(ptv, hf_woww_trade_slot_count, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_trade_slot_count, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_money_in_trade, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_spell_on_lowest_slot, 4, ENC_LITTLE_ENDIAN);
            for (guint32 i1 = 0; i1 < 7; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TradeSlot %i", i1);
                ptvcursor_add(ptv, hf_woww_trade_slot_number, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_display_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_stack_count, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_wrapped, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_gift_wrapper, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_enchantment, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_creator, 8, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_spell_charges, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_suffix_factor, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_item_random_properties_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_lock_id, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_max_durability, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_durability, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_TRAINER_BUY_FAILED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_training_failure_reason, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_TRAINER_BUY_SUCCEEDED:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_TRAINER_LIST:
            ptvcursor_add(ptv, hf_woww_guid, 8, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_trainer_type, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_spells, 4, ENC_LITTLE_ENDIAN, &amount_of_spells);
            for (guint32 i1 = 0; i1 < amount_of_spells; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "TrainerSpell %i", i1);
                ptvcursor_add(ptv, hf_woww_spell, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_trainer_spell_state, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_spell_cost, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_talent_point_cost, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_first_rank, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_level, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_skill, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_required_skill_value, 4, ENC_LITTLE_ENDIAN);
                for (guint32 i2 = 0; i2 < 3; ++i2) {
                    ptvcursor_add(ptv, hf_woww_required_spells, 4, ENC_LITTLE_ENDIAN);
                }
                ptvcursor_pop_subtree(ptv);
            }
            add_cstring(ptv, &hf_woww_greeting);
            break;
        case SMSG_TRANSFER_ABORTED:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_transfer_abort_reason, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_argument, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_TRANSFER_PENDING:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            if (len > 0) {
                ptvcursor_add(ptv, hf_woww_transport, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_TRIGGER_CINEMATIC:
            ptvcursor_add(ptv, hf_woww_cinematic_sequence_id, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_TURN_IN_PETITION_RESULTS:
            ptvcursor_add(ptv, hf_woww_petition_result, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_TUTORIAL_FLAGS:
            for (guint32 i1 = 0; i1 < 8; ++i1) {
                ptvcursor_add(ptv, hf_woww_tutorial_data, 4, ENC_LITTLE_ENDIAN);
            }
            break;
        case SMSG_UPDATE_AURA_DURATION:
            ptvcursor_add(ptv, hf_woww_aura_slot, 1, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_aura_duration, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_UPDATE_INSTANCE_OWNERSHIP:
            ptvcursor_add(ptv, hf_woww_player_is_saved_to_a_raid, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_UPDATE_LAST_INSTANCE:
            ptvcursor_add(ptv, hf_woww_map, 4, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_UPDATE_OBJECT:
            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_objects, 4, ENC_LITTLE_ENDIAN, &amount_of_objects);
            ptvcursor_add(ptv, hf_woww_has_transport, 1, ENC_LITTLE_ENDIAN);
            for (guint32 i1 = 0; i1 < amount_of_objects; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Object %i", i1);
                ptvcursor_add_ret_uint(ptv, hf_woww_update_type, 1, ENC_LITTLE_ENDIAN, &update_type);
                if (update_type == UPDATE_TYPE_VALUES) {
                    add_packed_guid(ptv, pinfo);
                    add_update_mask(ptv, pinfo);
                }
                else if (update_type == UPDATE_TYPE_MOVEMENT) {
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementBlock");
                    ptvcursor_add_ret_uint(ptv, hf_woww_update_flag, 1, ENC_LITTLE_ENDIAN, &update_flag);
                    if (update_flag & UPDATE_FLAG_LIVING) {
                        ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                        ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                        ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_pop_subtree(ptv);
                        ptvcursor_add(ptv, hf_woww_living_orientation, 4, ENC_LITTLE_ENDIAN);
                        if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                            add_packed_guid(ptv, pinfo);
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                            ptvcursor_add(ptv, hf_woww_transport_orientation, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (flags & MOVEMENT_FLAGS_SWIMMING) {
                            ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                        }
                        ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                        if (flags & MOVEMENT_FLAGS_JUMPING) {
                            ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                            ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                        }
                        ptvcursor_add(ptv, hf_woww_walking_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_running_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_backwards_running_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_swimming_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_backwards_swimming_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_turn_rate, 4, ENC_LITTLE_ENDIAN);
                        if (flags & MOVEMENT_FLAGS_SPLINE_ENABLED) {
                            ptvcursor_add_ret_uint(ptv, hf_woww_spline_flag, 4, ENC_LITTLE_ENDIAN, &spline_flags);
                            if (spline_flags & SPLINE_FLAG_FINAL_ANGLE) {
                                ptvcursor_add(ptv, hf_woww_angle, 4, ENC_LITTLE_ENDIAN);
                            }
                            else if (spline_flags & SPLINE_FLAG_FINAL_TARGET) {
                                ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                            }
                            else if (spline_flags & SPLINE_FLAG_FINAL_POINT) {
                                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_pop_subtree(ptv);
                            }
                            ptvcursor_add(ptv, hf_woww_time_passed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_nodes, 4, ENC_LITTLE_ENDIAN, &amount_of_nodes);
                            for (guint32 i2 = 0; i2 < amount_of_nodes; ++i2) {
                                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d %i", i2);
                                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_pop_subtree(ptv);
                            }
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                        }
                    }
                    else if (update_flag & UPDATE_FLAG_HAS_POSITION) {
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                        ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_pop_subtree(ptv);
                        ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    }
                    if (update_flag & UPDATE_FLAG_HIGH_GUID) {
                        ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                    }
                    if (update_flag & UPDATE_FLAG_ALL) {
                        ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                    }
                    if (update_flag & UPDATE_FLAG_MELEE_ATTACKING) {
                        add_packed_guid(ptv, pinfo);
                    }
                    if (update_flag & UPDATE_FLAG_TRANSPORT) {
                        ptvcursor_add(ptv, hf_woww_transport_progress_in_ms, 4, ENC_LITTLE_ENDIAN);
                    }
                    ptvcursor_pop_subtree(ptv);
                }
                else if (update_type == UPDATE_TYPE_CREATE_OBJECT
                 || update_type == UPDATE_TYPE_CREATE_OBJECT2) {
                    add_packed_guid(ptv, pinfo);
                    ptvcursor_add(ptv, hf_woww_object_type, 1, ENC_LITTLE_ENDIAN);
                    ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "MovementBlock");
                    ptvcursor_add_ret_uint(ptv, hf_woww_update_flag, 1, ENC_LITTLE_ENDIAN, &update_flag);
                    if (update_flag & UPDATE_FLAG_LIVING) {
                        ptvcursor_add_ret_uint(ptv, hf_woww_movement_flags, 4, ENC_LITTLE_ENDIAN, &flags);
                        ptvcursor_add(ptv, hf_woww_timestamp, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                        ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_pop_subtree(ptv);
                        ptvcursor_add(ptv, hf_woww_living_orientation, 4, ENC_LITTLE_ENDIAN);
                        if (flags & MOVEMENT_FLAGS_ON_TRANSPORT) {
                            add_packed_guid(ptv, pinfo);
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                            ptvcursor_add(ptv, hf_woww_transport_orientation, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (flags & MOVEMENT_FLAGS_SWIMMING) {
                            ptvcursor_add(ptv, hf_woww_pitch, 4, ENC_LITTLE_ENDIAN);
                        }
                        ptvcursor_add(ptv, hf_woww_fall_time, 4, ENC_LITTLE_ENDIAN);
                        if (flags & MOVEMENT_FLAGS_JUMPING) {
                            ptvcursor_add(ptv, hf_woww_z_speed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_cos_angle, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_sin_angle, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_xy_speed, 4, ENC_LITTLE_ENDIAN);
                        }
                        if (flags & MOVEMENT_FLAGS_SPLINE_ELEVATION) {
                            ptvcursor_add(ptv, hf_woww_spline_elevation, 4, ENC_LITTLE_ENDIAN);
                        }
                        ptvcursor_add(ptv, hf_woww_walking_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_running_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_backwards_running_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_swimming_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_backwards_swimming_speed, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_turn_rate, 4, ENC_LITTLE_ENDIAN);
                        if (flags & MOVEMENT_FLAGS_SPLINE_ENABLED) {
                            ptvcursor_add_ret_uint(ptv, hf_woww_spline_flag, 4, ENC_LITTLE_ENDIAN, &spline_flags);
                            if (spline_flags & SPLINE_FLAG_FINAL_ANGLE) {
                                ptvcursor_add(ptv, hf_woww_angle, 4, ENC_LITTLE_ENDIAN);
                            }
                            else if (spline_flags & SPLINE_FLAG_FINAL_TARGET) {
                                ptvcursor_add(ptv, hf_woww_target, 8, ENC_LITTLE_ENDIAN);
                            }
                            else if (spline_flags & SPLINE_FLAG_FINAL_POINT) {
                                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_pop_subtree(ptv);
                            }
                            ptvcursor_add(ptv, hf_woww_time_passed, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_duration, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_id, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add_ret_uint(ptv, hf_woww_amount_of_nodes, 4, ENC_LITTLE_ENDIAN, &amount_of_nodes);
                            for (guint32 i2 = 0; i2 < amount_of_nodes; ++i2) {
                                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d %i", i2);
                                ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                                ptvcursor_pop_subtree(ptv);
                            }
                            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                            ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                            ptvcursor_pop_subtree(ptv);
                        }
                    }
                    else if (update_flag & UPDATE_FLAG_HAS_POSITION) {
                        ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "Vector3d");
                        ptvcursor_add(ptv, hf_woww_x, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_y, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_add(ptv, hf_woww_z, 4, ENC_LITTLE_ENDIAN);
                        ptvcursor_pop_subtree(ptv);
                        ptvcursor_add(ptv, hf_woww_orientation, 4, ENC_LITTLE_ENDIAN);
                    }
                    if (update_flag & UPDATE_FLAG_HIGH_GUID) {
                        ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                    }
                    if (update_flag & UPDATE_FLAG_ALL) {
                        ptvcursor_add(ptv, hf_woww_unknown_int, 4, ENC_LITTLE_ENDIAN);
                    }
                    if (update_flag & UPDATE_FLAG_MELEE_ATTACKING) {
                        add_packed_guid(ptv, pinfo);
                    }
                    if (update_flag & UPDATE_FLAG_TRANSPORT) {
                        ptvcursor_add(ptv, hf_woww_transport_progress_in_ms, 4, ENC_LITTLE_ENDIAN);
                    }
                    ptvcursor_pop_subtree(ptv);
                    add_update_mask(ptv, pinfo);
                }
                else if (update_type == UPDATE_TYPE_OUT_OF_RANGE_OBJECTS
                 || update_type == UPDATE_TYPE_NEAR_OBJECTS) {
                    ptvcursor_add_ret_uint(ptv, hf_woww_count, 4, ENC_LITTLE_ENDIAN, &count);
                    for (guint32 i2 = 0; i2 < count; ++i2) {
                        add_packed_guid(ptv, pinfo);
                    }
                }
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_UPDATE_WORLD_STATE:
            ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "WorldState");
            ptvcursor_add(ptv, hf_woww_state, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_value, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_pop_subtree(ptv);
            break;
        case SMSG_WARDEN_DATA:
            len = offset_packet_end - ptvcursor_current_offset(ptv);
            ptvcursor_add(ptv, hf_woww_encrypted_data, len, ENC_NA);
            break;
        case SMSG_WEATHER:
            ptvcursor_add(ptv, hf_woww_weather_type, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_grade, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_sound_id, 4, ENC_LITTLE_ENDIAN);
            ptvcursor_add(ptv, hf_woww_weather_change_type, 1, ENC_LITTLE_ENDIAN);
            break;
        case SMSG_WHO:
            ptvcursor_add_ret_uint(ptv, hf_woww_listed_players, 4, ENC_LITTLE_ENDIAN, &listed_players);
            ptvcursor_add(ptv, hf_woww_online_players, 4, ENC_LITTLE_ENDIAN);
            for (guint32 i1 = 0; i1 < listed_players; ++i1) {
                ptvcursor_add_text_with_subtree(ptv, SUBTREE_UNDEFINED_LENGTH, ett_message, "WhoPlayer %i", i1);
                add_cstring(ptv, &hf_woww_name);
                add_cstring(ptv, &hf_woww_guild);
                ptvcursor_add(ptv, hf_woww_level, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_class, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_race, 1, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_add(ptv, hf_woww_party_status, 4, ENC_LITTLE_ENDIAN);
                ptvcursor_pop_subtree(ptv);
            }
            break;
        case SMSG_WHOIS:
            add_cstring(ptv, &hf_woww_message);
            break;
        case SMSG_ZONE_UNDER_ATTACK:
            ptvcursor_add(ptv, hf_woww_area, 4, ENC_LITTLE_ENDIAN);
            break;

        default:
            break;
    }
/* AUTOGENERATED_END_PARSER */

    ptvcursor_free(ptv);
}

static gint
add_header_to_tree(WowwDecryptedHeader_t* decrypted_header,
                   proto_tree* tree,
                   tvbuff_t* tvb,
                   packet_info* pinfo,
                   guint8 headerSize,
                   gint start_offset)
{
    // Size field does not count in the reported size, so we need to add it.
    const guint16 packet_size = (decrypted_header->size[0] << 8 | decrypted_header->size[1]) + WOWW_HEADER_SIZE_FIELD_WIDTH;

    proto_tree* ti = proto_tree_add_item(tree, proto_woww, tvb, start_offset, packet_size, ENC_NA);

    proto_tree* woww_tree = proto_item_add_subtree(ti, ett_message);

    // Add to tree
    tvbuff_t *next_tvb = tvb_new_child_real_data(tvb, (guint8*)decrypted_header, headerSize, headerSize);
    add_new_data_source(pinfo, next_tvb, "Decrypted Header");

    // We're indexing into another tvb
    gint offset = 0;
    gint len = WOWW_HEADER_SIZE_FIELD_WIDTH;
    proto_tree_add_item(woww_tree, hf_woww_size, next_tvb,
                        offset, len, ENC_BIG_ENDIAN);
    offset += len;

    guint32 opcode = 0;
    if (WOWW_SERVER_TO_CLIENT) {
        len = 2;
        opcode = tvb_get_guint16(next_tvb, offset, ENC_LITTLE_ENDIAN);
    } else if (WOWW_CLIENT_TO_SERVER) {
        len = 4;
        opcode = tvb_get_guint32(next_tvb, offset, ENC_LITTLE_ENDIAN);
    }

    proto_tree_add_item(woww_tree, hf_woww_opcode, next_tvb,
                        offset, len, ENC_LITTLE_ENDIAN);

    if (start_offset == 0) {
        // First message
        col_set_str(pinfo->cinfo, COL_INFO, val_to_str_const(opcode,
                                                             world_packet_strings,
                                                             "Encrypted Header"));
    }
    else {
        col_append_str(pinfo->cinfo, COL_INFO, " | ");
        col_append_str(pinfo->cinfo, COL_INFO, val_to_str_const(opcode,
                                                                world_packet_strings,
                                                                "Encrypted Header"));
    }

    proto_item_set_text(woww_tree, "%s", val_to_str_const(opcode,
                                                    world_packet_strings,
                                                    "Encrypted Header"));

    gint offset_packet_end = start_offset + (gint)packet_size;

    // Remember to go back to original tvb
    add_body_fields(opcode, woww_tree, tvb, start_offset + headerSize, offset_packet_end, pinfo);

    return offset_packet_end;
}

static int
dissect_woww(tvbuff_t *tvb,
             packet_info *pinfo,
             proto_tree *tree,
             void *data _U_)
{
    if (tvb_reported_length(tvb) < WOWW_MIN_LENGTH)
        return 0;

    if (tvb_captured_length(tvb) < 1)
        return 0;

    col_set_str(pinfo->cinfo, COL_PROTOCOL, "WOWW");

    col_set_str(pinfo->cinfo, COL_INFO, "Session Key Not Known Yet");

    // Get conversation data
    conversation_t* conv = find_or_create_conversation(pinfo);
    WowwConversation_t* wowwConversation = (WowwConversation_t *)conversation_get_proto_data(conv,
                                                                                             proto_woww);
    if (wowwConversation == NULL) {
        // Assume that file scope means for the lifetime of the dissection
        wowwConversation = (WowwConversation_t*) wmem_new0(wmem_file_scope(), WowwConversation_t);
        conversation_add_proto_data(conv, proto_woww, wowwConversation);
        wowwConversation->decrypted_headers = wmem_map_new(wmem_file_scope(), g_int64_hash, g_int64_equal);
        wowwConversation->headers_need_decryption = wmem_map_new(wmem_file_scope(), g_int64_hash, g_int64_equal);
    }

    // Isolate session key for packet
    WowwParticipant_t* participant;
    guint8 headerSize = 4;

    if (WOWW_SERVER_TO_CLIENT) {
        participant = &wowwConversation->server;
        headerSize = 4;
    } else {
        participant = &wowwConversation->client;
        headerSize = 6;
    }

    proto_tree* ti = proto_tree_add_item(tree, proto_woww, tvb, 0, -1, ENC_NA);

    proto_tree* woww_tree = proto_item_add_subtree(ti, ett_woww);

    gint pdu_offset = 0;
    gint reported_length = (gint)tvb_reported_length(tvb);
    guint8 header_index = 0;
    do {
        WowwDecryptedHeader_t* decrypted_header = handle_packet_header(pinfo, tvb, participant, wowwConversation, headerSize, header_index, pdu_offset);
        if (!decrypted_header) {
            return tvb_captured_length(tvb);
        }
        const gint message_size = (decrypted_header->size[0] << 8 | decrypted_header->size[1]) + WOWW_HEADER_SIZE_FIELD_WIDTH;
        if ((pdu_offset + message_size) > reported_length) {
            return pdu_offset;
        }

        pdu_offset = add_header_to_tree(decrypted_header, woww_tree, tvb, pinfo, headerSize, pdu_offset);

        header_index++;
    } while (pdu_offset < reported_length);

    return tvb_captured_length(tvb);
}

void
proto_register_woww(void)
{
    static hf_register_info hf[] = {
        { &hf_woww_size,
          { "Size", "woww.size",
            FT_UINT16, BASE_HEX_DEC, NULL, 0,
            "Size of the packet including opcode field but not including size field", HFILL }
        },
        { &hf_woww_opcode,
          { "Opcode", "woww.opcode",
            FT_UINT32, BASE_HEX, VALS(world_packet_strings), 0,
            "Opcode of the packet", HFILL }
        },
        { &hf_woww_string_length,
          { "String Length", "woww.string_length",
            FT_UINT32, BASE_DEC_HEX, NULL, 0,
            NULL, HFILL }
        },
        { &hf_woww_aura_mask,
          { "Aura Mask", "woww.aura_mask",
            FT_UINT32, BASE_DEC_HEX, NULL, 0,
            NULL, HFILL }
        },
        { &hf_woww_aura,
          { "Aura", "woww.aura",
            FT_UINT32, BASE_DEC_HEX, NULL, 0,
            NULL, HFILL }
        },
        { &hf_woww_amount_of_masks,
          { "Amount of Masks", "woww.amount_of_masks",
            FT_UINT8, BASE_DEC_HEX, NULL, 0,
            NULL, HFILL }
        },
        { &hf_woww_update_mask,
          { "Update Mask", "woww.update_mask",
            FT_UINT32, BASE_DEC_HEX, NULL, 0,
            NULL, HFILL }
        },
        { &hf_woww_update_mask_index,
          { "Update Mask Index", "woww.update_mask_index",
            FT_UINT32, BASE_DEC_HEX, NULL, 0,
            NULL, HFILL }
        },
        { &hf_woww_update_mask_value,
          { "Update Mask Value", "woww.update_mask_value",
            FT_UINT32, BASE_DEC_HEX, NULL, 0,
            NULL, HFILL }
        },
        { &hf_woww_amount_of_splines,
          { "Amount Of Splines", "woww.amount.of.splines",
            FT_UINT32, BASE_HEX_DEC, NULL, 0,
            NULL, HFILL }
        },
        { &hf_woww_packed_spline,
          { "Packed Spline", "woww.packed_spline",
            FT_UINT32, BASE_HEX_DEC, NULL, 0,
            NULL, HFILL }
        },
/* AUTOGENERATED_START_REGISTER */
        { &hf_woww_absorb,
            { "Absorb", "woww.absorb",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_absorbed,
            { "Absorbed", "woww.absorbed",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_absorbed_damage,
            { "Absorbed Damage", "woww.absorbed.damage",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_account_data_type,
            { "Account Data Type", "woww.account.data.type",
                FT_UINT32, BASE_HEX_DEC, VALS(e_account_data_type_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_action,
            { "Action", "woww.action",
                FT_UINT16, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_action_bar,
            { "Action Bar", "woww.action.bar",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_action_bars,
            { "Action Bars", "woww.action.bars",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_action_type,
            { "Action Type", "woww.action.type",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_activate_taxi_reply,
            { "Activate Taxi Reply", "woww.activate.taxi.reply",
                FT_UINT32, BASE_HEX_DEC, VALS(e_activate_taxi_reply_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_actual_roll,
            { "Actual Roll", "woww.actual.roll",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_addon_crc,
            { "Addon Crc", "woww.addon.crc",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_addon_extra_crc,
            { "Addon Extra Crc", "woww.addon.extra.crc",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_addon_has_signature,
            { "Addon Has Signature", "woww.addon.has.signature",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_addon_name,
            { "Addon Name", "woww.addon.name",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_addon_type,
            { "Addon Type", "woww.addon.type",
                FT_UINT8, BASE_HEX_DEC, VALS(e_addon_type_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_agility,
            { "Agility", "woww.agility",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_ai_reaction,
            { "Ai Reaction", "woww.ai.reaction",
                FT_UINT32, BASE_HEX_DEC, VALS(e_ai_reaction_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_allow_movement,
            { "Allow Movement", "woww.allow.movement",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_allowed_class,
            { "Allowed Class", "woww.allowed.class",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_allowed_genders,
            { "Allowed Genders", "woww.allowed.genders",
                FT_UINT16, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_allowed_guild_id,
            { "Allowed Guild Id", "woww.allowed.guild.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_allowed_maximum_level,
            { "Allowed Maximum Level", "woww.allowed.maximum.level",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_allowed_minimum_level,
            { "Allowed Minimum Level", "woww.allowed.minimum.level",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_allowed_race,
            { "Allowed Race", "woww.allowed.race",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_ammo_display_id,
            { "Ammo Display Id", "woww.ammo.display.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_ammo_inventory_type,
            { "Ammo Inventory Type", "woww.ammo.inventory.type",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_ammo_type,
            { "Ammo Type", "woww.ammo.type",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount,
            { "Amount", "woww.amount",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_bought,
            { "Amount Bought", "woww.amount.bought",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_for_sale,
            { "Amount For Sale", "woww.amount.for.sale",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_accounts_in_guild,
            { "Amount Of Accounts In Guild", "woww.amount.of.accounts.in.guild",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_auras,
            { "Amount Of Auras", "woww.amount.of.auras",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_carriers,
            { "Amount Of Carriers", "woww.amount.of.carriers",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_characters,
            { "Amount Of Characters", "woww.amount.of.characters",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_characters_in_guild,
            { "Amount Of Characters In Guild", "woww.amount.of.characters.in.guild",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_choice_item_rewards,
            { "Amount Of Choice Item Rewards", "woww.amount.of.choice.item.rewards",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_cooldowns,
            { "Amount Of Cooldowns", "woww.amount.of.cooldowns",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_damages,
            { "Amount Of Damages", "woww.amount.of.damages",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_effects,
            { "Amount Of Effects", "woww.amount.of.effects",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_emotes,
            { "Amount Of Emotes", "woww.amount.of.emotes",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_entries,
            { "Amount Of Entries", "woww.amount.of.entries",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_events,
            { "Amount Of Events", "woww.amount.of.events",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_extra_fields,
            { "Amount Of Extra Fields", "woww.amount.of.extra.fields",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_faction_standings,
            { "Amount Of Faction Standings", "woww.amount.of.faction.standings",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_factions,
            { "Amount Of Factions", "woww.amount.of.factions",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_friends,
            { "Amount Of Friends", "woww.amount.of.friends",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_gossip_items,
            { "Amount Of Gossip Items", "woww.amount.of.gossip.items",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_hits,
            { "Amount Of Hits", "woww.amount.of.hits",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_ignored,
            { "Amount Of Ignored", "woww.amount.of.ignored",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_item_rewards,
            { "Amount Of Item Rewards", "woww.amount.of.item.rewards",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_items,
            { "Amount Of Items", "woww.amount.of.items",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_logs,
            { "Amount Of Logs", "woww.amount.of.logs",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_mails,
            { "Amount Of Mails", "woww.amount.of.mails",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_members,
            { "Amount Of Members", "woww.amount.of.members",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_misses,
            { "Amount Of Misses", "woww.amount.of.misses",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_nodes,
            { "Amount Of Nodes", "woww.amount.of.nodes",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_objects,
            { "Amount Of Objects", "woww.amount.of.objects",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_outbid_items,
            { "Amount Of Outbid Items", "woww.amount.of.outbid.items",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_petitions,
            { "Amount Of Petitions", "woww.amount.of.petitions",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_pets,
            { "Amount Of Pets", "woww.amount.of.pets",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_players,
            { "Amount Of Players", "woww.amount.of.players",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_quests,
            { "Amount Of Quests", "woww.amount.of.quests",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_raid_infos,
            { "Amount Of Raid Infos", "woww.amount.of.raid.infos",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_reactions,
            { "Amount Of Reactions", "woww.amount.of.reactions",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_records,
            { "Amount Of Records", "woww.amount.of.records",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_required_items,
            { "Amount Of Required Items", "woww.amount.of.required.items",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_rights,
            { "Amount Of Rights", "woww.amount.of.rights",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_signatures,
            { "Amount Of Signatures", "woww.amount.of.signatures",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_spells,
            { "Amount Of Spells", "woww.amount.of.spells",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_states,
            { "Amount Of States", "woww.amount.of.states",
                FT_UINT16, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_strings,
            { "Amount Of Strings", "woww.amount.of.strings",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_targets,
            { "Amount Of Targets", "woww.amount.of.targets",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_teammates,
            { "Amount Of Teammates", "woww.amount.of.teammates",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_amount_of_zones,
            { "Amount Of Zones", "woww.amount.of.zones",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_angle,
            { "Angle", "woww.angle",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_animation_id,
            { "Animation Id", "woww.animation.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_answer,
            { "Answer", "woww.answer",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_answer_comment,
            { "Answer Comment", "woww.answer.comment",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_apply,
            { "Apply", "woww.apply",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_arcane_resistance,
            { "Arcane Resistance", "woww.arcane.resistance",
                FT_INT32, BASE_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_area,
            { "Area", "woww.area",
                FT_UINT32, BASE_HEX_DEC, VALS(e_area_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_argument,
            { "Argument", "woww.argument",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_armor,
            { "Armor", "woww.armor",
                FT_INT32, BASE_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_attacker,
            { "Attacker", "woww.attacker",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_command_action,
            { "Auction Command Action", "woww.auction.command.action",
                FT_UINT32, BASE_HEX_DEC, VALS(e_auction_command_action_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_command_result,
            { "Auction Command Result", "woww.auction.command.result",
                FT_UINT32, BASE_HEX_DEC, VALS(e_auction_command_result_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_command_result_two,
            { "Auction Command Result Two", "woww.auction.command.result.two",
                FT_UINT32, BASE_HEX_DEC, VALS(e_auction_command_result_two_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_duration_in_minutes,
            { "Auction Duration In Minutes", "woww.auction.duration.in.minutes",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_house,
            { "Auction House", "woww.auction.house",
                FT_UINT32, BASE_HEX_DEC, VALS(e_auction_house_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_id,
            { "Auction Id", "woww.auction.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_main_category,
            { "Auction Main Category", "woww.auction.main.category",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_out_bid,
            { "Auction Out Bid", "woww.auction.out.bid",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_outbid,
            { "Auction Outbid", "woww.auction.outbid",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_slot_id,
            { "Auction Slot Id", "woww.auction.slot.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auction_sub_category,
            { "Auction Sub Category", "woww.auction.sub.category",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auctioneer,
            { "Auctioneer", "woww.auctioneer",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_aura_duration,
            { "Aura Duration", "woww.aura.duration",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_aura_slot,
            { "Aura Slot", "woww.aura.slot",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_aura_type,
            { "Aura Type", "woww.aura.type",
                FT_UINT32, BASE_HEX_DEC, VALS(e_aura_type_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auto_decline_time,
            { "Auto Decline Time", "woww.auto.decline.time",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_auto_finish,
            { "Auto Finish", "woww.auto.finish",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_autocast_enabled,
            { "Autocast Enabled", "woww.autocast.enabled",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_average_wait_time_in_ms,
            { "Average Wait Time In Ms", "woww.average.wait.time.in.ms",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_background_color,
            { "Background Color", "woww.background.color",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_backwards_running_speed,
            { "Backwards Running Speed", "woww.backwards.running.speed",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_backwards_swimming_speed,
            { "Backwards Swimming Speed", "woww.backwards.swimming.speed",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bag,
            { "Bag", "woww.bag",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bag_family,
            { "Bag Family", "woww.bag.family",
                FT_UINT32, BASE_HEX_DEC, VALS(e_bag_family_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bag_index,
            { "Bag Index", "woww.bag.index",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bag_slot,
            { "Bag Slot", "woww.bag.slot",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bag_type_subclass,
            { "Bag Type Subclass", "woww.bag.type.subclass",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_battlefield_port_action,
            { "Battlefield Port Action", "woww.battlefield.port.action",
                FT_UINT8, BASE_HEX_DEC, VALS(e_battlefield_port_action_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_battleground_bracket,
            { "Battleground Bracket", "woww.battleground.bracket",
                FT_UINT8, BASE_HEX_DEC, VALS(e_battleground_bracket_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_battleground_end_status,
            { "Battleground End Status", "woww.battleground.end.status",
                FT_UINT8, BASE_HEX_DEC, VALS(e_battleground_end_status_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_battleground_winner,
            { "Battleground Winner", "woww.battleground.winner",
                FT_UINT8, BASE_HEX_DEC, VALS(e_battleground_winner_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_battlegrounds,
            { "Battlegrounds", "woww.battlegrounds",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_battlemaster,
            { "Battlemaster", "woww.battlemaster",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bg_type_id,
            { "Bg Type Id", "woww.bg.type.id",
                FT_UINT32, BASE_HEX_DEC, VALS(e_bg_type_id_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bid,
            { "Bid", "woww.bid",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bidder,
            { "Bidder", "woww.bidder",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_billing_flags,
            { "Billing Flags", "woww.billing.flags",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_billing_rested,
            { "Billing Rested", "woww.billing.rested",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_billing_time,
            { "Billing Time", "woww.billing.time",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_block,
            { "Block", "woww.block",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_blocked,
            { "Blocked", "woww.blocked",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_blocked_amount,
            { "Blocked Amount", "woww.blocked.amount",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_body,
            { "Body", "woww.body",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_body_text,
            { "Body Text", "woww.body.text",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bonding,
            { "Bonding", "woww.bonding",
                FT_UINT32, BASE_HEX_DEC, VALS(e_bonding_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bonus_honor,
            { "Bonus Honor", "woww.bonus.honor",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_border_color,
            { "Border Color", "woww.border.color",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_border_style,
            { "Border Style", "woww.border.style",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_bug_type,
            { "Bug Type", "woww.bug.type",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_build,
            { "Build", "woww.build",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_button,
            { "Button", "woww.button",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_buy_bank_slot_result,
            { "Buy Bank Slot Result", "woww.buy.bank.slot.result",
                FT_UINT32, BASE_HEX_DEC, VALS(e_buy_bank_slot_result_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_buy_price,
            { "Buy Price", "woww.buy.price",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_buy_result,
            { "Buy Result", "woww.buy.result",
                FT_UINT8, BASE_HEX_DEC, VALS(e_buy_result_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_buyback_slot,
            { "Buyback Slot", "woww.buyback.slot",
                FT_UINT32, BASE_HEX_DEC, VALS(e_buyback_slot_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_buyout,
            { "Buyout", "woww.buyout",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_buyout_amount,
            { "Buyout Amount", "woww.buyout.amount",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cash_on_delivery_amount,
            { "Cash On Delivery Amount", "woww.cash.on.delivery.amount",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cast_failure_reason,
            { "Cast Failure Reason", "woww.cast.failure.reason",
                FT_UINT8, BASE_HEX_DEC, VALS(e_cast_failure_reason_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cast_flags,
            { "Cast Flags", "woww.cast.flags",
                FT_UINT16, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cast_item,
            { "Cast Item", "woww.cast.item",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_caster,
            { "Caster", "woww.caster",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_category_cooldown,
            { "Category Cooldown", "woww.category.cooldown",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_channel,
            { "Channel", "woww.channel",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_channel_flags,
            { "Channel Flags", "woww.channel.flags",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_channel_member_flags,
            { "Channel Member Flags", "woww.channel.member.flags",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_channel_name,
            { "Channel Name", "woww.channel.name",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_channel_password,
            { "Channel Password", "woww.channel.password",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_character_flags,
            { "Character Flags", "woww.character.flags",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_character_guid,
            { "Character Guid", "woww.character.guid",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_character_name,
            { "Character Name", "woww.character.name",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_charter_display_id,
            { "Charter Display Id", "woww.charter.display.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_charter_entry,
            { "Charter Entry", "woww.charter.entry",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_charter_owner,
            { "Charter Owner", "woww.charter.owner",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_chat_credit,
            { "Chat Credit", "woww.chat.credit",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_chat_data_line_count,
            { "Chat Data Line Count", "woww.chat.data.line.count",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_chat_notify,
            { "Chat Notify", "woww.chat.notify",
                FT_UINT8, BASE_HEX_DEC, VALS(e_chat_notify_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_chat_type,
            { "Chat Type", "woww.chat.type",
                FT_UINT32, BASE_HEX_DEC, VALS(e_chat_type_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_checked_timestamp,
            { "Checked Timestamp", "woww.checked.timestamp",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cinematic_sequence_id,
            { "Cinematic Sequence Id", "woww.cinematic.sequence.id",
                FT_UINT32, BASE_HEX_DEC, VALS(e_cinematic_sequence_id_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_civilian,
            { "Civilian", "woww.civilian",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_class,
            { "Class", "woww.class",
                FT_UINT32, BASE_HEX_DEC, VALS(e_class_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_class_mask,
            { "Class Mask", "woww.class.mask",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_client_instance_id,
            { "Client Instance Id", "woww.client.instance.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_client_proof,
            { "Client Proof", "woww.client.proof",
                FT_BYTES, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_client_seed,
            { "Client Seed", "woww.client.seed",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_code,
            { "Code", "woww.code",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_coded,
            { "Coded", "woww.coded",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_comment,
            { "Comment", "woww.comment",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_compressed_chat_data,
            { "Compressed Chat Data", "woww.compressed.chat.data",
                FT_BYTES, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_compressed_data,
            { "Compressed Data", "woww.compressed.data",
                FT_BYTES, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_compressed_move_opcode,
            { "Compressed Move Opcode", "woww.compressed.move.opcode",
                FT_UINT16, BASE_HEX_DEC, VALS(e_compressed_move_opcode_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_container_slots,
            { "Container Slots", "woww.container.slots",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_content,
            { "Content", "woww.content",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cooldown,
            { "Cooldown", "woww.cooldown",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cooldown_count,
            { "Cooldown Count", "woww.cooldown.count",
                FT_UINT16, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cooldown_time,
            { "Cooldown Time", "woww.cooldown.time",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_corpse,
            { "Corpse", "woww.corpse",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_corpse_query_result,
            { "Corpse Query Result", "woww.corpse.query.result",
                FT_UINT8, BASE_HEX_DEC, VALS(e_corpse_query_result_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cos_angle,
            { "Cos Angle", "woww.cos.angle",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_cost_in_copper,
            { "Cost In Copper", "woww.cost.in.copper",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_count,
            { "Count", "woww.count",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_countdown_time,
            { "Countdown Time", "woww.countdown.time",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_counter,
            { "Counter", "woww.counter",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_created_day,
            { "Created Day", "woww.created.day",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_created_month,
            { "Created Month", "woww.created.month",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_created_year,
            { "Created Year", "woww.created.year",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_creature,
            { "Creature", "woww.creature",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_creature_entry,
            { "Creature Entry", "woww.creature.entry",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_creature_family,
            { "Creature Family", "woww.creature.family",
                FT_UINT32, BASE_HEX_DEC, VALS(e_creature_family_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_creature_id,
            { "Creature Id", "woww.creature.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_creature_rank,
            { "Creature Rank", "woww.creature.rank",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_creature_type,
            { "Creature Type", "woww.creature.type",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_critical,
            { "Critical", "woww.critical",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_current_health,
            { "Current Health", "woww.current.health",
                FT_UINT16, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_current_power,
            { "Current Power", "woww.current.power",
                FT_UINT16, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_damage,
            { "Damage", "woww.damage",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_damage_float,
            { "Damage Float", "woww.damage.float",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_damage_maximum,
            { "Damage Maximum", "woww.damage.maximum",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_damage_minimum,
            { "Damage Minimum", "woww.damage.minimum",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_damage_state,
            { "Damage State", "woww.damage.state",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_damage_uint,
            { "Damage Uint", "woww.damage.uint",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_data,
            { "Data", "woww.data",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_data_type,
            { "Data Type", "woww.data.type",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_datetime,
            { "Datetime", "woww.datetime",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_days_since_last_updated,
            { "Days Since Last Updated", "woww.days.since.last.updated",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_days_since_oldest_ticket_creation,
            { "Days Since Oldest Ticket Creation", "woww.days.since.oldest.ticket.creation",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_days_since_ticket_creation,
            { "Days Since Ticket Creation", "woww.days.since.ticket.creation",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_deadline,
            { "Deadline", "woww.deadline",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_deaths,
            { "Deaths", "woww.deaths",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_debug_log_format,
            { "Debug Log Format", "woww.debug.log.format",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_decompressed_size,
            { "Decompressed Size", "woww.decompressed.size",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_delay,
            { "Delay", "woww.delay",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_delay_time,
            { "Delay Time", "woww.delay.time",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_description,
            { "Description", "woww.description",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_destination_bag,
            { "Destination Bag", "woww.destination.bag",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_destination_node,
            { "Destination Node", "woww.destination.node",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_destination_slot,
            { "Destination Slot", "woww.destination.slot",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_destionation_slot,
            { "Destionation Slot", "woww.destionation.slot",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_details,
            { "Details", "woww.details",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_dismount_result,
            { "Dismount Result", "woww.dismount.result",
                FT_UINT32, BASE_HEX_DEC, VALS(e_dismount_result_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_display_id,
            { "Display Id", "woww.display.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_duel_winner_reason,
            { "Duel Winner Reason", "woww.duel.winner.reason",
                FT_UINT8, BASE_HEX_DEC, VALS(e_duel_winner_reason_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_durability,
            { "Durability", "woww.durability",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_duration,
            { "Duration", "woww.duration",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_eff,
            { "Eff", "woww.eff",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_emblem_color,
            { "Emblem Color", "woww.emblem.color",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_emblem_style,
            { "Emblem Style", "woww.emblem.style",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_emote_delay,
            { "Emote Delay", "woww.emote.delay",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_emote_enum,
            { "Emote Enum", "woww.emote.enum",
                FT_UINT32, BASE_HEX_DEC, VALS(e_emote_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_emote_int,
            { "Emote Int", "woww.emote.int",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_enable_pvp,
            { "Enable Pvp", "woww.enable.pvp",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_enchantment,
            { "Enchantment", "woww.enchantment",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_encrypted_data,
            { "Encrypted Data", "woww.encrypted.data",
                FT_BYTES, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_end_text,
            { "End Text", "woww.end.text",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_ended_without_interruption,
            { "Ended Without Interruption", "woww.ended.without.interruption",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_enemy,
            { "Enemy", "woww.enemy",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_energize_amount,
            { "Energize Amount", "woww.energize.amount",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_energize_power,
            { "Energize Power", "woww.energize.power",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_energy,
            { "Energy", "woww.energy",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_entry,
            { "Entry", "woww.entry",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_entry_id,
            { "Entry Id", "woww.entry.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_environmental_damage_type,
            { "Environmental Damage Type", "woww.environmental.damage.type",
                FT_UINT8, BASE_HEX_DEC, VALS(e_environmental_damage_type_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_equip_error,
            { "Equip Error", "woww.equip.error",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_equipment_display_id,
            { "Equipment Display Id", "woww.equipment.display.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_equipped_item_class,
            { "Equipped Item Class", "woww.equipped.item.class",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_equipped_item_inventory_type_mask,
            { "Equipped Item Inventory Type Mask", "woww.equipped.item.inventory.type.mask",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_equipped_item_subclass_mask,
            { "Equipped Item Subclass Mask", "woww.equipped.item.subclass.mask",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_event_descriptions,
            { "Event Descriptions", "woww.event.descriptions",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_exp_group_bonus,
            { "Exp Group Bonus", "woww.exp.group.bonus",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_experience,
            { "Experience", "woww.experience",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_experience_award_type,
            { "Experience Award Type", "woww.experience.award.type",
                FT_UINT8, BASE_HEX_DEC, VALS(e_experience_award_type_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_experience_reward,
            { "Experience Reward", "woww.experience.reward",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_experience_without_rested,
            { "Experience Without Rested", "woww.experience.without.rested",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_expiration_time,
            { "Expiration Time", "woww.expiration.time",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_extend_flag,
            { "Extend Flag", "woww.extend.flag",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_extra_attacks,
            { "Extra Attacks", "woww.extra.attacks",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_face,
            { "Face", "woww.face",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_facial_hair,
            { "Facial Hair", "woww.facial.hair",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_faction,
            { "Faction", "woww.faction",
                FT_UINT32, BASE_HEX_DEC, VALS(e_faction_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_faction_flag,
            { "Faction Flag", "woww.faction.flag",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_fall_time,
            { "Fall Time", "woww.fall.time",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_far_sight_operation,
            { "Far Sight Operation", "woww.far.sight.operation",
                FT_UINT8, BASE_HEX_DEC, VALS(e_far_sight_operation_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_feed_pet_item,
            { "Feed Pet Item", "woww.feed.pet.item",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_fields,
            { "Fields", "woww.fields",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_fire_resistance,
            { "Fire Resistance", "woww.fire.resistance",
                FT_INT32, BASE_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_first_bag_display_id,
            { "First Bag Display Id", "woww.first.bag.display.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_first_bag_inventory_id,
            { "First Bag Inventory Id", "woww.first.bag.inventory.id",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_first_login,
            { "First Login", "woww.first.login",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_first_rank,
            { "First Rank", "woww.first.rank",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_flags,
            { "Flags", "woww.flags",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_focus,
            { "Focus", "woww.focus",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_friend_result,
            { "Friend Result", "woww.friend.result",
                FT_UINT8, BASE_HEX_DEC, VALS(e_friend_result_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_friend_status,
            { "Friend Status", "woww.friend.status",
                FT_UINT8, BASE_HEX_DEC, VALS(e_friend_status_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_frost_resistance,
            { "Frost Resistance", "woww.frost.resistance",
                FT_INT32, BASE_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gain_multiplier,
            { "Gain Multiplier", "woww.gain.multiplier",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gameobject,
            { "Gameobject", "woww.gameobject",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gender,
            { "Gender", "woww.gender",
                FT_UINT32, BASE_HEX_DEC, VALS(e_gender_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gift_bag_index,
            { "Gift Bag Index", "woww.gift.bag.index",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gift_slot,
            { "Gift Slot", "woww.gift.slot",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gift_wrapper,
            { "Gift Wrapper", "woww.gift.wrapper",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gm_ticket_escalation_status,
            { "Gm Ticket Escalation Status", "woww.gm.ticket.escalation.status",
                FT_UINT8, BASE_HEX_DEC, VALS(e_gm_ticket_escalation_status_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gm_ticket_queue_status,
            { "Gm Ticket Queue Status", "woww.gm.ticket.queue.status",
                FT_UINT32, BASE_HEX_DEC, VALS(e_gm_ticket_queue_status_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gm_ticket_response,
            { "Gm Ticket Response", "woww.gm.ticket.response",
                FT_UINT32, BASE_HEX_DEC, VALS(e_gm_ticket_response_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gm_ticket_status,
            { "Gm Ticket Status", "woww.gm.ticket.status",
                FT_UINT32, BASE_HEX_DEC, VALS(e_gm_ticket_status_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gm_ticket_status_response,
            { "Gm Ticket Status Response", "woww.gm.ticket.status.response",
                FT_UINT32, BASE_HEX_DEC, VALS(e_gm_ticket_status_response_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gm_ticket_type,
            { "Gm Ticket Type", "woww.gm.ticket.type",
                FT_UINT8, BASE_HEX_DEC, VALS(e_gm_ticket_type_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gold,
            { "Gold", "woww.gold",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_gossip_list_id,
            { "Gossip List Id", "woww.gossip.list.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_grade,
            { "Grade", "woww.grade",
                FT_FLOAT, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_greeting,
            { "Greeting", "woww.greeting",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_group_loot_setting,
            { "Group Loot Setting", "woww.group.loot.setting",
                FT_UINT32, BASE_HEX_DEC, VALS(e_group_loot_setting_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_group_member_online_status,
            { "Group Member Online Status", "woww.group.member.online.status",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_group_number,
            { "Group Number", "woww.group.number",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_group_type,
            { "Group Type", "woww.group.type",
                FT_UINT8, BASE_HEX_DEC, VALS(e_group_type_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_group_update_flags,
            { "Group Update Flags", "woww.group.update.flags",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guid,
            { "Guid", "woww.guid",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guids,
            { "Guids", "woww.guids",
                FT_UINT64, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild,
            { "Guild", "woww.guild",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild_charter_cost,
            { "Guild Charter Cost", "woww.guild.charter.cost",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild_command,
            { "Guild Command", "woww.guild.command",
                FT_UINT32, BASE_HEX_DEC, VALS(e_guild_command_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild_command_result,
            { "Guild Command Result", "woww.guild.command.result",
                FT_UINT32, BASE_HEX_DEC, VALS(e_guild_command_result_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild_emblem_result,
            { "Guild Emblem Result", "woww.guild.emblem.result",
                FT_UINT32, BASE_HEX_DEC, VALS(e_guild_emblem_result_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild_event,
            { "Guild Event", "woww.guild.event",
                FT_UINT8, BASE_HEX_DEC, VALS(e_guild_event_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild_id,
            { "Guild Id", "woww.guild.id",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild_info,
            { "Guild Info", "woww.guild.info",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild_member_status,
            { "Guild Member Status", "woww.guild.member.status",
                FT_UINT8, BASE_HEX_DEC, VALS(e_guild_member_status_strings), 0,
                NULL, HFILL
            }
        },
        { &hf_woww_guild_name,
            { "Guild Name", "woww.guild.name",
                FT_STRINGZ, BASE_NONE, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_hair_color,
            { "Hair Color", "woww.hair.color",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_hair_style,
            { "Hair Style", "woww.hair.style",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_happiness,
            { "Happiness", "woww.happiness",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_has_transport,
            { "Has Transport", "woww.has.transport",
                FT_UINT8, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_heal_amount,
            { "Heal Amount", "woww.heal.amount",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
        { &hf_woww_heal_critical,
            { "Heal Critical", "woww.heal.critical",
                FT_UINT32, BASE_HEX_DEC, NULL, 0,
                NULL, HFILL
            }
        },
      