/*******************************************************************************
* localinput.cpp: Local streams
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: localinput.cpp,v 1.13.4.1 2003/06/04 19:06:14 alexis Exp $
*
* Authors: Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../../core/defs.h"

#include "../../core/core.h"

#include "../../mpeg/mpeg.h"
#include "../../mpeg/ts.h"
#include "../../mpeg/rtp.h"
#include "../../server/program.h"
#include "../../server/buffer.h"
#include "../../server/output.h"
#include "../../server/channel.h"
#include "../../server/broadcast.h"
#include "../../server/request.h"
#include "../../server/input.h"
#include "../../server/tsstreamer.h"

#include "../../mpeg/reader.h"
#include "../../mpeg/converter.h"

#include "localinput.h"

#define MAX_FILES_NUMBER 4096

//------------------------------------------------------------------------------
// Library declaration
//------------------------------------------------------------------------------
#ifdef __PLUGIN__
GENERATE_LIB_ARGS(C_LocalInputModule, handle);
#endif


//------------------------------------------------------------------------------
// Builtin declaration
//------------------------------------------------------------------------------
#ifdef __BUILTIN__
C_Module* NewBuiltin_localinput(handle hLog)
{
  return new C_LocalInputModule(hLog);
}
#endif


/*******************************************************************************
* C_LocalInput class
********************************************************************************
*
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_LocalInput::C_LocalInput(C_Module* pModule,
                           const C_String& strName) : C_Input(pModule,
                                                              strName)
{
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_LocalInput::~C_LocalInput()
{
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
void C_LocalInput::OnInit()
{
  C_Settings m_cSettings;
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);

  // Build the program list

  for(int ui = 1;
       ui <= pApp->GetSetting(GetName()+".ProgramCount", "0").ToInt(); ui++)
  {
    C_String strName = pApp->GetSetting(GetName()+"."+C_String(ui)
		                + ".Name",  C_String("Pgrm") + ui);
    C_String* pStr = new C_String(strName);
    m_vProgramNames.Add(pStr);
    Log(m_hLog, LOG_NOTE, "Added program '" + *pStr+"'");
  }
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
void C_LocalInput::OnDestroy()
{
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_List<C_Program> C_LocalInput::OnGetAvailablePgrms()
{
  C_List<C_Program> cPgrmList;

  for(unsigned int ui = 0; ui < m_vProgramNames.Size(); ui++)
  {
    C_Program* pProgram = new C_Program(/*ui + 1,*/ m_vProgramNames[ui]);
    ASSERT(pProgram);
    cPgrmList.PushEnd(pProgram);
  }

  return cPgrmList;
}


//------------------------------------------------------------------------------
// Start the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnStartStreaming(C_Broadcast* pBroadcast)
{
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);
  ASSERT(pBroadcast);

  // We choose a TS packet buffer able to store up to 3s of stream at 8 Mbits/s
  C_NetList* pTsProvider = new C_NetList(3*3*797);

  // The netlist must be at least of the size of the Reader buffer +
  // the size of the output buffer + 2 to be sure that there are always free
  // packets in it
  const C_Channel* pChannel = pBroadcast->GetChannel();
  ASSERT(pTsProvider->Capacity() - pChannel->GetBuffCapacity() - 2 > 0);
  unsigned int uiSize = pTsProvider->Capacity() -pChannel->GetBuffCapacity()-2;
  C_SyncFifo* pBuffer;

  // Get the type of the program
  unsigned int uiId =
        m_vProgramNames.Find(pBroadcast->GetProgram()->GetName()) + 1;
  ASSERT(uiId > 0);
  C_String strId = GetName() + "." + uiId;
  C_String strType = pApp->GetSetting(strId + ".Type",
                                            "Mpeg2-TS").ToLower();

  C_String strReaderType;
  C_String strConverterType;

  // Specific behaviour depends on the type
  if(strType == "mpeg1-ps")             // Mpeg 1 - Program Stream
  {
    // Update the size of the buffer and create it
    uiSize -= 2;
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "file";
    C_String strFile = m_strFilesPath +
                                   pApp->GetSetting(strId+".FileName", "");
    pBroadcast->SetOption("filename", strFile);

    // Converter configuration
    strConverterType = "ps2ts";
    pBroadcast->SetOption("mpegversion", "1");
    pBroadcast->SetOption("preparse", "1");
  }
  else if(strType == "mpeg2-ps")        // Mpeg 2 - Program Stream
  {
    // Update the size of the buffer and create it
    uiSize -= 2;
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "file";
    C_String strFile = m_strFilesPath +
                       pApp->GetSetting(strId+".FileName", "");
    pBroadcast->SetOption("filename", strFile);

    // Converter configuration
    strConverterType = "ps2ts";
    pBroadcast->SetOption("mpegversion", "2");
    pBroadcast->SetOption("preparse", "1");
  }
  else if(strType == "mpeg2-ts")        // Mpeg 2 - Transport Stream
  {
    // Update the size of the buffer and create it
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "file";
    C_String strFile = m_strFilesPath +
                       pApp->GetSetting(strId+".FileName", "");
    pBroadcast->SetOption("filename", strFile);

    // Converter configuration
    strConverterType = "ts2ts";
  }
  else if(strType == "dvd")             // DVD device (Mpeg2 PS)
  {
    // Update the size of the buffer and create it
    uiSize -= 2;
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "dvd";
    C_String strDevice = pApp->GetSetting(strId+".Device", "");
    pBroadcast->SetOption("device", strDevice);
    C_String strDvdTitle = pApp->GetSetting(strId+".dvdtitle","");
    pBroadcast->SetOption("dvdtitle", strDvdTitle);
    C_String strDvdChapter = pApp->GetSetting(strId+".dvdchapter","");
    pBroadcast->SetOption("dvdchapter", strDvdChapter);

    // Converter configuration
    strConverterType = "ps2ts";
    pBroadcast->SetOption("mpegversion", "2");
    pBroadcast->SetOption("preparse", "0");
  }
  else
  {
    delete pTsProvider;
    throw E_Exception(GEN_ERR, "Unsupported or unknown type : " + strType);
  }

  C_MpegReader* pReader;

  // Create the converter
  C_MpegReaderModule* pReaderModule = (C_MpegReaderModule*)
                                C_Application::GetModuleManager()
                                        ->GetModule("mpegreader",
                                                    strReaderType);

  if(pReaderModule)
  {
    pReader = pReaderModule->NewMpegReader(pBroadcast);
    ASSERT(pReader);
  }
  else
  {
    throw E_Exception(GEN_ERR,
                      "Module mpegreader:" + strConverterType +
                      " not present");
  }

  C_MpegConverter* pConverter;

  // Create the converter
  C_MpegConverterModule* pConverterModule = (C_MpegConverterModule*)
                                C_Application::GetModuleManager()
                                        ->GetModule("mpegconverter",
                                                    strConverterType);

  if(pConverterModule)
  {
    C_MpegConverterConfig cConfig;
    cConfig.m_hLog = m_hLog;
    cConfig.m_pBroadcast = pBroadcast;
    cConfig.m_pReader = pReader;
    cConfig.m_pTsProvider = pTsProvider;
    cConfig.m_pHandler = pBuffer;
    cConfig.m_iInitFill = pBuffer->Capacity();
    cConfig.m_pEventHandler = m_pEventHandler;
    pConverter = pConverterModule->NewMpegConverter(cConfig);
    ASSERT(pConverter);
  }
  else
  {
    throw E_Exception(GEN_ERR,
                      "Module mpegconverter:" + strConverterType +
                      " not present");
  }

  pReader->SetConverter(pConverter);

  // Create the streamer
  C_TsStreamer* pStreamer = new C_TsStreamer(m_hLog, pBroadcast,
                                             pTsProvider, pBuffer,
                                             m_pEventHandler, true, true);
  ASSERT(pStreamer);

  m_cConverters.Add(pBroadcast, pConverter);
  m_cStreamers.Add(pBroadcast, pStreamer);

  try
  {
    pConverter->Create();
    pStreamer->Create();
  }
  catch(E_Exception e)
  {
    pConverter->Stop();
    pStreamer->Stop();

    // Unregister the 2 thread and delete them
    m_cConverters.Delete(pBroadcast);
    m_cStreamers.Delete(pBroadcast);

    throw E_Exception(GEN_ERR, "unable to start streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


//------------------------------------------------------------------------------
// Resume the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnResumeStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // Find the converter that receive the pgrm
  C_MpegConverter* pConverter = m_cConverters.Get(pBroadcast);
  ASSERT(pConverter);

  // Resume the thread
  try
  {
    pConverter->Resume();
  }
  catch(E_Exception e)
  {
    throw E_Exception(GEN_ERR, "Unable to resume streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


//------------------------------------------------------------------------------
// Suspend the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnSuspendStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // Find the converter that receive the pgrm
  C_MpegConverter* pConverter = m_cConverters.Get(pBroadcast);
  ASSERT(pConverter);

  C_String strPgrmName = pBroadcast->GetProgram()->GetName();

  // Suspend the thread
  try
  {
    pConverter->Suspend();
  }
  catch(E_Exception e)
  {
    throw E_Exception(GEN_ERR, "Unable to resume streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


//------------------------------------------------------------------------------
// Stop the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnStopStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // Find the reader and the streamer that receive the pgrm
  C_MpegConverter* pConverter = m_cConverters.Remove(pBroadcast);
  ASSERT(pConverter);
  C_TsStreamer* pStreamer = m_cStreamers.Remove(pBroadcast);
  ASSERT(pStreamer);

  // Stop the threads
  try
  {
    pConverter->Stop();
    delete pConverter;
    pStreamer->Stop();
    delete pStreamer;
  }
  catch(E_Exception e)
  {
    throw E_Exception(GEN_ERR, "Unable to stop streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_LocalInput::OnUpdateProgram(C_String strProgram,
                                     C_String strFileName, C_String strType)
{
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);
  C_String strPgrmCount = pApp->GetSetting("Input.ProgramCount", "0");
  unsigned int uiPgrmCount = strPgrmCount.ToInt();

  unsigned int uiId = m_vProgramNames.Find(strProgram) + 1;
  if (uiId>0)
  {
    Log(m_hLog, LOG_WARN, "The program already exists in the list for this input");
  }

  if(uiPgrmCount)
  {
      uiPgrmCount++;
      C_String strKey = C_String(uiPgrmCount)+ ".Name";
      pApp->SetSettings(strKey,strProgram);
      strKey = C_String(uiPgrmCount)+ ".FileName";
      pApp->SetSettings(strKey, strFileName);
      strKey = C_String(uiPgrmCount)+ ".Type";
      pApp->SetSettings(strKey, strType);
      strKey = C_String("Input.ProgramCount");
      C_String strPgrmCount=C_String(uiPgrmCount);
      pApp->SetSettings(strKey, strPgrmCount);

      // add to m_vProgramNames
      C_String strName = pApp->GetSetting(C_String(uiPgrmCount) + ".Name",
                                                C_String("Pgrm") + uiPgrmCount);
      C_String* pStr = new C_String(strName);
      m_vProgramNames.Add(pStr);
  }
  else
  {
    throw E_Exception(GEN_ERR, "No files defined in vls.cfg");
  }
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_LocalInput::OnDeleteProgram(C_String strProgram)
{
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);
  // Get the type of the program
  unsigned int uiId = m_vProgramNames.Find(strProgram) + 1;
  if (uiId>0)
  {
    // Delete settings from m_cSettings
    C_String strPgrmCount = pApp->GetSetting("Input.ProgramCount", "0");
    unsigned int uiPgrmCount = strPgrmCount.ToInt();

    if(uiPgrmCount)
    {
      C_String strDelKey = C_String(uiId)+ ".Name";
      pApp->DeleteSetting(strDelKey);
      if (pApp->GetSetting(C_String(uiId)+".FileName", "") != "" )
      {
        strDelKey = C_String(uiId)+ ".FileName";
        pApp->DeleteSetting(strDelKey);
      }
      if (pApp->GetSetting(C_String(uiId)+".Device", "") != "" )
      {
        strDelKey = C_String(uiId)+ ".Device";
        pApp->DeleteSetting(strDelKey);
      }
      strDelKey = C_String(uiId)+ ".Type";
      pApp->DeleteSetting(strDelKey);

      // Move all other settings one item down
      for(unsigned int ui = uiId; ui <= uiPgrmCount; ui++)
      {
        C_String strProgram = pApp->GetSetting(C_String(ui+1) + ".Name",C_String("Pgrm") + ui+1);
        C_String strFileName = pApp->GetSetting(C_String(ui+1)+".FileName", "");
        C_String strType = pApp->GetSetting(C_String(ui+1) + ".Type", "Mpeg2-TS");
        C_String strDevice = pApp->GetSetting(C_String(ui+1)+".Device", "");

        // update settings
        C_String strKey = C_String(ui)+ ".Name";
        pApp->SetSettings(strKey,strProgram);
        strKey = C_String(ui)+ ".FileName";
        pApp->SetSettings(strKey, strFileName);
        strKey = C_String(ui)+ ".Device";
        pApp->SetSettings(strKey, strDevice);
        strKey = C_String(ui)+ ".Type";
        pApp->SetSettings(strKey, strType);
      }
      // Lower program count
      uiPgrmCount--;
      strDelKey = C_String("Input.ProgramCount");
      C_String strProgramCount=C_String(uiPgrmCount);
      pApp->SetSettings(strDelKey, strPgrmCount);
      //pApp->SetSettings(strDelKey, C_String(uiPgrmCount));
      m_vProgramNames.Delete(uiId);

      // ** DEBUG **
      Log(m_hLog, LOG_NOTE, "Debugging inputs m_cSettings list.");
      C_String strPgrmCount = pApp->GetSetting("Input.ProgramCount", "0");
      unsigned int uiPgrmCount = strPgrmCount.ToInt();
      for(unsigned int ui = 1; ui <= uiPgrmCount; ui++)
      {
        C_String strProgram = pApp->GetSetting(C_String(ui) + ".Name",C_String("Pgrm") + ui);
        C_String strFileName = pApp->GetSetting(C_String(ui)+".FileName", "");
        C_String strDevice = pApp->GetSetting(C_String(ui)+".Device", "");
        C_String strType = pApp->GetSetting(C_String(ui) + ".Type", "Mpeg2-TS").ToLower();
        Log(m_hLog, LOG_NOTE, "Program: \"" + strProgram +"\"");
        Log(m_hLog, LOG_NOTE, "  FileName: \"" + strFileName+"\"");
        Log(m_hLog, LOG_NOTE, "  Device: \"" + strDevice+"\"");
        Log(m_hLog, LOG_NOTE, "  Type: \"" + strType+"\"");
      }
    }
    else
    {
      throw E_Exception(GEN_ERR,"Deleting program information from input failed");
    }
  }
  else
  {
    throw E_Exception(GEN_ERR, "The program is not known by this input.");
  }
}
