// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;

#include "gzraw.h"
#include "zlib.h"
#include "util.h"

template<class T> FitsGzrawm<T>::FitsGzrawm(FitsFile* fits) 
  : FitsCompressm<T>(fits)
{
  FitsCompressm<T>::uncompress(fits);
}

template <class T> int FitsGzrawm<T>::compressed(T* dest, char* sptr, 
						char* heap,
						int kkstart, int kkstop, 
						int jjstart, int jjstop, 
						int iistart, int iistop)
{
  int icnt=0;
  unsigned char* ibuf = (unsigned char*)((FitsBinColumnArray*)FitsCompressm<T>::compress_)->get(heap, sptr, &icnt);

  // ibuf can be NULL
  if (ibuf && icnt>0) {
    int ocnt = FitsCompressm<T>::ww_ * FitsCompressm<T>::hh_ * FitsCompressm<T>::dd_;
    char obuf[ocnt*sizeof(T)];

    z_stream zstrm;
    zstrm.next_in = NULL;
    zstrm.avail_in = 0;
    zstrm.zalloc = NULL;
    zstrm.zfree = NULL;
    zstrm.opaque = NULL;

    // look for both zlib and gzraw headers
    if (inflateInit2(&zstrm, MAX_WBITS+32) != Z_OK) {
      internalError("Fitsy++ gzraw inflateInit error");
      return 0;
    }

    zstrm.avail_in = icnt;
    zstrm.next_in = ibuf;
    zstrm.avail_out = ocnt*sizeof(T);
    zstrm.next_out = (Bytef*)obuf;

    if (DebugCompress)
      cerr << "  inflate START: avail_in " << zstrm.avail_in
	   << " avail_out " << zstrm.avail_out
	   << " total_in " << zstrm.total_in 
	   << " total_out " << zstrm.total_out << endl;

    int result = ::inflate(&zstrm, Z_FINISH);

    switch (result) {
    case Z_OK:
      if (DebugCompress)
	cerr << "  inflate OK: avail_in " << zstrm.avail_in
	     << " avail_out " << zstrm.avail_out
	     << " total_in " << zstrm.total_in 
	     << " total_out " << zstrm.total_out << endl;
      break;
    case Z_STREAM_END:
      if (DebugCompress)
	cerr << "  inflate STREAM_END: avail_in " << zstrm.avail_in
	     << " avail_out " << zstrm.avail_out
	     << " total_in " << zstrm.total_in 
	     << " total_out " << zstrm.total_out << endl;
      break;
    case Z_BUF_ERROR:
      if (DebugCompress)
	cerr << "  inflate BUF_ERROR: avail_in " << zstrm.avail_in
	     << " avail_out " << zstrm.avail_out << endl;
      return 0;
    default:
      internalError("Fitsy++ gzraw inflate error");
      return 0;
    }

    inflateEnd(&zstrm);

    int ll=0;
    for (int kk=kkstart; kk<kkstop; kk++) {
      for (int jj=jjstart; jj<jjstop; jj++) {
	for (int ii=iistart; ii<iistop; ii++,ll++) {
	  // swap if needed
	  if (FitsCompressm<T>::byteswap_)
	    *((T*)obuf+ll) = swap((T*)obuf+ll);
	  dest[kk*FitsCompressm<T>::width_*FitsCompressm<T>::height_ + jj*FitsCompressm<T>::width_ + ii] = *((T*)obuf+ll);
	}
      }
    }
  }

  return 1;
}

template class FitsGzrawm<float>;
template class FitsGzrawm<double>;

template <> float FitsGzrawm<float>::swap(float* ptr)
{
  const char* p = (const char*)ptr;
  union {
    char c[4];
    float f;
  } u;

  u.c[3] = *p++;
  u.c[2] = *p++;
  u.c[1] = *p++;
  u.c[0] = *p;

  return u.f;
}

template <> double FitsGzrawm<double>::swap(double* ptr)
{
  const char* p = (const char*)ptr;
  union {
    char c[8];
    double d;
  } u;

  u.c[7] = *p++;
  u.c[6] = *p++;
  u.c[5] = *p++;
  u.c[4] = *p++;
  u.c[3] = *p++;
  u.c[2] = *p++;
  u.c[1] = *p++;
  u.c[0] = *p;

  return u.d;
}
