// Test case inputs for: __int128_t __fixtfti (long double)
// Conversion from long double (IBM double-double) to 128 bit integer.

#define INIT_U128(HI, LO) (((__uint128_t) (HI) << 64) | (LO))

struct testCase {
  double hiInput;
  double loInput;
  __uint128_t result128;
};

// This test case tests long doubles generated using the following approach:
// For each possible exponent in double precision, we generate a random number
// in [1,2) interval as a mantissa for the high double. Given that the low
// double must be less than half an ULP of the high double, we choose the
// exponent of the low double to be less than the exponent of the high double
// minus 52. For the given exponent in the low double, we generate a random
// number in [1,2) interval as a mantissa. From this generated long double,
// we generate four long doubles by setting the sign of the high and low
// doubles to + or -.
struct testCase testList[] = {
  { 0x0p+0, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { -0x0p+0, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { -0x0p+0, -0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x0p+0, -0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { -0x1p+0, 0x0p+0, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1p+0, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000001 ) },
  { -0x1p+127, 0x0p+0, INIT_U128( 0x8000000000000000, 0x0000000000000000 ) },
  { 0x1.9a22227b34444p+0, 0x1.05d9d02c0bb3ap-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.be1c2ac37c385p+0, -0x1.2254205c44a84p-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.8b79847916f30p+0, 0x1.515b66aaa2b6dp-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.a322b51546457p+0, -0x1.6d405aa0da80cp-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.4f9b9df49f374p+0, 0x1.a6d9873d4db31p-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.26b4f6de4d69fp+0, -0x1.a6e6262b4dcc5p-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.ed76bc33daed8p+1, 0x1.ac3a776d5874fp-52, INIT_U128( 0x0, 0x3 ) },
  { 0x1.87f39b850fe74p+1, -0x1.fc15cd99f82b9p-52, INIT_U128( 0x0, 0x3 ) },
  { 0x1.c52002bd8a401p+1, 0x1.a783df434f07cp-52, INIT_U128( 0x0, 0x3 ) },
  { 0x1.46a0aef28d416p+1, -0x1.9dbb2fbf3b766p-52, INIT_U128( 0x0, 0x2 ) },
  { 0x1.421e4dee843cap+1, 0x1.97155e672e2acp-52, INIT_U128( 0x0, 0x2 ) },
  { 0x1.a51ea2714a3d4p+1, -0x1.0c7e935c18fd2p-52, INIT_U128( 0x0, 0x3 ) },
  { 0x1.0fef3d221fde8p+2, 0x1.bea2cd2b7d45ap-51, INIT_U128( 0x0, 0x4 ) },
  { 0x1.f9918603f3231p+2, -0x1.18de594e31bcbp-51, INIT_U128( 0x0, 0x7 ) },
  { 0x1.37ae92126f5d2p+2, 0x1.1fba676c3f74dp-51, INIT_U128( 0x0, 0x4 ) },
  { 0x1.0136db2e026dcp+2, -0x1.d1d0b48fa3a16p-51, INIT_U128( 0x0, 0x4 ) },
  { 0x1.2594379a4b287p+2, 0x1.bf5305f37ea61p-51, INIT_U128( 0x0, 0x4 ) },
  { 0x1.f424d4ede849ap+2, -0x1.b10dbc15621b8p-51, INIT_U128( 0x0, 0x7 ) },
  { 0x1.10ae6c9c215cep+3, 0x1.97478a012e8f1p-50, INIT_U128( 0x0, 0x8 ) },
  { 0x1.1e4e16b63c9c3p+3, -0x1.604a0a92c0942p-50, INIT_U128( 0x0, 0x8 ) },
  { 0x1.3e21113e7c422p+3, 0x1.45030d948a062p-50, INIT_U128( 0x0, 0x9 ) },
  { 0x1.073030be0e606p+3, -0x1.2b97cd72572fap-50, INIT_U128( 0x0, 0x8 ) },
  { 0x1.f13d8503e27b0p+3, 0x1.8f23cc451e47ap-50, INIT_U128( 0x0, 0xf ) },
  { 0x1.27ad9b0c4f5b4p+3, -0x1.b65d87c16cbb1p-50, INIT_U128( 0x0, 0x9 ) },
  { 0x1.36a7ab866d4f6p+4, 0x1.6e4e4a00dc9cap-49, INIT_U128( 0x0, 0x13 ) },
  { 0x1.5418b780a8317p+4, -0x1.78c5f976f18bfp-49, INIT_U128( 0x0, 0x15 ) },
  { 0x1.0563a8680ac75p+4, 0x1.f93759f3f26ebp-49, INIT_U128( 0x0, 0x10 ) },
  { 0x1.85448cd50a892p+4, -0x1.1552e58c2aa5cp-49, INIT_U128( 0x0, 0x18 ) },
  { 0x1.b7da271d6fb45p+4, 0x1.7ed612d8fdac2p-49, INIT_U128( 0x0, 0x1b ) },
  { 0x1.2e313edc5c628p+4, -0x1.ef88b931df117p-49, INIT_U128( 0x0, 0x12 ) },
  { 0x1.9e907a473d20fp+5, 0x1.2f681a0a5ed04p-48, INIT_U128( 0x0, 0x33 ) },
  { 0x1.c1596dc182b2ep+5, -0x1.c424b1e388496p-48, INIT_U128( 0x0, 0x38 ) },
  { 0x1.eebb1fffdd764p+5, 0x1.b57e42af6afc8p-48, INIT_U128( 0x0, 0x3d ) },
  { 0x1.75326df6ea64ep+5, -0x1.ff698c9dfed31p-48, INIT_U128( 0x0, 0x2e ) },
  { 0x1.4cc5ccb0998bap+5, 0x1.938084af27010p-48, INIT_U128( 0x0, 0x29 ) },
  { 0x1.5fb39730bf673p+5, -0x1.8ae6ddf115cdcp-48, INIT_U128( 0x0, 0x2b ) },
  { 0x1.d8839245b1072p+6, 0x1.66e3c7f0cdc79p-47, INIT_U128( 0x0, 0x76 ) },
  { 0x1.b67eab656cfd6p+6, -0x1.c9c1174993823p-47, INIT_U128( 0x0, 0x6d ) },
  { 0x1.f6079435ec0f2p+6, 0x1.de2c4673bc589p-47, INIT_U128( 0x0, 0x7d ) },
  { 0x1.801e9f09003d4p+6, -0x1.2caf3fbc595e8p-47, INIT_U128( 0x0, 0x60 ) },
  { 0x1.4c7a907498f52p+6, 0x1.d437568ba86ebp-47, INIT_U128( 0x0, 0x53 ) },
  { 0x1.a57539354aea7p+6, -0x1.51de38dea3bc7p-47, INIT_U128( 0x0, 0x69 ) },
  { 0x1.cd6eb07d9add6p+7, 0x1.2ab9c60a55739p-46, INIT_U128( 0x0, 0xe6 ) },
  { 0x1.67311e6cce624p+7, -0x1.8548d79b0a91bp-46, INIT_U128( 0x0, 0xb3 ) },
  { 0x1.a98fd8e5531fbp+7, 0x1.95625d492ac4cp-46, INIT_U128( 0x0, 0xd4 ) },
  { 0x1.74d190cce9a32p+7, -0x1.4820532c9040ap-46, INIT_U128( 0x0, 0xba ) },
  { 0x1.c4ca77558994fp+7, 0x1.362afe8e6c560p-46, INIT_U128( 0x0, 0xe2 ) },
  { 0x1.0c450510188a0p+7, -0x1.b3859755670b3p-46, INIT_U128( 0x0, 0x86 ) },
  { 0x1.eba9413fd7528p+8, 0x1.8177730902eeep-45, INIT_U128( 0x0, 0x1eb ) },
  { 0x1.b2ef23f165de4p+8, -0x1.386f0dee70de2p-45, INIT_U128( 0x0, 0x1b2 ) },
  { 0x1.14f7f7b629effp+8, 0x1.12a2bb2225458p-45, INIT_U128( 0x0, 0x114 ) },
  { 0x1.c355ebe186abep+8, -0x1.ed7b89fddaf71p-45, INIT_U128( 0x0, 0x1c3 ) },
  { 0x1.79868312f30d0p+8, 0x1.79d49744f3a93p-45, INIT_U128( 0x0, 0x179 ) },
  { 0x1.b73ca1456e794p+8, -0x1.6f8e1748df1c3p-45, INIT_U128( 0x0, 0x1b7 ) },
  { 0x1.5800ddb8b001cp+9, 0x1.375b404e6eb68p-44, INIT_U128( 0x0, 0x2b0 ) },
  { 0x1.9c3d5707387abp+9, -0x1.a47b6e2d48f6ep-44, INIT_U128( 0x0, 0x338 ) },
  { 0x1.0b9296da17253p+9, 0x1.41b2903c83652p-44, INIT_U128( 0x0, 0x217 ) },
  { 0x1.f659a70decb35p+9, -0x1.d9014759b2029p-44, INIT_U128( 0x0, 0x3ec ) },
  { 0x1.6c6c5648d8d8bp+9, 0x1.5d8ac578bb158p-44, INIT_U128( 0x0, 0x2d8 ) },
  { 0x1.5e4a1ef4bc944p+9, -0x1.15fc2f2e2bf86p-44, INIT_U128( 0x0, 0x2bc ) },
  { 0x1.f2ed22c3e5da5p+10, 0x1.d31bafc3a6376p-43, INIT_U128( 0x0, 0x7cb ) },
  { 0x1.5e1c825cbc390p+10, -0x1.b7b79a196f6f3p-43, INIT_U128( 0x0, 0x578 ) },
  { 0x1.0946ae52128d6p+10, 0x1.6a61876ed4c31p-43, INIT_U128( 0x0, 0x425 ) },
  { 0x1.6b3d316ed67a6p+10, -0x1.f4efef91e9dfep-43, INIT_U128( 0x0, 0x5ac ) },
  { 0x1.ebd8fa77d7b20p+10, 0x1.1e2c55603c58ap-43, INIT_U128( 0x0, 0x7af ) },
  { 0x1.2d5efd805abe0p+10, -0x1.00d0f22e01a1ep-43, INIT_U128( 0x0, 0x4b5 ) },
  { 0x1.30cf1082619e2p+11, 0x1.f0b4e963e169dp-42, INIT_U128( 0x0, 0x986 ) },
  { 0x1.ee4c7c2ddc990p+11, -0x1.aa55939754ab2p-42, INIT_U128( 0x0, 0xf72 ) },
  { 0x1.6a8980a8d5130p+11, 0x1.a5891d734b124p-42, INIT_U128( 0x0, 0xb54 ) },
  { 0x1.19e3aae033c76p+11, -0x1.c28a5e618514cp-42, INIT_U128( 0x0, 0x8cf ) },
  { 0x1.dce45dc3b9c8cp+11, 0x1.b62101e96c420p-42, INIT_U128( 0x0, 0xee7 ) },
  { 0x1.6705cc8ece0bap+11, -0x1.e876f49dd0edep-42, INIT_U128( 0x0, 0xb38 ) },
  { 0x1.f7d39227efa73p+12, 0x1.5d5f739cbabeep-41, INIT_U128( 0x0, 0x1f7d ) },
  { 0x1.c1783bf782f08p+12, -0x1.643fe870c87fdp-41, INIT_U128( 0x0, 0x1c17 ) },
  { 0x1.baea391775d47p+12, 0x1.1fc25aea3f84cp-41, INIT_U128( 0x0, 0x1bae ) },
  { 0x1.7a3c17a0f4783p+12, -0x1.add75bdb5baecp-41, INIT_U128( 0x0, 0x17a3 ) },
  { 0x1.f2de0aafe5bc2p+12, 0x1.d85899ffb0b13p-41, INIT_U128( 0x0, 0x1f2d ) },
  { 0x1.658ef5f4cb1dep+12, -0x1.fc24b901f8497p-41, INIT_U128( 0x0, 0x1658 ) },
  { 0x1.42ba216e85744p+13, 0x1.d0cc2763a1985p-40, INIT_U128( 0x0, 0x2857 ) },
  { 0x1.f114a893e2295p+13, -0x1.2e931be65d264p-40, INIT_U128( 0x0, 0x3e22 ) },
  { 0x1.eef87cc5ddf0fp+13, 0x1.1e73a6143ce75p-40, INIT_U128( 0x0, 0x3ddf ) },
  { 0x1.2f640e165ec82p+13, -0x1.1fc78d483f8f2p-40, INIT_U128( 0x0, 0x25ec ) },
  { 0x1.0f288f781e512p+13, 0x1.1db08a423b612p-40, INIT_U128( 0x0, 0x21e5 ) },
  { 0x1.ca1d20e1943a4p+13, -0x1.79b21020f3642p-40, INIT_U128( 0x0, 0x3943 ) },
  { 0x1.c339a7b786735p+14, 0x1.e400cbc3c801ap-39, INIT_U128( 0x0, 0x70ce ) },
  { 0x1.3c3c6e827878ep+14, -0x1.a01e0579403c1p-39, INIT_U128( 0x0, 0x4f0f ) },
  { 0x1.9a21cdad3443ap+14, 0x1.a3423b1346848p-39, INIT_U128( 0x0, 0x6688 ) },
  { 0x1.8894fa171129fp+14, -0x1.91f55fd523eacp-39, INIT_U128( 0x0, 0x6225 ) },
  { 0x1.00a1aeb001436p+14, 0x1.4a0d360e941a7p-39, INIT_U128( 0x0, 0x4028 ) },
  { 0x1.f6354847ec6a9p+14, -0x1.bb8be7c77717dp-39, INIT_U128( 0x0, 0x7d8d ) },
  { 0x1.69cf90fad39f2p+15, 0x1.70715fe8e0e2cp-38, INIT_U128( 0x0, 0xb4e7 ) },
  { 0x1.696343a2d2c68p+15, -0x1.5cad9ba8b95b4p-38, INIT_U128( 0x0, 0xb4b1 ) },
  { 0x1.48c9f9e89193fp+15, 0x1.cef239979de47p-38, INIT_U128( 0x0, 0xa464 ) },
  { 0x1.58d254aab1a4ap+15, -0x1.8b841b5717084p-38, INIT_U128( 0x0, 0xac69 ) },
  { 0x1.9fbe60fb3f7ccp+15, 0x1.49e6fd3e93ce0p-38, INIT_U128( 0x0, 0xcfdf ) },
  { 0x1.ba17373f742e7p+15, -0x1.8eb6ddc11d6dcp-38, INIT_U128( 0x0, 0xdd0b ) },
  { 0x1.685aa354d0b54p+16, 0x1.e97b4555d2f68p-37, INIT_U128( 0x0, 0x1685a ) },
  { 0x1.ae7e6b095cfcep+16, -0x1.778332ceef066p-37, INIT_U128( 0x0, 0x1ae7e ) },
  { 0x1.072bcf980e57ap+16, 0x1.789b8beaf1372p-37, INIT_U128( 0x0, 0x1072b ) },
  { 0x1.f13542f1e26a9p+16, -0x1.ca65775194cafp-37, INIT_U128( 0x0, 0x1f135 ) },
  { 0x1.5c9c57acb938bp+16, 0x1.2d71b1745ae36p-37, INIT_U128( 0x0, 0x15c9c ) },
  { 0x1.6a837ccad5070p+16, -0x1.f6be255bed7c4p-37, INIT_U128( 0x0, 0x16a83 ) },
  { 0x1.d0b3ec89a167dp+17, 0x1.f31876e3e630fp-36, INIT_U128( 0x0, 0x3a167 ) },
  { 0x1.88b3852f11671p+17, -0x1.d1559b39a2ab4p-36, INIT_U128( 0x0, 0x31167 ) },
  { 0x1.3d25acf07a4b6p+17, 0x1.7ee8cd76fdd1ap-36, INIT_U128( 0x0, 0x27a4b ) },
  { 0x1.8ce1830119c30p+17, -0x1.a6f53ded4dea8p-36, INIT_U128( 0x0, 0x319c3 ) },
  { 0x1.ef5a2d5fdeb45p+17, 0x1.f644c6a5ec899p-36, INIT_U128( 0x0, 0x3deb4 ) },
  { 0x1.03a2d9960745bp+17, -0x1.3824274870485p-36, INIT_U128( 0x0, 0x20745 ) },
  { 0x1.c537003d8a6e0p+18, 0x1.5c752e42b8ea6p-35, INIT_U128( 0x0, 0x714dc ) },
  { 0x1.c3a842a187509p+18, -0x1.64f56138c9eacp-35, INIT_U128( 0x0, 0x70ea1 ) },
  { 0x1.a65c2b834cb86p+18, 0x1.7dbf79defb7efp-35, INIT_U128( 0x0, 0x69970 ) },
  { 0x1.f3ac3ef9e7588p+18, -0x1.0eb6ced81d6dap-35, INIT_U128( 0x0, 0x7ceb0 ) },
  { 0x1.673abf34ce758p+18, 0x1.a22aaa0344555p-35, INIT_U128( 0x0, 0x59cea ) },
  { 0x1.863a01a70c740p+18, -0x1.6e668ecedccd2p-35, INIT_U128( 0x0, 0x618e8 ) },
  { 0x1.1ac635ee358c6p+19, 0x1.776f4498eede8p-34, INIT_U128( 0x0, 0x8d631 ) },
  { 0x1.38cc062071981p+19, -0x1.77628d9ceec52p-34, INIT_U128( 0x0, 0x9c660 ) },
  { 0x1.a4e0f60d49c1fp+19, 0x1.a34f82eb469f0p-34, INIT_U128( 0x0, 0xd2707 ) },
  { 0x1.6e20b2f6dc416p+19, -0x1.7594e296eb29cp-34, INIT_U128( 0x0, 0xb7105 ) },
  { 0x1.9667f9cf2ccffp+19, 0x1.f6d43827eda87p-34, INIT_U128( 0x0, 0xcb33f ) },
  { 0x1.46ea7dd08dd50p+19, -0x1.d38da6c1a71b5p-34, INIT_U128( 0x0, 0xa3753 ) },
  { 0x1.75bd059eeb7a0p+20, 0x1.d3cc00d3a7980p-33, INIT_U128( 0x0, 0x175bd0 ) },
  { 0x1.7a82e4acf505cp+20, -0x1.651732feca2e6p-33, INIT_U128( 0x0, 0x17a82e ) },
  { 0x1.ca10c77794219p+20, 0x1.7e25be44fc4b8p-33, INIT_U128( 0x0, 0x1ca10c ) },
  { 0x1.9d3b5a473a76bp+20, -0x1.8f47c7b51e8f9p-33, INIT_U128( 0x0, 0x19d3b5 ) },
  { 0x1.da9aec87b535dp+20, 0x1.60e39ce8c1c74p-33, INIT_U128( 0x0, 0x1da9ae ) },
  { 0x1.f43d28ede87a5p+20, -0x1.80f7180b01ee3p-33, INIT_U128( 0x0, 0x1f43d2 ) },
  { 0x1.dea316c9bd463p+21, 0x1.e7e967e7cfd2dp-32, INIT_U128( 0x0, 0x3bd462 ) },
  { 0x1.8fade6e31f5bdp+21, -0x1.45ddd5fe8bbbap-32, INIT_U128( 0x0, 0x31f5bc ) },
  { 0x1.8a1411eb14282p+21, 0x1.15e7e88a2bcfdp-32, INIT_U128( 0x0, 0x314282 ) },
  { 0x1.b44464cf6888cp+21, -0x1.a4da47db49b49p-32, INIT_U128( 0x0, 0x36888c ) },
  { 0x1.8d572d251aae6p+21, 0x1.ed106aefda20ep-32, INIT_U128( 0x0, 0x31aae5 ) },
  { 0x1.1c3cd5763879ap+21, -0x1.3238233264704p-32, INIT_U128( 0x0, 0x23879a ) },
  { 0x1.905feddb20bfep+22, 0x1.8a5d886f14bb1p-31, INIT_U128( 0x0, 0x6417fb ) },
  { 0x1.a7cd05c94f9a1p+22, -0x1.26c7e7aa4d8fdp-31, INIT_U128( 0x0, 0x69f341 ) },
  { 0x1.8131c09702638p+22, 0x1.eeb7839bdd6f1p-31, INIT_U128( 0x0, 0x604c70 ) },
  { 0x1.fa558e6bf4ab2p+22, -0x1.f6eec637eddd9p-31, INIT_U128( 0x0, 0x7e9563 ) },
  { 0x1.c283846d85070p+22, 0x1.74d87318e9b0ep-31, INIT_U128( 0x0, 0x70a0e1 ) },
  { 0x1.5c05f934b80bfp+22, -0x1.8c944be51928ap-31, INIT_U128( 0x0, 0x57017e ) },
  { 0x1.f2fe315fe5fc6p+23, 0x1.f941537df282bp-30, INIT_U128( 0x0, 0xf97f18 ) },
  { 0x1.1bdc139a37b82p+23, -0x1.cd991de79b324p-30, INIT_U128( 0x0, 0x8dee09 ) },
  { 0x1.5a24f066b449ep+23, 0x1.10193c8820328p-30, INIT_U128( 0x0, 0xad1278 ) },
  { 0x1.4e9035f09d206p+23, -0x1.38c8072e71901p-30, INIT_U128( 0x0, 0xa7481a ) },
  { 0x1.f802a471f0054p+23, 0x1.de3e1cd5bc7c3p-30, INIT_U128( 0x0, 0xfc0152 ) },
  { 0x1.7ea5dd2efd4bcp+23, -0x1.8216b65d042d7p-30, INIT_U128( 0x0, 0xbf52ee ) },
  { 0x1.f4260561e84c0p+24, 0x1.32ea41fc65d48p-29, INIT_U128( 0x0, 0x1f42605 ) },
  { 0x1.26f0fe364de20p+24, -0x1.817d328702fa6p-29, INIT_U128( 0x0, 0x126f0fe ) },
  { 0x1.1ffaacec3ff56p+24, 0x1.432dd3ae865bap-29, INIT_U128( 0x0, 0x11ffaac ) },
  { 0x1.b3b555b9676abp+24, -0x1.b05a9ac960b53p-29, INIT_U128( 0x0, 0x1b3b555 ) },
  { 0x1.cd8b5c1f9b16cp+24, 0x1.5b825b82b704cp-29, INIT_U128( 0x0, 0x1cd8b5c ) },
  { 0x1.d7228275ae450p+24, -0x1.4ebd1cb69d7a4p-29, INIT_U128( 0x0, 0x1d72282 ) },
  { 0x1.4a39df1e9473cp+25, 0x1.d17810e1a2f02p-28, INIT_U128( 0x0, 0x29473be ) },
  { 0x1.45abe2ee8b57cp+25, -0x1.0cbd638c197acp-28, INIT_U128( 0x0, 0x28b57c5 ) },
  { 0x1.465083fa8ca10p+25, 0x1.a1b1dc254363cp-28, INIT_U128( 0x0, 0x28ca107 ) },
  { 0x1.9951af7332a36p+25, -0x1.394eb558729d6p-28, INIT_U128( 0x0, 0x332a35e ) },
  { 0x1.9af32fa335e66p+25, 0x1.b52dc30d6a5b8p-28, INIT_U128( 0x0, 0x335e65f ) },
  { 0x1.06e7d3400dcfap+25, -0x1.f5182009ea304p-28, INIT_U128( 0x0, 0x20dcfa6 ) },
  { 0x1.c546ff978a8e0p+26, 0x1.55780932aaf01p-27, INIT_U128( 0x0, 0x7151bfe ) },
  { 0x1.2de027385bc05p+26, -0x1.bd49a4077a934p-27, INIT_U128( 0x0, 0x4b7809c ) },
  { 0x1.33b62608676c5p+26, 0x1.ac860469590c0p-27, INIT_U128( 0x0, 0x4ced898 ) },
  { 0x1.3f605a647ec0cp+26, -0x1.27dc2b6a4fb86p-27, INIT_U128( 0x0, 0x4fd8169 ) },
  { 0x1.df704667bee09p+26, 0x1.6651589ccca2bp-27, INIT_U128( 0x0, 0x77dc119 ) },
  { 0x1.a9595df152b2cp+26, -0x1.ff6af355fed5fp-27, INIT_U128( 0x0, 0x6a56577 ) },
  { 0x1.a97f2c4d52fe6p+27, 0x1.5d0eb3aaba1d6p-26, INIT_U128( 0x0, 0xd4bf962 ) },
  { 0x1.afc58e1f5f8b2p+27, -0x1.770a886cee151p-26, INIT_U128( 0x0, 0xd7e2c70 ) },
  { 0x1.b8910f2f71222p+27, 0x1.86e810930dd02p-26, INIT_U128( 0x0, 0xdc48879 ) },
  { 0x1.c5c7a5258b8f4p+27, -0x1.02caee4e0595ep-26, INIT_U128( 0x0, 0xe2e3d29 ) },
  { 0x1.58602174b0c04p+27, 0x1.0b73881216e71p-26, INIT_U128( 0x0, 0xac3010b ) },
  { 0x1.0f8851da1f10ap+27, -0x1.9eb0a56f3d615p-26, INIT_U128( 0x0, 0x87c428e ) },
  { 0x1.ad2564395a4acp+28, 0x1.a8da014f51b40p-25, INIT_U128( 0x0, 0x1ad25643 ) },
  { 0x1.d7ff5ac9affecp+28, -0x1.481893da90312p-25, INIT_U128( 0x0, 0x1d7ff5ac ) },
  { 0x1.5e07c45cbc0f8p+28, 0x1.dee72e0bbdce6p-25, INIT_U128( 0x0, 0x15e07c45 ) },
  { 0x1.3d30b3e47a616p+28, -0x1.234ec3f2469d8p-25, INIT_U128( 0x0, 0x13d30b3e ) },
  { 0x1.345f373c68be7p+28, 0x1.6b10db1ed621cp-25, INIT_U128( 0x0, 0x1345f373 ) },
  { 0x1.e09867fbc130dp+28, -0x1.68155d46d02acp-25, INIT_U128( 0x0, 0x1e09867f ) },
  { 0x1.1796df882f2dcp+29, 0x1.1ff8c2603ff18p-24, INIT_U128( 0x0, 0x22f2dbf1 ) },
  { 0x1.8ff188b11fe31p+29, -0x1.f70725efee0e5p-24, INIT_U128( 0x0, 0x31fe3116 ) },
  { 0x1.864a1afd0c943p+29, 0x1.394ae6a27295dp-24, INIT_U128( 0x0, 0x30c9435f ) },
  { 0x1.f6721945ece43p+29, -0x1.2fc9474a5f929p-24, INIT_U128( 0x0, 0x3ece4328 ) },
  { 0x1.96599a4f2cb33p+29, 0x1.a637a7634c6f5p-24, INIT_U128( 0x0, 0x32cb3349 ) },
  { 0x1.a6d3cfb74da7ap+29, -0x1.4eeaa97e9dd55p-24, INIT_U128( 0x0, 0x34da79f6 ) },
  { 0x1.2d1fb7f65a3f7p+30, 0x1.6c9b74dcd936ep-23, INIT_U128( 0x0, 0x4b47edfd ) },
  { 0x1.51b597cea36b3p+30, -0x1.857fd96d0affbp-23, INIT_U128( 0x0, 0x546d65f3 ) },
  { 0x1.780efcf0f01e0p+30, 0x1.613de9aac27bdp-23, INIT_U128( 0x0, 0x5e03bf3c ) },
  { 0x1.24307edc48610p+30, -0x1.253461284a68cp-23, INIT_U128( 0x0, 0x490c1fb7 ) },
  { 0x1.0a17eaf8142fep+30, 0x1.fc2fd9a7f85fbp-23, INIT_U128( 0x0, 0x4285fabe ) },
  { 0x1.6d4cf504da99ep+30, -0x1.681faf56d03f6p-23, INIT_U128( 0x0, 0x5b533d41 ) },
  { 0x1.db1707b1b62e1p+31, 0x1.aa7da8eb54fb5p-22, INIT_U128( 0x0, 0xed8b83d8 ) },
  { 0x1.8b6e734b16dcep+31, -0x1.0f763eee1eec8p-22, INIT_U128( 0x0, 0xc5b739a5 ) },
  { 0x1.158e840a2b1d0p+31, 0x1.6c06db46d80dcp-22, INIT_U128( 0x0, 0x8ac74205 ) },
  { 0x1.778b9c36ef174p+31, -0x1.2d2563945a4acp-22, INIT_U128( 0x0, 0xbbc5ce1b ) },
  { 0x1.8bff9da717ff4p+31, 0x1.077d24e80efa4p-22, INIT_U128( 0x0, 0xc5ffced3 ) },
  { 0x1.242c898448591p+31, -0x1.1249b8a424937p-22, INIT_U128( 0x0, 0x921644c2 ) },
  { 0x1.40c80b8c81902p+32, 0x1.1a9ea2c8353d4p-21, INIT_U128( 0x0, 0x140c80b8c ) },
  { 0x1.4212fff084260p+32, -0x1.d721a5ddae435p-21, INIT_U128( 0x0, 0x14212fff0 ) },
  { 0x1.c0d64a2d81ac9p+32, 0x1.e8e6cfa9d1cdap-21, INIT_U128( 0x0, 0x1c0d64a2d ) },
  { 0x1.260c4f5c4c18ap+32, -0x1.edf60225dbec0p-21, INIT_U128( 0x0, 0x1260c4f5c ) },
  { 0x1.55824484ab048p+32, 0x1.77c1e72aef83dp-21, INIT_U128( 0x0, 0x155824484 ) },
  { 0x1.7375db92e6ebcp+32, -0x1.cd9141779b228p-21, INIT_U128( 0x0, 0x17375db92 ) },
  { 0x1.d8ffd0d7b1ffap+33, 0x1.97a65a6f2f4cbp-20, INIT_U128( 0x0, 0x3b1ffa1af ) },
  { 0x1.f2c5a2a1e58b5p+33, -0x1.e317bbd1c62f8p-20, INIT_U128( 0x0, 0x3e58b4543 ) },
  { 0x1.93d6341727ac6p+33, 0x1.601297a6c0253p-20, INIT_U128( 0x0, 0x327ac682e ) },
  { 0x1.4b376b26966eep+33, -0x1.6eed3bc8ddda8p-20, INIT_U128( 0x0, 0x2966ed64d ) },
  { 0x1.0a90409215208p+33, 0x1.d6a5de73ad4bcp-20, INIT_U128( 0x0, 0x215208124 ) },
  { 0x1.7e637ab8fcc70p+33, -0x1.201ec49e403d8p-20, INIT_U128( 0x0, 0x2fcc6f571 ) },
  { 0x1.42acb9b885597p+34, 0x1.f5ef16efebde3p-19, INIT_U128( 0x0, 0x50ab2e6e2 ) },
  { 0x1.4d4428329a885p+34, -0x1.ceac57619d58bp-19, INIT_U128( 0x0, 0x53510a0ca ) },
  { 0x1.fd6c3bd9fad88p+34, 0x1.7170b1b8e2e16p-19, INIT_U128( 0x0, 0x7f5b0ef67 ) },
  { 0x1.e0f61007c1ec2p+34, -0x1.3e3447647c689p-19, INIT_U128( 0x0, 0x783d8401f ) },
  { 0x1.8c0f5483181eap+34, 0x1.23090ff846122p-19, INIT_U128( 0x0, 0x6303d520c ) },
  { 0x1.1555ed422aabep+34, -0x1.4ce59eb099cb4p-19, INIT_U128( 0x0, 0x45557b508 ) },
  { 0x1.0befdfa017dfcp+35, 0x1.3b623c9e76c48p-18, INIT_U128( 0x0, 0x85f7efd00 ) },
  { 0x1.3adba0d675b74p+35, -0x1.61722426c2e44p-18, INIT_U128( 0x0, 0x9d6dd06b3 ) },
  { 0x1.b32423cb66484p+35, 0x1.ae517c695ca30p-18, INIT_U128( 0x0, 0xd99211e5b ) },
  { 0x1.2f03659e5e06cp+35, -0x1.8c23f4a31847ep-18, INIT_U128( 0x0, 0x9781b2cf2 ) },
  { 0x1.45f7d6d88befbp+35, 0x1.21c9083e43921p-18, INIT_U128( 0x0, 0xa2fbeb6c4 ) },
  { 0x1.6f3bb072de776p+35, -0x1.d9ce0af1b39c2p-18, INIT_U128( 0x0, 0xb79dd8396 ) },
  { 0x1.81db9fe703b74p+36, 0x1.ba731b9b74e64p-17, INIT_U128( 0x0, 0x181db9fe70 ) },
  { 0x1.48bb46b891769p+36, -0x1.8c6be07918d7cp-17, INIT_U128( 0x0, 0x148bb46b89 ) },
  { 0x1.9118fa0f2231fp+36, 0x1.c71ec1738e3d8p-17, INIT_U128( 0x0, 0x19118fa0f2 ) },
  { 0x1.ac8aa51959155p+36, -0x1.d3c5b3c9a78b6p-17, INIT_U128( 0x0, 0x1ac8aa5195 ) },
  { 0x1.bd0dd8637a1bbp+36, 0x1.f9cecd4df39d9p-17, INIT_U128( 0x0, 0x1bd0dd8637 ) },
  { 0x1.a2720b2b44e42p+36, -0x1.ba7a143b74f42p-17, INIT_U128( 0x0, 0x1a2720b2b4 ) },
  { 0x1.6b3f7f04d67f0p+37, 0x1.83681b3306d04p-16, INIT_U128( 0x0, 0x2d67efe09a ) },
  { 0x1.f8ae5ba1f15ccp+37, -0x1.f74ed389ee9dbp-16, INIT_U128( 0x0, 0x3f15cb743e ) },
  { 0x1.a6712b2b4ce26p+37, 0x1.0e0448e41c089p-16, INIT_U128( 0x0, 0x34ce256569 ) },
  { 0x1.443472388868ep+37, -0x1.b2610acd64c21p-16, INIT_U128( 0x0, 0x28868e4711 ) },
  { 0x1.4966858492cd0p+37, 0x1.7f101310fe202p-16, INIT_U128( 0x0, 0x292cd0b092 ) },
  { 0x1.13d32ef427a66p+37, -0x1.fc8aa951f9155p-16, INIT_U128( 0x0, 0x227a65de84 ) },
  { 0x1.2d618f7a5ac32p+38, 0x1.42d6949485ad2p-15, INIT_U128( 0x0, 0x4b5863de96 ) },
  { 0x1.88ae9df3115d4p+38, -0x1.73596654e6b2dp-15, INIT_U128( 0x0, 0x622ba77cc4 ) },
  { 0x1.649f9cd4c93f4p+38, 0x1.f7ea4e67efd4ap-15, INIT_U128( 0x0, 0x5927e73532 ) },
  { 0x1.4d2771449a4eep+38, -0x1.e67b3cbbccf67p-15, INIT_U128( 0x0, 0x5349dc5126 ) },
  { 0x1.b25d6a0364badp+38, 0x1.95e2b7ff2bc57p-15, INIT_U128( 0x0, 0x6c975a80d9 ) },
  { 0x1.72e71d34e5ce4p+38, -0x1.4376641286eccp-15, INIT_U128( 0x0, 0x5cb9c74d39 ) },
  { 0x1.5d017972ba02fp+39, 0x1.1f71c24e3ee38p-14, INIT_U128( 0x0, 0xae80bcb95d ) },
  { 0x1.173c17da2e783p+39, -0x1.1a0f51d4341eap-14, INIT_U128( 0x0, 0x8b9e0bed17 ) },
  { 0x1.19408f9a32812p+39, 0x1.b9ae2307735c4p-14, INIT_U128( 0x0, 0x8ca047cd19 ) },
  { 0x1.d4896611a912dp+39, -0x1.84800ea109002p-14, INIT_U128( 0x0, 0xea44b308d4 ) },
  { 0x1.3d163e387a2c8p+39, 0x1.cd37c1b39a6f8p-14, INIT_U128( 0x0, 0x9e8b1f1c3d ) },
  { 0x1.7bcee22af79dcp+39, -0x1.e3876965c70edp-14, INIT_U128( 0x0, 0xbde771157b ) },
  { 0x1.db3c0a99b6782p+40, 0x1.c00354658006ap-13, INIT_U128( 0x0, 0x1db3c0a99b6 ) },
  { 0x1.95e874332bd0ep+40, -0x1.24a2901c49452p-13, INIT_U128( 0x0, 0x195e874332b ) },
  { 0x1.3f1367747e26dp+40, 0x1.a9bba32953774p-13, INIT_U128( 0x0, 0x13f1367747e ) },
  { 0x1.212d396e425a7p+40, -0x1.6c74f6f0d8e9fp-13, INIT_U128( 0x0, 0x1212d396e42 ) },
  { 0x1.ec9237bfd9247p+40, 0x1.a9dbd76353b7bp-13, INIT_U128( 0x0, 0x1ec9237bfd9 ) },
  { 0x1.dfce5d45bf9cbp+40, -0x1.5131d492a263ap-13, INIT_U128( 0x0, 0x1dfce5d45bf ) },
  { 0x1.a1c625cb438c5p+41, 0x1.2cf5ce8a59ebap-12, INIT_U128( 0x0, 0x3438c4b9687 ) },
  { 0x1.37ad114e6f5a2p+41, -0x1.54238fcea8472p-12, INIT_U128( 0x0, 0x26f5a229cde ) },
  { 0x1.c2d8868b85b11p+41, 0x1.aa942e8555286p-12, INIT_U128( 0x0, 0x385b10d170b ) },
  { 0x1.9799f3e12f33ep+41, -0x1.1fd9fffe3fb40p-12, INIT_U128( 0x0, 0x32f33e7c25e ) },
  { 0x1.ca05f0d5940bep+41, 0x1.aed60bbd5dac2p-12, INIT_U128( 0x0, 0x3940be1ab28 ) },
  { 0x1.39e951d473d2ap+41, -0x1.e3468071c68d0p-12, INIT_U128( 0x0, 0x273d2a3a8e7 ) },
  { 0x1.68842ec4d1086p+42, 0x1.0e72b7e61ce57p-11, INIT_U128( 0x0, 0x5a210bb1344 ) },
  { 0x1.67c68f12cf8d2p+42, -0x1.a59a6cff4b34ep-11, INIT_U128( 0x0, 0x59f1a3c4b3e ) },
  { 0x1.dbaa799bb754fp+42, 0x1.e395713dc72aep-11, INIT_U128( 0x0, 0x76ea9e66edd ) },
  { 0x1.8921d6651243bp+42, -0x1.b6e684336dcd0p-11, INIT_U128( 0x0, 0x62487599449 ) },
  { 0x1.a398c5a347319p+42, 0x1.1f0a4a983e14ap-11, INIT_U128( 0x0, 0x68e63168d1c ) },
  { 0x1.db891451b7122p+42, -0x1.e2ac2d19c5585p-11, INIT_U128( 0x0, 0x76e245146dc ) },
  { 0x1.c8c5edc3918bep+43, 0x1.fcb64dc1f96c9p-10, INIT_U128( 0x0, 0xe462f6e1c8c ) },
  { 0x1.4f2124089e424p+43, -0x1.b55825816ab05p-10, INIT_U128( 0x0, 0xa79092044f2 ) },
  { 0x1.a69225234d245p+43, 0x1.2eef814e5ddf0p-10, INIT_U128( 0x0, 0xd3491291a69 ) },
  { 0x1.0689fa140d140p+43, -0x1.48584aee90b0ap-10, INIT_U128( 0x0, 0x8344fd0a068 ) },
  { 0x1.5be54fdab7caap+43, 0x1.3e370b647c6e2p-10, INIT_U128( 0x0, 0xadf2a7ed5be ) },
  { 0x1.9c7bdabd38f7bp+43, -0x1.1ebdd6663d7bbp-10, INIT_U128( 0x0, 0xce3ded5e9c7 ) },
  { 0x1.e8cb39ebd1967p+44, 0x1.3415ce9e682bap-9, INIT_U128( 0x0, 0x1e8cb39ebd19 ) },
  { 0x1.3e97d6667d2fbp+44, -0x1.b2d12ac165a25p-9, INIT_U128( 0x0, 0x13e97d6667d2 ) },
  { 0x1.e2ebda5dc5d7cp+44, 0x1.7881283af1025p-9, INIT_U128( 0x0, 0x1e2ebda5dc5d ) },
  { 0x1.9617ab312c2f6p+44, -0x1.4b70092296e01p-9, INIT_U128( 0x0, 0x19617ab312c2 ) },
  { 0x1.86f3e0230de7cp+44, 0x1.fdde3a45fbbc8p-9, INIT_U128( 0x0, 0x186f3e0230de ) },
  { 0x1.f10a7621e214fp+44, -0x1.629e59bcc53cbp-9, INIT_U128( 0x0, 0x1f10a7621e21 ) },
  { 0x1.e173ab83c2e76p+45, 0x1.7a2c4c86f458ap-8, INIT_U128( 0x0, 0x3c2e7570785c ) },
  { 0x1.1984ddf03309cp+45, -0x1.850c71250a18ep-8, INIT_U128( 0x0, 0x23309bbe0661 ) },
  { 0x1.9936971f326d3p+45, 0x1.f50df243ea1bfp-8, INIT_U128( 0x0, 0x3326d2e3e64d ) },
  { 0x1.67b89e3ecf714p+45, -0x1.61fc84e2c3f90p-8, INIT_U128( 0x0, 0x2cf713c7d9ee ) },
  { 0x1.f1c44005e3888p+45, 0x1.b6d9c2496db38p-8, INIT_U128( 0x0, 0x3e388800bc71 ) },
  { 0x1.1754aa642ea96p+45, -0x1.9856652930acdp-8, INIT_U128( 0x0, 0x22ea954c85d5 ) },
  { 0x1.e5c94399cb929p+46, 0x1.1ddea47a3bbd4p-7, INIT_U128( 0x0, 0x797250e672e4 ) },
  { 0x1.e0aeb017c15d6p+46, -0x1.41ae9374835d2p-7, INIT_U128( 0x0, 0x782bac05f057 ) },
  { 0x1.5c15afe0b82b6p+46, 0x1.7d2a8ad4fa552p-7, INIT_U128( 0x0, 0x57056bf82e0a ) },
  { 0x1.606c69dcc0d8dp+46, -0x1.44e0e9cc89c1dp-7, INIT_U128( 0x0, 0x581b1a773036 ) },
  { 0x1.b14059756280bp+46, 0x1.9f0440433e088p-7, INIT_U128( 0x0, 0x6c50165d58a0 ) },
  { 0x1.f7b2148def642p+46, -0x1.942c33d328586p-7, INIT_U128( 0x0, 0x7dec85237bd9 ) },
  { 0x1.1785687e2f0adp+47, 0x1.8dc9bfbf1b938p-6, INIT_U128( 0x0, 0x8bc2b43f1785 ) },
  { 0x1.df9c3cf7bf387p+47, -0x1.9e7948a33cf29p-6, INIT_U128( 0x0, 0xefce1e7bdf9c ) },
  { 0x1.8ba7fa05174ffp+47, 0x1.d0a6b319a14d7p-6, INIT_U128( 0x0, 0xc5d3fd028ba7 ) },
  { 0x1.0117a260022f4p+47, -0x1.6d1ba586da374p-6, INIT_U128( 0x0, 0x808bd1300117 ) },
  { 0x1.4ad2ff9e95a60p+47, 0x1.1a9c805235390p-6, INIT_U128( 0x0, 0xa5697fcf4ad3 ) },
  { 0x1.f22816ade4503p+47, -0x1.e40996bfc8133p-6, INIT_U128( 0x0, 0xf9140b56f228 ) },
  { 0x1.0faa72961f54ep+48, 0x1.59e0b4ecb3c16p-5, INIT_U128( 0x0, 0x10faa72961f54 ) },
  { 0x1.46061ba28c0c4p+48, -0x1.016eb29a02dd6p-5, INIT_U128( 0x0, 0x146061ba28c0c ) },
  { 0x1.10ba77342174fp+48, 0x1.5d5ac606bab59p-5, INIT_U128( 0x0, 0x110ba77342174 ) },
  { 0x1.99c517c5338a3p+48, -0x1.f683bacfed078p-5, INIT_U128( 0x0, 0x199c517c5338a ) },
  { 0x1.7b396a82f672ep+48, 0x1.7a73c7dcf4e79p-5, INIT_U128( 0x0, 0x17b396a82f672 ) },
  { 0x1.38324e427064ap+48, -0x1.d74f1a63ae9e3p-5, INIT_U128( 0x0, 0x138324e427064 ) },
  { 0x1.f79e52b3ef3cbp+49, 0x1.989eb821313d7p-4, INIT_U128( 0x0, 0x3ef3ca567de79 ) },
  { 0x1.1c30043038600p+49, -0x1.a8deac2951bd6p-4, INIT_U128( 0x0, 0x23860086070bf ) },
  { 0x1.37a2f8646f45fp+49, 0x1.13e218be27c43p-4, INIT_U128( 0x0, 0x26f45f0c8de8b ) },
  { 0x1.3d0adffc7a15cp+49, -0x1.4f2860e49e50cp-4, INIT_U128( 0x0, 0x27a15bff8f42b ) },
  { 0x1.a1abd87f4357bp+49, 0x1.1ca71cf2394e4p-4, INIT_U128( 0x0, 0x34357b0fe86af ) },
  { 0x1.eda1a987db435p+49, -0x1.2d53d2825aa7ap-4, INIT_U128( 0x0, 0x3db43530fb686 ) },
  { 0x1.ed4dd00dda9bap+50, 0x1.eeb94be9dd72ap-3, INIT_U128( 0x0, 0x7b53740376a6e ) },
  { 0x1.9a6f780b34defp+50, -0x1.3820cfac7041ap-3, INIT_U128( 0x0, 0x669bde02cd37b ) },
  { 0x1.0955cc3a12abap+50, 0x1.8c978eb9192f2p-3, INIT_U128( 0x0, 0x4255730e84aae ) },
  { 0x1.1c2a666c3854dp+50, -0x1.19003ed232008p-3, INIT_U128( 0x0, 0x470a999b0e153 ) },
  { 0x1.e67843a7ccf09p+50, 0x1.6230820ec4610p-3, INIT_U128( 0x0, 0x799e10e9f33c2 ) },
  { 0x1.1a6ef16e34ddep+50, -0x1.f732baddee658p-3, INIT_U128( 0x0, 0x469bbc5b8d377 ) },
  { 0x1.bf8b48177f169p+51, 0x1.b434093368681p-2, INIT_U128( 0x0, 0xdfc5a40bbf8b4 ) },
  { 0x1.edb675abdb6cep+51, -0x1.3ac4b29a75896p-2, INIT_U128( 0x0, 0xf6db3ad5edb66 ) },
  { 0x1.2ea7d1105d4fap+51, 0x1.25e8c3ee4bd18p-2, INIT_U128( 0x0, 0x9753e8882ea7d ) },
  { 0x1.1213ccfe2427ap+51, -0x1.264623c24c8c4p-2, INIT_U128( 0x0, 0x8909e67f1213c ) },
  { 0x1.be877c117d0f0p+51, 0x1.2e43d12e5c87ap-2, INIT_U128( 0x0, 0xdf43be08be878 ) },
  { 0x1.cf24cfed9e49ap+51, -0x1.f62a7475ec54ep-2, INIT_U128( 0x0, 0xe79267f6cf24c ) },
  { 0x1.4225e9ee844bdp+52, 0x1.0c2fafe8185f6p-1, INIT_U128( 0x0, 0x14225e9ee844bd ) },
  { 0x1.4552d55a8aa5ap+52, -0x1.e3d1c26dc7a39p-1, INIT_U128( 0x0, 0x14552d55a8aa59 ) },
  { 0x1.28d6ccdc51adap+52, 0x1.32f47ee265e90p-1, INIT_U128( 0x0, 0x128d6ccdc51ada ) },
  { 0x1.112594f0224b2p+52, -0x1.16249cea2c494p-1, INIT_U128( 0x0, 0x1112594f0224b1 ) },
  { 0x1.21adcaca435bap+52, 0x1.ab950cfd572a2p-1, INIT_U128( 0x0, 0x121adcaca435ba ) },
  { 0x1.4e456c969c8aep+52, -0x1.2b03970656073p-1, INIT_U128( 0x0, 0x14e456c969c8ad ) },
  { 0x1.382c9bb670594p+53, 0x1.6a6fd304d4dfap+0, INIT_U128( 0x0, 0x27059376ce0b29 ) },
  { 0x1.4403d5da8807ap+53, -0x1.1ea4df5e3d49cp+0, INIT_U128( 0x0, 0x28807abb5100f2 ) },
  { 0x1.cca055159940ap+53, 0x1.5e72c748bce59p+0, INIT_U128( 0x0, 0x39940aa2b32815 ) },
  { 0x1.15d79a3a2baf4p+53, -0x1.8c2b9d2d18574p+0, INIT_U128( 0x0, 0x22baf3474575e6 ) },
  { 0x1.12a6d07c254dap+53, 0x1.038e0fe6071c2p+0, INIT_U128( 0x0, 0x2254da0f84a9b5 ) },
  { 0x1.c6f04fc78de0ap+53, -0x1.414ea18e829d4p+0, INIT_U128( 0x0, 0x38de09f8f1bc12 ) },
  { 0x1.2976d3d652edap+54, 0x1.d9041cc7b2083p+1, INIT_U128( 0x0, 0x4a5db4f594bb6b ) },
  { 0x1.6794c552cf298p+54, -0x1.e7fe4fe9cffcap+1, INIT_U128( 0x0, 0x59e53154b3ca5c ) },
  { 0x1.0747edf60e8fep+54, 0x1.36b9953e6d732p+1, INIT_U128( 0x0, 0x41d1fb7d83a3fa ) },
  { 0x1.de28c45fbc518p+54, -0x1.496dd7e492dbbp+1, INIT_U128( 0x0, 0x778a3117ef145d ) },
  { 0x1.42e9452885d28p+54, 0x1.237b99b846f73p+1, INIT_U128( 0x0, 0x50ba514a2174a2 ) },
  { 0x1.2d3f9a525a7f4p+54, -0x1.6bc01206d7802p+1, INIT_U128( 0x0, 0x4b4fe694969fcd ) },
  { 0x1.5670189cace03p+55, 0x1.3e652f447cca6p+2, INIT_U128( 0x0, 0xab380c4e56701c ) },
  { 0x1.81e4aef303c96p+55, -0x1.781de368f03bcp+2, INIT_U128( 0x0, 0xc0f2577981e4aa ) },
  { 0x1.e9fa3c43d3f47p+55, 0x1.d02a1b8ba0544p+2, INIT_U128( 0x0, 0xf4fd1e21e9fa3f ) },
  { 0x1.a3217a714642fp+55, -0x1.2226d7fa444dbp+2, INIT_U128( 0x0, 0xd190bd38a32173 ) },
  { 0x1.3a99ee8e7533ep+55, 0x1.e725504bce4aap+2, INIT_U128( 0x0, 0x9d4cf7473a99f7 ) },
  { 0x1.40cbb75a81977p+55, -0x1.073a43a40e748p+2, INIT_U128( 0x0, 0xa065dbad40cbb3 ) },
  { 0x1.459817948b303p+56, 0x1.56a35196ad46ap+3, INIT_U128( 0x0, 0x1459817948b303a ) },
  { 0x1.9365e0d126cbcp+56, -0x1.583ee812b07ddp+3, INIT_U128( 0x0, 0x19365e0d126cbb5 ) },
  { 0x1.5a31617eb462cp+56, 0x1.5a563098b4ac6p+3, INIT_U128( 0x0, 0x15a31617eb462ca ) },
  { 0x1.998d89a3331b1p+56, -0x1.83a8355707507p+3, INIT_U128( 0x0, 0x1998d89a3331b03 ) },
  { 0x1.335a4d6066b4ap+56, 0x1.01224ef40244ap+3, INIT_U128( 0x0, 0x1335a4d6066b4a8 ) },
  { 0x1.6ba6853ed74d0p+56, -0x1.3aa23b5675448p+3, INIT_U128( 0x0, 0x16ba6853ed74cf6 ) },
  { 0x1.37dbe4346fb7cp+57, 0x1.49cf499e939e9p+4, INIT_U128( 0x0, 0x26fb7c868df6f94 ) },
  { 0x1.8410133d08202p+57, -0x1.7ac5295ef58a5p+4, INIT_U128( 0x0, 0x30820267a104028 ) },
  { 0x1.6d4ae358da95cp+57, 0x1.b14fad91629f6p+4, INIT_U128( 0x0, 0x2da95c6b1b52b9b ) },
  { 0x1.e6853b67cd0a8p+57, -0x1.c3bafbf587760p+4, INIT_U128( 0x0, 0x3cd0a76cf9a14e3 ) },
  { 0x1.efb4dcd7df69bp+57, 0x1.6869ea5ad0d3ep+4, INIT_U128( 0x0, 0x3df69b9afbed376 ) },
  { 0x1.3bd8df5e77b1cp+57, -0x1.d9af191bb35e3p+4, INIT_U128( 0x0, 0x277b1bebcef6362 ) },
  { 0x1.389405e471280p+58, 0x1.defa59cfbdf4bp+5, INIT_U128( 0x0, 0x4e2501791c4a03b ) },
  { 0x1.fbd5f117f7abep+58, -0x1.732df474e65bep+5, INIT_U128( 0x0, 0x7ef57c45fdeaf51 ) },
  { 0x1.c61faa1b8c3f5p+58, 0x1.3ca1a87279435p+5, INIT_U128( 0x0, 0x7187ea86e30fd67 ) },
  { 0x1.7a68381af4d07p+58, -0x1.0bb7c1b0176f8p+5, INIT_U128( 0x0, 0x5e9a0e06bd3419e ) },
  { 0x1.9344fa092689fp+58, 0x1.0dce18ec1b9c3p+5, INIT_U128( 0x0, 0x64d13e8249a27e1 ) },
  { 0x1.63f6a9c4c7ed5p+58, -0x1.ed765b89daeccp+5, INIT_U128( 0x0, 0x58fdaa7131fb502 ) },
  { 0x1.68244986d0489p+59, 0x1.fd843369fb087p+6, INIT_U128( 0x0, 0xb41224c368244ff ) },
  { 0x1.711e90dee23d2p+59, -0x1.9b7e96e736fd3p+6, INIT_U128( 0x0, 0xb88f486f711e899 ) },
  { 0x1.fea6825ffd4d1p+59, 0x1.dcc5161fb98a3p+6, INIT_U128( 0x0, 0xff53412ffea68f7 ) },
  { 0x1.d620d23fac41ap+59, -0x1.368266946d04dp+6, INIT_U128( 0x0, 0xeb10691fd620cb2 ) },
  { 0x1.26945fbe4d28cp+59, 0x1.5a30e57cb461cp+6, INIT_U128( 0x0, 0x934a2fdf2694656 ) },
  { 0x1.b1478ff5628f2p+59, -0x1.93df431727be8p+6, INIT_U128( 0x0, 0xd8a3c7fab14789b ) },
  { 0x1.0b80930e17012p+60, 0x1.97cccb5d2f99ap+7, INIT_U128( 0x0, 0x10b80930e17012cb ) },
  { 0x1.579a3f0caf348p+60, -0x1.fb356fe7f66aep+7, INIT_U128( 0x0, 0x1579a3f0caf34702 ) },
  { 0x1.0036b5b6006d6p+60, 0x1.93731e6d26e64p+7, INIT_U128( 0x0, 0x10036b5b6006d6c9 ) },
  { 0x1.d4e48903a9c91p+60, -0x1.38cabb9c71958p+7, INIT_U128( 0x0, 0x1d4e48903a9c9063 ) },
  { 0x1.726d783ce4dafp+60, 0x1.d0ba7a1ba174fp+7, INIT_U128( 0x0, 0x1726d783ce4dafe8 ) },
  { 0x1.abf8b01357f16p+60, -0x1.388d2258711a4p+7, INIT_U128( 0x0, 0x1abf8b01357f1563 ) },
  { 0x1.0d5c228e1ab84p+61, 0x1.2660e82e4cc1dp+8, INIT_U128( 0x0, 0x21ab8451c3570926 ) },
  { 0x1.4444e40a8889cp+61, -0x1.a0a11c9941424p+8, INIT_U128( 0x0, 0x28889c815111365f ) },
  { 0x1.342f011a685e0p+61, 0x1.a83b8dd150772p+8, INIT_U128( 0x0, 0x2685e0234d0bc1a8 ) },
  { 0x1.8e1778211c2efp+61, -0x1.9c534aa338a69p+8, INIT_U128( 0x0, 0x31c2ef042385dc63 ) },
  { 0x1.a5bfc1394b7f8p+61, 0x1.ff360901fe6c1p+8, INIT_U128( 0x0, 0x34b7f827296ff1ff ) },
  { 0x1.37d1e18a6fa3cp+61, -0x1.a466439948cc8p+8, INIT_U128( 0x0, 0x26fa3c314df4765b ) },
  { 0x1.dbfb1f21b7f64p+62, 0x1.0df2b3ca1be56p+9, INIT_U128( 0x0, 0x76fec7c86dfd921b ) },
  { 0x1.dae8aa2db5d15p+62, -0x1.028f7ee0051f0p+9, INIT_U128( 0x0, 0x76ba2a8b6d7451fa ) },
  { 0x1.68239948d0473p+62, 0x1.8c303a2318607p+9, INIT_U128( 0x0, 0x5a08e6523411cf18 ) },
  { 0x1.966ec1f92cdd8p+62, -0x1.73a42c58e7486p+9, INIT_U128( 0x0, 0x659bb07e4b375d18 ) },
  { 0x1.23fd057e47fa0p+62, 0x1.ee090107dc120p+9, INIT_U128( 0x0, 0x48ff415f91fe83dc ) },
  { 0x1.6ed99c40ddb34p+62, -0x1.2433bb3648678p+9, INIT_U128( 0x0, 0x5bb66710376ccdb7 ) },
  { 0x1.817c1f7302f84p+63, 0x1.43be2544877c4p+10, INIT_U128( 0x0, 0xc0be0fb9817c250e ) },
  { 0x1.5cfe76d4b9fcfp+63, -0x1.f3e997b1e7d33p+10, INIT_U128( 0x0, 0xae7f3b6a5cfe7030 ) },
  { 0x1.14789f6028f14p+63, 0x1.08f726e811ee5p+10, INIT_U128( 0x0, 0x8a3c4fb01478a423 ) },
  { 0x1.2c76ba0858ed8p+63, -0x1.21d4c1ee43a98p+10, INIT_U128( 0x0, 0x963b5d042c76bb78 ) },
  { 0x1.2f580f185eb02p+63, 0x1.70bb9e14e1774p+10, INIT_U128( 0x0, 0x97ac078c2f5815c2 ) },
  { 0x1.c275de8984ebcp+63, -0x1.63871034c70e2p+10, INIT_U128( 0x0, 0xe13aef44c275da71 ) },
  { 0x1.18f72be431ee6p+64, 0x1.508d56aaa11abp+11, INIT_U128( 0x1, 0x18f72be431ee6a84 ) },
  { 0x1.ffda5ad7ffb4cp+64, -0x1.beb6ed1f7d6dep+11, INIT_U128( 0x1, 0xffda5ad7ffb4b20a ) },
  { 0x1.4fc35fac9f86cp+64, 0x1.37ac3c626f588p+11, INIT_U128( 0x1, 0x4fc35fac9f86c9bd ) },
  { 0x1.631d30b8c63a6p+64, -0x1.2bbe7ece577d0p+11, INIT_U128( 0x1, 0x631d30b8c63a56a2 ) },
  { 0x1.459ef02c8b3dep+64, 0x1.3e05dae67c0bcp+11, INIT_U128( 0x1, 0x459ef02c8b3de9f0 ) },
  { 0x1.2e4dfdae5c9c0p+64, -0x1.adc289775b851p+11, INIT_U128( 0x1, 0x2e4dfdae5c9bf291 ) },
  { 0x1.ca361509946c2p+65, 0x1.c4bcbfa989798p+12, INIT_U128( 0x3, 0x946c2a1328d85c4b ) },
  { 0x1.2166b5ce42cd6p+65, -0x1.ee331a89dc664p+12, INIT_U128( 0x2, 0x42cd6b9c859aa11c ) },
  { 0x1.b2c5c0af658b8p+65, 0x1.904052112080ap+12, INIT_U128( 0x3, 0x658b815ecb171904 ) },
  { 0x1.1266d5c024cdap+65, -0x1.3441e0226883cp+12, INIT_U128( 0x2, 0x24cdab80499b2cbb ) },
  { 0x1.d3fe7755a7fcfp+65, 0x1.6f654c96decaap+12, INIT_U128( 0x3, 0xa7fceeab4ff9f6f6 ) },
  { 0x1.282b77d45056fp+65, -0x1.e87716b7d0ee3p+12, INIT_U128( 0x2, 0x5056efa8a0adc178 ) },
  { 0x1.785c7380f0b8ep+66, 0x1.54a231dea9446p+13, INIT_U128( 0x5, 0xe171ce03c2e3aa94 ) },
  { 0x1.0a4bd8a61497bp+66, -0x1.a7b4829b4f690p+13, INIT_U128( 0x4, 0x292f6298525e8b09 ) },
  { 0x1.c55dcff38abbap+66, 0x1.ccc1b72f99837p+13, INIT_U128( 0x7, 0x15773fce2aeeb998 ) },
  { 0x1.0b3b92ce16772p+66, -0x1.de54fbd9bcaa0p+13, INIT_U128( 0x4, 0x2cee4b3859dc4435 ) },
  { 0x1.1d4f0dda3a9e2p+66, 0x1.0b15eda6162bep+13, INIT_U128( 0x4, 0x753c3768ea78a162 ) },
  { 0x1.9d0be8f93a17dp+66, -0x1.6d126d86da24ep+13, INIT_U128( 0x6, 0x742fa3e4e85f125d ) },
  { 0x1.42c22a0a85846p+67, 0x1.002919b200523p+14, INIT_U128( 0xa, 0x161150542c23400a ) },
  { 0x1.98d0ec5731a1ep+67, -0x1.88611a3910c23p+14, INIT_U128( 0xc, 0xc68762b98d0e9de7 ) },
  { 0x1.3e2ef4987c5dep+67, 0x1.c71eea078e3ddp+14, INIT_U128( 0x9, 0xf177a4c3e2ef71c7 ) },
  { 0x1.3623a3ae6c474p+67, -0x1.086509a210ca1p+14, INIT_U128( 0x9, 0xb11d1d736239bde6 ) },
  { 0x1.8bdba9af17b75p+67, 0x1.578a597eaf14bp+14, INIT_U128( 0xc, 0x5edd4d78bdbad5e2 ) },
  { 0x1.f698242ded304p+67, -0x1.3ea16a5e7d42ep+14, INIT_U128( 0xf, 0xb4c1216f6981b057 ) },
  { 0x1.e7caab91cf956p+68, 0x1.08fef9ee11fdfp+15, INIT_U128( 0x1e, 0x7caab91cf956847f ) },
  { 0x1.72e34a82e5c6ap+68, -0x1.bbc922e777924p+15, INIT_U128( 0x17, 0x2e34a82e5c69221b ) },
  { 0x1.78644686f0c89p+68, 0x1.9b0ec259361d8p+15, INIT_U128( 0x17, 0x8644686f0c89cd87 ) },
  { 0x1.a440399d48807p+68, -0x1.f0c0ba09e1817p+15, INIT_U128( 0x1a, 0x440399d48806079f ) },
  { 0x1.efb0f437df61ep+68, 0x1.ae8c12455d182p+15, INIT_U128( 0x1e, 0xfb0f437df61ed746 ) },
  { 0x1.98753ca530ea8p+68, -0x1.b50ec42b6a1d8p+15, INIT_U128( 0x19, 0x8753ca530ea72578 ) },
  { 0x1.7b4dc974f69b9p+69, 0x1.a3b0cf754761ap+16, INIT_U128( 0x2f, 0x69b92e9ed373a3b0 ) },
  { 0x1.9363c00526c78p+69, -0x1.989cd74f3139bp+16, INIT_U128( 0x32, 0x6c7800a4d8ee6763 ) },
  { 0x1.aec6bd1d5d8d8p+69, 0x1.306fa8fe60df5p+16, INIT_U128( 0x35, 0xd8d7a3abb1b1306f ) },
  { 0x1.05af44d60b5e8p+69, -0x1.f188e727e311dp+16, INIT_U128( 0x20, 0xb5e89ac16bce0e77 ) },
  { 0x1.3098c2b261318p+69, 0x1.9e80312f3d006p+16, INIT_U128( 0x26, 0x1318564c26319e80 ) },
  { 0x1.79ea0160f3d40p+69, -0x1.6ec7b74cdd8f7p+16, INIT_U128( 0x2f, 0x3d402c1e7a7e9138 ) },
  { 0x1.659f1b34cb3e4p+70, 0x1.b00da511601b5p+17, INIT_U128( 0x59, 0x67c6cd32cf93601b ) },
  { 0x1.772cc0f0ee598p+70, -0x1.f17ac4e5e2f58p+17, INIT_U128( 0x5d, 0xcb303c3b965c1d0a ) },
  { 0x1.0797fe900f300p+70, 0x1.6dc4e51edb89cp+17, INIT_U128( 0x41, 0xe5ffa403cc02db89 ) },
  { 0x1.301c2f4460386p+70, -0x1.ef62aa21dec55p+17, INIT_U128( 0x4c, 0x70bd1180e14213a ) },
  { 0x1.76c3df0eed87cp+70, 0x1.a2ff79c945fefp+17, INIT_U128( 0x5d, 0xb0f7c3bb61f345fe ) },
  { 0x1.ab2bcd095657ap+70, -0x1.ef282597de504p+17, INIT_U128( 0x6a, 0xcaf3425595e421af ) },
  { 0x1.3e0e3c227c1c8p+71, 0x1.4f6c06a69ed81p+18, INIT_U128( 0x9f, 0x71e113e0e453db0 ) },
  { 0x1.dfe8dfa1bfd1cp+71, -0x1.2dbf305a5b7e6p+18, INIT_U128( 0xef, 0xf46fd0dfe8db4903 ) },
  { 0x1.fdf818ebfbf03p+71, 0x1.785e1c44f0bc4p+18, INIT_U128( 0xfe, 0xfc0c75fdf81de178 ) },
  { 0x1.e2cdf487c59bep+71, -0x1.7945e260f28bcp+18, INIT_U128( 0xf1, 0x66fa43e2cdea1ae8 ) },
  { 0x1.1c0eebba381dep+71, 0x1.7631b48cec636p+18, INIT_U128( 0x8e, 0x775dd1c0ef5d8c6 ) },
  { 0x1.11e2b9b023c57p+71, -0x1.cad5a8d795ab5p+18, INIT_U128( 0x88, 0xf15cd811e2b0d4a9 ) },
  { 0x1.a6a15d8d4d42cp+72, 0x1.1791fe882f240p+19, INIT_U128( 0x1a6, 0xa15d8d4d42c8bc8f ) },
  { 0x1.bc5e8fff78bd2p+72, -0x1.d73a203fae744p+19, INIT_U128( 0x1bc, 0x5e8fff78bd11462e ) },
  { 0x1.b6122fb76c246p+72, 0x1.364891606c912p+19, INIT_U128( 0x1b6, 0x122fb76c2469b244 ) },
  { 0x1.4601d78c8c03bp+72, -0x1.1bb14aec3762ap+19, INIT_U128( 0x146, 0x1d78c8c03a72275 ) },
  { 0x1.e6563671ccac7p+72, 0x1.bd2e987d7a5d3p+19, INIT_U128( 0x1e6, 0x563671ccac7de974 ) },
  { 0x1.0d2c0fd21a582p+72, -0x1.edee3503dbdc6p+19, INIT_U128( 0x10d, 0x2c0fd21a5810908e ) },
  { 0x1.2af37d9c55e70p+73, 0x1.3d483f187a908p+20, INIT_U128( 0x255, 0xe6fb38abce13d483 ) },
  { 0x1.dd50df25baa1cp+73, -0x1.a1b75cab436ecp+20, INIT_U128( 0x3ba, 0xa1be4b754365e48a ) },
  { 0x1.e047b8e3c08f7p+73, 0x1.887cac2d10f96p+20, INIT_U128( 0x3c0, 0x8f71c7811ef887ca ) },
  { 0x1.90df824321bf0p+73, -0x1.1e55f5063cabep+20, INIT_U128( 0x321, 0xbf0486437dee1aa0 ) },
  { 0x1.d7e8b2d5afd17p+73, 0x1.70c861e2e190cp+20, INIT_U128( 0x3af, 0xd165ab5fa2f70c86 ) },
  { 0x1.4c152560982a4p+73, -0x1.d5e0cbc1abc1ap+20, INIT_U128( 0x298, 0x2a4ac1305462a1f3 ) },
  { 0x1.e9267e29d24d0p+74, 0x1.2ee319e85dc63p+21, INIT_U128( 0x7a4, 0x99f8a7493425dc63 ) },
  { 0x1.4f26ae209e4d6p+74, -0x1.053569e40a6adp+21, INIT_U128( 0x53c, 0x9ab88279355f5952 ) },
  { 0x1.a514ce754a29ap+74, 0x1.610967d2c212dp+21, INIT_U128( 0x694, 0x5339d528a6ac212c ) },
  { 0x1.d00b12bda0163p+74, -0x1.4bb62c02976c6p+21, INIT_U128( 0x740, 0x2c4af6805896893a ) },
  { 0x1.789b665af136dp+74, 0x1.8c69a2bd18d34p+21, INIT_U128( 0x5e2, 0x6d996bc4db718d34 ) },
  { 0x1.22f04c4245e0ap+74, -0x1.2ead96125d5b3p+21, INIT_U128( 0x48b, 0xc1310917825a2a4d ) },
  { 0x1.c2b2341d85646p+75, 0x1.68f223ced1e44p+22, INIT_U128( 0xe15, 0x91a0ec2b235a3c88 ) },
  { 0x1.4a5ee10094bdcp+75, -0x1.a9086a8f5210dp+22, INIT_U128( 0xa52, 0xf70804a5ed95bde5 ) },
  { 0x1.2620bc4c4c418p+75, 0x1.578af0d4af15ep+22, INIT_U128( 0x931, 0x5e262620c55e2bc ) },
  { 0x1.96f69f932ded4p+75, -0x1.511439eaa2287p+22, INIT_U128( 0xcb7, 0xb4fc996f69abbaf1 ) },
  { 0x1.94d32fef29a66p+75, 0x1.74477eb6e88f0p+22, INIT_U128( 0xca6, 0x997f794d335d11df ) },
  { 0x1.f2cb9697e5973p+75, -0x1.751ae8d2ea35dp+22, INIT_U128( 0xf96, 0x5cb4bf2cb922b945 ) },
  { 0x1.fcc42ae5f9886p+76, 0x1.83ab18d907563p+23, INIT_U128( 0x1fcc, 0x42ae5f9886c1d58c ) },
  { 0x1.9370dddb26e1cp+76, -0x1.d4acddbba959cp+23, INIT_U128( 0x1937, 0xdddb26e1b15a991 ) },
  { 0x1.f4737abde8e70p+76, 0x1.df86033dbf0c1p+23, INIT_U128( 0x1f47, 0x37abde8e70efc301 ) },
  { 0x1.aa8da97b551b5p+76, -0x1.dd99f8e7bb33fp+23, INIT_U128( 0x1aa8, 0xda97b551b4113303 ) },
  { 0x1.0e691d241cd24p+76, 0x1.f9b4579df368bp+23, INIT_U128( 0x10e6, 0x91d241cd24fcda2b ) },
  { 0x1.e2cf62cdc59edp+76, -0x1.b37deb9b66fbep+23, INIT_U128( 0x1e2c, 0xf62cdc59ec26410a ) },
  { 0x1.5abdbf70b57b8p+77, 0x1.b2da758965b4fp+24, INIT_U128( 0x2b57, 0xb7ee16af71b2da75 ) },
  { 0x1.ff34179ffe683p+77, -0x1.d35925cda6b25p+24, INIT_U128( 0x3fe6, 0x82f3ffcd042ca6da ) },
  { 0x1.3f4418467e883p+77, 0x1.222463e04448cp+24, INIT_U128( 0x27e8, 0x8308cfd107222463 ) },
  { 0x1.0206bbe0040d8p+77, -0x1.01f64c6403ecap+24, INIT_U128( 0x2040, 0xd77c0081aefe09b3 ) },
  { 0x1.8b1687b1162d1p+77, 0x1.4c659ce298cb4p+24, INIT_U128( 0x3162, 0xd0f622c5a34c659c ) },
  { 0x1.aafeb6f555fd7p+77, -0x1.b13743ff626e8p+24, INIT_U128( 0x355f, 0xd6deaabfac4ec8bc ) },
  { 0x1.a3f08db947e12p+78, 0x1.41f5568683eabp+25, INIT_U128( 0x68fc, 0x236e51f84a83eaad ) },
  { 0x1.024b7dea04970p+78, -0x1.38c3bda871878p+25, INIT_U128( 0x4092, 0xdf7a8125bd8e7884 ) },
  { 0x1.b63cd53f6c79bp+78, 0x1.f5171481ea2e2p+25, INIT_U128( 0x6d8f, 0x354fdb1e6fea2e29 ) },
  { 0x1.addea67d5bbd5p+78, -0x1.b301002566020p+25, INIT_U128( 0x6b77, 0xa99f56ef5099fdff ) },
  { 0x1.78c22d5af1846p+78, 0x1.414f8456829f0p+25, INIT_U128( 0x5e30, 0x8b56bc611a829f08 ) },
  { 0x1.87addde10f5bcp+78, -0x1.6d35a818da6b5p+25, INIT_U128( 0x61eb, 0x777843d6ed2594af ) },
  { 0x1.20d5879441ab1p+79, 0x1.323b875c64771p+26, INIT_U128( 0x906a, 0xc3ca20d58cc8ee1d ) },
  { 0x1.4acfa100959f4p+79, -0x1.2f3ea4ba5e7d4p+26, INIT_U128( 0xa567, 0xd0804acf9b43056d ) },
  { 0x1.2befdefa57dfcp+79, 0x1.2d9f03ce5b3e0p+26, INIT_U128( 0x95f7, 0xef7d2befe4b67c0f ) },
  { 0x1.e2bc9057c5792p+79, -0x1.86ad9e6b0d5b4p+26, INIT_U128( 0xf15e, 0x482be2bc89e54986 ) },
  { 0x1.e0797959c0f2fp+79, 0x1.e1f0a7f5c3e15p+26, INIT_U128( 0xf03c, 0xbcace0797f87c29f ) },
  { 0x1.5a06c438b40d8p+79, -0x1.1fbd919e3f7b2p+26, INIT_U128( 0xad03, 0x621c5a06bb8109b9 ) },
  { 0x1.04150bd6082a2p+80, 0x1.5c0d8018b81b0p+27, INIT_U128( 0x10415, 0xbd6082a2ae06c00 ) },
  { 0x1.21b3de044367cp+80, -0x1.8f2b93891e572p+27, INIT_U128( 0x121b3, 0xde044367b386a363 ) },
  { 0x1.a8731cfd50e64p+80, 0x1.ccb294f999652p+27, INIT_U128( 0x1a873, 0x1cfd50e64e6594a7 ) },
  { 0x1.4062d78880c5bp+80, -0x1.4c63aab498c76p+27, INIT_U128( 0x14062, 0xd78880c5a59ce2aa ) },
  { 0x1.0ea7eb801d4fep+80, 0x1.42ae5574855cap+27, INIT_U128( 0x10ea7, 0xeb801d4fea1572ab ) },
  { 0x1.8527685f0a4edp+80, -0x1.c4e4c0c189c98p+27, INIT_U128( 0x18527, 0x685f0a4ec1d8d9f9 ) },
  { 0x1.37c569f66f8adp+81, 0x1.33060eda660c2p+28, INIT_U128( 0x26f8a, 0xd3ecdf15b33060ed ) },
  { 0x1.77e5c0e6efcb8p+81, -0x1.b0879753610f3p+28, INIT_U128( 0x2efcb, 0x81cddf96e4f7868a ) },
  { 0x1.74559332e8ab2p+81, 0x1.ff939ec9ff274p+28, INIT_U128( 0x2e8ab, 0x2665d1565ff939ec ) },
  { 0x1.1dbd3f6c3b7a8p+81, -0x1.952b1ac92a563p+28, INIT_U128( 0x23b7a, 0x7ed876f4e6ad4e53 ) },
  { 0x1.31cf2624639e5p+81, 0x1.688ce06ed119cp+28, INIT_U128( 0x2639e, 0x4c48c73cb688ce06 ) },
  { 0x1.c469bf8388d38p+81, -0x1.5dbf0520bb7e0p+28, INIT_U128( 0x388d3, 0x7f0711a6ea240fad ) },
  { 0x1.962be43d2c57cp+82, 0x1.a7264fdb4e4cap+29, INIT_U128( 0x658af, 0x90f4b15f34e4c9fb ) },
  { 0x1.e46ca38dc8d95p+82, -0x1.76a55d96ed4acp+29, INIT_U128( 0x791b2, 0x8e372365112b544d ) },
  { 0x1.8916f7d1122dfp+82, 0x1.3e7367c47ce6dp+29, INIT_U128( 0x6245b, 0xdf4448b7e7ce6cf8 ) },
  { 0x1.9662ef372cc5ep+82, -0x1.8d2c03a91a580p+29, INIT_U128( 0x6598b, 0xbcdcb3174e5a7f8a ) },
  { 0x1.9a80e7df3501dp+82, 0x1.b816cd3b702dap+29, INIT_U128( 0x66a03, 0x9f7cd4077702d9a7 ) },
  { 0x1.1c57973238af3p+82, -0x1.42f404dc85e80p+29, INIT_U128( 0x4715e, 0x5cc8e2bc97a17f64 ) },
  { 0x1.84c962350992cp+83, 0x1.5cba6ebcb974ep+30, INIT_U128( 0xc264b, 0x11a84c96572e9baf ) },
  { 0x1.8f57af931eaf6p+83, -0x1.93714db726e2ap+30, INIT_U128( 0xc7abd, 0x7c98f57a9b23ac92 ) },
  { 0x1.9f68c4313ed18p+83, 0x1.147f17f028fe3p+30, INIT_U128( 0xcfb46, 0x2189f68c451fc5fc ) },
  { 0x1.58560e78b0ac2p+83, -0x1.d72e2e29ae5c6p+30, INIT_U128( 0xac2b0, 0x73c585608a347475 ) },
  { 0x1.478526cc8f0a5p+83, 0x1.d03bcf61a077ap+30, INIT_U128( 0xa3c29, 0x36647852f40ef3d8 ) },
  { 0x1.87b5c57b0f6b9p+83, -0x1.bbdaba0177b57p+30, INIT_U128( 0xc3dae, 0x2bd87b5c1109517f ) },
  { 0x1.cfcf6e2b9f9eep+84, 0x1.a57304e94ae60p+31, INIT_U128( 0x1cfcf6, 0xe2b9f9eed2b98274 ) },
  { 0x1.5105d4caa20bap+84, -0x1.019e9450033d2p+31, INIT_U128( 0x15105d, 0x4caa20b97f30b5d7 ) },
  { 0x1.0dffe5561bffcp+84, 0x1.156f944e2adf2p+31, INIT_U128( 0x10dffe, 0x5561bffc8ab7ca27 ) },
  { 0x1.5f5d9970bebb3p+84, -0x1.a42bc99548579p+31, INIT_U128( 0x15f5d9, 0x970bebb22dea1b35 ) },
  { 0x1.bc95e429792bcp+84, 0x1.43ca3cfe87948p+31, INIT_U128( 0x1bc95e, 0x429792bca1e51e7f ) },
  { 0x1.1ad1272c35a25p+84, -0x1.45acdbfa8b59cp+31, INIT_U128( 0x11ad12, 0x72c35a245d299202 ) },
  { 0x1.823da4c5047b4p+85, 0x1.b134166362683p+32, INIT_U128( 0x3047b4, 0x98a08f69b1341663 ) },
  { 0x1.d2d8dfa3a5b1cp+85, -0x1.1cbe8ca4397d2p+32, INIT_U128( 0x3a5b1b, 0xf474b636e341735b ) },
  { 0x1.694ae39ed295cp+85, 0x1.ef3076d7de60fp+32, INIT_U128( 0x2d295c, 0x73da52b9ef3076d7 ) },
  { 0x1.5fb29180bf652p+85, -0x1.ee1445d5dc289p+32, INIT_U128( 0x2bf652, 0x3017eca211ebba2a ) },
  { 0x1.4beae59297d5cp+85, 0x1.ef0a4115de148p+32, INIT_U128( 0x297d5c, 0xb252fab9ef0a4115 ) },
  { 0x1.8185938d030b2p+85, -0x1.eb53a9c5d6a75p+32, INIT_U128( 0x3030b2, 0x71a0616214ac563a ) },
  { 0x1.038959060712bp+86, 0x1.d9dba207b3b74p+33, INIT_U128( 0x40e256, 0x4181c4afb3b7440f ) },
  { 0x1.c281d7ef8503bp+86, -0x1.4b0e7fd4961d0p+33, INIT_U128( 0x70a075, 0xfbe140e969e30056 ) },
  { 0x1.aa177169542eep+86, 0x1.4a379d6a946f4p+33, INIT_U128( 0x6a85dc, 0x5a550bba946f3ad5 ) },
  { 0x1.06e939d60dd27p+86, -0x1.79e6df94f3cdcp+33, INIT_U128( 0x41ba4e, 0x758374990c3240d6 ) },
  { 0x1.efaaa253df555p+86, 0x1.57ef0ea0afde2p+33, INIT_U128( 0x7beaa8, 0x94f7d556afde1d41 ) },
  { 0x1.7b8573e4f70aep+86, -0x1.fdaa87abfb551p+33, INIT_U128( 0x5ee15c, 0xf93dc2b404aaf0a8 ) },
  { 0x1.6d5ea2f0dabd4p+87, 0x1.dae30d57b5c61p+34, INIT_U128( 0xb6af51, 0x786d5ea76b8c355e ) },
  { 0x1.a1d6514143acap+87, -0x1.29f4871a53e91p+34, INIT_U128( 0xd0eb28, 0xa0a1d64b582de396 ) },
  { 0x1.1ead4a563d5aap+87, 0x1.bca7786f794efp+34, INIT_U128( 0x8f56a5, 0x2b1ead56f29de1bd ) },
  { 0x1.6fa16314df42cp+87, -0x1.a0eaef1b41d5ep+34, INIT_U128( 0xb7d0b1, 0x8a6fa1597c544392 ) },
  { 0x1.6ddb8ed2dbb72p+87, 0x1.427a42b884f48p+34, INIT_U128( 0xb6edc7, 0x696ddb9509e90ae2 ) },
  { 0x1.bda97bc17b530p+87, -0x1.d7267273ae4cep+34, INIT_U128( 0xded4bd, 0xe0bda978a3663631 ) },
  { 0x1.31aab99063557p+88, 0x1.1d5c0d423ab82p+35, INIT_U128( 0x131aab9, 0x90635578eae06a11 ) },
  { 0x1.c53ab8498a757p+88, -0x1.7d959f24fb2b4p+35, INIT_U128( 0x1c53ab8, 0x498a7564135306d8 ) },
  { 0x1.0c664e5618ccap+88, 0x1.46c04bd68d80ap+35, INIT_U128( 0x10c664e, 0x5618ccaa36025eb4 ) },
  { 0x1.68e948ead1d29p+88, -0x1.0fefa75e1fdf5p+35, INIT_U128( 0x168e948, 0xead1d2878082c50f ) },
  { 0x1.209bedde4137ep+88, 0x1.2b6b20d856d64p+35, INIT_U128( 0x1209bed, 0xde4137e95b5906c2 ) },
  { 0x1.5afe2732b5fc5p+88, -0x1.cab35f499566cp+35, INIT_U128( 0x15afe27, 0x32b5fc41aa6505b3 ) },
  { 0x1.75a2be44eb458p+89, 0x1.61e7610ac3cecp+36, INIT_U128( 0x2eb457c, 0x89d68b161e7610ac ) },
  { 0x1.449a3edc89348p+89, -0x1.654d6098ca9acp+36, INIT_U128( 0x289347d, 0xb91268e9ab29f673 ) },
  { 0x1.b9d66fad73acep+89, 0x1.c01fb2c3803f7p+36, INIT_U128( 0x373acdf, 0x5ae759dc01fb2c38 ) },
  { 0x1.62f7e844c5efdp+89, -0x1.2735129c4e6a2p+36, INIT_U128( 0x2c5efd0, 0x898bdf8d8caed63b ) },
  { 0x1.9b02c01b36058p+89, 0x1.04ce3986099c7p+36, INIT_U128( 0x3360580, 0x366c0b104ce39860 ) },
  { 0x1.512999b8a2533p+89, -0x1.b9b00a7373601p+36, INIT_U128( 0x2a25333, 0x7144a64464ff58c8 ) },
  { 0x1.c175b1f582eb6p+90, 0x1.c0cafccf8195fp+37, INIT_U128( 0x705d6c7, 0xd60badb8195f99f0 ) },
  { 0x1.5a9af98eb535fp+90, -0x1.2f5140c85ea28p+37, INIT_U128( 0x56a6be6, 0x3ad4d79a15d7e6f4 ) },
  { 0x1.03453f88068a8p+90, 0x1.1844754e3088ep+37, INIT_U128( 0x40d14fe, 0x201a2a23088ea9c6 ) },
  { 0x1.0677b33a0cef6p+90, -0x1.34eff91869dffp+37, INIT_U128( 0x419decc, 0xe833bd596200dcf2 ) },
  { 0x1.35a082926b410p+90, 0x1.cbb26b859764ep+37, INIT_U128( 0x4d6820a, 0x49ad0439764d70b2 ) },
  { 0x1.b285983f650b3p+90, -0x1.4206d0ea840dap+37, INIT_U128( 0x6ca1660, 0xfd942c97bf25e2af ) },
  { 0x1.1272b75a24e57p+91, 0x1.1b6ee12836ddcp+38, INIT_U128( 0x89395ba, 0xd1272bc6dbb84a0d ) },
  { 0x1.51f67848a3ecfp+91, -0x1.330ea53a661d4p+38, INIT_U128( 0xa8fb3c2, 0x451f67333c56b166 ) },
  { 0x1.46da02028db40p+91, 0x1.acf49f7d59e94p+38, INIT_U128( 0xa36d010, 0x146da06b3d27df56 ) },
  { 0x1.fdc20483fb840p+91, -0x1.bc7cc04978f98p+38, INIT_U128( 0xfee1024, 0x1fdc1f90e0cfeda1 ) },
  { 0x1.0edc00881db80p+91, 0x1.425c436084b88p+38, INIT_U128( 0x876e004, 0x40edc0509710d821 ) },
  { 0x1.21ca20e043944p+91, -0x1.c8b270359164ep+38, INIT_U128( 0x90e5107, 0x21ca18dd363f29b ) },
  { 0x1.027bf62404f7fp+92, 0x1.84c2092709841p+39, INIT_U128( 0x1027bf62, 0x404f7fc261049384 ) },
  { 0x1.efe782d3dfcf1p+92, -0x1.9d7eb6413afd7p+39, INIT_U128( 0x1efe782d, 0x3dfcf03140a4df62 ) },
  { 0x1.899042ad13208p+92, 0x1.41111c8c82224p+39, INIT_U128( 0x1899042a, 0xd13208a0888e4641 ) },
  { 0x1.572ec0b4ae5d8p+92, -0x1.4b05f4a2960bep+39, INIT_U128( 0x1572ec0b, 0x4ae5d75a7d05aeb4 ) },
  { 0x1.01dc195a03b83p+92, 0x1.b1c9ba4363937p+39, INIT_U128( 0x101dc195, 0xa03b83d8e4dd21b1 ) },
  { 0x1.7a57356cf4ae6p+92, -0x1.052158e40a42bp+39, INIT_U128( 0x17a57356, 0xcf4ae57d6f538dfa ) },
  { 0x1.ca0e2f93941c6p+93, 0x1.80cee8a9019ddp+40, INIT_U128( 0x3941c5f2, 0x72838d80cee8a901 ) },
  { 0x1.3a1151fe7422ap+93, -0x1.ffaeb227ff5d7p+40, INIT_U128( 0x27422a3f, 0xce845200514dd800 ) },
  { 0x1.4c81542c9902ap+93, 0x1.ec96ea3dd92ddp+40, INIT_U128( 0x29902a85, 0x932055ec96ea3dd9 ) },
  { 0x1.41b58310836b0p+93, -0x1.5ef40b88bde82p+40, INIT_U128( 0x2836b062, 0x106d5ea10bf47742 ) },
  { 0x1.0805cb66100bap+93, 0x1.93abfb5b27580p+40, INIT_U128( 0x2100b96c, 0xc2017593abfb5b27 ) },
  { 0x1.9202b0c524056p+93, -0x1.4edfcd6a9dbfap+40, INIT_U128( 0x32405618, 0xa480aab120329562 ) },
  { 0x1.40a09ad881414p+94, 0x1.8fc4b5491f897p+41, INIT_U128( 0x502826b6, 0x2050531f896a923f ) },
  { 0x1.0b5c8db416b92p+94, -0x1.4f7c9b629ef94p+41, INIT_U128( 0x42d7236d, 0x5ae456106c93ac2 ) },
  { 0x1.d220f8aba441fp+94, 0x1.2d26ae965a4d6p+41, INIT_U128( 0x74883e2a, 0xe9107e5a4d5d2cb4 ) },
  { 0x1.182f0b98305e2p+94, -0x1.d49ceed1a939ep+41, INIT_U128( 0x460bc2e6, 0xc178456c6225cad ) },
  { 0x1.b1e8b7bd63d17p+94, 0x1.08168e6c102d2p+41, INIT_U128( 0x6c7a2def, 0x58f45e102d1cd820 ) },
  { 0x1.721ba512e4374p+94, -0x1.3b78fa6a76f20p+41, INIT_U128( 0x5c86e944, 0xb90dcd890e0b2b12 ) },
  { 0x1.4927aaf8924f6p+95, 0x1.c94a65c79294dp+42, INIT_U128( 0xa493d57c, 0x4927b72529971e4a ) },
  { 0x1.867eef0f0cfdep+95, -0x1.4b03c45496078p+42, INIT_U128( 0xc33f7787, 0x867eead3f0eeada7 ) },
  { 0x1.7b14200af6284p+95, 0x1.00d6247c01ac4p+42, INIT_U128( 0xbd8a1005, 0x7b1424035891f006 ) },
  { 0x1.50251d3aa04a4p+95, -0x1.45224f9e8a44ap+42, INIT_U128( 0xa8128e9d, 0x50251aeb76c185d6 ) },
  { 0x1.81a50d25034a2p+95, 0x1.8a366f39146cep+42, INIT_U128( 0xc0d28692, 0x81a51628d9bce451 ) },
  { 0x1.44d101c689a20p+95, -0x1.ce2661539c4ccp+42, INIT_U128( 0xa26880e3, 0x44d0f8c7667ab18e ) },
  { 0x1.76cd5976ed9abp+96, 0x1.868684d90d0d0p+43, INIT_U128( 0x176cd5976, 0xed9abc343426c868 ) },
  { 0x1.2d1a6cdc5a34ep+96, -0x1.7ed324defda64p+43, INIT_U128( 0x12d1a6cdc, 0x5a34d40966d90812 ) },
  { 0x1.1a32803434650p+96, 0x1.bf1d1da37e3a4p+43, INIT_U128( 0x11a328034, 0x34650df8e8ed1bf1 ) },
  { 0x1.cdb2f24b9b65ep+96, -0x1.5ad31b0eb5a64p+43, INIT_U128( 0x1cdb2f24b, 0x9b65d52967278a52 ) },
  { 0x1.4ee1d2ea9dc3ap+96, 0x1.d90f8001b21f0p+43, INIT_U128( 0x14ee1d2ea, 0x9dc3aec87c000d90 ) },
  { 0x1.aa4fb671549f7p+96, -0x1.2102cb964205ap+43, INIT_U128( 0x1aa4fb671, 0x549f66f7e9a34def ) },
  { 0x1.06362e960c6c6p+97, 0x1.c27ec20b84fd8p+44, INIT_U128( 0x20c6c5d2c, 0x18d8dc27ec20b84f ) },
  { 0x1.f59fba67eb3f8p+97, -0x1.b81ee655703ddp+44, INIT_U128( 0x3eb3f74cf, 0xd67ee47e119aa8fc ) },
  { 0x1.ca955077952aap+97, 0x1.67bb5f7acf76cp+44, INIT_U128( 0x3952aa0ef, 0x2a55567bb5f7acf7 ) },
  { 0x1.f397e0c1e72fcp+97, -0x1.13bcfb70277a0p+44, INIT_U128( 0x3e72fc183, 0xce5f6ec43048fd88 ) },
  { 0x1.3105c542620b8p+97, 0x1.7a16cfd0f42dap+44, INIT_U128( 0x2620b8a84, 0xc41717a16cfd0f42 ) },
  { 0x1.5ec0bfc6bd818p+97, -0x1.ac19e54f5833dp+44, INIT_U128( 0x2bd817f8d, 0x7b02e53e61ab0a7c ) },
  { 0x1.7aecf44ef5d9ep+98, 0x1.aee5dd035dcbcp+45, INIT_U128( 0x5ebb3d13b, 0xd767b5dcbba06bb9 ) },
  { 0x1.f13c34efe2786p+98, -0x1.fc920175f9240p+45, INIT_U128( 0x7c4f0d3bf, 0x89e1406dbfd140db ) },
  { 0x1.391c4c3c7238ap+98, 0x1.360d36b46c1a7p+45, INIT_U128( 0x4e47130f1, 0xc8e2a6c1a6d68d83 ) },
  { 0x1.cab862c99570dp+98, -0x1.afe9a5b55fd35p+45, INIT_U128( 0x72ae18b26, 0x55c30a02cb495405 ) },
  { 0x1.bc93bb8f79278p+98, 0x1.f7d2cfa7efa5ap+45, INIT_U128( 0x6f24eee3d, 0xe49e3efa59f4fdf4 ) },
  { 0x1.2ebcca945d79ap+98, -0x1.d6c63bc3ad8c8p+45, INIT_U128( 0x4baf32a51, 0x75e6452738878a4e ) },
  { 0x1.b6efed496ddfep+99, 0x1.fc6fbce1f8df7p+46, INIT_U128( 0xdb77f6a4b, 0x6eff7f1bef387e37 ) },
  { 0x1.319956d26332bp+99, -0x1.05d1c0340ba38p+46, INIT_U128( 0x98ccab693, 0x19953e8b8ff2fd17 ) },
  { 0x1.d57f3ce1aafe7p+99, 0x1.dbe90d45b7d21p+46, INIT_U128( 0xeabf9e70d, 0x57f3f6fa43516df4 ) },
  { 0x1.26d48bcc4da92p+99, -0x1.dbb56b79b76aep+46, INIT_U128( 0x936a45e62, 0x6d488912a5219225 ) },
  { 0x1.9e67cf4f3ccfap+99, 0x1.1c74463438e89p+46, INIT_U128( 0xcf33e7a79, 0xe67d471d118d0e3a ) },
  { 0x1.93d451cf27a8ap+99, -0x1.68fd1fc6d1fa4p+46, INIT_U128( 0xc9ea28e79, 0x3d44a5c0b80e4b81 ) },
  { 0x1.842fa5af085f5p+100, 0x1.876c32910ed86p+47, INIT_U128( 0x1842fa5af0, 0x85f5c3b61948876c ) },
  { 0x1.7cba1b38f9744p+100, -0x1.b5356af36a6adp+47, INIT_U128( 0x17cba1b38f, 0x974325654a864aca ) },
  { 0x1.0183026203060p+100, 0x1.db7adb01b6f5cp+47, INIT_U128( 0x1018302620, 0x3060edbd6d80db7a ) },
  { 0x1.614f5040c29eap+100, -0x1.7c6ff6b2f8dffp+47, INIT_U128( 0x1614f5040c, 0x29e941c804a68390 ) },
  { 0x1.8a60b80514c17p+100, 0x1.528b8530a5170p+47, INIT_U128( 0x18a60b8051, 0x4c17a945c298528b ) },
  { 0x1.7901f826f203fp+100, -0x1.c37d044186fa0p+47, INIT_U128( 0x17901f826f, 0x203e1e417ddf3c83 ) },
  { 0x1.8898bbe711318p+101, 0x1.43ba5af08774cp+48, INIT_U128( 0x3113177ce2, 0x263143ba5af08774 ) },
  { 0x1.7366a9f6e6cd5p+101, -0x1.452c77748a58fp+48, INIT_U128( 0x2e6cd53edc, 0xd9a8bad3888b75a7 ) },
  { 0x1.3b8d2a96771a6p+101, 0x1.a223748b4446ep+48, INIT_U128( 0x2771a552ce, 0xe34da223748b4446 ) },
  { 0x1.1bf2b33837e56p+101, -0x1.f27d17e1e4fa3p+48, INIT_U128( 0x237e566706, 0xfcaa0d82e81e1b05 ) },
  { 0x1.9e93316d3d266p+101, 0x1.4d8c0a0c9b182p+48, INIT_U128( 0x33d2662da7, 0xa4cd4d8c0a0c9b18 ) },
  { 0x1.f84ed817f09dbp+101, -0x1.74126e4ee824ep+48, INIT_U128( 0x3f09db02fe, 0x13b48bed91b117db ) },
  { 0x1.2975175052ea3p+102, 0x1.1f2363e43e46cp+49, INIT_U128( 0x4a5d45d414, 0xba8e3e46c7c87c8d ) },
  { 0x1.4fc7d9c89f8fbp+102, -0x1.c7dce6a18fb9dp+49, INIT_U128( 0x53f1f67227, 0xe3e8704632bce08c ) },
  { 0x1.3b97aa18772f6p+102, 0x1.e39c2b99c7386p+49, INIT_U128( 0x4ee5ea861d, 0xcbdbc73857338e70 ) },
  { 0x1.30da066861b41p+102, -0x1.bfc74fc97f8eap+49, INIT_U128( 0x4c36819a18, 0x6d008071606d00e2 ) },
  { 0x1.6b085e2ad610cp+102, 0x1.ad9421a35b284p+49, INIT_U128( 0x5ac2178ab5, 0x84335b284346b650 ) },
  { 0x1.74fcbabce9f98p+102, -0x1.6c8b608ed916cp+49, INIT_U128( 0x5d3f2eaf3a, 0x7e5d26e93ee24dd2 ) },
  { 0x1.890efca5121e0p+103, 0x1.d64c0affac982p+50, INIT_U128( 0xc4877e5289, 0xf0759302bfeb260 ) },
  { 0x1.e8fb5741d1f6bp+103, -0x1.136fb4aa26df6p+50, INIT_U128( 0xf47daba0e8, 0xfb53b2412d576482 ) },
  { 0x1.28d7902e51af2p+103, 0x1.61fd4f68c3faap+50, INIT_U128( 0x946bc81728, 0xd79587f53da30fea ) },
  { 0x1.d6ecb8edadd97p+103, -0x1.b1704c1762e0ap+50, INIT_U128( 0xeb765c76d6, 0xecb13a3ecfa2747d ) },
  { 0x1.a5b7aa5b4b6f5p+103, 0x1.4a5362e294a6cp+50, INIT_U128( 0xd2dbd52da5, 0xb7ad294d8b8a529b ) },
  { 0x1.f69cc38ded399p+103, -0x1.e144d4f3c289ap+50, INIT_U128( 0xfb4e61c6f6, 0x9cc07aecac30f5d9 ) },
  { 0x1.ec76d76fd8edbp+104, 0x1.128f76c6251efp+51, INIT_U128( 0x1ec76d76fd8, 0xedb8947bb63128f7 ) },
  { 0x1.d3c1ecd5a783dp+104, -0x1.8b0a08dd16141p+51, INIT_U128( 0x1d3c1ecd5a7, 0x83c3a7afb9174f5f ) },
  { 0x1.601b80d2c0370p+104, 0x1.cc10c4ed98218p+51, INIT_U128( 0x1601b80d2c0, 0x370e6086276cc10c ) },
  { 0x1.ff1c772bfe38fp+104, -0x1.8990982513213p+51, INIT_U128( 0x1ff1c772bfe, 0x38e3b37b3ed766f6 ) },
  { 0x1.eb3428d1d6685p+104, 0x1.4ee450f49dc8ap+51, INIT_U128( 0x1eb3428d1d6, 0x685a772287a4ee45 ) },
  { 0x1.516d7ec6a2db0p+104, -0x1.26cbd2ea4d97ap+51, INIT_U128( 0x1516d7ec6a2, 0xdaf6c9a168ad9343 ) },
  { 0x1.32807c8c65010p+105, 0x1.82478531048f1p+52, INIT_U128( 0x26500f918ca, 0x2182478531048f1 ) },
  { 0x1.e69322b5cd265p+105, -0x1.9d88dab93b11bp+52, INIT_U128( 0x3cd26456b9a, 0x4c862772546c4ee5 ) },
  { 0x1.2fdba6d45fb75p+105, 0x1.5b8fdd72b71fcp+52, INIT_U128( 0x25fb74da8bf, 0x6eb5b8fdd72b71fc ) },
  { 0x1.0a143b4614288p+105, -0x1.b8eaa37b71d54p+52, INIT_U128( 0x21428768c28, 0x50e47155c848e2ac ) },
  { 0x1.31dbe87063b7dp+105, 0x1.f30f9289e61f3p+52, INIT_U128( 0x263b7d0e0c7, 0x6fbf30f9289e61f3 ) },
  { 0x1.cc5a582798b4bp+105, -0x1.5ab378a0b566fp+52, INIT_U128( 0x398b4b04f31, 0x694a54c875f4a991 ) },
  { 0x1.550ce1f2aa19cp+106, 0x1.a3471113468e2p+53, INIT_U128( 0x5543387caa8, 0x673468e22268d1c4 ) },
  { 0x1.0c23c4b818478p+106, -0x1.fac48c4ff5891p+53, INIT_U128( 0x4308f12e061, 0x1dc0a76e76014ede ) },
  { 0x1.ed9a73f7db34ep+106, 0x1.02c0884405811p+53, INIT_U128( 0x7b669cfdf6c, 0xd3a058110880b022 ) },
  { 0x1.dc096141b812cp+106, -0x1.da114b67b422ap+53, INIT_U128( 0x770258506e0, 0x4ac4bdd693097bac ) },
  { 0x1.154fff0c2aa00p+106, 0x1.afcb4e175f96ap+53, INIT_U128( 0x4553ffc30aa, 0x8035f969c2ebf2d4 ) },
  { 0x1.651b5444ca36ap+106, -0x1.756b7fe0ead70p+53, INIT_U128( 0x5946d511328, 0xda51529003e2a520 ) },
  { 0x1.7bdc1302f7b82p+107, 0x1.6437cb70c86fap+54, INIT_U128( 0xbdee09817bd, 0xc1590df2dc321be8 ) },
  { 0x1.fefc1807fdf83p+107, -0x1.67103bd4ce208p+54, INIT_U128( 0xff7e0c03fef, 0xc1263bf10acc77e0 ) },
  { 0x1.b31c1c6566384p+107, 0x1.506996cea0d33p+54, INIT_U128( 0xd98e0e32b31, 0xc2541a65b3a834cc ) },
  { 0x1.8ddc0ec11bb82p+107, -0x1.e59c98f3cb393p+54, INIT_U128( 0xc6ee07608dd, 0xc08698d9c30d31b4 ) },
  { 0x1.d2b11c01a5624p+107, 0x1.746f3176e8de6p+54, INIT_U128( 0xe9588e00d2b, 0x125d1bcc5dba3798 ) },
  { 0x1.832573ad064aep+107, -0x1.028cc2d605198p+54, INIT_U128( 0xc192b9d6832, 0x56bf5ccf4a7eb9a0 ) },
  { 0x1.cfff0eeb9ffe2p+108, 0x1.8d39aef51a736p+55, INIT_U128( 0x1cfff0eeb9ff, 0xe2c69cd77a8d39b0 ) },
  { 0x1.bb77665376eedp+108, -0x1.01daf75c03b5fp+55, INIT_U128( 0x1bb77665376e, 0xec7f128451fe2508 ) },
  { 0x1.8049417f00928p+108, 0x1.87d1be790fa38p+55, INIT_U128( 0x18049417f009, 0x28c3e8df3c87d1c0 ) },
  { 0x1.5c8e6ffeb91cep+108, -0x1.d5562373aaac5p+55, INIT_U128( 0x15c8e6ffeb91, 0xcd1554ee462aa9d8 ) },
  { 0x1.2b18cf8c5631ap+108, 0x1.68b234b6d1646p+55, INIT_U128( 0x12b18cf8c563, 0x1ab4591a5b68b230 ) },
  { 0x1.d01aafc1a0356p+108, -0x1.18b3438431668p+55, INIT_U128( 0x1d01aafc1a03, 0x5573a65e3de74cc0 ) },
  { 0x1.6b72bcfcd6e58p+109, 0x1.ac24110358482p+56, INIT_U128( 0x2d6e579f9adc, 0xb1ac241103584820 ) },
  { 0x1.f2c48eede5892p+109, -0x1.80c2bc0b01858p+56, INIT_U128( 0x3e5891ddbcb1, 0x227f3d43f4fe7a80 ) },
  { 0x1.5bef5f1ab7decp+109, 0x1.57dfe332afbfcp+56, INIT_U128( 0x2b7debe356fb, 0xd957dfe332afbfc0 ) },
  { 0x1.f62e3bebec5c8p+109, -0x1.d7cb721faf96ep+56, INIT_U128( 0x3ec5c77d7d8b, 0x8e28348de0506920 ) },
  { 0x1.bc17aea3782f6p+109, 0x1.f52a53f3ea54ap+56, INIT_U128( 0x3782f5d46f05, 0xedf52a53f3ea54a0 ) },
  { 0x1.3edbadf47db76p+109, -0x1.6f33cb08de67ap+56, INIT_U128( 0x27db75be8fb6, 0xea90cc34f7219860 ) },
  { 0x1.4593eac08b27ep+110, 0x1.ccb7bcb7996f7p+57, INIT_U128( 0x5164fab022c9, 0xfb996f796f32dee0 ) },
  { 0x1.54d063fca9a0cp+110, -0x1.184506c2308a1p+57, INIT_U128( 0x553418ff2a68, 0x2dcf75f27b9eebe0 ) },
  { 0x1.4126ee2e824dep+110, 0x1.d7f5d7abafebbp+57, INIT_U128( 0x5049bb8ba093, 0x7bafebaf575fd760 ) },
  { 0x1.1ad1c99835a39p+110, -0x1.1125fd18224c0p+57, INIT_U128( 0x46b472660d68, 0xe1ddb405cfbb6800 ) },
  { 0x1.652f869eca5f1p+110, 0x1.d6492fedac926p+57, INIT_U128( 0x594be1a7b297, 0xc7ac925fdb5924c0 ) },
  { 0x1.1300f4762601ep+110, -0x1.e578c81fcaf19p+57, INIT_U128( 0x44c03d1d8980, 0x74350e6fc06a1ce0 ) },
  { 0x1.5e1aee66bc35ep+111, 0x1.6f8f6474df1ecp+58, INIT_U128( 0xaf0d77335e1a, 0xf5be3d91d37c7b00 ) },
  { 0x1.baceeb91759dep+111, -0x1.8933bdf512678p+58, INIT_U128( 0xdd6775c8bace, 0xe9db31082bb66200 ) },
  { 0x1.d841992db0833p+111, 0x1.8ae99b5315d34p+58, INIT_U128( 0xec20cc96d841, 0x9e2ba66d4c574d00 ) },
  { 0x1.a1e7017b43ce0p+111, -0x1.43b4562a8768bp+58, INIT_U128( 0xd0f380bda1e6, 0xfaf12ea755e25d40 ) },
  { 0x1.370dac566e1b6p+111, 0x1.94ab906929572p+58, INIT_U128( 0x9b86d62b370d, 0xb652ae41a4a55c80 ) },
  { 0x1.c477123788ee2p+111, -0x1.92fd0b7325fa2p+58, INIT_U128( 0xe23b891bc477, 0x9b40bd233681780 ) },
  { 0x1.ec8b739bd916fp+112, 0x1.baa54e23754aap+59, INIT_U128( 0x1ec8b739bd916, 0xfdd52a711baa5500 ) },
  { 0x1.6ac87d90d5910p+112, -0x1.a8a3223f51464p+59, INIT_U128( 0x16ac87d90d590, 0xf2bae6ee0575ce00 ) },
  { 0x1.afcfa2155f9f4p+112, 0x1.a9a42b8753486p+59, INIT_U128( 0x1afcfa2155f9f, 0x4d4d215c3a9a4300 ) },
  { 0x1.17d6ed482fadep+112, -0x1.f5638cd7eac71p+59, INIT_U128( 0x117d6ed482fad, 0xd054e39940a9c780 ) },
  { 0x1.765be83cecb7dp+112, 0x1.6ca75146d94eap+59, INIT_U128( 0x1765be83cecb7, 0xdb653a8a36ca7500 ) },
  { 0x1.0da8939a1b512p+112, -0x1.b782d66d6f05bp+59, INIT_U128( 0x10da8939a1b51, 0x1243e94c9487d280 ) },
  { 0x1.449d28f2893a5p+113, 0x1.287a5d3450f4cp+60, INIT_U128( 0x2893a51e51274, 0xb287a5d3450f4c00 ) },
  { 0x1.c8a8d3879151ap+113, -0x1.a9ccaf9353996p+60, INIT_U128( 0x39151a70f22a3, 0x25633506cac66a00 ) },
  { 0x1.fec38d21fd871p+113, 0x1.dba9c7fdb7539p+60, INIT_U128( 0x3fd871a43fb0e, 0x3dba9c7fdb753900 ) },
  { 0x1.8f4577b11e8afp+113, -0x1.5cde7b88b9bd0p+60, INIT_U128( 0x31e8aef623d15, 0xca32184774643000 ) },
  { 0x1.4b392c7296726p+113, 0x1.4a1463429428cp+60, INIT_U128( 0x2967258e52ce4, 0xd4a1463429428c00 ) },
  { 0x1.e612397dcc247p+113, -0x1.237ac5a046f58p+60, INIT_U128( 0x3cc2472fb9848, 0xcdc853a5fb90a800 ) },
  { 0x1.d4fdfe97a9fc0p+114, 0x1.87f93af90ff27p+61, INIT_U128( 0x753f7fa5ea7f0, 0x30ff275f21fe4e00 ) },
  { 0x1.672f1bccce5e4p+114, -0x1.0c0baaee18176p+61, INIT_U128( 0x59cbc6f333978, 0xde7e8aa23cfd1400 ) },
  { 0x1.1ca4cb603949ap+114, 0x1.2ba62e04574c6p+61, INIT_U128( 0x472932d80e526, 0xa574c5c08ae98c00 ) },
  { 0x1.9f08b6633e117p+114, -0x1.09303efc12608p+61, INIT_U128( 0x67c22d98cf845, 0x9ed9f8207db3f000 ) },
  { 0x1.e64b7c27cc970p+114, 0x1.09d133f413a26p+61, INIT_U128( 0x7992df09f325c, 0x213a267e82744c00 ) },
  { 0x1.b1d3613d63a6cp+114, -0x1.961b1e3d2c364p+61, INIT_U128( 0x6c74d84f58e9a, 0xcd3c9c385a793800 ) },
  { 0x1.b3755f7b66eacp+115, 0x1.c9aa4e879354ap+62, INIT_U128( 0xd9baafbdb3756, 0x726a93a1e4d52800 ) },
  { 0x1.8b7eab1516fd6p+115, -0x1.29d147b853a29p+62, INIT_U128( 0xc5bf558a8b7ea, 0xb58bae11eb175c00 ) },
  { 0x1.36519fce6ca34p+115, 0x1.99273eb1324e8p+62, INIT_U128( 0x9b28cfe73651a, 0x6649cfac4c93a000 ) },
  { 0x1.e1541e27c2a84p+115, -0x1.7aeec8c2f5dd9p+62, INIT_U128( 0xf0aa0f13e1541, 0xa1444dcf42889c00 ) },
  { 0x1.c1a19be583434p+115, 0x1.a9fcf1ad53f9ep+62, INIT_U128( 0xe0d0cdf2c1a1a, 0x6a7f3c6b54fe7800 ) },
  { 0x1.24bb78544976fp+115, -0x1.c065290780ca5p+62, INIT_U128( 0x925dbc2a24bb7, 0xfe6b5be1fcd6c00 ) },
  { 0x1.85a6b9ab0b4d7p+116, 0x1.b400f0076801ep+63, INIT_U128( 0x185a6b9ab0b4d7, 0xda007803b400f000 ) },
  { 0x1.1d43a4903a874p+116, -0x1.d0dfe61fa1bfdp+63, INIT_U128( 0x11d43a4903a873, 0x17900cf02f201800 ) },
  { 0x1.fe15534bfc2abp+116, 0x1.0355de0e06abcp+63, INIT_U128( 0x1fe15534bfc2ab, 0x81aaef070355e000 ) },
  { 0x1.f45aabc1e8b56p+116, -0x1.d31351e3a626ap+63, INIT_U128( 0x1f45aabc1e8b55, 0x1676570e2cecb000 ) },
  { 0x1.8b4f1907169e3p+116, 0x1.5b89c78eb7139p+63, INIT_U128( 0x18b4f1907169e3, 0xadc4e3c75b89c800 ) },
  { 0x1.df1efcd3be3dfp+116, -0x1.a7f3e4694fe7cp+63, INIT_U128( 0x1df1efcd3be3de, 0x2c060dcb580c2000 ) },
  { 0x1.872ff5950e5ffp+117, 0x1.7e27b336fc4f6p+64, INIT_U128( 0x30e5feb2a1cbff, 0x7e27b336fc4f6000 ) },
  { 0x1.b124236762484p+117, -0x1.6d7b71bcdaf6ep+64, INIT_U128( 0x3624846cec4906, 0x92848e4325092000 ) },
  { 0x1.87f8e72b0ff1dp+117, 0x1.62f784a4c5ef0p+64, INIT_U128( 0x30ff1ce561fe3b, 0x62f784a4c5ef0000 ) },
  { 0x1.03968ffa072d2p+117, -0x1.3b6e46a676dc9p+64, INIT_U128( 0x2072d1ff40e5a2, 0xc491b95989237000 ) },
  { 0x1.2ca1d32c5943ap+117, 0x1.8f153b111e2a8p+64, INIT_U128( 0x25943a658b2875, 0x8f153b111e2a8000 ) },
  { 0x1.653f8e60ca7f2p+117, -0x1.62f372fcc5e6ep+64, INIT_U128( 0x2ca7f1cc194fe2, 0x9d0c8d033a192000 ) },
  { 0x1.283c79c05078fp+118, 0x1.4693ac888d276p+65, INIT_U128( 0x4a0f1e70141e3e, 0x8d2759111a4ec000 ) },
  { 0x1.dde23bc1bbc48p+118, -0x1.e9de15a7d3bc2p+65, INIT_U128( 0x77788ef06ef11c, 0x2c43d4b05887c000 ) },
  { 0x1.f0909e37e1214p+118, 0x1.029db416053b6p+65, INIT_U128( 0x7c24278df84852, 0x53b682c0a76c000 ) },
  { 0x1.aa433eaf54868p+118, -0x1.763757e2ec6ebp+65, INIT_U128( 0x6a90cfabd5219d, 0x1391503a2722a000 ) },
  { 0x1.b69ea41f6d3d4p+118, 0x1.c786e3418f0ddp+65, INIT_U128( 0x6da7a907db4f53, 0x8f0dc6831e1ba000 ) },
  { 0x1.47173e028e2e8p+118, -0x1.b4b3f76b6967fp+65, INIT_U128( 0x51c5cf80a38b9c, 0x969811292d302000 ) },
  { 0x1.cadcc14f95b98p+119, 0x1.3b71e9c476e3dp+66, INIT_U128( 0xe56e60a7cadcc4, 0xedc7a711db8f4000 ) },
  { 0x1.87c7494f0f8e9p+119, -0x1.562bda56ac57cp+66, INIT_U128( 0xc3e3a4a787c742, 0xa75096a54ea10000 ) },
  { 0x1.96fbb1552df76p+119, 0x1.66e64622cdcc9p+66, INIT_U128( 0xcb7dd8aa96fbb5, 0x9b99188b37324000 ) },
  { 0x1.fe1fbec1fc3f8p+119, -0x1.1e2ba6e83c575p+66, INIT_U128( 0xff0fdf60fe1fbb, 0x8751645f0ea2c000 ) },
  { 0x1.e50df959ca1bfp+119, 0x1.af43e2295e87cp+66, INIT_U128( 0xf286fcace50dfe, 0xbd0f88a57a1f0000 ) },
  { 0x1.8ba718a7174e3p+119, -0x1.6d06e084da0dcp+66, INIT_U128( 0xc5d38c538ba712, 0x4be47dec97c90000 ) },
  { 0x1.123b66ce2476dp+120, 0x1.8f3da8a11e7b5p+67, INIT_U128( 0x1123b66ce2476dc, 0x79ed4508f3da8000 ) },
  { 0x1.a875272d50ea5p+120, -0x1.3edd39fc7dba7p+67, INIT_U128( 0x1a875272d50ea46, 0x916301c122c8000 ) },
  { 0x1.1e52e3b23ca5cp+120, 0x1.0db4b58e1b696p+67, INIT_U128( 0x11e52e3b23ca5c8, 0x6da5ac70db4b0000 ) },
  { 0x1.a1d0acfb43a16p+120, -0x1.468f5d748d1ecp+67, INIT_U128( 0x1a1d0acfb43a155, 0xcb85145b970a0000 ) },
  { 0x1.54486218a890cp+120, 0x1.7fb2e8bcff65dp+67, INIT_U128( 0x154486218a890cb, 0xfd9745e7fb2e8000 ) },
  { 0x1.306d731c60daep+120, -0x1.44d9004e89b20p+67, INIT_U128( 0x1306d731c60dad5, 0xd937fd8bb2700000 ) },
  { 0x1.82509cd304a14p+121, 0x1.dab0b1cdb5616p+68, INIT_U128( 0x304a139a609429d, 0xab0b1cdb56160000 ) },
  { 0x1.bb1058337620bp+121, -0x1.38ef40f271de8p+68, INIT_U128( 0x37620b066ec414c, 0x710bf0d8e2180000 ) },
  { 0x1.a237950d446f3p+121, 0x1.02279634044f3p+68, INIT_U128( 0x3446f2a1a88de70, 0x2279634044f30000 ) },
  { 0x1.eda3385ddb467p+121, -0x1.6d14565cda28bp+68, INIT_U128( 0x3db4670bbb68cc9, 0x2eba9a325d750000 ) },
  { 0x1.3d997ffa7b330p+121, 0x1.756a81acead50p+68, INIT_U128( 0x27b32fff4f66617, 0x56a81acead500000 ) },
  { 0x1.c34030b586806p+121, -0x1.d495314fa92a6p+68, INIT_U128( 0x38680616b0d00a2, 0xb6aceb056d5a0000 ) },
  { 0x1.dc50c7cfb8a19p+122, 0x1.c15fef7782bfep+69, INIT_U128( 0x771431f3ee28678, 0x2bfdeef057fc0000 ) },
  { 0x1.f2c0d839e581bp+122, -0x1.c15ec12982bd8p+69, INIT_U128( 0x7cb0360e7960687, 0xd427dacfa8500000 ) },
  { 0x1.70a3d1a0e147ap+122, 0x1.7e67f0e0fccfep+69, INIT_U128( 0x5c28f4683851eaf, 0xccfe1c1f99fc0000 ) },
  { 0x1.2e65a1ae5ccb4p+122, -0x1.82df387105be7p+69, INIT_U128( 0x4b99686b9732ccf, 0xa418f1df48320000 ) },
  { 0x1.0da599821b4b3p+122, 0x1.d6dac8ddadb59p+69, INIT_U128( 0x4369666086d2cfa, 0xdb591bb5b6b20000 ) },
  { 0x1.c1bc726f8378ep+122, -0x1.2bf87d3457f10p+69, INIT_U128( 0x706f1c9be0de35a, 0x80f0597501e00000 ) },
  { 0x1.e48f7e6dc91f0p+123, 0x1.638d1f6ac71a4p+70, INIT_U128( 0xf247bf36e48f858, 0xe347dab1c6900000 ) },
  { 0x1.7287daaae50fcp+123, -0x1.38d7e08671afcp+70, INIT_U128( 0xb943ed557287db1, 0xca07de6394100000 ) },
  { 0x1.e3400829c6801p+123, 0x1.a2f54dc745eaap+70, INIT_U128( 0xf1a00414e3400e8, 0xbd5371d17aa80000 ) },
  { 0x1.7db0e0d6fb61cp+123, -0x1.6590a4fccb214p+70, INIT_U128( 0xbed8706b7db0da6, 0x9bd6c0cd37b00000 ) },
  { 0x1.7da5ff94fb4c0p+123, 0x1.38c1390871827p+70, INIT_U128( 0xbed2ffca7da604e, 0x304e421c609c0000 ) },
  { 0x1.9e7fe5ef3cffdp+123, -0x1.1fdd94a23fbb2p+70, INIT_U128( 0xcf3ff2f79e7fe38, 0x89ad77011380000 ) },
  { 0x1.3ae8cf3e75d1ap+124, 0x1.8c231e4d18464p+71, INIT_U128( 0x13ae8cf3e75d1ac6, 0x118f268c23200000 ) },
  { 0x1.8cf1eb0119e3ep+124, -0x1.78824f3af104ap+71, INIT_U128( 0x18cf1eb0119e3d43, 0xbed862877db00000 ) },
  { 0x1.018d9ffa031b4p+124, 0x1.50321bb6a0644p+71, INIT_U128( 0x1018d9ffa031b4a8, 0x190ddb5032200000 ) },
  { 0x1.4d17808a9a2f0p+124, -0x1.ddde67cbbbbcdp+71, INIT_U128( 0x14d17808a9a2ef11, 0x10cc1a2221980000 ) },
  { 0x1.11920b2e23242p+124, 0x1.3fd858c47fb0bp+71, INIT_U128( 0x111920b2e232429f, 0xec2c623fd8580000 ) },
  { 0x1.9f3d28f53e7a5p+124, -0x1.8235dcd1046bcp+71, INIT_U128( 0x19f3d28f53e7a43e, 0xe511977dca200000 ) },
  { 0x1.be4049a57c809p+125, 0x1.cda2caa59b45ap+72, INIT_U128( 0x37c80934af9013cd, 0xa2caa59b45a00000 ) },
  { 0x1.051515420a2a2p+125, -0x1.cbe5e32997cbdp+72, INIT_U128( 0x20a2a2a841454234, 0x1a1cd66834300000 ) },
  { 0x1.a47d938348fb2p+125, 0x1.c6d187b38da31p+72, INIT_U128( 0x348fb270691f65c6, 0xd187b38da3100000 ) },
  { 0x1.f7de605fefbccp+125, -0x1.890d11f1121a2p+72, INIT_U128( 0x3efbcc0bfdf79676, 0xf2ee0eede5e00000 ) },
  { 0x1.2a5ea71e54bd5p+125, 0x1.6a663b08d4cc8p+72, INIT_U128( 0x254bd4e3ca97ab6a, 0x663b08d4cc800000 ) },
  { 0x1.c1e4f27783c9ep+125, -0x1.0d9eaf461b3d6p+72, INIT_U128( 0x383c9e4ef0793af2, 0x6150b9e4c2a00000 ) },
  { 0x1.0d5b88d01ab71p+126, 0x1.3f95d34e7f2bap+73, INIT_U128( 0x4356e23406adc67f, 0x2ba69cfe57400000 ) },
  { 0x1.732f5444e65eap+126, -0x1.8b01886716031p+73, INIT_U128( 0x5ccbd5113997a4e9, 0xfcef31d3f9e00000 ) },
  { 0x1.78570c58f0ae2p+126, 0x1.11af3a34235e8p+73, INIT_U128( 0x5e15c3163c2b8a23, 0x5e746846bd000000 ) },
  { 0x1.aadf1d0955be4p+126, -0x1.b33fdb99667fcp+73, INIT_U128( 0x6ab7c742556f8c99, 0x8048cd3300800000 ) },
  { 0x1.9dd258833ba4bp+126, 0x1.37d1080a6fa21p+73, INIT_U128( 0x67749620cee92e6f, 0xa21014df44200000 ) },
  { 0x1.2bc22ad257846p+126, -0x1.9f5ff87d3ebffp+73, INIT_U128( 0x4af08ab495e114c1, 0x400f058280200000 ) },
  { -0x1.9a22227b34444p+0, -0x1.05d9d02c0bb3ap-53, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { -0x1.be1c2ac37c385p+0, 0x1.2254205c44a84p-53, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { -0x1.8b79847916f30p+0, -0x1.515b66aaa2b6dp-53, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { -0x1.a322b51546457p+0, 0x1.6d405aa0da80cp-53, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { -0x1.4f9b9df49f374p+0, -0x1.a6d9873d4db31p-53, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { -0x1.26b4f6de4d69fp+0, 0x1.a6e6262b4dcc5p-53, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { -0x1.ed76bc33daed8p+1, -0x1.ac3a776d5874fp-52, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffd ) },
  { -0x1.87f39b850fe74p+1, 0x1.fc15cd99f82b9p-52, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffd ) },
  { -0x1.c52002bd8a401p+1, -0x1.a783df434f07cp-52, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffd ) },
  { -0x1.46a0aef28d416p+1, 0x1.9dbb2fbf3b766p-52, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffe ) },
  { -0x1.421e4dee843cap+1, -0x1.97155e672e2acp-52, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffe ) },
  { -0x1.a51ea2714a3d4p+1, 0x1.0c7e935c18fd2p-52, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffd ) },
  { -0x1.0fef3d221fde8p+2, -0x1.bea2cd2b7d45ap-51, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffc ) },
  { -0x1.f9918603f3231p+2, 0x1.18de594e31bcbp-51, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffff9 ) },
  { -0x1.37ae92126f5d2p+2, -0x1.1fba676c3f74dp-51, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffc ) },
  { -0x1.0136db2e026dcp+2, 0x1.d1d0b48fa3a16p-51, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffc ) },
  { -0x1.2594379a4b287p+2, -0x1.bf5305f37ea61p-51, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffffc ) },
  { -0x1.f424d4ede849ap+2, 0x1.b10dbc15621b8p-51, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffff9 ) },
  { -0x1.10ae6c9c215cep+3, -0x1.97478a012e8f1p-50, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffff8 ) },
  { -0x1.1e4e16b63c9c3p+3, 0x1.604a0a92c0942p-50, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffff8 ) },
  { -0x1.3e21113e7c422p+3, -0x1.45030d948a062p-50, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffff7 ) },
  { -0x1.073030be0e606p+3, 0x1.2b97cd72572fap-50, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffff8 ) },
  { -0x1.f13d8503e27b0p+3, -0x1.8f23cc451e47ap-50, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffff1 ) },
  { -0x1.27ad9b0c4f5b4p+3, 0x1.b65d87c16cbb1p-50, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffff7 ) },
  { -0x1.36a7ab866d4f6p+4, -0x1.6e4e4a00dc9cap-49, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffed ) },
  { -0x1.5418b780a8317p+4, 0x1.78c5f976f18bfp-49, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffeb ) },
  { -0x1.0563a8680ac75p+4, -0x1.f93759f3f26ebp-49, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffff0 ) },
  { -0x1.85448cd50a892p+4, 0x1.1552e58c2aa5cp-49, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffe8 ) },
  { -0x1.b7da271d6fb45p+4, -0x1.7ed612d8fdac2p-49, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffe5 ) },
  { -0x1.2e313edc5c628p+4, 0x1.ef88b931df117p-49, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffee ) },
  { -0x1.9e907a473d20fp+5, -0x1.2f681a0a5ed04p-48, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffcd ) },
  { -0x1.c1596dc182b2ep+5, 0x1.c424b1e388496p-48, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffc8 ) },
  { -0x1.eebb1fffdd764p+5, -0x1.b57e42af6afc8p-48, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffc3 ) },
  { -0x1.75326df6ea64ep+5, 0x1.ff698c9dfed31p-48, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffd2 ) },
  { -0x1.4cc5ccb0998bap+5, -0x1.938084af27010p-48, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffd7 ) },
  { -0x1.5fb39730bf673p+5, 0x1.8ae6ddf115cdcp-48, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffd5 ) },
  { -0x1.d8839245b1072p+6, -0x1.66e3c7f0cdc79p-47, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff8a ) },
  { -0x1.b67eab656cfd6p+6, 0x1.c9c1174993823p-47, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff93 ) },
  { -0x1.f6079435ec0f2p+6, -0x1.de2c4673bc589p-47, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff83 ) },
  { -0x1.801e9f09003d4p+6, 0x1.2caf3fbc595e8p-47, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffa0 ) },
  { -0x1.4c7a907498f52p+6, -0x1.d437568ba86ebp-47, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffad ) },
  { -0x1.a57539354aea7p+6, 0x1.51de38dea3bc7p-47, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff97 ) },
  { -0x1.cd6eb07d9add6p+7, -0x1.2ab9c60a55739p-46, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff1a ) },
  { -0x1.67311e6cce624p+7, 0x1.8548d79b0a91bp-46, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff4d ) },
  { -0x1.a98fd8e5531fbp+7, -0x1.95625d492ac4cp-46, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff2c ) },
  { -0x1.74d190cce9a32p+7, 0x1.4820532c9040ap-46, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff46 ) },
  { -0x1.c4ca77558994fp+7, -0x1.362afe8e6c560p-46, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff1e ) },
  { -0x1.0c450510188a0p+7, 0x1.b3859755670b3p-46, INIT_U128( 0xffffffffffffffff, 0xffffffffffffff7a ) },
  { -0x1.eba9413fd7528p+8, -0x1.8177730902eeep-45, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffe15 ) },
  { -0x1.b2ef23f165de4p+8, 0x1.386f0dee70de2p-45, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffe4e ) },
  { -0x1.14f7f7b629effp+8, -0x1.12a2bb2225458p-45, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffeec ) },
  { -0x1.c355ebe186abep+8, 0x1.ed7b89fddaf71p-45, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffe3d ) },
  { -0x1.79868312f30d0p+8, -0x1.79d49744f3a93p-45, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffe87 ) },
  { -0x1.b73ca1456e794p+8, 0x1.6f8e1748df1c3p-45, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffe49 ) },
  { -0x1.5800ddb8b001cp+9, -0x1.375b404e6eb68p-44, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffd50 ) },
  { -0x1.9c3d5707387abp+9, 0x1.a47b6e2d48f6ep-44, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffcc8 ) },
  { -0x1.0b9296da17253p+9, -0x1.41b2903c83652p-44, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffde9 ) },
  { -0x1.f659a70decb35p+9, 0x1.d9014759b2029p-44, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffc14 ) },
  { -0x1.6c6c5648d8d8bp+9, -0x1.5d8ac578bb158p-44, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffd28 ) },
  { -0x1.5e4a1ef4bc944p+9, 0x1.15fc2f2e2bf86p-44, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffd44 ) },
  { -0x1.f2ed22c3e5da5p+10, -0x1.d31bafc3a6376p-43, INIT_U128( 0xffffffffffffffff, 0xfffffffffffff835 ) },
  { -0x1.5e1c825cbc390p+10, 0x1.b7b79a196f6f3p-43, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffa88 ) },
  { -0x1.0946ae52128d6p+10, -0x1.6a61876ed4c31p-43, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffbdb ) },
  { -0x1.6b3d316ed67a6p+10, 0x1.f4efef91e9dfep-43, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffa54 ) },
  { -0x1.ebd8fa77d7b20p+10, -0x1.1e2c55603c58ap-43, INIT_U128( 0xffffffffffffffff, 0xfffffffffffff851 ) },
  { -0x1.2d5efd805abe0p+10, 0x1.00d0f22e01a1ep-43, INIT_U128( 0xffffffffffffffff, 0xfffffffffffffb4b ) },
  { -0x1.30cf1082619e2p+11, -0x1.f0b4e963e169dp-42, INIT_U128( 0xffffffffffffffff, 0xfffffffffffff67a ) },
  { -0x1.ee4c7c2ddc990p+11, 0x1.aa55939754ab2p-42, INIT_U128( 0xffffffffffffffff, 0xfffffffffffff08e ) },
  { -0x1.6a8980a8d5130p+11, -0x1.a5891d734b124p-42, INIT_U128( 0xffffffffffffffff, 0xfffffffffffff4ac ) },
  { -0x1.19e3aae033c76p+11, 0x1.c28a5e618514cp-42, INIT_U128( 0xffffffffffffffff, 0xfffffffffffff731 ) },
  { -0x1.dce45dc3b9c8cp+11, -0x1.b62101e96c420p-42, INIT_U128( 0xffffffffffffffff, 0xfffffffffffff119 ) },
  { -0x1.6705cc8ece0bap+11, 0x1.e876f49dd0edep-42, INIT_U128( 0xffffffffffffffff, 0xfffffffffffff4c8 ) },
  { -0x1.f7d39227efa73p+12, -0x1.5d5f739cbabeep-41, INIT_U128( 0xffffffffffffffff, 0xffffffffffffe083 ) },
  { -0x1.c1783bf782f08p+12, 0x1.643fe870c87fdp-41, INIT_U128( 0xffffffffffffffff, 0xffffffffffffe3e9 ) },
  { -0x1.baea391775d47p+12, -0x1.1fc25aea3f84cp-41, INIT_U128( 0xffffffffffffffff, 0xffffffffffffe452 ) },
  { -0x1.7a3c17a0f4783p+12, 0x1.add75bdb5baecp-41, INIT_U128( 0xffffffffffffffff, 0xffffffffffffe85d ) },
  { -0x1.f2de0aafe5bc2p+12, -0x1.d85899ffb0b13p-41, INIT_U128( 0xffffffffffffffff, 0xffffffffffffe0d3 ) },
  { -0x1.658ef5f4cb1dep+12, 0x1.fc24b901f8497p-41, INIT_U128( 0xffffffffffffffff, 0xffffffffffffe9a8 ) },
  { -0x1.42ba216e85744p+13, -0x1.d0cc2763a1985p-40, INIT_U128( 0xffffffffffffffff, 0xffffffffffffd7a9 ) },
  { -0x1.f114a893e2295p+13, 0x1.2e931be65d264p-40, INIT_U128( 0xffffffffffffffff, 0xffffffffffffc1de ) },
  { -0x1.eef87cc5ddf0fp+13, -0x1.1e73a6143ce75p-40, INIT_U128( 0xffffffffffffffff, 0xffffffffffffc221 ) },
  { -0x1.2f640e165ec82p+13, 0x1.1fc78d483f8f2p-40, INIT_U128( 0xffffffffffffffff, 0xffffffffffffda14 ) },
  { -0x1.0f288f781e512p+13, -0x1.1db08a423b612p-40, INIT_U128( 0xffffffffffffffff, 0xffffffffffffde1b ) },
  { -0x1.ca1d20e1943a4p+13, 0x1.79b21020f3642p-40, INIT_U128( 0xffffffffffffffff, 0xffffffffffffc6bd ) },
  { -0x1.c339a7b786735p+14, -0x1.e400cbc3c801ap-39, INIT_U128( 0xffffffffffffffff, 0xffffffffffff8f32 ) },
  { -0x1.3c3c6e827878ep+14, 0x1.a01e0579403c1p-39, INIT_U128( 0xffffffffffffffff, 0xffffffffffffb0f1 ) },
  { -0x1.9a21cdad3443ap+14, -0x1.a3423b1346848p-39, INIT_U128( 0xffffffffffffffff, 0xffffffffffff9978 ) },
  { -0x1.8894fa171129fp+14, 0x1.91f55fd523eacp-39, INIT_U128( 0xffffffffffffffff, 0xffffffffffff9ddb ) },
  { -0x1.00a1aeb001436p+14, -0x1.4a0d360e941a7p-39, INIT_U128( 0xffffffffffffffff, 0xffffffffffffbfd8 ) },
  { -0x1.f6354847ec6a9p+14, 0x1.bb8be7c77717dp-39, INIT_U128( 0xffffffffffffffff, 0xffffffffffff8273 ) },
  { -0x1.69cf90fad39f2p+15, -0x1.70715fe8e0e2cp-38, INIT_U128( 0xffffffffffffffff, 0xffffffffffff4b19 ) },
  { -0x1.696343a2d2c68p+15, 0x1.5cad9ba8b95b4p-38, INIT_U128( 0xffffffffffffffff, 0xffffffffffff4b4f ) },
  { -0x1.48c9f9e89193fp+15, -0x1.cef239979de47p-38, INIT_U128( 0xffffffffffffffff, 0xffffffffffff5b9c ) },
  { -0x1.58d254aab1a4ap+15, 0x1.8b841b5717084p-38, INIT_U128( 0xffffffffffffffff, 0xffffffffffff5397 ) },
  { -0x1.9fbe60fb3f7ccp+15, -0x1.49e6fd3e93ce0p-38, INIT_U128( 0xffffffffffffffff, 0xffffffffffff3021 ) },
  { -0x1.ba17373f742e7p+15, 0x1.8eb6ddc11d6dcp-38, INIT_U128( 0xffffffffffffffff, 0xffffffffffff22f5 ) },
  { -0x1.685aa354d0b54p+16, -0x1.e97b4555d2f68p-37, INIT_U128( 0xffffffffffffffff, 0xfffffffffffe97a6 ) },
  { -0x1.ae7e6b095cfcep+16, 0x1.778332ceef066p-37, INIT_U128( 0xffffffffffffffff, 0xfffffffffffe5182 ) },
  { -0x1.072bcf980e57ap+16, -0x1.789b8beaf1372p-37, INIT_U128( 0xffffffffffffffff, 0xfffffffffffef8d5 ) },
  { -0x1.f13542f1e26a9p+16, 0x1.ca65775194cafp-37, INIT_U128( 0xffffffffffffffff, 0xfffffffffffe0ecb ) },
  { -0x1.5c9c57acb938bp+16, -0x1.2d71b1745ae36p-37, INIT_U128( 0xffffffffffffffff, 0xfffffffffffea364 ) },
  { -0x1.6a837ccad5070p+16, 0x1.f6be255bed7c4p-37, INIT_U128( 0xffffffffffffffff, 0xfffffffffffe957d ) },
  { -0x1.d0b3ec89a167dp+17, -0x1.f31876e3e630fp-36, INIT_U128( 0xffffffffffffffff, 0xfffffffffffc5e99 ) },
  { -0x1.88b3852f11671p+17, 0x1.d1559b39a2ab4p-36, INIT_U128( 0xffffffffffffffff, 0xfffffffffffcee99 ) },
  { -0x1.3d25acf07a4b6p+17, -0x1.7ee8cd76fdd1ap-36, INIT_U128( 0xffffffffffffffff, 0xfffffffffffd85b5 ) },
  { -0x1.8ce1830119c30p+17, 0x1.a6f53ded4dea8p-36, INIT_U128( 0xffffffffffffffff, 0xfffffffffffce63d ) },
  { -0x1.ef5a2d5fdeb45p+17, -0x1.f644c6a5ec899p-36, INIT_U128( 0xffffffffffffffff, 0xfffffffffffc214c ) },
  { -0x1.03a2d9960745bp+17, 0x1.3824274870485p-36, INIT_U128( 0xffffffffffffffff, 0xfffffffffffdf8bb ) },
  { -0x1.c537003d8a6e0p+18, -0x1.5c752e42b8ea6p-35, INIT_U128( 0xffffffffffffffff, 0xfffffffffff8eb24 ) },
  { -0x1.c3a842a187509p+18, 0x1.64f56138c9eacp-35, INIT_U128( 0xffffffffffffffff, 0xfffffffffff8f15f ) },
  { -0x1.a65c2b834cb86p+18, -0x1.7dbf79defb7efp-35, INIT_U128( 0xffffffffffffffff, 0xfffffffffff96690 ) },
  { -0x1.f3ac3ef9e7588p+18, 0x1.0eb6ced81d6dap-35, INIT_U128( 0xffffffffffffffff, 0xfffffffffff83150 ) },
  { -0x1.673abf34ce758p+18, -0x1.a22aaa0344555p-35, INIT_U128( 0xffffffffffffffff, 0xfffffffffffa6316 ) },
  { -0x1.863a01a70c740p+18, 0x1.6e668ecedccd2p-35, INIT_U128( 0xffffffffffffffff, 0xfffffffffff9e718 ) },
  { -0x1.1ac635ee358c6p+19, -0x1.776f4498eede8p-34, INIT_U128( 0xffffffffffffffff, 0xfffffffffff729cf ) },
  { -0x1.38cc062071981p+19, 0x1.77628d9ceec52p-34, INIT_U128( 0xffffffffffffffff, 0xfffffffffff639a0 ) },
  { -0x1.a4e0f60d49c1fp+19, -0x1.a34f82eb469f0p-34, INIT_U128( 0xffffffffffffffff, 0xfffffffffff2d8f9 ) },
  { -0x1.6e20b2f6dc416p+19, 0x1.7594e296eb29cp-34, INIT_U128( 0xffffffffffffffff, 0xfffffffffff48efb ) },
  { -0x1.9667f9cf2ccffp+19, -0x1.f6d43827eda87p-34, INIT_U128( 0xffffffffffffffff, 0xfffffffffff34cc1 ) },
  { -0x1.46ea7dd08dd50p+19, 0x1.d38da6c1a71b5p-34, INIT_U128( 0xffffffffffffffff, 0xfffffffffff5c8ad ) },
  { -0x1.75bd059eeb7a0p+20, -0x1.d3cc00d3a7980p-33, INIT_U128( 0xffffffffffffffff, 0xffffffffffe8a430 ) },
  { -0x1.7a82e4acf505cp+20, 0x1.651732feca2e6p-33, INIT_U128( 0xffffffffffffffff, 0xffffffffffe857d2 ) },
  { -0x1.ca10c77794219p+20, -0x1.7e25be44fc4b8p-33, INIT_U128( 0xffffffffffffffff, 0xffffffffffe35ef4 ) },
  { -0x1.9d3b5a473a76bp+20, 0x1.8f47c7b51e8f9p-33, INIT_U128( 0xffffffffffffffff, 0xffffffffffe62c4b ) },
  { -0x1.da9aec87b535dp+20, -0x1.60e39ce8c1c74p-33, INIT_U128( 0xffffffffffffffff, 0xffffffffffe25652 ) },
  { -0x1.f43d28ede87a5p+20, 0x1.80f7180b01ee3p-33, INIT_U128( 0xffffffffffffffff, 0xffffffffffe0bc2e ) },
  { -0x1.dea316c9bd463p+21, -0x1.e7e967e7cfd2dp-32, INIT_U128( 0xffffffffffffffff, 0xffffffffffc42b9e ) },
  { -0x1.8fade6e31f5bdp+21, 0x1.45ddd5fe8bbbap-32, INIT_U128( 0xffffffffffffffff, 0xffffffffffce0a44 ) },
  { -0x1.8a1411eb14282p+21, -0x1.15e7e88a2bcfdp-32, INIT_U128( 0xffffffffffffffff, 0xffffffffffcebd7e ) },
  { -0x1.b44464cf6888cp+21, 0x1.a4da47db49b49p-32, INIT_U128( 0xffffffffffffffff, 0xffffffffffc97774 ) },
  { -0x1.8d572d251aae6p+21, -0x1.ed106aefda20ep-32, INIT_U128( 0xffffffffffffffff, 0xffffffffffce551b ) },
  { -0x1.1c3cd5763879ap+21, 0x1.3238233264704p-32, INIT_U128( 0xffffffffffffffff, 0xffffffffffdc7866 ) },
  { -0x1.905feddb20bfep+22, -0x1.8a5d886f14bb1p-31, INIT_U128( 0xffffffffffffffff, 0xffffffffff9be805 ) },
  { -0x1.a7cd05c94f9a1p+22, 0x1.26c7e7aa4d8fdp-31, INIT_U128( 0xffffffffffffffff, 0xffffffffff960cbf ) },
  { -0x1.8131c09702638p+22, -0x1.eeb7839bdd6f1p-31, INIT_U128( 0xffffffffffffffff, 0xffffffffff9fb390 ) },
  { -0x1.fa558e6bf4ab2p+22, 0x1.f6eec637eddd9p-31, INIT_U128( 0xffffffffffffffff, 0xffffffffff816a9d ) },
  { -0x1.c283846d85070p+22, -0x1.74d87318e9b0ep-31, INIT_U128( 0xffffffffffffffff, 0xffffffffff8f5f1f ) },
  { -0x1.5c05f934b80bfp+22, 0x1.8c944be51928ap-31, INIT_U128( 0xffffffffffffffff, 0xffffffffffa8fe82 ) },
  { -0x1.f2fe315fe5fc6p+23, -0x1.f941537df282bp-30, INIT_U128( 0xffffffffffffffff, 0xffffffffff0680e8 ) },
  { -0x1.1bdc139a37b82p+23, 0x1.cd991de79b324p-30, INIT_U128( 0xffffffffffffffff, 0xffffffffff7211f7 ) },
  { -0x1.5a24f066b449ep+23, -0x1.10193c8820328p-30, INIT_U128( 0xffffffffffffffff, 0xffffffffff52ed88 ) },
  { -0x1.4e9035f09d206p+23, 0x1.38c8072e71901p-30, INIT_U128( 0xffffffffffffffff, 0xffffffffff58b7e6 ) },
  { -0x1.f802a471f0054p+23, -0x1.de3e1cd5bc7c3p-30, INIT_U128( 0xffffffffffffffff, 0xffffffffff03feae ) },
  { -0x1.7ea5dd2efd4bcp+23, 0x1.8216b65d042d7p-30, INIT_U128( 0xffffffffffffffff, 0xffffffffff40ad12 ) },
  { -0x1.f4260561e84c0p+24, -0x1.32ea41fc65d48p-29, INIT_U128( 0xffffffffffffffff, 0xfffffffffe0bd9fb ) },
  { -0x1.26f0fe364de20p+24, 0x1.817d328702fa6p-29, INIT_U128( 0xffffffffffffffff, 0xfffffffffed90f02 ) },
  { -0x1.1ffaacec3ff56p+24, -0x1.432dd3ae865bap-29, INIT_U128( 0xffffffffffffffff, 0xfffffffffee00554 ) },
  { -0x1.b3b555b9676abp+24, 0x1.b05a9ac960b53p-29, INIT_U128( 0xffffffffffffffff, 0xfffffffffe4c4aab ) },
  { -0x1.cd8b5c1f9b16cp+24, -0x1.5b825b82b704cp-29, INIT_U128( 0xffffffffffffffff, 0xfffffffffe3274a4 ) },
  { -0x1.d7228275ae450p+24, 0x1.4ebd1cb69d7a4p-29, INIT_U128( 0xffffffffffffffff, 0xfffffffffe28dd7e ) },
  { -0x1.4a39df1e9473cp+25, -0x1.d17810e1a2f02p-28, INIT_U128( 0xffffffffffffffff, 0xfffffffffd6b8c42 ) },
  { -0x1.45abe2ee8b57cp+25, 0x1.0cbd638c197acp-28, INIT_U128( 0xffffffffffffffff, 0xfffffffffd74a83b ) },
  { -0x1.465083fa8ca10p+25, -0x1.a1b1dc254363cp-28, INIT_U128( 0xffffffffffffffff, 0xfffffffffd735ef9 ) },
  { -0x1.9951af7332a36p+25, 0x1.394eb558729d6p-28, INIT_U128( 0xffffffffffffffff, 0xfffffffffccd5ca2 ) },
  { -0x1.9af32fa335e66p+25, -0x1.b52dc30d6a5b8p-28, INIT_U128( 0xffffffffffffffff, 0xfffffffffcca19a1 ) },
  { -0x1.06e7d3400dcfap+25, 0x1.f5182009ea304p-28, INIT_U128( 0xffffffffffffffff, 0xfffffffffdf2305a ) },
  { -0x1.c546ff978a8e0p+26, -0x1.55780932aaf01p-27, INIT_U128( 0xffffffffffffffff, 0xfffffffff8eae402 ) },
  { -0x1.2de027385bc05p+26, 0x1.bd49a4077a934p-27, INIT_U128( 0xffffffffffffffff, 0xfffffffffb487f64 ) },
  { -0x1.33b62608676c5p+26, -0x1.ac860469590c0p-27, INIT_U128( 0xffffffffffffffff, 0xfffffffffb312768 ) },
  { -0x1.3f605a647ec0cp+26, 0x1.27dc2b6a4fb86p-27, INIT_U128( 0xffffffffffffffff, 0xfffffffffb027e97 ) },
  { -0x1.df704667bee09p+26, -0x1.6651589ccca2bp-27, INIT_U128( 0xffffffffffffffff, 0xfffffffff8823ee7 ) },
  { -0x1.a9595df152b2cp+26, 0x1.ff6af355fed5fp-27, INIT_U128( 0xffffffffffffffff, 0xfffffffff95a9a89 ) },
  { -0x1.a97f2c4d52fe6p+27, -0x1.5d0eb3aaba1d6p-26, INIT_U128( 0xffffffffffffffff, 0xfffffffff2b4069e ) },
  { -0x1.afc58e1f5f8b2p+27, 0x1.770a886cee151p-26, INIT_U128( 0xffffffffffffffff, 0xfffffffff281d390 ) },
  { -0x1.b8910f2f71222p+27, -0x1.86e810930dd02p-26, INIT_U128( 0xffffffffffffffff, 0xfffffffff23b7787 ) },
  { -0x1.c5c7a5258b8f4p+27, 0x1.02caee4e0595ep-26, INIT_U128( 0xffffffffffffffff, 0xfffffffff1d1c2d7 ) },
  { -0x1.58602174b0c04p+27, -0x1.0b73881216e71p-26, INIT_U128( 0xffffffffffffffff, 0xfffffffff53cfef5 ) },
  { -0x1.0f8851da1f10ap+27, 0x1.9eb0a56f3d615p-26, INIT_U128( 0xffffffffffffffff, 0xfffffffff783bd72 ) },
  { -0x1.ad2564395a4acp+28, -0x1.a8da014f51b40p-25, INIT_U128( 0xffffffffffffffff, 0xffffffffe52da9bd ) },
  { -0x1.d7ff5ac9affecp+28, 0x1.481893da90312p-25, INIT_U128( 0xffffffffffffffff, 0xffffffffe2800a54 ) },
  { -0x1.5e07c45cbc0f8p+28, -0x1.dee72e0bbdce6p-25, INIT_U128( 0xffffffffffffffff, 0xffffffffea1f83bb ) },
  { -0x1.3d30b3e47a616p+28, 0x1.234ec3f2469d8p-25, INIT_U128( 0xffffffffffffffff, 0xffffffffec2cf4c2 ) },
  { -0x1.345f373c68be7p+28, -0x1.6b10db1ed621cp-25, INIT_U128( 0xffffffffffffffff, 0xffffffffecba0c8d ) },
  { -0x1.e09867fbc130dp+28, 0x1.68155d46d02acp-25, INIT_U128( 0xffffffffffffffff, 0xffffffffe1f67981 ) },
  { -0x1.1796df882f2dcp+29, -0x1.1ff8c2603ff18p-24, INIT_U128( 0xffffffffffffffff, 0xffffffffdd0d240f ) },
  { -0x1.8ff188b11fe31p+29, 0x1.f70725efee0e5p-24, INIT_U128( 0xffffffffffffffff, 0xffffffffce01ceea ) },
  { -0x1.864a1afd0c943p+29, -0x1.394ae6a27295dp-24, INIT_U128( 0xffffffffffffffff, 0xffffffffcf36bca1 ) },
  { -0x1.f6721945ece43p+29, 0x1.2fc9474a5f929p-24, INIT_U128( 0xffffffffffffffff, 0xffffffffc131bcd8 ) },
  { -0x1.96599a4f2cb33p+29, -0x1.a637a7634c6f5p-24, INIT_U128( 0xffffffffffffffff, 0xffffffffcd34ccb7 ) },
  { -0x1.a6d3cfb74da7ap+29, 0x1.4eeaa97e9dd55p-24, INIT_U128( 0xffffffffffffffff, 0xffffffffcb25860a ) },
  { -0x1.2d1fb7f65a3f7p+30, -0x1.6c9b74dcd936ep-23, INIT_U128( 0xffffffffffffffff, 0xffffffffb4b81203 ) },
  { -0x1.51b597cea36b3p+30, 0x1.857fd96d0affbp-23, INIT_U128( 0xffffffffffffffff, 0xffffffffab929a0d ) },
  { -0x1.780efcf0f01e0p+30, -0x1.613de9aac27bdp-23, INIT_U128( 0xffffffffffffffff, 0xffffffffa1fc40c4 ) },
  { -0x1.24307edc48610p+30, 0x1.253461284a68cp-23, INIT_U128( 0xffffffffffffffff, 0xffffffffb6f3e049 ) },
  { -0x1.0a17eaf8142fep+30, -0x1.fc2fd9a7f85fbp-23, INIT_U128( 0xffffffffffffffff, 0xffffffffbd7a0542 ) },
  { -0x1.6d4cf504da99ep+30, 0x1.681faf56d03f6p-23, INIT_U128( 0xffffffffffffffff, 0xffffffffa4acc2bf ) },
  { -0x1.db1707b1b62e1p+31, -0x1.aa7da8eb54fb5p-22, INIT_U128( 0xffffffffffffffff, 0xffffffff12747c28 ) },
  { -0x1.8b6e734b16dcep+31, 0x1.0f763eee1eec8p-22, INIT_U128( 0xffffffffffffffff, 0xffffffff3a48c65b ) },
  { -0x1.158e840a2b1d0p+31, -0x1.6c06db46d80dcp-22, INIT_U128( 0xffffffffffffffff, 0xffffffff7538bdfb ) },
  { -0x1.778b9c36ef174p+31, 0x1.2d2563945a4acp-22, INIT_U128( 0xffffffffffffffff, 0xffffffff443a31e5 ) },
  { -0x1.8bff9da717ff4p+31, -0x1.077d24e80efa4p-22, INIT_U128( 0xffffffffffffffff, 0xffffffff3a00312d ) },
  { -0x1.242c898448591p+31, 0x1.1249b8a424937p-22, INIT_U128( 0xffffffffffffffff, 0xffffffff6de9bb3e ) },
  { -0x1.40c80b8c81902p+32, -0x1.1a9ea2c8353d4p-21, INIT_U128( 0xffffffffffffffff, 0xfffffffebf37f474 ) },
  { -0x1.4212fff084260p+32, 0x1.d721a5ddae435p-21, INIT_U128( 0xffffffffffffffff, 0xfffffffebded0010 ) },
  { -0x1.c0d64a2d81ac9p+32, -0x1.e8e6cfa9d1cdap-21, INIT_U128( 0xffffffffffffffff, 0xfffffffe3f29b5d3 ) },
  { -0x1.260c4f5c4c18ap+32, 0x1.edf60225dbec0p-21, INIT_U128( 0xffffffffffffffff, 0xfffffffed9f3b0a4 ) },
  { -0x1.55824484ab048p+32, -0x1.77c1e72aef83dp-21, INIT_U128( 0xffffffffffffffff, 0xfffffffeaa7dbb7c ) },
  { -0x1.7375db92e6ebcp+32, 0x1.cd9141779b228p-21, INIT_U128( 0xffffffffffffffff, 0xfffffffe8c8a246e ) },
  { -0x1.d8ffd0d7b1ffap+33, -0x1.97a65a6f2f4cbp-20, INIT_U128( 0xffffffffffffffff, 0xfffffffc4e005e51 ) },
  { -0x1.f2c5a2a1e58b5p+33, 0x1.e317bbd1c62f8p-20, INIT_U128( 0xffffffffffffffff, 0xfffffffc1a74babd ) },
  { -0x1.93d6341727ac6p+33, -0x1.601297a6c0253p-20, INIT_U128( 0xffffffffffffffff, 0xfffffffcd85397d2 ) },
  { -0x1.4b376b26966eep+33, 0x1.6eed3bc8ddda8p-20, INIT_U128( 0xffffffffffffffff, 0xfffffffd699129b3 ) },
  { -0x1.0a90409215208p+33, -0x1.d6a5de73ad4bcp-20, INIT_U128( 0xffffffffffffffff, 0xfffffffdeadf7edc ) },
  { -0x1.7e637ab8fcc70p+33, 0x1.201ec49e403d8p-20, INIT_U128( 0xffffffffffffffff, 0xfffffffd03390a8f ) },
  { -0x1.42acb9b885597p+34, -0x1.f5ef16efebde3p-19, INIT_U128( 0xffffffffffffffff, 0xfffffffaf54d191e ) },
  { -0x1.4d4428329a885p+34, 0x1.ceac57619d58bp-19, INIT_U128( 0xffffffffffffffff, 0xfffffffacaef5f36 ) },
  { -0x1.fd6c3bd9fad88p+34, -0x1.7170b1b8e2e16p-19, INIT_U128( 0xffffffffffffffff, 0xfffffff80a4f1099 ) },
  { -0x1.e0f61007c1ec2p+34, 0x1.3e3447647c689p-19, INIT_U128( 0xffffffffffffffff, 0xfffffff87c27bfe1 ) },
  { -0x1.8c0f5483181eap+34, -0x1.23090ff846122p-19, INIT_U128( 0xffffffffffffffff, 0xfffffff9cfc2adf4 ) },
  { -0x1.1555ed422aabep+34, 0x1.4ce59eb099cb4p-19, INIT_U128( 0xffffffffffffffff, 0xfffffffbaaa84af8 ) },
  { -0x1.0befdfa017dfcp+35, -0x1.3b623c9e76c48p-18, INIT_U128( 0xffffffffffffffff, 0xfffffff7a0810300 ) },
  { -0x1.3adba0d675b74p+35, 0x1.61722426c2e44p-18, INIT_U128( 0xffffffffffffffff, 0xfffffff62922f94d ) },
  { -0x1.b32423cb66484p+35, -0x1.ae517c695ca30p-18, INIT_U128( 0xffffffffffffffff, 0xfffffff266dee1a5 ) },
  { -0x1.2f03659e5e06cp+35, 0x1.8c23f4a31847ep-18, INIT_U128( 0xffffffffffffffff, 0xfffffff687e4d30e ) },
  { -0x1.45f7d6d88befbp+35, -0x1.21c9083e43921p-18, INIT_U128( 0xffffffffffffffff, 0xfffffff5d041493c ) },
  { -0x1.6f3bb072de776p+35, 0x1.d9ce0af1b39c2p-18, INIT_U128( 0xffffffffffffffff, 0xfffffff486227c6a ) },
  { -0x1.81db9fe703b74p+36, -0x1.ba731b9b74e64p-17, INIT_U128( 0xffffffffffffffff, 0xffffffe7e2460190 ) },
  { -0x1.48bb46b891769p+36, 0x1.8c6be07918d7cp-17, INIT_U128( 0xffffffffffffffff, 0xffffffeb744b9477 ) },
  { -0x1.9118fa0f2231fp+36, -0x1.c71ec1738e3d8p-17, INIT_U128( 0xffffffffffffffff, 0xffffffe6ee705f0e ) },
  { -0x1.ac8aa51959155p+36, 0x1.d3c5b3c9a78b6p-17, INIT_U128( 0xffffffffffffffff, 0xffffffe53755ae6b ) },
  { -0x1.bd0dd8637a1bbp+36, -0x1.f9cecd4df39d9p-17, INIT_U128( 0xffffffffffffffff, 0xffffffe42f2279c9 ) },
  { -0x1.a2720b2b44e42p+36, 0x1.ba7a143b74f42p-17, INIT_U128( 0xffffffffffffffff, 0xffffffe5d8df4d4c ) },
  { -0x1.6b3f7f04d67f0p+37, -0x1.83681b3306d04p-16, INIT_U128( 0xffffffffffffffff, 0xffffffd298101f66 ) },
  { -0x1.f8ae5ba1f15ccp+37, 0x1.f74ed389ee9dbp-16, INIT_U128( 0xffffffffffffffff, 0xffffffc0ea348bc2 ) },
  { -0x1.a6712b2b4ce26p+37, -0x1.0e0448e41c089p-16, INIT_U128( 0xffffffffffffffff, 0xffffffcb31da9a97 ) },
  { -0x1.443472388868ep+37, 0x1.b2610acd64c21p-16, INIT_U128( 0xffffffffffffffff, 0xffffffd77971b8ef ) },
  { -0x1.4966858492cd0p+37, -0x1.7f101310fe202p-16, INIT_U128( 0xffffffffffffffff, 0xffffffd6d32f4f6e ) },
  { -0x1.13d32ef427a66p+37, 0x1.fc8aa951f9155p-16, INIT_U128( 0xffffffffffffffff, 0xffffffdd859a217c ) },
  { -0x1.2d618f7a5ac32p+38, -0x1.42d6949485ad2p-15, INIT_U128( 0xffffffffffffffff, 0xffffffb4a79c216a ) },
  { -0x1.88ae9df3115d4p+38, 0x1.73596654e6b2dp-15, INIT_U128( 0xffffffffffffffff, 0xffffff9dd458833c ) },
  { -0x1.649f9cd4c93f4p+38, -0x1.f7ea4e67efd4ap-15, INIT_U128( 0xffffffffffffffff, 0xffffffa6d818cace ) },
  { -0x1.4d2771449a4eep+38, 0x1.e67b3cbbccf67p-15, INIT_U128( 0xffffffffffffffff, 0xffffffacb623aeda ) },
  { -0x1.b25d6a0364badp+38, -0x1.95e2b7ff2bc57p-15, INIT_U128( 0xffffffffffffffff, 0xffffff9368a57f27 ) },
  { -0x1.72e71d34e5ce4p+38, 0x1.4376641286eccp-15, INIT_U128( 0xffffffffffffffff, 0xffffffa34638b2c7 ) },
  { -0x1.5d017972ba02fp+39, -0x1.1f71c24e3ee38p-14, INIT_U128( 0xffffffffffffffff, 0xffffff517f4346a3 ) },
  { -0x1.173c17da2e783p+39, 0x1.1a0f51d4341eap-14, INIT_U128( 0xffffffffffffffff, 0xffffff7461f412e9 ) },
  { -0x1.19408f9a32812p+39, -0x1.b9ae2307735c4p-14, INIT_U128( 0xffffffffffffffff, 0xffffff735fb832e7 ) },
  { -0x1.d4896611a912dp+39, 0x1.84800ea109002p-14, INIT_U128( 0xffffffffffffffff, 0xffffff15bb4cf72c ) },
  { -0x1.3d163e387a2c8p+39, -0x1.cd37c1b39a6f8p-14, INIT_U128( 0xffffffffffffffff, 0xffffff6174e0e3c3 ) },
  { -0x1.7bcee22af79dcp+39, 0x1.e3876965c70edp-14, INIT_U128( 0xffffffffffffffff, 0xffffff42188eea85 ) },
  { -0x1.db3c0a99b6782p+40, -0x1.c00354658006ap-13, INIT_U128( 0xffffffffffffffff, 0xfffffe24c3f5664a ) },
  { -0x1.95e874332bd0ep+40, 0x1.24a2901c49452p-13, INIT_U128( 0xffffffffffffffff, 0xfffffe6a178bccd5 ) },
  { -0x1.3f1367747e26dp+40, -0x1.a9bba32953774p-13, INIT_U128( 0xffffffffffffffff, 0xfffffec0ec988b82 ) },
  { -0x1.212d396e425a7p+40, 0x1.6c74f6f0d8e9fp-13, INIT_U128( 0xffffffffffffffff, 0xfffffeded2c691be ) },
  { -0x1.ec9237bfd9247p+40, -0x1.a9dbd76353b7bp-13, INIT_U128( 0xffffffffffffffff, 0xfffffe136dc84027 ) },
  { -0x1.dfce5d45bf9cbp+40, 0x1.5131d492a263ap-13, INIT_U128( 0xffffffffffffffff, 0xfffffe2031a2ba41 ) },
  { -0x1.a1c625cb438c5p+41, -0x1.2cf5ce8a59ebap-12, INIT_U128( 0xffffffffffffffff, 0xfffffcbc73b46979 ) },
  { -0x1.37ad114e6f5a2p+41, 0x1.54238fcea8472p-12, INIT_U128( 0xffffffffffffffff, 0xfffffd90a5dd6322 ) },
  { -0x1.c2d8868b85b11p+41, -0x1.aa942e8555286p-12, INIT_U128( 0xffffffffffffffff, 0xfffffc7a4ef2e8f5 ) },
  { -0x1.9799f3e12f33ep+41, 0x1.1fd9fffe3fb40p-12, INIT_U128( 0xffffffffffffffff, 0xfffffcd0cc183da2 ) },
  { -0x1.ca05f0d5940bep+41, -0x1.aed60bbd5dac2p-12, INIT_U128( 0xffffffffffffffff, 0xfffffc6bf41e54d8 ) },
  { -0x1.39e951d473d2ap+41, 0x1.e3468071c68d0p-12, INIT_U128( 0xffffffffffffffff, 0xfffffd8c2d5c5719 ) },
  { -0x1.68842ec4d1086p+42, -0x1.0e72b7e61ce57p-11, INIT_U128( 0xffffffffffffffff, 0xfffffa5def44ecbc ) },
  { -0x1.67c68f12cf8d2p+42, 0x1.a59a6cff4b34ep-11, INIT_U128( 0xffffffffffffffff, 0xfffffa60e5c3b4c2 ) },
  { -0x1.dbaa799bb754fp+42, -0x1.e395713dc72aep-11, INIT_U128( 0xffffffffffffffff, 0xfffff89156199123 ) },
  { -0x1.8921d6651243bp+42, 0x1.b6e684336dcd0p-11, INIT_U128( 0xffffffffffffffff, 0xfffff9db78a66bb7 ) },
  { -0x1.a398c5a347319p+42, -0x1.1f0a4a983e14ap-11, INIT_U128( 0xffffffffffffffff, 0xfffff9719ce972e4 ) },
  { -0x1.db891451b7122p+42, 0x1.e2ac2d19c5585p-11, INIT_U128( 0xffffffffffffffff, 0xfffff891dbaeb924 ) },
  { -0x1.c8c5edc3918bep+43, -0x1.fcb64dc1f96c9p-10, INIT_U128( 0xffffffffffffffff, 0xfffff1b9d091e374 ) },
  { -0x1.4f2124089e424p+43, 0x1.b55825816ab05p-10, INIT_U128( 0xffffffffffffffff, 0xfffff586f6dfbb0e ) },
  { -0x1.a69225234d245p+43, -0x1.2eef814e5ddf0p-10, INIT_U128( 0xffffffffffffffff, 0xfffff2cb6ed6e597 ) },
  { -0x1.0689fa140d140p+43, 0x1.48584aee90b0ap-10, INIT_U128( 0xffffffffffffffff, 0xfffff7cbb02f5f98 ) },
  { -0x1.5be54fdab7caap+43, -0x1.3e370b647c6e2p-10, INIT_U128( 0xffffffffffffffff, 0xfffff520d5812a42 ) },
  { -0x1.9c7bdabd38f7bp+43, 0x1.1ebdd6663d7bbp-10, INIT_U128( 0xffffffffffffffff, 0xfffff31c212a1639 ) },
  { -0x1.e8cb39ebd1967p+44, -0x1.3415ce9e682bap-9, INIT_U128( 0xffffffffffffffff, 0xffffe1734c6142e7 ) },
  { -0x1.3e97d6667d2fbp+44, 0x1.b2d12ac165a25p-9, INIT_U128( 0xffffffffffffffff, 0xffffec168299982e ) },
  { -0x1.e2ebda5dc5d7cp+44, -0x1.7881283af1025p-9, INIT_U128( 0xffffffffffffffff, 0xffffe1d1425a23a3 ) },
  { -0x1.9617ab312c2f6p+44, 0x1.4b70092296e01p-9, INIT_U128( 0xffffffffffffffff, 0xffffe69e854ced3e ) },
  { -0x1.86f3e0230de7cp+44, -0x1.fdde3a45fbbc8p-9, INIT_U128( 0xffffffffffffffff, 0xffffe790c1fdcf22 ) },
  { -0x1.f10a7621e214fp+44, 0x1.629e59bcc53cbp-9, INIT_U128( 0xffffffffffffffff, 0xffffe0ef589de1df ) },
  { -0x1.e173ab83c2e76p+45, -0x1.7a2c4c86f458ap-8, INIT_U128( 0xffffffffffffffff, 0xffffc3d18a8f87a4 ) },
  { -0x1.1984ddf03309cp+45, 0x1.850c71250a18ep-8, INIT_U128( 0xffffffffffffffff, 0xffffdccf6441f99f ) },
  { -0x1.9936971f326d3p+45, -0x1.f50df243ea1bfp-8, INIT_U128( 0xffffffffffffffff, 0xffffccd92d1c19b3 ) },
  { -0x1.67b89e3ecf714p+45, 0x1.61fc84e2c3f90p-8, INIT_U128( 0xffffffffffffffff, 0xffffd308ec382612 ) },
  { -0x1.f1c44005e3888p+45, -0x1.b6d9c2496db38p-8, INIT_U128( 0xffffffffffffffff, 0xffffc1c777ff438f ) },
  { -0x1.1754aa642ea96p+45, 0x1.9856652930acdp-8, INIT_U128( 0xffffffffffffffff, 0xffffdd156ab37a2b ) },
  { -0x1.e5c94399cb929p+46, -0x1.1ddea47a3bbd4p-7, INIT_U128( 0xffffffffffffffff, 0xffff868daf198d1c ) },
  { -0x1.e0aeb017c15d6p+46, 0x1.41ae9374835d2p-7, INIT_U128( 0xffffffffffffffff, 0xffff87d453fa0fa9 ) },
  { -0x1.5c15afe0b82b6p+46, -0x1.7d2a8ad4fa552p-7, INIT_U128( 0xffffffffffffffff, 0xffffa8fa9407d1f6 ) },
  { -0x1.606c69dcc0d8dp+46, 0x1.44e0e9cc89c1dp-7, INIT_U128( 0xffffffffffffffff, 0xffffa7e4e588cfca ) },
  { -0x1.b14059756280bp+46, -0x1.9f0440433e088p-7, INIT_U128( 0xffffffffffffffff, 0xffff93afe9a2a760 ) },
  { -0x1.f7b2148def642p+46, 0x1.942c33d328586p-7, INIT_U128( 0xffffffffffffffff, 0xffff82137adc8427 ) },
  { -0x1.1785687e2f0adp+47, -0x1.8dc9bfbf1b938p-6, INIT_U128( 0xffffffffffffffff, 0xffff743d4bc0e87b ) },
  { -0x1.df9c3cf7bf387p+47, 0x1.9e7948a33cf29p-6, INIT_U128( 0xffffffffffffffff, 0xffff1031e1842064 ) },
  { -0x1.8ba7fa05174ffp+47, -0x1.d0a6b319a14d7p-6, INIT_U128( 0xffffffffffffffff, 0xffff3a2c02fd7459 ) },
  { -0x1.0117a260022f4p+47, 0x1.6d1ba586da374p-6, INIT_U128( 0xffffffffffffffff, 0xffff7f742ecffee9 ) },
  { -0x1.4ad2ff9e95a60p+47, -0x1.1a9c805235390p-6, INIT_U128( 0xffffffffffffffff, 0xffff5a968030b52d ) },
  { -0x1.f22816ade4503p+47, 0x1.e40996bfc8133p-6, INIT_U128( 0xffffffffffffffff, 0xffff06ebf4a90dd8 ) },
  { -0x1.0faa72961f54ep+48, -0x1.59e0b4ecb3c16p-5, INIT_U128( 0xffffffffffffffff, 0xfffef0558d69e0ac ) },
  { -0x1.46061ba28c0c4p+48, 0x1.016eb29a02dd6p-5, INIT_U128( 0xffffffffffffffff, 0xfffeb9f9e45d73f4 ) },
  { -0x1.10ba77342174fp+48, -0x1.5d5ac606bab59p-5, INIT_U128( 0xffffffffffffffff, 0xfffeef4588cbde8c ) },
  { -0x1.99c517c5338a3p+48, 0x1.f683bacfed078p-5, INIT_U128( 0xffffffffffffffff, 0xfffe663ae83acc76 ) },
  { -0x1.7b396a82f672ep+48, -0x1.7a73c7dcf4e79p-5, INIT_U128( 0xffffffffffffffff, 0xfffe84c6957d098e ) },
  { -0x1.38324e427064ap+48, 0x1.d74f1a63ae9e3p-5, INIT_U128( 0xffffffffffffffff, 0xfffec7cdb1bd8f9c ) },
  { -0x1.f79e52b3ef3cbp+49, -0x1.989eb821313d7p-4, INIT_U128( 0xffffffffffffffff, 0xfffc10c35a982187 ) },
  { -0x1.1c30043038600p+49, 0x1.a8deac2951bd6p-4, INIT_U128( 0xffffffffffffffff, 0xfffdc79ff79f8f41 ) },
  { -0x1.37a2f8646f45fp+49, -0x1.13e218be27c43p-4, INIT_U128( 0xffffffffffffffff, 0xfffd90ba0f372175 ) },
  { -0x1.3d0adffc7a15cp+49, 0x1.4f2860e49e50cp-4, INIT_U128( 0xffffffffffffffff, 0xfffd85ea40070bd5 ) },
  { -0x1.a1abd87f4357bp+49, -0x1.1ca71cf2394e4p-4, INIT_U128( 0xffffffffffffffff, 0xfffcbca84f017951 ) },
  { -0x1.eda1a987db435p+49, 0x1.2d53d2825aa7ap-4, INIT_U128( 0xffffffffffffffff, 0xfffc24bcacf0497a ) },
  { -0x1.ed4dd00dda9bap+50, -0x1.eeb94be9dd72ap-3, INIT_U128( 0xffffffffffffffff, 0xfff84ac8bfc89592 ) },
  { -0x1.9a6f780b34defp+50, 0x1.3820cfac7041ap-3, INIT_U128( 0xffffffffffffffff, 0xfff996421fd32c85 ) },
  { -0x1.0955cc3a12abap+50, -0x1.8c978eb9192f2p-3, INIT_U128( 0xffffffffffffffff, 0xfffbdaa8cf17b552 ) },
  { -0x1.1c2a666c3854dp+50, 0x1.19003ed232008p-3, INIT_U128( 0xffffffffffffffff, 0xfffb8f56664f1ead ) },
  { -0x1.e67843a7ccf09p+50, -0x1.6230820ec4610p-3, INIT_U128( 0xffffffffffffffff, 0xfff8661ef160cc3e ) },
  { -0x1.1a6ef16e34ddep+50, 0x1.f732baddee658p-3, INIT_U128( 0xffffffffffffffff, 0xfffb96443a472c89 ) },
  { -0x1.bf8b48177f169p+51, -0x1.b434093368681p-2, INIT_U128( 0xffffffffffffffff, 0xfff203a5bf44074c ) },
  { -0x1.edb675abdb6cep+51, 0x1.3ac4b29a75896p-2, INIT_U128( 0xffffffffffffffff, 0xfff0924c52a1249a ) },
  { -0x1.2ea7d1105d4fap+51, -0x1.25e8c3ee4bd18p-2, INIT_U128( 0xffffffffffffffff, 0xfff68ac1777d1583 ) },
  { -0x1.1213ccfe2427ap+51, 0x1.264623c24c8c4p-2, INIT_U128( 0xffffffffffffffff, 0xfff76f61980edec4 ) },
  { -0x1.be877c117d0f0p+51, -0x1.2e43d12e5c87ap-2, INIT_U128( 0xffffffffffffffff, 0xfff20bc41f741788 ) },
  { -0x1.cf24cfed9e49ap+51, 0x1.f62a7475ec54ep-2, INIT_U128( 0xffffffffffffffff, 0xfff186d980930db4 ) },
  { -0x1.4225e9ee844bdp+52, -0x1.0c2fafe8185f6p-1, INIT_U128( 0xffffffffffffffff, 0xffebdda16117bb43 ) },
  { -0x1.4552d55a8aa5ap+52, 0x1.e3d1c26dc7a39p-1, INIT_U128( 0xffffffffffffffff, 0xffebaad2aa5755a7 ) },
  { -0x1.28d6ccdc51adap+52, -0x1.32f47ee265e90p-1, INIT_U128( 0xffffffffffffffff, 0xffed7293323ae526 ) },
  { -0x1.112594f0224b2p+52, 0x1.16249cea2c494p-1, INIT_U128( 0xffffffffffffffff, 0xffeeeda6b0fddb4f ) },
  { -0x1.21adcaca435bap+52, -0x1.ab950cfd572a2p-1, INIT_U128( 0xffffffffffffffff, 0xffede523535bca46 ) },
  { -0x1.4e456c969c8aep+52, 0x1.2b03970656073p-1, INIT_U128( 0xffffffffffffffff, 0xffeb1ba936963753 ) },
  { -0x1.382c9bb670594p+53, -0x1.6a6fd304d4dfap+0, INIT_U128( 0xffffffffffffffff, 0xffd8fa6c8931f4d7 ) },
  { -0x1.4403d5da8807ap+53, 0x1.1ea4df5e3d49cp+0, INIT_U128( 0xffffffffffffffff, 0xffd77f8544aeff0e ) },
  { -0x1.cca055159940ap+53, -0x1.5e72c748bce59p+0, INIT_U128( 0xffffffffffffffff, 0xffc66bf55d4cd7eb ) },
  { -0x1.15d79a3a2baf4p+53, 0x1.8c2b9d2d18574p+0, INIT_U128( 0xffffffffffffffff, 0xffdd450cb8ba8a1a ) },
  { -0x1.12a6d07c254dap+53, -0x1.038e0fe6071c2p+0, INIT_U128( 0xffffffffffffffff, 0xffddab25f07b564b ) },
  { -0x1.c6f04fc78de0ap+53, 0x1.414ea18e829d4p+0, INIT_U128( 0xffffffffffffffff, 0xffc721f6070e43ee ) },
  { -0x1.2976d3d652edap+54, -0x1.d9041cc7b2083p+1, INIT_U128( 0xffffffffffffffff, 0xffb5a24b0a6b4495 ) },
  { -0x1.6794c552cf298p+54, 0x1.e7fe4fe9cffcap+1, INIT_U128( 0xffffffffffffffff, 0xffa61aceab4c35a4 ) },
  { -0x1.0747edf60e8fep+54, -0x1.36b9953e6d732p+1, INIT_U128( 0xffffffffffffffff, 0xffbe2e04827c5c06 ) },
  { -0x1.de28c45fbc518p+54, 0x1.496dd7e492dbbp+1, INIT_U128( 0xffffffffffffffff, 0xff8875cee810eba3 ) },
  { -0x1.42e9452885d28p+54, -0x1.237b99b846f73p+1, INIT_U128( 0xffffffffffffffff, 0xffaf45aeb5de8b5e ) },
  { -0x1.2d3f9a525a7f4p+54, 0x1.6bc01206d7802p+1, INIT_U128( 0xffffffffffffffff, 0xffb4b0196b696033 ) },
  { -0x1.5670189cace03p+55, -0x1.3e652f447cca6p+2, INIT_U128( 0xffffffffffffffff, 0xff54c7f3b1a98fe4 ) },
  { -0x1.81e4aef303c96p+55, 0x1.781de368f03bcp+2, INIT_U128( 0xffffffffffffffff, 0xff3f0da8867e1b56 ) },
  { -0x1.e9fa3c43d3f47p+55, -0x1.d02a1b8ba0544p+2, INIT_U128( 0xffffffffffffffff, 0xff0b02e1de1605c1 ) },
  { -0x1.a3217a714642fp+55, 0x1.2226d7fa444dbp+2, INIT_U128( 0xffffffffffffffff, 0xff2e6f42c75cde8d ) },
  { -0x1.3a99ee8e7533ep+55, -0x1.e725504bce4aap+2, INIT_U128( 0xffffffffffffffff, 0xff62b308b8c56609 ) },
  { -0x1.40cbb75a81977p+55, 0x1.073a43a40e748p+2, INIT_U128( 0xffffffffffffffff, 0xff5f9a2452bf344d ) },
  { -0x1.459817948b303p+56, -0x1.56a35196ad46ap+3, INIT_U128( 0xffffffffffffffff, 0xfeba67e86b74cfc6 ) },
  { -0x1.9365e0d126cbcp+56, 0x1.583ee812b07ddp+3, INIT_U128( 0xffffffffffffffff, 0xfe6c9a1f2ed9344b ) },
  { -0x1.5a31617eb462cp+56, -0x1.5a563098b4ac6p+3, INIT_U128( 0xffffffffffffffff, 0xfea5ce9e814b9d36 ) },
  { -0x1.998d89a3331b1p+56, 0x1.83a8355707507p+3, INIT_U128( 0xffffffffffffffff, 0xfe6672765ccce4fd ) },
  { -0x1.335a4d6066b4ap+56, -0x1.01224ef40244ap+3, INIT_U128( 0xffffffffffffffff, 0xfecca5b29f994b58 ) },
  { -0x1.6ba6853ed74d0p+56, 0x1.3aa23b5675448p+3, INIT_U128( 0xffffffffffffffff, 0xfe94597ac128b30a ) },
  { -0x1.37dbe4346fb7cp+57, -0x1.49cf499e939e9p+4, INIT_U128( 0xffffffffffffffff, 0xfd9048379720906c ) },
  { -0x1.8410133d08202p+57, 0x1.7ac5295ef58a5p+4, INIT_U128( 0xffffffffffffffff, 0xfcf7dfd985efbfd8 ) },
  { -0x1.6d4ae358da95cp+57, -0x1.b14fad91629f6p+4, INIT_U128( 0xffffffffffffffff, 0xfd256a394e4ad465 ) },
  { -0x1.e6853b67cd0a8p+57, 0x1.c3bafbf587760p+4, INIT_U128( 0xffffffffffffffff, 0xfc32f5893065eb1d ) },
  { -0x1.efb4dcd7df69bp+57, -0x1.6869ea5ad0d3ep+4, INIT_U128( 0xffffffffffffffff, 0xfc20964650412c8a ) },
  { -0x1.3bd8df5e77b1cp+57, 0x1.d9af191bb35e3p+4, INIT_U128( 0xffffffffffffffff, 0xfd884e4143109c9e ) },
  { -0x1.389405e471280p+58, -0x1.defa59cfbdf4bp+5, INIT_U128( 0xffffffffffffffff, 0xfb1dafe86e3b5fc5 ) },
  { -0x1.fbd5f117f7abep+58, 0x1.732df474e65bep+5, INIT_U128( 0xffffffffffffffff, 0xf810a83ba02150af ) },
  { -0x1.c61faa1b8c3f5p+58, -0x1.3ca1a87279435p+5, INIT_U128( 0xffffffffffffffff, 0xf8e7815791cf0299 ) },
  { -0x1.7a68381af4d07p+58, 0x1.0bb7c1b0176f8p+5, INIT_U128( 0xffffffffffffffff, 0xfa165f1f942cbe62 ) },
  { -0x1.9344fa092689fp+58, -0x1.0dce18ec1b9c3p+5, INIT_U128( 0xffffffffffffffff, 0xf9b2ec17db65d81f ) },
  { -0x1.63f6a9c4c7ed5p+58, 0x1.ed765b89daeccp+5, INIT_U128( 0xffffffffffffffff, 0xfa702558ece04afe ) },
  { -0x1.68244986d0489p+59, -0x1.fd843369fb087p+6, INIT_U128( 0xffffffffffffffff, 0xf4beddb3c97dbb01 ) },
  { -0x1.711e90dee23d2p+59, 0x1.9b7e96e736fd3p+6, INIT_U128( 0xffffffffffffffff, 0xf4770b7908ee1767 ) },
  { -0x1.fea6825ffd4d1p+59, -0x1.dcc5161fb98a3p+6, INIT_U128( 0xffffffffffffffff, 0xf00acbed00159709 ) },
  { -0x1.d620d23fac41ap+59, 0x1.368266946d04dp+6, INIT_U128( 0xffffffffffffffff, 0xf14ef96e029df34e ) },
  { -0x1.26945fbe4d28cp+59, -0x1.5a30e57cb461cp+6, INIT_U128( 0xffffffffffffffff, 0xf6cb5d020d96b9aa ) },
  { -0x1.b1478ff5628f2p+59, 0x1.93df431727be8p+6, INIT_U128( 0xffffffffffffffff, 0xf275c38054eb8765 ) },
  { -0x1.0b80930e17012p+60, -0x1.97cccb5d2f99ap+7, INIT_U128( 0xffffffffffffffff, 0xef47f6cf1e8fed35 ) },
  { -0x1.579a3f0caf348p+60, 0x1.fb356fe7f66aep+7, INIT_U128( 0xffffffffffffffff, 0xea865c0f350cb8fe ) },
  { -0x1.0036b5b6006d6p+60, -0x1.93731e6d26e64p+7, INIT_U128( 0xffffffffffffffff, 0xeffc94a49ff92937 ) },
  { -0x1.d4e48903a9c91p+60, 0x1.38cabb9c71958p+7, INIT_U128( 0xffffffffffffffff, 0xe2b1b76fc5636f9d ) },
  { -0x1.726d783ce4dafp+60, -0x1.d0ba7a1ba174fp+7, INIT_U128( 0xffffffffffffffff, 0xe8d9287c31b25018 ) },
  { -0x1.abf8b01357f16p+60, 0x1.388d2258711a4p+7, INIT_U128( 0xffffffffffffffff, 0xe54074feca80ea9d ) },
  { -0x1.0d5c228e1ab84p+61, -0x1.2660e82e4cc1dp+8, INIT_U128( 0xffffffffffffffff, 0xde547bae3ca8f6da ) },
  { -0x1.4444e40a8889cp+61, 0x1.a0a11c9941424p+8, INIT_U128( 0xffffffffffffffff, 0xd777637eaeeec9a1 ) },
  { -0x1.342f011a685e0p+61, -0x1.a83b8dd150772p+8, INIT_U128( 0xffffffffffffffff, 0xd97a1fdcb2f43e58 ) },
  { -0x1.8e1778211c2efp+61, 0x1.9c534aa338a69p+8, INIT_U128( 0xffffffffffffffff, 0xce3d10fbdc7a239d ) },
  { -0x1.a5bfc1394b7f8p+61, -0x1.ff360901fe6c1p+8, INIT_U128( 0xffffffffffffffff, 0xcb4807d8d6900e01 ) },
  { -0x1.37d1e18a6fa3cp+61, 0x1.a466439948cc8p+8, INIT_U128( 0xffffffffffffffff, 0xd905c3ceb20b89a5 ) },
  { -0x1.dbfb1f21b7f64p+62, -0x1.0df2b3ca1be56p+9, INIT_U128( 0xffffffffffffffff, 0x8901383792026de5 ) },
  { -0x1.dae8aa2db5d15p+62, 0x1.028f7ee0051f0p+9, INIT_U128( 0xffffffffffffffff, 0x8945d574928bae06 ) },
  { -0x1.68239948d0473p+62, -0x1.8c303a2318607p+9, INIT_U128( 0xffffffffffffffff, 0xa5f719adcbee30e8 ) },
  { -0x1.966ec1f92cdd8p+62, 0x1.73a42c58e7486p+9, INIT_U128( 0xffffffffffffffff, 0x9a644f81b4c8a2e8 ) },
  { -0x1.23fd057e47fa0p+62, -0x1.ee090107dc120p+9, INIT_U128( 0xffffffffffffffff, 0xb700bea06e017c24 ) },
  { -0x1.6ed99c40ddb34p+62, 0x1.2433bb3648678p+9, INIT_U128( 0xffffffffffffffff, 0xa44998efc8933249 ) },
  { -0x1.817c1f7302f84p+63, -0x1.43be2544877c4p+10, INIT_U128( 0xffffffffffffffff, 0x3f41f0467e83daf2 ) },
  { -0x1.5cfe76d4b9fcfp+63, 0x1.f3e997b1e7d33p+10, INIT_U128( 0xffffffffffffffff, 0x5180c495a3018fd0 ) },
  { -0x1.14789f6028f14p+63, -0x1.08f726e811ee5p+10, INIT_U128( 0xffffffffffffffff, 0x75c3b04feb875bdd ) },
  { -0x1.2c76ba0858ed8p+63, 0x1.21d4c1ee43a98p+10, INIT_U128( 0xffffffffffffffff, 0x69c4a2fbd3894488 ) },
  { -0x1.2f580f185eb02p+63, -0x1.70bb9e14e1774p+10, INIT_U128( 0xffffffffffffffff, 0x6853f873d0a7ea3e ) },
  { -0x1.c275de8984ebcp+63, 0x1.63871034c70e2p+10, INIT_U128( 0xffffffffffffffff, 0x1ec510bb3d8a258f ) },
  { -0x1.18f72be431ee6p+64, -0x1.508d56aaa11abp+11, INIT_U128( 0xfffffffffffffffe, 0xe708d41bce11957c ) },
  { -0x1.ffda5ad7ffb4cp+64, 0x1.beb6ed1f7d6dep+11, INIT_U128( 0xfffffffffffffffe, 0x25a528004b4df6 ) },
  { -0x1.4fc35fac9f86cp+64, -0x1.37ac3c626f588p+11, INIT_U128( 0xfffffffffffffffe, 0xb03ca05360793643 ) },
  { -0x1.631d30b8c63a6p+64, 0x1.2bbe7ece577d0p+11, INIT_U128( 0xfffffffffffffffe, 0x9ce2cf4739c5a95e ) },
  { -0x1.459ef02c8b3dep+64, -0x1.3e05dae67c0bcp+11, INIT_U128( 0xfffffffffffffffe, 0xba610fd374c21610 ) },
  { -0x1.2e4dfdae5c9c0p+64, 0x1.adc289775b851p+11, INIT_U128( 0xfffffffffffffffe, 0xd1b20251a3640d6f ) },
  { -0x1.ca361509946c2p+65, -0x1.c4bcbfa989798p+12, INIT_U128( 0xfffffffffffffffc, 0x6b93d5ecd727a3b5 ) },
  { -0x1.2166b5ce42cd6p+65, 0x1.ee331a89dc664p+12, INIT_U128( 0xfffffffffffffffd, 0xbd3294637a655ee4 ) },
  { -0x1.b2c5c0af658b8p+65, -0x1.904052112080ap+12, INIT_U128( 0xfffffffffffffffc, 0x9a747ea134e8e6fc ) },
  { -0x1.1266d5c024cdap+65, 0x1.3441e0226883cp+12, INIT_U128( 0xfffffffffffffffd, 0xdb32547fb664d345 ) },
  { -0x1.d3fe7755a7fcfp+65, -0x1.6f654c96decaap+12, INIT_U128( 0xfffffffffffffffc, 0x58031154b006090a ) },
  { -0x1.282b77d45056fp+65, 0x1.e87716b7d0ee3p+12, INIT_U128( 0xfffffffffffffffd, 0xafa910575f523e88 ) },
  { -0x1.785c7380f0b8ep+66, -0x1.54a231dea9446p+13, INIT_U128( 0xfffffffffffffffa, 0x1e8e31fc3d1c556c ) },
  { -0x1.0a4bd8a61497bp+66, 0x1.a7b4829b4f690p+13, INIT_U128( 0xfffffffffffffffb, 0xd6d09d67ada174f7 ) },
  { -0x1.c55dcff38abbap+66, -0x1.ccc1b72f99837p+13, INIT_U128( 0xfffffffffffffff8, 0xea88c031d5114668 ) },
  { -0x1.0b3b92ce16772p+66, 0x1.de54fbd9bcaa0p+13, INIT_U128( 0xfffffffffffffffb, 0xd311b4c7a623bbcb ) },
  { -0x1.1d4f0dda3a9e2p+66, -0x1.0b15eda6162bep+13, INIT_U128( 0xfffffffffffffffb, 0x8ac3c89715875e9e ) },
  { -0x1.9d0be8f93a17dp+66, 0x1.6d126d86da24ep+13, INIT_U128( 0xfffffffffffffff9, 0x8bd05c1b17a0eda3 ) },
  { -0x1.42c22a0a85846p+67, -0x1.002919b200523p+14, INIT_U128( 0xfffffffffffffff5, 0xe9eeafabd3dcbff6 ) },
  { -0x1.98d0ec5731a1ep+67, 0x1.88611a3910c23p+14, INIT_U128( 0xfffffffffffffff3, 0x39789d4672f16219 ) },
  { -0x1.3e2ef4987c5dep+67, -0x1.c71eea078e3ddp+14, INIT_U128( 0xfffffffffffffff6, 0xe885b3c1d108e39 ) },
  { -0x1.3623a3ae6c474p+67, 0x1.086509a210ca1p+14, INIT_U128( 0xfffffffffffffff6, 0x4ee2e28c9dc6421a ) },
  { -0x1.8bdba9af17b75p+67, -0x1.578a597eaf14bp+14, INIT_U128( 0xfffffffffffffff3, 0xa122b28742452a1e ) },
  { -0x1.f698242ded304p+67, 0x1.3ea16a5e7d42ep+14, INIT_U128( 0xfffffffffffffff0, 0x4b3ede90967e4fa9 ) },
  { -0x1.e7caab91cf956p+68, -0x1.08fef9ee11fdfp+15, INIT_U128( 0xffffffffffffffe1, 0x835546e306a97b81 ) },
  { -0x1.72e34a82e5c6ap+68, 0x1.bbc922e777924p+15, INIT_U128( 0xffffffffffffffe8, 0xd1cb57d1a396dde5 ) },
  { -0x1.78644686f0c89p+68, -0x1.9b0ec259361d8p+15, INIT_U128( 0xffffffffffffffe8, 0x79bb9790f3763279 ) },
  { -0x1.a440399d48807p+68, 0x1.f0c0ba09e1817p+15, INIT_U128( 0xffffffffffffffe5, 0xbbfc662b77f9f861 ) },
  { -0x1.efb0f437df61ep+68, -0x1.ae8c12455d182p+15, INIT_U128( 0xffffffffffffffe1, 0x4f0bc8209e128ba ) },
  { -0x1.98753ca530ea8p+68, 0x1.b50ec42b6a1d8p+15, INIT_U128( 0xffffffffffffffe6, 0x78ac35acf158da88 ) },
  { -0x1.7b4dc974f69b9p+69, -0x1.a3b0cf754761ap+16, INIT_U128( 0xffffffffffffffd0, 0x9646d1612c8c5c50 ) },
  { -0x1.9363c00526c78p+69, 0x1.989cd74f3139bp+16, INIT_U128( 0xffffffffffffffcd, 0x9387ff5b2711989d ) },
  { -0x1.aec6bd1d5d8d8p+69, -0x1.306fa8fe60df5p+16, INIT_U128( 0xffffffffffffffca, 0x27285c544e4ecf91 ) },
  { -0x1.05af44d60b5e8p+69, 0x1.f188e727e311dp+16, INIT_U128( 0xffffffffffffffdf, 0x4a17653e9431f189 ) },
  { -0x1.3098c2b261318p+69, -0x1.9e80312f3d006p+16, INIT_U128( 0xffffffffffffffd9, 0xece7a9b3d9ce6180 ) },
  { -0x1.79ea0160f3d40p+69, 0x1.6ec7b74cdd8f7p+16, INIT_U128( 0xffffffffffffffd0, 0xc2bfd3e185816ec8 ) },
  { -0x1.659f1b34cb3e4p+70, -0x1.b00da511601b5p+17, INIT_U128( 0xffffffffffffffa6, 0x983932cd306c9fe5 ) },
  { -0x1.772cc0f0ee598p+70, 0x1.f17ac4e5e2f58p+17, INIT_U128( 0xffffffffffffffa2, 0x34cfc3c469a3e2f6 ) },
  { -0x1.0797fe900f300p+70, -0x1.6dc4e51edb89cp+17, INIT_U128( 0xffffffffffffffbe, 0x1a005bfc33fd2477 ) },
  { -0x1.301c2f4460386p+70, 0x1.ef62aa21dec55p+17, INIT_U128( 0xffffffffffffffb3, 0xf8f42ee7f1ebdec6 ) },
  { -0x1.76c3df0eed87cp+70, -0x1.a2ff79c945fefp+17, INIT_U128( 0xffffffffffffffa2, 0x4f083c449e0cba02 ) },
  { -0x1.ab2bcd095657ap+70, 0x1.ef282597de504p+17, INIT_U128( 0xffffffffffffff95, 0x350cbdaa6a1bde51 ) },
  { -0x1.3e0e3c227c1c8p+71, -0x1.4f6c06a69ed81p+18, INIT_U128( 0xffffffffffffff60, 0xf8e1eec1f1bac250 ) },
  { -0x1.dfe8dfa1bfd1cp+71, 0x1.2dbf305a5b7e6p+18, INIT_U128( 0xffffffffffffff10, 0xb902f201724b6fd ) },
  { -0x1.fdf818ebfbf03p+71, -0x1.785e1c44f0bc4p+18, INIT_U128( 0xffffffffffffff01, 0x3f38a0207e21e88 ) },
  { -0x1.e2cdf487c59bep+71, 0x1.7945e260f28bcp+18, INIT_U128( 0xffffffffffffff0e, 0x9905bc1d3215e518 ) },
  { -0x1.1c0eebba381dep+71, -0x1.7631b48cec636p+18, INIT_U128( 0xffffffffffffff71, 0xf88a22e3f10a273a ) },
  { -0x1.11e2b9b023c57p+71, 0x1.cad5a8d795ab5p+18, INIT_U128( 0xffffffffffffff77, 0xea327ee1d4f2b57 ) },
  { -0x1.a6a15d8d4d42cp+72, -0x1.1791fe882f240p+19, INIT_U128( 0xfffffffffffffe59, 0x5ea272b2bd374371 ) },
  { -0x1.bc5e8fff78bd2p+72, 0x1.d73a203fae744p+19, INIT_U128( 0xfffffffffffffe43, 0xa170008742eeb9d2 ) },
  { -0x1.b6122fb76c246p+72, -0x1.364891606c912p+19, INIT_U128( 0xfffffffffffffe49, 0xedd04893db964dbc ) },
  { -0x1.4601d78c8c03bp+72, 0x1.1bb14aec3762ap+19, INIT_U128( 0xfffffffffffffeb9, 0xfe287373fc58dd8b ) },
  { -0x1.e6563671ccac7p+72, -0x1.bd2e987d7a5d3p+19, INIT_U128( 0xfffffffffffffe19, 0xa9c98e335382168c ) },
  { -0x1.0d2c0fd21a582p+72, 0x1.edee3503dbdc6p+19, INIT_U128( 0xfffffffffffffef2, 0xd3f02de5a7ef6f72 ) },
  { -0x1.2af37d9c55e70p+73, -0x1.3d483f187a908p+20, INIT_U128( 0xfffffffffffffdaa, 0x1904c75431ec2b7d ) },
  { -0x1.dd50df25baa1cp+73, 0x1.a1b75cab436ecp+20, INIT_U128( 0xfffffffffffffc45, 0x5e41b48abc9a1b76 ) },
  { -0x1.e047b8e3c08f7p+73, -0x1.887cac2d10f96p+20, INIT_U128( 0xfffffffffffffc3f, 0x708e387ee1077836 ) },
  { -0x1.90df824321bf0p+73, 0x1.1e55f5063cabep+20, INIT_U128( 0xfffffffffffffcde, 0x40fb79bc8211e560 ) },
  { -0x1.d7e8b2d5afd17p+73, -0x1.70c861e2e190cp+20, INIT_U128( 0xfffffffffffffc50, 0x2e9a54a05d08f37a ) },
  { -0x1.4c152560982a4p+73, 0x1.d5e0cbc1abc1ap+20, INIT_U128( 0xfffffffffffffd67, 0xd5b53ecfab9d5e0d ) },
  { -0x1.e9267e29d24d0p+74, -0x1.2ee319e85dc63p+21, INIT_U128( 0xfffffffffffff85b, 0x660758b6cbda239d ) },
  { -0x1.4f26ae209e4d6p+74, 0x1.053569e40a6adp+21, INIT_U128( 0xfffffffffffffac3, 0x65477d86caa0a6ae ) },
  { -0x1.a514ce754a29ap+74, -0x1.610967d2c212dp+21, INIT_U128( 0xfffffffffffff96b, 0xacc62ad75953ded4 ) },
  { -0x1.d00b12bda0163p+74, 0x1.4bb62c02976c6p+21, INIT_U128( 0xfffffffffffff8bf, 0xd3b5097fa76976c6 ) },
  { -0x1.789b665af136dp+74, -0x1.8c69a2bd18d34p+21, INIT_U128( 0xfffffffffffffa1d, 0x9266943b248e72cc ) },
  { -0x1.22f04c4245e0ap+74, 0x1.2ead96125d5b3p+21, INIT_U128( 0xfffffffffffffb74, 0x3ecef6e87da5d5b3 ) },
  { -0x1.c2b2341d85646p+75, -0x1.68f223ced1e44p+22, INIT_U128( 0xfffffffffffff1ea, 0x6e5f13d4dca5c378 ) },
  { -0x1.4a5ee10094bdcp+75, 0x1.a9086a8f5210dp+22, INIT_U128( 0xfffffffffffff5ad, 0x8f7fb5a126a421b ) },
  { -0x1.2620bc4c4c418p+75, -0x1.578af0d4af15ep+22, INIT_U128( 0xfffffffffffff6ce, 0xfa1d9d9df3aa1d44 ) },
  { -0x1.96f69f932ded4p+75, 0x1.511439eaa2287p+22, INIT_U128( 0xfffffffffffff348, 0x4b0366909654450f ) },
  { -0x1.94d32fef29a66p+75, -0x1.74477eb6e88f0p+22, INIT_U128( 0xfffffffffffff359, 0x668086b2cca2ee21 ) },
  { -0x1.f2cb9697e5973p+75, 0x1.751ae8d2ea35dp+22, INIT_U128( 0xfffffffffffff069, 0xa34b40d346dd46bb ) },
  { -0x1.fcc42ae5f9886p+76, -0x1.83ab18d907563p+23, INIT_U128( 0xffffffffffffe033, 0xbd51a067793e2a74 ) },
  { -0x1.9370dddb26e1cp+76, 0x1.d4acddbba959cp+23, INIT_U128( 0xffffffffffffe6c8, 0xf2224d91e4ea566f ) },
  { -0x1.f4737abde8e70p+76, -0x1.df86033dbf0c1p+23, INIT_U128( 0xffffffffffffe0b8, 0xc85421718f103cff ) },
  { -0x1.aa8da97b551b5p+76, 0x1.dd99f8e7bb33fp+23, INIT_U128( 0xffffffffffffe557, 0x25684aae4beeccfd ) },
  { -0x1.0e691d241cd24p+76, -0x1.f9b4579df368bp+23, INIT_U128( 0xffffffffffffef19, 0x6e2dbe32db0325d5 ) },
  { -0x1.e2cf62cdc59edp+76, 0x1.b37deb9b66fbep+23, INIT_U128( 0xffffffffffffe1d3, 0x9d323a613d9bef6 ) },
  { -0x1.5abdbf70b57b8p+77, -0x1.b2da758965b4fp+24, INIT_U128( 0xffffffffffffd4a8, 0x4811e9508e4d258b ) },
  { -0x1.ff34179ffe683p+77, 0x1.d35925cda6b25p+24, INIT_U128( 0xffffffffffffc019, 0x7d0c0032fbd35926 ) },
  { -0x1.3f4418467e883p+77, -0x1.222463e04448cp+24, INIT_U128( 0xffffffffffffd817, 0x7cf7302ef8dddb9d ) },
  { -0x1.0206bbe0040d8p+77, 0x1.01f64c6403ecap+24, INIT_U128( 0xffffffffffffdfbf, 0x2883ff7e5101f64d ) },
  { -0x1.8b1687b1162d1p+77, -0x1.4c659ce298cb4p+24, INIT_U128( 0xffffffffffffce9d, 0x2f09dd3a5cb39a64 ) },
  { -0x1.aafeb6f555fd7p+77, 0x1.b13743ff626e8p+24, INIT_U128( 0xffffffffffffcaa0, 0x2921554053b13744 ) },
  { -0x1.a3f08db947e12p+78, -0x1.41f5568683eabp+25, INIT_U128( 0xffffffffffff9703, 0xdc91ae07b57c1553 ) },
  { -0x1.024b7dea04970p+78, 0x1.38c3bda871878p+25, INIT_U128( 0xffffffffffffbf6d, 0x20857eda4271877c ) },
  { -0x1.b63cd53f6c79bp+78, -0x1.f5171481ea2e2p+25, INIT_U128( 0xffffffffffff9270, 0xcab024e19015d1d7 ) },
  { -0x1.addea67d5bbd5p+78, 0x1.b301002566020p+25, INIT_U128( 0xffffffffffff9488, 0x5660a910af660201 ) },
  { -0x1.78c22d5af1846p+78, -0x1.414f8456829f0p+25, INIT_U128( 0xffffffffffffa1cf, 0x74a9439ee57d60f8 ) },
  { -0x1.87addde10f5bcp+78, 0x1.6d35a818da6b5p+25, INIT_U128( 0xffffffffffff9e14, 0x8887bc2912da6b51 ) },
  { -0x1.20d5879441ab1p+79, -0x1.323b875c64771p+26, INIT_U128( 0xffffffffffff6f95, 0x3c35df2a733711e3 ) },
  { -0x1.4acfa100959f4p+79, 0x1.2f3ea4ba5e7d4p+26, INIT_U128( 0xffffffffffff5a98, 0x2f7fb53064bcfa93 ) },
  { -0x1.2befdefa57dfcp+79, -0x1.2d9f03ce5b3e0p+26, INIT_U128( 0xffffffffffff6a08, 0x1082d4101b4983f1 ) },
  { -0x1.e2bc9057c5792p+79, 0x1.86ad9e6b0d5b4p+26, INIT_U128( 0xffffffffffff0ea1, 0xb7d41d43761ab67a ) },
  { -0x1.e0797959c0f2fp+79, -0x1.e1f0a7f5c3e15p+26, INIT_U128( 0xffffffffffff0fc3, 0x43531f8680783d61 ) },
  { -0x1.5a06c438b40d8p+79, 0x1.1fbd919e3f7b2p+26, INIT_U128( 0xffffffffffff52fc, 0x9de3a5f9447ef647 ) },
  { -0x1.04150bd6082a2p+80, -0x1.5c0d8018b81b0p+27, INIT_U128( 0xfffffffffffefbea, 0xf429f7d5d51f9400 ) },
  { -0x1.21b3de044367cp+80, 0x1.8f2b93891e572p+27, INIT_U128( 0xfffffffffffede4c, 0x21fbbc984c795c9d ) },
  { -0x1.a8731cfd50e64p+80, -0x1.ccb294f999652p+27, INIT_U128( 0xfffffffffffe578c, 0xe302af19b19a6b59 ) },
  { -0x1.4062d78880c5bp+80, 0x1.4c63aab498c76p+27, INIT_U128( 0xfffffffffffebf9d, 0x28777f3a5a631d56 ) },
  { -0x1.0ea7eb801d4fep+80, -0x1.42ae5574855cap+27, INIT_U128( 0xfffffffffffef158, 0x147fe2b015ea8d55 ) },
  { -0x1.8527685f0a4edp+80, 0x1.c4e4c0c189c98p+27, INIT_U128( 0xfffffffffffe7ad8, 0x97a0f5b13e272607 ) },
  { -0x1.37c569f66f8adp+81, -0x1.33060eda660c2p+28, INIT_U128( 0xfffffffffffd9075, 0x2c1320ea4ccf9f13 ) },
  { -0x1.77e5c0e6efcb8p+81, 0x1.b0879753610f3p+28, INIT_U128( 0xfffffffffffd1034, 0x7e3220691b087976 ) },
  { -0x1.74559332e8ab2p+81, -0x1.ff939ec9ff274p+28, INIT_U128( 0xfffffffffffd1754, 0xd99a2ea9a006c614 ) },
  { -0x1.1dbd3f6c3b7a8p+81, 0x1.952b1ac92a563p+28, INIT_U128( 0xfffffffffffdc485, 0x8127890b1952b1ad ) },
  { -0x1.31cf2624639e5p+81, -0x1.688ce06ed119cp+28, INIT_U128( 0xfffffffffffd9c61, 0xb3b738c3497731fa ) },
  { -0x1.c469bf8388d38p+81, 0x1.5dbf0520bb7e0p+28, INIT_U128( 0xfffffffffffc772c, 0x80f8ee5915dbf053 ) },
  { -0x1.962be43d2c57cp+82, -0x1.a7264fdb4e4cap+29, INIT_U128( 0xfffffffffff9a750, 0x6f0b4ea0cb1b3605 ) },
  { -0x1.e46ca38dc8d95p+82, 0x1.76a55d96ed4acp+29, INIT_U128( 0xfffffffffff86e4d, 0x71c8dc9aeed4abb3 ) },
  { -0x1.8916f7d1122dfp+82, -0x1.3e7367c47ce6dp+29, INIT_U128( 0xfffffffffff9dba4, 0x20bbb74818319308 ) },
  { -0x1.9662ef372cc5ep+82, 0x1.8d2c03a91a580p+29, INIT_U128( 0xfffffffffff9a674, 0x43234ce8b1a58076 ) },
  { -0x1.9a80e7df3501dp+82, -0x1.b816cd3b702dap+29, INIT_U128( 0xfffffffffff995fc, 0x60832bf888fd2659 ) },
  { -0x1.1c57973238af3p+82, 0x1.42f404dc85e80p+29, INIT_U128( 0xfffffffffffb8ea1, 0xa3371d43685e809c ) },
  { -0x1.84c962350992cp+83, -0x1.5cba6ebcb974ep+30, INIT_U128( 0xfffffffffff3d9b4, 0xee57b369a8d16451 ) },
  { -0x1.8f57af931eaf6p+83, 0x1.93714db726e2ap+30, INIT_U128( 0xfffffffffff38542, 0x83670a8564dc536e ) },
  { -0x1.9f68c4313ed18p+83, -0x1.147f17f028fe3p+30, INIT_U128( 0xfffffffffff304b9, 0xde760973bae03a04 ) },
  { -0x1.58560e78b0ac2p+83, 0x1.d72e2e29ae5c6p+30, INIT_U128( 0xfffffffffff53d4f, 0x8c3a7a9f75cb8b8b ) },
  { -0x1.478526cc8f0a5p+83, -0x1.d03bcf61a077ap+30, INIT_U128( 0xfffffffffff5c3d6, 0xc99b87ad0bf10c28 ) },
  { -0x1.87b5c57b0f6b9p+83, 0x1.bbdaba0177b57p+30, INIT_U128( 0xfffffffffff3c251, 0xd42784a3eef6ae81 ) },
  { -0x1.cfcf6e2b9f9eep+84, -0x1.a57304e94ae60p+31, INIT_U128( 0xffffffffffe30309, 0x1d4606112d467d8c ) },
  { -0x1.5105d4caa20bap+84, 0x1.019e9450033d2p+31, INIT_U128( 0xffffffffffeaefa2, 0xb355df4680cf4a29 ) },
  { -0x1.0dffe5561bffcp+84, -0x1.156f944e2adf2p+31, INIT_U128( 0xffffffffffef2001, 0xaa9e4003754835d9 ) },
  { -0x1.5f5d9970bebb3p+84, 0x1.a42bc99548579p+31, INIT_U128( 0xffffffffffea0a26, 0x68f4144dd215e4cb ) },
  { -0x1.bc95e429792bcp+84, -0x1.43ca3cfe87948p+31, INIT_U128( 0xffffffffffe436a1, 0xbd686d435e1ae181 ) },
  { -0x1.1ad1272c35a25p+84, 0x1.45acdbfa8b59cp+31, INIT_U128( 0xffffffffffee52ed, 0x8d3ca5dba2d66dfe ) },
  { -0x1.823da4c5047b4p+85, -0x1.b134166362683p+32, INIT_U128( 0xffffffffffcfb84b, 0x675f70964ecbe99d ) },
  { -0x1.d2d8dfa3a5b1cp+85, 0x1.1cbe8ca4397d2p+32, INIT_U128( 0xffffffffffc5a4e4, 0xb8b49c91cbe8ca5 ) },
  { -0x1.694ae39ed295cp+85, -0x1.ef3076d7de60fp+32, INIT_U128( 0xffffffffffd2d6a3, 0x8c25ad4610cf8929 ) },
  { -0x1.5fb29180bf652p+85, 0x1.ee1445d5dc289p+32, INIT_U128( 0xffffffffffd409ad, 0xcfe8135dee1445d6 ) },
  { -0x1.4beae59297d5cp+85, -0x1.ef0a4115de148p+32, INIT_U128( 0xffffffffffd682a3, 0x4dad054610f5beeb ) },
  { -0x1.8185938d030b2p+85, 0x1.eb53a9c5d6a75p+32, INIT_U128( 0xffffffffffcfcf4d, 0x8e5f9e9deb53a9c6 ) },
  { -0x1.038959060712bp+86, -0x1.d9dba207b3b74p+33, INIT_U128( 0xffffffffffbf1da9, 0xbe7e3b504c48bbf1 ) },
  { -0x1.c281d7ef8503bp+86, 0x1.4b0e7fd4961d0p+33, INIT_U128( 0xffffffffff8f5f8a, 0x41ebf16961cffaa ) },
  { -0x1.aa177169542eep+86, -0x1.4a379d6a946f4p+33, INIT_U128( 0xffffffffff957a23, 0xa5aaf4456b90c52b ) },
  { -0x1.06e939d60dd27p+86, 0x1.79e6df94f3cdcp+33, INIT_U128( 0xffffffffffbe45b1, 0x8a7c8b66f3cdbf2a ) },
  { -0x1.efaaa253df555p+86, -0x1.57ef0ea0afde2p+33, INIT_U128( 0xffffffffff841557, 0x6b082aa95021e2bf ) },
  { -0x1.7b8573e4f70aep+86, 0x1.fdaa87abfb551p+33, INIT_U128( 0xffffffffffa11ea3, 0x6c23d4bfb550f58 ) },
  { -0x1.6d5ea2f0dabd4p+87, -0x1.dae30d57b5c61p+34, INIT_U128( 0xffffffffff4950ae, 0x8792a1589473caa2 ) },
  { -0x1.a1d6514143acap+87, 0x1.29f4871a53e91p+34, INIT_U128( 0xffffffffff2f14d7, 0x5f5e29b4a7d21c6a ) },
  { -0x1.1ead4a563d5aap+87, -0x1.bca7786f794efp+34, INIT_U128( 0xffffffffff70a95a, 0xd4e152a90d621e43 ) },
  { -0x1.6fa16314df42cp+87, 0x1.a0eaef1b41d5ep+34, INIT_U128( 0xffffffffff482f4e, 0x75905ea683abbc6e ) },
  { -0x1.6ddb8ed2dbb72p+87, -0x1.427a42b884f48p+34, INIT_U128( 0xffffffffff491238, 0x9692246af616f51e ) },
  { -0x1.bda97bc17b530p+87, 0x1.d7267273ae4cep+34, INIT_U128( 0xffffffffff212b42, 0x1f4256875c99c9cf ) },
  { -0x1.31aab99063557p+88, -0x1.1d5c0d423ab82p+35, INIT_U128( 0xfffffffffece5546, 0x6f9caa87151f95ef ) },
  { -0x1.c53ab8498a757p+88, 0x1.7d959f24fb2b4p+35, INIT_U128( 0xfffffffffe3ac547, 0xb6758a9becacf928 ) },
  { -0x1.0c664e5618ccap+88, -0x1.46c04bd68d80ap+35, INIT_U128( 0xfffffffffef399b1, 0xa9e73355c9fda14c ) },
  { -0x1.68e948ead1d29p+88, 0x1.0fefa75e1fdf5p+35, INIT_U128( 0xfffffffffe9716b7, 0x152e2d787f7d3af1 ) },
  { -0x1.209bedde4137ep+88, -0x1.2b6b20d856d64p+35, INIT_U128( 0xfffffffffedf6412, 0x21bec816a4a6f93e ) },
  { -0x1.5afe2732b5fc5p+88, 0x1.cab35f499566cp+35, INIT_U128( 0xfffffffffea501d8, 0xcd4a03be559afa4d ) },
  { -0x1.75a2be44eb458p+89, -0x1.61e7610ac3cecp+36, INIT_U128( 0xfffffffffd14ba83, 0x762974e9e189ef54 ) },
  { -0x1.449a3edc89348p+89, 0x1.654d6098ca9acp+36, INIT_U128( 0xfffffffffd76cb82, 0x46ed971654d6098d ) },
  { -0x1.b9d66fad73acep+89, -0x1.c01fb2c3803f7p+36, INIT_U128( 0xfffffffffc8c5320, 0xa518a623fe04d3c8 ) },
  { -0x1.62f7e844c5efdp+89, 0x1.2735129c4e6a2p+36, INIT_U128( 0xfffffffffd3a102f, 0x76742072735129c5 ) },
  { -0x1.9b02c01b36058p+89, -0x1.04ce3986099c7p+36, INIT_U128( 0xfffffffffcc9fa7f, 0xc993f4efb31c67a0 ) },
  { -0x1.512999b8a2533p+89, 0x1.b9b00a7373601p+36, INIT_U128( 0xfffffffffd5daccc, 0x8ebb59bb9b00a738 ) },
  { -0x1.c175b1f582eb6p+90, -0x1.c0cafccf8195fp+37, INIT_U128( 0xfffffffff8fa2938, 0x29f45247e6a06610 ) },
  { -0x1.5a9af98eb535fp+90, 0x1.2f5140c85ea28p+37, INIT_U128( 0xfffffffffa959419, 0xc52b2865ea28190c ) },
  { -0x1.03453f88068a8p+90, -0x1.1844754e3088ep+37, INIT_U128( 0xfffffffffbf2eb01, 0xdfe5d5dcf771563a ) },
  { -0x1.0677b33a0cef6p+90, 0x1.34eff91869dffp+37, INIT_U128( 0xfffffffffbe62133, 0x17cc42a69dff230e ) },
  { -0x1.35a082926b410p+90, -0x1.cbb26b859764ep+37, INIT_U128( 0xfffffffffb297df5, 0xb652fbc689b28f4e ) },
  { -0x1.b285983f650b3p+90, 0x1.4206d0ea840dap+37, INIT_U128( 0xfffffffff935e99f, 0x26bd36840da1d51 ) },
  { -0x1.1272b75a24e57p+91, -0x1.1b6ee12836ddcp+38, INIT_U128( 0xfffffffff76c6a45, 0x2ed8d4392447b5f3 ) },
  { -0x1.51f67848a3ecfp+91, 0x1.330ea53a661d4p+38, INIT_U128( 0xfffffffff5704c3d, 0xbae098ccc3a94e9a ) },
  { -0x1.46da02028db40p+91, -0x1.acf49f7d59e94p+38, INIT_U128( 0xfffffffff5c92fef, 0xeb925f94c2d820aa ) },
  { -0x1.fdc20483fb840p+91, 0x1.bc7cc04978f98p+38, INIT_U128( 0xfffffffff011efdb, 0xe023e06f1f30125f ) },
  { -0x1.0edc00881db80p+91, -0x1.425c436084b88p+38, INIT_U128( 0xfffffffff7891ffb, 0xbf123faf68ef27df ) },
  { -0x1.21ca20e043944p+91, 0x1.c8b270359164ep+38, INIT_U128( 0xfffffffff6f1aef8, 0xfde35e722c9c0d65 ) },
  { -0x1.027bf62404f7fp+92, -0x1.84c2092709841p+39, INIT_U128( 0xffffffffefd8409d, 0xbfb0803d9efb6c7c ) },
  { -0x1.efe782d3dfcf1p+92, 0x1.9d7eb6413afd7p+39, INIT_U128( 0xffffffffe10187d2, 0xc2030fcebf5b209e ) },
  { -0x1.899042ad13208p+92, -0x1.41111c8c82224p+39, INIT_U128( 0xffffffffe766fbd5, 0x2ecdf75f7771b9bf ) },
  { -0x1.572ec0b4ae5d8p+92, 0x1.4b05f4a2960bep+39, INIT_U128( 0xffffffffea8d13f4, 0xb51a28a582fa514c ) },
  { -0x1.01dc195a03b83p+92, -0x1.b1c9ba4363937p+39, INIT_U128( 0xffffffffefe23e6a, 0x5fc47c271b22de4f ) },
  { -0x1.7a57356cf4ae6p+92, 0x1.052158e40a42bp+39, INIT_U128( 0xffffffffe85a8ca9, 0x30b51a8290ac7206 ) },
  { -0x1.ca0e2f93941c6p+93, -0x1.80cee8a9019ddp+40, INIT_U128( 0xffffffffc6be3a0d, 0x8d7c727f311756ff ) },
  { -0x1.3a1151fe7422ap+93, 0x1.ffaeb227ff5d7p+40, INIT_U128( 0xffffffffd8bdd5c0, 0x317badffaeb22800 ) },
  { -0x1.4c81542c9902ap+93, -0x1.ec96ea3dd92ddp+40, INIT_U128( 0xffffffffd66fd57a, 0x6cdfaa136915c227 ) },
  { -0x1.41b58310836b0p+93, 0x1.5ef40b88bde82p+40, INIT_U128( 0xffffffffd7c94f9d, 0xef92a15ef40b88be ) },
  { -0x1.0805cb66100bap+93, -0x1.93abfb5b27580p+40, INIT_U128( 0xffffffffdeff4693, 0x3dfe8a6c5404a4d9 ) },
  { -0x1.9202b0c524056p+93, 0x1.4edfcd6a9dbfap+40, INIT_U128( 0xffffffffcdbfa9e7, 0x5b7f554edfcd6a9e ) },
  { -0x1.40a09ad881414p+94, -0x1.8fc4b5491f897p+41, INIT_U128( 0xffffffffafd7d949, 0xdfaface076956dc1 ) },
  { -0x1.0b5c8db416b92p+94, 0x1.4f7c9b629ef94p+41, INIT_U128( 0xffffffffbd28dc92, 0xfa51ba9ef936c53e ) },
  { -0x1.d220f8aba441fp+94, -0x1.2d26ae965a4d6p+41, INIT_U128( 0xffffffff8b77c1d5, 0x16ef81a5b2a2d34c ) },
  { -0x1.182f0b98305e2p+94, 0x1.d49ceed1a939ep+41, INIT_U128( 0xffffffffb9f43d19, 0xf3e87ba939dda353 ) },
  { -0x1.b1e8b7bd63d17p+94, -0x1.08168e6c102d2p+41, INIT_U128( 0xffffffff9385d210, 0xa70ba1efd2e327e0 ) },
  { -0x1.721ba512e4374p+94, 0x1.3b78fa6a76f20p+41, INIT_U128( 0xffffffffa37916bb, 0x46f23276f1f4d4ee ) },
  { -0x1.4927aaf8924f6p+95, -0x1.c94a65c79294dp+42, INIT_U128( 0xffffffff5b6c2a83, 0xb6d848dad668e1b6 ) },
  { -0x1.867eef0f0cfdep+95, 0x1.4b03c45496078p+42, INIT_U128( 0xffffffff3cc08878, 0x7981152c0f115259 ) },
  { -0x1.7b14200af6284p+95, -0x1.00d6247c01ac4p+42, INIT_U128( 0xffffffff4275effa, 0x84ebdbfca76e0ffa ) },
  { -0x1.50251d3aa04a4p+95, 0x1.45224f9e8a44ap+42, INIT_U128( 0xffffffff57ed7162, 0xafdae514893e7a2a ) },
  { -0x1.81a50d25034a2p+95, -0x1.8a366f39146cep+42, INIT_U128( 0xffffffff3f2d796d, 0x7e5ae9d726431baf ) },
  { -0x1.44d101c689a20p+95, 0x1.ce2661539c4ccp+42, INIT_U128( 0xffffffff5d977f1c, 0xbb2f073899854e72 ) },
  { -0x1.76cd5976ed9abp+96, -0x1.868684d90d0d0p+43, INIT_U128( 0xfffffffe8932a689, 0x126543cbcbd93798 ) },
  { -0x1.2d1a6cdc5a34ep+96, 0x1.7ed324defda64p+43, INIT_U128( 0xfffffffed2e59323, 0xa5cb2bf69926f7ee ) },
  { -0x1.1a32803434650p+96, -0x1.bf1d1da37e3a4p+43, INIT_U128( 0xfffffffee5cd7fcb, 0xcb9af2071712e40f ) },
  { -0x1.cdb2f24b9b65ep+96, 0x1.5ad31b0eb5a64p+43, INIT_U128( 0xfffffffe324d0db4, 0x649a2ad698d875ae ) },
  { -0x1.4ee1d2ea9dc3ap+96, -0x1.d90f8001b21f0p+43, INIT_U128( 0xfffffffeb11e2d15, 0x623c513783fff270 ) },
  { -0x1.aa4fb671549f7p+96, 0x1.2102cb964205ap+43, INIT_U128( 0xfffffffe55b0498e, 0xab609908165cb211 ) },
  { -0x1.06362e960c6c6p+97, -0x1.c27ec20b84fd8p+44, INIT_U128( 0xfffffffdf393a2d3, 0xe72723d813df47b1 ) },
  { -0x1.f59fba67eb3f8p+97, 0x1.b81ee655703ddp+44, INIT_U128( 0xfffffffc14c08b30, 0x29811b81ee655704 ) },
  { -0x1.ca955077952aap+97, -0x1.67bb5f7acf76cp+44, INIT_U128( 0xfffffffc6ad55f10, 0xd5aaa9844a085309 ) },
  { -0x1.f397e0c1e72fcp+97, 0x1.13bcfb70277a0p+44, INIT_U128( 0xfffffffc18d03e7c, 0x31a0913bcfb70278 ) },
  { -0x1.3105c542620b8p+97, -0x1.7a16cfd0f42dap+44, INIT_U128( 0xfffffffd9df4757b, 0x3be8e85e9302f0be ) },
  { -0x1.5ec0bfc6bd818p+97, 0x1.ac19e54f5833dp+44, INIT_U128( 0xfffffffd427e8072, 0x84fd1ac19e54f584 ) },
  { -0x1.7aecf44ef5d9ep+98, -0x1.aee5dd035dcbcp+45, INIT_U128( 0xfffffffa144c2ec4, 0x28984a23445f9447 ) },
  { -0x1.f13c34efe2786p+98, 0x1.fc920175f9240p+45, INIT_U128( 0xfffffff83b0f2c40, 0x761ebf92402ebf25 ) },
  { -0x1.391c4c3c7238ap+98, -0x1.360d36b46c1a7p+45, INIT_U128( 0xfffffffb1b8ecf0e, 0x371d593e5929727d ) },
  { -0x1.cab862c99570dp+98, 0x1.afe9a5b55fd35p+45, INIT_U128( 0xfffffff8d51e74d9, 0xaa3cf5fd34b6abfb ) },
  { -0x1.bc93bb8f79278p+98, -0x1.f7d2cfa7efa5ap+45, INIT_U128( 0xfffffff90db111c2, 0x1b61c105a60b020c ) },
  { -0x1.2ebcca945d79ap+98, 0x1.d6c63bc3ad8c8p+45, INIT_U128( 0xfffffffb450cd5ae, 0x8a19bad8c77875b2 ) },
  { -0x1.b6efed496ddfep+99, -0x1.fc6fbce1f8df7p+46, INIT_U128( 0xfffffff2488095b4, 0x910080e410c781c9 ) },
  { -0x1.319956d26332bp+99, 0x1.05d1c0340ba38p+46, INIT_U128( 0xfffffff67335496c, 0xe66ac174700d02e9 ) },
  { -0x1.d57f3ce1aafe7p+99, -0x1.dbe90d45b7d21p+46, INIT_U128( 0xfffffff1540618f2, 0xa80c0905bcae920c ) },
  { -0x1.26d48bcc4da92p+99, 0x1.dbb56b79b76aep+46, INIT_U128( 0xfffffff6c95ba19d, 0x92b776ed5ade6ddb ) },
  { -0x1.9e67cf4f3ccfap+99, -0x1.1c74463438e89p+46, INIT_U128( 0xfffffff30cc18586, 0x1982b8e2ee72f1c6 ) },
  { -0x1.93d451cf27a8ap+99, 0x1.68fd1fc6d1fa4p+46, INIT_U128( 0xfffffff3615d7186, 0xc2bb5a3f47f1b47f ) },
  { -0x1.842fa5af085f5p+100, -0x1.876c32910ed86p+47, INIT_U128( 0xffffffe7bd05a50f, 0x7a0a3c49e6b77894 ) },
  { -0x1.7cba1b38f9744p+100, 0x1.b5356af36a6adp+47, INIT_U128( 0xffffffe8345e4c70, 0x68bcda9ab579b536 ) },
  { -0x1.0183026203060p+100, -0x1.db7adb01b6f5cp+47, INIT_U128( 0xffffffefe7cfd9df, 0xcf9f1242927f2486 ) },
  { -0x1.614f5040c29eap+100, 0x1.7c6ff6b2f8dffp+47, INIT_U128( 0xffffffe9eb0afbf3, 0xd616be37fb597c70 ) },
  { -0x1.8a60b80514c17p+100, -0x1.528b8530a5170p+47, INIT_U128( 0xffffffe759f47fae, 0xb3e856ba3d67ad75 ) },
  { -0x1.7901f826f203fp+100, 0x1.c37d044186fa0p+47, INIT_U128( 0xffffffe86fe07d90, 0xdfc1e1be8220c37d ) },
  { -0x1.8898bbe711318p+101, -0x1.43ba5af08774cp+48, INIT_U128( 0xffffffceece8831d, 0xd9cebc45a50f788c ) },
  { -0x1.7366a9f6e6cd5p+101, 0x1.452c77748a58fp+48, INIT_U128( 0xffffffd1932ac123, 0x2657452c77748a59 ) },
  { -0x1.3b8d2a96771a6p+101, -0x1.a223748b4446ep+48, INIT_U128( 0xffffffd88e5aad31, 0x1cb25ddc8b74bbba ) },
  { -0x1.1bf2b33837e56p+101, 0x1.f27d17e1e4fa3p+48, INIT_U128( 0xffffffdc81a998f9, 0x355f27d17e1e4fb ) },
  { -0x1.9e93316d3d266p+101, -0x1.4d8c0a0c9b182p+48, INIT_U128( 0xffffffcc2d99d258, 0x5b32b273f5f364e8 ) },
  { -0x1.f84ed817f09dbp+101, 0x1.74126e4ee824ep+48, INIT_U128( 0xffffffc0f624fd01, 0xec4b74126e4ee825 ) },
  { -0x1.2975175052ea3p+102, -0x1.1f2363e43e46cp+49, INIT_U128( 0xffffffb5a2ba2beb, 0x4571c1b938378373 ) },
  { -0x1.4fc7d9c89f8fbp+102, 0x1.c7dce6a18fb9dp+49, INIT_U128( 0xffffffac0e098dd8, 0x1c178fb9cd431f74 ) },
  { -0x1.3b97aa18772f6p+102, -0x1.e39c2b99c7386p+49, INIT_U128( 0xffffffb11a1579e2, 0x342438c7a8cc7190 ) },
  { -0x1.30da066861b41p+102, 0x1.bfc74fc97f8eap+49, INIT_U128( 0xffffffb3c97e65e7, 0x92ff7f8e9f92ff1e ) },
  { -0x1.6b085e2ad610cp+102, -0x1.ad9421a35b284p+49, INIT_U128( 0xffffffa53de8754a, 0x7bcca4d7bcb949b0 ) },
  { -0x1.74fcbabce9f98p+102, 0x1.6c8b608ed916cp+49, INIT_U128( 0xffffffa2c0d150c5, 0x81a2d916c11db22e ) },
  { -0x1.890efca5121e0p+103, -0x1.d64c0affac982p+50, INIT_U128( 0xffffff3b7881ad76, 0xf0f8a6cfd4014da0 ) },
  { -0x1.e8fb5741d1f6bp+103, 0x1.136fb4aa26df6p+50, INIT_U128( 0xffffff0b82545f17, 0x4ac4dbed2a89b7e ) },
  { -0x1.28d7902e51af2p+103, -0x1.61fd4f68c3faap+50, INIT_U128( 0xffffff6b9437e8d7, 0x286a780ac25cf016 ) },
  { -0x1.d6ecb8edadd97p+103, 0x1.b1704c1762e0ap+50, INIT_U128( 0xffffff1489a38929, 0x134ec5c1305d8b83 ) },
  { -0x1.a5b7aa5b4b6f5p+103, -0x1.4a5362e294a6cp+50, INIT_U128( 0xffffff2d242ad25a, 0x4852d6b27475ad65 ) },
  { -0x1.f69cc38ded399p+103, 0x1.e144d4f3c289ap+50, INIT_U128( 0xffffff04b19e3909, 0x633f851353cf0a27 ) },
  { -0x1.ec76d76fd8edbp+104, -0x1.128f76c6251efp+51, INIT_U128( 0xfffffe1389289027, 0x12476b8449ced709 ) },
  { -0x1.d3c1ecd5a783dp+104, 0x1.8b0a08dd16141p+51, INIT_U128( 0xfffffe2c3e132a58, 0x7c3c585046e8b0a1 ) },
  { -0x1.601b80d2c0370p+104, -0x1.cc10c4ed98218p+51, INIT_U128( 0xfffffe9fe47f2d3f, 0xc8f19f79d8933ef4 ) },
  { -0x1.ff1c772bfe38fp+104, 0x1.8990982513213p+51, INIT_U128( 0xfffffe00e388d401, 0xc71c4c84c128990a ) },
  { -0x1.eb3428d1d6685p+104, -0x1.4ee450f49dc8ap+51, INIT_U128( 0xfffffe14cbd72e29, 0x97a588dd785b11bb ) },
  { -0x1.516d7ec6a2db0p+104, 0x1.26cbd2ea4d97ap+51, INIT_U128( 0xfffffeae9281395d, 0x2509365e97526cbd ) },
  { -0x1.32807c8c65010p+105, -0x1.82478531048f1p+52, INIT_U128( 0xfffffd9aff06e735, 0xfde7db87acefb70f ) },
  { -0x1.e69322b5cd265p+105, 0x1.9d88dab93b11bp+52, INIT_U128( 0xfffffc32d9ba9465, 0xb379d88dab93b11b ) },
  { -0x1.2fdba6d45fb75p+105, -0x1.5b8fdd72b71fcp+52, INIT_U128( 0xfffffda048b25740, 0x914a470228d48e04 ) },
  { -0x1.0a143b4614288p+105, 0x1.b8eaa37b71d54p+52, INIT_U128( 0xfffffdebd78973d7, 0xaf1b8eaa37b71d54 ) },
  { -0x1.31dbe87063b7dp+105, -0x1.f30f9289e61f3p+52, INIT_U128( 0xfffffd9c482f1f38, 0x9040cf06d7619e0d ) },
  { -0x1.cc5a582798b4bp+105, 0x1.5ab378a0b566fp+52, INIT_U128( 0xfffffc674b4fb0ce, 0x96b5ab378a0b566f ) },
  { -0x1.550ce1f2aa19cp+106, -0x1.a3471113468e2p+53, INIT_U128( 0xfffffaabcc783557, 0x98cb971ddd972e3c ) },
  { -0x1.0c23c4b818478p+106, 0x1.fac48c4ff5891p+53, INIT_U128( 0xfffffbcf70ed1f9e, 0xe23f589189feb122 ) },
  { -0x1.ed9a73f7db34ep+106, -0x1.02c0884405811p+53, INIT_U128( 0xfffff84996302093, 0x2c5fa7eef77f4fde ) },
  { -0x1.dc096141b812cp+106, 0x1.da114b67b422ap+53, INIT_U128( 0xfffff88fda7af91f, 0xb53b42296cf68454 ) },
  { -0x1.154fff0c2aa00p+106, -0x1.afcb4e175f96ap+53, INIT_U128( 0xfffffbaac003cf55, 0x7fca06963d140d2c ) },
  { -0x1.651b5444ca36ap+106, 0x1.756b7fe0ead70p+53, INIT_U128( 0xfffffa6b92aeecd7, 0x25aead6ffc1d5ae0 ) },
  { -0x1.7bdc1302f7b82p+107, -0x1.6437cb70c86fap+54, INIT_U128( 0xfffff4211f67e842, 0x3ea6f20d23cde418 ) },
  { -0x1.fefc1807fdf83p+107, 0x1.67103bd4ce208p+54, INIT_U128( 0xfffff0081f3fc010, 0x3ed9c40ef5338820 ) },
  { -0x1.b31c1c6566384p+107, -0x1.506996cea0d33p+54, INIT_U128( 0xfffff2671f1cd4ce, 0x3dabe59a4c57cb34 ) },
  { -0x1.8ddc0ec11bb82p+107, 0x1.e59c98f3cb393p+54, INIT_U128( 0xfffff3911f89f722, 0x3f7967263cf2ce4c ) },
  { -0x1.d2b11c01a5624p+107, -0x1.746f3176e8de6p+54, INIT_U128( 0xfffff16a771ff2d4, 0xeda2e433a245c868 ) },
  { -0x1.832573ad064aep+107, 0x1.028cc2d605198p+54, INIT_U128( 0xfffff3e6d46297cd, 0xa940a330b5814660 ) },
  { -0x1.cfff0eeb9ffe2p+108, -0x1.8d39aef51a736p+55, INIT_U128( 0xffffe3000f114600, 0x1d3963288572c650 ) },
  { -0x1.bb77665376eedp+108, 0x1.01daf75c03b5fp+55, INIT_U128( 0xffffe448899ac891, 0x1380ed7bae01daf8 ) },
  { -0x1.8049417f00928p+108, -0x1.87d1be790fa38p+55, INIT_U128( 0xffffe7fb6be80ff6, 0xd73c1720c3782e40 ) },
  { -0x1.5c8e6ffeb91cep+108, 0x1.d5562373aaac5p+55, INIT_U128( 0xffffea371900146e, 0x32eaab11b9d55628 ) },
  { -0x1.2b18cf8c5631ap+108, -0x1.68b234b6d1646p+55, INIT_U128( 0xffffed4e73073a9c, 0xe54ba6e5a4974dd0 ) },
  { -0x1.d01aafc1a0356p+108, 0x1.18b3438431668p+55, INIT_U128( 0xffffe2fe5503e5fc, 0xaa8c59a1c218b340 ) },
  { -0x1.6b72bcfcd6e58p+109, -0x1.ac24110358482p+56, INIT_U128( 0xffffd291a8606523, 0x4e53dbeefca7b7e0 ) },
  { -0x1.f2c48eede5892p+109, 0x1.80c2bc0b01858p+56, INIT_U128( 0xffffc1a76e22434e, 0xdd80c2bc0b018580 ) },
  { -0x1.5bef5f1ab7decp+109, -0x1.57dfe332afbfcp+56, INIT_U128( 0xffffd482141ca904, 0x26a8201ccd504040 ) },
  { -0x1.f62e3bebec5c8p+109, 0x1.d7cb721faf96ep+56, INIT_U128( 0xffffc13a38828274, 0x71d7cb721faf96e0 ) },
  { -0x1.bc17aea3782f6p+109, -0x1.f52a53f3ea54ap+56, INIT_U128( 0xffffc87d0a2b90fa, 0x120ad5ac0c15ab60 ) },
  { -0x1.3edbadf47db76p+109, 0x1.6f33cb08de67ap+56, INIT_U128( 0xffffd8248a417049, 0x156f33cb08de67a0 ) },
  { -0x1.4593eac08b27ep+110, -0x1.ccb7bcb7996f7p+57, INIT_U128( 0xffffae9b054fdd36, 0x466908690cd2120 ) },
  { -0x1.54d063fca9a0cp+110, 0x1.184506c2308a1p+57, INIT_U128( 0xffffaacbe700d597, 0xd2308a0d84611420 ) },
  { -0x1.4126ee2e824dep+110, -0x1.d7f5d7abafebbp+57, INIT_U128( 0xffffafb644745f6c, 0x84501450a8a028a0 ) },
  { -0x1.1ad1c99835a39p+110, 0x1.1125fd18224c0p+57, INIT_U128( 0xffffb94b8d99f297, 0x1e224bfa30449800 ) },
  { -0x1.652f869eca5f1p+110, -0x1.d6492fedac926p+57, INIT_U128( 0xffffa6b41e584d68, 0x38536da024a6db40 ) },
  { -0x1.1300f4762601ep+110, 0x1.e578c81fcaf19p+57, INIT_U128( 0xffffbb3fc2e2767f, 0x8bcaf1903f95e320 ) },
  { -0x1.5e1aee66bc35ep+111, -0x1.6f8f6474df1ecp+58, INIT_U128( 0xffff50f288cca1e5, 0xa41c26e2c838500 ) },
  { -0x1.baceeb91759dep+111, 0x1.8933bdf512678p+58, INIT_U128( 0xffff22988a374531, 0x1624cef7d4499e00 ) },
  { -0x1.d841992db0833p+111, -0x1.8ae99b5315d34p+58, INIT_U128( 0xffff13df336927be, 0x61d45992b3a8b300 ) },
  { -0x1.a1e7017b43ce0p+111, 0x1.43b4562a8768bp+58, INIT_U128( 0xffff2f0c7f425e19, 0x50ed158aa1da2c0 ) },
  { -0x1.370dac566e1b6p+111, -0x1.94ab906929572p+58, INIT_U128( 0xffff647929d4c8f2, 0x49ad51be5b5aa380 ) },
  { -0x1.c477123788ee2p+111, 0x1.92fd0b7325fa2p+58, INIT_U128( 0xffff1dc476e43b88, 0xf64bf42dcc97e880 ) },
  { -0x1.ec8b739bd916fp+112, -0x1.baa54e23754aap+59, INIT_U128( 0xfffe13748c6426e9, 0x22ad58ee455ab00 ) },
  { -0x1.6ac87d90d5910p+112, 0x1.a8a3223f51464p+59, INIT_U128( 0xfffe9537826f2a6f, 0xd451911fa8a3200 ) },
  { -0x1.afcfa2155f9f4p+112, -0x1.a9a42b8753486p+59, INIT_U128( 0xfffe50305deaa060, 0xb2b2dea3c565bd00 ) },
  { -0x1.17d6ed482fadep+112, 0x1.f5638cd7eac71p+59, INIT_U128( 0xfffee82912b7d052, 0x2fab1c66bf563880 ) },
  { -0x1.765be83cecb7dp+112, -0x1.6ca75146d94eap+59, INIT_U128( 0xfffe89a417c31348, 0x249ac575c9358b00 ) },
  { -0x1.0da8939a1b512p+112, 0x1.b782d66d6f05bp+59, INIT_U128( 0xfffef2576c65e4ae, 0xedbc16b36b782d80 ) },
  { -0x1.449d28f2893a5p+113, -0x1.287a5d3450f4cp+60, INIT_U128( 0xfffd76c5ae1aed8b, 0x4d785a2cbaf0b400 ) },
  { -0x1.c8a8d3879151ap+113, 0x1.a9ccaf9353996p+60, INIT_U128( 0xfffc6eae58f0dd5c, 0xda9ccaf935399600 ) },
  { -0x1.fec38d21fd871p+113, -0x1.dba9c7fdb7539p+60, INIT_U128( 0xfffc0278e5bc04f1, 0xc2456380248ac700 ) },
  { -0x1.8f4577b11e8afp+113, 0x1.5cde7b88b9bd0p+60, INIT_U128( 0xfffce175109dc2ea, 0x35cde7b88b9bd000 ) },
  { -0x1.4b392c7296726p+113, -0x1.4a1463429428cp+60, INIT_U128( 0xfffd698da71ad31b, 0x2b5eb9cbd6bd7400 ) },
  { -0x1.e612397dcc247p+113, 0x1.237ac5a046f58p+60, INIT_U128( 0xfffc33db8d0467b7, 0x3237ac5a046f5800 ) },
  { -0x1.d4fdfe97a9fc0p+114, -0x1.87f93af90ff27p+61, INIT_U128( 0xfff8ac0805a1580f, 0xcf00d8a0de01b200 ) },
  { -0x1.672f1bccce5e4p+114, 0x1.0c0baaee18176p+61, INIT_U128( 0xfffa634390ccc687, 0x2181755dc302ec00 ) },
  { -0x1.1ca4cb603949ap+114, -0x1.2ba62e04574c6p+61, INIT_U128( 0xfffb8d6cd27f1ad9, 0x5a8b3a3f75167400 ) },
  { -0x1.9f08b6633e117p+114, 0x1.09303efc12608p+61, INIT_U128( 0xfff983dd267307ba, 0x612607df824c1000 ) },
  { -0x1.e64b7c27cc970p+114, -0x1.09d133f413a26p+61, INIT_U128( 0xfff866d20f60cda3, 0xdec5d9817d8bb400 ) },
  { -0x1.b1d3613d63a6cp+114, 0x1.961b1e3d2c364p+61, INIT_U128( 0xfff938b27b0a7165, 0x32c363c7a586c800 ) },
  { -0x1.b3755f7b66eacp+115, -0x1.c9aa4e879354ap+62, INIT_U128( 0xfff264550424c8a9, 0x8d956c5e1b2ad800 ) },
  { -0x1.8b7eab1516fd6p+115, 0x1.29d147b853a29p+62, INIT_U128( 0xfff3a40aa7574815, 0x4a7451ee14e8a400 ) },
  { -0x1.36519fce6ca34p+115, -0x1.99273eb1324e8p+62, INIT_U128( 0xfff64d73018c9ae5, 0x99b63053b36c6000 ) },
  { -0x1.e1541e27c2a84p+115, 0x1.7aeec8c2f5dd9p+62, INIT_U128( 0xfff0f55f0ec1eabe, 0x5ebbb230bd776400 ) },
  { -0x1.c1a19be583434p+115, -0x1.a9fcf1ad53f9ep+62, INIT_U128( 0xfff1f2f320d3e5e5, 0x9580c394ab018800 ) },
  { -0x1.24bb78544976fp+115, 0x1.c065290780ca5p+62, INIT_U128( 0xfff6da243d5db448, 0xf0194a41e0329400 ) },
  { -0x1.85a6b9ab0b4d7p+116, -0x1.b400f0076801ep+63, INIT_U128( 0xffe7a594654f4b28, 0x25ff87fc4bff1000 ) },
  { -0x1.1d43a4903a874p+116, 0x1.d0dfe61fa1bfdp+63, INIT_U128( 0xffee2bc5b6fc578c, 0xe86ff30fd0dfe800 ) },
  { -0x1.fe15534bfc2abp+116, -0x1.0355de0e06abcp+63, INIT_U128( 0xffe01eaacb403d54, 0x7e5510f8fcaa2000 ) },
  { -0x1.f45aabc1e8b56p+116, 0x1.d31351e3a626ap+63, INIT_U128( 0xffe0ba5543e174aa, 0xe989a8f1d3135000 ) },
  { -0x1.8b4f1907169e3p+116, -0x1.5b89c78eb7139p+63, INIT_U128( 0xffe74b0e6f8e961c, 0x523b1c38a4763800 ) },
  { -0x1.df1efcd3be3dfp+116, 0x1.a7f3e4694fe7cp+63, INIT_U128( 0xffe20e1032c41c21, 0xd3f9f234a7f3e000 ) },
  { -0x1.872ff5950e5ffp+117, -0x1.7e27b336fc4f6p+64, INIT_U128( 0xffcf1a014d5e3400, 0x81d84cc903b0a000 ) },
  { -0x1.b124236762484p+117, 0x1.6d7b71bcdaf6ep+64, INIT_U128( 0xffc9db7b9313b6f9, 0x6d7b71bcdaf6e000 ) },
  { -0x1.87f8e72b0ff1dp+117, -0x1.62f784a4c5ef0p+64, INIT_U128( 0xffcf00e31a9e01c4, 0x9d087b5b3a110000 ) },
  { -0x1.03968ffa072d2p+117, 0x1.3b6e46a676dc9p+64, INIT_U128( 0xffdf8d2e00bf1a5d, 0x3b6e46a676dc9000 ) },
  { -0x1.2ca1d32c5943ap+117, -0x1.8f153b111e2a8p+64, INIT_U128( 0xffda6bc59a74d78a, 0x70eac4eee1d58000 ) },
  { -0x1.653f8e60ca7f2p+117, 0x1.62f372fcc5e6ep+64, INIT_U128( 0xffd3580e33e6b01d, 0x62f372fcc5e6e000 ) },
  { -0x1.283c79c05078fp+118, -0x1.4693ac888d276p+65, INIT_U128( 0xffb5f0e18febe1c1, 0x72d8a6eee5b14000 ) },
  { -0x1.dde23bc1bbc48p+118, 0x1.e9de15a7d3bc2p+65, INIT_U128( 0xff8887710f910ee3, 0xd3bc2b4fa7784000 ) },
  { -0x1.f0909e37e1214p+118, -0x1.029db416053b6p+65, INIT_U128( 0xff83dbd87207b7ad, 0xfac497d3f5894000 ) },
  { -0x1.aa433eaf54868p+118, 0x1.763757e2ec6ebp+65, INIT_U128( 0xff956f30542ade62, 0xec6eafc5d8dd6000 ) },
  { -0x1.b69ea41f6d3d4p+118, -0x1.c786e3418f0ddp+65, INIT_U128( 0xff925856f824b0ac, 0x70f2397ce1e46000 ) },
  { -0x1.47173e028e2e8p+118, 0x1.b4b3f76b6967fp+65, INIT_U128( 0xffae3a307f5c7463, 0x6967eed6d2cfe000 ) },
  { -0x1.cadcc14f95b98p+119, -0x1.3b71e9c476e3dp+66, INIT_U128( 0xff1a919f5835233b, 0x123858ee2470c000 ) },
  { -0x1.87c7494f0f8e9p+119, 0x1.562bda56ac57cp+66, INIT_U128( 0xff3c1c5b587838bd, 0x58af695ab15f0000 ) },
  { -0x1.96fbb1552df76p+119, -0x1.66e64622cdcc9p+66, INIT_U128( 0xff3482275569044a, 0x6466e774c8cdc000 ) },
  { -0x1.fe1fbec1fc3f8p+119, 0x1.1e2ba6e83c575p+66, INIT_U128( 0xff00f0209f01e044, 0x78ae9ba0f15d4000 ) },
  { -0x1.e50df959ca1bfp+119, -0x1.af43e2295e87cp+66, INIT_U128( 0xff0d7903531af201, 0x42f0775a85e10000 ) },
  { -0x1.8ba718a7174e3p+119, 0x1.6d06e084da0dcp+66, INIT_U128( 0xff3a2c73ac7458ed, 0xb41b821368370000 ) },
  { -0x1.123b66ce2476dp+120, -0x1.8f3da8a11e7b5p+67, INIT_U128( 0xfeedc49931db8923, 0x8612baf70c258000 ) },
  { -0x1.a875272d50ea5p+120, 0x1.3edd39fc7dba7p+67, INIT_U128( 0xfe578ad8d2af15b9, 0xf6e9cfe3edd38000 ) },
  { -0x1.1e52e3b23ca5cp+120, -0x1.0db4b58e1b696p+67, INIT_U128( 0xfee1ad1c4dc35a37, 0x925a538f24b50000 ) },
  { -0x1.a1d0acfb43a16p+120, 0x1.468f5d748d1ecp+67, INIT_U128( 0xfe5e2f5304bc5eaa, 0x347aeba468f60000 ) },
  { -0x1.54486218a890cp+120, -0x1.7fb2e8bcff65dp+67, INIT_U128( 0xfeabb79de7576f34, 0x268ba1804d18000 ) },
  { -0x1.306d731c60daep+120, 0x1.44d9004e89b20p+67, INIT_U128( 0xfecf928ce39f252a, 0x26c802744d900000 ) },
  { -0x1.82509cd304a14p+121, -0x1.dab0b1cdb5616p+68, INIT_U128( 0xfcfb5ec659f6bd62, 0x54f4e324a9ea0000 ) },
  { -0x1.bb1058337620bp+121, 0x1.38ef40f271de8p+68, INIT_U128( 0xfc89df4f9913beb3, 0x8ef40f271de80000 ) },
  { -0x1.a237950d446f3p+121, -0x1.02279634044f3p+68, INIT_U128( 0xfcbb90d5e577218f, 0xdd869cbfbb0d0000 ) },
  { -0x1.eda3385ddb467p+121, 0x1.6d14565cda28bp+68, INIT_U128( 0xfc24b98f44497336, 0xd14565cda28b0000 ) },
  { -0x1.3d997ffa7b330p+121, -0x1.756a81acead50p+68, INIT_U128( 0xfd84cd000b0999e8, 0xa957e53152b00000 ) },
  { -0x1.c34030b586806p+121, 0x1.d495314fa92a6p+68, INIT_U128( 0xfc797f9e94f2ff5d, 0x495314fa92a60000 ) },
  { -0x1.dc50c7cfb8a19p+122, -0x1.c15fef7782bfep+69, INIT_U128( 0xf88ebce0c11d7987, 0xd402110fa8040000 ) },
  { -0x1.f2c0d839e581bp+122, 0x1.c15ec12982bd8p+69, INIT_U128( 0xf834fc9f1869f978, 0x2bd8253057b00000 ) },
  { -0x1.70a3d1a0e147ap+122, -0x1.7e67f0e0fccfep+69, INIT_U128( 0xfa3d70b97c7ae150, 0x3301e3e066040000 ) },
  { -0x1.2e65a1ae5ccb4p+122, 0x1.82df387105be7p+69, INIT_U128( 0xfb466979468cd330, 0x5be70e20b7ce0000 ) },
  { -0x1.0da599821b4b3p+122, -0x1.d6dac8ddadb59p+69, INIT_U128( 0xfbc96999f792d305, 0x24a6e44a494e0000 ) },
  { -0x1.c1bc726f8378ep+122, 0x1.2bf87d3457f10p+69, INIT_U128( 0xf8f90e3641f21ca5, 0x7f0fa68afe200000 ) },
  { -0x1.e48f7e6dc91f0p+123, -0x1.638d1f6ac71a4p+70, INIT_U128( 0xf0db840c91b707a7, 0x1cb8254e39700000 ) },
  { -0x1.7287daaae50fcp+123, 0x1.38d7e08671afcp+70, INIT_U128( 0xf46bc12aa8d7824e, 0x35f8219c6bf00000 ) },
  { -0x1.e3400829c6801p+123, -0x1.a2f54dc745eaap+70, INIT_U128( 0xf0e5ffbeb1cbff17, 0x42ac8e2e85580000 ) },
  { -0x1.7db0e0d6fb61cp+123, 0x1.6590a4fccb214p+70, INIT_U128( 0xf41278f94824f259, 0x64293f32c8500000 ) },
  { -0x1.7da5ff94fb4c0p+123, -0x1.38c1390871827p+70, INIT_U128( 0xf412d00358259fb1, 0xcfb1bde39f640000 ) },
  { -0x1.9e7fe5ef3cffdp+123, 0x1.1fdd94a23fbb2p+70, INIT_U128( 0xf30c00d0861801c7, 0xf765288feec80000 ) },
  { -0x1.3ae8cf3e75d1ap+124, -0x1.8c231e4d18464p+71, INIT_U128( 0xec51730c18a2e539, 0xee70d973dce00000 ) },
  { -0x1.8cf1eb0119e3ep+124, 0x1.78824f3af104ap+71, INIT_U128( 0xe730e14fee61c2bc, 0x41279d7882500000 ) },
  { -0x1.018d9ffa031b4p+124, -0x1.50321bb6a0644p+71, INIT_U128( 0xefe726005fce4b57, 0xe6f224afcde00000 ) },
  { -0x1.4d17808a9a2f0p+124, 0x1.ddde67cbbbbcdp+71, INIT_U128( 0xeb2e87f7565d10ee, 0xef33e5ddde680000 ) },
  { -0x1.11920b2e23242p+124, -0x1.3fd858c47fb0bp+71, INIT_U128( 0xeee6df4d1dcdbd60, 0x13d39dc027a80000 ) },
  { -0x1.9f3d28f53e7a5p+124, 0x1.8235dcd1046bcp+71, INIT_U128( 0xe60c2d70ac185bc1, 0x1aee688235e00000 ) },
  { -0x1.be4049a57c809p+125, -0x1.cda2caa59b45ap+72, INIT_U128( 0xc837f6cb506fec32, 0x5d355a64ba600000 ) },
  { -0x1.051515420a2a2p+125, 0x1.cbe5e32997cbdp+72, INIT_U128( 0xdf5d5d57bebabdcb, 0xe5e32997cbd00000 ) },
  { -0x1.a47d938348fb2p+125, -0x1.c6d187b38da31p+72, INIT_U128( 0xcb704d8f96e09a39, 0x2e784c725cf00000 ) },
  { -0x1.f7de605fefbccp+125, 0x1.890d11f1121a2p+72, INIT_U128( 0xc10433f402086989, 0xd11f1121a200000 ) },
  { -0x1.2a5ea71e54bd5p+125, -0x1.6a663b08d4cc8p+72, INIT_U128( 0xdab42b1c35685495, 0x99c4f72b33800000 ) },
  { -0x1.c1e4f27783c9ep+125, 0x1.0d9eaf461b3d6p+72, INIT_U128( 0xc7c361b10f86c50d, 0x9eaf461b3d600000 ) },
  { -0x1.0d5b88d01ab71p+126, -0x1.3f95d34e7f2bap+73, INIT_U128( 0xbca91dcbf9523980, 0xd4596301a8c00000 ) },
  { -0x1.732f5444e65eap+126, 0x1.8b01886716031p+73, INIT_U128( 0xa3342aeec6685b16, 0x310ce2c06200000 ) },
  { -0x1.78570c58f0ae2p+126, -0x1.11af3a34235e8p+73, INIT_U128( 0xa1ea3ce9c3d475dc, 0xa18b97b943000000 ) },
  { -0x1.aadf1d0955be4p+126, 0x1.b33fdb99667fcp+73, INIT_U128( 0x954838bdaa907366, 0x7fb732ccff800000 ) },
  { -0x1.9dd258833ba4bp+126, -0x1.37d1080a6fa21p+73, INIT_U128( 0x988b69df3116d190, 0x5defeb20bbe00000 ) },
  { -0x1.2bc22ad257846p+126, 0x1.9f5ff87d3ebffp+73, INIT_U128( 0xb50f754b6a1eeb3e, 0xbff0fa7d7fe00000 ) }
};

static const int numTests = sizeof(testList) / sizeof(struct testCase);
