use core:lang;
use core:asm;
use lang:asm;
use lang:bs:macro;

/**
 * Memory management helpers.
 */

on Compiler:

/**
 * Markers we use to mark allocations in C++.
 *
 * We put these in the 'filled' member of the arrays we allocate.
 */
enum AllocFlags : bitmask {
	// Mask for the size.
	sizeMask = 0x1FFFFFFF,

	// This was allocated on the heap.
	heapAlloc = 0x80000000,

	// This was allocated using new[].
	arrayAlloc = 0x40000000,

	// Static allocation (e.g. a string).
	staticAlloc = 0x20000000,
}

/**
 * Variable allocated in C++.
 */
class CppVar on Compiler {
	// Dummy value.
	init() {
		init {}
	}

	// No initialization required.
	init(Var var) {
		init {
			v = var;
			needsPart = false;
		}
	}

	init(Var var, Bool needsPart) {
		init {
			v = var;
			needsPart = needsPart;
		}
	}

	// The variable itself.
	Var v;

	// Do we need to put it in a new part?
	Bool needsPart;

	// Call when the variable has been initialized properly.
	void created(CodeGen g) {
		if (!needsPart)
			return;

		Listing l = g.l;

		// Mark the variable as 'created' by setting 'filled' to 1.
		l << mov(ptrA, v);
		l << mov(ptrRel(ptrA, Offset(sPtr)), ptrConst(1));

		// Activate exception handling.
		l << activate(v);

		needsPart = false;
	}

	// Place the adjusted value of this variable in 'dst'.
	void adjust(CodeGen gen, Operand dst) {
		gen.l << mov(dst, v);
		// Skip over the array header.
		gen.l << add(dst, ptrConst(sPtr * 2));
	}

	// Put pointer data in ptrB (base) and ecx (offset).
	void ptr(CodeGen gen) {
		gen.l << mov(ptrB, v);
		gen.l << mov(ecx, natConst(sPtr * 2));
	}
}

// Allocate memory for a C++ type. Return the variable where it was created.
CppVar allocType(CodeGen gen, Type type) {
	// We attach a destructor to all objects so that we can keep track of which are alive.
	var dtor = type.cppDtor();

	Listing l = gen.l;
	Var v = l.createVar(gen.block, sPtr, dtor.ref, FreeOpt:default + FreeOpt:inactive);

	// Allocate memory.
	l << fnParam(ptrDesc, type.typeRef);
	l << fnParam(ptrDesc, ptrConst(1));
	l << fnCall(ref(BuiltIn:allocArray), false, ptrDesc, v);

	CppVar(v, true);
}

// Allocate memory for a C++ type that will be used as a temporary (making it show up on the stack).
CppVar allocTemporary(CodeGen gen, Type type) {
	CppVar v = allocType(gen, type);

	Listing:VarInfo info("-", type, true, SrcPos());
	gen.l.varInfo(v.v, info);
	v;
}

