/*
    Copyright (C) 2002 Paul Davis

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id$
*/

#include <set>
#include <string>
#include <cstring>
#include <stdint.h>

#include "pbd/pthread_utils.h"
#ifdef WINE_THREAD_SUPPORT
#include <fst.h>
#endif

#ifdef COMPILER_MSVC
DECLARE_DEFAULT_COMPARISONS(pthread_t)  // Needed for 'DECLARE_DEFAULT_COMPARISONS'. Objects in an STL container can be
                                        // searched and sorted. Thus, when instantiating the container, MSVC complains
                                        // if the type of object being contained has no appropriate comparison operators
                                        // defined (specifically, if operators '<' and '==' are undefined). This seems
                                        // to be the case with ptw32 'pthread_t' which is a simple struct.
#endif

using namespace std;

typedef std::list<pthread_t> ThreadMap;
static ThreadMap all_threads;
static pthread_mutex_t thread_map_lock = PTHREAD_MUTEX_INITIALIZER;
static Glib::Threads::Private<char> thread_name (free);

namespace PBD {
	PBD::Signal3<void,pthread_t,std::string,uint32_t> ThreadCreatedWithRequestSize;
}

using namespace PBD;

static int thread_creator (pthread_t* thread_id, const pthread_attr_t* attr, void *(*function)(void*), void* arg)
{
#ifdef WINE_THREAD_SUPPORT
       return wine_pthread_create (thread_id, attr, function, arg);
#else
       return pthread_create (thread_id, attr, function, arg);
#endif
}


void
PBD::notify_event_loops_about_thread_creation (pthread_t thread, const std::string& emitting_thread_name, int request_count)
{
	/* notify threads that may exist in the future (they may also exist
	 * already, in which case they will catch the
	 * ThreadCreatedWithRequestSize signal)
	 */
	EventLoop::pre_register (emitting_thread_name, request_count);

	/* notify all existing threads */
	ThreadCreatedWithRequestSize (thread, emitting_thread_name, request_count);
}

struct ThreadStartWithName {
    void* (*thread_work)(void*);
    void* arg;
    std::string name;

    ThreadStartWithName (void* (*f)(void*), void* a, const std::string& s)
	    : thread_work (f), arg (a), name (s) {}
};

static void*
fake_thread_start (void* arg)
{
	ThreadStartWithName* ts = (ThreadStartWithName*) arg;
	void* (*thread_work)(void*) = ts->thread_work;
	void* thread_arg = ts->arg;

	/* name will be deleted by the default handler for GStaticPrivate, when the thread exits */

	pthread_set_name (ts->name.c_str());

	/* we don't need this object anymore */

	delete ts;

	/* actually run the thread's work function */

	void* ret = thread_work (thread_arg);

	/* cleanup */

	pthread_mutex_lock (&thread_map_lock);

	for (ThreadMap::iterator i = all_threads.begin(); i != all_threads.end(); ++i) {
		if (pthread_equal ((*i), pthread_self())) {
			all_threads.erase (i);
			break;
		}
	}

	pthread_mutex_unlock (&thread_map_lock);

	/* done */

	return ret;
}

int
pthread_create_and_store (string name, pthread_t  *thread, void * (*start_routine)(void *), void * arg)
{
	pthread_attr_t default_attr;
	int ret;

	// set default stack size to sensible default for memlocking
	pthread_attr_init(&default_attr);
	pthread_attr_setstacksize(&default_attr, 500000);

	ThreadStartWithName* ts = new ThreadStartWithName (start_routine, arg, name);

	if ((ret = thread_creator (thread, &default_attr, fake_thread_start, ts)) == 0) {
		pthread_mutex_lock (&thread_map_lock);
		all_threads.push_back (*thread);
		pthread_mutex_unlock (&thread_map_lock);
	}

	pthread_attr_destroy(&default_attr);

	return ret;
}

void
pthread_set_name (const char *str)
{
	/* copy string and delete it when exiting */

	thread_name.set (strdup (str));
}

const char *
pthread_name ()
{
	const char* str = thread_name.get ();

	if (str) {
		return str;
	}
	return "unknown";
}

void
pthread_kill_all (int signum)
{
	pthread_mutex_lock (&thread_map_lock);
	for (ThreadMap::iterator i = all_threads.begin(); i != all_threads.end(); ++i) {
		if (!pthread_equal ((*i), pthread_self())) {
			pthread_kill ((*i), signum);
		}
	}
	all_threads.clear();
	pthread_mutex_unlock (&thread_map_lock);
}

void
pthread_cancel_all ()
{
	pthread_mutex_lock (&thread_map_lock);

	for (ThreadMap::iterator i = all_threads.begin(); i != all_threads.end(); ) {

		ThreadMap::iterator nxt = i;
		++nxt;

		if (!pthread_equal ((*i), pthread_self())) {
			pthread_cancel ((*i));
		}

		i = nxt;
	}
	all_threads.clear();
	pthread_mutex_unlock (&thread_map_lock);
}

void
pthread_cancel_one (pthread_t thread)
{
	pthread_mutex_lock (&thread_map_lock);
	for (ThreadMap::iterator i = all_threads.begin(); i != all_threads.end(); ++i) {
		if (pthread_equal ((*i), thread)) {
			all_threads.erase (i);
			break;
		}
	}

	pthread_cancel (thread);
	pthread_mutex_unlock (&thread_map_lock);
}
