// Copyright (c) 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_BROWSER_WATCHER_WATCHER_CLIENT_WIN_H_
#define COMPONENTS_BROWSER_WATCHER_WATCHER_CLIENT_WIN_H_

#include "base/callback.h"
#include "base/macros.h"
#include "base/process/process.h"

namespace base {
class CommandLine;
}  // namespace base

namespace browser_watcher {

// An interface class to take care of the details in launching a browser
// watch process.
class WatcherClient {
 public:
  // A CommandLineGenerator generates command lines that will launch a separate
  // process and pass the supplied HANDLE to ExitCodeWatcher in that process.
  typedef base::Callback<base::CommandLine(HANDLE)> CommandLineGenerator;

  // Constructs a watcher client that launches its watcher process using the
  // command line generated by |command_line_generator|.
  explicit WatcherClient(const CommandLineGenerator& command_line_generator);

  ~WatcherClient();

  // Launches the watcher process such that the child process is able to inherit
  // a handle to the current process. If use_legacy_launch() is true, this uses
  // a non-threadsafe legacy launch mode that's compatible with Windows XP.
  void LaunchWatcher();

  // Ensures that |handle| may be inherited by the watcher process. |handle|
  // must still be inheritable, and it's the client's responsibility to
  // communicate the value of |handle| to the launched process.
  void AddInheritedHandle(HANDLE handle);

  // Returns the launched process.
  const base::Process& process() const { return process_; }

  // Accessors, exposed only for testing.
  bool use_legacy_launch() const { return use_legacy_launch_; }
  void set_use_legacy_launch(bool use_legacy_launch) {
      use_legacy_launch_ = use_legacy_launch;
  }

 private:
  // If true, the watcher process will be launched with XP legacy handle
  // inheritance. This is not thread safe and can leak random handles into the
  // child process, but it's the best we can do on XP.
  bool use_legacy_launch_;

  // The CommandLineGenerator passed to the constructor.
  CommandLineGenerator command_line_generator_;

  // A handle to the launched watcher process. Valid after a successful
  // LaunchWatcher() call.
  base::Process process_;

  std::vector<HANDLE> inherited_handles_;

  DISALLOW_COPY_AND_ASSIGN(WatcherClient);
};

}  // namespace browser_watcher

#endif  // COMPONENTS_BROWSER_WATCHER_WATCHER_CLIENT_WIN_H_
