// ****************************************************************************
// * YaNuCa Version 1.0 | (c) 2003-2008 by Dominic König (nursix.org)         *
// ****************************************************************************
// * yacore.js : Kernfunktionen der Ernährungskalkulation                     *
// ****************************************************************************
// * This program is free software: you can redistribute it and/or modify     *
// * it under the terms of the GNU Affero General Public License as published *
// * by the Free Software Foundation, either version 3 of the License, or     *
// * (at your option) any later version.                                      *
// *                                                                          *
// * This program is distributed in the hope that it will be useful,          *
// * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
// * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
// * GNU Affero General Public License for more details.                      *
// *                                                                          *
// * You should have received a copy of the GNU Affero General Public License *
// * along with this program.  If not, see <http://www.gnu.org/licenses/>.    *
// *                                                                          *
// * For more information about nursix.org, please visit us at                *
// * our website at http://www.nursix.org                                     *
// ****************************************************************************

// ****************************************************************************
// Globale Variablen
//

// Input
var iGeschlecht = 0;      // Geschlecht, 0=weiblich, 1=männlich
var iAlter      = 70;     // Alter in Jahren
var iLaenge     = 165;    // Größe in cm
var iStatur     = 22;     // geschätzter BMI
var iGewicht    = 0;      // Körpergewicht (Normalgewicht) in kg, wenn bekannt, sonst 0
var iFieber     = 1.0;    // Fieberfaktor (0,5..1,4)
var iDarm       = 0;      // enterale Zufuhrmöglichkeit (Code)
var iGlucose    = 0;      // Glucosetoleranzgrenze (Code)
var iMaxRate    = 150;    // maximale Laufrate der Sondenkost
var iManProtein = 0;      // manuelle Proteinmenge (ArrayIndex)
var iManEnergie = 0;      // manuelle Energiezufuhr (Array-Index)
var iOral       = 0;      // orale Kostaufbaustufe (Array-Index)

var iGruppe     = "IntensivPassiv"; // Patientengruppe (String)
var iSondenkost = "Standard";       // Sondenkost (String)

// Output
var oGewicht    = 0.0;                     // Angenommenes Patientengewicht in kg
var oProteinAbs = 0.0, oProteinRel = 0.0;  // Proteinzufuhr absolut/relativ in g/d
var oKHAbs      = 0.0, oKHRel      = 0.0;  // Kohlenhydratbedarf absolut/relativ in g/d
var oFettAbs    = 0.0, oFettRel    = 0.0;  // Fettbedarf absolut/relativ in g/d
var oBedarfAbs  = 0.0;                     // Energiebedarf absolut
var oZufuhrAbs  = 0.0, oZufuhrRel  = 0.0;  // Energiezufuhr absolut/relativ in kcal/d

var oStufeSK   = new Array( 0, 0, 0, 0, 0 );  // Laufraten Sondenkost in ml/h
var oStufeH2O  = new Array( 0, 0, 0, 0, 0 );  // Wasseranteil der Sondenkost in ml/h
var oStufePort = new Array( 5, 5, 5, 5, 5 );  // Anzahl Portionen á 4h

var oStufeA10  = new Array( 0, 0, 0, 0, 0 );  // Laufrate Aminosre.lsg. 10% in ml/h
var oStufeG40  = new Array( 0, 0, 0, 0, 0 );  // Laufrate Glucoselsg. 40% in ml/h
var oStufeL20  = new Array( 0, 0, 0, 0, 0 );  // Laufrate Fettemuls. 20% in ml/h

// ****************************************************************************
// yaBedarfsberechnung: Berechnung des Energiebedarfs/Nährstoffbedarfs
//
// Parameter: keine

function yaBedarfsberechnung() {

  // Übernahme der Parameter

  var alter      = iAlter;
  var laenge     = iLaenge;
  var bmi        = iStatur;
  var geschlecht = iGeschlecht;
  var fieber     = iFieber;
  var gruppe     = iGruppe;

  // Berechne Normgewicht

  var gewicht = 0;

  if ( iGewicht != 0 ) {
    gewicht = iGewicht;
  } else {
    gewicht = yaNormalgewicht( alter, laenge, bmi );
  }

  // Berechne Grundumsatz

  var grundumsatz = yaGrundumsatz( geschlecht, alter, laenge, gewicht );

  // Berechne Gesamtbedarf

  var bedarf = yaBedarfsEskalation( gewicht, grundumsatz, yaStressFaktor[gruppe], fieber );
  var zufuhr = bedarf;

  // Berücksichtige manuelle Energievorgabe

  if ( iManEnergie != 0 ) {
    var energie = yaManEnergie[ iManEnergie-1 ];
    zufuhr = energie * gewicht;
  }

  // Bestimme aktuellen Proteinbereich

  var aktProteinMin, aktProteinStd, aktProteinMax;

  if ( iManProtein != 0 ) {
    aktProteinMin = yaManProteinMin[ iManProtein-1 ];
    aktProteinStd = yaManProteinStd[ iManProtein-1 ];
    aktProteinMax = yaManProteinMax[ iManProtein-1 ];
  } else {
    aktProteinMin = yaProteinZufuhr.autoMinimum;
    aktProteinStd = yaProteinZufuhr.Standard;
    aktProteinMax = yaProteinZufuhr.autoMaximum;
  }

  // Bestimme aktuellen Glucosebereich

  var aktGlucoseMin, aktGlucoseMax;

  if ( iGlucose != 0 ) {
    aktGlucoseMin = yaManGlucoseMin[ iGlucose-1 ];
    aktGlucoseMax = yaManGlucoseMax[ iGlucose-1 ];
  } else {
    aktGlucoseMin = yaGlucoseZufuhr.autoMinimum;
    aktGlucoseMax = yaGlucoseZufuhr.autoMaximum;
  }

  // Berechne Zufuhrmengen

  // Das hier ist richtig kompliziert, und kann gern noch "mathematisch" überarbeitet
  // werden. Genau genommen handelt es sich um ein Ungleichungssystem aus
  // 11 Ungleichungen mit 3 Unbekannten und einigen "vagen" Nebenbedingungen,
  // das ich hier durch Fallunterscheidungen zu lösen versuche. Nicht schön,
  // aber es funktioniert soweit.
  //
  // Wer eine bessere Lösungsidee hat: nur zu!

  // Verwende Protein-Standard-Wert

  var autoProteinEnergie = aktProteinStd * gewicht * yaSpezEnergie.Protein;

  // Berechne Defizit

  var defizit = zufuhr - autoProteinEnergie;

  // Glucose-Index: Anpassung der Glucosemenge an die gewählte
  // Sondennahrung, damit so wenig wie möglich parenteral
  // ergänzt werden muß, dieser muß jedoch zwingend zwischen
  // 0.33 und 0.85 liegen, sonst wird 0.65 angenommen

  var glucoseindex = (  skk[iSondenkost] * yaSpezEnergie.KH ) /
                     (( skenergie[iSondenkost] * 100 ) -
                      ( skp[iSondenkost] * yaSpezEnergie.Protein ));

  if ( glucoseindex < 0.33 || glucoseindex > 0.85 ) {
    glucoseindex = 0.65;
  }

  var autoGlucoseEnergie = defizit * glucoseindex;
  var autoFettEnergie    = defizit - autoGlucoseEnergie;

  // Korrekturrechnung: Protein-Utilisations-Limit

  if (( autoProteinEnergie / zufuhr ) > yaProteinUtilisationsLimit ) {

    if ((( aktProteinMin * gewicht * yaSpezEnergie.Protein ) / zufuhr ) > yaProteinUtilisationsLimit ) {

      // Falls Proteinzufuhr nicht reduziert werden kann:

      // Erhöhe Gesamtzufuhr auf Utilisationsminimum
      autoProteinEnergie = aktProteinMin * gewicht * yaSpezEnergie.Protein;
      zufuhr = autoProteinEnergie / yaProteinUtilisationsLimit;

    } else {

      // Vermindere Proteinmenge so, daß Utilisationslimit erreicht wird
      autoProteinEnergie = zufuhr * yaProteinUtilisationsLimit;

    }
  }

  defizit = zufuhr - autoProteinEnergie;

  // Korrekturrechnung: Glucose-Minimum

  var glucoseminimum = aktGlucoseMin * gewicht * yaSpezEnergie.KH;
  var proteinminimum = aktProteinMin * gewicht * yaSpezEnergie.Protein;

  if ( defizit < glucoseminimum ) {

    if (( zufuhr - proteinminimum  ) < glucoseminimum ) {

      // Falls Proteinzufuhr nicht vermindert werden kann:

      // Erhöhe Zufuhr, so daß Glucoseminimum erreicht wird
      zufuhr = proteinminimum + glucoseminimum;

      // und minimiere Protein
      autoProteinEnergie = proteinminimum;

    } else {

      // Verringere Proteinzufuhr, so daß Glucoseminimum erreicht wird
      autoProteinEnergie = zufuhr - glucoseminimum;

    }
  }

  // Berechne Defizit neu

  defizit = zufuhr - autoProteinEnergie;

  // Korrekturrechnung: Maxima für Glucose und Fett

  var fettmaximum    = yaFettZufuhr.autoMaximum * gewicht * yaSpezEnergie.Fett;
  var glucosemaximum = aktGlucoseMax * gewicht * yaSpezEnergie.KH;
  var proteinmaximum = Math.min(( zufuhr * yaProteinUtilisationsLimit ),
                                ( aktProteinMax * gewicht * yaSpezEnergie.Protein ));

  if ( glucosemaximum < ( defizit * glucoseindex )) {

    // Glucose-Maximum genügt nicht für vorgegebenen Glucoseindex

    // versuche zunächst, Protein zu steigern, bis Glucosemaximum für
    // Glucoseindex ausreicht
    autoProteinEnergie = Math.min(( zufuhr - glucosemaximum * ( 1/glucoseindex )), proteinmaximum );

    // und maximiere Glucose
    autoGlucoseEnergie = glucosemaximum;

    // Berechne Rest-Defizit
    defizit = zufuhr - autoProteinEnergie - autoGlucoseEnergie;

    if ( fettmaximum < defizit ) {

      // Fettmaximum genügt nicht für verbleibendes Defizit:

      if ( autoProteinEnergie < proteinmaximum ) {

        // Falls Proteinmaximum noch nicht ausgeschöpft, versuche weitere Proteinsteigerung
        autoProteinEnergie = Math.min( proteinmaximum, ( autoProteinEnergie + ( defizit - fettmaximum )));

        // Berechne Defizit neu
        defizit = zufuhr - autoProteinEnergie - autoGlucoseEnergie;
      }

      if (( fettmaximum < defizit ) && (( defizit - fettmaximum ) > yaMaximumDefizit )) {

        // Fettmaximum genügt trotzdem nicht:

        yaAlertDeficit( Math.round( defizit - fettmaximum ));

        // nimm Fettmaximum und reduziere Zufuhrmenge um Restdefizit
        autoFettEnergie = fettmaximum;

        zufuhr -= ( defizit - fettmaximum );

      } else {

        // berechne Fettmenge aus Defizit
        autoFettEnergie = defizit;
      }
    } else {

      // berechne Fettmenge aus Defizit
      autoFettEnergie = defizit;
    }

  } else {

    // Versuche Glucose an Glucoseindex anzupassen
    autoGlucoseEnergie = Math.min( glucosemaximum, defizit * glucoseindex );

    // Glucoseminimum nicht unterschreiten!
    if ( autoGlucoseEnergie < glucoseminimum ) {
      autoGlucoseEnergie = glucoseminimum;
    }

    // Berechne Restdefizit
    defizit = zufuhr - autoProteinEnergie - autoGlucoseEnergie;

    if ( defizit > fettmaximum ) {

      // Restdefizit ist durch Fettmaximum nicht zu decken:

      if (( autoProteinEnergie + ( defizit - fettmaximum )) > proteinmaximum ) {

        // Restdefizit ist auch mit Proteinmaximierung nicht durch Fett zu decken

        // maximiere Protein
        autoProteinEnergie = proteinmaximum;

        // maximiere Fett
        autoFettEnergie    = fettmaximum;

        // Berechne Restdefizit
        defizit = zufuhr - autoProteinEnergie - autoFettEnergie;

        if (( defizit > glucosemaximum ) && ( defizit > yaMaximumDefizit )) {

          // Restdefizit durch Glucosemaximum nicht zu decken:

          yaAlertDeficit( Math.round( defizit - glucosemaximum ));

          // maximiere Glucose
          autoGlucoseEnergie = glucosemaximum;

          // Reduziere Zufuhr auf maximal mögliche Menge
          zufuhr = autoProteinEnergie + autoGlucoseEnergie + autoFettEnergie;

        } else {

          // berechne Glucosemenge aus Defizit
          autoGlucoseEnergie = defizit;
        }
      } else {

        // erhöhe Proteinmenge so, daß Fettmaximum ausreicht
        autoProteinEnergie += defizit - fettmaximum;

        // maximiere Fett
        autoFettEnergie = fettmaximum;
      }
    } else {

      // bereche Fettmenge aus Defizit
      autoFettEnergie = defizit;
    }
  }

  // Reduktion für oralen Kostaufbau

  if ( iOral != 0 ) {

    var redMenge = yaReduktionsStufe[ iOral -1 ];

    // Wenn das Defizit infolge Reduktion das Limit übersteigt
    if (( zufuhr - redMenge ) > yaMaximumOralDefizit ) {

      // Ermittle Preferenzen der Makronährstoffe in der oralen Nahrung
      var prefsum = yaOraleNahrung.Protein + yaOraleNahrung.KH + yaOraleNahrung.Fett;
      var ProtRF  = yaOraleNahrung.Protein / prefsum;
      var KHRF    = yaOraleNahrung.KH      / prefsum;
      var FettRF  = yaOraleNahrung.Fett    / prefsum;

      // Reduziere Zufuhr um entsprechenden Anteil
      autoProteinEnergie -= ProtRF * redMenge;
      autoGlucoseEnergie -= KHRF   * redMenge;
      autoFettEnergie    -= FettRF * redMenge;

      // Reduziere Gesamtzufuhr
      zufuhr -= redMenge;

    } else {

      // sonst keine künstliche Ernährung erforderlich
      autoProteinEnergie = 0;
      autoGlucoseEnergie = 0;
      autoFettEnergie    = 0;

      zufuhr = 0;

    }
  }

  // Ergebnisse

  oProteinAbs = autoProteinEnergie / yaSpezEnergie.Protein;
  oProteinRel = oProteinAbs / gewicht;
  oKHAbs      = autoGlucoseEnergie / yaSpezEnergie.KH;
  oKHRel      = oKHAbs / gewicht;
  oFettAbs    = autoFettEnergie / yaSpezEnergie.Fett;
  oFettRel    = oFettAbs / gewicht;

  oGewicht   = gewicht;
  oBedarfAbs = bedarf;

  oZufuhrAbs = zufuhr;
  oZufuhrRel = zufuhr / gewicht;
}

// ****************************************************************************
// yaErnaehrungsplanung: Berechnung der Laufraten
//
// Parameter: keine

function yaErnaehrungsplanung() {

  // Berechne Tagesmenge Sondennahrung
  var MengeEnergie  = oZufuhrAbs  / skenergie[iSondenkost];
  var MengeProtein  = oProteinAbs / ( skp[iSondenkost] / 100 );
  var MengeKH       = oKHAbs      / ( skk[iSondenkost] / 100 );

  // Berechne Toleranzbereich
  var MengeEnergieMin = MengeEnergie * ( 1 - yaMaxEnteraleToleranz );
  var MengeEnergieMax = MengeEnergie * ( 1 + yaMaxEnteraleToleranz );

  var MengeProteinMin = MengeProtein * ( 1 - yaMaxEnteraleToleranz );
  var MengeProteinMax = MengeProtein * ( 1 + yaMaxEnteraleToleranz );

  var MengeKHMin      = MengeKH      * ( 1 - yaMaxEnteraleToleranz );
  var MengeKHMax      = MengeKH      * ( 1 + yaMaxEnteraleToleranz );

  // Keine der Mengen sollte überschritten werden!
  var MengeMaximal = Math.min(
                     Math.min( MengeKHMax, MengeProteinMax ),
                     Math.min( MengeKHMax, MengeEnergieMax ));

  // Falls eines der Minima größer als das Maximum ist, dann nimm das Maximum
  if (( MengeEnergieMin > MengeMaximal ) ||
      ( MengeProteinMin > MengeMaximal ) ||
      ( MengeKHMin      > MengeMaximal )) {
    MengeGesamt = MengeMaximal;
  } else {
    // Sonst nimm das größte der Minima und berechne Optimum
    MengeMinimal = Math.max(
                   Math.max( MengeEnergieMin, MengeProteinMin ),
                   Math.max( MengeEnergieMin, MengeKHMin      ));

    MengeGesamt  = ( MengeMinimal + MengeMaximal ) / 2;
  }

  // Berechne Sondenkost-Laufraten

  var RateMinimal  = 10;
  var RateMaximal  = Math.min( skmaxrate[iSondenkost], iMaxRate );

  // Berechne Rate für Vollaufbau

  var AnzahlPortionen = 4;
  var AnzahlStufen    = 5;

  var VollAufbau = Math.max( RateMinimal, ( MengeGesamt / ( AnzahlPortionen * 4 )));

  // Ueberschreitung der maximalen Laufgeschwindigkeit?
  if ( Math.round( VollAufbau ) > RateMaximal ) {

    // Versuche Verteilung auf 5 Gaben...
    VollAufbau      = Math.max( RateMinimal, ( VollAufbau * 16 / 20 ));
    AnzahlPortionen = 5;

    // Immer noch zu hohe Laufgeschwindigkeit?
    // Begrenze auf maximale Laufgeschwindigkeit
    if ( Math.round( VollAufbau ) > RateMaximal ) {
      VollAufbau = RateMaximal;
    }
  }

  // Berechne Aufbaustufen
  for ( var i=0; i < AnzahlStufen; i++ ) {

    oStufeSK[i]  = yaStandardRate[i];
    oStufeA10[i] = 0; // erforderlich! (Rücksetzung der Werte)
    oStufeG40[i] = 0; // erforderlich!
    oStufeL20[i] = 0; // erforderlich!

    if ( oStufeSK[i] > VollAufbau ) {
      oStufeSK[i]   = VollAufbau;
      oStufePort[i] = AnzahlPortionen;
    } else {
      oStufePort[i] = 4;
    }

    oStufePort[0] = 5;

    oStufeH2O[i] = Math.round( oStufeSK[i] * skwasser[iSondenkost] / 500 ) * 5;

    // Tagesmenge an Sondenkost (wird in Stufe 0 immer als 0ml angenommen)
    var TagesmengeSK = 0;

    if ( i > 0 ) {
      TagesmengeSK = oStufeSK[i] * oStufePort[i] * 4;
    }

    // Tagesmengen der Makronährstoffe in der Sondenkost
    var Tagesmenge = {
      Protein : TagesmengeSK * ( skp[iSondenkost] / 100 ),
      KH      : TagesmengeSK * ( skk[iSondenkost] / 100 ),
      Fett    : TagesmengeSK * ( skf[iSondenkost] / 100 )
    }

    // Defizite bei der Sondenkost
    var Defizit = {
      Protein : Math.max( 0, oProteinAbs - Tagesmenge.Protein ),
      KH      : Math.max( 0, oKHAbs      - Tagesmenge.KH      ),
      Fett    : Math.max( 0, oFettAbs    - Tagesmenge.Fett    ),

      Energie : Math.max( 0,
                ( oKHAbs * yaSpezEnergie.KH + oFettAbs * yaSpezEnergie.Fett ) -
                ( Tagesmenge.KH * yaSpezEnergie.KH + Tagesmenge.Fett * yaSpezEnergie.Fett ))
    }

    // Fehlt mehr Protein als Toleranzgrenze?
    if (( Defizit.Protein > 0 ) && (( Defizit.Protein / oProteinAbs ) > yaMaxEnteraleToleranz )) {

      // Dann ergänze Aminosäuren!
      oStufeA10[i] = ( Defizit.Protein / 0.1 ) / 24;
    }

    // Fehlt Energie?
    if ( Defizit.Energie > yaMaximumDefizit ) {

      // Fülle zuerst fehlende KH auf
      if ( Defizit.KH > 0 ) {
        oStufeG40[i] = Math.min(
                         ( Defizit.KH / 0.4 ) / 24,
                         (( Defizit.Energie / yaSpezEnergie.KH ) / 0.4 ) / 24 );
        Defizit.Energie = Defizit.Energie - (( oStufeG40[i] * 24 ) * 0.4 ) * yaSpezEnergie.KH;
      }

      // Nur, wenn dann trotzdem noch Energie fehlt, ergänze noch Fett
      if ( Defizit.Energie > 0 ) {
        oStufeL20[i] = Math.min(
                         ( Defizit.Fett / 0.2 ) / 24,
                         (( Defizit.Energie / yaSpezEnergie.Fett ) / 0.2 ) / 24 );
      }
    }
    // Hier könnte ja immer noch Energie fehlen, oder? Nein - natürlich nicht!
  }
}

// ****************************************************************************
// yaNormalgewicht: Normalgewicht entsprechend geschätztem BMI und Alter
//
// Parameter:
//
//   alter      : Alter in Jahren
//   laenge     : Körperlänge in cm
//   bmi        : Body-Mass-Index = (Gewicht in kg) / (Körperlänge in m)²

function yaNormalgewicht( alter, laenge, bmi ) {

  var korrektur     = alter / 20.0;
  var normalgewicht = ( bmi + korrektur ) * (laenge/100.0)*(laenge/100.0);

  return( normalgewicht );
}

// ****************************************************************************
// yaGrundumsatz: Schätzung des Grundumsatzes nach Harris und Benedict
//
// Kommentar: Diese Formeln unterschätzen den Grundumsatz normalerweise
//            geringfügig (im Mittel um etwa 10%), deshalb muß eventuell
//            ein Korrekturfaktor erwogen werden.
//
// Parameter:
//
//   geschlecht : 0=weiblich, 1=männlich
//   alter      : Alter in Jahren
//   laenge     : Körperlänge in cm
//   gewicht    : Körpergewicht in kg

function yaGrundumsatz( geschlecht, alter, laenge, gewicht ) {

  var korrekturfaktor = 1.025; // prinzipiell diskutabel!
  var grundumsatz     = 0.0;

  if ( geschlecht == "0" ) {
    // weiblich
    grundumsatz = 655.0  +  9.6 * gewicht + 1.8 * laenge - 4.7 * alter;
  } else {
    // männlich
    grundumsatz = 66.5 + 13.8 * gewicht + 5.0 * laenge - 6.8 * alter;
  }

  return( grundumsatz * korrekturfaktor );
}

// ****************************************************************************
// yaBedarfsEskalation: Berechnung des durch Streß und Fieber eskalierten
//                      Gesamtenergiebedarfs
//
// Parameter:
//   gewicht      = das angenommene Gewicht des Patienten in kg (30..250 kg)
//   grundumsatz  = der geschätzte Grundumsatz
//   stressfaktor = ein Streßfaktor  (kann zwischen 0,5 und 1,4 liegen)
//   fieber       = der Fieberfaktor (kann zwischen 0,5 und 1,4 liegen)

function yaBedarfsEskalation( gewicht, grundbedarf, stressfaktor, fieberfaktor ) {

  // Konstanten

  var grundrelationMin = 15; // Minimum Grundrelation in kcal/kg*d
  var grundrelationMax = 40; // Maximum Grundrelation in kcal/kg*d
  var skalierungsLimit = 47; // Maximal mögliche Energiezufuhr in kcal/kg*d

  // Variablen

  var skalierungsfaktor = 1.0;         // Skalierungsfaktor, s.u.
  var umsatz            = grundbedarf; // Variable zum Berechnen
  var stress            = 1.0;         // skalierter Gesamtstreßfaktor

  var masse = gewicht;      // korrigierter Wert für Gewicht (muß zwischen 30 und 250kg liegen)
  var sf    = stressfaktor; // korrigierter Wert für Streßfaktor
  var tf    = fieberfaktor; // korrigierter Wert für Fieberfaktor

  // Korrigiere ggf. Grenzwertüberschreitungen

  if ( masse < 30  ) { masse = 30;  }
  if ( masse > 250 ) { masse = 250; }

  if ( sf < 0.5 ) { sf = 0.5; }
  if ( sf > 1.4 ) { sf = 1.4; }

  if ( tf < 0.5 ) { tf = 0.5; }
  if ( tf > 1.4 ) { tf = 1.4; }

  // Berechne die Grundrelation = Grundumsatz je kg Körpergewicht

  var grundrelation = umsatz / masse;

  // Limitiere die Grundrelation innerhalb der vorgegebenen Grenzen

  if ( grundrelation > grundrelationMax ) {
    umsatz        = masse * grundrelationMax;
    grundrelation = grundrelationMax;
  }

  if ( grundrelation < grundrelationMin ) {
    umsatz        = masse * grundrelationMin;
    grundrelation = grundrelationMin;
  }

  // Skalierungsfaktor berechnen
  // Der Skalierungsfaktor soll sicherstellen, daß Umsatzeskalationen
  // durch Streßfaktoren über das äußerste Zufuhrlimit (max. mögliche
  // Energiezufuhr in kcal/kg*d) hinaus nicht erreicht werden. Die
  // streßbedingten Steigerungen werden entgegengesetzt zur
  // Grundrelation skaliert, d.h. je niedriger die Grundrelation ist,
  // desto stärker wirken sich Streßfaktoren aus - und umgekehrt: die
  // Annahme ist also, daß große Menschen mit hoher Körpermassemasse
  // ihren Bedarf durch Streß stärker steigern als kleine Menschen mit
  // wenig Masse. Die gilt analog auch für "negative" Streßfaktoren wie
  // etwa kontrollierte Hypothermie (= Bedarfsreduktion).
  // Ich bin unsicher, inwieweit diese Skalierung tatsächlich berechtigt
  // ist, da ich (noch) nicht über entsprechende Daten oder Erfahrungen
  // verfüge. Aber sie ist ein mir sehr plausibler Weg, das Zufuhrlimit
  // definitiv einzuhalten.

  skalierungsfaktor = ( skalierungsLimit - grundrelation ) / ( grundrelation );

  stress = 1.0 + skalierungsfaktor * ( sf * tf - 1.0 );

  // Berechne Gesamtbedarf

  umsatz = umsatz * stress;

  return( umsatz );
}

// ****************************************************************************
// ENDE
// ****************************************************************************
