<?php

namespace ZeroPortal;

use ApiBase;
use ApiFormatBase;
use ApiMain;
use ApiResult;
use FauxRequest;
use JsonConfig\JCSingleton;
use TitleValue;
use ZeroBanner\ApiRawJsonPrinter;
use ZeroBanner\ZeroConfig;

/** @noinspection PhpInconsistentReturnPointsInspection */
class ApiZeroPortal extends ApiBase {
	/**
	 * Override built-in handling of format parameter.
	 * Only JSON is supported.
	 * @return ApiFormatBase
	 */
	public function getCustomPrinter() {
		$params = $this->extractRequestParams();
		if ( $params['type'] !== 'analyticsconfig' ) {
			return new ApiRawJsonPrinter( $this->getMain(), 'json', $this->getModuleName() );
		}
		return null;
	}

	public function execute() {
		$this->getMain()->setCacheMaxAge( 1 * 60 ); // seconds
		$this->getMain()->setCacheMode( 'public' );
		$result = $this->getResult();

		$params = $this->extractRequestParams();
		$type = $params['type'];
		$moduleName = $this->getModuleName();
		if ( !$this->getUser()->isAllowed( 'zero-script' ) ) {
			$this->dieUsage( "Must be authenticated with zero-script right to use this API", 'login', 401 );
		}

		switch ( $type ) {
			default:
				ApiBase::dieDebug( __METHOD__, 'Unhandled type' );
			case 'proxies':
			case 'carriers':
				$processor = function ( ZeroConfig $content, $title ) use ( $type, $result, $moduleName ) {
					foreach ( $content->getIpsets() as $name => $ipset ) {
						$result->addValue( $moduleName, $name === 'default' ? $title : $title . '|' . $name,
							$ipset, ApiResult::NO_SIZE_CHECK );
					}
				};
				break;
			case 'carriersnoips':
				$processor = function ( ZeroConfig $content, $title ) use ( $type, $result, $moduleName ) {
					$json = $content->getDataWithDefaults();
					unset( $json->ipsets );
					$result->addValue( $moduleName, $title, $json, ApiResult::NO_SIZE_CHECK );
				};
				break;
			case 'analyticsconfig':
				$processor = function ( ZeroConfig $content, $title ) use ( $type, $result, $moduleName ) {
					// @fixme: should not disable size check above
					$val = array();
					foreach ( $content->getConfigIds() as $id ) {
						$content->selectConfigById( $id );
						$val[] = array(
							'from' => wfTimestamp( TS_ISO_8601, '2000-01-01 00:00:00' ),
							'before' => null,
							'https' => $content->enableHttps(),
							'languages' => $content->whitelistedLangs() ?: true,
							'sites' => $content->sites(),
							'via' => $content->proxies(),
							'ipsets' => $content->ipsetNames(),
						);
					}
					$result->addValue( $moduleName, $title, $val, ApiResult::NO_SIZE_CHECK );
				};
				break;
		}

		$res = $this->iterateAllConfigs( $type === 'proxies', $processor );
		if ( $res ) {
			call_user_func_array( array( $this, 'dieUsage' ), $res );
		}
	}

	public function getAllowedParams() {
		return array(
			'type' => array(
				ApiBase::PARAM_DFLT => 'config',
				ApiBase::PARAM_TYPE => array(
					'carriersnoips',
					'proxies',
					'carriers',
					'analyticsconfig',
				)
			),
		);
	}

	public function getParamDescription() {
		return array(
			'type' => array(
				'What kind of Zero info is needed',
				'  carriersnoips  - core configs for all carriers, excluding IPs',
				'  proxies  - list of all ips of the proxies (pages begin with "-")',
				'  carriers - list of all ips of all carriers',
				'  analyticsconfig - data in the analytics-required format',
			),
		);
	}

	public function getDescription() {
		return 'Zero portal api';
	}

	public function getExamples() {
		return array(
			'api.php?action=zeroportal&type=carriers',
		);
	}

	/**
	 * @param bool $isProxies
	 * @param $processCfg
	 * @throws \UsageException
	 * @return false|array
	 */
	public static function iterateAllConfigs( $isProxies, $processCfg ) {
		$reqParamsOriginal = array(
			'action' => 'query',
			'generator' => 'allpages',
			'gaplimit' => 'max',
			'gapnamespace' => '480',
			'prop' => 'revisions',
			'rvprop' => 'content',
			'continue' => '',
		);
		if ( $isProxies ) {
			$reqParamsOriginal['gapprefix'] = '-';
		} else {
			// All carriers are in NNN-NN format, which means we can safely enumerate
			// beginning from 0 to avoid proxies that start with a '-'.
			$reqParamsOriginal['gapfrom'] = '0';
		}
		$reqParams = $reqParamsOriginal;

		$ctx = new \DerivativeContext( \RequestContext::getMain() );
		global $wgZeroPortalImpersonateUser;
		if ( $wgZeroPortalImpersonateUser ) {
			$ctx->setUser( \User::newFromName( $wgZeroPortalImpersonateUser ) );
		}

		$hasWarnings = false;
		while ( true ) {
			// Get needed data
			$ctx->setRequest( new FauxRequest( $reqParams ) );
			$api = new ApiMain( $ctx );
			$api->execute();
			$data = $api->getResultData();
			// @bug: errors are not returned, they are thrown, when calling api with faux request
			if ( array_key_exists( 'error', $data ) ) {
				return array( 'Error getting data', 'data', 500, $data['error'] );
			}
			if ( isset( $data['query']['pages'] ) ) {
				foreach ( (array)$data['query']['pages'] as $pageId => $page ) {
					$rawText = $page['revisions'][0]['*'];
					$title = str_replace( 'Zero:', '', $page['title'] );
					/** @var ZeroConfig $content */
					$content =
						JCSingleton::getContent( new TitleValue( NS_ZERO, str_replace( ' ', '_', $title ) ),
							$rawText );
					if ( !$content || !$content->isValid() ) {
						if ( !$hasWarnings ) {
							wfLogWarning( 'ZeroAPI: Unable to parse json of page ' . $page['title'] );
							$hasWarnings = true; // only log once per api request
						}
						continue;
					}
					if ( $isProxies ) {
						$title =
							ucfirst( strtolower( substr( $title, 1 ) ) ); // remove '-', and make it "Opera"
					}
					$res = $processCfg( $content, $title );
					if ( $res ) {
						return $res;
					}
				}
			}
			if ( !array_key_exists( 'continue', $data ) ) {
				return false;
			}
			$reqParams = array_merge( $reqParamsOriginal, $data['continue'] );
		}
	}
}
