<?php

namespace ZeroBanner;
use Language;
use UnlistedSpecialPage;
use Html;
use MobileContext;
use WebRequest;
use Xml;

/**
 * Default startup page for ZeroBanner extension that shows the list of available languages.
 *
 * @file
 * @ingroup Extensions
 */
class ZeroSpecialPage extends UnlistedSpecialPage {

	/**
	 * Constructor — set up the SpecialPage:ZeroRatedMobileAccess page
	 */
	public function __construct() {
		parent::__construct( 'ZeroRatedMobileAccess' );
	}

	/**
	 * Show the special page
	 *
	 * @param null|string $par parameter passed to the special page or null
	 */
	public function execute( $par ) {
		$out = $this->getOutput();

		// todo: vary-headers should be removed from the mobile frontend
		$this->setHeaders();
		$out->addVaryHeader( 'X-CS' );
		$out->addVaryHeader( 'X-Subdomain' );
		$out->addVaryHeader( 'X-Forwarded-By' );
		$out->addVaryHeader( 'X-Forwarded-Proto' );

		$state = PageRenderingHooks::getState( $this->getContext() );
		$zcmd = $this->getRequest()->getVal( 'zcmd' );
		if ( $zcmd && $this->echoBanner( $state, $zcmd ) ) {
			return;
		}
		$redir = $state->getRedirectInfo();
		if ( array_key_exists( 'redirect', $redir ) ) {
			$out->redirect( $redir['redirect'], $redir['code'] );
			return;
		}

		$out->setPageTitle( null );
		if ( array_key_exists( 'warn', $redir ) ) {
			$html = $this->renderWarning( $redir );
			$out->addHTML( $html );
		} else {
			if ( array_key_exists( 'softredirect', $redir ) ) {
				$r = $redir['softredirect'];
				$rSafe = preg_match( '#^(https?://|/)#i', $r );
				$r = $rSafe ? wfUrlencode( $r ) : urlencode( $r );
				$output = '<a href="' . $r . '">' . htmlentities( $r ) . '</a>';
				$out->addHTML( $output );
			} else {
				$config = $state->getZeroConfig();
				if ( $config && $state->isZeroSubdomain() ) {
					$out->disable();
					$req = $this->getRequest();
					$resp = $req->response();
					$resp->header( 'Content-Type: text/html; charset=UTF-8' );
					$this->setCacheControl( $req );

					// $out->addVaryHeader is a no-op, so let's do this manually.
					// We don't really care about other headers here, like Cookie
					$resp->header( 'Vary: Accept-Encoding,X-CS,X-Subdomain,X-Forwarded-By,X-Forwarded-Proto' );

					$output = $this->renderSimplerSpecialPage( $state );
					echo $output;
				} elseif ( $config || $state->isZeroSubdomain() ) {
					$out->addModules( 'zerobanner.special.scripts' );
					$out->addModuleStyles( 'zerobanner.special.styles' );
					if ( $config ) {
						$output = $this->renderSpecialPage( $state );
					} else {
						$output = $this->renderUnknownZeroPartnerPage( $state );
					}
					$out->addHTML( $output );
				}
			}
		}
	}

	public static function createImageBanner( $background, $foreground, $text, $isError = false ) {
		global $wgZeroBannerFontSize, $wgZeroBannerImageSize, $wgZeroBannerErrImageSize, $wgZeroBannerFont;
		// todo: Switch to "pango:" when it becomes available in a few months
		$cmd = 'convert' .
		       ' +repage' .
		       ' -trim' .
		       ' -border 3' .
		       ' -bordercolor ' . wfEscapeShellArg( $background ) .
		       ' -background ' . wfEscapeShellArg( $background ) .
		       ' -fill ' . wfEscapeShellArg( $foreground ) .
		       ( $wgZeroBannerFont ? ' -font ' . wfEscapeShellArg( $wgZeroBannerFont ) : '' ) .
		       ( $wgZeroBannerFontSize ? ( ' -pointsize ' . wfEscapeShellArg( $wgZeroBannerFontSize ) ) :
			       ( ' -size ' . wfEscapeShellArg( $isError ? $wgZeroBannerErrImageSize : $wgZeroBannerImageSize ) ) ) .
		       // These arguments must be after all other "-xxxx" ones
		       ' ' . wfEscapeShellArg( ( $isError ? 'caption:' : 'label:' ) . $text ) .
		       ' gif:-';
		$output = wfShellExecWithStderr( $cmd, $retval );

		// All gif images must begin with 'GIF87/GIF89'
		if ( $retval === 0 && strncmp( $output, 'GIF8', 4 ) === 0 ) {
			return $output;
		} else {
			wfLogWarning( "Error $retval executing $cmd: $output" );
			return false;
		}
	}

	/**
	 * @param PageRendering $state
	 * @throws \MWException
	 * @return string
	 */
	private function renderSpecialPage( PageRendering $state ) {
		$output = '';

		// CSS hack to get rid of the header that comes from the mobile frontend
		$output .= '<style type="text/css">.header {display: none}</style>';
		$output .= $this->msg( 'zero-landing-welcome' )->parseAsBlock() . Html::element( 'hr' );

		$config = $state->getZeroConfig();

		$languageNames = $this->getLangNamesWithOverrides( $config );
		$output .= $this->renderPrimaryLangs( $state, $config, $languageNames);
		$output .= Html::element( 'hr' );
		$output .= wfMessage( 'zero-home-page-selection-text' )->escaped();
		$output .= Html::openElement( 'form' );
		$currentTitle = $this->getPageTitle();
		$defaultUrl = MobileContext::singleton()->getMobileUrl( $currentTitle->getFullURL() );
		$output .= Html::element( 'input', array( 'type' => 'hidden',
			'name' => 'from',
			'value' => $currentTitle,
		) );
		// TODO: Remove unnecessary 'id' attributes
		$output .= Html::openElement( 'select',
			array( 'id' => 'languageselection',
				'onchange' => 'javascript:window.location = this.options[this.selectedIndex].value;',
				'class' => 'mw-zero-language',
				'name' => 'to',
			)
		);
		$output .= Html::element( 'option',
			array( 'value' => $defaultUrl ),
			wfMessage( 'zero-language-selection' )->text()
		);

		$freeLangs = $config->whitelistedLangs();
		foreach ( $languageNames as $languageCode => $languageName ) {
			$isFree = count( $freeLangs ) === 0 || in_array( $languageCode, $freeLangs );
			$url = $state->getStartPageUrl( $languageCode, $isFree ? 0 : PageRendering::FORCE_MDOT );
			if ( !$isFree ) {
				$url = $state->makeRedirect( $url );
			}
			$output .= Html::element( 'option', array( 'value' => $url ), $languageName );
		}
		$output .= Html::closeElement( 'select' );
		$output .= Html::element( 'input',
			array( 'type' => 'submit',
				'value' => wfMessage( 'go' )->text(),
				'class' => 'mw-zero-language',
			)
		);
		$output .= Html::closeElement( 'form' );
		return $output;
	}

	/**
	 * Renders HTML hyperlinks for Main Pages considered primary for given operator's geo.
	 * @param PageRendering $state
	 * @param ZeroConfig $config
	 * @param array $languageNames language code to language name associative array
	 * @param string $tag tag, like 'p', for each Main Page language link in method response
	 * @return string the HTML
	 *
	 */
	private function renderPrimaryLangs( $state, $config, $languageNames, $tag = 'h3' ) {
		$primaryLangs = '';
		foreach ( $config->showLangs() as $languageCode ) {
			if ( !array_key_exists( $languageCode, $languageNames ) ) {
				continue;
			}
			$languageLink = Html::element( 'a',
				array( 'href' => $state->getStartPageUrl( $languageCode ) ),
				wfMessage( 'zero-home-page-selection',
					ucfirst( $languageNames[$languageCode] ) )->inLanguage( $languageCode )
			);
			$primaryLangs .= Html::rawElement( $tag, null, $languageLink );
		}
		return $primaryLangs;
	}

	/**
	 * Assembles a speedier landing page with a simple search box
	 * @param PageRendering $state
	 * @throws \MWException
	 * @return string
	 */
	private function renderSimplerSpecialPage( $state ) {
		global $wgScript, $wgSitename;

		$brTag = Html::element( 'br' );
		$hrTag = Html::element( 'hr' );

		$config = $state->getZeroConfig();
		/** @var \SkinMinerva $skin - Assuming this is a mobile skin */
		$skin = $this->getContext()->getSkin();
		$dir = $skin->getLanguage()->isRTL() ? 'rtl' : 'ltr';
		$lang = $skin->getLanguage()->getCode();
		$output = Html::htmlHeader( array( 'lang' => $lang, 'dir' => $dir ) );
		$output .= Html::openElement( 'head' );
		$output .= Html::element( 'meta', array( 'charset' =>'UTF-8' ) );
		$output .= Html::element( 'meta', array( 'name' => 'viewport',
			'content' =>'width=device-width' ) );
		$output .= Html::element( 'title', null, $wgSitename );
		$output .= Html::element( 'style', null, 'body{text-align:center} #center{width:100%;margin:0 auto} hr{width:80%;height:1px} p,small {font-family:Tahoma,sans-serif}' );
		$output .= Html::element( 'link', array( 'rel' => 'shortcut icon', 'href' => '#' ) );
		$output .= Html::closeElement( 'head' );
		$output .= Html::openElement( 'body', array( 'onLoad' => 'document.searchform.search.focus();' ) );
		$output .= Html::element( 'p', null, $state->getBannerText( $config ) );
		$output .= Html::openElement( 'div', array( 'id' => 'center' ) );
		$output .= Html::openElement( 'form', array( 'name' => 'searchform',
		                                             'action' => $wgScript ) );
		$output .= Html::element( 'input', array( 'type' => 'search',
			'name' => 'search',
			'size' => '21',
			'value' => '',
			'accesskey' => 'f',
			'autocomplete' => 'off',
			'autofocus' => 'autofocus', // dummy value - it's HTML5
		) );
		$output .= str_repeat( $brTag, 2 );
		$output .= Html::element( 'input',
			array( 'type' => 'submit',
			       'name' => 'fulltext',
			       'value' => wfMessage( 'zero-search' )->text(),
			)
		);
		$output .= Html::closeElement( 'form' );
		$output .= $hrTag;
		$languageNames = $this->getLangNamesWithOverrides( $config );
		$output .= $this->renderPrimaryLangs( $state, $config, $languageNames, 'p' );
		$output .= $hrTag;
		$output .= Html::openElement( 'small' );

		// START Adapted from SkinMinerva.php
		// Generate the licensing text displayed in the footer of each page
		$link = \SkinMinerva::getLicenseLink( 'footer' );
		// The license message is displayed in the content language rather than the user
		// language. See Skin::getCopyright.
		if ( $link ) {
			$licenseText = $skin->msg( 'mobile-frontend-copyright' )->rawParams(
				$link )->inContentLanguage()->text();
		} else {
			$licenseText = '';
		}
		// END Adapted from SkinMinerva.php
		if ( $licenseText ) {
			$output .= $state->createWarningLink( $licenseText );
			$output .= str_repeat( $brTag, 2 );
		}

		$termsLink = $skin->getTermsLink();
		if ( $termsLink ) {
			$output .= $state->createWarningLink( $termsLink ) . ' | ';
		}

		$privacyLink = $skin->footerLink( 'mobile-frontend-privacy-link-text', 'privacypage' );
		if ( $privacyLink ) {
			$output .= $state->createWarningLink( $privacyLink );
		}

		$output .= Html::closeElement( 'small' );

		$output .= Html::closeElement( 'div' );
		$output .= Html::closeElement( 'body' );
		$output .= Html::closeElement( 'html' );
		return $output;
	}

	/**
	 * Create HTML warning of navigation to a paid resource
	 * @param array $redir that describes how redirect should be rendered
	 * @return string HTML
	 */
	private function renderWarning( $redir ) {
		// Show warning text (navigating away from the free site)
		$reject = Html::rawElement(
				'a',
				array(
					'class' => 'cancel inline zerobutton',
					'role' => 'button',
					'href' => $redir['from'] ),
				$this->msg( 'zero-go-back' )->parse() ) . '';

		$accept = Html::rawElement(
				'a',
				array(
					'class' => 'button zerobutton',
					'role' => 'button',
					'href' => $redir['to'],
					'tabindex' => '0' ),
				$this->msg( 'zero-accept' )->parse() );

		$lang = $this->getLanguage();
		$divOpen = '<div class="button-bar button-bar-centered buttonBar zerobar">';
		$divClose = '</div>';
		if ( $lang->isRTL() ) {
			$buttons = $divOpen . $accept . ' ' . $reject . $divClose;
		} else {
			$buttons = $divOpen . $reject . ' ' . $accept . $divClose;
		}
		$dir = $lang->getDir();
		$msg = $this->msg( $redir['warn'] === 'file'
			? 'zero-file-auth'
			: 'zero-charge-auth' )->parse();
		$html = <<<END
<div dir='$dir' class='header mw-mf-banner mw-mf-banner-undismissable'>$msg</div>
<div dir='$dir' class="zero-init-mw-viewPageTarget-toolbar-actions">$buttons</div>
END;
		return $html;
	}

	/**
	 * Render the warning text for the unknown IP when accessing zero.*
	 * @param PageRendering $state
	 * @throws \MWException
	 * @return string
	 */
	private function renderUnknownZeroPartnerPage( PageRendering $state ) {
		// This page contains client IP and should not be cached
		$this->getOutput()->enableClientCache( false );
		$req = $this->getRequest();

		$host = implode( '.', $state->getWikiInfo() );
		$bannerText = $this->msg( 'zero-sorry', $host )->text() . '<br />';

		$ip = $req->getIP();
		$bannerText .= $this->msg( 'zero-sorry-ip', $ip )->text() . '<br />';

		$url = false;
		$from = $req->getVal( 'from' );
		if ( $from ) {
			$urlParts = wfParseUrl( $from );
			if ( $urlParts && isset( $urlParts['host'] ) ) {
				// Check that this domain name is handled by local servers (disabled in testing)
				// Last two or three parts of the domain name must be found in the local vhost
				$hp = explode( '.', $urlParts['host'] );
				global $wgZeroSiteOverride, $wgConf;
				if ( $wgZeroSiteOverride ||
					$wgConf->isLocalVHost( implode( '.', array_slice( $hp, -2, 2 ) ) ) ||
					$wgConf->isLocalVHost( implode( '.', array_slice( $hp, -3, 3 ) ) )
				) {
					if ( $hp[1] === 'zero' ) {
						$hp[1] = 'm';
					} elseif ( $hp[0] === 'zero' ) {
						$hp[0] = 'm';
					}
					$urlParts['host'] = implode( '.', $hp );
					$url = wfUrlencode( wfAssembleUrl( $urlParts ) );
				}
			}
		}
		if ( !$url ) {
			$url = $state->getStartPageUrl( null, PageRendering::FORCE_MDOT );
		}
		$link = Html::element( 'a', array( 'href' => $url ), wfExpandUrl( $url ) );
		$bannerText .= $this->msg( 'zero-sorry-goto', $link )->text();

		return Html::rawElement( 'div',
			array( 'class' => 'zrma-unknown' ),
			$bannerText );
	}

	/**
	 * Render raw HTML snippet or GIF image content for unified design JS inclusion
	 * @param PageRendering $state
	 * @param string $zcmd
	 * @return bool
	 */
	private function echoBanner( PageRendering $state, $zcmd ) {
		if ( $zcmd !== 'js-banner' && $zcmd !== 'img-banner' ) {
			return false;
		}
		$isJsBanner = $zcmd === 'js-banner';
		$out = $this->getOutput();
		$out->disable();
//		@fixme: needed?
//		ob_clean();
//		flush();

		$request = $this->getRequest();
		$response = $request->response();

		$out->addVaryHeader( 'X-CS' ); // Carrier ID
		$out->addVaryHeader( 'X-Subdomain' ); // ZERO vs M
		$out->addVaryHeader( 'X-Forwarded-By' ); // Proxy
		$out->addVaryHeader( 'X-Forwarded-Proto' ); // HTTPS
		$response->header( $out->getVaryHeader() );

		$id = $state->getConfigId();
		$config = $state->getZeroConfig();
		$isFilePage = $request->getCheck( 'file' );
		$errors = false;

		if ( $isJsBanner ) {
			//
			// JavaScript banner together with configuration (or a local redirect)
			//
			$response->header( 'Content-type: application/javascript; charset=UTF-8' );
			$banner = '';
			if ( $state->isZeroSubdomain() && !$state->getZeroConfig() ) {
				// If zerodot isn't supported here and the user isn't already on
				// Special:ZeroRatedMobileAccess, send the user to Special:ZeroRatedMobileAccess
				// and include the original URL so that the user may continue to the m. server.
				$info = $state->getWikiInfo();
				$flags = PageRendering::GET_LANDING | PageRendering::FORCE_HTTP;
				$url = $state->getStartPageUrl( $info[0], $flags );
				$banner = "window.location='$url?from='+encodeURIComponent(window.location)";
			} else {
				if ( $config && $config->enabled() ) {
					$bannerHtml = PageRendering::renderBanner( $state, $config, null, null, $isFilePage );
					$cfg = PageRendering::getJsConfigBlock( $id, $config, (bool)$bannerHtml );
					if ( $bannerHtml ) {
						$banner = 'document.write(' . Xml::encodeJsVar( $bannerHtml ) . ')';
					}
				} else {
					$cfg = PageRendering::getJsConfigBlock( $id, $config, false );
				}
				$banner = $cfg . $banner;
			}
		} else {
			//
			// Returning content of a GIF image
			//
			$response->header( 'Content-type: image/gif' );
			$lang = $this->getRequest()->getVal( 'zlang' );
			$text = PageRendering::getBannerText( $config, $isFilePage, $lang );
			$banner = false;
			if ( $text ) {
				$banner = self::createImageBanner( $config->background(), $config->foreground(), $text );
				$errors = !$banner;
			} elseif ( $state->isZeroSubdomain() ) {
				$host = implode( '.', $state->getWikiInfo() );
				$bannerText = $this->msg( 'zero-sorry', $host )->text() . "\n\n";

				$ip = $request->getIP();
				$bannerText .= $this->msg( 'zero-sorry-ip', $ip )->text();

				$banner = self::createImageBanner( 'red', 'white', $bannerText, true );
				$errors = !$banner;
			}
			if ( !$banner ) {
				// This tiny file resides in the same dir
				// We pretend like there is no banner with a 1x1 pixel img
				$banner = file_get_contents( __DIR__ . DIRECTORY_SEPARATOR . 'blank.gif' );
			}
		}

		if ( $errors ) {
			$this->setCacheControl( $request, 5, 15 ); // shorter error caching
		} else {
			$this->setCacheControl( $request );
		}

		echo $banner;
		return true;
	}

	/**
	 * @param WebRequest $request
	 * @param int $min
	 * @param int $max
	 */
	private function setCacheControl( WebRequest $request, $min = 5, $max = 900 ) {
		// Both maxage and smaxage has to be between $min and $max
		$smaxage = min( $max, max( $min, abs( $request->getInt( 'smaxage', $max ) ) ) );
		$maxage = min( $max, max( $min, abs( $request->getInt( 'maxage', $max ) ) ) );
		$request->response()->header( 'Cache-Control: public, s-maxage=' . $smaxage . ', max-age=' . $maxage );
	}

	/**
	 * Given an operator configuration, returns an array of language code to language name, with overrides like kg => Kikongo
	 * @param ZeroConfig $config
	 * @return array langcode => langname
	 */
	private function getLangNamesWithOverrides( $config ) {
		$languageNames = Language::fetchLanguageNames();
		$overrides = $config->langNameOverrides();
		// Allow language name overrides, as requested by Orange Congo ('kg' => 'Kikongo')
		// Do it one at a time to keep the original order of the array
		foreach ( $overrides as $l => $n ) {
			$languageNames[$l] = $n;
		}
		return $languageNames;
	}
}
