<?php

namespace Wikibase\Test;

use ContentHandler;
use Wikibase\DataModel\Entity\EntityId;
use Wikibase\DataModel\Entity\Item;
use Wikibase\DataModel\Entity\ItemId;
use Wikibase\DataModel\Entity\Property;
use Wikibase\DataModel\Entity\PropertyId;
use Wikibase\EntityRevision;
use Wikibase\Lib\Store\EntityRedirect;
use Wikibase\Lib\Store\EntityRevisionLookup;
use Wikibase\Lib\Store\UnresolvedRedirectException;

/**
 * Base class for testing EntityRevisionLookup implementations
 *
 * @group Wikibase
 *
 * @licence GNU GPL v2+
 * @author Daniel Kinzler
 */
abstract class EntityRevisionLookupTest extends \PHPUnit_Framework_TestCase {

	/**
	 * @return EntityRevision[]
	 */
	protected function getTestRevisions() {
		$entities = array();

		$item = Item::newEmpty();
		$item->setId( 42 );

		$entities[11] = new EntityRevision( $item, 11, '20130101001100' );

		$item = $item->copy();
		$item->setLabel( 'en', "Foo" );

		$entities[12] = new EntityRevision( $item, 12, '20130101001200' );

		$prop = Property::newFromType( "string" );
		$prop->setId( 753 );

		$entities[13] = new EntityRevision( $prop, 13, '20130101001300' );

		return $entities;
	}

	/**
	 * @return EntityRedirect[]
	 */
	protected function getTestRedirects() {
		$redirects = array();

		$redirects[] = new EntityRedirect( new ItemId( 'Q23' ), new ItemId( 'Q42' ) );

		return $redirects;
	}

	protected function resolveLogicalRevision( $revision ) {
		return $revision;
	}

	/**
	 * @return EntityRevisionLookup
	 */
	protected function getEntityRevisionLookup() {
		$revisions = $this->getTestRevisions();
		$redirects = $this->getTestRedirects();

		$lookup = $this->newEntityRevisionLookup( $revisions, $redirects );

		return $lookup;
	}

	/**
	 * @param EntityRevision[] $entityRevisions
	 * @param EntityRedirect[] $entityRedirects
	 *
	 * @return EntityRevisionLookup
	 */
	protected abstract function newEntityRevisionLookup( array $entityRevisions, array $entityRedirects );

	protected function itemSupportsRedirect() {
		if ( !defined( 'CONTENT_MODEL_WIKIBASE_ITEM' ) ) {
			// We currently cannot determine whether redirects are supported if
			// no repo code is available. Just skip the corresponding tests in that case.
			return false;
		}

		$handler = ContentHandler::getForModelID( CONTENT_MODEL_WIKIBASE_ITEM );
		return $handler->supportsRedirects();
	}

	public static function provideGetEntityRevision() {
		$cases = array(
			array( // #0: any revision
				new ItemId( 'q42' ), 0, true,
			),
			array( // #1: first revision
				new ItemId( 'q42' ), 11, true,
			),
			array( // #2: second revision
				new ItemId( 'q42' ), 12, true,
			),
			array( // #3: bad revision
				new ItemId( 'q42' ), 600000, false, 'Wikibase\Lib\Store\StorageException',
			),
			array( // #4: wrong type
				new ItemId( 'q753' ), 0, false,
			),
			array( // #5: mismatching revision
				new PropertyId( 'p753' ), 11, false, 'Wikibase\Lib\Store\StorageException',
			),
			array( // #6: some revision
				new PropertyId( 'p753' ), 0, true,
			),
		);

		return $cases;
	}

	/**
	 * @dataProvider provideGetEntityRevision
	 *
	 * @param EntityId $id    The entity to get
	 * @param int             $revision The revision to get (or 0)
	 * @param bool            $shouldExist
	 * @param string|null     $expectException
	 */
	public function testGetEntityRevision( $id, $revision, $shouldExist, $expectException = null ) {
		if ( $expectException !== null ) {
			$this->setExpectedException( $expectException );
		}

		$revision = $this->resolveLogicalRevision( $revision );

		$lookup = $this->getEntityRevisionLookup();
		$entityRev = $lookup->getEntityRevision( $id, $revision );

		if ( $shouldExist == true ) {
			$this->assertNotNull( $entityRev, "ID " . $id->__toString() );
			$this->assertEquals( $id->__toString(), $entityRev->getEntity()->getId()->__toString() );
		} else {
			$this->assertNull( $entityRev, "ID " . $id->__toString() );
		}
	}

	public function provideGetEntityRevision_redirect() {
		$redirects = $this->getTestRedirects();
		$cases = array();

		foreach ( $redirects as $redirect ) {
			$cases[] = array( $redirect->getEntityId(), $redirect->getTargetId() );
		}

		return $cases;
	}

	/**
	 * @dataProvider provideGetEntityRevision_redirect
	 */
	public function testGetEntityRevision_redirect( EntityId $entityId, EntityId $expectedRedirect ) {
		if ( !$this->itemSupportsRedirect() ) {
			$this->markTestSkipped( 'redirects not supported' );
		}

		$lookup = $this->getEntityRevisionLookup();

		try {
			$lookup->getEntityRevision( $entityId );
			$this->fail( 'Expected an UnresolvedRedirectException exception when looking up a redirect.' );
		} catch ( UnresolvedRedirectException $ex ) {
			$this->assertEquals( $expectedRedirect, $ex->getRedirectTargetId() );
		}
	}

	public function provideGetLatestRevisionId() {
		$cases = array(
			array( // #0
				new ItemId( 'q42' ), 12,
			),
			array( // #1
				new PropertyId( 'p753' ), 13,
			),
		);

		return $cases;
	}

	/**
	 * @dataProvider provideGetLatestRevisionId
	 *
	 * @param EntityId $id The entity to check
	 * @param int $expected
	 */
	public function testGetLatestRevisionId( EntityId $id, $expected ) {
		$lookup = $this->getEntityRevisionLookup();
		$result = $lookup->getLatestRevisionId( $id );

		$expected = $this->resolveLogicalRevision( $expected );

		$this->assertInternalType( 'int', $result );
		$this->assertEquals( $expected, $result );

		$entityRev = $lookup->getEntityRevision( $id );
		$this->assertInstanceOf( 'Wikibase\EntityRevision', $entityRev );
	}

	public function testGetLatestRevisionForMissing() {
		$lookup = $this->getEntityRevisionLookup();
		$itemId = new ItemId( 'Q753' );

		$result = $lookup->getLatestRevisionId( $itemId );
		$expected = $this->resolveLogicalRevision( false );

		$this->assertEquals( $expected, $result );

		$entityRev = $lookup->getEntityRevision( $itemId );
		$this->assertNull( $entityRev );
	}

}

