/**
 * @licence GNU GPL v2+
 * @author H. Snater < mediawiki@snater.com >
 */
( function( mw, wb, $, util ) {
	'use strict';

	var PARENT = $.TemplatedWidget;

	/**
	 * The node of the rank selector menu to select a rank from.
	 * @type {jQuery}
	 */
	var $menu = null;

	/**
	 * Selector for choosing a statement rank.
	 * @since 0.5
	 *
	 * @option [rank] {boolean} The rank that shall be selected.
	 *         Default: wb.datamodel.Statement.RANK.NORMAL
	 *
	 * @option [isRTL] {boolean} Defines whether the widget is displayed in right-to-left context.
	 *         If not specified, the context is detected by checking whether the 'rtl' css class is
	 *         set on the HTML body element.
	 *         Default: undefined
	 *
	 * @event afterchange Triggered after the snak type got changed
	 *        (1) {jQuery.Event}
	 */
	$.wikibase.statementview.RankSelector = util.inherit( PARENT, {
		widgetName: 'wikibase-rankselector',
		widgetBaseClass: 'wb-rankselector',

		/**
		 * @type {Object}
		 */
		options: {
			template: 'wb-rankselector',
			templateParams: [
				'',
				'',
				''
			],

			templateShortCuts: {
				'$icon': '.ui-icon-rankselector'
			},

			rank: wb.datamodel.Statement.RANK.NORMAL,
			isRtl: undefined
		},

		/**
		 * The rank currently featured by the rank selector.
		 * @see wb.datamodel.Statement.RANK
		 * @type {number}
		 */
		_rank: null,

		/**
		 * @see jQuery.Widget._create
		 */
		_create: function() {
			var self = this;

			PARENT.prototype._create.call( this );

			if( !$menu ) {
				$menu = this._buildMenu().appendTo( 'body' ).hide();

				$menu.on( 'click.' + this.widgetName, function( event ) {
					var $li = $( event.target ).closest( 'li' ),
						rank = $li.data( self.widgetName + '-menuitem-rank' );

					if( rank !== undefined ) {
						$.data( this, self.widgetName ).rank( rank );
					}
				} );
			}

			this.element
			.addClass( this.widgetBaseClass )
			.on( 'mouseover.' + this.widgetName, function( event ) {
				if( !self.isDisabled() ) {
					self.element.addClass( 'ui-state-hover' );
				}
			} )
			.on( 'mouseout.' + this.widgetName, function( event ) {
				if( !self.isDisabled() ) {
					self.element.removeClass( 'ui-state-hover' );
				}
			} )
			.on( 'click.' + this.widgetName, function( event ) {
				if( self.isDisabled() || $menu.is( ':visible' ) ) {
					$menu.hide();
					return;
				}

				$menu.data( self.widgetName, self );
				$menu.show();
				self._updateMenuCss();
				self.repositionMenu();

				self.element.addClass( 'ui-state-active' );

				// Close the menu when clicking, regardless of whether the click is performed on the
				// menu itself or outside of it:
				var degrade = function( event ) {
					if ( event.target !== self.element.get( 0 ) ) {
						$menu.hide();
						self.element.removeClass( 'ui-state-active' );
					}
					self._unbindGlobalEventListeners();
				};

				$( document ).on( 'mouseup.' + self.widgetName, degrade  );
				$( window ).on( 'resize.' + self.widgetName, degrade );
			} );

			this._setRank( this.options.rank );
		},

		/**
		 * @see jQuery.Widget.destroy
		 */
		destroy: function() {
			if( $( '.' + this.widgetBaseClass ).length === 0 ) {
				$menu.data( 'menu' ).destroy();
				$menu.remove();
				$menu = null;
			}
			this.$icon.remove();

			this.element.removeClass( 'ui-state-default ui-state-hover ' + this.widgetBaseClass );

			this._unbindGlobalEventListeners();

			PARENT.prototype.destroy.call( this );
		},

		/**
		 * @see jQuery.Widget._setOption
		 * @triggers afterchange
		 */
		_setOption: function( key, value ) {
			PARENT.prototype._setOption.apply( this, arguments );
			if( key === 'rank' ) {
				this._setRank( value );
				this._trigger( 'afterchange' );
			}
		},

		/**
		 * Removes all global event listeners generated by the rank selector.
		 */
		_unbindGlobalEventListeners: function() {
			$( document ).add( $( window ) ).off( '.' + this.widgetName );
		},

		/**
		 * Generates the menu the rank may be chosen from.
		 *
		 * @return {jQuery}
		 */
		_buildMenu: function() {
			var self = this,
				$menu = $( '<ul/>' ).addClass( this.widgetBaseClass + '-menu' );

			$.each( wb.datamodel.Statement.RANK, function( rankId, i ) {
				rankId = rankId.toLowerCase();

				$menu.append(
					$( '<li/>' )
					.addClass( self.widgetBaseClass + '-menuitem-' + rankId )
					.data( self.widgetName + '-menuitem-rank', i )
					.append(
						$( '<a/>' )
						.text( mw.msg( 'wikibase-statementview-rank-' + rankId ) )
						.on( 'click.' + self.widgetName, function( event ) {
							event.preventDefault();
						} )
					)
				);
			} );

			return $menu.menu();
		},

		/**
		 * Sets the rank if a rank is specified or gets the current rank if parameter is omitted.
		 * @since 0.5
		 *
		 * @param {number} [rank]
		 * @return {number|undefined}
		 *
		 * @triggers afterchange
		 */
		rank: function( rank ) {
			if( rank === undefined ) {
				return this._rank;
			}

			this._setRank( rank );

			this._trigger( 'afterchange' );
		},

		/**
		 * Sets the rank activating the menu item representing the specified rank.
		 *
		 * @param {number} rank
		 */
		_setRank: function( rank ) {
			this._rank = rank;

			if( $menu && $menu.data( this.widgetName ) === this ) {
				this._updateMenuCss();
			}

			this._updateIcon();
		},

		/**
		 * Updates the menu's css classes.
		 */
		_updateMenuCss: function() {
			$menu.children().removeClass( 'ui-state-active' );
			$menu
			.children( '.' + this.widgetBaseClass + '-menuitem-' + getRankString( this.rank() ) )
			.addClass( 'ui-state-active' );
		},

		/**
		 * Updates the rank icon to reflect the rank currently set.
		 */
		_updateIcon: function() {
			var self = this,
				rankString = getRankString( this.rank() );

			$.each( wb.datamodel.Statement.RANK, function( rankId, i ) {
				self.$icon.removeClass( 'wb-rankselector-' + rankId.toLowerCase() );
			} );

			this.$icon
			.addClass( 'wb-rankselector-' + rankString )
			.attr( 'title', mw.msg( 'wikibase-statementview-rank-' + rankString ) );
		},

		/**
		 * Positions the menu.
		 * @since 0.5
		 */
		repositionMenu: function() {
			var isRtl = ( this.options.isRTL )
				? this.options.isRTL
				: $( 'body' ).hasClass( 'rtl' );

			$menu.position( {
				of: this.$icon,
				my: ( isRtl ? 'right' : 'left' ) + ' top',
				at: ( isRtl ? 'right' : 'left' ) + ' bottom',
				offset: '0 1',
				collision: 'none'
			} );
		},

		/**
		 * @see jQuery.Widget.disable
		 * @since 0.5
		 */
		disable: function() {
			if( $menu && $menu.data( this.widgetName ) === this ) {
				// Disabling the rank selector the menu currently references to.
				$menu.hide();
			}
			this.element.removeClass( 'ui-state-active ui-state-hover' );
			this.element.addClass( 'ui-state-disabled' );
			return PARENT.prototype.disable.call( this );
		},

		/**
		 * @see jQuery.Widget.enable
		 * @since 0.5
		 */
		enable: function() {
			this.element.removeClass( 'ui-state-disabled' );
			return PARENT.prototype.enable.call( this );
		},

		/**
		 * Returns whether the widget is currently disabled.
		 * @return 0.5
		 */
		isDisabled: function() {
			return this.element.hasClass( 'ui-state-disabled' );
		}

	} );

	/**
	 * Returns a rank's serialized string.
	 * @see wb.datamodel.Statement.RANK
	 *
	 * @param {number} rank
	 * @return {string|null}
	 */
	function getRankString( rank ) {
		var rankString = null;

		$.each( wb.datamodel.Statement.RANK, function( rankId, i ) {
			if( rank === i ) {
				rankString = rankId.toLowerCase();
				return false;
			}
		} );

		return rankString;
	}

}( mediaWiki, wikibase, jQuery, util ) );
