import re
import sys
import netaddr
import utils

def write(dst, line):
    dst.write(bytes(line, 'UTF-8'))

def toVclStr(ip, comment):
    s = str(ip)
    s = s.replace('/32','') if ':' not in s else s.replace('/128','')
    s = s.replace('/','"/')
    if '"' not in s: s = s + '"'
    return '\t"%s;%s\n' % (s,comment)

def writeHdr(dst, id, conf, comment = ''):
    comment = comment.strip(' \t/')
    if conf is not None:
        comment = conf['comment'] if 'comment' in conf else comment
    comment = comment.strip()
    if comment != '':
        comment = ' // ' + comment
    write(dst, 'acl carrier_%s {%s\n' % (id.replace('-','_'), comment))

def printConflicts(confs, extra = None):
    sets = dict([(xcs, IPSet(ips['ips'])) for xcs, ips in confs.items()])
    if extra is not None:
        sets['extra'] = netaddr.IPSet(extra)
    for xcs, ips in sets.items():
        for xcs2, ips2 in sets.items():
            if xcs == xcs2:
                continue
            conflcts = ips & ips2
            if conflcts.size == 0:
                continue
            print('*****\n***** {0} conflicts with {1}:'.format(xcs,xcs2))
            for ip in ips:
                if ip in conflcts:
                    print('{0}: {1}'.format(xcs, ip))
            for ip in ips2:
                if ip in conflcts:
                    print('{0}: {1}'.format(xcs2, ip))


if __name__ == '__main__':

    if len(sys.argv) != 2:
        raise Exception('Expects VCL file as the first argument')

    confs = utils.getAllConfigs()

    with open(sys.argv[1], 'r') as src, open(sys.argv[1]+'.out', 'wb') as dst:
        pending = []
        acl = None
        for line in src:
            if acl is None:
                r = re.match(r'^acl ([_a-z0-9]+) \{(.*)\n$', line, re.I)
                if r and r.group(1) != 'carrier_testing':
                    acl = (r.group(1), '' if r.group(2) is None else r.group(2))
                    continue
                elif line.startswith('acl '):
                    print('Skipping ' + line.strip())
                write(dst, line)
            else:
                r = re.match(r'^\t"([0-9a-f.:]+)"(/[0-9]+)?;(.*)\n$', line)
                if r:
                    bk = '' if r.group(2) is None else r.group(2)
                    pending.append((netaddr.IPNetwork(r.group(1) + bk), r.group(3)))
                    continue
                if line == '}\n':
                    id = acl[0].replace('carrier_','').replace('_','-')
                    print('Optimizing ' + id)
#                    if id == '646-02':
#                        pending.append((netaddr.IPNetwork('41.63.128.0'), ''))
#                        pending.append((netaddr.IPNetwork('41.74.220.0'), ''))
#                        pending.append((netaddr.IPNetwork('41.63.143.255'), ''))
                    comments = dict([(v[0], v[1]) for v in pending if v[1]])
                    vals = netaddr.cidr_merge([v[0] for v in pending])
                    if id in confs:
                        conf = confs[id]
                        del confs[id]
                        vals2 = netaddr.cidr_merge([netaddr.IPNetwork(v) for v in conf['ips']])
                        if netaddr.IPSet(vals) != netaddr.IPSet(vals2):
                            print('IP range has changed for %s' % id)
                            for ip in netaddr.cidr_merge(netaddr.IPSet(vals) ^ netaddr.IPSet(vals2)):
                                print(ip)
                        writeHdr(dst, id, conf, acl[1])
                    else:
                        print(acl[0] + ' not found in configuration')
                        writeHdr(dst, id, None, acl[1])
                    for ip in vals:
                        if ip in comments:
                            c = comments[ip]
                            del comments[ip]
                        else:
                            c = ''
                        write(dst, toVclStr(ip, c))
                    for c in comments:
                        print('Deleted comment "%s" for ip %s' % (comments[c], str(c)))
                else:
                    id = acl[0].replace('carrier_','').replace('_','-')
                    print('Cannot parse %s (%s): %s' % (acl[0], id, line))
                    writeHdr(dst, id, conf, acl[1])
                    for v in pending:
                        write(dst, toVclStr(v[0], v[1]))
                write(dst, line)
                acl = None
                pending = []
        for id in confs:
            if id == 'WMF':
                continue
            print('ID %s was not found' % id)
            conf = confs[id]
            vals = netaddr.cidr_merge([netaddr.IPNetwork(v) for v in conf['ips']])
            writeHdr(dst, id, conf)
            for v in vals:
                write(dst, toVclStr(v, ''))
            write(dst, '}\n\n')
