<?php

namespace ZeroBanner;

use ApiBase;
use ApiFormatBase;

/**
 * Utility class to avoid any warnings or any other additions by the API framework
 * Class ApiRawJsonPrinter
 * @package ZeroBanner
 */
class ApiRawJsonPrinter extends \ApiFormatJson {
	private $moduleName;

	public function __construct( $main, $format, $moduleName ) {
		parent::__construct( $main, $format );
		$this->moduleName = $moduleName;
	}

	public function execute() {
		// Make sure we only output the data we
		$res = $this->getResult();
		$data = $res->getData();
		$res->reset();
		// @todo: add ApiResult::NO_SIZE_CHECK when core update is live
		if ( array_key_exists( $this->moduleName, $data ) ) {
			foreach ( $data[$this->moduleName] as $k => $v ) {
				$res->addValue( null, $k, $v );
			}
		}
		parent::execute();
	}
}

class ApiZeroBanner extends ApiBase {
	/**
	 * Override built-in handling of format parameter.
	 * Only JSON is supported.
	 * @return ApiFormatBase
	 */
	public function getCustomPrinter() {
		return new ApiRawJsonPrinter( $this->getMain(), 'json', $this->getModuleName() );
	}

	public function execute() {
		$this->getMain()->setCacheMaxAge( 1 * 60 ); // seconds
		$this->getMain()->setCacheMode( 'public' );
		$result = $this->getResult();

		$params = $this->extractRequestParams();
		$type = $params['type'];
		$moduleName = $this->getModuleName();

		switch ( $type ) {
			case 'config':
				$state = PageRenderingHooks::getState( $this );
				$config = $state->getZeroConfig();
				if ( $config ) {
					$result->addValue( $moduleName, 'enabled', true );
					$result->addValue( $moduleName, 'id', $state->getConfigId() );
					$result->addValue( $moduleName, 'showImages', $config->showImages() );
					$result->addValue( $moduleName, 'whitelistedLangs', $config->whitelistedLangs() );
					$result->addValue( $moduleName, 'enableHttps', $config->enableHttps() );
					$result->addValue( $moduleName, 'sites', $config->sites() );
					$result->addValue( $moduleName, 'domains', $config->domains() );
				} else {
					$result->addValue( $moduleName, 'enabled', false );
				}
				$out = $result->getOutput();
				$out->addVaryHeader( 'X-CS' );
				$out->addVaryHeader( 'X-Subdomain' );
				$out->addVaryHeader( 'X-Forwarded-By' );
				$out->addVaryHeader( 'X-Forwarded-Proto' );
				break;

			case 'message':
				if ( ! ( isset( $params['agent'] ) && strlen( $params['agent'] ) ) ) {
					$this->dieUsage( 'Non-empty agent parameter is required when type param is message.',
						'missing_valid_required_parameter', 500 );
				}
				$state = PageRenderingHooks::getState( $this );
				$config = $state->getZeroConfig();
				if ( $config && !$config->disableApps() && $config->showImages() ) {
					// @fixme: we should pass in the $isFilePage
					$text = $state->getBannerText( $config );
					if ( $text ) {
						$result->addValue( $moduleName, 'message', $text );
					}
				}
				// else json_encode will ultimately return as an empty array
				$out = $result->getOutput();
				$out->addVaryHeader( 'X-CS' );
				$out->addVaryHeader( 'X-Subdomain' );
				$out->addVaryHeader( 'X-Forwarded-By' );
				$out->addVaryHeader( 'X-Forwarded-Proto' );
				break;
		}
	}

	public function getAllowedParams() {
		return array(
			'type' => array(
				ApiBase::PARAM_DFLT => 'config',
				ApiBase::PARAM_TYPE => array(
					'config',
					'message',
				)
			),
			'agent' => array(
				ApiBase::PARAM_DFLT => '',
			),
		);
	}

	public function getParamDescription() {
		return array(
			'type' => array(
				'What kind of Zero info is needed',
				'  config   - get a trimmed-down carrier configuration based on the X-CS header',
				'  message  - get lang-appropriate verbiage based on X-CS header; requires agent param',
			),
			'agent' => array(
				'When setting type to message, include a source agent value as well',
			),
		);
	}

	public function getDescription() {
		return 'Get configuration of the Zero extension';
	}

	public function getExamples() {
		return array(
			'api.php?action=zeroconfig&type=config',
			'api.php?action=zeroconfig&type=message&agent=neat-app',
		);
	}
}
