// Posix_error.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/support/error/Posix_error.hpp>
#include <tovero/support/common.hpp>
#include <string>
#include <cstring>
#include <cstdio> // Standards Rule 22 deviation (check_code_ignore)

using std::string;
using namespace Roan_trail::Tovero_support;

//
// Constructor/destructor/copy
//

Posix_error::Posix_error(int posix_code,
                         const char* posix_function_name,
                         const char* file_path,
                         const Error* base_error)
  : Error(0,
          posix_function_name,
          0,
          posix_code,
          base_error)
{
  const int buffer_size = 1024;
  char error_string_buffer[buffer_size];
  const char* buffer_used;
  const int posix_error_code = posix_code;

// assign error description string

#if ((_POSIX_C_SOURCE >= 200112L || _XOPEN_SOURCE >= 600) && ! _GNU_SOURCE)
  // XSI version of strerror_r
  const int ret = strerror_r(posix_error_code,
                             error_string_buffer,
                             buffer_size);
  if (Posix_error::error_return == ret)
  {
    snprintf(error_string_buffer,
             buffer_size,
             "Posix error %d",
             posix_error_code);
  }
  buffer_used = error_string_buffer;
#else
  // GNU version of strerror_r
  buffer_used = strerror_r(posix_error_code,
                           error_string_buffer,
                           buffer_size);
#endif
  Error_dictionary &dict = error_dictionary();
  if (buffer_used)
  {
    dict[Error::diagnostic_error_key] = string(buffer_used);
  }

  if (file_path)
  {
    dict[Error::file_path_error_key] = string(file_path);
  }

  postcondition(mf_invariant(false));
}

//
// Class constants
//

const int Posix_error::error_return = -1;

//
// Protected member functions
//

bool Posix_error::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!file().empty() || line())
  {
    goto exit_point;
  }
  else
  {
    return_value = (!check_base_class || Error::mf_invariant(check_base_class));
  }

 exit_point:

  return return_value;
}
