// Reference_counting_map.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Class for maps of reference counted classes.

#ifndef TOVERO_SUPPORT_REFERENCE_COUNTING_MAP_HPP_
#define TOVERO_SUPPORT_REFERENCE_COUNTING_MAP_HPP_

#include <algorithm>
#include <map>

namespace Roan_trail
{
  namespace Tovero_support
  {
    template <class Key_type, class Mapped_type> class Reference_counting_map
    {
      // TODO: boost concepts for  (reference/dereference)
    public:
      // typedefs
      typedef typename std::map<const Key_type*, Mapped_type*>::const_iterator const_iterator;
      typedef typename std::map<const Key_type*, Mapped_type*>::iterator iterator;
      // constructor/destructor/copy
      Reference_counting_map() : m_map() {}
      Reference_counting_map(const Reference_counting_map& map);
      Reference_counting_map& operator=(const Reference_counting_map& map);
      virtual ~Reference_counting_map();
      // functions
      void add(const Key_type& key, Mapped_type& value);
      bool remove(const Key_type& key);
      bool get(const Key_type& key, Mapped_type*& value);
      void clear();
      //   STL equivalents
      size_t size() const { return m_map.size(); }
      const_iterator begin() const { return m_map.begin(); }
      iterator begin() { return m_map.begin(); }
      const_iterator end() const { return m_map.end(); }
      iterator end() { return m_map.end(); }
      const_iterator find(const Key_type& key) const { return m_map.find(&key); }
      iterator find(const Key_type& key) { return m_map.find(&key); }
      void erase(iterator position);
    private:
      std::map<const Key_type*, Mapped_type*> m_map;
    };

    //
    // Constructor/destructor/copy
    //

    template <class Key_type, class Mapped_type>
    Reference_counting_map<Key_type, Mapped_type>::Reference_counting_map(const Reference_counting_map& map)
    {
      for (typename std::map<const Key_type*, Mapped_type*>::const_iterator i = map.m_map.begin();
           i != map.m_map.end();
           ++i)
      {
        i->first->reference();
        i->second->reference();
        m_map[i->first] = i->second;
      }
    }

    template <class Key_type, class Mapped_type>
    Reference_counting_map<Key_type, Mapped_type>::~Reference_counting_map()
    {
      for (typename std::map<const Key_type*, Mapped_type*>::const_iterator i = m_map.begin();
           i != m_map.end();
           ++i)
      {
        i->second->unreference();
        i->first->unreference();
      }
    }

    template <class Key_type, class Mapped_type> Reference_counting_map<Key_type, Mapped_type>&
    Reference_counting_map<Key_type, Mapped_type>::operator=(const Reference_counting_map& map)
    {
      if (this != &map)
      {
        for (typename std::map<const Key_type*, Mapped_type*>::const_iterator i = map.m_map.begin();
             i != map.m_map.end();
             ++i)
        {
          i->first->reference();
          i->second->reference();
          m_map[i->first] = i->second;
        }
      }

      return *this;
    }

    //
    // Functions
    //

    template <class Key_type, class Mapped_type>
    void Reference_counting_map<Key_type, Mapped_type>::add(const Key_type& key, Mapped_type& value)
    {
      typename std::map<const Key_type*, Mapped_type*>::const_iterator i = m_map.find(&key);
      if (m_map.end() != i)
      {
        if (i->second != &value)
        {
          i->second->unreference();
          m_map[&key] = &value;
          value.reference();
        }
      }
      else
      {
        key.reference();
        value.reference();
        m_map[&key] = &value;
      }
    }

    template <class Key_type, class Mapped_type>
    bool Reference_counting_map<Key_type, Mapped_type>::remove(const Key_type& key)
    {
      bool return_value = false;

      iterator i = find(key);
      if (i != m_map.end())
      {
        i->second->unreference();
        i->first->unreference();
        erase(i);
        return_value = true;
      }

      return return_value;
    }

    template <class Key_type, class Mapped_type>
    bool Reference_counting_map<Key_type, Mapped_type>::get(const Key_type& key, Mapped_type*& return_mapped)
    {
      bool return_value = false;

      iterator i = find(key);
      if (i != m_map.end())
      {
        return_mapped = i->second;
        return_value = true;
      }

      return return_value;
    }

    template <class Key_type, class Mapped_type>
    void Reference_counting_map<Key_type, Mapped_type>::clear()
    {
      for (const_iterator i = m_map.begin(); i != m_map.end(); ++i)
      {
        i->second->unreference();
        i->first->unreference();
      }
      m_map.clear();
    }

    //   STL equivalents

    template <class Key_type, class Mapped_type>
    void Reference_counting_map<Key_type, Mapped_type>::erase(iterator position)
    {
      if (position != m_map.end())
      {
        position->second->unreference();
        position->first->unreference();
      }
      m_map.erase(position);
    }
  }
}

#endif // TOVERO_SUPPORT_REFERENCE_COUNTING_MAP_HPP_
