// Reference_counting_list.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Class for storing reference counted nodes in a list as pointers.

#ifndef TOVERO_SUPPORT_REFERENCE_COUNTING_LIST_HPP_
#define TOVERO_SUPPORT_REFERENCE_COUNTING_LIST_HPP_

#include <algorithm>
#include <list>
#include <cstddef>

namespace Roan_trail
{
  namespace Tovero_support
  {
    template <class Value_type> class Reference_counting_list
    {
      // TODO: boost concepts for Value_type (non pointer, reference/dereference)
    public:
      // typedefs
      typedef typename std::list<Value_type*>::const_iterator const_iterator;
      typedef typename std::list<Value_type*>::iterator iterator;
      // constructor/destructor/copy
      Reference_counting_list() {}
      Reference_counting_list(const Reference_counting_list& list);
      Reference_counting_list& operator=(const Reference_counting_list& list);
      virtual ~Reference_counting_list();
      // operators
      inline Reference_counting_list& operator<<(Value_type& value);
      // functions
      void push_back(Value_type& value);
      void push_front(Value_type& value);
      bool remove_first(Value_type& value);
      void clear();
      //   STL equivalents
      size_t size() const { return m_list.size(); }
      const_iterator begin() const { return m_list.begin(); }
      iterator begin() { return m_list.begin(); }
      const_iterator end() const { return m_list.end(); }
      iterator end() { return m_list.end(); }
      void erase(iterator position);
    private:
      std::list<Value_type*> m_list;
    };

    //
    // Constructor/destructor/copy
    //

    template <class Value_type>
    Reference_counting_list<Value_type>::Reference_counting_list(const Reference_counting_list& list)
    {
      for (typename std::list<Value_type*>::const_iterator i = list.m_list.begin();
           i != list.m_list.end();
           ++i)
      {
        (*i)->reference();
        m_list.push_back(*i);
      }
    }

    template <class Value_type> Reference_counting_list<Value_type>::~Reference_counting_list()
    {
      for (typename std::list<Value_type*>::const_iterator i = m_list.begin();
           i != m_list.end();
           ++i)
      {
        (*i)->unreference();
      }
    }

    template <class Value_type> Reference_counting_list<Value_type>&
    Reference_counting_list<Value_type>::operator=(const Reference_counting_list& list)
    {
      if (this != &list)
      {
        for (typename std::list<Value_type*>::const_iterator i = list.m_list.begin();
             i != list.m_list.end();
             ++i)
        {
          (*i)->reference();
          m_list.push_back(*i);
        }
      }

      return *this;
    }

    template <class Value_type> inline Reference_counting_list<Value_type>&
    Reference_counting_list<Value_type>::operator<<(Value_type& value)
    {
      push_back(value);
      return *this;
    }

    //
    // Functions
    //

    template <class Value_type> void Reference_counting_list<Value_type>::push_back(Value_type& value)
    {
      value.reference();
      m_list.push_back(&value);
    }

    template <class Value_type> void Reference_counting_list<Value_type>::push_front(Value_type& value)
    {
      value.reference();
      m_list.push_front(&value);
    }

    template <class Value_type> bool Reference_counting_list<Value_type>::remove_first(Value_type& value)
    {
      bool return_value = false;

      iterator i = find(m_list.begin(), m_list.end(), &value);
      if (i != m_list.end())
      {
        erase(i);
        return_value = true;
      }

      return return_value;
    }

    template <class Value_type> void Reference_counting_list<Value_type>::clear()
    {
      for (const_iterator i = m_list.begin(); i != m_list.end(); ++i)
      {
        (*i)->unreference();
      }
      m_list.clear();
    }

    //
    //   STL equivalents
    //

    template <class Value_type> void Reference_counting_list<Value_type>::erase(iterator position)
    {
      (*position)->unreference();
      m_list.erase(position);
    }
  }
}

#endif // TOVERO_SUPPORT_REFERENCE_COUNTING_LIST_HPP_
