// Unitless.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#ifndef TOVERO_MATH_UNITLESS_HPP_
#define TOVERO_MATH_UNITLESS_HPP_

#include <tovero/math/geometry/Quantity.hpp>
#include <cmath>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Unitless : public Quantity
    {
    public:
      // constructors
      Unitless() : Quantity() {}
      explicit Unitless(double v) : Quantity(v) {}
      static Unitless from_value(double value) { return Unitless(value); }
      //  Note: use compiler generated copy constructor and destructor
      // operators
      inline Unitless& operator=(const Unitless& q);
      Unitless operator+(const Unitless& q) const
      { return Unitless::from_value(value() + q.value()); }
      Unitless operator-(const Unitless& q) const
      { return Unitless::from_value(value() - q.value()); }
      Unitless operator-() const { return Unitless::from_value(-value()); }
      Unitless operator*(const Unitless& factor) const { return Unitless::from_value(factor.value() * value()); }
      Unitless operator*(double factor) const { return Unitless::from_value(factor * value()); }
      Unitless operator/(const Unitless& divisor) const
      { return Unitless::from_value(value() / divisor.value()); }
      Unitless operator/(double divisor) const { return Unitless::from_value(value() / divisor); }
      inline Unitless& operator+=(const Unitless& addend);
      inline Unitless& operator-=(const Unitless& subtrahend);
      inline Unitless& operator*=(const Unitless& factor);
      inline Unitless& operator*=(double factor);
      inline Unitless& operator/=(const Unitless& divisor);
      inline Unitless& operator/=(double divisor);
    };

    // free functions
    inline Unitless operator-(double minuend, const Unitless& subtrahend)
    { return Unitless::from_value(minuend - subtrahend.value()); }

    inline Unitless operator*(double factor, const Unitless& q)
    { return Unitless::from_value(factor * q.value()); }

    inline Unitless sqrt(const Unitless& q)
    { return Unitless::from_value(std::sqrt(q.value())); }


    //
    // Inline definitions
    //

    //
    //   Operators
    //

    inline Unitless& Unitless::operator=(const Unitless& d)
    {
      if (this != &d)
      {
        set_value(d.value());
      }

      return *this;
    }

    inline Unitless& Unitless::operator+=(const Unitless& addend)
    {
      set_value(value() + addend.value());

      return *this;
    }

    inline Unitless& Unitless::operator-=(const Unitless& subtrahend)
    {
      set_value(value() - subtrahend.value());

      return *this;
    }

    inline Unitless& Unitless::operator*=(const Unitless& factor)
    {
      set_value(value() * factor.value());

      return *this;
    }

    inline Unitless& Unitless::operator*=(double factor)
    {
      set_value(value() * factor);

      return *this;
    }

    inline Unitless& Unitless::operator/=(const Unitless& divisor)
    {
      set_value(value() / divisor.value());

      return *this;
    }

    inline Unitless& Unitless::operator/=(double divisor)
    {
      set_value(value() / divisor);

      return *this;
    }
  }
}

#endif // TOVERO_MATH_UNITLESS_HPP_
