// Transformation.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Used to geometrically transform operands in compound solids
// Evaluates to a 4x4 "projective" matrix

#include <tovero/math/geometry/Transformation.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Matrix.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/Reference_counting_base.hpp>
#include <tovero/support/common.hpp>

using Roan_trail::Tovero_support::Reference_counting_base;
using namespace Roan_trail::Tovero_math;

//
// Constructors/copy
//

Transformation::Transformation()
  : Reference_counting_base(),
    m_matrix(new Matrix),
    m_translation_units(new Distance)
{
  *m_matrix = Matrix::I;
  *m_translation_units = Distance::meter;

  postcondition(mf_invariant(false));
}

Transformation::Transformation(const Matrix& matrix, const Distance& translation_units)
  : Reference_counting_base(),
    m_matrix(new Matrix),
    m_translation_units(new Distance)
{
  *m_matrix = matrix;
  *m_translation_units = Distance::meter;

  postcondition(mf_invariant(false));
}

Transformation::Transformation(const Transformation& t)
  : Reference_counting_base(t)
{
  if (this != &t)
  {
    *m_matrix = *t.m_matrix;
    *m_translation_units = *t.m_translation_units;
  }

  postcondition(mf_invariant(false));
}

Transformation& Transformation::operator=(const Transformation& t)
{
  precondition(mf_invariant(false));

  if (this != &t)
  {
    Reference_counting_base::operator=(t);
    *m_matrix = *t.m_matrix;
    *m_translation_units = *t.m_translation_units;
  }

  postcondition(mf_invariant(false));
  return *this;
}

//
// Accessors/mutators
//

void Transformation::set_matrix(const Matrix& matrix)
{
  precondition(mf_invariant());

  delete m_matrix;
  m_matrix = new Matrix;
  *m_matrix = matrix;

  postcondition(mf_invariant());
}

void Transformation::set_translation_units(const Distance& units)
{
  precondition(mf_invariant());

  delete m_translation_units;
  m_translation_units = new Distance;
  *m_translation_units = units;

  postcondition(mf_invariant());
}

//
// Protected member functions
//

//
//   Destructor
//

Transformation::~Transformation()
{
  precondition(mf_invariant(false));

  delete m_translation_units;
  delete m_matrix;
}

//
// Functions
//

// functions
void Transformation::set_rotation(const Unit_vector& axis,
                                  const Angle& a,
                                  const Geometric_tolerances& tolerances)
{
  precondition(mf_invariant());

  const Unitless axis_length = axis.length();

  if (tolerances.is_unitless_near_zero(axis_length))
  {
    // invalid rotation axis (length zero), set this transformation to identity
    *m_matrix = Matrix::I;
  }

  m_matrix->set_rotation(axis, a);

  postcondition(mf_invariant());
}

Transformation& Transformation::rotate(const Unit_vector& axis,
                                       const Angle& a,
                                       const Geometric_tolerances& tolerances)
{
  precondition(mf_invariant());

  Transformation rotation;
  rotation.set_rotation(axis,
                        a,
                        tolerances);

  m_matrix->multiply_left(*rotation.m_matrix);

  postcondition(mf_invariant());
  return *this;
}

void Transformation::set_translation(const Vector& tv)
{
  precondition(mf_invariant());

  set_translation(tv[0],
                  tv[1],
                  tv[2]);

  postcondition(mf_invariant());
}

void Transformation::set_translation(const Distance& tx,
                                     const Distance& ty,
                                     const Distance& tz)
{
  precondition(mf_invariant());

  m_matrix->set_translation((tx / *m_translation_units).value(),
                            (ty / *m_translation_units).value(),
                            (tz / *m_translation_units).value());

  postcondition(mf_invariant());
}

Transformation& Transformation::translate(const Vector& tv)
{
  precondition(mf_invariant());

  translate(tv[0],
            tv[1],
            tv[2]);

  postcondition(mf_invariant());

  return *this;
}

Transformation& Transformation::translate(const Distance& tx,
                                          const Distance& ty,
                                          const Distance& tz)
{
  precondition(mf_invariant());

  Transformation translation;
  translation.set_translation(tx,
                              ty,
                              tz);

  m_matrix->multiply_left(*translation.m_matrix);

  postcondition(mf_invariant());

  return *this;
}

void Transformation::set_scale(const Unitless& sx,
                               const Unitless& sy,
                               const Unitless& sz)
{
  precondition(mf_invariant());

  m_matrix->set_scale(sx.value(),
                      sy.value(),
                      sz.value());

  postcondition(mf_invariant());
}

Transformation& Transformation::scale(const Unitless& sx,
                                      const Unitless& sy,
                                      const Unitless& sz)
{
  precondition(mf_invariant());

  Transformation scale;
  scale.set_scale(sx,
                  sy,
                  sz);

  m_matrix->multiply_left(*scale.m_matrix);

  postcondition(mf_invariant());

  return *this;
}

void Transformation::set_uniform_scale(const Unitless& s)
{
  precondition(mf_invariant());

  m_matrix->set_uniform_scale(s.value());

  postcondition(mf_invariant());
}

Transformation& Transformation::scale_uniform(const Unitless& s)
{
  precondition(mf_invariant());

  Transformation scale;
  scale.set_uniform_scale(s);

  m_matrix->multiply_left(*scale.m_matrix);

  postcondition(mf_invariant());

  return *this;
}

//
//   Class invariant
//

bool Transformation::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  if (!m_matrix || !m_translation_units)
  {
    goto exit_point;
  }

  return_value = (!check_base_class || Reference_counting_base::mf_invariant(check_base_class));;
  goto exit_point;

 exit_point:
  return return_value;
}
