// Point.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// affine 3D point

#ifndef TOVERO_MATH_POINT_HPP_
#define TOVERO_MATH_POINT_HPP_

#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <iostream>
#include <cstddef>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Point
    {
    public:
      // constructors
      inline Point();
      inline Point(const Distance& x,
                   const Distance& y,
                   const Distance& z);
      inline Point(double x,
                   double y,
                   double z,
                   const Distance& units);
      inline Point(const Point& P);
      // operators
      Point& operator=(const Point& P);
      const Distance& operator[](size_t index) const { return m_P[index]; }
      Distance& operator[](size_t index) { return m_P[index]; }
      inline Point operator+(const Vector& v) const;
      inline Vector operator-(const Point& P) const;
      inline Point operator-(const Vector& v) const;
      // predefined points
      static const Point O;
    private:
      Distance m_P[3];
    };

    //
    // Definitions
    //

    //
    //   Constructors
    //

    inline Point::Point()
    {
      m_P[0] = Distance::from_value(0.0);
      m_P[1] = Distance::from_value(0.0);
      m_P[2] = Distance::from_value(0.0);
    }

    inline Point::Point(const Distance& x,
                        const Distance& y,
                        const Distance& z)
    {
      m_P[0] = x;
      m_P[1] = y;
      m_P[2] = z;
    }

    inline Point::Point(double x,
                        double y,
                        double z,
                        const Distance& units)
    {
      m_P[0] = x * units;
      m_P[1] = y * units;
      m_P[2] = z * units;
    }

    inline Point::Point(const Point& P)
    {
      m_P[0] = P.m_P[0];
      m_P[1] = P.m_P[1];
      m_P[2] = P.m_P[2];
    }

    //
    //   Operators
    //
    inline Point& Point::operator=(const Point& P)
    {
      if (this != &P)
      {
        m_P[0] = P.m_P[0];
        m_P[1] = P.m_P[1];
        m_P[2] = P.m_P[2];
      }
      return *this;
    }

    inline Point Point::operator+(const Vector& v) const
    {
      return Point(m_P[0] + v[0],
                   m_P[1] + v[1],
                   m_P[2] + v[2]);
    }

    inline Vector Point::operator-(const Point& P) const
    {
      return Vector(m_P[0] - P[0],
                    m_P[1] - P[1],
                    m_P[2] - P[2]);
    }

    inline Point Point::operator-(const Vector& v) const
    {
      return Point(m_P[0] - v[0],
                   m_P[1] - v[1],
                   m_P[2] - v[2]);
    }
  }
}

#endif // TOVERO_MATH_POINT_HPP_
