// Elliptical_cone.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Solid validation based on ./src/librt/primitives/tgc/tgc.c from
// BRL-CAD (version 7.20.4) source:
//
//   Copyright (c) 1985-2011 United States Government as represented by
//   the U.S. Army Research Laboratory.
//
//   This library is free software; you can redistribute it and/or
//   modify it under the terms of the GNU Lesser General Public License
//   version 2.1 as published by the Free Software Foundation.
//
//   This library is distributed in the hope that it will be useful, but
//   WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Lesser General Public License for more details.
//
//   You should have received a copy of the GNU Lesser General Public
//   License along with this file; see the file named COPYING for more
//   information.

#include <tovero/math/geometry/Elliptical_cone.hpp>
#include <tovero/math/geometry/Cone.hpp>
#include <tovero/math/geometry/Cylinder.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Elliptical_cylinder.hpp>
#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>

using std::string;
using std::stringstream;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using namespace Roan_trail::Tovero_math;

//
// Constructor
//

Elliptical_cone::Elliptical_cone(const string& name)
  : General_cone_base(name),
    m_base(Point::O),
    m_base_a(Unit_vector::x * Distance::meter),
    m_base_b(Unit_vector::y * Distance::meter),
    m_height(Distance(1.0 * Distance::meter)),
    m_ratio(Unitless(1.0))
{
}

Elliptical_cone::Elliptical_cone(const Point& base,
                                 const Vector& base_a,
                                 const Vector& base_b,
                                 const Distance& height,
                                 const Unitless& ratio,
                                 const string& name)
  : General_cone_base(name),
    m_base(base),
    m_base_a(base_a),
    m_base_b(base_b),
    m_height(height),
    m_ratio(ratio)
{
}

//
// Predicates
//

bool Elliptical_cone::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  Unit_vector unit_a;
  const Distance& base_a_length = m_base_a.normalize(unit_a);
  Unit_vector unit_b;
  const Distance& base_b_length = m_base_b.normalize(unit_b);

  // check non-zero length base vectors
  if (tolerances.is_distance_near_zero(base_a_length)
      || tolerances.is_distance_near_zero(base_b_length))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the base vectors must be non-zero ";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check that both bases aren't degenerate (i.e. < 2D)
  if (tolerances.is_area_near_zero(base_a_length * base_b_length))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": base is degenerate (< 2D)";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check that the base vectors are perpendicular
  if (!tolerances.is_cosine_near_zero(unit_a.dot(unit_b)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the base vectors are not perpendicular";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check for positive height
  if (tolerances.is_distance_near_zero(m_height)
      || m_height < Distance(0.0 * Distance::meter))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the height is non-positive";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check for positive ratio
  if (tolerances.is_unitless_near_zero(m_ratio)
      || m_ratio < Unitless(0.0))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the ratio is non-positive";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

General_cone& Elliptical_cone::generalize() const
{
  Unit_vector unit_a;
  m_base_a.normalize(unit_a);
  Unit_vector unit_b;
  m_base_b.normalize(unit_b);

  Vector height = unit_a.cross(unit_b) * m_height;

  General_cone* cone = new General_cone(m_base,
                                        height,
                                        m_base_a,
                                        m_base_b,
                                        m_base_a / m_ratio,
                                        m_base_b / m_ratio,
                                        name());
  return *cone;
}

Solid& Elliptical_cone::specialize(const Geometric_tolerances& tolerances) const
{
  Solid* return_value = const_cast<Solid*>(static_cast<const Solid*>(this));

  Unit_vector unit_a;
  const Distance& base_a_length = m_base_a.normalize(unit_a);

  // check for equal base radii
  if (tolerances.is_distance_near_zero(base_a_length - m_base_b.length()))
  {
    Unit_vector unit_b;
    m_base_b.normalize(unit_b);
    Vector height = unit_a.cross(unit_b) * m_height;
    // right circular cone
    Cone* cone = new Cone(m_base,
                          height,
                          base_a_length,
                          base_a_length / m_ratio,
                          name());
    return_value = &cone->specialize(tolerances);
    if (return_value != cone)
    {
      cone->reference();
      cone->unreference();
    }
  }
  // check that base and top are the same shape
  else if (tolerances.is_unitless_near_zero(1.0 - m_ratio))
  {
    // right elliptical cylinder
    Elliptical_cylinder* cylinder = new Elliptical_cylinder(m_base,
                                                            m_base_a,
                                                            m_base_b,
                                                            m_height,
                                                            name());
    return_value = cylinder;
  }

  return *return_value;
}
