// BC_database_reader.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.
//
// Based on the following files from BRL-CAD (version 7.20.4) source:
//
//   ./src/conv/g-dot.c
//     Copyright (c) 2011 United States Government as represented by
//     the U.S. Army Research Laboratory.
//
// and documentation from:
//
//   ./include/rtgeom.h
//     Copyright (c) 2004-2011 United States Government as represented by
//     the U.S. Army Research Laboratory.
//
// under the following the license:
//
//   This library is free software; you can redistribute it and/or
//   modify it under the terms of the GNU Lesser General Public License
//   version 2.1 as published by the Free Software Foundation.
//
//   This library is distributed in the hope that it will be useful, but
//   WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Lesser General Public License for more details.
//
//   You should have received a copy of the GNU Lesser General Public
//   License along with this file; see the file named COPYING for more
//   information.

// BRL-CAD headers (declared first to avoid macro conflicts)
#include <wdb.h>
#include <bu.h>
//
#include <tovero/graphics/brlcad/BC_database_reader.hpp>
#include <tovero/graphics/brlcad/BC_combination_attributes.hpp>
#include <tovero/graphics/base/Combination.hpp>
#include <tovero/graphics/base/Combination_attributes.hpp>
#include <tovero/graphics/base/Node_mapper.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/Logger.hpp>
#include <tovero/math/base/math.hpp>
#include <tovero/math/geometry/Boundary_triangle_mesh.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Ellipsoid.hpp>
#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Half_space.hpp>
#include <tovero/math/geometry/Matrix.hpp>
#include <tovero/math/geometry/Plate_triangle_mesh.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Polyhedron.hpp>
#include <tovero/math/geometry/Solid.hpp>
#include <tovero/math/geometry/Solid_combination.hpp>
#include <tovero/math/geometry/Solid_member.hpp>
#include <tovero/math/geometry/Solid_operand.hpp>
#include <tovero/math/geometry/Solid_operator.hpp>
#include <tovero/math/geometry/Torus.hpp>
#include <tovero/math/geometry/Transformation.hpp>
#include <tovero/math/geometry/Triangle_face.hpp>
#include <tovero/math/geometry/Triangle_mesh.hpp>
#include <tovero/math/geometry/Units_mapper.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/Reference_counting_list.hpp>
#include <tovero/support/error/Graphics_error.hpp>
#include <sstream>
#include <string>
#include <vector>
#include <iostream>
#include <cstddef>

using std::endl;
using std::stringstream;
using std::string;
using std::vector;
using Roan_trail::Tovero_support::Error;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Graphics_error;
using Roan_trail::Tovero_support::Logger;
using Roan_trail::Tovero_support::Reference_counting_list;
using Roan_trail::Tovero_math::Boundary_triangle_mesh;
using Roan_trail::Tovero_math::Distance;
using Roan_trail::Tovero_math::Ellipsoid;
using Roan_trail::Tovero_math::General_cone;
using Roan_trail::Tovero_math::Geometric_tolerances;
using Roan_trail::Tovero_math::Half_space;
using Roan_trail::Tovero_math::Matrix;
using Roan_trail::Tovero_math::Plate_triangle_mesh;
using Roan_trail::Tovero_math::Point;
using Roan_trail::Tovero_math::Polyhedron;
using Roan_trail::Tovero_math::Solid;
using Roan_trail::Tovero_math::Solid_combination;
using Roan_trail::Tovero_math::Solid_member;
using Roan_trail::Tovero_math::Solid_operand;
using Roan_trail::Tovero_math::Solid_operator;
using Roan_trail::Tovero_math::Torus;
using Roan_trail::Tovero_math::Transformation;
using Roan_trail::Tovero_math::Triangle_face;
using Roan_trail::Tovero_math::Triangle_mesh;
using Roan_trail::Tovero_math::Units_mapper;
using Roan_trail::Tovero_math::Unit_vector;
using Roan_trail::Tovero_math::Unitless;
using Roan_trail::Tovero_math::Vector;
using namespace Roan_trail::Tovero_graphics;

//
// Internal helpers
//

namespace
{
  // constants
  const int ic_CPU_count_for_walk = 1; // TODO: implement multiple CPUs?
  const string ic_BC_default_length_units = "mm";

  Solid_operator::Solid_operator_type ih_convert_from_BC_op(int BC_op)
  {
    Solid_operator::Solid_operator_type op = Solid_operator::union_op;
    switch (BC_op)
    {
    case OP_INTERSECT:
      op = Solid_operator::intersection_op;
      break;
    case OP_SUBTRACT:
      op = Solid_operator::difference_op;
      break;
    default:
      break;
    }
    return op;
  }

  void ih_copy_matrix(const matp_t BC_matrix, Matrix& return_matrix)
  {
    for (size_t i = 0; i < 16; ++i)
    {
      return_matrix[i] = BC_matrix[i];
    }
  }
}

//
// Constructor/destructor/initialization
//

BC_database_reader::BC_database_reader(bool simplify_solids,
                                       bool tolerate_database_errors,
                                       const Geometric_tolerances& tolerances)
  : m_file(),
    m_objects(),
    m_node_mapper(0),
    m_simplify_solids(simplify_solids),
    m_tolerate_database_errors(tolerate_database_errors),
    m_tolerances(tolerances),
    m_IO_error(0),
    m_logger(Logger::default_logger()),
    m_distance_conversion_factor(new Distance(1.0 * Distance::meter))
{
  // Note: setting debug options with RT_G_DEBUG = DEBUG_DB | DEBUG_REGIONS | DEBUG_SOLIDS | DEBUG_TREEWALK;
  //       helps debug BC tree walking

  postcondition(mf_invariant(false));
}

BC_database_reader::File_struct::File_struct()
  : path(""),
    DB_instance(0)
{
}

BC_database_reader::~BC_database_reader()
{
  precondition(mf_invariant(false));

  delete m_distance_conversion_factor;
  delete m_IO_error;

  if (m_file.DB_instance)
  {
    db_close(m_file.DB_instance);
  }

  mf_cleanup_nodes();
}

//
// Read
//

bool BC_database_reader::read(const string& path,
                              const vector<string>& objects,
                              Reference_counting_list<Solid>& return_solids,
                              Units_mapper& return_units_mapper,
                              Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  on_error(path == "", new Graphics_error(error_location(),
                                          Graphics_error::read,
                                          "empty path for BRL-CAD database when opening for read"));

  for (size_t i = 0; i < objects.size(); ++i)
  {
    on_error("" == objects[i], new Graphics_error(error_location(),
                                                  Graphics_error::read,
                                                  "empty object path requested from BRL-CAD database"));
  }

  mf_setup_for_read(path,
                    objects,
                    return_units_mapper);

  Error_param error;

  string length_units;
  const bool opened = mf_open_for_read(return_units_mapper,
                                       length_units,
                                       error);
  on_error(!opened, new Graphics_error(error_location(),
                                       Graphics_error::general,
                                       error()));

  const bool database_read = mf_read_database(return_solids, error);
  on_error(!database_read, new Graphics_error(error_location(),
                                              Graphics_error::general,
                                              error()));

  return_units_mapper.set_length_units(length_units);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (m_IO_error)
  {
    delete m_IO_error;
    m_IO_error = 0;
  }
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
// Protected member functions
//

bool BC_database_reader::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  if (!m_distance_conversion_factor)
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions
//

bool BC_database_reader::mf_open_for_read(const Units_mapper& units_mapper,
                                          string& return_length_units,
                                          Error_param& return_error)
{
  precondition(!return_error()
               && !m_file.DB_instance);
  struct db_i *stream = 0;

  bool return_value = false;

  start_error_block();

  // open database
  stream = db_open(m_file.path.c_str(), "r");
  on_error(!stream, new Graphics_error(error_location(),
                                       Graphics_error::read,
                                       "could not open BRL-CAD database",
                                       m_file.path));

  on_error((db_version(stream) < 5), new Graphics_error(error_location(),
                                                        Graphics_error::read,
                                                        "BRL-CAD database version less than 5 not supported, "
                                                        "upgrade to read",
                                                        m_file.path));

  // build database directory
  const int build_return = db_dirbuild(stream);
  on_error(build_return, new Graphics_error(error_location(),
                                            Graphics_error::read,
                                            "could not build BRL-CAD database directory",
                                            m_file.path));

  // update count of references to objects by combinations
  // (the d_nref field of a directory entry)
  // tree tops will then have zero references
  db_update_nref(stream, &rt_uniresource);

  if (m_objects.empty())
  {
    // if no objects requested, assume all objects, get "tree tops"
    for (size_t i = 0; i < RT_DBNHASH; i++)
    {
      for (struct directory* dir = stream->dbi_Head[i];
           dir != RT_DIR_NULL; // (check_code_ignore)
           dir = dir->d_forw)
      {
        if (!dir->d_nref // zero references, it is a tree top
            && !(dir->d_flags & RT_DIR_HIDDEN) && (dir->d_addr != RT_DIR_PHONY_ADDR)) // entries to ignore
        {
          m_objects.push_back(dir->d_namep);
        }
      }
    }
  }

  const char* length_units = bu_units_string(stream->dbi_local2base);
  const Units_mapper::Length_map_type& length_map = units_mapper.length_map;
  Units_mapper::Length_map_type::const_iterator i = length_map.end();
  if (length_units)
  {
    i = length_map.find(string(length_units));
  }

  if (!length_units
      || (i == length_map.end()))
  {
    stringstream diagnostic;
    diagnostic << "Unsupported length unit conversion factor: " << stream->dbi_local2base;
    if (length_units)
    {
      diagnostic << " (\"" << length_units << "\") ";
    }
    diagnostic << "in BRL-CAD input database";
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::read,
                                                             diagnostic.str()));
    m_logger << Logger::warning << diagnostic << ", using BRL-CAD default length units)" << endl;
    return_length_units = ic_BC_default_length_units;
  }
  else
  {
    return_length_units = string(length_units);
  }

  assert((length_map.find(string(return_length_units)) != length_map.end())
         && "invalid length units set in units mapper");

  m_file.DB_instance = stream;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (stream)
  {
    db_close(stream);
  }
  goto exit_point;

 exit_point:
  postcondition(((return_value && m_file.DB_instance)
                 || (!return_value && !m_file.DB_instance))
                && return_error.is_valid_at_return(return_value));
  return return_value;
}

bool BC_database_reader::mf_read_database(Reference_counting_list<Solid>& return_solids,
                                          Error_param& return_error)
{
  precondition(!return_error()
               && m_file.DB_instance);

  bool return_value = false;

  start_error_block();

  // process all the specified objects in the BRL-CAD database
  const size_t object_count = m_objects.size();
  for (size_t i = 0; i < object_count; ++i)
  {
    const char* object_name = m_objects[i].c_str();
    struct directory *dir =  db_lookup(m_file.DB_instance, object_name, 1);
    // if found, traverse the tree for the object to create its Tovero
    // representation
    if (dir)
    {
      db_functree(m_file.DB_instance,
                  dir,
                  &BC_database_reader::mf_do_combination,
                  &BC_database_reader::mf_do_leaf,
                  0,
                  this);
      Solid* current_solid = m_node_mapper->find_node_by_name(string(object_name));
      if (current_solid)
      {
        return_solids.push_back(*current_solid);
      }
    }
    else
    {
      const string diagnostic = string("Unable to locate: ") + object_name + string(" in input database");
      on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                               Graphics_error::read,
                                                               diagnostic));
      m_logger << Logger::warning << diagnostic << endl;
    }
  }

  // set the database title
  if (m_file.DB_instance->dbi_title)
  {
    m_title = m_file.DB_instance->dbi_title;
  }
  else
  {
    m_title = "";
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

void BC_database_reader::mf_setup_for_read(const string& path,
                                           const vector<string>& objects,
                                           const Units_mapper& units_mapper)
{
  m_file.path = path;
  if (m_file.DB_instance)
  {
    db_close(m_file.DB_instance);
    m_file.DB_instance = 0;
  }
  m_objects = objects;

  mf_cleanup_nodes();
  m_node_mapper = new Node_mapper<Solid*>;

  delete m_IO_error;
  m_IO_error = 0;

  // BRL-CAD stores all length values internally as millimeters, set up a
  // conversion factor
  Units_mapper::Length_map_type::const_iterator i = units_mapper.length_map.find("mm");
  assert((i != units_mapper.length_map.end()) && "error, mm unit conversion factor not found");
  *m_distance_conversion_factor = i->second;
}

void BC_database_reader::mf_cleanup_after_read()
{
  db_close(m_file.DB_instance);
  m_file.DB_instance = 0;
  m_file.path = "";
  m_objects.clear();

  mf_cleanup_nodes();
}

void BC_database_reader::mf_cleanup_nodes()
{
  if (m_node_mapper)
  {
    Node_mapper<Solid*>::Node_name_map_type node_map = m_node_mapper->nodes();
    for (Node_mapper<Solid*>::Node_name_map_type::left_map::const_iterator i = node_map.left.begin();
         i != node_map.left.end();
         ++i)
    {
      i->first->unreference(); // make sure any unused nodes are deleted from memory
    }
    delete m_node_mapper;
    m_node_mapper = 0;
  }
}

//
// Tree walk callbacks
//

void BC_database_reader::mf_do_combination(struct db_i *database,
                                           struct directory *dir,
                                           genptr_t client_data)
{
  precondition(database
               && dir
               && client_data);

  BC_database_reader* reader = reinterpret_cast<BC_database_reader*>(client_data);

  bool need_internal_free = false;
  struct rt_db_internal internal;

  start_error_block();

  // skip further processing if there is already an error
  if (reader->m_IO_error)
  {
    goto exit_point;
  }

  const string combination_name = dir->d_namep;
  if (!reader->m_node_mapper->find_node_by_name(combination_name))
  {
    // this combination not created yet, create it

    // get the combination from the BRL-CAD database, returning its internal representation
    // (function returns object ID or < 0 on failure)
    const int object_ID = rt_db_get_internal(&internal,
                                             dir,
                                             reader->m_file.DB_instance,
                                             0,
                                             &rt_uniresource);
    if (object_ID < 0)
    {
      const string diagnostic = string("Could not get get database object: ") + combination_name;
      on_error(!reader->m_tolerate_database_errors, new Graphics_error(error_location(),
                                                                       Graphics_error::read,
                                                                       diagnostic));
      reader->m_logger << Logger::warning << diagnostic << endl;
    }

    need_internal_free = true;
    struct rt_comb_internal* BC_combination = reinterpret_cast<struct rt_comb_internal *>(internal.idb_ptr);

    if (BC_combination->tree)
    {
      struct bu_vls vls = BU_VLS_INIT_ZERO;
      struct rt_tree_array *tree_array = 0;

      if (db_ck_v4gift_tree(BC_combination->tree) < 0)
      {
        // make tree top all union operations, move non-union operations down
        db_non_union_push(BC_combination->tree, &rt_uniresource);
        // From the BRL-CAD documentation the following function:
        // 1) Checks that if unions are present they are all at the root of tree (db_non_union_push above).
        // 2) Checks that non-union children of union nodes are all left-heavy
        //    (nothing but solid nodes permitted on rhs of binary operators).
        // (returns 0 if error, < 0 if error)
        const int checked = db_ck_v4gift_tree(BC_combination->tree);
        if (checked < 0)
        {
          const string diagnostic = string("Could not flatten tree of combination: ") + combination_name;
          on_error(!reader->m_tolerate_database_errors, new Graphics_error(error_location(),
                                                                           Graphics_error::read,
                                                                           diagnostic));
          reader->m_logger << Logger::warning << diagnostic << endl;
          goto exit_point;
        }
      }
      size_t actual_count = 0;
      const size_t node_count = db_tree_nleaves(BC_combination->tree);
      if (node_count > 0)
      {
        tree_array = reinterpret_cast<struct rt_tree_array *>(bu_calloc(node_count,
                                                                           sizeof(struct rt_tree_array),
                                                                           "tree list"));
        actual_count = reinterpret_cast<struct rt_tree_array *>(db_flatten_tree(tree_array,
                                                                                BC_combination->tree,
                                                                                OP_UNION,
                                                                                1,
                                                                                &rt_uniresource))
          - tree_array;
        assert((actual_count == node_count) && "Error, actual count does not equal node count");
        BC_combination->tree = TREE_NULL; // (check_code_ignore)
      }
      else
      {
        actual_count = 0;
        tree_array = 0;
      }

      // setup combination attributes
      int material = 0;
      if (BC_combination->material.vls_str
          && ("" != string(BC_combination->material.vls_str)))
      {
        stringstream material_stream(BC_combination->material.vls_str);
        material_stream >> material;
      }
      BC_combination_attributes BC_attributes((BC_combination->shader.vls_str ? BC_combination->shader.vls_str
                                               : ""),
                                              material,
                                              (BC_combination->rgb_valid ?
                                               &BC_combination->rgb[0] : 0),
                                              BC_combination->region_flag,
                                              BC_combination->inherit);
      Combination_attributes* attributes = BC_attributes.attributes();

      // create and add the combination
      Combination* combination = new Combination(attributes);
      reader->mf_add_solid(combination_name.c_str(), combination);

      // add children to the combination
      for (size_t i = 0; i < actual_count; i++)
      {
        const char* child_node_name = tree_array[i].tl_tree->tr_l.tl_name;
        Solid* solid = reader->m_node_mapper->find_node_by_name(string(child_node_name));
        assert(solid && "error, solid not found in mapper");
        // determine transform, if any, for edge
        const matp_t edge_matrix = tree_array[i].tl_tree->tr_l.tl_mat;
        Transformation* transformation = 0;
        if (edge_matrix)
        {
          Matrix matrix(Matrix::I);
          ih_copy_matrix(edge_matrix, matrix);
          transformation = new Transformation(matrix, *reader->m_distance_conversion_factor);
        }
        // setup the member
        //   get operator for this member
        const Solid_operator::Solid_operator_type op = ih_convert_from_BC_op(tree_array[i].tl_op);
        Solid_operand* operand = new Solid_operand(*solid, transformation);
        Solid_member* member = new Solid_member(*operand, op);
        combination->add_member(*member);
        db_free_tree(tree_array[i].tl_tree, &rt_uniresource);
      }
      bu_vls_free(&vls);
      if (tree_array)
      {
        bu_free((char *)tree_array, "printnode: rt_tree_array");
      }
    }
  }

  end_error_block();

 error_handler:
  delete reader->m_IO_error;
  reader->m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  goto exit_point;

 exit_point:
  if (need_internal_free)
  {
    rt_db_free_internal(&internal);
  }
  return;
}

void BC_database_reader::mf_do_leaf(struct db_i *database,
                                    struct directory *dir,
                                    genptr_t client_data)
{
  precondition(database && dir && client_data);

  BC_database_reader* reader = reinterpret_cast<BC_database_reader*>(client_data);

  bool need_internal_free = false;
  struct rt_db_internal internal;

  start_error_block();

  // skip further processing if there is already an error
  if (reader->m_IO_error)
  {
    goto exit_point;
  }

  const string node_name = dir->d_namep;
  if (!reader->m_node_mapper->find_node_by_name(node_name))
  {
    // this leaf not created yet in the solid tree, create it
    need_internal_free = true;
    // get an object from the BRL-CAD database, returning its internal representation
    // (function returns object ID or < 0 on failure)
    const int object_ID = rt_db_get_internal(&internal,
                                             dir,
                                             reader->m_file.DB_instance,
                                             0,
                                             &rt_uniresource);
    if (object_ID < 0)
    {
      const string diagnostic = string("Could not get get database object: ") + node_name;
      on_error(!reader->m_tolerate_database_errors, new Graphics_error(error_location(),
                                                                       Graphics_error::read,
                                                                       diagnostic));
      reader->m_logger << Logger::warning << diagnostic << endl;
      goto exit_point;
    }

    bool valid_solid = false;
    Error_param error;
    const Conversion_struct conversion = { internal.idb_ptr,
                                           internal.idb_type,
                                           node_name };
    // TODO: use function objects or function pointer array?
    switch (internal.idb_type) {
    case ID_ELL: // ellipsoid
    case ID_SPH: // sphere
      valid_solid = reader->mf_make_ellipsoid(conversion, error);
      break;
    case ID_TOR: // torus
      valid_solid = reader->mf_make_torus(conversion, error);
      break;
    case ID_TGC: // truncated general cone
    case ID_REC: // right elliptical cylinder
      valid_solid = reader->mf_make_general_cone(conversion, error);
      break;
    case ID_ARB8: // convex primitive with from four to six faces
      // this primitive may have degenerate faces
      // faces are: 0123, 7654, 0347, 1562, 0451, 3267
      // (points listed above in counter-clockwise order)
      valid_solid = reader->mf_make_polyhedron(conversion, error);
      break;
    case ID_BOT: // bag of triangles
      valid_solid = reader->mf_make_BOT(conversion, error);
      break;
    case ID_ARS: // series of curves, each with the same number of points
    case ID_HALF: // half-space defined by a plane and a distance from origin
      valid_solid = reader->mf_make_half_space(conversion, error);
      break;
    case ID_POLY: // polygons (up to 5 vertices per)
    case ID_BSPLINE: // NURB surfaces
    case ID_NMG: // N-manifold geometry
    case ID_ARBN:
    case ID_DSP: // displacement map (terrain primitive), may reference an external file or binunif object
    case ID_HF: // height field (terrain primitive), references an external file
    case ID_EBM: // extruded bit-map, references an external file
    case ID_VOL: // volume of revolution, references an external file
    case ID_PIPE:
    case ID_PARTICLE:
    case ID_RPC:
    case ID_RHC:
    case ID_EPA:
    case ID_EHY:
    case ID_ETO:
    case ID_GRIP:
    case ID_SKETCH:
    case ID_EXTRUDE: // extrusion references a sketch, also convert the sketch
      std::cout << "Primitive " << node_name << " type "; // TODO: implement handler
      std::cout << internal.idb_type << " not yet supported" << std::endl; // TODO: implement handler
      break;
    default:
      std::cout << "Primitive " << node_name << " type "; // TODO: implement handler
      std::cout << internal.idb_type << " not recognized" << std::endl; // TODO: implement handler
      break;
    }

    on_error(!valid_solid, new Graphics_error(error_location(),
                                              Graphics_error::general,
                                              error()));
  }


  end_error_block();

 error_handler:
  delete reader->m_IO_error;
  reader->m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  goto exit_point;

 exit_point:
  if (need_internal_free)
  {
    rt_db_free_internal(&internal);
  }
  return;
}

//
// BRL-CAD -> Tovero conversion functions
//

// Documentation for BRL-CAD BOT (bag of triangles) triangular mesh structure

// struct rt_bot_internal
// {
//  uint32_t magic;
//  unsigned char mode;        // RT_BOT_SURFACE, RT_BOT_SOLID, RT_BOT_PLATE,
//                             // RT_BOT_PLATE_NOCOS (unsupported in Tovero)
//  unsigned char orientation; // RT_BOT_UNORIENTED, RT_BOT_CCW, RT_BOT_CW
//  unsigned char bot_flags;   // flags, (indicates surface normals available, for example):
//                             // RT_BOT_HAS_SURFACE_NORMALS -- this primitive may have surface normals
//                             //                               at each face vertex
//                             // RT_BOT_USE_NORMALS -- use the surface normals if they exist
//                             // RT_BOT_USE_FLOATS -- use the single precision version of "tri_specific"
//                             //                      during prep, ignored by Tovero
//  size_t num_vertices;
//  size_t num_faces;
//  int *faces;                // array of ints for faces [num_faces*3]
//  fastf_t *vertices;         // array of floats for vertices [num_vertices*3]
//  fastf_t *thickness;        // array of plate mode thicknesses (corresponds to array of faces)
//                             // for RT_BOT_PLATE
//                             // (null for modes RT_BOT_SURFACE and RT_BOT_SOLID)
//  struct bu_bitv *face_mode; // a flag for each face indicating thickness is appended to
//                             // hit point in ray direction (if bit is set), otherwise thickness is
//                             // centered about hit point (null for modes RT_BOT_SURFACE and RT_BOT_SOLID).
//  size_t num_normals;
//  fastf_t *normals;          // array of unit surface normals [num_normals*3]
//  size_t num_face_normals;   // current size of the face_normals array below (number of faces in the array)
//  int *face_normals;         // array of indices into the "normals" array, one per face vertex
//                             // [num_face_normals*3]
//  void *tie;                 // triangle index, ignored by Tovero
// };

bool BC_database_reader::mf_make_BOT(const Conversion_struct& conversion, Error_param& return_error)
{
  precondition(conversion.solid
               && (conversion.solid_ID == ID_BOT)
               && ("" != conversion.solid_name)
               && !return_error()
               && !m_node_mapper->find_node_by_name(conversion.solid_name));

  bool return_value = false;

  start_error_block();

  const struct rt_bot_internal* BC_BOT = static_cast<const rt_bot_internal*>(conversion.solid);
  RT_BOT_CK_MAGIC(BC_BOT);

  Triangle_face::Triangle_direction_type triangle_direction = Triangle_face::direction_none;
  switch (BC_BOT->orientation)
  {
  case RT_BOT_UNORIENTED:
    break;
  case RT_BOT_CCW:
    triangle_direction = Triangle_face::direction_counterclockwise;
    break;
  case RT_BOT_CW:
    triangle_direction = Triangle_face::direction_clockwise;
    break;
  default:
    {
      stringstream diagnostic;
      diagnostic << "invalid orientation for BOT: " << BC_BOT->orientation;
      on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                               Graphics_error::validation,
                                                               diagnostic.str()));
      m_logger << Logger::warning << diagnostic.str() << endl;
    }
    break;
  }

  // convert vertices
  vector<Point> vertices;
  for (size_t i = 0; i < BC_BOT->num_vertices; ++i)
  {
    const size_t index = i * 3;
    vertices.push_back(Point(BC_BOT->vertices[index] * *m_distance_conversion_factor,
                             BC_BOT->vertices[index + 1] * *m_distance_conversion_factor,
                             BC_BOT->vertices[index + 2] * *m_distance_conversion_factor));

  }
  // convert faces
  vector<Triangle_face> faces;
  for (size_t i = 0; i < BC_BOT->num_faces; ++i)
  {
    const size_t index = i * 3;
    faces.push_back(Triangle_face(BC_BOT->faces[index],
                                  BC_BOT->faces[index + 1],
                                  BC_BOT->faces[index + 2]));
  }
  // convert normals
  vector<Vector> normals;
  for (size_t i = 0; i < BC_BOT->num_normals; ++i)
  {
    const size_t index = i * 3;
    normals.push_back(Vector(BC_BOT->normals[index] * *m_distance_conversion_factor,
                             BC_BOT->normals[index + 1] * *m_distance_conversion_factor,
                             BC_BOT->normals[index + 2] * *m_distance_conversion_factor));
  }
  vector<Triangle_face> face_normals;
  for (size_t i = 0; i < BC_BOT->num_face_normals; ++i)
  {
    const size_t index = i * 3;
    face_normals.push_back(Triangle_face(BC_BOT->face_normals[index],
                                         BC_BOT->face_normals[index + 1],
                                         BC_BOT->face_normals[index + 2]));
  }
  const bool has_surface_normals = (BC_BOT->bot_flags & RT_BOT_HAS_SURFACE_NORMALS);
  const bool use_surface_normals = (BC_BOT->bot_flags & RT_BOT_USE_NORMALS);

  Triangle_mesh* mesh;
  switch (BC_BOT->mode)
  {
  case RT_BOT_SOLID:
    {
      mesh = new Triangle_mesh(triangle_direction,
                               vertices,
                               faces,
                               normals,
                               face_normals,
                               has_surface_normals,
                               use_surface_normals);
    }
    break;
  case RT_BOT_SURFACE:
      mesh = new Boundary_triangle_mesh(triangle_direction,
                                        vertices,
                                        faces,
                                        normals,
                                        face_normals,
                                        has_surface_normals,
                                        use_surface_normals);
    break;
  case RT_BOT_PLATE:
    {
      // convert thicknesses
      vector<Distance> thicknesses;
      for (size_t i = 0; i < BC_BOT->num_faces; ++i)
      {
        Distance thickness = BC_BOT->thickness[i] * *m_distance_conversion_factor;
        thicknesses.push_back(thickness);
      }
      // convert thickness modes
      vector<bool> append_thicknesses;
      for (size_t i = 0; i < BC_BOT->num_faces; ++i)
      {
        append_thicknesses.push_back(BC_BOT->face_mode->bits[i]);
      }
      mesh = new Plate_triangle_mesh(triangle_direction,
                                     vertices,
                                     faces,
                                     normals,
                                     face_normals,
                                     has_surface_normals,
                                     use_surface_normals,
                                     thicknesses,
                                     append_thicknesses);
    }
    break;
  default:
    {
      stringstream diagnostic;
      diagnostic << "invalid mode for BOT: " << BC_BOT->mode;
      diagnostic << " defaulting to " << RT_BOT_SOLID;
      on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                               Graphics_error::validation,
                                                               diagnostic.str()));
      m_logger << Logger::warning << diagnostic.str() << endl;
      // if only warning, default to RT_BOT_SOLID
      mesh = new Triangle_mesh(triangle_direction,
                               vertices,
                               faces,
                               normals,
                               face_normals,
                               has_surface_normals,
                               use_surface_normals);
    }
    break;
  }

  // validate
  Error_param error;
  const bool valid = mesh->is_valid(m_tolerances, error);
  if (!valid)
  {
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::validation,
                                                             error()));
    string diagnostic = error()->error_dictionary()[Error::diagnostic_error_key];
    m_logger << Logger::warning << diagnostic << endl;
  }

  Solid* solid_to_add = mesh;
  mf_add_solid(conversion.solid_name, solid_to_add);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

// Documentation for BRL-CAD ellipsoid structure:
//
// struct rt_ell_internal  {
//     uint32_t magic;
//     point_t     v; // center point (vertex)
//     vect_t      a; // axis 1 vector (basis 1, mutally perpendicular to b and c)
//     vect_t      b; // axis 2 vector (basis 2, mutally perpendicular to a and c)
//     vect_t      c; // axis 3 vector (basis 3, mutally perpendicular to a and b)
// };

bool BC_database_reader::mf_make_ellipsoid(const Conversion_struct& conversion, Error_param& return_error)
{
  precondition(conversion.solid
               && ((conversion.solid_ID == ID_ELL) || (conversion.solid_ID == ID_SPH))
               && ("" != conversion.solid_name)
               && !return_error()
               && !m_node_mapper->find_node_by_name(conversion.solid_name));

  bool return_value = false;

  start_error_block();

  const struct rt_ell_internal* BC_ellipsoid = static_cast<const rt_ell_internal*>(conversion.solid);
  RT_ELL_CK_MAGIC(BC_ellipsoid);

  const Point V(BC_ellipsoid->v[X] * *m_distance_conversion_factor,
                BC_ellipsoid->v[Y] * *m_distance_conversion_factor,
                BC_ellipsoid->v[Z] * *m_distance_conversion_factor);
  const Vector a(BC_ellipsoid->a[X] * *m_distance_conversion_factor,
                 BC_ellipsoid->a[Y] * *m_distance_conversion_factor,
                 BC_ellipsoid->a[Z] * *m_distance_conversion_factor);
  const Vector b(BC_ellipsoid->b[X] * *m_distance_conversion_factor,
                 BC_ellipsoid->b[Y] * *m_distance_conversion_factor,
                 BC_ellipsoid->b[Z] * *m_distance_conversion_factor);
  const Vector c(BC_ellipsoid->c[X] * *m_distance_conversion_factor,
                 BC_ellipsoid->c[Y] * *m_distance_conversion_factor,
                 BC_ellipsoid->c[Z] * *m_distance_conversion_factor);

  Ellipsoid* ellipsoid = new Ellipsoid(V,
                                       a,
                                       b,
                                       c);

  // validate
  Error_param error;
  const bool valid = ellipsoid->is_valid(m_tolerances, error);
  if (!valid)
  {
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::validation,
                                                             error()));
    string diagnostic = error()->error_dictionary()[Error::diagnostic_error_key];
    m_logger << Logger::warning << diagnostic << endl;
  }

  Solid* solid_to_add = ellipsoid;
  if (m_simplify_solids)
  {
    // determine if there is a more specialized solid to use
    solid_to_add = &ellipsoid->specialize(m_tolerances);
    if (solid_to_add != ellipsoid)
    {
      ellipsoid->reference();
      ellipsoid->unreference();
    }
  }
  mf_add_solid(conversion.solid_name, solid_to_add);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

// Documentation for BRL-CAD half-space structure:

// struct rt_half_internal  {
//  uint32_t magic;
//  plane_t     eqn;
// };

bool BC_database_reader::mf_make_half_space(const Conversion_struct& conversion, Error_param& return_error)
{
  precondition(conversion.solid && (conversion.solid_ID == ID_HALF) && ("" != conversion.solid_name)
               && !return_error()
               && !m_node_mapper->find_node_by_name(conversion.solid_name));

  bool return_value = false;

  start_error_block();

  const struct rt_half_internal* BC_half_space = static_cast<const rt_half_internal *>(conversion.solid);
  RT_HALF_CK_MAGIC(BC_half_space);

  const Unit_vector normal(Unitless::from_value(BC_half_space->eqn[X]),
                           Unitless::from_value(BC_half_space->eqn[Y]),
                           Unitless::from_value(BC_half_space->eqn[Z]));
  const Distance distance = BC_half_space->eqn[W] * *m_distance_conversion_factor;

  Half_space* half_space = new Half_space(normal, distance);

  // validate
  Error_param error;
  const bool valid = half_space->is_valid(m_tolerances, error);
  if (!valid)
  {
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::validation,
                                                             error()));
    string diagnostic = error()->error_dictionary()[Error::diagnostic_error_key];
    m_logger << Logger::warning << diagnostic << endl;
  }
  mf_add_solid(conversion.solid_name, half_space);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

// Documentation for BRL-CAD ARB structure:
//
// struct rt_arb_internal  {
//     uint32_t magic;
//     point_t     pt[8]; // vertices
// };

bool BC_database_reader::mf_make_polyhedron(const Conversion_struct& conversion,
                                            Error_param& return_error)
{
  precondition(conversion.solid
               && (conversion.solid_ID == ID_ARB8)
               && ("" != conversion.solid_name)
               && !return_error()
               && !m_node_mapper->find_node_by_name(conversion.solid_name));

  bool return_value = false;

  start_error_block();

  struct rt_arb_internal *BC_ARB = static_cast<struct rt_arb_internal *>(conversion.solid);
  RT_ARB_CK_MAGIC(BC_ARB);

  vector<Point> vertices;
  for (size_t i = 0; i < 8; ++i)
  {
    vertices.push_back(Point(BC_ARB->pt[i][X] * *m_distance_conversion_factor,
                             BC_ARB->pt[i][Y] * *m_distance_conversion_factor,
                             BC_ARB->pt[i][Z] * *m_distance_conversion_factor));
  }

  Error_param error;

  Polyhedron* polyhedron = new Polyhedron(vertices);

  // validate
  const bool valid = polyhedron->is_valid(m_tolerances, error);
  if (!valid)
  {
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::validation,
                                                             error()));
    string diagnostic = error()->error_dictionary()[Error::diagnostic_error_key];
    m_logger << Logger::warning << diagnostic << endl;
  }

  Solid* solid_to_add = polyhedron;
  if (m_simplify_solids)
  {
    // determine if there is a more specialized solid to use
    solid_to_add = &polyhedron->specialize(m_tolerances);
    if (solid_to_add != polyhedron)
    {
      polyhedron->reference();
      polyhedron->unreference();
    }
  }
  mf_add_solid(conversion.solid_name, solid_to_add);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

// Documentation for BRL-CAD torus structure:
//
// struct rt_tor_internal  {
//     uint32_t magic;
//     point_t     v;   // center point (vertex)
//     vect_t      h;   // normal unit vector (hole points this way)
//     fastf_t     r_h; // radius in H direction (r2, major radius)
//     fastf_t     r_a; // radius in A direction (r1, minor radius)
//                      // REMAINING ELEMENTS PROVIDED BY IMPORT, UNUSED BY EXPORT (ignored by Tovero)
//     vect_t      a;   // r_a length (ignored by Tovero)
//     vect_t      b;   // r_b length (ignored by Tovero)
//     fastf_t     r_b; // radius in B direction (typically == r_a, ignored by Tovero)
// };

bool BC_database_reader::mf_make_torus(const Conversion_struct& conversion, Error_param& return_error)
{
  precondition(conversion.solid && (conversion.solid_ID == ID_TOR) && ("" != conversion.solid_name)
               && !return_error()
               && !m_node_mapper->find_node_by_name(conversion.solid_name));

  bool return_value = false;

  start_error_block();

  const struct rt_tor_internal* BC_torus = static_cast<const rt_tor_internal *>(conversion.solid);
  RT_TOR_CK_MAGIC(BC_torus);

  const Point V(BC_torus->v[X] * *m_distance_conversion_factor,
                BC_torus->v[Y] * *m_distance_conversion_factor,
                BC_torus->v[Z] * *m_distance_conversion_factor);
  const Vector normal_vector(BC_torus->h[X] * *m_distance_conversion_factor,
                             BC_torus->h[Y] * *m_distance_conversion_factor,
                             BC_torus->h[Z] * *m_distance_conversion_factor);
  Unit_vector normal;
  normal_vector.normalize(normal);
  const Distance major_radius = BC_torus->r_h * *m_distance_conversion_factor;
  const Distance minor_radius = BC_torus->r_a * *m_distance_conversion_factor;

  Torus* torus = new Torus(V,
                           normal,
                           major_radius,
                           minor_radius);

  // validate
  Error_param error;
  const bool valid = torus->is_valid(m_tolerances, error);
  if (!valid)
  {
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::validation,
                                                             error()));
    string diagnostic = error()->error_dictionary()[Error::diagnostic_error_key];
    m_logger << Logger::warning << diagnostic << endl;
  }
  mf_add_solid(conversion.solid_name, torus);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

// Documentation for BRL-CAD truncated general cone structure:
//
//  struct rt_tgc_internal {
//      uint32_t magic;
//      vect_t      v; // base point (vertex)
//      vect_t      h; // height vector
//      vect_t      a; // base vector a
//      vect_t      b; // base vector b
//      vect_t      c; // top vector c (in the same direction as a)
//      vect_t      d; // top vector d (in the same direction as b)
//  };

bool BC_database_reader::mf_make_general_cone(const Conversion_struct& conversion,
                                              Error_param& return_error)
{
  precondition(conversion.solid
               && ((conversion.solid_ID == ID_REC) || (conversion.solid_ID == ID_TGC))
               && ("" != conversion.solid_name)
               && !return_error()
               && !m_node_mapper->find_node_by_name(conversion.solid_name));

  bool return_value = false;

  start_error_block();

  const struct rt_tgc_internal* cylinder = static_cast<const rt_tgc_internal *>(conversion.solid);
  RT_TGC_CK_MAGIC(cylinder);

  const Point V(cylinder->v[X] * *m_distance_conversion_factor,
                cylinder->v[Y] * *m_distance_conversion_factor,
                cylinder->v[Z] * *m_distance_conversion_factor);
  const Vector h(cylinder->h[X] * *m_distance_conversion_factor,
                 cylinder->h[Y] * *m_distance_conversion_factor,
                 cylinder->h[Z] * *m_distance_conversion_factor);
  const Vector a(cylinder->a[X] * *m_distance_conversion_factor,
                 cylinder->a[Y] * *m_distance_conversion_factor,
                 cylinder->a[Z] * *m_distance_conversion_factor);
  const Vector b(cylinder->b[X] * *m_distance_conversion_factor,
                 cylinder->b[Y] * *m_distance_conversion_factor,
                 cylinder->b[Z] * *m_distance_conversion_factor);
  const Vector c(cylinder->c[X] * *m_distance_conversion_factor,
                 cylinder->c[Y] * *m_distance_conversion_factor,
                 cylinder->c[Z] * *m_distance_conversion_factor);
  const Vector d(cylinder->d[X] * *m_distance_conversion_factor,
                 cylinder->d[Y] * *m_distance_conversion_factor,
                 cylinder->d[Z] * *m_distance_conversion_factor);

  General_cone* cone = new General_cone(V,
                                        h,
                                        a,
                                        b,
                                        c,
                                        d);

  // validate
  Error_param error;
  const bool valid = cone->is_valid(m_tolerances, error);
  if (!valid)
  {
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::validation,
                                                             error()));
    string diagnostic = error()->error_dictionary()[Error::diagnostic_error_key];
    m_logger << Logger::warning << diagnostic << endl;
  }

  Solid* solid_to_add = cone;
  if (m_simplify_solids)
  {
    // determine if there is a more specialized solid to use
    solid_to_add = &cone->specialize(m_tolerances);
    if (solid_to_add != cone)
    {
      cone->reference();
      cone->unreference();
    }
  }
  mf_add_solid(conversion.solid_name, solid_to_add);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

void BC_database_reader::mf_add_solid(const string& solid_name, Solid* solid)
{
  solid->reference();
  solid->set_name(solid_name);
  string added_solid_name;
  m_node_mapper->add_node(solid,
                          solid->solid_class(),
                          added_solid_name);
  solid->set_name(added_solid_name);
}
