/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include "path.h"

@implementation Path

-(Path *) init
{
    char *list[] = {"degree", "resolution"};

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->degree = 2;
    self->resolution = 10;
    self->length = 0;
    self->samples = NULL;

    return self;
}

-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "degree")) {
	lua_pushnumber (_L, self->degree);
    } else if (!xstrcmp(k, "resolution")) {
	lua_pushnumber (_L, self->resolution);
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "degree")) {
	self->degree = lua_tointeger (_L, 3);
    } else if (!xstrcmp(k, "resolution")) {
	self->resolution = lua_tointeger (_L, 3);
    } else {
	[super set];
    }

    if (self->size > self->degree && (lua_isnumber (_L, 2) ||
				      !xstrcmp(k, "degree") ||
				      !xstrcmp(k, "resolution"))) {
	int i, j, k, l;
	float t, c;

	/* printf ("%d, %d, %d\n", self->size, self->degree, self->resolution); */

    	self->length = (self->size - 1) / self->degree * self->resolution + 1;
    	self->samples = realloc (self->samples, 3 * self->length *
    				                sizeof (float));

    	for (i = 0 ; i < (self->size - 1) / self->degree ; i += 1) {
    	    float (*p)[3];

    	    p = (float (*)[3])&self->vertices[3 * self->degree * i];

    	    for (j = 0, t = 0;
    		 j <= self->resolution;
    		 j += 1, t += 1.0 / self->resolution) {
    		self->samples[3 * (i * self->resolution + j)] = 0;
    		self->samples[3 * (i * self->resolution + j) + 1] = 0;
    		self->samples[3 * (i * self->resolution + j) + 2] = 0;
		
    		for (k = 0, c = pow (1 - t, self->degree);
    		     k <= self->degree;
    		     c *= (self->degree - k) * t / (k + 1) / (1 - t), k += 1) {
		    
    		    for (l = 0 ; l < 3 ; l += 1) {
    			self->samples[3 * (i * self->resolution + j) + l] +=
    			    c * p[k][l];
    		    }
    		}
    	    }
    	}
    }
}

-(void) traversePass: (int)pass
{
    if (pass == 2) {
	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixd ([self matrix]);

	glUseProgramObjectARB(0);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_BLEND);
	glLineWidth ([self width]);
	glDepthMask (GL_FALSE);

	glColor4fv([self color]);

	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 0, self->samples);
	glDrawArrays(GL_LINE_STRIP, 0, self->length);
	glDisableClientState(GL_VERTEX_ARRAY);
    
	glDepthMask (GL_TRUE);
	glDisable(GL_BLEND);
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_DEPTH_TEST);
    
	glMatrixMode (GL_MODELVIEW);
	glPopMatrix();
    }
    
    [super traversePass: pass];
}

@end
