/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ode/ode.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "hinge.h"
#include "body.h"

@implementation Hinge

-(Joint *) init
{
    char *list[] = {
	"anchor", "axis", "motor", "state", "stops", "tolerance"
    };

    self->joint = dJointCreateHinge (_WORLD, NULL);
    
    self->anchor[0] = 0;
    self->anchor[1] = 0;
    self->anchor[2] = 0;
    
    self->stops[0] = -dInfinity;
    self->stops[1] = dInfinity;

    self->axis[0] = 0;
    self->axis[1] = 0;
    self->axis[2] = 1;

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    return self;
}

-(void) update
{
    [super update];
    
    /* Axis and anchor should be set after the
       joint has been attached. */
	
    dJointSetHingeAxis ([self joint],
			self->axis[0],
			self->axis[1],
			self->axis[2]);

    dJointSetHingeAnchor ([self joint],
			  self->anchor[0],
			  self->anchor[1],
			  self->anchor[2]);
}

-(void) get
{
    const char *k;
    int i;
    
    k = lua_tostring (_L, -1);

    if (!xstrcmp(k, "anchor")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->anchor[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "axis")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->axis[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "motor")) {
        lua_newtable (_L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->motor[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "stops")) {
        lua_newtable (_L);

        lua_newtable (_L);
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->stops[i]);
            lua_rawseti (_L, -2, i + 1);
        }
	lua_rawseti (_L, -2, 1);
	
        lua_newtable (_L);
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->hardness[i]);
            lua_rawseti (_L, -2, i + 1);
        }
	lua_rawseti (_L, -2, 2);

	lua_pushnumber (_L, self->bounce);
	lua_rawseti (_L, -2, 3);
    } else if (!xstrcmp(k, "tolerance")) {
	lua_pushnumber (_L, self->tolerance);
    } else if (!xstrcmp(k, "state")) {
	dReal state[2];

	if ([self joint]) {
	    state[0] = dJointGetHingeAngle ([self joint]);
	    state[1] = dJointGetHingeAngleRate ([self joint]);

	    lua_newtable (_L);
        
	    for(i = 0 ; i < 2 ; i += 1) {
		lua_pushnumber (_L, state[i]);
		lua_rawseti (_L, -2, i + 1);
	    }
	} else {
	    lua_pushnil (_L);
	}
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;
    int i;

    k = lua_tostring (_L, -2);

    if (!xstrcmp(k, "axis")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->axis[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }

	    dSafeNormalize3 (self->axis);
	    dJointSetHingeAxis ([self joint],
				self->axis[0],
				self->axis[1],
				self->axis[2]);
        }
    } else if (!xstrcmp(k, "anchor")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->anchor[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }

	    dJointSetHingeAnchor ([self joint],
				  self->anchor[0],
				  self->anchor[1],
				  self->anchor[2]);
        }
    } else if (!xstrcmp(k, "motor")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->motor[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }

	    dJointSetHingeParam ([self joint], dParamVel, self->motor[0]);
	    dJointSetHingeParam ([self joint], dParamFMax, self->motor[1]);
	}
    } else if (!xstrcmp(k, "stops")) {
        if(lua_istable (_L, 3)) {
	    lua_rawgeti (_L, 3, 1);
	    for(i = 0 ; i < 2 ; i += 1) {
		lua_rawgeti (_L, -1, i + 1);
		
		self->stops[i] = lua_tonumber (_L, -1);
		
		lua_pop (_L, 1);
	    }
	    lua_pop (_L, 1);

	    lua_rawgeti (_L, 3, 2);
	    for(i = 0 ; i < 2 ; i += 1) {
		lua_rawgeti (_L, -1, i + 1);

		self->hardness[i] = lua_tonumber (_L, -1);

		lua_pop (_L, 1);
	    }
	    lua_pop (_L, 1);
	
	    lua_rawgeti (_L, 3, 3);
	    self->bounce = lua_tonumber (_L, -1);
	    lua_pop (_L, 1);

	    /* Resetting the stops makes sure that lo remains
	       smaller than hi between calls. */
	
	    dJointSetHingeParam ([self joint], dParamLoStop, -dInfinity);
	    dJointSetHingeParam ([self joint], dParamHiStop, dInfinity);
	    
	    dJointSetHingeParam ([self joint], dParamLoStop,
				 self->stops[0]);
	    dJointSetHingeParam ([self joint], dParamHiStop,
				 self->stops[1]);
	    dJointSetHingeParam ([self joint], dParamStopCFM,
				 self->hardness[0]);
	    dJointSetHingeParam ([self joint], dParamStopERP,
				 self->hardness[1]);
	    dJointSetHingeParam ([self joint], dParamBounce, self->bounce);
	}
    } else if (!xstrcmp(k, "tolerance")) {
	self->tolerance = lua_tonumber (_L, 3);

	dJointSetHingeParam ([self joint], dParamCFM, self->tolerance);
    } else {
	[super set];
    }

}

-(void) traversePass: (int)pass
{
    if (pass == 2 && self->debug) {
	dBodyID a, b;
	const dReal *q;
	dVector3 p, x;

	a = dJointGetBody ([self joint], 0);
	b = dJointGetBody ([self joint], 1);

	assert (a || b);

	dJointGetHingeAnchor ([self joint], p);
	dJointGetHingeAxis ([self joint], x);

	glUseProgramObjectARB(0);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_POINT_SMOOTH);
	glEnable(GL_BLEND);
	glDepthMask (GL_FALSE);

	glPushMatrix();
	glTranslatef(p[0], p[1], p[2]);
	
	/* Draw the dots. */

	glPointSize (3);

	glColor3f(0, 0, 1);

	glBegin (GL_POINTS);
	glVertex3f (x[0], x[1], x[2]);
	glEnd ();	
	
	/* Draw the lines. */

	glLineWidth (1);

	glColor3f(0, 0, 1);

	glBegin (GL_LINES);
	glVertex3f (0, 0, 0);
	glVertex3f (x[0], x[1], x[2]);

	glEnd ();	

	glPopMatrix ();

	if (b) {
	    q = dBodyGetPosition (b);

	    /* Draw one arm. */
	    
	    glColor3f(1, 0, 0);
	    glLineWidth (1);

	    glBegin (GL_LINES);
	    glVertex3f (p[0], p[1], p[2]);
	    glVertex3f (q[0], q[1], q[2]);
	    glEnd();
	}

	if (a) {
	    q = dBodyGetPosition (a);

	    /* Draw the other arm and anchor. */
	
	    glColor3f(0, 1, 0);
	    glLineWidth (3);

	    glBegin (GL_LINES);
	    glVertex3f (p[0], p[1], p[2]);
	    glVertex3f (q[0], q[1], q[2]);
	    glEnd();

	    glPointSize (8);
	    
	    glBegin(GL_POINTS);
	    glVertex3f (p[0], p[1], p[2]);
	    glEnd();
	}

	glDepthMask (GL_TRUE);
	glDisable(GL_BLEND);
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_POINT_SMOOTH);
	glDisable(GL_DEPTH_TEST);
    }
    
    [super traversePass: pass];
}

@end
