/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <ctype.h>

#include "event.h"

@implementation Event

-(Event *) init
{
    char *list[] = {
	"buttonpress", "buttonrelease", "keypress", "keyrelease",
	"motion", "scroll"
    };
    
    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->buttonpress = LUA_REFNIL;
    self->buttonrelease = LUA_REFNIL;
    self->keypress = LUA_REFNIL;
    self->keyrelease = LUA_REFNIL;
    self->motion = LUA_REFNIL;
    self->scroll = LUA_REFNIL;

    return self;
}

-(void) input
{
    GdkEvent *event;
    int i, h_0, h_1;
	
    event = gdk_event_get ();
    assert (event);

    h_0 = lua_gettop (_L) + 1;

    /* Prepare the bindings based on the event type. */
	
    if (event->type == GDK_BUTTON_PRESS ||
	event->type == GDK_BUTTON_RELEASE) {
	if (event->type == GDK_BUTTON_PRESS) {
	    lua_rawgeti (_L, LUA_REGISTRYINDEX, self->buttonpress);
	} else {
	    lua_rawgeti (_L, LUA_REGISTRYINDEX, self->buttonrelease);
	}

	lua_pushnumber (_L, ((GdkEventButton *)event)->button);
	lua_pushnumber (_L, ((GdkEventButton *)event)->x);
	lua_pushnumber (_L, ((GdkEventButton *)event)->y);
    } else if (event->type == GDK_SCROLL) {
	lua_rawgeti (_L, LUA_REGISTRYINDEX, self->scroll);

	if (((GdkEventScroll *)event)->direction == GDK_SCROLL_UP) {
	    lua_pushstring (_L, "up");
	} else if (((GdkEventScroll *)event)->direction == GDK_SCROLL_DOWN) {
	    lua_pushstring (_L, "down");
	} else if (((GdkEventScroll *)event)->direction == GDK_SCROLL_LEFT) {
	    lua_pushstring (_L, "left");
	} else if (((GdkEventScroll *)event)->direction == GDK_SCROLL_RIGHT) {
	    lua_pushstring (_L, "right");
	}

	lua_pushnumber (_L, ((GdkEventScroll *)event)->x);
	lua_pushnumber (_L, ((GdkEventScroll *)event)->y);	
    } else if (event->type == GDK_MOTION_NOTIFY) {
	int i;
	    
	lua_rawgeti (_L, LUA_REGISTRYINDEX, motion);

	for (i = 0;
	     (1 << i) - 1 < ((GdkEventMotion *)event)->state >> 8;
	     i += 1);

	if (i > 0) {
	    lua_pushnumber (_L, i);
	} else {
	    lua_pushnil(_L);
	}
		
	lua_pushnumber (_L, ((GdkEventMotion *)event)->x);
	lua_pushnumber (_L, ((GdkEventMotion *)event)->y);
    } else if (event->type == GDK_KEY_PRESS ||
	       event->type == GDK_KEY_RELEASE) {
	char *s;
	int i;

	if (event->type == GDK_KEY_PRESS) {
	    lua_rawgeti (_L, LUA_REGISTRYINDEX, self->keypress);
	} else {
	    lua_rawgeti (_L, LUA_REGISTRYINDEX, self->keyrelease);
	}
		
	s = strdup(gdk_keyval_name (((GdkEventKey *)event)->keyval));

	/* Convert all strings to lowercase to
	   make things more consistent. */
		    
	for (i = 0 ; i < strlen (s) ; i += 1) {
	    s[i] = tolower(s[i]);
	}

	lua_pushstring (_L, s);
	free(s);
    }

    /* Fire the relevant bindings if this event
       was interesting. */
	
    if (lua_gettop (_L) >= h_0) {
	lua_getfield (_L, LUA_REGISTRYINDEX, "userdata");
	lua_pushlightuserdata (_L, self);
	lua_gettable (_L, -2);
	lua_insert (_L, h_0 + 1);
	lua_pop (_L, 1);

	h_1 = lua_gettop (_L);
	    
	if (lua_isfunction (_L, h_0)) {
	    luaX_call (_L, h_1 - h_0, 0);
	} else if (lua_istable (_L, h_0)) {
	    lua_pushnil (_L);
	    while (lua_next (_L, h_0) != 0) {
		for (i = 0 ; i < h_1 - h_0 ; i += 1) {
		    lua_pushvalue (_L, h_0 + i + 1);
		}

		luaX_call (_L, h_1 - h_0, 0);
	    }
	}

	lua_settop (_L, h_0 - 1);
    }

    /* Put the event back for others to see. */
	
    gdk_event_put (event);
    gdk_event_free (event);
    
    [super input];
}

-(void) get
{
    const char *k;

    k = lua_tostring(_L, 2);
    
    if (!xstrcmp(k, "buttonpress")) {
	lua_rawgeti(_L, LUA_REGISTRYINDEX, self->buttonpress);
    } else if (!xstrcmp(k, "buttonrelease")) {
	lua_rawgeti(_L, LUA_REGISTRYINDEX, self->buttonrelease);
    } else if (!xstrcmp(k, "keypress")) {
	lua_rawgeti(_L, LUA_REGISTRYINDEX, self->keypress);
    } else if (!xstrcmp(k, "keyrelease")) {
	lua_rawgeti(_L, LUA_REGISTRYINDEX, self->keyrelease);
    } else if (!xstrcmp(k, "motion")) {
	lua_rawgeti(_L, LUA_REGISTRYINDEX, motion);
    } else if (!xstrcmp(k, "scroll")) {
	lua_rawgeti(_L, LUA_REGISTRYINDEX, scroll);
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;

    k = lua_tostring(_L, 2);
    
    if (!xstrcmp(k, "buttonpress")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, self->buttonpress);
	self->buttonpress = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "buttonrelease")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, self->buttonrelease);
	self->buttonrelease = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "keypress")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, self->keypress);
	self->keypress = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "keyrelease")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, self->keyrelease);
	self->keyrelease = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "motion")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, motion);
	motion = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "scroll")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, scroll);
	scroll = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else {
	[super set];
    }
}

@end
