# Copyright (C) 2012, 2013 Julian Marchant <onpon4@riseup.net>
# 
# This file is part of the Pygame SGE.
# 
# The Pygame SGE is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# The Pygame SGE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public License
# along with the Pygame SGE.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import division
from __future__ import absolute_import
from __future__ import print_function
from __future__ import unicode_literals

import sge


__all__ = ['View']


class View(object):

    """Class for room views.

    This class controls what the player sees in a room at any given
    time.  Multiple views can exist in a room, and this can be used to
    create a split-screen effect.

    .. attribute:: x

       The horizontal position of the view in the room.  When set, if it
       brings the view outside of the room it is in, it will be
       re-adjusted so that the view is completely inside the room.

    .. attribute:: y

       The vertical position of the view in the room.  When set, if it
       brings the view outside of the room it is in, it will be
       re-adjusted so that the view is completely inside the room.

    .. attribute:: xport

       The horizontal position of the view port on the window.

    .. attribute:: yport

       The vertical position of the view port on the window.

    .. attribute:: width

       The width of the view.  When set, if it results in the view being
       outside of the room it is in, :attr:`x` will be adjusted so that
       the view is completely inside the room.

    .. attribute:: height

       The height of the view.  When set, if it results in the view
       being outside the room it is in, :attr:`y` will be adjusted so
       that the view is completely inside the room.

    .. attribute:: wport

       The width of the view port.  Set to :const:`None` to make it the
       same as :attr:`width`.  If this value differs from :attr:`width`,
       the image will be horizontally scaled so that it fills the port.

    .. attribute:: hport

       The height of the view port.  Set to :const:`None` to make it the
       same as :attr:`height`.  If this value differs from
       :attr:`height`, the image will be vertically scaled so that it
       fills the port.

    """

    @property
    def x(self):
        return self._x

    @x.setter
    def x(self, value):
        self._x = value
        self._limit()

    @property
    def y(self):
        return self._y

    @y.setter
    def y(self, value):
        self._y = value
        self._limit()

    @property
    def width(self):
        return self._width

    @width.setter
    def width(self, value):
        self._width = value
        self._limit()

    @property
    def height(self):
        return self._height

    @height.setter
    def height(self, value):
        self._height = value
        self._limit()

    @property
    def wport(self):
        return self._wport if self._wport is not None else self.width

    @wport.setter
    def wport(self, value):
        self._wport = value

    @property
    def hport(self):
        return self._hport if self._hport is not None else self.height

    @hport.setter
    def hport(self, value):
        self._hport = value

    def __init__(self, x, y, xport=0, yport=0, width=None, height=None,
                 wport=None, hport=None):
        """Constructor method.

        Arguments:

        - ``width`` -- The width of the view.  If set to :const:`None`,
          it will become ``sge.game.width - xport``.
        - ``height`` -- The height of the view.  If set to
          :const:`None`, it will become ``sge.game.height - yport``.

        All other arugments set the respective initial attributes of the
        view.  See the documentation for :class:`sge.View` for more
        information.

        """
        self._x = x
        self._y = y
        self.xport = xport
        self.yport = yport
        self._width = width if width else sge.game.width - xport
        self._height = height if height else sge.game.height - yport
        self.wport = wport
        self.hport = hport
        self._start_x = self.x
        self._start_y = self.y
        self._start_xport = self.xport
        self._start_yport = self.yport
        self._start_width = self.width
        self._start_height = self.height
        self._start_wport = self._wport
        self._start_hport = self._hport

    def _limit(self):
        if sge.game.current_room is not None:
            if self.x < 0:
                self._x = 0
            elif self.x + self.width > sge.game.current_room.width:
                self._x = sge.game.current_room.width - self.width

            if self.y < 0:
                self._y = 0
            elif self.y + self.height > sge.game.current_room.height:
                self._y = sge.game.current_room.height - self.height

    def _reset(self):
        # Reset the view to its original state.
        self.x = self._start_x
        self.y = self._start_y
        self.xport = self._start_xport
        self.yport = self._start_yport
        self.width = self._start_width
        self.height = self._start_height
        self.wport = self._start_wport
        self.hport = self._start_hport
