/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* startjob.c -- Start a job */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <lwc_log.h>
#include <net_utils.h>
#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_hosts.h>
#include <request.h>
#include <lib_functions.h>
#include <startjob.h>

/*
 * Error callback function for the sql_host_get_main_row() function
 */
static void
sql_host_get_main_row_error_logger (	void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while retrieving the agent host details"));
	}
}


/*
 * Callback for the SQL function which retrieves the command arguments
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to log an
 *         error message)
 *   1 --> Stop the sql function.  No arguments for this job
 */
static int
argument_callback (void *obj, row_item_t *row)
{
	startjob_request_ptr request = (startjob_request_ptr) obj;

#if HAVE_ASSERT_H
	assert (request != NULL);
#endif

	if (row == NULL) {
		return 0;
	}

	/*
	 * row[0] --> Position
	 * row[1] --> Argument
	 */

	if (sql_row_item2ll (&(row[0])) == -1) {
		/* No argument for the command */
		return 1;
	}
	return startjob_request_add_argument (request,
					"1", 1, /* Ignored */
					row[1].value_string, row[1].len);
}


/*
 * Start a job
 *
 * Return:
 *   0 --> No error.  Job successfully started on the remote client.  If not
 *         NULL, out_message contains a reply message sent by the agent.  It
 *         may be NULL if the agent didn't send back a message.  out_message
 *         must be freed by the caller by free()
 *   1 --> The agent cannot start the job.  If not NULL, out_message
 *         contains a reply message sent by the agent.  It may be NULL if the
 *         agent didn't send back a message.  out_message must be freed by
 *         the caller by free()
 *   2 --> The job does not have an associated command.  out_message is not
 *         changed
 *  -1 --> Error (a message is logged using lwc_writeLog()).  If not NULL,
 *         out_message contains an error message.  It may be NULL if there
 *         is not enought memory to allocate a buffer for this message.
 *         out_message must be freed by the caller by free()
 */
int
startjob (	int workload_date, unsigned long long int job_id,
		char **out_message)
{
	lwc_LL *hierarchy_list;
	row_item_t value;
	unsigned long long int host_id;
	char *s, *err_msg = NULL;
	int ret;
	startjob_request_ptr request;
	row_item_t *host_details;
	schedwi_BIO *b;

	if (request_job_init (	workload_date, job_id,
				&host_id, &request, &hierarchy_list) != 0)
	{
		if (out_message != NULL) {
			err_msg =
			_("Server error: cannot retrieve job parameters");
			*out_message = (char *)malloc (schedwi_strlen (err_msg)
								+ 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}

	/* Get the command */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_command_s", "command",
				&value, &err_msg) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (LOG_CRIT,
				_("Job %lld: cannot retrieve the command"),
				job_id);
			if (out_message != NULL) {
				err_msg =
				_("Server error: cannot retrieve job command");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
		}
		return -1;
	}

	if (	   value.type != RES_END
		&& value.value_string != NULL
		&& value.value_string[0] != '\0'
		&& (	   value.value_string[0] != '-'
			|| value.value_string[1] != '\0'))
	{
		if (startjob_request_add_command (request,
						value.value_string,
						value.len) != 0)
		{
			free (value.value_string);
			lwc_delLL (	hierarchy_list,
					(void (*)(const void *))free);
			startjob_request_destroy (request);
			if (out_message != NULL) {
				err_msg =
				_("Server error: cannot retrieve job command");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
			return -1;
		}
		free (value.value_string);
	}
	else {
		/* No command, the job is then considered completed */
		if (value.type != RES_END && value.value_string != NULL) {
			free (value.value_string);
		}
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		return 2;
	}

	/* Get the username */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_username_s", "username",
				&value, &err_msg) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (LOG_CRIT,
				_("Job %lld: cannot retrieve the username"),
				job_id);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the username");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		if (startjob_request_add_username (request,
						value.value_string,
						value.len) != 0)
		{
			if (value.value_string != NULL) {
				free (value.value_string);
			}
			lwc_delLL (	hierarchy_list,
					(void (*)(const void *))free);
			startjob_request_destroy (request);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the username");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
			return -1;
		}
		if (value.value_string != NULL) {
			free (value.value_string);
		}
	}
	else {
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		lwc_writeLog (	LOG_CRIT,
				_("Job %lld: cannot retrieve the username"),
				job_id);
		if (out_message != NULL) {
			err_msg =
			_("Server error: cannot retrieve the username");
			*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}

	/* Get the output file name */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_file_out_s", "file_out",
				&value, &err_msg) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (LOG_CRIT,
				_("Job %lld: cannot retrieve the output file"),
				job_id);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the output file");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		if (startjob_request_add_fileout (request,
						value.value_string,
						value.len) != 0)
		{
			if (value.value_string != NULL) {
				free (value.value_string);
			}
			lwc_delLL (	hierarchy_list,
					(void (*)(const void *))free);
			startjob_request_destroy (request);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the output file");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
			return -1;
		}
		if (value.value_string != NULL) {
			free (value.value_string);
		}
	}

	/* Get the error file name */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_file_err_s", "file_err",
				&value, &err_msg) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (LOG_CRIT,
				_("Job %lld: cannot retrieve the error file"),
				job_id);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the error file");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		if (startjob_request_add_filerr (request,
						value.value_string,
						value.len) != 0)
		{
			if (value.value_string != NULL) {
				free (value.value_string);
			}
			lwc_delLL (	hierarchy_list,
					(void (*)(const void *))free);
			startjob_request_destroy (request);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the error file");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
			return -1;
		}
		if (value.value_string != NULL) {
			free (value.value_string);
		}
	}

	/* Get the Linux Control Group name */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_control_group_s", "control_group",
				&value, &err_msg) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (LOG_CRIT,
			_("Job %lld: cannot retrieve the control group"),
				job_id);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the control group");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		if (startjob_request_add_cgroup (request,
						value.value_string,
						value.len) != 0)
		{
			if (value.value_string != NULL) {
				free (value.value_string);
			}
			lwc_delLL (	hierarchy_list,
					(void (*)(const void *))free);
			startjob_request_destroy (request);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the control group");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
			return -1;
		}
		if (value.value_string != NULL) {
			free (value.value_string);
		}
	}

	/* Get the load environment flag */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_loadenv_s", "loadenv",
				&value, &err_msg) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (LOG_CRIT,
	_("Job %lld: cannot retrieve the `load user environment' flag"),
				job_id);
			if (out_message != NULL) {
				err_msg =
	_("Server error: cannot retrieve the `load user environment' flag");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		s = (sql_row_item2ll(&value) == 0) ? "0" : "1";
		if (startjob_request_add_loadenv (request, s, 1) != 0) {
			lwc_delLL (	hierarchy_list,
					(void (*)(const void *))free);
			startjob_request_destroy (request);
			if (out_message != NULL) {
				err_msg =
	_("Server error: cannot retrieve the `load user environment' flag");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
			return -1;
		}
	}

	/* Get the command arguments */
	ret = get_job_parameters_multi (workload_date, hierarchy_list,
				"job_arguments_s", "position,argument",
				"position",
				argument_callback, request, &err_msg);
	if (ret != 0 && ret != 2) {
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (LOG_CRIT,
			_("Job %lld: cannot retrieve the command arguments"),
				job_id);
			if (out_message != NULL) {
				err_msg =
		_("Server error: cannot retrieve the command arguments");
				*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
				if (*out_message != NULL) {
					strcpy (*out_message, err_msg);
				}
			}
		}
		return -1;
	}

	lwc_delLL (hierarchy_list, (void (*)(const void *))free);

	/* Get the host details */
	ret= sql_host_get_main_row (host_id, &host_details,
					sql_host_get_main_row_error_logger,
					NULL);
	if (ret != 0) {
		startjob_request_destroy (request);
		if (out_message != NULL) {
			err_msg =
		_("Server error: cannot retrieve the remote agent details");
			*out_message = (char *)malloc (
					schedwi_strlen (err_msg) + 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}

	/* Establish the connection to the agent */
	b = net_client (host_details[2].value_string,  /* TCP port */
			host_details[1].value_string,  /* Agent host name */
			(char)sql_row_item2ll (&(host_details[3])),
			host_details[4].value_string,  /* Certificate */
			(unsigned int)(host_details[4].len));
	sql_free_row (host_details);
	if (b == NULL) {
		startjob_request_destroy (request);
		if (out_message != NULL) {
			err_msg =
_("Server error: cannot establish the connection with the remote agent");
			*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}

	/* Send the request */
	if (net_write_request (request, b, "runjob") != 0) {
		net_close (b);
		startjob_request_destroy (request);
		if (out_message != NULL) {
			err_msg =
	_("Server error: cannot send the request to the remote agent");
			*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}
	startjob_request_destroy (request);

	/* Read the result */
	ret = net_read_result (b, out_message);
	net_close (b);
	return ret;
}

/*-----------------============== End Of File ==============-----------------*/
