/* Schedwi
   Copyright (C) 2007-2011 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwica.c -- Add or update an agent host in the database */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) schedwi_strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif


#include <sql_common.h>
#include <utils.h>
#include <conf.h>
#include <conf_srv.h>
#include <net_utils.h>
#include <net_utils_sock.h>
#include <net_utils_ssl.h>
#include <sql_hosts.h>
#include <module.h>
#include <lib_functions.h>
#include <cert_utils.h>
#include <cert_ca.h>

static const char *configuration_file = NULL;
static const char *ca_crt, *ca_key, *ca_crl;
static const char *client_cert_dir, *client_csr_dir, *client_savecsr_dir;
static const char *descr;
static char force, quiet;


/*
 * Print a help message to stderr
 */
static void
help (const char * prog_name)
{
	int i;
	char * msg[] = {
N_("Sign a Schedwi agent certificate and define it in the database or"),
N_("define a Schedwi agent with a custom-made certificate."),
N_("This program must be run on the same host as the Schedwi server."),
"",
#if HAVE_GETOPT_LONG
N_("  -c, --config=FILE  use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -l, --list         list pending signing requests"),
N_("  -s, --sign         sign requests of specified hosts"),
N_("  -S, --sign-all     sign all pending requests"),
N_("  -r, --clean        clean out all certs or csrs for the specified hosts"),
N_("  -d, --descr=TEXT   also add an agent description in the database"),
N_("  -a, --agent=NAME   agent host name.  If not set, the host name in the"),
N_("                     certificate (in the Alternate Name or in the"),
N_("                     Common Name field) will be used"),
N_("  -f, --force        force the database upload even if there is a"),
N_("                     mismatch between the provided host name and the one"),
N_("                     in the certificate file (in the Alternate Name or"),
N_("                     in the Common Name field) or if the agent is"),
N_("                     already defined in the database"),
N_("  -q, --quiet        suppress all normal output"),
N_("  -h, --help         display this help and exit"),
N_("  -V, --version      print version information and exit"),
#else /* HAVE_GETOPT_LONG */
N_("  -c FILE            use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -l                 list pending signing requests"),
N_("  -s                 sign requests of specified hosts"),
N_("  -S                 sign all pending requests"),
N_("  -r                 clean out all certs or csrs for the specified hosts"),
N_("  -d DESCRIPTION     also add an agent description in the database"),
N_("  -a HOSTNAME[:PORT] agent host name.  If not set, the host name in the"),
N_("                     certificate (in the Alternate Name or in the"),
N_("                     Common Name field) will be used"),
N_("  -f                 force the database upload even if there is a"),
N_("                     mismatch between the provided host name and the one"),
N_("                     in the certificate file (in the Alternate Name or"),
N_("                     in the Common Name field) or if the agent is"),
N_("                     already defined in the database"),
N_("  -q                 suppress all normal output"),
N_("  -h                 display this help and exit"),
N_("  -V                 print version information and exit"),
#endif /* ! HAVE_GETOPT_LONG */
"",
"~",
N_("Exit status is 1 on error (0 means no error)."),
NULL
	};

#if HAVE_ASSERT_H
	assert (prog_name != NULL && configuration_file != NULL);
#endif

	fprintf (stderr, _("Usage: %s [OPTION]... -l|-S\n"), prog_name);
	fprintf (stderr, _("   OR  %s [OPTION]... -s|-r HOSTNAME[:PORT]...\n"),
			prog_name);
	fprintf (stderr, _("   OR  %s [OPTION]... CERTFILE\n"),
			prog_name);

	for (i = 0; msg[i] != NULL; i++) {
		if (msg[i][0] == '~') {
			fputs (	_("The default configuration file is "),
				stderr);
			fputs (configuration_file, stderr);
		}
		else {
			fputs ((msg[i][0] == '\0')? msg[i]: _(msg[i]), stderr);
		}
		fputc ('\n', stderr);
	}

	fputc ('\n', stderr);
	fputs (_("Report bugs to "), stderr);
	fputs (PACKAGE_BUGREPORT, stderr);
	fputc ('\n', stderr);
}


/*
 * Error callback function for the sql_host_replace() function
 */
static void
sql_host_replace_error (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		fputs (msg, stderr);
		fputc ('\n', stderr);
	}
	else {
		fputs (
		_("Database error while saving the agent host details\n"),
			stderr);
	}
}


/*
 * Save the agent host details in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been printed on stderr
 */
static int
certificate_to_database (	const char *filename,
				const char *hostname,
				const char *port,
				char ssl)
{
	const char *h[2];
	char *cert, *client_name;
	int ret;
	size_t l;
	gnutls_x509_crt_t cert_obj;

#if HAVE_ASSERT_H
	assert (hostname != NULL && port != NULL);
#endif

	/* Check the port */
	if (check_port (port) != 0) {
		fprintf (stderr, _("Cannot get the port number for `%s'\n"),
			port);
		return -1;
	}

	cert = client_name = NULL;
	/* Retrieve the certificate */
	if (filename != NULL) {
		cert = read_file (filename, &l);
		if (cert == NULL) {
			return -1;
		}
		cert_obj = load_cert_from_mem (cert, l, filename);
		if (cert_obj == NULL) {
			free (cert);
			return -1;
		}

		/* Get the agent hostname from the certificate */
		client_name = get_hostname_from_cert (cert_obj);
		if (client_name == NULL) {
			gnutls_x509_crt_deinit (cert_obj);
			free (cert);
			fprintf (stderr,
	_("Cannot retrieve the agent hostname from the certificate %s\n"),
				filename);
			return -1;
		}

		/* Check that the provided agent name matches the cert */
		if (force == 0 && hostname != NULL) {
			h[0] = hostname;
			h[1] = NULL;
			if (my_gnutls_x509_crt_check_hostname (cert_obj, CERT,
								h) == 0)
			{
				gnutls_x509_crt_deinit (cert_obj);
				free (cert);
				fprintf (stderr,
	_("Hostname `%s' is different from the one in certicate (`%s')\n"),
					hostname, client_name);
				free (client_name);
				return -1;
			}
		}
		gnutls_x509_crt_deinit (cert_obj);
	}

	/* Update the database */
	ret = sql_host_replace ((hostname != NULL) ? hostname: client_name,
				port,
				ssl,
				(filename != NULL) ? cert : "",
				descr,
				sql_host_replace_error, NULL);
	if (filename != NULL) {
		free (cert);
		free (client_name);
	}
	return (ret == 0) ? 0: -1;
}


/*
 * Move the provided CSR file in the SSLAgentSaveRequestDir directory
 *
 * Return:
 *   The new filename (to be freed by free()) OR
 *   NULL in case of error (a message has been printed)
 */
static char *
save_csr (const char *filecsr)
{
	char *s;
	const char *base;

#if HAVE_ASSERT_H
	assert (filecsr != NULL);
#endif

	base = base_name (filecsr);

	s = (char *) malloc (	  schedwi_strlen (client_savecsr_dir)
				+ schedwi_strlen (DIR_SEP)
				+ schedwi_strlen (base)
				+ 1);
	if (s == NULL) {
		fputs (_("Memory allocation error\n"), stderr);
		return NULL;
	}

	strcpy (s, client_savecsr_dir);
	strcat (s, DIR_SEP);
	strcat (s, base);

	if (rename (filecsr, s) != 0) {
		fprintf (stderr, _("Cannot move %s to %s: %s\n"),
			filecsr, s, strerror (errno));
		free (s);
		return NULL;
	}
	return s;
}


/*
 * Return the full path in `outfilename' of the `basedir/nameext1' file or
 * `basedir/nameext2' file.
 *
 * Return:
 *    0 --> No such file
 *    1 --> A file with the `ext1' extension has been found. `outfilename' is
 *          set and must be freed by free()
 *    2 --> A file with the `ext2' extension has been found. `outfilename' is
 *          set and must be freed by free()
 *   -1 --> Memory allocation error
 */
static int
find_file (	const char *basedir, const char *name, const char *port,
		const char *ext1, const char *ext2,
		char **outfilename)
{
	char *s;
	size_t l_base, l;


#if HAVE_ASSERT_H
	assert (   basedir != NULL && name != NULL && port != NULL
		&& outfilename != NULL);
#endif

	l_base =  schedwi_strlen (basedir)
		+ schedwi_strlen (DIR_SEP)
		+ schedwi_strlen (port)
		+ 1 /* Separator `_' */
		+ schedwi_strlen (name);
	l =	  l_base
		+ ((ext1 != NULL) ? schedwi_strlen (ext1) : 0)
		+ ((ext2 != NULL) ? schedwi_strlen (ext2) : 0)
		+ 1;
	s = (char *) malloc (l);
	if (s == NULL) {
		return -1;
	}

	strcpy (s, basedir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);

	if (ext1 != NULL) {
		strcpy (s + l_base, ext1);
		if (access (s, R_OK) == 0) {
			*outfilename = s;
			return 1;
		}
	}

	if (ext2 != NULL) {
		strcpy (s + l_base, ext2);
		if (access (s, R_OK) == 0) {
			*outfilename = s;
			return 2;
		}
	}

	if (ext1 == NULL && ext2 == NULL) {
		if (access (s, R_OK) == 0) {
			*outfilename = s;
			return 1;
		}
	}

	free (s);
	return 0;
}


/*
 * Return the full path in `outfilename' of a CSR file.
 *
 * Return:
 *    0 --> No such file
 *    1 --> A CSR file has been found (path in `outfilename' - use free())
 *    2 --> A file (no SSL) has been found (path in `outfilename' - use free())
 *   -1 --> Memory allocation error
 */
static int
find_csr (const char *name, const char *port, char **outfilename)
{
#if HAVE_ASSERT_H
	assert (name != NULL && outfilename != NULL);
#endif

	return find_file (	client_csr_dir, name, port,
				CSR_EXTENSION, NOSSL_EXTENSION,
				outfilename);
}


/*
 * Sign a request and generate the associated CRT file
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been printed)
 *  -2 --> No CSR file found (a message has been printed)
 *  -3 --> Already defined in DB (a message has been printed)
 */
static int
sign_csr (const char *agent_name)
{
	char **names;
	char *filecsr, *filenewcsr, *filecrt, *port, *name, *p;
	row_item_t *host_details;
	int ret, typecsr;

	if (parse_hostname (agent_name, &name, &port) != 0) {
		return -1;
	}

	/* Get the file name of the pending request */
	typecsr = find_csr (name, port, &filecsr);
	if (typecsr < 0) {
		free (name);
		free (port);
		fputs (_("Memory allocation error\n"), stderr);
		return -1;
	}
	if (typecsr == 0) {
		free (name);
		free (port);
		fprintf (stderr, _("No pending request found for %s\n"),
				agent_name);
		return -2;
	}

	/* Retrieve all the names of the agent from the DNS */
	if (build_client_names_from_name (agent_name, &names, &p) != 0) {
		free (filecsr);
		free (name);
		free (port);
		return -1;
	}

	/* Retrieve the agent details from the database */
	host_details = NULL;
	ret = sql_host_get_main_row_names (	names, p, &host_details,
						NULL, NULL);
	free (p);
	destroy_client_names (names);
	sql_free_row (host_details);
	if (ret == -1) {
		free (filecsr);
		free (name);
		free (port);
		fputs (_("Memory allocation error\n"), stderr);
		return -1;
	}
	if (ret == 0 && force == 0) { /* The agent already exists in the DB */
		free (filecsr);
		free (name);
		free (port);
		fprintf (stderr,
	_("%s is already defined in the database. Use -f to overwrite.\n"),
				agent_name);
		return -3;
	}

	/* Compose the certificate file name */
	filecrt = (char *) malloc (	  schedwi_strlen (client_cert_dir)
					+ schedwi_strlen (DIR_SEP)
					+ schedwi_strlen (port)
					+ 1 /* Separator `_' */
					+ schedwi_strlen (name)
					+ schedwi_strlen (CRT_EXTENSION)
					+ schedwi_strlen (NOSSL_EXTENSION));
	if (filecrt == NULL) {
		free (filecsr);
		free (name);
		free (port);
		fputs (_("Memory allocation error\n"), stderr);
		return -1;
	}
	strcpy (filecrt, client_cert_dir);
	strcat (filecrt, DIR_SEP);
	strcat (filecrt, port);
	strcat (filecrt, "_");
	strcat (filecrt, name);
	strcat (filecrt, (typecsr == 1) ? CRT_EXTENSION : NOSSL_EXTENSION);

	/* Move the request file in the SSLAgentSaveRequestDir directory */
	filenewcsr = save_csr (filecsr);
	if (filenewcsr == NULL) {
		free (filecrt);
		free (filecsr);
		free (name);
		free (port);
		return -1;
	}

	/* Sign the request and build the certificate */
	if (typecsr == 1) {
		/* SSL */
		if (sign_request (ca_key, ca_crt, filenewcsr, filecrt) != 0) {
			/* Try to move back the CSR file */
			rename (filenewcsr, filecsr);
			free (filenewcsr);
			free (filecrt);
			free (filecsr);
			free (name);
			free (port);
			return -1;
		}
	}
	else {
		/* No SSL */
		FILE *f;
		f = fopen (filecrt, "w");
		if (f == NULL) {
			fprintf (stderr, _("Cannot create file %s: %s\n"),
					filecrt, strerror (errno));
			/* Try to move back the CSR file */
			rename (filenewcsr, filecsr);
			free (filenewcsr);
			free (filecrt);
			free (filecsr);
			free (name);
			free (port);
			return -1;
		}
		fclose (f);
	}

	/* Add the agent in the DB */
	if (certificate_to_database (	NULL, name, port,
					(typecsr == 1) ? 1: 0) != 0)
	{
		/* Try to move back the CSR file and remove the CRT file */
		rename (filenewcsr, filecsr);
		my_unlink (filecrt);
		free (filenewcsr);
		free (filecrt);
		free (filecsr);
		free (name);
		free (port);
		return -1;
	}

	free (filenewcsr);
	free (filecrt);
	free (filecsr);
	free (name);
	free (port);
	return 0;
}


/*
 * Remove all the certificates/requests/pending requests for the
 * provided agent. If the certificate exists it is revoked.
 *
 * Return:
 *   0 --> No error
 *   1 --> No file found for the specified agent
 *  -1 --> Error (a message has been printed)
 *  -3 --> Still defined in DB (a message has been printed)
 */
static int
revoke_crt (const char *agent_name)
{
	char **names;
	char *s, *port, *name;
	char found;
	row_item_t *host_details;
	int ret;


	/* Retrieve all the names of the agent from the DNS */
	if (build_client_names_from_name (agent_name, &names, &port) != 0) {
		return -1;
	}

	/* Retrieve the agent details from the database */
	host_details = NULL;
	ret = sql_host_get_main_row_names (	names, port, &host_details,
						NULL, NULL);
	free (port);
	destroy_client_names (names);
	if (ret == -1) {
		fputs (_("Memory allocation error\n"), stderr);
		return -1;
	}
	/* The agent is already defined in the DB */
	if (	   ret == 0
		&& sql_row_item2ll (&(host_details[3])) != 0
		&& force == 0)
	{
		sql_free_row (host_details);
		fprintf (stderr,
_("%s is still defined in the database. You can use the -f option\nto bypass but be aware that the agent will have to request a new certificate.\n"),
				agent_name);
		return -3;
	}
	sql_free_row (host_details);

	/* Split the agent_name */
	if (parse_hostname (agent_name, &name, &port) != 0) {
		return -1;
	}

	found = 0;
	s =  (char *) malloc (	  schedwi_strlen (client_cert_dir)
				+ schedwi_strlen (client_csr_dir)
				+ schedwi_strlen (client_savecsr_dir)
				+ schedwi_strlen (DIR_SEP)
				+ schedwi_strlen (port)
				+ 1  /* Separator `_' */
				+ schedwi_strlen (name)
				+ schedwi_strlen (CSR_EXTENSION)
				+ schedwi_strlen (NOSSL_EXTENSION)
				+ schedwi_strlen (CRT_EXTENSION));
	if (s == NULL) {
		free (name);
		free (port);
		fputs (_("Memory allocation error\n"), stderr);
		return -1;
	}

	/* Remove from SSLAgentRequestDir */
	strcpy (s, client_csr_dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, CSR_EXTENSION);
	if (access (s, F_OK) == 0) {
		my_unlink (s);
		found = 1;
	}

	strcpy (s, client_csr_dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, NOSSL_EXTENSION);
	if (access (s, F_OK) == 0) {
		my_unlink (s);
		found = 1;
	}

	strcpy (s, client_csr_dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, CRT_EXTENSION);
	if (access (s, F_OK) == 0) {
		my_unlink (s);
		found = 1;
	}

	/* Remove from  SSLAgentSaveRequestDir */
	strcpy (s, client_savecsr_dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, CSR_EXTENSION);
	if (access (s, F_OK) == 0) {
		my_unlink (s);
		found = 1;
	}

	strcpy (s, client_savecsr_dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, NOSSL_EXTENSION);
	if (access (s, F_OK) == 0) {
		my_unlink (s);
		found = 1;
	}

	strcpy (s, client_savecsr_dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, CRT_EXTENSION);
	if (access (s, F_OK) == 0) {
		my_unlink (s);
		found = 1;
	}

	/* Remove from  SSLAgentCertificateDir (but revoke the cert first) */
	strcpy (s, client_cert_dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, CSR_EXTENSION);
	if (access (s, F_OK) == 0) {
		my_unlink (s);
		found = 1;
	}

	strcpy (s, client_cert_dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, NOSSL_EXTENSION);
	if (access (s, F_OK) == 0) {
		my_unlink (s);
		found = 1;
	}

	/* Revoke the certificate before removing the file */
	strcpy (s, client_cert_dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, CRT_EXTENSION);
	if (access (s, F_OK) == 0) {
		if (revoke_certificate (ca_key, ca_crt, s, ca_crl) != 0) {
			free (s);
			free (name);
			free (port);
			return -1;
		}
		my_unlink (s);
		found = 1;
	}

	free (s);
	free (name);
	free (port);

	if (found == 0) {
		fprintf (stderr, _("Nothing find for %s\n"), agent_name);
		return 1;
	}
	return 0;
}


/*
 * Callbacks for the sign-all (-S) option
 */
static int
sign_ssl (const char *file, int idx)
{
	char *s, *u;
	int ret;

#if HAVE_ASSERT_H
	assert (file != NULL);
#endif

	s = (char *) malloc (idx + 1);
	if (s == NULL) {
		fputs (_("Memory allocation error\n"), stderr);
		return -1;
	}

	u = strchr (file, '_');
	if (u == NULL) {
		strncpy (s, file, idx);
		s[idx] = '\0';
	}
	else {
		strncpy (s, u + 1, idx - (u - file + 1));
		strcpy (s + idx - (u - file + 1), ":");
		strncpy (s + idx - (u - file + 1) + 1, file, u - file);
		s[idx] = '\0';
	}
	ret = sign_csr (s);
	if (ret == 0 && quiet == 0) {
		printf (_("%s successfully signed and defined\n"), s);
	}
	free (s);
	return (ret == -1) ? -1 : 0;
}

static int
sign_end (int nb)
{
	switch (nb) {
		case 0:
			fputs (_("No pending request\n"), stdout);
			break;
		case 1:
			fputs (_("\n1 request signed\n"), stdout);
			break;
		default:
			fprintf (stdout, _("\n%d requests signed\n"), nb);
			break;
	}
	return 0;
}


/*
 * Callbacks for the list (-l) option
 */
static int
print_ssl (const char *file, int idx)
{
	char *s;

#if HAVE_ASSERT_H
	assert (file != NULL);
#endif

	s = strchr (file, '_');
	if (s == NULL) {
		fwrite (file, 1, idx , stdout);
	}
	else {
		fwrite (s + 1, 1, idx - (s - file + 1), stdout);
		/* Only print the port if it's not the default one */
		if (schedwi_strncmp (	file, SCHEDWI_DEFAULT_AGTPORT,
					s - file) != 0)
		{
			fwrite (":", 1, 1, stdout);
			fwrite (file, 1, s - file, stdout);
		}
	}
	fputs (_("\t(SSL)\n"), stdout);
	return 0;
}

static int
print_nossl (const char *file, int idx)
{
	char *s;

#if HAVE_ASSERT_H
	assert (file != NULL);
#endif

	s = strchr (file, '_');
	if (s == NULL) {
		fwrite (file, 1, idx , stdout);
	}
	else {
		fwrite (s + 1, 1, idx - (s - file + 1), stdout);
		/* Only print the port if it's not the default one */
		if (schedwi_strncmp (	file, SCHEDWI_DEFAULT_AGTPORT,
					s - file) != 0)
		{
			fwrite (":", 1, 1, stdout);
			fwrite (file, 1, s - file, stdout);
		}
	}
	fputs (_("\t(NO SSL)\n"), stdout);
	return 0;
}

static int
print_end (int nb)
{
	switch (nb) {
		case 0:
			fputs (_("No pending request\n"), stdout);
			break;
		case 1:
			fputs (_("\n1 pending request\n"), stdout);
			break;
		default:
			fprintf (stdout, _("\n%d pending requests\n"), nb);
			break;
	}
	return 0;
}


/*
 * For each pending request in the `SSLAgentCertificateDir' directory, call
 * the `callback_ssl' function for SSL requests or `callback_nossl' for not
 * SSL requests.  At the end, call the `callback_end' function with the
 * number of pending requests detected.
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (a message has been printed)
 *  -2 --> One of the callback function returned a value != 0
 */
static int
csr_list (	int (*callback_ssl)(const char *, int),
		int (*callback_nossl)(const char *, int),
		int (*callback_end)(int))
{
	DIR *d;
	struct dirent *f;
	size_t ext_ssl_length, ext_nossl_length, name_length;
	int idx, nb;

	d = opendir (client_csr_dir);
	if (d == NULL) {
		fprintf (stderr,
		_("Cannot open the pending request directory %s: opendir: %s"),
			client_csr_dir, strerror (errno));
		return -1;
	}

	ext_ssl_length   = schedwi_strlen (CSR_EXTENSION);
	ext_nossl_length = schedwi_strlen (NOSSL_EXTENSION);
	nb = 0;

	while ((f = readdir (d)) != NULL) {
		name_length = schedwi_strlen (f->d_name);
		idx = name_length - ext_ssl_length;
		if (idx > 0 && strcmp (f->d_name + idx, CSR_EXTENSION) == 0) {
			if (	   callback_ssl != NULL
				&& callback_ssl (f->d_name, idx) != 0)
			{
				closedir (d);
				return -2;
			}
			nb++;
			continue;
		}
		idx = name_length - ext_nossl_length;
		if (idx > 0 && strcmp (f->d_name + idx, NOSSL_EXTENSION) == 0)
		{
			if (	   callback_nossl != NULL
				&& callback_nossl (f->d_name, idx) != 0)
			{
				closedir (d);
				return -2;
			}
			nb++;
			continue;
		}
	}
	closedir (d);

	if (callback_end != NULL && callback_end (nb) != 0) {
		return -2;
	}

	return 0;
}


/*
 * Main function
 *
 * The exit code is 1 in case of error or 0 if killed (SIGTERM)
 */
int
main (int argc, char **argv)
{
	const char *server_key, *server_crt;
	const char *agent_host, *certificate_file;
	const char *prog_name;
	int ret, i;
	char *err_msg, *port, *name;
	char quick_random, list, sign, sign_all, clean, load_cert;

#if HAVE_GETOPT_LONG
	int option_index;
	struct option long_options[] =
	{
		{"help",       0, 0, 'h'},
		{"version",    0, 0, 'V'},
		{"config",     1, 0, 'c'},
		{"agent",      1, 0, 'a'},
		{"descr",      1, 0, 'd'},
		{"force",      0, 0, 'f'},
		{"list",       0, 0, 'l'},
		{"sign",       0, 0, 's'},
		{"sign-all",   0, 0, 'S'},
		{"clean",      0, 0, 'r'},
		{"quiet",      0, 0, 'q'},
		{0, 0, 0, 0}
	};
#endif

#if HAVE_SETLOCALE
	setlocale (LC_ALL, "");
#endif
#if HAVE_BINDTEXTDOMAIN
	bindtextdomain (PACKAGE, LOCALEDIR);
#endif
#if HAVE_TEXTDOMAIN
	textdomain (PACKAGE);
#endif


	/* Set default values for options */
	configuration_file = SCHEDWI_DEFAULT_CONFFILE_SRV;
	agent_host = descr = NULL;
	force = list = sign = sign_all = clean = quiet = 0;

	prog_name = base_name (argv[0]);

	/* Parse options */
	while (1) {
#if HAVE_GETOPT_LONG
		option_index = 0;
		ret = getopt_long (argc, argv, "hVc:a:d:flsSrq",
					long_options, &option_index);
#else
		ret = getopt (argc, argv, "hVc:a:d:flsSrq");
#endif

		if (ret == -1) {
			break;
		}

		switch (ret) {
			case 'h':
				help (prog_name);
				return 0;
			case 'V':
				version (prog_name);
				return 0;
			case 'c':
				configuration_file = optarg;
				break;
			case 'a':
				agent_host = optarg;
				break;
			case 'd':
				descr = optarg;
				break;
			case 'f':
				force = 1;
				break;
			case 'l':
				list = 1;
				break;
			case 's':
				sign = 1;
				break;
			case 'S':
				sign_all = 1;
				break;
			case 'r':
				clean = 1;
				break;
			case 'q':
				quiet = 1;
				break;
			default:
				help (prog_name);
				return 1;
		}
	}

	/* Missing parameters */
	if (optind >= argc && sign_all == 0 && list == 0) {
		if (sign == 1 || clean == 1) {
			fputs (	_("At least one host name is required\n"),
				stderr);
		}
		else {
			fputs (_("Certificate file name required\n"), stderr);
		}
		help (prog_name);
		return 1;
	}

	/* Too many parameters */
	if (	   optind + 1 != argc && sign == 0 && clean == 0
		&& sign_all == 0 && list == 0)
	{
		fputs (_("Too many parameters\n"), stderr);
		help (prog_name);
		return 1;
	}


	/*
	 * Initialize the module engine
	 */
	if (module_init () != 0) {
		return 1;
	}

	/*
	 * Read the configuration file
	 */
	ret = conf_init_srv (configuration_file);
	switch (ret) {
		case -1:
			fputs (_("Memory allocation error\n"), stderr);
			break;

		case -2:
			perror (configuration_file);
			break;
	}
	if (ret != 0) {
		module_exit ();
		return 1;
	}

	ret =  conf_get_param_string ("SSLCACertificateFile", &ca_crt);
	ret += conf_get_param_string ("SSLCACertificateKeyFile", &ca_key);
	ret += conf_get_param_string ("SSLCertificateFile", &server_crt);
	ret += conf_get_param_string ("SSLCACRLFile", &ca_crl);
	ret += conf_get_param_string ("SSLCertificateKeyFile", &server_key);
	ret += conf_get_param_bool ("SSLQuickRandom", &quick_random);

	ret += conf_get_param_string (	"SSLAgentCertificateDir",
					&client_cert_dir);
	ret += conf_get_param_string (	"SSLAgentRequestDir",
					&client_csr_dir);
	ret += conf_get_param_string (	"SSLAgentSaveRequestDir",
					&client_savecsr_dir);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	/*
	 * Initialize the GnuTLS layer
	 */
	if (net_init_gnutls (quick_random) != 0) {
		conf_destroy_srv ();
		module_exit ();
		return 1;
	}


	/*
	 * Initialize the network (only needed for the SSL init)
	 */
	if (net_init (	server_crt, server_key, ca_crt, ca_crl,
			quick_random) != 0)
	{
		net_destroy ();
		conf_destroy_srv ();
		module_exit ();
		return 1;
	}

	/*
	 * Database connection
	 */
	err_msg = NULL;
	if (begin_sql (&err_msg) == NULL) {
		if (err_msg != NULL) {
			fprintf (stderr,
				_("Failed to connect to database: %s\n"),
				err_msg);
			free (err_msg);
		}
		else {
			fputs (_("Failed to connect to database\n"), stderr);
		}
		conf_destroy_srv ();
		module_exit ();
		net_destroy ();
		return 1;
	}

	load_cert = 1;

	/* Remove and revoke certificates */
	if (clean == 1) {
		for (i = optind; i < argc; i++) {
			ret = revoke_crt (argv[i]);
			if (ret < 0) {
				end_sql ();
				conf_destroy_srv ();
				module_exit ();
				net_destroy ();
				return 1;
			}
			if (ret == 0 && quiet == 0) {
				printf (
				_("%s SSL details successfully removed\n"),
					argv[i]);
			}
		}
		load_cert = 0;
	}

	/* List all pending requests */
	if (list == 1) {
		if (csr_list (	print_ssl, print_nossl,
				(quiet == 0) ? print_end : NULL) != 0)
		{
			end_sql ();
			conf_destroy_srv ();
			module_exit ();
			net_destroy ();
			return 1;
		}
		load_cert = 0;
	}

	/* Sign pending requests */
	if (sign == 1) {
		for (i = optind; i < argc; i++) {
			ret = sign_csr (argv[i]);
			if (ret < 0) {
				end_sql ();
				conf_destroy_srv ();
				module_exit ();
				net_destroy ();
				return 1;
			}
			if (ret == 0 && quiet == 0) {
				printf (
				_("%s successfully signed and defined\n"),
					argv[i]);
			}
		}
		load_cert = 0;
	}

	/* Sign all pending requests */
	if (sign_all == 1) {
		if (csr_list (	sign_ssl, sign_ssl,
				(quiet == 0) ? sign_end : NULL) != 0)
		{
			end_sql ();
			conf_destroy_srv ();
			module_exit ();
			net_destroy ();
			return 1;
		}
		load_cert = 0;
	}


	/* Add/Update the new agent */
	if (load_cert != 0) {
		certificate_file = argv[optind];
		if (parse_hostname (agent_host, &name, &port) != 0) {
			end_sql ();
			conf_destroy_srv ();
			module_exit ();
			net_destroy ();
			return 1;
		}
		if (certificate_to_database (	certificate_file, name, port,
						1) != 0)
		{
			free (name);
			free (port);
			end_sql ();
			conf_destroy_srv ();
			module_exit ();
			net_destroy ();
			return 1;
		}
		free (name);
		free (port);
	}

	end_sql ();
	conf_destroy_srv ();
	module_exit ();
	net_destroy ();
	return 0;
}

/*------------------------======= End Of File =======------------------------*/
