/* Schedwi
   Copyright (C) 2007-2011 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* reg_getmycrt.c -- Parse the `getmycrt' request */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <utils.h>
#include <net_utils_sock.h>
#include <cert_ca.h>
#include <cert_utils.h>
#include <conf.h>
#include <lwc_log.h>
#include <JSON_parser.h>
#include <reg_error_code.h>
#include <reg_getmycrt.h>


enum steps {
	S_START,
	S_OUTER_ARRAY,
	S_OUTER_ARRAY_MODULE,
	S_DATA,
	S_KEY_PORT,
	S_END
};

struct json_cb {
	enum steps step;
	char error;
	int port;
};


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	struct json_cb *s = (struct json_cb *)ctx;

	switch (type) {
		case JSON_T_ARRAY_BEGIN:
			if (s->step == S_START) {
				s->step = S_OUTER_ARRAY;
			}
			break;
		case JSON_T_STRING:
			if (s->step == S_OUTER_ARRAY) {
				s->step = S_OUTER_ARRAY_MODULE;
			}
			break;
		case JSON_T_OBJECT_BEGIN:
			if (s->step == S_OUTER_ARRAY_MODULE) {
				s->step = S_DATA;
			}
			break;
		case JSON_T_KEY:
			if (s->step == S_DATA) {
				if (schedwi_strcasecmp (value->vu.str.value,
							"port") == 0)
				{
					s->step = S_KEY_PORT;
				}
			}
			break;
		case JSON_T_INTEGER:
			if (s->step == S_KEY_PORT) {
				s->port = value->vu.integer_value;
				s->step = S_DATA;
			}
			break;
		case JSON_T_OBJECT_END:
			if (s->step == S_DATA) {
				s->step = S_END;
			}
			break;
	}

	return 1;
}


static void
send_error (int sock, const char *error_message, error_reason_t reason)
{
	const char *s;

	s = "{ \"success\" : false, \"reason\" : ";
	net_write_sock (sock, s, schedwi_strlen (s));
	s = reg_error_to_string (reason);
	net_write_sock (sock, s, schedwi_strlen (s));
	s = ", \"data\" : \"";
	net_write_sock (sock, s, schedwi_strlen (s));
	if (error_message != NULL) {
		net_write_sock (sock, error_message,
				schedwi_strlen (error_message));
	}
	else {
		s = _("Cannot get the agent certificate");
		net_write_sock (sock, s, schedwi_strlen (s));
	}
	s = "\" }";
	net_write_sock (sock, s, schedwi_strlen (s));
}


/*
 * Check if the file `dir'/`port'_`name'`ext' exists
 *
 * Return:
 *   0 --> file exists (`outfilenames' contains the file name - use free())
 *   1 --> file does not exists
 *  -1 --> Error (a message has been logged)
 */
static int
check_file (	const char *dir, const char *name, 
		const char *port, const char *ext, char **outfilename)
{
	size_t l;
	char *s;

#if HAVE_ASSERT_H
	assert (dir != NULL && name != NULL && port != NULL && ext != NULL);
#endif

	l =	  schedwi_strlen (dir)
		+ schedwi_strlen (DIR_SEP)
		+ schedwi_strlen (port)
		+ 1 /* "_" separator */
		+ schedwi_strlen (name)
		+ schedwi_strlen (ext)
		+ 1;
	s  = (char *) malloc (l);
	if (s == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	strcpy (s, dir);
	strcat (s, DIR_SEP);
	strcat (s, port);
	strcat (s, "_");
	strcat (s, name);
	strcat (s, ext);
	if (access (s, F_OK) == 0) {
		if (outfilename != NULL) {
			*outfilename = s;
		}
		else {
			free (s);
		}
		return 0;
	}
	free (s);
	return 1;
}


int
reg_getmycrt (	int sock, const char *buff, ssize_t buff_len,
		const char *const *client_names)
{
	char *s, *serialized, *content;
	const char *client_cert_dir, *client_csr_dir;
	int ret, i;
	JSON_config config;
	struct JSON_parser_struct *jc;
	struct json_cb cb_data;
	char sport[7];
	unsigned int l_sport;


	ret =  conf_get_param_string (	"SSLAgentCertificateDir",
					&client_cert_dir);
	ret += conf_get_param_string (	"SSLAgentRequestDir",
					&client_csr_dir);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	if (client_names == NULL || client_names[0] == NULL) {
		send_error (	sock, _("Cannot get agent IP/name details"),
				REG_SYSTEM_ERROR);
		return -1;
	}

	/* JSON parser initialization */
	init_JSON_config (&config);
	cb_data.step                  = S_START;
	cb_data.error                 = 0;
	cb_data.port                  = atoi (SCHEDWI_DEFAULT_AGTPORT);
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &cb_data;

	jc = new_JSON_parser (&config);

	/* Parse the JSON string */
	for (i = 0; i < buff_len && cb_data.step != S_END; i++) {
		/*
		 * No need to check the return code as the JSON string has
		 * already been parsed once in net_parse_reg.c
		 */
		JSON_parser_char (jc, buff[i]);
		if (cb_data.error != 0) {
			delete_JSON_parser (jc);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			send_error (sock, NULL, REG_SYSTEM_ERROR);
			return -1;
		}
	}
	delete_JSON_parser (jc);

	l_sport = copy_ulltostr (cb_data.port, sport);

	/* First, check if the agent certificate is ready */
	for (i = 0; client_names[i] != NULL; i++) {
		/* SSL */
		ret = check_file (	client_cert_dir, client_names[i],
					sport, CRT_EXTENSION, &s);
		if (ret < 0) {
			send_error (sock, NULL, REG_SYSTEM_ERROR);
			return -1;
		}
		if (ret == 0) {
			content = read_file (s, NULL);
			free (s);
			if (content == NULL) {
				send_error (sock, NULL, REG_SYSTEM_ERROR);
				return -1;
			}
			serialized = PEM2string (content);
			free (content);
			if (serialized == NULL) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				send_error (sock, NULL, REG_SYSTEM_ERROR);
				return -1;
			}

			s = "{ \"success\" : true, \"data\" : \"";
			net_write_sock (sock, s, schedwi_strlen (s));
			net_write_sock (	sock, serialized,
						schedwi_strlen (serialized));
			free (serialized);
			s = "\" }";
			net_write_sock (sock, s, schedwi_strlen (s));
			return 0;
		}
		/* No SSL */
		ret = check_file (	client_cert_dir, client_names[i],
					sport, NOSSL_EXTENSION, NULL);
		if (ret < 0) {
			send_error (sock, NULL, REG_SYSTEM_ERROR);
			return -1;
		}
		if (ret == 0) {
			s = "{ \"success\" : true, \"data\" : \"OK\" }";
			net_write_sock (sock, s, schedwi_strlen (s));
			return 0;
		}
	}

	/* No certificate. Just check if at least we've got the request */
	for (i = 0; client_names[i] != NULL; i++) {
		/* SSL */
		ret = check_file (	client_csr_dir, client_names[i],
					sport, CSR_EXTENSION, NULL);
		if (ret < 0) {
			send_error (sock, NULL, REG_SYSTEM_ERROR);
			return -1;
		}
		if (ret == 0) {
			send_error (	sock, _("Certificate still not ready"),
					REG_NOT_READY_YET);
			return 0;
		}
		/* No SSL */
		ret = check_file (	client_csr_dir, client_names[i],
					sport, NOSSL_EXTENSION, NULL);
		if (ret < 0) {
			send_error (sock, NULL, REG_SYSTEM_ERROR);
			return -1;
		}
		if (ret == 0) {
			send_error (	sock, _("Certificate still not ready"),
					REG_NOT_READY_YET);
			return 0;
		}
	}

	/* No certificate and no request! */
	lwc_writeLog (LOG_INFO,
_("Agent %s (port %s) is asking for its certificate but I don't even have the pending request"), client_names[0], sport);
	send_error (sock,
		_("I don't have a request for you. Send a request first"),
		REG_NO_CSR);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
