/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_utils.c -- Useful network functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#if HAVE_NETDB_H
#include <netdb.h>
#endif

#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_TCPD_H
#include <tcpd.h>
#endif

#if HAVE_SYSLOG_H
#include <syslog.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#ifndef HAVE_CLOSESOCKET
#define closesocket(x) close(x)
#endif

#include <pthread.h>

#include <thread_init.h>
#include <lib_functions.h>
#include <lwc_log.h>
#include <sql_hosts.h>
#include <net_utils_sock.h>
#include <net_utils_ssl.h>
#include <parse_json_result.h>
#include <addrmatch.h>
#include <net_utils.h>


#if HAVE_LIBWRAP
int allow_severity;
int deny_severity;
#endif


/*
 * Structure used to exchange parameters between the net_accept() function and
 * the new thread which manages the agent connection
 */
struct _net_utils_data {
	struct sockaddr_storage in;
	socklen_t inlen;
	int (*run)(schedwi_BIO *, void *);
	void *user_data;
	int s;
};
typedef struct _net_utils_data net_utils_data;
typedef struct _net_utils_data *net_utils_data_ptr;


/*
 * Create a new net_utils_data_ptr object
 *
 * Return:
 *   The new object (to be freed by the caller by net_utils_data_destroy()) or
 *   NULL in case of memory allocation error.
 */
static net_utils_data_ptr
net_utils_data_new (	int s, struct sockaddr_storage *in, socklen_t inlen,
			int (*run)(schedwi_BIO *, void *), void *user_data)
{
	net_utils_data_ptr ptr;

#if HAVE_ASSERT_H
	assert (in != NULL && run != NULL);
#endif

	ptr = (net_utils_data_ptr) malloc (sizeof (net_utils_data));
	if (ptr == NULL) {
		return NULL;
	}
	ptr->in        = *in;
	ptr->inlen     = inlen;
	ptr->s         = s;
	ptr->run       = run;
	ptr->user_data = user_data;
	return ptr;
}


/*
 * Free the provided net_utils_data_ptr object.  The associated socket is
 * also closed.
 */
static void
net_utils_data_destroy (net_utils_data_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->s >= 0) {
			net_close_sock (ptr->s);
		}
		free (ptr);
	}
}


/*
 * Error callback function for the sql_host_certificate() function
 */
static void
sql_host_certificate_error_logger (	void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
	_("Database error while trying to retrieve the agent parameters"));
	}
}


/*
 * Thread function used to manage the new agent connection
 */
static void *
net_utils_server_thread (void *data)
{
	net_utils_data_ptr ptr = (net_utils_data_ptr) data;
	char **names;
	char use_ssl, *client_certificate;
	unsigned long int len_client_certificate;
	schedwi_BIO in;

	if (ptr == NULL) {
		pthread_exit (NULL);
	}

	/* Initialize the thread */
	thread_init ();

	/* Install the cleanup callback */
	pthread_cleanup_push (thread_clean, NULL);

	/* Retrieve the names associated with the in_addr structure */
	if (build_client_names_from_addr (&(ptr->in), ptr->inlen, &names) != 0)
	{
		net_utils_data_destroy (ptr);
		pthread_exit (NULL);
	}

	/*
	 * Retrieve whether SSL must be used for the connexion and retrieve
	 * the associated certificate
	 */
	client_certificate = NULL;
	if (sql_host_certificate (	names, &use_ssl, &client_certificate,
					&len_client_certificate,
					sql_host_certificate_error_logger,
					NULL) != 0)
	{
		net_utils_data_destroy (ptr);
		destroy_client_names (names);
		pthread_exit (NULL);
	}

	/*
	 * No SSL
	 */
	if (use_ssl == 0) {
		in.use_ssl = 0;
		in.trusted_cert_ptr = NULL;
		in.fd = ptr->s;
		if (client_certificate != NULL) {
			free (client_certificate);
		}
		destroy_client_names (names);
		(ptr->run)(&in, ptr->user_data);
		net_utils_data_destroy (ptr);
		pthread_exit (NULL);
	}


	/*
	 * SSL
	 */
	in.trusted_cert_ptr = (gnutls_certificate_credentials_t *) malloc (
				sizeof (gnutls_certificate_credentials));
	if (in.trusted_cert_ptr == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		if (client_certificate != NULL) {
			free (client_certificate);
		}
		destroy_client_names (names);
		net_utils_data_destroy (ptr);
		pthread_exit (NULL);
	}
	in.use_ssl = 1;
	in.fd = ptr->s;

	if (net_server_ssl (&(in.session), in.fd, in.trusted_cert_ptr,
			(const char **)names,
			client_certificate, len_client_certificate) != 0)
	{
		if (client_certificate != NULL) {
			free (client_certificate);
		}
		destroy_client_names (names);
		net_utils_data_destroy (ptr);
		free (in.trusted_cert_ptr);
		pthread_exit (NULL);
	}

	if (client_certificate != NULL) {
		free (client_certificate);
	}
	destroy_client_names (names);
	(ptr->run)(&in, ptr->user_data);
	net_close_ssl (in.session, in.fd, in.trusted_cert_ptr);
	ptr->s = -1; /* Socket already closed */
	net_utils_data_destroy (ptr);

	pthread_cleanup_pop (1);
	pthread_exit (NULL);
}


/*
 * Retrieve the next waiting connection from the provided set.
 * sock_set and nb_set are updated accordingly.
 *   sock_set --> Set of waiting sockets
 *   max_set  --> Highest socket number + 1 in the set
 *   nb_set   --> Number of waiting sockets in the set
 *   allow_from --> Only accept connections from the allowed clients
 *
 * Return :
 *   The next socket
 *   -1 in case of error (errno is set and a message is displayed using
 *      lwc_writeLog())
 *   -2 No more socket waiting (the set is empty)
 */
static int
net_accept_next (fd_set *sock_set, int max_set, int *nb_set,
		const char *allow_from,
		struct sockaddr_storage *out_sockname, socklen_t *out_len)
{
	int i, nb, save_errno, s, ret;
	struct sockaddr_storage sockname;
	char name[NI_MAXHOST];
	socklen_t len;
#if HAVE_LIBWRAP
	struct request_info req;
#endif

	nb = *nb_set;
	for (i = 0 ; i < max_set && nb > 0; i++) {
		if (FD_ISSET (i, sock_set) == 0) {
			continue;
		}
		nb--;

		do {
			len = sizeof (sockname);
			s = accept (i, (struct sockaddr *)&sockname, &len);
			if (s < 0 && errno != EINTR && errno != EAGAIN) {
				save_errno = errno;
				lwc_writeLog (	LOG_CRIT,
						_("Network error: accept: %s"),
						strerror (errno));
				errno = save_errno;
				return -1;
			}
		} while (s < 0);

		FD_CLR (i, sock_set);
		(*nb_set)--;

		/*
		 * Only the allowed clients, defined by the configuration
		 * option ALLOW_FROM are allowed.
		 * It's always allowed if the variable is not set
		 */
		if (allow_from == NULL || allow_from[0] == '\0') {
			if (out_sockname != NULL) {
				*out_sockname = sockname;
			}
			if (out_len != NULL) {
				*out_len = len;
			}
			return s;
		}

		ret = getnameinfo (	(struct sockaddr *)&sockname, len,
					name, sizeof (name), NULL, 0,
					NI_NUMERICHOST);
		if (ret != 0) {
			lwc_writeLog (	LOG_NOTICE,
					_("getnameinfo: %s"),
					gai_strerror (ret));
			net_close_sock (s);
			continue;
		}

#if HAVE_LIBWRAP
		/* Check the network access using the TCP Wrappers */
		allow_severity = LOG_INFO;
		deny_severity  = LOG_WARNING;

		request_init (	&req,
				RQ_DAEMON, SCHEDWI_TCP_WRAPPERS_DAEMON,
				RQ_FILE, s, 0);
		fromhost (&req);
		if (hosts_access (&req) == 0) {
			lwc_writeLog (	LOG_DEBUG,
			_("Connection refused by the TCP Wrappers for %s"),
					name);
			net_close_sock (s);
			continue;
		}
#endif /* HAVE_LIBWRAP */

		/*
		 * Check through the internal network access list if the agent
		 * is allowed.  This list is set through the configuration
		 * variable ALLOW_FROM.
		 */
		ret = addr_match_list (name, allow_from);
		switch (ret) {
			case 1:
				/* Allowed */
				if (out_sockname != NULL) {
					*out_sockname = sockname;
				}
				if (out_len != NULL) {
					*out_len = len;
				}
				return s;
			case 0:
				lwc_writeLog (	LOG_DEBUG,
						_("Connection refused for %s"),
						name);
				break;
			case -1:
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				break;
			case -2:
				/*
				 * Should not occur (checked at start time
				 * by check_netlist())
				 */
				lwc_writeLog (	LOG_ERR,
	_("Inconsistent mask length in one of the address in ALLOW_FROM"));
				break;
		}

		net_close_sock (s);
	}
	return -2;
}


/*
 * Wait for a network request and return the socket associated with the
 * accepted connection.  The agent host is check to be sure that it's an
 * allowed client.
 *
 * Return:
 *   The new socket or
 *   -1 in case of error (errno is set and a message is displayed using
 *      lwc_writeLog())
 */
int
net_accept_sock (fd_set *sock_set, int sock_max,
		const char *allow_from, char ***names)
{
	static fd_set waiting_set;
	static int still_waiting = 0;
	int s, ret, save_errno;
	fd_set rfds;
	struct sockaddr_storage sockname;
	socklen_t len;

#if HAVE_ASSERT_H
	assert (sock_set != NULL);
#endif

	/* First, manage the connections still waiting from a previous call */
	s = net_accept_next (	&waiting_set, sock_max, &still_waiting,
				allow_from, &sockname, &len);
	if (s == -1) {
		return -1;
	}
	if (s >= 0) {
		if (names != NULL) {
			build_client_names_from_addr (&sockname, len, names);
		}
		return s;
	}

	/* No more pending connections from a previous call */

	while (1) {
		rfds = *sock_set;
		ret = select (sock_max, &rfds, NULL, NULL, NULL);
		if (ret < 0) {
			if (errno == EINTR || errno == EAGAIN) {
				continue;
			}
			save_errno = errno;
			lwc_writeLog (	LOG_CRIT,
					_("Network error: select: %s"),
					strerror (errno));
			errno = save_errno;
			return -1;
		}
		waiting_set = rfds;
		still_waiting = ret;

		s = net_accept_next (	&waiting_set, sock_max, &still_waiting,
					allow_from, &sockname, &len);
		if (s == -1) {
			return -1;
		}
		if (s >= 0) {
			if (names != NULL) {
				build_client_names_from_addr (	&sockname, len,
								names);
			}
			return s;
		}
	}
	return -1;
}


/*
 * Manage all the network request on the provided socket.  For each request
 * a new thread is started.
 * The run function is called to interract with the agent (its second
 * parameter is the provided user_data parameter).
 *
 * Return:
 *   Never or
 *   -1 in case of error (errno is set and a message is displayed using
 *      lwc_writeLog())
 */
int
net_accept (	fd_set *sock_set, int sock_max, const char *allow_from,
		int (*run)(schedwi_BIO *, void *), void *user_data)
{
	pthread_attr_t attr;
	pthread_t thread;
	net_utils_data_ptr ptr;
	int s, ret, save_errno;
	fd_set rfds;
	struct sockaddr_storage sockname;
	socklen_t len;

#if HAVE_ASSERT_H
	assert (run != NULL && sock_set != NULL);
#endif

	/* Initialize the thread attribute object */
	pthread_attr_init (&attr);
	pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_DETACHED);

	/*
	 * In case this function is in a thread, make sure that if the thread
	 * is cancelled the attr object if freed
	 */
	pthread_cleanup_push (	(void (*) (void *))pthread_attr_destroy,
				(void *)&attr);

	/*
	 * For each received connection, a new detached thread is started
	 * to manage the new connection
	 */
	while (1) {
		pthread_testcancel ();
		rfds = *sock_set;
		ret = select (sock_max, &rfds, NULL, NULL, NULL);
		if (ret < 0) {
			if (errno == EINTR || errno == EAGAIN) {
				continue;
			}
			save_errno = errno;
			lwc_writeLog (	LOG_CRIT,
					_("Network error: select: %s"),
					strerror (errno));
			pthread_attr_destroy (&attr);
			errno = save_errno;
			return -1;
		}

		do {
			pthread_testcancel ();
			s = net_accept_next (	&rfds, sock_max,
						&ret, allow_from,
						&sockname, &len);
			if (s == -1) {
				save_errno = errno;
				pthread_attr_destroy (&attr);
				errno = save_errno;
				return -1;
			}
			if (s >= 0) {
				ptr = net_utils_data_new (s, &sockname, len,
							run, user_data);
				if (ptr == NULL) {
					closesocket (s);
					lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
					continue;
				}
				if (pthread_create (	&thread, &attr,
							net_utils_server_thread,
							ptr) != 0)
				{
					lwc_writeLog (	LOG_CRIT,
_("Cannot create a new thread to manage the network connection: %s"),
						strerror (errno));
					net_utils_data_destroy (ptr);
					continue;
				}
			}

		} while (s != -2);
	}
	/* Never reach this point */
	pthread_attr_destroy (&attr);
	pthread_cleanup_pop (0);
	return -1;
}


/*
 * Initialize the network layer (SSL algorithms) for a server
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
net_init_server ()
{
	return net_init_server_ssl ();
}


/*
 * Initialize the network layer.  Warning: this function is not thread safe
 * and must then be called only once in the main thread before the creation
 * of the other one.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
net_init (	const char *schedwisrv_crt_file,
		const char *schedwisrv_key_file,
		const char *schedwisrv_ca_file,
		const char *schedwisrv_crl_file,
		char quick_random)
{
	return net_init_ssl (	schedwisrv_crt_file, schedwisrv_key_file,
				schedwisrv_ca_file, schedwisrv_crl_file,
				quick_random);
}


/*
 * Free the network layer.  This function may be called before exiting the
 * schedwisrv program
 */
void
net_destroy ()
{
	net_destroy_ssl ();
}


/*
 * Create a socket connected to a server.
 * port_number is the port number to connect to (a number or a service name
 * in /etc/services)
 *
 * Return:
 *   The schedwi_BIO handler to be freed by the caller by net_close() or
 *   NULL in case of error (an error message has been logged by lwc_writeLog())
 */
schedwi_BIO *
net_client (	const char *port_number, const char *hostname, char use_ssl,
		char *remote_crt, unsigned int remote_crt_len)
{
	schedwi_BIO *out;
	int ret;

#if HAVE_ASSERT_H
	assert (port_number != NULL && hostname != NULL);
#endif

	/* Allocate space for the returned schedwi_BIO object */
	out = (schedwi_BIO *) malloc (sizeof (schedwi_BIO));
	if (out == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}

	/*
	 * No SSL, only socket
	 */
	if (use_ssl == 0) {
		out->use_ssl = 0;
		out->trusted_cert_ptr = NULL;
		out->fd = net_client_sock (port_number, hostname);
		if (out->fd < 0) {
			free (out);
			return NULL;
		}
		return out;
	}

	/*
	 * SSL
	 */
	out->use_ssl = 1;
	out->trusted_cert_ptr = (gnutls_certificate_credentials_t *) malloc (
				sizeof (gnutls_certificate_credentials));
	if (out->trusted_cert_ptr == NULL) {
		free (out);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}

	ret = net_client_ssl (	port_number, hostname,
				&(out->session), &(out->fd),
				out->trusted_cert_ptr,
				remote_crt, remote_crt_len);
	if (ret != 0) {
		free (out->trusted_cert_ptr);
		free (out);
		return NULL;
	}

	return out;
}


/*
 * Read data from the provided schedwi_BIO handler.
 * The data are store in *buff (of size *len). If buffer is NULL or not big
 * enough, it is reallocated to a bigger size. It must be freed by the caller
 *
 * Return:
 *    The total number of characters read and stored in *buff or
 *    -1 on error (a message has been displayed using lwc_writeLog())
 *    -2 if too many data are received (a message has been displayed using
 *       lwc_writeLog())
 */
int
net_read (schedwi_BIO *in, char **buff, size_t *len)
{
#if HAVE_ASSERT_H
	assert (in != NULL && buff != NULL && len != NULL);
#endif

	if (in->use_ssl == 1) {
		return net_read_ssl (in->session, buff, len);
	}
	else {
		return net_read_sock (in->fd, buff, len);
	}
}


/*
 * Read the reply
 *
 * The format of the JSON reply looks like:
 *     {
 *         "success" : false,
 *          "data" : "Database unavailable"
 *     }
 *
 * No reply (nothing to read) means failure.
 *
 * Return:
 *   0 --> The reply means the remote action was successfull.  If result_msg is
 *         not NULL, it is set with the message sent by the remote socket.  It
 *         must be freed by the caller.
 *   1 --> The reply means the remote action failed.  If result_msg is not
 *         NULL, it is set with the message sent by the remote socket.  It may
 *         be NULL if no message was sent.  It must be freed by the caller.
 *  -1 --> Error (a message has been logged by lwc_writeLog())
 */
int
net_read_result (schedwi_BIO *in, char **result_msg)
{
	char *buff;
	size_t len;
	int nb_read, ret;

	if (result_msg != NULL) {
		*result_msg = NULL;
	}

	/* Read the reply */
	buff = NULL;
	len = 0;
	nb_read = net_read (in, &buff, &len);
	if (nb_read < 0 ) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	if (nb_read > 0) {
		ret = parse_json_result (buff, (ssize_t)nb_read, result_msg);
	}
	else {
		ret = 1;
	}
	if (buff != NULL) {
		free (buff);
	}
	return ret;
}


/*
 * Write data to the provided handler.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message has been logged by lwc_writeLog())
 */
int
net_write (schedwi_BIO *out, const char *buff, int len)
{
	if (buff == NULL || len == 0) {
		return 0;
	}

#if HAVE_ASSERT_H
	assert (out != NULL);
#endif

	if (out->use_ssl == 1) {
		return net_write_ssl (out->session, buff, len);
	}
	else {
		return net_write_sock (out->fd, buff, len);
	}
}


/*
 * Close the handler
 */
void
net_close (schedwi_BIO *out)
{
	if (out != NULL) {
		if (out->use_ssl == 1) {
			net_close_ssl (	out->session, out->fd,
					out->trusted_cert_ptr);
		}
		else {
			net_close_sock (out->fd);
		}
		free (out);
	}
}

/*------------------------======= End Of File =======------------------------*/
