/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_parse.c -- Parse a network request */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_hierarchy.h>
#include <lib_functions.h>
#include <lwc_log.h>
#include <result_mgnt.h>
#include <schedwi_time.h>
#include <load_workload_class.h>
#include <job_status_state.h>
#include <workload_class.h>
#include <net_parse.h>
#include <JSON_parser.h>


#define WORKLOAD_LEN 8


/*
 * Build a message and find out the new status of the job from the
 * provided result_t object.
 *
 * Return:
 *   0 --> No error.  new_state and job_err_msg are set.  job_err_msg must be
 *         freed by the caller (by free()).
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
static int
result_to_status (	schedwi_date workload_date,
			result_t *ptr,
			job_status_state *new_state,
			char **job_err_msg)
{
	unsigned int len;
	char *status_msg, *msg, *err_msg;
	row_item_t success_ret;
	int ret, d;
	lwc_LL *hierarchy_list;
	unsigned long long int job_id;

#if HAVE_ASSERT_H
	assert (ptr != NULL && new_state != NULL && job_err_msg != NULL);
#endif

	/* Build the status message */
	if (ptr->was_killed != 0) {
		status_msg = _("Killed by signal %d");
	}
	else {
		status_msg = _("Exit with return code %d");
	}

	len = schedwi_strlen (status_msg) + 11;
	msg = (char *) malloc (len);
	if (msg == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	ret = snprintf (msg, len, status_msg, ptr->exit_code);
	if (ret >= len || ret < 0) {
		msg[0] = '\0';
	}

	/* If the job was kill */
	if (ptr->was_killed != 0) {
		*job_err_msg = msg;
		*new_state = JOB_STATUS_STATE_FAILED;
		return 0;
	}

	/*
	 * Retrieve the error code from the database to find out if the
	 * job is successful or not
	 */
	d = schedwi_date_to_int (workload_date);
	job_id = strtoull (ptr->job_id + WORKLOAD_LEN + 1, NULL, 0);
	err_msg = NULL;
	hierarchy_list = get_hierarchy_list (d, job_id, &err_msg);
	if (hierarchy_list == NULL) {
		free (msg);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
					_("Job %s: cannot retrieve details"),
					ptr->job_id + WORKLOAD_LEN + 1);
		}
		return -1;
	}

	err_msg = NULL;
	if (get_job_parameter (	d, hierarchy_list,
				"job_success_return_code_s", "success_ret",
				&success_ret, &err_msg) != 0)
	{
		free (msg);
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
	_("Job %s: error while retrieving the job success return code"),
					ptr->job_id + WORKLOAD_LEN + 1);
		}
		return -1;
	}
	lwc_delLL (hierarchy_list, (void (*)(const void *))free);

	if (	   success_ret.type == RES_END
		|| ptr->exit_code > sql_row_item2ll (&success_ret))
	{
		*new_state = JOB_STATUS_STATE_FAILED;
	}
	else {
		*new_state = JOB_STATUS_STATE_COMPLETED;
	}

	*job_err_msg = msg;
	return 0;
}


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	char *s = (char *)ctx;

	if (type == JSON_T_STRING) {
		strncpy (s, value->vu.str.value, SCHEDWI_NET_FUNCTION_LENGHT);
		s[SCHEDWI_NET_FUNCTION_LENGHT] = '\0';
	}
	return 1;
}


/*
 * Parse the agent request
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  A message has been logged by lwc_writeLog()
 */
int
net_parse (schedwi_BIO *b, void *obj)
{
	load_workload_class_ptr workload_list = (load_workload_class_ptr) obj;
	char *buff, *err_msg, *s;
	size_t len;
	int nb_read, i;
	char module[SCHEDWI_NET_FUNCTION_LENGHT + 1];
	result_t *ptr;
	char workload[WORKLOAD_LEN + 1];
	schedwi_date workload_date;
	workload_class_ptr workload_obj;
	job_status_state new_state;
	JSON_config config;
	struct JSON_parser_struct *jc;


#if HAVE_ASSERT_H
	assert (workload_list != NULL);
#endif

	if (b == NULL) {
		return 0;
	}

	/* Read the request */
	buff = NULL;
	len = 0;
	nb_read = net_read (b, &buff, &len);
	if (nb_read < 0) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	/* JSON parser initialization */
	init_JSON_config (&config);

	module[0]                     = '\0';
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = module;

	jc = new_JSON_parser (&config);

	/* Look for the first JSON string which is the module name */
	for (i = 0; i < nb_read && module[0]  == '\0'; i++) {
		if (! JSON_parser_char (jc, buff[i])) {
			if (buff != NULL) {
				free (buff);
			}
			lwc_writeLog (LOG_ERR,
			_("Wrong network request: JSON syntax error"));
			delete_JSON_parser (jc);
			return -1;
		}
	}
	delete_JSON_parser (jc);

	if (module[0] == '\0') {
		if (buff != NULL) {
			free (buff);
		}
		lwc_writeLog (LOG_ERR,
			_("Wrong network request: module name not specified"));
		return -1;
	}

	if (strcmp (module, "result") != 0) {
		if (buff != NULL) {
			free (buff);
		}
		lwc_writeLog (	LOG_ERR,
			_("Wrong network request: unknown module `%s'"),
				module);
		return -1;
	}

	/* Convert the string to a result_t structure */
	ptr = NULL;
	if (str_to_result (buff, nb_read, &ptr) != 0) {
		if (buff != NULL) {
			free (buff);
		}
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (b, s, schedwi_strlen (s));
		s = _("Memory allocation error");
		net_write (b, s, schedwi_strlen (s));
		lwc_writeLog (LOG_CRIT, s);
		s = "\" }";
		net_write (b, s, schedwi_strlen (s));
		return -1;
	}

	if (buff != NULL) {
		free (buff);
	}

	/* Update the status of the job */

	strncpy (workload, ptr->job_id, WORKLOAD_LEN);
	workload[WORKLOAD_LEN] = '\0';

	if (	   ptr->job_id_len < WORKLOAD_LEN + 2
		|| (ptr->job_id)[WORKLOAD_LEN] != SCHEDWI_WORKLOAD_SEPARATOR
		|| schedwi_date_from_string (workload, &workload_date) != 0)
	{
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (b, s, schedwi_strlen (s));
		s = _("Wrong job ID");
		net_write (b, s, schedwi_strlen (s));
		s = "\" }";
		net_write (b, s, schedwi_strlen (s));
		lwc_writeLog (LOG_ERR, _("Wrong job ID `%s'"), ptr->job_id);
		destroy_result (ptr);
		return -1;
	}

	/* Convert the received result to a status */
	if (result_to_status (workload_date, ptr, &new_state, &err_msg) != 0) {
		destroy_result (ptr);
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (b, s, schedwi_strlen (s));
		s = _("Server internal error");
		net_write (b, s, schedwi_strlen (s));
		s = "\" }";
		net_write (b, s, schedwi_strlen (s));
		return -1;
	}

	/* Retrieve the workload tree */
	workload_obj = load_workload_class_find (workload_list, workload_date);
	if (	   workload_obj != NULL
		&& workload_class_job_finished (workload_obj,
				strtoull (	ptr->job_id + WORKLOAD_LEN + 1,
						NULL, 0),
				new_state,
				(long int)ptr->duration,
				err_msg) != 0)
	{
		free (err_msg);
		destroy_result (ptr);
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (b, s, schedwi_strlen (s));
		s = _("Server internal error");
		net_write (b, s, schedwi_strlen (s));
		s = "\" }";
		net_write (b, s, schedwi_strlen (s));
		return -1;
	}
	free (err_msg);
	destroy_result (ptr);

	s = "{ \"success\" : true }";
	net_write (b, s, schedwi_strlen (s));

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
