# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Job and jobset status."""

from datetime import datetime

import sqlalchemy.orm.session
import babel.dates

from tables.job_status import job_status
import colors
import locale_utils

WAITING, RUNNING, COMPLETED, FAILED, COMPLETED_OR_FAILED = 1, 2, 3, 4, 5


def status_list():
    return [WAITING, RUNNING, COMPLETED, FAILED, COMPLETED_OR_FAILED]


def status2string(st):
    """Convert a status number to a string."""
    if st == WAITING:
        return _("waiting")
    elif st == RUNNING:
        return _("running")
    elif st == COMPLETED:
        return _("completed")
    elif st == COMPLETED_OR_FAILED:
        return _("done")
    else:
        return _("failed")


def string2status(str_status):
    """Convert a string to a status number."""
    c = str_status[0].lower()
    if c == 'w':
        return WAITING
    elif c == 'r':
        return RUNNING
    elif c == 'c':
        return COMPLETED
    elif c == 'f':
        return FAILED
    elif c == 'd':
        return COMPLETED_OR_FAILED
    return 0


def status2colorname(st):
    """Convert a status number to a color name."""
    if st == WAITING:
        return "yellow"
    elif st == RUNNING:
        return "blue"
    elif st == COMPLETED:
        return "green"
    elif st == FAILED:
        return "red"
    elif st == COMPLETED_OR_FAILED:
        return "black"
    return ''


def string2colorname(str_status):
    """Convert a status string to a color name."""
    return status2colorname(string2status(str_status))


def status2char(st):
    """Convert a status number to a single character."""
    if st == WAITING:
        return _("w")
    elif st == RUNNING:
        return _("r")
    elif st == COMPLETED:
        return _("c")
    elif st == COMPLETED_OR_FAILED:
        return _("d")
    else:
        return _("f")


def status2color(st):
    """Convert a status number to a color code."""
    if st == WAITING:
        return colors.YELLOW
    elif st == RUNNING:
        return colors.BLUE
    elif st == COMPLETED:
        return colors.GREEN
    elif st == COMPLETED_OR_FAILED:
        return colors.BLACK
    else:
        return colors.RED


def get_status(sql_session, job_id, workload):
    """Return the job_status object of the specified job_id."""
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session
    query = session.query(job_status).filter(job_status.job_id == job_id)
    query = query.filter(job_status.workload_date == workload)
    try:
        status = query.one()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return ''
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return status


def get_status_string(sql_session, job_id, workload):
    """Return a string representing the status of the given job_id."""
    status = get_status(sql_session, job_id, workload)
    if not status:
        return "waiting"
    time_str = babel.dates.format_time(
                                datetime.fromtimestamp(status.start_time),
                                format="short",
                                locale=locale_utils.get_locale())
    return "%s (%s - try %d%s)" % (
           status2string(status.status),
           time_str.encode('utf-8'),
           status.retry_num,
           (" - %s" % status.error_msg) if status.error_msg else '')
