# Schedwi
# Copyright (C) 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Simple database queries."""

import sqlalchemy.orm.session

from tables.job_main import job_main
from tables.job_main_s import job_main_s


def sql_get_job(sql_session, job_id, workload=None):
    """Retrieve a job or jobset from the database.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param job_id:
                the job ID to look for in the database.
    @param workload:
                workload to consider.
    @return:    the job.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given job ID is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    if workload is None:
        query = session.query(job_main)
    else:
        query = session.query(job_main_s)
        query = query.filter(job_main_s.workload_date == workload)
    query = query.filter_by(id=job_id)
    try:
        job = query.one()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return job


def sql_get_children(session, jobset_id, only_jobset=False, workload=None):
    """Return the list of children of the specified jobset.

    @param session:
                an opened SQLAlchemy session.
    @param jobset_id:
                parent jobset ID.
    @param only_jobset:
                only return the children that are jobsets.
    @param workload:
                workload to consider.
    @return:    the list of children jobs/jobsets.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given jobset ID is not in the database.
    """
    if workload is None:
        query = session.query(job_main).order_by(job_main.type, job_main.name)
    else:
        query = session.query(job_main_s)
        query = query.order_by(job_main_s.type, job_main_s.name)
        query = query.filter(job_main_s.workload_date == workload)
    if only_jobset:
        query = query.filter_by(type=0)
    return query.filter_by(parent=jobset_id).all()


def sql_count_children(sql_session, jobset_id, only_jobset=False,
                       workload=None):
    """Return the number of children of the specified jobset.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param jobset_id:
                parent jobset ID.
    @param only_jobset:
                only count the children that are jobsets.
    @param workload:
                workload to consider.
    @return:    the number of children.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given jobset ID is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    if workload is None:
        query = session.query(job_main)
    else:
        query = session.query(job_main_s)
        query = query.filter(job_main_s.workload_date == workload)
    if only_jobset:
        query = query.filter_by(type=0)
    try:
        n = query.filter_by(parent=jobset_id).count()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return n


def sql_count_children2(sql_session, jobset_id, workload=None):
    """Return the number of children of the specified jobset.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param jobset_id:
                parent jobset ID.
    @param workload:
                workload to consider.
    @return:    the tuple (#jobs, #jobsets)
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given jobset ID is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session
    # Number of Jobsets
    try:
        nb_jobsets = sql_count_children(session, jobset_id, True, workload)
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise
    # Number of Jobs
    if workload is None:
        query = session.query(job_main)
    else:
        query = session.query(job_main_s)
        query = query.filter(job_main_s.workload_date == workload)
    query = query.filter_by(type=1)
    try:
        nb_jobs = query.filter_by(parent=jobset_id).count()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return (nb_jobs, nb_jobsets)
