# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Main command loop."""

try:
    import readline
except ImportError:
    _has_readline = False
else:
    _has_readline = True

import cmd
import shlex
from subprocess import Popen
import os
import sys

import config
import path
import path_cal
import cmd_job.cd
import cmd_job.ls
import cmd_job.tree
import cmd_job.cat
import cmd_job.stat
import cmd_job.mk
import cmd_job.mkjs
import cmd_job.set
import cmd_job.rm
import cmd_job.ln
import cmd_job.rmln
import cmd_job.mv
import cmd_job.cp
import cmd_job.find
import cmd_job.complete
import cmd_hosts.ls
import cmd_hosts.cat
import cmd_hosts.whatuses
import cmd_hosts.set
import cmd_hosts.mk
import cmd_hosts.rm
import cmd_hosts.complete
import cmd_env.ls
import cmd_env.cat
import cmd_env.set
import cmd_env.mk
import cmd_env.whatuses
import cmd_env.rm
import cmd_env.complete
import cmd_cal.cd
import cmd_cal.ls
import cmd_cal.tree
import cmd_cal.cat
import cmd_cal.whatuses
import cmd_cal.rm
import cmd_cal.mv
import cmd_cal.cp
import cmd_cal.mkdir
import cmd_cal.rmdir
import cmd_cal.mk
import cmd_cal.set
import cmd_cal.show
import cmd_cal.complete
import cmd_wl.wl
import cmd_wl.start
import cmd_wl.stop
import cmd_wl.complete
import cmd_wl.status

_history_file = "~/.schedwi_history"

JOB, HOST, CAL, ENV = 0, 1, 2, 3


class Loop(cmd.Cmd):

    def __init__(self, sql_session, interactive=True):
        cmd.Cmd.__init__(self)
        self.interactive = interactive
        self.last_return_code = 0
        self.context = JOB
        self.sql_session = sql_session
        if config.ISATTY and _has_readline:
            try:
                readline.read_history_file(os.path.expanduser(_history_file))
            except IOError:
                pass
        self.cwd = path.Path(self.sql_session, '/', only_dir=True)
        self.previous_cwd = None
        self.cwd_cal = path_cal.PathCal(self.sql_session, '/')
        self.previous_cwd_cal = None
        self.prompt = self.cwd.path + '> '
        self.workload = None
        self.cwd_workload = None
        self.previous_cwd_workload = None
        self.cwd_cal_workload = None
        self.previous_cwd_cal_workload = None

    def _my_shlex(self, s, comments=False):
        try:
            return shlex.split(s, comments)
        except Exception as e:
            sys.stderr.write(str(e) + "\n")
            raise

    def postcmd(self, stop, line):
        if self.interactive:
            return False
        else:
            return stop

    def emptyline(self):
        pass

    def help_exit(self):
        if self.context == JOB:
            print _("Exit the program.")
        elif self.context == HOST:
            print _("Exit the `hosts' sub-command.")
        elif self.context == ENV:
            print _("Exit the `env' sub-command.")
        elif self.context == CAL:
            print _("Exit the `cal' sub-command.")

    def help_quit(self):
        self.help_exit()

    def help_bye(self):
        self.help_exit()

    def do_exit(self, s):
        """Exit from schedwi or, if in a sub-command, go back up"""
        if self.context == JOB and self.workload is None:
            if config.ISATTY and _has_readline:
                readline.write_history_file(os.path.expanduser(_history_file))
            sys.exit(self.last_return_code)
        if self.context == JOB:
            self.prompt = self.cwd.path + '> '
            self.workload = None
        elif self.workload is None:
            self.context = JOB
            self.prompt = self.cwd.path + '> '
        else:
            self.context = JOB
            self.prompt = '%s %s> ' % (self.workload, self.cwd.path)

    def do_quit(self, s):
        """Exit from schedwi or, if in a sub-command, go back up"""
        self.do_exit(s)

    def do_bye(self, s):
        """Exit from schedwi or, if in a sub-command, go back up"""
        self.do_exit(s)

    def do_EOF(self, s):
        """Exit from schedwi or, if in a sub-command, go back up"""
        print "exit"
        self.do_exit(s)

    def help_shell(self):
        print _("""Usage: !COMMAND
   or: shell COMMAND
Launch the provided shell command.""")

    def do_shell(self, s):
        """Launch a shell command"""
        p = Popen(s, shell=True)
        os.waitpid(p.pid, 0)

    def help_help(self):
        print _("""Usage: help [CMD]
Print help for command CMD, or a list of all available commands.""")

    def do_help(self, s):
        """Print help"""
        if not s:
            print _("""
Available commands (type help <topic>):
=======================================

cat       print job and jobset parameters
cd        change the current jobset
cp        copy jobs and jobsets
find      find jobs and jobsets
help      print command help
ln        create dependency links between jobs and jobsets
ls        list jobs and jobsets
mk        create a new job
mkjs      create a new jobset
mv        rename or move jobs and jobsets
pwd       print the current jobset
quit, bye or exit   leave the program
rm        remove jobs and jobsets
set       change job and jobset parameters
shell or !   launch a shell command
start     force start jobs and jobsets
stat      print job and jobset execution statistics
stop      stop jobs and jobsets
status    list job and jobset status
tree      list contents of jobsets in a tree-like format
wl        list workload dates and change the current workload

hosts     enter the hosts sub-command.  In this mode, the following commands
          are available:

          quit, bye or exit   leave the hosts sub-command
          cat       print host parameters
          ls        list hosts
          mk        create a new host
          rm        remove hosts
          set       change host parameters
          whatuses  list the jobs, jobsets and constraint files
                    that use a host

env       enter the environment group sub-command.  In this mode, the
          following commands are available:

          quit, bye or exit   leave the environment group sub-command
          cat       print environment group parameters
          ls        list environment groups
          mk        create a new environment group
          rm        remove environment groups
          set       change environment group parameters
          whatuses  list the jobs, jobsets and hosts that use an
                    environment group

cal       enter the calendar sub-command.  In this mode, the following
          commands are available:

          quit, bye or exit   leave the calendar sub-command
          cat       print calendar parameters
          cd        change the current directory
          cp        copy calendars and calendar directories
          ls        list calendars and calendar directories
          mk        create a new calendar
          mkdir     create a new calendar directory
          mv        rename or move calendars and calendar directories
          pwd       print the current calendar directory
          rm        remove calendars and calendar directories
          rmdir     remove empty calendar directories
          set       change calendar parameters
          show      display a calendar view
          tree      list contents of directories in a tree-like format
          whatuses  list the jobs and jobsets associated with a calendar
            """)
        elif s == '!':
            cmd.Cmd.do_help(self, 'shell')
        else:
            cmd.Cmd.do_help(self, s)

    def help_find(self):
        cmd_job.find.usage()

    def do_find(self, s):
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.workload is None:
            self.last_return_code = cmd_job.find.find(self.sql_session,
                                                self.cwd, args)
        else:
            self.last_return_code = cmd_job.find.find(self.sql_session,
                                                self.cwd, args,
                                                self.workload)
        return self.last_return_code

    def help_stat(self):
        cmd_job.stat.usage()

    def complete_stat(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        else:
            return []

    def do_stat(self, s):
        """Display the stats of a job/jobset"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        self.last_return_code = cmd_job.stat.stat(self.sql_session,
                                                  self.cwd, args)
        return self.last_return_code

    def help_pwd(self):
        if self.context == JOB:
            print _("Print name of current/working jobset.")
        elif self.context == CAL:
            print _("Print name of current/working calendar directory.")

    def do_pwd(self, s):
        """Print name of current jobset or current calendar directory"""
        if self.context == JOB:
            if self.workload is None:
                print unicode(self.cwd).encode('utf-8')
            else:
                print unicode(self.cwd_workload).encode('utf-8')
        elif self.context == CAL:
            if self.workload is None:
                print unicode(self.cwd_cal)
            else:
                print unicode(self.cwd_cal_workload)
        else:
            self.default("pwd")

    def help_cd(self):
        if self.context == JOB:
            cmd_job.cd.usage()
        elif self.context == CAL:
            cmd_cal.cd.usage()

    def complete_cd(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload, True)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload, True)
        else:
            return []

    def do_cd(self, s):
        """Change working jobset or calendar directory"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        self.last_return_code = 0
        if self.context == JOB:
            if self.workload is None:
                new_cwd = cmd_job.cd.cd(self.sql_session,
                                        self.cwd, self.previous_cwd, args)
            else:
                new_cwd = cmd_job.cd.cd(self.sql_session, self.cwd_workload,
                                        self.previous_cwd_workload,
                                        args, self.workload)
            if new_cwd is None:
                self.last_return_code = 1
            else:
                if self.workload is None:
                    if new_cwd != self.cwd:
                        self.previous_cwd = self.cwd
                        self.cwd = new_cwd
                        self.prompt = unicode(self.cwd.path).encode('utf-8') \
                                      + '> '
                else:
                    if new_cwd != self.cwd_workload:
                        self.previous_cwd_workload = self.cwd_workload
                        self.cwd_workload = new_cwd
                        self.prompt = '%s %s> ' % (self.workload,
                            unicode(self.cwd_workload.path).encode('utf-8'))
        elif self.context == CAL:
            if self.workload is None:
                new_cwd = cmd_cal.cd.cd(self.sql_session, self.cwd_cal,
                                        self.previous_cwd_cal, args)
            else:
                new_cwd = cmd_cal.cd.cd(self.sql_session,
                                        self.cwd_cal_workload,
                                        self.previous_cwd_cal_workload, args,
                                        self.workload)
            if new_cwd is None:
                self.last_return_code = 1
            else:
                if self.workload is None:
                    if new_cwd != self.cwd_cal:
                        self.previous_cwd_cal = self.cwd_cal
                        self.cwd_cal = new_cwd
                        self.prompt = 'cal:%s> ' % \
                                    unicode(self.cwd_cal.path).encode('utf-8')
                else:
                    if new_cwd != self.cwd_cal_workload:
                        self.previous_cwd_cal_workload = self.cwd_cal_workload
                        self.cwd_cal_workload = new_cwd
                        self.prompt = '%s cal:%s> ' % (self.workload,
                           unicode(self.cwd_cal_workload.path).encode('utf-8'))
        else:
            self.default("cd")
        return self.last_return_code

    def help_cat(self):
        if self.context == JOB:
            cmd_job.cat.usage()
        elif self.context == HOST:
            cmd_hosts.cat.usage()
        elif self.context == ENV:
            cmd_env.cat.usage()
        elif self.context == CAL:
            cmd_cal.cat.usage()

    def complete_cat(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == ENV:
            return cmd_env.complete.complete(self.sql_session,
                                             text, self.workload)
        elif self.context == HOST:
            return cmd_hosts.complete.complete(self.sql_session, text)
        elif self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload)
        else:
            return []

    def do_cat(self, s):
        """Display a job/jobset, host, environment or calendar parameters"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB:
            if self.workload is None:
                self.last_return_code = cmd_job.cat.cat(self.sql_session,
                                                    self.cwd, args,
                                                    self.workload)
            else:
                self.last_return_code = cmd_job.cat.cat(self.sql_session,
                                                    self.cwd_workload, args,
                                                    self.workload)
        elif self.context == HOST:
            self.last_return_code = cmd_hosts.cat.cat(self.sql_session, args)
        elif self.context == ENV:
            self.last_return_code = cmd_env.cat.cat(self.sql_session, args,
                                                    self.workload)
        elif self.context == CAL:
            if self.workload is None:
                self.last_return_code = cmd_cal.cat.cat(self.sql_session,
                                                        self.cwd_cal, args,
                                                        self.workload)
            else:
                self.last_return_code = cmd_cal.cat.cat(self.sql_session,
                                                        self.cwd_cal_workload,
                                                        args, self.workload)
        return self.last_return_code

    def help_tree(self):
        if self.context == JOB:
            cmd_job.tree.usage()
        elif self.context == CAL:
            cmd_cal.tree.usage()

    def complete_tree(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload, True)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload, True)
        else:
            return []

    def do_tree(self, s):
        """List jobsets in a tree-like format"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB:
            if self.workload is None:
                self.last_return_code = cmd_job.tree.tree(self.sql_session,
                                                      self.cwd, args,
                                                      self.workload)
            else:
                self.last_return_code = cmd_job.tree.tree(self.sql_session,
                                                      self.cwd_workload, args,
                                                      self.workload)
        elif self.context == CAL:
            if self.workload is None:
                self.last_return_code = cmd_cal.tree.tree(self.sql_session,
                                                      self.cwd_cal, args,
                                                      self.workload)
            else:
                self.last_return_code = cmd_cal.tree.tree(self.sql_session,
                                                      self.cwd_cal_workload,
                                                      args, self.workload)
        else:
            self.default("tree")
        return self.last_return_code

    def help_ls(self):
        if self.context == JOB:
            cmd_job.ls.usage()
        elif self.context == HOST:
            cmd_hosts.ls.usage()
        elif self.context == ENV:
            cmd_env.ls.usage()
        elif self.context == CAL:
            cmd_cal.ls.usage()

    def complete_ls(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == ENV:
            return cmd_env.complete.complete(self.sql_session,
                                             text, self.workload)
        elif self.context == HOST:
            return cmd_hosts.complete.complete(self.sql_session, text)
        elif self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload)
        else:
            return []

    def do_ls(self, s):
        """List jobset content"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB:
            if self.workload is None:
                self.last_return_code = cmd_job.ls.ls(self.sql_session,
                                                      self.cwd, args,
                                                      self.workload)
            else:
                self.last_return_code = cmd_job.ls.ls(self.sql_session,
                                                      self.cwd_workload, args,
                                                      self.workload)
        elif self.context == HOST:
            self.last_return_code = cmd_hosts.ls.ls(self.sql_session, args)
        elif self.context == ENV:
            self.last_return_code = cmd_env.ls.ls(self.sql_session, args,
                                                  self.workload)
        elif self.context == CAL:
            if self.workload is None:
                self.last_return_code = cmd_cal.ls.ls(self.sql_session,
                                                      self.cwd_cal, args,
                                                      self.workload)
            else:
                self.last_return_code = cmd_cal.ls.ls(self.sql_session,
                                                      self.cwd_cal_workload,
                                                      args, self.workload)
        return self.last_return_code

    def complete_ll(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == ENV:
            return cmd_env.complete.complete(self.sql_session,
                                             text, self.workload)
        elif self.context == HOST:
            return cmd_hosts.complete.complete(self.sql_session, text)
        elif self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload)
        else:
            return []

    def do_ll(self, s):
        return self.do_ls("-l %s" % s)

    def help_rm(self):
        if self.context == JOB:
            cmd_job.rm.usage()
        elif self.context == HOST:
            cmd_hosts.rm.usage()
        elif self.context == ENV:
            cmd_env.rm.usage()
        elif self.context == CAL:
            cmd_cal.rm.usage()

    def complete_rm(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == ENV:
            return cmd_env.complete.complete(self.sql_session,
                                             text, self.workload)
        elif self.context == HOST:
            return cmd_hosts.complete.complete(self.sql_session, text)
        elif self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload)
        else:
            return []

    def do_rm(self, s):
        """Remove jobs/jobsets, hosts, environment groups or calndars"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB:
            if self.workload is None:
                self.last_return_code = cmd_job.rm.rm(self.sql_session,
                                                      self.cwd, args)
            else:
                self.default("rm")
        elif self.context == HOST:
            self.last_return_code = cmd_hosts.rm.rm(self.sql_session, args)
        elif self.context == ENV:
            self.last_return_code = cmd_env.rm.rm(self.sql_session, args,
                                                  self.workload)
        elif self.context == CAL:
            if self.workload is None:
                self.last_return_code = cmd_cal.rm.rm(self.sql_session,
                                                      self.cwd_cal, args,
                                                      self.workload)
            else:
                self.last_return_code = cmd_cal.rm.rm(self.sql_session,
                                                      self.cwd_cal_workload,
                                                      args, self.workload)
        return self.last_return_code

    def help_mv(self):
        if self.context == JOB:
            cmd_job.mv.usage()
        elif self.context == CAL:
            cmd_cal.mv.usage()

    def complete_mv(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload)
        else:
            return []

    def do_mv(self, s):
        """Move (rename) jobs/jobsets and calendars/directories"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB and self.workload is None:
            self.last_return_code = cmd_job.mv.mv(self.sql_session, self.cwd,
                                                  args)
        elif self.context == CAL:
            if self.workload is None:
                self.last_return_code = cmd_cal.mv.mv(self.sql_session,
                                                      self.cwd_cal, args,
                                                      self.workload)
            else:
                self.last_return_code = cmd_cal.mv.mv(self.sql_session,
                                                      self.cwd_cal_workload,
                                                      args, self.workload)
        else:
            self.default("mv")
        return self.last_return_code

    def help_cp(self):
        if self.context == JOB:
            cmd_job.cp.usage()
        elif self.context == CAL:
            cmd_cal.cp.usage()

    def complete_cp(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload)
        else:
            return []

    def do_cp(self, s):
        """Copy jobs/jobsets and calendars/directories"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB and self.workload is None:
            self.last_return_code = cmd_job.cp.cp(self.sql_session, self.cwd,
                                                  args)
        elif self.context == CAL and self.workload is None:
            self.last_return_code = cmd_cal.cp.cp(self.sql_session,
                                                  self.cwd_cal, args)
        else:
            self.default("cp")
        return self.last_return_code

    def help_ln(self):
        cmd_job.ln.usage()

    def complete_ln(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        else:
            return []

    def do_ln(self, s):
        """Create links between jobs and jobsets"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB and self.workload is None:
            self.last_return_code = cmd_job.ln.ln(self.sql_session, self.cwd,
                                                  args)
        else:
            self.default("ln")
        return self.last_return_code

    def help_rmln(self):
        cmd_job.rmln.usage()

    def complete_rmln(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        else:
            return []

    def do_rmln(self, s):
        """Remove links between jobs and jobsets"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB and self.workload is None:
            self.last_return_code = cmd_job.rmln.rmln(self.sql_session,
                                                      self.cwd, args)
        else:
            self.default("rmln")
        return self.last_return_code

    def help_mk(self):
        if self.context == JOB:
            cmd_job.mk.usage()
        elif self.context == HOST:
            cmd_hosts.mk.usage()
        elif self.context == ENV:
            cmd_env.mk.usage()
        elif self.context == CAL:
            cmd_cal.mk.usage()

    def complete_mk(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload, True)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload, True)
        else:
            return []

    def do_mk(self, s):
        """Create new jobs, hosts, environment groups and calendars"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.workload is None:
            if self.context == JOB:
                self.last_return_code = cmd_job.mk.mk(self.sql_session,
                                                      self.cwd, self.cwd_cal,
                                                      args)
            elif self.context == HOST:
                self.last_return_code = cmd_hosts.mk.mk(self.sql_session, args)
            elif self.context == ENV:
                self.last_return_code = cmd_env.mk.mk(self.sql_session, args)
            elif self.context == CAL:
                self.last_return_code = cmd_cal.mk.mk(self.sql_session,
                                                      self.cwd_cal, args)
        else:
            self.default("mk")
        return self.last_return_code

    def help_mkjs(self):
        cmd_job.mkjs.usage()

    def complete_mkjs(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload, True)
        else:
            return []

    def do_mkjs(self, s):
        """Create a new jobset"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB and self.workload is None:
            self.last_return_code = cmd_job.mkjs.mkjs(self.sql_session,
                                                      self.cwd, self.cwd_cal,
                                                      args)
        else:
            self.default("mkjs")
        return self.last_return_code

    def help_mkdir(self):
        cmd_cal.mkdir.usage()

    def complete_mkdir(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload, True)
        else:
            return []

    def do_mkdir(self, s):
        """Create a new calendar directory"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == CAL and self.workload is None:
            self.last_return_code = cmd_cal.mkdir.mkdir(self.sql_session,
                                                        self.cwd_cal, args)
        else:
            self.default("mkdir")
        return self.last_return_code

    def help_rmdir(self):
        cmd_cal.rmdir.usage()

    def complete_rmdir(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload, True)
        else:
            return []

    def do_rmdir(self, s):
        """Remove an empty calendar directory"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == CAL:
            if self.workload is None:
                self.last_return_code = cmd_cal.rmdir.rmdir(self.sql_session,
                                                            self.cwd_cal, args,
                                                            self.workload)
            else:
                self.last_return_code = cmd_cal.rmdir.rmdir(self.sql_session,
                                                        self.cwd_cal_workload,
                                                        args, self.workload)
        else:
            self.default("rmdir")
        return self.last_return_code

    def help_show(self):
        cmd_cal.show.usage()

    def complete_show(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload)
        else:
            return []

    def do_show(self, s):
        """Show a calendar"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == CAL:
            if self.workload is None:
                self.last_return_code = cmd_cal.show.show(self.sql_session,
                                                          self.cwd_cal, args,
                                                          self.workload)
            else:
                self.last_return_code = cmd_cal.show.show(self.sql_session,
                                                         self.cwd_cal_workload,
                                                         args, self.workload)
        else:
            self.default("show")
        return self.last_return_code

    def help_set(self):
        if self.context == JOB:
            if self.workload is None:
                cmd_job.set_options.usage()
            else:
                cmd_job.set_options_w.usage()
        elif self.context == HOST:
            cmd_hosts.set_options.usage()
        elif self.context == ENV:
            cmd_env.set_options.usage()
        elif self.context == CAL:
            cmd_cal.set_options.usage()

    def complete_set(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == ENV:
            return cmd_env.complete.complete(self.sql_session,
                                             text, self.workload)
        elif self.context == HOST:
            return cmd_hosts.complete.complete(self.sql_session, text)
        elif self.context == JOB:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload)
        else:
            return []

    def do_set(self, s):
        """Change job parameters"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB:
            if self.workload is None:
                self.last_return_code = cmd_job.set.set(self.sql_session,
                                                        self.cwd, self.cwd_cal,
                                                        args, self.workload)
            else:
                self.last_return_code = cmd_job.set.set(self.sql_session,
                                                        self.cwd_workload,
                                                        self.cwd_cal_workload,
                                                        args, self.workload)
        elif self.context == HOST:
            self.last_return_code = cmd_hosts.set.set(self.sql_session, args)
        elif self.context == ENV:
            self.last_return_code = cmd_env.set.set(self.sql_session, args,
                                                    self.workload)
        elif self.context == CAL and self.workload is None:
            self.last_return_code = cmd_cal.set.set(self.sql_session,
                                                    self.cwd_cal, args)
        else:
            self.default("set")
        return self.last_return_code

    def help_start(self):
        cmd_wl.start.usage()

    def complete_start(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB and self.workload is not None:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        else:
            return []

    def do_start(self, s):
        """Force the start jobs and jobsets"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB and self.workload is not None:
                self.last_return_code = cmd_wl.start.start(self.sql_session,
                                                        self.cwd_workload,
                                                        args, self.workload)
        else:
            self.default("start")

    def help_stop(self):
        cmd_wl.stop.usage()

    def complete_stop(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == JOB and self.workload is not None:
            return cmd_job.complete.complete(self.sql_session, text, line,
                            self.cwd_workload if self.workload else self.cwd,
                            self.workload)
        else:
            return []

    def do_stop(self, s):
        """Stop jobs and jobsets"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == JOB and self.workload is not None:
                self.last_return_code = cmd_wl.stop.stop(self.sql_session,
                                                        self.cwd_workload,
                                                        args, self.workload)
        else:
            self.default("stop")

    def help_status(self):
        cmd_wl.status.usage()

    def do_status(self, s):
        """List the status of jobs and jobsets"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        self.last_return_code = cmd_wl.status.status(self.sql_session, args,
                                                     self.workload)

    def help_hosts(self):
        print _("""Usage: hosts [CMD]
Enter the hosts sub-command.  If a hosts command is specified (CMD), it is
directly executed instead of entering the sub-command.""")

    def do_hosts(self, s):
        try:
            t = self._my_shlex(s, True)
        except:
            self.last_return_code = 1
            return self.last_return_code
        self.context = HOST
        if not t and config.ISATTY:
            print _("Type exit to leave the hosts sub-command")
            self.prompt = 'hosts> '
        if t:
            cmd = t.pop(0)
            try:
                func = getattr(self, "do_%s" % cmd)
            except AttributeError:
                ret = self.default(cmd)
            else:
                if t:
                    ret = func(s.split(None, 1)[1])
                else:
                    ret = func('')
            self.context = JOB
            return ret

    def help_env(self):
        print _("""Usage: env [CMD]
Enter the environment group sub-command.  If a command is specified (CMD),
it is directly executed instead of entering the sub-command.""")

    def do_env(self, s):
        try:
            t = self._my_shlex(s, True)
        except:
            self.last_return_code = 1
            return self.last_return_code
        self.context = ENV
        if not t and config.ISATTY:
            print _("Type exit to leave the env sub-command")
            if self.workload is None:
                self.prompt = 'env> '
            else:
                self.prompt = '%s env> ' % self.workload
        if t:
            cmd = t.pop(0)
            try:
                func = getattr(self, "do_%s" % cmd)
            except AttributeError:
                ret = self.default(cmd)
            else:
                if t:
                    ret = func(s.split(None, 1)[1])
                else:
                    ret = func('')
            self.context = JOB
            return ret

    def help_cal(self):
        print _("""Usage: cal [CMD]
Enter the calendar sub-command.  If a calendar command is specified (CMD),
it is directly executed instead of entering the sub-command.""")

    def do_cal(self, s):
        try:
            t = self._my_shlex(s, True)
        except:
            self.last_return_code = 1
            return self.last_return_code
        self.context = CAL
        if not t and config.ISATTY:
            print _("Type exit to leave the cal sub-command")
            if self.workload is None:
                self.prompt = 'cal:%s> ' % self.cwd_cal.path
            else:
                self.prompt = '%s cal:%s> ' % (self.workload,
                                               self.cwd_cal_workload.path)
        if t:
            cmd = t.pop(0)
            try:
                func = getattr(self, "do_%s" % cmd)
            except AttributeError:
                ret = self.default(cmd)
            else:
                if t:
                    ret = func(s.split(None, 1)[1])
                else:
                    ret = func('')
            self.context = JOB
            return ret

    def help_whatuses(self):
        if self.context == HOST:
            cmd_hosts.whatuses.usage()
        elif self.context == ENV:
            cmd_env.whatuses.usage()
        elif self.context == CAL:
            cmd_cal.whatuses.usage()

    def complete_whatuses(self, text, line, start_index, end_index):
        """Readline argument completion"""
        if self.context == ENV:
            return cmd_env.complete.complete(self.sql_session,
                                             text, self.workload)
        elif self.context == HOST:
            return cmd_hosts.complete.complete(self.sql_session, text)
        elif self.context == CAL:
            return cmd_cal.complete.complete(self.sql_session, text, line,
                    self.cwd_cal_workload if self.workload else self.cwd_cal,
                    self.workload)
        else:
            return []

    def do_whatuses(self, s):
        """Show what uses the provided host, environment group or calendar"""
        try:
            args = self._my_shlex(s)
        except:
            self.last_return_code = 1
            return self.last_return_code
        if self.context == HOST:
            self.last_return_code = cmd_hosts.whatuses.whatuses(
                                              self.sql_session, args)
        elif self.context == ENV:
            self.last_return_code = cmd_env.whatuses.whatuses(self.sql_session,
                                                              args,
                                                              self.workload)
        elif self.context == CAL:
            if self.workload is None:
                self.last_return_code = cmd_cal.whatuses.whatuses(
                                                              self.sql_session,
                                                              self.cwd_cal,
                                                              args,
                                                              self.workload)
            else:
                self.last_return_code = cmd_cal.whatuses.whatuses(
                                                         self.sql_session,
                                                         self.cwd_cal_workload,
                                                         args,
                                                         self.workload)
        else:
            self.default("whatuses")
        return self.last_return_code

    def help_wl(self):
        cmd_wl.wl.usage()

    def complete_wl(self, text, line, start_index, end_index):
        """Readline argument completion"""
        return cmd_wl.complete.complete(self.sql_session, text)

    def do_wl(self, s):
        """List workloads and change the current workload"""
        try:
            t = self._my_shlex(s, True)
        except:
            self.last_return_code = 1
            return self.last_return_code
        workload = cmd_wl.wl.wl(self.sql_session, t)
        if workload:
            save_cwd_workload = self.cwd_workload
            save_cwd_cal_workload = self.cwd_cal_workload
            save_workload = self.workload
            save_context = self.context
            self.cwd_workload = path.Path(self.sql_session, '/', only_dir=True,
                                          workload=workload)
            self.cwd_cal_workload = path_cal.PathCal(self.sql_session, '/',
                                                     workload=workload)
            self.workload = workload
            self.context = JOB
            t.pop(0)  # Drop the workload (first argument)
            if not t and config.ISATTY:
                print _("Type exit to leave the workload sub-command")
                self.prompt = '%s %s> ' % (workload, self.cwd_workload.path)
            if t:
                # Run the command given in parameter
                cmd = t.pop(0)
                try:
                    func = getattr(self, "do_%s" % cmd)
                except AttributeError:
                    ret = self.default(cmd)
                else:
                    if t:
                        ret = func(s.split(None, 2)[2])
                    else:
                        ret = func('')
                # Set back the previous environment
                self.cwd_workload = save_cwd_workload
                self.cwd_cal_workload = save_cwd_cal_workload
                self.workload = save_workload
                self.context = save_context
                return ret
        return 0

    def help_formula(self):
        print _("""Calendar formula

A calendar formula is a way to define a calendar; it specifies days in a year.
A formula may contains several lines, each of them defining a set of days.
A comment can be added, on a line by itself, and must start by the # character.

The syntax of each line may take several forms.

The day number form
===================

This form can be used to define a day number in a month.  The syntax is as
follow:
        MONTH/DAY[/SHIFT]
with
  MONTH
       month name    Jan, February, ...  Only the first 3 characters are
                     required.
       month number  1, 2, ..., 12
       *             all months of the year.
       range         1-6, Jan-March, 11-4 (ie. from November to April).
       list          1,3,Apr,Oct-Dec
  DAY
       day           day number (1-31)
       *             all the days of the month
       -day          offset from the end of the month.  For instance, -1 is the
                     last day of the month, -2 is the day before the last day
                     of the month
       range         1-20, 15--1 (from the 15th to the last day of the month)
       list          1,7,28

  SHIFT (optional)   DAYNAME+|-NUM,...
       Sat+1         if the day selected by the MONTH/DAY part falls a
                     Saturday, move it by one day (Sunday).
       Mon-2         if the day selected by the MONTH/DAY part falls a Monday,
                     move it to 2 days back (Saturday).
       Sat+2,Sun+1   if the day selected by the MONTH/DAY part falls a Saturday
                     or a Sunday, move it to the next Monday.

Examples:
  Mar/14        14th of March.
  */15          Every 15th.
  Aug/*         Every day in August.
  1-6/23        Every 23rd from January to June. The `-' character is used to
                specify a range.
  Dec/15-31     Every day between the 15th and the 31st of December (included).
  */-1          Last day of every month. A negative day number specifies a
                day from the end of the month. -1 stands for the last day of
                the month, -2 for the day before the last day of the month
                and so on.
Jan,Jun/1,15,-1 The 1st, 15th and last day of January and June.
                The , character is used to specify a list of months or days.
  Oct/23/Sat+2  23rd of October or, if the 23rd is a Saturday, 25th of
                October. The optional last part (Sat+2 in this example)
                specifies that if a day falls on a specific week day, it must
                be moved by the number of specified days.
1/1/Sat+2,Sun+1 1st January or, if it's a week-end (a Saturday or a Sunday),
                the following Monday.
7/4/Sat-1,Sun+1 4th of July. If it's a Saturday then the previous Friday or if
                it's a Sunday then the following Monday.


The weekday name form
=====================

This form can be used to define day names in a month.  The syntax is as
follow:
        MONTH/DAYNAME[/WEEK]
with
  MONTH
       month name    Jan, February, ...  Only the first 3 characters are
                     required.
       month number  1, 2, ..., 12
       *             all months of the year.
       range         1-6, Jan-March, 11-4 (ie. from November to April).
       list          1,3,Apr,Oct-Dec
  DAYNAME
       day name      Mon, Tuesday, ... Only the first 3 characters are
                     required.
       *             all the days of the week
       range         Mon-Fri, Sat-Mon
       list          Mon,Wed,Fri-Sun
  WEEK (optional)
       week number   first, second, third, fourth, fifth or 1, 2, 3, 4, 5
       last or -1    last week of the month

Examples:

  Jan/Mon        Every Monday in January.
  Nov/Fri-Sun    All Fridays, Saturdays and Sundays in November.
  May/Mon/first  First Monday of May. The optional last part (first or 1 in
   or May/Mon/1  this example) specifies the week number in the month.
  Apr/Fri/Last   Last Friday of April.
   or Apr/Fri/-1

The open day form
=================

This syntax is used to define calendars based on open days. The syntax is as
follow:
        MONTH/Open[/OPENDAY]
with
  MONTH
       month name    Jan, February, ...  Only the first 3 characters are
                     required.
       month number  1, 2, ..., 12
       *             all months of the year.
       range         1-6, Jan-March, 11-4 (ie. from November to April).
       list          1,3,Apr,Oct-Dec
  OPENDAY
       day number    first, second, third, fourth, fifth or 1, 2, 3, 4, ...
                     Open day number in the month
       last          last open day of the month.
       negative num  -1, -2, ... Last open day of the month, open day before
                     the last open day of the month.

Examples:

  */Open        All open days. The Open keyword specifies all the days of the
                week except Saturday and Sunday (it is a shortcut for Mon-Fri).
  */Open/Last   The last open day in every month.
   or */Open/-1

The Easter and Paskha form
==========================

This form can be used to define calendars based on Easter and Paskha (Orthodox
Easter). The syntax is as follow:
        Easter[+|-NUM]
        Paskha[+|-NUM]
with
  NUM   number of days before (-) or after (+) Easter or Paskha

Examples:

  Easter         Easter day.
  Easter+2       Two days after Easter.
  Paskha-2       Two days before Paskha.

Logical operator
================

The special operator Not (or !) can be added in front of a formula to inverse
the day selection.

Examples:

  Not Jan/15     Every day of the year except the 15th of January.
   or ! Jan/15
  Not */Wed      Every day except Wednesdays.
        """)
