# Schedwi
# Copyright (C) 2011, 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to print the execution statistics of jobs and jobsets."""

import sys
import getopt

from sqlalchemy.orm.exc import NoResultFound
import sqlalchemy.orm.session

import path
from tables.job_stat import job_stat
from help import print_trim
from simple_queries_job import sql_get_job


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: stat [JOB|JOBSET]...
    Print execution statistics of the provided jobs or jobsets.
    If JOB|JOBSET is not specified, print the statistics of the current jobset.

    Option:
      -h, --help  display this help.
    """))

_FIELD_WIDTH = 36


def get_stat(sql_session, job_id):
    """Retrieve a job or jobset statistics from the database.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param job_id:
                the job ID to look for in the database.
    @return:
                the L{job_stat} object.  If the job statistics cannot
                be found in the database a L{job_stat} object with all
                the values at 0 is returned.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    try:
        st = session.query(job_stat).filter_by(job_id=job_id).one()
    except NoResultFound:
        st = job_stat(0, 0, 0, 0)

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return st


def stat2strings(st):
    """From a L{job_stat} object, returns representation on the values.

    @param st:  the L{job_stat} object
    @return:    a dictionnary with the string representation of the
                stats. ['num_success'], ['num_failed'], ['avg_duration'],
                ['total_duration'] and ['last_duration']
    """
    r = dict()
    tot_runs = st.num_success + st.num_failed
    if tot_runs:
        r['num_success'] = "%d (%.2f%%)" % (st.num_success,
                                            st.num_success * 100.0 / tot_runs)
        r['num_failed'] = "%d (%.2f%%)" % (st.num_failed,
                                           st.num_failed * 100.0 / tot_runs)
    else:
        r['num_success'] = "%d (%.2f%%)" % (st.num_success, 0.0)
        r['num_failed'] = "%d (%.2f%%)" % (st.num_failed, 0.0)

    # Average duration
    if st.num_success:
        i = st.total_duration / st.num_success
        days = i / (60 * 60 * 24)
        i = i % (60 * 60 * 24)
        hours = i / (60 * 60)
        i = i % (60 * 60)
        minutes = i / 60
        secs = i % 60
        r['avg_duration'] = "%dd %02dh %02dm %02ds" % (days, hours,
                                                       minutes, secs)
    else:
        r['avg_duration'] = "%dd %02dh %02dm %02ds" % (0, 0, 0, 0)

    # Total duration (when successful)
    days = st.total_duration / (60 * 60 * 24)
    i = st.total_duration % (60 * 60 * 24)
    hours = i / (60 * 60)
    i = i % (60 * 60)
    minutes = i / 60
    secs = i % 60
    r['total_duration'] = "%dd %02dh %02dm %02ds" % (days, hours,
                                                     minutes, secs)

    # Last duration
    days = st.last_duration / (60 * 60 * 24)
    i = st.last_duration % (60 * 60 * 24)
    hours = i / (60 * 60)
    i = i % (60 * 60)
    minutes = i / 60
    secs = i % 60
    r['last_duration'] = "%dd %02dh %02dm %02ds" % (days, hours,
                                                    minutes, secs)
    return r


def _print_stat(title, val):
    print "%s :" % title.rjust(_FIELD_WIDTH), val


def stat(sql_session, current_cwd, arguments):
    """Display the stat of jobs and jobsets.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the stat command (list
                 of jobs/jobsets)

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("stat: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            paths.extend(path.get_paths(sql_session, arg, current_cwd))
    else:
        paths.append(current_cwd)
    if not paths:
        sys.stderr.write(_("stat: no such job or jobset\n"))
        return 1
    first = True
    for p in paths:
        if first == False:
            sys.stdout.write("\n")
        else:
            first = False
        session = sql_session.open_session()
        try:
            job = sql_get_job(session, p.id[-1])
        except:
            sys.stderr.write(_("stat: no such job or jobset\n"))
            sql_session.cancel_session(session)
            return 1
        st = get_stat(session, p.id[-1])
        sql_session.close_session(session)
        a = stat2strings(st)
        _print_stat(_("Name"), job.name)
        _print_stat(_("Number of successful runs"), a['num_success'])
        _print_stat(_("Number of failed runs"), a['num_failed'])
        _print_stat(_("Average duration of successful runs"),
                    a['avg_duration'])
        _print_stat(_("Total duration (when successful)"), a['total_duration'])
        _print_stat(_("Last duration"), a['last_duration'])
    return 0
