# Schedwi
# Copyright (C) 2011, 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to set host parameters."""

import getopt
import sys
import cert

from sqlalchemy.orm.exc import NoResultFound

import host_utils
from tables.host_environment import host_environment
import environments_utils


def usage_options(with_hostname=True):
    """Print the option list on STDOUT."""
    print _("\nOptions:")
    if with_hostname:
        print _("""  --host=HOSTNAME[:PORT]  host name.
  --port=PORT             port number or port name.""")
    print _("""  --ssl=True|False        use SSL (True) or not (False).
  --sslcert=FILE          load the provided certificate.
  --addenv=ENV_NAME       add an environment group to the host.  This
                          option can be specified several times.
  --delenv=ENV_NAME       remove the provided environment group.
  -f, --force             force even if errors are detected.
  --description=TEXT      a description.
  -h, --help              display this help.
    """)


def usage():
    """Print a usage message on STDOUT."""
    print _("""Usage: set [OPTION]... HOST
Change host parameters.
HOST must have the following format:
  HOSTNAME[:PORT]
For IPv6, the address must be enclosed between square brackets (ie.
[fe80::210:a2ff:fa1d:aabb:ccdd]:2006)""")
    usage_options()


def parse_args(argv):
    """Parse the argument list.

    Argument:
    argv --  argument list to parse

    Exception:
    getopt.getopterror -- option error (unknow option or missing parameter)

    """
    optlist, args = getopt.getopt(argv, "hf",
            [
                "force",
                "host=",
                "port=",
                "ssl=",
                "sslcert=",
                "addenv=",
                "delenv=",
                "description=",
                "help"
            ])
    return (optlist, args)


def set(session, host, opts):
    """Change host parameters.

    Arguments:
    session -- SQLAlchemy session
    host -- host object to change
    opts -- option list (from getopt.getopt() - see parse_args())

    """
    force = False
    port = None
    sslfile = None
    envs = list()
    for o, a in opts:
        if o in ("-f", "--force"):
            force = True
        elif o == "--ssl":
            if a[0] in (_('Y'), _('y'), '1', _('t'), _('T')):
                host.sslenable = 1
            else:
                host.sslenable = 0
        elif o == "--sslcert":
            sslfile = a
        elif o == "--port":
            port = a
        elif o == "--description":
            host.description = a.decode('utf-8')
        elif o in ("-h", "--help"):
            usage()
        elif o == "--host":
            host.hostname = a.decode('utf-8')
        elif o == "--addenv":
            try:
                env = environments_utils.name2env(session, a)
            except NoResultFound:
                sys.stderr.write(_("No such environment %s (--addenv)\n") % a)
                return 1
            try:
                envs.remove(env.id)
            except:
                pass
            envs.append(env.id)
        elif o == "--delenv":
            try:
                env = environments_utils.name2env(session, a)
            except NoResultFound:
                sys.stderr.write(_("No such environment %s (--delenv)\n") % a)
                return 1
            for e in host.host_environment:
                if env.id == e.env_id:
                    host.host_environment.remove(e)
                    break
            envs = filter(lambda id: id != env.id, envs)
    error = False
    if port is not None:
        if host_utils.is_known_port(port) or force == True:
            host.portnum = port.decode('utf-8')
        else:
            sys.stderr.write(
            _("Unknown port (--port=%s). Use --force to bypass this check.\n")
                            % port)
            error = True
    if envs:
        new_envs = list()
        last_position = 0
        for e in host.host_environment:
            if e.position < 0:
                continue
            try:
                # Skip the environment group associated with the host
                # as it is provided by --addenv
                envs.index(e.env_id)
            except:
                pass
            else:
                continue
            if e.position > last_position:
                last_position = e.position
            new_envs.append(e)
        for env_id in envs:
            last_position += 1
            new_envs.append(host_environment(env_id, last_position))
        host.host_environment = new_envs
    if sslfile is not None:
        try:
            certX509 = cert.schedwiX509(sslfile, cert.FILE)
        except IOError, e:
            sys.stderr.write(_("Could not open certificate file: %s\n") % e)
            error = True
        except cert.schedwiX509Error as e:
            sys.stderr.write(_("Invalid certificate file `%s': %s\n") %
                             (sslfile, e))
            error = True
        else:
            h = host.hostname.lower()
            if certX509.get_CN().lower() != h:
                for altname in certX509.get_list_dns():
                    if altname.lower() == h:
                        break
                else:
                    if force == False:
                        sys.stderr.write(_("The hostname (`%s') does not \
match the certificate CN (`%s') nor the certificate alternative DNS names \
(%s)\n")
                         % (host.hostname, certX509.get_CN(),
                            certX509.get_dns()))
                        error = True
            if certX509.not_yet_active() and not force:
                sys.stderr.write(_("The certificate is not yet activated (it \
will be valid on %s)\n")
                         % certX509.get_not_before())
                error = True
            if certX509.has_expired() and not force:
                sys.stderr.write(_("The certificate has expired on %s\n") %
                                 certX509.get_not_after())
                error = True
            host.sslcert = open(sslfile).read()
    return 1 if error else 0
