# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list caendars and calendar directories."""

import sys
import getopt

import config
import console_size
import simple_queries_cal
import path_cal
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: ls [OPTION]... [CAL|DIR]...
    List information about the CALs or DIRs (the current directory by default).
    Sort entries alphabetically.

    Options:
      -l          use a long listing format according to the following schema:
                      - or d     whether the entry is a calendar or a directory
                      NAME       calendar/directory name
      -h, --help  display this help.

    Example:
      > ls -l
      - Every Friday
      - Every Monday
      - Every day
      - First open day of the month
      d Worldwide
    """))

_INTERSPACE = 2


def _max(a, b):
    i = len(b.name)
    if i > a:
        return i
    return a


def _print1cal(session, cal, long_listing):
    """Print the provided calendar name."""
    if not long_listing:
        print cal.name
        return
    print "%(dir)s %(name)s" % \
            {"dir":   'd' if cal.entry_type == 1 else '-',
             "name":     cal.name}


def _print_cals(session, cals, long_listing):
    """Print the given calendars."""
    if config.ISATTY and not long_listing:
        m = reduce(_max, cals, 0) + _INTERSPACE
        w = console_size.width
        n = w / m
        i = 0
        for cal in cals:
            f = m - len(cal.name)
            sys.stdout.write(cal.name + ' ' * f)
            i += 1
            if i == n:
                i = 0
                sys.stdout.write("\n")
        if i:
            sys.stdout.write("\n")
    else:
        for cal in cals:
            _print1cal(session, cal, long_listing)


def ls(sql_session, current_cwd, arguments, workload=None):
    """List calendars.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the ls command (list of calendars)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, 'lh', ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("ls: ") + str(err) + "\n")
        return 1
    long_listing = False
    for o, a in optlist:
        if o == "-l":
            long_listing = True
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path_cal.get_paths(sql_session, arg, current_cwd,
                                   workload=workload)
            if not p:
                sys.stderr.write(
                        _("ls: `%s': no such calendar or directory\n") % arg)
            else:
                paths.extend(p)
    else:
        paths.append(current_cwd)
    if not paths:
        return 1
    session = sql_session.open_session()
    previous = 0
    error = False
    for p in paths:
        try:
            cal = simple_queries_cal.sql_get_cal(session, p.id[-1], workload)
        except:
            sys.stderr.write(_("ls: no such calendar or directory\n"))
            error = True
            continue
        if cal.entry_type == 0:
            if previous == 1:  # The previous item was a directory
                sys.stdout.write("\n")
            _print1cal(session, cal, long_listing)
            previous = 2
        else:
            if len(paths) > 1:
                if previous:
                    sys.stdout.write("\n")
                print "%s:" % p
            previous = 1
            cals = simple_queries_cal.sql_get_cal_children(session, p.id[-1],
                                                           False, workload,
                                                           False)
            _print_cals(session, cals, long_listing)
    sql_session.close_session(session)
    return 0 if not error else 1
