# Schedwi
# Copyright (C) 2011, 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to change the current calendar directory."""

import sys
import getopt

from sqlalchemy.orm.exc import NoResultFound

import path_cal
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: cd [-h] [DIR]
        Change the current calendar directory to DIR.
        If DIR is not specified, change to the root directory (`/').
        If DIR is `-', change to the previous working calendar directory.

        Option:
          -h, --help  display this help.
    """))


def cd(sql_session, current_cwd, previous_cwd, arguments, workload=None):
    """Change the current calendar directory.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    previous_cwd -- previous working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the cd command (a directory)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("cd: ") + str(err) + "\n")
        return None
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return None
    if not args:
        new_path = '/'
    else:
        new_path = args[0]
    if new_path == '-':
        if previous_cwd is not None:
            return previous_cwd
        sys.stderr.write(_("cd: no previous calendar directory\n"))
        return None
    try:
        new_cwd = path_cal.PathCal(sql_session, new_path, current_cwd,
                                   only_dir=True, workload=workload)
        if not new_cwd:
            sys.stderr.write(_("cd: no such calendar directory\n"))
            return current_cwd
        return new_cwd
    except NoResultFound:
        sys.stderr.write(_("cd: `%s': no such calendar directory\n")
                         % new_path)
        return None
