/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* thread_init.c -- Thread initialization functions */

#include <schedwi.h>

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#include <pthread.h>

#include <lib_functions.h>
#include <sql_common.h>
#include <thread_init.h>

/*
 * Empty signal handler just to allow the exit from system calls
 */
static void
sigusr2_foo (int sig)
{
                return;
}


/*
 * Initialize the thread
 */
void
thread_init ()
{
	sigset_t mask;
	struct sigaction sa;

	/* 
	 * Be sure the thread can be cancelled only
	 * at cancellation points
	 */
	pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL);
	pthread_setcanceltype (PTHREAD_CANCEL_DEFERRED, NULL);

	/* SQL thread initialization */
	sql_thread_init ();

	/* 
	 * SIGUSR2 handler
	 * To cancel a thread the following functions are called in this order:
	 *     pthread_cancel (<thread_id>)
	 *     pthread_kill(<thread_id>, SIGUSR2)
	 *     pthread_join(<thread_id>, NULL)
	 * The pthread_kill() is required by some thread to exit from
	 * blocking system calls (like accept() or sleep()) to take the
	 * cancellation into account.  Without this call to pthread_kill()
	 * the cancelling thread will stay stuck waiting for the system call
	 * to end.
	 */
	schedwi_memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sigusr2_foo;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	sigaction (SIGUSR2, &sa, NULL);

	/* Remove the mask for SIGUSR2 */
	sigemptyset (&mask);
	sigaddset (&mask, SIGUSR2);
	sigprocmask (SIG_UNBLOCK, &mask, 0);
}


/*
 * Cleanup the thread
 */
void
thread_clean (void *foo)
{
	sql_thread_end ();
}

/*-----------------============== End Of File ==============-----------------*/
