/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* testfile.c -- Test the presence of a file on a client */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <lwc_log.h>
#include <net_utils.h>
#include <net_module_name.h>
#include <sql_common.h>
#include <sql_hosts.h>
#include <request.h>
#include <lib_functions.h>
#include <testfile.h>

/*
 * Error callback function for the sql_host_get_main_row() function
 */
static void
sql_host_get_main_row_error_logger (	void *data, const char *msg,
					unsigned int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while retrieving the client host details"));
	}
}


/*
 * Check the existance of the provided file on the provided host ID
 *
 * Return:
 *   0 --> No error.  The file exists on the remote host.  If not
 *         NULL, out_message contains a reply message sent by the client.  It
 *         may be NULL if the client didn't send back a message.  out_message
 *         must be freed by the caller by free()
 *   1 --> The file does not exists on the remote client (or error on the
 *         client side).  If not NULL, out_message contains a reply message
 *         sent by the client.  It may be NULL if the client didn't send back
 *         a message.  out_message must be freed by the caller by free()
 *  -1 --> Error (a message is logged using lwc_writeLog()).  If not NULL,
 *         out_message contains an error message.  It may be NULL if there
 *         is not enought memory to allocate a buffer for this message.
 *         out_message must be freed by the caller by free()
 */
int
testfile (	int workload_date,
		unsigned long long int job_id,
		unsigned long long int host_id,
		const char *filename,
		char **out_message)
{
	char host_id_str[25];
	startjob_request_ptr request;
	char *err_msg = NULL;
	char **host_details;
	unsigned long int *host_details_len;
	char *module;
	int module_len, ret_read;
	schedwi_BIO *b;
	unsigned int len, ret;
	const char *req;


#if HAVE_ASSERT_H
	assert (filename != NULL);
#endif

	/* Convert the host ID to a string */
	copy_ulltostr (host_id, host_id_str);

	/* Build the request */
	if (request_testfile_init (	workload_date, job_id, host_id_str,
					filename, &request) != 0)
	{
		if (out_message != NULL) {
			err_msg =
			_("Server error: cannot retrieve job parameters");
			*out_message = (char *)malloc (schedwi_strlen (err_msg)
								+ 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}

	/* Get the host details */
	ret = sql_host_get_main_row (	host_id_str,
					&host_details, &host_details_len,
					sql_host_get_main_row_error_logger,
					NULL);
	if (ret != 0) {
		startjob_request_destroy (request);
		if (out_message != NULL) {
			err_msg =
		_("Server error: cannot retrieve the remote agent details");
			*out_message = (char *)malloc (
					schedwi_strlen (err_msg) + 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}

	/* Compose the module name */
	module = NULL;
	module_len = build_module_name ("testfile-0.1", &module);
	if (module_len < 0) {
		sql_free_row (host_details);
		free (host_details_len);
		startjob_request_destroy (request);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		if (out_message != NULL) {
			err_msg =
				_("Server error: memory allocation error");
			*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}

	/* Establish the connection to the client */
	b = net_client (host_details[2],  /* TCP port */
			host_details[1],  /* Client host name */
			(host_details[3][0] == '0') ? 0: 1, /* SSL */
			host_details[4],  /* Certificate */
			(unsigned int)(host_details_len[4]));
	free (host_details_len);
	sql_free_row (host_details);
	if (b == NULL) {
		free (module);
		startjob_request_destroy (request);
		if (out_message != NULL) {
			err_msg =
_("Server error: cannot establish the connection with the remote agent");
			*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}

	/* Get the request string */
	len = get_request_string (request, &req);

	/* Send the request */
	if (	   net_write (b, module, (int)module_len) != 0
		|| net_write (b, req, (int)len) != 0
		|| net_write_eof (b) != 0)
	{
		net_close (b);
		free (module);
		startjob_request_destroy (request);
		if (out_message != NULL) {
			err_msg =
	_("Server error: cannot send the request to the remote agent");
			*out_message = (char *)malloc (
						schedwi_strlen (err_msg) + 1);
			if (*out_message != NULL) {
				strcpy (*out_message, err_msg);
			}
		}
		return -1;
	}
	free (module);
	startjob_request_destroy (request);

	/* Read the result */
	ret_read = net_read_result (b, out_message);
	net_close (b);
	return ret_read;
}

/*-----------------============== End Of File ==============-----------------*/
