/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * startjob_request.c -- Build the string which contains the network request
 *                       to start a job
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lwc_log.h>
#include <lib_functions.h>
#include <startjob_request.h>


/*
 * Create a new startjob_request_ptr structure
 *
 * Return:
 *   The new structure (to be freed by startjob_request_destroy()) or
 *   NULL in case of memory allocation error
 */
startjob_request_ptr
startjob_request_new ()
{
	startjob_request_ptr ptr;

	ptr = (startjob_request_ptr ) malloc (sizeof (startjob_request));
	if (ptr != NULL) {
		ptr->req = NULL;
		ptr->idx = ptr->size = 0;
	}
	return ptr;
}


/*
 * Destroy (free) the provided startjob_request_ptr structure
 */
void
startjob_request_destroy (startjob_request_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->req != NULL && ptr->size > 0) {
			free (ptr->req);
		}
		free (ptr);
	}
}


/*
 * Add an argument to the string
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to log an
 *         error message)
 */
int
startjob_request_add_argument (	startjob_request_ptr ptr,
				const char *pos, unsigned int len_pos,
				const char *val, unsigned int len_val)
{
	unsigned int len;
	char *tmp;

#if HAVE_ASSERT_H
	assert (ptr != NULL && pos != NULL && val != NULL);
#endif

	/* Check the buffer size */
	len = len_val + len_pos + 7;
	if (len + ptr->idx >= ptr->size) {
		tmp = (char *) realloc (ptr->req, len + ptr->idx + 1024);
		if (tmp == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		ptr->req = tmp;
		ptr->size = len + ptr->idx + 1024;
	}

	/*
	 * Build the argument string.  This string looks like:
	 *    ARG:2:-c prog.cfg
	 */
	strncpy (ptr->req + ptr->idx, "ARG:", 4);
	ptr->idx += 4;
	strncpy (ptr->req + ptr->idx, pos, len_pos);
	ptr->idx += len_pos;
	ptr->req [ptr->idx++] = ':';
	strncpy (ptr->req + ptr->idx, val, len_val);
	ptr->idx += len_val;
	ptr->req [ptr->idx++] = '\n';
	return 0;
}


/*
 * Add an environment variable to the string
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an
 *         error message)
 */
int
startjob_request_add_environment (	startjob_request_ptr ptr,
					const char *pos,
					const char *key, const char *value)
{
	unsigned int len, len_pos, len_key, len_value;
	char *tmp;

#if HAVE_ASSERT_H
	assert (ptr != NULL && pos != NULL && key != NULL && value != NULL);
#endif

	/* Check the buffer space */
	len_pos   = schedwi_strlen (pos);
	len_key   = schedwi_strlen (key);
	len_value = schedwi_strlen (value);
	len = len_pos + len_key + len_value + 8;
	if (len + ptr->idx >= ptr->size) {
		tmp = (char *) realloc (ptr->req, len + ptr->idx + 1024);
		if (tmp == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		ptr->req = tmp;
		ptr->size = len + ptr->idx + 1024;
	}

	/*
	 * Build the environment string.  This string looks like:
	 *    ENV:2:9:PI:3.14
	 */
	strncpy (ptr->req + ptr->idx, "ENV:", 4);
	ptr->idx += 4;
	strncpy (ptr->req + ptr->idx, pos, len_pos);
	ptr->idx += len_pos;
	ptr->req [ptr->idx++] = ':';
	strncpy (ptr->req + ptr->idx, key, len_key);
	ptr->idx += len_key;
	ptr->req [ptr->idx++] = ':';
	strncpy (ptr->req + ptr->idx, value, len_value);
	ptr->idx += len_value;
	ptr->req [ptr->idx++] = '\n';
	return 0;
}


/*
 * Add a parameter to the string
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an
 *         error message)
 */
int
startjob_request_add_parameter (startjob_request_ptr ptr,
				const char *name,
				const char *val, unsigned int len_val)
{
	unsigned int len;
	char *tmp;

#if HAVE_ASSERT_H
	assert (ptr != NULL && name != NULL && val != NULL);
#endif

	/* Check the buffer space */
	len = len_val + 6;
	if (len + ptr->idx >= ptr->size) {
		tmp = (char *) realloc (ptr->req, len + ptr->idx + 1024);
		if (tmp == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		ptr->req = tmp;
		ptr->size = len + ptr->idx + 1024;
	}

	/*
	 * Build the parameter
	 */
	strncpy (ptr->req + ptr->idx, name, 3);
	ptr->idx += 3;
	ptr->req [ptr->idx++] = ':';
	strncpy (ptr->req + ptr->idx, val, len_val);
	ptr->idx += len_val;
	ptr->req [ptr->idx++] = '\n';
	return 0;
}


/*
 * Add the Job ID parameter to the string
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an
 *         error message)
 */
int
startjob_request_add_jobid (	startjob_request_ptr ptr,
				const char *workload,
				const char *job_id_str,
				unsigned int len_job_id_str)
{
	unsigned int len, len_workload;
	char *tmp;

#if HAVE_ASSERT_H
	assert (ptr != NULL && workload != NULL);
#endif

	/* Check the buffer space */
	len_workload = schedwi_strlen (workload);
	len = len_workload + len_job_id_str + 7;
	if (len + ptr->idx >= ptr->size) {
		tmp = (char *) realloc (ptr->req, len + ptr->idx + 1024);
		if (tmp == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		ptr->req = tmp;
		ptr->size = len + ptr->idx + 1024;
	}

	/*
	 * Build the Job ID string
	 */
	strncpy (ptr->req + ptr->idx, "JID:", 4);
	ptr->idx += 4;
	strncpy (ptr->req + ptr->idx, workload, len_workload);
	ptr->idx += len_workload;
	ptr->req [ptr->idx++] = SCHEDWI_WORKLOAD_SEPARATOR;
	strncpy (ptr->req + ptr->idx, job_id_str, len_job_id_str);
	ptr->idx += len_job_id_str;
	ptr->req [ptr->idx++] = '\n';
	return 0;
}


/*
 * Return the size of the string (and the string in req)
 */
unsigned int
get_request_string (startjob_request_ptr ptr, const char **req)
{
	if (ptr == NULL || ptr->req == NULL || ptr->size <= 0) {
		if (req != NULL) {
			*req = NULL;
		}
		return 0;
	}
	ptr->req [ptr->idx] = '\0';
	if (req != NULL) {
		*req = ptr->req;
	}
	return ptr->idx;
}

/*-----------------============== End Of File ==============-----------------*/
