/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* request.c -- Common functions to build a request to a client */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <lib_functions.h>
#include <lwc_log.h>
#include <sql_get_environment.h>
#include <sql_hierarchy.h>
#include <request.h>


/*
 * Callback for the sql_get_environment() function which retrieves the command
 * environment variables
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to log an
 *         error message)
 */
static int
environment_callback (	void *obj, unsigned int pos,
			const char *name, const char *value)
{
	startjob_request_ptr request = (startjob_request_ptr) obj;
	char pos_str[25];

#if HAVE_ASSERT_H
	assert (request != NULL && name != NULL && value != NULL);
#endif

	copy_ulltostr (pos, pos_str);
	return startjob_request_add_environment (	request, pos_str,
							name, value);
}

/*
 * Error callback function for the sql_get_environment() function
 */
static void
sql_get_environment_error_logger (	void *data, const char *msg,
					unsigned int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
_("Database error while trying to retrieve a job environment variables"));
	}
}


/*
 * Create a new request object and initialize it
 *
 * Return:
 *   0 --> No error.  host_id_out, request_out and hierarchy_list_out are set
 *         by this function and must be freed by the caller by
 *             free (host_id_out)
 *             startjob_request_destroy(request_out)
 *             lwc_delLL(hierarchy_list_out, (void (*)(const void *))free)
 *  -1 --> Error.  An error message has been logged by lwc_writeLog().
 */
int
request_job_init (int workload_date, unsigned long long int job_id,
		char **host_id_out,
		startjob_request_ptr *request_out,
		lwc_LL **hierarchy_list_out)
{
	lwc_LL *hierarchy_list;
	startjob_request_ptr request;
	char workload_str[25], job_id_str[25], *host_id_str, *err_msg = NULL;
	unsigned int len, last_pos;

#if HAVE_ASSERT_H
	assert (   host_id_out != NULL
		&& request_out != NULL && hierarchy_list_out != NULL);
#endif

	/* Create the startjob_request structure */
	request = startjob_request_new ();
	if (request == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* Convert the job ID to a string */
	len = copy_ulltostr (job_id, job_id_str);

	/* Get the job hierarchy */
	hierarchy_list = get_hierarchy_list (	workload_date, job_id_str,
						&err_msg);
	if (hierarchy_list == NULL) {
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
					_("Job %s: cannot retrieve details"),
					job_id_str);
		}
		return -1;
	}

	/* Convert the workload date to a string */
	copy_ulltostr (workload_date, workload_str);

	/* Add the job ID to the startjob_request structure */
	if (startjob_request_add_jobid (request, workload_str,
					job_id_str, len) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		return -1;
	}

	/* Get the host ID associated with the job */
	if  (get_job_parameter (workload_date, hierarchy_list,
				"job_host_s", "host_id",
				&host_id_str, NULL, &err_msg) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (LOG_CRIT,
		_("Job %s: error while retrieving the client host name"),
				job_id_str);
		}
		return -1;
	}
	if (host_id_str == NULL) {
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		lwc_writeLog (LOG_CRIT,
			_("Job %s: client host name missing in the database"),
				job_id_str);
		return -1;
	}

	/* Get the host environment */
	if (sql_get_environment_host (	workload_date, host_id_str,
					environment_callback,
					request,
					&last_pos,
					sql_get_environment_error_logger,
					NULL) != 0)
	{
		free (host_id_str);
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		return -1;
	}

	/* Get the job environment variables */
	if (sql_get_environment (	workload_date, hierarchy_list,
					last_pos,
					environment_callback,
					request,
					sql_get_environment_error_logger,
					NULL) != 0)
	{
		free (host_id_str);
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		return -1;
	}

	*host_id_out = host_id_str;
	*request_out = request;
	*hierarchy_list_out = hierarchy_list;
	return 0;
}


/*
 * Create a new request object (for testing a remote file) and initialize it
 *
 * Return:
 *   0 --> No error.  host_id_out, request_out and hierarchy_list_out are set
 *         by this function and must be freed by the caller by
 *             free (host_id_out)
 *             startjob_request_destroy(request_out)
 *             lwc_delLL(hierarchy_list_out, (void (*)(const void *))free)
 *  -1 --> Error.  An error message has been logged by lwc_writeLog().
 */
int
request_testfile_init (	int workload_date,
			unsigned long long int job_id,
			const char *host_id_str,
			const char *filename,
			startjob_request_ptr *request_out)
{
	lwc_LL *hierarchy_list;
	startjob_request_ptr request;
	char job_id_str[25], *err_msg = NULL;
	unsigned int last_pos;

#if HAVE_ASSERT_H
	assert (   host_id_str != NULL && filename != NULL
		&& request_out != NULL);
#endif

	/* Create the startjob_request structure */
	request = startjob_request_new ();
	if (request == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* Add the filename to the request object */
	if (startjob_request_add_filename (
					request, filename,
					schedwi_strlen (filename)) != 0)
	{
		startjob_request_destroy (request);
		return -1;
	}

	/* Convert the job ID to a string */
	copy_ulltostr (job_id, job_id_str);

	/* Get the job hierarchy */
	hierarchy_list = get_hierarchy_list (	workload_date, job_id_str,
						&err_msg);
	if (hierarchy_list == NULL) {
		startjob_request_destroy (request);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
					_("Job %s: cannot retrieve details"),
					job_id_str);
		}
		return -1;
	}

	/* Get the host environment */
	if (sql_get_environment_host (	workload_date,
					host_id_str, environment_callback,
					request,
					&last_pos,
					sql_get_environment_error_logger,
					NULL) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		return -1;
	}

	/* Get the job environment variables */
	if (sql_get_environment (	workload_date,
					hierarchy_list, last_pos,
					environment_callback,
					request,
					sql_get_environment_error_logger,
					NULL) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		startjob_request_destroy (request);
		return -1;
	}
	lwc_delLL (hierarchy_list, (void (*)(const void *))free);

	*request_out = request;
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
