/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * net_utils_ssl.c -- Useful network SSL functions
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#ifndef HAVE_CLOSESOCKET
#define closesocket(x) close(x)
#endif

#include <gnutls/gnutls.h>
#include <gnutls/x509.h>
#include <gcrypt.h>
#include <pthread.h>
GCRY_THREAD_OPTION_PTHREAD_IMPL;

#include <lib_functions.h>
#include <lwc_log.h>
#include <net_utils_sock.h>
#include <net_utils_ssl.h>


#define BUFFER_LEN 1024
#define ENF_OF_REQUEST "\nbYe"


/* Our certificate and its associated key file */
static char *schedwisrv_crt = NULL;
static char *schedwisrv_key = NULL;

/* Diffie Hellman parameters */
static gnutls_dh_params_t dh_params;

/* RSA parameters */
static gnutls_rsa_params_t rsa_params;

static char params_set = 0;


/*
 * Verify the remote certificate
 *
 * Return:
 *   0 --> No error, certificate verified
 *  -1 --> Error (an error message is displayed using lwc_writeLog())
 */
static int
net_utils_ssl_check_certificate_client (gnutls_session_t *session,
					char **names)
{
	int ret;
	unsigned int status, cert_list_size, i, j;
	gnutls_x509_crt_t cert;
	const gnutls_datum_t *cert_list;
	time_t now;
	char dnsname[512];
	size_t dnsnamesize;
	char found_dnsname;

#if HAVE_ASSERT_H
	assert (names != NULL && session != NULL);
#endif

	/* Verifiy the certificate */
	ret = gnutls_certificate_verify_peers2 (*session, &status);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
	_("SSL: Failed to verify the cerficate of the remote client: %s"),
				gnutls_strerror (ret));
		return -1;
	}

	if (status & GNUTLS_CERT_INVALID) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: The certificate of the remote client is invalid."));
		return -1;
	}

	if (status & GNUTLS_CERT_SIGNER_NOT_FOUND) {
		lwc_writeLog (	LOG_ERR,
_("SSL: The certificate of the remote client does not have a known issuer."));
		return -1;
	}

	if (status & GNUTLS_CERT_REVOKED) {
		lwc_writeLog (	LOG_ERR,
	_("SSL: The certificate of the remote client has been revoked."));
		return -1;
	}

	/* Retrieve the x509 certificate */
	ret = gnutls_x509_crt_init (&cert);
	if (ret != 0) {
		lwc_writeLog (	LOG_ERR,
	_("SSL: Cannot initialize the certificate: %s"),
				gnutls_strerror (ret));
		return -1;
	}

	cert_list = gnutls_certificate_get_peers (*session, &cert_list_size);
	if (cert_list == NULL) {
		lwc_writeLog (LOG_ERR,
			_("SSL: No certificate found for the remote client."));
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	/* Only check the first certificate */
	ret = gnutls_x509_crt_import (	cert, &cert_list[0],
					GNUTLS_X509_FMT_PEM);
	if (	   ret != 0
		&& gnutls_x509_crt_import (	cert, &cert_list[0],
						GNUTLS_X509_FMT_DER) != 0)
	{
		lwc_writeLog (	LOG_ERR,
	_("SSL: Cannot retrieve the certificate of the remote client: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	/* Check the dates */
	now = time (NULL);
	if (gnutls_x509_crt_get_expiration_time (cert) < now) {
		lwc_writeLog (LOG_ERR,
		_("SSL: The certificate of the remote client has expired."));
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	if (gnutls_x509_crt_get_activation_time (cert) > now) {
		lwc_writeLog (	LOG_ERR,
	_("SSL: The certificate of the remote client is not yet activated."));
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	/*
	 * Check the hostname.  One could use the provided
	 * gnutls_x509_crt_check_hostname() function.  However, we will check
	 * all the provided names.
	 * The following lines are then inspired from this function
	 * in GNUTLS (see rfc2818_hostname.c)
	 */
	ret = 0;
	found_dnsname = 0;
	for (i = 0; ret >= 0; i++) {
		dnsnamesize = sizeof (dnsname);
		ret = gnutls_x509_crt_get_subject_alt_name (cert, i,
						dnsname, &dnsnamesize, NULL);
		if (ret == GNUTLS_SAN_DNSNAME) {
			found_dnsname = 1;
			for (j = 0; names[j] != NULL; j++) {
				if (schedwi_strcasecmp (names[j],
							dnsname) == 0)
				{
					gnutls_x509_crt_deinit (cert);
					return 0;
				}
			}
		}
	}

	if (found_dnsname == 0) {
		dnsnamesize = sizeof (dnsname);
		ret = gnutls_x509_crt_get_dn_by_oid (cert,
						GNUTLS_OID_X520_COMMON_NAME,
						0, 0,
						dnsname, &dnsnamesize);
		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
_("SSL: Cannot retrieve the certificate hostname for the remote client: %s"),
				gnutls_strerror (ret));
			gnutls_x509_crt_deinit (cert);
			return -1;
		}
		for (j = 0; names[j] != NULL; j++) {
			if (schedwi_strcasecmp (names[j], dnsname) == 0) {
				gnutls_x509_crt_deinit (cert);
				return 0;
			}
		}
	}

	gnutls_x509_crt_deinit (cert);
	lwc_writeLog (LOG_ERR,
_("SSL: The certificate hostname does not match the known names of the remote client."));
	return -1;
}


/*
 * Create a SSL session connected to a client
 *
 * Return:
 *   0 on success
 *   -1 in case of error (an error message is displayed using lwc_writeLog())
 */
int
net_server_ssl (gnutls_session_t *session, int fd,
		gnutls_certificate_credentials_t *trusted_cert_ptr,
		char **names,
		char *remote_crt, unsigned long int remote_crt_len)
{
	int ret;
	gnutls_datum_t crt;
	const int cert_type_priority[2] = { GNUTLS_CRT_X509, 0 };

#if HAVE_ASSERT_H
	assert (   session != NULL && trusted_cert_ptr != NULL
		&& names != NULL && remote_crt != NULL && remote_crt_len > 0);
#endif

	/* Trusted certificate (the client one).  Try in PEM and DER format */
	ret = gnutls_certificate_allocate_credentials (trusted_cert_ptr);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot initialize the credentials of the remote client: %s"),
			gnutls_strerror (ret));
		return -1;
	}

	crt.data = (unsigned char *)remote_crt;
	crt.size = remote_crt_len;
	ret = gnutls_certificate_set_x509_trust_mem (*trusted_cert_ptr, &crt,
							GNUTLS_X509_FMT_PEM);
	if (	   ret < 0
		&& gnutls_certificate_set_x509_trust_mem (*trusted_cert_ptr,
						&crt, GNUTLS_X509_FMT_DER) < 0)
	{
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot load the certificate of the remote client: %s"),
				gnutls_strerror (ret));
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	/*
	 * Load the local certificate and key files.  Try in PEM and DER
	 * format
	 */
	ret = gnutls_certificate_set_x509_key_file (	*trusted_cert_ptr,
							schedwisrv_crt,
							schedwisrv_key,
							GNUTLS_X509_FMT_PEM);
	if (	   ret != 0
		&& gnutls_certificate_set_x509_key_file (*trusted_cert_ptr,
						schedwisrv_crt,
						schedwisrv_key,
						GNUTLS_X509_FMT_DER) != 0)
	{
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot load the provided certificate `%s' with key `%s': %s"),
			schedwisrv_crt, schedwisrv_key,
			gnutls_strerror (ret));
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	gnutls_certificate_set_dh_params (*trusted_cert_ptr, dh_params);
	gnutls_certificate_set_rsa_export_params (	*trusted_cert_ptr,
							rsa_params);

	/* Initialize the session */
	ret = gnutls_init (session, GNUTLS_SERVER);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot initialize the session with the remote client: %s"),
			gnutls_strerror (ret));
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	/* Use default priority */
	gnutls_set_default_priority (*session);
	gnutls_certificate_type_set_priority (*session, cert_type_priority);

	/* Put the x509 credentials to the current session */
	ret = gnutls_credentials_set (	*session, GNUTLS_CRD_CERTIFICATE,
					*trusted_cert_ptr);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
		_("SSL: Cannot associate certificate with the session: %s"),
			gnutls_strerror (ret));
		gnutls_deinit (*session);
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	/* The remote client certificate is required */
	gnutls_certificate_server_set_request (*session, GNUTLS_CERT_REQUIRE);
	gnutls_dh_set_prime_bits (*session, 1024);

	/* Associate the socket to the session */
	gnutls_transport_set_ptr (*session, (gnutls_transport_ptr_t) fd);

	/* Handshake */
	do {
		ret = gnutls_handshake (*session);
	} while (ret == GNUTLS_E_AGAIN || ret == GNUTLS_E_INTERRUPTED);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
	_("SSL: Connection handshake failed with the remote client: %s"),
				gnutls_strerror (ret));
		gnutls_bye (*session, GNUTLS_SHUT_RDWR);
		gnutls_deinit (*session);
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	/* Check the client name against the received certificate */
	if (net_utils_ssl_check_certificate_client (session, names) != 0) {
		gnutls_bye (*session, GNUTLS_SHUT_RDWR);
		gnutls_deinit (*session);
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}
	return 0;
}


/*
 * Free the memory used by SSL
 */
void
net_destroy_ssl ()
{
	if (schedwisrv_key != NULL) {
		free (schedwisrv_key);
	}
	if (schedwisrv_crt != NULL) {
		free (schedwisrv_crt);
	}
	schedwisrv_key = schedwisrv_crt = NULL;

	if (params_set != 0) {
		gnutls_rsa_params_deinit (rsa_params);
		gnutls_dh_params_deinit (dh_params);
		params_set = 0;
	}
	gnutls_global_deinit ();
}


/*
 * Initialize the Diffie Hellman and RSA algorithms (only required for server)
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged by lwc_writeLog())
 */
int
net_init_server_ssl ()
{
	int ret;

	/* Build the Diffie Hellman parameters */
	ret = gnutls_dh_params_init (&dh_params);
	if  (ret != 0) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: Failed to allocate the Diffie Hellman parameters: %s"),
				gnutls_strerror (ret));
		return -1;
	}

	ret = gnutls_dh_params_generate2 (dh_params, 1024);
	if  (ret != 0) {
		lwc_writeLog (	LOG_ERR,
_("SSL: Failed to generate the parameters for the Diffie Hellman algorithms: %s"),
				gnutls_strerror (ret));
		gnutls_dh_params_deinit (dh_params);
		return -1;
	}

	/* Build the RSA parameters */
	ret = gnutls_rsa_params_init (&rsa_params);
	if  (ret != 0) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: Failed to allocate the RSA parameters: %s"),
				gnutls_strerror (ret));
		gnutls_dh_params_deinit (dh_params);
		return -1;
	}

	ret = gnutls_rsa_params_generate2 (rsa_params, 512);
	if  (ret != 0) {
		lwc_writeLog (	LOG_ERR,
_("SSL: Failed to generate the parameters for the RSA algorithms: %s"),
				gnutls_strerror (ret));
		gnutls_rsa_params_deinit (rsa_params);
		gnutls_dh_params_deinit (dh_params);
		return -1;
	}
	params_set = 1;
	return 0;
}


/*
 * Initialize the SSL layer
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged by lwc_writeLog())
 */
int
net_init_ssl (	const char *schedwisrv_crt_file,
		const char *schedwisrv_key_file,
		char quick_random)
{
	int ret;
	gnutls_certificate_credentials_t x509_cred;

#if HAVE_ASSERT_H
	assert (schedwisrv_crt_file != NULL && schedwisrv_key_file != NULL);
#endif

	/* For the support of threads in libgcrypt */
	gcry_control (GCRYCTL_SET_THREAD_CBS, &gcry_threads_pthread);

	/* Fast random mode (less secure) */
	if (quick_random != 0) {
		gcry_control (GCRYCTL_ENABLE_QUICK_RANDOM, 0);
	}

	/* GNUTLS init */
	ret = gnutls_global_init ();
	if (ret != 0) {
		lwc_writeLog (LOG_CRIT,
			_("SSL: Cannot initialize the SSL library: %s"),
			gnutls_strerror (ret));
		return -1;
	}

	/*
	 * Try the load the provided certificate and its key to be sure that
	 * everything is fine
	 */

	/* Allocate the credentials for the provided certificate */
	ret = gnutls_certificate_allocate_credentials (&x509_cred);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot initialize the credentials for the current host: %s"),
			gnutls_strerror (ret));
		gnutls_global_deinit ();
		return -1;
	}

	/*
	 * Try to load the certificate in PEM format.  If it does not work,
	 * try to load it in DER format
	 */
	ret = gnutls_certificate_set_x509_key_file (	x509_cred,
							schedwisrv_crt_file,
							schedwisrv_key_file,
							GNUTLS_X509_FMT_PEM);
	if (	   ret != 0
		&& gnutls_certificate_set_x509_key_file (x509_cred,
						schedwisrv_crt_file,
						schedwisrv_key_file,
						GNUTLS_X509_FMT_DER) != 0)
	{
		lwc_writeLog (LOG_ERR,
_("SSL: Cannot load the provided certificat `%s' with the key `%s': %s"),
				schedwisrv_crt_file, schedwisrv_key_file,
				gnutls_strerror (ret));
		gnutls_certificate_free_credentials (x509_cred);
		gnutls_global_deinit ();
		return -1;
	}
	gnutls_certificate_free_credentials (x509_cred);

	/* Save the path to the provided files */
	schedwisrv_crt = (char *) malloc (
				schedwi_strlen (schedwisrv_crt_file) + 1);
	if (schedwisrv_crt == NULL) {
		gnutls_global_deinit ();
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	strcpy (schedwisrv_crt, schedwisrv_crt_file);

	schedwisrv_key = (char *) malloc (
				schedwi_strlen (schedwisrv_key_file) + 1);
	if (schedwisrv_key == NULL) {
		free (schedwisrv_crt);
		schedwisrv_crt = NULL;
		gnutls_global_deinit ();
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	strcpy (schedwisrv_key, schedwisrv_key_file);

	return 0;
}


/*
 * Verify the remote certificate
 *
 * Return:
 *   0 --> No error, certificate verified
 *  -1 --> Error (an error message is displayed using lwc_writeLog())
 */
static int
net_utils_ssl_check_certificate (	gnutls_session_t *session,
					const char *port_number,
					const char *hostname)
{
	int ret, i;
	unsigned int status, cert_list_size;
	gnutls_x509_crt_t cert;
	const gnutls_datum_t *cert_list;
	time_t now;
	char dnsname[512];
	size_t dnsnamesize;
	char found_dnsname;

#if HAVE_ASSERT_H
	assert (port_number != NULL && hostname != NULL && session != NULL);
#endif

	/* Verifiy the certificate */
	ret = gnutls_certificate_verify_peers2 (*session, &status);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
	_("SSL: Failed to verify the certificate for %s (port %s): %s"),
				hostname, port_number,
				gnutls_strerror (ret));
		return -1;
	}

	if (status & GNUTLS_CERT_INVALID) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: The certificate for %s (port %s) is invalid."),
				hostname, port_number);
		return -1;
	}

	if (status & GNUTLS_CERT_SIGNER_NOT_FOUND) {
		lwc_writeLog (	LOG_ERR,
_("SSL: The certificate for %s (port %s) does not have a known issuer."),
				hostname, port_number);
		return -1;
	}

	if (status & GNUTLS_CERT_REVOKED) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: The certificate for %s (port %s) has been revoked."),
				hostname, port_number);
		return -1;
	}

	/* Retrieve the x509 certificate */
	ret = gnutls_x509_crt_init (&cert);
	if (ret != 0) {
		lwc_writeLog (	LOG_ERR,
	_("SSL: Cannot initialize the certificate for %s (port %s): %s"),
				hostname, port_number,
				gnutls_strerror (ret));
		return -1;
	}

	cert_list = gnutls_certificate_get_peers (*session, &cert_list_size);
	if (cert_list == NULL) {
		lwc_writeLog (	LOG_ERR,
			_("SSL: No certificate found for %s (port %s)"),
				hostname, port_number);
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	/* Only check the first certificate */
	ret = gnutls_x509_crt_import (	cert, &cert_list[0],
					GNUTLS_X509_FMT_DER);
	if (ret != 0) {
		lwc_writeLog (	LOG_ERR,
	_("SSL: Cannot retrieve the certificate for %s (port %s): %s"),
				hostname, port_number,
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	/* Check the dates */
	now = time (NULL);
	if (gnutls_x509_crt_get_expiration_time (cert) < now) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: The certificate for %s (port %s) has expired."),
				hostname, port_number);
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	if (gnutls_x509_crt_get_activation_time (cert) > now) {
		lwc_writeLog (	LOG_ERR,
	_("SSL: The certificate for %s (port %s) is not yet activated."),
				hostname, port_number);
		gnutls_x509_crt_deinit (cert);
		return -1;
	}

	/* Log a message if the certificate expires in less than a month */
	if (gnutls_x509_crt_get_expiration_time (cert)
			< now + 30 * 24 * 60 * 60)
	{
		lwc_writeLog (	LOG_NOTICE,
_("SSL: The certificate for %s (port %s) is going to expire in less than a month."),
				hostname, port_number);
	}

	/*
	 * Check the hostname.  One could use the provided
	 * gnutls_x509_crt_check_hostname() function.  However, we will check
	 * all the known names of the host (from the DNS, /etc/hosts, ...)
	 * The following lines are then inspired from this function
	 * in GNUTLS (see rfc2818_hostname.c)
	 */
	found_dnsname = 0;
	for (i = ret = 0; ret >= 0; i++) {
		dnsnamesize = sizeof (dnsname);
		ret = gnutls_x509_crt_get_subject_alt_name (cert, i,
						dnsname, &dnsnamesize, NULL);
		if (ret == GNUTLS_SAN_DNSNAME) {
			found_dnsname = 1;
			if (hostname_cmp (hostname, dnsname) == 0) {
				gnutls_x509_crt_deinit (cert);
				return 0;
			}
		}
	}

	if (found_dnsname == 0) {
		dnsnamesize = sizeof (dnsname);
		ret = gnutls_x509_crt_get_dn_by_oid (cert,
						GNUTLS_OID_X520_COMMON_NAME,
						0, 0,
						dnsname, &dnsnamesize);
		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
_("SSL: Cannot retrieve the certificate hostname for %s (port %s): %s"),
				hostname, port_number,
				gnutls_strerror (ret));
			gnutls_x509_crt_deinit (cert);
			return -1;
		}
		if (hostname_cmp (hostname, dnsname) == 0) {
			gnutls_x509_crt_deinit (cert);
			return 0;
		}
	}

	gnutls_x509_crt_deinit (cert);
	lwc_writeLog (	LOG_ERR,
	_("SSL: The certificate hostname does not match %s (port %s)."),
			hostname, port_number);
	return -1;
}


/*
 * Create a SSL session connected to a server
 * port_number is the port number to connect to (a number or a service name
 *             in /etc/services)
 *
 * Return:
 *   0 on success
 *   -1 in case of error (an error message is displayed using lwc_writeLog())
 */
int
net_client_ssl (const char *port_number, const char *hostname,
		gnutls_session_t *session, int *fd,
		gnutls_certificate_credentials_t *trusted_cert_ptr,
		char *remote_crt, unsigned int remote_crt_len)
{
	int ret;
	gnutls_datum_t crt;
	const int cert_type_priority[2] = { GNUTLS_CRT_X509, 0 };

#if HAVE_ASSERT_H
	assert (   port_number != NULL && hostname != NULL
		&& session != NULL && fd != NULL && trusted_cert_ptr != NULL
		&& remote_crt != NULL && remote_crt_len > 0);
#endif

	/* Trusted certificate (the server one).  Try in PEM and DER format */
	ret = gnutls_certificate_allocate_credentials (trusted_cert_ptr);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot initialize the credentials for %s (port %s): %s"),
			hostname, port_number,
			gnutls_strerror (ret));
		return -1;
	}

	crt.data = (unsigned char *)remote_crt;
	crt.size = remote_crt_len;
	ret = gnutls_certificate_set_x509_trust_mem (*trusted_cert_ptr, &crt,
							GNUTLS_X509_FMT_PEM);
	if (	   ret < 0
		&& gnutls_certificate_set_x509_trust_mem (*trusted_cert_ptr,
						&crt, GNUTLS_X509_FMT_DER) < 0)
	{
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot load the certificate for %s (port %s): %s"),
				hostname, port_number,
				gnutls_strerror (ret));
		gnutls_certificate_free_credentials ( *trusted_cert_ptr);
		return -1;
	}

	/*
	 * Load the local certificate and key files.  Try in PEM and DER
	 * format
	 */
	ret = gnutls_certificate_set_x509_key_file (	*trusted_cert_ptr,
							schedwisrv_crt,
							schedwisrv_key,
							GNUTLS_X509_FMT_PEM);
	if (	   ret != 0
		&& gnutls_certificate_set_x509_key_file (*trusted_cert_ptr,
						schedwisrv_crt,
						schedwisrv_key,
						GNUTLS_X509_FMT_DER) != 0)
	{
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot load the provided certificate `%s' with key `%s': %s"),
			schedwisrv_crt, schedwisrv_key,
			gnutls_strerror (ret));
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	/* Initialize the session */
	ret = gnutls_init (session, GNUTLS_CLIENT);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
	_("SSL: Cannot initialize the session for %s (port %s): %s"),
			hostname, port_number,
			gnutls_strerror (ret));
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	/* Use default priority */
	gnutls_set_default_priority (*session);
	gnutls_certificate_type_set_priority (*session, cert_type_priority);

	/* Put the x509 credentials to the current session */
	ret = gnutls_credentials_set (	*session, GNUTLS_CRD_CERTIFICATE,
					*trusted_cert_ptr);
	if (ret != 0) {
		lwc_writeLog (LOG_ERR,
_("SSL: Cannot associate certificate with the session for %s (port %s): %s"),
			hostname, port_number,
			gnutls_strerror (ret));
		gnutls_deinit (*session);
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	/* Connect to the peer */
	*fd = net_client_sock (port_number, hostname);
	if (*fd < 0) {
		gnutls_deinit (*session);
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	/* Associate the socket to the session */
	gnutls_transport_set_ptr (*session, (gnutls_transport_ptr_t) *fd);

	/* Handshake */
	do {
		ret = gnutls_handshake (*session);
	} while (ret == GNUTLS_E_AGAIN || ret == GNUTLS_E_INTERRUPTED);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
		_("SSL: Connection handshake failed with %s (port %s): %s"),
				hostname, port_number,
				gnutls_strerror (ret));
		gnutls_bye (*session, GNUTLS_SHUT_RDWR);
		closesocket (*fd);
		gnutls_deinit (*session);
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}

	/* Check the certificate */
	if (net_utils_ssl_check_certificate (	session,
						port_number, hostname) != 0)
	{
		gnutls_bye (*session, GNUTLS_SHUT_RDWR);
		closesocket (*fd);
		gnutls_deinit (*session);
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		return -1;
	}
	return 0;
}


/*
 * Read data from the provided SSL session handler.
 * The data are store in *buff (of size *len). If buffer is NULL or not big
 * enough, it is reallocated to a bigger size. It must be freed by the caller
 *
 * Return:
 *    The total number of characters read and stored in *buff or
 *    -1 on error (a message is displayed using lwc_writeLog())
 *    -2 if too many data are received (a message is displayed using
 *       lwc_writeLog())
 */
ssize_t
net_read_ssl (gnutls_session_t session, char **buff, unsigned int *len)
{
	ssize_t nb_read, total;
	size_t to_read;
	char *b, *tmp;

#if HAVE_ASSERT_H
	assert (buff != NULL && len != NULL);
#endif

	if (*buff == NULL || *len == 0) {
		*buff = (char *) malloc (BUFFER_LEN);
		if (*buff == NULL) {
			*len = 0;
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		*len = BUFFER_LEN;
	}

	total = 0;
	to_read = *len;
	b = *buff;
	do {
		nb_read = gnutls_record_recv (session, b, to_read);
		if (nb_read == 0) {
			return total;
		}

		if (nb_read < 0) {
			if (	   nb_read != GNUTLS_E_AGAIN
				&& nb_read != GNUTLS_E_INTERRUPTED)
			{
				lwc_writeLog (LOG_ERR,
					_("Error while reading data: %s"),
					gnutls_strerror (nb_read));
				return nb_read;
			}
			continue;
		}

		total += nb_read;

		/* Find the EOF tag */
		tmp = schedwi_strnstr (*buff, ENF_OF_REQUEST, total);
		if (tmp != NULL) {
			return (ssize_t)(tmp - *buff);
		}

		if (nb_read < to_read) {
			to_read -= nb_read;
			b += nb_read;
		}
		else {
			if (*len + BUFFER_LEN > SCHEDWI_MAX_BUFFER_LEN) {
				/* Too many data */
				lwc_writeLog (LOG_NOTICE,
		_("Failed to read the network request: buffer overflow"));
				return -2;
			}

			tmp = (char *) realloc (*buff, *len + BUFFER_LEN);
			if (tmp == NULL) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				return -1;
			}
			*buff = tmp;
			*len += BUFFER_LEN;
			b = *buff + total;
			to_read = BUFFER_LEN;
		}
	} while (1);

	return total;
}


/*
 * Write data to the provided SSL session handler.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_ssl (gnutls_session_t session, const char *buff, size_t len)
{
	ssize_t nb_write;

	if (buff == NULL || len == 0) {
		return 0;
	}

	do {
		nb_write = gnutls_record_send (session, buff, len);
		if (nb_write == len) {
			return 0;
		}

		if (nb_write < 0) {
			if (	   nb_write != GNUTLS_E_AGAIN
				&& nb_write != GNUTLS_E_INTERRUPTED)
			{
				lwc_writeLog (LOG_ERR,
					_("Error while sending data: %s"),
					gnutls_strerror (nb_write));
				return -1;
			}
		}
		else {
			buff += nb_write;
			len -= nb_write;
		}

	} while (1);

	return 0;
}


/*
 * Close the SSL session
 */
int
net_close_ssl (	gnutls_session_t session, int fd,
		gnutls_certificate_credentials_t *trusted_cert_ptr)
{
	int ret;

	do {
		ret = gnutls_bye (session, GNUTLS_SHUT_RDWR);
	} while (ret == GNUTLS_E_AGAIN || ret == GNUTLS_E_INTERRUPTED);
	net_close_sock (fd);
	gnutls_deinit (session);
	if (trusted_cert_ptr != NULL) {
		gnutls_certificate_free_credentials (*trusted_cert_ptr);
		free (trusted_cert_ptr);
		trusted_cert_ptr = NULL;
	}
	return 0;
}

/*------------------------======= End Of File =======------------------------*/
