/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
   USA.
*/

/* stopjob.c -- Ask to stop a job */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <lwc_log.h>
#include <net_utils.h>
#include <net_module_name.h>
#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_hosts.h>
#include <lib_functions.h>
#include <stopjob.h>

/*
 * Error callback function for the sql_host_get_main_row() function
 */
static void
sql_host_get_main_row_error_logger (	void *data, const char *msg,
					unsigned int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while retrieving the client host details"));
	}
}


/*
 * Stop a job
 *
 * Return:
 *   0 --> No error.  The TERM signal has been successfully sent to the job
 *         on the remote client.  This does not mean that the job will stopped.
 *         If not NULL, out_message contains a reply message sent by the
 *         client.  It  may be NULL if the client didn't send back a message.
 *         out_message must be freed by the caller by free()
 *   1 --> The client cannot stop the job.  If not NULL, out_message
 *         contains a reply message sent by the client.  It may be NULL if the
 *         client didn't send back a message.  out_message must be freed by
 *         the caller by free()
 *  -1 --> Error (a message is logged using lwc_writeLog()).
 */
int
stopjob (	int workload_date, unsigned long long int job_id,
		char **out_message)
{
	schedwi_BIO *b;
	unsigned int len, len_workload, ret;
	int module_len, ret_read;
	char workload_str[25], job_id_str[25], *host_id_str;
	char *err_msg = NULL, *module, *req;
	lwc_LL *hierarchy_list;
	char **host_details;
	unsigned long int *host_details_len;

	/* Convert the job ID to a string */
	len = copy_ulltostr (job_id, job_id_str);

	/* Get the job hierarchy */
	hierarchy_list = get_hierarchy_list (	workload_date, job_id_str,
						&err_msg);
	if (hierarchy_list == NULL) {
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
					_("Job %s: cannot retrieve details"),
					job_id_str);
		}
		return -1;
	}

	/* Get the host ID associated with the job */
	if  (get_job_parameter (workload_date, hierarchy_list,
				"job_host_s", "host_id",
				&host_id_str, NULL, &err_msg) != 0)
	{
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (LOG_CRIT,
		_("Job %s: error while retrieving the client host name"),
				job_id_str);
		}
		return -1;
	}
	if (host_id_str == NULL) {
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		lwc_writeLog (LOG_CRIT,
			_("Job %s: client host name missing in the database"),
				job_id_str);
		return -1;
	}
	lwc_delLL (hierarchy_list, (void (*)(const void *))free);

	/* Get the host details */
	ret = sql_host_get_main_row (	host_id_str,
					&host_details, &host_details_len,
					sql_host_get_main_row_error_logger,
					NULL);
	free (host_id_str);
	if (ret != 0) {
		return -1;
	}

	/* Compose the module name */
	module = NULL;
	module_len = build_module_name ("killjob-0.1", &module);
	if (module_len < 0) {
		free (host_details_len);
		sql_free_row (host_details);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* Build the full job ID */
	len_workload = copy_ulltostr (workload_date, workload_str);
	len += len_workload + 1;
	req = (char *) malloc (len + 1);
	if (req == NULL) {
		free (module);
		free (host_details_len);
		sql_free_row (host_details);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	strcpy (req, workload_str);
	req[len_workload] = SCHEDWI_WORKLOAD_SEPARATOR;
	strcpy (req + len_workload + 1, job_id_str);

	/* Establish the connection to the client */
	b = net_client (host_details[2],  /* TCP port */
			host_details[1],  /* Client host name */
			(host_details[3][0] == '0') ? 0: 1, /* SSL */
			host_details[4], /* Certificate */
			(unsigned int)(host_details_len[4]));
	free (host_details_len);
	sql_free_row (host_details);
	if (b == NULL) {
		free (req);
		free (module);
		return -1;
	}

	/* Send the request */
	if (	   net_write (b, module, (int)module_len) != 0
		|| net_write (b, req, (int)len) != 0
		|| net_write_eof (b) != 0)
	{
		net_close (b);
		free (req);
		free (module);
		return -1;
	}
	free (req);
	free (module);

	/* Read the result */
	ret_read = net_read_result (b, out_message);
	net_close (b);
	return ret_read;
}

/*-----------------============== End Of File ==============-----------------*/
