/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
   USA.
*/

/* sql_link.c -- functions to retrieve link details from the database */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_link.h>

#define SQL_LINK_LIST "SELECT job_id_source,job_id_destination,required_status FROM link_s WHERE workload_date=%d AND job_id_source IN (%s) ORDER BY job_id_source"


/*
 * Get the links which source jobs/jobsets are in the provided list
 *
 * Return:
 *   0 --> No error.  rows contains the retrieved links and must be freed
 *         by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_link_get_list (	lwc_LL **rows,
			int workload_date,
			const char *id_list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL && id_list != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, rows, NULL,
				SQL_LINK_LIST,
				SQL_INT, (long int)workload_date,
				SQL_STRING_NON_ESCAPE, id_list,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
