/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* load_workload_class.c -- Functions to load and run the workloads */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lwc_log.h>
#include <calendar_list.h>
#include <sql_status.h>
#include <load_workload_class.h>


/*
 * Error callback function for the sql_status_workload_list() function
 */
static void
sql_error_logger (void *data, const char *msg, unsigned int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
	    _("Database error while trying to retrieve the workloads"));
	}
}


/*
 * Error callback function for the sql_status_delete() function
 */
static void
sql_status_delete_error_logger (void *data, const char *msg,
				unsigned int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		    _("Database error while trying to delete a workload"));
	}
}


/*
 * Error callback function for the sql_status_copy_to_workload() function
 */
static void
sql_status_copy_error_logger (	void *data, const char *msg,
				unsigned int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
	    _("Database error while trying to build the workload"));
	}
}


/*
 * Fill the provided object list with the workload_class objects in the
 * job_status database table.
 *
 * Return:
 *   0 --> No error.
 *  -1 --> Error.  An error message has been logged by using lwc_writeLog()
 */
int
load_workload_class_from_status_table (load_workload_class_ptr ptr)
{
	lwc_LL *rows;
	char **row;
	schedwi_date workload_date;
	unsigned short int previous_workload_year;
	calendar_list_t_ptr calendars;
	workload_class_ptr workload_ptr;

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	/* Retrieve the ordered workload list */
	if (sql_status_workload_list (&rows, sql_error_logger, NULL) != 0) {
		return -1;
	}

	/* For each workload date in the job_status database table */
	previous_workload_year = 0;
	calendars = NULL;
	pthread_mutex_lock (&(ptr->mutex));
	while ((row = (char **)lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] is the workload date (YYYYMMDD)
		 */

		/*
		 * Convert the date. In case of syntax error a warning message
		 * is logged and the date is ignored
		 */
		if (schedwi_date_from_string (row[0], &workload_date) != 0) {
			lwc_writeLog (LOG_WARNING,
		_("Wrong date format `%s' in the job_status database table"),
					row[0]);
			sql_free_row (row);
			continue;
		}
		sql_free_row (row);

		/* Retrieve the calendar list for the workload year */
		if (workload_date.year != previous_workload_year) {
			destroy_calendar_list (calendars);
			calendars = new_calendar_list (workload_date);
			if (calendars == NULL) {
				lwc_delLL (rows,
					(void (*)(const void *)) sql_free_row);
				lwc_emptyLL (	ptr->list,
						(void (*)(const void *))
						workload_class_destroy);
				pthread_mutex_unlock (&(ptr->mutex));
				return -1;
			}
			previous_workload_year = workload_date.year;
		}

		/* Build the workload_class object */
		workload_ptr = workload_class_new (workload_date, calendars);
		if (workload_ptr == NULL) {
			destroy_calendar_list (calendars);
			lwc_delLL (rows,
				(void (*)(const void *)) sql_free_row);
			lwc_emptyLL (	ptr->list, (void (*)(const void *))
					workload_class_destroy);
			pthread_mutex_unlock (&(ptr->mutex));
			return -1;
		}

		/* Add the new object to the list */
		if (lwc_addEndLL (ptr->list, workload_ptr) != 0) {
			workload_class_destroy (workload_ptr);
			destroy_calendar_list (calendars);
			lwc_delLL (rows,
				(void (*)(const void *)) sql_free_row);
			lwc_emptyLL (	ptr->list, (void (*)(const void *))
					workload_class_destroy);
			pthread_mutex_unlock (&(ptr->mutex));
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}
	pthread_mutex_unlock (&(ptr->mutex));
	lwc_delLL (rows, NULL);
	destroy_calendar_list (calendars);
	return 0;
}


/*
 * Build the workload_class object for the provided date and add it to the
 * list of the provided load_workload_class_ptr object.
 *
 * Return:
 *   0 --> No error.
 *  -1 --> Error.  An error message has been logged by using lwc_writeLog()
 */
int
load_workload_class_for_day (	load_workload_class_ptr ptr,
				schedwi_date workload_date)

{
	calendar_list_t_ptr calendars;
	workload_class_ptr workload_ptr;
	int d;

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	/* Build the workload by copying the tables */
	d = schedwi_date_to_int (workload_date);
	if (sql_status_copy_to_workload (	d,
						sql_status_copy_error_logger,
						NULL) != 0)
	{
		return -1;
	}

	/* Retrieve the calendars */
	calendars = new_calendar_list (workload_date);
	if (calendars == NULL) {
		sql_status_delete (d, NULL, NULL);
		return -1;
	}

	/* Build the workload_class object */
	workload_ptr = workload_class_new (workload_date, calendars);
	destroy_calendar_list (calendars);
	if (workload_ptr == NULL) {
		sql_status_delete (d, NULL, NULL);
		return -1;
	}

	/* Add the new workload_class object to the list */
	pthread_mutex_lock (&(ptr->mutex));
	if (lwc_addEndLL (ptr->list, workload_ptr) != 0) {
		pthread_mutex_unlock (&(ptr->mutex));
		workload_class_destroy (workload_ptr);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		sql_status_delete (d, NULL, NULL);
		return -1;
	}
	pthread_mutex_unlock (&(ptr->mutex));

	return 0;
}


/*
 * Find the workload_class object in the workload list for the provided date
 *
 * Return:
 *   The workload_class_ptr object or
 *   NULL is not found
 */
workload_class_ptr
load_workload_class_find (	load_workload_class_ptr ptr,
				schedwi_date workload_date)
{
	workload_class_ptr workload_ptr;

	if (ptr == NULL) {
		return NULL;
	}

	pthread_mutex_lock (&(ptr->mutex));
	workload_ptr = lwc_searchLL (ptr->list, &workload_date,
	     (int (*)(const void *, const void *))workload_class_date_compar);
	pthread_mutex_unlock (&(ptr->mutex));

	return workload_ptr;
}


/*
 * Create a new load_workload_class_ptr object.
 *
 * Return:
 *   The new object to be freed by the caller by load_workload_class_destroy()
 *   or NULL in case of error (a message has been logged by lwc_writeLog())
 */
load_workload_class_ptr
load_workload_class_new ()
{
	load_workload_class_ptr ptr;

	ptr = (load_workload_class_ptr) malloc (sizeof (load_workload_class));
	if (ptr == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}

	ptr->list = lwc_newLL ();
	if (ptr->list == NULL) {
		free (ptr);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}
	
	if (pthread_mutex_init (&(ptr->mutex), NULL) != 0) {
		lwc_writeLog (  LOG_ERR,
				_("Internal error: mutex initialization: %s"),
				strerror (errno));
		lwc_delLL (ptr->list, NULL);
		free (ptr);
		return NULL;
	}
	return ptr;
}


/*
 * Destroy the provided load_workload_class_ptr object
 */
void
load_workload_class_destroy (load_workload_class_ptr ptr)
{
	if (ptr != NULL) {
		pthread_mutex_lock (&(ptr->mutex));
		lwc_delLL ( ptr->list,
			(void (*)(const void *)) workload_class_destroy);
		ptr->list = NULL;
		pthread_mutex_unlock (&(ptr->mutex));
		pthread_mutex_destroy (&(ptr->mutex));
		free (ptr);
	}
}


/*
 * Remove from the list all the workloads for which retention is over.
 * retention_completed and retention_failed are in days
 */
void
load_workload_class_clean_list (load_workload_class_ptr ptr,
				schedwi_date current_date,
				long int retention_completed,
				long int retention_failed)
{
	workload_class_ptr workload_ptr;
	schedwi_date last_date_completed, last_date_failed;
	int d;

	if (ptr == NULL) {
		return;
	}

	/* Compute the last date to keep */
	schedwi_date_add_days (current_date, &last_date_completed,
				(int)(-retention_completed));
	schedwi_date_add_days (current_date, &last_date_failed,
				(int)(-retention_failed));

	pthread_mutex_lock (&(ptr->mutex));
	lwc_rewindLL (ptr->list);
	while ((workload_ptr = (workload_class_ptr)
				lwc_nextLL (ptr->list)) != NULL)
	{
		/* Workload completed */
		if (workload_class_is_completed (workload_ptr) != 0) {
			/* Retention date passed - delete the workload */
			if (schedwi_date_compar (
						&(workload_ptr->workload_date),
						&last_date_completed) < 0)
			{
				d = schedwi_date_to_int (
						workload_ptr->workload_date);
				lwc_delCurrentLL (ptr->list);
				workload_class_destroy (workload_ptr);
				sql_status_delete (d,
						sql_status_delete_error_logger,
						NULL);
			}
		}
		else {
			/* Retention date passed - delete the workload */
			if (schedwi_date_compar (
						&(workload_ptr->workload_date),
						&last_date_failed) < 0)
			{
				d = schedwi_date_to_int (
						workload_ptr->workload_date);
				lwc_delCurrentLL (ptr->list);
				workload_class_destroy (workload_ptr);
				sql_status_delete (d,
						sql_status_delete_error_logger,
						NULL);
			}
		}
	}
	pthread_mutex_unlock (&(ptr->mutex));
}


/*
 * Print the job/jobset status for each workloads
 */
void
load_workload_class_print (load_workload_class_ptr ptr)
{
	workload_class_ptr workload_ptr;

	if (ptr != NULL) {
		pthread_mutex_lock (&(ptr->mutex));
		lwc_rewindLL (ptr->list);
		while ((workload_ptr = (workload_class_ptr)
					lwc_nextLL (ptr->list)) != NULL)
		{
			workload_class_print (workload_ptr);
		}
		pthread_mutex_unlock (&(ptr->mutex));
	}
}

/******************************** End Of File ********************************/
