/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* commands.c -- commands management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <lwc_log.h>
#include <sql_commands.h>
#include <lib_functions.h>
#include <commands.h>


/*
 * Convert a command to an int
 */
static int
command_to_int (command_t cmd)
{
	switch (cmd) {
		case COMMAND_CHANGE_STATUS:	return 1;
		case COMMAND_STOP:		return 2;
		default:			return 0;
	}
}


/*
 * Convert an int to a command
 */
static command_t
command_from_int (int cmd)
{
	switch (cmd) {
		case 1:		return COMMAND_CHANGE_STATUS;
		case 2:		return COMMAND_STOP;
		default:	return COMMAND_UNKNOWN;
	}
}


/*
 * Create a new command_action object
 *
 * Return:
 *   The new object (to be freed by the caller by command_action_destroy()) or
 *   NULL in case of memory allocation error
 */
static command_action_ptr
command_action_new (	const char *workload_str,
			const char *job_id_str,
			const char *user_details,
			int cmd,
			int parameter)
{
	command_action_ptr ptr;

#if HAVE_ASSERT_H
	assert (workload_str != NULL && job_id_str != NULL);
#endif

	ptr = (command_action_ptr) malloc (sizeof (command_action));
	if (ptr == NULL) {
		return NULL;
	}
	if (user_details != NULL) {
		ptr->user_details = (char *) malloc (
					schedwi_strlen (user_details) + 1);
		if (ptr->user_details == NULL) {
			free (ptr);
			return NULL;
		}
		strcpy (ptr->user_details, user_details);
	}
	else {
		ptr->user_details = NULL;
	}

	schedwi_date_from_string (workload_str, &(ptr->workload));
	ptr->job_id = strtoull (job_id_str, NULL, 0);
	ptr->cmd = command_from_int (cmd);
	ptr->parameter = parameter;
	return ptr;
}


/*
 * Destroy the provided command_action object
 */
static void
command_action_destroy (command_action_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->user_details != NULL) {
			free (ptr->user_details);
		}
		free (ptr);
	}
}


/*
 * Compare two command_action objects.  This function is used by
 * lwc_replaceUnsortedLL()
 */
static int
command_action_cmp (const void *key, const void *obj)
{
	const command_action_ptr k = (command_action_ptr)key;
	const command_action_ptr o = (command_action_ptr)obj;
	int ret;

	ret = schedwi_date_compar (&(k->workload), &(o->workload));
	return (ret != 0)? ret: (k->job_id - o->job_id);
}


/*
 * Error callback function for the sql_command_done() and sql_command_failed()
 * functions
 */
static void
sql_command_update_error_logger (	void *data, const char *msg,
					unsigned int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
	    _("Database error while trying to set the status of a command"));
	}
}


/*
 * Update the status of the command in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
command_done (command_action_ptr ptr)
{
	char id_str[25];
	int wl_date;
	unsigned int ret;

	if (ptr != NULL) {
		wl_date = schedwi_date_to_int (ptr->workload);
		copy_ulltostr (ptr->job_id, id_str);
		ret =  sql_command_done (wl_date, id_str,
					command_to_int (ptr->cmd),
					ptr->parameter,
					sql_command_update_error_logger, NULL);
		return (ret != 0)? -1: 0;
	}
	return 0;
}


/*
 * Update the status of the command in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
command_failed (command_action_ptr ptr)
{
	char id_str[25];
	int wl_date;
	unsigned int ret;

	if (ptr != NULL) {
		wl_date = schedwi_date_to_int (ptr->workload);
		copy_ulltostr (ptr->job_id, id_str);
		ret =  sql_command_failed (wl_date, id_str,
					command_to_int (ptr->cmd),
					ptr->parameter,
					sql_command_update_error_logger, NULL);
		return (ret != 0)? -1: 0;
	}
	return 0;
}


/*
 * Add a new command to the database commands table
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  If not NULL, err_msg contains the error message and
 *         must be freed by the caller by free()
 */
int
command_add (	schedwi_date workload, unsigned long long int job_id,
		const char *username, command_t cmd, int parameter,
		char **err_msg)
{
	char id_str[25];
	int wl_date;

	wl_date = schedwi_date_to_int (workload);
	copy_ulltostr (job_id, id_str);
	if (sql_command_add (	wl_date, id_str,
				username,
				command_to_int (cmd),
				parameter,
				err_msg) != 0)
	{
		return -1;
	}
	return 0;
}


/*
 * Remove the old commands
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  If not NULL, err_msg contains the error message and
 *         must be freed by the caller by free()
 */
int
command_purge (char **err_msg)
{
	if (sql_command_purge (err_msg) != 0) {
		return -1;
	}
	return 0;
}


/*
 * Error callback function for the sql_command_get_commands() function
 */
static void
sql_command_get_error_logger (	void *data, const char *msg,
				unsigned int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		    _("Database error while trying to get the command list"));
	}
}


/*
 * Callback function for the sql_command_get_commands() function.  Add the
 * new command to the list
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error.  An error message has been logged by
 *         lwc_writeLog()
 */
static int
command_get_list_cb (	void *lst,
			const char *workload_str, const char *job_id_str,
			const char *user_name, int cmd, int parameter)
{
	lwc_LL *list = (lwc_LL *) lst;
	command_action_ptr ptr;
	void *old;

	ptr = command_action_new (	workload_str, job_id_str, user_name,
					cmd, parameter);
	if (ptr == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	old = NULL;
	if (lwc_replaceUnsortedLL (list, ptr, command_action_cmp, &old) != 0) {
		command_action_destroy (ptr);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	/* Ignore the previous command for the same job/jobset */
	if (old != NULL) {
		/* Say that it's done */
		command_done ((command_action_ptr)old);
		command_action_destroy ((command_action_ptr)old);
	}
	return 0;
}


/*
 * Retrieve and return the commands to be run
 *
 * Return:
 *    The command list.  Each node is a command_action object.  This list must
 *    be freed by the caller by command_destroy_list()
 *   OR
 *    NULL in case of error (a message has been logged by lwc_writeLog())
 */
lwc_LL *
command_get_list ()
{
	lwc_LL *list;
	unsigned int ret;

	list = lwc_newLL ();
	if (list == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}

	ret = sql_command_get_commands (command_get_list_cb, list,
					sql_command_get_error_logger, NULL);
	if (ret != 0) {
		lwc_delLL (list,
			(void (*)(const void *))command_action_destroy);
		return NULL;
	}


	return list;
}


/*
 * Destroy the provided command list
 */
void
command_destroy_list (lwc_LL *list)
{
	lwc_delLL (list, (void (*)(const void *))command_action_destroy);
}

/*-----------------============== End Of File ==============-----------------*/
