# Schedwi
#   Copyright (C) 2007 Herve Quatremain
#     
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#     
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU Library General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# MySql tables definitions for the Schedwi server
#
# Process this script with the mysql command to create the required tables.
# The mysql command must be run from the directory containing this script.
#
# Example:
#
#  mysql --user=user_name --password=your_password database < schedwi_mysql.sql
#
# The database and the user must have been already created prior to
# running the previous command.  As an example, the following commands create
# a schedwidb database and a schedwi user:
#   mysqladmin --user=root --password=<root_password> create schedwidb
#   mysql --user=root --password=<root_password>
#   mysql> GRANT ALL PRIVILEGES ON schedwidb.*  
#   mysql> TO schedwi@localhost IDENTIFIED BY 'your_password';  
#   mysql> quit 
#

#
# Version of the schedwi database and installation date
#
CREATE TABLE schedwi_version (
 version_number varchar(10),
 installed_date DATETIME,

 UNIQUE (version_number)
);


#
# Main job/jobset parameters
#
#           id --> Job/jobset ID
#       parent --> Parent jobset ID.  O means the jobset is the root one
#         name --> Name
#         type --> 0 (jobset) or 1 (job)
#      enabled --> 0 (disabled - this job/jobset is ignored) or 1 (enabled)
#  description --> Description
#      x and y --> Coordinates of the job/jobset icon in the canvas
#       cal_id --> Calendar ID (see table calendars).  0 means no calendar and
#                  the job/jobset will be run everyday
#   start_time --> Start time (HHMM). For example, for the job to start at
#                  9:17am, the start_time will be 917.  If -1 the job/jobset
#                  will start as soon as its parent jobset is running
#
CREATE TABLE job_main (
 id             bigint UNSIGNED NOT NULL auto_increment,
 parent         bigint UNSIGNED DEFAULT 0 NOT NULL,
 name           varchar(255) DEFAULT '' NOT NULL,
 type           tinyint DEFAULT 1 NOT NULL,
 enabled        tinyint DEFAULT 0 NOT NULL,
 description    text NOT NULL,
 x              int,
 y              int,
 cal_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 start_time     smallint DEFAULT -1 NOT NULL,

 PRIMARY KEY (id)
);


#
# Number of minutes for the job to be started past it start time
# (see start_time in table job_main).  0 means no limit
#
CREATE TABLE job_start_limit (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 start_limit    smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id)
);


#
# Max number of minutes of the job/jobset to complete once running.  0 means
# no limit
#
CREATE TABLE job_max_duration (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 max_duration   smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id)
);


#
# Username of the account under which the job command will be run
# on the remote host
#
CREATE TABLE job_username (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 username       varchar(100) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id)
);


#
# Output file name (stdout) for the job command on the remote host
#
CREATE TABLE job_file_out (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 file_out       varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id)
);


#
# Error file name (stderr) for the job command on the remote host
#
CREATE TABLE job_file_err (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 file_err       varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id)
);


#
# Specifies if the user account environment ($HOME/.profile or $HOME/.basrc for
# instance) on the remote host must be loaded before running the job command
# 0 --> Do not load the environment
# 1 --> Load the environment
#
CREATE TABLE job_loadenv (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 loadenv        tinyint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id)
);


#
# Job command return code which means that the job is successful.  All
# returned code lower or equal to this value mean that the job is successful.
# All returned code greater than this value mean the job failed.
#
CREATE TABLE job_success_return_code (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 success_ret    smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id)
);


#
# Remote command.  If command is `-' then there is no command associated
# with the job and the job will immediately completed.  This may be used
# to define steps like the Start or the End of a chain.
#
CREATE TABLE job_command (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 command        varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id)
);


#
# Command arguments
# A position of -1 means that the command has no arguments.
# If for a specified job there is no row in this table, its parent
# command arguments are used instead (and recursively up to the root
# jobset until arguments are found).
#
CREATE TABLE job_arguments (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 argument       varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id,position)
);


#
# Group of environment variables
#
CREATE TABLE environments (
 id             bigint UNSIGNED NOT NULL auto_increment,
 name           varchar(100) DEFAULT '' NOT NULL,
 description    text NOT NULL,

 PRIMARY KEY (id),
 UNIQUE (name)
);


#
# Environment variables
#
#     env_id --> Group of the environment variable (see table environments)
#   position --> Position in the group
#    env_key --> Name
#  env_value --> Value
#
CREATE TABLE environment_var (
 env_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 env_key        varchar(255) DEFAULT '' NOT NULL,
 env_value      varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (env_id,position)
);


#
# Associate environments with jobs.  A position of -1 means that the job
# does not have any environment.
# The final job environment is the concatenation of all its parents 
# environments plus its associated host environement (see
# table host_environment)
#
CREATE TABLE job_environment (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 env_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id,env_id)
);


#
# Associate environments with a host
#
CREATE TABLE host_environment (
 host_id        bigint UNSIGNED DEFAULT 0 NOT NULL,
 env_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (host_id,env_id)
);


#
# Schedwi client hosts
#
#          id --> Host ID
#    hostname --> Client host name (a name of an address in dot notation) 
#     portnum --> Client TCP port number (or a name in /etc/services)
#   sslenable --> Use SSL for the connexion (1 use SSL or O dont use SSL)
#     sslcert --> Client's certificate (only used if sslenable is 1)
# description --> Host description
#
CREATE TABLE hosts (
 id             bigint UNSIGNED NOT NULL auto_increment,
 hostname       varchar(255) DEFAULT '' NOT NULL,
 portnum        varchar(50) DEFAULT '2006' NOT NULL,
 sslenable      tinyint DEFAULT 1 NOT NULL,
 sslcert        text NOT NULL,
 description    text NOT NULL,

 PRIMARY KEY (id),
 UNIQUE (hostname,portnum)
);


#
# Associate a job/jobset to a host
#
CREATE TABLE job_host (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 host_id        bigint UNSIGNED DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id)
);


#
# Calendar definition.  Calendars are organized in a tree in this table
#
#          id --> Calendar ID
#      parent --> Parent folder
#        name --> Calendar/folder name
#  entry_type --> 0 (calendar) or 1 (folder)
# description --> Host description
#     formula --> Calendar formula (if entry_type is 0)
#      
CREATE TABLE calendars (
  id            bigint UNSIGNED NOT NULL auto_increment,
  parent        bigint UNSIGNED DEFAULT 0 NOT NULL,
  name          varchar(255) DEFAULT '' NOT NULL,
  entry_type    tinyint DEFAULT 0 NOT NULL,
  description   text NOT NULL,
  formula       text NOT NULL,

  PRIMARY KEY (id),
  UNIQUE (parent,name)
);


#
# File name to check
#
#      job_id --> Job ID
#     host_id --> Host ID on which the file presence must be checked
#    filename --> File name (variables are allowed in the name)
#       exist --> 0: file mustn't exist    1: file must exist
#
CREATE TABLE constraint_file (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 host_id        bigint UNSIGNED DEFAULT 0 NOT NULL,
 filename       text,
 exist          tinyint
);


#
# Canvas background image for a jobset
#
#                   job_id --> Jobset ID
#                    image --> Serialized GDK image
#          background_mode --> 0: Solid color
#                              1: Vertical gradient
#                              2: Horizontal gradient
#                              3: Image in top/left cornet
#                              4: Image centered
#                              5: Image tiled all over the canvas
#                              6: Image scaled (ratio width/height keeped)
#                              7: Image stretched to fill all the canvas
#   gradient_color_primary --> Background color (or first color for the
#                              gradient). Format: #RRGGBB
# gradient_color_secondary --> Last color of the gradient
#             canvas_width --> Width of the canvas for this jobset
#            canvas_height --> Height of the canvas for this jobset
#
CREATE TABLE jobset_background (
 job_id                    bigint UNSIGNED DEFAULT 0 NOT NULL,
 image                     MEDIUMBLOB,
 background_mode           tinyint DEFAULT 0 NOT NULL,
 gradient_color_primary    VARCHAR(100) DEFAULT '#FFFFFF' NOT NULL, 
 gradient_color_secondary  VARCHAR(100) DEFAULT '#808080' NOT NULL,
 canvas_width              int DEFAULT 640 NOT NULL,
 canvas_height             int DEFAULT 480 NOT NULL,
 
 PRIMARY KEY (job_id)
);
 

#
# Job icon (used during design)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_default (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Job icon (Completed)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_completed (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Job icon (Failed)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_failed (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Job icon (Running)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_running (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Job icon (Waiting)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_waiting (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Jobset icon (used during design)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_default (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Jobset icon (Completed)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_completed (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Jobset icon (Failed)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_failed (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Jobset icon (Running)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_running (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Jobset icon (Waiting)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_waiting (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 
 PRIMARY KEY (job_id)
);


#
# Chaining between jobs/jobsets
#
CREATE TABLE link (
 job_id_source             bigint UNSIGNED DEFAULT 0 NOT NULL,
 job_id_destination        bigint UNSIGNED DEFAULT 0 NOT NULL,
 required_status           tinyint DEFAULT 1 NOT NULL,

 PRIMARY KEY (job_id_source,job_id_destination)
);


#
# User commands
#
#      workload_date --> Workload date
#             job_id --> Job/jobset ID
#           username --> Name of the user who submits the command
#        insert_time --> Date/time at which the command was added in this table
#     command_status --> 0 (waiting to be run), 1 (Done) or 2 (Failed)
#            command --> 1 (change the status of the job) or 2 (stop the job)
#          parameter --> If command is 1, requested new status for the job
#
CREATE TABLE commands (
 workload_date  int DEFAULT 0 NOT NULL,
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 username       varchar(100) DEFAULT '' NOT NULL,
 insert_time    TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
 command_status tinyint DEFAULT 0 NOT NULL, # 0: To run   1: Done   2: Failed
 command        tinyint DEFAULT 0 NOT NULL,
 parameter      tinyint DEFAULT 0 NOT NULL
);


#
# Statistics
#
CREATE TABLE job_stat (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 num_success    bigint UNSIGNED DEFAULT 0 NOT NULL,
 num_failed     bigint UNSIGNED DEFAULT 0 NOT NULL,
 total_duration bigint UNSIGNED DEFAULT 0 NOT NULL,
 last_duration  bigint UNSIGNED DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id)
);



###
### Tables for the running job status
###

#
# Main job/jobset parameters
#
#  workload_date --> Workload date
#             id --> Job/jobset ID
#         parent --> Parent jobset ID.  O means the jobset is the root one
#           name --> Name
#           type --> 0 (jobset) or 1 (job)
#        enabled --> 0 (disabled - this job/jobset is ignored) or 1 (enabled)
#    description --> Description
#        x and y --> Coordinates of the job/jobset icon in the canvas
#         cal_id --> Calendar ID (see table calendars).  0 means no calendar and
#                    the job/jobset will be run everyday
#     start_time --> Start time (HHMM). For example, for the job to start at
#                    9:17am, the start_time will be 917.  If -1 the job/jobset
#                    will start as soon as its parent jobset is running
#
CREATE TABLE job_main_s (
 id             bigint UNSIGNED DEFAULT 0 NOT NULL,
 parent         bigint UNSIGNED DEFAULT 0 NOT NULL,
 name           varchar(255) DEFAULT '' NOT NULL,
 type           tinyint DEFAULT 1 NOT NULL,
 enabled        tinyint DEFAULT 0 NOT NULL,
 description    text NOT NULL,
 x              int,
 y              int,
 cal_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 start_time     smallint DEFAULT -1 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,id)
);


#
# Number of minutes for the job to be started past it start time
# (see start_time in table job_main_s).  0 means no limit
#
CREATE TABLE job_start_limit_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 start_limit    smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Max number of minutes of the job/jobset to complete once running.  0 means
# no limit
#
CREATE TABLE job_max_duration_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 max_duration   smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Username of the account under which the job command will be run
# on the remote host
#
CREATE TABLE job_username_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 username       varchar(100) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Output file name (stdout) for the job command on the remote host
#
CREATE TABLE job_file_out_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 file_out       varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Error file name (stderr) for the job command on the remote host
#
CREATE TABLE job_file_err_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 file_err       varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Specifies if the user account environment ($HOME/.profile or $HOME/.basrc for
# instance) on the remote host must be loaded before running the job command
# 0 --> Do not load the environment
# 1 --> Load the environment
#
CREATE TABLE job_loadenv_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 loadenv        tinyint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Job command return code which means that the job is successful.  All
# returned code lower or equal to this value mean that the job is successful.
# All returned code greater than this value mean the job failed.
#
CREATE TABLE job_success_return_code_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 success_ret    smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Remote command.  If command is `-' then there is no command associated
# with the job and the job will immediately completed.  This may be used
# to define steps like the Start or the End of a chain.
#
CREATE TABLE job_command_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 command        varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Command arguments
# A position of -1 means that the command has no arguments.
# If for a specified job there is no row in this table, its parent
# command arguments are used instead (and recursively up to the root
# jobset until arguments are found).
#
CREATE TABLE job_arguments_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 argument       varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id,position)
);


#
# Group of environment variables
#
CREATE TABLE environments_s (
 id             bigint UNSIGNED DEFAULT 0 NOT NULL,
 name           varchar(100) DEFAULT '' NOT NULL,
 description    text NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,id),
 UNIQUE (workload_date,name)
);


#
# Environment variables
#
#     env_id --> Group of the environment variable (see table environments_s)
#   position --> Position in the group
#    env_key --> Name
#  env_value --> Value
#
CREATE TABLE environment_var_s (
 env_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 env_key        varchar(255) DEFAULT '' NOT NULL,
 env_value      varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,env_id,position)
);


#
# Associate environments with jobs.  A position of -1 means that the job
# does not have any environment.
# The final job environment is the concatenation of all its parents 
# environments plus its associated host environement (see
# table host_environment_s)
#
CREATE TABLE job_environment_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 env_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id,env_id)
);


#
# Associate environments with a host
#
CREATE TABLE host_environment_s (
 host_id        bigint UNSIGNED DEFAULT 0 NOT NULL,
 env_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,host_id,env_id)
);


#
# Associate a job/jobset to a host
#
CREATE TABLE job_host_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 host_id        bigint UNSIGNED DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Calendar definition.  Calendars are organized in a tree in this table
#
#          id --> Calendar ID
#      parent --> Parent folder
#        name --> Calendar/folder name
#  entry_type --> 0 (calendar) or 1 (folder)
# description --> Host description
#     formula --> Calendar formula (if entry_type is 0)
#      
CREATE TABLE calendars_s (
 id             bigint UNSIGNED DEFAULT 0 NOT NULL,
 parent         bigint UNSIGNED DEFAULT 0 NOT NULL,
 name           varchar(255) DEFAULT '' NOT NULL,
 entry_type     tinyint DEFAULT 0 NOT NULL,
 description    text NOT NULL,
 formula        text NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,id),
 UNIQUE (workload_date,parent,name)
);


#
# File name to check
#
#      job_id --> Job ID
#     host_id --> Host ID on which the file presence must be checked
#    filename --> File name (variables are allowed in the name)
#       exist --> 0: file mustn't exist    1: file must exist
#
CREATE TABLE constraint_file_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 host_id        bigint UNSIGNED DEFAULT 0 NOT NULL,
 filename       text,
 exist          tinyint,
 workload_date  int DEFAULT 0 NOT NULL
);


#
# Current status of jobs/jobsets
#
#      workload_date --> Workload date
#             job_id --> Job/jobset ID
#             status --> 1: Waiting
#                        2: Running
#                        3: Completed
#                        4: Failed
#         start_time --> Date/time at which this status was set
#          error_msg --> Message giving a reason of the current status
#
CREATE TABLE job_status (
 workload_date  int DEFAULT 0 NOT NULL,
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 status         tinyint DEFAULT 1 NOT NULL,
 start_time     bigint UNSIGNED DEFAULT 0 NOT NULL,
 error_msg      varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (workload_date,job_id)
);


#
# Canvas background image for a jobset
#
#                   job_id --> Jobset ID
#                    image --> Serialized GDK image
#          background_mode --> 0: Solid color
#                              1: Vertical gradient
#                              2: Horizontal gradient
#                              3: Image in top/left cornet
#                              4: Image centered
#                              5: Image tiled all over the canvas
#                              6: Image scaled (ratio width/height keeped)
#                              7: Image stretched to fill all the canvas
#   gradient_color_primary --> Background color (or first color for the
#                              gradient). Format: #RRGGBB
# gradient_color_secondary --> Last color of the gradient
#             canvas_width --> Width of the canvas for this jobset
#            canvas_height --> Height of the canvas for this jobset
#
CREATE TABLE jobset_background_s (
 job_id                    bigint UNSIGNED DEFAULT 0 NOT NULL,
 image                     MEDIUMBLOB,
 background_mode           tinyint DEFAULT 0 NOT NULL,
 gradient_color_primary    VARCHAR(100) DEFAULT '#FFFFFF' NOT NULL, 
 gradient_color_secondary  VARCHAR(100) DEFAULT '#808080' NOT NULL,
 canvas_width              int DEFAULT 640 NOT NULL,
 canvas_height             int DEFAULT 480 NOT NULL,
 workload_date             int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);
 

#
# Job icon (used during design)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_default_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Job icon (Completed)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_completed_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Job icon (Failed)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_failed_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Job icon (Running)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_running_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Job icon (Waiting)
# The icon is a serialized GDK image
#
CREATE TABLE job_icon_waiting_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Jobset icon (used during design)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_default_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Jobset icon (Completed)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_completed_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Jobset icon (Failed)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_failed_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Jobset icon (Running)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_running_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Jobset icon (Waiting)
# The icon is a serialized GDK image
#
CREATE TABLE jobset_icon_waiting_s (
 job_id         bigint UNSIGNED DEFAULT 0 NOT NULL,
 icon           MEDIUMBLOB,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id)
);


#
# Chaining between jobs/jobsets
#
CREATE TABLE link_s (
 job_id_source             bigint UNSIGNED DEFAULT 0 NOT NULL,
 job_id_destination        bigint UNSIGNED DEFAULT 0 NOT NULL,
 required_status           tinyint DEFAULT 1 NOT NULL,
 workload_date             int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id_source,job_id_destination)
);



#
# Create the root job
#
INSERT INTO job_main VALUES (1,0,"Root",0,1,"Root Jobset",100,100,4,0);
INSERT INTO job_start_limit VALUES (1, 1440);
INSERT INTO job_max_duration VALUES (1, 0);
INSERT INTO job_username VALUES (1,"nobody");
INSERT INTO job_file_out VALUES (1,"/dev/null");
INSERT INTO job_file_err VALUES (1,"/dev/null");
INSERT INTO job_loadenv  VALUES (1,0);
INSERT INTO job_success_return_code VALUES (1,0);
INSERT INTO job_arguments VALUES (1,-1,"");
INSERT INTO environments VALUES (1,"Default","Default environment");
INSERT INTO job_environment VALUES (1,1,-1);
INSERT INTO hosts (id,hostname,description)
       VALUES (1,"localhost","Local client");
INSERT INTO job_host VALUES (1,1);
INSERT INTO jobset_background VALUES (1,"",2,"#678cc2","#d7e1ef",800,600);
INSERT INTO calendars (id,parent,name,entry_type,formula) VALUES
(1,0,"Calendars",1,""),
(2,1,"Custom",1,""),
(3,1,"Samples",1,""),
(4,3,"Every day",0,"*"),
(5,3,"Every Monday",0,"*/Monday"),
(6,3,"Every Tuesday",0,"*/Tuesday"),
(7,3,"Every Wednesday",0,"*/Wednesday"),
(8,3,"Every Thursday",0,"*/Thursday"),
(9,3,"Every Friday",0,"*/Friday"),
(10,3,"Every Saturday",0,"*/Saturday"),
(11,3,"Every Sunday",0,"*/Sunday"),
(12,3,"Every weekday",0,"*/Open"),
(13,3,"Every weekend",0,"*/Saturday,Sunday"),
(14,3,"First day of the month",0,"*/1"),
(15,3,"Last day of the month",0,"*/-1"),
(16,3,"First open day of the month",0,"*/Open/First"),
(17,3,"Last open day of the month",0,"*/Open/Last"),
(18,3,"Worldwide",1,""),
(19,18,"Germany",1,""),
(20,19,"New Year's Day",0,"1/1"),
(21,19,"Epiphany",0,"1/6"),
(22,19,"Good Friday",0,"Easter-2"),
(23,19,"Easter",0,"Easter"),
(24,19,"Easter Monday",0,"Easter+1"),
(25,19,"Labour Day",0,"5/1"),
(26,19,"Ascension Day",0,"Easter+39"),
(27,19,"Whit Sunday",0,"Easter+49"),
(28,19,"Whit Monday",0,"Easter+50"),
(29,19,"Corpus Christi Day",0,"Easter+60"),
(30,19,"Assumption Day",0,"8/15"),
(31,19,"German Reunification Day",0,"10/3"),
(32,19,"Reformation Day",0,"10/31"),
(33,19,"All Saints' Day",0,"11/1"),
(34,19,"Repentance Day",0,"11/Wed/Third"),
(35,19,"Christmas Eve",0,"12/24"),
(36,19,"Christmas Day",0,"12/25"),
(37,19,"Second Day of Christmas",0,"12/26"),
(38,19,"New Year's Eve",0,"12/31"),
(39,19,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(40,19,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(41,18,"Czech Republic",1,""),
(42,41,"New Year's Day",0,"1/1"),
(43,41,"Easter",0,"Easter"),
(44,41,"Easter Monday",0,"Easter+1"),
(45,41,"Labour Day",0,"5/1"),
(46,41,"Liberation Day",0,"5/8"),
(47,41,"Saint Cyril and Saint Methodius Day",0,"7/5"),
(48,41,"Jan Hus Day",0,"7/6"),
(49,41,"Czech Statehood Day",0,"9/28"),
(50,41,"National Day",0,"10/28"),
(51,41,"Struggle for Freedom and Democracy Day",0,"11/17"),
(52,41,"Christmas Eve",0,"12/24"),
(53,41,"Christmas Day",0,"12/25"),
(54,41,"Boxing Day",0,"12/26"),
(55,41,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(56,41,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(57,18,"Belgium",1,""),
(58,57,"New Year's Day",0,"1/1"),
(59,57,"Easter",0,"Easter"),
(60,57,"Easter Monday",0,"Easter+1"),
(61,57,"Labour Day",0,"5/1"),
(62,57,"Ascension Day",0,"Easter+39"),
(63,57,"Whit Sunday",0,"Easter+49"),
(64,57,"Whit Monday",0,"Easter+50"),
(65,57,"Flemish Community Day",0,"7/11"),
(66,57,"National Day",0,"7/21"),
(67,57,"Assumption Day",0,"8/15"),
(68,57,"French Community Day",0,"9/27"),
(69,57,"All Saints' Day",0,"11/1"),
(70,57,"All Souls' Day",0,"11/2"),
(71,57,"Armistice Day",0,"11/11"),
(72,57,"Dynasty Day",0,"11/15"),
(73,57,"Christmas Day",0,"12/25"),
(74,57,"Saint Stephen's Day",0,"12/26"),
(75,57,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(76,57,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(77,18,"Austria",1,""),
(78,77,"New Year's Day",0,"1/1"),
(79,77,"Epiphany",0,"1/6"),
(80,77,"Good Friday",0,"Easter-2"),
(81,77,"Easter",0,"Easter"),
(82,77,"Easter Monday",0,"Easter+1"),
(83,77,"Labour Day",0,"5/1"),
(84,77,"Ascension Day",0,"Easter+39"),
(85,77,"Whit Sunday",0,"Easter+49"),
(86,77,"Whit Monday",0,"Easter+50"),
(87,77,"Corpus Christi Day",0,"Easter+60"),
(88,77,"Assumption Day",0,"8/15"),
(89,77,"National Day",0,"10/26"),
(90,77,"All Saints' Day",0,"11/1"),
(91,77,"Immaculate Conception Day",0,"12/8"),
(92,77,"Christmas Day",0,"12/25"),
(93,77,"Saint Stephen's Day",0,"12/26"),
(94,77,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(95,77,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(96,18,"Bulgaria",1,""),
(97,96,"New Year's Day",0,"1/1"),
(98,96,"Liberation Day",0,"3/3"),
(99,96,"Labour Day",0,"5/1"),
(100,96,"Orthodox Easter",0,"Paskha"),
(101,96,"Orthodox Easter Monday",0,"Paskha+1"),
(102,96,"Saint George's Day",0,"5/6"),
(103,96,"Saint Cyril and Saint Methodius Day",0,"5/24"),
(104,96,"Unification Day",0,"9/6"),
(105,96,"Independence Day",0,"9/22"),
(106,96,"Leaders of the Bulgarian Revival Day",0,"11/1"),
(107,96,"Christmas Eve",0,"12/24"),
(108,96,"Christmas Day",0,"12/25"),
(109,96,"Second Day of Christmas",0,"12/26"),
(110,96,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(111,96,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(112,18,"Croatia",1,""),
(113,112,"New Year's Day",0,"1/1"),
(114,112,"Epiphany",0,"1/6"),
(115,112,"Easter",0,"Easter"),
(116,112,"Easter Monday",0,"Easter+1"),
(117,112,"Labour Day",0,"5/1"),
(118,112,"Antifacism Day",0,"6/22"),
(119,112,"National Day",0,"6/25"),
(120,112,"Thanksgiving Day",0,"8/5"),
(121,112,"Assumption Day",0,"8/15"),
(122,112,"All Saints' Day",0,"11/1"),
(123,112,"Christmas Day",0,"12/25"),
(124,112,"Saint Stephen's Day",0,"12/26"),
(125,112,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(126,112,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(127,18,"Denmark",1,""),
(128,127,"New Year's Day",0,"1/1"),
(129,127,"Maundy Thursday",0,"Easter-3"),
(130,127,"Good Friday",0,"Easter-2"),
(131,127,"Easter",0,"Easter"),
(132,127,"Easter Monday",0,"Easter+1"),
(133,127,"Store Bededag",0,"Easter+26"),
(134,127,"Ascension Day",0,"Easter+39"),
(135,127,"Whit Sunday",0,"Easter+49"),
(136,127,"Whit Monday",0,"Easter+50"),
(137,127,"Constitution Day",0,"6/5"),
(138,127,"Christmas Eve",0,"12/24"),
(139,127,"Christmas Day",0,"12/25"),
(140,127,"Second Day of Christmas",0,"12/26"),
(141,127,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(142,127,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(143,18,"Estonia",1,""),
(144,143,"New Year's Day",0,"1/1"),
(145,143,"Independence Day",0,"2/24"),
(146,143,"Good Friday",0,"Easter-2"),
(147,143,"Easter",0,"Easter"),
(148,143,"Spring Day",0,"5/1"),
(149,143,"Whit Sunday",0,"Easter+49"),
(150,143,"Victory Day",0,"6/23"),
(151,143,"Midsummer's Day",0,"6/24"),
(152,143,"Independence Restoration Day",0,"8/20"),
(153,143,"Christmas Day",0,"12/25"),
(154,143,"Boxing Day",0,"12/26"),
(155,143,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(156,143,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(157,18,"France",1,""),
(158,157,"New Year's Day",0,"1/1"),
(159,157,"Good Friday",0,"Easter-2"),
(160,157,"Easter",0,"Easter"),
(161,157,"Easter Monday",0,"Easter+1"),
(162,157,"Labour Day",0,"5/1"),
(163,157,"Ascension Day",0,"Easter+39"),
(164,157,"Victory Day",0,"5/8"),
(165,157,"Whit Sunday",0,"Easter+49"),
(166,157,"Whit Monday",0,"Easter+50"),
(167,157,"Bastille Day",0,"7/14"),
(168,157,"Assumption Day",0,"8/15"),
(169,157,"All Saints' Day",0,"11/1"),
(170,157,"Armistice Day",0,"11/11"),
(171,157,"Christmas Day",0,"12/25"),
(172,157,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(173,157,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(174,18,"Hungary",1,""),
(175,174,"New Year's Day",0,"1/1"),
(176,174,"Revolution Day",0,"3/15"),
(177,174,"Easter",0,"Easter"),
(178,174,"Easter Monday",0,"Easter+1"),
(179,174,"Labour Day",0,"5/1"),
(180,174,"Whit Sunday",0,"Easter+49"),
(181,174,"Whit Monday",0,"Easter+50"),
(182,174,"Constitution Day",0,"8/20"),
(183,174,"Republic Day",0,"10/23"),
(184,174,"All Saints' Day",0,"11/1"),
(185,174,"Christmas Day",0,"12/25"),
(186,174,"Second Day of Christmas",0,"12/26"),
(187,174,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(188,174,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(189,18,"Italy",1,""),
(190,189,"New Year's Day",0,"1/1"),
(191,189,"Epiphany",0,"1/6"),
(192,189,"Easter",0,"Easter"),
(193,189,"Easter Monday",0,"Easter+1"),
(194,189,"Liberation Day",0,"4/25"),
(195,189,"Labour Day",0,"5/1"),
(196,189,"Republic Day",0,"6/2"),
(197,189,"Assumption Day",0,"8/15"),
(198,189,"All Saints' Day",0,"11/1"),
(199,189,"Immaculate Conception Day",0,"12/8"),
(200,189,"Christmas Day",0,"12/25"),
(201,189,"Saint Stephen's Day",0,"12/26"),
(202,189,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(203,189,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(204,18,"Latvia",1,""),
(205,204,"New Year's Day",0,"1/1"),
(206,204,"Good Friday",0,"Easter-2"),
(207,204,"Easter",0,"Easter"),
(208,204,"Easter Monday",0,"Easter+1"),
(209,204,"Labour Day",0,"5/1"),
(210,204,"Ligo Day",0,"6/23"),
(211,204,"Saint John's Day",0,"6/24"),
(212,204,"National Day",0,"11/18"),
(213,204,"Christmas Day",0,"12/25"),
(214,204,"Saint Stephen's Day",0,"12/26"),
(215,204,"New Year's Eve",0,"12/31"),
(216,204,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(217,204,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(218,18,"The Netherlands",1,""),
(219,218,"New Year's Day",0,"1/1"),
(220,218,"Good Friday",0,"Easter-2"),
(221,218,"Easter",0,"Easter"),
(222,218,"Easter Monday",0,"Easter+1"),
(223,218,"Queen's Day",0,"4/30"),
(224,218,"Liberation Day",0,"5/5/Sat+2,Sun+1"),
(225,218,"Ascension Day",0,"Easter+39"),
(226,218,"Whit Sunday",0,"Easter+49"),
(227,218,"Whit Monday",0,"Easter+50"),
(228,218,"Christmas Day",0,"12/25"),
(229,218,"Boxing Day",0,"12/26"),
(230,218,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(231,218,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(232,18,"Poland",1,""),
(233,232,"New Year's Day",0,"1/1"),
(234,232,"Easter",0,"Easter"),
(235,232,"Easter Monday",0,"Easter+1"),
(236,232,"Labour Day",0,"5/1"),
(237,232,"Constitution Day",0,"5/3"),
(238,232,"Ascension Day",0,"Easter+39"),
(239,232,"Corpus Christi Day",0,"Easter+60"),
(240,232,"Assumption Day",0,"8/15"),
(241,232,"All Saints' Day",0,"11/1"),
(242,232,"Independence Day",0,"11/11"),
(243,232,"Christmas Day",0,"12/25"),
(244,232,"Second Day of Christmas",0,"12/26"),
(245,232,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(246,232,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(247,18,"Portugal",1,""),
(248,247,"New Year's Day",0,"1/1"),
(249,247,"Shrove Tuesday",0,"Easter-47"),
(250,247,"Maundy Thursday",0,"Easter-3"),
(251,247,"Good Friday",0,"Easter-2"),
(252,247,"Easter",0,"Easter"),
(253,247,"Easter Monday",0,"Easter+1"),
(254,247,"Liberation Day",0,"4/25"),
(255,247,"Labour Day",0,"5/1"),
(256,247,"Corpus Christi Day",0,"Easter+60"),
(257,247,"National Day",0,"6/10"),
(258,247,"Saint Anthony's Day",0,"6/13"),
(259,247,"Saint John's Day",0,"6/24"),
(260,247,"Assumption Day",0,"8/15"),
(261,247,"Republic Day",0,"10/5"),
(262,247,"All Saints' Day",0,"11/1"),
(263,247,"Independence Day",0,"12/1"),
(264,247,"Immaculate Conception Day",0,"12/8"),
(265,247,"Christmas Eve",0,"12/24"),
(266,247,"Christmas Day",0,"12/25"),
(267,247,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(268,247,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(269,18,"Romania",1,""),
(270,269,"New Year's Day",0,"1/1"),
(271,269,"Second Day of New Year",0,"1/2"),
(272,269,"Labour Day",0,"5/1"),
(273,269,"Orthodox Easter",0,"Paskha"),
(274,269,"Orthodox Easter Monday",0,"Paskha+1"),
(275,269,"National Day",0,"12/1"),
(276,269,"Christmas Day",0,"12/25"),
(277,269,"Second Day of Christmas",0,"12/26"),
(278,269,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(279,269,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(280,18,"Slovakia",1,""),
(281,280,"New Year's Day",0,"1/1"),
(282,280,"Epiphany",0,"1/6"),
(283,280,"Good Friday",0,"Easter-2"),
(284,280,"Easter",0,"Easter"),
(285,280,"Easter Monday",0,"Easter+1"),
(286,280,"Labour Day",0,"5/1"),
(287,280,"Liberation Day",0,"5/8"),
(288,280,"Saint Cyril and Saint Methodius Day",0,"7/5"),
(289,280,"Slovak National Uprising Anniversary",0,"8/29"),
(290,280,"Constitution Day",0,"9/1"),
(291,280,"Our Lady of Sorrows' Day",0,"9/15"),
(292,280,"All Saints' Day",0,"11/1"),
(293,280,"Freedom and Democracy Day",0,"11/17"),
(294,280,"Christmas Eve",0,"12/24"),
(295,280,"Christmas Day",0,"12/25"),
(296,280,"Boxing Day",0,"12/26"),
(297,280,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(298,280,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(299,18,"Slovenia",1,""),
(300,299,"New Year's Day",0,"1/1"),
(301,299,"Second Day of New Year",0,"1/2"),
(302,299,"Cultural Day",0,"2/8"),
(303,299,"Easter",0,"Easter"),
(304,299,"Easter Monday",0,"Easter+1"),
(305,299,"Resistance Day",0,"4/27"),
(306,299,"Labour Day",0,"5/1-2"),
(307,299,"National Day",0,"6/25"),
(308,299,"Assumption Day",0,"8/15"),
(309,299,"Reformation Day",0,"10/31"),
(310,299,"Remembrance Day",0,"11/1"),
(311,299,"Christmas Day",0,"12/25"),
(312,299,"Independence Day",0,"12/26"),
(313,299,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(314,299,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(315,18,"Spain",1,""),
(316,315,"New Year's Day",0,"1/1"),
(317,315,"Epiphany",0,"1/6"),
(318,315,"Maundy Thursday",0,"Easter-3"),
(319,315,"Good Friday",0,"Easter-2"),
(320,315,"Easter",0,"Easter"),
(321,315,"Easter Monday",0,"Easter+1"),
(322,315,"Labour Day",0,"5/1"),
(323,315,"Assumption Day",0,"8/15"),
(324,315,"National Day",0,"10/12"),
(325,315,"All Saints' Day",0,"11/1"),
(326,315,"Constitution Day",0,"12/6"),
(327,315,"Immaculate Conception Day",0,"12/8"),
(328,315,"Christmas Day",0,"12/25"),
(329,315,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(330,315,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(331,18,"Switzerland",1,""),
(332,331,"New Year's Day",0,"1/1"),
(333,331,"Saint Berchtold's Day",0,"1/2"),
(334,331,"Good Friday",0,"Easter-2"),
(335,331,"Easter",0,"Easter"),
(336,331,"Easter Monday",0,"Easter+1"),
(337,331,"Ascension Day",0,"Easter+39"),
(338,331,"Whit Sunday",0,"Easter+49"),
(339,331,"Whit Monday",0,"Easter+50"),
(340,331,"National Day",0,"8/1"),
(341,331,"Assumption Day",0,"8/15"),
(342,331,"All Saints' Day",0,"11/1"),
(343,331,"Immaculate Conception Day",0,"12/8"),
(344,331,"Christmas Eve",0,"12/24"),
(345,331,"Christmas Day",0,"12/25"),
(346,331,"Saint Stephen's Day",0,"12/26"),
(347,331,"New Year's Eve",0,"12/31"),
(348,331,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(349,331,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(350,18,"United Kingdom",1,""),
(351,350,"New Year's Day",0,"1/1/Sat+2,Sun+1"),
(352,350,"Good Friday",0,"Easter-2"),
(353,350,"Easter",0,"Easter"),
(354,350,"Easter Monday",0,"Easter+1"),
(355,350,"Early May Bank Holiday",0,"5/1/Sat+2,Sun+1"),
(356,350,"Spring Bank Holiday",0,"5/Mon/Last"),
(357,350,"Summer Bank Holiday",0,"8/Mon/Last"),
(358,350,"Christmas Day",0,"12/25"),
(359,350,"Boxing Day",0,"12/26/Sat+2,Sun+1"),
(360,350,"Christmas Holiday",0,"12/25/Sat+3,Sun+2"),
(361,350,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(362,350,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(363,18,"Brazil",1,""),
(364,363,"New Year's Day",0,"1/1"),
(365,363,"Carnival (first day)",0,"Easter-48"),
(366,363,"Carnival (second day)",0,"Easter-47"),
(367,363,"Good Friday",0,"Easter-2"),
(368,363,"Easter",0,"Easter"),
(369,363,"Tiradentes Day",0,"4/21"),
(370,363,"Labour Day",0,"5/1"),
(371,363,"Corpus Christi Day",0,"Easter+60"),
(372,363,"Independence Day",0,"9/7"),
(373,363,"Our Lady Aparecida Day",0,"10/12"),
(374,363,"All Souls' Day",0,"11/2"),
(375,363,"Republic Day",0,"11/15"),
(376,363,"Christmas Day",0,"12/25"),
(377,363,"Daylight Saving Time (start)",0,"11/Sun/First"),
(378,363,"Daylight Saving Time (end)",0,"2/Sun/Third"),
(379,18,"Russia",1,""),
(380,379,"New Year's Day",0,"1/1/Sat+2,Sun+1"),
(381,379,"Second Day of New Year",0,"1/2/Sat+2,Sun+2,Mon+1"),
(382,379,"Orthodox Christmas Day",0,"1/7/Sat+2,Sun+1"),
(383,379,"Orthodox New Year",0,"1/14"),
(384,379,"Defenders of the Motherland Day",0,"2/23/Sat+2,Sun+1"),
(385,379,"Women's Day",0,"3/8/Sat+2,Sun+1"),
(386,379,"Orthodox Good Friday",0,"Paskha-2"),
(387,379,"Orthodox Easter",0,"Paskha"),
(388,379,"Orthodox Easter Monday",0,"Paskha+1"),
(389,379,"Labour Day",0,"5/1-2/Sat+2,Sun+2"),
(390,379,"Victory Day",0,"5/9/Sat+2,Sun+1"),
(391,379,"Independence Day",0,"6/12/Sat+2,Sun+1"),
(392,379,"Accord and Reconciliation Day",0,"11/7/Sat+2,Sun+1"),
(393,379,"Constitution Day",0,"12/12/Sat+2,Sun+1"),
(394,379,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(395,379,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(396,18,"Mexico",1,""),
(397,396,"New Year's Day",0,"1/1"),
(398,396,"Constitution Day",0,"2/5"),
(399,396,"Benito Juarez's Birthday",0,"3/21"),
(400,396,"Maundy Thursday",0,"Easter-3"),
(401,396,"Good Friday",0,"Easter-2"),
(402,396,"Easter",0,"Easter"),
(403,396,"Labour Day",0,"5/1"),
(404,396,"Battle of Puebla Day",0,"5/5"),
(405,396,"President's State of the Nation Address",0,"9/1"),
(406,396,"Independence Day",0,"9/16"),
(407,396,"Dia de la Raza / Columbus Day",0,"10/12"),
(408,396,"All Souls' Day",0,"11/2"),
(409,396,"Revolution Day",0,"11/20"),
(410,396,"Virgin of Guadalupe Day",0,"12/12"),
(411,396,"Christmas Day",0,"12/25"),
(412,396,"Daylight Saving Time (start)",0,"4/Sun/First"),
(413,396,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(414,18,"Philippines",1,""),
(415,414,"New Year's Day",0,"1/1"),
(416,414,"Maundy Thursday",0,"Easter-3"),
(417,414,"Good Friday",0,"Easter-2"),
(418,414,"Easter",0,"Easter"),
(419,414,"Day of Valor (Bataan Day)",0,"4/9"),
(420,414,"Labour Day",0,"5/1"),
(421,414,"Independence Day",0,"6/12"),
(422,414,"Manila Day",0,"12/24"),
(423,414,"Heroes' Day",0,"8/Sun/Last"),
(424,414,"All Saints' Day",0,"11/1"),
(425,414,"Bonifacio Day",0,"11/30"),
(426,414,"Christmas Day",0,"12/25"),
(427,414,"Rizal Day",0,"12/30"),
(428,414,"New Year's Eve",0,"12/31"),
(429,18,"United States",1,""),
(430,429,"New Year's Day",0,"1/1"),
(431,429,"New Year's Day Holiday",0,"1/1/Sun+1"),
(432,429,"Martin Luther King's Day",0,"1/Mon/Third"),
(433,429,"Presidents'Day",0,"2/Mon/Third"),
(434,429,"Memorial Day",0,"5/Mon/Last"),
(435,429,"Independence Day",0,"7/4/Sat-1,Sun+1"),
(436,429,"Labour Day",0,"9/Mon/First"),
(437,429,"Columbus Day",0,"10/Mon/Second"),
(438,429,"Veterans Day",0,"11/11/Sat-1,Sun+1"),
(439,429,"Thanksgiving Day",0,"11/Thu/Fourth"),
(440,429,"Christmas Day",0,"12/25"),
(441,429,"Christmas Holiday",0,"12/25/Sat-1,Sun+1"),
(442,429,"New Year's Day Holiday (next year)",0,"12/25/Sat+6"),
(443,429,"Daylight Saving Time (start)",0,"4/Sun/First"),
(444,429,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(445,18,"Congo (Dem. Rep.)",1,""),
(446,445,"New Year's Day",0,"1/1"),
(447,445,"Martyr's Day",0,"1/4"),
(448,445,"Labour Day",0,"5/1"),
(449,445,"Independence Day",0,"6/30"),
(450,445,"Parents' Day",0,"8/1"),
(451,445,"Youth Day",0,"10/14"),
(452,445,"Army Day",0,"11/17"),
(453,445,"New Regime Day",0,"11/24"),
(454,445,"Christmas Day",0,"12/25"),
(455,18,"Ukraine",1,""),
(456,455,"New Year's Day",0,"1/1/Sat+2,Sun+1"),
(457,455,"Orthodox Christmas Day",0,"1/7/Sat+2,Sun+1"),
(458,455,"Orthodox New Year",0,"1/14/Sun+1"),
(459,455,"Women's Day",0,"3/8/Sat+2,Sun+1"),
(460,455,"Orthodox Good Friday",0,"Paskha-2"),
(461,455,"Orthodox Easter",0,"Paskha"),
(462,455,"Orthodox Easter Monday",0,"Paskha+1"),
(463,455,"Labour Day",0,"5/1/Sat+2,Sun+1"),
(464,455,"Victory Day",0,"5/9/Sat+2,Sun+1"),
(465,455,"Radunitsa (Orthodox Trinity)",0,"Paskha+49"),
(466,455,"Radunitsa Monday",0,"Paskha+50"),
(467,455,"Constitution Day",0,"6/28/Sat+2,Sun+1"),
(468,455,"Independence Day",0,"8/24/Sat+2,Sun+1"),
(469,455,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(470,455,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(471,18,"South Africa",1,""),
(472,471,"New Year's Day",0,"1/1/Sun+1"),
(473,471,"Human Righths's Day",0,"3/21/Sun+1"),
(474,471,"Good Friday",0,"Easter-2"),
(475,471,"Easter",0,"Easter"),
(476,471,"Easter Monday",0,"Easter+1"),
(477,471,"Freedom Day",0,"4/27/Sun+1"),
(478,471,"Workers' Day",0,"5/1/Sun+1"),
(479,471,"Youth Day",0,"6/16/Sun+1"),
(480,471,"Women's Day",0,"8/9/Sun+1"),
(481,471,"Heritage Day",0,"9/25/Sun+1"),
(482,471,"Reconciliation Day",0,"12/16/Sun+1"),
(483,471,"Christmas Day",0,"12/25/Sun+1"),
(484,471,"Day of Goodwill",0,"12/26/Mon+1"),
(485,18,"Colombia",1,""),
(486,485,"New Year's Day",0,"1/1"),
(487,485,"Epiphany",0,"1/Mon/Second"),
(488,485,"San Jose Day",0,"3/Mon/Third"),
(489,485,"Maundy Thursday",0,"Easter-3"),
(490,485,"Good Friday",0,"Easter-2"),
(491,485,"Easter",0,"Easter"),
(492,485,"Labour Day",0,"5/1"),
(493,485,"Ascension Day",0,"Easter+39"),
(494,485,"Sagrado Corazon Day",0,"Easter+71"),
(495,485,"Saint Peter and Saint Paul's Day",0,"7/Mon/First"),
(496,485,"Independence Day",0,"7/20"),
(497,485,"Battle of Boyaca",0,"8/7"),
(498,485,"Assumption Day",0,"8/15"),
(499,485,"Dia de la Raza",0,"10/Mon/Third"),
(500,485,"All Saints' Day",0,"11/Mon/First"),
(501,485,"Cartagena Independence Day",0,"11/Mon/-3"),
(502,485,"Immaculate Conception Day",0,"12/8"),
(503,485,"Christmas Day",0,"12/25"),
(504,18,"Argentina",1,""),
(505,504,"New Year's Day",0,"1/1"),
(506,504,"Good Friday",0,"Easter-2"),
(507,504,"Labour Day",0,"5/1"),
(508,504,"May Day Revolution",0,"5/25"),
(509,504,"Malvinas Day",0,"6/Mon/Second"),
(510,504,"Flag Day",0,"6/Mon/Third"),
(511,504,"Independence Day",0,"7/9"),
(512,504,"Death of General Jose de San Martin",0,"8/Mon/Third"),
(513,504,"Columbus Day",0,"10/12"),
(514,504,"Immaculate Conception Day",0,"12/8"),
(515,504,"Christmas Day",0,"12/25"),
(516,18,"Canada",1,""),
(517,516,"New Year's Day",0,"1/1"),
(518,516,"Good Friday",0,"Easter-2"),
(519,516,"Easter",0,"Easter"),
(520,516,"Easter Monday",0,"Easter+1"),
(521,516,"Victoria Day",0,"5/Mon/-2"),
(522,516,"Saint Jean Baptiste",0,"6/24"),
(523,516,"Canada Day",0,"7/1"),
(524,516,"Civic Holiday",0,"8/Mon/First"),
(525,516,"Labour Day",0,"9/Mon/First"),
(526,516,"Thanksgiving Day",0,"10/Mon/Second"),
(527,516,"Remembrance Day",0,"11/11"),
(528,516,"Christmas Day",0,"12/25"),
(529,516,"Boxing Day",0,"12/26"),
(530,516,"Daylight Saving Time (start)",0,"4/Sun/First"),
(531,516,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(532,18,"Peru",1,""),
(533,532,"New Year's Day",0,"1/1"),
(534,532,"Maundy Thursday",0,"Easter-3"),
(535,532,"Good Friday",0,"Easter-2"),
(536,532,"Easter",0,"Easter"),
(537,532,"Labour Day",0,"5/1"),
(538,532,"Inti Raymi Day",0,"6/24"),
(539,532,"Saint Peter and Saint Paul's Day",0,"6/29"),
(540,532,"Independence Day",0,"7/28-29"),
(541,532,"Santa Rosa de Lima Day",0,"8/30"),
(542,532,"Battle of Angamos Day",0,"10/8"),
(543,532,"All Saints' Day",0,"11/1"),
(544,532,"Immaculate Conception Day",0,"12/8"),
(545,532,"Christmas Day",0,"12/25"),
(546,18,"Venezuela",1,""),
(547,546,"New Year's Day",0,"1/1"),
(548,546,"Epiphany",0,"1/Mon/Second"),
(549,546,"Carnival (first day)",0,"Easter-48"),
(550,546,"Carnival (second day)",0,"Easter-47"),
(551,546,"San Jose Day",0,"3/Mon/Third"),
(552,546,"Maundy Thursday",0,"Easter-3"),
(553,546,"Good Friday",0,"Easter-2"),
(554,546,"Easter",0,"Easter"),
(555,546,"Declaration of Independence Day",0,"4/19"),
(556,546,"Labour Day",0,"5/1"),
(557,546,"Ascension Day",0,"Easter+39"),
(558,546,"Corpus Christi Day",0,"Easter+60"),
(559,546,"Battle of Carabobo Day",0,"6/24"),
(560,546,"Saint Peter and Saint Paul's Day",0,"6/29"),
(561,546,"Independence Day",0,"7/5"),
(562,546,"Simon Bolivar's Birthday",0,"7/24"),
(563,546,"Assumption Day",0,"8/15"),
(564,546,"Dia de la Raza / Columbus Day",0,"10/12"),
(565,546,"All Saints' Day",0,"11/1"),
(566,546,"Immaculate Conception Day",0,"12/8"),
(567,546,"Christmas Day",0,"12/25"),
(568,546,"New Year's Eve",0,"12/31"),
(569,18,"Mozambique",1,""),
(570,569,"New Year's Day",0,"1/1"),
(571,569,"Heroes' Day",0,"2/3"),
(572,569,"Women's Day",0,"4/7"),
(573,569,"Labour Day",0,"5/1"),
(574,569,"Independence Day",0,"6/25"),
(575,569,"Victory Day",0,"9/7"),
(576,569,"Armed Forces Day",0,"9/25"),
(577,569,"Maputo Day",0,"11/10"),
(578,569,"Christmas Day / Family Day",0,"12/25"),
(579,18,"Australia",1,""),
(580,579,"New Year's Day",0,"1/1/Sat+2,Sun+1"),
(581,579,"Australia Day",0,"1/26/Sat+2,Sun+1"),
(582,579,"Good Friday",0,"Easter-2"),
(583,579,"Holy Saturday",0,"Easter-1"),
(584,579,"Easter",0,"Easter"),
(585,579,"Easter Monday",0,"Easter+1"),
(586,579,"ANZAC Day",0,"4/25/Sat+2,Sun+1"),
(587,579,"Queen's Birthday",0,"6/Mon/Second"),
(588,579,"Melbourne Cup Day",0,"11/Tue/First"),
(589,579,"Christmas Day",0,"12/25"),
(590,579,"Christmas Holiday",0,"12/25/Sat+2,Sun+2"),
(591,579,"Boxing Day",0,"12/26/Sat+2,Sun+2"),
(592,579,"Daylight Saving Time (start)",0,"10/Sun/Last"),
(593,579,"Daylight Saving Time (end)",0,"3/Sun/Last"),
(594,18,"Madagascar",1,""),
(595,594,"New Year's Day",0,"1/1"),
(596,594,"Easter",0,"Easter"),
(597,594,"Easter Monday",0,"Easter+1"),
(598,594,"Memorial Day",0,"3/29"),
(599,594,"Labour Day",0,"5/1"),
(600,594,"Ascension Day",0,"Easter+39"),
(601,594,"Whit Sunday",0,"Easter+49"),
(602,594,"Whit Monday",0,"Easter+50"),
(603,594,"O .A.U. Day",0,"5/25"),
(604,594,"Independence Day",0,"6/26"),
(605,594,"Assumption Day",0,"8/15"),
(606,594,"All Saints' Day",0,"11/1"),
(607,594,"Christmas Day",0,"12/25"),
(608,18,"Chile",1,""),
(609,608,"New Year's Day",0,"1/1"),
(610,608,"Good Friday",0,"Easter-2"),
(611,608,"Holy Saturday",0,"Easter-1"),
(612,608,"Easter",0,"Easter"),
(613,608,"Labour Day",0,"5/1"),
(614,608,"Navy day",0,"5/21"),
(615,608,"Corpus Christi Day",0,"Easter+60"),
(616,608,"Saint Peter and Saint Paul's Day",0,"6/26/Sat+2,Sun+1"),
(617,608,"Assumption Day",0,"8/15"),
(618,608,"Independence Day",0,"9/18"),
(619,608,"Armed Forces Day",0,"9/19"),
(620,608,"Columbus Day",0,"10/Mon/Second"),
(621,608,"All Saints' Day",0,"11/1"),
(622,608,"Immaculate Conception Day",0,"12/8"),
(623,608,"Christmas Day",0,"12/25"),
(624,608,"Daylight Saving Time (start)",0,"10/Sat/Second"),
(625,608,"Daylight Saving Time (end)",0,"3/Sat/Second"),
(626,18,"Ecuador",1,""),
(627,626,"New Year's Day",0,"1/1"),
(628,626,"Carnival (first day)",0,"Easter-48"),
(629,626,"Carnival (second day)",0,"Easter-47"),
(630,626,"Maundy Thursday",0,"Easter-3"),
(631,626,"Good Friday",0,"Easter-2"),
(632,626,"Easter",0,"Easter"),
(633,626,"Labour Day",0,"5/1"),
(634,626,"Battle of Pichincha Day",0,"5/24"),
(635,626,"Simon Bolivar Day",0,"7/24"),
(636,626,"Independence Day",0,"8/10"),
(637,626,"Guayaquil Day",0,"10/9"),
(638,626,"Dia de la Raza / Columbus Day",0,"10/12"),
(639,626,"All Souls' Day",0,"11/2"),
(640,626,"Cuenca Day",0,"11/3"),
(641,626,"Quito Day",0,"12/6"),
(642,626,"Christmas Day",0,"12/25"),
(643,626,"New Year's Eve",0,"12/31"),
(644,18,"Guatemala",1,""),
(645,644,"New Year's Day",0,"1/1"),
(646,644,"Maundy Thursday",0,"Easter-3"),
(647,644,"Good Friday",0,"Easter-2"),
(648,644,"Holy Saturday",0,"Easter-1"),
(649,644,"Easter",0,"Easter"),
(650,644,"Labour Day",0,"5/1"),
(651,644,"Army Day",0,"6/30"),
(652,644,"Assumption Day",0,"8/15"),
(653,644,"Independence Day",0,"9/15"),
(654,644,"Revolution Day",0,"10/20"),
(655,644,"All Saints' Day",0,"11/1"),
(656,644,"Christmas Eve",0,"12/24"),
(657,644,"Christmas Day",0,"12/25"),
(658,644,"New Year's Eve",0,"12/31"),
(659,18,"Angola",1,""),
(660,659,"New Year's Day",0,"1/1"),
(661,659,"Memorial Day",0,"1/4"),
(662,659,"Liberation Day",0,"2/4"),
(663,659,"Mardi Gras",0,"Easter-40"),
(664,659,"Women's Day",0,"3/8"),
(665,659,"Good Friday",0,"Easter-2"),
(666,659,"Easter",0,"Easter"),
(667,659,"Peace Day",0,"4/4"),
(668,659,"Labour Day",0,"5/1"),
(669,659,"O .A.U. Day",0,"5/25"),
(670,659,"children's Day",0,"6/1"),
(671,659,"National Day",0,"9/17"),
(672,659,"All Souls' Day",0,"11/2"),
(673,659,"Independence Day",0,"11/11"),
(674,659,"Christmas Day",0,"12/25"),
(675,18,"Zimbabwe",1,""),
(676,675,"New Year's Day",0,"1/1"),
(677,675,"Good Friday",0,"Easter-2"),
(678,675,"Holy Saturday",0,"Easter-1"),
(679,675,"Easter",0,"Easter"),
(680,675,"Easter Monday",0,"Easter+1"),
(681,675,"Independence Day",0,"4/18"),
(682,675,"Labour Day",0,"5/1"),
(683,675,"Africa day",0,"5/25"),
(684,675,"Heroes' Day",0,"8/11"),
(685,675,"Armed Forces Day",0,"8/12"),
(686,675,"Unity Day",0,"12/22/Sat+2,Sun+1"),
(687,675,"Christmas Day",0,"12/25"),
(688,675,"Boxing Day",0,"12/26"),
(689,18,"Cuba",1,""),
(690,689,"Liberation Day",0,"1/1"),
(691,689,"Labour Day",0,"5/1"),
(692,689,"Rebellion Anniversary",0,"7/25-27"),
(693,689,"Independence Wars Day",0,"10/10"),
(694,689,"Christmas Day",0,"12/25"),
(695,689,"Daylight Saving Time (start)",0,"4/1"),
(696,689,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(697,18,"Greece",1,""),
(698,697,"New Year's Day",0,"1/1"),
(699,697,"Epiphany",0,"1/6"),
(700,697,"Shrove Monday",0,"Paskha-48"),
(701,697,"Ash Wednesday",0,"Paskha-46"),
(702,697,"Independence Day",0,"3/25"),
(703,697,"Orthodox Good Friday",0,"Paskha-2"),
(704,697,"Labour Day",0,"5/1"),
(705,697,"Orthodox Easter",0,"Paskha"),
(706,697,"Orthodox Easter Monday",0,"Paskha+1"),
(707,697,"Whit Sunday",0,"Paskha+49"),
(708,697,"Whit Monday",0,"Paskha+50"),
(709,697,"Assumption Day",0,"8/15"),
(710,697,"Ochi Day",0,"10/28"),
(711,697,"Christmas Day",0,"12/25"),
(712,697,"Second Day of Christmas",0,"12/26"),
(713,697,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(714,697,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(715,18,"Luxembourg",1,""),
(716,715,"New Year's Day",0,"1/1"),
(717,715,"Carnival",0,"Easter-48"),
(718,715,"Easter",0,"Easter"),
(719,715,"Easter Monday",0,"Easter+1"),
(720,715,"May Day",0,"5/1"),
(721,715,"Ascension Day",0,"Easter+39"),
(722,715,"Whit Sunday",0,"Easter+49"),
(723,715,"Whit Monday",0,"Easter+50"),
(724,715,"National Day / Grand Duc's Birthday",0,"6/23/Sat+2,Sun+1"),
(725,715,"Assumption Day",0,"8/15"),
(726,715,"Luxembourg City Kermesse",0,"9/1"),
(727,715,"All Saints' Day",0,"11/1"),
(728,715,"Christmas Day",0,"12/25"),
(729,715,"Saint Stephen's Day",0,"12/26"),
(730,715,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(731,715,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(732,18,"Ireland",1,""),
(733,732,"New Year's Day",0,"1/1"),
(734,732,"Saint Patrick's Day",0,"3/17/Sat+2,Sun+1"),
(735,732,"Good Friday",0,"Easter-2"),
(736,732,"Easter",0,"Easter"),
(737,732,"Easter Monday",0,"Easter+1"),
(738,732,"May Bank Holiday",0,"5/Mon/First"),
(739,732,"June Bank Holiday",0,"6/Mon/First"),
(740,732,"Summer Bank Holiday",0,"8/Mon/First"),
(741,732,"Hallowe'en Bank Holiday",0,"10/Mon/Last"),
(742,732,"Christmas Day",0,"12/25"),
(743,732,"Saint Stephen's Day",0,"12/26"),
(744,732,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(745,732,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(746,18,"Cyprus",1,""),
(747,746,"New Year's Day",0,"1/1"),
(748,746,"Epiphany",0,"1/6"),
(749,746,"Clean Monday / Start of Lent",0,"Paskha-48"),
(750,746,"Greek National Day",0,"3/25"),
(751,746,"Greek Cypriot National Day",0,"4/1"),
(752,746,"Orthodox Good Friday",0,"Paskha-2"),
(753,746,"Labour Day",0,"5/1"),
(754,746,"Orthodox Easter",0,"Paskha"),
(755,746,"Orthodox Easter Monday",0,"Paskha+1"),
(756,746,"Orthodox Easter Tuesday",0,"Paskha+2"),
(757,746,"Whit Sunday",0,"Paskha+49"),
(758,746,"Whit Monday",0,"Paskha+50"),
(759,746,"Assumption Day",0,"8/15"),
(760,746,"Independence Day",0,"10/1"),
(761,746,"Ochi Day",0,"10/28"),
(762,746,"Christmas Day",0,"12/25"),
(763,746,"Second Day of Christmas",0,"12/26"),
(764,746,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(765,746,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(766,18,"Malta",1,""),
(767,766,"New Year's Day",0,"1/1"),
(768,766,"Saint Paul's Shipwreck",0,"2/10"),
(769,766,"Saint Joseph's Day",0,"3/19"),
(770,766,"Good Friday",0,"Easter-2"),
(771,766,"Easter",0,"Easter"),
(772,766,"Freedom Day",0,"3/31"),
(773,766,"Labour Day",0,"5/1"),
(774,766,"Commemoration of 1919 Uprising",0,"6/7"),
(775,766,"Saint Peter and Saint Paul's Day",0,"6/29"),
(776,766,"Assumption Day",0,"8/15"),
(777,766,"Our Lady of the Victories' Day",0,"9/8"),
(778,766,"Independence Day",0,"9/21"),
(779,766,"Immaculate Conception Day",0,"12/8"),
(780,766,"Republic Day",0,"12/13"),
(781,766,"Christmas Day",0,"12/25"),
(782,766,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(783,766,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(784,18,"Lithuania",1,""),
(785,784,"New Year's Day",0,"1/1"),
(786,784,"Independence Day",0,"2/16"),
(787,784,"Restoration Day",0,"3/11"),
(788,784,"Easter",0,"Easter"),
(789,784,"Easter Monday",0,"Easter+1"),
(790,784,"Labour Day",0,"5/1"),
(791,784,"Statehood Day / Mindaugas' Coronation",0,"7/6"),
(792,784,"Assumption Day",0,"8/15"),
(793,784,"All Saints' Day",0,"11/1"),
(794,784,"Christmas Day",0,"12/25"),
(795,784,"Boxing Day",0,"12/26"),
(796,784,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(797,784,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(798,18,"Sweden",1,""),
(799,798,"New Year's Day",0,"1/1"),
(800,798,"Epiphany Eve",0,"1/5"),
(801,798,"Epiphany",0,"1/6"),
(802,798,"Maundy Thursday",0,"Easter-3"),
(803,798,"Good Friday",0,"Easter-2"),
(804,798,"Easter",0,"Easter"),
(805,798,"Easter Monday",0,"Easter+1"),
(806,798,"Valborg's Eve",0,"4/30"),
(807,798,"Labour Day",0,"5/1"),
(808,798,"Ascension Day",0,"Easter+39"),
(809,798,"Whit Sunday",0,"Easter+49"),
(810,798,"National Day",0,"6/6"),
(811,798,"Midsummer's Eve",0,"6/Fri/Fourth"),
(812,798,"Midsummer's Day",0,"6/Sat/Fourth"),
(813,798,"All Hallow's Eve",0,"11/Fri/First"),
(814,798,"All Saints' Day",0,"11/Sat/First"),
(815,798,"Christmas Eve",0,"12/24"),
(816,798,"Christmas Day",0,"12/25"),
(817,798,"Second Day of Christmas",0,"12/26"),
(818,798,"New Year's Eve",0,"12/31"),
(819,798,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(820,798,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(821,18,"Finland",1,""),
(822,821,"New Year's Day",0,"1/1"),
(823,821,"Epiphany",0,"1/6"),
(824,821,"Good Friday",0,"Easter-2"),
(825,821,"Easter",0,"Easter"),
(826,821,"Easter Monday",0,"Easter+1"),
(827,821,"May Day / Vappu",0,"5/1"),
(828,821,"Ascension Day",0,"Easter+39"),
(829,821,"Whit Sunday",0,"Easter+49"),
(830,821,"Midsummer's Eve",0,"6/Fri/Fourth"),
(831,821,"Midsummer's Day",0,"6/Sat/Fourth"),
(832,821,"All Saints' Day",0,"11/Sat/First"),
(833,821,"Independence Day",0,"12/6"),
(834,821,"Christmas Eve",0,"12/24"),
(835,821,"Christmas Day",0,"12/25"),
(836,821,"Boxing Day",0,"12/26"),
(837,821,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(838,821,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(839,18,"Norway",1,""),
(840,839,"New Year's Day",0,"1/1"),
(841,839,"Maundy Thursday",0,"Easter-3"),
(842,839,"Good Friday",0,"Easter-2"),
(843,839,"Easter",0,"Easter"),
(844,839,"Easter Monday",0,"Easter+1"),
(845,839,"Labour Day",0,"5/1"),
(846,839,"Ascension Day",0,"Easter+39"),
(847,839,"Whit Sunday",0,"Easter+49"),
(848,839,"Whit Monday",0,"Easter+50"),
(849,839,"Constitution Day",0,"5/17"),
(850,839,"Christmas Eve",0,"12/24"),
(851,839,"Christmas Day",0,"12/25"),
(852,839,"Boxing Day",0,"12/26"),
(853,839,"New Year's Eve",0,"12/31"),
(854,18,"Bolivia",1,""),
(855,854,"New Year's Day",0,"1/1"),
(856,854,"Carnival (first day)",0,"Easter-48"),
(857,854,"Carnival (second day)",0,"Easter-47"),
(858,854,"Good Friday",0,"Easter-2"),
(859,854,"Easter",0,"Easter"),
(860,854,"Labour Day",0,"5/1"),
(861,854,"Corpus Christi Day",0,"Easter+60"),
(862,854,"Independence Day",0,"8/6"),
(863,854,"All Souls' Day",0,"11/2"),
(864,854,"Christmas Day",0,"12/25"),
(865,18,"Paraguay",1,""),
(866,865,"New Year's Day",0,"1/1"),
(867,865,"Battle of Cerro Cora Day",0,"3/1"),
(868,865,"Maundy Thursday",0,"Easter-3"),
(869,865,"Good Friday",0,"Easter-2"),
(870,865,"Easter",0,"Easter"),
(871,865,"Labour Day",0,"5/1"),
(872,865,"Independence Day",0,"5/15"),
(873,865,"Chaco Armistice Day",0,"6/12"),
(874,865,"Founding of Asuncion Day",0,"8/15"),
(875,865,"Battle of Boqueron Day",0,"9/29"),
(876,865,"Immaculate Conception Day",0,"12/8"),
(877,865,"Christmas Day",0,"12/25"),
(878,865,"Daylight Saving Time (start)",0,"9/Sun/First"),
(879,865,"Daylight Saving Time (end)",0,"4/Sun/First"),
(880,18,"New Zealand",1,""),
(881,880,"New Year's Day",0,"1/1/Sat+2,Sun+1"),
(882,880,"Second Day of New Year",0,"1/2/Sat+2,Sun+2,Mon+1"),
(883,880,"Waitangi Day",0,"2/6"),
(884,880,"Good Friday",0,"Easter-2"),
(885,880,"Easter",0,"Easter"),
(886,880,"Easter Monday",0,"Easter+1"),
(887,880,"ANZAC Day",0,"4/25"),
(888,880,"Queen's Birthday",0,"6/Mon/First"),
(889,880,"Labour Day",0,"10/Mon/Fourth"),
(890,880,"Christmas Day",0,"12/25"),
(891,880,"Boxing Day",0,"12/26/Sat+2,Sun+1"),
(892,880,"Christmas Holiday",0,"12/25/Sat+3,Sun+2"),
(893,880,"Daylight Saving Time (start)",0,"10/Sun/First"),
(894,880,"Daylight Saving Time (end)",0,"3/Sun/Third"),
(895,18,"Bahamas",1,""),
(896,895,"New Year's Day",0,"1/1"),
(897,895,"New Year's Holiday",0,"1/1/Sat+2,Sun+1"),
(898,895,"Good Friday",0,"Easter-2"),
(899,895,"Easter",0,"Easter"),
(900,895,"Easter Monday",0,"Easter+1"),
(901,895,"Whit Sunday",0,"Easter+49"),
(902,895,"Whit Monday",0,"Easter+50"),
(903,895,"Labour Day",0,"6/Fri/First"),
(904,895,"Independence Day",0,"7/10"),
(905,895,"Emancipation Day",0,"8/Mon/First"),
(906,895,"Discovery Day",0,"10/Mon/Second"),
(907,895,"Christmas Day",0,"12/25"),
(908,895,"Christmas Holiday",0,"12/25/Sat+2,Sun+1"),
(909,895,"Boxing Day",0,"12/26/Sat+2,Sun+2"),
(910,895,"Daylight Saving Time (start)",0,"4/Sun/First"),
(911,895,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(912,18,"Barbados",1,""),
(913,912,"New Year's Day",0,"1/1"),
(914,912,"New Year's Holiday",0,"1/1/Sun+1"),
(915,912,"Errol Barrow Day",0,"1/21/Sun+1"),
(916,912,"Good Friday",0,"Easter-2"),
(917,912,"Easter",0,"Easter"),
(918,912,"Easter Monday",0,"Easter+1"),
(919,912,"Heroes'Day",0,"4/28/Sun+1"),
(920,912,"Labour Day",0,"5/1/Sun+1"),
(921,912,"Whit Sunday",0,"Easter+49"),
(922,912,"Whit Monday",0,"Easter+50"),
(923,912,"Kadooment Day",0,"8/Mon/First"),
(924,912,"Emancipation Day",0,"8/Tue/First"),
(925,912,"Independence Day",0,"11/30/Sun+1"),
(926,912,"Christmas Day",0,"12/25"),
(927,912,"Christmas Holiday",0,"12/25/Sun+2"),
(928,912,"Boxing Day",0,"12/26/Sun+1"),
(929,18,"Belize",1,""),
(930,929,"New Year's Day",0,"1/1/Sun+1"),
(931,929,"Baron Bliss Day",0,"3/9/Sun+1"),
(932,929,"Good Friday",0,"Easter-2"),
(933,929,"Holy Saturday",0,"Easter-1"),
(934,929,"Easter",0,"Easter"),
(935,929,"Easter Monday",0,"Easter+1"),
(936,929,"Labour Day",0,"5/1/Sun+1"),
(937,929,"Commonwealth Day",0,"5/24/Sun+1"),
(938,929,"National Day",0,"9/10/Sun+1"),
(939,929,"Independence Day",0,"9/21/Sun+1"),
(940,929,"Pan American Day / Columbus Day",0,"10/12/Sun+1"),
(941,929,"Garifuna Day",0,"11/19/Sun+1"),
(942,929,"Bank Holiday",0,"12/1/Sun+1"),
(943,929,"Christmas Day",0,"12/25/Sun+1"),
(944,929,"Second Day of Christmas",0,"12/26/Sun+1,Mon+1"),
(945,18,"Bermuda",1,""),
(946,945,"New Year's Day",0,"1/1/Sat+2,Sun+1"),
(947,945,"Good Friday",0,"Easter-2"),
(948,945,"Easter",0,"Easter"),
(949,945,"Bermuda Day",0,"5/24/Sat+2,Sun+1"),
(950,945,"Queen's Birthday",0,"6/Mon/Second"),
(951,945,"Emancipation Day",0,"7/Thu/Last"),
(952,945,"Somers Day",0,"7/Fri/Last"),
(953,945,"Labour Day",0,"9/Mon/First"),
(954,945,"Remembrance Day",0,"11/11/Sat+2,Sun+1"),
(955,945,"Christmas Day",0,"12/25/Sat+2,Sun+2"),
(956,945,"Boxing Day",0,"12/26/Sat+2,Sun+2"),
(957,18,"Botswana",1,""),
(958,957,"New Year's Day",0,"1/1"),
(959,957,"Second Day of New Year",0,"1/2"),
(960,957,"Good Friday",0,"Easter-2"),
(961,957,"Easter",0,"Easter"),
(962,957,"Easter Monday",0,"Easter+1"),
(963,957,"Labour Day",0,"5/1"),
(964,957,"Ascension Day",0,"Easter+39"),
(965,957,"Sir Seretse Khama Day",0,"7/1"),
(966,957,"President's Day (first day)",0,"7/Tue/Third"),
(967,957,"President's Day (second day)",0,"7/Wed/Third"),
(968,957,"Botswana Day (first day)",0,"9/30"),
(969,957,"Botswana Day (second day)",0,"10/1"),
(970,957,"Christmas Day",0,"12/25"),
(971,957,"Boxing Day",0,"12/26"),
(972,957,"Third Day of Christmas",0,"12/27"),
(973,18,"Costa Rica",1,""),
(974,973,"New Year's Day",0,"1/1"),
(975,973,"San Jose Day",0,"3/19"),
(976,973,"Maundy Thursday",0,"Easter-3"),
(977,973,"Good Friday",0,"Easter-2"),
(978,973,"Holy Saturday",0,"Easter-1"),
(979,973,"Easter",0,"Easter"),
(980,973,"Juan Santamaria Day",0,"4/11"),
(981,973,"Labour Day",0,"5/1"),
(982,973,"Corpus Christi Day",0,"Easter+60"),
(983,973,"Saint Peter and Saint Paul's Day",0,"6/29"),
(984,973,"Guanacoste Annexation Day",0,"7/25"),
(985,973,"Virgin of the Angels Day",0,"8/2"),
(986,973,"Independence Day",0,"9/15"),
(987,973,"Dia de la Raza / Columbus Day",0,"10/12"),
(988,973,"Immaculate Conception Day",0,"12/8"),
(989,973,"Christmas Eve",0,"12/24"),
(990,973,"Christmas Day",0,"12/25"),
(991,973,"New Year's Eve",0,"12/31"),
(992,18,"Zambia",1,""),
(993,992,"New Year's Day",0,"1/1/Sun+1"),
(994,992,"Youth Day",0,"3/Fri/Second"),
(995,992,"Good Friday",0,"Easter-2"),
(996,992,"Holy Saturday",0,"Easter-1"),
(997,992,"Easter",0,"Easter"),
(998,992,"Easter Monday",0,"Easter+1"),
(999,992,"Labour Day",0,"5/1/Sun+1"),
(1000,992,"Africa Day",0,"5/25/Sun+1"),
(1001,992,"Heroes' Day",0,"7/Mon/First"),
(1002,992,"Unity Day",0,"7/Tue/First"),
(1003,992,"Farmers' Day",0,"8/Mon/First"),
(1004,992,"Independence Day",0,"10/24/Sun+1"),
(1005,992,"Christmas Day",0,"12/25/Sun+1"),
(1006,18,"Uruguay",1,""),
(1007,1006,"New Year's Day",0,"1/1"),
(1008,1006,"Epiphany",0,"1/Mon/-4"),
(1009,1006,"Carnival (first day)",0,"Easter-48"),
(1010,1006,"Carnival (second day)",0,"Easter-47"),
(1011,1006,"Maundy Thursday",0,"Easter-3"),
(1012,1006,"Good Friday",0,"Easter-2"),
(1013,1006,"Holy Saturday",0,"Easter-1"),
(1014,1006,"Easter",0,"Easter"),
(1015,1006,"Landing of the 33 Patriots Day",0,"4/17/Sat+2,Sun+1"),
(1016,1006,"Labour Day",0,"5/1"),
(1017,1006,"Battle of Las Piedras Day",0,"5/18"),
(1018,1006,"Jose Artigas' Birthday",0,"6/19"),
(1019,1006,"Constitution Day",0,"7/18"),
(1020,1006,"Independence Day",0,"8/25"),
(1021,1006,"Dia de la Raza / Columbus Day",0,"10/10/Sat+2,Sun+1,Tue+6,Wed+5,Thu+4,Fri+3"),
(1022,1006,"All Souls' Day",0,"11/2/Tue-1,Wed+5,Thu+4,Fri+3,Sat+2,Sun+1"),
(1023,1006,"Christmas Day",0,"12/25"),
(1024,18,"Seychelles",1,""),
(1025,1024,"New Year's Day",0,"1/1"),
(1026,1024,"Second Day of New Year",0,"1/2"),
(1027,1024,"Good Friday",0,"Easter-2"),
(1028,1024,"Holy Saturday",0,"Easter-1"),
(1029,1024,"Easter",0,"Easter"),
(1030,1024,"Labour Day",0,"5/1"),
(1031,1024,"Liberation Day",0,"6/5"),
(1032,1024,"Corpus Christi Day",0,"Easter+60"),
(1033,1024,"National Day",0,"6/18"),
(1034,1024,"Independence Day",0,"6/29"),
(1035,1024,"Assumption Day",0,"8/15"),
(1036,1024,"All Saints' Day",0,"11/1"),
(1037,1024,"Immaculate Conception Day",0,"12/8"),
(1038,1024,"Christmas Day",0,"12/25"),
(1039,18,"Puerto Rico",1,""),
(1040,1039,"New Year's Day",0,"1/1"),
(1041,1039,"Epiphany",0,"1/6"),
(1042,1039,"De Hostos' Birthday",0,"1/Mon/Second"),
(1043,1039,"Martin Luther King's Day",0,"1/Mon/Third"),
(1044,1039,"Presidents'Day",0,"2/Mon/Third"),
(1045,1039,"Emancipation Day",0,"3/22"),
(1046,1039,"Good Friday",0,"Easter-2"),
(1047,1039,"Easter",0,"Easter"),
(1048,1039,"De Diego's Birthday",0,"4/Mon/Third"),
(1049,1039,"Memorial Day",0,"5/Mon/Last"),
(1050,1039,"U.S. Independence Day",0,"7/4"),
(1051,1039,"Munoz Rivera's Birthday",0,"7/Mon/Third"),
(1052,1039,"Constitution Day",0,"7/25"),
(1053,1039,"Barbosa's Birthday",0,"7/28"),
(1054,1039,"Labour Day",0,"9/Mon/First"),
(1055,1039,"Columbus Day",0,"10/Mon/Second"),
(1056,1039,"Veterans' Day",0,"11/11"),
(1057,1039,"Discovery Day",0,"11/20"),
(1058,1039,"Thanksgiving Day",0,"11/Thu/Fourth"),
(1059,1039,"Christmas Day",0,"12/25"),
(1060,18,"Papua New Guinea",1,""),
(1061,1060,"New Year's Day",0,"1/1/Sun+1"),
(1062,1060,"Good Friday",0,"Easter-2"),
(1063,1060,"Holy Saturday",0,"Easter-1"),
(1064,1060,"Easter",0,"Easter"),
(1065,1060,"Easter Monday",0,"Easter+1"),
(1066,1060,"Queen's Birthday",0,"6/Mon/Second"),
(1067,1060,"Remembrance Day",0,"7/23/Sun+1"),
(1068,1060,"Independence Day",0,"9/16/Sun+1"),
(1069,1060,"Christmas Day",0,"12/25/Sun+2"),
(1070,1060,"Boxing Day",0,"12/26/Sun+1"),
(1071,18,"Panama",1,""),
(1072,1071,"New Year's Day",0,"1/1/Sun+1"),
(1073,1071,"Martyrs' Day",0,"1/9/Sun+1"),
(1074,1071,"Carnival (first day)",0,"Easter-48"),
(1075,1071,"Carnival (second day)",0,"Easter-47"),
(1076,1071,"Ash Wednesday",0,"Easter-46"),
(1077,1071,"Maundy Thursday",0,"Easter-3"),
(1078,1071,"Good Friday",0,"Easter-2"),
(1079,1071,"Holy Saturday",0,"Easter-1"),
(1080,1071,"Easter",0,"Easter"),
(1081,1071,"Labour Day",0,"5/1/Sun+1"),
(1082,1071,"Assumption Day",0,"8/15"),
(1083,1071,"Independence from Columbia Day",0,"11/3/Sun+1"),
(1084,1071,"Flag Day",0,"11/4/Sun+1"),
(1085,1071,"Colon Independence Day",0,"11/5/Sun+1"),
(1086,1071,"Cry of Independence Day",0,"11/10/Sun+1"),
(1087,1071,"Independence from Spain Day",0,"11/28/Sun+1"),
(1088,1071,"Christmas Eve",0,"12/24"),
(1089,1071,"Christmas Day",0,"12/25/Sun+1"),
(1090,1071,"New Year's Eve",0,"12/31"),
(1091,18,"Nicaragua",1,""),
(1092,1091,"New Year's Day",0,"1/1"),
(1093,1091,"Maundy Thursday",0,"Easter-3"),
(1094,1091,"Good Friday",0,"Easter-2"),
(1095,1091,"Holy Saturday",0,"Easter-1"),
(1096,1091,"Easter",0,"Easter"),
(1097,1091,"Labour Day",0,"5/1/Sun+1"),
(1098,1091,"Revolution Day",0,"7/19"),
(1099,1091,"Santo Domingo Day",0,"8/1"),
(1100,1091,"San Jacinto Battle Day",0,"9/14"),
(1101,1091,"Independence Day",0,"9/15"),
(1102,1091,"All Souls' Day",0,"11/2"),
(1103,1091,"Immaculate Conception Day",0,"12/8"),
(1104,1091,"Christmas Day",0,"12/25"),
(1105,18,"Monaco",1,""),
(1106,1105,"New Year's Day",0,"1/1"),
(1107,1105,"Saint Devote's Day",0,"1/27"),
(1108,1105,"Mardi Gras",0,"Easter-40"),
(1109,1105,"Mi-careme",0,"Easter-19"),
(1110,1105,"Good Friday",0,"Easter-2"),
(1111,1105,"Easter",0,"Easter"),
(1112,1105,"Easter Monday",0,"Easter+1"),
(1113,1105,"Labour Day",0,"5/1"),
(1114,1105,"Ascension Day",0,"Easter+39"),
(1115,1105,"Whit Sunday",0,"Easter+49"),
(1116,1105,"Whit Monday",0,"Easter+50"),
(1117,1105,"Corpus Christi Day",0,"Easter+60"),
(1118,1105,"All Saints' Day",0,"11/1"),
(1119,1105,"Immaculate Conception Day",0,"12/8"),
(1120,1105,"Christmas Eve",0,"12/24"),
(1121,1105,"Christmas Day",0,"12/25"),
(1122,1105,"New Year's Eve",0,"12/31"),
(1123,1105,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(1124,1105,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(1125,18,"North Korea",1,""),
(1126,1125,"New Year's Day",0,"1/1"),
(1127,1125,"Kim Jong II's Anniversary",0,"2/16-17"),
(1128,1125,"Kim II Sung's Birthday",0,"4/15"),
(1129,1125,"Army Day",0,"4/25"),
(1130,1125,"Labour Day",0,"5/1"),
(1131,1125,"Victory Day",0,"7/27"),
(1132,1125,"Freedom Day",0,"8/15"),
(1133,1125,"Communist Party Foundation",0,"9/9"),
(1134,1125,"Workers' Party Foundation",0,"10/10"),
(1135,1125,"Constitution Day",0,"12/27"),
(1136,18,"Jamaica",1,""),
(1137,1136,"New Year's Day",0,"1/1/Sun+1"),
(1138,1136,"Ash Wednesday",0,"Easter-46"),
(1139,1136,"Good Friday",0,"Easter-2"),
(1140,1136,"Easter",0,"Easter"),
(1141,1136,"Easter Monday",0,"Easter+1"),
(1142,1136,"Labour Day",0,"5/23/Sun+1"),
(1143,1136,"Emancipation Day",0,"8/1/Sun+1"),
(1144,1136,"Independence Day",0,"8/6/Sun+1"),
(1145,1136,"National Heroes'Day",0,"10/16"),
(1146,1136,"Christmas Day",0,"12/25/Sun+1"),
(1147,1136,"Boxing Day",0,"12/26/Sun+1,Mon+1"),
(1148,18,"Iceland",1,""),
(1149,1148,"New Year's Day",0,"1/1"),
(1150,1148,"Maundy Thursday",0,"Easter-3"),
(1151,1148,"Good Friday",0,"Easter-2"),
(1152,1148,"Easter",0,"Easter"),
(1153,1148,"Easter Monday",0,"Easter+1"),
(1154,1148,"First Day of Summer",0,"4/Thu/-2"),
(1155,1148,"Labour Day",0,"5/1"),
(1156,1148,"Ascension Day",0,"Easter+39"),
(1157,1148,"Whit Sunday",0,"Easter+49"),
(1158,1148,"Whit Monday",0,"Easter+50"),
(1159,1148,"National Day",0,"6/17"),
(1160,1148,"Commerce Day",0,"8/Mon/First"),
(1161,1148,"Christmas Eve",0,"12/24"),
(1162,1148,"Christmas Day",0,"12/25"),
(1163,1148,"Boxing Day",0,"12/26"),
(1164,1148,"New Year's Eve",0,"12/31"),
(1165,18,"Honduras",1,""),
(1166,1165,"New Year's Day",0,"1/1"),
(1167,1165,"Maundy Thursday",0,"Easter-3"),
(1168,1165,"Good Friday",0,"Easter-2"),
(1169,1165,"Holy Saturday",0,"Easter-1"),
(1170,1165,"Easter",0,"Easter"),
(1171,1165,"America's Day",0,"4/14"),
(1172,1165,"Labour Day",0,"5/1"),
(1173,1165,"Independence Day",0,"9/15"),
(1174,1165,"Francisco Morozan's Birthday",0,"10/3"),
(1175,1165,"Columbus Day",0,"10/12"),
(1176,1165,"Armed Forces Day",0,"10/21"),
(1177,1165,"Christmas Day",0,"12/25"),
(1178,18,"Haiti",1,""),
(1179,1178,"Independence Day",0,"1/1"),
(1180,1178,"Ancestors'Day",0,"1/2"),
(1181,1178,"Carnival (first day)",0,"Easter-48"),
(1182,1178,"Carnival (second day)",0,"Easter-47"),
(1183,1178,"Ash Wednesday",0,"Easter-46"),
(1184,1178,"Good Friday",0,"Easter-2"),
(1185,1178,"Easter",0,"Easter"),
(1186,1178,"Pan American Day",0,"4/14"),
(1187,1178,"Labour Day",0,"5/1"),
(1188,1178,"Ascension Day",0,"Easter+39"),
(1189,1178,"Flag Day / University Day",0,"5/18"),
(1190,1178,"Corpus Christi Day",0,"Easter+60"),
(1191,1178,"Assumption Day",0,"8/15"),
(1192,1178,"Jean-Jacques Dessalines' Death",0,"10/17"),
(1193,1178,"United Nations' Day",0,"10/24"),
(1194,1178,"All Saints' Day",0,"11/1"),
(1195,1178,"All Souls' Day",0,"11/2"),
(1196,1178,"Vertieres Battle Day",0,"18/11"),
(1197,1178,"Christmas Day",0,"12/25"),
(1198,18,"El Salvador",1,""),
(1199,1198,"New Year's Day",0,"1/1"),
(1200,1198,"Holy Wednesday",0,"Easter-4"),
(1201,1198,"Maundy Thursday",0,"Easter-3"),
(1202,1198,"Good Friday",0,"Easter-2"),
(1203,1198,"Holy Saturday",0,"Easter-1"),
(1204,1198,"Easter",0,"Easter"),
(1205,1198,"Labour Day",0,"5/1"),
(1206,1198,"Bank Holiday",0,"6/30/Sun+1"),
(1207,1198,"Transfiguration",0,"8/4"),
(1208,1198,"San Salvador Days",0,"8/5-6"),
(1209,1198,"Independence Day",0,"9/15"),
(1210,1198,"Columbus Day",0,"10/12"),
(1211,1198,"All Souls' Day",0,"11/2"),
(1212,1198,"Cry of Independence Day",0,"11/5"),
(1213,1198,"Christmas Eve",0,"12/24"),
(1214,1198,"Christmas Day",0,"12/25"),
(1215,1198,"New Year's Eve",0,"12/31"),
(1216,18,"Egypt",1,""),
(1217,1216,"Daylight Saving Time (start)",0,"4/Fri/Last"),
(1218,1216,"Daylight Saving Time (end)",0,"9/Thu/Last"),
(1219,18,"Namibia",1,""),
(1220,1219,"Daylight Saving Time (start)",0,"9/Sun/First"),
(1221,1219,"Daylight Saving Time (end)",0,"4/Sun/First"),
(1222,18,"Iraq",1,""),
(1223,1222,"Daylight Saving Time (start)",0,"4/1"),
(1224,1222,"Daylight Saving Time (end)",0,"10/1"),
(1225,18,"Lebanon",1,""),
(1226,1225,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(1227,1225,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(1228,18,"Kirgizstan",1,""),
(1229,1228,"Daylight Saving Time (start)",0,"3/Sun/Last"),
(1230,1228,"Daylight Saving Time (end)",0,"10/Sun/Last"),
(1231,18,"Syria",1,""),
(1232,1231,"Daylight Saving Time (start)",0,"4/1"),
(1233,1231,"Daylight Saving Time (end)",0,"10/1"),
(1234,18,"Tonga",1,""),
(1235,1234,"Daylight Saving Time (start)",0,"11/Sun/First"),
(1236,1234,"Daylight Saving Time (end)",0,"1/Sun/Last");

#
# Insert the icons (binary SQL scripts)
#
source schedwi_binary_data.sql

INSERT INTO schedwi_version VALUES ("0.1.0",NOW());

#******************************** End Of File ********************************
