/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
     
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
     
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*
 * job_parameters.c -- Excution parameters for a job
 *
 * The parameters (members of the job_parameters structure) are the
 * following:
 *
 *        command --> The command to be run
 *       username --> User name to use to run the command
 *    file_stdout --> File name of the command stdout redirection 
 *    file_stderr --> File name of the command stderr redirection 
 *       load_env --> Do the user environement should be load (1) or
 *                    not (0).  If yes (1), the user shell is run with the
 *                    command as a parameter (ie. /bin/sh -c command)
 *      arguments --> Arguments list
 *        environ --> Environement to load (several other variables will be
 *                    added, like SHELL, PATH, LOGNAME, ...)
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <job_parameters.h>


/*
 * Initialize a job_parameters structure
 */
void
init_job_parameters (job_parameters_t *j)
{
	if (j != NULL) {
		j->command = NULL;
		j->username = NULL;
		j->file_stdout = NULL;
		j->file_stderr = NULL;
		j->load_env = 0;
		init_argument (&(j->arguments));
		init_environment (&(j->environ));
	}
}


/*
 * Free the content of a job_paramters structure
 */
void
destroy_job_parameters (job_parameters_t *j)
{
	if (j != NULL) {
		if (j->command != NULL) {
			free (j->command);
		}
		if (j->username != NULL) {
			free (j->username);
		}
		if (j->file_stdout != NULL) {
			free (j->file_stdout);
		}
		if (j->file_stderr != NULL) {
			free (j->file_stderr);
		}
		destroy_argument (&(j->arguments));
		destroy_environment (&(j->environ));
		init_job_parameters (j);
	}
}


/*
 * Copy a string
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
static int
add_attr_job_parameters (char **attr, const char *str)
{
	char *tmp;
	size_t len;

#if HAVE_ASSERT_H
	assert (attr != NULL);
#endif

	if (str == NULL) {
		if (*attr != NULL) {
			free (*attr);
			*attr = NULL;
		}
		return 0;
	}

	len = schedwi_strlen (str);

	if (*attr == NULL || len >= schedwi_strlen (*attr)) {
		tmp = (char *)malloc (len + 1);
		if (tmp == NULL) {
			return -1;
		}
		if (*attr != NULL) {
			free (*attr);
		}
		*attr = tmp;
	}
	strcpy (*attr, str);
	return 0;
}


/*
 * Add the command to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_command_job_parameters (job_parameters_t *j, const char *command)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->command), command);
}


/*
 * Add the username to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_username_job_parameters (job_parameters_t *j, const char *username)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->username), username);
}


/*
 * Add the output file to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_stdout_job_parameters (job_parameters_t *j, const char *file)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->file_stdout), file);
}


/*
 * Add the output error file to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_stderr_job_parameters (job_parameters_t *j, const char *file)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->file_stderr), file);
}


/*
 * Set the `load user environement' flag
 */
void
add_loadenv_job_parameters (job_parameters_t *j, int flag)
{
	if (j != NULL) {
		j->load_env = flag;
	}
}


/*
 * Add an argument
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_arg_job_parameters (job_parameters_t *j, const char *arg)
{
	if (j == NULL) {
		return 0;
	}
	return add_arg (&(j->arguments), arg);
}


/*
 * Add an environment variable
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_env_job_parameters (job_parameters_t *j, const char *key, const char *val)
{
	if (j == NULL) {
		return 0;
	}
	return add_env (&(j->environ), key, val);
}

/*-----------------============== End Of File ==============-----------------*/
