/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* child_mgnt.c -- Manage running child processes in a linked list */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <lwc_linkedlist.h>
#include <child_mgnt.h>


/* List of the running jobs */
static lwc_LL *list_jobs = NULL;


/* Internal structure that define a running job */
struct child_str {
#if HAVE_PID_T
	pid_t child;
#else
	int child;
#endif
	char *job_id;
};
typedef struct child_str child_t;


/*
 * Comparison function between a process id and a child_t structure
 */
static int
compar_pid (const void *a, const void *b)
{
#if HAVE_PID_T
	const pid_t *p = a;
#else
	const int *p = a;
#endif
	const child_t *c = b;

#if HAVE_ASSERT_H
	assert (p != NULL && c != NULL);
#endif

	return *p - c->child;
}

/*
 * Comparison function between a job id and a child_t structure
 */
static int
compar_job_id (const void *a, const void *b)
{
	const char *id = a;
	const child_t *c = b;

#if HAVE_ASSERT_H
	assert (c != NULL);
#endif

	if (id == NULL) {
		if (c->job_id == NULL) {
			return 0;
		}
		else {
			return -1;
		}
	}
	if (c->job_id == NULL) {
		return 1;
	}
	return strcmp (id, c->job_id);
}

/*
 * Comparison function between two child_t structures
 */
static int
compar_child (const void *a, const void *b)
{
	const child_t *k = a;
	const child_t *c = b;

#if HAVE_ASSERT_H
	assert (k != NULL && c != NULL);
#endif

	return k->child - c->child;
}


/*
 * Add a new child to the list of the running children
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
#if HAVE_PID_T
int
add_child (pid_t child, const char *job_id)
#else
int
add_child (int child, const char *job_id)
#endif
{
	child_t *new_child;
	void *old;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	if (list_jobs == NULL) {
		list_jobs = lwc_newLL ();
		if (list_jobs == NULL) {
			return -1;
		}
	}

	new_child = (child_t *) malloc (sizeof (child_t));
	if (new_child == NULL) {
		return -1;
	}

	new_child->job_id = (char *) malloc (schedwi_strlen (job_id) + 1);
	if (new_child->job_id == NULL) {
		free (new_child);
		return -1;
	}
	strcpy (new_child->job_id, job_id);
	new_child->child = child;

	if (lwc_replaceLL (	list_jobs,
				new_child,
				compar_child,
				&old) != 0)

	{
		free (new_child->job_id);
		free (new_child);
		return -1;
	}
	if (old != NULL) {
		if (((child_t *)old)->job_id != NULL) {
			free (((child_t *)old)->job_id);
		}
		free (old);
	}

	return 0;
}


/*
 * Remove a child from the list of the running children.  Return in
 * *job_id the job_id of the removed child (must be freed my the caller using
 * free()).  If job_id is NULL, no value is returned in these variable.
 *
 * Return
 *    0 --> No error (*job_id is set if not NULL and must be freed by free())
 *   -1 --> Child not found in the list of children
 */
#if HAVE_PID_T
int
delete_child (pid_t child, char **job_id)
#else
int
delete_child (int child, char **job_id)
#endif
{
	child_t *ret_child;

	ret_child = lwc_delNodeLL (list_jobs, &child, compar_pid);
	if (ret_child == NULL) {
		return -1;
	}

	if (job_id != NULL) {
		*job_id = ret_child->job_id;
	}
	else {
		if (ret_child->job_id != NULL) {
			free (ret_child->job_id);
		}
	}

	free (ret_child);
	return 0;
}


/*
 * Terminate a job
 *
 * Return:
 *    0 --> No error
 *   -1 --> kill error (errno is set)
 *    1 --> Unknown job
 */
int
terminate_job (const char *job_id)
{
	child_t *ret_child;

	ret_child = lwc_searchLL (list_jobs, job_id, compar_job_id);
	if (ret_child == NULL) {
		return 1;
	}

	return kill (ret_child->child, SIGTERM);
}


/*
 * Write the list of all the running jobs to the provided socket
 */
void
print_jobs (net_id *fd)
{
	child_t *ret_child;

	lwc_rewindLL (list_jobs);
	while ((ret_child = lwc_nextLL (list_jobs)) != NULL) {
		if (ret_child->job_id != NULL) {
			net_write (	fd, ret_child->job_id,
					schedwi_strlen (ret_child->job_id));
			net_write (fd, " ", 1);
		}
	}
}

/*-----------------============== End Of File ==============-----------------*/
